/*****************************************************************************/
/*
                                 File.c

This module implements a full multi-threaded, AST-driven, asynchronous file 
send.  The AST-driven nature makes the code a little more difficult to follow, 
but creates a powerful, event-driven, multi-threaded server.  All of the 
necessary functions implementing this module are designed to be non-blocking. 

It uses the same buffer space as, and interworks with, NetWriteBuffered().  If 
there is already data (text) buffered in this area the file module will, for 
record-oriented, non-HTML-escaped transfers, continue to fill the area (using 
its own buffering function), flushing when and if necessary.  At end-of-file 
explicitly flush the buffer only if escaping HTML-forbidden characters, 
otherwise allow subsequent processing to do it as necessary.  For binary-mode 
files the buffer is explicitly flushed before commencing the file transfer. 

It handles both VARIABLE file (record-by-record) and STREAM file (virtual block-
oriented) transfers.  Even with record mode efficiency is increased by building
up a series of file records in available buffer space before transfering them
as one I/O to the network.  The binary mode reads as many virtual blocks as
will fit into the available buffer, then transfers these as one I/O to the
network.  STREAM FORMAT FILES ARE BY FAR THE MOST EFFICIENT TO HANDLE.

For text mode transfers (record-oriented) the completion of each read of the 
next record using FileNextRecord() calls FileNextRecordAST().  This function 
checks if the record buffer is full (by detecting if the last read failed with 
a "record too big for user's buffer" error).  If full it writes the buffer to 
the network using NetWrite(), which when complete calls FileNextRecord() as 
an AST routine.  This begins by getting the next record again (the one that 
failed).  If the buffer is not full FileNextRecordAST() just calls 
FileNextRecord() to get the next record, until the buffer fills. 

For binary mode transfers (non-record oriented) the completion of each read of 
the next series of blocks using FileNextBlocks() calls FileNextBlocksAST().  
This function writes the blocks (binary) using NetWrite() which when 
complete calls FileNextBlocks() as an AST.  This in turn calls 
FileNextBlocksAST(), etc., which drives the transfer until EOF. 

Once the file has been opened, the transfer becomes event-driven.

The module can encapsulate plain-text and escape HTML-forbidden characters. 

The only non-success status codes returned to the calling function are RMS 
codes.  These can be used to determine the success of the the file access.

Can work in conjunction with the Cache.c module.  Two different buffer spaces
are used.  A file read can be simultaneously used to send the data to the
client and load a cache buffer.  The request structure fields 'CacheContentPtr'
and 'CacheContentCount' being used in both block IO and record mode access to
track through the available buffer space, for this located in the cache
structure.  For non-cache-load reads uses the standard buffer space pointed to
by 'OutputBufferPtr' and for record mode access tracked using
'OutputBufferCurrentPtr' and 'OutputBufferRemaining' fields.

Implements defacto HTTP/1.0 persistent connections.  Only provide "keep-alive"
response if we're sending a file in binary mode and know its precise length.
An accurate "Content-Length:" field is vital for the client's correct reception
of the transmitted data.  Currently the only other time a "keep-alive" response
can be provided is when a "304 not-modified" header is returned (it has a
content length of precisely zero!).  October 1997: noted that Netscape
Navigator 3.n seems to pay no attention to a "Content-Length: 0" with a
"Keep-Alive:" connection, it just sits there waiting until the keep-alive time
closes the connection, after which it reports "document contains no data".
(IE 3.02 seems to behave correctly ;^)  I have therefore disabled persistent
connections for zero-length files.


VERSION HISTORY
---------------
14-MAY-98  MGD  request-specified content-type ("httpd=content&type=")
19-MAR-98  MGD  buffer VBN and first free byte for use by cache module
19-JAN-98  MGD  new NetWriteBuffered() and structures
07-JAN-98  MGD  groan, bugfix; record processing for files > 4096 bytes
                completely brain-dead ... sorry
22-NOV-97  MGD  sigh, bugfix; heap corruption by file cache
05-OCT-97  MGD  file cache,
                keep-alive now not used if the content-length is zero
17-SEP-97  MGD  if block-mode open is locked retry open in record-mode
17-AUG-97  MGD  message database,
                SYSUAF-authenticated users security-profile,
                addressed potential problem with FIXed and odd-byte records
08-JUN-97  MGD  if request "Pragma: no-cache" then always return 
27-FEB-97  MGD  delete on close for "temporary" files
01-FEB-97  MGD  HTTPd version 4
01-AUG-96  MGD  Variable to stream-LF file conversion "on-the-fly"
12-APR-96  MGD  determine read method (record or binary) from record format;
                implemented persistent connections ("keep-alive")
01-DEC-95  MGD  HTTPd version 3
27-SEP-95  MGD  added If-Modified-Since: functionality;
                changed carriage-control on records from <CR><LF> to single
                <LF> ('\n' ... newline), to better comply with some browsers
                (Netscape was spitting on X-bitmap files, for example!)
07-AUG-95  MGD  ConfigIncludeCommentedInfo to allow physical file
                specification to be included as commentary within an HTML file
13-JUL-95  MGD  bugfix; occasionally a record was re-read after flushing
                the records accumulated in the buffer NOT due to RMS$_RTB
20-DEC-94  MGD  initial development for multi-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* VMS related header files */
#include <atrdef.h>
#include <descrip.h>
#include <fibdef.h>
#include <iodef.h>
#include <libdef.h>
#include <rms.h>
#include <rmsdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application header files */
#include "wasd.h"
#include "auth.h"
#include "cache.h"
#include "httpd.h"
#include "error.h"
#include "file.h"
#include "mapurl.h"
#include "msg.h"
#include "net.h"
#include "support.h"
#include "stmlf.h"
#include "vm.h"

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern boolean  CacheEnabled;
extern int  OutputBufferSize;
extern char  ConfigContentTypeUnknown[];
extern char  ConfigDefaultFileContentType[];
extern char  HttpProtocol[];
extern char  SoftwareID[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/*****************************************************************************/
/*
Get record format, size and revision time of the file. As fixed documents
(files) can have revision date/times easily checked, any "If-Modified-Since:"
request field date/time supplied in the request is  processed and the file only
sent if modified later than specified. A "Last-Modified:" field is included in
any response header.

Get the file information using an ACP routine.  Any error status reported by
that routine, in particular "file not found", is returned to the calling
routine.  This is used to determine whether the file exists, as when
attempting to open one of multiple, possible home pages.  The calling routine
must also check for any error message generated by calling this function.

When successfully opened and connected generate an HTTP header, if required.
Once open and connected the transfer becomes I/O event-driven. 
*/ 
 
int FileBegin
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
void *FileOpenErrorFunction,
char *FileName,
char *ContentTypePtr,
boolean CacheAllowed,
boolean PreTagFileContents,
boolean EscapeHtml
)
{
   register char  *cptr, *sptr, *zptr;
   register struct FileTaskStruct  *tkptr;

   int  status,
        ContentLength,
        Length,
        MultiBlockCount;
   char  *KeepAlivePtr;
   void  *AstFunctionPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "FileBegin() %d |%s|%s|\n",
               rqptr->ErrorMessagePtr, FileName, ContentTypePtr);

   if (rqptr->ErrorMessagePtr != NULL)
   {
      /* previous error, cause threaded processing to unravel */
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /* set up the task structure (possibly multiple serially) */
   if (rqptr->FileTaskPtr == NULL)
   {
      rqptr->FileTaskPtr = tkptr = (struct FileTaskStruct*)
         VmGetHeap (rqptr, sizeof(struct FileTaskStruct));
   }
   else
   {
      tkptr = rqptr->FileTaskPtr;
      memset (tkptr, 0, sizeof(struct FileTaskStruct));
   }
   tkptr->NextTaskFunction = NextTaskFunction;

   cptr = FileName;
   zptr = (sptr = tkptr->FileName) + sizeof(tkptr->FileName);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      FileEnd (rqptr);
      return;
   }
   *sptr = '\0';
   tkptr->FileNameLength = sptr - tkptr->FileName;

   if (strsame (ContentTypePtr, ConfigContentTypeUnknown, -1))
   {
      /* if the content-type is not known drop over to the default */
      if (Config.ContentTypeDefaultPtr[0])
         tkptr->ContentTypePtr = Config.ContentTypeDefaultPtr;
      else
         tkptr->ContentTypePtr = ConfigDefaultFileContentType;
   }
   else
   {
      /* otherwise it's whatever it was initially resolved to */
      tkptr->ContentTypePtr = ContentTypePtr;
   }

   if (tkptr->PreTagFileContents = PreTagFileContents)
      tkptr->EscapeHtml = true;
   else
      tkptr->EscapeHtml = EscapeHtml;

   /*****************************************/
   /* check VMS-authenticated user's access */
   /*****************************************/

   if (rqptr->AuthVmsUserProfileLength)
   {
      status = AuthCheckVmsUserAccess (rqptr, tkptr->FileName,
                                       tkptr->FileNameLength);
      if (status == RMS$_PRV)
         tkptr->AuthVmsUserHasAccess = false;
      else
      if (VMSok (status))
         tkptr->AuthVmsUserHasAccess = true;
      else
      {
         /* error reported by access check */
         FileEnd (rqptr);
         return;
      }
   }
   else
      tkptr->AuthVmsUserHasAccess = false;

   /*****************/
   /* get file info */
   /*****************/

   tkptr->ParseInUse = true;

   tkptr->FileFab = cc$rms_fab;
   tkptr->FileFab.fab$l_fna = tkptr->FileName;  
   tkptr->FileFab.fab$b_fns = tkptr->FileNameLength;
   tkptr->FileFab.fab$l_nam = &tkptr->FileNam;

   /* initialize the NAM block */
   tkptr->FileNam = cc$rms_nam;
   tkptr->FileNam.nam$l_esa = tkptr->ExpandedFileName;
   tkptr->FileNam.nam$b_ess = sizeof(tkptr->ExpandedFileName)-1;

   if (VMSok (status = sys$parse (&tkptr->FileFab, 0, 0)))
   {
      if (strsame (tkptr->FileNam.nam$l_type,
                   HTA_FILE_TYPE, sizeof(HTA_FILE_TYPE)-1) &&
          *(tkptr->FileNam.nam$l_type+sizeof(HTA_FILE_TYPE)-1) == ';')
      {
         /* attempt to retrieve an HTA authorization file, scotch that! */
         status = SS$_NOPRIV;
      }
   }

   if (VMSnok (status))
   {
      if (FileOpenErrorFunction != NULL)
      {
         /* do not report the error, the alternate function will handle it */
         tkptr->NextTaskFunction = FileOpenErrorFunction;
         FileEnd (rqptr);
         return;
      }
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      FileEnd (rqptr);
      return;
   }

   /* this acp access to file size, date, etc., is significantly faster */
   if (VMSnok (status = FileAcpInfo (rqptr)))
   {
      if (status == RMS$_FNF && FileOpenErrorFunction != NULL)
      {
         /* do not report an error, the alternate function will handle it */
         tkptr->NextTaskFunction = FileOpenErrorFunction;
         FileEnd (rqptr);
         return;
      }
      else
      {
         if (!rqptr->AccountingDone)
            rqptr->AccountingDone = ++Accounting.DoFileCount;
         rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
         rqptr->ErrorHiddenTextPtr = tkptr->FileName;
         ErrorVmsStatus (rqptr, status, FI_LI);
         FileEnd (rqptr);
         return;
      }
   }

   /***************/
   /* file exists */
   /***************/

   if (rqptr->ResponseHeaderPtr == NULL)
   {
      /****************************/
      /* response header required */
      /****************************/

      /*
         For variable-length, etc. files (accessed using record-mode) file size
         cannot be accurately determined.  Indicate this using length of -1.
      */
      if (tkptr->RecordIO)
         ContentLength = -1;
      else
         ContentLength = tkptr->SizeInBytes;
      if (Debug) fprintf (stdout, "ContentLength: %d\n", ContentLength);

      if (rqptr->HttpMethod == HTTP_METHOD_GET &&
          (rqptr->IfModifiedSinceBinaryTime[0] ||
           rqptr->IfModifiedSinceBinaryTime[1]) &&
           !rqptr->PragmaNoCache &&
           !rqptr->DeleteOnClose)

      {
         /*********************/
         /* if modified since */
         /*********************/

         if (VMSnok (status =
             HttpIfModifiedSince (rqptr, &tkptr->ModifiedBinaryTime,
                                  ContentLength)))
         {
            /* task status LIB$_NEGTIM if not modified/sent */
            if (!rqptr->AccountingDone)
            {
               rqptr->AccountingDone = ++Accounting.DoFileCount;
               Accounting.DoFileNotModifiedCount++;
            }
            FileEnd (rqptr);
            return;
         }
      }

      if (!rqptr->AccountingDone)
         rqptr->AccountingDone = ++Accounting.DoFileCount;

      if (Config.KeepAliveTimeoutSeconds &&
          rqptr->KeepAliveRequest &&
          ContentLength > 0)
      {
         rqptr->KeepAliveResponse = true;
         rqptr->KeepAliveCount++;
         KeepAlivePtr = KeepAliveHttpHeader;
      }
      else
         KeepAlivePtr = "";

      if (rqptr->QueryStringPtr[0] &&
          tolower(rqptr->QueryStringPtr[0]) == 'h' &&
          strsame (rqptr->QueryStringPtr, "httpd=content", 13))
      {
         /* request-specified content-type (default to plain-text) */
         if (strsame (rqptr->QueryStringPtr+13, "&type=", 6))
            ContentTypePtr = rqptr->QueryStringPtr + 19;
         else
            ContentTypePtr = "text/plain";
      }
      else
         ContentTypePtr = tkptr->ContentTypePtr;

      rqptr->ResponseStatusCode = 200;
      if ((rqptr->ResponseHeaderPtr =
           HttpHeader (rqptr, 200, ContentTypePtr, ContentLength,
                       &tkptr->ModifiedBinaryTime, KeepAlivePtr))
          == NULL)
      {
         FileEnd (rqptr);
         return;
      }

      /* quit here if the HTTP method was HEAD */
      if (rqptr->HttpMethod == HTTP_METHOD_HEAD)
      {
         FileEnd (rqptr);
         return;
      }
   }

   /*****************/
   /* NOW open file */
   /*****************/

   tkptr->ParseInUse = false;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   /* "temporary" file, automatic delete on closing it */
   if (rqptr->DeleteOnClose) tkptr->FileFab.fab$l_fop |= FAB$M_DLT;

   if (tkptr->RecordIO)
   {
      tkptr->FileFab.fab$b_fac = FAB$M_GET;
      tkptr->FileFab.fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT;

      status = sys$open (&tkptr->FileFab, 0, 0);
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
   }
   else
   {
      tkptr->FileFab.fab$b_fac = FAB$M_GET | FAB$M_BIO; 
      tkptr->FileFab.fab$b_shr = FAB$M_SHRGET;

      status = sys$open (&tkptr->FileFab, 0, 0);
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);

      if (status == RMS$_FLK)
      {
         /* RMS$_FLK, block-mode cannot SHRPUT, try record-mode! */
         tkptr->RecordIO = true;
         tkptr->FileFab.fab$b_fac = FAB$M_GET;
         tkptr->FileFab.fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT;

         status = sys$open (&tkptr->FileFab, 0, 0);
         if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      }
   }

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      /* ensure the response header created above is not used for the error */
      rqptr->ResponseHeaderPtr = NULL;
      rqptr->ResponseHeaderLength = 0;
      rqptr->ResponseStatusCode = 0;

      /* if its a search list treat directory not found as if file not found */
      if ((tkptr->FileNam.nam$l_fnb & NAM$M_SEARCH_LIST) && status == RMS$_DNF)
         status = RMS$_FNF;

      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      FileEnd (rqptr);
      return;
   }

   tkptr->FileRab = cc$rms_rab;
   /* set the RAB context to contain the request thread pointer */
   tkptr->FileRab.rab$l_ctx = rqptr;
   tkptr->FileRab.rab$l_fab = &tkptr->FileFab;
   if (tkptr->RecordIO)
   {
      /* divide by 512 and one for trailing bytes */
      MultiBlockCount = (tkptr->SizeInBytes >> 9) + 1;
      if (MultiBlockCount > 127)
      {
         tkptr->FileRab.rab$b_mbc = 127;
         tkptr->FileRab.rab$b_mbf = 2;
      }
      else
      {
         /* less than 127 * 512 bytes in length, one buffer enough! */
         tkptr->FileRab.rab$b_mbc = MultiBlockCount;
         tkptr->FileRab.rab$b_mbf = 1;
      }
      tkptr->FileRab.rab$l_rop = RAB$M_RAH | RAB$M_ASY;
      if (Debug)
         fprintf (stdout, "rab$b_mbc: %d rab$b_mbf: %d\n",
                  tkptr->FileRab.rab$b_mbc, tkptr->FileRab.rab$b_mbf);
   }
   else
   {
      tkptr->FileRab.rab$l_bkt = 0;
      tkptr->FileRab.rab$l_rop = RAB$M_RAH | RAB$M_BIO | RAB$M_ASY;
   }

   if (VMSnok (status = sys$connect (&tkptr->FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      FileEnd (rqptr);
      return;
   }

   /******************************/
   /* stream-LF file conversion? */
   /******************************/

   if (Config.StreamLfConversionMaxKbytes &&
       tkptr->RecordIO &&
       tkptr->FileFab.fab$b_rfm == FAB$C_VAR &&
       strsame (tkptr->ContentTypePtr, "text/", 5))
   {
      /* divide by two to get the number of kilobytes (1024) in the file */
      if ((tkptr->SizeInBytes >> 10) <= Config.StreamLfConversionMaxKbytes)
         StmLfBegin (MapVmsPath(tkptr->FileName),
                     tkptr->FileFab.fab$l_fna,
                     tkptr->FileFab.fab$b_fns,
                     tkptr->AuthVmsUserHasAccess);
   }

   /********************/
   /* begin processing */
   /********************/

   /* network writes are checked for success, fudge the first one! */
   rqptr->NetWriteIOsb.Status = SS$_NORMAL;

   /*
      Cache load is not initiated if this is not a stand-alone file request
      (i.e. not part of some other activity, e.g. a directory read-me file,
      'CacheHashValue' zero), if we are not interested in caching the file
      ('CacheHashValue' zero), if it is already being loaded via another
      request, if a "temporary" file ('DeleteOnClose'), or via requests with
      a VMS authentication profile attached.
   */
   if (CacheEnabled &&
       CacheAllowed &&
       rqptr->CacheHashValue &&
       !tkptr->EscapeHtml &&
       !rqptr->CacheLoading &&
       !rqptr->DeleteOnClose &&
       !rqptr->AuthVmsUserProfileLength)
   {
      /* get a cache structure allocated */
      CacheLoadBegin (rqptr, tkptr->SizeInBytes);
      /* 'rqptr->CachePtr' will now be used to control cache load behaviour */
   }
   else
      rqptr->CachePtr = NULL;

   if (tkptr->PreTagFileContents)
   {
      tkptr->PreTagEndFileContents = true;
      NetWriteBuffered (rqptr, &FileEmptyBuffer, "<PRE>", 5);
      return;
   }

   FileEmptyBuffer (rqptr);
}

/*****************************************************************************/
/*
*/

FileEnd (struct RequestStruct *rqptr)

{
   register struct FileTaskStruct  *tkptr;

   int  status,
        SetPrvStatus;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FileEnd()\n");

   tkptr = rqptr->FileTaskPtr;

   if (rqptr->CachePtr != NULL) CacheLoadEnd (rqptr, tkptr);

   if (tkptr->ParseInUse)
   {
      /* ensure parse internal data structures are released */
      tkptr->FileFab.fab$l_fna = "a:[b]c.d;";
      tkptr->FileFab.fab$b_fns = 9;
      tkptr->FileFab.fab$b_dns = 0;
      tkptr->FileNam.nam$b_nop = NAM$M_SYNCHK;
      sys$parse (&tkptr->FileFab, 0, 0);
   }
   else
   if (tkptr->FileFab.fab$w_ifi)
   {
      if (rqptr->DeleteOnClose)
      {
         /* use SYSPRV to ensure deletion-on-close of file */
         EnableSysPrv ();
         sys$close (&tkptr->FileFab, 0, 0);
         DisableSysPrv ();
      }
      else
         sys$close (&tkptr->FileFab, 0, 0);
   }

   if (rqptr->HttpMethod != HTTP_METHOD_HEAD && tkptr->PreTagEndFileContents)
   {
      /* encapsulating a file, add the end tag, declaring next task as AST */
      NetWriteBuffered (rqptr, tkptr->NextTaskFunction, "</PRE>\n", 7);
   }
   else
   {
      /* declare the next task */
      SysDclAst (tkptr->NextTaskFunction, rqptr);
   }
} 

/*****************************************************************************/
/*
Ensure any output buffer contents are flushed.
*/

FileEmptyBuffer (struct RequestStruct *rqptr)

{
   register struct FileTaskStruct  *tkptr;

   int  status;
   void  *AstFunctionPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FileEmptyBuffer()\n");

   tkptr = rqptr->FileTaskPtr;

   if (tkptr->RecordIO)
      AstFunctionPtr = &FileNextRecord;
   else
      AstFunctionPtr = &FileNextBlocks;
   if (rqptr->OutputBufferCount)
   {
      /* need exclusive use, flush the current contents */
      NetWriteBuffered (rqptr, AstFunctionPtr, NULL, 0);
   }
   else
      SysDclAst (AstFunctionPtr, rqptr);
}

/*****************************************************************************/
/*
Queue a read of the next record from the file.  When the read completes call 
FileNextRecordAST() function to post-process the read and/or send the data 
to the client.  Don't bother to test any status here, the AST routine will do 
that!
*/ 

FileNextRecord (struct RequestStruct *rqptr)

{
   register struct FileTaskStruct  *tkptr;

   /*********/
   /* begin */
   /*********/

   /** if (Debug) fprintf (stdout, "FileNextRecord()\n"); **/

   if (VMSnok (rqptr->NetWriteIOsb.Status))
   {
      /* network write has failed (as AST), bail out now */
      if (Debug)
         fprintf (stdout, "NetWriteIOsb.Status %%X%08.08X\n",
                  rqptr->NetWriteIOsb.Status);
      FileEnd (rqptr);
      return;
   }

   tkptr = rqptr->FileTaskPtr;

   if (!tkptr->FileRab.rab$l_ubf)
   {
      /* first IO in file transfer, organise the buffer */
      if (rqptr->CachePtr == NULL)
      {
         /* not loading cache, initialize output buffer */
         NetWriteBuffered (rqptr, 0, NULL, 0);
      }
      else
      {
         /* using a cache buffer */
         rqptr->CacheContentPtr = rqptr->CacheContentCurrentPtr =
            rqptr->CachePtr->ContentPtr;
         rqptr->CacheContentCount = rqptr->CachePtr->CacheSize;
      }
   }

   if (rqptr->CachePtr == NULL)
   {
      /* not loading cache */
      tkptr->FileRab.rab$l_ubf = rqptr->OutputBufferCurrentPtr;
      tkptr->FileRab.rab$w_usz = rqptr->OutputBufferRemaining;
   }
   else
   {
      /* loading cache */
      tkptr->FileRab.rab$l_ubf = rqptr->CacheContentCurrentPtr;
      tkptr->FileRab.rab$w_usz = rqptr->CacheContentCount;
   }

   /* GET the next record (or last read if RFA is set) */
   sys$get (&tkptr->FileRab, &FileNextRecordAST, &FileNextRecordAST);
}

/*****************************************************************************/
/*
The read of the next record from the file has completed.  Post-process and/or 
queue a network write to the client.  When the network write completes it will 
call the function FileNextRecord() to queue a read of the next record. Record-
by-record processing builds up as many file records as possible into available 
buffer space before transfering them to the network as one I/O.  The 
sys$get()s use the buffer area pointed to by
'rqptr->OutputBufferCurrentPtr' and the size specified by
'rqptr->OutputBufferRemaining'.  When a sys$get() fails with a "record 
too big for user's buffer", because we're running out of buffer space, the 
accumulated records are flushed to the network and the pointer and available 
space values are reset.  The FileNextRecord() AST called when the 
NetWrite() completes will re-read a record that failed with the "record too 
big" error by forcing the read to be by RFA for that one record.
*/ 

FileNextRecordAST (struct RAB *RabPtr)

{
   register int  rsz;
   register struct RequestStruct  *rqptr;
   register struct FileTaskStruct  *tkptr;

   int  status,
        CacheBufferRemaining;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
         "FileNextRecordAST() sts: %%X%08.08X stv: %%X%08.08X rsz: %d\n",
         RabPtr->rab$l_sts, RabPtr->rab$l_stv, RabPtr->rab$w_rsz);

   rqptr = RabPtr->rab$l_ctx;
   tkptr = rqptr->FileTaskPtr;

   if (VMSnok (tkptr->FileRab.rab$l_sts))
   {
      /*
         Test for "record too big for user's buffer", indicates that
         its time to flush the records accumulated in the buffer.
         This should never happen when reading into a cache buffer!!!
         (only possibly if the file was to be extended, etc.)
      */
      if (tkptr->FileRab.rab$l_sts == RMS$_RTB)
      {
         if (Debug) fprintf (stdout, "RMS$_RTB\n");

         if (rqptr->CachePtr == NULL)
         {
            if (rqptr->OutputBufferCount == 0)
            {
               /* legitimate "record too big for user's buffer" error */
               rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
               rqptr->ErrorHiddenTextPtr = tkptr->FileName;
               ErrorVmsStatus (rqptr, RMS$_RTB, FI_LI);
               FileEnd (rqptr);
               return;
            }

            /* insufficient space for next record, flush buffer contents */
            FileFlushBuffer (rqptr, &FileNextRecord);

            /* re-read the record that just failed by forcing a read by RFA */
            tkptr->FileRab.rab$b_rac |= RAB$C_RFA;

            return;
         }
         else
         {
            /*
               This is not supposed to happen!
               Possibly in between getting the file size and opening and
               reading this far the file has been extended or rewritten.
               Shouldn't happen often enough to be a worry!
            */
            rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
            rqptr->ErrorHiddenTextPtr = tkptr->FileName;
            ErrorVmsStatus (rqptr, RMS$_RTB, FI_LI);
            FileEnd (rqptr);
            return;
         }
      }

      if (tkptr->FileRab.rab$l_sts == RMS$_EOF)
      {
         if (Debug) fprintf (stdout, "RMS$_EOF\n");
         /* cache load becomes valid only after we get to end-of-file */
         if (rqptr->CachePtr != NULL) rqptr->CachePtr->CacheLoadValid = true;
         FileFlushBuffer (rqptr, &FileEnd);
         return;
      }

      /* hmmm, file transfer error */
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, tkptr->FileRab.rab$l_sts, FI_LI);
      FileEnd (rqptr);
      return;
   }

   /* force the next read to be sequential (in case the last was by RFA) */
   tkptr->FileRab.rab$b_rac &= ~RAB$C_RFA;

   rsz = tkptr->FileRab.rab$w_rsz;
   if (rsz)
   {
      /* add newline if last character in line is not a newline! */
      if (tkptr->FileRab.rab$l_ubf[rsz-1] != '\n')
         tkptr->FileRab.rab$l_ubf[rsz++] = '\n';
   }
   else
   {
      /* must be a blank line (empty record), add a newline */
      tkptr->FileRab.rab$l_ubf[rsz++] = '\n';
   }

   if (Debug)
      fprintf (stdout, "rsz: %d remain: %d count: %d\n",
               rsz, rqptr->OutputBufferRemaining, rqptr->CacheContentCount);

   if (rqptr->CachePtr != NULL)
   {
      /* has not filled or overflowed this portion of the cache buffer */
      rqptr->CacheContentCount -= rsz;
      rqptr->CacheContentCurrentPtr += rsz;

      /* flush if we have arrived at a chunk of output */
      if (rqptr->CacheContentCurrentPtr >=
          rqptr->CacheContentPtr + OutputBufferSize)
         FileFlushBuffer (rqptr, &FileNextRecord);
      else
         FileNextRecord (rqptr);
   }
   else
   if (rqptr->OutputBufferRemaining > rsz)
   {
      /* has not filled or overflowed the standard output buffer */
      rqptr->OutputBufferRemaining -= rsz;
      rqptr->OutputBufferCurrentPtr += rsz;
      rqptr->OutputBufferCount += rsz;
      FileNextRecord (rqptr);
   }
   else
   {
      /* insufficient space for next record, flush buffer contents */
      FileFlushBuffer (rqptr, &FileNextRecord);

      /* re-read the record that just failed by forcing a read by RFA */
      tkptr->FileRab.rab$b_rac |= RAB$C_RFA;
   }
}

/*****************************************************************************/
/*
Flush the file records accumulated in the 'Buffer'.  If plain text and 
encapsulated then escape any HTML-forbidden characters using a separate 
function.
*/

int FileFlushBuffer
(
struct RequestStruct *rqptr,
void *AstFunctionPtr
)
{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FileFlushBuffer()\n");

   if (rqptr->FileTaskPtr->EscapeHtml)
      return (FileWriteBufferEscapeHtml (rqptr, AstFunctionPtr));
   else
   {
      if (rqptr->CachePtr == NULL)
      {
         if (!rqptr->OutputBufferCount)
         {
            /* nothing to flush! */
            SysDclAst (AstFunctionPtr, rqptr);
            return;
         }

         NetWrite (rqptr, AstFunctionPtr,
                   rqptr->OutputBufferPtr,
                   rqptr->OutputBufferCount);

         /* reset standard output buffer */
         NetWriteBuffered (rqptr, NULL, NULL, 0);
         return;
      }
      else
      {
         if (!(rqptr->CacheContentCurrentPtr - rqptr->CacheContentPtr))
         {
            /* nothing to flush! */
            SysDclAst (AstFunctionPtr, rqptr);
            return;
         }

         NetWrite (rqptr, AstFunctionPtr,
                   rqptr->CacheContentPtr,
                   rqptr->CacheContentCurrentPtr - rqptr->CacheContentPtr);

         /* reset the pointer to the new flush position */
         rqptr->CacheContentPtr = rqptr->CacheContentCurrentPtr;
         return;
      }
   }
}

/*****************************************************************************/
/*
Queue a read of the next series of Virtual Blocks from the file.  When the 
read completes call FileNextBlocksAST() function to post-process the read 
and/or send the data to the client.  Don't bother to test any status here, the 
AST routine will do that!
*/ 

FileNextBlocks (struct RequestStruct *rqptr)

{
   register struct FileTaskStruct  *tkptr;

   unsigned int  BufferSize;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FileNextBlocks()\n");

   if (VMSnok (rqptr->NetWriteIOsb.Status))
   {
      /* network write has failed (as AST), bail out now */
      if (Debug)
         fprintf (stdout, "NetWriteIOsb.Status %%X%08.08X\n",
                  rqptr->NetWriteIOsb.Status);
      FileEnd (rqptr);
      return;
   }

   tkptr = rqptr->FileTaskPtr;

   if (tkptr->FileRab.rab$l_bkt)
   {
      /* subsequent reads */
      tkptr->FileRab.rab$l_bkt += tkptr->FileRab.rab$w_usz >> 9;

      if (rqptr->CachePtr != NULL)
      {
         /* adjust according to size of previous read */
         rqptr->CacheContentPtr += tkptr->FileRab.rab$w_rsz;
         rqptr->CacheContentCount -= tkptr->FileRab.rab$w_rsz;
         if (Debug) fprintf (stdout, "count: %d\n", rqptr->CacheContentCount);
         if (rqptr->CacheContentCount <= 0)
         {
            /*
               This is not supposed to happen!
               Possibly in between getting the file size and opening and
               reading this far the file has been extended or rewritten.
               Shouldn't happen often enough to be a worry!
            */
            rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
            rqptr->ErrorHiddenTextPtr = tkptr->FileName;
            ErrorVmsStatus (rqptr, RMS$_RTB, FI_LI);
            FileEnd (rqptr);
            return;
         }
      }
   }
   else
   {
      /* first read */
      tkptr->FileRab.rab$l_bkt = 1;
      if (rqptr->CachePtr == NULL)
      {
         /* not using a cache buffer, initialize standard output buffer */
         NetWriteBuffered (rqptr, 0, NULL, 0);
      }
      else
      {
         /* using a cache buffer */
         rqptr->CacheContentPtr = rqptr->CachePtr->ContentPtr;
         rqptr->CacheContentCount = rqptr->CachePtr->CacheSize;
      }
   }

   /* make buffer size an even number of 512 byte blocks */
   BufferSize = OutputBufferSize & 0xfe00;

   if (rqptr->CachePtr == NULL)
   {
      /* not using a cache buffer, standard output buffer */
      tkptr->FileRab.rab$l_ubf = rqptr->OutputBufferPtr;
      tkptr->FileRab.rab$w_usz = BufferSize;
   }
   else
   {
      /* using a cache buffer, load it progressively */
      if (Debug) fprintf (stdout, "count: %d\n", rqptr->CacheContentCount);
      tkptr->FileRab.rab$l_ubf = rqptr->CacheContentPtr;
      if (rqptr->CacheContentCount > BufferSize)
         tkptr->FileRab.rab$w_usz = BufferSize;
      else
         tkptr->FileRab.rab$w_usz = rqptr->CacheContentCount;
   }

   sys$read (&tkptr->FileRab, &FileNextBlocksAST, &FileNextBlocksAST);
}

/*****************************************************************************/
/*
The read of the next series of Virtual Blocks from the file has completed.  
Post-process and/or queue a network write to the client.  When the network 
write completes it will call the function FileNextBlocks() to queue a read 
of the next series of blocks.
*/ 

FileNextBlocksAST (struct RAB *RabPtr)

{
   register struct RequestStruct  *rqptr;
   register struct FileTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "FileNextBlocksAST() sts: %%X%08.08X stv: %%X%08.08X\n",
               RabPtr->rab$l_sts,
               RabPtr->rab$l_stv);

   rqptr = RabPtr->rab$l_ctx;
   tkptr = rqptr->FileTaskPtr;

   if (VMSnok (tkptr->FileRab.rab$l_sts))
   {
      if (tkptr->FileRab.rab$l_sts == RMS$_EOF)
      {
         if (rqptr->CachePtr == NULL)
         {
            /* reset the standard output buffer */
            NetWriteBuffered (rqptr, NULL, NULL, 0);
         }
         else
         {
            /* a cache load becomes valid only after we get to end-of-file */
            rqptr->CachePtr->CacheLoadValid = true;
         }
         FileEnd (rqptr);
         return;
      }
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, tkptr->FileRab.rab$l_sts, FI_LI);
      FileEnd (rqptr);
      return;
   }

   /* queue a network write to the client, AST to FileNextBlocks() */
   if (rqptr->FileTaskPtr->EscapeHtml)
   {
      rqptr->OutputBufferCount = tkptr->FileRab.rab$w_rsz;
      FileWriteBufferEscapeHtml (rqptr, &FileNextBlocks);
   }
   else
      NetWrite (rqptr, &FileNextBlocks,
                tkptr->FileRab.rab$l_ubf, tkptr->FileRab.rab$w_rsz);
}

/*****************************************************************************/
/*
Send the buffer contents escaping any HTML-forbidden characters.
*/ 

int FileWriteBufferEscapeHtml
(
struct RequestStruct *rqptr,
void *AstFunctionPtr
)
{
   register int  bcnt, ecnt;
   unsigned register char  *bptr, *sptr, *zptr;
   register struct FileTaskStruct  *tkptr;

   int  status;
                         
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FileWriteBufferEscapeHtml()\n");

   tkptr = rqptr->FileTaskPtr;

   /* allocate a worst-case escaped HTML buffer (i.e. all "&amp;"s) */
   if (tkptr->EscapeHtmlPtr == NULL)
      tkptr->EscapeHtmlPtr = VmGetHeap (rqptr, OutputBufferSize * 5);

   sptr = tkptr->EscapeHtmlPtr;
   bptr = rqptr->OutputBufferPtr;
   bcnt = rqptr->OutputBufferCount;
   while (bcnt)
   {
      switch (*bptr)
      {
         case '<' :
            memcpy (sptr, "&lt;", 4); sptr += 4; bptr++; break;
         case '>' :
            memcpy (sptr, "&gt;", 4); sptr += 4; bptr++; break;
         case '&' :
            memcpy (sptr, "&amp;", 5); sptr += 5; bptr++; break;
         default :
            *sptr++ = *bptr++;
      }
      bcnt--;
   }

   /* flush the HTML-escaped buffer contents */
   NetWrite (rqptr, AstFunctionPtr, 
             tkptr->EscapeHtmlPtr, sptr - tkptr->EscapeHtmlPtr);

   /* reuse standard output buffer */
   NetWriteBuffered (rqptr, NULL, NULL, 0);
}

/*****************************************************************************/
/*
Get the file's record format byte, file size and revision date.

Check the specified file's record format byte.  If VARIABLE or VFC the file is
read by record.  If UNDEFINED, STMLF, STMCR, STREAM, or FIXED it can be read
as a binary file (the stream formats having embedded carriage-control suitable
for HTTP).  Set the '->RecordIO' boolean accordingly.  Relies on the
calling routine to have sys$parse()d file specification into the NAM block.

This function uses the ACP-QIO interface detailed in the "OpenVMS I/O User's 
Reference Manual", and is probably as fast as we can get for this type of file
system functionality!
*/ 

#define FAT$C_UNDEFINED 0
#define FAT$C_FIXED 1
#define FAT$C_VARIABLE 2
#define FAT$C_VFC 3
#define FAT$C_STREAM 4
#define FAT$C_STMLF 5
#define FAT$C_STMCR 6

int FileAcpInfo (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (DeviceDsc, "");

   static unsigned long  AtrRdt [2];

   static struct {
      unsigned char  RecordType;
      unsigned char  OfNoInterest1;
      unsigned short  RecordSize;
      unsigned char  OfNoInterest2 [4];
      unsigned short  EndOfFileVbnHi;
      unsigned short  EndOfFileVbnLo;
      unsigned short  FirstFreeByte;
      unsigned char   OfNoInterest3 [18];
   } AtrRecord;

   static struct fibdef  SearchFib;

   static struct atrdef  FileAtr [] =
   {
      { sizeof(AtrRecord), ATR$C_RECATTR, &AtrRecord },
      { sizeof(AtrRdt), ATR$C_REVDATE, &AtrRdt },
      { 0, 0, 0 }
   };

   register struct FileTaskStruct  *tkptr;

   int  status;
   unsigned short  AcpChannel;
   unsigned long  EndOfFileVbn;

   struct fibdef  FileFib;

   struct {
      unsigned short  Length;
      unsigned short  Unused;
      unsigned long  Address;
   } FileFibAcpDsc,
     FileNameAcpDsc;
     
   struct {
      unsigned short  Status;
      unsigned short  Unused1;
      unsigned long  Unused2;
   } AcpIOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      tkptr = rqptr->FileTaskPtr;
      tkptr->FileNam.nam$l_ver[tkptr->FileNam.nam$b_ver] = '\0';
      fprintf (stdout, "FileAcpInfo() |%s|\n", tkptr->FileNam.nam$l_esa);
   }

   tkptr = rqptr->FileTaskPtr;

   /* initialize return values */
   tkptr->RecordIO = false;
   tkptr->SizeInBytes = 0;
   memset (tkptr->ModifiedBinaryTime, 0, 8);

   if (!tkptr->FileNam.nam$b_name &&
       tkptr->FileNam.nam$b_type == 1 &&
       tkptr->FileNam.nam$b_ver == 1) return (SS$_NORMAL);

   /* assign a channel to the disk device containing the file */
   DeviceDsc.dsc$a_pointer = tkptr->FileNam.nam$l_dev;
   DeviceDsc.dsc$w_length = tkptr->FileNam.nam$b_dev;
   if (Debug)
      fprintf (stdout, "device |%*.*s|\n",
               tkptr->FileNam.nam$b_dev, tkptr->FileNam.nam$b_dev,
               tkptr->FileNam.nam$l_dev);

   status = sys$assign (&DeviceDsc, &AcpChannel, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$assign() %%X%08.08X\n", status);
   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }

   /* set up the File Information Block for the ACP interface */
   memset (&FileFib, 0, sizeof(struct fibdef));
   FileFibAcpDsc.Length = sizeof(FileFib);
   FileFibAcpDsc.Address = &FileFib;

   /* quick work around, different syntax for this structure with DEC C? */
#ifdef __DECC
   memcpy (&FileFib.fib$w_did, &tkptr->FileNam.nam$w_did, 6);
#else
   memcpy (&FileFib.fib$r_did_overlay.fib$w_did, &tkptr->FileNam.nam$w_did, 6);
#endif

   FileNameAcpDsc.Address = tkptr->FileNam.nam$l_name;
   FileNameAcpDsc.Length = tkptr->FileNam.nam$b_name +
                           tkptr->FileNam.nam$b_type +
                           tkptr->FileNam.nam$b_ver;
   if (Debug)
      fprintf (stdout, "name |%*.*s|\n",
               FileNameAcpDsc.Length, FileNameAcpDsc.Length,
               FileNameAcpDsc.Address);

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   /* read the file attributes */
   status = sys$qiow (0, AcpChannel, IO$_ACCESS, &AcpIOsb, 0, 0, 
                      &FileFibAcpDsc, &FileNameAcpDsc, 0, 0,
                      &FileAtr, 0);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (Debug)
      fprintf (stdout,
"sys$qio() %%X%08.08X IOsb: %%X%08.08X RecordSize: %d RecordType: %02.02X\n",
         status, AcpIOsb.Status, AtrRecord.RecordSize, AtrRecord.RecordType);

   sys$dassgn (AcpChannel);

   if (VMSok (status)) status = AcpIOsb.Status;
   if (status == SS$_NOSUCHFILE) return (RMS$_FNF);
   if (VMSnok (status)) return (status);

   if (AtrRecord.RecordType == FAT$C_VARIABLE ||
       AtrRecord.RecordType == FAT$C_VFC ||
       (AtrRecord.RecordType == FAT$C_FIXED && (AtrRecord.RecordSize & 1)))
   {
      if (Debug) fprintf (stdout, "record!\n");
      tkptr->RecordIO = true;
   }

   EndOfFileVbn = AtrRecord.EndOfFileVbnLo + (AtrRecord.EndOfFileVbnHi << 16);
   if (Debug)
      fprintf (stdout, "EndOfFileVbn: %d FirstFreeByte %d\n",
               EndOfFileVbn, AtrRecord.FirstFreeByte);

   /* buffer these for access by the cache module */
   tkptr->EndOfFileVbn = EndOfFileVbn;
   tkptr->FirstFreeByte = AtrRecord.FirstFreeByte;

   if (EndOfFileVbn <= 1)
      tkptr->SizeInBytes = AtrRecord.FirstFreeByte;
   else
      tkptr->SizeInBytes = ((EndOfFileVbn-1) << 9) + AtrRecord.FirstFreeByte;
   if (Debug) fprintf (stdout, "bytes %d\n", tkptr->SizeInBytes); 

   memcpy (tkptr->ModifiedBinaryTime, &AtrRdt, 8);

   if (Debug)
   {
      char  GmtRdt [48],
            VmsRdt [32];
      $DESCRIPTOR (VmsRdtFaoDsc, "!%D\0");
      $DESCRIPTOR (VmsRdtDsc, VmsRdt);

      sys$fao (&VmsRdtFaoDsc, 0, &VmsRdtDsc, tkptr->ModifiedBinaryTime);
      HttpGmTimeString (GmtRdt, tkptr->ModifiedBinaryTime);
      fprintf (stdout, "|%s|%s|\n", VmsRdt, GmtRdt); 
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/

