/*****************************************************************************/
/*
                                 Graph.c


This module provides some core plotting and GIF image functions, and although
it's a slight anomaly, activity statistic recording, for which the graphing
is used to produce a GIF image of server activity used in a simple "snapshot"
report.

The activity report is potentially JavaScript enhanced. Works well with
Netscape Navigator 3.0ff, slightly less well with Microsoft Internet Explorer
3.02ff. MSIE 3.02 doesn't seem to honour the "onFocus" and "onBlur" window
events.


ACTIVITY STATISTICS NOTES
-------------------------

Server activity is gathered by accumlating all requests and bytes transmitted
on a per-minute basis. These statistics are kept in a two dynamically allocated
arrays of longwords. The arrays are sized on a per-day basis, 1440 longwords
per-day, minimum 1 day, maximum 28 days (although that's an arbitrary number I
can't imagine anyone usefully using this facility over a longer period) a total
of 2880 longwords, or 11520 bytes per day, 80640 per week. The index into the
array is based on the day of activity (zero to whatever day it is), plus the
hour and minute. Days are the VMS absolute day (day number from start of
epoch). The index into the array is calculated using a modulas of the day of
activity by the number of days in the array (hence ranges from zero to the
number of days in the array) multiplied by the number of minutes in the day,
plus the hour multiplied by sixty, plus the minute.


ACTIVITY REPORT/PLOT NOTES
--------------------------

Activity statistics are reported/plotted from an absolute day and hour BACK
for a specified number of hours. That is, the date and hour specified is the
date and hour the report/plot ends, the start is calculated as a relative
offset backwards from that. Slightly curious, but the idea was for a quick
snapshot of activity up until the current time, so default behaviour (without
any parameters) is for the current hour, beginning at minute 00 and ending
at minute 59. If a duration is specified it is the number of hours prior to
current time, so a duration of 4 at time 11:15 ranges from 08:00 to 11:59!
If a time is specified then it is the period leading up to that, etc.


GIF/PLOTTING NOTES
------------------

The plot functions have been designed to be a general as possible, so although
only used for activity reports at this stage another HTTPd use may be found
for them sometime in the future!

NOTE: The GIF functions in this module are not designed to be reentrant and
cannot be used for multi-threaded processing. All processing for a single
request must begin and end before returning from AST delivery.

The GIF code in these functions is implemented in accordance with Compuserve's
Graphic Interchange Format Programming Reference specification, version 89a,
31st July 1990. 

The LZW compression employed by the GIF algorithm is implemented using code
derived from the PBM suite. This code is copyright by the original author and
used within the specified licensing conditions, as noted immediately above the
applicable functions.

The graphing functions functions use a lazy and faster approach to plotting.
Although only providing 16 colours (4 bits) it represents each pixel in 8
bits, thus trading off memory against ease of programming and to a certain
extent plotting speed.


VERSION HISTORY
---------------
07-JAN-98  MGD  same bugfix (obviously wasn't); in GraphActivityClearDay()
                (and a plethora of other annoyances/problems ...
                bit of a holiday does you the world of good :^)
01-DEC-97  MGD  bugfix; in GraphActivityClearDay()
18-OCT-97  MGD  remove dependence on Unix time functions after irritating
                experience related to VMS 7.1,
                add JavaScript-driven descriptions to client-side map links
01-AUG-97  MGD  new for v4.3 (hope my plotting functions are not too
                brain-dead, I'm only a graphics novice, sigh!)
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>

/* VMS header files */
#include <ssdef.h>
#include <stsdef.h>

/* application header files */
#include "wasd.h"
#include "graph.h"
#include "httpd.h"
#include "support.h"
#include "vm.h"

unsigned char  GraphicRgbRed [] =
{ 000, 255, 000, 000, 255, 255, 000, 255, 
  192, 128, 000, 000, 128, 128, 128, 250 };
unsigned char  GraphicRgbGreen [] =
{ 000, 000, 255, 000, 255, 000, 255, 255,
  192, 000, 128, 000, 128, 000, 128, 240 };
unsigned char  GraphicRgbBlue [] =
{ 000, 000, 000, 255, 000, 255, 255, 255,
  192, 000, 000, 128, 000, 128, 128, 230 };

#define JAVASCRIPT_ENABLED 1
#define JAVASCRIPT_ONMOUSEOVER_ENABLED 1

#define ACTIVITY_GRAPH_WIDTH 480
#define ACTIVITY_GRAPH_HEIGHT 200

#define MAX_POSSIBLE_DAYS 28

#define MINUTES_IN_DAY 1440
#define MINUTES_IN_HOUR 60

/******************/
/* global storage */
/******************/

boolean  GraphDebug = false;

int  ActivityNumberOfDays,
     ActivityTotalMinutes;
unsigned long  ActivityStartAbsVmsDay,
               ActivityStartMinute;
unsigned long  *ActivityBytesPtr,
               *ActivityRequestsPtr;
unsigned long  ActivityStartBinTime [2];

int  GraphicBitsPerPixel = 4,
     GraphicMaximumHeight = 1000,
     GraphicMaximumWidth = 1000;

char  ErrorGraphNotInit [] = "Activity statistics not initialized!",
      ErrorGraphPeriod [] = "Period problem.",
      ErrorGraphQuery [] = "Query not understood",
      ErrorGraphFuture [] = "Future history!",
      ErrorGraphHistory [] = "Too far back in history!";

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern struct ConfigStruct  Config;
extern char  HtmlSgmlDoctype[];
extern char  ServerHostPort[];
extern struct ConfigStruct  Config;

/*****************************************************************************/
/*
Initialize the per-minute activity statistics data arrays.
*/

int GraphActivityInit ()

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphActivityInit()\n");

   ActivityTotalMinutes = 0;
   if (Config.ActivityNumberOfDays < 0 ||
       Config.ActivityNumberOfDays > MAX_POSSIBLE_DAYS)
   {
      Config.ActivityNumberOfDays = 0;
      return;
   }
   ActivityNumberOfDays = Config.ActivityNumberOfDays;
   ActivityTotalMinutes = ActivityNumberOfDays * MINUTES_IN_DAY;

   sys$gettim (&ActivityStartBinTime);
   lib$day (&ActivityStartAbsVmsDay, &ActivityStartBinTime,
            &ActivityStartMinute);
   /* adjust from ten milli-second to one minute units */
   ActivityStartMinute = ActivityStartAbsVmsDay * MINUTES_IN_DAY +
                         ActivityStartMinute / 6000;

   if (Debug)
      fprintf (stdout, "%d %d %d %d %d %d\n",
               Config.ActivityNumberOfDays, ActivityTotalMinutes,
               ActivityStartAbsVmsDay, ActivityStartMinute,
               ActivityStartBinTime[1], ActivityStartBinTime[0]);

   ActivityRequestsPtr = VmGet (sizeof(unsigned int) * ActivityTotalMinutes);
   ActivityBytesPtr = VmGet (sizeof(unsigned int) * ActivityTotalMinutes);
}

/*****************************************************************************/
/*
Called with an absolute VMS day number. Return the day number relative to the
start of activity data collection.  Will be in the range of 0 ... maximum
number of days in the activity data.  If the day specified occurs before the
current day if will be folded back into the data so it is up to the calling
code to ensure reading data from there is legitimate.
*/

#ifdef __DECC
#pragma inline(GraphActivityDataIdx)
#endif

unsigned long GraphActivityDataIdx
(
unsigned long AbsVmsDay,
unsigned long StartHour
)
{
   unsigned long  DataDay;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "GraphActivityDataIdx() %d %d\n", AbsVmsDay, StartHour);

   DataDay = AbsVmsDay % ActivityNumberOfDays;
   if (Debug)
      fprintf (stdout, "DataDay: %d idx: %d\n",
               DataDay, DataDay*MINUTES_IN_DAY+StartHour*MINUTES_IN_HOUR);
   return (DataDay * MINUTES_IN_DAY + StartHour * MINUTES_IN_HOUR);
}

/*****************************************************************************/
/*
Resets to zero the activity request and byte acumulators for all days between
the last day that was cleared and this day.
*/

int GraphActivityClearDay ()

{
   static unsigned long  PrevAbsVmsDay = 0,
                         DaysWorth = MINUTES_IN_DAY * sizeof(unsigned long);

   int  status,
        idx;
   unsigned long  AbsVmsDay,
                  Day;
   unsigned long  BinTime [2];
   unsigned short  NumTime [7];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphActivityClearDay()\n");

   sys$gettim (&BinTime);
   lib$day (&AbsVmsDay, &BinTime, 0);
   if (!PrevAbsVmsDay) PrevAbsVmsDay = AbsVmsDay;

   /* just return if there are no intervening days to be cleared */
   if (PrevAbsVmsDay == AbsVmsDay) return;

   for (Day = PrevAbsVmsDay+1; Day <= AbsVmsDay; Day++)
   {
      idx = GraphActivityDataIdx (Day, 0);
      memset (&ActivityRequestsPtr[idx], 0, DaysWorth);
      memset (&ActivityBytesPtr[idx], 0, DaysWorth);
   }
   PrevAbsVmsDay = AbsVmsDay;

   /* check if the data buffer has "wrapped around" */
   if (AbsVmsDay >= ActivityStartAbsVmsDay + ActivityNumberOfDays)
   {
      /* yep, data is now available from a new start date at midnight */
      sys$gettim (&ActivityStartBinTime);
      if (VMSnok (status =
          GraphActivityOffsetTime (-(ActivityTotalMinutes-MINUTES_IN_DAY),
                                   &ActivityStartBinTime, &BinTime)))
         ErrorExitVmsStatus (status, "GraphActivityOffsetTime()", FI_LI);

      sys$numtim (&NumTime, &BinTime);
      NumTime[3] = NumTime[4] = NumTime[5] = NumTime[6] = 0;

      if (VMSnok (status = lib$cvt_vectim (&NumTime, &ActivityStartBinTime)))
         ErrorExitVmsStatus (status, "lib$cvt_vectim()", FI_LI);

      lib$day (&ActivityStartAbsVmsDay, &ActivityStartBinTime, 0);
      ActivityStartMinute = ActivityStartAbsVmsDay * MINUTES_IN_DAY;
   }
}

/*****************************************************************************/
/*
Called when concluding each request.  Adjusts per-minute activity statistics.
*/

int GraphActivityIncrement (struct RequestStruct *rqptr)

{
   static unsigned short  PrevDate = 0;
   static unsigned long  AbsVmsDay;

   register int  idx;
   unsigned long  Day;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphActivityIncrement()\n");

   if (ActivityBytesPtr == NULL || ActivityRequestsPtr == NULL) return;

   /* save the overhead of the lib$day() call if the date has not changed! */
   if (rqptr->NumericTime[2] != PrevDate)
   {
      if (PrevDate) GraphActivityClearDay ();
      lib$day (&AbsVmsDay, &rqptr->BinaryTime, 0);
      PrevDate = rqptr->NumericTime[2];
   }

   if (Debug)
      fprintf (stdout, "%d %d %d %d %d\n",
               ActivityStartAbsVmsDay, AbsVmsDay,
               ActivityStartAbsVmsDay - AbsVmsDay,
               rqptr->NumericTime[3], rqptr->NumericTime[4]);

   Day = AbsVmsDay % ActivityNumberOfDays;
   if (Debug) fprintf (stdout, "Day: %d\n", Day);
   idx = Day * MINUTES_IN_DAY +
         rqptr->NumericTime[3] * MINUTES_IN_HOUR +
         rqptr->NumericTime[4];

   ActivityRequestsPtr[idx]++;
   ActivityBytesPtr[idx] += rqptr->BytesTx;

   if (Debug)
      fprintf (stdout, "[%d] %d %d\n",
               idx, ActivityRequestsPtr[idx], ActivityBytesPtr[idx]);
}

/*****************************************************************************/
/*
Scan the specified range in the activity statistics finding the peak-per-
minute and total requests and bytes. Returns the number of minutes in the
specified range, -1 for an error. 'StartAbsVmsDay' is the VMS absolute day,
and 'Hour' the hour of the day (0..23) for the start of the scan.
'NumberOfHours' specifies the duration of the scan. 'MinuteGranularity', if
greater than one, specifies that the simple mean of the data for that period
of minutes is to be used as peak values (totals are raw!). This function
relies on calling functions to ensure the time and range makes sense!
*/

int GraphActivityDataScan
(
int StartAbsVmsDay,
int StartHour,
int NumberOfHours,
int MinuteGranularity,
boolean FindMinuteAverage,
unsigned int *PeakRequestPtr,
unsigned int *PeakBytePtr,
unsigned long *QuadTotalRequestsPtr,
unsigned long *QuadTotalBytesPtr
)
{
   static long  Addx2 = 2;

   register int  cnt, idx, mcnt;

   unsigned int  AbsDay,
                 ByteCount,
                 PeakBytes,
                 PeakRequests,
                 NumberOfMinutes,
                 RequestCount;
   unsigned long  QuadScratch [2];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "GraphActivityDataScan() %d %d %d %d\n",
               StartAbsVmsDay, StartHour, NumberOfHours, MinuteGranularity);

   if (ActivityBytesPtr == NULL || ActivityRequestsPtr == NULL)
      return (-1);

   if (NumberOfHours <= 0 || NumberOfHours > MAX_POSSIBLE_DAYS * 24)
      return (-1);

   /* ensure any days' data between the last request and now are cleared */
   GraphActivityClearDay ();

   NumberOfMinutes = NumberOfHours * MINUTES_IN_HOUR;

   idx = GraphActivityDataIdx (StartAbsVmsDay, StartHour);

   PeakRequests = PeakBytes = 0;
   if (QuadTotalBytesPtr != NULL) memset (QuadTotalBytesPtr, 0, 8);
   if (QuadTotalRequestsPtr != NULL) memset (QuadTotalRequestsPtr, 0, 8);

   for (mcnt = 0; mcnt < NumberOfMinutes; mcnt += MinuteGranularity)
   {
      if (idx >= ActivityTotalMinutes) idx = 0;
      if (Debug) fprintf (stdout, "%d idx: %d\n", mcnt, idx);

      if (MinuteGranularity == 1)
      {
         if (QuadTotalBytesPtr != NULL && QuadTotalRequestsPtr != NULL)
         {
            if (QuadScratch[0] = ActivityBytesPtr[idx])
            {
               QuadScratch[1] = 0;
               lib$addx (&QuadScratch, QuadTotalBytesPtr,
                         QuadTotalBytesPtr, &Addx2);
            }
            if (QuadScratch[0] = ActivityRequestsPtr[idx])
            {
               QuadScratch[1] = 0;
               lib$addx (&QuadScratch, QuadTotalRequestsPtr,
                         QuadTotalRequestsPtr, &Addx2);
            }
         }
         ByteCount = ActivityBytesPtr[idx];
         RequestCount = ActivityRequestsPtr[idx];
         idx++;
      }
      else
      {
         /* find the peak or average of the minute counts */
         RequestCount = ByteCount = 0;
         for (cnt = 0; cnt < MinuteGranularity; cnt++)
         {
            if (QuadTotalBytesPtr != NULL && QuadTotalRequestsPtr != NULL)
            {
               if (QuadScratch[0] = ActivityBytesPtr[idx])
               {
                  QuadScratch[1] = 0;
                  lib$addx (&QuadScratch, QuadTotalBytesPtr,
                            QuadTotalBytesPtr, &Addx2);
               }
               if (QuadScratch[0] = ActivityRequestsPtr[idx])
               {
                  QuadScratch[1] = 0;
                  lib$addx (&QuadScratch, QuadTotalRequestsPtr,
                            QuadTotalRequestsPtr, &Addx2);
               }
            }
            if (FindMinuteAverage)
            {
               ByteCount += ActivityBytesPtr[idx];
               RequestCount += ActivityRequestsPtr[idx];
            }
            else
            {
               if (ActivityBytesPtr[idx] > ByteCount)
                  ByteCount = ActivityBytesPtr[idx];
               if (ActivityRequestsPtr[idx] > RequestCount)
                  RequestCount = ActivityRequestsPtr[idx];
            }
            idx++;
         }
         if (FindMinuteAverage)
         {
            ByteCount /= MinuteGranularity;
            RequestCount /= MinuteGranularity;
         }
         if (Debug) fprintf (stdout, "%d %d\n", ByteCount, RequestCount);
      }

      if (ByteCount > PeakBytes) PeakBytes = ByteCount;
      if (RequestCount > PeakRequests) PeakRequests = RequestCount;
      if (Debug) fprintf (stdout, "%d %d\n", PeakBytes, PeakRequests);
   }

   if (Debug)
      fprintf (stdout, "PeakRequests: %d PeakBytes: %d\n",
               PeakRequests, PeakBytes);

   *PeakBytePtr = PeakBytes;
   *PeakRequestPtr = PeakRequests;

   return (NumberOfMinutes);
}

/*****************************************************************************/
/*
Round the request and byte maxima up to the next whole digit in the range
(e.g. 8745 to 9000, 320 to 400)
*/

int GraphActivityMaxima
(
unsigned int *MaxRequestPtr,
unsigned int *MaxBytePtr
)
{
   unsigned int  MaxBytes,
                 MaxRequests;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "GraphActivityMaxima() %d %d\n",
               *MaxRequestPtr, *MaxBytePtr);

   MaxRequests = *MaxRequestPtr;
   MaxBytes = *MaxBytePtr;

   if (MaxRequests < 10)
      MaxRequests = 10;
   else
   if (MaxRequests < 100)
      MaxRequests = ((MaxRequests / 10) * 10) + 10;
   else
   if (MaxRequests < 1000)
      MaxRequests = ((MaxRequests / 100) * 100) + 100;
   else
   if (MaxRequests < 10000)
      MaxRequests = ((MaxRequests / 1000) * 1000) + 1000;
   else
   if (MaxRequests < 100000)
      MaxRequests = ((MaxRequests / 10000) * 10000) + 10000;
   else
   if (MaxRequests < 1000000)
      MaxRequests = ((MaxRequests / 100000) * 100000) + 100000;

   if (MaxBytes < 10)
      MaxBytes = 10;
   else
   if (MaxBytes < 100)
      MaxBytes = ((MaxBytes / 10) * 10) + 10;
   else
   if (MaxBytes < 1000)
      MaxBytes = ((MaxBytes / 100) * 100) + 100;
   else
   if (MaxBytes < 10000)
      MaxBytes = ((MaxBytes / 1000) * 1000) + 1000;
   else
   if (MaxBytes < 100000)
      MaxBytes = ((MaxBytes / 10000) * 10000) + 10000;
   else
   if (MaxBytes < 1000000)
      MaxBytes = ((MaxBytes / 100000) * 100000) + 100000;
   else
   if (MaxBytes < 10000000)
      MaxBytes = ((MaxBytes / 1000000) * 1000000) + 1000000;
   else
   if (MaxBytes < 100000000)
      MaxBytes = ((MaxBytes / 10000000) * 10000000) + 10000000;
   else
   if (MaxBytes < 1000000000)
      MaxBytes = ((MaxBytes / 100000000) * 100000000) + 100000000;

   *MaxBytePtr = MaxBytes;
   *MaxRequestPtr = MaxRequests;
}

/*****************************************************************************/
/*
Set the offset binary time to the base binary time plus or minus the number of
minutes specified.
*/

int GraphActivityOffsetTime
(
int NumberOfMinutes,
unsigned long *BaseBinTimePtr,
unsigned long *OffsetBinTimePtr
)
{
   static unsigned long  Addend = 0,
                         OneSecond = -10000000;

   int  status;
   unsigned long  Minutes,
                  Seconds;
   unsigned long  DeltaBinTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "GraphActivityOffsetTime() %d\n", NumberOfMinutes);

   if (NumberOfMinutes >= 0)
      Minutes = NumberOfMinutes;
   else
      Minutes = NumberOfMinutes * -1;
   if (Minutes)
      Seconds = Minutes * MINUTES_IN_HOUR;
   else
      Seconds = 1;

   status = lib$emul (&Seconds, &OneSecond, &Addend, &DeltaBinTime);
   if (Debug) fprintf (stdout, "lib$emul() %%X%08.08X\n", status);
   if (NumberOfMinutes >= 0)
   {
      status = lib$add_times (BaseBinTimePtr, &DeltaBinTime, OffsetBinTimePtr);
      if (Debug) fprintf (stdout, "lib$add_times() %%X%08.08X\n", status);
      return (status);
   }
   else
   {
      status = lib$sub_times (BaseBinTimePtr, &DeltaBinTime, OffsetBinTimePtr);
      if (Debug) fprintf (stdout, "lib$sub_times() %%X%08.08X\n", status);
      return (status);
   }
}

/*****************************************************************************/
/*
Generate the HTML activity report page.  The supplied query string specifies
the END TIME for the the display, NOT THE START.  The period actually
specifies for what duration prior to the specified end time the display should
be!

The JavaScript code activates when a duration-only request is made. The
JavaScript provides an automatic update of the activity report at a frequency
appropriate to the period of the report. If the browser is not in focus then
the update is not made (at least that's how the script is designed and
Navigator 3.0 behaves, MSIE 3.02 has still a ways to go :^), it is defered
until it is brought back into focus. This save "background" updates that would
essentially be wasted processing and bandwidth. JavaScript 1.1 is required,
and the script is only enabled for versions of Navigator and MSIE that are
known to support the required functionality (primarily the "location.replace"
method). JavaScript may be completely disabled using the JAVASCRIPT_ENABLED
define and recompiling.

The "uniquifier" performs two tasks.  First, although the images and reports
are pre-expired MSIE (3.02) continues to use images from the cache over and
again!  Second, it prevents a browser doing a JavaSciprt automatic update
using a cached report or image if the server becomes unavailable.
*/ 

int GraphActivityReport
(
struct RequestStruct  *rqptr,
void *NextTaskFunction
)
{
   static char *MonthName [] =
      { "", "JAN", "FEB", "MAR", "APR", "MAY", "JUN",
            "JUL", "AUG", "SEP", "OCT", "NOV", "DEC" };

char  UpdateJavaScript [] =
"<SCRIPT LANGUAGE=\"JavaScript\">\n\
<!--\n\
\
var activityUrl = null;\n\
var canDo = false;\n\
var errorOccured = false;\n\
var needsUpdating = false;\n\
var windowInFocus = true;\n\
\
function updatePage()\n\
{\n\
   if (windowInFocus)\n\
      location.replace(activityUrl);\n\
   else\n\
      needsUpdating = true;\n\
}\n\
\
function windowFocus()\n\
{\n\
   if (canDo && needsUpdating && !errorOccured) location.replace(activityUrl);\n\
   windowInFocus = true;\n\
}\n\
\
function startUpdatePage(Url,updateSeconds)\n\
{\n\
   activityUrl = Url;\n\
   if (navigator.appName.substring(0,8) == \"Netscape\")\n\
   {\n\
       version = parseFloat(navigator.appVersion);\n\
       if (version >= 3.0) canDo = true;\n\
   }\n\
   else\n\
   if ((idx = navigator.appVersion.indexOf(\"MSIE\")) > 0)\n\
   {\n\
      version = parseFloat(\
navigator.appVersion.substring(idx+5,navigator.appVersion.length));\n\
      if (version >= 3.02) canDo = true;\n\
   }\n\
   if (canDo && !errorOccured) setTimeout(\"updatePage()\",updateSeconds*1000);\n\
}\n\
\
function suppressError()\n\
{\n\
   errorOccured = true;\n\
   return true;\n\
}\n\
\
//-->\n\
</SCRIPT>\n";

   static $DESCRIPTOR (OnLoadJavaScriptFaoDsc,
" ONLOAD=\"startUpdatePage(\'!AZ?!UL!AZ\',!UL)\"\
 ONERROR=\"suppressError()\"\
 ONFOCUS=\"windowFocus()\"\
 ONBLUR=\"windowInFocus = false\"\
\0");

   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
!AZ\
<TITLE>HTTPd !AZ ... Server Activity</TITLE>\n\
</HEAD>\n\
<BODY!AZ>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Server Activity</H2>\n\
!AZ\n\
\
<P><TABLE CELLPADDING=0 CELLSPACING=0 BORDER=0>\n\
\
<TR>\
<TD ALIGN=right>!UL</TD>\
<TD>&nbsp;</TD>\
<TD ALIGN=center COLSPAN=3></TD>\
<TD>&nbsp;</TD>\
<TD ALIGN=left>!UL!AZ</TD>\
</TR>\n\
\
<TR>\
<TD ALIGN=right>!AZ</TD>\
<TD>&nbsp;</TD>\
<TD COLSPAN=3>\
<IMG WIDTH=!UL HEIGHT=!UL BORDER=0 ALT=\"[activity graph]\" \
USEMAP=\"#1\" SRC=\"!AZ?!4ZL!2ZL!2ZL!2ZL+!UL+!UL+!UL!AZ\">\
</TD>\
<TD>&nbsp;</TD>\
<TD ALIGN=left>!AZ</TD>\
</TR>\n\
\
<TR>\
<TD ALIGN=right>!UL</TD>\
<TD>&nbsp;</TD>\
<TD ALIGN=left>&nbsp;&nbsp;<FONT SIZE=-2>!UL-!AZ !2ZL:!2ZL</FONT></TD>\
<TD ALIGN=center></TD>\
<TD ALIGN=right><FONT SIZE=-2>!UL-!AZ !2ZL:!2ZL</FONT>&nbsp;&nbsp;</TD>\
<TD>&nbsp;</TD>\
<TD ALIGN=left>!UL</TD>\
</TR>\n\
\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLPSPACING=1 BORDER=0>\n\
<TR><TH ALIGN=right>Period:</TH><TD>&nbsp;!AZ &nbsp;(!UL !AZ)\
<TR><TH ALIGN=right>Requests:</TH><TD>&nbsp;!AZ total; &nbsp;!AZ peak</TD></TR>\n\
<TR><TH ALIGN=right>Bytes:</TH><TD>&nbsp;!AZ total; &nbsp;!AZ peak</TD></TR>\n\
<TR><TH></TH>\
<TD ALIGN=left>&nbsp;<FONT SIZE=-2>(Data available from !AZ)</FONT></TD>\
</TR>\n\
</TABLE>\n\
<MAP NAME=\"1\">\n");

   static $DESCRIPTOR (AreaFaoDsc,
"<AREA SHAPE=\"rect\" COORDS=\"!UL,!UL,!UL,!UL\" \
HREF=\"!AZ?!4ZL!2ZL!2ZL!2ZL+!UL\"!AZ>\n");

   static char  EndPage [] =
"</MAP>\n\
</BODY>\n\
</HTML>\n";

   static $DESCRIPTOR (MultipleDaysFaoDsc, "!AZ to !AZ\0");
   static $DESCRIPTOR (UniquifierFaoDsc, "+!2ZL!2ZL!2ZL!2ZL!2ZL\0");
   static $DESCRIPTOR (WithinOneDayFaoDsc, "!AZ to !2ZL:!2ZL\0");

   register char  *cptr;
   register unsigned long  *vecptr;

   boolean  FindMinuteAverage,
            JavaScriptEnabled,
            HourSupplied;
   int  status,
        AtX,
        Bytes,
        ColumnWidth,
        Count,
        Day,
        GraphHeight,
        GraphWidth,
        Hour,
        IncrementMinutes,
        JavaScriptUniquifier,
        MapSections,
        MinuteGranularity,
        Minutes,
        Month,
        NumberOfDays,
        NumberOfHours,
        NumberOfMinutes,
        PeriodHours,
        StartHour,
        UpdateSeconds,
        Year;
   unsigned short  Length;
   unsigned short  CurrentNumTime [7],
                   EndNumTime [7],
                   NumTime [7],
                   StartNumTime [7];
   unsigned long  AbsVmsDay,
                  CurrentAbsVmsDay,
                  CurrentMinute,
                  CurrentSecond,
                  EndMinute,
                  EndSecond,
                  MaxBytes,
                  MaxRequests,
                  Minute,
                  TenmSecSinceMidnight,
                  PeakBytes,
                  PeakRequests,
                  StartAbsVmsDay,
                  Second;
   unsigned long  FaoVector [64];
   unsigned long  BinTime [2],
                  CurrentBinTime [2],
                  EndBinTime [2],
                  StartBinTime [2],
                  EndThisPeriodBinTime [2],
                  QuadTotalBytes [2],
                  QuadTotalRequests [2];
   char  *BytesPtr,
         *EndPtr,
         *HoursPtr,
         *StartPtr;
   char  Buffer [4096],
         DeltaTime [32],
         EndDayDateTime [32],
         OnLoadJavaScript [256],
         PeakBytesString [32],
         PeakRequestsString [32],
         Period [128],
         StartDayDateTime [32],
         ActivityStartDayDateTime [32],
         TotalBytesString [32],
         TotalRequestsString [32],
         Uniquifier [16];
   $DESCRIPTOR (BufferDsc, Buffer);
   $DESCRIPTOR (DeltaTimeDsc, DeltaTime);
   $DESCRIPTOR (OnLoadJavaScriptDsc, OnLoadJavaScript);
   $DESCRIPTOR (PeriodDsc, Period);
   $DESCRIPTOR (UniquifierDsc, Uniquifier);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphActivityReport()\n");

   JavaScriptEnabled = false;

   /**************************/
   /* parse the query string */
   /**************************/

   HourSupplied = false;
   Year = Month = Day = Hour = NumberOfHours = JavaScriptUniquifier = 0;
   if (rqptr->QueryStringPtr[0])
   {
      if (isdigit(*(cptr = rqptr->QueryStringPtr)))
      {
         /* "keyword"-based query string */
         while (isdigit(*cptr)) cptr++;
         if (cptr <= rqptr->QueryStringPtr + 3)
         {
            /* assume just a duration (and possibly a uniquifier) */
            sscanf (rqptr->QueryStringPtr, "%d+%d",
                    &NumberOfHours, &JavaScriptUniquifier);

            /* enable JavaScript for duration-only based requests */
            if (JAVASCRIPT_ENABLED) JavaScriptEnabled = true;
         }
         else
         if (sscanf (rqptr->QueryStringPtr, "%4d%2d%2d%2d+%d+%d",
                     &Year, &Month, &Day, &Hour, &NumberOfHours,
                     &JavaScriptUniquifier) < 5)
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, ErrorGraphQuery, FI_LI);
            SysDclAst (NextTaskFunction, rqptr);
            return;
         }
         else
            HourSupplied = true;
      }
      else
      {
         /* "form"-based query string */
         while (*cptr)
         {
            if (strsame (cptr, "yr=", 3))
               Year = atoi(cptr+3);
            else
            if (strsame (cptr, "mn=", 3))
               Month = atoi(cptr+3);
            else
            if (strsame (cptr, "dy=", 3))
               Day = atoi(cptr+3);
            else
            if (strsame (cptr, "hr=", 3))
            {
               Hour = atoi(cptr+3);
               HourSupplied = true;
            }
            else
            if (strsame (cptr, "du=", 3))
               NumberOfHours = atoi(cptr+3);
            else
            {
               rqptr->ResponseStatusCode = 400;
               ErrorGeneral (rqptr, ErrorGraphQuery, FI_LI);
               SysDclAst (NextTaskFunction, rqptr);
               return;
            }
            while (*cptr && *cptr != '&') cptr++;
            if (*cptr) cptr++;
         }

         /* disable javascript for specific-period requests */
         JavaScriptEnabled = false;
      }
   }

   /***********/
   /* process */
   /***********/

   FindMinuteAverage = false;

   GraphWidth = ACTIVITY_GRAPH_WIDTH + 2;
   GraphHeight = ACTIVITY_GRAPH_HEIGHT + 2;

   sys$gettim (&CurrentBinTime);
   sys$numtim (&CurrentNumTime, &CurrentBinTime);
   lib$day (&CurrentAbsVmsDay, &CurrentBinTime, 0);

   /* defaults for any time components not supplied */
   if (!Year) Year = CurrentNumTime[0];
   if (!Month) Month = CurrentNumTime[1];
   if (!Day) Day = CurrentNumTime[2];
   if (!Hour && !HourSupplied) Hour = CurrentNumTime[3];
   if (!NumberOfHours) NumberOfHours = 1;

   /* make it a multiple of 4 (which divides the graph up nicely :^) */
   if (NumberOfHours > 2) while (NumberOfHours % 4) NumberOfHours++;
   NumberOfMinutes = NumberOfHours * MINUTES_IN_HOUR;
   if (Debug) fprintf (stdout, "NumberOfMinutes: %d\n", NumberOfMinutes);

   EndNumTime[0] = Year;
   EndNumTime[1] = Month;
   EndNumTime[2] = Day;
   EndNumTime[3] = Hour;
   /* always ends after the 59th minute! */
   EndNumTime[4] = 59;
   EndNumTime[5] = EndNumTime[6] = 0;

   if (VMSnok (status = lib$cvt_vectim (&EndNumTime, &EndBinTime)))
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorGraphPeriod, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }
   strcpy (EndDayDateTime, DayDateTime (&EndBinTime, 17));

   /* get the start time as the number of minutes before the end time */
   if (VMSnok (status =
       GraphActivityOffsetTime (-(NumberOfMinutes-1),
                                &EndBinTime, &StartBinTime)))
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorGraphPeriod, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   if (VMSnok (status = lib$day (&StartAbsVmsDay, &StartBinTime, 0)))
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorGraphPeriod, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   sys$numtim (&StartNumTime, &StartBinTime);
   StartHour = StartNumTime[3];
   strcpy (StartDayDateTime, DayDateTime (&StartBinTime, 17));

   if (Debug)
      fprintf (stdout, "day: %d %d hour: %d %d\n",
               CurrentAbsVmsDay,  StartAbsVmsDay, CurrentNumTime[3], StartHour);

   if (ActivityStartAbsVmsDay - StartAbsVmsDay < 0)
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorGraphHistory, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   if (StartAbsVmsDay > CurrentAbsVmsDay ||
       (StartAbsVmsDay == CurrentAbsVmsDay && StartHour > CurrentNumTime[3]))
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorGraphFuture, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /***********************/
   /* make some estimates */
   /***********************/

   /* width of the bar graph */
   ColumnWidth = (GraphWidth-2) / (NumberOfMinutes * 2);
   if (!ColumnWidth) ColumnWidth = 1;

   /* calculate simple mean for this number of minutes when duration large */
   MinuteGranularity =
      (int)(1.0 / ((float)(GraphWidth-2) / ((float)NumberOfMinutes * 2.0)));
   if (!MinuteGranularity) MinuteGranularity = 1;

   if (Debug)
      fprintf (stdout, "ColumnWidth %d MinuteGranularity: %d\n",
               ColumnWidth, MinuteGranularity);

   GraphActivityDataScan (StartAbsVmsDay, StartHour, NumberOfHours,
                          MinuteGranularity, FindMinuteAverage,
                          &PeakRequests, &PeakBytes,
                          &QuadTotalRequests, &QuadTotalBytes);

   MaxBytes = PeakBytes;
   MaxRequests = PeakRequests;
   GraphActivityMaxima (&MaxRequests, &MaxBytes);

   if (MaxBytes < 1000)
   {
      Bytes = MaxBytes;
      BytesPtr = "";
   }
   else
   if (MaxBytes < 1000000)
   {
      Bytes = MaxBytes / 1000;
      BytesPtr = " K";
   }
   else
   {
      Bytes = MaxBytes / 1000000;
      BytesPtr = " M";
   }

   if (StartNumTime[0] != EndNumTime[0] ||
       StartNumTime[1] != EndNumTime[1] ||
       StartNumTime[2] != EndNumTime[2])
      sys$fao (&MultipleDaysFaoDsc, 0, &PeriodDsc, 
               StartDayDateTime, EndDayDateTime);
   else
      sys$fao (&WithinOneDayFaoDsc, 0, &PeriodDsc, 
               StartDayDateTime, EndNumTime[3], EndNumTime[4]);

   CommaNumber (64, &QuadTotalRequests, TotalRequestsString);
   CommaNumber (64, &QuadTotalBytes, TotalBytesString);
   CommaNumber (32, PeakBytes, PeakBytesString);
   CommaNumber (32, PeakRequests, PeakRequestsString);

   strcpy (ActivityStartDayDateTime, DayDateTime (&ActivityStartBinTime, 17));

   /**********************/
   /* generate HTML page */
   /**********************/

   if (NumberOfHours <= 4)
      UpdateSeconds = 60;
   else
   if (NumberOfHours <= 8)
      UpdateSeconds = 120;
   else
   if (NumberOfHours <= 24)
      UpdateSeconds = 300;
   else
      UpdateSeconds = 3600;

   vecptr = FaoVector;
   *vecptr++ = CurrentNumTime[1];
   *vecptr++ = CurrentNumTime[2];
   *vecptr++ = CurrentNumTime[3];
   *vecptr++ = CurrentNumTime[4];
   *vecptr++ = CurrentNumTime[5];
   sys$faol (&UniquifierFaoDsc, 0, &UniquifierDsc, &FaoVector);

   if (JavaScriptEnabled)
   {
      vecptr = FaoVector;
      *vecptr++ = HttpdInternalReportActivity;
      *vecptr++ = NumberOfHours;
      *vecptr++ = Uniquifier;
      *vecptr++ = UpdateSeconds - CurrentNumTime[5];

      status = sys$faol (&OnLoadJavaScriptFaoDsc, 0, &OnLoadJavaScriptDsc,
                         &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }
   }

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);
   if (JavaScriptEnabled)
      *vecptr++ = UpdateJavaScript;
   else
      *vecptr++ = "";
   *vecptr++ = ServerHostPort;
   if (JavaScriptEnabled)
      *vecptr++ = OnLoadJavaScript;
   else
      *vecptr++ = "";
   *vecptr++ = ServerHostPort;
   *vecptr++ = DayDateTime (&rqptr->BinaryTime, 20);

   *vecptr++ = MaxRequests;
   *vecptr++ = Bytes;
   *vecptr++ = BytesPtr;
   *vecptr++ =
"<FONT COLOR=\"#00bbbb\">Requests</FONT><BR>\
<FONT SIZE=-1 COLOR=\"#888800\">(mean)</FONT><BR>\
<FONT SIZE=-3>per-minute</FONT>";            
   *vecptr++ = GraphWidth;
   *vecptr++ = GraphHeight;
   *vecptr++ = HttpdInternalGraphicActivity;
   *vecptr++ = Year;
   *vecptr++ = Month;
   *vecptr++ = Day;
   *vecptr++ = Hour;
   *vecptr++ = NumberOfHours;
   *vecptr++ = MaxRequests;
   *vecptr++ = MaxBytes;
   *vecptr++ = Uniquifier;
   *vecptr++ =
"<FONT COLOR=\"#0000ff\">Bytes</FONT><BR>\
<FONT SIZE=-1 COLOR=\"#ff0000\">(mean)</FONT><BR>\
<FONT SIZE=-3>per-minute</FONT>";
   *vecptr++ = 0;

   *vecptr++ = StartNumTime[2];
   *vecptr++ = MonthName[StartNumTime[1]];
   *vecptr++ = StartNumTime[3];
   *vecptr++ = StartNumTime[4];

   *vecptr++ = EndNumTime[2];
   *vecptr++ = MonthName[EndNumTime[1]];
   *vecptr++ = EndNumTime[3];
   *vecptr++ = EndNumTime[4];

   *vecptr++ = 0;

   *vecptr++ = Period;
   *vecptr++ = NumberOfHours;
   if (NumberOfHours == 1)
      *vecptr++ = "hour";
   else
      *vecptr++ = "hours";
   *vecptr++ = TotalRequestsString;
   *vecptr++ = PeakRequestsString;
   *vecptr++ = TotalBytesString;
   *vecptr++ = PeakBytesString;

   *vecptr++ = ActivityStartDayDateTime;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }
   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   /*************************/
   /* image map coordinates */
   /*************************/

   NumberOfDays = NumberOfHours / 24;
   if (NumberOfDays > 1)
   {
      MapSections = NumberOfDays;
      IncrementMinutes = MINUTES_IN_DAY;
   }
   else
   {
      MapSections = NumberOfHours;
      IncrementMinutes = MINUTES_IN_HOUR;
   }
   Minute = 0;

   if (NumberOfHours > 1)
   {
      /***************************/
      /* reduce period map areas */
      /***************************/

      switch (NumberOfHours)
      {
         case 1 : { PeriodHours = 1; break; }
         case 2 : { PeriodHours = 1; break; }
         case 4 : { PeriodHours = 2; break; }
         case 12 : { PeriodHours = 4; break; }
         case 24 : { PeriodHours = 12; break; }
         case 72 : { PeriodHours = 24; break; }
         case 168 : { PeriodHours = 72; break; }
         case 672 : { PeriodHours = 168; break; }
         default : PeriodHours = NumberOfHours / 2;
      }
      if (!PeriodHours) PeriodHours = 1;

      if (VMSnok (status =
          GraphActivityOffsetTime (PeriodHours*60,
                                   &CurrentBinTime, &EndThisPeriodBinTime)))
      {
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }

      Count = 0;
      while (Count < MapSections)
      {
         AtX = (int)((float)Count * (float)GraphWidth / (float)MapSections);
         Count++;
         Minute += IncrementMinutes;

         /* 'BinTime' will contain the period end time */
         if (VMSnok (status =
             GraphActivityOffsetTime (Minute-1, &StartBinTime, &BinTime)))
         {
            ErrorVmsStatus (rqptr, status, FI_LI);
            SysDclAst (NextTaskFunction, rqptr);
            return;
         }

         /*
            If this period end time is later than server start time, and
            the end time is earlier than the end of this period, then ...
         */
         if (CompareVmsBinTimes (&BinTime, &ActivityStartBinTime) > 0 &&
             CompareVmsBinTimes (&CurrentBinTime, &EndThisPeriodBinTime) < 0)
         {             
            sys$numtim (&NumTime, &BinTime);

            vecptr = FaoVector;
            *vecptr++ = AtX;
            *vecptr++ = GraphHeight / 2;
            *vecptr++ = AtX + (GraphWidth / MapSections);
            *vecptr++ = GraphHeight;
            *vecptr++ = HttpdInternalReportActivity;
            *vecptr++ = NumTime[0];
            *vecptr++ = NumTime[1];
            *vecptr++ = NumTime[2];
            *vecptr++ = NumTime[3];
            *vecptr++ = PeriodHours;
            /* enable JavaScript for duration-only based requests */
            if (JAVASCRIPT_ONMOUSEOVER_ENABLED)
               *vecptr++ = GraphActivityOnMouseOver (&BinTime, PeriodHours);
            else
               *vecptr++ = "";

            status = sys$faol (&AreaFaoDsc, &Length, &BufferDsc, &FaoVector);
            if (VMSnok (status) || status == SS$_BUFFEROVF)
            {
               rqptr->ErrorTextPtr = "sys$faol()";
               ErrorVmsStatus (rqptr, status, FI_LI);
               SysDclAst (NextTaskFunction, rqptr);
               return;
            }
            Buffer[Length] = '\0';
            NetWriteBuffered (rqptr, 0, Buffer, Length);
         }
      }
   }

   /************************/
   /* less recent map area */
   /************************/

   if (CompareVmsBinTimes (&StartBinTime, &ActivityStartBinTime) > 0)
   {
      if (VMSnok (status =
         GraphActivityOffsetTime (-(NumberOfHours*MINUTES_IN_HOUR),
                                  &EndBinTime, &BinTime)))
      {
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }
      sys$numtim (&NumTime, &BinTime);

      vecptr = FaoVector;
      *vecptr++ = 0;
      *vecptr++ = 0;
      *vecptr++ = GraphWidth / 4;
      *vecptr++ = GraphHeight / 2;
      *vecptr++ = HttpdInternalReportActivity;
      *vecptr++ = NumTime[0];
      *vecptr++ = NumTime[1];
      *vecptr++ = NumTime[2];
      *vecptr++ = NumTime[3];
      *vecptr++ = NumberOfHours;
      if (JAVASCRIPT_ONMOUSEOVER_ENABLED)
         *vecptr++ = GraphActivityOnMouseOver (&BinTime, NumberOfHours);
      else
         *vecptr++ = "";

      status = sys$faol (&AreaFaoDsc, &Length, &BufferDsc, &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }
      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   if (NumberOfDays < ActivityNumberOfDays)
   {
      /****************************/
      /* increase period map area */
      /****************************/

      switch (NumberOfHours)
      {
         case 1 : { PeriodHours = 2; break; }
         case 2 : { PeriodHours = 4; break; }
         case 4 : { PeriodHours = 12; break; }
         case 12 : { PeriodHours = 24; break; }
         case 24 : { PeriodHours = 72; break; }
         case 72 : { PeriodHours = 168; break; }
         case 168 : { PeriodHours = 672; break; }
         default : PeriodHours = NumberOfHours * 2;
      }
      if (PeriodHours > ActivityNumberOfDays * 24)
         PeriodHours = ActivityNumberOfDays * 24;

      vecptr = FaoVector;
      *vecptr++ = GraphWidth / 4;
      *vecptr++ = 0;
      *vecptr++ = GraphWidth / 2 + GraphWidth / 4;
      *vecptr++ = GraphHeight / 2;
      *vecptr++ = HttpdInternalReportActivity;
      *vecptr++ = EndNumTime[0];
      *vecptr++ = EndNumTime[1];
      *vecptr++ = EndNumTime[2];
      *vecptr++ = EndNumTime[3];
      *vecptr++ = PeriodHours;
      if (JAVASCRIPT_ONMOUSEOVER_ENABLED)
         *vecptr++ = GraphActivityOnMouseOver (&EndBinTime, PeriodHours);
      else
         *vecptr++ = "";

      status = sys$faol (&AreaFaoDsc, &Length, &BufferDsc, &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }
      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   /************************/
   /* more recent map area */
   /************************/

   if (CompareVmsBinTimes (&CurrentBinTime, &EndBinTime) > 0)
   {
      if (VMSnok (status =
         GraphActivityOffsetTime (NumberOfHours*MINUTES_IN_HOUR,
                                  &EndBinTime, &BinTime)))
      {
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }
      sys$numtim (&NumTime, &BinTime);

      vecptr = FaoVector;
      *vecptr++ = GraphWidth / 2 + GraphWidth / 4;
      *vecptr++ = 0;
      *vecptr++ = GraphWidth;
      *vecptr++ = GraphHeight / 2;
      *vecptr++ = HttpdInternalReportActivity;
      *vecptr++ = NumTime[0];
      *vecptr++ = NumTime[1];
      *vecptr++ = NumTime[2];
      *vecptr++ = NumTime[3];
      *vecptr++ = NumberOfHours;
      if (JAVASCRIPT_ONMOUSEOVER_ENABLED)
         *vecptr++ = GraphActivityOnMouseOver (&BinTime, NumberOfHours);
      else
         *vecptr++ = "";

      status = sys$faol (&AreaFaoDsc, &Length, &BufferDsc, &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }
      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   /************/
   /* end page */
   /************/

   NetWriteBuffered (rqptr, NextTaskFunction, EndPage, sizeof(EndPage)-1);
}

/*****************************************************************************/
/*
*/

char* GraphActivityOnMouseOver
(
unsigned long *EndBinTimePtr,
int NumberOfHours
)
{
   static $DESCRIPTOR (OnMouseOverFaoDsc,
" ONMOUSEOVER=\"window.status=\'!17%D  to  !17%D  (!UL hour!AZ)\';\
 return true\"\
 ONMOUSEOUT=\"window.status=\'\'\; return true\"\0");
   static char OnMouseOverError [] =
" ONMOUSEOVER=\"window.status=\'*ERROR*\'; return true\"\
 ONMOUSEOUT=\"window.status=\'\'\ return true\"";

   static char  OnMouseOver [256];
   static $DESCRIPTOR (OnMouseOverDsc, OnMouseOver);

   register unsigned long  *vecptr;

   int  status;
   unsigned long  StartBinTime [2];
   unsigned long  FaoVector [8];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "GraphActivityOnMouseOver() %d\n", NumberOfHours);

   if (VMSnok (status =
       GraphActivityOffsetTime (-(NumberOfHours*MINUTES_IN_HOUR-1),
                                EndBinTimePtr, &StartBinTime)))
   {
      if (Debug) fprintf (stdout, "GraphOffsetTime() %%X%08.08X\n", status);
      return (OnMouseOverError);
   }

   vecptr = FaoVector;
   *vecptr++ = &StartBinTime;
   *vecptr++ = EndBinTimePtr;
   *vecptr++ = NumberOfHours;
   if (NumberOfHours == 1)
      *vecptr++ = "";
   else
      *vecptr++ = "s";
   status = sys$faol (&OnMouseOverFaoDsc, 0, &OnMouseOverDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      if (Debug) fprintf (stdout, "sys$faol() %%X%08.08X\n", status);
      return (OnMouseOverError);
   }

   if (Debug) fprintf (stdout, "|%s|\n", OnMouseOver);
   return (OnMouseOver);
}

/*****************************************************************************/
/*
Generate the activity graph GIF.  An <IMG...> tag in the HTML report page
results in this function begin called.
*/

int GraphActivityPlotBegin
(
struct RequestStruct  *rqptr,
void *NextTaskFunction
)
{
   static $DESCRIPTOR (DeltaTimeFaoDsc, "!UL !2ZL:!2ZL:00.00");

   register int  cnt, idx, mcnt;
   register char  *cptr;
   register struct GraphStruct *grptr;
   register struct GraphTaskStruct  *tkptr;

   boolean  FindMinuteAverage,
            HourSupplied;
   int  status,
        AbsDay,
        AtX,
        AxisBytes,
        AxisRequests,
        ByteCount,
        ByteHeight,
        ByteMean,
        ByteTotal,
        ColourAxis,
        ColourByte,
        ColourByteMean,
        ColourNoData,
        ColourRequest,
        ColourRequestMean,
        ColumnWidth,
        CurrentMinute,
        Day,
        DisplayMean,
        GraphHeight,
        GraphWidth,
        Hour,
        MinuteGranularity,
        Month,
        NumberOfHours,
        NumberOfDays,
        NumberOfMinutes,
        PrevAtX,
        PrevByteMean,
        PrevRequestMean,
        RequestCount,
        RequestHeight,
        RequestMean,
        RequestTotal,
        SampleCount,
        StartHour,
        Year;
   unsigned long  AbsActivityDay,
                  CurrentAbsVmsDay,
                  DeltaDays,
                  DeltaHours,
                  DeltaMinutes,
                  EndMinute,
                  EndSecond,
                  MaxRequests,
                  MaxBytes,
                  PeakBytes,
                  PeakRequests,
                  StartAbsVmsDay,
                  StartMinute,
                  StartSecond,
                  Minute;
   unsigned long  FaoVector [32];
   unsigned long  BinTime [2],
                  CurrentBinTime [2],
                  EndBinTime [2],
                  StartBinTime [2];
   unsigned short  CurrentNumTime [7],
                   EndNumTime [7],
                   NumTime [7],
                   StartNumTime [7];
   float  ByteFactor,
          RequestFactor;
   char  DeltaTime [32];
   $DESCRIPTOR (DeltaTimeDsc, DeltaTime);

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "GraphActivityPlotBegin() %d %d %d %d %d\n",
               Day, Hour, NumberOfHours, MaxRequests, MaxBytes);

   if (ActivityBytesPtr == NULL || ActivityRequestsPtr == NULL)
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorGraphNotInit, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /**************************/
   /* parse the query string */
   /**************************/

   HourSupplied = false;
   Year = Month = Day = Hour = NumberOfHours = MaxRequests = MaxBytes = 0;
   if (rqptr->QueryStringPtr[0])
   {
      cptr = rqptr->QueryStringPtr;
      if (isdigit(*cptr))
      {
         /* "keyword"-based query string */
         if (strlen (cptr) <= 3)
         {
            /* assume just a duration has been supplied */
            NumberOfHours = atoi(cptr);
         }
         else
         if (sscanf (cptr, "%4d%2d%2d%2d+%d+%d+%d",
                     &Year, &Month, &Day, &Hour, &NumberOfHours,
                     &MaxRequests, &MaxBytes) != 7)
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, ErrorGraphQuery, FI_LI);
            SysDclAst (NextTaskFunction, rqptr);
            return;
         }
         else
            HourSupplied = true;
      }
      else
      {
         /* "form"-based query string */
         while (*cptr)
         {
            if (strsame (cptr, "yr=", 3))
               Year = atoi(cptr+3);
            else
            if (strsame (cptr, "mn=", 3))
               Month = atoi(cptr+3);
            else
            if (strsame (cptr, "dy=", 3))
               Day = atoi(cptr+3);
            else
            if (strsame (cptr, "hr=", 3))
            {
               Hour = atoi(cptr+3);
               HourSupplied = true;
            }
            else
            if (strsame (cptr, "du=", 3))
               NumberOfHours = atoi(cptr+3);
            else
            if (strsame (cptr, "mr=", 3))
               MaxRequests = atoi(cptr+3);
            else
            if (strsame (cptr, "mb=", 3))
               MaxBytes = atoi(cptr+3);
            else
            {
               rqptr->ResponseStatusCode = 400;
               ErrorGeneral (rqptr, ErrorGraphQuery, FI_LI);
               SysDclAst (NextTaskFunction, rqptr);
               return;
            }
            while (*cptr && *cptr != '&') cptr++;
            if (*cptr) cptr++;
         }
      }
   }

   /**************************/
   /* create graph structure */
   /**************************/

   /* set up the task structure (only ever one per request!) */
   rqptr->GraphTaskPtr = tkptr = (struct GraphTaskStruct*)
      VmGetHeap (rqptr, sizeof(struct GraphTaskStruct));
   tkptr->NextTaskFunction = NextTaskFunction;

   /* allocate heap memory for the graph structure */
   grptr = rqptr->GraphTaskPtr->GraphPtr = (struct GraphStruct*)
        VmGetHeap (rqptr, sizeof(struct GraphStruct));

   /***********/
   /* process */
   /***********/

   FindMinuteAverage = false;

   GraphWidth = ACTIVITY_GRAPH_WIDTH + 2;
   GraphHeight = ACTIVITY_GRAPH_HEIGHT + 2;

   ColourAxis= COLOUR_BLACK;
   ColourByte = COLOUR_BLUE;
   ColourNoData = COLOUR_GREY;
   ColourRequest = COLOUR_CYAN;
   ColourByteMean = COLOUR_RED;
   ColourRequestMean = COLOUR_DYELLOW;

   sys$gettim (&CurrentBinTime);
   sys$numtim (&CurrentNumTime, &CurrentBinTime);
   lib$day (&CurrentAbsVmsDay, &CurrentBinTime, 0);

   /* defaults for any time components not supplied */
   if (!Year) Year = CurrentNumTime[0];
   if (!Month) Month = CurrentNumTime[1];
   if (!Day) Day = CurrentNumTime[2];
   if (!Hour && !HourSupplied) Hour = CurrentNumTime[3];
   if (!NumberOfHours) NumberOfHours = 1;

   /* make it a multiple of 4 (which divides the graph up nicely :^) */
   if (NumberOfHours > 2)
      while (NumberOfHours % 4) NumberOfHours++;
   NumberOfMinutes = NumberOfHours * MINUTES_IN_HOUR;
   if (Debug) fprintf (stdout, "NumberOfMinutes: %d\n", NumberOfMinutes);

   EndNumTime[0] = Year;
   EndNumTime[1] = Month;
   EndNumTime[2] = Day;
   EndNumTime[3] = Hour;
   /* always ends after the 59th minute! */
   EndNumTime[4] = 59;
   EndNumTime[5] = EndNumTime[6] = 0;

   if (VMSnok (status = lib$cvt_vectim (&EndNumTime, &EndBinTime)))
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorGraphPeriod, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /* get the start time as the number of minutes before the end time */
   if (VMSnok (status =
       GraphActivityOffsetTime (-(NumberOfMinutes-1),
                                &EndBinTime, &StartBinTime)))
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorGraphPeriod, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   if (VMSnok (status = lib$day (&StartAbsVmsDay, &StartBinTime, 0)))
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorGraphPeriod, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /* calculate the start time */
   AbsDay = (StartAbsVmsDay - ActivityStartAbsVmsDay) % ActivityNumberOfDays;
   sys$numtim (&StartNumTime, &StartBinTime);
   StartHour = StartNumTime[3];

   if (Debug)
      fprintf (stdout, "day: %d %d hour: %d %d minute: %d %d\n",
               CurrentAbsVmsDay,  StartAbsVmsDay,
               CurrentNumTime[3], StartHour, StartMinute, EndMinute);

   if (ActivityStartAbsVmsDay - StartAbsVmsDay < 0)
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorGraphHistory, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   if (StartAbsVmsDay > CurrentAbsVmsDay ||
       (StartAbsVmsDay == CurrentAbsVmsDay && StartHour > CurrentNumTime[3]))
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorGraphFuture, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /***********************/
   /* make some estimates */
   /***********************/

   /* width of the bar graph */
   ColumnWidth = (GraphWidth-2) / (NumberOfMinutes * 2);
   if (!ColumnWidth) ColumnWidth = 1;

   /* calculate simple mean for this number of minutes when duration large */
   MinuteGranularity =
      (int)(1.0 / ((float)(GraphWidth-2) / ((float)NumberOfMinutes * 2.0)));
   if (!MinuteGranularity) MinuteGranularity = 1;

   if (Debug)
      fprintf (stdout, "ColumnWidth %d MinuteGranularity: %d\n",
               ColumnWidth, MinuteGranularity);

   if (!MaxRequests || !MaxBytes)
   {
      /* image is not being requested by the report, don't barf, fudge! */
      GraphActivityDataScan (StartAbsVmsDay, StartHour, NumberOfHours,
                             MinuteGranularity, FindMinuteAverage,
                             &PeakRequests, &PeakBytes, NULL, NULL);
      MaxBytes = PeakBytes;
      MaxRequests = PeakRequests;
      GraphActivityMaxima (&MaxRequests, &MaxBytes);
   }

   /* ratio for calculating request bar graph height */
   if (MaxRequests)
      RequestFactor = (float)(GraphHeight-2) / (float)MaxRequests;
   else
      RequestFactor = 1.0;

   /* ratio for calculating byte bar graph height */
   if (MaxBytes)
      ByteFactor = (float)(GraphHeight-2) / (float)MaxBytes;
   else
      ByteFactor = 1.0;

   if (Debug)
      fprintf (stdout, "RequestFactor: %f ByteFactor: %f\n",
               RequestFactor, ByteFactor);

   /**************/
   /* plot graph */
   /**************/

   if (Debug) fprintf (stdout, "plot\n");

   if (GraphNew (rqptr, grptr, GraphWidth, GraphHeight, 7)) 
   {
      GraphActivityPlotEnd (rqptr);
      return;
   }

   StartMinute = StartAbsVmsDay * MINUTES_IN_DAY + StartHour * MINUTES_IN_HOUR;
   EndMinute = StartMinute + NumberOfHours * MINUTES_IN_HOUR - 1;
   CurrentMinute = CurrentAbsVmsDay * MINUTES_IN_DAY +
                   CurrentNumTime[3] * MINUTES_IN_HOUR +
                   CurrentNumTime[4];
   if (Debug)
      fprintf (stdout, "start: %d end: %d current: %d\n",
               StartMinute, EndMinute, CurrentMinute);

   ByteTotal = DisplayMean = RequestTotal = SampleCount = 0;
   AtX = 1;
   Minute = StartMinute;
   idx = GraphActivityDataIdx (StartAbsVmsDay, StartHour);

   while (Minute <= EndMinute)
   {
      if (idx >= ActivityTotalMinutes) idx = 0;
      if (Debug) fprintf (stdout, "%d idx: %d\n", Minute, idx);

      if (Minute < ActivityStartMinute)
      {
         GraphDrawBlock (grptr, AtX, 1, AtX+ColumnWidth+ColumnWidth-1,
                         GraphHeight-1, ColourNoData);

         PrevAtX = AtX;
         AtX += ColumnWidth + ColumnWidth;
         idx += MinuteGranularity;
         Minute += MinuteGranularity;

         continue;
      }

      if (Minute > CurrentMinute) break;
      Minute += MinuteGranularity;

      if (MinuteGranularity == 1)
      {
         ByteCount = ActivityBytesPtr[idx];
         RequestCount = ActivityRequestsPtr[idx];
         idx++;
      }
      else
      if (FindMinuteAverage)
      {
         /* find the average of the minute counts */
         ByteCount = RequestCount = 0;
         for (cnt = 0; cnt < MinuteGranularity; cnt++)
         {
            ByteCount += ActivityBytesPtr[idx];
            RequestCount += ActivityRequestsPtr[idx];
            idx++;
         }
         ByteCount /= MinuteGranularity;
         RequestCount /= MinuteGranularity;
      }
      else
      {
         /* find the peak of the minute counts */
         ByteCount = RequestCount = 0;
         for (cnt = 0; cnt < MinuteGranularity; cnt++)
         {
            if (ActivityBytesPtr[idx] > ByteCount)
               ByteCount = ActivityBytesPtr[idx];
            if (ActivityRequestsPtr[idx] > RequestCount)
               RequestCount = ActivityRequestsPtr[idx];
            idx++;
         }
      }

      RequestTotal += RequestCount;
      ByteTotal += ByteCount;

      /* calculate means */
      SampleCount++;
      RequestMean = (int)((float)(RequestTotal / SampleCount) * RequestFactor);
      ByteMean = (int)((float)(ByteTotal / SampleCount) * ByteFactor);

      /* limit the range (just in case) */
      if (RequestCount > MaxRequests) RequestCount = MaxRequests;
      if (ByteCount > MaxBytes) ByteCount = MaxBytes;

      RequestHeight = (int)((float)RequestCount * RequestFactor);
      ByteHeight = (int)((float)ByteCount * ByteFactor);

      if (Debug)
         fprintf (stdout, "%d %d %d %d %d %d\n",
                  RequestTotal, ByteTotal, RequestHeight, ByteHeight,
                  RequestMean, ByteMean);

      GraphDrawBlock (grptr, AtX, 1, AtX+ColumnWidth-1,
                      RequestHeight+1, ColourRequest);

      if (DisplayMean)
         GraphDrawLine (grptr, PrevAtX+ColumnWidth, PrevRequestMean,
                        AtX+ColumnWidth, RequestMean, ColourRequestMean, 1);
                  
      GraphDrawBlock (grptr, AtX+ColumnWidth, 1,
                      AtX+ColumnWidth+ColumnWidth-1,
                      ByteHeight+1, ColourByte);

      if (DisplayMean++)
         GraphDrawLine (grptr, PrevAtX+ColumnWidth+ColumnWidth, PrevByteMean,
                        AtX+ColumnWidth+ColumnWidth, ByteMean,
                        ColourByteMean, 1);

      PrevRequestMean = RequestMean;
      PrevByteMean = ByteMean;
      PrevAtX = AtX;
      AtX += ColumnWidth + ColumnWidth;
   }

   if (Minute >= CurrentMinute)
      GraphDrawBlock (grptr, AtX, 1, GraphWidth-1, GraphHeight-1, ColourNoData);

   /****************/
   /* finish graph */
   /****************/

   if (MaxBytes < 10)
      AxisBytes = MaxBytes;
   else
   if (MaxBytes < 100)
      AxisBytes = MaxBytes / 10;
   else
   if (MaxBytes < 1000)
      AxisBytes = MaxBytes / 100;
   else
   if (MaxBytes < 10000)
      AxisBytes = MaxBytes / 1000;
   else
   if (MaxBytes < 100000)
      AxisBytes = MaxBytes / 10000;
   else
   if (MaxBytes < 1000000)
      AxisBytes = MaxBytes / 100000;
   else
   if (MaxBytes < 10000000)
      AxisBytes = MaxBytes / 1000000;
   else
   if (MaxBytes < 100000000)
      AxisBytes = MaxBytes / 10000000;
   else
      AxisBytes = MaxBytes / 100000000;

   if (MaxRequests < 10)
      AxisRequests = MaxRequests;
   else
   if (MaxRequests < 100)
      AxisRequests = MaxRequests / 10;
   else
   if (MaxRequests < 1000)
      AxisRequests = MaxRequests / 100;
   else
   if (MaxRequests < 10000)
      AxisRequests = MaxRequests / 1000;
   else
   if (MaxRequests < 100000)
      AxisRequests = MaxRequests / 10000;
   else
   if (MaxRequests < 1000000)
      AxisRequests = MaxRequests / 100000;
   else
      AxisRequests = MaxRequests / 1000000;

   if (AxisRequests <= 2)
      GraphGraduateYAxis (grptr, GRAPH_YAXIS_LEFT,
                          AxisRequests*10, 4, ColourAxis);
   GraphGraduateYAxis (grptr, GRAPH_YAXIS_LEFT, AxisRequests*2, 7, ColourAxis);
   GraphGraduateYAxis (grptr, GRAPH_YAXIS_LEFT, AxisRequests, 10, ColourAxis);

   if (AxisBytes <= 2)
      GraphGraduateYAxis (grptr, GRAPH_YAXIS_RIGHT,
                          AxisBytes*10, 4, ColourAxis);
   GraphGraduateYAxis (grptr, GRAPH_YAXIS_RIGHT, AxisBytes*2, 7, ColourAxis);
   GraphGraduateYAxis (grptr, GRAPH_YAXIS_RIGHT, AxisBytes, 10, ColourAxis);

   if (NumberOfHours == 1)
   {
      GraphGraduateXAxis (grptr, GRAPH_XAXIS_BOTTOM, 60, 4, ColourAxis);
      GraphGraduateXAxis (grptr, GRAPH_XAXIS_BOTTOM, 12, 7, ColourAxis);
      GraphGraduateXAxis (grptr, GRAPH_XAXIS_BOTTOM, 2, 10, ColourAxis);
      GraphGraduateXAxis (grptr, GRAPH_XAXIS_TOP, 60, 4, ColourAxis);
      GraphGraduateXAxis (grptr, GRAPH_XAXIS_TOP, 12, 7, ColourAxis);
      GraphGraduateXAxis (grptr, GRAPH_XAXIS_TOP, 2, 10, ColourAxis);
   }
   else
   if (NumberOfHours < 24)
   {
      if (NumberOfHours <= 8)
      {
         GraphGraduateXAxis (grptr, GRAPH_XAXIS_BOTTOM,
                             NumberOfHours*12, 4, ColourAxis);
         GraphGraduateXAxis (grptr, GRAPH_XAXIS_TOP,
                             NumberOfHours*12, 4, ColourAxis);
      }
      GraphGraduateXAxis (grptr, GRAPH_XAXIS_BOTTOM,
                          NumberOfHours*2, 7, ColourAxis);
      GraphGraduateXAxis (grptr, GRAPH_XAXIS_BOTTOM,
                          NumberOfHours, 10, ColourAxis);
      GraphGraduateXAxis (grptr, GRAPH_XAXIS_TOP,
                          NumberOfHours*2, 7, ColourAxis);
      GraphGraduateXAxis (grptr, GRAPH_XAXIS_TOP,
                          NumberOfHours, 10, ColourAxis);
   }
   else
   {
      NumberOfDays = NumberOfHours / 24;
      if (NumberOfDays <= 3)
      {
         GraphGraduateXAxis (grptr, GRAPH_XAXIS_BOTTOM,
                             NumberOfDays*24, 4, ColourAxis);
         GraphGraduateXAxis (grptr, GRAPH_XAXIS_TOP,
                             NumberOfDays*24, 4, ColourAxis);
      }
      GraphGraduateXAxis (grptr, GRAPH_XAXIS_BOTTOM,
                          NumberOfDays*4, 6, ColourAxis);
      GraphGraduateXAxis (grptr, GRAPH_XAXIS_BOTTOM,
                          NumberOfDays, 8, ColourAxis);
      GraphGraduateXAxis (grptr, GRAPH_XAXIS_TOP,
                          NumberOfDays*4, 6, ColourAxis);
      GraphGraduateXAxis (grptr, GRAPH_XAXIS_TOP,
                          NumberOfDays, 8, ColourAxis);
   }

   GraphDrawBorder (grptr, ColourAxis, 1);

   GraphActivityGifStream (rqptr);
}

/*****************************************************************************/
/*
*/

int GraphActivityPlotEnd (struct RequestStruct *rqptr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphActivityPlotEnd()\n");

   SysDclAst (rqptr->GraphTaskPtr->NextTaskFunction, rqptr);
}

/*****************************************************************************/
/*
Convert the image pointed to by grptr->GraphPlotPtr of grptr->PixelCount size
into a GIF image and send it to the client.
*/ 

int GraphActivityGifStream (struct RequestStruct *rqptr)

{
   register struct GraphStruct  *grptr;

   char  *KeepAlivePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphActivityGifStream()\n");

   /* get the pointer to this request's graph structure */
   grptr = rqptr->GraphTaskPtr->GraphPtr;

   /* we can usually count on heaps of compression using LZW :^) */
   grptr->GifBufferLength = (grptr->PixelCount / GIF_EXPECTED_COMPRESSION) +
                            GIF_OVERHEAD;

   /* allocate heap memory for the GIF image */
   grptr->GifBufferPtr =
      VmGetHeap (rqptr, grptr->GifBufferLength);

   /* set marker so we don't over run the end of the buffer */
   grptr->GifBufferEndPtr = grptr->GifBufferPtr + grptr->GifBufferLength;

   if (GraphGifImage (grptr))
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, grptr->ErrorTextPtr, FI_LI);
      return;
   }

   /*******************/
   /* write to client */
   /*******************/

   if (Config.KeepAliveTimeoutSeconds && rqptr->KeepAliveRequest)
   {
      rqptr->KeepAliveResponse = true;
      rqptr->KeepAliveCount++;
      KeepAlivePtr = KeepAliveHttpHeader;
   }
   else
      KeepAlivePtr = "";

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = true;
   if ((rqptr->ResponseHeaderPtr =
       HttpHeader (rqptr, 200, "image/gif", grptr->GifContentLength,
                   rqptr->BinaryTime, KeepAlivePtr))
       == NULL)
   {
      GraphActivityPlotEnd (rqptr);
      return;
   }

   NetWrite (rqptr, GraphActivityPlotEnd,
             grptr->GifBufferPtr, grptr->GifContentLength);
}

/*****************************************************************************/
/*
Allocate memory for a graphic of the specified size, then set all the pixels
to the specified colour.
*/

char* GraphNew
(
struct RequestStruct *rqptr,
struct GraphStruct *grptr,
int Width,
int Height,
int Colour
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "GraphNew() %dx%d %d\n", Width, Height, Colour);

   if (Width <= 0 || Width >= GraphicMaximumWidth)
      return (grptr->ErrorTextPtr = "Graph: width problem.");
   if (Height <= 0 || Height >= GraphicMaximumHeight)
      return (grptr->ErrorTextPtr = "Graph: height problem.");

   /* some initializations */
   grptr->PreExpired = true;
   grptr->BitsPerPixel = GraphicBitsPerPixel;
   grptr->GifColourBg = grptr->ColourBg = Colour;
   grptr->ColourFg = 0;
   grptr->ColourTr = -1;
   grptr->Width = Width;
   grptr->Height = Height;
   grptr->PixelCount = grptr->Width * grptr->Height;
   if (GraphDebug) fprintf (stdout, "PixelCount: %d\n", grptr->PixelCount);

   /* allocate heap memory for the graph structure */
   grptr->PlotBufferPtr = VmGetHeap (rqptr, grptr->PixelCount);

   /* set all the pixels to background colour */
   memset (grptr->PlotBufferPtr, Colour, grptr->PixelCount);

   return (0);
}

/*****************************************************************************/
/*
Set all the pixels in the graphic to the specified colour.
*/

char* GraphSetBackground
(
struct GraphStruct *grptr,
int Colour
)
{
   /*********/
   /* begin */
   /*********/

   if (GraphDebug) fprintf (stdout, "GraphSetBackground() %d\n", Colour);

   memset (grptr->PlotBufferPtr, Colour, grptr->PixelCount);
   return (0);
}

/*****************************************************************************/
/*
The basic point drawing/plotting function.  Draws a point at (x,y) in the
specified colour.  Width specifies the size of the point, 1 being one pixel,
2 being 2x2 pixels, 3 being 3x3 pixels, etc.  Points with widths greater than
1 begin at the specified (x,y) and grow in the x and y directions.
*/

char* GraphDrawPoint
(
struct GraphStruct *grptr,
int atx,
int aty,
int Colour,
int Width
)
{
   /*********/
   /* begin */
   /*********/

   if (GraphDebug) fprintf (stdout, "GraphDrawPoint() %d,%d\n", atx, aty);

   if (Width <= 0 || Width >= grptr->Width)
      return (grptr->ErrorTextPtr = "Graph: point width problem.");
   if (atx < 0 || atx >= grptr->Width || aty < 0 || aty >= grptr->Height)
      return (grptr->ErrorTextPtr = "Graph: point X,Y problem.");

   if (Width == 1)
   {
      *(grptr->PlotBufferPtr + (aty * grptr->Width) + atx) = Colour;
      return (0);
   }
   else
   {
      /* local storage */
      register int  x, y, tox, toy;

      if ((tox = atx + Width) >= grptr->Width) tox = grptr->Width - 1;
      if ((toy = aty + Width) >= grptr->Height) tox = grptr->Height - 1;
      for (x = atx; x < tox; x++)
         for (y = aty; y < toy; y++)
            *(grptr->PlotBufferPtr + (y * grptr->Width) + x) = Colour;
      return (0);
   }
}

/*****************************************************************************/
/*
Draws a straight line from (x1,y1) to (x2,y2) of the specified colour and
width. For efficiency plots its own points without calling GraphDrawPoint().
*/

char* GraphDrawLine
(
struct GraphStruct *grptr,
int x1,
int y1,
int x2,
int y2,
int Colour,
int Width
)
{
   register boolean  SlopePos;
   register int  thy, x, xinc, y, yinc;
   register float  Slope, yf;

   /*********/
   /* begin */
   /*********/

   if (GraphDebug)
      fprintf (stdout, "GraphDrawLine() %d,%d %d,%d %d %d\n",
               x1, y1, x2, y2, Colour,
      Width);

   if (Width <= 0 || Width >= grptr->Width)
      return (grptr->ErrorTextPtr = "Graph: line width problem.");
   if (x1 < 0 || x1 >= grptr->Width || y1 < 0 || y1 >= grptr->Height)
      return (grptr->ErrorTextPtr = "Graph: line X1,Y1 problem.");
   if (x2 < 0 || x2 >= grptr->Width || y2 < 0 || y2 >= grptr->Height)
      return (grptr->ErrorTextPtr = "Graph: line X2,Y2 problem.");

   if (x1 > x2)
   {
      x = x1;
      x1 = x2;
      x2 = x;
      y = y1;
      y1 = y2;
      y2 = y;
   }

   if (x2 - x1)
   {
      xinc = 1;
      Slope = (float)(y2 -y1) / (float)(x2 - x1);
   }
   else
   {
      xinc = 0;
      if (y2 > y1)
         Slope = 1.0;
      else
         Slope = -1.0;
   }
   if (Slope >= 0.0)
   {
      SlopePos = true;
     yinc = 1;
   }
   else
   {
      SlopePos = false;
      yinc = -1;
   }

   yf = (float)(y = thy = y1);
   x = x1;
   while (x <= x2 && ((SlopePos && y <= y2) || (!SlopePos && y >= y2)))
   {
      if (GraphDebug) fprintf (stdout, "%d,%d (%f %f)\n", x, y, yf, Slope);

      while ((SlopePos && thy <= y) || (!SlopePos && thy >= y))
      {
         /* plot the point */
         if (Width == 1)
            *(grptr->PlotBufferPtr + (thy * grptr->Width) + x) = Colour;
         else
         {
            /* local storage */
            register int  xx, yy, tox, toy;

            if ((tox = x + Width) >= grptr->Width) tox = grptr->Width - 1;
            if ((toy = thy + Width) >= grptr->Height) toy = grptr->Height - 1;
            for (xx = x; xx < tox; xx++)
               for (yy = thy; yy < toy; yy++)
                  *(grptr->PlotBufferPtr + (yy * grptr->Width) + xx) = Colour;
         }
         thy += yinc;
      }

      thy -= yinc;
      x += xinc;
      y = (int)(yf = yf + Slope);
   }

   return (0);
}

/*****************************************************************************/
/*
Draw a rectangle with opposite corners defined by (x1,y1) and (x2,y2) of the
specified colour and width.
*/

char* GraphDrawRect
(
struct GraphStruct *grptr,
int x1,
int y1,
int x2,
int y2,
int Colour,
int Width
)
{
   register int  x, y;

   /*********/
   /* begin */
   /*********/

   if (GraphDebug)
      fprintf (stdout, "GraphDrawRect() %d,%d %d,%d %d\n",
               x1, y1, x2, y2, Colour);

   if (Width <= 0 || Width >= grptr->Width)
      return (grptr->ErrorTextPtr = "Graph: rectangle width problem.");
   if (x1 < 0 || x1 >= grptr->Width || y1 < 0 || y1 >= grptr->Height)
      return (grptr->ErrorTextPtr = "Graph: rectangle X1,Y1 problem.");
   if (x2 < 0 || x2 >= grptr->Width || y2 < 0 || y2 >= grptr->Height)
      return (grptr->ErrorTextPtr = "Graph: rectangle X2,Y2 problem.");

   GraphDrawLine (grptr, x1, y1, x1, y2, Colour, Width);
   GraphDrawLine (grptr, x1, y2, x2, y2, Colour, Width);
   GraphDrawLine (grptr, x2, y2, x2, y1, Colour, Width);
   GraphDrawLine (grptr, x2, y1, x1, y1, Colour, Width);

   return (0);
}

/*****************************************************************************/
/*
Draw a rectangle with opposite corners defined by (x1,y1) and (x2,y2) filled
with the specified colour.
*/

char* GraphDrawBlock
(
struct GraphStruct *grptr,
int x1,
int y1,
int x2,
int y2,
int Colour
)
{
   register int  x, y;

   /*********/
   /* begin */
   /*********/

   if (GraphDebug)
      fprintf (stdout, "GraphDrawBlock() %d,%d %d,%d %d\n",
               x1, y1, x2, y2, Colour);

   if (x1 < 0 || x1 >= grptr->Width || y1 < 0 || y1 >= grptr->Height)
      return (grptr->ErrorTextPtr = "Graph: block X1,Y1 problem.");
   if (x2 < 0 || x2 >= grptr->Width || y2 < 0 || y2 >= grptr->Height)
      return (grptr->ErrorTextPtr = "Graph: block X2,Y2 problem.");

   if (x1 > x2)
   {
      x = x1;
      x1 = x2;
      x2 = x;
   }
   if (y1 > y2)
   {
      y = y1;
      y1 = y2;
      y2 = y;
   }

   for (x = x1; x <= x2; x++)
      GraphDrawLine (grptr, x, y1, x, y2, Colour, 1);

   return (0);
}

/*****************************************************************************/
/*
Draw graduations along either the top or bottom X axis.
*/

char* GraphGraduateXAxis
(
struct GraphStruct *grptr,
int TopOrBottom,
int Graduations,
int Length,
int Colour
)
{
   register int  x, FromY, ToY;
   register float  gr, xf;

   /*********/
   /* begin */
   /*********/

   if (GraphDebug)
      fprintf (stdout, "GraphGraduateXAxis() %d %d %d\n",
               TopOrBottom, Graduations, Colour);

   if (TopOrBottom == GRAPH_XAXIS_BOTTOM)
   {
      FromY = 0;
      ToY = Length - 1;
   }
   else
   {
      FromY = grptr->Height - Length;
      ToY = grptr->Height - 1;
   }
   xf = 0.0;
   gr = (float)grptr->Width / ((float)Graduations + 0.001);
   for (x = 0; x < grptr->Width; x = (int)(xf += gr))   
      GraphDrawLine (grptr, x, FromY, x, ToY, Colour, 1); 

   return (0);
}

/*****************************************************************************/
/*
Draw graduations along either the left or right Y axis.
*/

char* GraphGraduateYAxis
(
struct GraphStruct *grptr,
int LeftOrRight,
int Graduations,
int Length,
int Colour
)
{
   register int  y, FromX, ToX;
   register float  gr, yf;

   /*********/
   /* begin */
   /*********/

   if (GraphDebug)
      fprintf (stdout, "GraphGraduateYAxis() %d %d %d\n",
               LeftOrRight, Graduations, Colour);

   if (LeftOrRight == GRAPH_YAXIS_LEFT)
   {
      FromX = 0;
      ToX = Length - 1;
   }
   else
   {
      FromX = grptr->Width - Length;
      ToX = grptr->Width - 1;
   }
   yf = 0.0;
   gr = (float)grptr->Height / (float)Graduations;
   for (y = 0; y < grptr->Height; y = (int)(yf += gr))   
      GraphDrawLine (grptr, FromX, y, ToX, y, Colour, 1); 

   return (0);
}

/*****************************************************************************/
/*
Place a border of the specified colour and width around the entire graphic.
*/

char* GraphDrawBorder
(
struct GraphStruct *grptr,
int Colour,
int Width
)
{
   register int  x, y, cnt;
   register unsigned char  *bptr;

   /*********/
   /* begin */
   /*********/

   if (GraphDebug) fprintf (stdout, "GraphDrawBorder() %d %d\n", Colour, Width);

   if (Width <= 0 || Width >= grptr->Width || Width >= grptr->Height)
      return (grptr->ErrorTextPtr = "Graph: border width problem.");

   for (cnt = 0; cnt < Width; cnt++)
   {
      bptr = grptr->PlotBufferPtr + (cnt * grptr->Width);
      for (x = 0; x < grptr->Width; x++) *bptr++ = Colour;
   }
   for (cnt = Width; cnt; cnt--)
   {
      bptr = grptr->PlotBufferPtr + (grptr->Height - cnt) * grptr->Width;
      for (x = 0; x < grptr->Width; x++) *bptr++ = Colour;
   }
   for (cnt = 0; cnt < Width; cnt++)
   {
      bptr = grptr->PlotBufferPtr + cnt;
      for (y = 0; y < grptr->Height; y++)
        *(bptr + (y * grptr->Width)) = Colour;
   }
   for (cnt = Width; cnt; cnt--)
   {
      bptr = grptr->PlotBufferPtr + grptr->Width - cnt;
      for (y = 0; y < grptr->Height; y++)
        *(bptr + (y * grptr->Width)) = Colour;
   }

   return (0);
}

/*****************************************************************************/
/*
Convert a pixmap image to a GIF image.

The code in these functions is implemented in accordance with Compuserve's 
Graphic Interchange Format Programming Reference specification, version 89a, 
31st July 1990. 

The LZW compression employed by the GIF algorithm is implemented using code 
derived from the PBM suite.  Two functions, slightly modified and definitely
reformatted, are employed, GraphCompress() ... formally called 'compgif()',
and GraphPackBits() ...  formally called 'pack_bits()'.  The original
commentary and copyright notice remains as per the code author's request.
*/ 

char* GraphGifImage (struct GraphStruct *grptr)

{
   register unsigned char  *gifptr;
   register int  idx,
                 Byte;

   int  LeftOffset,
        TopOffset,
        Resolution,
        ColorMapSize,
        InitCodeSize;
   char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (GraphDebug) fprintf (stdout, "GraphGifImage()\n");

   grptr->GifContentLength = 0;

   /* initialize non-reentrant functions() */
   GraphPackBits (NULL);
   GraphNextPixel (NULL);

   ColorMapSize = 1 << grptr->BitsPerPixel;
   LeftOffset = TopOffset = 0;
   Resolution = grptr->BitsPerPixel;

   /* the initial code size */
   if (grptr->BitsPerPixel <= 1)
      InitCodeSize = 2;
   else
      InitCodeSize = grptr->BitsPerPixel;

   /**************************/
   /* GIF Data Stream header */
   /**************************/

   gifptr = grptr->GifBufferPtr;

   memcpy (gifptr, "GIF89a", 6);
   gifptr += 6;

   /*****************************/
   /* Logical Screen Descriptor */
   /*****************************/

   /* width and height of logical screen */
   *gifptr++ = grptr->Width & 0xff;
   *gifptr++ = (grptr->Width >> 8) & 0xff;
   *gifptr++ = grptr->Height & 0xff;
   *gifptr++ = (grptr->Height >> 8) & 0xff;

   /* indicate that there is a global colour map */
   Byte = 0x80;
   /* OR in the resolution */
   Byte |= (Resolution - 1) << 5;
   /* OR in the Bits per Pixel */
   Byte |= (grptr->BitsPerPixel - 1);
   /* buffer it */
   *gifptr++ = Byte;

   /* background colour */
   *gifptr++ = grptr->GifColourBg;

   /* pixel aspect ratio */
   *gifptr++ = 0;

   /***********************/
   /* Global Colour Table */
   /***********************/

   for (idx = 0; idx < ColorMapSize; idx++)
   {
      *gifptr++ = GraphicRgbRed[idx];
      *gifptr++ = GraphicRgbGreen[idx];
      *gifptr++ = GraphicRgbBlue[idx];
   }

   /****************************************/
   /* Graphic Control Extension descriptor */
   /****************************************/

   if (grptr->ColourTr >= 0)
   {
      /* extension introducer and graphic control label */
      *gifptr++ = 0x21;
      *gifptr++ = 0xf9;
      /* fixed size of the following data block */
      *gifptr++ = 0x04;
      /* Transparency Index is provided */
      *gifptr++ = 0x01;
      /* no data in these */
      *gifptr++ = 0x00;
      *gifptr++ = 0x00;
      /* Transparent Color Index value, background SHOULD BE TRANSPARENT */
      *gifptr++ = grptr->ColourTr;
      /* block terminator */
      *gifptr++ = 0x00;
   }

   /********************/
   /* image descriptor */
   /********************/

   /* write an image separator */
   *gifptr++ = 0x2c;

   /* location of image within logical screen */
   *gifptr++ = LeftOffset & 0xff;
   *gifptr++ = (LeftOffset >> 8) & 0xff;
   *gifptr++ = TopOffset & 0xff;
   *gifptr++ = (TopOffset >> 8) & 0xff;

   /* width and height of image within logical screen */
   *gifptr++ = grptr->Width & 0xff;
   *gifptr++ = (grptr->Width >> 8) & 0xff;
   *gifptr++ = grptr->Height & 0xff;
   *gifptr++ = (grptr->Height >> 8) & 0xff;

   /* no local color table, image is not interlaced, not ordered, etc. */
   *gifptr++ = 0x00;

   /**************************/
   /* table-based image data */
   /**************************/

   /* write out the initial code size */
   *gifptr++ = InitCodeSize;

   /* LZW compress the data using PBM-derived algorithm and code */
   if (GraphDebug) fprintf (stdout, "%d bytes\n", gifptr - grptr->GifBufferPtr);
   grptr->GifPtr = gifptr;
   if (cptr = GraphCompress (grptr, InitCodeSize + 1))
      return (cptr);
   if (GraphDebug)
      fprintf (stdout, "%d bytes\n", grptr->GifPtr - grptr->GifBufferPtr);

   /****************************************/
   /* end of image data and GIF terminator */
   /****************************************/

   /* write out a zero-length packet (to end the series), and terminator */
   if (grptr->GifPtr < grptr->GifBufferEndPtr) *grptr->GifPtr++ = 0;
   if (grptr->GifPtr < grptr->GifBufferEndPtr) *grptr->GifPtr++ = ';';

   if (grptr->GifPtr >= grptr->GifBufferEndPtr)
      return("GIF: image buffer too small.");

   grptr->GifContentLength = grptr->GifPtr - grptr->GifBufferPtr;

   return (0);
}

/*****************************************************************************/
/*
Get a series of pixels from the bitmap.  It is done in such a way that the
origin (0,0) is in the bottom, left hand corner, although the memory origin is
at byte zero.
*/ 

int GraphNextPixel (struct GraphStruct *grptr)

{
   static int  CountX,
               CountY;
   static unsigned char  *PixelPtr;

   /*********/
   /* begin */
   /*********/

   /* if (GraphDebug) fprintf (stdout, "GraphNextPixel()\n"); */

   if (grptr == NULL)
   {
      /* initialize */
      CountX = 0;
      CountY = -1;
      return (0);
   }

   if (!CountX--)
   {
      if (CountY < 0) CountY = grptr->Height;
      if (!CountY--) return (EOF);
      CountX = grptr->Width - 1;
      PixelPtr = grptr->PlotBufferPtr + (CountY * grptr->Width);
   }

   return (*PixelPtr++);
}

/****************************************************************************/
/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */

/*  compgif.c */
/*
 *
 * GIF Image compression - LZW algorithm implemented with Tree type
 *                         structure.
 *                         Written by Bailey Brown, Jr.
 *                         last change May 24, 1990
 *                         file: compgif.c
 *
 *  You may use or modify this code as you wish, as long as you mention
 *  my name in your documentation.
 *
 *                  - Bailey Brown, Jr.
 *
 */

#define MAXIMUMCODE 4095   /* 2**maximum_code_size */
#define BLOCKSIZE 256   /* max block byte count + 1 */
#define NULLPREFIX -1

typedef struct str_table_entry
{
   int code;
   int prefix;
   int suffix;
}  strTableEntry;

typedef struct str_table_node
{
   strTableEntry entry;
   struct str_table_node *left;
   struct str_table_node *right;
   struct str_table_node *children;
} strTableNode, *strTableNodePtr, **strTable;

/*
 ********************************************************************
 * compgif() recieves pointers to an input function and an output   *
 * stream, and the code size as parameters and outputs successive   *
 * blocks of LZW compressed gif data.  The calling routine should   *
 * have aready written the GIF file header out to the output file.  *
 * It assumes that there will be no more than 8 bits/pixel and that *
 * each data item comes from successive bytes returned by infun.    *
 ********************************************************************
 */

char* GraphCompress
(
struct GraphStruct *grptr,
int code_size
)
{
   strTable heap; /* our very own memory manager */
   int heap_index;
   int clear_code, end_code, cur_code;
   int i, found, num_colors, prefix, compress_size;
   int cur_char, end_of_data, bits_per_pix;
   strTableNodePtr cur_node;
   strTable root;  /* root of string table for LZW compression is */
                   /* an array of 2**bits_per_pix pointers to atomic nodes */

   /*********/
   /* begin */
   /*********/

   if (GraphDebug) fprintf (stdout, "GraphCompress()\n");

   heap_index = 0;
   heap = (strTable)VmGet(sizeof(strTableNodePtr)*MAXIMUMCODE);
   if (heap == NULL) return("GIF: VmGet() failure, heap");

   for (i=0; i < MAXIMUMCODE; i++)
   {
      heap[i] = (strTableNodePtr)VmGet(sizeof(strTableNode));
      if (heap[i] == NULL) return("GIF: VmGet() failure, heap[idx]");
   }

   bits_per_pix = code_size - 1;
   compress_size = code_size;
   num_colors = 1<<(bits_per_pix);
   clear_code = num_colors;
   end_code = clear_code + 1;
   cur_code = end_code + 1;
   prefix = NULLPREFIX;
   root = (strTable)VmGet(sizeof(strTableNodePtr)*num_colors);
   if (!root) return("GIF: VmGet() failure, root");

   for(i=0; i<num_colors; i++)
   {
      root[i] = heap[heap_index++];
      root[i]->entry.code = i;
      root[i]->entry.prefix = NULLPREFIX;
      root[i]->entry.suffix = i;
      root[i]->left = NULL;
      root[i]->right = NULL;
      root[i]->children = NULL;
   }

   /* initialize  output block */
   GraphPackBits(grptr, compress_size, -1);
   GraphPackBits(grptr, compress_size, clear_code);
   end_of_data = 0;

   if ((cur_char = GraphNextPixel(grptr)) == EOF)
      return("GIF: premature end of data.");

   while (!end_of_data)
   {
      prefix = cur_char;
      cur_node = root[prefix];
      found = 1;
      if((cur_char = GraphNextPixel(grptr)) == EOF)
      {
         end_of_data = 1;
         break;
      }

      while(cur_node->children && found)
      {
         cur_node = cur_node->children;
         while(cur_node->entry.suffix != cur_char)
         {
            if (cur_char < cur_node->entry.suffix)
            {
               if (cur_node->left)
                  cur_node = cur_node->left;
               else
               {
                  cur_node->left = heap[heap_index++];
                  cur_node = cur_node->left;
                  found = 0; break;
               }
            }
            else
            {
               if (cur_node->right)
                  cur_node = cur_node->right;
               else
               {
                  cur_node->right = heap[heap_index++];
                  cur_node = cur_node->right;
                  found = 0; break;
               }
            }
         }
         if (found)
         {
            prefix = cur_node->entry.code;
            if((cur_char = GraphNextPixel(grptr)) == EOF)
            {
               end_of_data = 1;
               break;
            }
         }
      }

      if (end_of_data) break;

      if (found)
      {
         cur_node->children = heap[heap_index++];
         cur_node = cur_node->children;
      }

      cur_node->children = NULL;
      cur_node->left = NULL;
      cur_node->right = NULL;
      cur_node->entry.code = cur_code;
      cur_node->entry.prefix = prefix;
      cur_node->entry.suffix = cur_char;
      GraphPackBits(grptr, compress_size, prefix);

      if (cur_code > ((1<<(compress_size))-1))
         compress_size++;
      if (cur_code < MAXIMUMCODE)
         cur_code++;
      else
      {
         heap_index = num_colors;  /* reinitialize string table */
         for (i=0; i < num_colors; i++ ) root[i]->children = NULL;
         GraphPackBits(grptr, compress_size, clear_code);
         compress_size = bits_per_pix + 1;
         cur_code = end_code + 1;
      }
   }

   GraphPackBits(grptr, compress_size, prefix);
   GraphPackBits(grptr, compress_size, end_code);
   GraphPackBits(grptr, compress_size, -1);

   for (i=0; i < MAXIMUMCODE; i++) VmFree(heap[i]);
   VmFree(heap);
   VmFree(root);

   return (0);
}

/****************************************************************************/
/*
 ************************************************************************
 * pack_bits() packs the bits of the codes generated by gifenc() into   *
 * a 1..256 byte output block.  The first byte of the block is the      *
 * number 0..255 of data bytes in the block.  To flush or initialize    *
 * the block, pass a negative argument.                                 *
 ************************************************************************
 */

int GraphPackBits
(
struct GraphStruct *grptr,
int compress_size,
int prefix
)
{
   static int cur_bit = 8;
   static unsigned char block[BLOCKSIZE] = { 0 };
   int i, left_over_bits;

   /*********/
   /* begin */
   /*********/

/*
   if (GraphDebug)
          fprintf (stdout, "GraphPackBits() %d %d %d\n",
                   grptr, compress_size, prefix);
*/

   if (grptr == NULL)
   {
      /* initialize */
      cur_bit = 8;
      block[0] = 0;
      return(1);
   }

   /* if we are about to excede the bounds of block or if the flush
      code (code_bis < 0) we output the block */

   if((cur_bit + compress_size > (BLOCKSIZE-1)*8) || (prefix < 0))
   {
       /* handle case of data overlapping blocks */
       if ((left_over_bits = (((cur_bit>>3) +
               ((cur_bit & 7) != 0))<<3) - cur_bit) != 0)
       {
           for (i=0; i < left_over_bits; i++)
           {
               if (prefix & (1<<i))
                  block[cur_bit>>3] |= (char)(1<<(cur_bit & 7));
               /* note n>>3 == n/8 and n & 7 == n % 8 */
               cur_bit++;
           }
       }

       compress_size -= left_over_bits;
       prefix = prefix>>left_over_bits;
       block[0] = (unsigned char)((cur_bit>>3) - 1);

       if (block[0]) 
       {
          /** if (GraphDebug) fprintf (stdout, "%d\n", block[0]+1); **/
          if (grptr->GifPtr + (block[0]+1) >= grptr->GifBufferEndPtr)
             grptr->GifPtr = grptr->GifBufferEndPtr;
          else
          {
             memcpy (grptr->GifPtr, block, block[0]+1);
             grptr->GifPtr = grptr->GifPtr + block[0]+1;
          }
       }

       memset (block, 0, BLOCKSIZE);
       cur_bit = 8;
   }

   if (prefix >= 0)
   {
       for (i=0; i < compress_size; i++)
       {
          if (prefix & (1<<i))
          block[cur_bit>>3] |= (unsigned char)(1<<(cur_bit & 7));
          /* note n>>3 == n/8 and n & 7 == n % 8 */
          cur_bit++;
       }
   }

   return (1);
}

/*****************************************************************************/
