/*****************************************************************************
/*
                                 HTAdmin.c

Administer the user authentication (HTA) Databases.


VERSION HISTORY
---------------
16-JUL-98  MGD  "https:" only flag,
                extend HTAdminPasswordChange() to VMS (SYSUAF)
09-AUG-97  MGD  message database
01-FEB-97  MGD  new for HTTPd version 4
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <time.h>

/* VMS related header files */
#include <descrip.h>
#include <jpidef.h>
#include <libdef.h>
#include <libdtdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application related header files */
#include "wasd.h"
#include "auth.h"
#include "error.h"
#include "HTAdmin.h"
#include "mapurl.h"
#include "msg.h"
#include "net.h"
#include "support.h"
#include "uaidef.h"
#include "vm.h"

/***********************/
/* module requirements */
/***********************/

#define DatabaseListSize 8
#define UserNameListSize 7

/******************/
/* global storage */
/******************/

char  ErrorHTAdminAction [] = "Unknown action.",
      ErrorHTAdminDatabase [] = "Authentication database.",
      ErrorHTAdminDatabaseEnter [] = "Enter a database name.",
      ErrorHTAdminDatabaseExists [] = "Database already exists.",
      ErrorHTAdminDatabaseNBG [] =
"Database name may contain only A..Z, 0..9, _ and - characters.",
      ErrorHTAdminDatabaseSelect[] = "Select a database.",
      ErrorHTAdminInsufficient [] = "Insufficient parameters.",
      ErrorHTAdminParameter [] = "Parameter out-of-range.",
      ErrorHTAdminPurgeCache [] = "purging authentication cache",
      ErrorHTAdminQuery [] = "Unknown query component.",
      ErrorHTAdminUserNotFound [] = "Username not found in database.",
      ErrorHTAdminUserEnter [] = "Enter a username.",
      ErrorHTAdminUserExists [] = "Username already exists.",
      ErrorHTAdminUserNBG [] =
"Username may contain only A..Z, 0..9, _ and - characters.",
      ErrorHTAdminUserSelect [] = "Select a username.",
      ErrorHTAdminVerify [] = "Password verification failure.";

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern boolean  AuthSysUafEnabled;
extern boolean  AuthSysUafIdentifier;
extern char  *DayName[];
extern char  HtmlSgmlDoctype[];
extern char  ServerHostPort[];
extern char  SoftwareID[];
extern char  Utility[];
extern struct AccountingStruct Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;
extern struct AuthCacheRecordStruct  *AuthCacheTreeHead;

/*****************************************************************************/
/*
*/ 

int HTAdminBegin
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   static $DESCRIPTOR (LocationDatabaseFaoDsc, "!AZ!AZ?do=!AZ\0");

   register char  *cptr, *sptr, *qptr, *zptr;
   register struct HTAdminTaskStruct  *tkptr;

   boolean  ForceUpperCase;
   int  status;
   unsigned long  FaoVector [32];
   unsigned short  Length;
   char  Access [32],
         Action [32],
         AsDatabaseName [MaxAuthRealmGroupNameLength+1],
         AsUserName [MaxAuthUserNameLength+1],
         Contact [MaxAuthContactLength+1],
         DatabaseName [MaxAuthRealmGroupNameLength+1],
         Email [MaxAuthEmailLength+1],
         Enabled [32],
         FieldName [32],
         FieldValue [256],
         FullName [MaxAuthFullNameLength+1],
         HttpsOnly [32],
         Location [512],
         PasswordNew [MaxAuthPasswordLength+1],
         PasswordCurrent [MaxAuthPasswordLength+1],
         PasswordVerify [MaxAuthPasswordLength+1],
         UserName [MaxAuthUserNameLength+1];
   $DESCRIPTOR (LocationDsc, Location);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminBegin()\n");

   if (!rqptr->AccountingDone)
      rqptr->AccountingDone = ++Accounting.DoServerAdminCount;

   if (rqptr->ErrorMessagePtr != NULL)
   {
      /* previous error, cause threaded processing to unravel */
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /* set up the task structure (only ever one per request!) */
   rqptr->HTAdminTaskPtr = tkptr = (struct HTAdminTaskStruct*)
      VmGetHeap (rqptr, sizeof(struct HTAdminTaskStruct));
   tkptr->NextTaskFunction = NextTaskFunction;

   tkptr->ParseInUse = false;

   /**********************/
   /* parse query string */
   /**********************/

   Action[0] = Access[0] = AsDatabaseName[0] = DatabaseName[0] =
      AsUserName[0] = Contact[0] = Email[0] = Enabled[0] = HttpsOnly[0] =
      PasswordCurrent[0] = PasswordNew[0] = PasswordVerify[0] =
      UserName[0] = '\0';

   if (rqptr->HttpMethod == HTTP_METHOD_POST)
      qptr = rqptr->ContentBufferPtr;
   else
      qptr = rqptr->QueryStringPtr;
   while (*qptr)
   {
      qptr = ParseQueryField (rqptr, qptr,
                              FieldName, sizeof(FieldName),
                              FieldValue, sizeof(FieldValue),
                              FI_LI);
      if (qptr == NULL)
      {
         /* error occured */
         HTAdminEnd (rqptr);
         return;
      }

      /********************/
      /* get field values */
      /********************/

      ForceUpperCase = false;
      sptr = NULL;
      if (strsame (FieldName, "ac", -1))
         zptr = (sptr = Access) + sizeof(Access);
      else
      if (strsame (FieldName, "ad", -1))
         zptr = (sptr = AsDatabaseName) + sizeof(AsDatabaseName);
      else
      if (ForceUpperCase = strsame (FieldName, "au", -1))
         zptr = (sptr = AsUserName) + sizeof(AsUserName);
      else
      if (strsame (FieldName, "co", -1))
         zptr = (sptr = Contact) + sizeof(Contact);
      else
      if (ForceUpperCase = strsame (FieldName, "cu", -1))
         zptr = (sptr = PasswordCurrent) + sizeof(PasswordCurrent);
      else
      if (strsame (FieldName, "da", -1))
         zptr = (sptr = DatabaseName) + sizeof(DatabaseName);
      else
      if (strsame (FieldName, "do", -1))
         zptr = (sptr = Action) + sizeof(Action);
      else
      if (strsame (FieldName, "em", -1))
         zptr = (sptr = Email) + sizeof(Email);
      else
      if (strsame (FieldName, "en", -1))
         zptr = (sptr = Enabled) + sizeof(Enabled);
      else
      if (strsame (FieldName, "f", -1))
         zptr = (sptr = FullName) + sizeof(FullName);
      else
      if (strsame (FieldName, "hs", -1))
         zptr = (sptr = HttpsOnly) + sizeof(HttpsOnly);
      else
      if (ForceUpperCase = strsame (FieldName, "p", -1))
         zptr = (sptr = PasswordNew) + sizeof(PasswordNew);
      else
      if (ForceUpperCase = strsame (FieldName, "v", -1))
         zptr = (sptr = PasswordVerify) + sizeof(PasswordVerify);
      else
      if (ForceUpperCase = strsame (FieldName, "u", -1))
         zptr = (sptr = UserName) + sizeof(UserName);

      if (sptr == NULL)
      {
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, ErrorHTAdminQuery, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
      else
      {
         cptr = FieldValue;
         if (ForceUpperCase)
            while (*cptr && sptr < zptr) *sptr++ = toupper(*cptr++);
         else
            while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         if (sptr >= zptr)
         {
            ErrorGeneralOverflow (rqptr, FI_LI);
            HTAdminEnd (rqptr);
            return;
         }
         *sptr = '\0';
      }
   }

   /**************************************/
   /* special case, user change password */
   /**************************************/

   if (strsame (rqptr->ResourcePtr, HttpdInternalPasswordChange,
                sizeof(HttpdInternalPasswordChange)-1))
   {
      if (rqptr->HttpMethod == HTTP_METHOD_POST)
          HTAdminPasswordChange (rqptr, PasswordCurrent,
                                 PasswordNew, PasswordVerify);
      else  
          HTAdminPasswordChangeForm (rqptr);
      return;
   }

   /****************************/
   /* administration functions */
   /****************************/

   Location[0] = '\0';

   if (!strcmp (Action, "HTALISTB") ||
       !strcmp (Action, "HTALISTF") ||
       !strcmp (Action, "HTAACCESS") ||
       !strcmp (Action, "HTADELETE") ||
       !strcmp (Action, "HTARESET"))
   {
      if (!DatabaseName[0])
      {
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, ErrorHTAdminDatabaseSelect, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
      for (cptr = Action; *cptr; cptr++) *cptr = tolower(*cptr);
      status = sys$fao (&LocationDatabaseFaoDsc, &Length, &LocationDsc,
                        rqptr->PathInfoPtr, DatabaseName, Action);
   }
   else
   if (!strcmp (Action, "HTACREATE"))
   {
      if (!AsDatabaseName[0])
      {
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, ErrorHTAdminDatabaseEnter, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
      for (cptr = Action; *cptr; cptr++) *cptr = tolower(*cptr);
      status = sys$fao (&LocationDatabaseFaoDsc, &Length, &LocationDsc,
                        rqptr->PathInfoPtr, AsDatabaseName, Action);
   }

   if (Location[0])
   {
      rqptr->LocationPtr = VmGetHeap (rqptr, Length);
      memcpy (rqptr->LocationPtr, Location, Length);
      HTAdminEnd (rqptr);
      return;
   }

   zptr = (sptr = DatabaseName) + sizeof(DatabaseName);
   for (cptr = rqptr->PathInfoPtr; *cptr; cptr++);
   while (cptr > rqptr->PathInfoPtr && *cptr != '/') cptr--;
   if (*cptr == '/')
   {
      cptr++;
      while (*cptr && *cptr != '/' && sptr < zptr) *sptr++ = toupper(*cptr++);
      if (sptr >= zptr)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
      *sptr = '\0';
   }
   else
      DatabaseName[0] = '\0';

   if (DatabaseName[0])
   {
      for (cptr = DatabaseName; *cptr; cptr++)
      {
         if (isalnum(*cptr) || *cptr == '_' || *cptr == '-') continue;
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, ErrorHTAdminDatabaseNBG, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
   }

   if (Action[0])
   {
      if (UserName[0] || AsUserName[0])
      {
         if (strsame (Action, "add", -1))
            cptr = AsUserName;
         else
            cptr = UserName;
         for (/* above */; *cptr; cptr++)
         {
            if (isalnum(*cptr) || *cptr == '_' || *cptr == '-') continue;
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, ErrorHTAdminUserNBG, FI_LI);
            HTAdminEnd (rqptr);
            return;
         }
      }

      if (strsame (Action, "view", -1) ||
          strsame (Action, "modify", -1) ||
          strsame (Action, "delete", -1))
      {
         if (!UserName[0])
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, ErrorHTAdminUserSelect, FI_LI);
            HTAdminEnd (rqptr);
            return;
         }
      }
      else
      if (strsame (Action, "add", -1))
      {
         if (!AsUserName[0])
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, ErrorHTAdminUserEnter, FI_LI);
            HTAdminEnd (rqptr);
            return;
         }
      }

      if (rqptr->HttpMethod == HTTP_METHOD_POST)
      {
         /***************/
         /* POST method */
         /***************/

         if (strsame (Action, "add", -1))
            HTAdminModifyUser (rqptr, true, DatabaseName, AsUserName,
                               FullName, Contact, Email, Enabled,
                               Access, HttpsOnly,
                               PasswordNew, PasswordVerify);
         else
         if (strsame (Action, "modify", -1))
            HTAdminModifyUser (rqptr, false, DatabaseName, UserName,
                               FullName, Contact, Email, Enabled,
                               Access, HttpsOnly,
                               PasswordNew, PasswordVerify);
         else
         if (strsame (Action, "htacreate", -1))
            HTAdminDatabaseCreate (rqptr, DatabaseName);
         else
         if (strsame (Action, "htadelete", -1))
            HTAdminDatabaseDelete (rqptr, DatabaseName);
         else
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, ErrorHTAdminAction, FI_LI);
            HTAdminEnd (rqptr);
            return;
         }
      }
      else
      {
         /**************/
         /* GET method */
         /**************/

         if (strsame (Action, "view", -1))
            HTAdminUserView (rqptr, DatabaseName, UserName);
         else
         if (strsame (Action, "modify", -1))
            HTAdminModifyUserForm (rqptr, false, DatabaseName, UserName);
         else
         if (strsame (Action, "add", -1))
            HTAdminModifyUserForm (rqptr, true, DatabaseName, AsUserName);
         else
         if (strsame (Action, "delete", -1))
            HTAdminUserDelete (rqptr, DatabaseName, UserName);
         else
         if (strsame (Action, "purge", -1))
            HTAdminCacheResetThis (rqptr, "", "");
         else
         if (strsame (Action, "reset", -1))
            HTAdminCacheResetThis (rqptr, DatabaseName, UserName);
         else
         if (strsame (Action, "htalistb", -1))
         {
            tkptr->BriefList = true;
            HTAdminListUsersBegin (rqptr, DatabaseName);
         }
         else
         if (strsame (Action, "htalistf", -1))
         {
            tkptr->BriefList = false;
            HTAdminListUsersBegin (rqptr, DatabaseName);
         }
         else
         if (strsame (Action, "htaaccess", -1))
            HTAdminDatabaseUsersBegin (rqptr, DatabaseName);
         else
         if (strsame (Action, "htacreate", -1))
            HTAdminDatabaseCreateForm (rqptr, DatabaseName);
         else
         if (strsame (Action, "htadelete", -1))
            HTAdminDatabaseDeleteForm (rqptr, DatabaseName);
         else
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, ErrorHTAdminAction, FI_LI);
            HTAdminEnd (rqptr);
            return;
         }
      }
   }
   else
   if (DatabaseName[0])
      HTAdminDatabaseUsersBegin (rqptr, DatabaseName);
   else
      HTAdminDatabaseBegin (rqptr);
}

/*****************************************************************************/
/*
*/ 

HTAdminEnd (struct RequestStruct *rqptr)

{
   register struct HTAdminTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminEnd()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   if (tkptr->ParseInUse)
   {
      /* ensure parse internal data structures are released */
      tkptr->SearchFab.fab$l_fna = "a:[b]c.d;";
      tkptr->SearchFab.fab$b_fns = 9;
      tkptr->SearchFab.fab$b_dns = 0;
      tkptr->SearchNam.nam$b_nop = NAM$M_SYNCHK;
      sys$parse (&tkptr->SearchFab, 0, 0);
   }

   if (tkptr->FileFab.fab$w_ifi)  sys$close (&tkptr->FileFab, 0, 0);

   SysDclAst (tkptr->NextTaskFunction, rqptr);
}

/*****************************************************************************/
/*
Begin a page providing a list of HTA Databases in a form for administering
them.  Set up search for authentication Database files.
*/

HTAdminDatabaseBegin (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (AuthFileSpecFaoDsc, "!AZ*!AZ;0");

   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Administer Authentication</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Administer Authentication</H2>\n\
\
<FORM METHOD=GET ACTION=\"!AZ\">\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Database</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD VALIGN=top>\n\
<SELECT SIZE=!UL NAME=da>\n");

   register unsigned long  *vecptr;
   register char  *cptr, *sptr, *zptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [2048];
   void  *AstFunctionPtr;
   $DESCRIPTOR (AuthFileSpecDsc, "");
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminDatabaseBegin()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   AuthFileSpecDsc.dsc$a_pointer = tkptr->AuthFileSpec;
   AuthFileSpecDsc.dsc$w_length = sizeof(tkptr->AuthFileSpec)-1;

   sys$fao (&AuthFileSpecFaoDsc, &Length, &AuthFileSpecDsc,
            HTA_DIRECTORY, HTA_FILE_TYPE);
   tkptr->AuthFileSpec[Length] = '\0';
   if (Debug) fprintf (stdout, "AuthFileSpec |%s|\n", tkptr->AuthFileSpec);

   tkptr->ParseInUse = true;

   tkptr->SearchFab = cc$rms_fab;
   tkptr->SearchFab.fab$l_ctx = rqptr;
   tkptr->SearchFab.fab$l_fna = tkptr->AuthFileSpec;
   tkptr->SearchFab.fab$b_fns = Length;
   tkptr->SearchFab.fab$l_fop = FAB$M_NAM;
   tkptr->SearchFab.fab$l_nam = &tkptr->SearchNam;

   tkptr->SearchNam = cc$rms_nam;
   tkptr->SearchNam.nam$l_esa = tkptr->ExpandedFileName;
   tkptr->SearchNam.nam$b_ess = sizeof(tkptr->ExpandedFileName)-1;
   tkptr->SearchNam.nam$l_rsa = tkptr->ResultantFileName;
   tkptr->SearchNam.nam$b_rss = sizeof(tkptr->ResultantFileName)-1;

   if (VMSnok (status = sys$parse (&tkptr->SearchFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->AuthFileSpec);
      rqptr->ErrorHiddenTextPtr = tkptr->AuthFileSpec;
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /**************/
   /* begin page */
   /**************/

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = rqptr->ScriptName;
   *vecptr++ = DatabaseListSize;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   tkptr->FileCount = 0;

   NetWriteBuffered (rqptr, &HTAdminDatabaseSearch, Buffer, Length);
}

/*****************************************************************************/
/*
(AST) function to invoke another sys$search() call when listing authentication
Databases.
*/ 

HTAdminDatabaseSearch (struct RequestStruct *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminDatabaseSearch()\n");

   /* call RMS directory search routine */
   sys$search (&rqptr->HTAdminTaskPtr->SearchFab,
               &HTAdminDatabaseSearchAst,
               &HTAdminDatabaseSearchAst);
}

/*****************************************************************************/
/*
AST completion routine called each time sys$search() completes.  It will 
either point to another file name found or have "no more files found" status 
(or an error!).
*/ 

HTAdminDatabaseSearchAst (struct FAB *FabPtr)

{
   register char  *cptr, *sptr;
   register struct RequestStruct  *rqptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status;
   char  Buffer [128];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
      "HTAdminDatabaseSearchAst() sts: %%X%08.08X stv: %%X%08.08X\n",
      FabPtr->fab$l_sts, FabPtr->fab$l_stv);

   /* retrieve the pointer to the client thread from the FAB user context */
   rqptr = FabPtr->fab$l_ctx;
   /* get the pointer to the task structure */
   tkptr = rqptr->HTAdminTaskPtr;

   if (VMSnok (status = tkptr->SearchFab.fab$l_sts))
   {
      if (status == RMS$_FNF || status == RMS$_NMF)
      {
         /* end of search */
         tkptr->ParseInUse = false;
         HTAdminDatabaseEnd (rqptr);
         return;
      }

      /* sys$search() error */
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->AuthFileSpec);
      rqptr->ErrorHiddenTextPtr = tkptr->AuthFileSpec;
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /* terminate following the last character in the version number */
   tkptr->SearchNam.nam$l_ver[tkptr->SearchNam.nam$b_ver] = '\0';
   if (Debug) fprintf (stdout, "Database |%s|\n", tkptr->ResultantFileName);

   tkptr->FileCount++;

   sptr = Buffer;
   strcpy (sptr, "<OPTION VALUE=\"");
   sptr += 15;
   for (cptr = tkptr->SearchNam.nam$l_name;
        *cptr && *cptr != '.';
        *sptr++ = tolower(*cptr++));
   strcpy (sptr, "\">");
   sptr += 2;
   for (cptr = tkptr->SearchNam.nam$l_name;
        *cptr && *cptr != '.';
        *sptr++ = toupper(*cptr++));
   *sptr++ = '\n';
   *sptr = '\0';

   NetWriteBuffered (rqptr, &HTAdminDatabaseSearch, Buffer, sptr - Buffer);
}

/*****************************************************************************/
/*
End authentication Database file search.  Conclude form's HTML.
*/

HTAdminDatabaseEnd (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (EndPageFaoDsc,
"</SELECT>\n\
</TD><TD VALIGN=top>\n\
<INPUT TYPE=radio NAME=do VALUE=HTAACCESS CHECKED>access<SUP>1</SUP>\n\
<BR><INPUT TYPE=radio NAME=do VALUE=HTALISTB>list/brief<SUP>1</SUP>\n\
<BR><INPUT TYPE=radio NAME=do VALUE=HTALISTF>list/full<SUP>1</SUP>\n\
<BR><INPUT TYPE=radio NAME=do VALUE=HTACREATE>create<SUP>2</SUP>\n\
<BR><INPUT TYPE=radio NAME=do VALUE=HTADELETE>delete<SUP>1</SUP>\n\
<BR><INPUT TYPE=radio NAME=do VALUE=reset>reset cache<SUP>1[or3]</SUP>\n\
<BR><INPUT TYPE=submit VALUE=\" Do \">\n\
<INPUT TYPE=reset VALUE=\" Reset \">\n\
</TD></TR>\n\
</TABLE>\n\
<SUP>1.</SUP> !AZ\n\
<BR><SUP>2.</SUP> enter name <INPUT TYPE=text NAME=ad SIZE=20>\n\
<BR><SUP>3.</SUP> if none selected then purges complete cache\n\
\
</TR>\n\
</TABLE>\n\
</FORM>\n\
\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [1024];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminDatabaseEnd()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   vecptr = FaoVector;
   if (tkptr->FileCount)
      *vecptr++ = "select from list";
   else
      *vecptr++ = "<I>none available</I>";

   status = sys$faol (&EndPageFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   NetWriteBuffered (rqptr, &HTAdminEnd, Buffer, Length);
}

/*****************************************************************************/
/*
*/

HTAdminDatabaseUsersBegin
(
struct RequestStruct *rqptr,
char *DatabaseName
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Administer Authentication</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Administer Authentication</H2>\n\
\
<FORM METHOD=GET ACTION=\"!AZ\">\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Users in !AZ</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD VALIGN=top>\n\
<SELECT SIZE=!UL NAME=u>\n");

   register unsigned long  *vecptr;
   register char  *cptr, *sptr, *zptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [2048];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminDatabaseBegin()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   if (VMSnok (HTAdminOpenDatabaseForRead (rqptr, DatabaseName)))
      return;

   /**************/
   /* begin page */
   /**************/

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = rqptr->PathInfoPtr;
   *vecptr++ = DatabaseName;
   *vecptr++ = UserNameListSize;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   tkptr->RecordCount = tkptr->UserCount = 0;

   NetWriteBuffered (rqptr, &HTAdminDatabaseUsersNext, Buffer, Length);
}

/*****************************************************************************/
/*
Queue a read of the next record from the file.  When the read completes call 
HTAdminDatabaseUsersNextAst() function.
*/ 

HTAdminDatabaseUsersNext (struct RequestStruct *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminDatabaseUsersNext()\n");

   sys$get (&rqptr->HTAdminTaskPtr->FileRab,
            &HTAdminDatabaseUsersNextAst,
            &HTAdminDatabaseUsersNextAst);
}

/*****************************************************************************/
/*
A user record has been read from the authentication Database.
*/ 

HTAdminDatabaseUsersNextAst (struct RAB *RabPtr)

{
   register int  cnt;
   register char  *cptr, *sptr;
   register struct RequestStruct  *rqptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status;
   char  Buffer [128];

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout,
"HTAdminDatabaseUsersNextAst sts: %%X%08.08X stv: %%X%08.08X rsz: %d\n",
      RabPtr->rab$l_sts, RabPtr->rab$l_stv,  RabPtr->rab$w_rsz);
   }

   rqptr = RabPtr->rab$l_ctx;
   tkptr = rqptr->HTAdminTaskPtr;

   if (VMSnok (tkptr->FileRab.rab$l_sts))
   {
      if (tkptr->FileRab.rab$l_sts == RMS$_EOF)
      {
         if (Debug) fprintf (stdout, "RMS$_EOF\n");
         sys$close (&tkptr->FileFab, 0, 0);
         HTAdminDatabaseUsersListSort (rqptr);
         return;
      }

      rqptr->ErrorTextPtr = MapVmsPath (tkptr->AuthFileName);
      rqptr->ErrorHiddenTextPtr = tkptr->AuthFileName;
      ErrorVmsStatus (rqptr, tkptr->FileRab.rab$l_sts, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /* check the version of the authorization database */
   if (tkptr->AuthHtRecord.DatabaseVersion &&
       tkptr->AuthHtRecord.DatabaseVersion != AuthCurrentDatabaseVersion)
   {
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->AuthFileName);
      rqptr->ErrorHiddenTextPtr = tkptr->AuthFileName;
      ErrorVmsStatus (rqptr, SS$_INCOMPAT & 0xfffffffe, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /* if the record has been removed (by zeroing) then ignore */
   if (!tkptr->AuthHtRecord.UserNameLength)
   {
      HTAdminDatabaseUsersNext (rqptr);
      return;
   }

   tkptr->UserCount++;

   if ((tkptr->UserCount * sizeof(tkptr->AuthHtRecord.UserName)) >
       tkptr->UserListLength)
   {
      /* need more (or some) list space */
      tkptr->UserListLength += 32 * sizeof(tkptr->AuthHtRecord.UserName);
      tkptr->UserListPtr =
         VmReallocHeap (rqptr, tkptr->UserListPtr, tkptr->UserListLength);
   }

   /* copy username including name terminating null */
   memcpy (tkptr->UserListPtr +
           ((tkptr->UserCount - 1) * sizeof(tkptr->AuthHtRecord.UserName)),
           tkptr->AuthHtRecord.UserName,
           sizeof(tkptr->AuthHtRecord.UserName));

   HTAdminDatabaseUsersNext (rqptr);
}

/*****************************************************************************/
/*
Humble bubble sort I'm afraid :^(  Gives the illusion that the database is
ordered (apart from that of entry sequence :^), although does provide the
advantage of viewing an ordered list.
*/ 

HTAdminDatabaseUsersListSort (struct RequestStruct *rqptr)

{
   register int  idx1, idx2, size;
   register char  *cptr1, *cptr2;
   register struct HTAdminTaskStruct  *tkptr;

   char UserName [sizeof(tkptr->AuthHtRecord.UserName)];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminDatabaseUsersListSort()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   size = sizeof(tkptr->AuthHtRecord.UserName);

   if (!tkptr->UserCount)
   {
      HTAdminDatabaseUsersEnd (rqptr);
      return;
   }

   for (idx1 = 0; idx1 < tkptr->UserCount-1; idx1++)
   {
      for (idx2 = idx1+1; idx2 < tkptr->UserCount; idx2++)
      {
          cptr1 = tkptr->UserListPtr + (idx1 * size);
          cptr2 = tkptr->UserListPtr + (idx2 * size);
          if (strcmp (cptr1, cptr2) <= 0) continue;
          memcpy (UserName, cptr1, size);
          memcpy (cptr1, cptr2, size);
          memcpy (cptr2, UserName, size);
      }
   }

   tkptr->UserListCount = 0;
   HTAdminDatabaseUsersList (rqptr);
}

/*****************************************************************************/
/*
This function is called for each username in the sorted list.  It reads them
sequentially, formats them as part of HTML selection list, then buffers the
output, ASTing back to this function for the next.
*/ 

HTAdminDatabaseUsersList (struct RequestStruct *rqptr)

{
   register char  *cptr, *sptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status;
   char  Buffer [128];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "HTAdminDatabaseUsersList() %d of %d\n",
               tkptr->UserListCount, tkptr->UserCount);

   tkptr = rqptr->HTAdminTaskPtr;

   if (tkptr->UserListCount >= tkptr->UserCount)
   {
      HTAdminDatabaseUsersEnd (rqptr);
      return;
   }

   sptr = Buffer;
   strcpy (sptr, "<OPTION VALUE=\"");
   sptr += 15;
   for (cptr = tkptr->UserListPtr +
               (tkptr->UserListCount * sizeof(tkptr->AuthHtRecord.UserName));
        *cptr;
        *sptr++ = tolower(*cptr++));
   strcpy (sptr, "\">");
   sptr += 2;
   for (cptr = tkptr->UserListPtr +
               (tkptr->UserListCount * sizeof(tkptr->AuthHtRecord.UserName));
        *cptr;
        *sptr++ = *cptr++);
   *sptr++ = '\n';
   *sptr = '\0';

   tkptr->UserListCount++;

   NetWriteBuffered (rqptr, &HTAdminDatabaseUsersList, Buffer, sptr-Buffer);
}

/*****************************************************************************/
/*
End user names in authentication Database.
*/

HTAdminDatabaseUsersEnd (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (EndPageFaoDsc,
"</SELECT>\n\
</TD><TD VALIGN=top>\n\
<INPUT TYPE=radio NAME=do VALUE=view CHECKED>view<SUP>1</SUP>\n\
<BR><INPUT TYPE=radio NAME=do VALUE=modify>modify<SUP>1</SUP>\n\
<BR><INPUT TYPE=radio NAME=do VALUE=add>add<SUP>2</SUP>\n\
<BR><INPUT TYPE=radio NAME=do VALUE=delete>delete<SUP>1</SUP>\n\
<BR><INPUT TYPE=radio NAME=do VALUE=reset>reset cache<SUP>1[or3]</SUP>\n\
<BR><INPUT TYPE=submit VALUE=\" Do \">\n\
<INPUT TYPE=reset VALUE=\" Reset \">\n\
</TD></TR>\n\
</TABLE>\n\
<SUP>1.</SUP> !AZ\n\
<BR><SUP>2.</SUP> enter name <INPUT TYPE=text NAME=au SIZE=20>\n\
<BR><SUP>3.</SUP> if none selected then resets all users\n\
\
</TR>\n\
</TABLE>\n\
</FORM>\n\
\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [1024];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminDatabaseUsersEnd()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   vecptr = FaoVector;
   if (tkptr->UserCount)
      *vecptr++ = "select from list";
   else
      *vecptr++ = "<I>none available</I>";

   status = sys$faol (&EndPageFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   NetWriteBuffered (rqptr, &HTAdminEnd, Buffer, Length);
}

/*****************************************************************************/
/*
*/

HTAdminListUsersBegin
(
struct RequestStruct *rqptr,
char *DatabaseName
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Administer Authentication</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Administer Authentication</H2>\n\
\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=!UL>Users in !AZ</TH></TR>\n\
!AZ");

   static char  BriefHeading [] =
"<TR><TH>Username</TH><TH>Full Name</TH><TH>Access</TH>\
<TH>Added</TH><TH COLSPAN=2>Accessed</TH></TR>\n\
<TR><TH COLSPAN=6></TH></TR>\n";

   static char  FullHeading [] =
"<TR><TH>Username</TH><TH COLSPAN=2>Full Name</TH>\
<TH COLSPAN=3>Access</TH><TH>Added</TH></TR>\n\
<TR><TD></TD><TH COLSPAN=6>Contact</TH></TR>\n\
<TR><TD></TD><TH COLSPAN=6>Email</TH></TR>\n\
<TR><TD></TD>\
<TH COLSPAN=2>Accessed</TH>\
<TH COLSPAN=2>Change</TH>\
<TH COLSPAN=2>Failure</TH></TR>\n\
<TR><TH COLSPAN=7></TH></TR>\n";

   register unsigned long  *vecptr;
   register char  *cptr, *sptr, *zptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status,
        SetPrvStatus;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [2048];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminListUsersBegin()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   if (VMSnok (HTAdminOpenDatabaseForRead (rqptr, DatabaseName)))
      return;

   /**************/
   /* begin page */
   /**************/

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;

   if (tkptr->BriefList)
      *vecptr++ = 6;
   else
      *vecptr++ = 7;

   *vecptr++ = DatabaseName;

   if (tkptr->BriefList)
      *vecptr++ = BriefHeading;
   else
      *vecptr++ = FullHeading;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   tkptr->RecordCount = tkptr->UserCount = 0;

   NetWriteBuffered (rqptr, &HTAdminListUsersNext, Buffer, Length);
}

/*****************************************************************************/
/*
Queue a read of the next record from the file.  When the read completes call 
HTAdminListUsersNextAst() function.
*/ 

HTAdminListUsersNext (struct RequestStruct *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminListUsersNext()\n");

   sys$get (&rqptr->HTAdminTaskPtr->FileRab,
            &HTAdminListUsersNextAst,
            &HTAdminListUsersNextAst);
}

/*****************************************************************************/
/*
A user record has been read from the authentication Database.
*/ 

HTAdminListUsersNextAst (struct RAB *RabPtr)

{
   register int  cnt;
   register char  *cptr, *sptr;
   register struct RequestStruct  *rqptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status;
   char  Buffer [128];

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout,
"HTAdminListUsersNextAst sts: %%X%08.08X stv: %%X%08.08X rsz: %d\n",
      RabPtr->rab$l_sts, RabPtr->rab$l_stv,  RabPtr->rab$w_rsz);
   }

   rqptr = RabPtr->rab$l_ctx;
   tkptr = rqptr->HTAdminTaskPtr;

   if (VMSnok (tkptr->FileRab.rab$l_sts))
   {
      if (tkptr->FileRab.rab$l_sts == RMS$_EOF)
      {
         if (Debug) fprintf (stdout, "RMS$_EOF\n");
         sys$close (&tkptr->FileFab, 0, 0);
         HTAdminListUsersListSort (rqptr);
         return;
      }

      rqptr->ErrorTextPtr = MapVmsPath (tkptr->AuthFileName);
      rqptr->ErrorHiddenTextPtr = tkptr->AuthFileName;
      ErrorVmsStatus (rqptr, tkptr->FileRab.rab$l_sts, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /* check the version of the authorization database */
   if (tkptr->AuthHtRecord.DatabaseVersion &&
       tkptr->AuthHtRecord.DatabaseVersion != AuthCurrentDatabaseVersion)
   {
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->AuthFileName);
      rqptr->ErrorHiddenTextPtr = tkptr->AuthFileName;
      ErrorVmsStatus (rqptr, SS$_INCOMPAT & 0xfffffffe, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /* if the record has been removed (by zeroing) then ignore */
   if (!tkptr->AuthHtRecord.UserNameLength)
   {
      HTAdminListUsersNext (rqptr);
      return;
   }

   tkptr->UserCount++;

   if ((tkptr->UserCount * sizeof(struct AuthHtRecordStruct)) >
       tkptr->UserListLength)
   {
      /* need more (or some) list space */
      tkptr->UserListLength += 32 * sizeof(struct AuthHtRecordStruct);
      tkptr->UserListPtr =
         VmReallocHeap (rqptr, tkptr->UserListPtr, tkptr->UserListLength);
   }

   /* copy the entire user record into the list */
   memcpy (tkptr->UserListPtr +
           ((tkptr->UserCount - 1) * sizeof(struct AuthHtRecordStruct)),
           &tkptr->AuthHtRecord, sizeof(struct AuthHtRecordStruct));

   HTAdminListUsersNext (rqptr);
}

/*****************************************************************************/
/*
Humble bubble sort I'm afraid :^(  Gives the illusion that the database is
ordered (apart from that of entry sequence :^), although does provide the
advantage of viewing an ordered list.  Bit more expensive this one, copying
aroung 512 byte records.  I'll improve it someday, sigh :^(
*/ 

HTAdminListUsersListSort (struct RequestStruct *rqptr)

{
   register int  idx1, idx2, size;
   register char  *cptr1, *cptr2;
   register struct AuthHtRecordStruct  *rptr1, *rptr2;
   register struct HTAdminTaskStruct  *tkptr;

   struct AuthHtRecordStruct  AuthHtRecord;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminListUsersListSort()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   if (!tkptr->UserCount)
   {
      HTAdminListUsersEnd (rqptr);
      return;
   }

   size = sizeof(struct AuthHtRecordStruct);

   for (idx1 = 0; idx1 < tkptr->UserCount-1; idx1++)
   {
      for (idx2 = idx1+1; idx2 < tkptr->UserCount; idx2++)
      {
          rptr1 = &((struct AuthHtRecordStruct*)tkptr->UserListPtr)[idx1];
          rptr2 = &((struct AuthHtRecordStruct*)tkptr->UserListPtr)[idx2];
          if (strcmp (rptr1->UserName, rptr2->UserName) <= 0) continue;
          memcpy (&AuthHtRecord, (char*)rptr1, size);
          memcpy ((char*)rptr1, (char*)rptr2, size);
          memcpy ((char*)rptr2, &AuthHtRecord, size);
      }
   }

   tkptr->UserListCount = 0;
   HTAdminListUsersList (rqptr);
}

/*****************************************************************************/
/*
This function is called for each username in the sorted list.  It reads them
sequentially, formats them as part of HTML table, then buffers the output,
ASTing back to this function for the next.
*/ 

HTAdminListUsersList (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (MailToFaoDsc, "<A HREF=\"mailto:!AZ\">!AZ</A>\0");

   static $DESCRIPTOR (BriefFaoDsc,
"<TR>\
<TD><B>!AZ!AZ!AZ</B></TD><TD>!AZ</TD><TD>!AZ!AZ</TD>\n\
<TD>!AZ</TD>\
<TD>!UL</TD><TD>!AZ</TD>\
</TR>\n");

   static $DESCRIPTOR (FullFaoDsc,
"<TR><TD><B>!AZ!AZ!AZ</B></TD><TD COLSPAN=2>!AZ</TD>\
<TD COLSPAN=3>!AZ!AZ</TD><TD>!AZ</TD></TR>\n\
<TR><TD></TD><TD COLSPAN=6><PRE>!AZ</PRE></TD></TR>\n\
<TR><TD></TD><TD COLSPAN=6>!AZ</TD></TR>\n\
<TR><TD></TD>\
<TD>!UL</TD><TD>!AZ</TD>\
<TD>!UL</TD><TD>!AZ</TD>\
<TD>!UL</TD><TD>!AZ</TD>\
</TR>\n");

   register unsigned long  *vecptr;
   register struct AuthHtRecordStruct  *rptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  *CanStringPtr;
   char  AddedDateTime [32],
         Buffer [1024],
         HtmlEscapedContact [256],
         HtmlEscapedEmail [256],
         HtmlEscapedFullName [128],
         LastAccessDateTime [32],
         LastChangeDateTime [32],
         LastFailureDateTime [32],
         MailTo [256];
   $DESCRIPTOR (BufferDsc, Buffer);
   $DESCRIPTOR (MailToDsc, MailTo);

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "HTAdminListUsersList() %d of %d\n",
               tkptr->UserListCount, tkptr->UserCount);

   tkptr = rqptr->HTAdminTaskPtr;

   if (tkptr->UserListCount >= tkptr->UserCount)
   {
      HTAdminListUsersEnd (rqptr);
      return;
   }

   rptr = &((struct AuthHtRecordStruct*)tkptr->UserListPtr)
             [tkptr->UserListCount];

   if (CopyToHtml (HtmlEscapedFullName, sizeof(HtmlEscapedFullName),
                   rptr->FullName, -1) < 0)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   strcpy (AddedDateTime, DateTime (&rptr->AddedBinTime, 17));
   strcpy (LastAccessDateTime, DateTime (&rptr->LastAccessBinTime, 17));

   if ((CanStringPtr =
        HTAdminCanString (rqptr, rptr->Flags, tkptr->BriefList)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   if (rptr->Flags & AUTH_FLAG_ENABLED)
      *vecptr++ = "";
   else
      *vecptr++ = "<I>";
   *vecptr++ = rptr->UserName;
   if (rptr->Flags & AUTH_FLAG_ENABLED)
      *vecptr++ = "";
   else
      *vecptr++ = "</I>";

   *vecptr++ = HtmlEscapedFullName;
   *vecptr++ = CanStringPtr;

   if (rptr->Flags & AUTH_FLAG_HTTPS_ONLY)
      *vecptr++ = " (&quot;https:&quot;&nbsp;only)";
   else
      *vecptr++ = "";

   if (tkptr->BriefList)
   {
      *vecptr++ = AddedDateTime;
      *vecptr++ = rptr->AccessCount;
      *vecptr++ = LastAccessDateTime;
   }
   else
   {
      if (CopyToHtml (HtmlEscapedContact, sizeof(HtmlEscapedContact),
                      rptr->Contact, -1) < 0)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }

      if (rptr->Email[0])
      {
         if (CopyToHtml (HtmlEscapedEmail, sizeof(HtmlEscapedEmail),
                         rptr->Email, -1) < 0)
         {
            ErrorGeneralOverflow (rqptr, FI_LI);
            HTAdminEnd (rqptr);
            return;
         }
         status = sys$fao (&MailToFaoDsc, 0, &MailToDsc,
                           HtmlEscapedEmail, HtmlEscapedEmail);
         if (VMSnok (status) || status == SS$_BUFFEROVF)
         {
               ErrorGeneralOverflow (rqptr, FI_LI);
               HTAdminEnd (rqptr);
               return;
         }
      }
      else
         MailTo[0] = '\0';

      strcpy (LastChangeDateTime, DateTime (&rptr->LastChangeBinTime, 17));
      strcpy (LastFailureDateTime, DateTime (&rptr->LastFailureBinTime, 17));

      *vecptr++ = AddedDateTime;
      *vecptr++ = HtmlEscapedContact;
      *vecptr++ = MailTo;

      *vecptr++ = rptr->AccessCount;
      *vecptr++ = LastAccessDateTime;
      *vecptr++ = rptr->ChangeCount;
      *vecptr++ = LastChangeDateTime;
      *vecptr++ = rptr->FailureCount;
      *vecptr++ = LastFailureDateTime;
   }

   if (tkptr->BriefList)
      status = sys$faol (&BriefFaoDsc, &Length, &BufferDsc, &FaoVector);
   else
      status = sys$faol (&FullFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   tkptr->UserListCount++;

   NetWriteBuffered (rqptr, &HTAdminListUsersList, Buffer, Length);
}

/*****************************************************************************/
/*
End list user information in authentication Database.
*/

HTAdminListUsersEnd (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (NoneFaoDsc,
"<TR><TD COLSPAN=!UL><I>(none)</I></TD></TR>\n\
</TABLE>\n\
\
</BODY>\n\
</HTML>\n");

   static $DESCRIPTOR (TotalFaoDsc,
"<TR><TH></TH></TR>\n\
<TR><TH COLSPAN=!UL>Total: !UL</TH></TR>\n\
</TABLE>\n\
\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [256];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminListUsersEnd()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   vecptr = FaoVector;

   if (tkptr->BriefList)
      *vecptr++ = 6;
   else
      *vecptr++ = 7;

   if (tkptr->UserCount)
   {
      *vecptr++ = tkptr->UserCount;
      status = sys$faol (&TotalFaoDsc, &Length, &BufferDsc, &FaoVector);
   }
   else
      status = sys$faol (&NoneFaoDsc, &Length, &BufferDsc, &FaoVector);

   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   NetWriteBuffered (rqptr, &HTAdminEnd, Buffer, Length);
}

/*****************************************************************************/
/*
Display an authentication database record.
*/

HTAdminUserView
(
struct RequestStruct *rqptr,
char *DatabaseName,
char *UserName
)
{
   static $DESCRIPTOR (MailToFaoDsc, "<A HREF=\"mailto:!AZ\">!AZ</A>\0");

   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Administer Authentication</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Administer Authentication</H2>\n\
!AZ\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=4>User !AZ in !AZ!AZ</TH></TR>\n\
<TR><TH>Full Name</TH><TD COLSPAN=3>!AZ</TD></TR>\n\
<TR><TH>Contact</TH><TD COLSPAN=3><PRE>!AZ</PRE></TD></TR>\n\
<TR><TH>E-mail</TH><TD COLSPAN=3>!AZ</TD></TR>\n\
<TR><TH>Access</TH><TD COLSPAN=3>!AZ!AZ</TD></TR>\n\
<TR><TH>Password</TH><TD>!AZ</TD><TD></TD><TD></TD></TR>\n\
<TR><TH>Added</TH><TD COLSPAN=2>!AZ</TD></TR>\n\
<TR><TH>Changed</TH><TD COLSPAN=2>!AZ</TD><TD ALIGN=right>!UL</TD></TR>\n\
<TR><TH>Accessed</TH><TD COLSPAN=2>!AZ</TD><TD ALIGN=right>!UL</TD></TR>\n\
<TR><TH>Failed</TH><TD COLSPAN=2>!AZ</TD><TD ALIGN=right>!UL</TD></TR>\n\
</TABLE>\n\
\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [64];
   char  *CanStringPtr;
   char  Buffer [2048],
         HtmlEscapedContact [256],
         HtmlEscapedEmail [256],
         MailTo [256],
         TimeCurrent [32],
         TimeAdded [32],
         TimeChange [32],
         TimeAccess [32],
         TimeFailure [32];
   $DESCRIPTOR (BufferDsc, Buffer);
   $DESCRIPTOR (MailToDsc, MailTo);
   struct AuthHtRecordStruct AuthHtRecord;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminUserView()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   if (!DatabaseName || !UserName[0])
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorHTAdminInsufficient, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /* look for the record, close the database file immediately */
   status = AuthAccessHtDatabase (false, DatabaseName, UserName,
                                  &AuthHtRecord, NULL, NULL);
   if (status == SS$_INVLOGIN)
   {
      rqptr->ResponseStatusCode = 404;
      ErrorGeneral (rqptr, ErrorHTAdminUserNotFound, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }
   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = ErrorHTAdminDatabase;
      rqptr->ErrorHiddenTextPtr = DatabaseName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /**************/
   /* begin page */
   /**************/

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = strcpy (TimeCurrent, DayDateTime (&rqptr->BinaryTime, 20));

   *vecptr++ = AuthHtRecord.UserName;
   *vecptr++ = DatabaseName;
   if (AuthHtRecord.Flags & AUTH_FLAG_ENABLED)
      *vecptr++ = "";
   else
      *vecptr++ = "<FONT COLOR=\"#ff0000\"> is DISABLED</FONT>";

   *vecptr++ = AuthHtRecord.FullName;

   if (CopyToHtml (HtmlEscapedContact, sizeof(HtmlEscapedContact),
                   AuthHtRecord.Contact, -1) < 0)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }
   *vecptr++ = HtmlEscapedContact;

   if (AuthHtRecord.Email[0])
   {
      if (CopyToHtml (HtmlEscapedEmail, sizeof(HtmlEscapedEmail),
                      AuthHtRecord.Email, -1) < 0)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
      status = sys$fao (&MailToFaoDsc, 0, &MailToDsc,
                        HtmlEscapedEmail, HtmlEscapedEmail);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
      *vecptr++ = MailTo;
   }
   else
      *vecptr++ = "";

   if ((CanStringPtr =
        HTAdminCanString (rqptr, AuthHtRecord.Flags, false)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }
   *vecptr++ = CanStringPtr;

   if (AuthHtRecord.Flags & AUTH_FLAG_HTTPS_ONLY)
      *vecptr++ = " (&quot;https:&quot;&nbsp;only)";
   else
      *vecptr++ = "";

   if (AuthHtRecord.HashedPwd[0] || AuthHtRecord.HashedPwd[1])
      *vecptr++ = "********";
   else
      *vecptr++ = "";

   *vecptr++ = strcpy (TimeAdded,
                       DayDateTime (&AuthHtRecord.AddedBinTime, 20));

   *vecptr++ = strcpy (TimeChange,
                       DayDateTime (&AuthHtRecord.LastChangeBinTime, 20));
   *vecptr++ = AuthHtRecord.ChangeCount;

   *vecptr++ = strcpy (TimeAccess,
                       DayDateTime (&AuthHtRecord.LastAccessBinTime, 20));
   *vecptr++ = AuthHtRecord.AccessCount;

   *vecptr++ = strcpy (TimeFailure,
                       DayDateTime (&AuthHtRecord.LastFailureBinTime, 20));
   *vecptr++ = AuthHtRecord.FailureCount;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWrite (rqptr, &HTAdminEnd, Buffer, Length);

   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
Form for modify or add an authentication database record.
*/

HTAdminModifyUserForm
(
struct RequestStruct *rqptr,
boolean AddUser,
char *DatabaseName,
char *UserName
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Administer Authentication</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Administer Authentication</H2>\n\
\
<FORM METHOD=POST ACTION=\"!AZ\">\n\
<INPUT TYPE=HIDDEN NAME=do VALUE=!AZ>\n\
<INPUT TYPE=HIDDEN NAME=!AZ VALUE=\"!AZ\">\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>!AZ !AZ !AZin !AZ</TH></TR>\n\
<TR><TD COLSPAN=2>\n\
\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH ALIGN=right>Full Name</TH>\
<TD><INPUT TYPE=text NAME=f SIZE=!UL MAXLENGTH=!UL \
VALUE=\"!AZ\"></TD></TR>\n\
<TR><TH ALIGN=right>Contact</TH>\
<TD><TEXTAREA NAME=co COLS=!UL ROWS=!UL>!AZ\
</TEXTAREA></TD></TR>\n\
<TR><TH ALIGN=right>E-mail</TH>\
<TD><INPUT TYPE=text NAME=em SIZE=!UL MAXLENGTH=!UL \
VALUE=\"!AZ\"></TD></TR>\n\
<TR><TH ALIGN=right>Enabled</TH><TD>\n\
<INPUT TYPE=radio NAME=en VALUE=yes!AZ>yes\n\
<INPUT TYPE=radio NAME=en VALUE=no!AZ>no\n\
</TD></TR>\n\
<TR><TH ALIGN=right>Access</TH><TD>\n\
<INPUT TYPE=radio NAME=ac VALUE=\"r\"!AZ>read-only\n\
<INPUT TYPE=radio NAME=ac VALUE=\"r+w\"!AZ>read <B>& write</B>\n\
<INPUT TYPE=radio NAME=ac VALUE=\"w\"!AZ>write-only\n\
</TD></TR>\n\
<TR><TH ALIGN=right>&quot;https:&quot; (SSL) Only</TH><TD>\n\
<INPUT TYPE=radio NAME=hs VALUE=yes!AZ>yes\n\
<INPUT TYPE=radio NAME=hs VALUE=no!AZ>no\n\
</TD></TR>\n\
<TR><TH ALIGN=right>Password</TH>\
<TD><INPUT TYPE=password NAME=p SIZE=!UL MAXLENGTH=!UL></TD></TR>\n\
<TR><TH ALIGN=right>Verify</TH>\
<TD><INPUT TYPE=password NAME=v SIZE=!UL MAXLENGTH=!UL></TD></TR>\n\
</TABLE>\n\
\
</TD></TR>\n\
<TR><TD>\n\
<TR><TD COLSPAN=2><INPUT TYPE=submit VALUE=\" !AZ \">\n\
<INPUT TYPE=reset VALUE=\" Reset \"></TD></TR>\n\
</TD></TR>\n\
</TABLE>\n\
</FORM>\n\
\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register struct HTAdminTaskStruct  *tkptr;

   boolean  AlreadyExists;
   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [64];
   char  Buffer [2048],
         HtmlEscapedContact [256],
         HtmlEscapedEmail [256],
         HtmlEscapedFullName [256];
   $DESCRIPTOR (BufferDsc, Buffer);
   struct AuthHtRecordStruct AuthHtRecord;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminModifyUserForm()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   if (!DatabaseName[0] || !UserName[0])
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorHTAdminInsufficient, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /* look for the record, close the database file immediately */
   if (AddUser)
   {
      status = AuthAccessHtDatabase (false, DatabaseName, UserName,
                                     &AuthHtRecord, NULL, NULL);
      if (VMSok (status))
         AlreadyExists = true;
      else
         AlreadyExists = false;
      if (status == SS$_INVLOGIN) status = SS$_NORMAL;
   }
   else
   {
      status = AuthAccessHtDatabase (false, DatabaseName, UserName,
                                     &AuthHtRecord, NULL, NULL);
      if (status == SS$_INVLOGIN)
      {
         rqptr->ResponseStatusCode = 404;
         ErrorGeneral (rqptr, ErrorHTAdminUserNotFound, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
      AlreadyExists = false;
   }
   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = ErrorHTAdminDatabase;
      rqptr->ErrorHiddenTextPtr = DatabaseName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }
      
   if (AddUser && !AlreadyExists)
   {
      memset (&AuthHtRecord, 0, sizeof(struct AuthHtRecordStruct));
      HtmlEscapedContact[0] = HtmlEscapedFullName[0] =
         HtmlEscapedEmail[0] = '\0';
   }
   else
   {
      if (CopyToHtml (HtmlEscapedFullName, sizeof(HtmlEscapedFullName),
                      AuthHtRecord.FullName, -1) < 0)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
      if (CopyToHtml (HtmlEscapedContact, sizeof(HtmlEscapedContact),
                      AuthHtRecord.Contact, -1) < 0)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
      if (CopyToHtml (HtmlEscapedEmail, sizeof(HtmlEscapedEmail),
                      AuthHtRecord.Email, -1) < 0)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
   }

   /**************/
   /* begin page */
   /**************/

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;

   *vecptr++ = rqptr->PathInfoPtr;
   if (AddUser && !AlreadyExists)
   {
      *vecptr++ = "add";
      *vecptr++ = "au";
      *vecptr++ = UserName;
      *vecptr++ = "<FONT COLOR=\"#ff0000\">New User</FONT>";
      *vecptr++ = UserName;
      *vecptr++ = "";
      *vecptr++ = DatabaseName;
   }
   else
   {     
      *vecptr++ = "modify";
      *vecptr++ = "u";
      *vecptr++ = UserName;
      *vecptr++ = "User";
      *vecptr++ = UserName;
      if (AlreadyExists)
         *vecptr++ =
"<FONT COLOR=\"#ff0000\"> &nbsp;&nbsp;&nbsp;\
ALREADY EXISTS&nbsp;&nbsp;&nbsp; </FONT>";
      else
         *vecptr++ = "";
      *vecptr++ = DatabaseName;
   }

   *vecptr++ = MaxAuthFullNameLength;
   *vecptr++ = MaxAuthFullNameLength;
   *vecptr++ = HtmlEscapedFullName;

   *vecptr++ = 40;
   *vecptr++ = 3;
   *vecptr++ = HtmlEscapedContact;

   *vecptr++ = 40;
   *vecptr++ = 40;
   *vecptr++ = HtmlEscapedEmail;

   if ((AuthHtRecord.Flags & AUTH_FLAG_ENABLED) ||
       (AddUser && !AlreadyExists))
   {
      *vecptr++ = " CHECKED";
      *vecptr++ = "";
   }
   else
   {
      *vecptr++ = "";
      *vecptr++ = " CHECKED";
   }

   if ((AuthHtRecord.Flags & AUTH_FLAG_DELETE  ||
        AuthHtRecord.Flags & AUTH_FLAG_POST ||
        AuthHtRecord.Flags & AUTH_FLAG_PUT) &&
       AuthHtRecord.Flags & AUTH_FLAG_GET)
   {
      *vecptr++ = "";
      *vecptr++ = " CHECKED";
      *vecptr++ = "";
   }
   else
   if (AuthHtRecord.Flags & AUTH_FLAG_GET)
   {
      *vecptr++ = " CHECKED";
      *vecptr++ = "";
      *vecptr++ = "";
   }
   else
   {
      *vecptr++ = " CHECKED";
      *vecptr++ = "";
      *vecptr++ = "";
   }

   if (AuthHtRecord.Flags & AUTH_FLAG_HTTPS_ONLY)
   {
      *vecptr++ = " CHECKED";
      *vecptr++ = "";
   }
   else
   {
      *vecptr++ = "";
      *vecptr++ = " CHECKED";
   }

   *vecptr++ = MaxAuthPasswordLength;
   *vecptr++ = MaxAuthPasswordLength;
   *vecptr++ = MaxAuthPasswordLength;
   *vecptr++ = MaxAuthPasswordLength;

   if (AddUser && !AlreadyExists)
      *vecptr++ = "Add";
   else
      *vecptr++ = "Modify";

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWrite (rqptr, &HTAdminEnd, Buffer, Length);

   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
*/

HTAdminModifyUser
(
struct RequestStruct *rqptr,
boolean AddUser,
char *DatabaseName,
char *UserName,
char *FullName,
char *Contact,
char *Email,
char *Enabled,
char *Access,
char *HttpsOnly,
char *PasswordNew,
char *PasswordVerify
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>Success 200</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>SUCCESS!!</H1>\n\
<P>Reported by server.\n\
<P>Record for !AZ in !AZ at !AZ !AZ.\n\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register char  *cptr, *sptr;

   int  status;
   unsigned long  FaoVector [32];
   unsigned short  Length;
   unsigned long  HashedPwd [2];
   unsigned char  A1DigestLoCase [16],
                  A1DigestUpCase [16];
   char  Buffer [2048];
   struct AuthHtRecordStruct AuthHtRecord;
   struct AuthCacheRecordStruct  AuthCacheRecord;
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminModifyUserForm()\n");

   if (!rqptr->RemoteUser[0])
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_REQUIRED), FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /*******************/
   /* process request */
   /*******************/

   if (!DatabaseName[0] || !UserName[0] || !FullName[0] ||
       !Enabled[0] || !Access[0])
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorHTAdminInsufficient, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   if ((toupper(Enabled[0]) != 'Y' && toupper(Enabled[0]) != 'N') ||
       (!strsame(Access, "r", -1) && !strsame(Access, "r+w", -1) &&
        !strsame(Access, "w", -1)))
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, ErrorHTAdminParameter, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   if (!strsame (PasswordNew, PasswordVerify, -1))
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_HTADMIN_PWD_INCORRECT), FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   if (PasswordNew[0])
   {
      if (VMSnok (status =
          AuthGenerateHashPassword (UserName, PasswordNew, &HashedPwd)))
      {
         rqptr->ErrorTextPtr = "password hash";
         ErrorVmsStatus (rqptr, status, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }

      if (VMSnok (status =
          AuthGenerateDigestPassword (rqptr->AuthRealmPtr, rqptr->RemoteUser,
                                      PasswordNew, &A1DigestLoCase,
                                      &A1DigestUpCase)))
      {
         rqptr->ErrorTextPtr = "password digest";
         ErrorVmsStatus (rqptr, status, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
   }

   /***********************/
   /* update the database */
   /***********************/

   /* look for the record, leave the database file open if found */
   status = AuthAccessHtDatabase (true, DatabaseName, UserName,
                                  &AuthHtRecord, NULL, NULL);
   if (AddUser)
   {
      if (VMSok (status))
      {
         /* ensure the database is closed */
         AuthAccessHtDatabase (false, NULL, NULL, NULL, NULL, NULL);
         rqptr->ResponseStatusCode = 404;
         ErrorGeneral (rqptr, ErrorHTAdminUserExists, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
      if (status == SS$_INVLOGIN) status = SS$_NORMAL;
      memset (&AuthHtRecord, 0, sizeof(struct AuthHtRecordStruct));
   }
   else
   {
      if (status == SS$_INVLOGIN)
      {
         /* ensure the database is closed */
         AuthAccessHtDatabase (false, NULL, NULL, NULL, NULL, NULL);
         rqptr->ResponseStatusCode = 404;
         ErrorGeneral (rqptr, ErrorHTAdminUserNotFound, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
   }
   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = ErrorHTAdminDatabase;
      rqptr->ErrorHiddenTextPtr = DatabaseName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   memcpy (&AuthHtRecord.UserName, UserName, sizeof(AuthHtRecord.UserName));
   AuthHtRecord.UserNameLength = strlen(UserName);
   memcpy (&AuthHtRecord.FullName, FullName, sizeof(AuthHtRecord.FullName));
   memcpy (&AuthHtRecord.Contact, Contact, sizeof(AuthHtRecord.Contact));
   memcpy (&AuthHtRecord.Email, Email, sizeof(AuthHtRecord.Email));
   if (AddUser)
   {
      memcpy (&AuthHtRecord.AddedBinTime, &rqptr->BinaryTime, 8);
      AuthHtRecord.DatabaseVersion = AuthCurrentDatabaseVersion;
   }
   else
   {
      memcpy (&AuthHtRecord.LastChangeBinTime, &rqptr->BinaryTime, 8);
      AuthHtRecord.ChangeCount++;
   }
   if (PasswordNew[0])
   {
      memcpy (&AuthHtRecord.HashedPwd, &HashedPwd, 8);
      memcpy (&AuthHtRecord.A1DigestLoCase, &A1DigestLoCase, 16);
      memcpy (&AuthHtRecord.A1DigestUpCase, &A1DigestUpCase, 16);
   }

   if (toupper(Enabled[0]) == 'Y')
      AuthHtRecord.Flags |= AUTH_FLAG_ENABLED;
   else
      AuthHtRecord.Flags &= ~AUTH_FLAG_ENABLED;

   if (toupper(HttpsOnly[0]) == 'Y')
      AuthHtRecord.Flags |= AUTH_FLAG_HTTPS_ONLY;
   else
      AuthHtRecord.Flags &= ~AUTH_FLAG_HTTPS_ONLY;

   /* reset all the method bits to zero */
   AuthHtRecord.Flags &= ~(AUTH_FLAG_DELETE | AUTH_FLAG_GET |
                           AUTH_FLAG_HEAD | AUTH_FLAG_POST | AUTH_FLAG_PUT);
   /* now set the relevant method bits on */
   if (strsame (Access, "r", -1))
      AuthHtRecord.Flags |= (AUTH_FLAG_GET | AUTH_FLAG_HEAD);
   else
   if (strsame (Access, "r+w", -1))
      AuthHtRecord.Flags |= (AUTH_FLAG_DELETE | AUTH_FLAG_GET |
                             AUTH_FLAG_HEAD | AUTH_FLAG_POST | AUTH_FLAG_PUT);
   else
   if (strsame (Access, "w", -1))
      AuthHtRecord.Flags |= (AUTH_FLAG_DELETE | AUTH_FLAG_POST | AUTH_FLAG_PUT);

   /* add/update the record, close the database file */
   if (AddUser)
      status = AuthAccessHtDatabase (false, NULL, NULL,
                                     NULL, &AuthHtRecord, NULL);
   else
      status = AuthAccessHtDatabase (false, NULL, NULL,
                                     NULL, NULL, &AuthHtRecord);
   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = ErrorHTAdminDatabase;
      rqptr->ErrorHiddenTextPtr = DatabaseName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /**************/
   /* successful */
   /**************/

   if (AddUser)
      fprintf (stdout, "%%%s-I-AUTHADD, %s, %s in %s by %s@%s\n",
               Utility, DateTime(NULL,20),
               UserName, DatabaseName, rqptr->RemoteUser, rqptr->ClientHostName);
   else
      fprintf (stdout, "%%%s-I-AUTHMOD, %s, %s in %s by %s@%s\n",
               Utility, DateTime(NULL,20),
               UserName, DatabaseName, rqptr->RemoteUser, rqptr->ClientHostName);

   /* reset relevant entries in the cache */
   if (VMSnok (HTAdminCacheReset (rqptr, DatabaseName, UserName)))
      return;

   /*****************/
   /* HTTP response */
   /*****************/

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = UserName;
   *vecptr++ = DatabaseName;
   *vecptr++ = ServerHostPort;
   if (AddUser)
      *vecptr++ = "added";
   else
      *vecptr++ = "modified";

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, &HTAdminEnd, Buffer, Length);
   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
*/

HTAdminUserDeleteForm
(
struct RequestStruct *rqptr,
char *DatabaseName,
char *UserName
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Administer Authentication</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Administer Authentication</H2>\n\
\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=3><FONT COLOR=\"#ff0000\">DELETE</FONT> \
User !AZ in !AZ</TH></TR>\n\
<TR><TH COLSPAN=2>Full Name</TH><TD>!AZ</TD></TR>\n\
<TR><TH COLSPAN=2>Contact</TH><TD><PRE>!AZ</PRE></TD></TR>\n\
<TR><TD></TD></TR>\n\
<TR><TD><INPUT TYPE=submit VALUE=\" Delete User \"></TD></TR>\n\
</TABLE>\n\
\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [2048],
         HtmlEscapedContact [256];
   $DESCRIPTOR (BufferDsc, Buffer);
   struct AuthHtRecordStruct AuthHtRecord;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminUserDelete()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   if (!DatabaseName || !UserName[0])
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorHTAdminInsufficient, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /* look for the record, close the database file immediately */
   status = AuthAccessHtDatabase (false, DatabaseName, UserName,
                                  &AuthHtRecord, NULL, NULL);
   if (status == SS$_INVLOGIN)
   {
      rqptr->ResponseStatusCode = 404;
      ErrorGeneral (rqptr, ErrorHTAdminUserNotFound, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }
   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = ErrorHTAdminDatabase;
      rqptr->ErrorHiddenTextPtr = DatabaseName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /**************/
   /* begin page */
   /**************/

   if (CopyToHtml (HtmlEscapedContact, sizeof(HtmlEscapedContact),
                   AuthHtRecord.Contact, -1) < 0)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;

   *vecptr++ = AuthHtRecord.UserName;
   *vecptr++ = DatabaseName;
   *vecptr++ = AuthHtRecord.FullName;
   *vecptr++ = HtmlEscapedContact;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWrite (rqptr, &HTAdminEnd, Buffer, Length);

   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
*/

HTAdminUserDelete
(
struct RequestStruct *rqptr,
char *DatabaseName,
char *UserName
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>Success 200</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>SUCCESS!!</H1>\n\
<P>Reported by server.\n\
<P><FONT COLOR=\"#ff0000\">Deleted</FONT> record for !AZ in !AZ at !AZ.\n\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register char  *cptr, *sptr;

   int  status;
   unsigned long  FaoVector [32];
   unsigned short  Length;
   char  Buffer [2048];
   struct AuthHtRecordStruct AuthHtRecord;
   struct AuthCacheRecordStruct  AuthCacheRecord;
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminUserDelete()\n");

   if (!rqptr->RemoteUser[0])
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_REQUIRED), FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   if (!DatabaseName[0] || !UserName[0])
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorHTAdminInsufficient, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /* look for the record, leave the database file open if found */
   status = AuthAccessHtDatabase (true, DatabaseName, UserName,
                                  &AuthHtRecord, NULL, NULL);
   if (status == SS$_INVLOGIN)
   {
      /* ensure the database is closed */
      AuthAccessHtDatabase (false, NULL, NULL, NULL, NULL, NULL);
      rqptr->ResponseStatusCode = 404;
      ErrorGeneral (rqptr, ErrorHTAdminUserNotFound, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }
   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = ErrorHTAdminDatabase;
      rqptr->ErrorHiddenTextPtr = DatabaseName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   memset (&AuthHtRecord, 0, sizeof(struct AuthHtRecordStruct));

   /* update the now zeroed record */
   status = AuthAccessHtDatabase (false, NULL, NULL,
                                  NULL, NULL, &AuthHtRecord);
   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = ErrorHTAdminDatabase;
      rqptr->ErrorHiddenTextPtr = DatabaseName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /**************/
   /* successful */
   /**************/

   fprintf (stdout, "%%%s-I-AUTHDEL, %s, %s in %s by %s@%s\n",
            Utility, DateTime(NULL,20),
            UserName, DatabaseName, rqptr->RemoteUser, rqptr->ClientHostName);

   /* reset relevant entries in the cache */
   if (VMSnok (HTAdminCacheReset (rqptr, DatabaseName, UserName)))
      return;

   /*****************/
   /* HTTP response */
   /*****************/

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = UserName;
   *vecptr++ = DatabaseName;
   *vecptr++ = ServerHostPort;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, &HTAdminEnd, Buffer, Length);
   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
Form for a user to change their own realm password.  The function that
actually performs the change, AuthPasswordChange(), is located in module
Auth.c
*/

HTAdminPasswordChangeForm (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ</H1>\n\
<P><FORM METHOD=POST ACTION=\"!AZ\">\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>\"!AZ\" ... !AZ@!AZ</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH ALIGN=RIGHT>!AZ</TH>\
<TD ALIGN=LEFT><INPUT TYPE=password SIZE=!UL MAXLENGTH=!UL NAME=cu></TD></TR>\n\
<TR><TH ALIGN=RIGHT>!AZ</TH>\
<TD ALIGN=LEFT><INPUT TYPE=password SIZE=!UL MAXLENGTH=!UL NAME=p></TD></TR>\n\
<TR><TH ALIGN=RIGHT>!AZ</TH>\
<TD ALIGN=LEFT><INPUT TYPE=password SIZE=!UL MAXLENGTH=!UL NAME=v></TD></TR>\n\
<TR><TD COLSPAN=2><INPUT TYPE=submit VALUE=\" !AZ \">\n\
<INPUT TYPE=reset VALUE=\" !AZ \"></TD></TR>\n\
</TABLE>\n\
</TABLE>\n\
</FORM>\n\
</BODY>\n\
</HTML>\n");

   register char  *cptr;
   register unsigned long  *vecptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [2048],
         Scratch [256];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminPasswordChangeForm()\n");

   if (!rqptr->RemoteUser[0])
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_REQUIRED), FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   strcpy (cptr = Scratch, MsgFor(rqptr,MSG_HTADMIN_PWD_CHANGE));

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   /* "Change Authentication" x 2 */
   *vecptr++ = cptr;
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   *vecptr++ = rqptr->PathInfoPtr;
   *vecptr++ = rqptr->AuthRealmPtr;
   *vecptr++ = rqptr->RemoteUser;
   /* the user is dealing with the host/port they specified, not the base */
   *vecptr++ = rqptr->ServicePtr->ServerHostPort;

   /* "Current" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';
   *vecptr++ = MaxAuthPasswordLength;
   *vecptr++ = MaxAuthPasswordLength;

   /* "New" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';
   *vecptr++ = MaxAuthPasswordLength;
   *vecptr++ = MaxAuthPasswordLength;

   /* "Verify" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';
   *vecptr++ = MaxAuthPasswordLength;
   *vecptr++ = MaxAuthPasswordLength;

   /* "Change" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* "Reset" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, &HTAdminEnd, Buffer, Length);
   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
Changes a user's password in the on-disk HTA database or the SYSUAF database
(if realm is VMS).  User determined by 'rqptr->RemoteUser', database by
'rqptr->AuthRealmPtr'.  The authorization cache is then searched for all
entries for the username and that realm and the password reset forcing it to be
revalidated the next time it is accessed. The form this request is generated by
comes from AdminPasswordChangeForm().
*/

HTAdminPasswordChange
(
struct RequestStruct *rqptr,
char *PasswordCurrent,
char *PasswordNew,
char *PasswordVerify
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ 200</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ!!</H1>\n\
<P>!AZ\n\
<P>\"!AZ\" ... !AZ@!AZ\n\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;

   int  status;
   unsigned long  FaoVector [32];
   unsigned short  Length;
   unsigned long  HashedPwd [2];
   unsigned char  A1DigestLoCase [16],
                  A1DigestUpCase [16];
   char  Buffer [2048];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminPasswordChange()\n");

   if (!rqptr->RemoteUser[0])
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_REQUIRED), FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /*******************/
   /* process request */
   /*******************/

   if (!(PasswordCurrent[0] && PasswordNew[0] && PasswordVerify[0]))
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_HTADMIN_PWD_INCOMPLETE), FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   if (!strsame (PasswordNew, PasswordVerify, -1))
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_HTADMIN_PWD_VERIFY), FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   if (strsame (PasswordCurrent, PasswordNew, -1))
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_HTADMIN_PWD_IDENTICAL), FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   if (!strcmp (rqptr->AuthRealmPtr, AUTH_VMS))
   {
      /*********************/
      /* update the SYSUAF */
      /*********************/

      static unsigned long  Context = -1;

      unsigned long  UaiPriv [2],
                     HashedPwd [2],
                     UaiPwd [2];
      unsigned short  UaiSalt;
      unsigned char  UaiEncrypt;
      struct {
         short BufferLength;
         short ItemCode;
         void  *BufferPtr;
         void  *LengthPtr;
      } GetItems [] = 
      {
         { sizeof(UaiPwd), UAI$_PWD, &UaiPwd, 0 },
         { sizeof(UaiEncrypt), UAI$_ENCRYPT, &UaiEncrypt, 0 },
         { sizeof(UaiSalt), UAI$_SALT, &UaiSalt, 0 },
         { 0, 0, 0, 0 }
      },
        SetItems [] = 
      {
         { sizeof(HashedPwd), UAI$_PWD, &HashedPwd, 0 },
         { 0, 0, 0, 0 }
      };
      $DESCRIPTOR (UserNameDsc, rqptr->RemoteUser);
      $DESCRIPTOR (PasswordDsc, PasswordNew);

      /* double-check! */
      if (!AuthSysUafEnabled)
      {
         rqptr->ResponseStatusCode = 403;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_GENERAL_DISABLED), FI_LI);
         HTAdminEnd (rqptr);
         return;
      }

      /* bit more checking */
      if (AuthSysUafIdentifier && !rqptr->AuthCanChangeSysUafPwd)
      {
         /* SYSUAF password change only allowed with appropriate identifier */
         rqptr->ResponseStatusCode = 403;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_ACCESS_DENIED), FI_LI);
         HTAdminEnd (rqptr);
         return;
      }

      /* yet more checking! */
      if (!strsame (PasswordCurrent, rqptr->RemoteUserPassword, -1))
      {
         rqptr->ResponseStatusCode = 403;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_HTADMIN_PWD_INCORRECT), FI_LI);
         HTAdminEnd (rqptr);
         return;
      }

      /* turn on SYSPRV to allow access to SYSUAF records */
      EnableSysPrv ();

      UserNameDsc.dsc$w_length = strlen(rqptr->RemoteUser);
      status = sys$getuai (0, &Context, &UserNameDsc, &GetItems, 0, 0, 0);
      if (Debug) fprintf (stdout, "sys$getuai() %%X%08.08X\n", status);

      /* turn off SYSPRV */
      DisableSysPrv ();

      if (VMSnok (status)) 
      {
         rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_AUTH_USER);
         ErrorVmsStatus (rqptr, status, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }

      PasswordDsc.dsc$w_length = strlen(PasswordNew);
      status = sys$hash_password (&PasswordDsc, UaiEncrypt,
                                  UaiSalt, &UserNameDsc, &HashedPwd);
      if (Debug) fprintf (stdout, "sys$hash_password() %%X%08.08X\n", status);
      if (VMSnok (status))
      {
         rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_AUTH_USER);
         ErrorVmsStatus (rqptr, status, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }

      /* turn on SYSPRV to allow access to SYSUAF records */
      EnableSysPrv ();

      status = sys$setuai (0, 0, &UserNameDsc, &SetItems, 0, 0, 0);
      if (Debug) fprintf (stdout, "sys$setuai() %%X%08.08X\n", status);

      /* turn off SYSPRV */
      DisableSysPrv ();

      if (VMSnok (status)) 
      {
         rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_AUTH_USER);
         ErrorVmsStatus (rqptr, status, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
   }
   else
   {
      /***************************/
      /* update the HTA database */
      /***************************/

      struct AuthHtRecordStruct AuthHtRecord;
      struct AuthCacheRecordStruct  AuthCacheRecord;

      /* check the correct current password has been supplied */
      if (!strsame (PasswordCurrent, rqptr->RemoteUserPassword, -1))
      {
         rqptr->ResponseStatusCode = 403;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_HTADMIN_PWD_INCORRECT), FI_LI);
         HTAdminEnd (rqptr);
         return;
      }

      if (VMSnok (status =
          AuthGenerateHashPassword (rqptr->RemoteUser, PasswordNew,
                                    &HashedPwd)))
      {
         rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_HTADMIN_PWD_ERROR);
         ErrorVmsStatus (rqptr, status, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }

      if (VMSnok (status =
          AuthGenerateDigestPassword (rqptr->AuthRealmPtr, rqptr->RemoteUser,
                                      PasswordNew, &A1DigestLoCase,
                                      &A1DigestUpCase)))
      {
         rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_HTADMIN_PWD_ERROR);
         ErrorVmsStatus (rqptr, status, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }

      /* look for the record, leave the database file open if found */
      status = AuthAccessHtDatabase (true, rqptr->AuthRealmPtr,
                                     rqptr->RemoteUser,
                                     &AuthHtRecord, NULL, NULL);
      if (status == SS$_INVLOGIN)
      {
         rqptr->ResponseStatusCode = 404;
         rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_HTADMIN_PWD_NOT_FOUND);
         HTAdminEnd (rqptr);
         return;
      }
      if (VMSnok (status))
      {
         rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_HTADMIN_DATABASE);
         rqptr->ErrorHiddenTextPtr = rqptr->AuthRealmPtr;
         ErrorVmsStatus (rqptr, status, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }

      memcpy (&AuthHtRecord.HashedPwd, &HashedPwd, 8);
      memcpy (&AuthHtRecord.A1DigestLoCase, &A1DigestLoCase, 16);
      memcpy (&AuthHtRecord.A1DigestUpCase, &A1DigestUpCase, 16);
      memcpy (&AuthHtRecord.LastChangeBinTime, &rqptr->BinaryTime, 8);
      AuthHtRecord.ChangeCount++;

      /* update the record, close the database file */
      status = AuthAccessHtDatabase (false, NULL, NULL, NULL, NULL,
                                     &AuthHtRecord);
      if (VMSnok (status))
      {
         rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_HTADMIN_DATABASE);
         rqptr->ErrorHiddenTextPtr = rqptr->AuthRealmPtr;
         ErrorVmsStatus (rqptr, status, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
   }

   /**************/
   /* successful */
   /**************/

   /* report this to the log */
   fprintf (stdout, "%%%s-I-PASSWORD, %s, %s in %s from %s\n",
            Utility, DateTime(NULL,20),
            rqptr->RemoteUser, rqptr->AuthRealmPtr, rqptr->ClientHostName);

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = MsgFor(rqptr,MSG_STATUS_SUCCESS);
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_SUCCESS);
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_REPORTED_BY_SERVER);
   *vecptr++ = rqptr->AuthRealmPtr;
   *vecptr++ = rqptr->RemoteUser;
   /* the user is dealing with the host/port they specified, not the base */
   *vecptr++ = rqptr->ServicePtr->ServerHostPort;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, &HTAdminEnd, Buffer, Length);
   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
Reset an entry in the authentication cache. The cache is searched for all
entries for the username and realm with the failure count and password reset
forcing it to be revalidated the next time it is accessed.
*/

HTAdminCacheResetThis
(
struct RequestStruct *rqptr,
char *DatabaseName,
char *UserName
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>Success 200</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>SUCCESS!!</H1>\n\
<P>Reported by server.\n\
<P>Authentication cache !AZ in !AZ at !AZ !AZ.\n\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register char  *cptr, *qptr, *sptr, *zptr;

   int  status;
   unsigned long  FaoVector [32];
   unsigned short  Length;
   char  Buffer [2048];
   $DESCRIPTOR (BufferDsc, Buffer);
   struct AuthCacheRecordStruct  AuthCacheRecord;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminCacheResetThis()\n");

   if (!rqptr->RemoteUser[0])
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_REQUIRED), FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   if (DatabaseName[0] || UserName[0])
   {
      /* reset relevant entries in the cache */
      if (VMSnok (HTAdminCacheReset (rqptr, DatabaseName, UserName)))
         return;
      cptr = "reset";
   }
   else
   {
      /* purge (release memory) for the entire cache */
      if (VMSnok (status = AuthCacheTreeFree ()))
      {
         rqptr->ErrorTextPtr = ErrorHTAdminPurgeCache;
         ErrorVmsStatus (rqptr, status, FI_LI);
         HTAdminEnd (rqptr);
         return;
      }
      cptr = "purged";
   }

   if (!UserName[0]) UserName = "*";
   if (!DatabaseName[0]) DatabaseName = "*";

   /* report this to the log */
   fprintf (stdout, "%%%s-I-AUTHCACHE, %s, %s in %s from %s@%s\n",
            Utility, DateTime(NULL,20), UserName, DatabaseName,
            rqptr->RemoteUser, rqptr->ClientHostName);

   /*****************/
   /* HTTP response */
   /*****************/

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = UserName;
   *vecptr++ = DatabaseName;
   *vecptr++ = ServerHostPort;
   *vecptr++ = cptr;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, &HTAdminEnd, Buffer, Length);
   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
Traverse the authentication cache tree reseting the password and failure count
for all entries with a matching realm and user name.  This effectively causes
the username to be revalidated next authorized path access.
*/

HTAdminCacheReset
(
struct RequestStruct *rqptr,
char *DatabaseName,
char *UserName
)
{
   register char  *cptr, *sptr;

   int  status;
   struct AuthHtRecordStruct AuthHtRecord;
   struct AuthCacheRecordStruct  AuthCacheRecord;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminCacheReset()\n");

   /* we know these'll fit, the size have been check during authentication */
   cptr = DatabaseName;
   sptr = AuthCacheRecord.Realm;
   while (*cptr) *sptr++ = *cptr++;
   *sptr = '\0';
   AuthCacheRecord.RealmLength = sptr - AuthCacheRecord.Realm;

   cptr = UserName;
   sptr = AuthCacheRecord.UserName;
   while (*cptr) *sptr++ = *cptr++;
   *sptr = '\0';
   AuthCacheRecord.UserNameLength = sptr - AuthCacheRecord.UserName;

   status = lib$traverse_tree (&AuthCacheTreeHead,
                               &HTAdminCacheResetEntry,
                               &AuthCacheRecord);
   if (Debug) fprintf (stdout, "lib$traverse_tree() %%X%08.08X\n", status);

   if (VMSnok (status) && status != LIB$_KEYNOTFOU)
   {
      rqptr->ErrorTextPtr = "lib$traverse_tree()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return (status);
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Called for each node by lib$traverse_tree() int other functions.  Reset the
password and failure count for each matching realm and username (group is not
checked so may be multiple entries if user has membership of multiple groups
and used those paths).  If realm or username in empty then that is a wildcard
match for that parameter, hence all users in a given realm can be reset, all
realms for a given user, or all users in all realms!
*/ 

int HTAdminCacheResetEntry
(
struct AuthCacheRecordStruct *TreeNodePtr,
struct AuthCacheRecordStruct *AuthCacheRecordPtr
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "AuthCacheTreeEntryReset() |%s|%s|%s|%s|\n",
               TreeNodePtr->Realm, TreeNodePtr->UserName,
               AuthCacheRecordPtr->Realm, AuthCacheRecordPtr->UserName);

   if (AuthCacheRecordPtr->RealmLength &&
       AuthCacheRecordPtr->RealmLength != TreeNodePtr->RealmLength)
      return (SS$_NORMAL);
   if (AuthCacheRecordPtr->UserNameLength &&
       AuthCacheRecordPtr->UserNameLength != TreeNodePtr->UserNameLength)
      return (SS$_NORMAL);
   if (AuthCacheRecordPtr->RealmLength &&
       strcmp (AuthCacheRecordPtr->Realm, TreeNodePtr->Realm))
      return (SS$_NORMAL);
   if (AuthCacheRecordPtr->UserNameLength &&
       strcmp (AuthCacheRecordPtr->UserName, TreeNodePtr->UserName))
      return (SS$_NORMAL);

   /* realm and username match (or wildcarded) */
   if (Debug) fprintf (stdout, "reset password\n");
   TreeNodePtr->Password[0]  = '\0';
   TreeNodePtr->FailureCount = 0;
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Set string text according to capability bits in on-disk HTA database.  These
may be different to the bits in the authorization capability vector, reported
by AuthCanString().
*/

char* HTAdminCanString
(
struct RequestStruct *rqptr,
unsigned long CanFlags,
boolean Brief
)
{
   static $DESCRIPTOR (CanBriefFaoDsc, "!AZ\0");
   static $DESCRIPTOR (CanFullFaoDsc, "!AZ!AZ!AZ!AZ!AZ!AZ!AZ!AZ\0");
   static char  Buffer [128];
   static $DESCRIPTOR (BufferDsc, Buffer);

   register unsigned long  *vecptr;

   int  status;
   unsigned long  FaoVector [32];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminCanString()\n");

   vecptr = FaoVector;

   if ((CanFlags & AUTH_FLAG_DELETE ||
        CanFlags & AUTH_FLAG_POST ||
        CanFlags & AUTH_FLAG_PUT) &&
       CanFlags & AUTH_FLAG_GET)
      *vecptr++ = "read <B>+ write</B>";
   else
   if ((CanFlags & AUTH_FLAG_DELETE ||
        CanFlags & AUTH_FLAG_POST ||
        CanFlags & AUTH_FLAG_PUT))
      *vecptr++ = "write-only";
   else
   if (CanFlags & AUTH_FLAG_GET)
      *vecptr++ = "read-only";
   else
      *vecptr++ = "<I>none!</I>";

   if (Brief)
      status = sys$faol (&CanBriefFaoDsc, 0, &BufferDsc, &FaoVector);
   else
   {
      if (CanFlags & AUTH_FLAG_DELETE ||
          CanFlags & AUTH_FLAG_GET ||
          CanFlags & AUTH_FLAG_HEAD ||
          CanFlags & AUTH_FLAG_POST ||
          CanFlags & AUTH_FLAG_PUT)
         *vecptr++ = " <FONT SIZE=1><NOBR>( ";
      else
         *vecptr++ = "";
      if (CanFlags & AUTH_FLAG_DELETE)
         *vecptr++ = " DELETE";
      else
         *vecptr++ = "";
      if (CanFlags & AUTH_FLAG_GET)
         *vecptr++ = " GET";
      else
         *vecptr++ = "";
      if (CanFlags & AUTH_FLAG_HEAD)
         *vecptr++ = " HEAD";
      else
         *vecptr++ = "";
      if (CanFlags & AUTH_FLAG_POST)
         *vecptr++ = " POST";
      else
         *vecptr++ = "";
      if (CanFlags & AUTH_FLAG_PUT)
         *vecptr++ = " PUT";
      else
         *vecptr++ = "";
      if (CanFlags & AUTH_FLAG_DELETE ||
          CanFlags & AUTH_FLAG_GET ||
          CanFlags & AUTH_FLAG_HEAD ||
          CanFlags & AUTH_FLAG_POST ||
          CanFlags & AUTH_FLAG_PUT)
         *vecptr++ = " )</NOBR></FONT>";
      else
         *vecptr++ = "";

      status = sys$faol (&CanFullFaoDsc, 0, &BufferDsc, &FaoVector);
   }

   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (NULL);
   }
   return (Buffer);
}

/*****************************************************************************/
/*
*/

HTAdminDatabaseCreateForm
(
struct RequestStruct *rqptr,
char *DatabaseName
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Administer Authentication</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Administer Authentication</H2>\n\
\
<FORM METHOD=POST ACTION=\"!AZ\">\n\
<INPUT TYPE=hidden NAME=do VALUE=htacreate>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH><FONT COLOR=\"#ff0000\">Create Database</FONT> !AZ</TH></TR>\n\
<TR><TD></TD></TR>\n\
<TR><TD ALIGN=center><INPUT TYPE=submit VALUE=\" Create \"></TD></TR>\n\
</TABLE>\n\
</FORM>\n\
\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [2048];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminDatabaseCreateForm()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   if (!DatabaseName)
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorHTAdminInsufficient, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;

   *vecptr++ = rqptr->PathInfoPtr;
   *vecptr++ = DatabaseName;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWrite (rqptr, &HTAdminEnd, Buffer, Length);

   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
*/

HTAdminDatabaseCreate
(
struct RequestStruct *rqptr,
char *DatabaseName,
char *UserName
)
{
   static $DESCRIPTOR (AuthFileNameFaoDsc, "!AZ!AZ!AZ");

   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>Success 200</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>SUCCESS!!</H1>\n\
<P>Reported by server.\n\
<P>Created !AZdatabase !AZ at !AZ.\n\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register char  *cptr, *sptr;

   int  status,
        EraseCount,
        SetPrvStatus;
   unsigned long  FaoVector [32];
   unsigned short  AuthFileNameLength,
                   Length;
   char  AuthFileName [64],
         Buffer [2048],
         ExpandedFileName [256];
   struct AuthHtRecordStruct AuthHtRecord;
   $DESCRIPTOR (AuthFileNameDsc, AuthFileName);
   $DESCRIPTOR (BufferDsc, Buffer);
   struct FAB  AuthFileFab;
   struct RAB  AuthFileRab;
   struct NAM  AuthFileNam;
   struct XABPRO  AuthFileXabPro;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminDatabaseCreate()\n");

   if (!rqptr->RemoteUser[0])
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_REQUIRED), FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   if (!DatabaseName[0])
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorHTAdminInsufficient, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   sys$fao (&AuthFileNameFaoDsc, &AuthFileNameLength, &AuthFileNameDsc,
            HTA_DIRECTORY, DatabaseName, HTA_FILE_TYPE);
   AuthFileName[AuthFileNameLength] = '\0';

   AuthFileFab = cc$rms_fab;
   AuthFileFab.fab$l_fna = AuthFileName;  
   AuthFileFab.fab$b_fns = AuthFileNameLength;
   AuthFileFab.fab$l_nam = &AuthFileNam;

   AuthFileNam = cc$rms_nam;
   AuthFileNam.nam$l_esa = ExpandedFileName;
   AuthFileNam.nam$b_ess = sizeof(ExpandedFileName)-1;
   
   if (VMSnok (status = sys$parse (&AuthFileFab, 0, 0)))
      exit (status);

   status = sys$search (&AuthFileFab, 0, 0);

   /* ensure parse internal data structures are released */
   AuthFileFab.fab$l_fna = "a:[b]c.d;";
   AuthFileFab.fab$b_fns = 9;
   AuthFileFab.fab$b_dns = 0;
   AuthFileNam.nam$b_nop = NAM$M_SYNCHK;
   sys$parse (&AuthFileFab, 0, 0);

   if (VMSnok (status) && status != RMS$_FNF)
   {
      rqptr->ErrorTextPtr = ErrorHTAdminDatabase;
      rqptr->ErrorHiddenTextPtr = DatabaseName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }
   else
   if (VMSok (status))
   {
      ErrorGeneral (rqptr, ErrorHTAdminDatabaseExists, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   AuthFileFab = cc$rms_fab;
   AuthFileFab.fab$l_fna = AuthFileName;  
   AuthFileFab.fab$b_fns = AuthFileNameLength;
   AuthFileFab.fab$l_fop = FAB$M_SQO;
   AuthFileFab.fab$w_mrs = sizeof(struct AuthHtRecordStruct);
   AuthFileFab.fab$l_nam = &AuthFileNam;
   AuthFileFab.fab$b_rfm = FAB$C_FIX;
   AuthFileFab.fab$l_xab = &AuthFileXabPro;

   AuthFileNam = cc$rms_nam;
   AuthFileNam.nam$l_esa = ExpandedFileName;
   AuthFileNam.nam$b_ess = sizeof(ExpandedFileName)-1;
   
   AuthFileXabPro = cc$rms_xabpro;
   /* ownded by SYSTEM ([1,4]) */
   AuthFileXabPro.xab$w_grp = 1;
   AuthFileXabPro.xab$w_mbm = 4;
   AuthFileXabPro.xab$l_nxt = 0;
   /* w:,g:,o:rwed,s:rwed */
   AuthFileXabPro.xab$w_pro = 0xff00;

   /* turn on SYSPRV to allow creation of database file */
   EnableSysPrv ();
   status = sys$create (&AuthFileFab, 0, 0);
   DisableSysPrv ();

    /* sys$create() status */
   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = ErrorHTAdminDatabase;
      rqptr->ErrorHiddenTextPtr = DatabaseName;
      ErrorVmsStatus (rqptr, AuthFileFab.fab$l_stv, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   sys$close (&AuthFileFab, 0, 0);

   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = ErrorHTAdminDatabase;
      rqptr->ErrorHiddenTextPtr = DatabaseName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /**************/
   /* successful */
   /**************/

   fprintf (stdout, "%%%s-I-AUTHDBCRE, %s, database %s by %s@%s\n",
            Utility, DateTime(NULL,20),
            DatabaseName, rqptr->RemoteUser, rqptr->ClientHostName);

   /*****************/
   /* HTTP response */
   /*****************/

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   if (AuthFileNam.nam$l_fnb & NAM$M_LOWVER)
      *vecptr++ = "<FONT COLOR=\"#ff0000\">new version</FONT> of ";
   else
      *vecptr++ = "";
   *vecptr++ = DatabaseName;
   *vecptr++ = ServerHostPort;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, &HTAdminEnd, Buffer, Length);
   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
*/

HTAdminDatabaseDeleteForm
(
struct RequestStruct *rqptr,
char *DatabaseName
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Administer Authentication</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Administer Authentication</H2>\n\
\
<FORM METHOD=POST ACTION=\"!AZ\">\n\
<INPUT TYPE=hidden NAME=do VALUE=htadelete>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH><FONT COLOR=\"#ff0000\">DELETE Database</FONT> !AZ</TH></TR>\n\
<TR><TD></TD></TR>\n\
<TR><TD ALIGN=CENTER><INPUT TYPE=submit VALUE=\" Delete \"></TD></TR>\n\
</TABLE>\n\
<FORM>\n\
\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register struct HTAdminTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [2048];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminDatabaseDeleteForm()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   if (!DatabaseName)
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorHTAdminInsufficient, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;

   *vecptr++ = rqptr->PathInfoPtr;
   *vecptr++ = DatabaseName;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWrite (rqptr, &HTAdminEnd, Buffer, Length);

   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
*/

HTAdminDatabaseDelete
(
struct RequestStruct *rqptr,
char *DatabaseName,
char *UserName
)
{
   static $DESCRIPTOR (AuthFileNameFaoDsc, "!AZ!AZ!AZ");

   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>Success 200</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>SUCCESS!!</H1>\n\
<P>Reported by server.\n\
<P><FONT COLOR=\"#ff0000\">Deleted</FONT> database !AZ at !AZ.\n\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register char  *cptr, *sptr;

   int  status,
        EraseCount,
        SetPrvStatus;
   unsigned long  FaoVector [32];
   unsigned short  AuthFileNameLength,
                   Length;
   char  AuthFileName [64],
         Buffer [2048],
         ExpandedFileName [256];
   $DESCRIPTOR (BufferDsc, Buffer);
   $DESCRIPTOR (AuthFileNameDsc, AuthFileName);
   struct FAB  AuthFileFab;
   struct NAM  AuthFileNam;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminDatabaseDelete()\n");

   if (!rqptr->RemoteUser[0])
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_AUTH_REQUIRED), FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   if (!DatabaseName[0])
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, ErrorHTAdminInsufficient, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   sys$fao (&AuthFileNameFaoDsc, &AuthFileNameLength, &AuthFileNameDsc,
            HTA_DIRECTORY, DatabaseName, HTA_FILE_TYPE);
   AuthFileName[AuthFileNameLength] = '\0';

   AuthFileFab = cc$rms_fab;
   AuthFileFab.fab$l_fna = AuthFileName;  
   AuthFileFab.fab$b_fns = AuthFileNameLength;
   AuthFileFab.fab$l_nam = &AuthFileNam;
   AuthFileNam = cc$rms_nam;
   AuthFileNam.nam$l_esa = ExpandedFileName;
   AuthFileNam.nam$b_ess = sizeof(ExpandedFileName)-1;
   
   if (VMSok (status = sys$parse (&AuthFileFab, 0, 0)))
   {
      *AuthFileNam.nam$l_ver = '\0'; 
      if (Debug) fprintf (stdout, "ExpandedFileName |%s|\n", ExpandedFileName);

      /* turn on SYSPRV to allow deletion of database file */
      EnableSysPrv ();

      EraseCount = 0;
      while (VMSok (status = sys$erase (&AuthFileFab, 0, 0)))
          EraseCount++;
      if (status == RMS$_FNF && EraseCount) status = SS$_NORMAL;

      DisableSysPrv ();
   }

   /* ensure parse internal data structures are released */
   AuthFileFab.fab$l_fna = "a:[b]c.d;";
   AuthFileFab.fab$b_fns = 9;
   AuthFileFab.fab$b_dns = 0;
   AuthFileNam.nam$b_nop = NAM$M_SYNCHK;
   sys$parse (&AuthFileFab, 0, 0);

   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = ErrorHTAdminDatabase;
      rqptr->ErrorHiddenTextPtr = DatabaseName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   /**************/
   /* successful */
   /**************/

   fprintf (stdout, "%%%s-I-AUTHDBDEL, %s, database %s by %s@%s\n",
            Utility, DateTime(NULL,20),
            DatabaseName, rqptr->RemoteUser, rqptr->ClientHostName);

   /* reset relevant entries in the cache */
   if (VMSnok (HTAdminCacheReset (rqptr, DatabaseName, "")))
      return;

   /*****************/
   /* HTTP response */
   /*****************/

   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      HTAdminEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = DatabaseName;
   *vecptr++ = ServerHostPort;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, &HTAdminEnd, Buffer, Length);
   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
*/

HTAdminOpenDatabaseForRead
(
struct RequestStruct *rqptr,
char *DatabaseName
)
{
   static $DESCRIPTOR (AuthFileNameFaoDsc, "!AZ!AZ!AZ");

   register struct HTAdminTaskStruct  *tkptr;

   int  status,
        SetPrvStatus;
   unsigned short  Length;
   $DESCRIPTOR (AuthFileNameDsc, "");

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HTAdminOpenDatabaseForRead()\n");

   tkptr = rqptr->HTAdminTaskPtr;

   AuthFileNameDsc.dsc$a_pointer = tkptr->AuthFileName;
   AuthFileNameDsc.dsc$w_length = sizeof(tkptr->AuthFileName)-1;

   status = sys$fao (&AuthFileNameFaoDsc, &Length, &AuthFileNameDsc,
                     HTA_DIRECTORY, DatabaseName, HTA_FILE_TYPE);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$fao()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return (status & 0xfffffffe);
   }
   tkptr->AuthFileName[Length] = '\0';
   if (Debug) fprintf (stdout, "AuthFileName |%s|\n", tkptr->AuthFileName);

   tkptr->FileFab = cc$rms_fab;
   tkptr->FileFab.fab$b_fac = FAB$M_GET;
   tkptr->FileFab.fab$l_fna = tkptr->AuthFileName;  
   tkptr->FileFab.fab$b_fns = Length;
   tkptr->FileFab.fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT | FAB$M_SHRUPD;

   /* turn on SYSPRV to allow access to authentication Database file */
   EnableSysPrv ();
   status = sys$open (&tkptr->FileFab, 0, 0);
   DisableSysPrv ();

   /* status from sys$open() */
   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->AuthFileName);
      rqptr->ErrorHiddenTextPtr = tkptr->AuthFileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return (status);
   }

   tkptr->ExpandedFileName[tkptr->FileNam.nam$b_rsl] = '\0';
   if (Debug)
      fprintf (stdout, "ExpandedFileName |%s|\n", tkptr->ExpandedFileName);

   tkptr->FileRab = cc$rms_rab;
   tkptr->FileRab.rab$l_ctx = rqptr;
   tkptr->FileRab.rab$l_fab = &tkptr->FileFab;
   tkptr->FileRab.rab$b_mbf = 2;
   tkptr->FileRab.rab$l_rop = RAB$M_RAH | RAB$M_ASY;
   tkptr->FileRab.rab$l_ubf = &tkptr->AuthHtRecord;
   tkptr->FileRab.rab$w_usz = sizeof(struct AuthHtRecordStruct);

   if (VMSnok (status = sys$connect (&tkptr->FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->AuthFileName);
      rqptr->ErrorHiddenTextPtr = tkptr->AuthFileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      HTAdminEnd (rqptr);
      return (status);
   }

   return (status);
}

/****************************************************************************/

