/*****************************************************************************/
/*
                                  HTTPd.c

WASD VMS Hypertext Transfer Protocol daemon.


OBLIGATORY COPYRIGHT NOTICE
---------------------------
WASD VMS Hypertext Services, Copyright (C) 1996,1998 Mark G.Daniel.
(prior to 01-JUL-1997, "HFRD VMS Hypertext Services")

This package is free software; you can redistribute it and/or modify it under 
the terms of the GNU General Public License as published by the Free Software 
Foundation; version 2 of the License, or any later version. 

This package is distributed in the hope that it will be useful, but WITHOUT 
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS 
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more 
details. 

You should have received a copy of the GNU General Public License along with 
this package; if not, write to the Free Software Foundation, Inc., 675 Mass 
Ave, Cambridge, MA 02139, USA. 


QUALIFIERS  (implemented in the CLI.c module)
----------
/ACCEPT=                comma separated list of accepted hosts/domains
/AUTHORIZATION=SSL      authorization may only be used with "https:"
/CGI_PREFIX=            prefix to CGI-script variable (symbol) names
/DBUG                   turn on all "if (Debug)" statements (if compiled DBUG)
/DO=                    command to be passed to server
/FILBUF=                number of bytes in record buffer
/FORMAT=                log record format
/REJECT=                comma separated list of rejected hosts/domains
/[NO]LOG[=]             logging enabled/disabled, optional log file name
/[NO]MONITOR            monitor logicals enabled/disabled
/NETBUF=                number of bytes in network read buffer
/OUTBUF=                number of bytes in output buffer
/PERIOD=                log file period
/PORT=                  server IP port number
/PRIORITY=              process priority
/[NO]PROFILE            allow/disallow SYSUAF-authenticated access control
/[NO]PROMISCUOUS[=pwd]  authenticates any user name for path authorization
                        (optional password required for authorization) 
/SERVICES=              list of [host-name:]port providing HTTP service
/SOFTWAREID=            overrides the server's software ID string
/[NO]SSL=               enables Secure Sockets Layer, sets protocol version(s)
/SUBBUF=                number of bytes in subprocess's SYS$OUTPUT buffer
/[NO]SYSUAF[=ID|SSL]    allow/disallow SYSUAF authentication, or
                        SYSUAF authentication is allowed by identifier, or
                        SYSUAF authentication is only allowed with "https:"
/[NO]SWAP               allow/disallow process to be swapped out
/VERSION                simply display the server version
/[NO]ZERO               accounting zeroed on startup (default is non-zeroed)


VERSION HISTORY
---------------
29-AUG-98  MGD  v5.2.0, reuse DECnet task connections,
                        allow specified hosts exclusion from logging,
                        stream-LF conversion only on specified paths,
                        bugfix; SYS$TIMEZONE_DIFFERENTIAL processing
                        bugfix; DECnet tasks not aborted at timeout
07-JUL-98  MGD  v5.1.0, add eXtended Server Side Includes processing,
                        design-problem; modify CGIplus script rundown,
                        SYSUAF authentication by identifier,
                        per-service logging,
                        rqptr->TimerTerminated (chasing occasional lib$get_vm()
                        %LIB-F-BADLOADR around connection expiry termination)
20-DEC-97  MGD  v5.0.0, optional Secure Sockets Layer (using SSLeay),
                        DECnet-based scripting including OSU emulation,
                        miscellaneous revisions and "improvements"
07-JAN-97  MGD  v4.5.2, bugfixes; record-mode file transfer, activity graph
06-DEC-97  MGD  v4.5.1, resolving a suspected inconsistent AST delivery
                        situation by requiring all $QIO()s with an AST routine
                        to ensure any queueing errors, etc. are reported via
                        the AST routine, by an explicit $DCLAST() ... this
                        removes some ambiguity about how $QIO() returns should
                        be handled ... drastic, but desperate times, etc.
                        (a more consistent and desirable model anyway :^)
02-NOV-97  MGD  v4.5.0, file cache,
                        logging periods,
                        HttpdSupervisor(),
                        configurable script run-time environments,
                        additional request header fields
18-OCT-97  MGD  v4.4.1, bugfixes; duration, logging period
01-OCT-97  MGD  v4.4.0, message module,
                        conditional rule mapping,
                        SYSUAF-authenticated user access control,
                        multi-homed/multi-port services
                        (some NETLIB supported packages now cannot DNS lookup),
                        echo and Xray internal scripts,
                        extensions to logging functionality,
                        additional command-line server control,
                        bugfix; redirection loop detection
01-AUG-97  MGD  v4.3.0, MadGoat NETLIB broadens TCP/IP package support,
                        server activity report
16-JUL-97  MGD  v4.2.2, bugfix; WORLD realm and access list
07-JUL-97  MGD  v4.2.1, minimum heap allocation chunk size,
                        prevent keep-alive timeout redefining request logical
01-JUL-97  MGD  v4.2.0, change name to WASD (Wide Area Surveillance Division)
                        persistant DCL subprocesses and CGIplus 
                        (see re-written DCL.C module),
                        scripting and client reports,
                        potential multi-thread problems in reports fixed
27-MAR-97  MGD  v4.1.0, rationalized HTTP response header generation;
                        delete on close for "temporary" files, to support
                        UPD module "preview" functionality ... WARNING, any
                        file with a name component comprising a leading hyphen,
                        sixteen digits and a trailing hyphen will be deleted!
01-FEB-97  MGD  v4.0.0, HTTPd version 4
01-OCT-96  MGD  v3.4.0, extended server reporting  
01-AUG-96  MGD  v3.3.0, realm/path-based authorization;
                        BASIC and DIGEST authentication;
                        PUT(/POST/DELETE) module;
                        StmLf module (variable to stream-LF file conversion)
12-APR-96  MGD  v3.2.0, file record/binary now determined by record format;
                        persistent connections ("Keep-Alive" within HTTP/1.0);
                        moved RMS parse structures into thread data;
                        improved local redirection detection;
                        observed Multinet disconnection/zero-byte behaviour
                        (request now aborts if network read returns zero bytes)
15-FEB-96  MGD  v3.1.1, fixed rediculous :^( bug in 302 HTTP header;
                        minor changes to request accounting and server report;
                        minor changes for user directory support;
                        minor changes to error reporting
03-JAN-96  MGD  v3.1.0, support for both DEC TCP/IP Services and TGV MultiNet
01-DEC-95  MGD  v3.0.0, single heap for each thread's dynamic memory management;
                        extensive rework of DCL subprocess functionality;
                        HTML pre-processsing module (aka Server Side Includes);
                        NCSA/CERN compliant image-mapping module;
                        NetWriteBuffered() for improving network IO;
                        miscellaneous reworks/rewrites
27-SEP-95  MGD  v2.3.0, carriage-control on non-header records from <CR><LF> 
                        to single <LF> ('\n' ... newline); some browsers expect
                        only this (e.g. Netscape 1.n was spitting on X-bitmaps)
                        added Greenwich Mean Time time-stamp functionality;
                        added 'Referer:', 'If-Modified-Since:', 'User-Agent:'
07-AUG-95  MGD  v2.2.2, optionally include commented VMS file specifications
                        in HTML documents and VMS-style directory listings
16-JUN-95  MGD  v2.2.1, added file type description to "Index of" (directory)
24-MAY-95  MGD  v2.2.0, minor changes to allow compilation on AXP platform
03-APR-95  MGD  v2.1.0, add SYSUAF authentication, POST method handling
20-DEC-94  MGD  v2.0.0, multi-threaded version
20-JUN-94  MGD  v1.0.0, single-threaded version
*/
/*****************************************************************************/

/* remember to keep 'HttpdAccountingVersion' in step with this version */
#define HttpdVersion "5.2.0"

/* used to check when accounting requires zeroing on change of version */
#define HttpdAccountingVersion 0x0520

#ifdef __ALPHA
#  define HttpdArch "AXP"
#else
#  define HttpdArch "VAX"
#endif

/* standard C header files */
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <descrip.h>
#include <jpidef.h>
#include <libdef.h>
#include <prvdef.h>
#include <psldef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application-related header files */
#include "wasd.h"
#include "cli.h"
#include "copyright.h"
#include "error.h"
#include "graph.h"
#include "httpd.h"
#include "logging.h"
#include "mapurl.h"
#include "msg.h"
#include "net.h"
#include "request.h"
#include "support.h"
#include "vm.h"

/******************/
/* global storage */
/******************/

/*
   These are the required privileges of the executing HTTP server.
   The server ACCOUNT should only have TMPMBX and NETMBX (just for
   extra security, policy ... keep privileged accounts to a minimum).
   The image should be installed with SYSPRV, PRMMBX, PSWAPM and SYSNAM.
   Subprocesses are spawned only with the process's current privileges,
   which are always maintained at TMPMBX and NETMBX.  If additional
   privileges are required for any particular purpose (e.g. binding to
   a privileged IP port) then they are enabled, the action performed,
   and then they are disabled again immediately.
*/
unsigned long  AvJoePrivMask [2] = { PRV$M_NETMBX | PRV$M_TMPMBX, 0 },
               ResetPrivMask [2] = { 0xffffffff, 0xffffffff };

char  Utility [] = "HTTPD";

/*
   Decided to be able to eliminate the debug statements from production
   executables completely.  This will eliminate some largely unused code
   from the images reducing the overall file size, but more importantly
   will eliminate the test and branch execution overhead each time a debug
   statement is encountered.  Do this by conditionally turning the integer
   Debug storage into a constant false value ... compiler optimization of
   an impossible-to-execute section of code does the rest!  Very little
   other code needs to be explicitly conditionally compiled.
*/
#ifdef DBUG
boolean  Debug;
#else
#define Debug 0
#endif

boolean  MonitorEnabled,
         NaturalLanguageEnglish,
         NoSwapOut = true;

int  DclSysOutputSize = DefaultDclSysOutputSize,
     ExitStatus,
     FileBufferSize = DefaultFileBufferSize,
     NetReadBufferSize = DefaultNetReadBufferSize,
     OutputBufferSize = DefaultOutputBufferSize,
     ProcessPriority = 4,
     RejectHostsLength,
     ServerPort = 80;

int  CurrentConnectCount,
     MaxConcurrentConnections = DefaultMaxConcurrentConnections;

char  BuildInfo [64],
      CommandLine [256],
      HttpdUserName [13],
      NaturalLanguage [64],
      ServerPortString [16],
      SoftwareID [256];

char  *AcceptHostsPtr = NULL,
      *RejectHostsPtr = NULL;

struct AnExitHandler  ExitHandler;

struct AccountingStruct  Accounting;
boolean  AccountingZeroOnStartup;

#ifdef DBUG
   USEC_TIMER_STORAGE
#endif

/********************/
/* external storage */
/********************/

extern boolean  AuthorizationEnabled;
extern boolean  CliMonitorDisabled;
extern boolean  CliMonitorEnabled;
extern boolean  CliVersion;
extern boolean  ControlExitRequested;
extern boolean  ControlRestartRequested;
extern boolean  LoggingEnabled;
extern boolean  ProtocolHttpsConfigured;
extern int  CliServerPort;
extern int  RequestHistoryMax;
extern int  ServerPort;
extern char  *CliAcceptHostsPtr;
extern char  *CliRejectHostsPtr;
extern char  CliLogFileName[];
extern char  ControlBuffer[];
extern char  ErrorSanityCheck[];
extern char  HttpdIpPackage[];
extern char  HttpdSesola[];
extern char  TimeGmtString[];
extern struct ConfigStruct  Config;
extern struct MsgDbStruct  MsgDb;
extern struct ListHeadStruct  RequestList;

/* for initial debugging only */
extern unsigned long  VmCacheVmZoneId,
                      VmGeneralVmZoneId,
                      VmRequestVmZoneId;

/*****************************************************************************/
/*
*/

int main ()

{
   static unsigned long  AltpriMask [2] = { PRV$M_ALTPRI, 0 },
                         PswapmMask [2] = { PRV$M_PSWAPM, 0 };

   static struct {
      unsigned short  BufferLength;
      unsigned short  ItemCode;
      unsigned long  BufferAddress;
      unsigned long  ReturnLengthAddress;
   } JpiItem [] =
       { { sizeof(HttpdUserName)-1, JPI$_USERNAME, HttpdUserName, 0 },
         { 0, 0, 0, 0 } };

   register char  *cptr;

   int  status;
   unsigned short  Length;
   char  TimeScratch [32],
         ProcessName [16];
   $DESCRIPTOR (NaturalLanguageDsc, NaturalLanguage);
   $DESCRIPTOR (ProcessNameDsc, ProcessName);
   $DESCRIPTOR (ProcessNameFaoDsc, "HTTPd:!UL");

   /*********/
   /* begin */
   /*********/

   if (!SoftwareID[0])
      sprintf (SoftwareID, "HTTPd-WASD/%s %OpenVMS/%s %s%s",
               HttpdVersion, HttpdArch, HttpdIpPackage, HttpdSesola);

#ifdef __DECC
   sprintf (BuildInfo, "%s %5.5s VMS %s DECC %d",
            __DATE__, __TIME__, __VMS_VERSION+1, __DECC_VER);
#else
   sprintf (BuildInfo, "%s %5.5s VMS %s VAXC ?",
            __DATE__, __TIME__, VMS_VERSION+1);
#endif

   if (VMSnok (status = ParseCommandLine ()))
      exit (status);

#ifdef DBUG
   if (!Debug)
      if (getenv ("HTTPD$DBUG") != NULL)
         Debug = true;
#endif

   if (ControlBuffer[0])
   {
      /*********************************************************/
      /* this image is being used to control the HTTPd process */
      /*********************************************************/

      if (CliServerPort) ServerPort = CliServerPort;
      if (ServerPort < 1 || ServerPort > 65535)
         ErrorExitVmsStatus (0, "IP port", FI_LI);
      exit (ControlCommand (ControlBuffer));
   }

   /****************/
   /* HTTPd server */
   /****************/

   /* output the GNU GENERAL PUBLIC LICENSE message */
   fprintf (stdout, "%%%s-I-SOFTWAREID, %s\n%s",
            Utility, SoftwareID, CopyRightMessageBrief);

   if (CliVersion) exit (SS$_NORMAL);

   /* initialize general-use virtual memory management */
   VmGeneralInit ();
   /** if (Debug) VmDebug (VmGeneralVmZoneId); **/

   /* get the username the server is executing under */
   if (VMSnok (status = sys$getjpi (0, 0, 0, &JpiItem, 0, 0, 0)))
      ErrorExitVmsStatus (status, "sys$getjpi()", FI_LI);
   HttpdUserName[12] = '\0';
   for (cptr = HttpdUserName; *cptr && !ISLWS(*cptr); cptr++);
   *cptr = '\0';
   if (Debug) fprintf (stdout, "HttpdUserName |%s|\n", HttpdUserName);

   /* make sure the process' privileges are those of a mere mortal */
   status = sys$setprv (0, &ResetPrivMask, 0, 0);
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "sys$setprv()", FI_LI);
   status = sys$setprv (1, &AvJoePrivMask, 0, 0);
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "sys$setprv()", FI_LI);

   /* set process priority */
   if (VMSnok (status = sys$setprv (1, &AltpriMask, 0, 0)))
      ErrorExitVmsStatus (status, "sys$setprv()", FI_LI);
   if (VMSnok (status = sys$setpri (0, 0, ProcessPriority, 0, 0, 0)))
      ErrorExitVmsStatus (status, "sys$setpri()", FI_LI);
   if (VMSnok (status = sys$setprv (0, &AltpriMask, 0, 0)))
      ErrorExitVmsStatus (status, "sys$setprv()", FI_LI);

   /* read the server configuration file */
   Configure (&Config);

   /* anywhere near before starting logging! */
   NetGetServerHostName ();

   if (Config.Busy) MaxConcurrentConnections = Config.Busy;

   if (CliServerPort)
      ServerPort = CliServerPort;
   else
   if (Config.ServerPort)
      ServerPort = Config.ServerPort;
   if (ServerPort < 1 || ServerPort > 65535)
      ErrorExitVmsStatus (0, "IP port", FI_LI);
   sprintf (ServerPortString, "%d", ServerPort);

   /* initialize Secure Sockets Layer, if available and if required */
   SeSoLaInit ();

   if (CliMonitorDisabled)
      MonitorEnabled = false;
   else
   if (Config.MonitorEnabled || CliMonitorEnabled)
      MonitorEnabled = true;
   else
      MonitorEnabled = false;

   if (CliAcceptHostsPtr != NULL)
      Config.AcceptHostsPtr = CliAcceptHostsPtr;

   if (CliRejectHostsPtr != NULL)
      Config.RejectHostsPtr = CliRejectHostsPtr;

   /* get and set the service names and ports */
   NetGetService ();

   /* set the process name */
   if (VMSnok (status =
       sys$fao (&ProcessNameFaoDsc, &Length, &ProcessNameDsc, ServerPort)))
      ErrorExitVmsStatus (status, "sys$fao()", FI_LI);
   ProcessName[ProcessNameDsc.dsc$w_length = Length] = '\0';
   if (Debug) fprintf (stdout, "ProcessName |%s|\n", ProcessName);
   if (VMSnok (status = sys$setprn (&ProcessNameDsc)))
      ErrorExitVmsStatus (status, "sys$setprn()", FI_LI);

   /* ensure the GMT time/logical is available */
   if (VMSnok (status = TimeSetGmt ()))
      ErrorExitVmsStatus (status, "TimeSetGmt()", FI_LI);
   fprintf (stdout, "%%%s-I-GMT, %s\n", Utility, TimeGmtString);

   /* initialize message database */
   MsgLoad (&MsgDb);

   /* authentication/authorization configuration */
   AuthPathConfig (NULL);

   /* load the rule mapping database */
   MapUrl_Map (NULL, NULL, NULL, NULL, NULL);

   /* initialize DCL processing */
   DclInit ();

   /* initialize request virtual memory management */
   VmRequestInit (MaxConcurrentConnections); 
   /** if (Debug) VmDebug (VmRequestVmZoneId); **/

   /* initialize file cache  */
   CacheInit (true);
   /** if (Debug) VmDebug (VmCacheVmZoneId); **/

   /** if (Debug) VmDebug (0); **/

   /* initialize activity statistics */
   GraphActivityInit ();

   /* initialize the language environment */
   if (VMSok (status = lib$get_users_language (&NaturalLanguageDsc)))
   {
      for (cptr = NaturalLanguage; *cptr && !ISLWS(*cptr); cptr++);
      *cptr = '\0';
      if (strsame (NaturalLanguage, "ENGLISH"))
         NaturalLanguageEnglish = true;
      else
      {
         NaturalLanguageEnglish = false;
         fprintf (stdout, "%%%s-I-LANGUAGE, natural language is %s\n",
                  Utility, NaturalLanguage);
      }
   }
   else
   {
      if (status == LIB$_ENGLUSED)
      {
         NaturalLanguageEnglish = true;
         strcpy (NaturalLanguage, "ENGLISH");
         fprintf (stdout,
            "%%%s-W-LANGUAGE, natural language has defaulted to %s\n",
            Utility, NaturalLanguage);
      }
      else
         ErrorExitVmsStatus (status, "lib$get_users_language()", FI_LI);
   }

   /* reset logical names provided to the HTTPd monitor utility */
   if (MonitorEnabled)
      if (VMSnok (status = DefineMonitorLogicals (NULL)))
         ErrorExitVmsStatus (status, "DefineMonitorLogicals()", FI_LI);

   /* must be done after DefineMonitorLogicals(), which reads previous values */
   if (AccountingZeroOnStartup ||
       HttpdAccountingVersion != Accounting.AccountingVersion ||
       !MonitorEnabled)
   {
      /* initialize accounting information without reseting the logicals */
      ZeroAccounting ();
   }
   Accounting.AccountingVersion = HttpdAccountingVersion;

   /* set up to allow the HTTPD process to be controlled */
   if (VMSnok (status = ControlHttpd ()))
      ErrorExitVmsStatus (status, "ControlHttpd()", FI_LI);

   /* initialize request history mechanism (ensure it's reasonable!) */
   RequestHistoryMax = Config.RequestHistory;
   if (RequestHistoryMax > 999) RequestHistoryMax = 0;

   /* disable process swapping */
   if (NoSwapOut)
   {
      if (VMSnok (status = sys$setprv (1, &PswapmMask, 0, 0)))
         ErrorExitVmsStatus (status, "sys$setprv()", FI_LI);
      if (VMSnok (status = sys$setswm (1)))
         ErrorExitVmsStatus (status, "sys$setswm()", FI_LI);
      if (VMSnok (status = sys$setprv (0, &PswapmMask, 0, 0)))
         ErrorExitVmsStatus (status, "sys$setprv()", FI_LI);
   }

   /* set up and declare the exit handler */
   ExitHandler.HandlerAddress = &HttpdExit;
   ExitHandler.ArgCount = 1;
   ExitHandler.ExitStatusPtr = &ExitStatus;
   if (VMSnok (status = sys$dclexh (&ExitHandler)))
      ErrorExitVmsStatus (status, "sys$dclexh()", FI_LI);

   /* disable AST's until we've created the listening sockets */
   sys$setast (0);

   /* need SYSPRV to bind to a well-known port */
   EnableSysPrv ();

   /* create network services */
   NetCreateService ();

   /* turn off SYSPRV after binding to well-known port */
   DisableSysPrv ();

   /* initialize logging after service creation (in case of per-service logs) */
   if (VMSnok (status = Logging (NULL, LOGGING_BEGIN)))
      ErrorExitVmsStatus (status, "Logging()", FI_LI);

   /*****************************/
   /* begin to process requests */
   /*****************************/

   /* reenable user-mode ASTs to allow use of our services */
   sys$setast (1);

   sys$hiber ();

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Exit the HTTP server, via this declared exit handler.
*/

HttpdExit ()

{
   if (Debug) fprintf (stdout, "HttpdExit() %%X%08.08X\n", ExitStatus);

   /* run down any subprocesses associated with script processing */
   DclExit ();

   if (LoggingEnabled)
   {
      /* provide a server exit entry */
      Logging (NULL, LOGGING_END);
   }

   if (MonitorEnabled)
   {
      /* set exit status in logical names used by HTTPd monitor utility */
      Accounting.LastExitStatus = ExitStatus;
      DefineMonitorLogicals (NULL);
   }
}

/*****************************************************************************/
/*
As the code gets more complex it's becoming increasingly possible a coding or
design error will leave privileges turned on somewhere. To help detect any such
problem ensure everthing's off occasionlly (when there are no more connections
to process).
*/

#ifdef __DECC
#pragma inline(HttpdCheckPriv)
#endif

int HttpdCheckPriv ()

{
   static long  Pid = 0;
   static unsigned long  JpiCurPriv [2];
   static struct
   {
      unsigned short  buf_len;
      unsigned short  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   }
      JpiItems [] =
   {
      { sizeof(JpiCurPriv), JPI$_CURPRIV, &JpiCurPriv, 0 },
      {0,0,0,0}
   };

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpdCheckPriv()\n");

   if (VMSnok (status = sys$getjpiw (0, &Pid, 0, &JpiItems, 0, 0, 0)))
      ErrorExitVmsStatus (status, "sys$getjpi()", FI_LI);

   if ((JpiCurPriv[0] & ~AvJoePrivMask[0]) || JpiCurPriv[1])
   {
      if (Debug)
         fprintf (stdout, "%08.08X%08.08X\n", JpiCurPriv[1], JpiCurPriv[0]);
      ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
   }
}

/*****************************************************************************/
/*
Set the timer counters for the request according to the function code. 
HttpdSupervisor() will monitor these counters.  Due to the way the seconds are
counted-down, and the setting of a timer may be in the middle of any one
second interval, increment very small periods to ensure they are at least that.
*/

int HttpdTimerSet
(
struct RequestStruct *rqptr,
int Function
)
{
   static boolean  Initialize = true;
   static unsigned long  InputSeconds,
                         OutputSeconds,
                         KeepAliveSeconds;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpdTimerSet() %d\n", Function);

   if (Initialize)
   {
      Initialize = false;
      if (Config.InputTimeoutMinutes)
         InputSeconds = Config.InputTimeoutMinutes * 60;
      else
         InputSeconds = DefaultInputTimeoutMinutes * 60;
      if (Config.OutputTimeoutMinutes)
         OutputSeconds = Config.OutputTimeoutMinutes * 60;
      else
         OutputSeconds = DefaultOutputTimeoutMinutes * 60;
      if (Config.KeepAliveTimeoutSeconds)
         KeepAliveSeconds = Config.KeepAliveTimeoutSeconds;
      else
         KeepAliveSeconds = DefaultKeepAliveTimeoutSeconds;
      /* ensure small periods are at least that duration */
      if (KeepAliveSeconds && KeepAliveSeconds < 10) KeepAliveSeconds++;
   }

   switch (Function)
   {
      case TIMER_INPUT :
         rqptr->TimerOutputCount = rqptr->TimerKeepAliveCount = 0;
         rqptr->TimerInputCount = InputSeconds;
         return;

      case TIMER_OUTPUT :
         rqptr->TimerInputCount = rqptr->TimerKeepAliveCount = 0;
         rqptr->TimerOutputCount = OutputSeconds;
         return;

      case TIMER_KEEPALIVE :
         rqptr->TimerInputCount = rqptr->TimerOutputCount = 0;
         rqptr->TimerKeepAliveCount = KeepAliveSeconds;
         return;

      default :
         ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
   }
   rqptr->TimerTerminated = false;
}

/*****************************************************************************/
/*
Whenever at least one request is being processed this function is called every
second to supervise the requests' progress.  Scanning the list of current
processes it decrements set timer counters.  When a counter reaches zero
network I/O or DCL processing is cancelled.  This results in the request being
terminated.

This function was introduced to address a suspected multi-thread timing problem
associated with setting and cancelling independent timers and ASTs for each
request.  It is probably no more expensive than setting and reseting VMS
timers for each request's processing, and does seem to give some measure (or
feeling of) more structure and control!
*/

HttpdSupervisor (boolean TimerExpired)

{
   static unsigned long  OneSecondDelta [2] = { -10000000, -1 };

   static boolean  TimerSet = false;

   register struct RequestStruct  *rqeptr;
   register struct ListEntryStruct  *leptr;

   int  status;
   struct AnIOsb  IOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpdSupervisor() %d\n", TimerExpired);

   if (Debug)
      fprintf (stdout, "Head %d\nTail %d\n",
               RequestList.HeadPtr, RequestList.TailPtr);

   if (!TimerExpired)
   {
      /* new request, if the supervisor is already running just return */
      if (TimerSet) return;

      /* kick the supervisor timer into life, and then return */
      if (VMSnok (status =
          sys$setimr (0, &OneSecondDelta, &HttpdSupervisor, true, 0)))
         ErrorExitVmsStatus (status, "sys$setimr()", FI_LI);
      TimerSet = true;
      return;
   }

   /* must have got here via the supervisor's timer expiring */
   TimerSet = false;

   /* if nothing in the current request list return without sys$setimr() */
   if ((leptr = RequestList.HeadPtr) == NULL) return;

   /* process the current request list entries */
   for (/* from above */; leptr != NULL; leptr = leptr->NextPtr)
   {
      rqeptr = (struct RequestStruct*)leptr;

      if (Debug)
         fprintf (stdout, "%d <- %d -> %d (%d) in: %d out: %d alive: %d\n",
                  leptr->PrevPtr, leptr, leptr->NextPtr, rqeptr,
                  rqeptr->TimerInputCount, rqeptr->TimerOutputCount,
                  rqeptr->TimerKeepAliveCount);

      /* ordered in the most common occurance of timings */
      if (rqeptr->TimerOutputCount)
         if (--rqeptr->TimerOutputCount)
            continue;
         else;
      else
      if (rqeptr->TimerKeepAliveCount)
         if (--rqeptr->TimerKeepAliveCount)
            continue;
         else;
      else
      if (rqeptr->TimerInputCount)
         if (--rqeptr->TimerInputCount)
            continue;

      if (Debug) fprintf (stdout, "rqeptr %d\n", rqeptr);

      if (!rqeptr->TimerTerminated)
      {
         if (Debug)
            fprintf (stdout, "terminate %d %d %d\n",
                     rqeptr, rqeptr->DclTaskPtr, rqeptr->DECnetTaskPtr);

         rqeptr->TimerTerminated = true;
         if (rqeptr->DclTaskPtr != NULL)
         {
            /* DCL involved, do any DCL-specific cleanup */
            DclConcludeTask (rqeptr->DclTaskPtr, true);
         }
         else
         if (rqeptr->DECnetTaskPtr != NULL)
         {
            /* DECnet (CGI or OSU) involved */
            DECnetEnd (rqeptr);
         }
         else
         {
            /* this should shake the connection up */
            if (rqeptr->ClientSocketShutdown)
               NetCloseSocket (rqeptr);
            else
               NetShutdownSocket (rqeptr, 0, true);
         }
      }
   }

   if (VMSnok (status =
       sys$setimr (0, &OneSecondDelta, &HttpdSupervisor, true, 0)))
      ErrorExitVmsStatus (status, "sys$setimr()", FI_LI);

   TimerSet = true;
}

/****************************************************************************/

