/*****************************************************************************/
/*
                                 IsMap.c

Image-mapping module, integrated into the HTTPd (at no great cost), removing 
the overhead of image mapping via a script.

This module is designed to be a single task within a request.  Multiple IsMap
tasks within the thread are not supported, and the IsMap module does not call
any other tasks.  Essentially the IsMap functionality provides all the
response required for an image mapping request.

It relies on the function RequestEnd() in httpd.c to do the actual
redirection, either locally, or to generate a response providing an absolute
path to the client. 

All of the "smarts" for this application have been plagiarized from the NCSA 
imagemap.c application (v1.8).  The routines directly lifted from that program 
are grouped together at the end of this code module.  Due acknowlegement to 
the original authors and maintainers.  Any copyright over sections of that 
code is also acknowleged: 

  ** mapper 1.2
  ** 7/26/93 Kevin Hughes, kevinh@pulua.hcc.hawaii.edu
  ** "macmartinized" polygon code copyright 1992 by Eric Haines, erich@eye.com

Up until this module was put together WASD was reliant on a pre-compiled 
version of the CERN htimage.exe program used as a script.  Having this gives a 
certain independence, functionality integrated with the HTTP server, 
documented code (have a look at imagemap.c!), and better error reporting. 

This application allows the image configuration file to be specified in either 
CERN and NCSA format (or a mix of the two for that matter).  It is slightly 
more efficient to use the NCSA format, and this is suggested anyway, but the 
CERN format is provided for backward compatibility.  The region type keywords 
must supply at least 3 characters (although four is generally peferable).  
Comments can be prefixed by '#' or '!' characters, and blank lines are 
ignored.  Lines may be continued by making a '\' character the last on the
line.  The maximum line length is 255 characters.

As an extension to both of the NCSA and CERN mapping applications, this HTTPd 
can accept and process server-side relative paths.   Hence it not only 
processes full URL's like "http://host/area/document.html", local absolute 
URL's like "/area/document.html", but also local relative URLs such as 
"../area/document.html", etc.  This provides for greater ease of mobility for 
mapped documents. 


NCSA FORMAT
-----------
default url
circle url x,y x,y              # centre-point then edge-point
point url x,y
polygon url x,y x,y [x,y ...]
rectangle url x,y x,y


CERN FORMAT
-----------
default url
circle (x,y) r url              # centre-point then radius
point (x,y) url
polygon (x,y) (x,y) [(x,y) ...] url
rectangle (x,y) (x,y) url


VERSION HISTORY
---------------
19-AUG-98  MGD  allow for continued lines
17-AUG-97  MGD  message database,
                SYSUAF-authenticated users security-profile
27-FEB-97  MGD  delete on close for "temporary" files
01-FEB-97  MGD  HTTPd version 4
01-DEC-95  MGD  new for HTTPd version 3
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <math.h>

/* VMS related header files */
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application-related header files */
#include "wasd.h"
#include "error.h"
#include "httpd.h"
#include "ismap.h"
#include "mapurl.h"
#include "msg.h"
#include "net.h"
#include "support.h"
#include "vm.h"

#define IsMapBufferSize 1024
#define MaxPathSize 512

/* required by the NCSA mapping functions, used by this module */
#define MAXVERTS 100
#define X 0
#define Y 1

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern char  SoftwareID[];
extern struct AccountingStruct  Accounting;
extern struct MsgStruct  Msgs;

/*****************************************************************************/
/*
Open the specified file.  If there is a problem opening it then report it and 
dispose of the thread.  Once open and connected the image mapping becomes
AST-driven.
*/

IsMapBegin
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
char *FileName
)
{
   register char  *cptr, *sptr, *zptr;
   register struct IsMapTaskStruct  *tkptr;

   int  status,
        Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "IsMapBegin() |%s|\n", FileName);

   if (!rqptr->AccountingDone)
      rqptr->AccountingDone = ++Accounting.DoIsMapCount;

   /* set up the task structure (only ever one per request!) */
   rqptr->IsMapTaskPtr = tkptr = (struct IsMapTaskStruct*)
      VmGetHeap (rqptr, sizeof(struct IsMapTaskStruct));
   tkptr->NextTaskFunction = NextTaskFunction;

   cptr = FileName;
   zptr = (sptr = tkptr->FileName) + sizeof(tkptr->FileName);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      IsMapEnd (rqptr);
      return;
   }
   *sptr = '\0';
   tkptr->FileNameLength = sptr - tkptr->FileName;

   /*****************************************/
   /* check VMS-authenticated user's access */
   /*****************************************/

   if (rqptr->AuthVmsUserProfileLength)
   {
      status = AuthCheckVmsUserAccess (rqptr, tkptr->FileName,
                                       tkptr->FileNameLength);
      if (status == RMS$_PRV)
         tkptr->AuthVmsUserHasAccess = false;
      else
      if (VMSok (status))
         tkptr->AuthVmsUserHasAccess = true;
      else
      {
         /* error reported by access check */
         IsMapEnd (rqptr);
         return;
      }
   }
   else
      tkptr->AuthVmsUserHasAccess = false;

   /*************/
   /* open file */
   /*************/

   tkptr->FileFab = cc$rms_fab;
   tkptr->FileFab.fab$b_fac = FAB$M_GET;
   tkptr->FileFab.fab$l_fna = tkptr->FileName;  
   tkptr->FileFab.fab$b_fns = tkptr->FileNameLength;
   tkptr->FileFab.fab$b_shr = FAB$M_SHRGET;

   /* "temporary" file, automatic delete on closing it */
   if (rqptr->DeleteOnClose) tkptr->FileFab.fab$l_fop |= FAB$M_DLT;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   status = sys$open (&tkptr->FileFab, 0, 0);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      IsMapEnd (rqptr);
      return;
   }

   tkptr->FileRab = cc$rms_rab;
   tkptr->FileRab.rab$l_fab = &tkptr->FileFab;
   /* 2 buffers and read ahead performance option */
   tkptr->FileRab.rab$b_mbf = 2;
   tkptr->FileRab.rab$l_rop = RAB$M_RAH;

   if (VMSnok (status = sys$connect (&tkptr->FileRab, 0, 0)))
   {
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      IsMapEnd (rqptr);
      return;
   }

   /********************/
   /* begin processing */
   /********************/

   /* get the coordinate of the mouse-click */
   cptr = rqptr->QueryStringPtr;
   while (ISLWS(*cptr)) cptr++;
   if (Debug) fprintf (stdout, "cptr |%s|\n", cptr);
   if (isdigit(*cptr))
   {
      tkptr->IsMapClickCoord[X] = (double)atoi(cptr);
      while (*cptr && isdigit(*cptr)) cptr++;
      while (ISLWS(*cptr) || *cptr == ',') cptr++;
      if (isdigit(*cptr))
         tkptr->IsMapClickCoord[Y] = (double)atoi(cptr);
      else
      {
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_ISMAP_COORDINATE), FI_LI);
         IsMapEnd (rqptr);
         return;
      }
   }
   else
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_ISMAP_COORDINATE), FI_LI);
      IsMapEnd (rqptr);
      return;
   }
   if (Debug)
      fprintf (stdout, "IsMapClickCoord |%f|%f|\n",
               tkptr->IsMapClickCoord[X], tkptr->IsMapClickCoord[Y]);

   /* allocate heap memory to accomodate the largest allowable path */
   rqptr->LocationPtr = VmGetHeap (rqptr, MaxPathSize+1);

   /* make it an empty string to indicate no default path specified (yet) */
   rqptr->LocationPtr[0] = '\0';

   tkptr->IsMapCharNumber = tkptr->IsMapLineNumber = 0;
   tkptr->IsMapClosestPoint = HUGE_VAL;

   /* set then RAB user context storage to the client thread pointer */
   tkptr->FileRab.rab$l_ctx = rqptr;

   /* queue the first record read */
   tkptr->ReadBuffer[0] = '\0';
   tkptr->FileRab.rab$l_ubf = tkptr->ReadBuffer;
   tkptr->FileRab.rab$w_usz = sizeof(tkptr->ReadBuffer)-1;
   sys$get (&tkptr->FileRab, &IsMapNextRecordAST, &IsMapNextRecordAST);
}

/*****************************************************************************/
/*
Ensure the source file is closed.  Conclude the connection, which will cause 
any local redirection to be effected, or send a redirection header to the 
client.
*/ 

IsMapEnd (struct RequestStruct *rqptr)

{
   register struct IsMapTaskStruct  *tkptr;

   int  status,
        SetPrvStatus;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "IsMapEnd()\n");

   /* get the pointer to the task structure from the thread storage */
   tkptr = rqptr->IsMapTaskPtr;

   if (Debug) fprintf (stdout, "LocationPtr |%s|\n", rqptr->LocationPtr);

   /* if an error was generated then forget any redirection */
   if (rqptr->ErrorMessagePtr) rqptr->LocationPtr = NULL;

   if (tkptr->FileFab.fab$w_ifi)
   {
      if (rqptr->DeleteOnClose)
      {
         /* use SYSPRV to ensure deletion-on-close of file */
         EnableSysPrv ();
         sys$close (&tkptr->FileFab, 0, 0);
         DisableSysPrv ();
      }
      else
         sys$close (&tkptr->FileFab, 0, 0);
   }

   SysDclAst (tkptr->NextTaskFunction, rqptr);
}

/*****************************************************************************/
/*
The asynchronous read of the next image-mapping configuration file record has 
completed.  Call a function to parse the configuration line.  Either end the 
task or queue another read depending on the results of the parse.
*/

IsMapNextRecordAST (struct RAB *RabPtr)

{
   register struct RequestStruct  *rqptr;
   register struct IsMapTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "IsMapNextRecordAST() sts: %%X%08.08X stv: %%X%08.08X\n",
               RabPtr->rab$l_sts,
               RabPtr->rab$l_stv);

   /* get the pointer to the thread from the RAB user context storage */
   rqptr = RabPtr->rab$l_ctx;
   /* get the pointer to the task structure from the thread storage */
   tkptr = rqptr->IsMapTaskPtr;

   if (VMSnok (tkptr->FileRab.rab$l_sts))
   {
      if (tkptr->FileRab.rab$l_sts == RMS$_EOF)
      {
         /***************/
         /* end of file */
         /***************/

         /* must be the last part of a continued line */
         if (tkptr->ReadBuffer[0]) IsMapParseLine (rqptr);

         if (!rqptr->LocationPtr[0] && rqptr->ErrorMessagePtr == NULL)
            IsMapExplainError (rqptr, MsgFor(rqptr,MSG_ISMAP_DEFAULT), FI_LI);

         IsMapEnd (rqptr);
         return;
      }

      /**********************/
      /* error reading file */
      /**********************/

      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, tkptr->FileRab.rab$l_sts, FI_LI);
      IsMapEnd (rqptr);
      return;
   }

   /*****************************/
   /* process the record (line) */
   /*****************************/

   /* terminate the line */
   tkptr->FileRab.rab$l_ubf[tkptr->FileRab.rab$w_rsz] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", tkptr->FileRab.rab$l_ubf);

   if (tkptr->FileRab.rab$w_rsz)
   {
      if (tkptr->FileRab.rab$l_ubf[tkptr->FileRab.rab$w_rsz-1] == '\\')
      {
         /* continued line, queue a read for more */
         tkptr->FileRab.rab$l_ubf += tkptr->FileRab.rab$w_rsz-1;
         tkptr->FileRab.rab$w_usz -= tkptr->FileRab.rab$w_rsz-1;
         sys$get (&tkptr->FileRab, &IsMapNextRecordAST, &IsMapNextRecordAST);
         return;
      }
   }

   IsMapParseLine (rqptr);

   if (Debug) fprintf (stdout, "LocationPtr |%s|\n", rqptr->LocationPtr);

   if (!tkptr->IsMapLineNumber || rqptr->ErrorMessagePtr != NULL)
   {
      /*
         Stop processing immediately because:
         The line number has reset to zero (when a successful mapping occured).
         An error has been reported.
      */
      IsMapEnd (rqptr);
   }
   else
   {
      /* queue the next record read */
      tkptr->ReadBuffer[0] = '\0';
      tkptr->FileRab.rab$l_ubf = tkptr->ReadBuffer;
      tkptr->FileRab.rab$w_usz = sizeof(tkptr->ReadBuffer)-1;
      sys$get (&tkptr->FileRab, &IsMapNextRecordAST, &IsMapNextRecordAST);
   }

   /*
      Return to the AST-interrupted routine now.  Continued processing
      is event-driven.  Routine completion ASTs drive the image-mapping.
   */
}

/*****************************************************************************/
/*
Parse the configuration file record (line) for mapping directives in either 
the NCSA or CERN format.  If a mapping successful reset the line number 
storage to zero to indicate this.
*/ 

IsMapParseLine (struct RequestStruct* rqptr)

{
   register int  ccnt;
   register char  *bptr, *cptr, *sptr, *zptr;
   register struct IsMapTaskStruct  *tkptr;

   boolean  CernCompliant;
   int  status,
        PathSize;
   double  Distance;
   double  CoordArray [MAXVERTS][2];
   char  Path [MaxPathSize+1],
         RegionKeyword [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "IsMapParseLine() |%s|\n",
               rqptr->IsMapTaskPtr->ReadBuffer);

   /* get the pointer to the task structure from the thread storage */
   tkptr = rqptr->IsMapTaskPtr;

   CernCompliant = false;
   PathSize = 0;

   tkptr->IsMapLineNumber++;

   bptr = tkptr->ReadBuffer;

   while (ISLWS(*bptr)) bptr++;
   if (!*bptr || *bptr == '#' || *bptr == '!') return;

   /**********************/
   /* get region keyword */
   /**********************/

   zptr = (sptr = RegionKeyword) + sizeof(RegionKeyword);
   while (*bptr && !ISLWS(*bptr) && sptr < zptr)
      *sptr++ = tolower(*bptr++);
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      return;
   }
   *sptr = '\0';
   if (Debug) fprintf (stdout, "RegionKeyword |%s|\n", RegionKeyword);

   if (!(!memcmp (RegionKeyword, "cir", 3) ||
         !memcmp (RegionKeyword, "poi", 3) ||
         !memcmp (RegionKeyword, "pol", 3) ||
         !memcmp (RegionKeyword, "rec", 3) ||
         !memcmp (RegionKeyword, "def", 3)))
   {
      IsMapExplainError (rqptr, MsgFor(rqptr,MSG_ISMAP_REGION), FI_LI);
      return;
   }

   while (ISLWS(*bptr)) bptr++;
   if (!*bptr || *bptr == '#' || *bptr == '!')
   {
      tkptr->IsMapCharNumber = bptr - tkptr->ReadBuffer + 1;
      IsMapExplainError (rqptr, MsgFor(rqptr,MSG_ISMAP_INCOMPLETE), FI_LI);
      return;
   }

   /************/
   /* get path */
   /************/

   if (isalpha(*bptr) || *bptr == '/' || *bptr == '.')
   {
      /******************/
      /* NCSA compliant */
      /******************/

      if (Debug) fprintf (stdout, "NCSA compliant\n");
      zptr = (sptr = Path) + sizeof(Path);
      while (*bptr && !ISLWS(*bptr) && sptr < zptr) *sptr++ = *bptr++;
      if (sptr >= zptr)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         return;
      }
      *sptr++ = '\0';
      PathSize = sptr - Path;
      if (Debug) fprintf (stdout, "Path |%s|\n", Path);
   }
   else
   if (isdigit(*bptr) || *bptr == '(')
   {
      /******************/
      /* CERN compliant */
      /******************/

      /*
          The CERN configuration file has the path following the
          coordinates.  Skip over any coordinates here looking
          for the path, terminate after them, get the path, resume.
          Turn any coordinate-grouping parentheses into spaces.
      */
      if (Debug) fprintf (stdout, "CERN compliant\n");
      CernCompliant = true;

      /* skip over everything looking like coordinates */
      cptr = bptr;
      while (*cptr && (ISLWS(*cptr) || isdigit(*cptr) ||
             *cptr == '(' || *cptr == ',' || *cptr == ')'))
      {
         if (*cptr == '(' || *cptr == ')')
            *cptr++ = ' ';
         else
            cptr++;
      }

      if (cptr[0] && ISLWS(cptr[-1]))
         cptr[-1] = '\0';
      else
      {
         tkptr->IsMapCharNumber = cptr - tkptr->ReadBuffer + 1;
         IsMapExplainError (rqptr, MsgFor(rqptr,MSG_ISMAP_CONFUSED), FI_LI);
         return;
      }

      zptr = (sptr = Path) + sizeof(Path);
      while (*cptr && !ISLWS(*cptr) && sptr < zptr) *sptr++ = *cptr++;
      *sptr++ = '\0';
      PathSize = sptr - Path;
      if (Debug) fprintf (stdout, "Path |%s|\n", Path);
   }
   else
   {
      /***********************/
      /* specification error */
      /***********************/

      tkptr->IsMapCharNumber = bptr - tkptr->ReadBuffer + 1;
      IsMapExplainError (rqptr, MsgFor(rqptr,MSG_ISMAP_CONFUSED), FI_LI);
      return;
   }

   /*****************/
   /* default path? */
   /*****************/

   if (RegionKeyword[0] == 'd')
   {
      /* know it will fit, path size is the same and already checked! */
      memcpy (rqptr->LocationPtr, Path, PathSize);

      /* not interested in any more of the line */
      return;
   }

   /*************************/
   /* process coordinate(s) */
   /*************************/

   ccnt = 0;
   while (*bptr)
   {
      if (Debug) fprintf (stdout, "bptr |%s|\n", bptr);
      while (ISLWS(*bptr)) bptr++;
      if (!*bptr || *bptr == '#' || *bptr == '!') break;

      if (ccnt >= MAXVERTS-1)
      {
         IsMapExplainError (rqptr, MsgFor(rqptr,MSG_ISMAP_LIMIT), FI_LI);
         return;
      }

      if (!isdigit(*bptr))
      {
         /**************/
         /* should be! */
         /**************/

         tkptr->IsMapCharNumber = bptr - tkptr->ReadBuffer + 1;
         IsMapExplainError (rqptr, MsgFor(rqptr,MSG_ISMAP_CONFUSED), FI_LI);
         return;
      }

      /********************/
      /* get X coordinate */
      /********************/

      if (Debug) fprintf (stdout, "bptr |%s|\n", bptr);
      CoordArray[ccnt][X] = (double)atoi(bptr);
      while (*bptr && isdigit(*bptr)) bptr++;

      /*************************************/
      /* find comma-separated Y coordinate */
      /*************************************/

      while (ISLWS(*bptr) || *bptr == ',') bptr++;
      if (Debug) fprintf (stdout, "bptr |%s|\n", bptr);

      if (isdigit(*bptr))
      {
         /********************/
         /* get Y coordinate */
         /********************/

         CoordArray[ccnt][Y] = (double)atoi(bptr);
         while (*bptr && isdigit(*bptr)) bptr++;
      }
      else
      {
         /*******************/
         /* no Y coordinate */
         /*******************/

         if (CernCompliant && RegionKeyword[0] == 'c' && ccnt)
         {
            /*
               CERN image mapping "circle" is "(X,Y) radius".
               Fudge it by creating an "X,Y" out of the radius.
            */
            CoordArray[ccnt][Y] = CoordArray[ccnt-1][Y] + CoordArray[ccnt][X];
            CoordArray[ccnt][X] = CoordArray[ccnt-1][X];
         }
         else
         {
            tkptr->IsMapCharNumber =
               bptr - tkptr->ReadBuffer + 1;
            IsMapExplainError (rqptr, MsgFor(rqptr,MSG_ISMAP_COORDINATE), FI_LI);
            return;
         }
      }
      if (Debug)
         fprintf (stdout, "CoordArray[%d] |%f|%f|\n",
                  ccnt, CoordArray[ccnt][X], CoordArray[ccnt][Y]);

      ccnt++;
   }

   if (!ccnt)
   {
      /* no coordinates have been supplied */
      tkptr->IsMapCharNumber = bptr - tkptr->ReadBuffer + 1;
      IsMapExplainError (rqptr, MsgFor(rqptr,MSG_ISMAP_INCOMPLETE), FI_LI);
      return;
   }

   /********************/
   /* lets try it out! */
   /********************/

   if (Debug) fprintf (stdout, "mapping |%s|\n", RegionKeyword);
   CoordArray[ccnt][X] = -1;

   if (RegionKeyword[0] == 'r')
   {
      if (ccnt != 2)
         IsMapExplainError (rqptr, MsgFor(rqptr,MSG_ISMAP_INCORRECT), FI_LI);
      else
      if (pointinrect (tkptr->IsMapClickCoord, CoordArray))
      {
         /* know it will fit, capacity is the same and already checked! */
         memcpy (rqptr->LocationPtr, Path, PathSize);
         /* indicate the mapping has been successful */
         tkptr->IsMapLineNumber = 0;
      }
      return;
   }

   if (RegionKeyword[0] == 'c')
   {
      if (ccnt != 2)
         IsMapExplainError (rqptr, MsgFor(rqptr,MSG_ISMAP_INCORRECT), FI_LI);
      else
      if (pointincircle (tkptr->IsMapClickCoord, CoordArray))
      {
         /* know it will fit, capacity is the same and already checked! */
         memcpy (rqptr->LocationPtr, Path, PathSize);
         /* indicate the mapping has been successful */
         tkptr->IsMapLineNumber = 0;
      }
      return;
   }

   if (!memcmp (RegionKeyword, "pol", 3))
   {
      if (ccnt < 3)
         IsMapExplainError (rqptr, MsgFor(rqptr,MSG_ISMAP_INCORRECT), FI_LI);
      else
      if (pointinpoly (tkptr->IsMapClickCoord, CoordArray))
      {
         /* know it will fit, capacity is the same and already checked! */
         memcpy (rqptr->LocationPtr, Path, PathSize);
         /* indicate the mapping has been successful */
         tkptr->IsMapLineNumber = 0;
      }
      return;
   }

   if (!memcmp (RegionKeyword, "poi", 3))
   {
      /* the essential functionality of this is also from NCSA imagemap.c */
      if (ccnt != 1)
         IsMapExplainError (rqptr, MsgFor(rqptr,MSG_ISMAP_INCORRECT), FI_LI);
      else
      {
         Distance = (tkptr->IsMapClickCoord[X] - CoordArray[0][X]) *
                    (tkptr->IsMapClickCoord[X] - CoordArray[0][X]) +
                    (tkptr->IsMapClickCoord[Y] - CoordArray[0][Y]) *
                    (tkptr->IsMapClickCoord[Y] - CoordArray[0][Y]);
         if (Distance < tkptr->IsMapClosestPoint)
         {
            tkptr->IsMapClosestPoint = Distance;
            /* know it will fit, capacity is the same and already checked! */
            memcpy (rqptr->LocationPtr, Path, PathSize);
         }
      }
      return;
   }
}

/*****************************************************************************/
/*
Generate a general error, with explanation about the image mapping error.
*/ 

int IsMapExplainError
(
struct RequestStruct *rqptr,
char *Explanation,
char *SourceFileName,
int SourceLineNumber
)
{
   static $DESCRIPTOR (ErrorMessageFaoDsc,
"!AZ.\n\
<P><I>(document requires correction)<I>\n");

   static $DESCRIPTOR (ErrorMessageLineFaoDsc,
"!AZ; at line !UL.\n\
<P><I>(document requires correction)<I>\n");

   static $DESCRIPTOR (ErrorMessageLineCharFaoDsc,
"!AZ; at line !UL, character !UL.\n\
<P><I>(document requires correction)<I>\n");

   register struct IsMapTaskStruct  *tkptr;

   unsigned short  Length;
   char  String [1024];
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "IsMapExplainError()\n");

   /* get the pointer to the task structure from the thread storage */
   tkptr = rqptr->IsMapTaskPtr;

   if (!rqptr->ResponseStatusCode) rqptr->ResponseStatusCode = 501;

   if (tkptr->IsMapLineNumber)
      if (tkptr->IsMapCharNumber)
         sys$fao (&ErrorMessageLineCharFaoDsc, &Length, &StringDsc,
                  Explanation, tkptr->IsMapLineNumber,
                  tkptr->IsMapCharNumber);
      else
         sys$fao (&ErrorMessageLineFaoDsc, &Length, &StringDsc,
                  Explanation, tkptr->IsMapLineNumber);
   else
      sys$fao (&ErrorMessageFaoDsc, &Length, &StringDsc,
               Explanation);

   String[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);

   ErrorGeneral (rqptr, String, SourceFileName, SourceLineNumber);
}

/*****************************************************************************/
/*
NCSA imagemap.c routines.
*/

int pointinrect(double point[2], double coords[MAXVERTS][2])
{
        return ((point[X] >= coords[0][X] && point[X] <= coords[1][X]) &&
        (point[Y] >= coords[0][Y] && point[Y] <= coords[1][Y]));
}

int pointincircle(double point[2], double coords[MAXVERTS][2])
{
        int radius1, radius2;

        radius1 = ((coords[0][Y] - coords[1][Y]) * (coords[0][Y] -
        coords[1][Y])) + ((coords[0][X] - coords[1][X]) * (coords[0][X] -
        coords[1][X]));
        radius2 = ((coords[0][Y] - point[Y]) * (coords[0][Y] - point[Y])) +
        ((coords[0][X] - point[X]) * (coords[0][X] - point[X]));
        return (radius2 <= radius1);
}

int pointinpoly(double point[2], double pgon[MAXVERTS][2])
{
        int i, numverts, inside_flag, xflag0;
        int crossings;
        double *p, *stop;
        double tx, ty, y;

        for (i = 0; pgon[i][X] != -1 && i < MAXVERTS; i++)
                ;
        numverts = i;
        crossings = 0;

        tx = point[X];
        ty = point[Y];
        y = pgon[numverts - 1][Y];

        p = (double *) pgon + 1;
        if ((y >= ty) != (*p >= ty)) {
                if ((xflag0 = (pgon[numverts - 1][X] >= tx)) ==
                (*(double *) pgon >= tx)) {
                        if (xflag0)
                                crossings++;
                }
                else {
                        crossings += (pgon[numverts - 1][X] - (y - ty) *
                        (*(double *) pgon - pgon[numverts - 1][X]) /
                        (*p - y)) >= tx;
                }
        }

        stop = pgon[numverts];

        for (y = *p, p += 2; p < stop; y = *p, p += 2) {
                if (y >= ty) {
                        while ((p < stop) && (*p >= ty))
                                p += 2;
                        if (p >= stop)
                                break;
                        if ((xflag0 = (*(p - 3) >= tx)) == (*(p - 1) >= tx)) {
                                if (xflag0)
                                        crossings++;
                        }
                        else {
                                crossings += (*(p - 3) - (*(p - 2) - ty) *
                                (*(p - 1) - *(p - 3)) / (*p - *(p - 2))) >= tx;
                        }
                }
                else {
                        while ((p < stop) && (*p < ty))
                                p += 2;
                        if (p >= stop)
                                break;
                        if ((xflag0 = (*(p - 3) >= tx)) == (*(p - 1) >= tx)) {
                                if (xflag0)
                                        crossings++;
                        }
                        else {
                                crossings += (*(p - 3) - (*(p - 2) - ty) *
                                (*(p - 1) - *(p - 3)) / (*p - *(p - 2))) >= tx;
                        }
                }
        }
        inside_flag = crossings & 0x01;
        return (inside_flag);
}

/*****************************************************************************/

