/*****************************************************************************/
/*
                                Logging.c

Logging functions for the WASD VMS HTTPd. By default, produces a Web "common"
format log suitable for analysis by tools capable of processing that format.
Secretes four pseudo-requests logging the BEGINing, OPENing, CLOSEing and
ENDing of logging activities. These shouldn't introduce problems for analysis
tools as they look like legitimate POST entries for user "HHTPd"!

Also can produce logs in the "common+server", "combined" and a user-defined
format.  If a user-defined format contains an error the server will exit the
first time a request is logged.  Caveat emptor!

Log file name must be supplied via 'LoggingFileName' or the logical name 
HTTPD$LOG (process, job, or system level).  An error is reporting if logging 
is enabled without either of these being supplied.  If the log file name
specified is "SYS$OUTPUT" log entries are written to SYS$OUTPUT.  This is
useful for checking user-defined log entries before committing files to that
format.

The log file can have a period associated with it.  This period determines
when a new log file is created.  The periods are daily, weekly and monthly.
A daily period is indicated by DAILY, a weekly period by MONDAY, TUESDAY,
WEDNESDAY, THURSDAY, FRIDAY, SATURDAY and SUNDAY (the day of the week on
which, at midnight or the first request thereafter, the log is recreated), and
finally a monthly recreation by MONTHLY.  The HTTPD$LOG logical is then used
merely for the directory component and the file name is fixed at
HOST_YYYYMMDD_ACCESS.LOG, where host is the first dot-separated component of
the primary IP host name and YYYY, MM, DD is the year, month and day the log
was created.


COMMON FORMAT
-------------
(NCSA, CERN, etc.)

'client rfc891 auth-user date/time request status bytes'


COMBINED FORMAT
---------------
(NCSA)

'client rfc891 auth-user date/time request status bytes referer user-agent'


COMMON FORMAT plus server name
-------------
(NCSA, CERN, etc.)

'client rfc891 auth-user date/time request status bytes server'


USER-DEFINED FORMAT
-------------------

Must begin with a character that is used as a substitute when a particular
field is empty (use "\0" for no substitute, as in the "windows log format"
example below).  Two different "escape" characters introduce the following
parameters of characters.

A '!' followed by ...

  'AR' :  authentication realm
  'AU' :  authenticated user name
  'BB' :  bytes in body (excluding response header)
  'BY' :  bytes transmitted to the client
  'CA' :  client numeric address
  'CN' :  client host name (or address if DNS lookup disabled)
  'CI' :  cache involvement (0 == none, 1 == could have but didn't, 2 == did!)
  'EM' :  request elapsed time in milliseconds
  'ES' :  request elapsed time in seconds
  'EU' :  request elapsed time in microseconds
  'ME' :  request method
  'PA' :  request path (NOTE ... this is not the full request path!  See 'R')
  'QS' :  request query string
  'RF' :  referer
  'RQ' :  FULL request path (including script and any query string)
  'RS' :  response status code
  'SC' :  script name
  'SM' :  request scheme name (i.e. "http:" or "https:") 
  'SN' :  server host name
  'SP' :  server port
  'TC' :  request time (common log format)
  'TG' :  request time (GMT)
  'TV' :  request time (VMS format)
  'UA' :  user agent

A '\' followed by ...

  '0' :  a null character (used to define the empty field character)
  '!' :  insert an '!'
  '\' :  insert a '\'
  'n' :  insert a newline
  'q' :  insert a quote (so that in the DCL the quotes won't need escaping!)
  't' :  insert a TAB

Any other character is directly inserted into the log entry.


Examples
--------

The equivalent of the common log format is:

  "-!CN - !AU [!TC] \q!RQ\q !RS !BY"

The combined log format:

  "-!CN - !AU [!TC] \q!RQ\q !RS !BY \q!RF\q \q!UA\q"

The WebSite "windows log format" would be created by:

  "\0!TC\t!CA\t!SN\t!AR\t!AU\t!ME\t!PA\t!RQ\t!EM\t!UA\t!RS\t!BB\t"

To get the common log format plus append request duration in seconds:

  "-!CN - !AU [!TC] \q!RQ\q !RS !BY !ES"


VERSION HISTORY
---------------
27-AUG-98  MGD  exclude specified hosts from being logged
17-MAY-98  MGD  add per-service logging
23-NOV-97  MGD  signal failed log file create/put (e.g. exceeded disk quota :^)
25-OCT-97  MGD  log file period new for v4.5,
                log file now opened for sharing GET, PUT and UPD
                added 'CI' (cache-involvement) user-format directive
                bugfix; user format from configuration file
25-SEP-97  MGD  bugfix; ACCVIO if request structure exists but empty
                (assumed if it existed all the pointers were valid - WRONG!)
06-SEP-97  MGD  added "common+server", "combined", "user-defined" log formats
10-JUL-97  MGD  minor modification for server-admin logging on/off/flush
10-JUL-96  MGD  ResourcePtr instead of ScriptPtr/PathInfoPtr/QueryStringPtr
01-DEC-95  MGD  HTTPd version 3, "common" log format
27-SEP-95  MGD  use time values in request thread structure
16-JUN-95  MGD  added node name, changed logging fields to be space-separated
07-APR-95  MGD  initial development for addition to multi-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <lnmdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application header files */
#include "wasd.h"
#include "httpd.h"
#include "error.h"
#include "logging.h"

#define LOGGING_NAMING_NAME 1
#define LOGGING_NAMING_ADDRESS 2

/******************/
/* global storage */
/******************/

char  ErrorLoggingFormatLength [] = "log format too long";

int  LoggingDayWeekBegins;

boolean  LoggingEnabled,
         LoggingFormatCommon,
         LoggingFormatCommonServer,
         LoggingFormatCombined,
         LoggingFlushTimerOutstanding,
         LoggingNaming,
         LoggingPerPeriod,
         LoggingPerService,
         LoggingPeriodDaily,
         LoggingPeriodWeekly,
         LoggingPeriodMonthly,
         LoggingSysOutput;

int  LoggingNaming;

char  LoggingFileName [256],
      LoggingFormatUser [128],
      LoggingPeriodName [128];

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern boolean  CliLoggingDisabled;
extern boolean  CliLoggingEnabled;
extern int  ExitStatus;
extern int  ServerPort;
extern char  CliLoggingPeriod[];
extern char  ErrorSanityCheck[];
extern char  ServerHostName[];
extern char  SoftwareID[];
extern char  TimeGmtString[];
extern char  Utility[];
extern struct ConfigStruct  Config;
extern struct ServiceStruct  *ServiceListHead;

/*****************************************************************************/
/*
Parameter 'Function' can be:

LOGGING_BEGIN           server startup
LOGGING_OPEN            open server logging, other than server startup
LOGGING_CLOSE           close server logging, other than server shutdown
LOGGING_END             server shutdown
LOGGING_ENTRY           log a request
LOGGING_FLUSH           close the log file, flushing the contents

With per-service logging most of these functions must be performed on each
service's log, logging a request entry occurs for the service involved only.
*/ 

int Logging
(
struct RequestStruct *rqptr,
int Function
)
{
   register char  *cptr, *sptr;

   boolean  PrevLoggingEnabled;
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "Logging() %d\n", Function);

   switch (Function)
   {
      case LOGGING_ENTRY :

         if (!LoggingEnabled) return (STS$K_ERROR);

         if (Config.LogExcludeHostsPtr != NULL &&
             rqptr != NULL)
         {
            sptr = Config.LogExcludeHostsPtr;
            while (*sptr)
            {
               /** if (Debug) fprintf (stdout, "sptr |%s|\n", sptr); **/
               if (isdigit(*sptr))
                  cptr = rqptr->ClientInternetAddress;
               else
                  cptr = rqptr->ClientHostName;
               while (*cptr)
               {
                  if (*sptr == ',') break;
                  if (*sptr == '*')
                  {
                     while (*sptr && *sptr == '*' && *sptr != ',') sptr++;
                     while (*cptr && *cptr != *sptr) cptr++;
                  }
                  if (tolower(*cptr) != tolower(*sptr)) break;
                  if (*cptr) cptr++;
                  if (*sptr) sptr++;
               }
               if (!*cptr && (!*sptr || *sptr == ',')) break;
               while (*sptr && *sptr != ',') sptr++;
               if (*sptr) sptr++;
            }
            /* if it was found then return without logging the request */
            if (!*cptr && (!*sptr || *sptr == ',')) return (SS$_NORMAL);
         }

         if (LoggingPerService)
            status = LoggingDo (rqptr, rqptr->ServicePtr, Function);
         else
            status = LoggingDo (rqptr, ServiceListHead, Function);

         return (status);

      case LOGGING_BEGIN :
      case LOGGING_CLOSE :
      case LOGGING_END :
      case LOGGING_FLUSH :
      case LOGGING_OPEN :

         switch (Function)
         {
            case LOGGING_BEGIN :

               if (Config.LoggingEnabled || CliLoggingEnabled)
                  LoggingEnabled = true;
               if (CliLoggingDisabled) LoggingEnabled = false;

               LoggingPerService = Config.LogPerService;

               /* in log using host part of file name, name or address? */
               LoggingNaming = LOGGING_NAMING_NAME;
               if (Config.LogNaming[0])
               {
                  if (strsame (Config.LogNaming, "NAME", -1))
                     LoggingNaming = LOGGING_NAMING_NAME;
                  else
                  if (strsame (Config.LogNaming, "ADDRESS", -1))
                     LoggingNaming = LOGGING_NAMING_ADDRESS;
                  else
                     fprintf (stdout, "%%%s-W-LOG, naming unknown\n \\%s\\\n",
                              Utility, Config.LogNaming);
               }

               break;

            case LOGGING_OPEN :

               if (LoggingEnabled) return (STS$K_ERROR);
               break;

            case LOGGING_CLOSE :
            case LOGGING_END :
            case LOGGING_FLUSH :

               if (!LoggingEnabled) return (STS$K_ERROR);
               break;
         }

         PrevLoggingEnabled = LoggingEnabled;

         if (LoggingPerService)
         {
            register struct ServiceStruct  *svptr;

            for (svptr = ServiceListHead;
                 svptr != NULL;
                 svptr = svptr->NextPtr)
               if (VMSnok (status = LoggingDo (rqptr, svptr, Function)))
                  break;
         }
         else
            status = LoggingDo (rqptr, ServiceListHead, Function);

         if (Debug) fprintf (stdout, "LoggingDo() %%X%08.08X\n", status);

         switch (Function)
         {
            case LOGGING_BEGIN :

               if (VMSnok (status)) LoggingEnabled = false;
               break;

            case LOGGING_OPEN :

               if (VMSok (status))
                  LoggingEnabled = true;
               else
                  LoggingEnabled = PrevLoggingEnabled;
               break;

            case LOGGING_CLOSE :
            case LOGGING_END :

               if (VMSok (status))
                  LoggingEnabled = false;
               else
                  LoggingEnabled = PrevLoggingEnabled;
         }

         return (status);

      default :

         ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
   }
}

/*****************************************************************************/
/*
What an incomprehensibly long function this has grown to be! :^(

The 'svptr' parameter points to the service structure associated with the log. 
If per-service logging is enabled this will be the appropriate one of however
many services are created.  If per-service logging is not enabled this will
always be the first service created (i.e. the "primary" service).  Where
service information is required for user-format logging it is accessed from the
request structure's service pointer.
*/ 

int LoggingDo
(
struct RequestStruct *rqptr,
struct ServiceStruct *svptr,
int Function
)
{
#  define MAX_FAO_VECTOR 64

   static char  *MonthName [] =
      { "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

   static $DESCRIPTOR (PeriodFileNameFaoDsc, "!AZ_!AZ_!UL!2ZL!2ZL_ACCESS.LOG");
   static $DESCRIPTOR (ServiceFileNameFaoDsc, "!AZ_!AZ_ACCESS.LOG");

   /* flush the log file one minute after the first entry after flushing */
   static $DESCRIPTOR (FlushDeltaTimeDsc, "0 00:01:00.00");

   static $DESCRIPTOR (BytesFaoDsc, "!UL\0");
   static $DESCRIPTOR (ElapsedResponseDurationDsc, "!UL\0");
   static $DESCRIPTOR (ElapsedSecondsDsc, "!UL.!3ZL\0");
   static $DESCRIPTOR (StatusFaoDsc, "!UL\0");
   static $DESCRIPTOR (VmsTimeFaoDsc, "!%D\0");

   static $DESCRIPTOR (LogFileTimeFaoDsc,
                       "!2ZL/!3AZ/!4ZL:!2ZL:!2ZL:!2ZL !AZ");

   /* server-host - - [date/time] \"what status\" 200 0 */
   static $DESCRIPTOR (LogFileBogusFaoDsc,
                       "!AZ - HTTPd [!AZ] \"POST !AZ-!8XL\" 200 -!AZ");

   /* host r-ident user [date/time] \"request\" status bytes */
   static $DESCRIPTOR (LogFileCommonFaoDsc,
                       "!AZ - !AZ [!AZ] \"!AZ !AZ\" !UL !AZ");

   /* as for "common" above, plus ' server'*/
   static $DESCRIPTOR (LogFileCommonServerFaoDsc,
                       "!AZ - !AZ [!AZ] \"!AZ !AZ\" !UL !AZ !AZ");

   /* as for "common" above, plus ' referer user-agent'*/
   static $DESCRIPTOR (LogFileCombinedFaoDsc,
                       "!AZ - !AZ [!AZ] \"!AZ !AZ\" !UL !AZ \"!AZ\" \"!AZ\"");

   /* user-defined log-file format */
   static $DESCRIPTOR (LogFileUserFaoDsc, "!#(AZ)");

   static $DESCRIPTOR (LnmFileDevDsc, "LNM$FILE_DEV");
   static $DESCRIPTOR (LoggingFileNameLogicalDsc, "HTTPD$LOG");

   static boolean  UserFormatProblem = false;

   static int  LoggingFileNameLength = 0,
               PreviousDay = 0,
               PreviousMonth = 0,
               PreviousYear = 0;
   static unsigned short  Length,
                          NumericTime [7],
                          WeeklyNumericTime [7];
   static unsigned long  BinaryTime [2],
                         FlushDeltaBinTime [2];

   static char  BytesString [32],
                Elapsed [32],
                FormatUser [256],
                StatusString [16],
                String [1024],
                TimeString [32],
                VmsTimeString [32];
   static $DESCRIPTOR (BytesStringDsc, BytesString);
   static $DESCRIPTOR (ElapsedDsc, Elapsed);
   static $DESCRIPTOR (StatusStringDsc, StatusString);
   static $DESCRIPTOR (StringDsc, String);
   static $DESCRIPTOR (TimeStringDsc, TimeString);
   static $DESCRIPTOR (VmsTimeStringDsc, VmsTimeString);

   static struct FAB  LogFileFab;
   static struct RAB  LogFileRab;

   static struct {
      short int  buf_len;
      short int  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   } LnmItems [] =
   {
      { sizeof(LoggingFileName)-1, LNM$_STRING, LoggingFileName, &Length },
      { 0,0,0,0 }
   };

   register unsigned long  *vecptr;

   boolean  PeriodChanged;
   int  status,
        SetimrStatus,
        SetPrvStatus;
   unsigned long  *BinaryTimePtr;
   unsigned long  FaoVector [MAX_FAO_VECTOR];
   char  *FunctionNamePtr;
   char  MiscChars [MAX_FAO_VECTOR * 2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "LoggingDo() %d\n", Function);

   /* logging to SYS$OUTPUT does not do many of the things that set 'status' */
   status = SS$_NORMAL;

   if (Function == LOGGING_BEGIN)
   {
      /*****************/
      /* begin logging */
      /*****************/

      register char  *cptr, *sptr, *zptr;

      /* if no logging file name from command line then use any config one */
      if (!LoggingFileName[0] && Config.LogFileName[0])
         strcpy (LoggingFileName, Config.LogFileName);

      if (strsame (LoggingFileName, "SYS$OUTPUT", -1))
         LoggingSysOutput = true;
      else
         LoggingSysOutput = false;

      /************************/
      /* determine log format */
      /************************/

      UserFormatProblem = false;
      if (LoggingFormatUser[0])
         strcpy (FormatUser, LoggingFormatUser);
      else
      if (Config.LogFormat[0])
         strcpy (FormatUser, Config.LogFormat);
      else
         FormatUser[0] = '\0';

      if (FormatUser[0])
      {
         if (strsame (FormatUser, "COMMON", -1))
         {
            LoggingFormatCommonServer = LoggingFormatCombined = false;
            LoggingFormatCommon = true;
            FormatUser[0] = '\0';
         }
         else
         if (strsame (FormatUser, "COMMON_SERVER", -1))
         {
            LoggingFormatCommon = LoggingFormatCombined = false;
            LoggingFormatCommonServer = true;
            FormatUser[0] = '\0';
         }
         else
         if (strsame (FormatUser, "COMBINED", -1))
         {
            LoggingFormatCommon = LoggingFormatCommonServer = false;
            LoggingFormatCombined = true;
            FormatUser[0] = '\0';
         }
         else
         if (FormatUser[0])
         {
            LoggingFormatCommon = LoggingFormatCommonServer =
               LoggingFormatCombined = false;
         }
         else
            LoggingFormatCommon = true;
      }
      else
         LoggingFormatCommon = true;

      /****************************/
      /* determine logging period */
      /****************************/

      LoggingPerPeriod = LoggingPeriodDaily = LoggingPeriodWeekly =
         LoggingPeriodMonthly = false;

      if (!LoggingPeriodName[0] && Config.LogPeriod[0])
         strcpy (LoggingPeriodName, Config.LogPeriod);

      if (LoggingPeriodName[0])
      {
         PreviousDay = PreviousMonth = PreviousYear = 0;

         if (strsame (LoggingPeriodName, "DAILY", 3))
            LoggingPerPeriod = LoggingPeriodDaily = true;
         else
         if (strsame (LoggingPeriodName, "SUNDAY", 3))
         {
            LoggingPerPeriod = LoggingPeriodWeekly = true;
            LoggingDayWeekBegins = 7;
         }
         else
         if (strsame (LoggingPeriodName, "MONDAY", 4))
         {
            LoggingPerPeriod = LoggingPeriodWeekly = true;
            LoggingDayWeekBegins = 1;
         }
         else
         if (strsame (LoggingPeriodName, "TUESDAY", 3))
         {
            LoggingPerPeriod = LoggingPeriodWeekly = true;
            LoggingDayWeekBegins = 2;
         }
         else
         if (strsame (LoggingPeriodName, "WEDNESDAY", 3))
         {
            LoggingPerPeriod = LoggingPeriodWeekly = true;
            LoggingDayWeekBegins = 3;
         }
         else
         if (strsame (LoggingPeriodName, "THURSDAY", 3))
         {
            LoggingPerPeriod = LoggingPeriodWeekly = true;
            LoggingDayWeekBegins = 4;
         }
         else
         if (strsame (LoggingPeriodName, "FRIDAY", 3))
         {
            LoggingPerPeriod = LoggingPeriodWeekly = true;
            LoggingDayWeekBegins = 5;
         }
         else
         if (strsame (LoggingPeriodName, "SATURDAY", 3))
         {
            LoggingPerPeriod = LoggingPeriodWeekly = true;
            LoggingDayWeekBegins = 6;
         }
         else
         if (strsame (LoggingPeriodName, "MONTH", 4))
            LoggingPerPeriod = LoggingPeriodMonthly = true;
         else
         {
            fprintf (stdout, "%%%s-W-LOG, period unknown\n \\%s\\\n",
                     Utility, LoggingPeriodName);
            LoggingPerPeriod = false;
         }
      }

      /*************************/
      /* get logging host name */
      /*************************/

      zptr = (sptr = svptr->LogHostName) + sizeof(svptr->LogHostName)-1;
      if (LoggingNaming == LOGGING_NAMING_ADDRESS)
      {
         /* internet address with periods replaced by hyphens */
         for (cptr = svptr->ServerInternetAddress;
              *cptr && sptr < zptr;
              cptr++)
            if (*cptr == '.') *sptr++ = '-'; else *sptr++ = *cptr;
      }
      else
      {
         /* just the first period-separated part of the full host name */
         for (cptr = svptr->ServerHostName;
              *cptr && *cptr != '.' && sptr < zptr;
              *sptr++ = toupper(*cptr++));
      }
      *sptr = '\0';
      if (Debug)
         fprintf (stdout, "LogHostName/Port |%s|%s| \n",
                  svptr->LogHostName, svptr->ServerPortString);

      if (!LoggingEnabled) return (SS$_NORMAL);
   }

   if (Function == LOGGING_FLUSH)
   {
      /******************/
      /* flush log file */
      /******************/

      if (LoggingSysOutput) return (SS$_NORMAL);
      if (!svptr->LogFileOpen) return (SS$_NORMAL);

      sys$close (&svptr->LogFileFab, 0, 0);
      svptr->LogFileOpen = false;
      return (SS$_NORMAL);
   }

   /*************************/
   /* log file name change? */
   /*************************/

   /* get numeric vector time for this log transaction */
   if (rqptr == NULL)
   {
      sys$gettim (BinaryTimePtr = &BinaryTime);
      sys$numtim (&NumericTime, &BinaryTime);
   }
   else
      sys$numtim (&NumericTime, BinaryTimePtr = &rqptr->BinaryTime);

   if (LoggingPerPeriod && NumericTime[2] != PreviousDay)
   {
      PeriodChanged = false;
      if (LoggingPeriodDaily)
      {
         PeriodChanged = true;
         PreviousYear = NumericTime[0];
         PreviousMonth = NumericTime[1];
         PreviousDay = NumericTime[2];
      }
      else
      if (LoggingPeriodWeekly)
      {
         LoggingWeek (BinaryTimePtr, &NumericTime, &WeeklyNumericTime);
         if (WeeklyNumericTime[0] != PreviousYear ||
             WeeklyNumericTime[1] != PreviousMonth ||
             WeeklyNumericTime[2] != PreviousDay)
         {
            PeriodChanged = true;
            PreviousYear = WeeklyNumericTime[0];
            PreviousMonth = WeeklyNumericTime[1];
            PreviousDay = WeeklyNumericTime[2];
         }
      }
      else
      if (LoggingPeriodMonthly)
      {
         if (NumericTime[1] != PreviousMonth)
         {
            PeriodChanged = true;
            PreviousYear = NumericTime[0];
            PreviousMonth = NumericTime[1];
            PreviousDay = NumericTime[2];
         }
      }
      else
         ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
   }

   if (!svptr->LogFileOpen && !LoggingSysOutput)
   {
      /*********************/
      /* open the log file */
      /*********************/

      if (Debug)
         fprintf (stdout,
"LoggingFileName |%s|\n\
svptr->LogFileName |%s|\n",
            LoggingFileName, svptr->LogFileName);

      if (!LoggingFileName[0])
      {
         if (VMSnok (status =
             sys$trnlnm (0, &LnmFileDevDsc,
                         &LoggingFileNameLogicalDsc, 0, &LnmItems)))
            return (status);
         LoggingFileName[Length] = '\0';
         if (Debug) fprintf (stdout, "LoggingFileName |%s|\n", LoggingFileName);
      }

      if (!LoggingFileNameLength)
         LoggingFileNameLength = strlen(LoggingFileName);

      /* initialize the FAB before setting up the file name! */
      svptr->LogFileFab = cc$rms_fab;

      if (LoggingPerPeriod)
      {
         /*************************************************/
         /* per-period logging (with/without per-service) */
         /*************************************************/

         if (PeriodChanged)
         {
            vecptr = FaoVector;
            *vecptr++ = svptr->LogHostName;
            *vecptr++ = svptr->ServerPortString;
            if (LoggingPeriodWeekly)
            {
               *vecptr++ = WeeklyNumericTime[0];
               *vecptr++ = WeeklyNumericTime[1];
               *vecptr++ = WeeklyNumericTime[2];
            }
            else
            {
               *vecptr++ = NumericTime[0];
               *vecptr++ = NumericTime[1];
               if (LoggingPeriodMonthly)
                  *vecptr++ = 1;
               else
                  *vecptr++ = NumericTime[2];
            }
         
            svptr->LogFileNameDsc.dsc$a_pointer = svptr->LogFileName;
            svptr->LogFileNameDsc.dsc$w_length = sizeof(svptr->LogFileName)-1;

            if (VMSnok (status =
                sys$faol (&PeriodFileNameFaoDsc, &svptr->LogFileNameLength,
                          &svptr->LogFileNameDsc, &FaoVector)))
               ErrorExitVmsStatus (status, "sys$faol()", FI_LI);
            svptr->LogFileName[svptr->LogFileNameLength] = '\0';
            if (Debug)
               fprintf (stdout, "svptr->LogFileName |%s|\n",
                        svptr->LogFileName);
         }

         /* logging file name is used for the directory component only */
         svptr->LogFileFab.fab$l_dna = LoggingFileName;  
         svptr->LogFileFab.fab$b_dns = LoggingFileNameLength;
         svptr->LogFileFab.fab$l_fna = svptr->LogFileName;  
         svptr->LogFileFab.fab$b_fns = svptr->LogFileNameLength;
      }
      else
      if (LoggingNaming)
      {
         /***************************************/
         /* per-service logging (no per-period) */
         /***************************************/

         if (!svptr->LogFileNameLength)
         {
            vecptr = FaoVector;
            *vecptr++ = svptr->LogHostName;
            *vecptr++ = svptr->ServerPortString;

            svptr->LogFileNameDsc.dsc$a_pointer = svptr->LogFileName;
            svptr->LogFileNameDsc.dsc$w_length = sizeof(svptr->LogFileName)-1;

            if (VMSnok (status =
                sys$faol (&ServiceFileNameFaoDsc, &svptr->LogFileNameLength,
                          &svptr->LogFileNameDsc, &FaoVector)))
               ErrorExitVmsStatus (status, "sys$faol()", FI_LI);
            svptr->LogFileName[svptr->LogFileNameLength] = '\0';
            if (Debug)
               fprintf (stdout, "svptr->LogFileName |%s|\n", svptr->LogFileName);
         }

         /* logging file name is used for the directory component only */
         svptr->LogFileFab.fab$l_dna = LoggingFileName;  
         svptr->LogFileFab.fab$b_dns = LoggingFileNameLength;
         svptr->LogFileFab.fab$l_fna = svptr->LogFileName;  
         svptr->LogFileFab.fab$b_fns = svptr->LogFileNameLength;
      }
      else
      {
         /******************/
         /* fixed log file */
         /******************/

         /* the logging file name must be complete */
         svptr->LogFileFab.fab$l_fna = LoggingFileName;  
         svptr->LogFileFab.fab$b_fns = LoggingFileNameLength;
      }

      /************************/
      /* create/open log file */
      /************************/

      svptr->LogFileFab.fab$l_fop = FAB$M_CIF | FAB$M_SQO;
      svptr->LogFileFab.fab$b_fac = FAB$M_PUT;
      svptr->LogFileFab.fab$b_rfm = FAB$C_STMLF;
      svptr->LogFileFab.fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT | FAB$M_SHRUPD;
      svptr->LogFileFab.fab$b_org = FAB$C_SEQ;
      svptr->LogFileFab.fab$b_rat = FAB$M_CR;
   
      /* turn on SYSPRV to allow access to the logging file */
      EnableSysPrv ();

      status = sys$create (&svptr->LogFileFab, 0, 0);

      /* turn off SYSPRV */
      DisableSysPrv ();

      if (VMSnok (status))
      {
         if (Debug) fprintf (stdout, "sys$create() %%X%08.08X\n", status);
         if (!svptr->LogFileError)
         {
            svptr->LogFileError = true;
            fprintf (stdout, "%%%s-W-LOG, file create error\n-%s\n \\%s\\\n",
                     Utility, SysGetMsg(status)+1, svptr->LogFileName);
         }
         return (status);
      }
      svptr->LogFileError = false;

      svptr->LogFileRab = cc$rms_rab;
      svptr->LogFileRab.rab$l_fab = &svptr->LogFileFab;
      svptr->LogFileRab.rab$b_mbf = 4;
      svptr->LogFileRab.rab$l_rop = RAB$M_EOF | RAB$M_WBH;

      if (VMSnok (status = sys$connect (&svptr->LogFileRab, 0, 0)))
      {
         if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
         sys$close (&svptr->LogFileFab, 0, 0);
         return (status);
      }

      svptr->LogFileOpen = true;
   }

   /***********************/
   /* act on the log file */
   /***********************/

   if (Debug) fprintf (stdout, "log file |%s|\n", svptr->LogFileName);

   status = sys$fao (&LogFileTimeFaoDsc, &Length, &TimeStringDsc,
                     NumericTime[2],
                     MonthName[NumericTime[1]],
                     NumericTime[0],
                     NumericTime[3],
                     NumericTime[4],
                     NumericTime[5],
                     TimeGmtString);
   /** if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status); **/
   TimeString[Length] = '\0';

   if (Function == LOGGING_BEGIN ||
       Function == LOGGING_OPEN ||
       Function == LOGGING_END ||
       Function == LOGGING_CLOSE)
   {
      if (Function == LOGGING_BEGIN)
         FunctionNamePtr = "BEGIN";
      else
      if (Function == LOGGING_OPEN)
         FunctionNamePtr = "OPEN";
      else
      if (Function == LOGGING_CLOSE)
         FunctionNamePtr = "CLOSE";
      else
      if (Function == LOGGING_END)
         FunctionNamePtr = "END";

      if (LoggingFormatCommon ||
          LoggingFormatCommonServer ||
          LoggingFormatCombined)
      {
         vecptr = FaoVector;
         *vecptr++ = ServerHostName;
         *vecptr++ = TimeString;
         *vecptr++ = FunctionNamePtr;
         *vecptr++ = ExitStatus;
         if (LoggingFormatCommonServer)
            *vecptr++ = " -";
         else
         if (LoggingFormatCombined)
            *vecptr++ = " - -";
         else
            *vecptr++ = "";

         status = sys$faol (&LogFileBogusFaoDsc, &Length, &StringDsc,
                            &FaoVector);
         /* if (Debug) fprintf (stdout, "sys$faol() %%X%08.08X\n", status); */
         if (VMSnok (status) || status == SS$_BUFFEROVF)
            ErrorExitVmsStatus (status, "sys$faol()", FI_LI);
         String[Length] = '\0';

         if (LoggingSysOutput)
            fprintf (stdout, "%s\n", String);
         else
         {
            svptr->LogFileRab.rab$l_rbf = String;
            svptr->LogFileRab.rab$w_rsz = Length;
            /** if (Debug) fprintf (stdout, "String |%s|\n", String); **/
            if (VMSnok (status = sys$put (&svptr->LogFileRab, 0, 0)))
            {
               if (Debug) fprintf (stdout, "sys$put() %%X%08.08X\n", status);
               if (!svptr->LogFileError)
               {
                  svptr->LogFileError = true;
                  fprintf (stdout, "%%%s-W-LOG, file write error\n-%s\n\\%s\\\n",
                           Utility, SysGetMsg(status)+1, svptr->LogFileName);
               }
            }
            else
               svptr->LogFileError = false;
         }
      }

      if (Function == LOGGING_END ||
          Function == LOGGING_CLOSE)
      {
         /******************/
         /* close log file */
         /******************/

         if (LoggingSysOutput) return (SS$_NORMAL);
         sys$close (&svptr->LogFileFab, 0, 0);
         svptr->LogFileOpen = false;
         return (SS$_NORMAL);
      }

      /* BEGIN and OPEN drop through to initialize a flush timer */
   }

   if (Function == LOGGING_ENTRY && rqptr != NULL)
   {
      /*************************/
      /* format request record */
      /*************************/

      if (LoggingFormatCommon ||
          LoggingFormatCommonServer ||
          LoggingFormatCombined)
      {
         /*******************/
         /* common/combined */
         /*******************/

         vecptr = FaoVector;

         *vecptr++ = rqptr->ClientHostName;

         /* if authentication has not been supplied "place-mark" remote user */
         if (rqptr->RemoteUser[0])
            *vecptr++ = rqptr->RemoteUser;
         else
            *vecptr++ = "-";

         *vecptr++ = TimeString;

         if (rqptr->HttpMethodName[0])
            *vecptr++ = rqptr->HttpMethodName;
         else
            *vecptr++ = "-";

         if (rqptr->ResourcePtr == NULL)
            *vecptr++ = "-";
         else
         if (!rqptr->ResourcePtr[0])
            *vecptr++ = "-";
         else
            *vecptr++ = rqptr->ResourcePtr;

         *vecptr++ = rqptr->ResponseStatusCode;

         if (rqptr->BytesTx)
         {
            sys$fao (&BytesFaoDsc, &Length, &BytesStringDsc, rqptr->BytesTx);
            *vecptr++ = BytesString;
         }
         else
            *vecptr++ = "0";

         if (LoggingFormatCommonServer)
            *vecptr++ = rqptr->ServicePtr->ServerHostName;
         else
         if (LoggingFormatCombined)
         {
            if (rqptr->HttpRefererPtr == NULL)
               *vecptr++ = "-";
            else
            if (!rqptr->HttpRefererPtr[0])
               *vecptr++ = "-";
            else
               *vecptr++ = rqptr->HttpRefererPtr;

            if (rqptr->HttpUserAgentPtr == NULL)
               *vecptr++ = "-";
            else
            if (!rqptr->HttpUserAgentPtr[0])
               *vecptr++ = "-";
            else
               *vecptr++ = rqptr->HttpUserAgentPtr;
         }
      }
      else
      if (FormatUser[0] || Config.LogFormat[0])
      {
         /***********************/
         /* user-defined format */
         /***********************/

         register int  cnt;
         register char  *cptr, *sptr;
         char  *AbsentPtr;

         if (UserFormatProblem) return (SS$_NORMAL);

         cnt = 0;
         sptr = MiscChars;
         AbsentPtr = "";
         if (FormatUser[0])
            cptr = FormatUser;
         else
            cptr = Config.LogFormat;

         vecptr = FaoVector;

         while (*cptr)
         {
            if (Debug) fprintf (stdout, "|%s|\n", cptr);

            if (cnt > MAX_FAO_VECTOR)
               ErrorExitVmsStatus (0, ErrorLoggingFormatLength, FI_LI);

            if (cnt == 1) AbsentPtr = FaoVector[0];

            switch (*cptr)
            {
               case '!' :
               {
                  cptr++;
                  switch (*(unsigned short*)cptr)
                  {
                     case 'AR' :
                        if (rqptr->AuthRealmPtr == NULL)
                           *vecptr++ = AbsentPtr;
                        else
                        if (!rqptr->AuthRealmPtr[0])
                           *vecptr++ = AbsentPtr;
                        else
                           *vecptr++ = rqptr->AuthRealmPtr;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'AU' :
                        if (!rqptr->RemoteUser[0])
                           *vecptr++ = AbsentPtr;
                        else
                           *vecptr++ = rqptr->RemoteUser;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'BB' :
                        if (rqptr->BytesTx)
                        {
                           sys$fao (&BytesFaoDsc, 0, &BytesStringDsc,
                              rqptr->BytesTx - rqptr->ResponseHeaderLength);
                           *vecptr++ = BytesString;
                        }
                        else
                           *vecptr++ = "0";
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'BY' :
                        if (rqptr->BytesTx)
                        {
                           sys$fao (&BytesFaoDsc, 0, &BytesStringDsc,
                                    rqptr->BytesTx);
                           *vecptr++ = BytesString;
                        }
                        else
                           *vecptr++ = "0";
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'CA' :
                        *vecptr++ = rqptr->ClientInternetAddress;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'CN' :
                        *vecptr++ = rqptr->ClientHostName;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'CI' :
                        if (!rqptr->CacheInvolvement)
                           *vecptr++ = "0";
                        else
                        if (rqptr->CacheInvolvement == 1)
                           *vecptr++ = "1";
                        else
                           *vecptr++ = "2";
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'EM' :
                        /* durations are measured in microseconds */
                        sys$fao (&ElapsedResponseDurationDsc, 0, &ElapsedDsc,
                                 rqptr->ResponseDuration / 1000);
                        *vecptr++ = Elapsed;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'ES' :
                        /* durations are measured in microseconds */
                        sys$fao (&ElapsedSecondsDsc, 0, &ElapsedDsc,
                                 rqptr->ResponseDuration / 1000000,
                                 (rqptr->ResponseDuration / 1000) % 1000);
                        *vecptr++ = Elapsed;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'EU' :
                        /* durations are measured in microseconds */
                        sys$fao (&ElapsedResponseDurationDsc, 0, &ElapsedDsc,
                                 rqptr->ResponseDuration);
                        *vecptr++ = Elapsed;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'ME' :
                        *vecptr++ = rqptr->HttpMethodName;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'PA' :
                        if (rqptr->PathInfoPtr == NULL)
                           *vecptr++ = AbsentPtr;
                        else
                        if (!rqptr->PathInfoPtr[0])
                           *vecptr++ = AbsentPtr;
                        else
                           *vecptr++ = rqptr->PathInfoPtr;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'QS' :
                        if (rqptr->QueryStringPtr == NULL)
                           *vecptr++ = AbsentPtr;
                        else
                        if (!rqptr->QueryStringPtr[0])
                           *vecptr++ = AbsentPtr;
                        else
                           *vecptr++ = rqptr->QueryStringPtr;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'RF' :
                        if (rqptr->HttpRefererPtr == NULL)
                           *vecptr++ = AbsentPtr;
                        else
                        if (!rqptr->HttpRefererPtr[0])
                           *vecptr++ = AbsentPtr;
                        else
                           *vecptr++ = rqptr->HttpRefererPtr;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'RQ' :
                        if (rqptr->ResourcePtr == NULL)
                           *vecptr++ = AbsentPtr;
                        else
                        if (!rqptr->ResourcePtr[0])
                           *vecptr++ = AbsentPtr;
                        else
                           *vecptr++ = rqptr->ResourcePtr;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'RS' :
                        sys$fao (&StatusFaoDsc, 0, &StatusStringDsc,
                                 rqptr->ResponseStatusCode);
                        *vecptr++ = StatusString;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'SC' :
                        if (!rqptr->ScriptName[0])
                           *vecptr++ = AbsentPtr;
                        else
                           *vecptr++ = rqptr->ScriptName;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'SM' :
                        /* must come from request's service pointer */
                        *vecptr++ = rqptr->ServicePtr->RequestSchemeNamePtr;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'SN' :
                        /* must come from request's service pointer */
                        *vecptr++ = rqptr->ServicePtr->ServerHostName;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'SP' :
                        /* must come from request's service pointer */
                        *vecptr++ = rqptr->ServicePtr->ServerPortString;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'TC' :
                        *vecptr++ = TimeString;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'TG' :
                        *vecptr++ = rqptr->GmDateTime;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'TV' :
                        sys$fao (&VmsTimeFaoDsc, 0, &VmsTimeStringDsc,
                                 &rqptr->BinaryTime);
                        *vecptr++ = VmsTimeString;
                        cnt++;
                        cptr += 2;
                        continue;
                     case 'UA' :
                        if (rqptr->HttpUserAgentPtr == NULL)
                           *vecptr++ = AbsentPtr;
                        else
                        if (!rqptr->HttpUserAgentPtr[0])
                           *vecptr++ = AbsentPtr;
                        else
                           *vecptr++ = rqptr->HttpUserAgentPtr;
                        cnt++;
                        cptr += 2;
                        continue;
                     default :
                        UserFormatProblem = true;
                        continue;
                  }
               }

               case '\\' :
               {
                  cptr++;
                  switch (*cptr)
                  {
                     case '0' :
                        *vecptr++ = "";
                        cnt++;
                        cptr++;
                        continue;
                     case 'n' :
                        *vecptr++ = "\n";
                        cnt++;
                        cptr++;
                        continue;
                     case 'q' :
                        *vecptr++ = "\"";
                        cnt++;
                        cptr++;
                        continue;
                     case 't' :
                        *vecptr++ = "\t";
                        cnt++;
                        cptr++;
                        continue;
                     case '!' :
                        *vecptr++ = "!";
                        cnt++;
                        cptr++;
                        continue;
                     case '\\' :
                        *vecptr++ = "\\";
                        cnt++;
                        cptr++;
                        continue;
                     default :
                        UserFormatProblem = true;
                        continue;
                  }
               }

               default :
               {
                  *sptr = *cptr++;
                  *vecptr++ = sptr++;
                  *sptr++ = '\0';
                  cnt++;
                  continue;
               }
            }
         }

         if (cnt <= 1) UserFormatProblem = true;

         if (UserFormatProblem)
         {
            fprintf (stdout, "%%%s-W-LOG, user format problem\n \\%s\\\n",
                     Utility, FormatUser);
            return (SS$_NORMAL);
         }

         /** if (Debug) fprintf (stdout, "cnt-1: %d\n", cnt-1); **/
         /* now set [0] to the repeat count */
         FaoVector[0] = cnt - 1;
      }
      else
         ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);

      /*************/
      /* sys$fao() */
      /*************/

      if (LoggingFormatCommon)
         status = sys$faol (&LogFileCommonFaoDsc, &Length, &StringDsc,
                            &FaoVector);
      else
      if (LoggingFormatCombined)
         status = sys$faol (&LogFileCombinedFaoDsc, &Length, &StringDsc,
                            &FaoVector);
      else
      if (LoggingFormatCommonServer)
         status = sys$faol (&LogFileCommonServerFaoDsc, &Length, &StringDsc,
                            &FaoVector);
      else
      if (FormatUser[0])
         status = sys$faol (&LogFileUserFaoDsc, &Length, &StringDsc,
                            &FaoVector);
      else
         ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);

      /** if (Debug) fprintf (stdout, "sys$faol() %%X%08.08X\n", status); **/

      if (VMSnok (status) || status == SS$_BUFFEROVF)
         ErrorExitVmsStatus (status, "sys$faol()", FI_LI);
      String[Length] = '\0';

      /********************/
      /* write the record */
      /********************/

      if (LoggingSysOutput)
      {
         fprintf (stdout, "%s\n", String);
         status = SS$_NORMAL;
      }
      else
      {
         svptr->LogFileRab.rab$l_rbf = String;
         svptr->LogFileRab.rab$w_rsz = Length;
         /** if (Debug) fprintf (stdout, "String |%s|\n", String); **/
         if (VMSnok (status = sys$put (&svptr->LogFileRab, 0, 0)))
         {
            if (Debug) fprintf (stdout, "sys$put() %%X%08.08X\n", status);
            if (!svptr->LogFileError)
            {
               svptr->LogFileError = true;
               fprintf (stdout, "%%%s-W-LOG, file write error\n-%s\n \\%s\\\n",
                        Utility, SysGetMsg(status)+1, svptr->LogFileName);
            }
         }
         else
            svptr->LogFileError = false;
      }
   }

   /*******************************/
   /* flush log file after period */
   /*******************************/

   if (!FlushDeltaBinTime[0])
   {
      if (VMSnok (status = sys$bintim (&FlushDeltaTimeDsc, &FlushDeltaBinTime)))
         ErrorExitVmsStatus (status, "sys$bintim()", FI_LI);
   }
   if (!LoggingFlushTimerOutstanding)
   {
      if (VMSnok (SetimrStatus =
          sys$setimr (0, &FlushDeltaBinTime, &LoggingFlushFileAST,
                      &LoggingFlushFileAST, 0)))
         ErrorExitVmsStatus (SetimrStatus, "sys$setimr()", FI_LI);
      LoggingFlushTimerOutstanding = true;
   }

   return (status);
}

/*****************************************************************************/
/*
Timer AST function to "flush" (close) the log file.
*/

LoggingFlushFileAST ()

{
   Logging (NULL, LOGGING_FLUSH);
   LoggingFlushTimerOutstanding = false;
}

/*****************************************************************************/
/*
Fills 'WeeklyNumericTime' with the date of the most recent
'LoggingDayWeekBegins' (i.e. the last Monday, Tuesday ... Sunday)
*/

int LoggingWeek
(
unsigned long *BinaryTimePtr,
unsigned short *NumericTimePtr,
unsigned short *WeeklyNumericTimePtr
)
{
   static unsigned long  DayOfWeek;
   static unsigned long  DeltaDaysBinaryTime [2],
                         ResultBinaryTime [2];
   static $DESCRIPTOR (DeltaDaysFaoDsc, "!UL 00:00:00.00");

   int  status,
        DeltaDays;
   unsigned short  Length;
   char  DeltaString [32];
   $DESCRIPTOR (DeltaStringDsc, DeltaString);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "LoggingWeek()\n");

   if (Debug)
   {
      char  Time [48];
      $DESCRIPTOR (TimeFaoDsc, "!%D\0");
      $DESCRIPTOR (TimeDsc, Time);

      sys$fao (&TimeFaoDsc, 0, &TimeDsc, BinaryTimePtr);
      fprintf (stdout, "time |%s|\n", Time); 
   }

   /* monday == 1, tuesday == 2 ... sunday == 7 */
   if (VMSnok (status =
       lib$day_of_week (BinaryTimePtr, &DayOfWeek)))
      ErrorExitVmsStatus (status, "lib$day_of_week()", FI_LI);

   if ((DeltaDays = DayOfWeek - LoggingDayWeekBegins) < 0)
      DeltaDays += 7;

   if (Debug)
      fprintf (stdout, "%d = %d - %d\n",
               DeltaDays, DayOfWeek, LoggingDayWeekBegins);

   if (!DeltaDays)
   {
      /* same bat-time, same bat-channel */
      memcpy (WeeklyNumericTimePtr, NumericTimePtr, 14);
      return;
   }

   if (VMSnok (status =
       sys$fao (&DeltaDaysFaoDsc, &Length, &DeltaStringDsc, DeltaDays)))
      ErrorExitVmsStatus (status, "sys$fao()", FI_LI);
   DeltaStringDsc.dsc$w_length = Length;
   if (Debug)
   {
      DeltaString[Length] = '\0';
      fprintf (stdout, "DeltaString |%s|\n", DeltaString);
   }

   if (VMSnok (status = sys$bintim (&DeltaStringDsc, DeltaDaysBinaryTime)))
      ErrorExitVmsStatus (status, "sys$bintim()", FI_LI);

   if (VMSnok (status =
       lib$sub_times (BinaryTimePtr, &DeltaDaysBinaryTime, &ResultBinaryTime)))
      ErrorExitVmsStatus (status, "lib$sub_times()", FI_LI);

   if (Debug)
   {
      char  Week [48];
      $DESCRIPTOR (WeekFaoDsc, "!%D\0");
      $DESCRIPTOR (WeekDsc, Week);

      sys$fao (&WeekFaoDsc, 0, &WeekDsc, &ResultBinaryTime);
      fprintf (stdout, "start of week |%s|\n", Week); 
   }

   /* overwrite the supplied numeric time with the start of week's */
   sys$numtim (WeeklyNumericTimePtr, &ResultBinaryTime);
}

/*****************************************************************************/

