/*****************************************************************************/
/*
                                MapUrl.c

Functions supporting mapping of URLs to VMS file specifications and VMS 
specifications to URLs, uses the HTTPd rule mapping file.  The number of rules 
it can store is not fixed, but is limited by available memory and the 
practical considerations of processing large, linear rule databases. 

This module is designed to be completely self-contained, only the global 
storage listed below is required for linking.  This is to allow other 
programs, particularly CGI scripts, to use the URL mapping and reverse-mapping 
functionality if they require it.  That is the primary reason for the separate 
rule-mapping database file with the WASD VMS HTTPd.

Comment lines may be provided by beginning a line with a '#' character.

Lines may be continued by ensuring the last character on the line is a '\'.


Mapping Rules
-------------

EXEC       template   result     [conditional] [conditional]
EXEC+      template   result     [conditional] [conditional]
FAIL       template              [conditional] [conditional]
MAP        template   result     [conditional] [conditional]
PASS       template   [result]   [conditional] [conditional]
REDIRECT   template   result     [conditional] [conditional]
SCRIPT     template   result     [conditional] [conditional]
SCRIPT+    template   result     [conditional] [conditional]

(Somewhat awkwardly the square brackets both denote optional fields and to
literally delimit a conditional.  Please do not confuse the two!)

PASS rules can map to a result comprising an HTTP status code (e.g. 403) with a
following textual message (including white-space).  Just delimit the result
string with double or single quotes, or within curly braces.  For example:

PASS /private/stuff/* "403 Can't go in there!" [!ho:my.host.name]
PASS /private/stuff/* '403 "/private/stuff/" is off-limits!' [!ho:my.host.name]


DECnet-based User Scripting
---------------------------

If DECnet/OSU scripting is enabled then the following rule will activate a
script in the specified user's directory (all other things being equal).
Always map to some unique and specific subdirectory of the home directory.
(In these examples commercial-at symbols substitute for wildcard asterisks!!)

exec /~@/cgi-bin/@ /0""::/where/ever/@/cgi-bin/@

It's a little like the user-mapping rules:

pass /~@ /where/ever/@


Mapping Conditionals
--------------------

THIS OFFERS A POWERFUL TOOL TO THE SERVER ADMINISTRATOR!

Conditionals are new to v4.4 and allow mapping rules to be applied
conditionally! That is, the conditionals contain strings that are matched to
specific elements of the request and only if matched are the corresponding
rules applied. The conditionals may contains the '*' and '%' wildcards, and
optionally be negated by an '!' at the start of the conditional string.

Conditionals must follow the rule and are delimited by '[' and ']'. Multiple,
space-separated conditions may be included within one '[...]'. This behaves as
a logical OR (i.e. the condition is true if only one is matched). Multiple
'[...]' conditionals may be included against a rule. These act as a logical
AND (i.e. all must have at least one condition matched). The result of an
entire conditional may be optionally negated by prefixing the '[' with a '!'.

If a conditional is not met the line is completely ignored.  Conditional rules
are not used for 'backward' translation (i.e. from VMS to URL path) so another
rule must exist somewhere to do these mappings.  Conditional rules are only
used by the HTTPd server, they are completely ignored by any scripts using the
MapUrl.c module.

[!]ac:accept             ('Accept:')
[!]al:accept-language    ('Accept-Language:')
[!]as:accept-charset     ('Accept-Charset:')
[!]fo:host-name/address  ('Forwarded:', proxies/gateways)
[!]ho:host-name/address  (client's of course!)
[!]hh:host-name/address  ('Host:', destination)
[!]me:http-method        (GET, POST, etc.)
[!]sc:scheme             (request scheme 'http:', 'http', 'https:' or 'https')
[!]sn:server-name        (server host name)
[!]sp:server-port        (server port)
[!]ru:remote-user        (if authenticated)
[!]ua:user-agent         ('User-Agent:')


VERSION HISTORY
---------------
12-AUG-98  MGD  MapUrl_VmsToUrl() now only optionally absorbs MFD
19-JUL-98  MGD  'E'xec DECnet-based user scripts
10-MAR-98  MGD  allow for local redirects (using '^' detect character)
07-FEB-98  MGD  added "sc:" scheme/protocol conditional for SSL support
20-DEC-97  MGD  added DECnet support in _UrlToVms(),
                result paths can now begin with an '*'
25-OCT-97  MGD  added "as:", "fo:", and "hh:" conditionals,
                bugfix; MsgFor() required request pointer to find langauge
15-SEP-97  MGD  MsgFor() was not returning a leading-null message (does now),
                added HTTP status code rules (e.g. "pass /no/* 403 forbidden"),
                removed two consecutive slashes inhibiting rule mapping
07-SEP-97  MGD  added "sn:" and "sp:" conditionals
09-AUG-97  MGD  v4.3, message database, mapping "conditionals",
                more cart-wheels, hand-stands and contortions (see 01-OCT-96)
05-JUN-97  MGD  v4.2, CGIplus "exec+" and "script+" rules
01-FEB-97  MGD  HTTPd version 4, major changes for form-based config
01-OCT-96  MGD  not proud of how I have shoe-horned the reporting features in,
                a good example of slop-down or object-disoriented programming,
                :^( will rework the whole thing one day, sigh!
                bugfix; requiring use of 'TheFirstRuleWillBeTheNextRule'
06-APR-96  MGD  modified conversion of non-VMS characters in MapUrl_UrlToVms()
15-FEB-95  MGD  bugfix; redirect rule
01-DEC-95  MGD  v3.0
24-MAR-95  MGD  bugfix; end-of-string sometimes not detected when rule matching
20-DEC-94  MGD  revised for multi-threaded HTTP daemon
20-JUN-94  MGD  single-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <prvdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* a little preprocessor config */
#undef IP_MAPURL
#ifdef IP_UCX
#   define IP_MAPURL 1
#endif /* IP_UCX */
#ifdef IP_TCPWARE
#   define IP_MAPURL 1
#endif /* IP_TCPWARE */
#ifdef IP_NETLIB
#   define IP_MAPURL 1
#endif /* IP_NETLIB */

/* application header files */
#include "wasd.h"
#include "mapurl.h"

#ifdef IP_MAPURL
#   include "vm.h"
#endif

/**********/
/* macros */
/**********/

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

/*******************/
/* data structures */
/*******************/

#ifdef __ALPHA
#   pragma member_alignment __save
#   pragma member_alignment
#endif

/* rule database is a singly-linked list of these structures */
struct MappingRuleStruct
{
   struct MappingRuleStruct  *NextPtr;
   boolean  IsCgiPlusScript;
   int  MapFileLineNumber;
   char  RuleType;
   char  *ConditionalPtr,
         *ResultPtr,
         *TemplatePtr;
   /* template/conditional/result strings stored straight after */
};

struct MappingRuleListStruct
{
   struct MappingRuleStruct  *HeadPtr;
   struct MappingRuleStruct  *TailPtr;
   int  LineNumber,
        ProblemCount,
        ProblemReportLength;
   unsigned long  LoadBinTime [2],
                  RevBinTime [2];
   char  *LinePtr,
         *ProblemReportPtr;
   char  LoadFileName [256],
         RuleName [32];
   /* only used when part of an HTTP request */
   struct RequestStruct  *RequestPtr;
};

#ifdef __ALPHA
#   pragma member_alignment __restore
#endif

/******************/
/* global storage */
/******************/

struct MappingRuleListStruct  MappingRulesList;

/*
   Point at the above storage as default for a rules list.
   This global pointer may be changed temporarily for using other rules.
*/
struct MappingRuleListStruct  *GlobalMappingRulesListPtr = &MappingRulesList;

/********************/
/* external storage */
/********************/

#ifdef IP_MAPURL

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

#else

extern boolean Debug;

#endif

extern char  SoftwareID[];
extern char  Utility[];

#ifdef IP_MAPURL

#include "msg.h"
#include "support.h"

extern char  HtmlSgmlDoctype[];
extern char  ServerHostPort[];
extern struct  MsgStruct  Msgs;

#endif /* IP_MAPURL */

/*****************************************************************************/
/*
This overly-long function (sorry) is not AST reentrant, but when executed 
within the context of the HTTPd, is executed either without the possibility of 
AST interruption, or during AST processing and so provides atomic 
functionality.  Pointers to any strings returned must be used immediately 
(before leaving AST context), or copied into storage maintained by the calling 
routine. 

Maps from URL format to VMS file specification, and REVERSE-MAPS from VMS file 
specification to URL format.  Maps scripts.

Always returns a pointer to char.  An error message is detected by being  
returned as a pointer to a string beginning with a null-character!  The 
message begins from character one. 

Call with 'PathPtr' pointing at a string containing the URL path and 'VmsPtr' 
pointing to the storage to contain the mapped VMS equivalent.

Call with 'VmsPtr' pointing to a VMS file specification (dev:[dir]file.ext) 
and 'PathPtr' pointing to an empty string, used as storage to contain the 
REVERSE-MAPPED file specification.

Call with 'PathPtr' pointing at a string containing the URL path, 'VmsPtr' 
pointing to storage to contain the mapped VMS equivalent, 'ScriptPtr' pointing 
to storage for the URL format script path and 'ScriptVmsPtr' pointing to 
storage to contain the VMS specification of the script procedure/image.

Alter-egos defined in MapUrl.h

        #define MapUrl(pp,vp,sp,svp) MapUrl_Map(pp,vp,sp,svp,NULL)
        #define MapVmsPath(vp) MapUrl_Map(NULL,vp,NULL,NULL,NULL)
*/ 

char* MapUrl_Map
(
char *PathPtr,
char *VmsPtr,
char *ScriptPtr,
char *ScriptVmsPtr,
void *rqptr
)
{
   static char  AccessDeniedDefault [] = 
                "\0Access denied (forbidden by default).",
                AccessDeniedInternal [] =
                "\0Access denied (INTERNAL processing problem!)",
                AccessDeniedNoRules [] = 
                "\0Access denied (NO RULES LOADED!)",
                AccessDeniedRule [] = 
                "\0Access denied (forbidden by rule).";

   static char  DerivedPathBuffer [512],
                PathBuffer [512],
                VmsBuffer [512];

   static struct MappingRuleStruct  TheFirstRuleWillBeTheNextRule;

   register char  PathChar,
                  RuleChar,
                  LastNonWildcardPathChar,
                  LastNonWildcardRuleChar;
   register char  *cptr, *pptr, *sptr, *zptr;
   register struct MappingRuleStruct  *mrptr;

   boolean  LoadingRules,
            MapPathToVms;
   int  status,
        TotalLength;
   unsigned short  Length;
   char  Scratch [512],
         WildBuffer [512];

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout, "MapUrl()");
      if (PathPtr != NULL) fprintf (stdout, " PathPtr: |%s|", PathPtr);
      if (VmsPtr != NULL) fprintf (stdout, " VmsPtr: |%s|", VmsPtr);
      if (ScriptPtr != NULL) fprintf (stdout, " (script)");
      fputc ('\n', stdout);
   }

   /* if just loading the rules then free any previous set */
   if (LoadingRules = (PathPtr == NULL && VmsPtr == NULL && ScriptPtr == NULL))
      MapUrl_FreeRules (GlobalMappingRulesListPtr);

   if (GlobalMappingRulesListPtr->HeadPtr == NULL)
   {
      /*********************/
      /* read mapping file */
      /*********************/

      cptr = MapUrl_LoadRules (GlobalMappingRulesListPtr);
      /* return message if an error reported */
      if (!cptr[0] && cptr[1]) return (cptr);
      /* point at the location of the first rule as the next rule */
      TheFirstRuleWillBeTheNextRule.NextPtr =
         GlobalMappingRulesListPtr->HeadPtr;
   }

   /* if just (re)loading rules then return now */
   if (LoadingRules) return ("\0\0");

   /*****************************************/
   /* map from the URL or VMS to VMS or URL */
   /*****************************************/

   if (PathPtr == NULL) *(PathBuffer) = *(PathPtr = PathBuffer+1) = '\0';
   if (VmsPtr == NULL) *(VmsPtr = VmsBuffer) = '\0';
   /* making 'ScriptPtr' non-empty means "exec" and "script" are ignored */
   if (ScriptPtr == NULL) ScriptPtr = "?";

   if (PathPtr[0])
   {
      /* if the URL is not an empty string then force the conversion to VMS */
      MapPathToVms = true;
      *VmsPtr = '\0';
   }
   else
   {
      /* URL was an empty string, convert from VMS to URL */
      MapPathToVms = false;
      /* generate a URL-style version of the VMS specification */
      MapUrl_VmsToUrl (PathPtr, VmsPtr, true);
   }

   /***********************/
   /* loop thru the rules */
   /***********************/

   if ((mrptr = GlobalMappingRulesListPtr->HeadPtr) == NULL)
#ifdef IP_MAPURL
      return (MsgFor(rqptr,MSG_MAPPING_DENIED_NO_RULES)-1);
#else
      return (AccessDeniedNoRules);
#endif

   /*
      This loop may be restarted by the "exec" or "script" rule after mapping
      the script component to further map the derived path.  This is done by
      setting up 'PathPtr' to the derived path and then repointing 'mrptr'
      to a scratch structure which has its 'NextPtr' set up to the first rule
      in the list and 'continue'ing the loop.
   */

   for (/* above! */; mrptr != NULL; mrptr = mrptr->NextPtr)
   {
      if (Debug)
         fprintf (stdout, "Rule |%c|%s|%s|\n",
                  mrptr->RuleType, mrptr->TemplatePtr, mrptr->ResultPtr);

      /* when mapping from VMS to URL then ignore all BUT "pass" rules */
      if (!MapPathToVms && mrptr->RuleType != 'P') continue;

#ifndef IP_MAPURL 

      /* when not mapping for HTTPd server completely ignore conditionals */
      if (mrptr->ConditionalPtr[0]) continue;

#endif  /* #ifndef IP_MAPURL */

      /* if 'ScriptPtr' is not to be used then ignore "exec" and "script" */
      if (ScriptPtr[0] && (mrptr->RuleType == 'E' || mrptr->RuleType == 'S'))
         continue;

      /*****************************************************************/
      /* compare the URL with the template/result in the mapping entry */
      /*****************************************************************/

      cptr = mrptr->TemplatePtr;
      /* if reverse-mapping VMS to URL use the result if available */
      if (!MapPathToVms && mrptr->ResultPtr[0]) cptr = mrptr->ResultPtr;

      /* 'WildBuffer' contains the null-separated wildcard-matched portions */
      zptr = (sptr = WildBuffer) + sizeof(WildBuffer)-1;
      LastNonWildcardPathChar = PathChar = *(pptr = PathPtr);
      LastNonWildcardRuleChar = RuleChar = *cptr;
      /** if (Debug) fprintf (stdout, "|%s|%s|\n", pptr, cptr); **/

      while (PathChar && RuleChar &&
             (tolower(PathChar) == tolower(RuleChar) || RuleChar == '*'))
      {
         if (RuleChar != '*')
         {
            LastNonWildcardPathChar = PathChar;
            LastNonWildcardRuleChar = RuleChar;
            PathChar = *++pptr;
            RuleChar = *++cptr;
            /** if (Debug) fprintf (stdout, "|%s|%s|\n", pptr, cptr); **/
            continue;
         }
         /** if (Debug) fprintf (stdout, "|%s|%s|\n", pptr, cptr); **/
         if ((mrptr->RuleType != 'S' && mrptr->RuleType != 'E') ||
             (mrptr->RuleType == 'S' || mrptr->RuleType == 'E') &&
             (tolower(LastNonWildcardPathChar) ==
              tolower(LastNonWildcardRuleChar) || !*pptr))
         {
            /* asterisk wildcard, multiple consecutive treated as single */
            while ((RuleChar = *cptr) == '*') cptr++;
            if (PathChar) *sptr++ = PathChar;
            if (sptr >= zptr)
#ifdef IP_MAPURL
               return (MsgFor(rqptr,MSG_MAPPING_DENIED_INTERNAL)-1);
#else
               return (AccessDeniedInternal);
#endif
            while ((PathChar = *++pptr) &&
                   tolower(PathChar) != tolower(RuleChar) &&
                   sptr < zptr)
               *sptr++ = PathChar;
            if (sptr >= zptr)
#ifdef IP_MAPURL
               return (MsgFor(rqptr,MSG_MAPPING_DENIED_INTERNAL)-1);
#else
               return (AccessDeniedInternal);
#endif
            *sptr++ = '\0';
            if (sptr >= zptr)
#ifdef IP_MAPURL
               return (MsgFor(rqptr,MSG_MAPPING_DENIED_INTERNAL)-1);
#else
               return (AccessDeniedInternal);
#endif
            /** if (Debug) fprintf (stdout, "|%s|%s|\n", pptr, cptr); **/
         }
         else
            while ((RuleChar = *cptr) == '*') cptr++;
      }

      /* account for trailing asterisk on rule template */
      if (RuleChar == '*')
         while ((RuleChar = *++cptr) == '*');

      /* if not matched then straight to next rule */
      if (PathChar || RuleChar)
      {
#ifdef IP_MAPURL
         if (GlobalMappingRulesListPtr->RequestPtr)
            MapUrl_ReportRule (mrptr, PathPtr, false);
#endif

         continue;
      }
#ifdef IP_MAPURL
      else
      {
         if (GlobalMappingRulesListPtr->RequestPtr)
            MapUrl_ReportRule (mrptr, PathPtr, true);
      }
#endif

      /* final, terminating null for 'WildBuffer' */
      *sptr = '\0';

      /****************/
      /* matched them */
      /****************/

#ifdef IP_MAPURL 

      /* when mapping for the HTTPd server apply any conditionals */
      if (mrptr->ConditionalPtr[0])
      {
         /* when not mapping using a request then ignore conditionals */
         if (rqptr != NULL)
         {
            /* if condition not met then continue */
            if (!MapUrl_Conditional ((struct RequestStruct*)rqptr,
                                     mrptr->ConditionalPtr))
               continue;
         }
      }

#endif

      if (MapPathToVms)
      {
         /***************************/
         /* mapping from URL to VMS */
         /***************************/

         switch (mrptr->RuleType)
         {
         case 'E' :

            /* exec directory mapping rule (different to "script") */

            /* get the virtual path to the script (CGI: "WWW_SCRIPT_NAME") */
            cptr = WildBuffer;
            sptr = mrptr->TemplatePtr;
            zptr = (pptr = ScriptPtr) + 255;

            if (*(unsigned short*)PathPtr == '/~')
            {
               /* e.g. exec /~@/cgi-bin/@ /0""::/web/user/@/cgi-bin/@ */
               while (*sptr && *sptr != '*' && pptr < zptr) *pptr++ = *sptr++;
               while (*sptr == '*') sptr++;
               if (Debug) fprintf (stdout, "WildBuffer |%s|\n", cptr);
               while (*cptr && pptr < zptr) *pptr++ = *cptr++;
               cptr++;
               /* directory name */
               while (*sptr && *sptr != '*' && pptr < zptr) *pptr++ = *sptr++;
               /* get the wildcard buffer up to the first slash (script name) */
               while (*cptr && *cptr != '/' && pptr < zptr) *pptr++ = *cptr++;
            }
            else
            {
               /* e.g. exec /cgi-bin/@ /cgi-bin/@ */
               while (*sptr && *sptr != '*' && pptr < zptr) *pptr++ = *sptr++;
               if (Debug) fprintf (stdout, "WildBuffer |%s|\n", cptr);
               while (*cptr && *cptr != '/' && pptr < zptr) *pptr++ = *cptr++;
            }
            if (pptr >= zptr)
#ifdef IP_MAPURL
               return (MsgFor(rqptr,MSG_MAPPING_DENIED_INTERNAL)-1);
#else
               return (AccessDeniedInternal);
#endif
            *pptr = '\0';

            cptr = WildBuffer;
            sptr = mrptr->ResultPtr;
            zptr = (pptr = Scratch) + 255;

            if (*(unsigned short*)PathPtr == '/~')
            {
               /* e.g. exec /~@/cgi-bin/@ /0""::/web/user/@/cgi-bin/@ */
               while (*sptr && *sptr != '*'  && *sptr != '\"' && pptr < zptr)
                  *pptr++ = *sptr++;
               if (*(unsigned long*)sptr == '\"\"::')
               {
                  if (Debug) fprintf (stdout, "WildBuffer |%s|\n", cptr);
                  if (pptr < zptr) *pptr++ = *sptr++;
                  while (*cptr && pptr < zptr) *pptr++ = *cptr++;
                  cptr = WildBuffer;
                  if (pptr < zptr) *pptr++ = *sptr++;
                  /* directory name */
                  while (*sptr && *sptr != '*' && pptr < zptr) *pptr++ = *sptr++;
               }
               /* username as subdirectory */
               while (*sptr == '*') sptr++;
               if (Debug) fprintf (stdout, "WildBuffer |%s|\n", cptr);
               while (*cptr && pptr < zptr) *pptr++ = *cptr++;
               cptr++;
               /* script directory, then substitute name */
               while (*sptr && *sptr != '*' && pptr < zptr) *pptr++ = *sptr++;
               while (*sptr == '*') sptr++;
               if (Debug) fprintf (stdout, "WildBuffer |%s|\n", cptr);
               while (*cptr && *cptr != '/' && pptr < zptr) *pptr++ = *cptr++;
            }
            else
            {
               /* e.g. exec /cgi-bin/@ /cgi-bin/@ */
               while (*sptr && *sptr != '*' && pptr < zptr) *pptr++ = *sptr++;
               if (Debug) fprintf (stdout, "WildBuffer |%s|\n", cptr);
               while (*cptr && *cptr != '/' && pptr < zptr) *pptr++ = *cptr++;
            }
            if (pptr >= zptr)
#ifdef IP_MAPURL
               return (MsgFor(rqptr,MSG_MAPPING_DENIED_INTERNAL)-1);
#else
               return (AccessDeniedInternal);
#endif
            *pptr = '\0';

            /* convert the URL-format script to a VMS-format specification */
            MapUrl_UrlToVms (Scratch, ScriptVmsPtr);

            /* indicate CGIplus script via leading '+' instead of '/' */
            if (mrptr->IsCgiPlusScript) ScriptPtr[0] = '+';

            /* get all including and after the first slash as the new path */
            zptr = (pptr = DerivedPathBuffer) + sizeof(DerivedPathBuffer)-1;
            while (*cptr && pptr < zptr) *pptr++ = *cptr++;
            if (pptr >= zptr)
#ifdef IP_MAPURL
               return (MsgFor(rqptr,MSG_MAPPING_DENIED_INTERNAL)-1);
#else
               return (AccessDeniedInternal);
#endif
            *pptr++ = '\0';
            TotalLength = pptr - DerivedPathBuffer;

            if (Debug)
               fprintf (stdout, "EXEC |%s|%s|%s|\n",
                        ScriptPtr, ScriptVmsPtr, DerivedPathBuffer);

            if (!DerivedPathBuffer[0]) return ("\0\0");

            /* restarting at first rule map the path derived from the script */
            *VmsPtr = '\0';
            PathPtr = DerivedPathBuffer;
            mrptr = &TheFirstRuleWillBeTheNextRule;
            continue;

         case 'F' :

            if (Debug) fprintf (stdout, "FAIL |%s|\n", PathPtr);
#ifdef IP_MAPURL
            return (MsgFor(rqptr,MSG_MAPPING_DENIED_RULE)-1);
#else
            return (AccessDeniedRule);
#endif

         case 'M' :

            /* map the result onto the original (even for "redirect") */

            cptr = WildBuffer;
            /* place this in a buffer so not to overwrite original path */
            zptr = (pptr = PathPtr = PathBuffer+1) + sizeof(PathBuffer)-2;
            sptr = mrptr->ResultPtr;
            /* scan through the result string */
            while (*sptr)
            {
               while (*sptr && *sptr != '*' && pptr < zptr) *pptr++ = *sptr++;
               if (pptr >= zptr)
#ifdef IP_MAPURL
               return (MsgFor(rqptr,MSG_MAPPING_DENIED_INTERNAL)-1);
#else
               return (AccessDeniedInternal);
#endif
               if (!*sptr) break;
               /* a wildcard asterisk, substitute from original path */
               while (*++sptr == '*');
               while (*cptr && pptr < zptr) *pptr++ = *cptr++;
               if (pptr >= zptr)
#ifdef IP_MAPURL
                  return (MsgFor(rqptr,MSG_MAPPING_DENIED_INTERNAL)-1);
#else
                  return (AccessDeniedInternal);
#endif
               cptr++;
            }
            *pptr = '\0';

            /* continue with the substituted URL mapping */
            if (Debug) fprintf (stdout, "MAP |%s|\n", PathPtr);
            continue;

         case 'P' :
         case 'R' :

            /* if there is no result to map just pass back the original path */
            if (!*(sptr = mrptr->ResultPtr)) 
            {
               /* convert the URL-style to a VMS-style specification */
               MapUrl_UrlToVms (PathPtr, VmsPtr);

               if (mrptr->RuleType == 'R')
               {
                  if (Debug)
                     fprintf (stdout, "REDIRECT |%s|%s|\n", PathPtr, VmsPtr);
                  /* indicate it's a local direct (which this has to be!) */
                  *PathPtr = '^';
               }
               else
               {
                  if (Debug)
                     fprintf (stdout, "PASS |%s|%s|\n", PathPtr, VmsPtr);
               }
               return (PathPtr);
            }

            cptr = WildBuffer;
            zptr = (pptr = PathBuffer+1) + sizeof(PathBuffer)-2;
            /* scan through the result string */
            while (*sptr)
            {
               if (mrptr->RuleType == 'P' && isdigit(*sptr))
               {
                  /* HTTP status code mapping, ignore any asterisks */
                  while (*sptr && pptr < zptr) *pptr++ = *sptr++;
               }
               else
               {
                  /* map wildcard asterisks */
                  while (*sptr && *sptr != '*' && pptr < zptr)
                      *pptr++ = *sptr++;
               }
               if (pptr >= zptr)
#ifdef IP_MAPURL
                  return (MsgFor(rqptr,MSG_MAPPING_DENIED_INTERNAL)-1);
#else
                  return (AccessDeniedInternal);
#endif
               if (!*sptr) break;
               /* a wildcard asterisk, substitute from original path */
               while (*++sptr == '*');
               while (*cptr && pptr < zptr) *pptr++ = *cptr++;
               if (pptr >= zptr)
#ifdef IP_MAPURL
                  return (MsgFor(rqptr,MSG_MAPPING_DENIED_INTERNAL)-1);
#else
                  return (AccessDeniedInternal);
#endif
               cptr++;
            }
            *pptr = '\0';

            if (mrptr->RuleType == 'P')
            {
               if (isdigit(*(PathBuffer+1))) return (PathBuffer);

               /* convert the URL-style to a VMS-style specification */
               MapUrl_UrlToVms (PathBuffer+1, VmsPtr);
               if (Debug) fprintf (stdout, "PASS |%s|%s|\n", PathPtr, VmsPtr);
               return (PathPtr);
            }
            else
            {
               if (Debug)
                  fprintf (stdout, "REDIRECT |%s|%s|\n", PathPtr, VmsPtr);
               /* indicate if it's a local direct */
               if (PathBuffer[1] == '/') PathBuffer[1] = '^';
               return (PathBuffer+1);
            }


         case 'S' :

            /* script file name mapping rule (different to "exec") */

            /* get the virtual path to the script (CGI: "WWW_SCRIPT_NAME") */
            sptr = mrptr->TemplatePtr;
            zptr = (pptr = ScriptPtr) + 255;
            while (*sptr && *sptr != '*' && pptr < zptr) *pptr++ = *sptr++;
            if (pptr >= zptr)
#ifdef IP_MAPURL
               return (MsgFor(rqptr,MSG_MAPPING_DENIED_INTERNAL)-1);
#else
               return (AccessDeniedInternal);
#endif
            if (pptr > ScriptPtr)
               if (pptr[-1] == '/')
                  pptr--;
            *pptr = '\0';

            /* now get the URL-format path to the physical script */
            zptr = (pptr = Scratch) + 255;
            sptr = mrptr->ResultPtr;
            /* 0x2a2f is the numeric equivalent of "/*" */
            while (*sptr && *sptr != '*' &&
                   *(unsigned short*)sptr != 0x2a2f && pptr < zptr)
               *pptr++ = *sptr++;
            if (pptr >= zptr)
#ifdef IP_MAPURL
               return (MsgFor(rqptr,MSG_MAPPING_DENIED_INTERNAL)-1);
#else
               return (AccessDeniedInternal);
#endif
            *pptr = '\0';

            /* convert the URL-format script to a VMS-format specification */
            MapUrl_UrlToVms (Scratch, ScriptVmsPtr);

            /* indicate CGIplus script via leading '+' instead of '/' */
            if (mrptr->IsCgiPlusScript) ScriptPtr[0] = '+';

            /* get wildcard matched second section of path as new path */
            zptr = (pptr = DerivedPathBuffer) + sizeof(DerivedPathBuffer)-1;
            if (*sptr == '/') *pptr++ = '/';
            cptr = WildBuffer;
            while (*cptr && pptr < zptr) *pptr++ = *cptr++;
            if (pptr >= zptr)
#ifdef IP_MAPURL
               return (MsgFor(rqptr,MSG_MAPPING_DENIED_INTERNAL)-1);
#else
               return (AccessDeniedInternal);
#endif
            *pptr++ = '\0';
            TotalLength = pptr - DerivedPathBuffer;

            if (Debug)
               fprintf (stdout, "SCRIPT |%s|%s|%s|\n",
                        ScriptPtr, ScriptVmsPtr, DerivedPathBuffer);

            if (!DerivedPathBuffer[0]) return ("\0\0");

            /* restarting at first rule map the path derived from the script */
            *VmsPtr = '\0';
            PathPtr = DerivedPathBuffer;
            mrptr = &TheFirstRuleWillBeTheNextRule;
            continue;
         }
      }
      else
      {
         /***************************/
         /* mapping from VMS to URL */
         /***************************/

         /* REVERSE maps a VMS "result" to a "template" :^) */

         cptr = WildBuffer;
         pptr = PathPtr;
         sptr = mrptr->TemplatePtr;
         /* scan through the template string */
         while (*sptr)
         {
            while (*sptr && *sptr != '*') *pptr++ = *sptr++;
            if (!*sptr) break;
            /* a wildcard asterisk, substitute from result path */
            while (*++sptr == '*');
            if (*cptr)
            {
               while (*cptr) *pptr++ = *cptr++;
               cptr++;
            }
         }
         *pptr = '\0';

         if (Debug) fprintf (stdout, "PASS |%s|\n", PathPtr);
         return (PathPtr);
      }
   }

   /***********************/
   /* a mapping not found */
   /***********************/

#ifdef IP_MAPURL
   return (MsgFor(rqptr,MSG_MAPPING_DENIED_DEFAULT)-1);
#else
   return (AccessDeniedDefault);
#endif
}

/*****************************************************************************/
/*
Open the mapping rule file and reading each line place mapping rules into the
singly-linked list pointed to by the supplied list pointer.  Close the file.
*/ 
 
char* MapUrl_LoadRules (struct MappingRuleListStruct *rlptr)

{
   static unsigned long  SysPrvMask [2] = { PRV$M_SYSPRV, 0 };

   static int  MappingRuleStructOverhead = sizeof(struct MappingRuleStruct);

   static char  ProblemConditional [] = "conditional problem",
                ProblemConfused [] = "confused",
                ProblemResultPath [] = "\"result\" path not absolute",
                ProblemResultRequired [] = "requires a \"result\"",
                ProblemTemplatePath [] = "\"template\" path not absolute",
                ProblemTemplateRequired [] = "requires a \"template\"",
                ProblemWildcardMapping [] = "wildcard mapping problem";

   static char  ErrorString [256];

   register char  RuleChar;
   register char  *cptr, *sptr, *zptr;
   register struct MappingRuleStruct  *mrptr;

   boolean  ConditionalProblem,
            DebugBuffer,
            IsCgiPlusScript,
            RequiresResult;
   int  status,
        ConditionalOffset,
        SetPrvStatus,
        TemplateWildcardCount,
        TotalLength,
        ResultOffset,
        ResultWildcardCount;
   unsigned short  Length;
   char  *ConditionalPtr,
         *ResultPtr,
         *TemplatePtr;
   char  ExpandedFileName [256],
         Line [512],
         LineBuffer [512];
   $DESCRIPTOR (ErrorStringDsc, ErrorString);
   struct FAB  MapFileFab;
   struct NAM  MapFileNam;
   struct RAB  MapFileRab;
   struct XABDAT  MapFileXabDat;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MapUrl_LoadRules() |%s|\n", MapFileName);

#ifdef DBUG
   /* not interested anymore in seeing debug information for rule load! */
   DebugBuffer = Debug;
   Debug = false;
#endif

   MapUrl_FreeRules (rlptr);

   if (rlptr->ProblemReportPtr != NULL) free (rlptr->ProblemReportPtr);

   memset (rlptr, 0, sizeof(struct MappingRuleListStruct));

   sys$gettim (&rlptr->LoadBinTime);

   MapFileFab = cc$rms_fab;
   MapFileFab.fab$b_fac = FAB$M_GET;
   MapFileFab.fab$l_fna = MapFileName;
   MapFileFab.fab$b_fns = strlen(MapFileName);
   MapFileFab.fab$l_nam = &MapFileNam;
   MapFileFab.fab$b_shr = FAB$M_SHRGET;
   MapFileFab.fab$l_xab = &MapFileXabDat;

   MapFileNam = cc$rms_nam;
   MapFileNam.nam$l_esa = ExpandedFileName;
   MapFileNam.nam$b_ess = sizeof(ExpandedFileName)-1;
   MapFileNam.nam$l_rsa = rlptr->LoadFileName;
   MapFileNam.nam$b_rss = sizeof(rlptr->LoadFileName)-1;

   /* initialize the date extended attribute block */
   MapFileXabDat = cc$rms_xabdat;

   /* turn on SYSPRV to allow access to possibly protected file */
   if (VMSnok (SetPrvStatus = sys$setprv (1, &SysPrvMask, 0, 0)))
#ifdef IP_MAPURL
      ErrorExitVmsStatus (SetPrvStatus, "sys$setprv()", FI_LI);
#else
      exit (SetPrvStatus);
#endif

   status = sys$open (&MapFileFab, 0, 0);

   /* turn off SYSPRV */
   if (VMSnok (SetPrvStatus = sys$setprv (0, &SysPrvMask, 0, 0)))
#ifdef IP_MAPURL
      ErrorExitVmsStatus (SetPrvStatus, "sys$setprv()", FI_LI);
#else
      exit (SetPrvStatus);
#endif

   /* status from sys$open() */
   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      cptr = MapUrl_ReportProblem (rlptr, NULL, status);
      ErrorString[0] = '\0';
      strcpy (ErrorString+1, cptr);
#ifdef DBUG
      Debug = DebugBuffer;
#endif
      return (ErrorString);
   }

   rlptr->LoadFileName[MapFileNam.nam$b_rsl] = '\0';
   if (Debug) fprintf (stdout, "LoadFileName |%s|\n", rlptr->LoadFileName);

   /* get the configuration file revision date and time */
   memcpy (&rlptr->RevBinTime,
           &MapFileXabDat.xab$q_rdt,
           sizeof(rlptr->RevBinTime));

   /* record access block */
   MapFileRab = cc$rms_rab;
   MapFileRab.rab$l_fab = &MapFileFab;
   /* try and optimise sequential read (next 3 lines) */
   MapFileRab.rab$b_mbc = 6;
   MapFileRab.rab$b_mbf = 2;
   MapFileRab.rab$l_rop = RAB$M_RAH;
   MapFileRab.rab$l_ubf = Line;
   MapFileRab.rab$w_usz = sizeof(Line)-1;

   if (VMSnok (status = sys$connect (&MapFileRab, 0, 0)))
   {
      sys$close (&MapFileFab, 0, 0);
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      cptr = MapUrl_ReportProblem (rlptr, NULL, status);
      ErrorString[0] = '\0';
      strcpy (ErrorString+1, cptr);
#ifdef DBUG
      Debug = DebugBuffer;
#endif
      return (ErrorString);
   }

   /*************************/
   /* read the mapping file */
   /*************************/

   rlptr->LineNumber = 0;

   while (VMSok (status = sys$get (&MapFileRab, 0, 0)))
   {
      rlptr->LineNumber++;
      MapFileRab.rab$l_ubf[MapFileRab.rab$w_rsz] = '\0';
      if (Debug)
         fprintf (stdout, "MapFileRab.rab$l_ubf |%s|\n", MapFileRab.rab$l_ubf);

      if (MapFileRab.rab$w_rsz)
      {
         if (MapFileRab.rab$l_ubf[MapFileRab.rab$w_rsz-1] == '\\')
         {
            /* line is continued, effectively remove backslash, get more */
            MapFileRab.rab$l_ubf += MapFileRab.rab$w_rsz-1;
            MapFileRab.rab$w_usz -= MapFileRab.rab$w_rsz-1;
            continue;
         }
      }
      /* reset the line buffer */
      MapFileRab.rab$l_ubf = Line;
      MapFileRab.rab$w_usz = sizeof(Line)-1;

      if (Debug) fprintf (stdout, "Line |%s|\n", Line);
      memcpy (rlptr->LinePtr = LineBuffer, Line, MapFileRab.rab$w_rsz+1);
      cptr = Line;
      while (ISLWS(*cptr)) cptr++;

      /* if a blank or comment line, then ignore */
      if (!*cptr || *cptr == '!' || *cptr == '#') continue;

      /* get the first character, then buffer the rule in case of error */
      RuleChar = toupper(*cptr);
      zptr = (sptr = rlptr->RuleName) + sizeof(rlptr->RuleName);
      while (*cptr && !ISLWS(*cptr) && sptr < zptr) *sptr++ = *cptr++;
      if (sptr >= zptr)
      {
         *--sptr = '\0';
         MapUrl_ReportProblem (rlptr, ProblemConfused, 0);
         continue;
      }
      *sptr = '\0';
      if (sptr > rlptr->RuleName)
         if (*(sptr-1) == '+')
            IsCgiPlusScript = true;
         else
            IsCgiPlusScript = false;
      while (ISLWS(*cptr)) cptr++;

      /* extract template */
      TemplatePtr = cptr;
      while (*cptr && !ISLWS(*cptr)) cptr++;
      /* note the position of the empty string in case neither of these! */
      ResultPtr = ConditionalPtr = cptr;
      ResultOffset = ConditionalOffset = cptr - TemplatePtr;
      /* terminate in the white-space following the template */
      if (*cptr) *cptr++ = '\0';
      /* skip intervening white-space */
      while (ISLWS(*cptr)) cptr++;

      if (*cptr != '[' && !(*cptr == '!' && *(cptr+1) == '['))
      {
         /* extract result */
         if (*cptr == '\"')
         {
            /* delimitted by double quotes */
            ResultPtr = ++cptr;
            ResultOffset = cptr - TemplatePtr;
            while (*cptr && *cptr != '\"') cptr++;
            if (*cptr) *cptr++ = '\0';
         }
         else
         if (*cptr == '\'')
         {
            /* delimitted by single quotes */
            ResultPtr = ++cptr;
            ResultOffset = cptr - TemplatePtr;
            while (*cptr && *cptr != '\'') cptr++;
            if (*cptr) *cptr++ = '\0';
         }
         else
         if (*cptr == '{')
         {
            /* delimitted by curly braces */
            ResultPtr = ++cptr;
            ResultOffset = cptr - TemplatePtr;
            while (*cptr && *cptr != '}') cptr++;
            if (*cptr) *cptr++ = '\0';
         }
         else
         {
            /* normal path mapping, up to first white-space */
            ResultPtr = cptr;
            ResultOffset = cptr - TemplatePtr;
            while (*cptr && !ISLWS(*cptr)) cptr++;
            /* terminate in the white-space following the path result */
            if (*cptr) *cptr++ = '\0';
         }
         /* skip intervening white-space */
         while (ISLWS(*cptr)) cptr++;
      }

      /* check for optional mapping conditional(s) */
      ConditionalProblem = false;

      if (*cptr == '[' || (*cptr == '!' && *(cptr+1) == '['))
      {
         ConditionalPtr = cptr;
         ConditionalOffset = cptr - TemplatePtr;
         while (*cptr == '[' || *cptr == '!')
         {
            while (*cptr == '[' || *cptr == '!') cptr++;
            while (*cptr && *cptr != ']')
            {
               while (ISLWS(*cptr)) cptr++;
               sptr = cptr;
               while (*cptr && *cptr != ':' && !ISLWS(*cptr) && *cptr != ']')
               {
                  *cptr = toupper(*cptr);
                  cptr++;
               }
               /* basic check of conditional rule */
               if (*sptr == '!') sptr++;
               if ((*(unsigned short*)sptr != 'AC' &&
                    *(unsigned short*)sptr != 'AL' &&
                    *(unsigned short*)sptr != 'AS' &&
                    *(unsigned short*)sptr != 'FO' &&
                    *(unsigned short*)sptr != 'HO' &&
                    *(unsigned short*)sptr != 'HH' &&
                    *(unsigned short*)sptr != 'ME' &&
                    *(unsigned short*)sptr != 'SC' &&
                    *(unsigned short*)sptr != 'SN' &&
                    *(unsigned short*)sptr != 'SP' &&
                    *(unsigned short*)sptr != 'RU' &&
                    *(unsigned short*)sptr != 'UA') ||
                   sptr[2] != ':')
               {
                  ConditionalProblem = true;
                  break;
               }
               while (*cptr && !ISLWS(*cptr) && *cptr != ']') cptr++;
               while (ISLWS(*cptr)) cptr++;
            }
            if (*cptr == ']') cptr++;
            /* look ahead */
            for (sptr = cptr; ISLWS(*sptr); sptr++);
            if (*sptr == '[' || *sptr == '!')
            {
               cptr = sptr;
               continue;
            }
         }
         /* terminate in the white-space following the conditional(s) */
         if (*cptr) *cptr = '\0';
      }

      TotalLength = cptr - TemplatePtr + 1;

      if (Debug)
         fprintf (stdout, "|%c|%d|%s|%s|%s|\n",
                  RuleChar, IsCgiPlusScript,
                  TemplatePtr, ConditionalPtr, ResultPtr);

      /******************************/
      /* basic consistency checking */
      /******************************/

      if (ConditionalProblem)
      {
         MapUrl_ReportProblem (rlptr, ProblemConditional, 0);
         continue;
      }

      TemplateWildcardCount = 0;
      for (cptr = TemplatePtr; *cptr; cptr++)
         if (*cptr == '*') TemplateWildcardCount++;

      ResultWildcardCount = 0;
      for (cptr = ResultPtr; *cptr; cptr++)
         if (*cptr == '*') ResultWildcardCount++;

      RequiresResult = false;
      switch (RuleChar)
      {
         case 'E' :
         case 'S' :
            if (ResultWildcardCount < 1 &&
                TemplateWildcardCount < 1 &&
                ResultWildcardCount != TemplateWildcardCount)
            {
               MapUrl_ReportProblem (rlptr, ProblemWildcardMapping, 0);
               continue;
            }
            RequiresResult = true;
            break;

         case 'F' :
         case 'P' :
            break;

         case 'M' :
         case 'R' :
            RequiresResult = true;
            break;

         default :
            MapUrl_ReportProblem (rlptr, "?", 0);
            continue;
      }

      if (!TemplatePtr[0])
      {
         /* there must be a "template" to map from */
         MapUrl_ReportProblem (rlptr, ProblemTemplateRequired, 0);
         continue;
      }

      if (TemplatePtr[0] != '/')
      {
         /* all paths must be absolute */
         MapUrl_ReportProblem (rlptr, ProblemTemplatePath, 0);
         continue;
      }

      if (RequiresResult && !ResultPtr[0])
      {
         /* there must be a "result" to map to */
         MapUrl_ReportProblem (rlptr, ProblemResultRequired, 0);
         continue;
      }

      if (RuleChar != 'P' && isdigit(ResultPtr[0]))
      {
         /* only pass rules may use the status code format */
         MapUrl_ReportProblem (rlptr, ProblemResultPath, 0);
         continue;
      }

      if ((RuleChar != 'R' && ResultPtr[0] &&
           ResultPtr[0] != '/' && ResultPtr[0] != '*') &&
          !(RuleChar == 'P' && isdigit(ResultPtr[0])))
      {
         /* all paths must be absolute (except for redirect) */
         MapUrl_ReportProblem (rlptr, ProblemResultPath, 0);
         continue;
      }

      if (ResultWildcardCount > TemplateWildcardCount &&
          !(RuleChar == 'P' && isdigit(ResultPtr[0])))
      {
         /* cannot wildcard map more elements than in template */
         MapUrl_ReportProblem (rlptr, ProblemWildcardMapping, 0);
         continue;
      }

      /************************************/
      /* enter the rule into the database */
      /************************************/

#ifdef IP_MAPURL
      mrptr = (struct MappingRuleStruct*)
         VmGet (MappingRuleStructOverhead+TotalLength);
#else
      if ((mrptr = (struct MappingRuleStruct*)
           malloc (MappingRuleStructOverhead+TotalLength)) == NULL)
      {
         /* no point in continuing if we can't get memory for essentials */
         exit (vaxc$errno);
      }
#endif

      cptr = (char*)mrptr + MappingRuleStructOverhead;
      memcpy (cptr, TemplatePtr, TotalLength);
      mrptr->NextPtr = NULL;
      mrptr->MapFileLineNumber = rlptr->LineNumber;
      mrptr->RuleType = RuleChar;
      mrptr->IsCgiPlusScript = IsCgiPlusScript;
      mrptr->TemplatePtr = cptr;
      mrptr->ConditionalPtr = cptr + ConditionalOffset;
      mrptr->ResultPtr = cptr + ResultOffset;
      if (rlptr->HeadPtr == NULL)
         rlptr->HeadPtr = mrptr;
      else
         rlptr->TailPtr->NextPtr = mrptr;
      rlptr->TailPtr = mrptr;
   }

   /**************************/
   /* close the mapping file */
   /**************************/

   /** if (Debug) fprintf (stdout, "sys$get() %%X%08.08X\n", status); **/
   if (status == RMS$_EOF) status = SS$_NORMAL;

   sys$close (&MapFileFab, 0, 0);

   if (VMSok (status))
      *(unsigned long*)ErrorString = 0;
   else
   {
      MapUrl_FreeRules (rlptr);
      cptr = MapUrl_ReportProblem (rlptr, NULL, status);
      ErrorString[0] = '\0';
      strcpy (ErrorString+1, cptr);
   }

#ifdef DBUG
   Debug = DebugBuffer;
#endif
   return (ErrorString);
}

/*****************************************************************************/
/*
This function formats an error report.  All lines are concatenated onto a
single string of dynamically allocated memory that (obviously) grows as
reports are added to it.  This string is then output if loading the server
configuration or is available for inclusion in an HTML page.
*/

char* MapUrl_ReportProblem
(
struct MappingRuleListStruct *rlptr,
char *Explanation,
int StatusValue
)
{
   static $DESCRIPTOR (RuleFaoDsc,
          "%HTTPD-W-MAPURL, \"!AZ\" rule !AZ at line !UL\n \\!AZ\\\n");
   static $DESCRIPTOR (ExplanationFaoDsc,
          "%HTTPD-W-MAPURL, !AZ at line !UL\n \\!AZ\\\n");

   static char  Buffer [512];
   static $DESCRIPTOR (BufferDsc, Buffer);

   int  status;
   unsigned short  Length;
   char  HtmlBuffer [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MapUrl_ReportProblem()\n");

   rlptr->ProblemCount++;

   if (StatusValue)
   {
      if (VMSok (status = sys$getmsg (StatusValue, &Length, &BufferDsc, 0, 0)))
         Buffer[Length++] = '\n';
   }
   else
   if (Explanation == NULL)
      status = sys$fao (&RuleFaoDsc, &Length, &BufferDsc,
                        rlptr->RuleName, "problem",
                        rlptr->LineNumber, rlptr->LinePtr);
   else
   if (Explanation[0] == '?')
      status = sys$fao (&RuleFaoDsc, &Length, &BufferDsc,
                        rlptr->RuleName, "unknown",
                        rlptr->LineNumber, rlptr->LinePtr);
   else
      status = sys$fao (&ExplanationFaoDsc, &Length, &BufferDsc,
                        Explanation,
                        rlptr->LineNumber, rlptr->LinePtr);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
#ifdef IP_MAPURL
         ErrorExitVmsStatus (status, "sys$fao()", FI_LI);
#else
         exit (status);
#endif
   else
      Buffer[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", Buffer);
   if (rlptr->RequestPtr == NULL) fputs (Buffer, stdout);

   if ((Length =
       MapUrl_CopyToHtml (HtmlBuffer, sizeof(HtmlBuffer), Buffer, -1)) < 0)
   {
#ifdef IP_MAPURL
      ErrorExitVmsStatus (SS$_BUFFEROVF, "CopyToHtml()", FI_LI);
#else
      exit (SS$_BUFFEROVF & 0xfffffffe);
#endif
   }

   if ((rlptr->ProblemReportPtr =
        realloc (rlptr->ProblemReportPtr, rlptr->ProblemReportLength+Length+1))
       == NULL)
#ifdef IP_MAPURL
         ErrorExitVmsStatus (vaxc$errno, "realloc()", FI_LI);
#else
         exit (status);
#endif
   /* include the terminating null */
   memcpy (rlptr->ProblemReportPtr+rlptr->ProblemReportLength,
           HtmlBuffer, Length+1);
   rlptr->ProblemReportLength += Length;

   return (Buffer);
}

/*****************************************************************************/
/*
Free all entries in the singly-linked rule list.
*/ 
 
MapUrl_FreeRules (struct MappingRuleListStruct *rlptr)

{
   register char  *cptr;
   register struct MappingRuleStruct  *mrptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MapUrl_FreeRules()\n");

   mrptr = rlptr->HeadPtr;
   while (mrptr != NULL)
   {
      cptr = (char*)mrptr;
      mrptr = mrptr->NextPtr;
#ifdef IP_MAPURL
      VmFree (cptr);
#else
      free (cptr);
#endif
   }
   rlptr->HeadPtr = rlptr->TailPtr = NULL;
}

/*****************************************************************************/
/*
Convert a URL-style specification into a RMS-style specification.  For 
example: "/disk/dir1/dir2/file.txt" into "disk:[dir1.dir2]file.txt".  Non-RMS 
characters (non-alphanumberic and non-"$_-" are converted to "$'.  Where a 
specification has multiple ".", all but the final one of the file component is 
converted to a "$".  Returns length of VMS specification.
*/ 
 
int MapUrl_UrlToVms
(
char *PathPtr,
char *VmsPtr
)
{
   register int  ncnt, ccnt;
   register char  *cptr, *pptr, *sptr;

   boolean  DECnet;
   char  PathComponents [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MapUrl_UrlToVms() |%s|\n", PathPtr);

   if (!PathPtr[0] || (PathPtr[0] == '/' && !PathPtr[1]))
   {
      VmsPtr[0] = VmsPtr[1] = '\0';
      return (0);
   }

   /******************************************/
   /* parse out each "/" delimited component */
   /******************************************/

   /* each component becomes a null-terminated string in a character array */
   DECnet = false;
   sptr = PathComponents;
   pptr = PathPtr;
   ccnt = 0;
   while (*pptr)
   {
      if (*pptr == '/') pptr++;
      /* copy, converting all non-RMS characters into exclamation marks */
      cptr = sptr;
      while (*pptr && *pptr != '/')
      {
         if (isalnum (*pptr))
            *sptr++ = toupper(*pptr++);
         else
         if (pptr[0] == '.' && pptr[1] == '.' && pptr[2] == '.')
         {
            *sptr++ = *pptr++;
            *sptr++ = *pptr++;
            *sptr++ = *pptr++;
         }
         else
         if (*pptr == '$' || *pptr == '_' || *pptr == '-' ||
             *pptr == '*' || *pptr == '%' || *pptr == ';' ||
             *pptr == '\"' || *pptr == '=')
            *sptr++ = *pptr++;
         else
         if (!ccnt && *(unsigned short*)pptr == '::')
         {
            /* DECnet (with or without access string) */
            DECnet = true;
            *sptr++ = *pptr++;
            *sptr++ = *pptr++;
         }
         else
         {
            *sptr++ = '!';
            pptr++;
         }
      }
      *sptr++ = '\0';
      ccnt++;
      if (Debug) fprintf (stdout, "%d |%s|\n", ccnt, cptr);
   }

   /********************************/
   /* build VMS file specification */
   /********************************/

   if (DECnet) ccnt--;
   if (ccnt == 1)
   {
      /* single component */
      cptr = PathComponents;
      sptr = VmsPtr;
      while (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
   }
   else
   if (ccnt == 2)
   {
      /***********************/
      /* top-level directory */
      /***********************/

      cptr = PathComponents;
      sptr = VmsPtr;
      if (DECnet)
      {
         /* copy first component as-is */
         while (*cptr) *sptr++ = *cptr++;
         cptr++;
      }
      /* start with the first component, the "device" */
      while (*cptr) *sptr++ = *cptr++;
      cptr++;
      /* append a Master File Directory component */
      strcpy (sptr, ":[000000]");
      while (*sptr) sptr++;
      /* add the second component, the "file" */
      while (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
   }
   else
   if (ccnt >= 3)
   {
      /****************/
      /* subdirectory */
      /****************/

      cptr = PathComponents;
      sptr = VmsPtr;
      if (DECnet)
      {
         /* copy first component as-is */
         while (*cptr) *sptr++ = *cptr++;
         cptr++;
      }
      /* start with the first component, the "device" */
      while (*cptr) *sptr++ = *cptr++;
      cptr++;
      *sptr++ = ':';
      *sptr++ = '[';
      if (cptr[0] == '.' && cptr[1] == '.' && cptr[2] == '.')
      {
         strcpy (sptr, "000000");
         sptr += 6;
      }
      for (ncnt = 2; ncnt < ccnt; ncnt++)
      {
         /* add the next component, a "directory" */
         if (cptr[0] == '.' && cptr[1] == '.' && cptr[2] == '.')
            while (*cptr) *sptr++ = *cptr++;
         else
         {
            if (ncnt > 2) *sptr++ = '.';
            while (*cptr) *sptr++ = *cptr++;
         }
         cptr++;
      }
      *sptr++ = ']';
      /* add the last component, the "file" */
      while (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
   }

   /*******************************************/
   /* convert intermediate non-RMS characters */
   /*******************************************/

   for (sptr = VmsPtr; *sptr && *sptr != ']'; sptr++)
      if (*sptr == '!') *sptr = '$';
   if (*sptr)
   {
      cptr = "";
      for (sptr++; *sptr; sptr++)
         if (*sptr == '!') *(cptr = sptr) = '$';
      if (*cptr) *cptr = '.';
   }

   if (!VmsPtr[0]) VmsPtr[1] = '\0';

   if (Debug) fprintf (stdout, "VmsPtr %d |%s|\n", sptr-VmsPtr, VmsPtr);
   return (sptr - VmsPtr);
}

/*****************************************************************************/
/*
Convert a VMS file specification into a URL-style specification.  For example:
"DEVICE:[DIR1.DIR2]FILE.TXT" into "/device/dir1/dir2/file.txt".  Returns length
of path.
*/ 
 
int MapUrl_VmsToUrl
(
char *PathPtr,
char *VmsPtr,
boolean AbsorbMfd
)
{
   register char  *pptr, *vptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MapUrl_VmsToUrl() |%s|\n", VmsPtr);

   vptr = VmsPtr;
   pptr = PathPtr;
   *pptr++ = '/';
   /* copy the device and directory components */
   while (*vptr)
   {
      if (*vptr == ':' && *(vptr+1) == '[')
      {
         vptr++;
         vptr++;
         /* remove any reference to a Master File Directory */
         if (AbsorbMfd && strncmp (vptr, "000000", 6) == 0)
            vptr += 6;
         else
            *pptr++ = '/';
      }
      if (*vptr == '.')
      {
         if (vptr[1] == '.' && vptr[2] == '.')
         {
            *pptr++ = '/';
            *pptr++ = *vptr++;
            *pptr++ = *vptr++;
            *pptr++ = *vptr++;
         }
         else
         {
            vptr++;
            *pptr++ = '/';
         }
      }
      else
      if (*vptr == ']')
      {
         vptr++;
         *pptr++ = '/';
         break;
      }
      else
         *pptr++ = tolower(*vptr++);
   }
   /* copy the file component */
   while (*vptr) *pptr++ = tolower(*vptr++);
   *pptr = '\0';
   *(pptr+1) = '\0';

   if (Debug) fprintf (stdout, "PathPtr %d |%s|\n", pptr-PathPtr, PathPtr);
   return (pptr - PathPtr);
}

/*****************************************************************************/
/*
Given the current document's URL path and the specified document's URL path, 
generate a resultant URL path based on whether the specified document's path 
is absolute or relative the the current document's path.  Insufficient space to
contain the result path will not crash the function and the result will be set
to an empty string.
*/ 

int MapUrl_VirtualPath
(
char *CurrentPath,
char *DocumentPath,
char *ResultPath,
int SizeOfResultPath
)
{
   register char  *cptr, *dptr, *sptr, *tptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "MapUrl_VirtualPath() |%s|%s|\n",
               CurrentPath, DocumentPath);

   if (*(dptr = DocumentPath) == '/' || strstr (DocumentPath, "//") != NULL) 
   {
      /*****************/
      /* absolute path */
      /*****************/

      strcpy (ResultPath, dptr);
      zptr = (sptr = ResultPath) + SizeOfResultPath;
      while (*dptr && sptr < zptr) *sptr++ = *dptr++;
      if (sptr >= zptr) return (-1);
      *sptr = '\0';
      if (Debug) fprintf (stdout, "ResultPath |%s|\n", ResultPath);
      return (sptr - ResultPath);
   }

   /* step over any "relative to this directory" syntax ("./") */
   if (dptr[0] == '.' && dptr[1] == '/') dptr += 2;
   if (*dptr != '.')
   {
      /*****************/
      /* inferior path */
      /*****************/

      zptr = (sptr = tptr = ResultPath) + SizeOfResultPath;
      cptr = CurrentPath;
      while (*cptr && sptr < zptr)
      {
         if (*cptr == '/') tptr = sptr+1;
         *sptr++ = *cptr++;
      }
      sptr = tptr;
      while (*dptr && sptr < zptr) *sptr++ = *dptr++;
      if (sptr >= zptr) return (-1);
      *sptr = '\0';
      if (Debug) fprintf (stdout, "ResultPath |%s|\n", ResultPath);
      return (sptr - ResultPath);
   }

   /*****************/
   /* relative path */
   /*****************/

   zptr = (sptr = tptr = ResultPath) + SizeOfResultPath;
   cptr = CurrentPath;
   while (*cptr && sptr < zptr)
   {
      if (*cptr == '/') tptr = sptr;
      *sptr++ = *cptr++;
   }
   sptr = tptr;
   /* loop, stepping back one level for each "../" in the document path */
   while (dptr[0] == '.' && dptr[1] == '.' && dptr[2] == '/')
   {
      dptr += 3;
      if (sptr > ResultPath) sptr--;
      while (sptr > ResultPath && *sptr != '/') sptr--;
   }
   if (sptr > ResultPath)
      sptr++;
   else
      *sptr++ = '/';
   while (*dptr && sptr < zptr) *sptr++ = *dptr++;
   if (sptr >= zptr) return (-1);
   *sptr = '\0';
   if (Debug) fprintf (stdout, "ResultPath |%s|\n", ResultPath);
   return (sptr - ResultPath);
}

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text in HTML.  For example the '<', '&', etc.  Convert these 
to the corresponding HTML character entities.  Returns the number of characters
added to the buffer or -1 to indicate buffer overflow.  Copies a maximum of
'Count' characters, or all id it is -1.
*/ 

int MapUrl_CopyToHtml
( 
char *Destination,
int SizeOfDestination,
char *Source,
int Count
)
{
    register char  *cptr, *sptr, *zptr;
    register int  ccnt;

    cptr = Source;
    zptr = (sptr = Destination) + SizeOfDestination;
    ccnt = Count;
    while (--ccnt && *cptr)
    {
       switch (*cptr)
       {
          case '<' :
             if (sptr+4 >= zptr) return (-1);
             memcpy (sptr, "&lt;", 4); sptr += 4; cptr++; break;
          case '>' :
             if (sptr+4 >= zptr) return (-1);
             memcpy (sptr, "&gt;", 4); sptr += 4; cptr++; break;
          case '&' :
             if (sptr+5 >= zptr) return (-1);
             memcpy (sptr, "&amp;", 5); sptr += 5; cptr++; break;
          case '\"' :
             if (sptr+6 >= zptr) return (-1);
             memcpy (sptr, "&quot;", 6); sptr += 6; cptr++; break;
          default :
             if (sptr >= zptr) return (-1);
             *sptr++ = *cptr++;
       }
    }
    if (sptr >= zptr) return (-1);
    *sptr = '\0';
    return (sptr-Destination);
}

#ifdef IP_MAPURL

/*****************************************************************************/
/*
A server administration report on the server's mapping rules. This function
just wraps the reporting function, loading a temporary database if necessary
for reporting from the rule file.
*/

MapUrl_Report
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
boolean UseServerDatabase
)
{
   int  status;
   char  *cptr;
   struct MappingRuleListStruct  LocalRulesList;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MapUrl_Report()\n");

   if (UseServerDatabase)
   {
      /* use mapping rules from supplied pointer */
      MapUrl_ReportNow (rqptr, GlobalMappingRulesListPtr);
   }
   else
   {
      /* load temporary set of rules from mapping file */
      memset (&LocalRulesList, 0, sizeof(struct MappingRuleListStruct)); 
      LocalRulesList.RequestPtr = rqptr;
      cptr = MapUrl_LoadRules (&LocalRulesList);
      if (!cptr[0] && cptr[1])
      {
         /* error reported */
         rqptr->ResponseStatusCode = 403;
         ErrorGeneral (rqptr, cptr+1, FI_LI);
      }
      else
         MapUrl_ReportNow (rqptr, &LocalRulesList);
      MapUrl_FreeRules (&LocalRulesList);
      if (LocalRulesList.ProblemReportPtr != NULL)
         free (LocalRulesList.ProblemReportPtr);
   }

   NetWriteBuffered (rqptr, NextTaskFunction, NULL, 0);
}

/*****************************************************************************/
/*
Return a report on the HTTPd server's mamping rules ... now!  This function
blocks while executing.
*/

MapUrl_ReportNow
(
struct RequestStruct *rqptr,
struct MappingRuleListStruct *rlptr
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Mapping Rules</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Mapping Rules</H2>\n\
!AZ\n");

   static $DESCRIPTOR (ProblemReportFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH><FONT COLOR=\"#ff0000\">!UL Problem!AZ !AZ</FONT></TH></TR>\n\
<TR><TD><PRE>!AZ</PRE></TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (CheckPathFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Check Path Against Rules</TH></TR>\n\
<TR><TD>\n\
<FORM METHOD=GET ACTION=\"!AZ\">\n\
<NOBR>\n\
Using <INPUT TYPE=radio NAME=server VALUE=yes CHECKED>server or \
<INPUT TYPE=radio NAME=server VALUE=no>file rules ...\n\
<BR><INPUT TYPE=text NAME=path SIZE=50>\n\
<INPUT TYPE=submit VALUE=\" check \">\n\
<INPUT TYPE=reset VALUE=\" reset \">\n\
</NOBR>\n\
</FORM>\n\
</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Source: &quot;!AZ&quot;</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH ALIGN=RIGHT>File:</TH>\
<TD ALIGN=LEFT>!AZ</TD>\
<TD>&nbsp;</TD><TH>[<A HREF=\"!AZ\">View</A>]</TH>\
</TR>\n\
<TR><TH ALIGN=RIGHT>!AZ</TH>\
<TD ALIGN=LEFT>!AZ</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH ROWSPAN=2>Rule</TH><TH>Path</TH><TH>Template</TH></TR>\n\
<TR><TH COLSPAN=2>Condition</TH></TR>\n\
<TR></TR>\n");

   static $DESCRIPTOR (RuleFaoDsc,
"<TR><TH>!AZ</TH><TD>!AZ</TD><TD>!AZ</TD></TR>\n");

   static $DESCRIPTOR (RuleConditionalFaoDsc,
"<TR><TH ROWSPAN=2>!AZ</TH><TD>!AZ</TD><TD>!AZ</TD></TR>\n\
<TR><TD COLSPAN=2><TT>!AZ</TT></TD></TR>\n");

   static $DESCRIPTOR (EndPageFaoDsc,
"!AZ\
</TABLE>\n\
</BODY>\n\
</HTML>\n");

   register struct MappingRuleStruct  *mrptr;
   register unsigned long  *vecptr;
   register char  *cptr;

   boolean  CheckFromFile;
   int  status,
        EntryCount;
   unsigned long  FaoVector [32];
   unsigned short  Length;
   char  Buffer [2048];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MapUrl_ReportNow()\n");

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
      return;

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = DayDateTime (&rqptr->BinaryTime, 20);

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   if (rlptr->ProblemReportLength)
   {
      vecptr = FaoVector;
      if ((*vecptr++ = rlptr->ProblemCount) == 1)
         *vecptr++ = "";
      else
         *vecptr++ = "s";
      if (rlptr == GlobalMappingRulesListPtr)
         *vecptr++ = "Reported At Startup";
      else
         *vecptr++ = "Detected During Load";
      *vecptr++ = rlptr->ProblemReportPtr;

      status = sys$faol (&ProblemReportFaoDsc, &Length, &BufferDsc,
                         &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   vecptr = FaoVector;

   *vecptr++ = HttpdInternalReportRules;
   if (rlptr == GlobalMappingRulesListPtr)
      *vecptr++ = "Server";
   else
      *vecptr++ = "File";
   *vecptr++ = rlptr->LoadFileName;
   *vecptr++ = MapUrl_Map (NULL, rlptr->LoadFileName, NULL, NULL, NULL);
   if (rlptr == GlobalMappingRulesListPtr)
   {
      *vecptr++ = "Loaded:";
      *vecptr++ = DayDateTime (&rlptr->LoadBinTime, 20);
   }
   else
   {
      *vecptr++ = "Revised:";
      *vecptr++ = DayDateTime (&rlptr->RevBinTime, 20);
   }

   status = sys$faol (&CheckPathFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   status = SS$_NORMAL;
   EntryCount = 0;

   for (mrptr = rlptr->HeadPtr; mrptr != NULL; mrptr = mrptr->NextPtr)
   {
      if (Debug)
         fprintf (stdout, "Rule |%c|%s|%s|\n",
                  mrptr->RuleType, mrptr->TemplatePtr, mrptr->ResultPtr);

      EntryCount++;

      if (mrptr->RuleType == 'E')
         if (mrptr->IsCgiPlusScript)
            cptr = "exec+";
         else
            cptr = "exec";
      else
      if (mrptr->RuleType == 'F')
         cptr = "fail";
      else
      if (mrptr->RuleType == 'M')
         cptr = "map";
      else
      if (mrptr->RuleType == 'P')
         cptr = "pass";
      else
      if (mrptr->RuleType == 'R')
         cptr = "redirect";
      else
      if (mrptr->RuleType == 'S')
         if (mrptr->IsCgiPlusScript)
            cptr = "script+";
         else
            cptr = "script";
      else
         cptr = "*ERROR*";

      vecptr = FaoVector;

      *vecptr++ = cptr;
      *vecptr++ = mrptr->TemplatePtr;
      *vecptr++ = mrptr->ResultPtr;

      if (mrptr->ConditionalPtr[0])
      {
         *vecptr++ = mrptr->ConditionalPtr;
         status = sys$faol (&RuleConditionalFaoDsc, &Length, &BufferDsc,
                            &FaoVector);
      }
      else
         status = sys$faol (&RuleFaoDsc, &Length, &BufferDsc,
                            &FaoVector);

      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   vecptr = FaoVector;
   if (EntryCount)
      *vecptr++ = "";
   else
      *vecptr++ = "<TR><TD COLSPAN=3><I>(none)</I></TD></TR>\n";

   status = sys$faol (&EndPageFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);
}

/*****************************************************************************/
/*
A server administration report on the server's mapping rules. This function
just wraps the reporting function, loading a temporary database if necessary
for reporting from the rule file.
*/

MapUrl_ReportRuleCheck
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
char *Path,
boolean UseServerDatabase
)
{
   int  status;
   char  *cptr;
   struct MappingRuleListStruct  LocalRulesList;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MapUrl_ReportRuleCheck()\n");

   if (UseServerDatabase)
   {
      /* use mapping rules from supplied pointer */
      MapUrl_ReportRuleCheckNow (rqptr, GlobalMappingRulesListPtr, Path);
   }
   else
   {
      /* load temporary set of rules from mapping file */
      memset (&LocalRulesList, 0, sizeof(struct MappingRuleListStruct)); 
      LocalRulesList.RequestPtr = rqptr;
      cptr = MapUrl_LoadRules (&LocalRulesList);
      if (!cptr[0] && cptr[1])
      {
         /* error reported */
         rqptr->ResponseStatusCode = 403;
         ErrorGeneral (rqptr, cptr+1, FI_LI);
      }
      else
         MapUrl_ReportRuleCheckNow (rqptr, &LocalRulesList, Path);
      MapUrl_FreeRules (&LocalRulesList);
      if (LocalRulesList.ProblemReportPtr != NULL)
         free (LocalRulesList.ProblemReportPtr);
   }

   NetWriteBuffered (rqptr, NextTaskFunction, NULL, 0);
}

/*****************************************************************************/
/*
Display mapping rules used to resolve the path in the query string. For HTTP
report, uses blocking network write.
*/

MapUrl_ReportRuleCheckNow
(
struct RequestStruct *rqptr,
struct MappingRuleListStruct *rlptr,
char *Path
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Mapping Rule Check</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Mapping Rules Check</H2>\n\
!AZ\n");

   static $DESCRIPTOR (ProblemReportFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH><FONT COLOR=\"#ff0000\">!UL Problem!AZ !AZ</FONT></TH></TR>\n\
<TR><TD><PRE>!AZ</PRE></TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (SourceFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Source: &quot;!AZ&quot;</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH ALIGN=RIGHT>File:</TH><TD ALIGN=LEFT>!AZ</TD></TR>\n\
<TR><TH ALIGN=RIGHT>!AZ</TH><TD ALIGN=LEFT>!AZ</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH ROWSPAN=3><FONT SIZE=+1>Path</FONT></TH>\
<TH ROWSPAN=3><FONT SIZE=+1>Match</FONT></TH>\
<TH COLSPAN=3><FONT SIZE=+1>Mapping</FONT></TH></TR>\n\
<TH ROWSPAN=2>Rule</TH><TH>Path</TH><TH>Template</TH></TR>\n\
<TR><TH COLSPAN=2>Condition</TH></TR>\n\
<TR></TR>\n");

   static $DESCRIPTOR (EndErrorFaoDsc,
"</TABLE>\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TD>!AZ</TD></TR>\n\
</TABLE>\n\
</BODY>\n\
</HTML>\n");

   static $DESCRIPTOR (EndPageFaoDsc,
"</TABLE>\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH></TH><TH>path</TH><TH>VMS</TH></TR>\n\
<TR><TH>path</TH><TD>!AZ</TD><TD>!AZ</TD></TR>\n\
<TR><TH>script!AZ</TH><TD>!AZ</TD><TD>!AZ</TD></TR>\n\
</TABLE>\n\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register char  *cptr;

   int  status;
   unsigned long  FaoVector [32];
   unsigned short  Length;
   char  Buffer [1024],
         Script [256],
         ScriptVms [256],
         Vms [256];
   $DESCRIPTOR (BufferDsc, Buffer);
   struct MappingRuleListStruct  *GlobalMappingRulesListPtrBuffer;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MapUrl_ReportRuleCheckNow()\n");

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
      return;

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = DayDateTime (&rqptr->BinaryTime, 20);

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   if (rlptr->ProblemReportLength)
   {
      vecptr = FaoVector;
      if ((*vecptr++ = rlptr->ProblemCount) == 1)
         *vecptr++ = "";
      else
         *vecptr++ = "s";
      if (rlptr == GlobalMappingRulesListPtr)
         *vecptr++ = "Reported At Startup";
      else
         *vecptr++ = "Detected During Load";
      *vecptr++ = rlptr->ProblemReportPtr;

      status = sys$faol (&ProblemReportFaoDsc, &Length, &BufferDsc,
                         &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   vecptr = FaoVector;

   if (rlptr == GlobalMappingRulesListPtr)
   {
      *vecptr++ = "Server";
      *vecptr++ = rlptr->LoadFileName;
      *vecptr++ = "Loaded:";
      *vecptr++ = DayDateTime (&rlptr->LoadBinTime, 20);
   }
   else
   {
      *vecptr++ = "File";
      *vecptr++ = rlptr->LoadFileName;
      *vecptr++ = "Revised:";
      *vecptr++ = DayDateTime (&rlptr->RevBinTime, 20);
   }

   status = sys$faol (&SourceFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   GlobalMappingRulesListPtrBuffer = GlobalMappingRulesListPtr;
   GlobalMappingRulesListPtr = rlptr;

   /* indicate this is a rule-check report */
   rlptr->RequestPtr = rqptr;

   Script[0] = ScriptVms[0] = Vms[0] = '\0';
   cptr = MapUrl_Map (Path, Vms, Script, ScriptVms, rqptr);

   /* indicate this is no longer a rule-check report */
   rlptr->RequestPtr = NULL;

   GlobalMappingRulesListPtr = GlobalMappingRulesListPtrBuffer;

   if (!cptr[0] && cptr[1])
      status = sys$fao (&EndErrorFaoDsc, &Length, &BufferDsc, cptr+1);
   else
   {
      if (Script[0] == '+')
      {
         Script[0] = '/';
         status = sys$fao (&EndPageFaoDsc, &Length, &BufferDsc,
                           cptr, Vms, " (CGIplus)", Script, ScriptVms);
      }
      else
         status = sys$fao (&EndPageFaoDsc, &Length, &BufferDsc,
                           cptr, Vms, "", Script, ScriptVms);
   }

   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$fao()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);
}

/*****************************************************************************/
/*
*/

MapUrl_ReportRule
(
register struct MappingRuleStruct  *mrptr,
char *PathPtr,
boolean Match
)
{
   static $DESCRIPTOR (RuleFaoDsc,
"<TR><TD>!AZ</TD><TH ALIGN=CENTER>!AZ</TH>\
<TH>!AZ</TH><TD>!AZ</TD><TD>!AZ</TD></TR>\n");

   static $DESCRIPTOR (RuleConditionalFaoDsc,
"<TR>\
<TD ROWSPAN=2>!AZ</TD>\
<TH ROWSPAN=2 ALIGN=CENTER>!AZ</TH>\
<TH ROWSPAN=2>!AZ</TH><TD>!AZ</TD><TD>!AZ</TD>\
</TR>\n\
<TR><TD COLSPAN=2><TT>!AZ</TT></TD></TR>");

   register unsigned long  *vecptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [1024];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MapUrl_ReportRule()\n");

   vecptr = FaoVector;

   *vecptr++ = PathPtr;

   if (Match)
      *vecptr++ = "X";
   else
      *vecptr++ = "";

   switch (mrptr->RuleType)
   {
      case 'E' :
         if (mrptr->IsCgiPlusScript)
            { *vecptr++ = "exec+"; break; }
         else
            { *vecptr++ = "exec"; break; }
      case 'F' :
         *vecptr++ = "fail"; break;
      case 'M' :
         *vecptr++ = "map"; break;
      case 'P' :
         *vecptr++ = "pass"; break;
      case 'R' :
         *vecptr++ = "redirect"; break;
      case 'S' :
         if (mrptr->IsCgiPlusScript)
            { *vecptr++ = "script+"; break; }
         else
            { *vecptr++ = "script"; break; }
      default :
         *vecptr++ = "*ERROR*";
   }

   *vecptr++ =  mrptr->TemplatePtr;
   *vecptr++ =  mrptr->ResultPtr;

   if (mrptr->ConditionalPtr[0])
   {
      *vecptr++ = mrptr->ConditionalPtr;
      status = sys$faol (&RuleConditionalFaoDsc, &Length, &BufferDsc,
                         &FaoVector);
   }
   else
      status = sys$faol (&RuleFaoDsc, &Length, &BufferDsc,
                         &FaoVector);

   if (VMSok (status) && status != SS$_BUFFEROVF)
   {
      Buffer[Length] = '\0';
      if (Debug) fprintf (stdout, "|%s|\n", Buffer);
      NetWriteBuffered (GlobalMappingRulesListPtr->RequestPtr, 0,
                       Buffer, Length);
      return;
   }
   NetWriteBuffered (GlobalMappingRulesListPtr->RequestPtr, 0,
                    "FAO() ERROR", 11);
}

/*****************************************************************************/
/*
Display mapping rules used to resolve the path in the query string.  Called
from the Request.c modules.  For HTTP report, uses blocking network write.
*/

MapUrl_ControlReload
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>Success 200</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>SUCCESS!!</H1>\n\
<P>Reported by server.\n\
<P>Server !AZ mapping rules reloaded.\n");

   static $DESCRIPTOR (ProblemReportFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH><FONT COLOR=\"#ff0000\">!UL Problem!AZ \
Detected During Load</FONT></TH></TR>\n\
<TR><TD><PRE>!AZ</PRE></TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (EndPageFaoDsc,
"</BODY>\n\
</HTML>\n");

   register char  *cptr;
   register unsigned long  *vecptr;

   int  status;
   unsigned long  FaoVector [16];
   unsigned short  Length;
   char  Buffer [4096];
   $DESCRIPTOR (BufferDsc, Buffer);
   struct MappingRuleListStruct  *rlptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MapUrl_ControlReload()\n");

   rlptr = GlobalMappingRulesListPtr;
   cptr = MapUrl_LoadRules (rlptr);
   /* if error reported */
   if (!cptr[0] && cptr[1])
   {
      rqptr->ResponseStatusCode = 403;
      ErrorGeneral (rqptr, cptr+1, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   NetWriteBuffered (rqptr, 0, Buffer, Length);

   if (rlptr->ProblemReportLength)
   {
      vecptr = FaoVector;
      if ((*vecptr++ = rlptr->ProblemCount) == 1)
         *vecptr++ = "";
      else
         *vecptr++ = "s";
      *vecptr++ = rlptr->ProblemReportPtr;

      status = sys$faol (&ProblemReportFaoDsc, &Length, &BufferDsc,
                         &FaoVector);

      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   NetWriteBuffered (rqptr, NextTaskFunction,
                    EndPageFaoDsc.dsc$a_pointer,
                    EndPageFaoDsc.dsc$w_length);
}

/*****************************************************************************/
/*
Scan the conditional string evaluating the conditions!  Return true or false.
Anything it cannot understand it ignores!
*/

boolean MapUrl_Conditional
(
struct RequestStruct  *rqptr,
char *Conditional
)
{
   register char  *cptr, *sptr, *zptr;

   boolean  NegateThisCondition,
            NegateEntireConditional,
            Result,
            SoFarSoGood;
   int  ConditionalCount;
   char  Scratch [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "MapUrl_Conditional() %d |%s|\n", rqptr, Conditional);

   ConditionalCount = 0;
   NegateEntireConditional = NegateThisCondition = SoFarSoGood = false;
   cptr = Conditional;
   while (*cptr)
   {
      while (ISLWS(*cptr)) cptr++;
      if (!*cptr) break;

      if (*cptr == '[' || (*cptr == '!' && *(cptr+1) == '['))
      {
         if (*cptr == '!')
         {
            NegateEntireConditional = true;
            cptr++;
         }
         else
            NegateEntireConditional = false;
         cptr++;
         ConditionalCount = 0;
         SoFarSoGood = false;
         continue;
      }
      if (*cptr == ']')
      {
         cptr++;
         if (NegateEntireConditional)
         {
            SoFarSoGood = !SoFarSoGood;
            NegateEntireConditional = false;
         }
         if (ConditionalCount && !SoFarSoGood) return (false);
         continue;
      }
      if (SoFarSoGood)
      {
         if (NegateEntireConditional)
         {
            SoFarSoGood = !SoFarSoGood;
            NegateEntireConditional = false;
         }
         /* at least one OK, skip to the end of the conditional */
         while (*cptr && *cptr != ']') cptr++;
         if (!*cptr) break;
      }

      NegateThisCondition = Result = false;
      zptr = (sptr = Scratch) + sizeof(Scratch);

      if (*cptr == '!')
      {
         cptr++;
         NegateThisCondition = true;
      }

      switch (*(unsigned short*)cptr)
      {
         case 'AC' :

            /*************/
            /* "Accept:" */
            /*************/

            ConditionalCount++;
            cptr += 3;
            while (*cptr && !ISLWS(*cptr) && *cptr != ']' && sptr < zptr)
               *sptr++ = *cptr++;
            if (sptr >= zptr) return (false);
            *sptr = '\0';
            if (rqptr->HttpAcceptPtr != NULL)
               Result = MapUrl_ConditionalList (rqptr, Scratch,
                                                rqptr->HttpAcceptPtr);
            break;
             
         case 'AL' :

            /**********************/
            /* "Accept-Language:" */
            /**********************/

            ConditionalCount++;
            cptr += 3;
            while (*cptr && !ISLWS(*cptr) && *cptr != ']' && sptr < zptr)
               *sptr++ = *cptr++;
            if (sptr >= zptr) return (false);
            *sptr = '\0';
            if (rqptr->HttpAcceptLangPtr != NULL)
               Result = MapUrl_ConditionalList (rqptr, Scratch,
                                                rqptr->HttpAcceptLangPtr);
            break;

         case 'AS' :

            /*********************/
            /* "Accept-Charset:" */
            /*********************/

            ConditionalCount++;
            cptr += 3;
            while (*cptr && !ISLWS(*cptr) && *cptr != ']' && sptr < zptr)
               *sptr++ = *cptr++;
            if (sptr >= zptr) return (false);
            *sptr = '\0';
            if (rqptr->HttpAcceptCharsetPtr != NULL)
               Result = MapUrl_ConditionalList (rqptr, Scratch,
                                                rqptr->HttpAcceptCharsetPtr);
            break;

         case 'HO' :

            /****************************/
            /* client host name/address */
            /****************************/

            ConditionalCount++;
            cptr += 3;
            while (*cptr && !ISLWS(*cptr) && *cptr != ']' && sptr < zptr)
               *sptr++ = *cptr++;
            if (sptr >= zptr) return (false);
            *sptr = '\0';
            for (sptr = Scratch; *sptr && !isalnum(*sptr); sptr++);
            if (isdigit(*sptr))
               Result = (SearchTextString (rqptr->ClientInternetAddress,
                                           Scratch, false, NULL) != NULL);
            else
               Result = (SearchTextString (rqptr->ClientHostName,
                                           Scratch, false, NULL) != NULL);
            break;

         case 'HH' :

            /*********************************/
            /* destination host name/address */
            /*********************************/

            ConditionalCount++;
            cptr += 3;
            while (*cptr && !ISLWS(*cptr) && *cptr != ']' && sptr < zptr)
               *sptr++ = *cptr++;
            if (sptr >= zptr) return (false);
            *sptr = '\0';
            for (sptr = Scratch; *sptr && !isalnum(*sptr); sptr++);
            if (rqptr->HttpHostPtr != NULL)
               Result = (SearchTextString (rqptr->HttpHostPtr,
                                           Scratch, false, NULL) != NULL);
            break;

         case 'FO' :

            /*******************/
            /* proxy forwarded */
            /*******************/

            ConditionalCount++;
            cptr += 3;
            while (*cptr && !ISLWS(*cptr) && *cptr != ']' && sptr < zptr)
               *sptr++ = *cptr++;
            if (sptr >= zptr) return (false);
            *sptr = '\0';
            if (rqptr->HttpForwardedPtr != NULL)
               Result = MapUrl_ConditionalList (rqptr, Scratch,
                                                rqptr->HttpForwardedPtr);
            break;

         case 'ME' :

            /***************/
            /* HTTP method */
            /***************/

            ConditionalCount++;
            cptr += 3;
            while (*cptr && !ISLWS(*cptr) && *cptr != ']' && sptr < zptr)
               *sptr++ = *cptr++;
            if (sptr >= zptr) return (false);
            *sptr = '\0';
            Result = (SearchTextString (rqptr->HttpMethodName,
                                        Scratch, false, NULL) != NULL);
            break;

         case 'SC' :

            /****************************************/
            /* request scheme ("http:" or "https:") */
            /****************************************/

            ConditionalCount++;
            cptr += 3;
            /* also ignore any trailing colon on the scheme string */
            while (*cptr && !ISLWS(*cptr) && *cptr != ':' && *cptr != ']' &&
                   sptr < zptr)
               *sptr++ = *cptr++;
            if (sptr >= zptr) return (false);
            *sptr = '\0';
            if (*cptr == ':') cptr++;
            /* any trailing colon has been stripped */
            if (strsame (Scratch, "https", -1) &&
                rqptr->RequestScheme == SCHEME_HTTPS)
               Result = true;
            else
            if (strsame (Scratch, "http", -1) &&
                rqptr->RequestScheme == SCHEME_HTTP)
               Result = true;
            else
               Result = false;

            break;

         case 'RU' :

            /***************/
            /* remote user */
            /***************/

            ConditionalCount++;
            cptr += 3;
            while (*cptr && !ISLWS(*cptr) && *cptr != ']' && sptr < zptr)
               *sptr++ = *cptr++;
            if (sptr >= zptr) return (false);
            *sptr = '\0';
            if (!rqptr->RemoteUser[0])
               Result = MapUrl_ConditionalList (rqptr, Scratch,
                                                rqptr->RemoteUser);
            break;

         case 'SN' :

            /***************/
            /* server name */
            /***************/

            ConditionalCount++;
            cptr += 3;
            while (*cptr && !ISLWS(*cptr) && *cptr != ']' && sptr < zptr)
               *sptr++ = *cptr++;
            if (sptr >= zptr) return (false);
            *sptr = '\0';
            Result = (SearchTextString (rqptr->ServicePtr->ServerHostName,
                                        Scratch, false, NULL) != NULL);
            break;

         case 'SP' :

            /***************/
            /* server port */
            /***************/

            ConditionalCount++;
            cptr += 3;
            while (*cptr && !ISLWS(*cptr) && *cptr != ']' && sptr < zptr)
               *sptr++ = *cptr++;
            if (sptr >= zptr) return (false);
            *sptr = '\0';
            Result = (SearchTextString (rqptr->ServicePtr->ServerPortString,
                                        Scratch, false, NULL) != NULL);
            break;

         case 'UA' :

            /*****************/
            /* "User-Agent:" */
            /*****************/

            ConditionalCount++;
            cptr += 3;
            while (*cptr && !ISLWS(*cptr) && *cptr != ']' && sptr < zptr)
               *sptr++ = *cptr++;
            if (sptr >= zptr) return (false);
            *sptr = '\0';
            if (rqptr->HttpUserAgentPtr != NULL)
               Result = (SearchTextString (rqptr->HttpUserAgentPtr,
                                           Scratch, false, NULL) != NULL);
            break;

         default :

            /*******************/
            /* unknown, ignore */
            /*******************/

            /* should never occur due to basic check in MapUrl_LoadRules()! */
            while (*cptr && !ISLWS(*cptr) && *cptr != ']') cptr++;
            continue;

      }

      if (NegateThisCondition)
         SoFarSoGood = SoFarSoGood || !Result;
      else
         SoFarSoGood = SoFarSoGood || Result;
   }

   if (Debug)
      fprintf (stdout, "%d %d %d\n",
               ConditionalCount, NegateEntireConditional, SoFarSoGood);

   if (NegateEntireConditional) SoFarSoGood = !SoFarSoGood;
   if (ConditionalCount)
      return (SoFarSoGood);
   else
      return (true);
}

/*****************************************************************************/
/*
Compare single "??:" conditional string to each of a comma-separated list in
the form "item" or "item1, item2, item3", etc. String may contain '*' and '%'
wildcards.  We can munge the list string like this because we're operating at
AST-delivery level and cannot be preempted!
*/

boolean MapUrl_ConditionalList
(
struct RequestStruct  *rqptr,
char *String,
char *List
)
{
   register char  *cptr, *lptr;
   char  c;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MapUrl_ConditionalList() |%s|\n", List);

   if ((lptr = List) == NULL) return (false);
   while (*lptr)
   {
      /* spaces and commas are element separators, step over */
      while (ISLWS(*lptr) || *lptr == ',') lptr++;
      /* if end of list (or empty) then finished */
      if (!*lptr) return (false);
      /* find end of this element, save next character, terminate element */
      for (cptr = lptr; *lptr && !ISLWS(*lptr) && *lptr != ','; lptr++);
      c = *lptr;
      *lptr = '\0';
      /* look for what we want */
      cptr = SearchTextString (cptr, String, false, NULL);
      /* restore list */
      *lptr = c;
      /* if found then finished */
      if (cptr != NULL) return (true);
   }
   /* ran out of list elements, so it can't be a match! */
   return (false);
}

/*****************************************************************************/

#endif /* IP_MAPURL */

