/*****************************************************************************/
/*
                                  Menu.c

This module implements a full multi-threaded, AST-driven, asynchronous "menu" 
file send.  The AST-driven nature makes the code a little more difficult to 
follow, but creates a powerful, event-driven, multi-threaded server.  All of 
the necessary functions implementing this module are designed to be non-
blocking. 

This module uses the NetWriteBuffered() function to buffer any output it can
into larger chunks before sending it to the client.

Implements the WASD menu file.  A WASD menu comprises a file with 1, 2, 3 or 
more blank-line delimitted sections.  The optional first is the document 
title.  The optional second is a menu description.  The mandatory section 
consists of the menu items.  Any blank-line delimtted sections thereafter 
alternate between descriptive paragraphs and menu sections.  For example: 

|This is the HTML <TITLE> line.  It must be only one line.|
||
|This is the descriptive paragraph.|
|It can be multi-line.|
||
|HT_DATA:FILE1.TXT   This is the first text file.|
|HT_DATA:FILE2.TXT   This is the second text file.|
|HT_DATA:FILE1.HTML  This would be an HTML file.|
||
|This is another (optional) descriptive paragraph.|
|It can also be multi-line.|
||
|HT_DATA:FILE1.TXT   This is the first file of the second menu.|
|HT_DATA:FILE2.TXT   This is the second file, etc.|
|!This is a comment|
|*.TXT?  This is a search item
|!The next item lists all matching files from their internal description|
|*.HTML|
 

VERSION HISTORY
---------------
17-AUG-97  MGD  message database,
                SYSUAF-authenticated users security-profile
27-FEB-97  MGD  delete on close for "temporary" files
01-FEB-97  MGD  HTTPd version 4
23-MAY-96  MGD  added listing from file descriptions (i.e. Description())
28-MAR-96  MGD  bugfix, MenuMextContent() caused RequestEnd() to be
                called twice (two log entries only deleterious effect :^)
01-DEC-95  MGD  HTTPd version 3
27-SEP-95  MGD  added "If-Modified-Since:" functionality
07-AUG-95  MGD  ConfigIncludeCommentedInfo to allow physical file
                specification to be included as commentary within menu output
20-DEC-94  MGD  initial development for multi-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>

/* VMS related header files */
#include <descrip.h>
#include <libdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application-related header files */
#include "wasd.h"
#include "descr.h"
#include "error.h"
#include "httpd.h"
#include "menu.h"
#include "mapurl.h"
#include "msg.h"
#include "net.h"
#include "support.h"
#include "vm.h"

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern int  FileBufferSize;
extern char  HtmlSgmlDoctype[];
extern char  SoftwareID[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/*****************************************************************************/
/*
Open the specified file.  If there is a problem opening it then just return 
the status, do not report an error or anything.  This is used to determine 
whether the file exists, as when attempting to open one of multiple, possible 
home pages.

As fixed documents (files) can have revision date/times easily checked any
"If-Modified-Since:" request field date/time supplied in the request is 
processed and the file only sent if modified later than specified.
A "Last-Modified:" field is included in any response header. 

When successfully opened generate an HTTP header if required.  Once open and 
connected the menu interpretation becomes I/O event-driven. 
*/

MenuBegin
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
void *FileOpenErrorFunction,
char *FileName
)
{
   register char  *cptr, *sptr, *zptr;
   register struct MenuTaskStruct  *tkptr;

   int  status;
   char  ExpandedFileName [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "MenuBegin() %d |%s|\n",
               rqptr->ErrorMessagePtr, FileName);

   if (rqptr->ErrorMessagePtr != NULL)
   {
      /* previous error, cause threaded processing to unravel */
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /* set up the task structure (only ever one per request!) */
   rqptr->MenuTaskPtr = tkptr = (struct MenuTaskStruct*)
      VmGetHeap (rqptr, sizeof(struct MenuTaskStruct));
   tkptr->NextTaskFunction = NextTaskFunction;

   cptr = FileName;
   zptr = (sptr = tkptr->FileName) + sizeof(tkptr->FileName);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      MenuEnd (rqptr);
      return;
   }
   *sptr = '\0';
   tkptr->FileNameLength = sptr - tkptr->FileName;

   /*****************************************/
   /* check VMS-authenticated user's access */
   /*****************************************/

   if (rqptr->AuthVmsUserProfileLength)
   {
      status = AuthCheckVmsUserAccess (rqptr, tkptr->FileName,
                                       tkptr->FileNameLength);
      if (status == RMS$_PRV)
         tkptr->AuthVmsUserHasAccess = false;
      else
      if (VMSok (status))
         tkptr->AuthVmsUserHasAccess = true;
      else
      {
         /* error reported by access check */
         MenuEnd (rqptr);
         return;
      }
   }
   else
      tkptr->AuthVmsUserHasAccess = false;

   /*************/
   /* open file */
   /*************/

   tkptr->FileFab = cc$rms_fab;
   tkptr->FileFab.fab$b_fac = FAB$M_GET;
   tkptr->FileFab.fab$l_fna = tkptr->FileName;  
   tkptr->FileFab.fab$b_fns = tkptr->FileNameLength;
   tkptr->FileFab.fab$b_shr = FAB$M_SHRGET;

   /* initialize the NAM block */
   tkptr->FileNam = cc$rms_nam;
   tkptr->FileFab.fab$l_nam = &tkptr->FileNam;
   tkptr->FileNam.nam$l_esa = ExpandedFileName;
   tkptr->FileNam.nam$b_ess = sizeof(ExpandedFileName)-1;

   /* initialize the date extended attribute block */
   tkptr->FileXabDat = cc$rms_xabdat;
   tkptr->FileFab.fab$l_xab = &tkptr->FileXabDat;

   /* "temporary" file, automatic delete on closing it */
   if (rqptr->DeleteOnClose) tkptr->FileFab.fab$l_fop |= FAB$M_DLT;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   status = sys$open (&tkptr->FileFab, 0, 0);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);

      /* if its a search list treat directory not found as if file not found */
      if ((tkptr->FileNam.nam$l_fnb & NAM$M_SEARCH_LIST) && status == RMS$_DNF)
         status = RMS$_FNF;

      if (FileOpenErrorFunction != NULL)
      {
         /* do not report the error, the alternate function will handle it */
         tkptr->NextTaskFunction = FileOpenErrorFunction;
         MenuEnd (rqptr);
         return;
      }
      else
      {
         if (!rqptr->AccountingDone)
            rqptr->AccountingDone = ++Accounting.DoMenuCount;
         rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
         rqptr->ErrorHiddenTextPtr = tkptr->FileName;
         ErrorVmsStatus (rqptr, status, FI_LI);
         MenuEnd (rqptr);
         return;
      }
   }

   if (!rqptr->AccountingDone)
      rqptr->AccountingDone = ++Accounting.DoMenuCount;

   /*******************************/
   /* connect record access block */
   /*******************************/

   tkptr->FileRab = cc$rms_rab;
   tkptr->FileRab.rab$l_fab = &tkptr->FileFab;
   /* 2 buffers and read ahead performance option */
   tkptr->FileRab.rab$b_mbf = 2;
   tkptr->FileRab.rab$l_rop = RAB$M_RAH;
   tkptr->FileRab.rab$l_ubf = tkptr->ReadBuffer;
   tkptr->FileRab.rab$w_usz = sizeof(tkptr->ReadBuffer)-1;

   if (VMSnok (status = sys$connect (&tkptr->FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      MenuEnd (rqptr);
      return;
   }

   /********************/
   /* begin processing */
   /********************/

   if (rqptr->ResponseHeaderPtr == NULL)
   {
      rqptr->ResponsePreExpired = PRE_EXPIRE_MENU;
      rqptr->ResponseStatusCode = 200;
      if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
      {
         MenuEnd (rqptr);
         return;
      }

      rqptr->RequestHeaderLength = strlen(rqptr->ResponseHeaderPtr);

      if (rqptr->HttpMethod == HTTP_METHOD_HEAD)
      {
         MenuEnd (rqptr);
         return;
      }
   }

   /* set then RAB user context storage to the client thread pointer */
   tkptr->FileRab.rab$l_ctx = rqptr;

   tkptr->MenuBlankLineCount = tkptr->MenuLineCount =
      tkptr->MenuSectionNumber = 0;

   /* network writes are checked for success, fudge the first one! */
   rqptr->NetWriteIOsb.Status = SS$_NORMAL;

   /* queue the first record read, count the number of sections in the menu */
   sys$get (&tkptr->FileRab, &MenuContents, &MenuContents);
}

/*****************************************************************************/
/*
Ensure the menu file is closed.  Flush the output buffer, activating the next 
task (if any).
*/ 

MenuEnd (struct RequestStruct *rqptr)

{
   register struct MenuTaskStruct  *tkptr;

   int  status,
        SetPrvStatus;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuEnd()\n");

   tkptr = rqptr->MenuTaskPtr;

   if (tkptr->FileFab.fab$w_ifi)
   {
      if (rqptr->DeleteOnClose)
      {
         /* use SYSPRV to ensure deletion-on-close of file */
         EnableSysPrv ();
         sys$close (&tkptr->FileFab, 0, 0);
         DisableSysPrv ();
      }
      else
         sys$close (&tkptr->FileFab, 0, 0);
   }

   if (tkptr->ParseInUse)
   {
      /* ensure parse internal data structures are released */
      tkptr->SearchFab.fab$l_fna = "a:[b]c.d;";
      tkptr->SearchFab.fab$b_fns = 9;
      tkptr->SearchFab.fab$b_dns = 0;
      tkptr->SearchNam.nam$b_nop = NAM$M_SYNCHK;
      sys$parse (&tkptr->SearchFab, 0, 0);
   }

   /* declare the next task */
   SysDclAst (tkptr->NextTaskFunction, rqptr);
}

/*****************************************************************************/
/*
Process the records (lines) of the menu file according to the total count of 
the sections (blank-line delimited text) in the file and the section number 
the current line falls within.  This routine is event-driven, using itself as 
an AST completion routine to sys$get()s.
*/ 

MenuContents (struct RAB *RabPtr)

{
   static char  MenuEndHtml [] = "</BODY>\n</HTML>\n",
                MenuEndListEndHtml [] = "</UL>\n</BODY>\n</HTML>\n";

   register unsigned char  *rptr;
   register struct RequestStruct  *rqptr;
   register struct MenuTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "MenuContents() sts: %%X%08.08X stv: %%X%08.08X\n",
               RabPtr->rab$l_sts, RabPtr->rab$l_stv);

   rqptr = RabPtr->rab$l_ctx;
   tkptr = rqptr->MenuTaskPtr;

   if (VMSnok (tkptr->FileRab.rab$l_sts))
   {
      if (tkptr->FileRab.rab$l_sts == RMS$_EOF)
      {
         /********************/
         /* end of menu file */
         /********************/

         /* disable implicit listing for now, not convinced it's way to go */
/*
         if (tkptr->MenuSectionNumber <= 2)
         {
            NetWriteBuffered (rqptr, &MenuBeginImplicitDescription,
                             "<P>\n<UL>\n", 9);
            return;
         }
*/

         if (!((tkptr->MenuSectionNumber-1) % 2))
         {
            NetWriteBuffered (rqptr, &MenuEnd,
                             MenuEndListEndHtml,
                             sizeof(MenuEndListEndHtml)-1);
            return;
         }
         else
         {
            NetWriteBuffered (rqptr, &MenuEnd,
                             MenuEndHtml, sizeof(MenuEndHtml)-1);
            return;
         }
      }

      /***************************/
      /* error reading menu file */
      /***************************/

      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, tkptr->FileRab.rab$l_sts, FI_LI);
      MenuEnd (rqptr);
      return;
   }

   /* terminate the line just read */
   (rptr = tkptr->FileRab.rab$l_ubf)[tkptr->FileRab.rab$w_rsz] = '\0';
   /* terminate on any carriage control that may be in the buffer */
   while (*rptr && *rptr != '\r' && *rptr != '\n') rptr++;
   *rptr = '\0';
   if (Debug)
      fprintf (stdout, "tkptr->FileRab.rab$l_ubf |%s|\n",
               tkptr->FileRab.rab$l_ubf);

   if (!tkptr->FileRab.rab$l_ubf[0])
   {
      /**************/
      /* empty line */
      /**************/

      tkptr->MenuBlankLineCount++;
      tkptr->MenuLineCount = 0;
      /* queue another read, completion AST back to this function again */
      sys$get (&tkptr->FileRab, &MenuContents, &MenuContents);
      return;
   }

   if (tkptr->FileRab.rab$l_ubf[0] == '!')
   {
      /*********************/
      /* comment-only line */
      /*********************/

      /* queue another read, completion AST back to this function again */
      sys$get (&tkptr->FileRab, &MenuContents, &MenuContents);
      return;
   }

   /******************/
   /* non-blank line */
   /******************/

   if (tkptr->MenuBlankLineCount) tkptr->MenuSectionNumber++;
   tkptr->MenuLineCount++;
   tkptr->MenuBlankLineCount = 0;
   if (!tkptr->MenuSectionNumber) tkptr->MenuSectionNumber++;

   /******************/
   /* interpret line */
   /******************/

   if (tkptr->MenuSectionNumber == 1 && tkptr->MenuLineCount == 1)
      MenuTitle (rqptr);
   else
   if (!(tkptr->MenuSectionNumber % 2))
   {
      /* ensure the description lines have HTTP carriage-control */
      *rptr++ = '\n';
      *rptr = '\0';
      MenuDescription (rqptr,
                       rptr - (unsigned char*)tkptr->FileRab.rab$l_ubf);
   }
   else
   if (!((tkptr->MenuSectionNumber-1) % 2))
   {
      if (tkptr->MenuLineCount == 1)
         NetWriteBuffered (rqptr, &MenuItems, "<P>\n<UL>\n", 9);
      else
         MenuItems (rqptr);
   }
}

/*****************************************************************************/
/*
*/

int MenuNextContents (struct RequestStruct *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuNextContents()\n");

   if (VMSnok (rqptr->NetWriteIOsb.Status))
   {
      /* network write has failed (as AST), bail out now */
      if (Debug)
         fprintf (stdout, "NetWriteIOsb.Status %%X%08.08X\n",
                  rqptr->NetWriteIOsb.Status);
      MenuEnd (rqptr);
      return;
   }

   /* queue another read, completion AST back to contents interpretation */
   sys$get (&rqptr->MenuTaskPtr->FileRab, &MenuContents, &MenuContents);
}

/*****************************************************************************/
/*
The line just read is the menu title.
*/

int MenuTitle (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (TitleFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ</H1>\n");

   static $DESCRIPTOR (StringDsc, "");

   register unsigned long  *vecptr;
   register struct MenuTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [16];
   char  String [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuTitle()\n");

   tkptr = rqptr->MenuTaskPtr;

   /* should only be one line, ignore all but the first */
   if (tkptr->MenuLineCount > 1)
      SysDclAst (&MenuNextContents, rqptr);

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, tkptr->FileName);

   *vecptr++ = tkptr->FileRab.rab$l_ubf;
   *vecptr++ = tkptr->FileRab.rab$l_ubf;

   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String)-1;

   status = sys$faol (&TitleFaoDsc, &Length, &StringDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      MenuEnd (rqptr);
      return;
   }
   String[Length] = '\0';

   /* output the title */
   NetWriteBuffered (rqptr, &MenuNextContents, String, Length);
}

/*****************************************************************************/
/*
The line just read forms part of a description section.  The calling routine 
has appended HTTP carriage-control, so this line is just output as-is.  If 
this is the first (and possibly only) description section separate using a 
paragraph, or <P>, tag.  If the second (or more) description section first 
terminate the unsigned list, or </UL> begun in the preceding item section 
before separating using the pragraph tag.
*/ 

int MenuDescription
(
struct RequestStruct *rqptr,
int LineLength
)
{
   register struct MenuTaskStruct  *tkptr;

   char  String [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuDescription()\n");

   tkptr = rqptr->MenuTaskPtr;

   if (tkptr->MenuLineCount == 1)
   {
      /* first line of the description section */
      if (tkptr->MenuSectionNumber == 2)
      {
         /* first description section of the menu */
         if (LineLength > sizeof(String)-5) LineLength -= 5;
         memcpy (String, "<P>\n", 4);
         memcpy (String+4, tkptr->FileRab.rab$l_ubf, LineLength);
         String[LineLength += 4] = '\0';
      }
      else
      {
         /* second or subsequent description section of the menu */
         if (LineLength > sizeof(String)-11) LineLength -= 11;
         memcpy (String, "</UL>\n<P>\n", 10);
         memcpy (String+10, tkptr->FileRab.rab$l_ubf, LineLength);
         String[LineLength += 10] = '\0';
      }
      NetWriteBuffered (rqptr, &MenuNextContents, String, LineLength);
   }
   else
   {
      /* second or subsequent line of the description section */
      NetWriteBuffered (rqptr, &MenuNextContents,
                       tkptr->FileRab.rab$l_ubf, LineLength);
   }
}

/*****************************************************************************/
/*
Interpret this line as a menu item comprising a URI, white-space, and a 
description comprising the rest of the line after the first non-space.  Each 
line in the item section is output as an item in an non-ordered list, or <UL>.
*/

int MenuItems (struct RequestStruct *rqptr)

{
   static char  NoDescription [] = "<I>(no description for this line)</I>";
   static $DESCRIPTOR (FileFaoDsc, "<LI><A HREF=\"!AZ\">!AZ</A>\n");
   static $DESCRIPTOR (StringDsc, "");

   register char  *cptr, *sptr, *zptr;
   register struct MenuTaskStruct  *tkptr;

   unsigned short  Length;
   char  *DescriptionPtr;
   char  Description [256],
         String [512],
         Uri [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuItems()\n");

   tkptr = rqptr->MenuTaskPtr;

   cptr = tkptr->FileRab.rab$l_ubf;
   /* skip leading white-space */
   while (*cptr && ISLWS(*cptr)) cptr++;
   zptr = (sptr = Uri) + sizeof(Uri)-2;
   while (*cptr && !ISLWS(*cptr) && sptr < zptr) *sptr++ = *cptr++;
   *sptr = '\0';

   /* skip intervening white-space */
   while (*cptr && ISLWS(*cptr)) cptr++;
   zptr = (sptr = Description) + sizeof(Description)-2;
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   *sptr = '\0';

   if (Description[0])
      DescriptionPtr = Description;
   else
      DescriptionPtr = NoDescription;

   if (Uri[0] == '\"')
   {
      /* full URI/URL specification, remove quotes and include as-is */
      for (cptr = Uri+1; *cptr && *cptr != '\"'; cptr++);
      if (*cptr == '\"') *cptr = '\0';
      StringDsc.dsc$a_pointer = String;
      StringDsc.dsc$w_length = sizeof(String)-1;
      sys$fao (&FileFaoDsc, &Length, &StringDsc,
               Uri+1, DescriptionPtr);
      String[Length] = '\0';
      NetWriteBuffered (rqptr, &MenuNextContents, String, Length);
   }
   else
   {
      for (cptr = Uri; *cptr && *cptr != '?'; cptr++);
      if (*cptr == '?')
      {
         /* search item */
         *cptr++ = '\0';
         MenuSearchItem (rqptr, &MenuNextContents, Uri, Description, cptr);
      }
      else
      if (Uri[0] == '*' && !*cptr)
      {
         /* wildcard description list item */
         MenuFileDescription (rqptr, &MenuNextContents, Uri);
      }
      else
      {
         /* file item */
         StringDsc.dsc$a_pointer = String;
         StringDsc.dsc$w_length = sizeof(String)-1;
         sys$fao (&FileFaoDsc, &Length, &StringDsc,
                  Uri, DescriptionPtr);
         String[Length] = '\0';
         NetWriteBuffered (rqptr, &MenuNextContents, String, Length);
      }
   }
}

/*****************************************************************************/
/*
The URI contains a question mark, which is menu-speak for "this is a search
item".  Create and output a search form.  Optional characters following the 
question mark include an "A" for and "about the search" link, "C" for case-
sensitivity buttons, and "O" for output style buttons.
*/

int MenuSearchItem
(
struct RequestStruct *rqptr,
void *AstFunctionPtr,
char *Uri,
char *Description,
char *SearchOptions
)
{
   static $DESCRIPTOR (StringDsc, "");

   static $DESCRIPTOR (BeginFormFaoDsc,
"<LI>\n\
<FORM ACTION=\"!AZ\">\n\
<INPUT TYPE=submit VALUE=\"!AZ\">\n\
<INPUT TYPE=text NAME=\"search\">\n\
<INPUT TYPE=reset VALUE=\"Reset\">\n");

   static char  AboutForm[] =
"<BR><I><A HREF=\"?about=search\">About</A> this search.</I>\n";

   static char  CaseForm[] =
"<BR><I>Output By:  \
line<INPUT TYPE=radio NAME=\"hits\" VALUE=\"line\" CHECKED>\
 document<INPUT TYPE=radio NAME=\"hits\" VALUE=\"document\"></I>\n";

   static char  OutputForm[] =
"<BR><I>Case Sensitive:  \
no<INPUT TYPE=radio NAME=\"case\" VALUE=\"no\" CHECKED>\
 yes<INPUT TYPE=radio NAME=\"case\" VALUE=\"yes\"></I>\n";

   static char EndForm[] = "</FORM>\n";

   int  status;
   unsigned short  Length;
   char  HtmlDescription [512],
         String [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuSearchItem()\n");

   CopyToHtml (HtmlDescription, sizeof(HtmlDescription), Description, -1);

   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String)-1;
   sys$fao (&BeginFormFaoDsc, &Length, &StringDsc, Uri, HtmlDescription);

   while (*SearchOptions)
   {
      if (toupper(*SearchOptions) == 'A')
      {
         if (sizeof(AboutForm)-1 < sizeof(String)-1 - Length)
         {
            memcpy (String+Length, AboutForm, sizeof(AboutForm)-1);
            Length += sizeof(AboutForm)-1;
         }
      }
      else
      if (toupper(*SearchOptions) == 'O')
      {
         if (sizeof(OutputForm)-1 < sizeof(String)-1 - Length)
         {
            memcpy (String+Length, OutputForm, sizeof(OutputForm)-1);
            Length += sizeof(OutputForm)-1;
         }
      }
      else
      if (toupper(*SearchOptions) == 'C')
      {
         if (sizeof(CaseForm)-1 < sizeof(String)-1 - Length)
         {
            memcpy (String+Length, CaseForm, sizeof(CaseForm)-1);
            Length += sizeof(CaseForm)-1;
         }
      }
      SearchOptions++;
   }

   if (sizeof(EndForm)-1 < sizeof(String)-1 - Length)
   {
      memcpy (String+Length, EndForm, sizeof(EndForm)-1);
      Length += sizeof(EndForm)-1;
   }

   String[Length] = '\0';
   NetWriteBuffered (rqptr, AstFunctionPtr, String, Length);
}

/*****************************************************************************/
/*
Only called, as an AST, if an implicit file contents listing is generated
because there was no third (item) section in the menu.
*/ 

MenuBeginImplicitDescription (struct RequestStruct *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuBeginImplicitDescription()\n");

   MenuFileDescription (rqptr, &MenuEndImplicitDescription, "*.*");
}

/*****************************************************************************/
/*
Only called, as an AST, if an implicit file contents listing is generated
because there was no third (item) section in the menu.
*/ 

MenuEndImplicitDescription (struct RequestStruct *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuEndImplicitDescription()\n");

   NetWriteBuffered (rqptr, &MenuEnd, "</UL>\n", 6);
}

/*****************************************************************************/
/*
*/ 

MenuFileDescription
(
struct RequestStruct *rqptr,
void *AstFunctionPtr,
char *FileWildcard
)
{
   register struct MenuTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuFileDescription()\n");

   tkptr = rqptr->MenuTaskPtr;

   tkptr->ParseInUse = true;

   tkptr->MenuAstFunctionPtr = AstFunctionPtr;

   tkptr->SearchFab = cc$rms_fab;
   /* set the FAB user context to the client thread pointer */
   tkptr->SearchFab.fab$l_ctx = rqptr;
   tkptr->SearchFab.fab$l_dna = tkptr->FileNam.nam$l_dev;
   tkptr->SearchFab.fab$b_dns = tkptr->FileNam.nam$b_dev +
                                tkptr->FileNam.nam$b_dir;
   tkptr->SearchFab.fab$l_fna = FileWildcard;
   tkptr->SearchFab.fab$b_fns = strlen(FileWildcard);
   tkptr->SearchFab.fab$l_fop = FAB$M_NAM;
   tkptr->SearchFab.fab$l_nam = &tkptr->SearchNam;

   tkptr->SearchNam = cc$rms_nam;
   tkptr->SearchNam.nam$l_esa = tkptr->ExpandedFileName;
   tkptr->SearchNam.nam$b_ess = sizeof(tkptr->ExpandedFileName)-1;
   tkptr->SearchNam.nam$l_rsa = tkptr->ResultantFileName;
   tkptr->SearchNam.nam$b_rss = sizeof(tkptr->ResultantFileName)-1;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   status = sys$parse (&tkptr->SearchFab, 0, 0);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
      *tkptr->FileNam.nam$l_name = '\0';
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileNam.nam$l_dev);
      rqptr->ErrorHiddenTextPtr = tkptr->FileNam.nam$l_dev;
      ErrorVmsStatus (rqptr, status, FI_LI);
      MenuEnd (rqptr);
      return;
   }

   tkptr->ExpandedFileName[tkptr->SearchNam.nam$b_esl] = '\0';

   MenuFileSearch (rqptr);
}

/*****************************************************************************/
/*
*/

MenuFileSearch (struct RequestStruct *rqptr)

{
   register struct MenuTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuFileSearch()\n");

   tkptr = rqptr->MenuTaskPtr;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   /* call RMS directory search routine, AST completion to DirFiles() */
   status = sys$search (&tkptr->SearchFab,
                        &MenuFileDescriptionOf, &MenuFileDescriptionOf);
   if (Debug) fprintf (stdout, "sys$search() %%X%08.08X\n", status);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();
}

/*****************************************************************************/
/*
*/

MenuFileDescriptionOf (struct FAB *FabPtr)

{
   register char  *cptr, *sptr;
   register struct RequestStruct  *rqptr;
   register struct MenuTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
      "MenuFileDescriptionOf() sts: %%X%08.08X stv: %%X%08.08X\n",
      FabPtr->fab$l_sts, FabPtr->fab$l_stv);

   /* retrieve the pointer to the client thread from the FAB user context */
   rqptr = FabPtr->fab$l_ctx;

   tkptr = rqptr->MenuTaskPtr;

   if (VMSnok (status = FabPtr->fab$l_sts))
   {
      /* if its a search list treat directory not found as if file not found */
      if ((tkptr->FileNam.nam$l_fnb & NAM$M_SEARCH_LIST) &&
          status == RMS$_DNF)
         status = RMS$_FNF;

      if (status == RMS$_FNF || status == RMS$_NMF)
      {
         /**********************/
         /* end of file search */
         /**********************/

         tkptr->ParseInUse = false;
         SysDclAst (tkptr->MenuAstFunctionPtr, rqptr);
         return;
      }

      /**********************/
      /* sys$search() error */
      /**********************/

      rqptr->ErrorTextPtr = MapVmsPath (tkptr->ExpandedFileName);
      rqptr->ErrorHiddenTextPtr = tkptr->ExpandedFileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      MenuEnd (rqptr);
      return;
   }

   /* terminate following the last character in the version number */
   tkptr->SearchNam.nam$l_ver[tkptr->SearchNam.nam$b_ver] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", tkptr->ResultantFileName);

   if (!strcmp (tkptr->SearchNam.nam$l_name, tkptr->FileNam.nam$l_name))
   {
      /* this is the same menu file, don't need a description of this! */
      MenuFileSearch (rqptr);
      return;
   }

   Description (rqptr,
                &MenuFileDescriptionDone,
                tkptr->SearchNam.nam$l_rsa,
                tkptr->SearchNam.nam$l_rsa,
                tkptr->Description,
                sizeof(tkptr->Description),
                DESCRIPTION_ALL);
}

/*****************************************************************************/
/*
*/ 

MenuFileDescriptionDone (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (LinkFaoDsc, "<LI><A HREF=\"!AZ\">!AZ!AZ!AZ</A>\n");

   register char  *cptr, *sptr;
   register struct MenuTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   char  LowerCaseName [128],
         Link [512];
   $DESCRIPTOR (LinkDsc, Link);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MenuFileDescriptionDone()\n");

   tkptr = rqptr->MenuTaskPtr;

   if (tkptr->Description[0] == DESCRIPTION_IMPOSSIBLE)
      SysDclAst (&MenuFileSearch, rqptr);

   sptr = LowerCaseName;
   for (cptr = tkptr->SearchNam.nam$l_name;
        *cptr && *cptr != ';';
        *sptr++ = tolower(*cptr++));
   *sptr = '\0';
   if (tkptr->Description[0])
      sys$fao (&LinkFaoDsc, &Length, &LinkDsc, 
               LowerCaseName, "&quot;", tkptr->Description, "&quot;");
   else
      sys$fao (&LinkFaoDsc, &Length, &LinkDsc, 
               LowerCaseName, "", LowerCaseName, "");
   Link[Length] = '\0';

   NetWriteBuffered (rqptr, &MenuFileSearch, Link, Length);
}

/*****************************************************************************/

