/*****************************************************************************/
/*
                                  msg.c

Provide a site-configurable message database.  This is particularly directed
towards non-English language sites.  No attempt has been made to completely
internationalize the server, only messages the average client is likely to be
exposed to are currently provided.

The only way of reloading the database is to restart the server. This is not
of great concern for once customized the messages should remain completely
stable between releases.


Current Message Groupings
-------------------------

  [auth]        authentication/authorization
  [dir]         directory listing
  [general]     messages belonging to no particular module or category
  [htadmin]     authentication database administration
  [ismap]       image mapping module
  [mapping]     mapping rules
  [put]         PUT/POST module
  [request]     request acceptance and initial parsing
  [script]      DCL/scripting module
  [ssi]         Server Side Includes
  [status]      error and other status reporting
  [upd]         update module


VERSION HISTORY
---------------
06-JUL-98  MGD  bugfix; MsgUnload(), VmFree() of 'HostListPtr'
14-FEB-98  MGD  message file format changed from v4.4 to v5.0
25-OCT-97  MGD  changes around MsgFor() when no request structure available
09-AUG-97  MGD  initial development (HTTPd v4.4)
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>

/* VMS related header files */
#include <ssdef.h>
#include <stsdef.h>

/* application related header files */
#include "wasd.h"
#include "net.h"
#include "mapurl.h"
#include "msg.h"
#include "support.h"

/******************/
/* global storage */
/******************/

#define MSG_VERSION "5.0"

char  MsgNotSet [] = "INTERNAL ERROR, please notify the administrator.";

struct MsgDbStruct  MsgDb;

char  ErrorMsgDuplicateLanguageNumber [] = "duplicate [language] number",
      ErrorMsgDuplicateMessageNumber [] = "duplicate message number",
      ErrorMsgLangDisabled [] = "[language] disabled",
      ErrorMsgLangName [] = "[language] name not specified",
      ErrorMsgLangNotFound [] = "[language] not found",
      ErrorMsgLangNotSpecified [] = "[language] not specified",
      ErrorMsgLangNumber [] = "[language] number out-of-range",
      ErrorMsgLangTooMany [] = "too many [language]s",
      ErrorMsgMessageNumber [] = "message number out-of-range",
      ErrorMsgNoGroup [] = "[group-name] has not been specified",
      ErrorMsgNoLangNumber [] = "no [language] number (order)",
      ErrorMsgNoLangName [] = "no [language] name",
      ErrorMsgNoMessageNumber [] = "no message number",
      ErrorMsgTooFew [] = "too few messages",
      ErrorMsgTooMany [] = "too many messages",
      ErrorMsgUnknownGroup [] = "unknown [group-name]",
      ErrorMsgVersion [] = "message file [version] mismatch",
      ErrorMsgVersionNotChecked [] = "no [version] for checking";

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern char  HtmlSgmlDoctype[];
extern char  ServerHostPort[];
extern char  SoftwareID[];
extern char  Utility[];
extern char  ErrorSanityCheck[];

/*****************************************************************************/
/*
Called from HTTPd.c during startup, or when creating a report from the file.
*/

MsgLoad (struct MsgDbStruct *mlptr)

{
   register char  *cptr, *sptr, *zptr;

   boolean  DebugBuffer,
            VersionChecked;
   int  status,
        Count,
        LanguageNumber,
        LineCount,
        MessageBase,
        MessageCount,
        MessageMax,
        MessageNumber,
        StartLineNumber;
   char  ExpandedFileName [256],
         LanguageName [32],
         Line [4096];

   struct FAB  MsgFileFab;
   struct NAM  MsgFileNam;
   struct RAB  MsgFileRab;
   struct XABDAT  MsgFileXabDat;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MsgLoad()\n");

#ifdef DBUG
   /* not interested anymore in seeing debug information for message load! */
   DebugBuffer = Debug;
   Debug = false;
#endif

   MsgUnLoad (mlptr);

   /*************************/
   /* open the message file */
   /*************************/

   MsgFileFab = cc$rms_fab;
   MsgFileFab.fab$b_fac = FAB$M_GET;
   MsgFileFab.fab$l_fna = MsgFileName;
   MsgFileFab.fab$b_fns = strlen(MsgFileName);
   MsgFileFab.fab$l_nam = &MsgFileNam;
   MsgFileFab.fab$b_shr = FAB$M_SHRGET;
   MsgFileFab.fab$l_xab = &MsgFileXabDat;

   MsgFileNam = cc$rms_nam;
   MsgFileNam.nam$l_esa = ExpandedFileName;
   MsgFileNam.nam$b_ess = sizeof(ExpandedFileName)-1;
   MsgFileNam.nam$l_rsa = mlptr->LoadFileName;
   MsgFileNam.nam$b_rss = sizeof(mlptr->LoadFileName)-1;

   /* initialize the date extended attribute block */
   MsgFileXabDat = cc$rms_xabdat;

   /* turn on SYSPRV to allow access to possibly protected file */
   EnableSysPrv ();
   status = sys$open (&MsgFileFab, 0, 0);
   DisableSysPrv ();

   /* status from sys$open() */
   if (VMSnok (status))
   {
      if (mlptr->RequestPtr)
      {
         MsgReportProblem (mlptr, MsgFileName, status);
         return;
      }
      else
         ErrorExitVmsStatus (status, MsgFileName, FI_LI);
   }

   mlptr->LoadFileName[MsgFileNam.nam$b_rsl] = '\0';
   if (Debug)
      fprintf (stdout, "mlptr->LoadFileName |%s|\n", mlptr->LoadFileName);

   sys$gettim (&mlptr->LoadBinTime);

   /* get the configuration file revision date and time */
   memcpy (&mlptr->RevBinTime,
           &MsgFileXabDat.xab$q_rdt,
           sizeof(mlptr->RevBinTime));

   /* record access block */
   MsgFileRab = cc$rms_rab;
   MsgFileRab.rab$l_fab = &MsgFileFab;
   /* try and optimise sequential read (next 3 lines) */
   MsgFileRab.rab$b_mbc = 6;
   MsgFileRab.rab$b_mbf = 2;
   MsgFileRab.rab$l_rop = RAB$M_RAH;

   if (VMSnok (status = sys$connect (&MsgFileRab, 0, 0)))
   {
      sys$close (&MsgFileFab, 0, 0);
      if (mlptr->RequestPtr)
      {
         sys$close (&MsgFileFab, 0, 0);
         MsgReportProblem (mlptr, MsgFileName, status);
         return;
      }
      else
         ErrorExitVmsStatus (status, MsgFileName, FI_LI);
   }

   mlptr->LanguageCount = mlptr->ProblemCount = 
      mlptr->LineNumber = mlptr->ProblemReportLength = 
      LanguageNumber = LineCount = StartLineNumber = MessageCount = 0;
   mlptr->ProblemReportPtr = NULL;
   MessageBase = -1;
   VersionChecked = false;

   /*************************/
   /* read the message file */
   /*************************/

   MsgFileRab.rab$l_ubf = Line;
   MsgFileRab.rab$w_usz = sizeof(Line)-1;

   while (VMSok (status = sys$get (&MsgFileRab, 0, 0)))
   {
      LineCount++;
      if (!StartLineNumber) StartLineNumber = LineCount;
      MsgFileRab.rab$l_ubf[MsgFileRab.rab$w_rsz] = '\0';
      if (Debug) fprintf (stdout, "%d |%s|\n", LineCount, MsgFileRab.rab$l_ubf);

      if (MsgFileRab.rab$w_rsz)
      {
         if (MsgFileRab.rab$l_ubf[MsgFileRab.rab$w_rsz-1] == '\\')
         {
            /* line is continued, change backslash int linefeed, get more */
            MsgFileRab.rab$l_ubf[MsgFileRab.rab$w_rsz-1] = '\n';
            MsgFileRab.rab$l_ubf += MsgFileRab.rab$w_rsz;
            MsgFileRab.rab$w_usz -= MsgFileRab.rab$w_rsz;
            continue;
         }
      }
      /* reset the line buffer */
      MsgFileRab.rab$l_ubf = Line;
      MsgFileRab.rab$w_usz = sizeof(Line)-1;

      mlptr->LineNumber = StartLineNumber;
      mlptr->LinePtr = Line;
      StartLineNumber = 0;
      if (Debug) fprintf (stdout, "Line %d |%s|\n", mlptr->LineNumber, Line);

      if (Line[0] == '#') continue;
      cptr = Line;
      while (*cptr && ISLWS(*cptr)) cptr++;
      if (!*cptr) continue;

      if (*cptr == '[')
      {
         if (strsame (cptr, "[version]", 9))
         {
            /*************/
            /* [version] */
            /*************/

            cptr += 9;
            while (*cptr && ISLWS(*cptr)) cptr++;
            if (!strsame (cptr, MSG_VERSION, -1))
            {
               if (mlptr->RequestPtr)
               {
                  MsgReportProblem (mlptr, ErrorMsgVersion, 0);
                  break;
               }
               else
                  ErrorExitVmsStatus (0, ErrorMsgVersion, FI_LI);
            }
            VersionChecked = true;
            continue;
         }

         if (strsame (cptr, "[language]", 10))
         {
            /**************/
            /* [language] */
            /**************/

            mlptr->LanguageCount++;
            if (Debug)
               fprintf (stdout, "mlptr->LanguageCount: %d\n",
                        mlptr->LanguageCount);

            /* check if the number of languages specified exceeds limits */
            if (mlptr->LanguageCount > MAX_LANGUAGES)
            {
               if (mlptr->RequestPtr)
               {
                  MsgReportProblem (mlptr, ErrorMsgLangTooMany, 0);
                  break;
               }
               else
                  ErrorExitVmsStatus (0, ErrorMsgLangTooMany, FI_LI);
            }

            /* allocate memory for this new language */
            mlptr->Msgs[mlptr->LanguageCount] = (struct MsgStruct*)
               VmGet (sizeof(struct MsgStruct));

            /* set all the messages in the language to NULL */
            for (Count = 0; Count <= MSG_RANGE; Count++)
               mlptr->Msgs[mlptr->LanguageCount]->TextPtr[Count] = NULL;

            /* step over the [language] and any trailing white-space */
            cptr += 10;
            while (*cptr && ISLWS(*cptr)) cptr++;

            if (!isdigit(*cptr))
            {
               /* no language number (order) has been specified */
               if (mlptr->RequestPtr)
               {
                  MsgReportProblem (mlptr, ErrorMsgNoLangNumber, 0);
                  continue;
               }
               else
                  ErrorExitVmsStatus (0, ErrorMsgNoLangNumber, FI_LI);
            }

            /* language number (order) */
            LanguageNumber = atol (cptr);
            if (LanguageNumber < 0 || LanguageNumber > mlptr->LanguageCount)
            {
               if (mlptr->RequestPtr)
               {
                  MsgReportProblem (mlptr, ErrorMsgLangNumber, 0);
                  continue;
               }
               else
                  ErrorExitVmsStatus (0, ErrorMsgLangNumber, FI_LI);
            }
            for (Count = 1; Count <= mlptr->LanguageCount; Count++)
            {
               if (mlptr->Msgs[Count]->LanguageNumber == LanguageNumber)
               ErrorExitVmsStatus (0, ErrorMsgDuplicateLanguageNumber, FI_LI);
            }
            mlptr->Msgs[LanguageNumber]->LanguageNumber = LanguageNumber;

            while (*cptr && !ISLWS(*cptr)) cptr++;
            while (*cptr && ISLWS(*cptr)) cptr++;

            /* get the language name */
            if (!*cptr)
            {
               if (mlptr->RequestPtr)
               {
                  MsgReportProblem (mlptr, ErrorMsgNoLangName, 0);
                  continue;
               }
               else
                  ErrorExitVmsStatus (0, ErrorMsgNoLangName, FI_LI);
            }
            zptr = (sptr = mlptr->Msgs[LanguageNumber]->LanguageName) +
                   sizeof(mlptr->Msgs[LanguageNumber]->LanguageName);
            while (*cptr && !ISLWS(*cptr) && sptr < zptr) *sptr++ = *cptr++;
            if (sptr > zptr) sptr--;
            *sptr = '\0';

            /* skip any white space and look for an optional host list */
            while (*cptr && ISLWS(*cptr)) cptr++;
            if (*cptr)
            {
               /* host list found */
               sptr = cptr;
               while (*cptr) cptr++;
               /* trim trailing white-space */
               if (cptr > sptr) cptr--;
               while (cptr > sptr && ISLWS(*cptr)) cptr--;
               if (cptr > sptr) cptr++;
               *cptr++ = '\0';

               /* allocate memory for the host list */
               zptr = mlptr->Msgs[mlptr->LanguageCount]->HostListPtr =
                  Vmget (cptr-sptr);

               memcpy (zptr, sptr, cptr-sptr);
            }

            if (mlptr->Msgs[LanguageNumber]->LanguageNumber == 0)
               MsgReportProblem (mlptr, ErrorMsgLangDisabled, 0);

            if (Debug)
               fprintf (stdout, "%d |%s|%s|\n",
                        mlptr->Msgs[LanguageNumber]->LanguageNumber,
                        mlptr->Msgs[LanguageNumber]->LanguageName,
                        mlptr->Msgs[LanguageNumber]->HostListPtr);
            continue;
         }

         /****************/
         /* [group-name] */
         /****************/

         /* the '\b' backspace is a sentinal, will never occur in a message */
         if (strsame (cptr, "[auth]", 6))
         {
            MessageBase = MSG_AUTH__BASE;
            MessageMax = MSG_AUTH__MAX;
            mlptr->Msgs[1]->TextPtr[MessageBase] = "\b[auth]";
         }
         else
         if (strsame (cptr, "[dir]", 6))
         {
            MessageBase = MSG_DIR__BASE;
            MessageMax = MSG_DIR__MAX;
            mlptr->Msgs[1]->TextPtr[MessageBase] = "\b[dir]";
         }
         else
         if (strsame (cptr, "[general]", 9))
         {
            MessageBase = MSG_GENERAL__BASE;
            MessageMax = MSG_GENERAL__MAX;
            mlptr->Msgs[1]->TextPtr[MessageBase] = "\b[general]";
         }
         else
         if (strsame (cptr, "[htadmin]", 9))
         {
            MessageBase = MSG_HTADMIN__BASE;
            MessageMax = MSG_HTADMIN__MAX;
            mlptr->Msgs[1]->TextPtr[MessageBase] = "\b[htadmin]";
         }
         else
         if (strsame (cptr, "[ismap]", 7))
         {
            MessageBase = MSG_ISMAP__BASE;
            MessageMax = MSG_ISMAP__MAX;
            mlptr->Msgs[1]->TextPtr[MessageBase] = "\b[ismap]";
         }
         else
         if (strsame (cptr, "[mapping]", 9))
         {
            MessageBase = MSG_MAPPING__BASE;
            MessageMax = MSG_MAPPING__MAX;
            mlptr->Msgs[1]->TextPtr[MessageBase] = "\b[mapping]";
         }
         else
         if (strsame (cptr, "[put]", 5))
         {
            MessageBase = MSG_PUT__BASE;
            MessageMax = MSG_PUT__MAX;
            mlptr->Msgs[1]->TextPtr[MessageBase] = "\b[put]";
         }
         else
         if (strsame (cptr, "[request]", 9))
         {
            MessageBase = MSG_REQUEST__BASE;
            MessageMax = MSG_REQUEST__MAX;
            mlptr->Msgs[1]->TextPtr[MessageBase] = "\b[request]";
         }
         else
         if (strsame (cptr, "[script]", 8))
         {
            MessageBase = MSG_SCRIPT__BASE;
            MessageMax = MSG_SCRIPT__MAX;
            mlptr->Msgs[1]->TextPtr[MessageBase] = "\b[script]";
         }
         else
         if (strsame (cptr, "[ssi]", 5))
         {
            MessageBase = MSG_SSI__BASE;
            MessageMax = MSG_SSI__MAX;
            mlptr->Msgs[1]->TextPtr[MessageBase] = "\b[ssi]";
         }
         else
         if (strsame (cptr, "[status]", 8))
         {
            MessageBase = MSG_STATUS__BASE;
            MessageMax = MSG_STATUS__MAX;
            mlptr->Msgs[1]->TextPtr[MessageBase] = "\b[status]";
         }
         else
         if (strsame (cptr, "[upd]", 5))
         {
            MessageBase = MSG_UPD__BASE;
            MessageMax = MSG_UPD__MAX;
            mlptr->Msgs[1]->TextPtr[MessageBase] = "\b[upd]";
         }
         else
         {
            MessageBase = -1;
            MsgReportProblem (mlptr, ErrorMsgUnknownGroup, 0);
         }

         continue;
      }

      /***********/
      /* message */
      /***********/

      if (!VersionChecked)
      {
         if (mlptr->RequestPtr)
         {
            MsgReportProblem (mlptr, ErrorMsgVersionNotChecked, 0);
            break;
         }
         else
            ErrorExitVmsStatus (0, ErrorMsgVersionNotChecked, FI_LI);
      }

      if (!mlptr->LanguageCount)
      {
         if (mlptr->RequestPtr)
         {
            MsgReportProblem (mlptr, ErrorMsgLangNotSpecified, 0);
            break;
         }
         else
            ErrorExitVmsStatus (0, ErrorMsgLangNotSpecified, FI_LI);
      }

      if (MessageBase < 0)
      {
         /* before any [goup-name] has been specified */
         MsgReportProblem (mlptr, ErrorMsgNoGroup, 0);
         continue;
      }

      if (!isalpha(*cptr))
      {
         /* language name is not being specified at start of line */
         MsgReportProblem (mlptr, ErrorMsgNoLangName, 0);
         continue;
      }

      zptr = (sptr = LanguageName) + sizeof(LanguageName);
      while (*cptr && !ISLWS(*cptr) && sptr < zptr) *sptr++ = *cptr++;
      if (sptr > zptr) sptr--;
      *sptr = '\0';
      /* find the index for the specified language */
      for (Count = 1; Count <= mlptr->LanguageCount; Count++)
      {
         if (Debug)
            fprintf (stdout, "%d |%s|%s|\n",
               Count, LanguageName,
               mlptr->Msgs[mlptr->LanguageCount]->LanguageName);
         if (strsame (LanguageName, mlptr->Msgs[Count]->LanguageName, -1))
         {
             LanguageNumber = Count;
             break;
         }
      }
      if (Count > mlptr->LanguageCount)
      {
         MsgReportProblem (mlptr, ErrorMsgLangNotFound, 0);
         continue;
      }

      while (*cptr && !ISLWS(*cptr)) cptr++;
      while (*cptr && ISLWS(*cptr)) cptr++;

      if (!isdigit(*cptr))
      {
         /* no message number has been specified */
         MsgReportProblem (mlptr, ErrorMsgNoMessageNumber, 0);
         continue;
      }

      MessageNumber = atol(cptr);
      if (Debug) fprintf (stdout, "MessageNumber: %d\n", MessageNumber);

      /* zero disables a message or whole language */
      if (!MessageNumber || !mlptr->Msgs[LanguageNumber]->LanguageNumber)
         continue;

      if (MessageNumber < 0 || MessageNumber > MessageMax)
      {
         MsgReportProblem (mlptr, ErrorMsgMessageNumber, 0);
         continue;
      }

      MessageNumber += MessageBase;
      if (Debug) fprintf (stdout, "Number+Base: %d\n", MessageNumber);

      while (*cptr && !ISLWS(*cptr)) cptr++;
      while (*cptr && ISLWS(*cptr)) cptr++;

      if (mlptr->Msgs[LanguageNumber]->TextPtr[MessageNumber] == NULL)
      {
         mlptr->Msgs[LanguageNumber]->TextPtr[MessageNumber] = MsgCreate (cptr);
         if (LanguageNumber == mlptr->LanguageCount) MessageCount++;
      }
      else
         MsgReportProblem (mlptr, ErrorMsgDuplicateMessageNumber, 0);
   }

   /**************************/
   /* close the message file */
   /**************************/

   sys$close (&MsgFileFab, 0, 0); 

   if (status == RMS$_EOF) status = SS$_NORMAL;
   if (VMSnok (status))
   {
      if (mlptr->RequestPtr)
      {
         MsgReportProblem (mlptr, MsgFileName, status);
         return;
      }
      else
         ErrorExitVmsStatus (status, ConfigFileName, FI_LI);
   }

   if (MessageCount < MSG_TOTAL)
   {
      if (mlptr->RequestPtr)
         MsgReportProblem (mlptr, ErrorMsgTooFew, 0);
      else
         ErrorExitVmsStatus (0, ErrorMsgTooFew, FI_LI);
   }
   else
   if (MessageCount > MSG_TOTAL)
   {
      /* technically, this shouldn't be possible! */
      if (mlptr->RequestPtr)
         MsgReportProblem (mlptr, ErrorMsgTooMany, 0);
      else
         ErrorExitVmsStatus (0, ErrorMsgTooMany, FI_LI);
   }

   mlptr->LanguageDefault = mlptr->LanguageCount;

#ifdef DBUG
   Debug = DebugBuffer;
#endif
}

/*****************************************************************************/
/*
Deallocate all memory used in this message database.
*/

MsgUnLoad (struct MsgDbStruct *mlptr)

{
   int  Count,
        LanguageCount;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MsgUnLoad()\n");

   for (LanguageCount = 1;
        LanguageCount <= mlptr->LanguageCount;
        LanguageCount++)
   {
      if (mlptr->Msgs[LanguageCount] != NULL)
      {
         for (Count = 0; Count <= MSG_RANGE; Count++)
         {
            /* the '\b' backspace is a sentinal, will never occur in a message */
            if (mlptr->Msgs[LanguageCount]->TextPtr[Count] != NULL &&
                *((unsigned short*)(mlptr->Msgs[LanguageCount]->TextPtr[Count])) != '\b[')
            {
               /* remember that MsgCreate() returns allocation plus one! */
               VmFree (mlptr->Msgs[LanguageCount]->TextPtr[Count]-1);
            }
         }

         if (mlptr->Msgs[LanguageCount]->HostListPtr != NULL)
            VmFree (mlptr->Msgs[LanguageCount]->HostListPtr);

         VmFree (mlptr->Msgs[LanguageCount]);
      }
   }

   /* clean it out completely, just in case it's to be reused! */
   memset (mlptr, 0, sizeof(struct MsgDbStruct));
}

/*****************************************************************************/
/*
Allocate dynamic memory for the text of the message, copy it into it and
return a pointer to it.  The MapUrl.c module (for horrible, historical reasons)
requires a leading null character before the message.  Fudge this by creating
a one character longer string with that leading null and returning a pointer to
the first character.  The MapUrl.c module will the just use the returned
pointer minus one! (Neat huh?  Well it works anyway!)
*/

char* MsgCreate (char *Text)

{
   char  *MsgPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MsgCreate() |%s|\n", Text);

   MsgPtr = VmGet (strlen(Text)+2);
   *MsgPtr = '\0';
   strcpy (MsgPtr+1, Text);
   return (MsgPtr+1);
}

/*****************************************************************************/
/*
This function formats an error report.  All lines are concatenated onto a
single string of dynamically allocated memory that (obviously) grows as
reports are added to it.  This string is then output if loading the server
configuration or is available for inclusion in an HTML page.
*/

int MsgReportProblem
(
struct MsgDbStruct *mlptr,
char *Explanation,
int StatusValue
)
{
   static $DESCRIPTOR (ExplanationFaoDsc,
          "%HTTPD-W-MSG, !AZ at line !UL\n");
   static $DESCRIPTOR (ExplanationLineFaoDsc,
          "%HTTPD-W-MSG, !AZ at line !UL\n \\!AZ\\\n");

   int  status;
   unsigned short  Length;
   char  Buffer [256],
         HtmlBuffer [512];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MsgReportProblem()\n");

   mlptr->ProblemCount++;

   if (StatusValue)
   {
      if (VMSok (status = sys$getmsg (StatusValue, &Length, &BufferDsc, 0)))
         Buffer[Length++] = '\n';
   }
   else
   if (mlptr->LinePtr[0])
      status = sys$fao (&ExplanationLineFaoDsc, &Length, &BufferDsc,
                        Explanation, mlptr->LineNumber, mlptr->LinePtr);
   else
      status = sys$fao (&ExplanationFaoDsc, &Length, &BufferDsc,
                        Explanation, mlptr->LineNumber);
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "sys$fao()", FI_LI);
   else
      Buffer[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", Buffer);

   if (mlptr->RequestPtr == NULL) fputs (Buffer, stdout);

   if ((Length = CopyToHtml (HtmlBuffer, sizeof(HtmlBuffer), Buffer, -1)) < 0)
      ErrorExitVmsStatus (SS$_BUFFEROVF, "CopyToHtml()", FI_LI);

   mlptr->ProblemReportPtr =
      VmRealloc (mlptr->ProblemReportPtr, mlptr->ProblemReportLength+Length+1);
   /* include the terminating null */
   memcpy (mlptr->ProblemReportPtr+mlptr->ProblemReportLength,
           HtmlBuffer, Length+1);
   mlptr->ProblemReportLength += Length;
}

/*****************************************************************************/
/*
Wraps a call to MsgFor() when there is no request structure.  Must initialize
all request structure fields accessed in MsgFor() and MsgInHostList() to some
reasonable value.  All others can be ignored.
*/

char* MsgForNoRequest
(
char *ClientHostName,
char *ClientInternetAddress,
int Message
)
{
   struct RequestStruct  BogusRequestStruct;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "MsgForNoRequest() |%s|%s|%d|\n",
               ClientHostName, ClientInternetAddress, Message);

   BogusRequestStruct.MsgLanguage = 0;
   BogusRequestStruct.HttpAcceptLangPtr = NULL;
   strcpy (BogusRequestStruct.ClientHostName, ClientHostName);
   strcpy (BogusRequestStruct.ClientInternetAddress, ClientInternetAddress);

   return (MsgFor (&BogusRequestStruct, Message));
}

/*****************************************************************************/
/*
Return a pointer to a character string for the message number supplied in
'Message'.  If multiple languages are in use then use any supplied client list
of accepted languages ("Accept-Language:" request header field) to see if the
message can be supplied in a prefered language.  If none supplied or if none
match then check if the language has geographical information against it (a
list of host/domain specifications that can be used to determine if a specific
language would be more appropriate.  If none of the "hit" then return the
configuration-prefered language.
*/

char* MsgFor
(
struct RequestStruct *rqptr,
int Message
)
{
   int  Count,
        Language;
   char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MsgFor() |%d|%d|\n", rqptr, Message);

   if (rqptr == NULL)
   {
      /* no request structure, use the default language */
      Language = MsgDb.LanguageDefault;
   }
   else
   if (rqptr->MsgLanguage)
   {
      /* request message language has previously been set */
      Language = rqptr->MsgLanguage;
   }
   else
   if (MsgDb.LanguageCount == 1)
   {
      /* only one language in use, use it! */
      Language = rqptr->MsgLanguage = MsgDb.LanguageDefault;
   }
   else
   {
      if (rqptr->HttpAcceptLangPtr != NULL)
      {
         /*******************************************/
         /* look for the client's prefered language */
         /*******************************************/

         register char  *sptr, *lptr;
         int  Count;

         lptr = rqptr->HttpAcceptLangPtr;
         while (*lptr)
         {
            for (Count = 1; Count <= MsgDb.LanguageCount; Count++)
            {
               if (Debug)
                  fprintf (stdout, "%d |%s|%s|\n",
                           Count, lptr, MsgDb.Msgs[Count]->LanguageName);

               cptr = lptr;
               sptr = MsgDb.Msgs[Count]->LanguageName;
               while (toupper(*cptr) == toupper(*sptr) &&
                      *cptr != ',' && !ISLWS(*cptr))
               {
                  cptr++;
                  sptr++;
               }
               if (!*sptr && (!*cptr || *cptr == ',' || ISLWS(*cptr)))
               {
                  Language = rqptr->MsgLanguage =
                     MsgDb.Msgs[Count]->LanguageNumber;
                  break;
               }
            }
            if (rqptr->MsgLanguage) break;
            while (*lptr && *lptr != ',' && !ISLWS(*lptr)) lptr++;
            while (*lptr && (*lptr == ',' || ISLWS(*lptr))) lptr++;
         }
      }

      if (!rqptr->MsgLanguage)
      {
         /************************/
         /* look for a host list */
         /************************/

         for (Count = 1; Count <= MsgDb.LanguageCount; Count++)
         {
            if (Debug)
               fprintf (stdout, "%d |%s|\n",
                        Count, MsgDb.Msgs[Count]->LanguageName);

            if (MsgDb.Msgs[Count]->HostListPtr != NULL)
            {
               if (MsgInHostList (rqptr, MsgDb.Msgs[Count]->HostListPtr))
               {
                  Language = rqptr->MsgLanguage =
                     MsgDb.Msgs[Count]->LanguageNumber;
                  break;
               }
            }
         }
      }

      /* if none matching then fall back to the default language */
      if (!rqptr->MsgLanguage)
         Language = rqptr->MsgLanguage = MsgDb.LanguageDefault;
   }

   if (Debug) fprintf (stdout, "Language: %d\n", Language);

   if (Message <= 0 || Message > MSG_RANGE ||
       Language < 1 || Language > MsgDb.LanguageCount)
      ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);

   /*******************/
   /* get the message */
   /*******************/

   /* if a message has been assigned then return it */
   if ((cptr = MsgDb.Msgs[Language]->TextPtr[Message]) != NULL)
   {
      if (Debug) fprintf (stdout, "|%s|\n", cptr);
      return (cptr);
   }

   /*********************************/
   /* fallback to the base language */
   /*********************************/

   if ((cptr = MsgDb.Msgs[MsgDb.LanguageCount]->TextPtr[Message]) != NULL)
   {
      if (Debug) fprintf (stdout, "|%s|\n", cptr);
      return (cptr);
   }

   /**************************************/
   /* no message was set for this event! */
   /**************************************/

   if (Debug) fprintf (stdout, "|%s|\n", MsgNotSet);
   return (MsgNotSet);
}

/*****************************************************************************/
/*
If the client's IP host name or address matches the wildcard string in
'HostList' then return true, else return false.
*/ 

boolean MsgInHostList
(
struct RequestStruct *rqptr,
char *HostList
)
{
   register char  c;
   register char  *cptr, *hptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "MsgInHostList() |%s|%s|%s|\n",
               rqptr->ClientHostName, rqptr->ClientInternetAddress, HostList);

   hptr = HostList;
   while (*hptr)
   {
      while (*hptr && (*hptr == ',' || ISLWS(*hptr))) hptr++;
      sptr = hptr;
      while (*hptr && *hptr != ',' && !ISLWS(*hptr)) hptr++;
      c = *hptr;
      *hptr = '\0';
      /* match against host address or name */
      if (isdigit(*sptr))
         cptr = rqptr->ClientInternetAddress;
      else
         cptr = rqptr->ClientHostName;
      if (Debug) fprintf (stdout, "|%s|%s|\n", cptr, sptr);
      if (SearchTextString (cptr, sptr, false, NULL) != NULL)
      {
         *hptr = c;
         return (true);
      }
      *hptr = c;
      if (*hptr) hptr++;
   }

   return (false);
}                             

/*****************************************************************************/
/*
A server administration report on the message database.  This function just
wraps the reporting function, loading a temporary database if necessary for
reporting from the configuration file.
*/ 

MsgReport
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
boolean UseServerDatabase
)
{
   int  status;
   struct  MsgDbStruct  LocalMsgDb;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MsgReport()\n");

   if (UseServerDatabase)
      MsgReportNow (rqptr, &MsgDb);
   else
   {
      /* ensure it's an empty database! */
      memset (&LocalMsgDb, 0, sizeof(struct MsgDbStruct));
      /* indicate it's being used for a report */
      LocalMsgDb.RequestPtr = rqptr;
      MsgLoad (&LocalMsgDb);
      MsgReportNow (rqptr, &LocalMsgDb);
      MsgUnLoad (&LocalMsgDb);
      if (LocalMsgDb.ProblemReportPtr != NULL)
         VmFree (LocalMsgDb.ProblemReportPtr);
   }

   NetWriteBuffered (rqptr, NextTaskFunction, NULL, 0);
}

/*****************************************************************************/
/*
A server administration report on the message database ... now.  This function
blocks while executing.
*/ 

MsgReportNow
(
struct RequestStruct *rqptr,
struct MsgDbStruct *mlptr
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Server Messages</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Server Messages</H2>\n\
!AZ\n");

   static $DESCRIPTOR (ProblemReportFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH><FONT COLOR=\"#ff0000\">!UL Problem!AZ !AZ</FONT></TH></TR>\n\
<TR><TD><PRE>!AZ</PRE></TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (MessageLoadFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Source: &quot;!AZ&quot;</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR>\
<TH ALIGN=right>File:</TH>\
<TD ALIGN=left>!AZ</TD>\
<TD>&nbsp;</TD><TH>[<A HREF=\"!AZ\">View</A>]</TH>\
<TH>[<A HREF=\"!AZ!AZ\">Edit</A>]</TH>\
</TR>\n\
<TR><TH ALIGN=right>!AZ</TH>\
<TD ALIGN=left>!AZ</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n");

   static char  LanguageTable [] =
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=3><FONT SIZE=+1>Languages</FONT></TH></TR>\n\
<TR><TH>Order</TH><TH>Language</TH><TH>Host List</TH></TR>\n";

   static char  EndLanguageTable [] = "</TABLE>\n";

   static $DESCRIPTOR (OneLanguageFaoDsc,
"<TR><TD>!UL</TD><TD>!AZ</TD><TD>!AZ</TD></TR>\n");

   static $DESCRIPTOR (GroupFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=3><FONT SIZE=+1>!AZ</FONT></TH></TR>\n\
<TR><TH>Number</TH><TH>Language</TH><TH>Text</TH></TR>\n");

   static $DESCRIPTOR (MessageRowSpanFaoDsc,
"<TR><TD ROWSPAN=!UL>!2ZL</TD><TD>!AZ</TD><TD>!AZ!AZ!AZ</TD></TR>\n");

   static $DESCRIPTOR (MessageMultipleFaoDsc,
"<TR><TD>!AZ</TD><TD>!AZ!AZ!AZ</TD></TR>\n");

   static $DESCRIPTOR (MessageOneFaoDsc,
"<TR><TD>!2ZL</TD><TD>!AZ</TD><TD>!AZ!AZ!AZ</TD></TR>\n");

   char  EndOfGroup [] = "</TABLE>";

   char  MessagesNotLoaded [] =
"<P><FONT SIZE=+1 COLOR=\"#ff0000\">No messages could be loaded!</FONT>\n\
</BODY>\n\
</HTML>\n";

   char  EndOfPage [] =
"</TABLE>\n\
</BODY>\n\
</HTML>\n";

   register char  *cptr;
   register unsigned long  *vecptr;

   boolean  MultipleLines;
   int  status,
        GroupCount,
        Count,
        Language,
        MessageCount,
        RowCount;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [16384],
         HtmlLanguageName [64],
         HtmlText [8192];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MsgReportNow()\n");

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
      return;

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = DayDateTime (&rqptr->BinaryTime, 20);

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   if (mlptr->ProblemReportLength)
   {
      vecptr = FaoVector;
      if ((*vecptr++ = mlptr->ProblemCount) == 1)
         *vecptr++ = "";
      else
         *vecptr++ = "s";
      if (mlptr == &MsgDb)
         *vecptr++ = "Reported At Startup";
      else
         *vecptr++ = "Detected During Load";
      *vecptr = mlptr->ProblemReportPtr;

      status = sys$faol (&ProblemReportFaoDsc, &Length, &BufferDsc,
                         &FaoVector);

      if (status == SS$_BUFFEROVF)
      {
         *vecptr = "TOO MANY PROBLEMS TO LIST!\nWhat have you done? :^)";
         status = sys$faol (&ProblemReportFaoDsc, &Length, &BufferDsc,
                            &FaoVector);
      }

      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   cptr = MapVmsPath (mlptr->LoadFileName);

   vecptr = FaoVector;

   /* source information (i.e. from executing server or static file) */
   if (mlptr == &MsgDb)
      *vecptr++ = "Server";
   else
      *vecptr++ = "File";
   *vecptr++ = mlptr->LoadFileName;
   *vecptr++ = cptr;
   *vecptr++ = HttpdInternalScriptUpd;
   *vecptr++ = cptr;

   if (mlptr == &MsgDb)
   {
      *vecptr++ = "Loaded:";
      *vecptr++ = DayDateTime (&mlptr->LoadBinTime, 20);
   }
   else
   {
      *vecptr++ = "Revised:";
      *vecptr++ = DayDateTime (&mlptr->RevBinTime, 20);
   }

   status = sys$faol (&MessageLoadFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   if (!mlptr->LanguageCount)
   {
      NetWriteBuffered (rqptr, 0, MessagesNotLoaded, sizeof(MessagesNotLoaded)-1);
      rqptr->ResponseStatusCode = 200;
      return;
   }

   /********************/
   /* language summary */
   /********************/

   NetWriteBuffered (rqptr, 0, LanguageTable, sizeof(LanguageTable)-1);

   for (Count = 1; Count <= MsgDb.LanguageCount; Count++)
   {
      vecptr = FaoVector;
      *vecptr++ = Count;
      *vecptr++ = MsgDb.Msgs[Count]->LanguageName;
      if (MsgDb.Msgs[Count]->HostListPtr != NULL)
         *vecptr++ = MsgDb.Msgs[Count]->HostListPtr;
      else
         *vecptr++ = "";

      status = sys$faol (&OneLanguageFaoDsc, &Length, &BufferDsc, &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   NetWriteBuffered (rqptr, 0, EndLanguageTable, sizeof(EndLanguageTable)-1);

   /*****************************/
   /* loop through all messages */
   /*****************************/

   Count = GroupCount = 0;

   for (Count = 0; Count < MSG_RANGE; Count++)
   {
      if (mlptr->Msgs[1]->TextPtr[Count] != NULL)
      {
         /* the '\b' backspace is a sentinal, will never occur in a message */
         if (*((unsigned short*)(mlptr->Msgs[1]->TextPtr[Count])) == '\b[')
         {
            if (GroupCount++)
               NetWriteBuffered (rqptr, 0, EndOfGroup, sizeof(EndOfGroup)-1);

            vecptr = FaoVector;
            *vecptr++ = mlptr->Msgs[1]->TextPtr[Count]+1;

            status = sys$faol (&GroupFaoDsc, &Length, &BufferDsc, &FaoVector);
            if (VMSnok (status) || status == SS$_BUFFEROVF)
            {
               rqptr->ErrorTextPtr = "sys$faol()";
               ErrorVmsStatus (rqptr, status, FI_LI);
               return;
            }

            Buffer[Length] = '\0';
            NetWriteBuffered (rqptr, 0, Buffer, Length);

            Count++;
            MessageCount = 0;
         }
      }

      RowCount = 0;
      for (Language = 1; Language <= mlptr->LanguageCount; Language++)
         if (mlptr->Msgs[Language]->TextPtr[Count] != NULL) RowCount++;

      for (Language = 1; Language <= mlptr->LanguageCount; Language++)
      {
         if (Debug) fprintf (stdout, "Language: %d\n", Language);
         if (mlptr->Msgs[Language]->TextPtr[Count] == NULL) continue;

         /* escape any HTML-forbidden characters */
         CopyToHtml (HtmlLanguageName, sizeof(HtmlLanguageName),
                     mlptr->Msgs[Language]->LanguageName, -1);
         CopyToHtml (HtmlText, sizeof(HtmlText),
                     mlptr->Msgs[Language]->TextPtr[Count], -1);

         /* '*cptr' will not be pointer to a null character if linefeeds */
         for (cptr = mlptr->Msgs[Language]->TextPtr[Count];
              *cptr && *cptr != '\n';
              cptr++);
         if (*cptr)
            MultipleLines = true;
         else
            MultipleLines = false;

         if (RowCount > 1)
         {
            /* first message for multiple languages */
            vecptr = FaoVector;
            *vecptr++ = RowCount;
            *vecptr++ = ++MessageCount;
            *vecptr++ = HtmlLanguageName;
            if (MultipleLines)
               *vecptr++ = "<PRE>";
            else
               *vecptr++ = "<TT>";
            *vecptr++ = HtmlText;
            if (MultipleLines)
               *vecptr++ = "</PRE>";
            else
               *vecptr++ = "</TT>";
            status = sys$faol (&MessageRowSpanFaoDsc, &Length, &BufferDsc,
                               &FaoVector);
            RowCount = 0;
         }
         else
         if (!RowCount)
         {
            /* subsequent message for multiple languages */
            vecptr = FaoVector;
            *vecptr++ = HtmlLanguageName;
            if (MultipleLines)
               *vecptr++ = "<PRE>";
            else
               *vecptr++ = "<TT>";
            *vecptr++ = HtmlText;
            if (MultipleLines)
               *vecptr++ = "</PRE>";
            else
               *vecptr++ = "</TT>";
            status = sys$faol (&MessageMultipleFaoDsc, &Length, &BufferDsc,
                               &FaoVector);
         }
         else
         {
            /* only message for just one language */
            vecptr = FaoVector;
            *vecptr++ = ++MessageCount;
            *vecptr++ = HtmlLanguageName;
            if (MultipleLines)
               *vecptr++ = "<PRE>";
            else
               *vecptr++ = "<TT>";
            *vecptr++ = HtmlText;
            if (MultipleLines)
               *vecptr++ = "</PRE>";
            else
               *vecptr++ = "</TT>";
            status = sys$faol (&MessageOneFaoDsc, &Length, &BufferDsc,
                               &FaoVector);
         }

         if (VMSnok (status) || status == SS$_BUFFEROVF)
         {
            rqptr->ErrorTextPtr = "sys$faol()";
            ErrorVmsStatus (rqptr, status, FI_LI);
            return;
         }

         Buffer[Length] = '\0';
         NetWriteBuffered (rqptr, 0, Buffer, Length);
      }
   }

   /**************/
   /* end report */
   /**************/

   NetWriteBuffered (rqptr, 0, EndOfPage, sizeof(EndOfPage)-1);
}

/*****************************************************************************/

