/*****************************************************************************/
/*
                                   Net.c

The networking essentials are based on DEC TCP/IP Services for OpenVMS (aka
UCX). The QIO interface was obviously chosen because of its asynchronous,
non-blocking capabilities. Although some functionalilty could have been
implemented using the BSD sockets abstraction and only the asynchronous I/O
done using the QIO I opted for working out the QIOs. It wasn't too difficult
and avoids the (slight) extra overhead of the sockets layer. The MadGoat
NETLIB port follows the basic outline of the DEC TCP/IP functionality. I
deliberately differentiated the UCX and NETLIB interfaces. It wasn't much
extra work and helps keep obvious the fact of the two APIs. NETLIB is a very
nice package! Wished I'd known about it three years ago. My thanks Matthew.

With resource wait explicitly enabled all sys$qio()s should wait until
resources become available. The only execption is ASTLM, a fundamental
requirement for this server. If SS$_EXQUOTA is returned from a sys$qio() (i.e.
without wait for completion, and therefore depending on AST delivery) then
EXIT THE SERVER with error status!

Some underlying NETLIB supporting TCP/IP packages do not support host-name
resolution at AST delivery level, which this module uses, and hence CANNOT
SUPPORT DNS LOOKUP from v4.4 onwards! (See the NETLIB documentation.)

The 'ServiceStruct' structure has two groups of fields and performs two basic
roles. First, it provides a linked list which can be traversed to determine
from the destination IP address of a request which host was specified in a
(multi-homed) request. Second, for each unique port specified for the same
server the client components of the structure are used for the accept()ing of
requests.  Note that there may be more 'service' structures in the list than
actual sockets created (and client components used) because only one is
allocated against a given port even though the services may specify multiple
host names using that port.

The server identity is derived from the 'ServerHostName' (gethostname()) name
and the 'ServerPort' (which is the first port specified as a service, or the
port specified in the configuration or /PORT= qualifier).


VERSION HISTORY
---------------
08-APR-98  MGD  allow legitimate connects to be CANCELed in NetAcceptAst()
19-JAN-98  MGD  redesigned NetWriteBuffer()
27-NOV-97  MGD  hmmm, rationalized AST processing by making network
                writes always deliver an AST (implicit or explicit)
                (causing ACCVIOs for the last couple of versions)
25-OCT-97  MGD  changes to MsgFor() function
06-SEP-97  MGD  multi-homed hosts and multi-port services
30-AUG-97  MGD  bugfix; get server host name before starting logging
                (woops, problem introduced with NETLIB)
09-AUG-97  MGD  message database
23-JUL-97  MGD  HTTPD v4.3, MultiNet dropped, using the NETLIB Library
01-FEB-97  MGD  HTTPd version 4
27-SEP-96  MGD  add dummy read for early error reporting
12-APR-96  MGD  observed Multinet disconnection/zero-byte behaviour
                (request now aborts if Multinet returns zero bytes)
03-JAN-96  MGD  support for both DEC TCP/IP Services and TGV MultiNet
01-DEC-95  MGD  NetWriteBuffered() for improving network I/O
20-DEC-94  MGD  multi-threaded version
20-JUN-94  MGD  single-threaded version
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <descrip.h>
#include <dvidef.h>
#include <iodef.h>
#include <psldef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application-related header files */
#include "wasd.h"
#include "msg.h"
#include "net.h"
#include "error.h"
#include "vm.h"

/******************/
/* global storage */
/******************/

int  QueueLength = 5,
     ServiceCount = 0,
     ServerHostNameLength;

char  ServerHostName [128],
      ServerHostPort [128],
      Services [256];

struct ServiceStruct  *ServiceListHead,
                      *ServiceListTail;

char  ErrorServiceList [] = "Service list confusing.";

#ifdef IP_UCX

char HttpdIpPackage [] = "Digital-TCPIP";

$DESCRIPTOR (InetDeviceDsc, "UCX$DEVICE");

int  OptionEnabled = 1;

struct {
   int  l_onoff;
   int  l_linger;
} LingerSeconds = { 1, 10 };

struct {
   unsigned short  Length;
   unsigned short  Parameter;
   char  *Address;
} ReuseAddress =
   { sizeof(OptionEnabled), UCX$C_REUSEADDR, &OptionEnabled },
  ReuseAddressSocketOption =
   { sizeof(ReuseAddress), UCX$C_SOCKOPT, &ReuseAddress },
  LingerOnClose =
   { sizeof(LingerSeconds), UCX$C_LINGER, &LingerSeconds },
  LingerOnCloseOption =
   { sizeof(LingerOnClose), UCX$C_SOCKOPT, &LingerOnClose };

unsigned short  TcpSocket [2] =
   { UCX$C_TCP, INET_PROTYP$C_STREAM };

#endif /* IP_UCX */


#ifdef IP_NETLIB

char HttpdIpPackage [] = "MadGoat-NETLIB";

unsigned long  NetLibAddressListSize = 16,
               NetLibAddressListCount,
               NetLibSocketType = NETLIB_K_TYPE_STREAM,
               NetLibSocketFamily = NETLIB_K_AF_INET,
               NetLibLevel = NETLIB_K_LEVEL_SOCKET,
               NetLibReUseOption = NETLIB_K_OPTION_REUSEADDR,
               NetLibShutdownRecv = NETLIB_K_SHUTDOWN_RECEIVER,
               NetLibShutdownBoth = NETLIB_K_SHUTDOWN_BOTH,
               NetLibOptionOn = 1,
               NetLibSizeofINADDRDEF = sizeof(struct INADDRDEF),
               NetLibSizeofSINDEF = sizeof(struct SINDEF),
               NetLibSizeofOptionOn = sizeof(unsigned long);

#endif /* IP_NETLIB */


/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern boolean  ControlExitRequested;
extern boolean  ControlRestartRequested;
extern boolean  ProtocolHttpsAvailable;
extern boolean  ProtocolHttpsConfigured;
extern int  CurrentConnectCount;
extern int  MaxConcurrentConnections;
extern int  OutputBufferSize;
extern int  ServerPort;
extern char  CliLogFileName[];
extern char  ControlBuffer[];
extern char  ErrorSanityCheck[];
extern char  Utility[];
extern char  *AcceptHostsPtr;
extern char  *RejectHostsPtr;
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/*****************************************************************************/
/*
Get local host (server) name.
*/ 

NetGetServerHostName ()

{
   register char  *cptr;

   int  status;
   unsigned short  Length;

#ifdef IP_NETLIB
   $DESCRIPTOR (ServerHostNameDsc, ServerHostName);
#endif

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetGetServerHostName()\n");

#ifdef IP_UCX

   if (gethostname (ServerHostName, sizeof(ServerHostName)) != 0)
      ErrorExitVmsStatus (0, strerror(errno), FI_LI);

#endif

#ifdef IP_NETLIB

   status = netlib_get_hostname (&ServerHostNameDsc, &Length);
   if (Debug) fprintf (stdout, "netlib_get_hostname() %%X%08.08X\n", status);
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "netlib_get_hostname()", FI_LI);
   ServerHostName[Length] = '\0';

#endif

   /* looks better if its all in lower case (host name is sometimes upper) */
   for (cptr = ServerHostName; *cptr; cptr++) *cptr = tolower(*cptr);
   ServerHostNameLength = cptr - ServerHostName;
   if (Debug) fprintf (stdout, "|%s|\n", ServerHostName);
}

/*****************************************************************************/
/*
Parse the 'Service's parameter to detmine which host(s)/port(s) services need
to be provided for.
*/ 

NetGetService ()

{
   register char  *cptr, *sptr;
   register struct ServiceStruct *svptr;

   boolean  SetServerPort;
   int  status,
        PortNumber,
        RequestScheme;
   char *RequestSchemeNamePtr;
   char  HostName [128];

#ifdef IP_UCX
   struct in_addr  ServerAddress;
#endif

#ifdef IP_NETLIB
   unsigned short  Length;
   unsigned long  NetLibSocket;
   struct  AnIOsb  IOsb;
   $DESCRIPTOR (HostNameDsc, HostName);
   $DESCRIPTOR (InternetAddressDsc, "");
#endif

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetGetService()\n");

   SetServerPort = false;

   if (Services[0])
      cptr = Services;
   else
   if (Config.ServicePtr == NULL)
      sprintf (cptr = Services, "%s:%d", ServerHostName, ServerPort);
   else
      cptr = Config.ServicePtr;

   while (*cptr)
   {
      while (*cptr && (ISLWS(*cptr) || *cptr == ',')) cptr++;
      if (!*cptr) break;

      HostName[0] = '\0';
      RequestScheme = PortNumber = 0;
      RequestSchemeNamePtr = NULL;

      if (Debug) fprintf (stdout, "|%s|\n", cptr);
      if (isalpha(*cptr))
      {
         if (strsame (cptr, "http:", 5))
         {
            RequestScheme = SCHEME_HTTP;
            RequestSchemeNamePtr = "http:";
            cptr += 5;
            while (*cptr == '/') cptr++;
         }
         else
         if (strsame (cptr, "https:", 6))
         {
            RequestScheme = SCHEME_HTTPS;
            RequestSchemeNamePtr = "https:";
            if (ProtocolHttpsAvailable) ProtocolHttpsConfigured = true;
            cptr += 6;
            while (*cptr == '/') cptr++;
         }
      }

      if (isalpha(*cptr))
      {
         sptr = HostName;
         while (*cptr && !ISLWS(*cptr) && *cptr != ',' && *cptr != ':')
            *sptr++ = *cptr++;
         *sptr = '\0';
         if (*cptr == ':')
         {
            cptr++;
            PortNumber = atol(cptr);
            while (*cptr && isdigit(*cptr)) cptr++;
         }
      }

      if (isdigit(*cptr))
      {
         PortNumber = atol(cptr);
         while (*cptr && isdigit(*cptr)) cptr++;
      }

      if (*cptr && *cptr != ',')
         ErrorExitVmsStatus (status, ErrorServiceList, FI_LI);

      if (!RequestScheme) RequestScheme = SCHEME_HTTP;
      if (RequestScheme == SCHEME_HTTP)
         RequestSchemeNamePtr = "http:";
      else
         RequestSchemeNamePtr = "https:";
      if (!HostName[0]) strcpy (HostName, ServerHostName);
      if (!PortNumber)
      {
         if (RequestScheme == SCHEME_HTTP)
            PortNumber = 80;
         else
            PortNumber = 443;
      }
      if (Debug) fprintf (stdout, "|%s:%d|\n", HostName, PortNumber);

      if (!ProtocolHttpsAvailable && RequestScheme == SCHEME_HTTPS)
      {
         fprintf (stdout,
            "%%%s-W-SERVICE, SSL not available, ignored\n \\%s//%s:%d\\\n",
            Utility, RequestSchemeNamePtr, HostName, PortNumber);
         continue;
      }

      /* allocate memory for a new service structure */
      svptr = VmGet (sizeof (struct ServiceStruct));
      if (Debug) fprintf (stdout, "svptr: %d\n", svptr);

      /* add it to the linked list */
      if (ServiceListHead == NULL)
         ServiceListHead = ServiceListTail = svptr;
      else
      {
         ServiceListTail->NextPtr = svptr;
         ServiceListTail = svptr;
      }
      svptr->NextPtr = NULL;
      ServiceCount++;

      svptr->ConnectCount = 0;

#ifdef IP_UCX

      if ((svptr->HostEntryPtr = gethostbyname (HostName)) == NULL)
      {
         fprintf (stdout, "%%%s-I-SERVICE, %s//%s:%d\n",
                  Utility, RequestSchemeNamePtr, HostName, PortNumber);
         ErrorExitVmsStatus (0, strerror(errno), FI_LI);
      }

      strcpy (svptr->ServerHostName, svptr->HostEntryPtr->h_name);
      memcpy (&svptr->ServerInternetAddress32bit,
              svptr->HostEntryPtr->h_addr, 4);
      memcpy (&ServerAddress, svptr->HostEntryPtr->h_addr, 4);
      strcpy (svptr->ServerInternetAddress, inet_ntoa (ServerAddress));

#endif

#ifdef IP_NETLIB

      /* NETLIB needs a socket to do these, create a temporary one */
      if (Debug) fprintf (stdout, "netlib_socket()\n");
      status = netlib_socket (&NetLibSocket, &NetLibSocketType,
                              &NetLibSocketFamily);
      if (Debug) fprintf (stdout, "netlib_socket() %%X%08.08X\n", status);
      if (VMSnok (status))
         ErrorExitVmsStatus (status, "netlib_socket()", FI_LI);

      /* turn the name into a binary address */
      HostNameDsc.dsc$a_pointer = HostName;
      HostNameDsc.dsc$w_length = strlen(HostName);

      status = netlib_name_to_address (&NetLibSocket, 0, &HostNameDsc,
                                       &svptr->NetLibAddressList,
                                       &NetLibAddressListSize,
                                       &NetLibAddressListCount,
                                       &IOsb, 0, 0);
      if (Debug)
         fprintf (stdout,
            "netlib_name_to_address() %%X%08.08X IOsb %%X%08.08X\n",
            status, IOsb.Status);
      if (VMSok (status) && VMSnok (IOsb.Status))
         status = IOsb.Status;
      if (VMSnok (status))
      {
         fprintf (stdout, "%%%s-I-SERVICE, %s//%s:%d\n",
                  Utility, RequestSchemeNamePtr, HostName, PortNumber);
         ErrorExitVmsStatus (status, "netlib_name_to_address()", FI_LI);
      }

      svptr->ServerInternetAddress32bit =
         svptr->NetLibAddressList[0].inaddr_l_addr;

      /* convert the binary address into a string */
      InternetAddressDsc.dsc$a_pointer = svptr->ServerInternetAddress;
      InternetAddressDsc.dsc$w_length = sizeof(svptr->ServerInternetAddress)-1;
      status = netlib_addrtostr (&svptr->NetLibAddressList[0],
                                 &InternetAddressDsc, &Length);

      if (Debug) fprintf (stdout, "netlib_addrtostr() %%X%08.08X\n", status);
      if (VMSnok (status))
         ErrorExitVmsStatus (status, "netlib_addrtostr()", FI_LI);
      svptr->ServerInternetAddress[Length] = '\0';
      if (Debug) fprintf (stdout, "|%s|\n", svptr->ServerInternetAddress);

      /* transform abbreviated host name to full ("host" into "host.domain") */
      HostNameDsc.dsc$w_length = sizeof(svptr->ServerHostName)-1;
      HostNameDsc.dsc$a_pointer = &svptr->ServerHostName;

      Length = 0;
      status = netlib_address_to_name (&NetLibSocket, 0,
                  &svptr->NetLibAddressList[0].inaddr_l_addr,
                  &NetLibSizeofINADDRDEF, &HostNameDsc, &Length, &IOsb, 0, 0);
      if (Debug)
         fprintf (stdout,
            "netlib_address_to_name() %%X%08.08X IOsb %%X%08.08X\n",
            status, IOsb.Status);
      svptr->ServerHostName[Length] = '\0';

      /* get rid of the temporary socket */
      status = netlib_close (&NetLibSocket);
      if (Debug) fprintf (stdout, "netlib_close() %%X%08.08X\n", status);

#endif

      svptr->RequestScheme = RequestScheme;
      svptr->RequestSchemeNamePtr = RequestSchemeNamePtr;
      svptr->ServerPort = PortNumber;
      sprintf (svptr->ServerPortString, "%d", svptr->ServerPort);

      /* looks better if its all in lower case (sometimes upper) */
      for (sptr = svptr->ServerHostName; *sptr; sptr++) *sptr = tolower(*sptr);

      sprintf (svptr->ServerHostPort, "%s:%d",
               svptr->ServerHostName, svptr->ServerPort);
      svptr->ServerHostNameLength = strlen(svptr->ServerHostName);

      if (Debug)
         fprintf (stdout, "%08.08X|%s|%d|%s|%s|%d|%s|\n",
                  svptr->ServerInternetAddress32bit,
                  svptr->ServerInternetAddress,
                  svptr->RequestScheme, svptr->RequestSchemeNamePtr,
                  svptr->ServerHostName, svptr->ServerPort,
                  svptr->ServerHostPort);

      /* process name, etc., generated from the primary port */
      if (!SetServerPort) SetServerPort = ServerPort = svptr->ServerPort;
   }

   if (!ServiceCount)
      ErrorExitVmsStatus (0, "No service specified!", FI_LI);

   /* the host and port for the "official" server */
   sprintf (ServerHostPort, "%s:%d", ServerHostName, ServerPort);
   if (Debug) fprintf (stdout, "|%s|\n", ServerHostPort);
}

/*****************************************************************************/
/*
It is only necessary to create one socket for each port port on the same
server, even if that port is specified against a different (multi-home) host
name because the accept()ing socket can be interrogated to determine which
host had been specified.  Then in a infinite loop accept connections from
clients.
*/ 

NetCreateService ()

{
   register char  *cptr, *sptr;
   register struct ServiceStruct  *svptr, *tsvptr;

   int  status,
        PortNumber;
   unsigned short  Length;
   char  *ProtocolPtr;
   
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetCreateService()\n");

   for (svptr = ServiceListHead; svptr != NULL; svptr = svptr->NextPtr)
   {
      fprintf (stdout, "%%%s-I-SERVICE, %s//%s\n",
               Utility, svptr->RequestSchemeNamePtr, svptr->ServerHostPort);

      /**************************************/
      /* check for previous channel on port */
      /**************************************/

      for (tsvptr = ServiceListHead; tsvptr != NULL; tsvptr = tsvptr->NextPtr)
      {
         /* skip over this entry */
         if (svptr == tsvptr)
         {
            /* got this far in the list, can't be a socket for this port! */
            tsvptr = NULL;
            break;
         }
         /* break if already a socket created for this port */
         if (svptr->ServerPort == tsvptr->ServerPort) break;
      }
      /* if another host already on the same port then just continue */
      if (tsvptr != NULL) continue;

      /*************************/
      /* create service socket */
      /*************************/

#ifdef IP_UCX

      /* assign a channel to the internet template device */
      if (VMSnok (status =
          sys$assign (&InetDeviceDsc, &svptr->ServerChannel, 0, 0)))
         ErrorExitVmsStatus (status, "sys$assign()", FI_LI);

      /* make the channel a TCP, connection-oriented socket */

      if (Debug) fprintf (stdout, "sys$qiow() IO$_SETMODE\n");
      status = sys$qiow (0, svptr->ServerChannel, IO$_SETMODE,
                         &svptr->ServerIOsb, 0, 0,
                         &TcpSocket, 0, 0, 0, &ReuseAddressSocketOption, 0);

      if (Debug)
         fprintf (stdout, "sys$qiow() %%X%08.08X IOsb %%X%08.08X\n",
                  status, svptr->ServerIOsb.Status);
      if (VMSok (status) && VMSnok (svptr->ServerIOsb.Status))
         status = svptr->ServerIOsb.Status;
      if (VMSnok (status))
         ErrorExitVmsStatus (status, "sys$qiow()", FI_LI);

#endif

#ifdef IP_NETLIB

      if (Debug) fprintf (stdout, "netlib_socket()\n");
      status = netlib_socket (&svptr->ServerNetLibSocket, &NetLibSocketType,
                              &NetLibSocketFamily);
      if (Debug) fprintf (stdout, "netlib_socket() %%X%08.08X\n", status);
      if (VMSnok (status))
         ErrorExitVmsStatus (status, "netlib_socket()", FI_LI);

      status = netlib_setsockopt (&svptr->ServerNetLibSocket, &NetLibLevel,
                                  &NetLibReUseOption,
                                  &NetLibOptionOn, &NetLibSizeofOptionOn,
                                  &svptr->ServerIOsb, 0, 0);
      if (Debug)
         fprintf (stdout, "netlib_setsockopt() %%X%08.08X IOsb %%X%08.08X\n",
                  status, svptr->ServerIOsb.Status);
      if (VMSok (status) && VMSnok (svptr->ServerIOsb.Status))
         status = svptr->ServerIOsb.Status;
      if (VMSnok (status))
         ErrorExitVmsStatus (status, "netlib_setsockopt()", FI_LI);

#endif

      /**************************/
      /* bind the server socket */
      /**************************/

#ifdef IP_UCX

      svptr->ServerSocketNameItem.Length = sizeof(svptr->ServerSocketName);
      svptr->ServerSocketNameItem.Address = &svptr->ServerSocketName;

      svptr->ServerSocketName.sin_family = AF_INET;
      svptr->ServerSocketName.sin_port = htons (svptr->ServerPort);
      svptr->ServerSocketName.sin_addr.s_addr = INADDR_ANY;

      if (Debug) fprintf (stdout, "sys$qiow() IO$_SETMODE\n");
      status = sys$qiow (0, svptr->ServerChannel, IO$_SETMODE,
                         &svptr->ServerIOsb, 0, 0,
                         0, 0, &svptr->ServerSocketNameItem, QueueLength, 0, 0);
      if (Debug)
         fprintf (stdout, "sys$qiow() %%X%08.08X IOsb %%X%08.08X\n",
                  status, svptr->ServerIOsb.Status);
      if (VMSok (status) && VMSnok (svptr->ServerIOsb.Status))
         status = svptr->ServerIOsb.Status;
      if (VMSnok (status))
         ErrorExitVmsStatus (status, "sys$qiow()", FI_LI);

#endif

#ifdef IP_NETLIB

      svptr->ServerNetLibName.sin_w_family = NETLIB_K_AF_INET;
      svptr->ServerNetLibName.sin_w_port =
         netlib_hton_word (&svptr->ServerPort);
      svptr->ServerNetLibName.sin_x_addr.inaddr_l_addr = 0;  /* INADDR_ANY */

      if (Debug) fprintf (stdout, "netlib_bind()\n");
      status = netlib_bind (&svptr->ServerNetLibSocket,
                            &svptr->ServerNetLibName,
                            &NetLibSizeofSINDEF, &svptr->ServerIOsb, 0, 0);
      if (Debug)
         fprintf (stdout, "netlib_bind %%X%08.08X IOsb %%X%08.08X\n",
                  status, svptr->ServerIOsb.Status);
      if (VMSok (status) && VMSnok (svptr->ServerIOsb.Status))
         status = svptr->ServerIOsb.Status;
      if (VMSnok (status))
         ErrorExitVmsStatus (status, "netlib_bind()", FI_LI);

      if (Debug) fprintf (stdout, "netlib_listen()\n");
      status = netlib_listen (&svptr->ServerNetLibSocket, &QueueLength,
                              &svptr->ServerIOsb, 0, 0);
      if (Debug)
         fprintf (stdout, "netlib_listen %%X%08.08X IOsb %%X%08.08X\n",
                  status, svptr->ServerIOsb.Status);
      if (VMSok (status) && VMSnok (svptr->ServerIOsb.Status))
         status = svptr->ServerIOsb.Status;
      if (VMSnok (status))
         ErrorExitVmsStatus (status, "netlib_listen()", FI_LI);

#endif

      /****************************/
      /* listen for first request */
      /****************************/

      NetAccept (svptr);
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Queue an accept() to the listening server socket.
*/ 

NetAccept (struct ServiceStruct *svptr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetAccept() %d\n", svptr);

   if (ControlExitRequested || ControlRestartRequested)
   {
      /* server exiting or restarting, no new accepts, just return */
      return;
   }

#ifdef IP_UCX

   /* assign a channel to the internet template device */
   if (VMSnok (status =
       sys$assign (&InetDeviceDsc, &svptr->ClientChannel, 0, 0)))
      ErrorExitVmsStatus (status, "sys$assign()", FI_LI);

   svptr->ClientSocketNameItem.Length = sizeof(svptr->ClientSocketName);
   svptr->ClientSocketNameItem.Address = &svptr->ClientSocketName;
   svptr->ClientSocketNameItem.LengthPtr = &svptr->ClientSocketNameLength;

   if (Debug) fprintf (stdout, "sys$qio() IO$_ACCESS | IO$M_ACCEPT\n");
   status = sys$qio (0, svptr->ServerChannel, IO$_ACCESS | IO$M_ACCEPT,
                     &svptr->ServerIOsb, &NetAcceptAst, svptr,
                     0, 0, &svptr->ClientSocketNameItem, &svptr->ClientChannel,
                     &LingerOnCloseOption, 0);
   if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
   if (VMSnok (status)) 
      ErrorExitVmsStatus (status, "sys$qio()", FI_LI);

/*
   if (Debug) fprintf (stdout, "sys$qio() IO$_SETMODE\n");
   status = sys$qiow (0, svptr->ClientChannel, IO$_SETMODE,
                     &svptr->ServerIOsb, 0, 0,
                     0, 0, 0, 0, &LingerOnCloseOption, 0);
   if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
   if (VMSnok (status)) 
      ErrorExitVmsStatus (status, "sys$qio()", FI_LI);
*/

#endif

#ifdef IP_NETLIB

   if (Debug) fprintf (stdout, "netlib_accept()\n");
   status = netlib_accept (&svptr->ServerNetLibSocket,
                           &svptr->ClientNetLibSocket,
                           &svptr->ClientNetLibName, &NetLibSizeofSINDEF,
                           &svptr->NetLibReturnedLength,
                           &svptr->ServerIOsb, &NetAcceptAst, svptr);
   if (Debug)
      fprintf (stdout, "netlib_accept() %%X%08.08X IOsb %%X%08.08X\n",
               status, svptr->ServerIOsb.Status);
   if (VMSok (status) && VMSnok (svptr->ServerIOsb.Status))
      status = svptr->ServerIOsb.Status;
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "netlib_accept()", FI_LI);

   /* option LINGER doesn't seem to be an option :^) ... perhaps default? */

#endif

}

/*****************************************************************************/
/*
A connection has been accept()ed on the specified server socket.
*/ 

NetAcceptAst (struct ServiceStruct *svptr)

{
   register struct RequestStruct  *rqptr;
   register struct ServiceStruct  *tsvptr;

   int  status,
        ServerSocketNameLength;
   unsigned short  PortNumber;
   char  ServerInternetAddress [32];
   struct AnIOsb  IOsb;

#ifdef IP_UCX
   int  SocketNameLength;
   struct sockaddr_in  SocketName;
   struct {
      unsigned int  Length;
      char  *Address;
      unsigned int  *LengthPtr;
   } SocketNameItem;
#endif

#ifdef IP_NETLIB
   static $DESCRIPTOR (InternetAddressDsc, "");
   unsigned short  Length;
   unsigned long  NetLibReturnedLength;
   struct SINDEF  NetLibName;
#endif

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetAcceptAst() %d\n", svptr);

#ifdef IP_UCX
   if (Debug)
      fprintf (stdout, "sys$qio() IOsb %%X%08.08X\n",
               svptr->ServerIOsb.Status);
#endif

#ifdef IP_NETLIB
   if (Debug)
      fprintf (stdout, "netlib_accept() IOsb %%X%08.08X\n",
               svptr->ServerIOsb.Status);
#endif
      
   if (VMSnok (svptr->ServerIOsb.Status)) 
   {
      if ((ControlExitRequested || ControlRestartRequested) &&
          svptr->ServerIOsb.Status == SS$_ABORT)
      {
         /* server exiting or restarting, no new accepts, just return */
         return;
      }

      if (svptr->ServerIOsb.Status == SS$_CANCEL ||
          svptr->ServerIOsb.Status == SS$_ABORT)
      {
         /* connect dropped, forget it, ready for next connection */
         NetAccept (svptr);
         return;
      }

      /* most often network/system shutting down ... SS$_SHUT */
      ErrorExitVmsStatus (svptr->ServerIOsb.Status, "accept()", FI_LI);
   }

   Accounting.ConnectCount++;

#ifdef IP_UCX

   svptr->ClientInternetAddress32bit =
      svptr->ClientSocketName.sin_addr.S_un.S_addr;
   strcpy (svptr->ClientInternetAddress,
           inet_ntoa (svptr->ClientSocketName.sin_addr));
   if (Debug) fprintf (stdout, "|%s|\n", svptr->ClientInternetAddress);

   if (ServiceCount == 1)
   {
      /* "they can have any color they want, provided it's black" */
      tsvptr = ServiceListHead;
   }
   else
   {
      /************************************/
      /* multiple services, get host/port */
      /************************************/

      SocketNameItem.Length = sizeof(SocketName);
      SocketNameItem.Address = &SocketName;
      SocketNameItem.LengthPtr = &SocketNameLength;

      if (Debug) fprintf (stdout, "sys$qiow() IO$_SENSEMODE\n");
      status = sys$qiow (0, svptr->ClientChannel, IO$_SENSEMODE,
                         &IOsb, 0, 0, 0, 0, &SocketNameItem, 0, 0, 0);
      if (Debug)
         fprintf (stdout, "sys$qiow() %%X%08.08X IOsb %%X%08.08X\n",
                  status, IOsb.Status);
      if (VMSok (status) && VMSnok (IOsb.Status))
         status = IOsb.Status;
      if (VMSnok (status))
      {
         /* hmmm, forget it, ready for next connection */
         NetAccept (svptr);
         return;
      }

      PortNumber = ntohs(SocketName.sin_port);

      if (Debug)
         fprintf (stdout, "|%s|%d|\n",
                  inet_ntoa (SocketName.sin_addr), PortNumber);

      for (tsvptr = ServiceListHead; tsvptr != NULL; tsvptr = tsvptr->NextPtr)
      {
         if (SocketName.sin_addr.s_addr ==
             tsvptr->ServerInternetAddress32bit &&
             PortNumber == tsvptr->ServerPort) break;
      }
      /* if none matches then 'tsvptr' is set to NULL */
   }

#endif

#ifdef IP_NETLIB

   InternetAddressDsc.dsc$a_pointer = svptr->ClientInternetAddress;
   InternetAddressDsc.dsc$w_length = sizeof(svptr->ClientInternetAddress)-1;

   svptr->ClientInternetAddress32bit =
      svptr->ClientNetLibName.sin_x_addr.inaddr_l_addr;

   status = netlib_addrtostr (&svptr->ClientNetLibName.sin_x_addr.inaddr_l_addr,
                              &InternetAddressDsc, &Length);
   if (Debug) fprintf (stdout, "netlib_addrtostr() %%X%08.08X\n", status);
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "netlib_addrtostr()", FI_LI);

   svptr->ClientInternetAddress[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", svptr->ClientInternetAddress);

   if (ServiceCount == 1)
   {
      /* ditto */
      tsvptr = ServiceListHead;
   }
   else
   {
      /************************************/
      /* multiple services, get host/port */
      /************************************/

      if (Debug) fprintf (stdout, "netlib_getsockname()\n");
      status = netlib_getsockname (&svptr->ClientNetLibSocket,
                                   &NetLibName, &NetLibSizeofSINDEF,
                                   &NetLibReturnedLength, &IOsb, 0, 0);
      if (Debug)
         fprintf (stdout, "netlib_getsockname %%X%08.08X IOsb %%X%08.08X\n",
                  status, IOsb.Status);
      if (VMSok (status) && VMSnok (IOsb.Status))
         status = IOsb.Status;
      if (VMSnok (status))
      {
         /* hmmm, forget it, ready for next connection */
         NetAccept (svptr);
         return;
      }

      PortNumber = netlib_ntoh_word (&NetLibName.sin_w_port);

      if (Debug)
      {
         char  AddressString [32];
         unsigned short  Length;
         $DESCRIPTOR (AddressStringDsc, AddressString);

         netlib_addrtostr (&NetLibName.sin_x_addr, &AddressStringDsc, &Length);
         AddressString[Length] = '\0';
         fprintf (stdout, "|%s|%d|\n", AddressString, PortNumber);
      }

      for (tsvptr = ServiceListHead; tsvptr != NULL; tsvptr = tsvptr->NextPtr)
      {
         if (NetLibName.sin_x_addr.inaddr_l_addr ==
             tsvptr->ServerInternetAddress32bit &&
             PortNumber == tsvptr->ServerPort) break;
      }
      /* if none matches then 'tsvptr' is set to NULL */
   }

#endif

   if (Debug && tsvptr != NULL)
      fprintf (stdout, "|%s|%s:%d|\n",
               tsvptr->ServerInternetAddress,
               tsvptr->ServerHostName, tsvptr->ServerPort);

   if (tsvptr == NULL)
      svptr->ConnectCount++;
   else
      tsvptr->ConnectCount++;

   if (svptr->RequestScheme == SCHEME_HTTPS)
      Accounting.ConnectSslCount++;

   if (Config.DnsLookup)
      NetHostLookup (svptr);
   else
      strcpy (svptr->ClientHostName, svptr->ClientInternetAddress);
   if (Debug) fprintf (stdout, "|%s|\n", svptr->ClientHostName);

   if (CurrentConnectCount >= MaxConcurrentConnections)
   {
      /************/
      /* too busy */
      /************/

      Accounting.ConnectTooBusyCount++;
      Accounting.ResponseStatusCodeCountArray[5]++;

#ifdef IP_UCX
      NetTooBusy (svptr->ClientChannel,
                  svptr->ClientHostName,
                  svptr->ClientInternetAddress);
      NetDummyRead (svptr->ClientChannel);
#endif

#ifdef IP_NETLIB
      NetTooBusy (&svptr->ClientNetLibSocket,
                  svptr->ClientHostName,
                  svptr->ClientInternetAddress);
      NetDummyRead (&svptr->ClientNetLibSocket);
#endif

      /* ready for next connection */
      NetAccept (svptr);

      return;
   }

   /* 'tsvptr' is NULL if the request did not match a known service */
   if (tsvptr == NULL ||
       !ConfigAcceptClientHostName (svptr->ClientInternetAddress,
                                    svptr->ClientHostName))
   {
      /************/
      /* rejected */
      /************/

      Accounting.ConnectRejectedCount++;
      Accounting.ResponseStatusCodeCountArray[4]++;

#ifdef IP_UCX
      NetAccessDenied (svptr->ClientChannel,
                       svptr->ClientHostName,
                       svptr->ClientInternetAddress);
      NetDummyRead (svptr->ClientChannel);
#endif

#ifdef IP_NETLIB
      NetAccessDenied (&svptr->ClientNetLibSocket,
                       svptr->ClientHostName,
                       svptr->ClientInternetAddress);
      NetDummyRead (&svptr->ClientNetLibSocket);
#endif

      /* ready for next connection */
      NetAccept (svptr);

      return;
   }

   Accounting.ConnectAcceptedCount++;

   /* allocate zeroed dynamic memory for the connection thread */
   rqptr = VmGetRequest ();

   /***************************/
   /* process HTTP connection */
   /***************************/

   /* once the thread structure has been created we have a client! */
   Accounting.ConnectCurrent = ++CurrentConnectCount;
   if (CurrentConnectCount > Accounting.ConnectPeak)
      Accounting.ConnectPeak = CurrentConnectCount;

   /* point at this service' structure */
   rqptr->ServicePtr = tsvptr;

   /* point at and copy the scheme (protocol) the request is using */
   rqptr->RequestScheme = tsvptr->RequestScheme;
   rqptr->RequestSchemeNamePtr = tsvptr->RequestSchemeNamePtr;

   /* copy the client's network information */
   rqptr->ClientInternetAddress32bit = svptr->ClientInternetAddress32bit;
   strcpy (rqptr->ClientInternetAddress, svptr->ClientInternetAddress);
   strcpy (rqptr->ClientHostName, svptr->ClientHostName);

#ifdef IP_UCX
   rqptr->ClientChannel = svptr->ClientChannel;
#endif

#ifdef IP_NETLIB
   rqptr->ClientNetLibSocket = svptr->ClientNetLibSocket;
#endif

   RequestBegin (rqptr, true);

   /* ready for next connection */
   NetAccept (svptr);
}

/*****************************************************************************/
/*
Some browsers report a network error (e.g. Windows Netscape Navigator 2.n) if
the server does not read anything from them before returning an error message.
This routine and it's AST ensure a read is done before reporting an error
earlier that reading the request from it.  Needless-to-say, no status checking
is performed and the data just discarded (hope it works consistently :^)
*/ 

#ifdef IP_UCX

NetDummyRead (unsigned short Channel)

{
   static char  DummyBuffer [1024];
   static struct AnIOsb  DummyIOsb;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetDummyRead() Channel: %d\n", Channel);

   status = sys$qio (0, Channel,
                     IO$_READVBLK, &DummyIOsb,
                     &NetDummyReadAst, Channel,
                     DummyBuffer, sizeof(DummyBuffer), 0, 0, 0, 0);

   if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);

}
#endif

#ifdef IP_NETLIB

NetDummyRead (unsigned long *NetLibSocketPtr)

{
   static char  DummyBuffer [1024];
   static struct AnIOsb  DummyIOsb;
   static $DESCRIPTOR (DummyBufferDsc, DummyBuffer);

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "NetDummyRead() *NetLibSocketPtr: %d\n",
               *NetLibSocketPtr);

   status = netlib_read (NetLibSocketPtr, &DummyBufferDsc, 0, 0, 0, 0,
                         &DummyIOsb, &NetDummyReadAst, NetLibSocketPtr);

   if (Debug) fprintf (stdout, "netlib_read %%X%08.08X\n", status);
}
#endif

/*****************************************************************************/
/*
See commentary in NetDummyRead().
*/ 

#ifdef IP_UCX

NetDummyReadAst (unsigned short Channel)

{
   int  status;
   struct  AnIOsb  IOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetDummyReadAst() Channel: %d\n", Channel);

   status = sys$qiow (0, Channel, IO$_DEACCESS | IO$M_SHUTDOWN, IOsb, 0, 0,
                      0, 0, 0, UCX$C_DSC_ALL, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb %%X%08.08X\n",
               status, IOsb.Status);

   status = sys$dassgn (Channel);
   if (Debug) fprintf (stdout, "sys$dassgn() %%X%08.08X\n", status);
}

#endif

#ifdef IP_NETLIB

NetDummyReadAst (unsigned long *NetLibSocketPtr)

{
   int  status;
   struct  AnIOsb  IOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "NetDummyReadAst() *NetLibSocketPtr: %d\n",
               *NetLibSocketPtr);

   status = netlib_shutdown (NetLibSocketPtr, &NetLibShutdownBoth, &IOsb, 0, 0);
   if (Debug)
      fprintf (stdout, "netlib_shutdown %%X%08.08X IOsb %%X%08.08X\n",
               status, IOsb.Status);

   status = netlib_close (NetLibSocketPtr);
   if (Debug) fprintf (stdout, "netlib_close() %%X%08.08X\n", status);
}

#endif

/*****************************************************************************/
/*
Perform the UCX get host name using QIO ACP functionality.  This can execute 
at AST-delivery level.  Doing this within NETLIB needs to cater for the
lowest-common-denominator package, which sometimes will not execute at AST
delivery level, therefore rendering some NETLIB packages incapable of
supporting DNS lookup!
*/ 

#ifdef IP_UCX

NetHostLookup (struct ServiceStruct *svptr) 

{
   static unsigned char ControlSubFunction [4] =
      { INETACP_FUNC$C_GETHOSTBYADDR, INETACP$C_TRANS, 0, 0 };
   static struct dsc$descriptor AddressDsc =
      { 0, DSC$K_DTYPE_T, DSC$K_CLASS_S, 0};
   static struct dsc$descriptor ControlSubFunctionDsc =
      { 4, DSC$K_DTYPE_T, DSC$K_CLASS_S, &ControlSubFunction };
   static struct dsc$descriptor ClientHostNameDsc =
      { 0, DSC$K_DTYPE_T, DSC$K_CLASS_S, 0};

   int  status;
   unsigned short  Length;
   struct  AnIOsb  IOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetHostLookup()\n");

   AddressDsc.dsc$w_length = sizeof(svptr->ClientSocketName.sin_addr);
   AddressDsc.dsc$a_pointer = &svptr->ClientSocketName.sin_addr;

   ClientHostNameDsc.dsc$w_length = sizeof(svptr->ClientHostName)-1;
   ClientHostNameDsc.dsc$a_pointer = &svptr->ClientHostName;

   Length = 0;
   if (Debug) fprintf (stdout, "sys$qio() IO$_ACPCONTROL\n");
   status = sys$qiow (0, svptr->ServerChannel, IO$_ACPCONTROL,
                      &IOsb, 0, 0, &ControlSubFunctionDsc,
                      &AddressDsc, &Length, &ClientHostNameDsc, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qio() %%X%08.08X IOsb %%X%08.08X\n",
               status, IOsb.Status);
   if (Length)
      svptr->ClientHostName[Length] = '\0';
   else
      strcpy (svptr->ClientHostName, svptr->ClientInternetAddress);
}
#endif

#ifdef IP_NETLIB

NetHostLookup (struct ServiceStruct *svptr) 

{
   static struct dsc$descriptor ClientHostNameDsc =
      { 0, DSC$K_DTYPE_T, DSC$K_CLASS_S, 0};

   int  status;
   unsigned short  Length;
   struct AnIOsb  IOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetHostLookup()\n");

   ClientHostNameDsc.dsc$w_length = sizeof(svptr->ClientHostName)-1;
   ClientHostNameDsc.dsc$a_pointer = &svptr->ClientHostName;

   Length = 0;
   status = netlib_address_to_name (&svptr->ClientNetLibSocket, 0,
               &svptr->ClientNetLibName.sin_x_addr.inaddr_l_addr,
               &NetLibSizeofINADDRDEF, &ClientHostNameDsc, &Length,
               &IOsb, 0, 0);
   if (Debug)
      fprintf (stdout, "netlib_address_to_name() %%X%08.08X IOsb %%X%08.08X\n",
               status, IOsb.Status);
   if (Length)
      svptr->ClientHostName[Length] = '\0';
   else
      strcpy (svptr->ClientHostName, svptr->ClientInternetAddress);
}
#endif

/****************************************************************************/
/*
If an AST routine is supplied it is always called.  If not suppliedm the for
UCX the shutdown occurs asynchronously anyway with no AST routine called, for
NETLIB it will be done synchronously.
*/

int NetShutdownSocket
(
struct RequestStruct *rqptr,
void *AstFunctionPtr,
boolean ShutdownBoth
)
{
#ifdef IP_UCX

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "NetShutdownSocketUcx() %d %d\n", rqptr, ShutdownBoth);

   rqptr->ClientSocketShutdown = true;

   if (ShutdownBoth)
      status = sys$qio (0, rqptr->ClientChannel, IO$_DEACCESS | IO$M_SHUTDOWN,
                        &rqptr->NetShutdownIOsb, AstFunctionPtr, rqptr,
                        0, 0, 0, UCX$C_DSC_ALL, 0, 0);
   else
      status = sys$qio (0, rqptr->ClientChannel, IO$_DEACCESS | IO$M_SHUTDOWN,
                        &rqptr->NetShutdownIOsb, AstFunctionPtr, rqptr,
                        0, 0, 0, UCX$C_DSC_RCV, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qio() %%X%08.08X IOsb %%X%08.08X\n",
               status, rqptr->NetShutdownIOsb.Status);

   if (AstFunctionPtr)
   {
      if (VMSnok (status))
      {
         rqptr->NetShutdownIOsb.Status = status;
         SysDclAst (AstFunctionPtr, rqptr);
         return;
      }
   }

   return (status);
}
#endif

#ifdef IP_NETLIB

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "NetShutdownSocketNetLib() %d %d\n",
               rqptr, ShutdownBoth);

   rqptr->ClientSocketShutdown = true;

   if (ShutdownBoth)
      status = netlib_shutdown (&rqptr->ClientNetLibSocket,
                                &NetLibShutdownRecv,
                                &rqptr->NetShutdownIOsb, AstFunctionPtr, rqptr);
   else
      status = netlib_shutdown (&rqptr->ClientNetLibSocket, &NetLibShutdownBoth,
                                &rqptr->NetShutdownIOsb, AstFunctionPtr, rqptr);
   if (Debug)
      fprintf (stdout, "netlib_shutdown %%X%08.08X IOsb %%X%08.08X\n",
               status, rqptr->NetShutdownIOsb.Status);

   if (AstFunctionPtr)
   {
      if (VMSnok (status))
      {
         rqptr->NetShutdownIOsb.Status = status;
         SysDclAst (AstFunctionPtr, rqptr);
         return;
      }
   }

   return (status);
}

#endif

/****************************************************************************/
/*
*/

int NetCloseSocket (struct RequestStruct *rqptr)

{
#ifdef IP_UCX
   NetCloseSocketUcx (rqptr->ClientChannel);
#endif

#ifdef IP_NETLIB
   NetCloseSocketNetLib (&rqptr->ClientNetLibSocket);
#endif
}

#ifdef IP_UCX

#ifdef __DECC
#pragma inline(NetCloseSocketUcx)
#endif

int NetCloseSocketUcx (unsigned short Channel)

{
   int  status;
   struct  AnIOsb  IOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetCloseSocketUcx() %d\n", Channel);

   status = sys$qiow (0, Channel, IO$_DEACCESS, &IOsb, 0, 0,
                      0, 0, 0, 0, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb %%X%08.08X\n",
               status, IOsb.Status);

   status = sys$dassgn (Channel);
   if (Debug) fprintf (stdout, "sys$dassgn() %%X%08.08X\n", status);
   return (status);
}
#endif

#ifdef IP_NETLIB

#ifdef __DECC
#pragma inline(NetCloseSocketNetLib)
#endif

NetCloseSocketNetLib (unsigned long *NetLibSocketPtr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "NetCloseSocketnetLib() *NetLibSocketPtr: %d\n",
               *NetLibSocketPtr);

   if (*NetLibSocketPtr == NULL) return (SS$_NORMAL);

   status = netlib_close (NetLibSocketPtr);
   if (Debug) fprintf (stdout, "netlib_close() %%X%08.08X\n", status);

   return (status);
}

#endif

/****************************************************************************/
/*
Stop the server from receiving incoming requests.
*/

#ifdef IP_UCX

NetShutdownServerSocket ()

{
   register struct ServiceStruct  *svptr;

   int  status;
   struct  AnIOsb  IOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetShutdownServerSocket()\n");

   for (svptr = ServiceListHead; svptr != NULL; svptr = svptr->NextPtr)
   {
      status = sys$qiow (0, svptr->ServerChannel, IO$_DEACCESS, &IOsb, 0, 0,
                         0, 0, 0, 0, 0, 0);

      if (Debug)
         fprintf (stdout, "sys$qiow() %%X%08.08X IOsb %%X%08.08X\n",
                  status, IOsb.Status);
   }
}
#endif

#ifdef IP_NETLIB

NetShutdownServerSocket ()

{
   register struct ServiceStruct  *svptr;

   int  status;
   struct  AnIOsb  IOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetShutdownServerSocket()\n");

   for (svptr = ServiceListHead; svptr != NULL; svptr = svptr->NextPtr)
   {
      status = netlib_shutdown (&svptr->ServerNetLibSocket,
                                &NetLibShutdownRecv, &IOsb, 0, 0);

      if (Debug)
         fprintf (stdout, "netlib_shutdown %%X%08.08X IOsb %%X%08.08X\n",
                  status, IOsb.Status);
   }
}
#endif

/*****************************************************************************/
/*
Unlike NetWrite() this function writes directly using a client channel/socket,
not using the full request data structure. This is for writing to the client
before any of the request data structures are created (e.g. when the server
denies access). This is a blocking function, but shouldn't introduce much
granularity as it is only used for short messages before a client thread is
set up.
*/ 

#ifdef IP_UCX

int NetDirectWrite
(
unsigned short Channel,
char *DataPtr,
int DataLength
)
{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetDirectWrite()\n");

   status = sys$qiow (0, Channel, IO$_WRITEVBLK, 0, 0, 0,
                      DataPtr, DataLength, 0, 0, 0, 0);

   if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);

   return (status);
}
#endif

#ifdef IP_NETLIB

int NetDirectWrite
(
unsigned long *NetLibSocketPtr,
char *DataPtr,
int DataLength
)
{
   static $DESCRIPTOR (DataPtrDsc, "");

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetDirectWrite()\n");

   DataPtrDsc.dsc$a_pointer = DataPtr;
   DataPtrDsc.dsc$w_length = DataLength;

   status = netlib_write (NetLibSocketPtr, &DataPtrDsc, 0, 0, 0, 0, 0);

   if (Debug) fprintf (stdout, "netlib_write() %%X%08.08X\n", status);

   return (status);
}
#endif

/*****************************************************************************/
/*
Called from NetWrite() is a response header needs to be sent before any data.
Response header has now been sent, send the data using the buffered
information about it.
*/ 

NetResponseHeaderAst (struct RequestStruct *rqptr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetResponseHeaderAst()\n");

   NetWrite (rqptr,
             rqptr->ResponseHeaderAstAddress,
             rqptr->ResponseHeaderDataPtr,
             rqptr->ResponseHeaderDataLength);
}

/*****************************************************************************/
/*
Write 'DataLength' bytes located at 'DataPtr' to the client either using either
the "raw" network or via the Secure Sockets Layer.

If 'AstFunctionPtr' zero then use sys$qiow(), waiting for completion. If
an AST completion address is supplied then use sys$qio().  If empty data
buffer is supplied (zero length) then declare an AST to service any AST
routine supplied.  If none then just return.

For responses generated by the server the HTTP header is a separate structure
which can be sent separately. If the HTTP method is "HEAD" only allow bytes in
the header to be sent, absorb any other, explicitly calling the AST completion
routine as necessary (this, in particular, is for scripts that don't recognise
this method).

An "Xray" returns a full response (header and body) as a plain text document
(allows diagnostics, etc.) With an Xray just output a hard-wired, plain-text
response with wait for completion (won't be happening enough for any
significant impact!)

Explicitly declares any AST routine if an error occurs. The calling function
must not do any error recovery if an AST routine has been supplied but the
associated AST routine must!  If an AST was not supplied then the return
status can be checked.
*/ 

int NetWrite
(
struct RequestStruct *rqptr,
void *AstFunctionPtr,
char *DataPtr,
int DataLength
)
{
   static char  XrayHeader [] =
"HTTP/1.0 200 Xray\r\n\
Content-Type: text/plain\r\n\
Expires: Thu, 01-Jan-1970 00:00:01 GMT\r\n\
\r\n";

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "NetWrite() %d %d %d %d bytes\n",
               rqptr, AstFunctionPtr, DataPtr, DataLength);
   /** if (Debug) fprintf (stdout, "|%*s|\n", DataLength, DataPtr); **/

   status = SS$_NORMAL;

   if (!rqptr->ResponseHeaderSent)
   {
      /*******************************************/
      /* nothing of response sent to client yet! */
      /*******************************************/

      rqptr->ResponseHeaderSent = true;

      if (rqptr->Xray)
      {
         /*********************************/
         /* header for "Xray" of response */
         /*********************************/

         if (rqptr->SeSoLaPtr == NULL)
            status = NetWriteRaw (rqptr, 0, XrayHeader, sizeof(XrayHeader)-1);
         else
            status = SeSoLaWrite (rqptr, 0, XrayHeader, sizeof(XrayHeader)-1);

         if (VMSnok (status))
         {
            /* if resource wait enabled only quota not waited for is ASTLM */
            if (status == SS$_EXQUOTA)
               ErrorExitVmsStatus (status, "sys$qio()", FI_LI);
            else
            if (AstFunctionPtr)
            {
               /* write failed, call AST explicitly, status in the IOsb */
               ErrorVmsStatus (rqptr, status, FI_LI);
               rqptr->NetWriteIOsb.Status = status;
               rqptr->NetWriteIOsb.Count = 0;
               SysDclAst (AstFunctionPtr, rqptr);
            }
            return (status);
         }
      }

      if (rqptr->ResponseHeaderPtr != NULL)
      {
         /************************/
         /* send response header */
         /************************/

         if (Debug)
            fprintf (stdout, "flushing rqptr->ResponseHeaderPtr\n|%s|\n",
                     rqptr->ResponseHeaderPtr);

         /* indicate to "send only response header" routine it's been sent */
         if (rqptr->HttpMethod == HTTP_METHOD_HEAD)
            rqptr->ResponseHeaderNewlineCount = 2;

         if (!rqptr->ResponseHeaderLength)
            rqptr->ResponseHeaderLength = strlen(rqptr->ResponseHeaderPtr);

         if (AstFunctionPtr)
         {
            /************************/
            /* AST routine, no wait */
            /************************/

            if (DataLength)
            {
               rqptr->ResponseHeaderAstAddress = AstFunctionPtr;
               rqptr->ResponseHeaderDataPtr = DataPtr;
               rqptr->ResponseHeaderDataLength = DataLength;
               AstFunctionPtr = &NetResponseHeaderAst;
            }

            if (rqptr->SeSoLaPtr == NULL)
               status = NetWriteRaw (rqptr, AstFunctionPtr,
                                     rqptr->ResponseHeaderPtr,
                                     rqptr->ResponseHeaderLength);
            else
               status = SeSoLaWrite (rqptr, AstFunctionPtr,
                                     rqptr->ResponseHeaderPtr,
                                     rqptr->ResponseHeaderLength);

            if (VMSnok (status))
            {
               /* write failed, call AST explicitly, status in the IOsb */
               ErrorVmsStatus (rqptr, status, FI_LI);
               rqptr->NetWriteIOsb.Status = status;
               rqptr->NetWriteIOsb.Count = 0;
               SysDclAst (AstFunctionPtr, rqptr);
            }

            /* return, the AST function will sys$qiow() the data */
            return (status);
         }
         else
         {
            /************************/
            /* no AST routine, wait */
            /************************/

            if (rqptr->SeSoLaPtr == NULL)
               status = NetWriteRaw (rqptr, 0,
                                     rqptr->ResponseHeaderPtr,
                                     rqptr->ResponseHeaderLength);
            else
               status = SeSoLaWrite (rqptr, 0,
                                     rqptr->ResponseHeaderPtr,
                                     rqptr->ResponseHeaderLength);

            if (VMSnok (status))
            {
               /* if resource wait enabled only quota not waited for ASTLM */
               if (status == SS$_EXQUOTA)
                  ErrorExitVmsStatus (status, "sys$qio()", FI_LI);
               else
                  return (status);
            }
            /* continue on to sys$qiow() the data */
         }
      }
   }

   /********/
   /* data */
   /********/

   if (!DataLength)
   {
      /* without a real network write just fudge the status */
      rqptr->NetWriteIOsb.Status = SS$_NORMAL;
      rqptr->NetWriteIOsb.Count = 0;
      if (!AstFunctionPtr) return (SS$_NORMAL);
      SysDclAst (AstFunctionPtr, rqptr);
      return (status);
   }

   if (rqptr->HttpMethod == HTTP_METHOD_HEAD)
   {
      /*****************************/
      /* send only response header */
      /*****************************/

      if (Debug) fprintf (stdout, "method HEAD\n");
      if (rqptr->ResponseHeaderNewlineCount <= 1)
      {
         /* local storage */
         register int  cnt, nlcnt;
         register char  *cptr;

         nlcnt = rqptr->ResponseHeaderNewlineCount;
         if (Debug) fprintf (stdout, "nlcnt: %d\n", nlcnt);
         cptr = DataPtr;
         cnt = DataLength;
         while (cnt--)
         {
            if (*cptr == '\n' && ++nlcnt == 2)
            {
               /* two successive end-of-lines, therefore end of header */
               if (Debug) fprintf (stdout, "end-of-header\n");
               cptr++;
               break;
            }
            else
            if (*cptr != '\r' && *cptr != '\n')
               nlcnt = 0;
            cptr++;
         }
         if ((rqptr->ResponseHeaderNewlineCount = nlcnt) == 2)
         {
            if (Debug) fprintf (stdout, "DataLength: %d\n", cptr - DataPtr);
            if (DataLength = cptr - DataPtr)
            {
               /* adjust data length to include only the HTTP header */
               DataLength = cptr - DataPtr;
            }
            else
            {
               /* no data in HTTP header left at all */
               rqptr->NetWriteIOsb.Status = STS$K_SUCCESS;
               rqptr->NetWriteIOsb.Count = 0;
               if (AstFunctionPtr) SysDclAst (AstFunctionPtr, rqptr);
               return (SS$_NORMAL);
            }
         }
      }
      else
      {
         /* HTTP header has been completely sent, absorb anything else */
         rqptr->NetWriteIOsb.Status = STS$K_SUCCESS;
         rqptr->NetWriteIOsb.Count = 0;
         if (AstFunctionPtr) SysDclAst (AstFunctionPtr, rqptr);
         return (SS$_NORMAL);
      }
   }

   /*************/
   /* send data */
   /*************/

   if (rqptr->SeSoLaPtr == NULL)
      status = NetWriteRaw (rqptr, AstFunctionPtr, DataPtr, DataLength);
   else
      status = SeSoLaWrite (rqptr, AstFunctionPtr, DataPtr, DataLength);

   return (status);
}

/*****************************************************************************/
/*
Write data to the network. Explicitly declares any AST routine if an error
occurs. The calling function must not do any error recovery if an AST routine
has been supplied but the associated AST routine must! If an AST was not
supplied then the return status can be checked.
*/

int NetWriteRaw
(
struct RequestStruct *rqptr,
void *AstFunctionPtr,
char *DataPtr,
int DataLength
)

{
#ifdef IP_NETLIB
   static $DESCRIPTOR (DataPtrDsc, "");
#endif

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "NetWriteRaw() %d %d %d\n", rqptr, DataPtr, DataLength);
   /** fprintf (stdout, "|%*.*s|\n", DataLength, DataLength, DataPtr); **/

   if (AstFunctionPtr)
   {
#ifdef IP_UCX
      status = sys$qio (0, rqptr->ClientChannel,
                        IO$_WRITEVBLK, &rqptr->NetWriteIOsb,
                        AstFunctionPtr, rqptr,
                        DataPtr, DataLength, 0, 0, 0, 0);

      if (Debug) fprintf (stdout, "sys$qio() %%%08.08X\n", status);
#endif

#ifdef IP_NETLIB
      DataPtrDsc.dsc$a_pointer = DataPtr;
      DataPtrDsc.dsc$w_length = DataLength;

      status = netlib_write (&rqptr->ClientNetLibSocket, &DataPtrDsc, 0, 0,
                             &rqptr->NetWriteIOsb,
                             AstFunctionPtr, rqptr);

      if (Debug) fprintf (stdout, "netlib_write() %%X%08.08X\n", status);
#endif

      if (VMSnok (status))
      {
         /* write failed, call AST explicitly, status in the IOsb */
         rqptr->NetWriteIOsb.Status = status;
         rqptr->NetWriteIOsb.Count = 0;
         SysDclAst (AstFunctionPtr, rqptr);
      }
   }
   else
   {
#ifdef IP_UCX
      status = sys$qiow (0, rqptr->ClientChannel,
                         IO$_WRITEVBLK, &rqptr->NetWriteIOsb, 0, 0,
                         DataPtr, DataLength, 0, 0, 0, 0);

      if (Debug) fprintf (stdout, "sys$qio() %%%08.08X\n", status);
#endif

#ifdef IP_NETLIB
      DataPtrDsc.dsc$a_pointer = DataPtr;
      DataPtrDsc.dsc$w_length = DataLength;

      status = netlib_write (&rqptr->ClientNetLibSocket, &DataPtrDsc, 0, 0,
                             &rqptr->NetWriteIOsb, 0, 0);

      if (Debug) fprintf (stdout, "netlib_write() %%X%08.08X\n", status);

      /* if resource wait enabled the only quota not waited for is ASTLM */
      if (status == SS$_EXQUOTA)
         ErrorExitVmsStatus (status, "sys$qio()", FI_LI);
#endif
   }

   if (VMSok (status)) rqptr->BytesTx += DataLength;

   return (status);
}

/*****************************************************************************/
/*
Wrapper for NetReadRaw().
*/ 

int NetRead
(
struct RequestStruct *rqptr,
void *AstFunctionPtr,
char *DataPtr,
int DataSize
)
{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetRead()\n");

   if (rqptr->SeSoLaPtr == NULL)
      status = NetReadRaw (rqptr, AstFunctionPtr, DataPtr, DataSize);
   else
      status = SeSoLaRead (rqptr, AstFunctionPtr, DataPtr, DataSize);
   if (Debug) fprintf (stdout, "status: %%X%08.08X\n", status);
   return (status);
}

/*****************************************************************************/
/*
Queue up a read from the client over the network. If 'AstFunctionPtr' 
is zero then no I/O completion AST routine is called.  If it is non-zero then 
the function pointed to by the parameter is called when the network write 
completes.

Explicitly declares any AST routine if an error occurs. The calling function
must not do any error recovery if an AST routine has been supplied but the
associated AST routine must!  If an AST was not supplied then the return
status can be checked.
*/ 

int NetReadRaw
(
struct RequestStruct *rqptr,
void *AstFunctionPtr,
char *DataPtr,
int DataSize
)
{
#ifdef IP_NETLIB
   static $DESCRIPTOR (DataPtrDsc, "");
#endif

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "NetReadRaw() %d %d %d\n", rqptr, DataPtr, DataSize);

   if (AstFunctionPtr)
   {
#ifdef IP_UCX
      status = sys$qio (0, rqptr->ClientChannel,
                        IO$_READVBLK, &rqptr->NetReadIOsb,
                        AstFunctionPtr, rqptr,
                        DataPtr, DataSize, 0, 0, 0, 0);

      if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
#endif

#ifdef IP_NETLIB
      DataPtrDsc.dsc$a_pointer = DataPtr;
      DataPtrDsc.dsc$w_length = DataSize;

      status = netlib_read (&rqptr->ClientNetLibSocket, &DataPtrDsc, 0, 0, 0, 0,
                            &rqptr->NetReadIOsb, AstFunctionPtr, rqptr);

      if (Debug) fprintf (stdout, "netlib_read() %%X%08.08X\n", status);
#endif

      /* with resource wait enabled the only quota not waited for is ASTLM */
      if (status == SS$_EXQUOTA)
         ErrorExitVmsStatus (status, "sys$qio()", FI_LI);

      if (VMSok (status)) return (status);

      /* queuing of read failed, call AST explicitly, status in the IOsb */
      ErrorVmsStatus (rqptr, status, FI_LI);
      rqptr->NetReadIOsb.Status = status;
      rqptr->NetReadIOsb.Count = 0;
      SysDclAst (AstFunctionPtr, rqptr);
      return (status);
   }
   else
   {
#ifdef IP_UCX
      status = sys$qiow (0, rqptr->
ClientChannel,
                         IO$_READVBLK, &rqptr->NetReadIOsb, 0, 0,
                         DataPtr, DataSize, 0, 0, 0, 0);

      if (Debug) fprintf (stdout, "sys$qio() %%X%08.08X\n", status);
#endif

#ifdef IP_NETLIB
      DataPtrDsc.dsc$a_pointer = DataPtr;
      DataPtrDsc.dsc$w_length = DataSize;

      status = netlib_read (&rqptr->ClientNetLibSocket, &DataPtrDsc, 0, 0, 0, 0,
                            &rqptr->NetReadIOsb, 0, 0);

      if (Debug) fprintf (stdout, "netlib_read() %%X%08.08X\n", status);
#endif

      return (status);
   }
}

/*****************************************************************************/
/*
This function buffers output without actually sending it to the client until
ready, either when the first buffer fills or when all output is completely
buffered. It will store any amount of output in a linked list of separately
allocated buffers. This is useful when a function that must not be interrupted
can rapidly store all required output without being slowed by actually
transfering it on the network, then flush it all asynchronously (e.g. the
server administration reports, for instance CacheReport(), which are
blocking).

Providing an AST parameter and a data parameter implies that after the first
buffer fills (and usually overflows into a second) the current buffered output
should be written to the client.

Providing an AST parameter and setting the data parameter to NULL indicates
all the currently buffered contents should be written to the client.

Providing all parameters as NULL and zero, (except 'rqptr') as appropriate,
results in the data buffer initialized (if it didn't exist) or reset (if it
did).
*/ 

int NetWriteBuffered
(
struct RequestStruct *rqptr,
void *AstFunctionPtr,
char *DataPtr,
int DataLength
)
{
   int  status;
   char  *BufferPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "NetWriteBuffered() %d %d %d %d\n",
               rqptr, AstFunctionPtr, DataPtr, DataLength);

   /* if all parameters essentially empty then reset the buffers */
   if (AstFunctionPtr == NULL && DataPtr == NULL && DataLength == 0)
   {
      NetWriteBufferedInit (rqptr, true);
      return;
   }

   /* first call, initialize a buffer */
   if (rqptr->OutputBufferStructPtr == NULL)
      NetWriteBufferedInit (rqptr, true);

   /**********************/
   /* buffer this output */
   /**********************/

   if (rqptr->NetWriteBufferedEscapeHtml)
   {
      /* escape HTML-forbidden characters */
      NetWriteBufferedEscapeHtml (rqptr, DataPtr, DataLength);
   }
   else
   {
      while (DataLength)
      {
         if (Debug)
            fprintf (stdout, "%d %d\n",
                     rqptr->OutputBufferRemaining, DataLength);

         if (DataLength <= rqptr->OutputBufferRemaining)
         {
            /* enough space in this buffer */
            memcpy (rqptr->OutputBufferCurrentPtr, DataPtr, DataLength);
            rqptr->OutputBufferCount += DataLength;
            rqptr->OutputBufferCurrentPtr += DataLength;
            rqptr->OutputBufferRemaining -= DataLength;
            DataLength = 0;
         }
         else
         {
            /* fill up any that's left */
            memcpy (rqptr->OutputBufferCurrentPtr,
                    DataPtr,
                    rqptr->OutputBufferRemaining);
            DataPtr += rqptr->OutputBufferRemaining;
            DataLength -= rqptr->OutputBufferRemaining;
            rqptr->OutputBufferCount += rqptr->OutputBufferRemaining;
            /* need another buffer */
            NetWriteBufferedInit (rqptr, false);
         }
      }
   }

   /* if an AST routine supplied then we can think about buffer flushing */
   if (AstFunctionPtr != NULL)
   {
      /* if the first buffer is full or if the flush is being forced */
      if (rqptr->OutputBufferCount > OutputBufferSize || DataPtr == NULL)
      {
         /***********************/
         /* flush the buffer(s) */
         /***********************/

         /* force all buffers to be flushed if 'DataPtr' is NULL */
         if (DataPtr == NULL) rqptr->OutputBufferFlush = true;
         /* buffer the AST function pointer */
         rqptr->OutputBufferAstFunctionPtr = AstFunctionPtr;
         /* get the "write now" pointer to the first buffer in the list */
         rqptr->OutputBufferNowStructPtr =
            LIST_HEAD (&rqptr->OutputBufferList);
         /* fudge this status for NetWriteBufferedNow() */
         rqptr->NetWriteIOsb.Status = SS$_NORMAL;
         NetWriteBufferedNow (rqptr);
         return;
      }

      /************************/
      /* just declare the AST */
      /************************/

      SysDclAst (AstFunctionPtr, rqptr);
   }
}

/*****************************************************************************/
/*
Copy the data into the buffer escaping HTML-forbidden characters.
*/ 

int NetWriteBufferedEscapeHtml
(
struct RequestStruct *rqptr,
char *DataPtr,
int DataLength
)
{
   register int  dcnt, bcnt, brcnt;
   register char  *bptr, *dptr, *eptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "NetWriteBufferedEscapeHtml() %d %d\n",
               DataPtr, DataLength);

   dptr = DataPtr;
   dcnt = DataLength;
   bptr = rqptr->OutputBufferCurrentPtr;
   brcnt = rqptr->OutputBufferRemaining;
   bcnt = rqptr->OutputBufferCount;
   eptr = "";

   while (dcnt)
   {
      /* if there is buffer space left then process this character */
      if (brcnt)
      {
         switch (*dptr)
         {
            case '<' :
               eptr = "&lt;"; dptr++; dcnt--; break;
            case '>' :
               eptr = "&gt;"; dptr++; dcnt--; break;
            case '&' :
               eptr = "&amp;"; dptr++; dcnt--; break;
            default :
               eptr = ""; *bptr++ = *dptr++; dcnt--; brcnt--; bcnt++;
         }

         /* buffer any escape sequence from this character */
         while (*eptr && brcnt)
         {
            *bptr++ = *eptr++;
            brcnt--;
            bcnt++;
         }
      }

      if (brcnt) continue;

      /* need another buffer */
      NetWriteBufferedInit (rqptr, false);
      bptr = rqptr->OutputBufferCurrentPtr;
      brcnt = rqptr->OutputBufferRemaining;

      /* buffer any escape sequence remaining from the previous character */
      while (*eptr && brcnt)
      {
         *bptr++ = *eptr++;
         brcnt--;
         bcnt++;
      }
   }

   rqptr->OutputBufferCurrentPtr = bptr;
   rqptr->OutputBufferRemaining = brcnt;
   rqptr->OutputBufferCount = bcnt;
}

/*****************************************************************************/
/*
This function processes the linked list of output buffers, writing them to the
client, and operates in two distinct modes. First, non-flush mode. In this
mode any full output buffers are written to the client. Any last,
partially-filled buffer is not, instead it is moved to the head of the list to
essentially become the first data in the output buffer to which more can then
be added. In this way when a buffer fills and overflows into a second buffer
the first is written but the second, scantily filled buffer is not. The second
mode is full-flush, where all buffers are written to the client. This is
forced by NetWriteBuffered() whenever an AST routine is supplied but no data
('DataPtr' is NULL).
*/ 

int NetWriteBufferedNow (struct RequestStruct *rqptr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "NetWriteBufferedNow() %d %d %d %d %d\n",
               rqptr, rqptr->OutputBufferStructPtr,
               rqptr->OutputBufferRemaining,
               rqptr->OutputBufferCount,
               rqptr->OutputBufferFlush);

   if (VMSnok (rqptr->NetWriteIOsb.Status))
   {
      /* network write has failed (as AST), bail out now */
      if (Debug)
         fprintf (stdout, "NetWriteIOsb.Status %%X%08.08X\n",
                  rqptr->NetWriteIOsb.Status);
      SysDclAst (rqptr->OutputBufferAstFunctionPtr, rqptr);
      return;
   }

   if ((rqptr->OutputBufferFlush &&
        rqptr->OutputBufferCount > OutputBufferSize) ||
       rqptr->OutputBufferCount > OutputBufferSize + OutputBufferSize)
   {
      /*
         If flushing all buffers and at least one full buffer in list, or
         if two or more full buffers in list.
      */
      NetWrite (rqptr, &NetWriteBufferedNow,
                LIST_DATA (rqptr->OutputBufferNowStructPtr),
                OutputBufferSize);
      rqptr->OutputBufferCount -= OutputBufferSize;
      /* set the "write now" buffer pointer to the next buffer */
      rqptr->OutputBufferNowStructPtr =
         LIST_NEXT (rqptr->OutputBufferNowStructPtr);
      return;
   }
   else
   if (rqptr->OutputBufferCount > OutputBufferSize)
   {
      /* if one full buffer, then one partially filled, NOTE the AST! */
      NetWrite (rqptr, rqptr->OutputBufferAstFunctionPtr,
                LIST_DATA (rqptr->OutputBufferNowStructPtr),
                OutputBufferSize);
      rqptr->OutputBufferCount -= OutputBufferSize;
      /*
         Move currently-being-written-to buffer to the head of the list,
         effectively to the front of the buffer.
      */
      ListMoveHead (&rqptr->OutputBufferList, rqptr->OutputBufferStructPtr);
      return;
   }
   else
   {
      /* if last buffer in linked list, write whatever remains in it */
      NetWrite (rqptr, rqptr->OutputBufferAstFunctionPtr,
                LIST_DATA (rqptr->OutputBufferNowStructPtr),
                rqptr->OutputBufferCount);
      /* reset the buffer(s) */
      NetWriteBufferedInit (rqptr, true);
      /* no more buffers therefore flush (if set) is complete */
      rqptr->OutputBufferFlush = false;
      return;
   }
}

/*****************************************************************************/
/*
Get another output buffer.
*/ 

int NetWriteBufferedInit
(
struct RequestStruct *rqptr,
boolean ResetBuffers
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "++++++++++++++++++++++++++++++++++\n");
   if (Debug) fprintf (stdout, "NetWriteBufferedInit()\n");

   if (ResetBuffers && !LIST_EMPTY (&rqptr->OutputBufferList))
   {
      /* reset a non-empty list to the first buffer */
      if (Debug) fprintf (stdout, "reset non-empty\n");
      rqptr->OutputBufferStructPtr = LIST_HEAD (&rqptr->OutputBufferList);
      rqptr->OutputBufferPtr = LIST_DATA (rqptr->OutputBufferStructPtr);
      rqptr->OutputBufferCurrentPtr = rqptr->OutputBufferPtr;
      rqptr->OutputBufferRemaining = OutputBufferSize;
      rqptr->OutputBufferCount = 0;
      return;
   }

   if (rqptr->OutputBufferStructPtr != NULL &&
       LIST_HAS_NEXT (rqptr->OutputBufferStructPtr))
   {
      /* reuse a previously allocated buffer from the list */
      if (Debug) fprintf (stdout, "reuse\n");
      rqptr->OutputBufferStructPtr =
         LIST_NEXT (rqptr->OutputBufferStructPtr);
      rqptr->OutputBufferPtr = LIST_DATA (rqptr->OutputBufferStructPtr);
      rqptr->OutputBufferCurrentPtr = rqptr->OutputBufferPtr;
      rqptr->OutputBufferRemaining = OutputBufferSize;
      return;
   }

   /* allocate the first or another buffer to the list */
   if (Debug) fprintf (stdout, "new\n");
   rqptr->OutputBufferStructPtr =
      VmGetHeap (rqptr, sizeof(struct ListEntryStruct) + OutputBufferSize);
   ListAddTail (&rqptr->OutputBufferList, rqptr->OutputBufferStructPtr);
   rqptr->OutputBufferPtr = LIST_DATA (rqptr->OutputBufferStructPtr);
   rqptr->OutputBufferCurrentPtr = rqptr->OutputBufferPtr;
   rqptr->OutputBufferRemaining = OutputBufferSize;
}

/*****************************************************************************/
/*
Report that client host name has been denied access to the server.
*/
 
#ifdef IP_UCX
   NetAccessDenied
(
unsigned short Channel,
char *HostName,
char *InternetAddress
)
#endif
#ifdef IP_NETLIB
   NetAccessDenied
(
unsigned long *NetLibSocketPtr,
char *HostName,
char *InternetAddress
)
#endif
 
{
   static $DESCRIPTOR (MessageFaoDsc,
"HTTP/1.0 403 Forbidden\r\n\
Content-Type: text/html\r\n\
\r\n\
<HTML>\n\
<HEAD>\n\
<TITLE>!AZ 403</TITLE>\n\
</HEAD>\n\
<BODY>\n\
!AZ\n\
</BODY>\n\
</HTML>\n");

   short  Length;
   char  Buffer [256];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   sys$fao (&MessageFaoDsc, &Length, &BufferDsc,
            MsgForNoRequest(HostName,InternetAddress,MSG_STATUS_ERROR),
            MsgForNoRequest(HostName,InternetAddress,MSG_GENERAL_ACCESS_DENIED));
   Buffer[Length] = '\0';

#ifdef IP_UCX
   NetDirectWrite (Channel, Buffer, Length);
#endif
#ifdef IP_NETLIB
   NetDirectWrite (NetLibSocketPtr, Buffer, Length);
#endif
}

/*****************************************************************************/
/*
Report that there is already the limit of client connections.
*/
 
#ifdef IP_UCX
   NetTooBusy
(
unsigned short Channel,
char *HostName,
char *InternetAddress
)
#endif
#ifdef IP_NETLIB
   NetTooBusy
(
unsigned long *NetLibSocketPtr,
char *HostName,
char *InternetAddress
)
#endif
 
{
   static $DESCRIPTOR (MessageFaoDsc,
"HTTP/1.0 502 Busy\r\n\
Content-Type: text/html\r\n\
\r\n\
<HTML>\n\
<HEAD>\n\
<TITLE>!AZ 502</TITLE>\n\
</HEAD>\n\
<BODY>\n\
!AZ\n\
</BODY>\n\
</HTML>\n");

   short  Length;
   char  Buffer [256];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   sys$fao (&MessageFaoDsc, &Length, &BufferDsc,
            MsgForNoRequest(HostName,InternetAddress,MSG_STATUS_ERROR),
            MsgForNoRequest(HostName,InternetAddress,MSG_GENERAL_TOO_BUSY));
   Buffer[Length] = '\0';

#ifdef IP_UCX
   NetDirectWrite (Channel, Buffer, Length);
#endif
#ifdef IP_NETLIB
   NetDirectWrite (NetLibSocketPtr, Buffer, Length);
#endif
}

/****************************************************************************/

