/*****************************************************************************/
/*
                                  Put.c

PUT, POST or DELETE a document.  Although the HTTP/1.1 specification
differentiates between the PUT and POST functionality this module does not.
Allows documents (files) and directories (i.e. specification ending in a slash)
to be created.  Either the DELETE method or a kludge allows these to be
deleted.  The kludge: if a wildcard version (";*") is included with the
specification the respective file or directory is deleted with either of the
non-DELETE methods.

Reads the entire content body from the client into dynamically allocated
memory, hence there may be a maximum size required/desired on file uploads,
etc.  This does make it much easier to parse "multipart/form-data" and possibly
a little more efficient in file writes (because of potentially bigger chunks
written at the one time).

Access Control
--------------

Relies on HTTPd authentication.  If a remote username has not been verified an
automatic no-privilege error is generated.

Access to create or delete documents is also controlled by any
permissions/protections/ACLs, etc. on the parent directory controlling access
by the HTTPd server account (usually HTTP$SERVER).  The explicit action
required to grant the server account access to a directory it needs to write
into is a bit of a nuisance, but deemed a good double check, preventing access
due to flawed authentication/authorization configuration (and hopefully even
a range of possible design problems or coding errors within this server :^)

Setting ACLs is preferable to granting world write access (obviously):

This ACE would grant PUT, POST or DELETE access to the server:

   $ SET ACL directory.DIR /ACL=(IDENT=HTTP$SERVER,ACCESS=READ+WRITE)

This ACE would explcitly deny POST access to the server:

   $ SET ACL directory.DIR /ACL=(IDENT=HTTP$SERVER,ACCESS=READ)

This ACE would explcitly deny ALL access to the server:

   $ SET ACL directory.DIR /ACL=(IDENT=HTTP$SERVER,ACCESS=NONE)

BE ULTRA-CAUTIOUS ... check access to directory before undertaking any action
that will alter anything within it, even if doing that potentially involves
some redundant processesing/checking!  In this way any programming oversights
will hopefully be ameliorated.

File Record Format
------------------

Text files are created with a STREAM_LF format and CR implied carriage-control. 
Binary files are created in undefined (UDF) record format with no record
attributes (carriage-control).  File writing is done using block I/O for
efficiency and record independence.


Content-Type: application/x-www-form-urlencoded
-----------------------------------------------

If the document 'Content-Type:' is "application/x-www-form-urlencoded" (i.e.
generated from an HTML form) all field names and delimiting symbols are
eliminated and the field(s) content only converted into plain text.  Hence a
text document can be POSTed using an HTML form with only the content ending up
in the file.

A field named "hexencoded" is expected to contain hexadecimal encoded bytes
(i.e. two hexadecimal digits per byte), as well as some url-encoded characters
(e.g. newlines, 0x0a).  The contents of this field are decoded before
inclusion in the file.  This was done to allow non-text files to be copied
around.

Content-Type: multipart/form-data
---------------------------------

This module can process a request body according to RFC-1867, "Form-based File
Upload in HTML".  As yet it is not a full implementation.  It will not process
"multipart/mixed" subsections.  The 'Content-Type:' is "multipart/form-data".
The implementation herein is very basic, a facility only to allow uploads of
files into the server administered file system.  The ACTION= tag of the form
must specify the directory (URL format) in which the uploaded file will be
created.

Sorry, but the MIME processing is so rudimentary and utilitarian as to be
embarressing, not so much so that I can't use it however :^)

The multi-threaded nature of the server is maintained by making file upload
writes AST-driven, and multiple multipart/form-data actions within the one
request driven by ASTs.

Multipart/form-data Fields
--------------------------

All file names supplied within a multipart/form-data request are relative to
the path information supplied in the header request.  Hence it must supply
a directory specification.  It is an error condition if this directory is not
specified, and all file specifications are generated by appending file names
to this directory.

In addition to file uploads, specific field names within a multipart/form-data
request are detected and used as additional information or to initiate actions.
These field names are case insensistive, as is any content within them.

When uploading files, the field "UploadFileName" contents, when received
before the file contents themselves (i.e. "text" field occurs in the form
before the "file" field) provides an alternative name to the original file.

The field named "action" allows the following actions to be initiated:

  o  "create", creates a new directory using the name field in 'PutFileName'
  o  "delete", deletes the file or directory named in field 'PutFileName'

Any other value present in the action field is reported as an error.

All required file names must be specified "file.type" prior to the action
being initiated (i.e. name fields must be present in the form before the
action field).  The following (hopefully self-documenting) field names can be
used to pass file names for action:

  o  "FileName"
  o  "UploadFileName"
  o  "DirectoryName"
  o  "Protection"
  o  "PreviewOnly"
  o  "PreviewNote"

Any other field names present in the request are reported as errors.

Other Considerations
--------------------

PUT/POSTed files automatically have a three version limit imposed.

If an error occurs (message generated) a file created by the request is
deleted.


VERSION HISTORY
---------------
12-MAR-98  MGD  file protection may now be specified (as hexadecimal value)
25-SEP-97  MGD  bugfix; PutProcessText() removed CRLF munging
17-AUG-97  MGD  message database,
                SYSUAF-authenticated users security-profile
27-MAR-97  MGD  provide file edit preview (see UPD.C)
01-FEB-97  MGD  HTTPd version 4
01-SEP-96  MGD  provide "Content-Type: multipart/form-data" for file upload
06-APR-96  MGD  initial development
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* VMS related header files */
#include <acldef.h>
#include <armdef.h>
#include <atrdef.h>
#include <chpdef.h>
#include <descrip.h>
#include <fibdef.h>
#include <iodef.h>
#include <jpidef.h>
#include <prvdef.h>
#include <rms.h>
#include <rmsdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application header files */
#include "wasd.h"
#include "auth.h"
#include "httpd.h"
#include "error.h"
#include "mapurl.h"
#include "msg.h"
#include "net.h"
#include "put.h"
#include "support.h"
#include "vm.h"

/***************/
/* definitions */
/***************/

#define PUT_FILE 0x01
#define PUT_DIRECTORY 0x02
#define PUT_UPLOAD 0x04
#define PUT_CREATED 0x10
#define PUT_DELETED 0x20
#define PUT_SUPERCEDED 0x40

/******************/
/* global storage */
/******************/

boolean  PutOnlyTextFilesStreamLf = true;

char  ErrorMultipartMixed [] =
"\"Multipart/mixed\" content is not currently supported.\
<P><I>Try reducing to a single file upload \
and/or the form to a single input field.</I>";

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern int  OutputBufferSize;
extern char  HtmlSgmlDoctype[];
extern char  SoftwareID[];
extern char  ErrorSanityCheck[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/*****************************************************************************/
/*
The file is initially created with undefined record attributes.  This allows
sys$put()ing to the file without having any record delimiters, etc. added by
RMS (i.e. we can write a binary stream with impunity :^)  The record attributes
are changed to stream-LF by PutWriteFileVersionLimit() in the file header after
it is closed in PutEnd().
*/ 
 
PutBegin
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   register char  *cptr;
   register struct PutTaskStruct  *tkptr;

   int  status,
        ReadSize;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "PutBegin() %d |%s|\n",
               rqptr->ContentLength, rqptr->RequestParseNam.nam$l_dev);

   if (!rqptr->AccountingDone)
      rqptr->AccountingDone = ++Accounting.DoPutCount;

   if (rqptr->ErrorMessagePtr != NULL)
   {
      /* previous error, cause threaded processing to unravel */
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /* authentication is mandatory for a PUT, DELETE or POST */
   if (!rqptr->RemoteUser[0] ||
       !((rqptr->AuthRequestCan & HTTP_METHOD_PUT) ||
         (rqptr->AuthRequestCan & HTTP_METHOD_POST) ||
         (rqptr->AuthRequestCan & HTTP_METHOD_DELETE)))
   {
      rqptr->ErrorTextPtr = rqptr->PathInfoPtr;
      ErrorVmsStatus (rqptr, SS$_NOPRIV, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /* set up the task structure (only ever one per request!) */
   rqptr->PutTaskPtr = tkptr = (struct PutTaskStruct*)
      VmGetHeap (rqptr, sizeof(struct PutTaskStruct));
   tkptr->NextTaskFunction = NextTaskFunction;

   tkptr->ProtectionMask = PUT_DEFAULT_FILE_PROTECTION;

   /**********************************************/
   /* delete (or file/directory deletion kludge) */
   /**********************************************/

   if ((rqptr->HttpMethod == HTTP_METHOD_DELETE) ||
       (rqptr->RequestParseNam.nam$l_ver[0] == ';' &&
        rqptr->RequestParseNam.nam$l_ver[1] == '*'))
   {
      /* terminate on either the directory or file name */
      if (!rqptr->RequestParseNam.nam$b_name &&
          rqptr->RequestParseNam.nam$b_type == 1)
         *rqptr->RequestParseNam.nam$l_name = '\0';
      else
         *rqptr->RequestParseNam.nam$l_ver = '\0';

      PutDelete (rqptr, rqptr->RequestParseNam.nam$l_dev);

      PutEnd (rqptr);
      return;
   }

   if (!rqptr->RequestParseNam.nam$b_name &&
       rqptr->RequestParseNam.nam$b_type == 1)
   {
      /********************/
      /* create directory */
      /********************/

      /* multipart/form-data supplies the parent directory as the path */
      if (!strsame (rqptr->HttpContentTypePtr, "multipart/form-data;", 20))
      {
         /* terminate immediately after the closing ']' */
         *rqptr->RequestParseNam.nam$l_name = '\0';

         /* there SHOULD be a file protection field */
         if (rqptr->ContentCount > 11)
            if (rqptr->ContentBufferPtr, "protection=", 11)
               if (isxdigit(rqptr->ContentBufferPtr[11]))
                  tkptr->ProtectionMask =
                     strtol (rqptr->ContentBufferPtr+11, NULL, 16);

         PutCreateDirectory (rqptr, rqptr->RequestParseNam.nam$l_dev);
         PutEnd (rqptr);
         return;
      }
   }

   /* terminate on version for file name, name for directory */
   if (!rqptr->RequestParseNam.nam$b_name &&
       rqptr->RequestParseNam.nam$b_type == 1 &&
       rqptr->RequestParseNam.nam$b_ver == 1)
      *rqptr->RequestParseNam.nam$l_name = '\0';
   else
      *rqptr->RequestParseNam.nam$l_ver = '\0';

   /* if not supplied in header set content type from file extension */
   if (rqptr->HttpContentTypePtr == NULL)
      rqptr->HttpContentTypePtr =
         ConfigContentType (NULL, rqptr->RequestParseNam.nam$l_type);
   if (Debug) fprintf (stdout, "|%s|\n", rqptr->HttpContentTypePtr);

   if (strsame (rqptr->HttpContentTypePtr, "text/", 5))
      PutProcessText (rqptr);
   else
   if (strsame (rqptr->HttpContentTypePtr,
                "application/x-www-form-urlencoded", -1))
      PutProcessUrlEncoded (rqptr);
   else
   if (strsame (rqptr->HttpContentTypePtr, "multipart/form-data;", 20))
      PutProcessMultipartFormData (rqptr);
   else
   {
      tkptr->FileNamePtr = rqptr->RequestFileName;
      tkptr->FileContentTypePtr = rqptr->HttpContentTypePtr;
      tkptr->FileContentPtr = rqptr->ContentBufferPtr;
      tkptr->FileContentLength = rqptr->ContentCount;

      PutWriteFile (rqptr);
   }
}

/*****************************************************************************/
/*
Conclude processing the request.  If a temporary file name/path was generated
then it's a preview only, turn the POST into a GET of the temporary file path.
*/

PutEnd (struct RequestStruct *rqptr)

{
   register struct PutTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PutEnd()\n");

   tkptr = rqptr->PutTaskPtr;
   SysDclAst (tkptr->NextTaskFunction, rqptr);
}

/*****************************************************************************/
/*
Process the body (now completely read into the buffer) according to its
content-type.
*/

PutProcessContent (struct RequestStruct *rqptr)

{
   register struct PutTaskStruct  *tkptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PutProcessContent()\n");

   tkptr = rqptr->PutTaskPtr;

   /* if not supplied in header set content type from file extension */
   if (rqptr->HttpContentTypePtr == NULL)
      rqptr->HttpContentTypePtr =
         ConfigContentType (NULL, rqptr->RequestParseNam.nam$l_type);
   if (Debug) fprintf (stdout, "|%s|\n", rqptr->HttpContentTypePtr);

   if (strsame (rqptr->HttpContentTypePtr, "text/", 5))
      PutProcessText (rqptr);
   else
   if (strsame (rqptr->HttpContentTypePtr,
                "application/x-www-form-urlencoded", -1))
      PutProcessUrlEncoded (rqptr);
   else
   if (strsame (rqptr->HttpContentTypePtr, "multipart/form-data;", 20))
      PutProcessMultipartFormData (rqptr);
   else
   {
      tkptr->FileNamePtr = rqptr->RequestFileName;
      tkptr->FileContentTypePtr = rqptr->HttpContentTypePtr;
      tkptr->FileContentPtr = rqptr->ContentBufferPtr;
      tkptr->FileContentLength = rqptr->ContentCount;

      PutWriteFile (rqptr);
   }
}

/*****************************************************************************/
/*
As this is supposed to be a text file.
*/

PutProcessText (struct RequestStruct *rqptr)

{
   register struct PutTaskStruct  *tkptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PutProcessText()\n");

   tkptr = rqptr->PutTaskPtr;

   tkptr->FileNamePtr = rqptr->RequestFileName;
   tkptr->FileContentTypePtr = rqptr->HttpContentTypePtr;
   tkptr->FileContentPtr = rqptr->ContentBufferPtr;
   tkptr->FileContentLength = rqptr->ContentCount;

   PutWriteFile (rqptr);
}

/*****************************************************************************/
/*
Decode URL-encoded byte stream.  Eliminates form field names, field equal
symbols and field-separating ampersands.  Converts '+' characters in field
content into spaces and hexadecimal, URL-encoded characters into the respective
character.  Requires no additional buffer space because all conversions require
less-than or equal to from whatever is being converted.

A field named "hexencoded" is expected to contain hexadecimal-encoded bytes,
and has these two-digit numbers converted back into bytes before inclusion in
the file.

The presence of a non-null field named "previewonly" causes a temporary,
delete-on-close file name to be generated (for previewing the POSTed file :^). 
A field "previewnote" contains text placed at the very beginning of a previewed
file. This can either be left in place or eliminated by pointer manipulation
depending on the value of "previewonly".  Also see UPD.C module.
*/ 

PutProcessUrlEncoded (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (TempFileNameFaoDsc, "!AZ!AZ!AZ\0");
   static $DESCRIPTOR (TempNameFaoDsc, "-!4ZL!2ZL!2ZL!2ZL!2ZL!2ZL!2ZL-\0");
   static $DESCRIPTOR (StringDsc, "");

   register boolean  InsideFormName;
   register int  cnt;
   register char  c;   
   register char  *cptr, *sptr, *zptr;
   register struct PutTaskStruct  *tkptr;

   boolean  PreviewOnly;
   int  status;
   char  *PreviewNoteBegin,
         *PreviewNoteEnd;
   char  TempName [32];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "PutProcessUrlEncoded() %d bytes\n",
               rqptr->ContentCount);

   tkptr = rqptr->PutTaskPtr;

   cptr = sptr = rqptr->ContentBufferPtr;
   cnt = rqptr->ContentCount;
   InsideFormName = true;
   PreviewOnly = false;
   PreviewNoteBegin = PreviewNoteEnd = NULL;

   while (cnt)
   {
      while (!InsideFormName && cnt)
      {
         if (*cptr != '+' && *cptr != '%' && *cptr != '&' && *cptr != '\r')
         {
            *sptr++ = *cptr++;
            cnt--;
            continue;
         }

         if (*cptr == '+')
         {
            *sptr++ = ' ';
            cptr++;
            cnt--;
            continue;
         }

         if (*cptr == '%')
         {
            /*****************************/
            /* convert from URL-encoding */
            /*****************************/

            cptr++;
            if (!--cnt)
            {
               ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_URL_ENC), FI_LI);
               PutEnd (rqptr);
               return;
            }

            c = 0;
            if (*cptr >= '0' && *cptr <= '9')
               c = (*cptr - (int)'0') << 4;
            else
            if (toupper(*cptr) >= 'A' && toupper(*cptr) <= 'F')
               c = (toupper(*cptr) - (int)'A' + 10) << 4;
            else
            {
               ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_URL_ENC), FI_LI);
               PutEnd (rqptr);
               return;
            }

            cptr++;
            if (!--cnt)
            {
               ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_URL_ENC), FI_LI);
               PutEnd (rqptr);
               return;
            }

            if (*cptr >= '0' && *cptr <= '9')
               c += (*cptr - (int)'0');
            else
            if (toupper(*cptr) >= 'A' && toupper(*cptr) <= 'F')
               c += (toupper(*cptr) - (int)'A' + 10);
            else
            {
               ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_URL_ENC), FI_LI);
               PutEnd (rqptr);
               return;
            }

            cptr++;
            cnt--;

            if (c == '\r')
               rqptr->ContentCount -= 3;
            else
            {
               *sptr++ = c;
               rqptr->ContentCount -= 2;
            }

            continue;
         }

         if (*cptr == '\r')
         {
            /* absorb extraneous carriage-returns (to be a stream-LF file) */
            cptr++;
            cnt--;
            rqptr->ContentCount--;
            continue;
         }

         if (*cptr == '&')
         {
            cptr++;
            cnt--;
            rqptr->ContentCount--;
            InsideFormName = true;
            break;
         }

         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_GENERAL_SANITY), FI_LI);
         PutEnd (rqptr);
         return;
      }

      /* note one after where the preview note ended (i.e. start of file) */
      if (PreviewNoteBegin != NULL && PreviewNoteEnd == NULL)
         PreviewNoteEnd = sptr;

      if (InsideFormName)
      {
         if (cnt >= 11 && strsame (cptr, "protection=", 11))
         {
            /*******************/
            /* file protection */
            /*******************/

            if (Debug) fprintf (stdout, "fileprotection\n");

            cnt -= 11;
            cptr += 11;
            rqptr->ContentCount -= 11;
            if (isxdigit(*cptr))
               tkptr->ProtectionMask = strtol (cptr, NULL, 16);
            while (cnt)
            {
               if (*cptr == '&') break;
               cnt--;
               cptr++;
               rqptr->ContentCount--;
            }
            InsideFormName = false;
         }
         else
         if (cnt >= 12 && strsame (cptr, "previewonly=", 12))
         {
            /****************/
            /* preview only */
            /****************/

            if (Debug) fprintf (stdout, "previewonly\n");

            cnt -= 12;
            cptr += 12;
            rqptr->ContentCount -= 12;
            /* if value is non-empty then preview, otherwise ignore */
            while (cnt)
            {
               if (*cptr == '&') break;
               cnt--;
               cptr++;
               rqptr->ContentCount--;
               PreviewOnly = true;
            }
            InsideFormName = false;
         }
         else
         if (cnt >= 12 && strsame (cptr, "previewnote=", 12))
         {
            /****************/
            /* preview note */
            /****************/

            if (Debug) fprintf (stdout, "previewnote\n");

            cnt -= 12;
            cptr += 12;
            rqptr->ContentCount -= 12;
            /* note the beginning and end of the preview note */
            PreviewNoteBegin = sptr;
            PreviewNoteEnd = NULL;
            InsideFormName = false;
         }
         else
         if (cnt >= 10 && strsame (cptr, "hexencoded", 10))
         {
            /*****************************/
            /* hexadecimal-encoded field */
            /*****************************/

            if (Debug) fprintf (stdout, "hexencoded\n");
 
            cnt -= 10;
            cptr += 10;
            rqptr->ContentCount -= 10;
            while (cnt && *cptr != '=')
            {
               cnt--;
               cptr++;
               rqptr->ContentCount--;
            }
            if (cnt)
            {
               cnt--;
               cptr++;
               rqptr->ContentCount--;
            }

            while (cnt)
            {
               if (*cptr == '&') break;

               if (*cptr == '%')
               {
                  /* URL-encoded hexadecimal character! */
                  if (cnt >= 3)
                  {
                     cnt -= 3;
                     cptr += 3;
                     rqptr->ContentCount -= 3;
                     continue;
                  }
                  ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_URL_ENC), FI_LI);
                  PutEnd (rqptr);
                  return;
               }

               c = 0;
               if (*cptr >= '0' && *cptr <= '9')
                  c = (*cptr - (int)'0') << 4;
               else
               if (toupper(*cptr) >= 'A' && toupper(*cptr) <= 'F')
                  c = (toupper(*cptr) - (int)'A' + 10) << 4;
               else
               {
                  ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_URL_ENC), FI_LI);
                  PutEnd (rqptr);
                  return;
               }

               if (!--cnt)
               {
                  ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_URL_ENC), FI_LI);
                  PutEnd (rqptr);
                  return;
               }
               cptr++;
 
               if (*cptr >= '0' && *cptr <= '9')
                  c += (*cptr - (int)'0');
               else
               if (toupper(*cptr) >= 'A' && toupper(*cptr) <= 'F')
                  c += (toupper(*cptr) - (int)'A' + 10);
               else
               {
                  ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_URL_ENC), FI_LI);
                  PutEnd (rqptr);
                  return;
               }

               cnt--;
               cptr++;
 
               *sptr++ = c;
               rqptr->ContentCount--;
            }

            InsideFormName = false;
         }
      }

      while (InsideFormName && cnt)
      {
         cnt--;
         rqptr->ContentCount--;
         if (*cptr++ != '=') continue;
         InsideFormName = false;
         break;
      }
   }

   tkptr->FileNamePtr = rqptr->RequestFileName;
   tkptr->FileContentTypePtr =
      ConfigContentType (NULL, rqptr->RequestParseNam.nam$l_type);
   tkptr->FileContentPtr = rqptr->ContentBufferPtr;
   tkptr->FileContentLength = rqptr->ContentCount;

   if (PreviewOnly)
   {
      /****************/
      /* preview only */
      /****************/

      /* create a "temporary" file name and path */

      StringDsc.dsc$a_pointer = TempName;
      StringDsc.dsc$w_length = sizeof(TempName) - 1;
      status = sys$fao (&TempNameFaoDsc, 0, &StringDsc,
                        rqptr->NumericTime[0],
                        rqptr->NumericTime[1],
                        rqptr->NumericTime[2],
                        rqptr->NumericTime[3],
                        rqptr->NumericTime[4],
                        rqptr->NumericTime[5],
                        rqptr->NumericTime[6]);

      c = *rqptr->RequestParseNam.nam$l_name;
      *rqptr->RequestParseNam.nam$l_name =
         *rqptr->RequestParseNam.nam$l_ver = '\0';
      StringDsc.dsc$a_pointer = tkptr->TemporaryFileName;
      StringDsc.dsc$w_length = sizeof(tkptr->TemporaryFileName) - 1;
      
      status = sys$fao (&TempFileNameFaoDsc, 0, &StringDsc,
                        rqptr->RequestParseNam.nam$l_dev,
                        TempName,
                        rqptr->RequestParseNam.nam$l_type);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = rqptr->PathInfoPtr;
         rqptr->ErrorHiddenTextPtr = tkptr->FileNamePtr;
         ErrorVmsStatus (rqptr, status, FI_LI);
         PutEnd (rqptr);
         return;
      }
      if (Debug)
         fprintf (stdout, "tkptr->TemporaryFileName |%s|\n",
                  tkptr->TemporaryFileName);

      zptr = (sptr = tkptr->TemporaryFilePath) +
             sizeof(tkptr->TemporaryFilePath);
      for (cptr = rqptr->PathInfoPtr; *cptr && sptr < zptr; *sptr++ = *cptr++);
      if (sptr < zptr)
      {
         *sptr = '\0';
         while (sptr > tkptr->TemporaryFilePath && *sptr != '/') sptr--;
         if (*sptr == '/') sptr++;
         for (cptr = TempName; *cptr && sptr < zptr; *sptr++ = *cptr++);
         for (cptr = rqptr->RequestParseNam.nam$l_type;
              *cptr && sptr < zptr;
              *sptr++ = tolower(*cptr++));
      }
      if (sptr >= zptr)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         PutEnd (rqptr);
         return;
      }
      *sptr = '\0';
      if (Debug)
         fprintf (stdout, "tkptr->TemporaryFilePath |%s|\n",
                  tkptr->TemporaryFilePath);

      *rqptr->RequestParseNam.nam$l_name = c;
      *rqptr->RequestParseNam.nam$l_ver = ';';

      tkptr->FileNamePtr = tkptr->TemporaryFileName;
   }
   else
   if (PreviewNoteBegin != NULL && PreviewNoteEnd != NULL)
   {
      if (Debug) fprintf (stdout, "previewnote removed\n");
      tkptr->FileContentPtr = PreviewNoteEnd;
      tkptr->FileContentLength -= PreviewNoteEnd - PreviewNoteBegin;
   }

   PutWriteFile (rqptr);
}

/*****************************************************************************/
/*
Sorry about this overly long function.

This function processes a request body according to draft RFC-1867,
"Form-based File Upload in HTML".  As yet it is not a full implementation.
It will not process "multipart/mixed" subsections.

References have been made to RFCs 1521, 1522 and 1802, although the
implementation herein is very basic, a facility only to allow uploads of files
into the server administered file system.

This function can be called multiple times to completely process a MIME
compliant HTTP request body.  The thread storage 'ContentPtr' points at
the current position in the body, and 'ContentCount' how many characters
still left to process.  The body is processed MIME-boundary delimited section
by MIME-boundary delimited section until the end boundary is encountered.
*/

PutProcessMultipartFormData (struct RequestStruct *rqptr)

{
   register int  cnt, lcnt;
   register unsigned char  *cptr, *sptr, *zptr;
   register struct PutTaskStruct  *tkptr;

   boolean  IsFileName,
            IsProtectionMask;
   int  status,
        Length,
        PeriodCount;
   unsigned char  *CharPtr,
                  *ContentTypePtr,
                  *NamePtr;
   unsigned char  **StorageAddress;
   char  FieldName [256],
         FieldValue [256],
         FileName [256],
         WriteFileName [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PutProcessMultipartFormData()\n");

   tkptr = rqptr->PutTaskPtr;

   if (tkptr->MultipartBoundaryPtr == NULL)
   {
      /*******************************/
      /* first call for this request */
      /*******************************/

      /* store the name of the directory the files will be created in */
      if ((tkptr->MultipartDirectoryPtr =
           PutStoreString (rqptr, rqptr->RequestFileName,
                           strlen(rqptr->RequestFileName), FI_LI)) == NULL)
      {
         PutEnd (rqptr);
         return;
      }
      if (Debug)
         fprintf (stdout, "|%s|\n", tkptr->MultipartDirectoryPtr);

      if (!PutGetMimeBoundary (rqptr, rqptr->HttpContentTypePtr))
      {
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_MULTIPART), FI_LI);
         PutEnd (rqptr);
         return;
      }

      /* keep track of where we are up to with 'ContentPtr' */
      rqptr->ContentPtr = rqptr->ContentBufferPtr;
   }

   /******************************/
   /* find leading MIME boundary */
   /******************************/

   if (!(cnt = rqptr->ContentCount))
   {
      PutEnd (rqptr);
      return;
   }
   cptr = rqptr->ContentPtr;

   if (Debug) fprintf (stdout, "cnt: %d cptr: %d\n", cnt, cptr);
   /** if (Debug) fprintf (stdout, "|%s|\n", cptr); **/

   while (cnt)
   {
      /* look for the MIME boundary's leading '--' */
      while (cnt && *(unsigned short*)cptr != 0x2d2d)
      {
         cnt--;
         cptr++;
      }
      if (cnt-2 >= tkptr->MultipartBoundaryLength &&
          !memcmp (cptr+2, tkptr->MultipartBoundaryPtr,
                   tkptr->MultipartBoundaryLength))
      {
         /* found a MIME stream boundary */
         break;
      }
      /* not a boundary, continue on */
      cnt--;
      cptr++;
   }
   if (!cnt)
   {
      /* shouldn't have run out of data here, must be an error! */
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_MULTIPART), FI_LI);
      PutEnd (rqptr);
      return;
   }

   cptr += tkptr->MultipartBoundaryLength + 2;
   cnt -= tkptr->MultipartBoundaryLength + 2;

   if (cnt >= 2 && *(unsigned short*)cptr == 0x2d2d)
   {
      /* end of MIME stream */
      PutEnd (rqptr);
      return;
   }

   if (Debug) fprintf (stdout, "cnt: %d cptr: %d\n", cnt, cptr);

   /*********************/
   /* process mime part */
   /*********************/

   ContentTypePtr = "";
   FileName[0] = '\0';

   /* while not having encountered a blank line (two successive 'cr/lf's) */
   lcnt = 0;
   while (lcnt < 2)
   {
      if (*cptr == '\r')
      {
         cptr++;
         cnt--;
         continue;
      }
      if (*cptr == '\n')
      {
         cptr++;
         cnt--;
         lcnt++;
         continue;
      }
      lcnt = 0;

      /** if (Debug) fprintf (stdout, "|%s|\n", cptr); **/

      if (strsame (cptr, "Content-Disposition:", 20))
      {
         /***********************/
         /* content-disposition */
         /***********************/

         /* note where we are, gonna use 'cptr' as scratch pointer */
         CharPtr = cptr;

         cptr += 20;
         while (ISLWS(*cptr)) cptr++;
         if (!strsame (cptr, "form-data;", 10))
         {
            /* should have had 'form-data' following the disposition */
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_MULTIPART), FI_LI);
            PutEnd (rqptr);
            return;
         }
         cptr += 10;

         while (*cptr && *cptr != '\r' && *cptr != '\n')
         {
            while (ISLWS(*cptr) || *cptr == ';') cptr++;
            if (strsame (cptr, "name=", 5))
            {
               /**************/
               /* field name */
               /**************/

               cptr = PutGetMultipartFieldName (rqptr, "name", cptr+5,
                                                FieldName, sizeof(FieldName));
               if (cptr == NULL)
               {
                  PutEnd (rqptr);
                  return;
               }
            }
            else
            if (strsame (cptr, "filename=", 9))
            {
               /*************/
               /* file name */
               /*************/

               cptr = PutGetMultipartFieldName (rqptr, "filename", cptr+9,
                                                FileName, sizeof(FileName));
               if (cptr == NULL)
               {
                  PutEnd (rqptr);
                  return;
               }

               /*
                  Initial development took place using Netscape Navigator
                  v2.02B for VMS from Digital.  This release would replace
                  a version delimiter (';') with a period ('.').  This code
                  eliminates both these version components.
               */
               PeriodCount = 0;
               for (cptr = FileName; *cptr && *cptr != ';'; cptr++)
                  if (*cptr == '.') PeriodCount++;
               if (*cptr)
                  *cptr = '\0';
               else
               if (PeriodCount > 1)
               {
                  for (cptr = FileName; *cptr; cptr++)
                     if (*cptr == '.')
                        if (!--PeriodCount) break;
                  *cptr = '\0';
               }
               if (Debug) fprintf (stdout, "FileName |%s|\n", FileName);
            }

            /* find the end of this section (';') or of the line */
            while (*cptr && *cptr != ';' && *cptr != '\r' && *cptr != '\n')
               cptr++;
         }

         /* go back where we were (so that 'cnt' can also be decremented) */
         cptr = CharPtr;
      }
      else
      if (strsame (cptr, "Content-Type:", 13))
      {
         /****************/
         /* content-type */
         /****************/

         cptr += 13;
         while (ISLWS(*cptr)) cptr++;
         CharPtr = cptr;
         while (*cptr && !ISLWS(*cptr) &&
                *cptr != '\r' && *cptr != '\n') cptr++;

         ContentTypePtr = sptr = VmGetHeap (rqptr, cptr-CharPtr+1);
         memcpy (sptr, CharPtr, cptr-CharPtr);
         sptr[cptr-CharPtr] = '\0';
         if (Debug) fprintf (stdout, "ContentTypePtr |%s|\n", ContentTypePtr);

         if (strsame (ContentTypePtr, "multipart/mixed", 15))
         {
            /* can't do 'multipart/mixed' ... yet */
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_MULTIPART_MIXED), FI_LI);
            PutEnd (rqptr);
            return;
         }
      }

      /* find the end of this line (just before the carriage-control) */
      while (cnt && *cptr != '\r' && *cptr != '\n')
      {
         cptr++;
         cnt--;
      }
   }

   /*******************************/
   /* find trailing MIME boundary */
   /*******************************/

   if (Debug) fprintf (stdout, "cnt: %d cptr: %d\n", cnt, cptr);
   /** if (Debug) fprintf (stdout, "|%s|\n", cptr); **/

   CharPtr = cptr;

   while (cnt)
   {
      /* the 'cr/lf' preceding the leading '--' are looked for here! */
      while (cnt && *(unsigned long*)cptr != 0x2d2d0a0d)
      {
         cnt--;
         cptr++;
      }
      if (cnt-4 >= tkptr->MultipartBoundaryLength &&
          !memcmp (cptr+4, tkptr->MultipartBoundaryPtr,
                   tkptr->MultipartBoundaryLength))
      {
         /* found a MIME stream boundary */
         break;
      }
      /* not a boundary, continue on */
      cnt--;
      cptr++;
   }
   if (!cnt)
   {
      /* shouldn't have run out of data here either, must be an error! */
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_MULTIPART), FI_LI);
      PutEnd (rqptr);
      return;
   }

   if (Debug) fprintf (stdout, "cnt: %d cptr: %d\n", cnt, cptr);
   /** if (Debug) fprintf (stdout, "|%s|\n", cptr); **/

   /**********************************/
   /* adjust for intervening content */
   /**********************************/

   Length = cptr - CharPtr;
   if (Debug) fprintf (stdout, "Length: %d bytes\n", Length);

   cptr += 4 + tkptr->MultipartBoundaryLength;
   cnt -= 4 + tkptr->MultipartBoundaryLength;

   if (cnt >= 2 && *(unsigned short*)cptr == 0x2d2d)
   {
      /* end of MIME stream */
      cnt = 0;
   }
   else
   {
      /* restore to what will be the leading MIME boundary */
      cptr -= tkptr->MultipartBoundaryLength + 2;
      cnt += tkptr->MultipartBoundaryLength + 2;
   }

   if (Debug) fprintf (stdout, "cnt: %d cptr: %d\n", cnt, cptr);
   /** if (Debug) fprintf (stdout, "|%s|\n", cptr); **/

   if (cnt)
   {
      tkptr->NextFunctionPtr = &PutProcessMultipartFormData;
      /* buffer these, we'll need them next time through */
      rqptr->ContentPtr = cptr;
      rqptr->ContentCount = cnt;
   }
   else
      tkptr->NextFunctionPtr = &PutEnd;

   /*******************/
   /* process content */
   /*******************/

   if (FileName[0])
   {
      /***********************/
      /* write file contents */
      /***********************/

      if (tkptr->MultipartFileNamePtr != NULL &&
          tkptr->MultipartFileNamePtr[0])
      {
         /* use the explicitly supplied file name */
         strcpy (WriteFileName, tkptr->MultipartFileNamePtr);
      }
      else
      {
         /*
            Try and ensure it's just the name and extension.
            Allow for directory terminators for DOS, Unix and VMS.
         */
         for (cptr = FileName; *cptr; cptr++);
         while (cptr > FileName &&
                *cptr != '\\' && *cptr != '/' && *cptr != ']') cptr--;
         if (*cptr == '\\' || *cptr == '/' || *cptr == ']') cptr++;

         if (!PutDirNameType (tkptr->MultipartDirectoryPtr,
                              cptr, WriteFileName, sizeof(WriteFileName)))
         {
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_MULTIPART_FILENAME), FI_LI);
            PutEnd (rqptr);
            return;
         }
      }

      /* ensure any upload file name explicitly specified is used only once */
      tkptr->MultipartFileNamePtr = NULL;

      if (!ContentTypePtr[0]) ContentTypePtr = "text/plain";

      tkptr->FileNamePtr = WriteFileName;
      tkptr->FileContentTypePtr = ContentTypePtr;
      tkptr->FileContentPtr = CharPtr;
      tkptr->FileContentLength = Length;

      PutWriteFile (rqptr);

      return;
   }

   /**********************/
   /* known field values */
   /**********************/

   if (strsame (FieldName, "filename", -1) ||
       strsame (FieldName, "uploadfilename", -1) ||
       strsame (FieldName, "directoryname", -1))
   {
      IsFileName = true;
      IsProtectionMask = false;
      StorageAddress = &tkptr->MultipartFileNamePtr;
   }
   else
   if (strsame (FieldName, "protection", -1))
   {
      IsFileName = false;
      IsProtectionMask = true;
   }
   else
   if (strsame (FieldName, "action", -1))
   {
      IsFileName = false;
      IsProtectionMask = false;
      StorageAddress = &tkptr->MultipartActionPtr;
   }
   else
   {
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_MULTIPART_FIELD), FI_LI);
      PutEnd (rqptr);
      return;
   }

   if (PutGetMultipartFieldValue (rqptr, FieldName, CharPtr,
                                  FieldValue, sizeof(FieldValue)) == NULL)
   {
      PutEnd (rqptr);
      return;
   }

   if (IsFileName && FieldValue[0])
   {
      if (!PutDirNameType (tkptr->MultipartDirectoryPtr,
                           FieldValue, FileName, sizeof(FileName)))
      {
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_MULTIPART_FILENAME), FI_LI);
         PutEnd (rqptr);
         return;
      }
      strcpy (FieldValue, FileName);
   }

   if (IsProtectionMask && FieldValue[0])
   {
      if (isxdigit(FieldValue[0]))
         tkptr->ProtectionMask = strtol (FieldValue, NULL, 16);
   }
   else
   {
      if ((*StorageAddress =
           PutStoreString (rqptr, FieldValue, strlen(FieldValue), FI_LI))
           == NULL)
      {
         PutEnd (rqptr);
         return;
      }
      if (Debug) fprintf (stdout, "*StorageAddress |%s|\n", *StorageAddress);
   }

   /**********/
   /* action */
   /**********/

   if (strsame (FieldName, "action", -1))
   {
      if (strsame (tkptr->MultipartActionPtr, "create", -1))
         status = PutCreateDirectory (rqptr, tkptr->MultipartFileNamePtr);
      else
      if (strsame (tkptr->MultipartActionPtr, "delete", -1))
         status = PutDelete (rqptr, tkptr->MultipartFileNamePtr);
      else
      {
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_MULTIPART_FIELD), FI_LI);
         PutEnd (rqptr);
         return;
      }

      /**************************/
      /* check for action error */
      /**************************/

      if (VMSnok (status))
      {
         PutEnd (rqptr);
         return;
      }

      /* ensure any parameters explicitly specified are used only once */
      tkptr->MultipartFileNamePtr = tkptr->MultipartActionPtr = NULL;
   }

   /*************************/
   /* declare next function */
   /*************************/

   /* declare an AST to execute the required function */
   SysDclAst (tkptr->NextFunctionPtr, rqptr);
}

/*****************************************************************************/
/*
Create a file specification for a directory and file/extension components.
*/

int PutDirNameType
(
char *Directory,
char *NameType,
char *FileName,
int SizeOfFileName
)
{
   register char  *cptr, *sptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "PutDirNameType() |%s|%s|\n", Directory, NameType);
   
   if (!Directory[0] || !NameType[0]) return (0);
   zptr = (sptr = FileName) + SizeOfFileName;
   for (cptr = Directory; *cptr && sptr < zptr; *sptr++ = toupper(*cptr++));
   for (cptr = NameType; *cptr && sptr < zptr; *sptr++ = toupper(*cptr++));
   if (sptr >= zptr) return (0);
   *sptr = '\0';
   return (sptr - FileName);
}

/*****************************************************************************/
/*
Get the name of a field from a "multipart/form-data" content body.
*/

char* PutGetMultipartFieldName
(
struct RequestStruct *rqptr,
char *Field,
char *CharPtr,
char *Name,
int SizeOfName
)
{
   register unsigned char  *cptr, *sptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "PutGetMultipartFieldName() |%s|\n", Field);
   
   zptr = (sptr = Name) + SizeOfName;
   cptr = CharPtr;

   if (*cptr == '\"')
   {
      cptr++;
      while (*cptr && *cptr != '\"' &&
             *cptr != '\r' && *cptr != '\n' &&
             sptr < zptr)
         *sptr++ = *cptr++;
   }
   else
   {
      while (*cptr && *cptr != ';' &&
             *cptr != '\r' && *cptr != '\n' &&
             sptr < zptr)
         *sptr++ = *cptr++;
   }

   if (sptr >= zptr)
   {
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_MULTIPART_FIELD), FI_LI);
      return (NULL);
   }
   *sptr = '\0';
   if (Debug) fprintf (stdout, "Name |%s|\n", Name);

   if (!Name[0])
   {
      if (strsame (Field, "filename", -1))
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_MULTIPART_UPLOAD), FI_LI);
      else
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_MULTIPART_FIELD), FI_LI);
      return (NULL);
   }

   return (cptr);
}

/*****************************************************************************/
/*
Get the value (i.e. that between the quotes) of a field from a
"multipart/form-data" content body.
*/

char* PutGetMultipartFieldValue
(
struct RequestStruct *rqptr,
char *Field,
char *CharPtr,
char *Value,
int SizeOfValue
)
{
   register unsigned char  *cptr, *sptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "PutGetMultipartFieldValue() |%s|\n", Field);
   
   zptr = (sptr = Value) + SizeOfValue;
   cptr = CharPtr;

   while (*cptr && *cptr != '\r' && *cptr != '\n' && sptr < zptr)
      *sptr++ = *cptr++;

   if (sptr >= zptr)
   {
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_MULTIPART_FIELD), FI_LI);
      return (NULL);
   }
   *sptr = '\0';
   if (Debug) fprintf (stdout, "Value |%s|\n", Value);

   return (cptr);
}

/*****************************************************************************/
/*
Allocate heap memory at the supplied pointer then copy the supplied string
into it.
*/

char* PutStoreString
(
struct RequestStruct *rqptr,
char *String,
int StringLength,
char *SourceFileName,
int SourceLineNumber
)
{
   char  *CharPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "PutStoreString() %d |%s|\n", StringLength, String);
   
   if (!StringLength) StringLength = strlen(String);

   /* allocate heap memory to store the string */
   CharPtr = VmGetHeap (rqptr, StringLength+1);
   memcpy (CharPtr, String, StringLength);
   CharPtr[StringLength] = '\0';
   return (CharPtr);
}

/*****************************************************************************/
/*
Scan through a "Content-Type:" header line looking for the MIME "boundary="
string.  If found allocate dynamic memory, copy it in, pointed to and sized by
'tkptr->MultipartBoundaryPtr', 'tkptr->MultipartBoundaryLength'
and return true.  If none found point to an empty string and return false.
Can be called multiple times with new memory being allocated for each boundary
detected.
*/

boolean PutGetMimeBoundary
(
struct RequestStruct *rqptr,
char *ContentTypePtr
)
{
   register unsigned char  *cptr;
   register struct PutTaskStruct  *tkptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PutGetMimeBoundary() |%s|\n", ContentTypePtr);

   tkptr = rqptr->PutTaskPtr;

   tkptr->MultipartBoundaryPtr = "";
   tkptr->MultipartBoundaryLength = 0;

   cptr = ContentTypePtr;
   while (*cptr)
   {
      while (*cptr && *cptr != ';') cptr++;
      if (!*cptr) break;
      cptr++;
      while (*cptr && ISLWS(*cptr)) cptr++;
      if (strsame (cptr, "boundary=", 9))
      {
         tkptr->MultipartBoundaryPtr = (cptr += 9);
         while (*cptr && *cptr != ';') cptr++;
         tkptr->MultipartBoundaryLength =
            cptr - tkptr->MultipartBoundaryPtr ;
      }
   }

   if (Debug)
      fprintf (stdout, "%d |%s|\n",
               tkptr->MultipartBoundaryLength,
               tkptr->MultipartBoundaryPtr);

   return (tkptr->MultipartBoundaryLength);
}

/*****************************************************************************/
/*
Create a file using '->FileName'.  Fill with '->ContentFileLength' bytes
from '->ContentFilePtr'.  Return to processing at '->NextFunctionPtr'
(providing there was no problem!)  Write using block I/O for efficiency and
record-type independence.
*/

PutWriteFile (struct RequestStruct *rqptr)

{
   register char  *cptr, *sptr, *zptr;
   register struct PutTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "PutWriteFile() |%s|\n",
               rqptr->PutTaskPtr->FileNamePtr);

   tkptr = rqptr->PutTaskPtr;

   /* ultra-cautious, check server's access to this file */
   /* check access to parent directory */
   if (VMSnok (status = AuthCheckWriteAccess (rqptr, tkptr->FileNamePtr, 0)))
   {
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileNamePtr);
      rqptr->ErrorHiddenTextPtr = tkptr->FileNamePtr;
      ErrorVmsStatus (rqptr, status, FI_LI);
      PutEnd (rqptr);
      return;
   }

   tkptr = rqptr->PutTaskPtr;

   cptr = tkptr->FileNamePtr;
   zptr = (sptr = tkptr->FileName) + sizeof(tkptr->FileName);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      PutEnd (rqptr);
      return;
   }
   *sptr = '\0';
   tkptr->FileNameLength = sptr - tkptr->FileName;

   tkptr->FileCount++;
   tkptr->FileCreated = false;
   tkptr->FileSizeBytes = tkptr->FileContentLength;

   tkptr->FileFab = cc$rms_fab;
   tkptr->FileFab.fab$b_fac = FAB$M_PUT | FAB$M_BIO;
   tkptr->FileFab.fab$l_fna = tkptr->FileName;
   tkptr->FileFab.fab$b_fns = tkptr->FileNameLength;
   tkptr->FileFab.fab$l_fop = FAB$M_SQO;
   tkptr->FileFab.fab$l_nam = &tkptr->FileNam;
   tkptr->FileFab.fab$b_rat = 0;

   if (strsame (tkptr->FileContentTypePtr, "text/", 5))
   {
      /* "textual" content-type, guess about the _best_ RMS record format! */
      register int  CrLfCount,
                    LfCount;

      CrLfCount = LfCount = 0;
      if (tkptr->FileContentLength > 1024)
         zptr = tkptr->FileContentPtr + 1024;
      else
         zptr = tkptr->FileContentPtr + tkptr->FileContentLength;
      for (cptr = tkptr->FileContentPtr+1; cptr < zptr; cptr++)
      {
         if (*cptr == '\n')
            if (*(cptr-1) == '\r')
               CrLfCount++;
            else
               LfCount++;
      }
      if (CrLfCount >= LfCount)
      {
         /* more CR+LFs than just LFs (DOS-style), STREAM */
         tkptr->FileFab.fab$b_rfm = FAB$C_STM;
      }
      else
      {
         /* STREAM-LF (Unix-style) */
         tkptr->FileFab.fab$b_rfm = FAB$C_STMLF;
      }
      tkptr->FileFab.fab$b_rat = FAB$M_CR;
   }
   else
   if (strsame (tkptr->FileContentTypePtr,
                "application/x-www-form-urlencoded", -1))
   {
      /* STREAM-LF (Unix-style) */
      tkptr->FileFab.fab$b_rfm = FAB$C_STMLF;
      tkptr->FileFab.fab$b_rat = FAB$M_CR;
   }
   else
   {
      /* undefined for binary ... everything else */
      tkptr->FileFab.fab$b_rfm = FAB$C_UDF;
      tkptr->FileFab.fab$b_rat = 0;
   }

   tkptr->FileFab.fab$b_shr = FAB$M_NIL;
   tkptr->FileFab.fab$l_xab = &tkptr->FileXabPro;

   /* initialize the name block */
   tkptr->FileNam = cc$rms_nam;
   tkptr->FileNam.nam$l_esa = tkptr->ExpandedFileName;
   tkptr->FileNam.nam$b_ess = sizeof(tkptr->ExpandedFileName)-1;

   if (Debug)
      fprintf (stdout, "ProtectionMask: %%X%04.04X\n",
               tkptr->ProtectionMask);

   /* initialize the protection extended attribute block */
   tkptr->FileXabPro = cc$rms_xabpro;
   tkptr->FileXabPro.xab$w_pro = tkptr->ProtectionMask;

   /* use SYSPRV to ensure creation of file (provided protection is S:RWED!) */
   EnableSysPrv ();

   status = sys$create (&tkptr->FileFab, 0, 0);

   DisableSysPrv ();

   if (VMSnok (status) && tkptr->FileFab.fab$l_stv)
      status = tkptr->FileFab.fab$l_stv;

   if (VMSnok (status))
   {
      /* sys$create() error */
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      PutWriteFileEnd (rqptr);
      return;
   }

   tkptr->FileCreated = true;

   tkptr->FileRab = cc$rms_rab;
   tkptr->FileRab.rab$l_fab = &tkptr->FileFab;
   tkptr->FileRab.rab$l_ctx = rqptr;
   tkptr->FileRab.rab$l_rop = RAB$M_BIO;

   if (VMSnok (status = sys$connect (&tkptr->FileRab, 0, 0)))
   {
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      PutWriteFileEnd (rqptr);
      return;
   }

   if (!tkptr->FileContentLength)
   {
      PutWriteFileEnd (rqptr);
      return;
   }

   tkptr->FileRab.rab$l_bkt = 1;

   if (tkptr->FileContentLength > 65024)
      tkptr->FileRab.rab$w_rsz = 65024;
   else
      tkptr->FileRab.rab$w_rsz = tkptr->FileContentLength;

   if (Debug)
      fprintf (stdout, "tkptr->FileRab.rab$w_rsz: %d\n",
               tkptr->FileRab.rab$w_rsz);

   tkptr->FileRab.rab$l_rbf = tkptr->FileContentPtr;

   tkptr->FileContentPtr += tkptr->FileRab.rab$w_rsz;
   tkptr->FileContentLength -= tkptr->FileRab.rab$w_rsz;

   sys$write (&tkptr->FileRab, &PutWriteFileAst, &PutWriteFileAst);
}

/*****************************************************************************/
/*
After each "record" (not really records because the file has an undefined
format and therefore the data is just written as-is to the file) is written to
the file this AST function is called to either write more data, or on conlusion
to call the end file function.  If an error is reported from the write the end
file function is called, with the presencne of the error message causing the
file to be deleted.
*/ 

PutWriteFileAst (struct RAB *RabPtr)

{
   register struct PutTaskStruct  *tkptr;

   int  status;
   struct RequestStruct *rqptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
         "PutWriteFileAst() sts: %%X%08.08X stv: %%X%08.08X rsz: %d\n",
         RabPtr->rab$l_sts,
         RabPtr->rab$l_stv,
         RabPtr->rab$w_rsz);

   rqptr = RabPtr->rab$l_ctx;
   tkptr = rqptr->PutTaskPtr;

   if (VMSnok (tkptr->FileRab.rab$l_sts))
   {
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, tkptr->FileRab.rab$l_sts, FI_LI);
      PutWriteFileEnd (rqptr);
      return;
   }

   if (Debug)
      fprintf (stdout, "ContentFileLength: %d\n", tkptr->FileContentLength);

   if (!tkptr->FileContentLength)
   {
      PutWriteFileEnd (rqptr);
      return;
   }

   /* writing in 65024 byte (127 block) lots */
   tkptr->FileRab.rab$l_bkt += 127;

   if (tkptr->FileContentLength > 65024)
      tkptr->FileRab.rab$w_rsz = 65024;
   else
      tkptr->FileRab.rab$w_rsz = tkptr->FileContentLength;

   if (Debug)
      fprintf (stdout, "tkptr->FileRab.rab$w_rsz: %d\n",
               tkptr->FileRab.rab$w_rsz);

   tkptr->FileRab.rab$l_rbf = tkptr->FileContentPtr;

   tkptr->FileContentPtr += tkptr->FileRab.rab$w_rsz;
   tkptr->FileContentLength -= tkptr->FileRab.rab$w_rsz;

   sys$write (&tkptr->FileRab, &PutWriteFileAst, &PutWriteFileAst);
}

/*****************************************************************************/
/*
Called when the file has been completely written or an error has been detected.
The presence of an error message results in the file being deleted. If OK the
file attributes are changed to limit versions and to stream-LF is "textual".

There is a small window between closing the file and changing the attributes
where it could conceivably be opened for write by another process and interfere
with that change.  Not a problem within this one server because all this
processing is occuring at user-AST-delivery level.  Don't know what to do about
it for the moment so I'll just say it's a low-risk scenario and live with it
for now!
*/ 

PutWriteFileEnd (struct RequestStruct *rqptr)

{
   register struct PutTaskStruct  *tkptr;

   int  status;
   char  *DispositionPtr,
         *PathPtr,
         *TypePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "PutWriteFileEnd() |%s|\n", rqptr->PutTaskPtr->FileName);

   tkptr = rqptr->PutTaskPtr;

   if (tkptr->FileFab.fab$w_ifi)
   {
      /* close, and indicate the file is closed */
      sys$close (&tkptr->FileFab, 0, 0);
      tkptr->FileFab.fab$w_ifi = 0;
   }

   if (rqptr->ErrorMessagePtr != NULL &&
       rqptr->ErrorMessageLength &&
       tkptr->FileCreated)
   {
      /*********************************************/
      /* an error has occured, delete created file */
      /*********************************************/

      /* use SYSPRV to ensure erasure of file */
      EnableSysPrv ();

      tkptr->FileFab.fab$l_fop = FAB$M_NAM;
      status = sys$erase (&tkptr->FileFab, 0, 0);
      if (Debug) fprintf (stdout, "sys$erase() %%X%08.08X\n", status);

      DisableSysPrv ();

      PutEnd (rqptr);
      return;
   }
   else
   {
      if (Config.PutVersionLimit)
      {
         if (VMSnok (PutWriteFileVersionLimit (rqptr)))
         {
            /* as an error has occured terminate the PUT processing */
            PutEnd (rqptr);
            return;
         }
      }

      if (tkptr->TemporaryFilePath[0])
      {
         /* preview only, redirect to temporary file */
         rqptr->LocationPtr =
            VmGetHeap (rqptr, strlen(tkptr->TemporaryFilePath)+6);
         strcpy (rqptr->LocationPtr, " GET ");
         strcpy (rqptr->LocationPtr+5, tkptr->TemporaryFilePath);
         PutEnd (rqptr);
         return;
      }

      if (strsame (tkptr->FileContentTypePtr, "text/", 5) ||
          strsame (tkptr->FileContentTypePtr,
                   "application/x-www-form-urlencoded", -1))
         TypePtr = MsgFor(rqptr,MSG_GENERAL_DOCUMENT);
      else
         TypePtr = MsgFor(rqptr,MSG_GENERAL_FILE);

      PathPtr = MapVmsPath (tkptr->FileName);

      if (tkptr->FileNam.nam$l_fnb & NAM$M_LOWVER)
         DispositionPtr = MsgFor(rqptr,MSG_PUT_SUPERCEDED);
      else
         DispositionPtr = MsgFor(rqptr,MSG_PUT_CREATED);

      PutResponse (rqptr, TypePtr, PathPtr,
                   DispositionPtr, tkptr->FileSizeBytes);

      PutEnd (rqptr);
      return;
   }
}

/*****************************************************************************/
/*
Set the version limit for the file specified in the tkptr->FileFab and
tkptr->FileNam structures.

This function uses the ACP-QIO interface detailed in the "OpenVMS I/O User's 
Reference Manual".
*/ 

int PutWriteFileVersionLimit (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (DeviceDsc, "");

   register struct PutTaskStruct  *tkptr;

   int  status;
   unsigned short  AcpChannel;

   struct fibdef  FileFib;

   struct {
      unsigned short  Length;
      unsigned short  Unused;
      unsigned long  Address;
   } FileFibAcpDsc,
     FileNameAcpDsc;
     
   struct {
      unsigned short  Status;
      unsigned short  Unused1;
      unsigned long  Unused2;
   } AcpIOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "PutWriteFileVersionLimit() |%s|%s|\n",
               rqptr->PutTaskPtr->FileName,
               rqptr->PutTaskPtr->FileContentTypePtr);

   tkptr = rqptr->PutTaskPtr;

   /* assign a channel to the disk device containing the file */
   DeviceDsc.dsc$a_pointer = tkptr->FileNam.nam$l_dev;
   DeviceDsc.dsc$w_length = tkptr->FileNam.nam$b_dev;
   if (Debug)
      fprintf (stdout, "device |%*.*s|\n",
               tkptr->FileNam.nam$b_dev, tkptr->FileNam.nam$b_dev,
               tkptr->FileNam.nam$l_dev);

   status = sys$assign (&DeviceDsc, &AcpChannel, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$assign() %%X%08.08X\n", status);
   if (VMSnok (status))
   {
      rqptr->ErrorHiddenTextPtr = tkptr->FileNam.nam$l_dev;
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }

   /* set up the File Information Block for the ACP interface */
   memset (&FileFib, 0, sizeof(struct fibdef));
   FileFibAcpDsc.Length = sizeof(FileFib);
   FileFibAcpDsc.Address = &FileFib;
   /* quick work around, different syntax for this structure with DEC C? */
#ifdef __DECC
   memcpy (&FileFib.fib$w_did, &tkptr->FileNam.nam$w_did, 6);
#else
   memcpy (&FileFib.fib$r_did_overlay.fib$w_did,
           tkptr->FileNam.nam$w_did, 6);
#endif

   FileNameAcpDsc.Address = tkptr->FileNam.nam$l_name;
   FileNameAcpDsc.Length = tkptr->FileNam.nam$b_name +
                           tkptr->FileNam.nam$b_type +
                           tkptr->FileNam.nam$b_ver;

   /* SYSPRV ensures access to the file (provided protection is S:RWED) */
   EnableSysPrv ();

   /* read the file attributes */
   status = sys$qiow (0, AcpChannel, IO$_ACCESS, &AcpIOsb, 0, 0, 
                      &FileFibAcpDsc, &FileNameAcpDsc, 0, 0, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qio() %%X%08.08X IOsb: %%X%08.08X\n",
               status, AcpIOsb.Status);
   if (VMSok (status)) status = AcpIOsb.Status;

   if (VMSok (status))
   {
      FileFib.fib$w_verlimit = Config.PutVersionLimit;

      status = sys$qiow (0, AcpChannel, IO$_MODIFY, &AcpIOsb, 0, 0, 
                         &FileFibAcpDsc, &FileNameAcpDsc, 0, 0, 0, 0);
      if (Debug)
         fprintf (stdout, "sys$qio() %%X%08.08X IOsb: %%X%08.08X\n",
                  status, AcpIOsb.Status);
      if (VMSok (status)) status = AcpIOsb.Status;
   }

   /* turn off SYSPRV */
   DisableSysPrv ();

   sys$dassgn (AcpChannel);

   if (VMSok (status)) status = AcpIOsb.Status;
   if (VMSnok (status))
   {
      rqptr->ErrorHiddenTextPtr = tkptr->FileNam.nam$l_dev;
      ErrorVmsStatus (rqptr, status, FI_LI);
   }
   return (status);
}

/*****************************************************************************/
/*
Create a directory!  Directory can be specified "device:[dir1.dir2]" or
"device:[dir1]dir2" (first preferable, only one sys$parse() required).
*/ 
 
int PutCreateDirectory
(
struct RequestStruct *rqptr,
char *DirectoryName
)
{
   static unsigned short  ProtectionEnable = 0xffff, /* alter all S,O,G,W */ 
                          EnsureSystemAccessMask = 0xfff0; /* S:RWED */
   static $DESCRIPTOR (DirectoryDsc, "");

   register struct PutTaskStruct  *tkptr;

   int  status,
        DirectoryFileLength;
   unsigned short  Length;
   char  DirectoryFile [256],
         ExpandedDirectoryName [256],
         ScratchDirectory [256];
   char  *DirectoryPtr,
         *PathInfoPtr;
   struct FAB  CreateFab;
   struct NAM  CreateNam;

   /*********/
   /* begin */
   /*********/

   if (Debug)
       fprintf (stdout, "PutCreateDirectory() |%s|\n", DirectoryName);

   tkptr = rqptr->PutTaskPtr;

   if (DirectoryName == NULL || !DirectoryName[0])
   {
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_DIR_NAME), FI_LI);
      return (STS$K_ERROR);
   }

   CreateFab = cc$rms_fab;
   CreateFab.fab$l_fna = DirectoryPtr = DirectoryName;
   CreateFab.fab$b_fns = strlen(DirectoryPtr);
   CreateFab.fab$l_nam = &CreateNam;

   CreateNam = cc$rms_nam;
   CreateNam.nam$l_esa = ExpandedDirectoryName;
   CreateNam.nam$b_ess = sizeof(ExpandedDirectoryName)-1;
   /* parse without disk I/O, syntax check only! */
   CreateNam.nam$b_nop = NAM$M_SYNCHK;

   if (VMSnok (status = sys$parse (&CreateFab, 0, 0)))
   {
      *CreateNam.nam$l_ver = '\0'; 
      rqptr->ErrorTextPtr = MapVmsPath (DirectoryPtr);
      rqptr->ErrorHiddenTextPtr = DirectoryPtr;
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }

   if (CreateNam.nam$b_name)
   {
      /* assume the new directory has been specified as a file name */
      CreateNam.nam$l_name[-1] == '.';
      CreateNam.nam$l_type[0] == ']';
      CreateNam.nam$l_type[1] == '\0';
      strcpy (ScratchDirectory, ExpandedDirectoryName);

      CreateFab = cc$rms_fab;
      CreateFab.fab$l_fna = DirectoryPtr = ScratchDirectory;
      CreateFab.fab$b_fns = strlen(DirectoryPtr);
      CreateFab.fab$l_nam = &CreateNam;

      CreateNam = cc$rms_nam;
      CreateNam.nam$l_esa = ExpandedDirectoryName;
      CreateNam.nam$b_ess = sizeof(ExpandedDirectoryName)-1;
      /* parse without disk I/O, syntax check only! */
      CreateNam.nam$b_nop = NAM$M_SYNCHK;

      if (VMSnok (status = sys$parse (&CreateFab, 0, 0)))
      {
         *CreateNam.nam$l_ver = '\0'; 
         rqptr->ErrorTextPtr = MapVmsPath (DirectoryPtr);
         rqptr->ErrorHiddenTextPtr = DirectoryPtr;
         ErrorVmsStatus (rqptr, status, FI_LI);
         return (status);
      }
   }

   *CreateNam.nam$l_name = '\0'; 
   if (Debug)
      fprintf (stdout, "CreateName.nam$b_dev |%s|\n", CreateNam.nam$l_dev);

   PathInfoPtr = MapVmsPath (DirectoryPtr);

   /* ultra-cautious, check server's access to this file (using parsed NAM) */
   PutGetDirectoryFile (DirectoryPtr, DirectoryFile, &DirectoryFileLength);
   if (Debug) fprintf (stdout, "DirectoryFile |%s|\n", DirectoryFile);
   if (VMSnok (status = AuthCheckWriteAccess (rqptr, DirectoryFile, 0)))
   {
      rqptr->ErrorTextPtr = PathInfoPtr;
      rqptr->ErrorHiddenTextPtr = DirectoryPtr;
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }

   DirectoryDsc.dsc$a_pointer = CreateNam.nam$l_dev;
   DirectoryDsc.dsc$w_length = CreateNam.nam$b_dev + CreateNam.nam$b_dir +
                               CreateNam.nam$b_name;

   if (Debug)
      fprintf (stdout, "ProtectionMask (before): %%X%04.04X\n",
               tkptr->ProtectionMask);

   tkptr->ProtectionMask &= EnsureSystemAccessMask;

   if (Debug)
      fprintf (stdout, "ProtectionMask (after): %%X%04.04X\n",
               tkptr->ProtectionMask);

   /* SYSPRV to ensure creation of file (provided protection is S:RWED) */
   EnableSysPrv ();

   status = lib$create_dir (&DirectoryDsc, 0, &ProtectionEnable,
                            &tkptr->ProtectionMask, 0, 0);

   DisableSysPrv ();

   if (status == SS$_CREATED)
      PutResponse (rqptr, MsgFor(rqptr,MSG_GENERAL_DIRECTORY),
                   PathInfoPtr, MsgFor(rqptr,MSG_PUT_CREATED), -1);
   else
   if (status == SS$_NORMAL)
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_DIR_EXISTS), FI_LI);
   else
      ErrorVmsStatus (rqptr, status, FI_LI);

   return (status);
}

/*****************************************************************************/
/*
Deletes both files and directories.
*/ 
 
PutDelete
(
struct RequestStruct *rqptr,
char *FileName
)
{
   register char  *cptr;
   register struct PutTaskStruct  *tkptr;

   boolean  DeletingDirectory;
   int  status,
        EraseCount,
        FileCount,
        DirFileLength;
   char  DirFile [256],
         DeleteFileName [256],
         DirectoryFileName [256],
         ScratchFileName [256],
         SearchFileName [256];
   char  *DirectoryVerPtr,
         *ScratchVerPtr;
   struct FAB  DeleteFab;
   struct NAM  DeleteNam;
   struct FAB  SearchFab;
   struct NAM  SearchNam;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PutDelete() |%s|\n", FileName);

   tkptr = rqptr->PutTaskPtr;

   if (FileName == NULL || !FileName[0])
   {
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_PUT_DELETE_NO_FILENAME), FI_LI);
      return (STS$K_ERROR);
   }

   SearchFab = cc$rms_fab;
   SearchFab.fab$l_dna = ";*";
   SearchFab.fab$b_dns = 2;
   SearchFab.fab$l_fna = FileName;
   SearchFab.fab$b_fns = strlen(FileName);
   SearchFab.fab$l_nam = &SearchNam;

   SearchNam = cc$rms_nam;
   SearchNam.nam$l_esa = ScratchFileName;
   SearchNam.nam$b_ess = sizeof(ScratchFileName)-1;
   SearchNam.nam$l_rsa = SearchFileName;
   SearchNam.nam$b_rss = sizeof(SearchFileName)-1;

   if (VMSnok (status = sys$parse (&SearchFab, 0, 0)))
   {
      rqptr->ErrorTextPtr = MapVmsPath (FileName);
      rqptr->ErrorHiddenTextPtr = FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }

   ScratchVerPtr = SearchNam.nam$l_ver;

   if (Debug)
   {
      *ScratchVerPtr = '\0'; 
      fprintf (stdout, "ScratchFileName |%s| %d %d\n",
               ScratchFileName, SearchNam.nam$b_name, SearchNam.nam$b_type);
      *ScratchVerPtr = ';'; 
   }

   if (DeletingDirectory = (!SearchNam.nam$b_name && SearchNam.nam$b_type == 1))
   {
      /* a directory is being specified */
      *ScratchVerPtr = '\0'; 

      PutGetDirectoryFile (ScratchFileName, DirFile, &DirFileLength);
      if (Debug) fprintf (stdout, "DirFile |%s|\n", DirFile);

      SearchFab = cc$rms_fab;
      SearchFab.fab$l_fna = DirFile;
      SearchFab.fab$b_fns = DirFileLength;
      SearchFab.fab$l_nam = &SearchNam;

      SearchNam = cc$rms_nam;
      SearchNam.nam$l_esa = DirectoryFileName;
      SearchNam.nam$b_ess = sizeof(DirectoryFileName)-1;
      SearchNam.nam$l_rsa = SearchFileName;
      SearchNam.nam$b_rss = sizeof(SearchFileName)-1;

      if (VMSnok (status = sys$parse (&SearchFab, 0, 0)))
      {
         *SearchNam.nam$l_esa = '\0';
         rqptr->ErrorTextPtr = MapVmsPath (DirFile);
         rqptr->ErrorHiddenTextPtr = DirFile;
         ErrorVmsStatus (rqptr, status, FI_LI);
         return (status);
      }

      DirectoryVerPtr = SearchNam.nam$l_ver;
   }

   /*******************/
   /* delete the file */
   /*******************/

   FileCount = 0;
   while (VMSok (status = sys$search (&SearchFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$search() %%X%08.08X\n", status);
      FileCount++;

      DeleteFab = cc$rms_fab;
      DeleteFab.fab$l_fna = SearchNam.nam$l_rsa;
      DeleteFab.fab$b_fns = SearchNam.nam$b_rsl;
      DeleteFab.fab$l_nam = &DeleteNam;

      DeleteNam = cc$rms_nam;
      DeleteNam.nam$l_esa = DeleteFileName;
      DeleteNam.nam$b_ess = sizeof(DeleteFileName)-1;
      /* parse without disk I/O, syntax check only! */
      DeleteNam.nam$b_nop = NAM$M_SYNCHK;

      if (VMSnok (status = sys$parse (&DeleteFab, 0, 0)))
      {
         if (DeletingDirectory)
         {
            rqptr->ErrorTextPtr = MapVmsPath (FileName);
            rqptr->ErrorHiddenTextPtr = FileName;
         }
         else
         {
            *SearchNam.nam$l_rsa = '\0';
            rqptr->ErrorTextPtr = MapVmsPath (SearchFileName);
            rqptr->ErrorHiddenTextPtr = SearchFileName;
         }
         ErrorVmsStatus (rqptr, status, FI_LI);
         return (status);
      }

      *DeleteNam.nam$l_ver = '\0'; 
      if (Debug) fprintf (stdout, "DeleteFileName |%s|\n", DeleteFileName);

      /* ultra-cautious, check server's access to parent directory */
      if (VMSnok (status = AuthCheckWriteAccess (rqptr, DeleteFileName, 0)))
      {
         if (DeletingDirectory)
         {
            rqptr->ErrorTextPtr = MapVmsPath (FileName);
            rqptr->ErrorHiddenTextPtr = FileName;
         }
         else
         {
            *DeleteNam.nam$l_ver = '\0'; 
            rqptr->ErrorTextPtr = MapVmsPath (DeleteFileName);
            rqptr->ErrorHiddenTextPtr = DeleteFileName;
         }
         ErrorVmsStatus (rqptr, status, FI_LI);
         return (status);
      }

      /* SYSPRV to ensure deletion of file (provided protection is S:RWED) */
      EnableSysPrv ();

      EraseCount = 0;
      while (VMSok (status = sys$erase (&DeleteFab, 0, 0)))
      {
         if (Debug) fprintf (stdout, "sys$erase() %%X%08.08X\n", status);
         EraseCount++;
      }
      if (Debug) fprintf (stdout, "sys$erase() %%X%08.08X\n", status);
      if (status == RMS$_FNF && EraseCount) status = SS$_NORMAL;
      if (Debug) fprintf (stdout, "EraseCount: %d\n", EraseCount);

      DisableSysPrv ();

      if (VMSnok (status) && DeleteFab.fab$l_stv)
         status = DeleteFab.fab$l_stv;

      if (VMSnok (status))
      {
         if (DeletingDirectory)
         {
            rqptr->ErrorTextPtr = MapVmsPath (FileName);
            rqptr->ErrorHiddenTextPtr = FileName;
         }
         else
         {
            *DeleteNam.nam$l_ver = '\0'; 
            rqptr->ErrorTextPtr = MapVmsPath (DeleteFileName);
            rqptr->ErrorHiddenTextPtr = DeleteFileName;
         }
         ErrorVmsStatus (rqptr, status, FI_LI);
         return (status);
      }
   }

   if (Debug) fprintf (stdout, "sys$search() %%X%08.08X\n", status);
   if (status == RMS$_NMF) status = SS$_NORMAL;

   if (VMSok (status))
   {
      if (DeletingDirectory)
         PutResponse (rqptr, MsgFor(rqptr,MSG_GENERAL_DIRECTORY),
                      MapVmsPath(FileName), MsgFor(rqptr,MSG_PUT_DELETED), -1); 
      else
         PutResponse (rqptr, MsgFor(rqptr,MSG_GENERAL_FILE),
                      MapVmsPath(FileName), MsgFor(rqptr,MSG_PUT_DELETED), -1); 
   }
   else
   {
      if (Debug) fprintf (stdout, "ScratchFileName |%s|\n", ScratchFileName);
      if (DeletingDirectory)
      {
         if (status == RMS$_FNF) status = RMS$_DNF;
         rqptr->ErrorTextPtr = MapVmsPath (FileName);
         rqptr->ErrorHiddenTextPtr = FileName;
      }
      else
      {
         *ScratchVerPtr = '\0';
         rqptr->ErrorTextPtr = MapVmsPath (ScratchFileName);
         rqptr->ErrorHiddenTextPtr = ScratchFileName;
      }
      ErrorVmsStatus (rqptr, status, FI_LI);
   }

   return (status);
}

/*****************************************************************************/
/*
Creates a directory file name from the parsed file specification.
e.g. "dev:[dir1.dir2]dir3.DIR" from "dev:[dir1.dir2.dir3]file.ext".
Parses the file name specification to reveal any concealed logicals.
*/ 
 
int PutGetDirectoryFile
(
char *FileName,
char *DirFileNamePtr,
int *DirFileLengthPtr
)
{
   register char  *cptr, *fptr, *sptr;

   int  status;
   char  ExpandedFileName [256];
   struct FAB  FileFab;
   struct NAM  FileNam;

   /*********/
   /* begin */
   /*********/

   if (Debug)
       fprintf (stdout, "PutGetDirectoryFile() |%s|\n", FileName);

   DirFileNamePtr[*DirFileLengthPtr = 0] = '\0';

   FileFab = cc$rms_fab;
   FileFab.fab$l_fna = FileName;
   FileFab.fab$b_fns = strlen(FileName);
   FileFab.fab$l_nam = &FileNam;

   FileNam = cc$rms_nam;
   FileNam.nam$l_esa = ExpandedFileName;
   FileNam.nam$b_ess = sizeof(ExpandedFileName)-1;
   /* parse without disk I/O, syntax check only, BUT WITH NOCONCEAL! */
   FileNam.nam$b_nop = NAM$M_SYNCHK | NAM$M_NOCONCEAL;

   if (VMSnok (status = sys$parse (&FileFab, 0, 0)))
      return (status);

   FileNam.nam$l_ver[FileNam.nam$b_ver] = '\0';
   if (Debug) fprintf (stdout, "ExpandedFileName |%s|\n", ExpandedFileName);

   fptr = FileNam.nam$l_name - 1;
   while (fptr > ExpandedFileName && *fptr != '[' && *fptr != '.') fptr--;
   if (fptr > ExpandedFileName && fptr[-1] == ']')
   {
      /* concealed, logical device */
      fptr -= 2;
   }

   sptr = DirFileNamePtr;
   if (!memcmp (fptr, "[000000]", 8) || !memcmp (fptr, "[000000.]", 9))
   {
      for (cptr = FileNam.nam$l_dev; cptr < fptr; *sptr++ = *cptr++);
      for (cptr = "[000000]000000.DIR"; *cptr; *sptr++ = *cptr++);
      *sptr = '\0';
   }
   else
   if (*fptr == '[')
   {
      for (cptr = FileNam.nam$l_dev; cptr < fptr; *sptr++ = *cptr++);
      for (cptr = "[000000]"; *cptr; *sptr++ = *cptr++);
      if (fptr[0] == '.' && fptr[1] == ']')
         fptr += 3;
      else
         fptr++;
      while (*fptr && *fptr != '.' && *fptr != ']') *sptr++ = *fptr++;
      for (cptr = ".DIR"; *cptr; *sptr++ = *cptr++);
      *sptr = '\0';
   }
   else
   {
      for (cptr = FileNam.nam$l_dev; cptr < fptr; *sptr++ = *cptr++);
      *sptr++ = ']';
      if (fptr[0] == '.' && fptr[1] == ']')
         fptr += 3;
      else
         fptr++;
      while (*fptr && *fptr != '.' && *fptr != ']') *sptr++ = *fptr++;
      for (cptr = ".DIR"; *cptr; *sptr++ = *cptr++);
      *sptr = '\0';
   }

   *DirFileLengthPtr = sptr - DirFileNamePtr;

   if (Debug)
      fprintf (stdout, "DirFileNamePtr %d |%s|\n",
               *DirFileLengthPtr, DirFileNamePtr);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/ 
 
int PutResponse
(
struct RequestStruct *rqptr,
char *Description,
char *Path,
char *Disposition,
int FileSize
)
{
   static $DESCRIPTOR (StringDsc, "");

   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ !UL</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ!!</H1>\n\
<P>!AZ\n\
<P>!AZ\n\
</BODY>\n\
</HTML>\n");

   static $DESCRIPTOR (NameFaoDsc, "!AZ &nbsp;<TT>!AZ</TT>&nbsp; !AZ\0");
   static $DESCRIPTOR (NameLinkFaoDsc,
          "!AZ &nbsp;<A HREF=\"!AZ\"><TT>!AZ</TT></A>&nbsp; !AZ\0");
   static $DESCRIPTOR (NameLinkSizeFaoDsc,
"!AZ &nbsp;<A HREF=\"!AZ\"><TT>!AZ</TT></A>&nbsp; !AZ \
(!UL bytes) &nbsp;<TT>(!AZ)</TT>\0");

   register unsigned long  *vecptr;
   register struct PutTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [64];
   char  *StringPtr;
   char  FileInfo [512],
         ProtectionString [32],
         ResponseString [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PutResponse()\n");

   tkptr = rqptr->PutTaskPtr;

   /*
      Strict compliance with HTTP 1.0/1 means this should be a 201 status
      (for PUT at least).  However this inhibits the response informational
      message in at least Netscape Navigator 3.0, so let's revert to 200!
   */
   if (rqptr->HttpMethod == HTTP_METHOD_POST)
      rqptr->ResponseStatusCode = 200;
   else
      rqptr->ResponseStatusCode = 200;
      /** rqptr->ResponseStatusCode = 201; **/

   StringDsc.dsc$a_pointer = FileInfo;
   StringDsc.dsc$w_length = sizeof(FileInfo);

   if (FileSize >= 0)
   {
      FormatProtection (tkptr->ProtectionMask, ProtectionString);
      status = sys$fao (&NameLinkSizeFaoDsc, 0, &StringDsc,
                        Description, Path, Path, Disposition,
                        FileSize, ProtectionString);
   }
   else
   if (FileSize == -1)
      status = sys$fao (&NameFaoDsc, 0, &StringDsc,
                        Description, Path, Disposition);
   else
      status = sys$fao (&NameLinkFaoDsc, 0, &StringDsc,
                        Description, Path, Path, Disposition);

   if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
   if (VMSnok (status))
   {
      ErrorInternal (rqptr, status, "sys$fao()", FI_LI);
      PutEnd (rqptr);
      return;
   }

   rqptr->ResponsePreExpired = PRE_EXPIRE_PUT;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      PutEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, tkptr->FileName);

   *vecptr++ = MsgFor(rqptr,MSG_STATUS_SUCCESS);
   *vecptr++ = rqptr->ResponseStatusCode;
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_SUCCESS);
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_REPORTED_BY_SERVER);

   *vecptr++ = FileInfo;

   StringDsc.dsc$a_pointer = ResponseString;
   StringDsc.dsc$w_length = sizeof(ResponseString);

   status = sys$faol (&ResponseFaoDsc, &Length, &StringDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      PutEnd (rqptr);
      return;
   }
   ResponseString[Length] = '\0';

   /* allocate heap memory for the response header */
   StringPtr = VmGetHeap (rqptr, Length);
   memcpy (StringPtr, ResponseString, Length+1);
   if (Debug) fprintf (stdout, "|%s|\n",rqptr->ResponseHeaderPtr);

   NetWriteBuffered (rqptr, NULL, StringPtr, Length);
}

/*****************************************************************************/

