/*****************************************************************************/
/*
                                Request.c

Get, process and execute the HTTP request from the client.

Some server directives are detected in this module.  Server directives take
the form of a special path or query string and are CASE SENSISTIVE.

  ?http=...                   any query string beginning "httpd=" will be
                              ignored by the default query script and the HTTPd
                              will pass it to the requested functionality

  /httpd/-/admin/             generates a server administration menu
  /httpd/-/admin/graphic/...  generates graphics
  /httpd/-/admin/report/...   generates various reports
  /httpd/-/admin/revise/...   allows form-based configuration
  /httpd/-/admin/control/...  allows server to be stopped, restarted, etc.

  /echo/                      echoes complete request (header and any body)
  /tree/                      directory tree
  /upd/                       activates the update module
  /where/                     reports mapped and parsed path
  /Xray/                      returns an "Xray" of the request (i.e. header
                              and body of response as a plain-text document)

A request for an HTTP/1.0 persistent connection is looked for in the request
header and if found a boolean set to indicate the request.

AUTHORIZATION MAY BE PERFORMED TWICE!

The first, to authorize the full path provided with the request (after it
has been mapped!)  The second, if the request path contained a script any
path specification following the script component is also checked.  In this
way access to controlled information is not inadvertantly allowed because
the script itself is not controlled.  In general, script specifications
should not be part of an authorized path if that script is used to access
other paths (i.e. does not return data generated by itself), apply
authorization to data paths.

Error counting within this module attempts to track the reason that any 
connection accepted fails before being passsed to another module for 
processing (i.e. request format error, path forbidden by rule).


VERSION HISTORY
---------------
12-JUL-98  MGD  bugfix; RequestEnd() no status returned from RequestBegin()!
14-MAY-98  MGD  ?httpd=... generalized from index processing to all requests
02-APR-98  MGD  no longer log internal redirects
28-MAR-98  MGD  declare an AST for local redirection parsing
                (in case a CGIplus script redirected and is exiting!)
28-JAN-98  MGD  moved more appropriate functions from HTTPd.C into here,
                header parsing now allows for hiatus in end-header blank line,
                allow for directories specified as "/dir1/dir2"
07-JAN-98  MGD  provide URL-encoded decode on path
05-OCT-97  MGD  file cache,
                added "Accept-Charset:", "Forwarded:" and "Host:"
18-SEP-97  MGD  HTTP status code mapping
09-AUG-97  MGD  message database
27-JUL-97  MGD  modified "Accept:" header lines processing
08-JUN-97  MGD  added "Pragma:" header field detection
27-MAR-97  MGD  added "temporary" file detection (for UPD/PUT preview)
01-FEB-97  MGD  HTTPd version 4
01-OCT-96  MGD  added more reports
25-JUL-96  MGD  use optional "length=" within "If-Modified-Since:" header
12-APR-96  MGD  RMS parse structures moved to thread data;
                persistent connections ("keep-alive");
                changed internal directive from query string to path;
                observed Multinet disconnection/zero-byte behaviour
                (request now aborts if Multinet returns zero bytes)
01-DEC-95  MGD  HTTPd version 3
27-SEP-95  MGD  extensive rework of some functions;
                added 'Referer:', 'User-Agent:', 'If-Modified-Since:'
01-APR-95  MGD  initial development for addition to multi-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application header files */
#include "wasd.h"
#include "admin.h"
#include "config.h"
#include "control.h"
#include "dcl.h"
#include "dir.h"
#include "error.h"
#include "file.h"
#include "httpd.h"
#include "ismap.h"
#include "menu.h"
#include "mapurl.h"
#include "msg.h"
#include "net.h"
#include "request.h"
#include "ssi.h"
#include "support.h"
#include "vm.h"

/******************/
/* global storage */
/******************/

struct  ListHeadStruct  RequestList;
struct  ListHeadStruct  RequestHistoryList;
int  RequestHistoryCount,
     RequestHistoryMax;

char  ErrorTimeSanityCheck[] = "Time sanity check failure!";

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern boolean  CacheEnabled;
extern boolean  ControlExitRequested;
extern boolean  ControlRestartRequested;
extern boolean  LoggingEnabled;
extern boolean  MonitorEnabled;
extern int  ExitStatus;
extern int  NetReadBufferSize;
extern int  OutputBufferSize;
extern int  CurrentConnectCount;
extern char  ConfigContentTypeIsMap[];
extern char  ConfigContentTypeMenu[];
extern char  ConfigContentTypeSsi[];
extern char  HtmlSgmlDoctype[];
extern char  HttpProtocol[];
extern char  ServerHostPort[];
extern char  SoftwareID[];
extern char  Utility[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/*****************************************************************************/
/*
This function can be called from two sources.  First, NetAccept() where a new
connection has been established and memory has just been allocated for this
connection's thread structure.  Second, from RequestEnd() where a
persistent connection (HTTP/1.0) is involved.

Both then queue a read from the network connection which calls an AST
function when the read completes, to process the request.  The keep-alive
read usually has a far shorter timeout associated with it so that rapid,
sequential reads (as happens when a document is being populated with images
for instance) are serviced from the one connection, but these are not left
open consuming resources on the off-chance of another request in the
not-too-distance future.

Persistent connections are implemented by disposing of all memory allocated
on a request's heap, and all data in the connection thread structure by
zeroing, EXCEPT that above the field 'RetainAboveZeroBelow' (client
connection information), then treating the almost-amnesic structure like a
brand-new connection (only with a shorter timeout period).

Every hour recheck the GMT time offset to crudely detect daylight saving and
other timezone changes.
*/ 

RequestBegin
(
struct RequestStruct *rqptr,
boolean  NewConnection
)
{
   static int  PrevDay = -1,
               PrevHour = -1,
               PrevMonth = -1,
               PrevYear = -1;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RequestBegin()\n");

   if (NewConnection)
   {
      /******************/
      /* new connection */
      /******************/

      /* add entry to the top of the request list */
      ListAddHead (&RequestList, rqptr);

      /* timestamp the transaction */
      sys$gettim (&rqptr->BinaryTime);
      sys$numtim (&rqptr->NumericTime, &rqptr->BinaryTime);
      HttpGmTimeString (rqptr->GmDateTime, &rqptr->BinaryTime);

      if (PrevYear != -1)
      {
         /* time sanity check with each transaction! */
         if (PrevYear > rqptr->NumericTime[0] ||
             (PrevYear ==  rqptr->NumericTime[0] &&
              PrevMonth > rqptr->NumericTime[1]) ||
             (PrevYear ==  rqptr->NumericTime[0] &&
              PrevMonth == rqptr->NumericTime[1] &&
              PrevDay > rqptr->NumericTime[2]))
            ErrorExitVmsStatus (0, ErrorTimeSanityCheck, FI_LI);
      }

      if (PrevHour != rqptr->NumericTime[3] ||
          PrevDay != rqptr->NumericTime[2] ||
          PrevMonth != rqptr->NumericTime[1] ||
          PrevYear != rqptr->NumericTime[0])
      {
         /* every hour recheck the GMT time offset */
         if (VMSnok (status = TimeSetGmt ()))
            ErrorExitVmsStatus (status, "TimeSetGmt()", FI_LI);
         PrevHour = rqptr->NumericTime[3];
         PrevDay = rqptr->NumericTime[2];
         PrevMonth = rqptr->NumericTime[1];
         PrevYear = rqptr->NumericTime[0];
      }

      /* initialize the timer for input */
      HttpdTimerSet (rqptr, TIMER_INPUT);

      /* initialize the HTTPd supervisor (if it's not already running) */
      HttpdSupervisor (false);

      if (rqptr->RequestScheme == SCHEME_HTTPS)
      {
         /* Secure Sockets Layer ("https://") transaction */
         SeSoLaBegin (rqptr);
         SeSoLaAccept (rqptr);
         return;
      }
   }
   else
   {
      /*************************/
      /* persistent connection */
      /*************************/

      memset ((unsigned char*)&rqptr->RetainAboveZeroBelow, 0,
              ((unsigned char*)&rqptr->ZeroedEnd -
               (unsigned char*)&rqptr->RetainAboveZeroBelow));

      /*
         Timestamp the keep-alive period for RequestReport()
         Transaction will again be time-stamped if/when request received!
      */
      sys$gettim (&rqptr->BinaryTime);

      /* initialize the timer for keep-alive */
      HttpdTimerSet (rqptr, TIMER_KEEPALIVE);
   }

   if (Debug) fprintf (stdout, "KeepAliveCount: %d\n", rqptr->KeepAliveCount);

   rqptr->NetReadBufferPtr = VmGetHeap (rqptr, NetReadBufferSize);
   rqptr->NetReadBufferSize = NetReadBufferSize;

   NetRead (rqptr, &RequestGet,
            rqptr->NetReadBufferPtr, rqptr->NetReadBufferSize);
}

/*****************************************************************************/
/*
This function may be AST-delivery executed ONE OR MORE TIMES, FROM ITSELF, 
before a connection is actually disposed of.

If a next-task function was specified, and there has been no error message
generated, then execute it.

If there are any characters remaining in the output buffer then flush it with
an AST leading straight back to this function!

If an error has been generated then send it to the client once again with an
AST leading straight back to this function!

Check for redirection.  If local redirection do NOT dispose of the thread
(we'll be re-using the thread context).

If a persistent connection then treat it like a new connection and go back
to listening for the next request.

Otherwise close the client network socket, release dynamic data structures
allocated, etc.
*/

RequestEnd (struct RequestStruct *rqptr)

{
   static long  Addx2 = 2,
                Subx2 = 2,
                EdivTen = 10;
   int  status,
        StatusCodeGroup;
   unsigned long  Remainder;
   unsigned long  BinaryTime [2],
                  QuadScratch [2],
                  ResultTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RequestEnd()\n");

   /*
      Flush anything currently remaining in the thread's output buffer.
      AST back to this same function to continue disposing of the thread!
   */
   if (rqptr->OutputBufferCount)
   {
      NetWriteBuffered (rqptr, &RequestEnd, NULL, 0);
      return;
   }

   /*
      If an error has been reported send this to the client.
      AST back to this same function to continue disposing of the thread!
   */
   if (rqptr->ErrorMessagePtr != NULL)
      if (VMSok (status = ErrorSendToClient (rqptr, &RequestEnd)))
         return;

   /*
      If the response header has not yet been sent then do it.
      AST back to this same function to continue disposing of the thread!
   */
   if (!rqptr->ResponseHeaderSent)
   {
      NetWrite (rqptr, &RequestEnd, 0, 0);
      return;
   }

   /*************************/
   /* response elapsed time */
   /*************************/

   if (!rqptr->KeepAliveTimeout)
   {
      sys$gettim (&BinaryTime);
      status = lib$subx (&BinaryTime, &rqptr->BinaryTime, 
                         &ResultTime, &Subx2);
      /* convert to microseconds (longword becomes 71 minutes max) */
      if (VMSok (status))
         status = lib$ediv (&EdivTen, &ResultTime,
                            &rqptr->ResponseDuration, &Remainder);
      if (Debug) fprintf (stdout, "elapsed: %%X%08.08X\n", status);
      if (VMSnok (status)) rqptr->ResponseDuration = 0;

      Accounting.ResponseDurationCount++;
      QuadScratch[0] = rqptr->ResponseDuration;
      QuadScratch[1] = 0;
      lib$addx (&QuadScratch, &Accounting.QuadResponseDuration,
                &Accounting.QuadResponseDuration, &Addx2);
      if (rqptr->ResponseDuration < Accounting.ResponseDurationMin)
         Accounting.ResponseDurationMin = rqptr->ResponseDuration;
      if (rqptr->ResponseDuration > Accounting.ResponseDurationMax)
         Accounting.ResponseDurationMax = rqptr->ResponseDuration;
   }

   /***************/
   /* redirection */
   /***************/

   if (rqptr->LocationPtr != NULL)
   {
      /* if local do NOT dispose of thread */
      if (VMSok (status = RequestRedirect (rqptr)))
      {
         /* AST call the parsing function to begin re-processing the request */
         SysDclAst (RequestParseAndExecute, rqptr);
         return;
      }

      /*
         If a genuine error reported when redirecting send this to the client.
         AST back to this same function to continue disposing of the thread!
      */
      if (rqptr->ErrorMessagePtr != NULL)
      {
         if (VMSok (ErrorSendToClient (rqptr, &RequestEnd)))
            return;
      }
      else
      {
         /*
            A status of SS$_UNREACHABLE is returned when non-local redirect.
            The network write will AST back to this same function to continue.
         */
         if (status == SS$_UNREACHABLE) return;
      }
   }

   /**************************/
   /* dispose of the REQUEST */
   /**************************/

   /* if Secure Sockets Layer request */
   if (rqptr->SeSoLaPtr != NULL)
   {
      /*
         If waiting on shutdown I/O just return, RequestEnd() will
         be called by SeSoLaEnd() to further conclude processing when that
         SSL shutdown I/O is complete.
      */
      if (!SeSoLaEnd (rqptr)) return;
      /* now free any SSLeay structures */
      SeSoLaFree (rqptr);
   }

   if (Debug) fprintf (stdout, "rqptr->BytesTx: %d\n", rqptr->BytesTx);

   /* a little more accounting */
   Accounting.ConnectCurrent = CurrentConnectCount-1;
   QuadScratch[0] = rqptr->BytesTx;
   QuadScratch[1] = 0;
   lib$addx (&QuadScratch, &Accounting.QuadBytesTx,
             &Accounting.QuadBytesTx, &Addx2);
   QuadScratch[0] = rqptr->BytesRx;
   QuadScratch[1] = 0;
   lib$addx (&QuadScratch, &Accounting.QuadBytesRx,
             &Accounting.QuadBytesRx, &Addx2);

   /* set logical names used by HTTPd monitor utility */
   if (MonitorEnabled) DefineMonitorLogicals (rqptr);

   if (!rqptr->KeepAliveTimeout)
   {
      StatusCodeGroup = rqptr->ResponseStatusCode / 100;
      if (StatusCodeGroup < 0 || StatusCodeGroup > 5) StatusCodeGroup = 0;
      Accounting.ResponseStatusCodeCountArray[StatusCodeGroup]++;

      if (LoggingEnabled) Logging (rqptr, LOGGING_ENTRY);

      /* if a request history is being kept then provide the entry */
      if (RequestHistoryMax) RequestHistory (rqptr);

      /* if keeping activity statistics */
      if (Config.ActivityNumberOfDays) GraphActivityIncrement (rqptr);
   }

   /* free dynamic memory allocated for the request's heap */
   VmFreeHeap (rqptr);

   if (rqptr->KeepAliveRequest &&
       rqptr->KeepAliveResponse &&
       rqptr->KeepAliveCount < MaxKeepAliveCount)
   {
      /*************************/
      /* PERSISTENT connection */
      /*************************/

      RequestBegin (rqptr, false);
      return;
   }

   /*****************************/
   /* dispose of the CONNECTION */
   /*****************************/

   if (rqptr->ClientSocketShutdown)
      RequestEndConnection (rqptr);
   else
      NetShutdownSocket (rqptr, &RequestEndConnection, true);
}

/*****************************************************************************/
/*
*/

int RequestEndConnection (struct RequestStruct *rqptr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RequestEndConnection() %d\n", rqptr);

   NetCloseSocket (rqptr);

   /* remove from the request list */
   ListRemove (&RequestList, rqptr);

   /* free dynamic memory allocated for the connection structure */
   VmFreeRequest (rqptr);

   /* one less to worry about! */
   if (CurrentConnectCount) CurrentConnectCount--;

   /* don't bother checking for exit or restart if connections still exist */
   if (CurrentConnectCount) return;

   /* if the control module has requested server exit or restart */
   if (ControlExitRequested)
   {
      fprintf (stdout, "%%%s-I-CONTROL, delayed server exit\n", Utility);
      ExitStatus = SS$_NORMAL;
      HttpdExit();
      sys$delprc (0, 0);
   }
   else
   if (ControlRestartRequested)
   {
      fprintf (stdout, "%%%s-I-CONTROL, delayed server restart\n", Utility);
      exit (SS$_NORMAL);
   }

   HttpdCheckPriv ();
}

/****************************************************************************/
/*
The 'rqptr->LocationPtr' is non-NULL indicating redirection.  This 
can be local, indicated by a leading '/', or it can be non-local, indicated
by anything other than a leading '/', usually a full URL including scheme
(e.g. "http:").

If authorization is enabled then all redirected requests should be returned
to the client in case authorization information needs to be applied to the
new request.  This may involve reformating a local redirection into a
non-local one.  If authentication is not enabled then check both local and
non-local redirections to see if it can be handled locally.

Return normal status to indicate local-redirection (and that the thread
should NOT be disposed of), or an error status to indicate client-handled,
non-local-redirection, or a legitimate error (and that the thread can go).
*/ 

int RequestRedirect (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (BufferDsc, "");

   static char  ContentLength [32];
   static $DESCRIPTOR (contentLengthDsc, ContentLength);
   static $DESCRIPTOR (ContentLengthFaoDsc, "Content-Length: !UL\r\n\0");

   static $DESCRIPTOR (Http302FaoDsc,
"!AZ 302 Redirection\r\n\
Location: !AZ\r\n\
Server: !AZ\r\n\
Date: !AZ\r\n\
\r\n\0");

   static $DESCRIPTOR (LocalFaoDsc,
"!AZ !AZ !AZ\r\n\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ!AZ!AZ\
!AZ\
!AZ\
\r\n\0");

   register unsigned long  *vecptr;
   register char  *cptr, *sptr;

   boolean  LocalRedirection;
   int  status,
        PortNumber;
   unsigned short  Length;
   unsigned long  FaoVector [64];
   char  *HttpMethodNamePtr;
   char  HttpMethodName [32],
         RedirectionUrl [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RequestRedirect() |%s|\n", rqptr->LocationPtr);

   if (rqptr->RedirectionCount++ > 1)
   {
      rqptr->LocationPtr = NULL;
      rqptr->ResponseStatusCode = 500;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_REDIRECTION), FI_LI);
      return (STS$K_ERROR);
   }

   rqptr->ResponseStatusCode = 302;

   /*
      A leading space is normally not possible.
      This indicates a specific HTTP method must be used.
      Format is: "<space>METHOD<space><redirection-path>".
   */
   if (rqptr->LocationPtr[0] == ' ')
   {
      HttpMethodNamePtr = sptr = HttpMethodName;
      for (cptr = rqptr->LocationPtr + 1;
           *cptr && *cptr != ' ';
           *sptr++ = *cptr++);
      *sptr = '\0';
      rqptr->LocationPtr = cptr + 1;
   }
   else
      HttpMethodNamePtr = rqptr->HttpMethodName;

   MapUrl_VirtualPath (rqptr->PathInfoPtr, rqptr->LocationPtr,
                       RedirectionUrl, sizeof(RedirectionUrl));

   /* now ignore old value of location redirection pointer */
   rqptr->LocationPtr = NULL;

#ifdef ALWAYS_LOCAL_REDIRECTION_IF_DETECTED

   LocalRedirection = false;

   if (RedirectionUrl[0] == '/')
      LocalRedirection = true;
   else
   {
      /*****************************/
      /* check for local host/port */
      /*****************************/

      if (!memcmp (cptr = RedirectionUrl, "http:", 5) ||
          !memcmp (cptr = RedirectionUrl, "https:", 6))
      {
         /* scan over the "http://" or "https://" */
         while (*cptr && *cptr != '/') cptr++;
         while (*cptr && *cptr == '/') cptr++;
         if (Debug)
            fprintf (stdout, "cptr |%s|%s|\n",
                     cptr, rqptr->ServicePtr->ServerHostName);
         if (strsame (cptr,
                      rqptr->ServicePtr->ServerHostName,
                      rqptr->ServicePtr->ServerHostNameLength) &&
             (cptr[rqptr->ServicePtr->ServerHostNameLength] == '/' ||
              cptr[rqptr->ServicePtr->ServerHostNameLength] == ':'))
         {
            cptr += rqptr->ServicePtr->ServerHostNameLength;
            if (*cptr == '/')
               LocalRedirection = true;
            else
            if (*cptr == ':')
            {
               PortNumber = atoi(++cptr);
               if (Debug) fprintf (stdout, "PortNumber: %d\n", PortNumber);
               if (PortNumber == ServerPort) LocalRedirection = true;
            }
            while (*cptr && *cptr != '/') cptr++;
         }

         if (LocalRedirection)
         {
            /* eliminate host details, copy just the path over the host detail */
            sptr = RedirectionUrl;
            while (*cptr) *sptr++ = *cptr++;
            *sptr = '\0';
            if (Debug) fprintf (stdout, "RedirectUrl |%s|\n", RedirectionUrl);
         }
      }
   }

#else

   if (RedirectionUrl[0] == '/')
      LocalRedirection = true;
   else
      LocalRedirection = false;

#endif

   if (LocalRedirection)
   {
      /*********************/
      /* local redirection */
      /*********************/

      Accounting.RedirectLocalCount++;

      vecptr = &FaoVector;

      *vecptr++ = HttpMethodNamePtr;
      *vecptr++ = RedirectionUrl;
      *vecptr++ = HttpProtocol;

      if (rqptr->HttpAcceptPtr != NULL)
      {
         *vecptr++ = "Accept: ";
         *vecptr++ = rqptr->HttpAcceptPtr;
         *vecptr++ = "\r\n";
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
      }

      if (rqptr->HttpAcceptLangPtr != NULL)
      {
         *vecptr++ = "Accept-Language: ";
         *vecptr++ = rqptr->HttpAcceptLangPtr;
         *vecptr++ = "\r\n";
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
      }

      if (rqptr->HttpAcceptCharsetPtr != NULL)
      {
         *vecptr++ = "Accept-Charset: ";
         *vecptr++ = rqptr->HttpAcceptCharsetPtr;
         *vecptr++ = "\r\n";
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
      }

      if (rqptr->HttpUserAgentPtr != NULL)
      {
         *vecptr++ = "User-Agent: ";
         *vecptr++ = rqptr->HttpUserAgentPtr;
         *vecptr++ = "\r\n";
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
      }

      if (rqptr->HttpRefererPtr != NULL)
      {
         *vecptr++ = "Referer: ";
         *vecptr++ = rqptr->HttpRefererPtr;
         *vecptr++ = "\r\n";
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
      }

      if (rqptr->HttpHostPtr != NULL)
      {
         *vecptr++ = "Host: ";
         *vecptr++ = rqptr->HttpHostPtr;
         *vecptr++ = "\r\n";
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
      }

      if (rqptr->HttpAuthorizationPtr != NULL)
      {
         *vecptr++ = "Authorization: ";
         *vecptr++ = rqptr->HttpAuthorizationPtr;
         *vecptr++ = "\r\n";
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
      }

      if (rqptr->HttpIfModifiedSincePtr != NULL)
      {
         *vecptr++ = "If-Modified-Since: ";
         *vecptr++ = rqptr->HttpIfModifiedSincePtr;
         *vecptr++ = "\r\n";
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
      }

      if (rqptr->HttpPragmaPtr != NULL)
      {
         *vecptr++ = "Pragma: ";
         *vecptr++ = rqptr->HttpPragmaPtr;
         *vecptr++ = "\r\n";
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
      }

      if (rqptr->HttpCookiePtr != NULL)
      {
         *vecptr++ = "Cookie: ";
         *vecptr++ = rqptr->HttpCookiePtr;
         *vecptr++ = "\r\n";
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
      }

      if (rqptr->HttpForwardedPtr != NULL)
      {
         *vecptr++ = "Forwarded: ";
         *vecptr++ = rqptr->HttpForwardedPtr;
         *vecptr++ = "\r\n";
      }
      else
      {
         *vecptr++ = "";
         *vecptr++ = "";
         *vecptr++ = "";
      }

      if (rqptr->ContentLength)
      {
         sys$fao (&ContentLengthFaoDsc, 0, ContentLength,
                  rqptr->ContentLength);
         *vecptr++ = ContentLength;
      }
      else
         *vecptr++ = "";

      if (rqptr->KeepAliveRequest)
         *vecptr++ = "Connection: Keep-Alive\r\n";
      else
         *vecptr++ = "";

      BufferDsc.dsc$a_pointer = rqptr->NetReadBufferPtr;
      BufferDsc.dsc$w_length = rqptr->NetReadBufferSize;

      status = sys$faol (&LocalFaoDsc, &Length, &BufferDsc, &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->LocationPtr = NULL;
         rqptr->ErrorTextPtr = "sys$faol()";
         rqptr->ResponseStatusCode = 500;
         ErrorVmsStatus (rqptr, status, FI_LI);
         return (STS$K_ERROR);
      }
      if (Debug) fprintf (stdout, "|%s|\n", rqptr->NetReadBufferPtr);

      /* request structure back to empty */
      memset ((unsigned char*)&rqptr->RetainAboveZeroBelow, 0,
              ((unsigned char*)&rqptr->ZeroedEnd -
               (unsigned char*)&rqptr->RetainAboveZeroBelow));

      /* re-timestamp the transaction */
      sys$gettim (&rqptr->BinaryTime);
      sys$numtim (&rqptr->NumericTime, &rqptr->BinaryTime);
      HttpGmTimeString (rqptr->GmDateTime, &rqptr->BinaryTime);

      /* initialize the timer for input */
      HttpdTimerSet (rqptr, TIMER_INPUT);

      /* use the previous buffer in which the new request has been put */
      rqptr->NetReadBufferPtr = BufferDsc.dsc$a_pointer;
      rqptr->NetReadBufferSize = BufferDsc.dsc$w_length;

      rqptr->RequestHeaderPtr = rqptr->NetReadBufferPtr;
      rqptr->RequestHeaderLength = Length;

      /* normal status indicates its a local redirection */
      return (SS$_NORMAL);
   }
   else
   {
      /*************************/
      /* non-local redirection */
      /*************************/

      Accounting.RedirectRemoteCount++;

      BufferDsc.dsc$a_pointer = rqptr->NetReadBufferPtr;
      BufferDsc.dsc$w_length = rqptr->NetReadBufferSize;

      status = sys$fao (&Http302FaoDsc, &Length, &BufferDsc,
                        HttpProtocol, RedirectionUrl, SoftwareID,
                        rqptr->GmDateTime);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->LocationPtr = NULL;
         rqptr->ErrorTextPtr = "sys$fao()";
         rqptr->ResponseStatusCode = 500;
         ErrorVmsStatus (rqptr, status, FI_LI);
         return (STS$K_ERROR);
      }
      if (Debug) fprintf (stdout, "|%s|\n", rqptr->NetReadBufferPtr);
      NetWrite (rqptr, &RequestEnd, rqptr->NetReadBufferPtr, Length-1);

      /* unreachable indicates it's non-local redirection :^) */
      return (SS$_UNREACHABLE);
   }
}

/*****************************************************************************/
/*
Process first packet read from the client.  In most cases this will contain 
the complete HTTP header and it can be processed without building up a 
buffered header.  If it does not contain the full header allocate heap memory
to contain the current packet and queue subsequent read(s), expanding the 
request header heap memory, until all is available (or it becomes completely 
rediculous!).
*/ 
 
RequestGet (struct RequestStruct *rqptr)

{
   register int  cnlcnt, cnt;
   register char  *cptr, *sptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
"RequestGet() Status: %%X%08.08X Count: %d BufferSize: %d HeaderLength: %d\n",
               rqptr->NetReadIOsb.Status,
               rqptr->NetReadIOsb.Count,
               rqptr->NetReadBufferSize,
               rqptr->RequestHeaderLength);

   /* zero bytes with a normal status is a definite no-no */
   if (VMSok (rqptr->NetReadIOsb.Status) && !rqptr->NetReadIOsb.Count)
      rqptr->NetReadIOsb.Status = SS$_ABORT;

   if (VMSnok (rqptr->NetReadIOsb.Status))
   {
      /*
         Most common cause will be broken connection.  Most common reason
         for the break ... "keep-alive" timeout on persistent connection.
         Don't bother to report!
      */
      if (rqptr->KeepAliveCount) rqptr->KeepAliveTimeout = true;
      RequestEnd (rqptr);
      return;
   }

   if (rqptr->KeepAliveCount)
   {
      /* request is underway, reinitialize for input (longer period) */
      HttpdTimerSet (rqptr, TIMER_INPUT);

      /* re-timestamp the kept-alive transaction */
      sys$gettim (&rqptr->BinaryTime);
      sys$numtim (&rqptr->NumericTime, &rqptr->BinaryTime);
      HttpGmTimeString (rqptr->GmDateTime, &rqptr->BinaryTime);
   }

   rqptr->BytesRx += (cnt = rqptr->NetReadIOsb.Count);

   if (rqptr->NetReadBufferSize >= 8192)
   {
      /* not likely to be a legitimate HTTP request! */
      RequestEnd (rqptr);
      return;
   }

   if (rqptr->RequestHeaderPtr == NULL)
   {
      rqptr->RequestHeaderPtr = rqptr->NetReadBufferPtr;
      rqptr->RequestHeaderLength = 0;
   }

   rqptr->RequestHeaderPtr[cnt] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", rqptr->RequestHeaderPtr); 

   /* look for the end of header blank line */
   cnlcnt = rqptr->HeaderConsecutiveNewLineCount;
   cptr = rqptr->RequestHeaderPtr;
   if (Debug) fprintf (stdout, "cnlcnt: %d *cptr: %d\n", cnlcnt, *cptr);
   while (*cptr && cnlcnt < 2)
   {
      if (*cptr == '\r') cptr++;
      if (!*cptr) break;
      if (*cptr != '\n')
      {
         cptr++;
         cnlcnt = 0;
         continue;
      }
      cptr++;
      cnlcnt++;
   }
   if (Debug) fprintf (stdout, "cnlcnt: %d\n", cnlcnt);

   if (cnlcnt >= 2)
   {
      /* point (possibly back) to the start of the header */
      rqptr->RequestHeaderPtr = rqptr->NetReadBufferPtr;
      /* make the header length the actual length of the header! */
      rqptr->RequestHeaderLength = cptr - (char*)rqptr->NetReadBufferPtr;

      RequestParseAndExecute (rqptr);

      return;
   }

   rqptr->HeaderConsecutiveNewLineCount = cnlcnt;

   /**********************************************/
   /* request header not completely received yet */
   /**********************************************/

   rqptr->RequestHeaderLength += cnt;

   if (rqptr->RequestHeaderLength >= rqptr->NetReadBufferSize)
   {
      /* need more buffer space, allow one for termination of the buffer */
      rqptr->NetReadBufferSize += NetReadBufferSize;
      rqptr->NetReadBufferPtr =
         VmReallocHeap (rqptr, rqptr->NetReadBufferPtr,
                        rqptr->NetReadBufferSize+1);
   }

   /* because a realloc() may move the memory recalculate the pointer */
   rqptr->RequestHeaderPtr = rqptr->NetReadBufferPtr +
                             rqptr->RequestHeaderLength;

   /* 
      Queue an asynchronous read to get more of the header from the client.
      When the read completes call RequestSubsequent() AST completion
      function again to further process the request.
   */
   NetRead (rqptr, &RequestGet, rqptr->RequestHeaderPtr,
            rqptr->NetReadBufferSize - rqptr->RequestHeaderLength);

   /* further processing of the request will be AST-driven */
}

/****************************************************************************/
/*
Get the method, path and query string, and some header field line values, then
execute the request.  This function can be called from two points.  First, from
the function RequestGet(), after it has received a complete HTTP header from
the client.  Second, from the function RequestRedirect(), where an HTTP header
has been reconstructed in the 'NetReadBufferPtr' in order to effect a local
redirection.
*/

RequestParseAndExecute (struct RequestStruct *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout, "RequestParseAndExecute() size: %d\n",
               rqptr->RequestHeaderLength);
/**
      fprintf (stdout, "|%.*s|\n",
               rqptr->RequestHeaderLength, rqptr->RequestHeaderPtr);
**/
   }

   Accounting.RequestTotalCount++;
   Accounting.RequestParseCount++;
   if (rqptr->KeepAliveCount)
      Accounting.RequestKeepAliveCount++;
   if (rqptr->KeepAliveCount > Accounting.RequestKeepAliveMax)
      Accounting.RequestKeepAliveMax = rqptr->KeepAliveCount;  

   if (!RequestMethodPathQuery (rqptr)) return;
   if (!RequestFields (rqptr)) return;

   /* evaluated in order of probable occurance */
   if (!strcmp (rqptr->HttpMethodName, "GET"))
   {
      Accounting.MethodGetCount++;
      rqptr->HttpMethod = HTTP_METHOD_GET;
      RequestExecute (rqptr);
      return;
   }
   if (!strcmp (rqptr->HttpMethodName, "PUT"))
   {
      Accounting.MethodPutCount++;
      rqptr->HttpMethod = HTTP_METHOD_PUT;
      rqptr->KeepAliveRequest = false;
      RequestBodyReadBegin (rqptr);
      return;
   }
   if (!strcmp (rqptr->HttpMethodName, "POST"))
   {
      Accounting.MethodPostCount++;
      rqptr->HttpMethod = HTTP_METHOD_POST;
      rqptr->KeepAliveRequest = false;
      RequestBodyReadBegin (rqptr);
      return;
   }
   if (!strcmp (rqptr->HttpMethodName, "DELETE"))
   {
      Accounting.MethodDeleteCount++;
      rqptr->HttpMethod = HTTP_METHOD_DELETE;
      rqptr->KeepAliveRequest = false;
      RequestExecute (rqptr);
      return;
   }
   if (!strcmp (rqptr->HttpMethodName, "HEAD"))
   {
      Accounting.MethodHeadCount++;
      rqptr->HttpMethod = HTTP_METHOD_HEAD;
      RequestExecute (rqptr);
      return;
   }

   Accounting.RequestErrorCount++;
   rqptr->ResponseStatusCode = 501;
   ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_METHOD), FI_LI);
   RequestEnd (rqptr);
}

/*****************************************************************************/
/*
Read the body of a request (POST, PUT, etc., methods) into a buffer of heap
memory pointed to by 'rqptr->ContentBufferPtr'.
*/ 
 
RequestBodyReadBegin (struct RequestStruct *rqptr)

{
   int  status,
        ReadSize;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "RequestBodyReadBegin() %d bytes\n",
               rqptr->ContentLength);

   if (rqptr->ContentBufferPtr != NULL)
   {
      /********************************/
      /* body exists from redirection */
      /********************************/

      RequestExecute (rqptr);
      return;
   }

   /****************************************/
   /* request body allowed to be this big? */
   /****************************************/

   if (Config.PutMaxKbytes &&
       rqptr->ContentLength / 1000 > Config.PutMaxKbytes)
   {
      /* local storage */
      char  String [128];
      $DESCRIPTOR (MaxLengthFaoDsc, MsgFor(rqptr,MSG_REQUEST_BODY_MAX));
      $DESCRIPTOR (StringDsc, String);

      MaxLengthFaoDsc.dsc$w_length = strlen(MsgFor(rqptr,MSG_REQUEST_BODY_MAX));
      status = sys$fao (&MaxLengthFaoDsc, 0, &StringDsc,
                        rqptr->HttpMethodName, rqptr->ContentLength / 1000,
                        Config.PutMaxKbytes);
      if (VMSnok (status))
      {
         rqptr->ErrorTextPtr = "sys$fao()";
         rqptr->ErrorHiddenTextPtr = "(possible msg problem)";
         ErrorVmsStatus (rqptr, RMS$_FNF, FI_LI);
      }
      else
      {
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, String, FI_LI);
      }

      /*
         Navigator 3 and MSIE 3 (all browsers?) spit if a POST is not read
         before returning an error, hence continue on if not an unreasonable
         number of bytes!  Disable this by defining NO_UPLOAD_GRACE.
         10 Mbytes doesn't seem an unreasonable limit!
      */

#ifndef NO_UPLOAD_GRACE
      if (rqptr->ContentLength < 10000000)
      {
         /* fake these for the discard AST routine checking */
         rqptr->NetReadIOsb.Status = SS$_NORMAL;
         rqptr->NetReadIOsb.Count = 1;

         if (VMSnok (status = ErrorSendToClient (rqptr, &RequestDiscardAst)))
            RequestEnd (rqptr);
      }
      else
#endif /* NO_UPLOAD_GRACE */
         if (VMSnok (status = ErrorSendToClient (rqptr, &RequestEnd)))
            RequestEnd (rqptr);

      return;
   }

   /****************************/
   /* allocate buffer for body */
   /****************************/

   /*
      We're going to stick an extra null char on the end
      so we can parse it as if it's a string if we want to!
   */
   rqptr->ContentBufferPtr = VmGetHeap (rqptr, rqptr->ContentLength+1);
   /* might as well do the above now! */
   rqptr->ContentBufferPtr[rqptr->ContentLength] = '\0';

   /*  allow for any content received along with the header */
   rqptr->ContentCount = rqptr->BytesRx - rqptr->RequestHeaderLength;
   /* ensure this does not exceed the specified content length */
   if (rqptr->ContentCount > rqptr->ContentLength)
      rqptr->ContentCount = rqptr->ContentLength;

   if (Debug)
      fprintf (stdout,
"ContentLength: %d BytesRx: %d RequestHeaderLength: %d ContentCount: %d\n",
          rqptr->ContentLength, rqptr->BytesRx,
          rqptr->RequestHeaderLength, rqptr->ContentCount);

   if (rqptr->ContentCount)
   {
      /* copy into the buffer any content received along with the header */
      memcpy (rqptr->ContentBufferPtr,
              rqptr->NetReadBufferPtr + rqptr->RequestHeaderLength,
              rqptr->ContentCount);
   }

   if (rqptr->ContentCount >= rqptr->ContentLength)
   {
      /******************************************************/
      /* looks like it all arrived with the request header! */
      /******************************************************/

      RequestExecute (rqptr);
      return;
   }

   /*******************************/
   /* read (more) from the client */
   /*******************************/

   rqptr->ContentPtr = rqptr->ContentBufferPtr + rqptr->ContentCount;
   if (rqptr->ContentLength - rqptr->ContentCount > 32767)
      ReadSize = 32767;
   else
      ReadSize = rqptr->ContentLength - rqptr->ContentCount;

   NetRead (rqptr, &RequestBodyReadAst, rqptr->ContentPtr, ReadSize);
}

/*****************************************************************************/
/*
A read from the client directly into the content buffer has completed.  Check
the status of that read.  If ok check whether all of the content body has been
read.  If not read more of the content body, otherwise declare an AST to
execute the next task.
*/ 

RequestBodyReadAst (struct RequestStruct *rqptr)

{
   int  status,
        ReadSize;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
         "RequestBodyReadAst() NetReadIOsb.Status %%X%08.08X .Count %d\n",
         rqptr->NetReadIOsb.Status, rqptr->NetReadIOsb.Count);

   /* zero bytes with a normal status is a definite no-no (TGV-Multinet) */
   if (VMSok (rqptr->NetReadIOsb.Status) && !rqptr->NetReadIOsb.Count)
      rqptr->NetReadIOsb.Status = SS$_ABORT;

   if (VMSnok (rqptr->NetReadIOsb.Status))
   {
      rqptr->ErrorTextPtr = MsgFor(rqptr,MSG_REQUEST_BODY_READ);
      ErrorVmsStatus (rqptr, rqptr->NetReadIOsb.Status, FI_LI);
      RequestEnd (rqptr);
      return;
   }

   rqptr->BytesRx += rqptr->NetReadIOsb.Count;
   rqptr->ContentCount += rqptr->NetReadIOsb.Count;

   if (Debug)
   {
      fprintf (stdout, "ContentLength: %d ContentCount: %d\n",
               rqptr->ContentLength, rqptr->ContentCount);
      /** if (Debug) fprintf (stdout, "|%s|\n", rqptr->ContentPtr); **/
   }

   if (rqptr->ContentCount >= rqptr->ContentLength)
   {
      /*******************************/
      /* looks like it's all arrived */
      /*******************************/

      RequestExecute (rqptr);
      return;
   }

   /*****************************/
   /* read more from the client */
   /*****************************/

   rqptr->ContentPtr = rqptr->ContentBufferPtr + rqptr->ContentCount;
   if (rqptr->ContentLength - rqptr->ContentCount > 32767)
      ReadSize = 32767;
   else
      ReadSize = rqptr->ContentLength - rqptr->ContentCount;

   NetRead (rqptr, &RequestBodyReadAst, rqptr->ContentPtr, ReadSize);
}

/*****************************************************************************/
/*
Throw-away anything read from the client.
*/ 

RequestDiscardAst (struct RequestStruct *rqptr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
         "RequestDiscardAst() NetReadIOsb.Status %%X%08.08X .Count %d\n",
         rqptr->NetReadIOsb.Status, rqptr->NetReadIOsb.Count);

   /* zero bytes with a normal status is a definite no-no (TGV-Multinet) */
   if (VMSok (rqptr->NetReadIOsb.Status) && !rqptr->NetReadIOsb.Count)
      rqptr->NetReadIOsb.Status = SS$_ABORT;

   if (VMSnok (rqptr->NetReadIOsb.Status))
   {
      RequestEnd (rqptr);
      return;
   }

   NetRead (rqptr, &RequestDiscardAst,
            rqptr->RequestHeaderPtr, rqptr->NetReadBufferSize);
}

/****************************************************************************/
/*
Parse the first <CR><LF> (strict HTTP) or newline (de facto HTTP) terminated
string from the client's request packet into HTTP method, path information and
query string.  Return true to continue processing the request, false to stop
immediately.  If false this function or any it called must have reported the
problem to the client and/or disposed of the thread.
*/ 
 
RequestMethodPathQuery (struct RequestStruct *rqptr)

{
   register unsigned char  c;
   register unsigned char  *cptr, *rptr, *sptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RequestMethodPathQuery()\n");

   cptr = rqptr->RequestHeaderPtr;

   zptr = (sptr = rqptr->HttpMethodName) + sizeof(rqptr->HttpMethodName);
   while (!ISLWS(*cptr) && NOTEOL(*cptr) && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr) sptr = rqptr->HttpMethodName;
   *sptr = '\0';

   if (!rqptr->HttpMethodName[0])
   {
      Accounting.RequestErrorCount++;
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_FORMAT), FI_LI);
      RequestEnd (rqptr);
      return (false);
   }

   /* skip across white-space between method and URI */
   while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;

   /* get the path information, noting the start of the requested resource */
   rptr = zptr = cptr;
   while (!ISLWS(*zptr) && *zptr != '?' && NOTEOL(*zptr)) zptr++;

   if (cptr == zptr)
   {
      /* no path supplied */
      Accounting.RequestErrorCount++;
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_FORMAT), FI_LI);
      RequestEnd (rqptr);
      return (false);
   }

   rqptr->PathInfoPtr = sptr =
      VmGetHeap (rqptr, (rqptr->PathInfoLength = zptr-cptr)+1);
   memcpy (sptr, cptr, zptr-cptr);
   sptr[zptr-cptr] = '\0';
   /* unescape any hex-encoded characters */
   for (cptr = sptr; *cptr && *cptr != '%'; cptr++);
   if (*cptr)
   {
      sptr = cptr;
      while (*cptr)
      {
         while (*cptr && *cptr != '%') *sptr++ = *cptr++;
         if (!*cptr) break;
         /* an escaped character ("%xx" where xx is a hex number) */
         cptr++;
         c = 0;
         if (*cptr >= '0' && *cptr <= '9')
            { c = (*cptr - (int)'0') << 4; cptr++; }
         else
         if (tolower(*cptr) >= 'a' && tolower(*cptr) <= 'f')
            { c = (tolower(*cptr) - (int)'a' + 10) << 4; cptr++; }
         else
         {
            Accounting.RequestErrorCount++;
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_URL_ENC), FI_LI);
            RequestEnd (rqptr);
            return (false);
         }
         if (*cptr >= '0' && *cptr <= '9')
            { c += (*cptr - (int)'0'); cptr++; }
         else
         if (tolower(*cptr) >= 'a' && tolower(*cptr) <= 'f')
            { c += (tolower(*cptr) - (int)'a' + 10); cptr++; }
         else
         {
            Accounting.RequestErrorCount++;
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_URL_ENC), FI_LI);
            RequestEnd (rqptr);
            return (false);
         }
         *sptr++ = c;
      }
      *sptr = '\0';
      rqptr->PathInfoLength = sptr - rqptr->PathInfoPtr;
   }
   /* point back into the request */ 
   cptr = zptr;

   /* get any query string, or create an empty one */
   if (*cptr == '?') cptr++;
   zptr = cptr;
   while (!ISLWS(*zptr) && NOTEOL(*zptr)) zptr++;
   rqptr->QueryStringPtr = sptr = VmGetHeap (rqptr, zptr-cptr+1);
   if (rqptr->QueryStringLength = zptr-cptr)
      memcpy (sptr, cptr, rqptr->QueryStringLength);
   sptr[rqptr->QueryStringLength] = '\0';

   /* store the requested resource */
   rqptr->ResourcePtr = sptr = VmGetHeap (rqptr, zptr-rptr+1);
   memcpy (sptr, rptr, rqptr->ResourceLength = zptr-rptr);
   sptr[zptr-rptr] = '\0';

   return (true);
}

/*****************************************************************************/
/*
Scan the header buffer looking for relevant fields. Return true to continue
processing the request, false to stop immediately.  If false this function or
any it called must have reported the problem to the client and/or disposed of
the thread.
*/ 
 
boolean RequestFields (struct RequestStruct *rqptr)

{
   register int  Length;
   register char  *cptr, *hptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RequestFields()\n");

   /* ensure any old field values are ignored (in the case of redirection) */

   rqptr->HttpAcceptPtr =
      rqptr->HttpAcceptLangPtr =
      rqptr->HttpAcceptCharsetPtr =
      rqptr->HttpAuthorizationPtr =
      rqptr->HttpContentTypePtr =
      rqptr->HttpForwardedPtr =
      rqptr->HttpHostPtr =
      rqptr->HttpIfModifiedSincePtr =
      rqptr->HttpRefererPtr =
      rqptr->HttpUserAgentPtr = NULL;

   rqptr->ContentLength =
      rqptr->HttpAcceptLength =
      rqptr->HttpAcceptCharsetLength =
      rqptr->HttpAcceptLangLength =
      rqptr->HttpForwardedLength =
      rqptr->IfModifiedSinceBinaryTime[0] =
      rqptr->IfModifiedSinceBinaryTime[1] = 0;

   rqptr->KeepAliveRequest = false;

   /* the header is already null-terminated */
   hptr = rqptr->RequestHeaderPtr;
   /* look for the request-terminating blank line */
   while (*hptr)
   {
      /** if (Debug) fprintf (stdout, "hptr |%s|\n", hptr); **/

      cptr = hptr;
      while (NOTEOL(*hptr)) hptr++;
      if (hptr[0] == '\n' || (hptr[0] == '\r' && hptr[1] == '\n'))
      {
         /* break if blank line (i.e. end-of-header) */
         if (cptr == hptr) break;
      }
      /* step over the carriage control at the end of the line */
      if (*hptr == '\r') hptr++;
      if (*hptr == '\n') hptr++;
      if (Debug)
         fprintf (stdout, "field |%*.*s|\n", hptr-cptr, hptr-cptr, cptr);

      /***********/
      /* Accept: */
      /***********/

      if (toupper(cptr[0]) == 'A' && cptr[6] == ':' &&
          strsame (cptr, "Accept:", 7))
      {
         cptr += 7;
         while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;
         sptr = cptr;
         while (NOTEOL(*sptr)) sptr++;
         Length = sptr - cptr;
         if (rqptr->HttpAcceptPtr == NULL)
            rqptr->HttpAcceptPtr = sptr = VmGetHeap (rqptr, Length+1);
         else
         {
            rqptr->HttpAcceptPtr =
               VmReallocHeap (rqptr, rqptr->HttpAcceptPtr, 
                              rqptr->HttpAcceptLength+Length+2);
            sptr = rqptr->HttpAcceptPtr + rqptr->HttpAcceptLength;
            *sptr++ = ',';
         }
         while (NOTEOL(*cptr))
         {
            if (ISLWS(*cptr))
               while (ISLWS(*cptr)) cptr++;
            else
               *sptr++ = *cptr++;
         }
         *sptr = '\0';
         rqptr->HttpAcceptLength = sptr - (char*)rqptr->HttpAcceptPtr;
         if (Debug)
            fprintf (stdout, "HttpAcceptPtr %d |%s|\n",
                     rqptr->HttpAcceptLength, rqptr->HttpAcceptPtr);
         continue;
      }

      /*******************/
      /* Accept-Charset: */
      /*******************/

      if (toupper(cptr[0]) == 'A' && tolower(cptr[7]) == 'c' &&
          strsame (cptr, "Accept-charset:", 15))
      {
         cptr += 15;
         while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;
         sptr = cptr;
         while (NOTEOL(*sptr)) sptr++;
         Length = sptr - cptr;
         if (rqptr->HttpAcceptCharsetPtr == NULL)
            rqptr->HttpAcceptCharsetPtr = sptr = VmGetHeap (rqptr, Length+1);
         else
         {
            rqptr->HttpAcceptCharsetPtr =
               VmReallocHeap (rqptr, rqptr->HttpAcceptCharsetPtr, 
                              rqptr->HttpAcceptCharsetLength+Length+2);
            sptr = rqptr->HttpAcceptCharsetPtr + rqptr->HttpAcceptCharsetLength;
            *sptr++ = ',';
         }
         while (NOTEOL(*cptr))
         {
            if (ISLWS(*cptr))
               while (ISLWS(*cptr)) cptr++;
            else
               *sptr++ = *cptr++;
         }
         *sptr = '\0';
         rqptr->HttpAcceptCharsetLength =
            sptr - (char*)rqptr->HttpAcceptCharsetPtr;
         if (Debug)
            fprintf (stdout, "HttpAcceptCharsetPtr %d |%s|\n",
                     rqptr->HttpAcceptCharsetLength,
                     rqptr->HttpAcceptCharsetPtr);
         continue;
      }

      /********************/
      /* Accept-Language: */
      /********************/

      if (toupper(cptr[0]) == 'A' && tolower(cptr[7]) == 'l' &&
          strsame (cptr, "Accept-language:", 16))
      {
         cptr += 16;
         while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;
         sptr = cptr;
         while (NOTEOL(*sptr)) sptr++;
         Length = sptr - cptr;
         if (rqptr->HttpAcceptLangPtr == NULL)
            rqptr->HttpAcceptLangPtr = sptr = VmGetHeap (rqptr, Length+1);
         else
         {
            rqptr->HttpAcceptLangPtr =
               VmReallocHeap (rqptr, rqptr->HttpAcceptLangPtr, 
                              rqptr->HttpAcceptLangLength+Length+2);
            sptr = rqptr->HttpAcceptLangPtr + rqptr->HttpAcceptLangLength;
            *sptr++ = ',';
         }
         while (NOTEOL(*cptr))
         {
            if (ISLWS(*cptr))
               while (ISLWS(*cptr)) cptr++;
            else
               *sptr++ = *cptr++;
         }
         *sptr = '\0';
         rqptr->HttpAcceptLangLength = sptr - (char*)rqptr->HttpAcceptLangPtr;
         if (Debug)
            fprintf (stdout, "HttpAcceptLangPtr %d |%s|\n",
                     rqptr->HttpAcceptLangLength, rqptr->HttpAcceptLangPtr);
         continue;
      }

      /******************/
      /* Authorization: */
      /******************/

      if (toupper(cptr[0]) == 'A' && tolower(cptr[1]) == 'u' &&
          strsame (cptr, "Authorization:", 14))
      {
         cptr += 14;
         while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;
         sptr = cptr;
         while (NOTEOL(*sptr)) sptr++;
         Length = sptr - cptr;
         rqptr->HttpAuthorizationPtr = sptr = VmGetHeap (rqptr, Length+1);
         memcpy (sptr, cptr, rqptr->HttpAuthorizationLength = Length);
         sptr[Length] = '\0';
         if (Debug) fprintf (stdout, "Authorization |%s|\n", sptr);

         continue;
      }

      /***************/
      /* Connection: */
      /***************/

      /*
         Only check for persistent connection request if it's configured.
         Persistent connection only allowed for object retrievals.
      */
      if (Config.KeepAliveTimeoutSeconds &&
          rqptr->RequestScheme != SCHEME_HTTPS)
      {
         if (toupper(cptr[0]) == 'C' && tolower(cptr[8]) == 'o' &&
             strsame (cptr, "Connection:", 11))
         {
            cptr += 11;
            while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;
            if (toupper(cptr[0]) == 'K' && strsame (cptr, "Keep-Alive", 10))
            {
               rqptr->KeepAliveRequest = true;
               if (Debug) fprintf (stdout, "KEEP-ALIVE!\n");
            }
            continue;
         }
      }

      /*******************/
      /* Content-Length: */
      /*******************/

      if (toupper(cptr[0]) == 'C' && tolower(cptr[8]) == 'l' &&
          strsame (cptr, "Content-length:", 15))
      {
         cptr += 15;
         while (!isdigit(*cptr) && NOTEOL(*cptr)) cptr++;
         if (!isdigit(*cptr))
         {
            Accounting.RequestErrorCount++;
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_FORMAT), FI_LI);
            RequestEnd (rqptr);
            return (false);
         }
         rqptr->ContentLength = atoi (cptr);
         if (Debug)
            fprintf (stdout, "ContentLength |%d|\n", rqptr->ContentLength);
         continue;
      }

      /*****************/
      /* Content-Type: */
      /*****************/

      if (toupper(cptr[0]) == 'C' && tolower(cptr[8]) == 't' &&
          strsame (cptr, "Content-type:", 13))
      {
         cptr += 13;
         while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;
         sptr = cptr;
         while (NOTEOL(*sptr)) sptr++;
         Length = sptr - cptr;
         rqptr->HttpContentTypePtr = sptr = VmGetHeap (rqptr, Length+1);
         memcpy (sptr, cptr, rqptr->ContentTypeLength = Length);
         sptr[Length] = '\0';
         if (Debug) fprintf (stdout, "HttpContentType |%s|\n", sptr);
         continue;
      }

      /***********/
      /* Cookie: */
      /***********/

      if (toupper(cptr[0]) == 'C' && cptr[6] == ':' &&
          strsame (cptr, "Cookie:", 7))
      {
         cptr += 7;
         while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;
         sptr = cptr;
         while (NOTEOL(*sptr)) sptr++;
         Length = sptr - cptr;
         rqptr->HttpCookiePtr = sptr = VmGetHeap (rqptr, Length+1);
         memcpy (sptr, cptr, Length);
         sptr[Length] = '\0';
         if (Debug) fprintf (stdout, "HttpCookie |%s|\n", sptr);
         continue;
      }

      /**************/
      /* Forwarded: */
      /**************/

      if (toupper(cptr[0]) == 'F' && tolower(cptr[1]) == 'o' &&
          strsame (cptr, "Forwarded:", 10))
      {
         cptr += 10;
         while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;
         sptr = cptr;
         while (NOTEOL(*sptr)) sptr++;
         Length = sptr - cptr;
         if (rqptr->HttpForwardedPtr == NULL)
            rqptr->HttpForwardedPtr = sptr = VmGetHeap (rqptr, Length+1);
         else
         {
            rqptr->HttpForwardedPtr =
               VmReallocHeap (rqptr, rqptr->HttpForwardedPtr,
                              rqptr->HttpForwardedLength+Length+2);
            sptr = rqptr->HttpForwardedPtr + rqptr->HttpForwardedLength;
            *sptr++ = ',';
         }
         memcpy (sptr, cptr, Length);
         sptr[Length] = '\0';
         rqptr->HttpForwardedLength =
            sptr - (char*)rqptr->HttpForwardedPtr + Length;
         if (Debug)
            fprintf (stdout, "HttpForwardedPtr %d |%s|\n",
                     rqptr->HttpForwardedLength, rqptr->HttpForwardedPtr);
         continue;
      }

      /**********************/
      /* If-Modified-Since: */
      /**********************/

      if (toupper(cptr[0]) == 'I' && tolower(cptr[1]) == 'f' &&
          strsame (cptr, "If-Modified-Since:", 18))
      {
         cptr += 18;
         while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;
         sptr = cptr;
         while (NOTEOL(*sptr)) sptr++;
         Length = sptr - cptr;
         rqptr->HttpIfModifiedSincePtr = sptr = VmGetHeap (rqptr, Length+1);
         memcpy (sptr, cptr, Length);
         sptr[Length] = '\0';
         if (Debug) fprintf (stdout, "HttpIfModifiedSince |%s|\n", sptr);

         if (VMSnok (HttpGmTime (sptr, &rqptr->IfModifiedSinceBinaryTime)))
         {
            if (Debug) fprintf (stdout, "If-Modified-Since: NBG!\n");
            rqptr->HttpIfModifiedSincePtr = NULL;
            rqptr->IfModifiedSinceBinaryTime[0] =
            rqptr->IfModifiedSinceBinaryTime[1] = 0;

            continue;
         }

         /* indicate that no "length=" was found (if none is found :^) */
         rqptr->IfModifiedSinceLength = -1;

         /* look for a "length=" parameter to this header field */
         while (*sptr)
         {
            while (*sptr && *sptr != ';') sptr++; 
            if (!*sptr) break;
            sptr++;
            while (*sptr && ISLWS(*sptr)) sptr++; 
            if (!*sptr) break;
            if (!strsame (sptr, "length", 6)) continue;
            sptr += 6;
            while (*sptr && *sptr != '=') sptr++; 
            if (!*sptr) break;
            sptr++;
            while (*sptr && ISLWS(*sptr)) sptr++; 
            if (!isdigit (*sptr)) continue;
            rqptr->IfModifiedSinceLength = atol(sptr);
            if (Debug)
               fprintf (stdout, "length=%d\n",
                        rqptr->IfModifiedSinceLength);
         }

         continue;
      }

      /*********/
      /* Host: */
      /*********/

      if (toupper(cptr[0]) == 'H' && tolower(cptr[1]) == 'o' &&
          strsame (cptr, "Host:", 5))
      {
         cptr += 5;
         while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;
         sptr = cptr;
         while (NOTEOL(*sptr)) sptr++;
         Length = sptr - cptr;
         rqptr->HttpHostPtr = sptr = VmGetHeap (rqptr, Length+1);
         memcpy (sptr, cptr, Length);
         sptr[Length] = '\0';
         if (Debug) fprintf (stdout, "HttpHost |%s|\n", sptr);
         continue;
      }

      /***********/
      /* Pragma: */
      /***********/

      if (toupper(cptr[0]) == 'P' && tolower(cptr[1]) == 'r' &&
          strsame (cptr, "Pragma:", 7))
      {
         cptr += 7;
         while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;
         sptr = cptr;
         while (NOTEOL(*sptr)) sptr++;
         Length = sptr - cptr;
         rqptr->HttpPragmaPtr = sptr = VmGetHeap (rqptr, Length+1);
         memcpy (sptr, cptr, Length);
         sptr[Length] = '\0';
         if (strsame (sptr, "no-cache", -1))
         {
            rqptr->PragmaNoCache = true;
            Accounting.RequestPragmaNoCacheCount++;
         }
         if (Debug)
            fprintf (stdout, "HttpPragma |%s| no-cache: %d\n",
                     sptr, rqptr->PragmaNoCache);
         continue;
      }

      /************/
      /* Referer: */
      /************/

      if (toupper(cptr[0]) == 'R' && tolower(cptr[1]) == 'e' &&
          strsame (cptr, "Referer:", 8))
      {
         cptr += 8;
         while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;
         sptr = cptr;
         while (NOTEOL(*sptr)) sptr++;
         Length = sptr - cptr;
         rqptr->HttpRefererPtr = sptr = VmGetHeap (rqptr, Length+1);
         memcpy (sptr, cptr, rqptr->HttpRefererLength = Length);
         sptr[Length] = '\0';
         if (Debug) fprintf (stdout, "Referer |%s|\n", sptr);
         continue;
      }

      /***************/
      /* User-Agent: */
      /***************/

      if (toupper(cptr[0]) == 'U' && tolower(cptr[1]) == 's' &&
          strsame (cptr, "User-Agent:", 11))
      {
         cptr += 11;
         while (ISLWS(*cptr) && NOTEOL(*cptr)) cptr++;
         sptr = cptr;
         while (NOTEOL(*sptr)) sptr++;
         Length = sptr - cptr;
         rqptr->HttpUserAgentPtr = sptr = VmGetHeap (rqptr, Length+1);
         memcpy (sptr, cptr, rqptr->HttpUserAgentLength = Length);
         sptr[Length] = '\0';
         if (Debug) fprintf (stdout, "UserAgent |%s|\n", sptr);
         continue;
      }
   }

   return (true);
}

/*****************************************************************************/
/*
If a file specification does not contain a file name the function looks for a
home page in the directory.  If no home page found it generates a directory
listing ("Index of" documents).

If the file specification contains a wildcard and there is no query string a
directory listing (index) is generated.  If there is a query string it must
begin with the literal "?httpd=index" for a directory listing (index) to be
generated (this allows directives in the query string to be passed to the
directory listing functions).  If it does not begin with this literal the
default query (search) script is invoked.

Detects file types that have an associated script.  If a script is returned by
ConfigContentType(), and the file specification does not include a specific
version number (even such as ";0"), an automatic redirection is generated so
that the specified document is processed by the script, and the output from
that returned to the client.  If it does contain a version number the file is
always directly returned to the client.
*/ 

int RequestExecute (struct RequestStruct *rqptr)

{
   register char  *cptr, *mpptr;

   int  status,
        Count;
   char  MappedFile [1024],
         MappedScript [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RequestExecute()\n");

   /** fprintf (stdout,"|%s|\n", rqptr->RequestHeaderPtr); **/

   /* ensure any old values are ignored, in case its a redirection */
   rqptr->LocationPtr = NULL;

   /* request is underway, initialize the output (response) timeout */
   HttpdTimerSet (rqptr, TIMER_OUTPUT);

   /****************/
   /* map the path */
   /****************/

   MappedFile[0] = rqptr->ScriptName[0] = MappedScript[0] = '\0';

   /*
      Map the path, converting it to a VMS file path (or script file name)
      and returning a pointer to a static buffer containing the mapped path.
   */
   mpptr = MapUrl_Map (rqptr->PathInfoPtr, MappedFile,
                       rqptr->ScriptName, MappedScript,
                       rqptr);
   if (!mpptr[0] && mpptr[1])
   {
      /* MapUrl_Map() returns errors with a leading null character */
      mpptr++;
      if (isdigit(*mpptr))
      {
         /* HTTP status code mapping, with following message */
         rqptr->ResponseStatusCode = atol(mpptr);
         while (*mpptr && isdigit(*mpptr)) mpptr++;
         while (*mpptr && ISLWS(*mpptr)) mpptr++;
         if (rqptr->ResponseStatusCode >= 300 &&
             rqptr->ResponseStatusCode <= 399)
         {
            /* redirection, message "text" should be the location */
            rqptr->LocationPtr = mpptr;
            RequestEnd (rqptr);
            return;
         }
         else
         if (rqptr->ResponseStatusCode >= 400 &&
             rqptr->ResponseStatusCode <= 599)
         {
            /* just a 400/500 message */
            if (rqptr->ResponseStatusCode == 401 ||
                rqptr->ResponseStatusCode == 402 ||
                rqptr->ResponseStatusCode == 403)
               Accounting.RequestForbiddenCount++;
            ErrorGeneral (rqptr, mpptr, FI_LI);
            RequestEnd (rqptr);
            return;
         }
         else
         {
            /* no point to codes other 3nn/4nn/5nn, use to abort connection */
            Accounting.RequestForbiddenCount++;
            RequestEnd (rqptr);
            return;
         }
      }
      else
      {
         /* just a rule-mapping message */
         Accounting.RequestForbiddenCount++;
         rqptr->ResponseStatusCode = 403;
         ErrorGeneral (rqptr, mpptr, FI_LI);
         RequestEnd (rqptr);
         return;
      }
   }

   /* MapUrl_Map() returns CGIplus mappings indicated by a leading '+' */
   if (rqptr->ScriptName[0] == '+')
   {
      rqptr->IsCgiPlusScript = true;
      rqptr->ScriptName[0] = '/';
   }
   else
      rqptr->IsCgiPlusScript = false;

   if (Debug)
      fprintf (stdout, "|%s|%s|%s|%d|%s|%s|\n",
               rqptr->PathInfoPtr, MappedFile,
               rqptr->ScriptName, rqptr->IsCgiPlusScript, MappedScript, mpptr);

   /********************************************/
   /* check for redirection (from the mapping) */
   /********************************************/

   if (mpptr[0] && mpptr[0] != '/')
   {
      /* local will begin with a '^', non-local with "http:", etc. */
      if (mpptr[0] == '^') mpptr[0] = '/';
      rqptr->LocationPtr = VmGetHeap (rqptr, strlen(mpptr));
      strcpy (rqptr->LocationPtr, mpptr);
      RequestEnd (rqptr);
      return;
   }

   /*********************************/
   /* authorization of request path */
   /*********************************/

   if (VMSnok (Authorize (rqptr)))
   {
      RequestEnd (rqptr);
      return;
   }

   /************************************/
   /* handled internally by the server */
   /************************************/

   if (strsame (rqptr->ResourcePtr, HttpdInternalAdmin,
                sizeof(HttpdInternalAdmin)-1))
   {
      AdminBegin (rqptr, &RequestEnd);
      return;
   }

   if (strsame (rqptr->ResourcePtr, HttpdInternalPasswordChange,
                sizeof(HttpdInternalPasswordChange)-1))
   {
      HTAdminBegin (rqptr, &RequestEnd);
      return;
   }

   /******************************/
   /* check for script execution */
   /******************************/

   if (rqptr->ScriptName[0])
   {
      /* get the path derived from the script specification */
      rqptr->PathInfoPtr =
         VmReallocHeap (rqptr, rqptr->PathInfoPtr, 
                        ((rqptr->PathInfoLength=strlen(mpptr))+1));
      memcpy (rqptr->PathInfoPtr, mpptr, rqptr->PathInfoLength+1);

      if (rqptr->PathInfoLength > 1)
      {
         /******************************************/
         /* authorization of script path parameter */
         /******************************************/

         if (VMSnok (Authorize (rqptr)))
         {
            RequestEnd (rqptr);
            return;
         }
      }

      RequestScript (rqptr, MappedScript, MappedFile);
      return;
   }

   /********************/
   /* check file cache */
   /********************/

   /* cache use never occurs for requests with a VMS authentication profile */
   if (CacheEnabled &&
       !rqptr->AuthVmsUserProfileLength &&
       (rqptr->HttpMethod == HTTP_METHOD_GET ||
        rqptr->HttpMethod == HTTP_METHOD_HEAD))
   {
      /* search the cache ('mpptr' points to the mapped, not request path) */
      if (CacheSearch (rqptr, mpptr))
         /* found, initiate transfer */
         if (CacheBegin (rqptr))
            /* if data valid and transfer begins return */
            return;

      /* buffer the mapped path in case it's needed during cache load */
      rqptr->CacheMappedPathPtr =
         VmGetHeap (rqptr, (rqptr->CacheMappedPathLength=strlen(mpptr))+1);
      memcpy (rqptr->CacheMappedPathPtr, mpptr, rqptr->CacheMappedPathLength+1);
   }

   /********************************/
   /* parse the file specification */
   /********************************/

   rqptr->RequestParseFab = cc$rms_fab;
   rqptr->RequestParseFab.fab$l_fna = MappedFile;
   rqptr->RequestParseFab.fab$b_fns = strlen(MappedFile);
   rqptr->RequestParseFab.fab$l_fop = FAB$M_NAM;
   rqptr->RequestParseFab.fab$l_nam = &rqptr->RequestParseNam;
   rqptr->RequestParseNam = cc$rms_nam;
   rqptr->RequestParseNam.nam$l_esa = rqptr->RequestFileName;
   rqptr->RequestParseNam.nam$b_ess = sizeof(rqptr->RequestFileName)-1;
   /* parse without disk I/O, syntax check only! */
   rqptr->RequestParseNam.nam$b_nop = NAM$M_SYNCHK;

   if (VMSnok (status = sys$parse (&rqptr->RequestParseFab, 0, 0)))
   {
      rqptr->ErrorHiddenTextPtr = MappedFile;
      ErrorVmsStatus (rqptr, status, FI_LI);
      RequestEnd (rqptr);
      return;
   }

   /* 
       If a file version was explicitly specified or a wildcard
       specified in any version component then terminate after that version.
       If there is no file name or type (i.e. a directory specification)
       then terminate after the ']'.
       Otherwise terminate after the file type.
   */
   if ((rqptr->RequestParseNam.nam$l_fnb & NAM$M_EXP_VER) ||
       (rqptr->RequestParseNam.nam$l_fnb & NAM$M_WILD_VER))
      rqptr->RequestParseNam.nam$l_ver[rqptr->RequestParseNam.nam$b_ver] = '\0';
   else
   if (rqptr->RequestParseNam.nam$l_name ==  rqptr->RequestParseNam.nam$l_type)
      rqptr->RequestParseNam.nam$l_name[0] = '\0';
   else
      rqptr->RequestParseNam.nam$l_ver[0] = '\0';
   if (Debug) fprintf (stdout, "FileName |%s|\n", rqptr->RequestFileName);

   /*
      If the file name component comprises a leading hyphen followed by
      16 digits and a trailing hyphen  ("-yyyymmddhhmmsshh-") then consider
      it a temporary file and delete it on close.
   */
   rqptr->DeleteOnClose = false;
   if (rqptr->RequestParseNam.nam$b_name == 18 &&
       *rqptr->RequestParseNam.nam$l_name == '-')
   {
      for (cptr = rqptr->RequestParseNam.nam$l_name+1;
           *cptr && isdigit(*cptr);
           cptr++);
      if (*cptr == '-')
      {
         rqptr->DeleteOnClose = true;
         rqptr->ResponsePreExpired = PRE_EXPIRE_DELETE_ON_CLOSE;
      }
   }

   /* all these "file write" methods are handled by the one set of functions */
   if (rqptr->HttpMethod == HTTP_METHOD_PUT ||
       rqptr->HttpMethod == HTTP_METHOD_POST ||
       rqptr->HttpMethod == HTTP_METHOD_DELETE)
   {
      PutBegin (rqptr, &RequestEnd);
      return;
   }

   if ((tolower(rqptr->QueryStringPtr[0]) == 'h' &&
        strsame (rqptr->QueryStringPtr, "httpd=index", 11)) ||
       ((rqptr->RequestParseNam.nam$l_fnb & NAM$M_WILDCARD) &&
        !rqptr->QueryStringPtr[0]))
   {
      /******************************/
      /* generate directory listing */
      /******************************/

      if (Config.DirWildcardEnabled)
      {
         DirBegin (rqptr, &RequestEnd, MappedFile, rqptr->QueryStringPtr, "");
         return;
      }
      else
      {
         Accounting.RequestForbiddenCount++;
         rqptr->ResponseStatusCode = 501;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_GENERAL_DISABLED), FI_LI);
         RequestEnd (rqptr);
         return;
      }
   }

   if (!rqptr->RequestParseNam.nam$b_name &&
       rqptr->RequestParseNam.nam$b_type == 1 &&
       !rqptr->QueryStringPtr[0])
   {
      /*********************************************/
      /* no file name supplied, look for home page */
      /*********************************************/

      RequestHomePage (rqptr);
      return;
   }

   /********************************/
   /* get the content-type details */
   /********************************/

   ConfigContentType (&rqptr->ContentType, rqptr->RequestParseNam.nam$l_type);

   if (!isdigit(rqptr->RequestParseNam.nam$l_ver[1]))
   {
      /* 
         If a file specification does not include a specific version
         number (which indicates send-me-this-file-regardless) and
         ConfigContentType() has returned a script name, indicating
         an automatic script handling for this file type, then redirect.
      */
      if (rqptr->ContentType.AutoScriptNamePtr[0] == '/')
      {
         /********************************/
         /* automatic script redirection */
         /********************************/

         Accounting.DoAutoScriptCount++;

         Count = strlen(rqptr->ContentType.AutoScriptNamePtr);
         if (rqptr->PathInfoPtr != NULL) Count += rqptr->PathInfoLength;
         if (rqptr->QueryStringPtr[0])
         {
             /* allow one extra character for the query-string leading '?' */
             Count++;
             Count += rqptr->QueryStringLength;
         }
         /* allow for the terminating null */
         Count++;

         rqptr->LocationPtr = mpptr = VmGetHeap (rqptr, Count);
         cptr = rqptr->ContentType.AutoScriptNamePtr;
         while (*cptr) *mpptr++ = *cptr++;
         if (rqptr->PathInfoPtr != NULL)
         {
            cptr = rqptr->PathInfoPtr;
            while (*cptr) *mpptr++ = *cptr++;
         }
         if (rqptr->QueryStringPtr[0])
         {
            *mpptr++ = '?';
            cptr = rqptr->QueryStringPtr;
            while (*cptr) *mpptr++ = *cptr++;
         }
         *mpptr = '\0';
         if (Debug) fprintf (stdout, "LocationPtr |%s|\n", rqptr->LocationPtr);

         /* dispose of thread function does the actual redirection */
         RequestEnd (rqptr);
         return;
      }
   }

   /*************************************/
   /* internal types with query strings *
   /*************************************/

   if (strsame (rqptr->ContentType.ContentTypePtr, ConfigContentTypeIsMap, -1))
   {
      /* image mapping configuration file (has a query string!) */
      IsMapBegin (rqptr, &RequestEnd, rqptr->RequestFileName);
      return;
   }

   /******************************************************************/
   /* query string (without script name), an implicit keyword search */
   /******************************************************************/

   if (rqptr->QueryStringPtr[0] &&
       !(tolower(rqptr->QueryStringPtr[0]) == 'h' &&
         strsame (rqptr->QueryStringPtr, "httpd=", 6)))
   {
      /*
         No need to authorize the path here.  No script component was
         supplied in the HTTP request, but a query string was, making
         this an ISINDEX search on the already authorized path.
      */
      strcpy (rqptr->ScriptName, Config.KeywordSearch);
      MapUrl_UrlToVms (rqptr->ScriptName, MappedScript);
      RequestScript (rqptr, MappedScript, MappedFile);
      return;
   }

   /****************************************/ 
   /* otherwise ... send the file contents */
   /****************************************/ 

   /* allow for directories specified as "/dir1/dir2" (i.e. no trailing '/') */
   if (rqptr->RequestParseNam.nam$b_type == 1)
      RequestFile (rqptr, &RequestEnd, &RequestFileNoType);
   else
      RequestFile (rqptr, &RequestEnd, NULL);
}

/*****************************************************************************/
/*
This function can be called one or more time per request.  It steps through all
possible home page files until either one is found or a default directory
listing is generated.
*/ 

int RequestHomePage (struct RequestStruct *rqptr)

{
   register char  *cptr;

   char  *HomePageNamePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "RequestHomePage() %d %d\n",
               rqptr->RequestHomePageIndex, rqptr->ResponseHeaderPtr);

   if (rqptr->ResponseHeaderPtr != NULL || rqptr->ErrorMessagePtr != NULL)
   {
      RequestEnd (rqptr);
      return;
   }

   if (!*(HomePageNamePtr = ConfigHomePage(rqptr->RequestHomePageIndex++)))
   {
      /********************************************/
      /* no home page, generate directory listing */
      /********************************************/

      if (!Config.DirAccess)
      {
         /* directory listing is disabled */
         rqptr->ErrorTextPtr = rqptr->PathInfoPtr;
         rqptr->ErrorHiddenTextPtr = rqptr->RequestParseNam.nam$l_dev;
         ErrorVmsStatus (rqptr, RMS$_FNF, FI_LI);
         RequestEnd (rqptr);
         return;
      }

      rqptr->RequestParseNam.nam$l_name[0] = '\0';
      DirBegin (rqptr, &RequestEnd,
                rqptr->RequestParseNam.nam$l_dev,
                rqptr->QueryStringPtr, "");
      return;
   }

   /********************************/
   /* generate home page file name */
   /********************************/

   /* overwrite any existing file name in the NAM block */
   strcpy (rqptr->RequestParseNam.nam$l_name, HomePageNamePtr);
   /* locate the start of the file type */
   for (cptr = rqptr->RequestParseNam.nam$l_name;
        *cptr && *cptr != '.';
        cptr++);
   /* set the mime content type for this file type */
   ConfigContentType (&rqptr->ContentType, cptr);

   RequestFile (rqptr, &RequestEnd, &RequestHomePage);
}

/*****************************************************************************/
/*
Send a file to the client.  Special cases are menu and pre-processed HTML.  
*/ 

int RequestFile
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
void *FileNotFoundFunction
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "RequestFile() |%s|%s|%s|\n",
               rqptr->RequestFileName,
               rqptr->ContentType.ContentTypePtr);

   if (strsame (rqptr->ContentType.ContentTypePtr, ConfigContentTypeMenu, -1))
   {
      MenuBegin (rqptr, NextTaskFunction, FileNotFoundFunction,
                 rqptr->RequestFileName);
      return;
   }

   if (strsame (rqptr->ContentType.ContentTypePtr, ConfigContentTypeSsi, -1))
   {
      SsiBegin (rqptr, NextTaskFunction, FileNotFoundFunction,
                rqptr->RequestFileName);
      return;
   }

   FileBegin (rqptr, NextTaskFunction, FileNotFoundFunction,
              rqptr->RequestFileName, rqptr->ContentType.ContentTypePtr,
              true, false, false);
}

/*****************************************************************************/
/*
The file requested had no type (extension) and was not found.  Check if there
is a directory of that name.  This allows for directories to be specified as
"/dir1/dir2" (i.e. no trailing '/') which is not strictly kosher in WWW syntax
but is allowed by some servers and so does occur in some documents.
*/ 

int RequestFileNoType (struct RequestStruct *rqptr)

{
   register char  *cptr, *sptr;

   char  c;
   char  DirName [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RequestFileNoType()\n");

   sptr = DirName;
   cptr = rqptr->RequestParseNam.nam$l_name;
   while (*cptr) *sptr++ = *cptr++;
   while (sptr > DirName && *sptr != '.') sptr--;
   cptr = ".DIR;";
   while (*cptr) *sptr++ = *cptr++;
   *sptr = '\0';

   c = *rqptr->RequestParseNam.nam$l_name;
   *rqptr->RequestParseNam.nam$l_name = '\0';

   if (VMSok (FileExists (rqptr->RequestParseNam.nam$l_dev, DirName)))
   {
      *rqptr->RequestParseNam.nam$l_name = c;
      rqptr->RequestParseNam.nam$l_name[-1] = '.';
      rqptr->RequestParseNam.nam$l_type[0] = ']';
      rqptr->RequestParseNam.nam$l_type[1] = '\0';

      sptr = DirName;
      *sptr++ = '.';
      *sptr++ = '/';
      cptr = rqptr->RequestParseNam.nam$l_name;
      while (*cptr && *cptr != ']') *sptr++ = tolower(*cptr++);
      *sptr++ = '/';
      *sptr = '\0';

      DirBegin (rqptr, &RequestEnd,
                rqptr->RequestParseNam.nam$l_dev,
                rqptr->QueryStringPtr, DirName);
   }
   else
   {
      *rqptr->RequestParseNam.nam$l_name = c;
      if (!rqptr->AccountingDone)
         rqptr->AccountingDone = ++Accounting.DoFileCount;
      rqptr->ErrorTextPtr = MapVmsPath (rqptr->RequestParseNam.nam$l_dev);
      rqptr->ErrorHiddenTextPtr = rqptr->RequestParseNam.nam$l_dev;
      ErrorVmsStatus (rqptr, RMS$_FNF, FI_LI);
      RequestEnd (rqptr);
   }
}

/*****************************************************************************/
/*
Process request by executing a script.  The 'MappedFile' was the non-script-name 
portion of the path, returned by the original MapUrl(), attempt to parse this
as a VMS file specification, but if that fails (and it well might not be a
file specification) then do not report it.
*/ 

RequestScript
(
struct RequestStruct *rqptr,
char *MappedScript,
char *MappedFile
)
{
   register char  *cptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "RequestScript() |%s|%s|%s|\n",
               rqptr->ScriptName, MappedFile, rqptr->PathInfoPtr);

   /***************************************/
   /* check for internal server "scripts" */
   /***************************************/

   if (tolower(rqptr->ScriptName[0]) == tolower(HttpdInternalScriptEcho[0]) &&
       strsame (rqptr->ScriptName, HttpdInternalScriptEcho,
                sizeof(HttpdInternalScriptEcho)-1))
   {
      RequestEcho (rqptr);
      return;
   }
   else
   if (tolower(rqptr->ScriptName[0]) == tolower(HttpdInternalScriptWhere[0]) &&
       strsame (rqptr->ScriptName, HttpdInternalScriptWhere,
                sizeof(HttpdInternalScriptWhere)-1))
   {
      RequestWhere (rqptr, MappedFile);
      return;
   }

   /********************************/
   /* parse any file specification */
   /********************************/

   if (MappedFile[0])
   {
      rqptr->RequestParseFab = cc$rms_fab;
      rqptr->RequestParseFab.fab$l_fna = MappedFile;
      rqptr->RequestParseFab.fab$b_fns = strlen(MappedFile);
      rqptr->RequestParseFab.fab$l_fop = FAB$M_NAM;
      rqptr->RequestParseFab.fab$l_nam = &rqptr->RequestParseNam;

      rqptr->RequestParseNam = cc$rms_nam;
      rqptr->RequestParseNam.nam$l_esa = rqptr->RequestFileName;
      rqptr->RequestParseNam.nam$b_ess = sizeof(rqptr->RequestFileName)-1;
      /* parse without disk I/O, syntax check only! */
      rqptr->RequestParseNam.nam$b_nop = NAM$M_SYNCHK;

      status = sys$parse (&rqptr->RequestParseFab, 0, 0);
      if (VMSok (status))
      {
         /* if its a directory specification then terminate after the ']' */
         if (rqptr->RequestParseNam.nam$l_name == 
             rqptr->RequestParseNam.nam$l_type)
            *rqptr->RequestParseNam.nam$l_name = '\0';
         else
            rqptr->RequestParseNam.nam$l_ver
               [rqptr->RequestParseNam.nam$b_ver] = '\0';
         ConfigContentType (&rqptr->ContentType,
                            rqptr->RequestParseNam.nam$l_type);
      }
      else
      {
         /* problem parsing (probably not intended as a file specification) */
         register char  *cptr, *sptr, *zptr;
         zptr = (sptr = rqptr->RequestFileName) +
                sizeof(rqptr->RequestFileName);
         for (cptr = MappedFile; *cptr && sptr < zptr; *sptr++ = *cptr++);
         if (sptr >= zptr) sptr = rqptr->RequestFileName;
         *sptr = '\0';
         while (sptr > rqptr->RequestFileName && *sptr != '.') sptr--;
         if (*sptr == '.') ConfigContentType (&rqptr->ContentType, sptr);
      }
   }
   else
      rqptr->RequestFileName[0] = '\0';

   if (Debug)
      fprintf (stdout, "rqptr->RequestFileName |%s|\n", rqptr->RequestFileName);

   /*********************************************/
   /* again check for internal server "scripts" */
   /*********************************************/

   if ((tolower(rqptr->ScriptName[0]) == tolower(HttpdInternalScriptUpd[0]) &&
        strsame (rqptr->ScriptName, HttpdInternalScriptUpd,
                 sizeof(HttpdInternalScriptUpd)-1)))
   {
      UpdBegin (rqptr, &RequestEnd);
      return;
   }
   else
   if ((tolower(rqptr->ScriptName[0]) == tolower(HttpdInternalScriptTree[0]) &&
        strsame (rqptr->ScriptName, HttpdInternalScriptTree,
                 sizeof(HttpdInternalScriptTree)-1)))
   {
      UpdBegin (rqptr, &RequestEnd);
      return;
   }
   else
   if (tolower(rqptr->ScriptName[0]) == tolower(HttpdInternalScriptXray[0]) &&
       strsame (rqptr->ScriptName, HttpdInternalScriptXray,
                sizeof(HttpdInternalScriptXray)-1))
   {
      /* indicate that it's an "Xray" and then cause a LOCAL redirection */
      if (Debug) fprintf (stdout, "Xray!\n");
      rqptr->Xray = true;
      /* the redirection location is the request minus "/xray" */
      rqptr->LocationPtr = rqptr->ResourcePtr +
                           sizeof(HttpdInternalScriptXray)-1;
      /* this can never be a legitimate keep-alive */
      rqptr->KeepAliveRequest = false;
      RequestEnd (rqptr);
      return;
   }

   /***************************/
   /* "real", external script */
   /***************************/

   for (cptr = MappedScript; *cptr && *cptr != ':'; cptr++);
   if (*(unsigned short*)cptr == '::')
      DECnetBegin (rqptr, &RequestEnd, MappedScript);
   else
      DclBegin (rqptr, &RequestEnd, NULL, MappedScript);
}

/*****************************************************************************/
/*
This simple little function is called directly and as an AST routine to echo
back to the client as a plain-text document the entire request header and any
body content. The call is set up in RequestScript() immediately above.
*/ 

RequestEcho (struct RequestStruct *rqptr)

{
   static char  EchoHeader [] =
"HTTP/1.0 200 echo\r\n\
Content-Type: text/plain\r\n\
Expires: Thu, 01-Jan-1970 00:00:01 GMT\r\n\
\r\n";

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "RequestEcho() %d %d\n",
               rqptr->ContentPtr, rqptr->ContentLength);

   if (!rqptr->ResponsePreExpired)
   {
      /* just detected that this is the setup call from RequestScript() */
      NetWrite (rqptr, &RequestEcho, EchoHeader, sizeof(EchoHeader)-1);
      rqptr->ResponseStatusCode = 200;
      rqptr->ResponsePreExpired = true;
      rqptr->ContentPtr = rqptr->ContentBufferPtr;
      return;
   }

   if (rqptr->RequestHeaderPtr != NULL)
   {
      /* AST from writing echo header, write response header */
      NetWrite (rqptr, &RequestEcho,
                rqptr->RequestHeaderPtr,
                rqptr->RequestHeaderLength);
      rqptr->RequestHeaderPtr = NULL;
      return;
   }

   /* write response body */
   if (rqptr->ContentLength <= OutputBufferSize)
   {
      /* queue a network write to the client, AST to end request processing */
      NetWrite (rqptr, &RequestEnd, rqptr->ContentPtr, rqptr->ContentLength);
      return;
   }
   else
   {   
      /* break it into smaller chunks */
      char  *cptr;
      cptr = rqptr->ContentPtr;
      rqptr->ContentPtr += OutputBufferSize;
      rqptr->ContentLength -= OutputBufferSize;
      /* queue a network write to the client, AST to echo more/rest */
      NetWrite (rqptr, &RequestEcho, cptr, OutputBufferSize);
   }
}

/*****************************************************************************/
/*
After the mapping the path, etc., parse it again (in case of error,
RequestScript() ignores errors) and return the VMS file name as a plain text
document.  This function merely translates any logicals, etc., and reports the
resulting name, it does not demonstrate the directory or file actually exists. 
The call is set up in RequestScript() immediately above.
*/ 

RequestWhere
(
struct RequestStruct *rqptr,
char *MappedFile
)
{
   static char  WhereHeader [] =
"HTTP/1.0 200 where\r\n\
Content-Type: text/plain\r\n\
Expires: Thu, 01-Jan-1970 00:00:01 GMT\r\n\
\r\n";

   int  status,
        Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RequestWhere()\n");

   rqptr->RequestParseFab = cc$rms_fab;
   rqptr->RequestParseFab.fab$l_fna = MappedFile;
   rqptr->RequestParseFab.fab$b_fns = strlen(MappedFile);
   rqptr->RequestParseFab.fab$l_fop = FAB$M_NAM;
   rqptr->RequestParseFab.fab$l_nam = &rqptr->RequestParseNam;

   rqptr->RequestParseNam = cc$rms_nam;
   rqptr->RequestParseNam.nam$l_esa = rqptr->RequestFileName;
   rqptr->RequestParseNam.nam$b_ess = sizeof(rqptr->RequestFileName)-1;
   /* parse without disk I/O, syntax check only! */
   rqptr->RequestParseNam.nam$b_nop = NAM$M_SYNCHK;

   if (VMSok (status = sys$parse (&rqptr->RequestParseFab, 0, 0)))
   {
      /* if its a directory specification then terminate after the ']' */
      if (rqptr->RequestParseNam.nam$l_name == 
          rqptr->RequestParseNam.nam$l_type)
      {
         *rqptr->RequestParseNam.nam$l_name = '\0';
         Length = rqptr->RequestParseNam.nam$l_name -
                  rqptr->RequestParseNam.nam$l_esa;
      }
      else
      {
         if (rqptr->RequestParseNam.nam$b_ver > 1)
         {
            rqptr->RequestParseNam.nam$l_ver
               [rqptr->RequestParseNam.nam$b_ver] = '\0';
            Length = rqptr->RequestParseNam.nam$b_ess;
         }
         else
         {
            *rqptr->RequestParseNam.nam$l_ver = '\0';
            Length = rqptr->RequestParseNam.nam$b_ess - 1;
         }
      }

      rqptr->ResponsePreExpired = true;
      rqptr->ResponseHeaderPtr = WhereHeader;
      rqptr->ResponseHeaderLength = sizeof(WhereHeader)-1;
      rqptr->ResponseStatusCode = 200;

      /* queue a network write to the client, AST to end processing */
      NetWrite (rqptr, &RequestEnd,
                rqptr->RequestParseNam.nam$l_esa, Length);
      return;
   }
   else
   {
      rqptr->ErrorTextPtr = rqptr->PathInfoPtr;
      ErrorVmsStatus (rqptr, status, FI_LI);
      RequestEnd (rqptr);
      return;
   }
}

/*****************************************************************************/
/*
Keeps a linked list history from most to least recent that can be reported on
by RequestReport().
*/

int RequestHistory (struct RequestStruct *rqptr)

{
   register char  *cptr, *sptr, *zptr;
   register struct RequestHistoryStruct  *hlptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RequestHistory()\n");

   if (RequestHistoryCount < RequestHistoryMax)
   {
      /* allocate memory for a new entry */
      hlptr = VmGet (sizeof (struct RequestHistoryStruct));
      RequestHistoryCount++;
   }
   else
   {
      /* reuse the tail entry (least recent) */
      hlptr = RequestHistoryList.TailPtr;
      ListRemove (&RequestHistoryList, hlptr);
   }

   /* add entry to the head of the history list (most recent) */
   ListAddHead (&RequestHistoryList, hlptr);

   memcpy (&hlptr->BinaryTime, &rqptr->BinaryTime, 8);
   hlptr->RequestScheme = rqptr->RequestScheme;
   hlptr->BytesRx = rqptr->BytesRx;
   hlptr->BytesTx = rqptr->BytesTx;
   hlptr->ResponseDuration = rqptr->ResponseDuration;
   hlptr->ResponseStatusCode = rqptr->ResponseStatusCode;

   zptr = (sptr = hlptr->ClientAndRequest) + sizeof(hlptr->ClientAndRequest);

   if (rqptr->RemoteUser[0])
   {
      cptr = rqptr->RemoteUser;
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      if (sptr < zptr) *sptr++ = '@';
   }
   cptr = rqptr->ClientHostName;
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr < zptr) *sptr++ = '\0';
   hlptr->RequestPtr = sptr;

   if (rqptr->PathInfoPtr != NULL && rqptr->QueryStringPtr != NULL)
   {
      cptr = rqptr->HttpMethodName;
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      if (sptr < zptr) *sptr++ = ' ';
      cptr = rqptr->ScriptName;
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      cptr = rqptr->PathInfoPtr;
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      if (rqptr->QueryStringPtr[0] && sptr < zptr) *sptr++ = '?';
      cptr = rqptr->QueryStringPtr;
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   }

   if (sptr >= zptr)
   {
      hlptr->Truncated = true;
      sptr--;
   }
   else
      hlptr->Truncated = false;
   *sptr = '\0';
}

/*****************************************************************************/
/*
Return a report on current and request history, listed most to least recent.
This function blocks while executing.
*/ 

int RequestReport
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   static long  Subx2 = 2,
                EdivTen = 10;

   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Request Report</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Request Report</H2>\n\
!AZ\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=7><FONT SIZE=+1>Current</FONT></TH></TR>\n\
<TR><TH></TH><TH>Client</TH><TH>SSL</TH><TH>Time</TH>\
<TH>Rx</TH><TH>Tx</TH><TH>Connected</TH></TR>\
<TR><TH></TH><TH COLSPAN=6>Request</TH></TR>\n\
<TR><TH COLSPAN=7></TH></TR>\n");

   static $DESCRIPTOR (CurrentRequestFaoDsc,
"<TR><TD ROWSPAN=2 ALIGN=right>!UL</TD>\
<TD ALIGN=left>!AZ</TD><TD>!AZ</TD><TD><FONT SIZE=-2>!AZ</FONT></TD>\
<TD ALIGN=right>!UL</TD>\
<TD ALIGN=right>!UL</TD>\
<TD ALIGN=right>!UL.!#ZL</TD></TR>\n\
<TR><TD ALIGN=left COLSPAN=6>\
<NOBR><TT>!AZ</TT></NOBR></TD></TR>\n");

   static $DESCRIPTOR (HistoryFaoDsc,
"</TABLE>\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=8><FONT SIZE=+1>History</FONT>\
 &nbsp;&nbsp;<FONT SIZE=-1>!AZ</FONT></TH></TR>\n\
<TR><TH></TH><TH>Client</TH><TH>SSL</TH><TH>Time</TH>\
<TH>Status</HT><TH>Rx</TH><TH>Tx</TH><TH>Duration</TH></TR>\n\
<TR><TH></TH><TH COLSPAN=7>Request</TH></TR>\n\
<TR><TH COLSPAN=7></TH></TR>\n");

   static $DESCRIPTOR (HistoryRequestFaoDsc,
"<TR><TD ROWSPAN=2 ALIGN=right>!UL</TD>\
<TD ALIGN=left>!AZ</TD><TD>!AZ</TD><TD><FONT SIZE=-2>!AZ</FONT></TD>\
<TD ALIGN=right>!UL</TD>\
<TD ALIGN=right>!UL</TD><TD ALIGN=right>!UL</TD>\
<TD ALIGN=right>!UL.!#ZL</TD></TR>\n\
<TR><TD ALIGN=left COLSPAN=7>\
<NOBR><TT>!AZ</TT></NOBR>!AZ</TD></TR>\n");

   static $DESCRIPTOR (EndPageFaoDsc,
"</TABLE>\n\
</BODY>\n\
</HTML>\n");

   static $DESCRIPTOR (KeepAliveFaoDsc, "&quot;Keep-Alive:&quot !UL\0");
   static $DESCRIPTOR (MaximumFaoDsc, "(maximum !UL)\0");

   register int  cnt;
   register unsigned long  *vecptr;
   register char  *cptr, *sptr, *zptr;
   register struct RequestStruct  *rqeptr;
   register struct RequestHistoryStruct  *rqhptr;
   register struct ListEntryStruct  *leptr;

   int  status,
        Count;
   unsigned short  Length;
   unsigned long  ResponseDuration,
                  Remainder;
   unsigned long  BinaryTime [2],
                  FaoVector [32],
                  ResultTime [2];
   char  Buffer [2048],
         Client [256],
         KeepAlive [64],
         Maximum [32],
         Request [1280],
         RequestBuffer [1024];
   $DESCRIPTOR (BufferDsc, Buffer);
   $DESCRIPTOR (KeepAliveDsc, KeepAlive);
   $DESCRIPTOR (MaximumDsc, Maximum);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RequestReport()\n");

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = DayDateTime (&rqptr->BinaryTime, 20);

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   /********************/
   /* current requests */
   /********************/

   Count = 0;

   sys$gettim (&BinaryTime);

   if (Debug)
      fprintf (stdout, "Head %d\nTail %d\n",
               RequestList.HeadPtr, RequestList.TailPtr);

   /* process the request list from least to most recent */
   for (leptr = RequestList.HeadPtr; leptr != NULL; leptr = leptr->NextPtr)
   {
      rqeptr = (struct RequestStruct*)leptr;

      if (Debug)
         fprintf (stdout, "%d <- %d -> %d (%d)\n",
                  leptr->PrevPtr, leptr, leptr->NextPtr, rqeptr);

      vecptr = FaoVector;

      *vecptr++ = ++Count;

      zptr = (sptr = Client) + sizeof(Client);
      if (rqeptr->RemoteUser[0])
      {
         cptr = rqeptr->RemoteUser;
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         if (sptr < zptr) *sptr++ = '@';
      }
      cptr = rqeptr->ClientHostName;
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      if (sptr >= zptr)
         ErrorExitVmsStatus (SS$_BUFFEROVF, "client", FI_LI);
      *sptr = '\0';
      *vecptr++ = Client;

      if (rqeptr->RequestScheme == SCHEME_HTTPS)
         *vecptr++ = "SSL";
      else
         *vecptr++ = "";
      *vecptr++ = DayDateTime (&rqeptr->BinaryTime, 23);
      *vecptr++ = rqeptr->BytesRx;
      *vecptr++ = rqeptr->BytesTx;

      status = lib$subx (&BinaryTime, &rqeptr->BinaryTime, 
                         &ResultTime, &Subx2);
      /* convert to microseconds (longword becomes 71 minutes max) */
      if (VMSok (status))
         status = lib$ediv (&EdivTen, &ResultTime,
                            &ResponseDuration, &Remainder);
      if (Debug) fprintf (stdout, "elapsed: %%X%08.08X\n", status);
      if (VMSnok (status)) ResponseDuration = 0;

      *vecptr++ = ResponseDuration / USECS_IN_A_SECOND;
      *vecptr++ = DURATION_DECIMAL_PLACES;
      *vecptr++ = (ResponseDuration % USECS_IN_A_SECOND) / DURATION_UNITS_USECS;

      if (rqeptr->PathInfoPtr == NULL || rqeptr->QueryStringPtr == NULL)
      {
         if (rqeptr->KeepAliveCount)
         {
            sys$fao (&KeepAliveFaoDsc, 0, &KeepAliveDsc,
                     rqeptr->KeepAliveCount);
            *vecptr++ = KeepAlive;
         }
         else
            *vecptr++ = "";
      }
      else
      {
         zptr = (sptr = Request) + sizeof(Request);
         cptr = rqeptr->HttpMethodName;
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         if (sptr < zptr) *sptr++ = ' ';
         cptr = rqeptr->ScriptName;
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         cptr = rqeptr->PathInfoPtr;
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         if (rqeptr->QueryStringPtr[0] && sptr < zptr) *sptr++ = '?';
         cptr = rqeptr->QueryStringPtr;
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         if (sptr >= zptr)
            ErrorExitVmsStatus (SS$_BUFFEROVF, "request", FI_LI);
         *sptr = '\0';

         /* provide a line-break every 80 characters */
         cnt = -1;
         zptr = (sptr = RequestBuffer) + sizeof(RequestBuffer);
         cptr = Request;
         while (*cptr && sptr < zptr)
         {
            while (*cptr && (cnt++ % 80) && sptr < zptr) *sptr++ = *cptr++;
            if (*cptr && cnt > 1 && sptr < zptr)
            {
               *sptr++ = '\n';
               if (sptr < zptr) *sptr++ = *cptr++;
            }
         }
         if (sptr >= zptr)
            ErrorExitVmsStatus (SS$_BUFFEROVF, "request", FI_LI);
         *sptr = '\0';

         if (CopyToHtml (Buffer, sizeof(Buffer), RequestBuffer, -1) < 0)
            ErrorExitVmsStatus (SS$_BUFFEROVF, "CopyToHtml()", FI_LI);

         /* continue with providing a line-break every 80 characters */
         zptr = (sptr = Request) + sizeof(Request);
         cptr = Buffer;
         while (*cptr && sptr < zptr)
         {
            while (*cptr && *cptr != '\n' && sptr < zptr) *sptr++ = *cptr++;
            if (*cptr && sptr+4 < zptr)
            {
               memcpy (sptr, "<BR>", 4);
               sptr += 4;
               *cptr++;
            }
         }
         if (sptr >= zptr)
            ErrorExitVmsStatus (SS$_BUFFEROVF, "request", FI_LI);
         *sptr = '\0';

         *vecptr++ = Request;
      }

      status = sys$faol (&CurrentRequestFaoDsc, &Length, &BufferDsc,
                         &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   /*******************/
   /* request history */
   /*******************/

   vecptr = FaoVector;
   if (RequestHistoryMax)
   {
      sys$fao (&MaximumFaoDsc, 0, &MaximumDsc, RequestHistoryMax);
      *vecptr++ = Maximum;
   }
   else
      *vecptr++ = "<I>(disabled)</I>";

   status = sys$faol (&HistoryFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   Count = 0;

   if (Debug)
      fprintf (stdout, "Head %d\nTail %d\n",
               RequestHistoryList.HeadPtr, RequestHistoryList.TailPtr);

   /* process the request list from least to most recent */
   for (leptr = RequestHistoryList.HeadPtr;
        leptr != NULL;
        leptr = leptr->NextPtr)
   {
      rqhptr = (struct RequestHistoryStruct*)leptr;

      if (Debug)
         fprintf (stdout, "%d <- %d -> %d (%d)\n",
                  leptr->PrevPtr, leptr, leptr->NextPtr, rqhptr);

      vecptr = FaoVector;
      *vecptr++ = ++Count;
      *vecptr++ = rqhptr->ClientAndRequest;
      if (rqhptr->RequestScheme == SCHEME_HTTPS)
         *vecptr++ = "SSL";
      else
         *vecptr++ = "";
      *vecptr++ = DayDateTime (&rqhptr->BinaryTime, 23);
      *vecptr++ = rqhptr->ResponseStatusCode;
      *vecptr++ = rqhptr->BytesRx;
      *vecptr++ = rqhptr->BytesTx;

      *vecptr++ = rqhptr->ResponseDuration / USECS_IN_A_SECOND;
      *vecptr++ = DURATION_DECIMAL_PLACES;
      *vecptr++ = (rqhptr->ResponseDuration % USECS_IN_A_SECOND) /
                  DURATION_UNITS_USECS;

      /* provide a line-break every 80 characters */
      cnt = -1;
      zptr = (sptr = RequestBuffer) + sizeof(RequestBuffer);
      cptr = rqhptr->RequestPtr;
      while (*cptr && sptr < zptr)
      {
         while (*cptr && (cnt++ % 80) && sptr < zptr) *sptr++ = *cptr++;
         if (*cptr && cnt > 1 && sptr < zptr)
         {
            *sptr++ = '\n';
            if (sptr < zptr) *sptr++ = *cptr++;
         }
      }
      if (sptr >= zptr)
         ErrorExitVmsStatus (SS$_BUFFEROVF, "request", FI_LI);
      *sptr = '\0';

      if (CopyToHtml (Buffer, sizeof(Buffer), RequestBuffer, -1) < 0)
         ErrorExitVmsStatus (SS$_BUFFEROVF, "CopyToHtml()", FI_LI);

      /* continue with providing a line-break every 80 characters */
      zptr = (sptr = Request) + sizeof(Request);
      cptr = Buffer;
      while (*cptr && sptr < zptr)
      {
         while (*cptr && *cptr != '\n' && sptr < zptr) *sptr++ = *cptr++;
         if (*cptr && sptr+4 < zptr)
         {
            memcpy (sptr, "<BR>", 4);
            sptr += 4;
            *cptr++;
         }
      }
      if (sptr >= zptr)
         ErrorExitVmsStatus (SS$_BUFFEROVF, "request", FI_LI);
      *sptr = '\0';

      *vecptr++ = Request;

      if (rqhptr->Truncated)
         *vecptr++ = "[truncated!]";
      else
         *vecptr++ = "";

      status = sys$faol (&HistoryRequestFaoDsc, &Length, &BufferDsc, 
                         &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }

      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   NetWriteBuffered (rqptr, NextTaskFunction,
                    EndPageFaoDsc.dsc$a_pointer,
                    EndPageFaoDsc.dsc$w_length);
}

/*****************************************************************************/

