/*****************************************************************************/
/*
                                 SeSoLa.c


    THE GNU GENERAL PUBLIC LICENSE APPLIES DOUBLY TO ANYTHING TO DO WITH
                    AUTHENTICATION AND AUTHORIZATION!

    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License, or any later
    version.

>   This package is distributed in the hope that it will be useful,
>   but WITHOUT ANY WARRANTY; without even the implied warranty of
>   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
>   GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


Secure Sockets Layer implemented using SSLeay v0.8.1
Named "SeSoLa" to avoid any confusion and/or conflict with SSLeay routines.

Supports SSL v2 and v3 protocols.

SSL I/O is implemented as a SSLeay BIO_METHOD, named "sesola_method". It
provides NON-BLOCKING SSL input/output. All routines that are part of this
functionality are named "sesola_..." and are grouped towards the end of this
module.


VERSION HISTORY
---------------
25-JAN-98  MGD  initial development for v5.0
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/* VMS related header files */
#include <ssdef.h>
#include <stsdef.h>

/* application header files */
#include "wasd.h"
#include "error.h"
#include "net.h"
#include "request.h"
#include "sesola.h"
#include "support.h"

/***************************************/
#ifdef SESOLA  /* secure sockets layer */
/***************************************/

/* SSLeay header files */
#include "bio.h"
#include "buffer.h"
#include "crypto.h"
#include "httpd.h"
#include "ssl.h"
#include "X509.h"

/**********/
/* macros */
/**********/

/* i.e. versions 2 and 3 */
#define DEFAULT_SSL_PROTOCOL_VERSION 23

#define SESOLA_DEBUG_STREAM 0

#define SESOLA_UNDEFINED           0
#define SESOLA_ACCEPT              1
#define SESOLA_SHUTDOWN            2
#define SESOLA_SHUT                3
#define SESOLA_READ_DATA_BLOCK     4
#define SESOLA_READ_DATA_NOBLOCK   5
#define SESOLA_WRITE_DATA_BLOCK    6
#define SESOLA_WRITE_DATA_NOBLOCK  7

/*******************/
/* data structures */
/*******************/

struct SeSoLaStruct
{
   /* SSLeay structures */
   SSL  *SslPtr;
   BIO  *BioPtr;
   BIO  *BioSslPtr;

   /* pointer back to the request owning this structure */
   struct RequestStruct  *rqptr;

   /*
       This "in-progress" flag is to provide an additional indication of
       when a SSLeay routine has used non-blocking I/O. It is only valid
       within the single AST-delivery context (i.e. to be checked immediately
       after using an SSLeay function, before returning out of the current
       AST delivery context), because as soon as the I/O AST is delivered
       the approriate flags will be reset.
   */
   boolean  IoInProgress;

   int  /* */
        AcceptCount,
        /* number of characters currently available in read buffer */
        ReadCount,
        /* size of read data buffer (for AST routine) */
        ReadDataSize,
        /* flag indicating purpose and/or type of network read */
        ReadType,
        /* number of characters actually written to network */
        WriteCount,
        /* number of characters to be written to network */
        WriteDataLength,
        /* flag indicating purpose and/or type of network write */
        WriteType;

  char  /* */
        *AcceptReadDataPtr,
        /* pointer to read data buffer (for AST routine use) */
        *ReadDataPtr,
        /* pointer to write data buffer (for AST routine use) */
        *WriteDataPtr;

   /* SSL post-processing, stores the pointer to the AST routine */
   void  *AstFunctionPtr;
};

/******************/
/* global storage */
/******************/

SSL_CTX  *SeSoLaCtx;
char  *SeSoLaCert,
      *SeSoLaCipher;

boolean  ProtocolHttpsAvailable,
         ProtocolHttpsConfigured;

int SeSoLaProtocolVersion = DEFAULT_SSL_PROTOCOL_VERSION;

char  HttpdSesola [] = " SSL";

char  ErrorSslPort [] = "This is an SSL (&quot;https:&quot;) port.";

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern int  NetReadBufferSize;
extern int  OutputBufferSize;
extern char  ErrorSanityCheck[];
extern char  HtmlSgmlDoctype[];
extern char  ServerHostPort[];
extern char  Utility[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/**********************************/
/* BIO method function prototypes */
/**********************************/

int sesola_write (BIO*, char*, int);
int sesola_write_ast (struct RequestStruct*);
int sesola_read (BIO*, char*, int);
int sesola_read_ast (struct RequestStruct*);
int sesola_puts (BIO*, char*);
int sesola_gets (BIO*, char*, int);
long sesola_ctrl (BIO*, int, long, char*);
int sesola_new (BIO*);
int sesola_free (BIO*);

BIO_METHOD *BIO_s_sesola();

BIO_METHOD sesola_method =
{
   BIO_TYPE_FD,
   "WASD SESOLA",
   sesola_write,
   sesola_read,
   sesola_puts,
   sesola_gets,
   sesola_ctrl,
   sesola_new,
   sesola_free,
};

/*****************************************************************************/
/*
*/

SeSoLaInit ()

{
   int  value;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SeSoLaInit()\n");

   if (SeSoLaProtocolVersion < 0) return;
   if (!SeSoLaProtocolVersion)
      SeSoLaProtocolVersion = DEFAULT_SSL_PROTOCOL_VERSION;
   if (SeSoLaProtocolVersion != 2 &&
       SeSoLaProtocolVersion != 3 &&
       SeSoLaProtocolVersion != 23)
   {
      fprintf (stdout, "%%%s-E-SSL, unknown SSL protocol version\n \\%d\\\n",
               Utility, SeSoLaProtocolVersion);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   ProtocolHttpsAvailable = true;

   fprintf (stdout, "%%%s-I-SSL, %s\n",
            Utility, SSLeay_version(SSLEAY_VERSION));

   if (Debug) SSL_load_error_strings ();

   SeSoLaCipher = getenv ("HTTPD$SSL_CIPHER");
   if (SeSoLaCipher == NULL)
      fprintf (stdout, "%%%s-I-SSL, using default ciphers\n", Utility);
   else
      fprintf (stdout, "%%%s-I-SSL, using cipher(s)\n \\%s\\\n",
               Utility, SeSoLaCipher);

   SeSoLaCert = getenv ("HTTPD$SSL_CERT");
   if (SeSoLaCert == NULL)
      ErrorExitVmsStatus (0, "SSL certificate not specified", FI_LI);

   fprintf (stdout, "%%%s-I-SSL, using server certificate\n \\%s\\\n",
            Utility, SeSoLaCert);

   SSLeay_add_ssl_algorithms(); 

   if (SeSoLaProtocolVersion == 2)
      SeSoLaCtx = SSL_CTX_new (SSLv2_server_method());
   else
   if (SeSoLaProtocolVersion == 3)
      SeSoLaCtx = SSL_CTX_new (SSLv3_server_method());
   else
   if (SeSoLaProtocolVersion == 23)
      SeSoLaCtx = SSL_CTX_new (SSLv23_server_method());
   if (SeSoLaCtx == NULL)
      ErrorExitVmsStatus (0, "SSL CTX returned null", FI_LI);

   /* turn on SYSPRV to ensure access to the certificate file */
   EnableSysPrv ();

   value = SSL_CTX_use_certificate_file (SeSoLaCtx, SeSoLaCert,
                                         SSL_FILETYPE_PEM);
   if (!value)
      ErrorExitVmsStatus (0, "problem with certificate", FI_LI);

   value = SSL_CTX_use_RSAPrivateKey_file (SeSoLaCtx, SeSoLaCert,
                                           SSL_FILETYPE_PEM);
   if (!value)
      ErrorExitVmsStatus (0, "problem with private key", FI_LI);

   DisableSysPrv ();
}

/*****************************************************************************/
/*
Begin an SSL transaction for a request. Create the SESOLA structure used to
store HTTPd SSL-related used during the transaction, initialize the SSLeay
structures required, then begin the SSLeay accept functionality.
*/

SeSoLaBegin (struct RequestStruct *rqptr)

{
   register struct SeSoLaStruct  *sslptr;

   int  value;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SeSoLaBegin() %d\n", rqptr);

   /* note that this IS NOT HEAP memory, and must be explicitly VmFree()d */
   rqptr->SeSoLaPtr = sslptr =
      (struct SeSoLaStruct*)VmGet (sizeof(struct SeSoLaStruct));

   if (Debug) fprintf (stdout, "SSL_new()\n");
   sslptr->SslPtr = SSL_new (SeSoLaCtx);
   if (sslptr->SslPtr == NULL)
   {
      fprintf (stdout, "%%%s-W-SSL, SSL_new(SeSoLaCtx) failed\n", Utility);
      SeSoLaFree (rqptr);
      return;
   }

   if (Debug) fprintf (stdout, "BIO_new()\n");
   sslptr->BioPtr = BIO_new (BIO_s_sesola());
   if (sslptr->BioPtr == NULL)
   {
      fprintf (stdout, "%%%s-W-SSL, BIO_new(BIO_s_sesola()) failed\n",
               Utility);
      SeSoLaFree (rqptr);
      return;
   }
   /* set the "BIO_s_sesola" pointer to the SESOLA structure */
   sslptr->BioPtr->ptr = sslptr;

   /* set up the SSL filter */
   if (Debug) fprintf (stdout, "BIO_new()\n");
   sslptr->BioSslPtr = BIO_new (BIO_f_ssl());
   if (sslptr->BioSslPtr == NULL)
   {
      fprintf (stdout, "%%%s-W-SSL, BIO_new(BIO_f_ssl()) failed\n", Utility);
      SeSoLaFree (rqptr);
      return;
   }

   SSL_set_bio (sslptr->SslPtr, sslptr->BioPtr, sslptr->BioPtr);

/*
   BIO_set_ssl (sslptr->BioSslPtr, sslptr->SslPtr, BIO_CLOSE);
*/

   /* set the SeSoLa structure request pointer */
   sslptr->rqptr = rqptr;

   SSL_set_accept_state (sslptr->SslPtr);

   SSL_set_shutdown (sslptr->SslPtr, 0);
}

/*****************************************************************************/
/*
Shutdown the SSL session with the client (via an SSL alert that we don't wait
around for a response on!)  Return true if RequestEnd() can continue
the request run-down, false if has to abort and wait for some more to happen!
*/

boolean SeSoLaEnd (struct RequestStruct *rqptr)

{
   register struct SeSoLaStruct  *sslptr;

   int  value;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SeSoLaEnd() %d\n", rqptr);

   /* the 'rqptr->SeSoLaPtr' is typeless! */
   sslptr = (struct SeSoLaStruct*)rqptr->SeSoLaPtr;

   if (sslptr->ReadType == SESOLA_SHUT &&
       sslptr->WriteType == SESOLA_SHUT)
   {
      /* final call from SysDclAst() below invoked RequestEnd() */
      return (true);
   }

   if (sslptr->ReadType == SESOLA_SHUTDOWN &&
       sslptr->WriteType == SESOLA_SHUTDOWN)
   {
      /* AST call from sesola_write() in SSL_shutdown() below */
      sslptr->ReadType = sslptr->WriteType = SESOLA_SHUT;
      SSL_set_shutdown (sslptr->SslPtr,
                        SSL_SENT_SHUTDOWN | SSL_RECEIVED_SHUTDOWN);
      /* just in case there's some non-block I/O I don't count on! */
      if (sslptr->IoInProgress) return (false);

      /* now, run down the request again */
      SysDclAst (&RequestEnd, rqptr);
      return (false);
   }

   sslptr->ReadType = sslptr->WriteType = SESOLA_SHUTDOWN;

   /* send the SSL close alert to the client */
   value = SSL_shutdown (sslptr->SslPtr);
   if (Debug) fprintf (stdout, "SSL_shutdown() %d\n", value);

   /* return if the shutdown is not yet complete (write underway) */
   if (value <= 0 || sslptr->IoInProgress) return (false);

   /* looks like we're all shutdown, just continue RequestEnd() */
   sslptr->ReadType = sslptr->WriteType = SESOLA_SHUT;
   return (true);
}

/*****************************************************************************/
/*
Free the SSLeay structures outside of any use of them (early mistake)!
*/

boolean SeSoLaFree (struct RequestStruct *rqptr)

{
   register struct SeSoLaStruct  *sslptr;

   int  value;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SeSoLaFree() %d\n", rqptr);

   /* the 'rqptr->SeSoLaPtr' is typeless! */
   sslptr = (struct SeSoLaStruct*)rqptr->SeSoLaPtr;

   if (sslptr->BioPtr != NULL) BIO_free (sslptr->BioPtr);

   /** if (sslptr->BioSslPtr != NULL) BIO_free (sslptr->BioSslPtr); **/

   if (sslptr->SslPtr != NULL)
   {
      /** sslptr->SslPtr->rbio = sslptr->SslPtr->wbio = NULL; **/
      SSL_free (sslptr->SslPtr);
   }

   /* request has finished with the SESOLA structure */
   VmFree (sslptr);
   rqptr->SeSoLaPtr = NULL;
}

/*****************************************************************************/
/*
This establishes the SSL transaction by providing the server "hello",
certificate and key exchange, etc. Due to the non-blocking I/O used by WASD
this function will be called multiple times to complete the SSLeay accept. 
*/

SeSoLaAccept (struct RequestStruct *rqptr)

{
   register struct SeSoLaStruct  *sslptr;

   int  status,
        value;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SeSoLaAccept() %d\n", rqptr);

   /* the 'rqptr->SeSoLaPtr' is typeless! */
   sslptr = (struct SeSoLaStruct*)rqptr->SeSoLaPtr;

   sslptr->ReadType = sslptr->WriteType = SESOLA_ACCEPT;

   if (++sslptr->AcceptCount == 2)
   {
      /* just finished the first lot of non-blocking I/O */
      if (!memcmp (sslptr->AcceptReadDataPtr, "GET ", 4) ||
          !memcmp (sslptr->AcceptReadDataPtr, "HEAD ", 5) ||
          !memcmp (sslptr->AcceptReadDataPtr, "POST ", 5) ||
          !memcmp (sslptr->AcceptReadDataPtr, "PUT ", 4) ||
          !memcmp (sslptr->AcceptReadDataPtr, "DELETE ", 7))
      {
         /* looks like it's a standard HTTP request on the wrong port! */
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, ErrorSslPort, FI_LI);
         SysDclAst (&RequestEnd, rqptr);
         SeSoLaFree (rqptr);
         return;
      }
   }

   value = SSL_accept (sslptr->SslPtr);
   if (Debug) fprintf (stdout, "SSL_accept() %d\n", value);

#ifdef DBUG
   if (Debug)
   {
      char  String [256];
      fprintf (stdout, "|%s|\n",
         ERR_error_string (SSL_get_error (sslptr->SslPtr, value), String));
      SeSoLaDebugSslEayError (sslptr, value);
   }
#endif

   switch (SSL_get_error (sslptr->SslPtr, value))
   {
      case SSL_ERROR_NONE :
         break;
      case SSL_ERROR_WANT_WRITE :
         return;
      case SSL_ERROR_WANT_READ :
         return;
      case SSL_ERROR_WANT_X509_LOOKUP :
         return;
      case SSL_ERROR_SYSCALL :
      case SSL_ERROR_SSL :
      case SSL_ERROR_ZERO_RETURN :
         if (value == -1 || sslptr->IoInProgress)
         {
            NetShutdownSocket (rqptr, 0, true);
            return;
         }
         SysDclAst (&RequestEnd, rqptr);
         return;
      default :
         ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
   }

#ifdef DBUG
   if (Debug) SeSoLaDebugSession (sslptr);
#endif

   rqptr->NetReadBufferPtr = VmGetHeap (rqptr, NetReadBufferSize);
   rqptr->NetReadBufferSize = NetReadBufferSize;

   NetRead (rqptr, &RequestGet,
            rqptr->NetReadBufferPtr, rqptr->NetReadBufferSize);
}

/*****************************************************************************/
/*
Read application data. This is the equivalent of the general NetRead() function
(and is indeed called from it) in that it reads data from the client via the
network. This function uses SSL_read() to read a stream of encrypted data from
the network then provide it decrypted in the supplied buffer.  This funtction
provides BLOCKING (non- AST) and NON-BLOCKING I/O conforming to the
functionality provided by NetRead(). The network read I/O status block status
and count fields are valid for the high-level calling routine.
*/

SeSoLaRead
(
struct RequestStruct *rqptr,
void *AstFunctionPtr,
char *DataPtr,
int DataSize
)
{
   register struct SeSoLaStruct  *sslptr;

   int  value;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SeSoLaRead() %d\n", rqptr);

   /* the 'rqptr->SeSoLaPtr' is typeless! */
   sslptr = (struct SeSoLaStruct*)rqptr->SeSoLaPtr;

   if (Debug) fprintf (stdout, "sslptr->ReadCount: %d\n", sslptr->ReadCount);

   if (!sslptr->ReadCount)
   {
      if (AstFunctionPtr == NULL)
         sslptr->ReadType = SESOLA_READ_DATA_BLOCK;
      else
      {
         sslptr->ReadType = SESOLA_READ_DATA_NOBLOCK;
         sslptr->AstFunctionPtr = AstFunctionPtr;
      }
   }

   value = SSL_read (sslptr->SslPtr,
                     sslptr->ReadDataPtr = DataPtr,
                     sslptr->ReadDataSize = DataSize);
   if (Debug) fprintf (stdout, "SSL_read() %d\n", value);

   /* return if non-blocking I/O is in progress */
   if (value == -1 || sslptr->IoInProgress) return (SS$_NORMAL);

   /* fudge the I/O status block count to reflect the application data count */
   rqptr->NetReadIOsb.Count = value;

   if (AstFunctionPtr == NULL) return (rqptr->NetReadIOsb.Status);

   SysDclAst (AstFunctionPtr, rqptr);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Write application data. This is the equivalent of the general NetWrite()
function (and is called from it) in that it writes data to the client via the
network. This function uses SSL_write() to write non-encrypted application data
to the network in encrypted form. This function provides BLOCKING (non-AST)
and NON-BLOCKING I/O conforming to the functionality provided by NetWrite().
The network write I/O status block status and count fields are valid for the
high-level calling routine.
*/

int SeSoLaWrite
(
struct RequestStruct *rqptr,
void *AstFunctionPtr,
char *DataPtr,
int DataLength
)
{
   register struct SeSoLaStruct  *sslptr;

   int  value;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SeSoLaWrite() %d %d %d\n", rqptr, DataPtr, DataLength);
   /** fprintf (stdout, "|%*.*s|\n", DataLength, DataLength, DataPtr); **/

   /* the 'rqptr->SeSoLaPtr' is typeless! */
   sslptr = (struct SeSoLaStruct*)rqptr->SeSoLaPtr;

   if (Debug) fprintf (stdout, "sslptr->WriteCount: %d\n", sslptr->WriteCount);

   if (!sslptr->WriteCount)
   {
      if (AstFunctionPtr == NULL)
         sslptr->WriteType = SESOLA_WRITE_DATA_BLOCK;
      else
      {
         sslptr->WriteType = SESOLA_WRITE_DATA_NOBLOCK;
         sslptr->AstFunctionPtr = AstFunctionPtr;
      }
   }

   value = SSL_write (sslptr->SslPtr,
                      sslptr->WriteDataPtr = DataPtr,
                      sslptr->WriteDataLength = DataLength);
   if (Debug) fprintf (stdout, "SSL_write() %d\n", value);

   /* return if non-blocking I/O is in progress */
   if (value == -1 || sslptr->IoInProgress) return (SS$_NORMAL);

   if (AstFunctionPtr == NULL) return (rqptr->NetReadIOsb.Status);

   SysDclAst (AstFunctionPtr, rqptr);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

SeSoLaReport
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... SSL Report</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>SSL Report</H2>\n\
!AZ\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=3><FONT SIZE=+1>General</FONT></TH></TR>\n\
<TR><TH>Version</TH><TD COLSPAN=2>!AZ</TD></TR>\n\
<TR><TH ROWSPAN=2>Server Accepts</TH><TH>Total</TH><TD>!UL</TD></TR>\n\
<TR><TH>Completed</TH><TD>!UL</TD></TR>\n\
<TR><TH ROWSPAN=3>Session Cache</TH><TH>Hits</TH><TD>!UL</TD></TR>\n\
<TR><TH>Misses</TH><TD>!UL</TD></TR>\n\
<TR><TH>Timeouts</TH><TD>!UL</TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (SupportedCiphersFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=3><FONT SIZE=+1>Supported Ciphers</FONT></TH></TR>\n\
<TR><TH></TH><TH>Version</TH><TH>Name</TH></TR>\n");

   static $DESCRIPTOR (CiphersFaoDsc,
"<TR><TH>!UL</TH><TD><TT>!AZ</TT></TD><TD><TT>!AZ</TT></TD></TR>\n");

   static $DESCRIPTOR (CurrentSession1FaoDsc,
"</TABLE>\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH COLSPAN=3><FONT SIZE=+1>Current Session</FONT></TH></TR>\n\
<TR><TH>Cache Hit</TH><TD COLSPAN=2>!AZ</TD></TR>\n\
<TR><TH>Version</TH><TD COLSPAN=2>!AZ</TD></TR>\n\
<TR><TH ROWSPAN=!UL>Shared Ciphers</TH>");

   static $DESCRIPTOR (SharedCipherFaoDsc,
"!AZ<TH>!UL</TH><TD><TT>!AZ</TT></TD></TR>\n");

   static $DESCRIPTOR (CurrentSession2FaoDsc,
"<TR><TH>Cipher</TH><TD COLSPAN=2>!AZ</TD></TR>\n\
<TR><TH>Session-ID</TH><TD COLSPAN=2><TT>!#(2XB)</TT></TD></TR>\n\
<TR><TH>Master-Key</TH><TD COLSPAN=2><TT>!#(2XB)</TT></TD></TR>\n\
<TR><TH>Key-Arg</TH><TD COLSPAN=2><TT>!#(2XB)</TT></TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (EndPageFaoDsc,
"</TABLE>\n\
</BODY>\n\
</HTML>\n");

   register unsigned long  *vecptr;
   register char  *cptr, *sptr, *zptr;
   register struct SeSoLaStruct  *sslptr;

   int  status,
        Count,
        Total;
   unsigned short  Length;
   unsigned long  BinaryTime [2],
                  FaoVector [256],
                  ResultTime [2];
   char  Buffer [2048],
         SharedCiphers [256];
   $DESCRIPTOR (BufferDsc, Buffer);
   STACK  *StackPtr;
   SSL_CIPHER  *CipherPtr;
   SSL_SESSION  *SessionPtr;
   X509  *PeerPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SeSolaReport()\n");

   /* the 'rqptr->SeSoLaPtr' is typeless! */
   sslptr = (struct SeSoLaStruct*)rqptr->SeSoLaPtr;

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = DayDateTime (&rqptr->BinaryTime, 20);

   *vecptr++ = SSLeay_version (SSLEAY_VERSION);
   *vecptr++ = SSL_CTX_sess_accept (SeSoLaCtx);
   *vecptr++ = SSL_CTX_sess_accept_good (SeSoLaCtx);
   *vecptr++ = SSL_CTX_sess_hits (SeSoLaCtx);
   *vecptr++ = SSL_CTX_sess_misses (SeSoLaCtx);
   *vecptr++ = SSL_CTX_sess_timeouts (SeSoLaCtx);

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   if (rqptr->RequestScheme != SCHEME_HTTPS)
   {
      NetWriteBuffered (rqptr, NextTaskFunction,
                       EndPageFaoDsc.dsc$a_pointer,
                       EndPageFaoDsc.dsc$w_length);
      return;
   }

   /*********************/
   /* supported ciphers */
   /*********************/

   NetWriteBuffered (rqptr, 0,
                     SupportedCiphersFaoDsc.dsc$a_pointer,
                     SupportedCiphersFaoDsc.dsc$w_length);

   StackPtr = SSL_get_ciphers (sslptr->SslPtr);
   Total = sk_num (StackPtr);
   for (Count = 0; Count < Total; Count++)
   {
      CipherPtr = (SSL_CIPHER *)sk_value (StackPtr, Count);

      vecptr = FaoVector;
      *vecptr++ = Count + 1;
      *vecptr++ = SSL_CIPHER_get_version (CipherPtr);
      *vecptr++ = SSL_CIPHER_get_name (CipherPtr);

      status = sys$faol (&CiphersFaoDsc, &Length, &BufferDsc, &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }
      Buffer[Length] = '\0';
      NetWriteBuffered (rqptr, 0, Buffer, Length);
   }

   /*******************/
   /* current session */
   /*******************/

   vecptr = FaoVector;

   if (sslptr->SslPtr->hit)
      *vecptr++ = "yes";
   else
      *vecptr++ = "no";

   CipherPtr = SSL_get_current_cipher (sslptr->SslPtr);

   /* can't really be null or we would not be here! */
   if (CipherPtr != NULL)
      *vecptr++ = SSL_CIPHER_get_version (CipherPtr);

   cptr = sptr = SSL_get_shared_ciphers (sslptr->SslPtr,
                    SharedCiphers, sizeof(SharedCiphers));
   Count = 0;
   /* there has to be at least one or we would not be here! */
   if (cptr != NULL)
   {
      while (*cptr)
      {
         while (*cptr && *cptr != ':') cptr++;
         if (*cptr) cptr++;
         Count++;
      }
   }
   *vecptr++ = Count;

   status = sys$faol (&CurrentSession1FaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }
   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   /* has to be at least one or we would not be here! */
   cptr = sptr;
   if (cptr != NULL)
   {
      Count = 0;
      while (*cptr)
      {
         vecptr = FaoVector;

         sptr = cptr;
         while (*cptr && *cptr != ':') cptr++;
         if (*cptr) *cptr++ = '\0';

         if (Count++)
            *vecptr++ = "<TR>";
         else
            *vecptr++ = "";

         *vecptr++ = Count;
         *vecptr++ = sptr;

         status = sys$faol (&SharedCipherFaoDsc, &Length, &BufferDsc,
                            &FaoVector);
         if (VMSnok (status) || status == SS$_BUFFEROVF)
         {
            rqptr->ErrorTextPtr = "sys$faol()";
            ErrorVmsStatus (rqptr, status, FI_LI);
            SysDclAst (NextTaskFunction, rqptr);
            return;
         }
         Buffer[Length] = '\0';
         NetWriteBuffered (rqptr, 0, Buffer, Length);
      }
   }

   vecptr = FaoVector;

   /* can't really be null or we would not be here! */
   if (CipherPtr != NULL)
      *vecptr++ = SSL_CIPHER_get_name (CipherPtr);

   /*******************************************/
   /* eqivalent to SSLeay SSL_session_print() */
   /*******************************************/

   SessionPtr = SSL_get_session (sslptr->SslPtr);

   *vecptr++ = SessionPtr->session_id_length;
   for (Count = 0; Count < (int)SessionPtr->session_id_length; Count++)
      *vecptr++ = SessionPtr->session_id[Count];

   *vecptr++ = SessionPtr->master_key_length;
   for (Count = 0; Count < (int)SessionPtr->master_key_length; Count++)
      *vecptr++ = SessionPtr->master_key[Count];

   *vecptr++ = SessionPtr->key_arg_length;
   for (Count = 0; Count < (int)SessionPtr->key_arg_length; Count++)
      *vecptr++ = SessionPtr->key_arg[Count];

   status = sys$faol (&CurrentSession2FaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }
   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   NetWriteBuffered (rqptr, NextTaskFunction,
                     EndPageFaoDsc.dsc$a_pointer,
                     EndPageFaoDsc.dsc$w_length);
}

/*****************************************************************************/
/*
*/

#ifdef DBUG

SeSoLaDebugSslEayError
(
struct SeSoLaStruct *sslptr,
int value
)
{
   int  number;

   /*********/
   /* begin */
   /*********/

   fprintf (stdout, "SeSoLaDebugSslEayError() %d\n", value);

   if (value >= 0)
   {
      fprintf (stdout, "(0) -NO ERROR-\n");
      return;
   }
   number = SSL_get_error (sslptr->SslPtr, value);
   fprintf (stdout, "(%d) %d == ", value, number);
   switch (number)
   {
      case SSL_ERROR_NONE :
         if (Debug) fprintf (stdout, "SSL_ERROR_NONE\n");
         return;
      case SSL_ERROR_WANT_WRITE :
         if (Debug) fprintf (stdout, "SSL_ERROR_WANT_WRITE\n");
         return;
      case SSL_ERROR_WANT_READ :
         if (Debug) fprintf (stdout, "SSL_ERROR_WANT_READ\n");
         return;
      case SSL_ERROR_WANT_X509_LOOKUP :
         if (Debug) fprintf (stdout, "SSL_ERROR_WANT_X509_LOOKUP\n");
         return;
      case SSL_ERROR_SYSCALL :
         if (Debug) fprintf (stdout, "SSL_ERROR_SYSCALL\n");
         return;
      case SSL_ERROR_SSL :
         if (Debug) fprintf (stdout, "SSL_ERROR_SSL\n");
         return;
      case SSL_ERROR_ZERO_RETURN :
         if (Debug) fprintf (stdout, "SSL_ERROR_ZERO_RETURN\n");
         return;
      default :
         if (Debug) fprintf (stdout, "?????\n");
         return;
   }
}

#endif

/*****************************************************************************/
/*
*/

#ifdef DBUG

SeSoLaDebugStream
(
char *DataPtr,
int DataLength
)
{
   /*********/
   /* begin */
   /*********/

   fprintf (stdout, "SeSoLaDebugStream() %d %d\n", DataPtr, DataLength);

   if (!SESOLA_DEBUG_STREAM) return;

   while (DataLength-- > 0)
      fprintf (stdout, "%02.02x ", (unsigned char)*DataPtr++);
   fprintf (stdout, "\n");
}

#endif

/*****************************************************************************/
/*
*/

#ifdef DBUG

SeSoLaDebugSession (struct SeSoLaStruct *sslptr)

{
   static BIO *bio_stdout = NULL;

   int  Count,
        Total;
   char  *cptr;
   char  Buffer [1024];
   STACK  *StackPtr;
   SSL_CIPHER  *CipherPtr;
   X509  *PeerPtr;

   /*********/
   /* begin */
   /*********/

   fprintf (stdout, "SeSoLaDebugSession()\n");

   if (bio_stdout == NULL)
      if ((bio_stdout = BIO_new (BIO_s_file())) != NULL)
         BIO_set_fp (bio_stdout, stdout, BIO_CLOSE);

   BIO_printf (bio_stdout, "----------\n");

   BIO_puts (bio_stdout, SSLeay_version (SSLEAY_VERSION));
   BIO_puts (bio_stdout, "\n");

   BIO_printf (bio_stdout, "Statistics:\n");
/*
   BIO_printf (bio_stdout, "%4ld items in the session cache\n",
               SSL_CTX_sess_number (SeSoLaCtx));
*/
   BIO_printf (bio_stdout, "%4d client connects (SSL_connect())\n",
               SSL_CTX_sess_connect (SeSoLaCtx));
   BIO_printf (bio_stdout, "%4d client connects that finished\n",
               SSL_CTX_sess_connect_good (SeSoLaCtx));
   BIO_printf (bio_stdout, "%4d server accepts (SSL_accept())\n",
               SSL_CTX_sess_accept (SeSoLaCtx));
   BIO_printf (bio_stdout, "%4d server accepts that finished\n",
               SSL_CTX_sess_accept_good (SeSoLaCtx));
   BIO_printf (bio_stdout, "%4d session cache hits\n",
               SSL_CTX_sess_hits (SeSoLaCtx));
   BIO_printf (bio_stdout, "%4d session cache misses\n",
               SSL_CTX_sess_misses (SeSoLaCtx));
   BIO_printf (bio_stdout, "%4d session cache timeouts\n",
               SSL_CTX_sess_timeouts (SeSoLaCtx));
   BIO_printf (bio_stdout, "%4d callback cache hits\n",
               SSL_CTX_sess_cb_hits (SeSoLaCtx));
   BIO_printf (bio_stdout, "\n");

   StackPtr = SSL_get_ciphers (sslptr->SslPtr);
   Total = sk_num (StackPtr);
   BIO_printf (bio_stdout, "%d ciphers supported:\n", Total);
   for (Count = 0; Count < Total; Count++)
   {
      CipherPtr = (SSL_CIPHER *)sk_value (StackPtr, Count);
      BIO_printf (bio_stdout, "%s:%-25s",
                  SSL_CIPHER_get_version (CipherPtr),
                  SSL_CIPHER_get_name (CipherPtr));
      if ((((Count+1)%3) == 0) && (Count+1 != Total))
         BIO_puts (bio_stdout, "\n");
   }
   BIO_puts (bio_stdout, "\n");
   cptr = SSL_get_shared_ciphers (sslptr->SslPtr, Buffer, sizeof(Buffer));
   if (cptr != NULL)
   {
      BIO_printf (bio_stdout,
         "\nCiphers common between both SSL end points:\n");
      Total = Count = 0;
      while (*cptr)
      {
         if (*cptr == ':')
         {
            BIO_write (bio_stdout, "               ", 15-Count);
            Total++;
            Count=0;
            BIO_write (bio_stdout, ((Total%3) ? " " : "\n"), 1);
         }
         else
         {
            BIO_write (bio_stdout, cptr, 1);
            Count++;
         }
         cptr++;
      }
      BIO_puts (bio_stdout, "\n");
   }
   BIO_printf (bio_stdout, ((sslptr->SslPtr->hit) ? "\nReused, " : "\nNew, "));
   CipherPtr = SSL_get_current_cipher (sslptr->SslPtr);
   if (CipherPtr == NULL)
      BIO_printf (bio_stdout, "(none current)\n");
   else
   {
      BIO_printf (bio_stdout, "%s, Cipher is %s\n",
                  SSL_CIPHER_get_version (CipherPtr),
                  SSL_CIPHER_get_name (CipherPtr));
      SSL_SESSION_print (bio_stdout, SSL_get_session (sslptr->SslPtr));
   }
   BIO_printf (bio_stdout, "\n");

   BIO_printf (bio_stdout, "Client certificate:\n");
   PeerPtr = SSL_get_peer_certificate (sslptr->SslPtr);
   if (PeerPtr == NULL)
      BIO_puts (bio_stdout, "(no client certificate available)\n");
   else
   {
      X509_print (bio_stdout, PeerPtr);
      PEM_write_bio_X509 (bio_stdout, PeerPtr);
   }

   BIO_printf (bio_stdout, "\n----------\n");
}

#endif

/*****************************************************************************/
/*
Implements a required function of a SSLeay BIO_METHOD.
*/

BIO_METHOD *BIO_s_sesola()
{
   return (&sesola_method);
}

/*****************************************************************************/
/*
Implements a required function of a SSLeay BIO_METHOD.
*/

int sesola_new (BIO *bioptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "sesola_new()\n");

   bioptr->init = 1;
   bioptr->shutdown = 1;
   bioptr->num = 0;
   bioptr->flags = 0;
   /* the BIO pointer is set to the SESOLA structure by the calling routine */
   return (1);
}

/*****************************************************************************/
/*
Implements a required function of a SSLeay BIO_METHOD.
*/

int sesola_free (BIO *bioptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "sesola_free()\n");

   if (bioptr == NULL) return (0);
   /* don't free anything, the BIO pointer is set to the SESOLA structure */
   bioptr->ptr = NULL;
   return (1);
}
	
/*****************************************************************************/
/*
Implements a required function of a SSLeay BIO_METHOD.
*/

int sesola_read
(
BIO *bioptr,
char *DataPtr,
int DataSize
)
{
   register struct SeSoLaStruct  *sslptr;
   register struct RequestStruct  *rqptr;

   int  status,
        Count;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "sesola_read() %d %d\n", DataPtr, DataSize);

   sslptr = (struct SeSoLaStruct*)bioptr->ptr;
   rqptr = sslptr->rqptr;

   BIO_clear_retry_flags (bioptr);

   if (Debug)
      fprintf (stdout, "sslptr->ReadCount: %d\n", sslptr->ReadCount);

   if (sslptr->ReadCount)
   {
      /* will only be non-zero due to a previous non-blocking read */
      Count = sslptr->ReadCount;
      sslptr->ReadCount = 0;
#ifdef DBUG
      if (Debug && Count >= 0) SeSoLaDebugStream (DataPtr, Count);
#endif
      return (Count);
   }

   if (sslptr->ReadType == SESOLA_READ_DATA_BLOCK)
   {
      /* blocking I/O */
      sslptr->IoInProgress = false;
      status = NetReadRaw (rqptr, NULL, DataPtr, DataSize);
      /* if success return the number of characters written */
      if (VMSok (status)) return (rqptr->NetReadIOsb.Count);
      /* whoa, error */
      rqptr->NetReadIOsb.Status = status;
      return (-2);
   }
   else
   {
      /* non-blocking I/O */
      sslptr->IoInProgress = true;
      BIO_set_retry_read (bioptr);
      if (sslptr->ReadType == SESOLA_ACCEPT)
         sslptr->AcceptReadDataPtr = DataPtr;
      NetReadRaw (rqptr, &sesola_read_ast, DataPtr, DataSize);
      /* return indicating non-blocking I/O */
      return (-1);
   }
}

/*****************************************************************************/
/*
*/

int sesola_read_ast (struct RequestStruct *rqptr)

{
   register struct SeSoLaStruct  *sslptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "sesola_read_ast() %%X%08.08X %d\n",
               rqptr->NetReadIOsb.Status, rqptr->NetReadIOsb.Count);

   sslptr = (struct SeSoLaStruct*)rqptr->SeSoLaPtr;

   sslptr->IoInProgress = false;

   if (VMSok (rqptr->NetReadIOsb.Status))
      sslptr->ReadCount = rqptr->NetReadIOsb.Count;
   else
      sslptr->ReadCount = -2;

   if (sslptr->ReadType == SESOLA_READ_DATA_NOBLOCK)
      SeSoLaRead (rqptr, sslptr->AstFunctionPtr,
                  sslptr->ReadDataPtr, sslptr->ReadDataSize);
   else
   if (sslptr->ReadType == SESOLA_ACCEPT)
      SeSoLaAccept (rqptr);
   else
   if (sslptr->ReadType == SESOLA_SHUTDOWN)
      SeSoLaEnd (rqptr);
   else
      ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
}

/*****************************************************************************/
/*
Implements a required function of a SSLeay BIO_METHOD.
*/

int sesola_write
(
BIO *bioptr,
char *DataPtr,
int DataLength
)
{
   register struct SeSoLaStruct  *sslptr;
   register struct RequestStruct  *rqptr;

   int  status,
        Count;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "sesola_write() %d %d\n", DataPtr, DataLength);

   sslptr = (struct SeSoLaStruct*)bioptr->ptr;
   rqptr = sslptr->rqptr;

   BIO_clear_retry_flags (bioptr);

   if (Debug)
      fprintf (stdout, "sslptr->WriteCount: %d\n", sslptr->WriteCount);

   if (sslptr->WriteCount)
   {
      /* will only be non-zero due to a previous non-blocking write */
      if ((Count = sslptr->WriteCount) == -2) Count = 0;
      sslptr->WriteCount = 0;
#ifdef DBUG
      if (Debug) SeSoLaDebugStream (DataPtr, Count);
#endif
      /* writes always "succeed" :^( */
      return (DataLength);
   }

   if (sslptr->WriteType == SESOLA_WRITE_DATA_BLOCK)
   {
      /* blocking I/O */
      sslptr->IoInProgress = false;
      status = NetWriteRaw (sslptr->rqptr, NULL, DataPtr, DataLength);
      /* if success return the number of characters written */
      if (VMSok (status)) return (rqptr->NetWriteIOsb.Count);
      /* whoa, error */
      rqptr->NetWriteIOsb.Status = status;
      return (-2);
   }
   else
   {
      /* non-blocking I/O */
      sslptr->IoInProgress = true;
      /* indicate to the calling routine this write should be retried */
      BIO_set_retry_write (bioptr);
      NetWriteRaw (sslptr->rqptr, &sesola_write_ast, DataPtr, DataLength);
      /* return indicating non-blocking I/O */
      return (-1);
   }
}

/*****************************************************************************/
/*
*/

int sesola_write_ast (struct RequestStruct *rqptr)

{
   register struct SeSoLaStruct  *sslptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "sesola_write_ast() %%X%08.08X %d\n",
               rqptr->NetWriteIOsb.Status, rqptr->NetWriteIOsb.Count);

   sslptr = (struct SeSoLaStruct*)rqptr->SeSoLaPtr;

   sslptr->IoInProgress = false;

   if (VMSok (rqptr->NetWriteIOsb.Status))
      sslptr->WriteCount = rqptr->NetWriteIOsb.Count;
   else
      sslptr->WriteCount = -2;

   if (sslptr->WriteType == SESOLA_WRITE_DATA_NOBLOCK)
      SeSoLaWrite (rqptr, sslptr->AstFunctionPtr,
                   sslptr->WriteDataPtr, sslptr->WriteDataLength);
   else
   if (sslptr->WriteType == SESOLA_ACCEPT)
      SeSoLaAccept (rqptr);
   else
   if (sslptr->WriteType == SESOLA_SHUTDOWN)
      SeSoLaEnd (rqptr);
   else
      ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
}

/*****************************************************************************/
/*
Implements a required function of a SSLeay BIO_METHOD.
*/

long sesola_ctrl
(
BIO *bioptr,
int Command,
long Number,
char *Pointer
)
{
   int  value;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "sesola_ctrl() %d %d %d\n", Command, Number, Pointer);

   value = 1;
   switch (Command)
   {
      case BIO_CTRL_RESET:
         if (Debug) fprintf (stdout, "BIO_CTRL_RESET\n");
         return (1);

      case BIO_CTRL_EOF:
         if (Debug) fprintf (stdout, "BIO_CTRL_EOF\n");
         return (1);

      case BIO_CTRL_SET:
         if (Debug) fprintf (stdout, "BIO_CTRL_SET\n");
         bioptr->num = Number;
         return (1);

      case BIO_CTRL_SET_CLOSE:
         if (Debug) fprintf (stdout, "BIO_CTRL_SET_CLOSE\n");
         return (1);

      case BIO_CTRL_FLUSH:
         if (Debug) fprintf (stdout, "BIO_CTRL_FLUSH\n");
         return (1);

      case BIO_CTRL_DUP:
         if (Debug) fprintf (stdout, "BIO_CTRL_DUP\n");
         return (1);

      case BIO_CTRL_GET_CLOSE:
         if (Debug) fprintf (stdout, "BIO_CTRL_GET_CLOSE\n");
         return (0);

      case BIO_CTRL_INFO:
         if (Debug) fprintf (stdout, "BIO_CTRL_INFO\n");
         return (0);

      case BIO_CTRL_GET:
         if (Debug) fprintf (stdout, "BIO_CTRL_GET\n");
         return (0);

      case BIO_CTRL_PENDING:
         if (Debug) fprintf (stdout, "BIO_CTRL_PENDING\n");
         return (0);

      case BIO_CTRL_POP:
         if (Debug) fprintf (stdout, "BIO_CTRL_POP\n");
         return (0);

      case BIO_CTRL_PUSH:
         if (Debug) fprintf (stdout, "BIO_CTRL_PUSH\n");
         return (0);

      case BIO_CTRL_WPENDING:
         if (Debug) fprintf (stdout, "BIO_CTRL_WPENDING\n");
         return (0);

      default:
         if (Debug) fprintf (stdout, "default:\n");
         return (0);
   }
}

/*****************************************************************************/
/*
Implements a required function of a SSLeay BIO_METHOD.
*/

int sesola_gets
(
BIO *bioptr,
char *StringPtr,
int StringSize
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "sesola_gets()\n");
   return(0);
}

/*****************************************************************************/
/*
Implements a required function of a SSLeay BIO_METHOD.
*/

int sesola_puts
(
BIO *bioptr,
char *StringPtr
)
{
    int  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "sesola_puts() |%s|\n", StringPtr);

    if (StringPtr == NULL) return (0);
    Length = strlen (StringPtr);
    if (Length == 0) return (0);
    return (sesola_write (bioptr, StringPtr, Length));
}

/*****************************************************************************/
/*
For compilations without SSL these functions provide LINKage stubs for the
rest of the HTTPd modules, allowing for just recompiling the SESOLA module to
integrate the SSL functionality.
*/

/*********************/
#else  /* not SESOLA */
/*********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

/* global storage */
boolean  ProtocolHttpsAvailable,
         ProtocolHttpsConfigured;
/* i.e. disabled */
int  SeSoLaProtocolVersion = -1;
char  HttpdSesola [] = "";

/* external storage */
extern char  ErrorSanityCheck[];
extern char  Utility[];

SeSoLaInit ()
{
   ProtocolHttpsAvailable = false;
   if (SeSoLaProtocolVersion <= 0) return;
   fprintf (stdout, "%%%s-E-SSL, non-SSL version\n", Utility);
   exit (STS$K_ERROR | STS$M_INHIB_MSG);
}

SeSoLaBegin (struct RequestStruct *rqptr)
{
   ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
}

SeSoLaAccept (struct RequestStruct *rqptr)
{
   ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
}

SeSoLaEnd (struct RequestStruct *rqptr)
{
   ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
}

SeSoLaFree (struct RequestStruct *rqptr)
{
   ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
}

SeSoLaShutdown (struct RequestStruct *rqptr)
{
   ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
}

int SeSoLaWrite
(
struct RequestStruct *rqptr,
void *AstFunctionPtr,
char *DataPtr,
int DataLength
)
{
   ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
}

SeSoLaRead
(
struct RequestStruct *rqptr,
void *AstFunctionPtr,
char *DataPtr,
int DataSize
)
{
   ErrorExitVmsStatus (0, ErrorSanityCheck, FI_LI);
}

SeSoLaReport
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   ErrorGeneral (rqptr, "This is a non-SSL version.", FI_LI);
   SysDclAst (NextTaskFunction, rqptr);
}

/************************/
#endif  /* ifdef SESOLA */
/************************/

/*****************************************************************************/

