/*****************************************************************************/
/*
                                   SSI.c

This module implements a full multi-threaded, AST-driven, asynchronous HTML
Server Side Includes pre-processor.  The AST-driven nature makes the code a
little more difficult to follow, but creates a powerful, event-driven,
multi-threaded server.  All of the necessary functions implementing this module
are designed to be non-blocking. 

The SSI functionality in the WASD server include the basics of NCSA SSI,
extensions similar to Apache's XSSI (eXtended SSI), as well as WASD and VMS
idiosyncratic elements, to provide a reasonably capable server-side document
processor including:

    o  inclusion of server information (e.g. date, name, software)
    o  inclusion of document information (e.g. path, last modification)
    o  inclusion of custom layout directory listings
    o  innocuous DCL command execution (e.g. "SHOW TIME")
    o  privileged DCL command execution (e.g. DCL procedures)
    o  expression evaluation and user variable assignment
    o  flow control in document output (e.g. if-then-else)
    o  document access counter

The flow control statements and associated evalutions were designed for
simplicity, both for the author (who didn't want to spend a huge amount of
time building a complex expression parser, etc.) and also for the server, with
a simple scheme presumably resulting in a lower execution overhead.  It is
assumed most documents will have simple internal flows so this shouldn't be an
issue.  If very complex decision making is required it is probably better
exported to a legitimate script.

To ease the clutter often present in XSSI documents (lots of SSI statements
all over the place) the <!--#ssi ... --> statement allows multiple SSI
statements to be grouped where the elipsis is shown.  See the example below.
Of course only SSI statements may be included in this structure!  No plain
HTML! The leading '#' of new SSI statement ends any previous statement. The
"-->" of the "<!--#ssi" ends the last statement. The '\' character may be used
inside "<!--#ssi" statements to escape forbidden characters (.e.g '#').

An SSI statement CAN BE SPLIT OVER MULTIPLE LINES, up to a maximum dictated by
the size of the statement buffer (currently 1024 characters).

Other SSI documents may be #included with full SSI processing up to a nested
limit imposed by SSI_MAX_DEPTH.

#included files are automatically encapsulated in <PRE></PRE> tags and HTML-
escaped if not "text/html" content-type (i.e. if they are "text/plain" 
content-type).  They can be forced to be directly included by including a 
'par="text/html"' parameter.  Conversly, "text/html" files can be forced to be 
included as plain-text using the 'par="text/plain"' parameter. 

Format ('fmt=""') specifications for time values follow those allowed by 
strftime().  If none is specified it defaults to a fairly standard looking 
VMS-style time.

*PRIVILEGED* statements are only allowed in documents owned by SYSTEM ([1,4])
and that are NOT world writable!

This module uses the NetWriteBuffered() function to buffer any output it can
into larger chunks before sending it to the client.


STATEMENTS
----------

<!--## -->                              comment (not in resulting document!)

<!--#accesses [ordinal] [since=""] [timefmt=""] -->  accesses of document

<!--#config errmsg="" -->               SSI error message header
<!--#config timefmt="" -->              set default time format
<!--#config sizefmt="" -->              set file size output format

<!--#dir file="" [par=""] -->           directory (index of) file spec
<!--#dir virtual="" [par=""] -->        directory (index of) virtual spec
<!--#index file="" [par=""] -->         synonym for the above
<!--#index virtual="" [par=""] -->      synonym for the above

<!--#echo created[=fmt] -->             current document creation date/time
<!--#echo date_local[=fmt] -->          current local date/time
<!--#echo date_gmt[=fmt] -->            current GMT date/time
<!--#echo document_name -->             current document VMS file path
<!--#echo document_uri - ->             current document URL path
<!--#echo file_name -->                 current document VMS file path
<!--#echo last_modified[=fmt] -->       current document modified date

<!--#echo http_accept -->               browser accepted content types
<!--#echo http_accept_charset -->       browser accepted character sets
<!--#echo http_accept_language -->      browser accepted languages
<!--#echo http_host -->                 destination server host/port
<!--#echo http_forwarded -->            forwarded by proxy/gateway host(s)
<!--#echo http_referer -->              refering document
<!--#echo http_user_agent -->           browser identification string
<!--#echo path_info -->                 request path
<!--#echo path_translated -->           current document VMS file name
<!--#echo query_string -->              request query string
<!--#echo query_string_unescaped -->    request query string
<!--#echo remote_addr -->               browser host address
<!--#echo remote_host -->               browser host name
<!--#echo remote_user -->               authenticated user
<!--#echo request_scheme -->            "http:" or "https:"
<!--#echo server_gmt -->                server GMT offset (e.g. "+09:30")
<!--#echo server_name -->               server host name
<!--#echo server_port -->               server host port
<!--#echo server_protocol -->           HTTP protocol version
<!--#echo server_software -->           server identification string
<!--#echo var="" -->                    print server or user variable value

NOTE: the "#dcl" and "#exec" statements are identical

<!--#exec dir="" [par=""] -->           DIRECTORY file-spec [qualifiers]
<!--#exec vdir="" [par=""] -->          DIRECTORY virtual-file-spec [qualifiers]
<!--#exec show="" -->                   SHOW command
<!--#exec say="" -->                    WRITE SYS$OUTPUT 'anything'

<!--#exec exec="" -->                   *PRIVILEGED* execute any DCL command
<!--#exec file="" [par=""] -->          *PRIVILEGED* execute command procedure
<!--#exec run="" [par=""] -->           *PRIVILEGED* execute image
<!--#exec virtual="" [par=""] -->       *PRIVILEGED* execute command procedure
<!--#exec vrun="" [par=""] -->          *PRIVILEGED* execute virtual image

<!--#fcreated file="" [fmt=""] -->      specified file creation date/time
<!--#fcreated virtual="" [fmt=""] -->   specified URL document creation date
<!--#flastmod file="" [fmt=""] -->      specified file last modified date/time
<!--#flastmod virtual="" [fmt=""] -->   specified URL document last modified 
<!--#fsize file="" -->                  specified file size (bytes, Kb, Mb)
<!--#fsize virtual="" -->               specified URL document size

<!--#include file="" [type=""] -->      include the file's contents
<!--#include virtual="" [type=""] -->   include the URL document contents

<!--#if value="" [...] -->              flow control
<!--#orif value="" [...] -->            flow control
<!--#elif value="" [...] -->            flow control
<!--#else -->                           flow control
<!--#endif -->                          flow control

<!--#printenv -->                       prints all assigned variables
<!--#set var="" value="" -->            sets the variable name to value
<!--#ssi [statement|...] -->            multiple SSI statements
<!--#stop -->                           stop processing the document here!
<!--#trace [ON|OFF] -->                 changes trace mode state


VARIABLES
---------
There are server and user variables. User variables may be assigned using the
"#set" statement. The values of both may be substituted into any tag value
associated with any statement. User variable names may comprise alphanumeric
and underscore characters, and are not case sensitive. Variable substitution
is indicated by delimitting the variable name with '{' and '}'. The variable
name may be followed by one or two comma separated numbers which serve as an
extraction start index and count. A single number results in the variable text
from zero for the specified. Two numbers specify a start index and count.

    <!--#set var=EXAMPLE1 value="This is an example!" -->
    <!--#set var=EXAMPLE2 value={EXAMPLE1,0,9} -->
    <!--#set var=EXAMPLE3 value="{EXAMPLE2}other example ..." -->
    <!--#set var=EXAMPLE4 value="{EXAMPLE1,9}other example!" -->
    <!--#echo "<P>" var={EXAMPLE3} "<BR>" var={EXAMPLE4} -->

The output from the "#echo" would be

    <P>This is another example ...
    <BR>This is another example!


EVALUATIONS and FLOW CONTROL
----------------------------
Flow control statements work in the same fashion to other implementations. The
possible WASD idiosyncrasy is "#orif" which, in the absence of a true
expression parser, allows multiple conditions to execute a single block of
statements.

    <!--#if value="" [...=""] -->
    <!--#orif value="" [...=""] -->
    <!--#elif value="" [...=""] -->
    <!--#else -->
    <!--#endif -->

The evalution of a flow control decision in an "#if", "#orif" or "#elif" is
based on one or more tests against one or more variables.

    <!--#if value="" -->                  if string not empty, or not numeric 0
    <!--#if value="" eqs="" -->           string equal to
    <!--#if value="" srch="" -->          string search (* and % wildcards)
    <!--#if value="" lt="" -->            numeric less than
    <!--#if value="" eq="" -->            numeric equal to
    <!--#if value="" gt="" -->            numeric greater than

If a single string/variable is supplied in the test then if it is numeric and
non-zero then the test is true, if a string and non-empty it is true.  Other
states are false.  The "eqs=" test does a case-insensitive string compare of
the supplied string and the "value=" string. The "srch=" test searches "value="
with the supplied string, which can contain the wildcards "*" matching
multiple characters and "%" matching any one character.  The "lt=", "eq=" and
"gt=" convert the parameters to integers and do an arithetic compare.
Multiple comparaisons, even with multiple variables, may occur in the one
decision statement, these will act as a logical AND.  Logical ORs may be
created using "#orif" statements.


SIMPLE EXAMPLE
--------------

    <!--##trace par=ON -->
    <HTML>
    <!--#ssi
    #set var=HOUR value={DATE_LOCAL,12,2}
    #if value={HOUR} lt=12
      #set var=GREETING value="Good morning"
    #elif value={HOUR} lt=19
      #set var=GREETING value="Good afternoon"
    #else
      #set var=GREETING value="Good evening"
    #endif
    -->
    <HEAD>
    <TITLE><!--#echo value={GREETING} -->
    <!--#echo value="{REMOTE_HOST}!" --></TITLE>
    </HEAD>
    <BODY>
    <H1>Simple XSSI Demonstration</H1>
    <!--#echo value={GREETING} --> <!--#echo value={REMOTE_HOST} -->,
    the time here is <!--#echo value={DATE_LOCAL,12,5} -->.
    </BODY>
    </HTML>


VERSION HISTORY
---------------
19-JUL-98  MGD  bugfix; GetTagValue() must return number of characters!
                bugfix; MapUrl_Map() pass 'rqptr' for conditionals to work
13-APR-98  MGD  add <!--#ssi ... --> block statement,
                variable assignment and flow-control eXtensions
23-JAN-98  MGD  <!--#echo server_gmt --> and ordinal accesses
25-OCT-97  MGD  <!--#echo http_accept_charset, http_host, http_forward -->
17-AUG-97  MGD  message database,
                SYSUAF-authenticated users security-profile,
                additional <!--#echo ... --> (just for the hang of it :^)
27-FEB-97  MGD  delete on close for "temporary" files
01-FEB-97  MGD  HTTPd version 4 (also changed module name to SSI.c);
                extensive rework of some functions;
                statements now allowed to span multiple lines
04-JUN-96  MGD  bugfix; SsiFileDetails() error reporting
01-DEC-95  MGD  new for HTTPd version 3
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <time.h>

/* VMS related header files */
#include <atrdef.h>
#include <descrip.h>
#include <fibdef.h>
#include <iodef.h>
#include <prvdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application-related header files */
#include "wasd.h"
#include "cgi.h"
#include "dcl.h"
#include "dir.h"
#include "error.h"
#include "file.h"
#include "httpd.h"
#include "mapurl.h"
#include "msg.h"
#include "net.h"
#include "ssi.h"
#include "support.h"
#include "vm.h"

#define SSI_DEFAULT_TIME_FORMAT "%Od-%b-%Y %T"  /* a little VMSish! */
#define SSI_DEFAULT_SIZE_FORMAT "abbrev"

#define FILE_FCREATED  1
#define FILE_FLASTMOD  2
#define FILE_FSIZE     3

#define SSI_STATE_DEFAULT  1
#define SSI_STATE_IF       2
#define SSI_STATE_ELIF     3
#define SSI_STATE_ELSE     4

#define SSI_MAX_DEPTH  5

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern char  ConfigContentTypeSsi[];
extern int  DclCgiVariablePrefixLength;
extern char  HttpProtocol[];
extern char  SoftwareID[];
extern char  TimeGmtString[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/*****************************************************************************/
/*
Open the specified file.  If there is a problem opening it then just return 
the status, do not report an error or anything.  This is used to determine 
whether the file exists, as when attempting to open one of multiple, possible 
home pages.

As pre-processed HTML files are by definition dynamic, no check of any
"If-Modified-Since:" request field date/time is made, and no "Last-Modified:" 
field is included in any response header. 

When successfully opened and connected generate an HTTP header, if required.
Once open and connected the pre-processing becomes AST-driven.
*/

SsiBegin
(
struct RequestStruct *rqptr,
void *NextTaskFunction,
void *FileOpenErrorFunction,
char *FileName
)
{
   static $DESCRIPTOR (StringDsc, "");

   register char  *cptr, *sptr, *zptr;
   register struct SsiTaskStruct  *tkptr;

   int  status;
   char  ExpandedFileName [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiBegin() |%s|\n", FileName);

   if (rqptr->ErrorMessagePtr != NULL)
   {
      /* previous error, cause threaded processing to unravel */
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   if (!Config.SsiEnabled)
   {
      Accounting.RequestForbiddenCount++;
      rqptr->ResponseStatusCode = 501;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_GENERAL_DISABLED), FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /* SSI documents can call other SSI documents ... infinite recursion! */
   if (LIST_COUNT (&rqptr->SsiTaskList) > SSI_MAX_DEPTH) 
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_SSI_RECURSION), FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /* set up the task structure (possibly multiple concurrent) */
   if (rqptr->SsiTaskPtr != NULL &&
       LIST_HAS_NEXT (rqptr->SsiTaskPtr))
   {
      rqptr->SsiTaskPtr = tkptr = LIST_NEXT (rqptr->SsiTaskPtr);
      memset (LIST_DATA(tkptr), 0,
              sizeof(struct SsiTaskStruct) - sizeof(struct ListEntryStruct));
   }
   else
   {
      rqptr->SsiTaskPtr = tkptr =
         VmGetHeap (rqptr, sizeof(struct SsiTaskStruct));
      ListAddTail (&rqptr->SsiTaskList, tkptr);
   }
   tkptr->NextTaskFunction = NextTaskFunction;

   cptr = FileName;
   zptr = (sptr = tkptr->FileName) + sizeof(tkptr->FileName);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      SsiEnd (rqptr);
      return;
   }
   *sptr = '\0';
   tkptr->FileNameLength = sptr - tkptr->FileName;

   /*****************************************/
   /* check VMS-authenticated user's access */
   /*****************************************/

   if (rqptr->AuthVmsUserProfileLength)
   {
      status = AuthCheckVmsUserAccess (rqptr, tkptr->FileName,
                                       tkptr->FileNameLength);
      if (status == RMS$_PRV)
         tkptr->AuthVmsUserHasAccess = false;
      else
      if (VMSok (status))
         tkptr->AuthVmsUserHasAccess = true;
      else
      {
         /* error reported by access check */
         SsiEnd (rqptr);
         return;
      }
   }
   else
      tkptr->AuthVmsUserHasAccess = false;

   /*************/
   /* go for it */
   /*************/

   tkptr->FileFab = cc$rms_fab;
   tkptr->FileFab.fab$b_fac = FAB$M_GET;
   tkptr->FileFab.fab$l_fna = tkptr->FileName;  
   tkptr->FileFab.fab$b_fns = tkptr->FileNameLength;
   tkptr->FileFab.fab$b_shr = FAB$M_SHRGET;

   /* initialize the NAM block */
   tkptr->FileNam = cc$rms_nam;
   tkptr->FileFab.fab$l_nam = &tkptr->FileNam;
   tkptr->FileNam.nam$l_esa = ExpandedFileName;
   tkptr->FileNam.nam$b_ess = sizeof(ExpandedFileName)-1;

   /* initialize the date extended attribute block */
   tkptr->FileXabDat = cc$rms_xabdat;
   tkptr->FileFab.fab$l_xab = &tkptr->FileXabDat;

   /* initialize and link in the protection extended attribute block */
   tkptr->FileXabPro = cc$rms_xabpro;
   tkptr->FileXabDat.xab$l_nxt = &tkptr->FileXabPro;

   /* "temporary" file, automatic delete on closing it */
   if (rqptr->DeleteOnClose) tkptr->FileFab.fab$l_fop |= FAB$M_DLT;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   status = sys$open (&tkptr->FileFab, 0, 0);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);

      /* if its a search list treat directory not found as if file not found */
      if ((tkptr->FileNam.nam$l_fnb & NAM$M_SEARCH_LIST) && status == RMS$_DNF)
         status = RMS$_FNF;

      if (FileOpenErrorFunction != NULL)
      {
         /* do not report the error, the alternate function will handle it */
         tkptr->NextTaskFunction = FileOpenErrorFunction;
      }
      else
      {
         if (!rqptr->AccountingDone)
            rqptr->AccountingDone = ++Accounting.DoSsiCount;
         rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
         rqptr->ErrorHiddenTextPtr = tkptr->FileName;
         ErrorVmsStatus (rqptr, status, FI_LI);
      }

      SsiEnd (rqptr);
      return;
   }

   if (!rqptr->AccountingDone)
      rqptr->AccountingDone = ++Accounting.DoSsiCount;

   /***************/
   /* file exists */
   /***************/

   tkptr->FileRab = cc$rms_rab;
   tkptr->FileRab.rab$l_fab = &tkptr->FileFab;
   /* 2 buffers and read ahead performance option */
   tkptr->FileRab.rab$b_mbf = 2;
   tkptr->FileRab.rab$l_rop = RAB$M_RAH;
   tkptr->FileRab.rab$l_ubf = tkptr->ReadBuffer;
   /* allow two bytes for carriage control and terminating null */
   tkptr->FileRab.rab$w_usz = sizeof(tkptr->ReadBuffer)-2;

   if (VMSnok (status = sys$connect (&tkptr->FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      SsiEnd (rqptr);
      return;
   }

   /********************/
   /* begin processing */
   /********************/

   if (rqptr->ResponseHeaderPtr == NULL)
   {
      rqptr->ResponsePreExpired = PRE_EXPIRE_SSI;
      rqptr->ResponseStatusCode = 200;
      if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
      {
         SsiEnd (rqptr);
         return;
      }

      if (rqptr->HttpMethod == HTTP_METHOD_HEAD)
      {
         SsiEnd (rqptr);
         return;
      }
   }

   if (VMSnok (CgiGenerateVariables (rqptr, CGI_VARIABLE_STREAM)))
   {
      SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_ERROR), FI_LI);
      return;
   }
   rqptr->SsiTaskPtr->CgiBufferPtr = rqptr->CgiBufferPtr;
   rqptr->CgiBufferPtr = NULL;

   /* no flow control structure always outputs */
   tkptr->FlowControlState[0] = SSI_STATE_DEFAULT;
   tkptr->FlowControlHasExecuted[0] = false;
   tkptr->FlowControlIsExecuting[0] = true;

   /* buffer the escape-HTML flag */
   tkptr->NetWriteBufferedEscapeHtml = rqptr->NetWriteBufferedEscapeHtml;

   tkptr->ErrMsgPtr = NULL;
   tkptr->SizeFmtPtr = SSI_DEFAULT_SIZE_FORMAT;
   tkptr->TimeFmtPtr = SSI_DEFAULT_TIME_FORMAT;

   tkptr->LineNumber = 0;

   /* set then RAB user context storage to the client thread pointer */
   tkptr->FileRab.rab$l_ctx = rqptr;

   /* network writes are checked for success, fudge the first one! */
   rqptr->NetWriteIOsb.Status = SS$_NORMAL;

   /* begin processing */
   SsiNextRecord (rqptr);
}

/*****************************************************************************/
/*
*/ 

SsiEnd (struct RequestStruct *rqptr)

{
   register struct SsiTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiEnd()\n");

   tkptr = rqptr->SsiTaskPtr;

   if (tkptr->FileFab.fab$w_ifi)
   {
      if (rqptr->DeleteOnClose)
      {
         /* use SYSPRV to ensure deletion-on-close of file */
         EnableSysPrv ();
         sys$close (&tkptr->FileFab, 0, 0);
         DisableSysPrv ();
      }
      else
         sys$close (&tkptr->FileFab, 0, 0);
   }

   /* restore the escape-HTML flag */
   rqptr->NetWriteBufferedEscapeHtml = tkptr->NetWriteBufferedEscapeHtml;

   SysDclAst (tkptr->NextTaskFunction, rqptr);

   /* restore previous SSI task (if any) */
   rqptr->SsiTaskPtr = LIST_PREV (tkptr);
}

/*****************************************************************************/
/*
Queue an asynchronous read of the next record.
*/ 
 
int SsiNextRecord (struct RequestStruct *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiNextRecord()\n");

   if (VMSnok (rqptr->NetWriteIOsb.Status))
   {
      /* network write has failed (as AST), bail out now */
      if (Debug)
         fprintf (stdout, "NetWriteIOsb.Status %%X%08.08X\n",
                  rqptr->NetWriteIOsb.Status);
      SsiEnd (rqptr);
      return;
   }

   /* queue the next record read */
   sys$get (&rqptr->SsiTaskPtr->FileRab, &SsiNextRecordAST, &SsiNextRecordAST);
}

/*****************************************************************************/
/*
The asynchronous read of the next record has completed.
*/ 

SsiNextRecordAST (struct RAB *RabPtr)

{
   register char  *cptr, *sptr;
   register struct SsiTaskStruct  *tkptr;

   int  status;
   struct RequestStruct  *rqptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
      "SsiNextRecordAST() sts: %%X%08.08X stv: %%X%08.08X\n",
      RabPtr->rab$l_sts, RabPtr->rab$l_stv);

   /* get the pointer to the thread from the RAB user context storage */
   rqptr = RabPtr->rab$l_ctx;

   tkptr = rqptr->SsiTaskPtr;

   if (VMSnok (tkptr->FileRab.rab$l_sts))
   {
      if (tkptr->FileRab.rab$l_sts == RMS$_EOF)
      {
         /***************/
         /* end of file */
         /***************/

         /* hmmm, got to end of file and there's still nested control */
         if (tkptr->FlowControlIndex)
         {
            tkptr->Statement[0] = '\0';
            tkptr->StatementLineNumber = tkptr->LineNumber;
            SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_FLOW_CONTROL), FI_LI);
         }

         SsiEnd (rqptr);
         return;
      }

      /**********************/
      /* error reading file */
      /**********************/

      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, tkptr->FileRab.rab$l_sts, FI_LI);
      SsiEnd (rqptr);
      return;
   }

   /*****************************/
   /* process the record (line) */
   /*****************************/

   /* first check if any error occured during any previous processing */
   if (rqptr->ErrorMessagePtr != NULL)
   {
      SsiEnd (rqptr);
      return;
   }

   /* terminate the line */
   tkptr->FileRab.rab$l_ubf[tkptr->FileRab.rab$w_rsz] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", tkptr->FileRab.rab$l_ubf);

   tkptr->LineNumber++;

   if (tkptr->TraceState) SsiTraceLine (rqptr);

   tkptr->PageHasBeenOutput = tkptr->SuppressLine =
      tkptr->TraceHasBeenOutput = false;

   /* this pointer keeps track of the current parse position in the line */
   tkptr->LineParsePtr = tkptr->FileRab.rab$l_ubf;

   /* begin parsing the line looking for commented SSI statements */
   SsiParseLine (rqptr);
}

/*****************************************************************************/
/*
'tkptr->LineParsePtr' points to the currently parsed-up-to position on 
the line.  Continue parsing that line looking for preprocessor statements.
*/ 

SsiParseLine (struct RequestStruct *rqptr)

{
   register char  ch;
   register char  *cptr, *sptr;
   register struct SsiTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiParseLine()\n|%s|\n",
               rqptr->SsiTaskPtr->LineParsePtr);

   tkptr = rqptr->SsiTaskPtr;

   /* first check if any error occured during any previous processing */
   if (rqptr->ErrorMessagePtr != NULL ||
       tkptr->SsiProblemEncountered)
   {
      SsiEnd (rqptr);
      return;
   }

   if (tkptr->StatementPtr == NULL)
   {
      /* ensure the escape-HTML flag is "off" from any previous processing */
      rqptr->NetWriteBufferedEscapeHtml = tkptr->TraceState;

      /* find the start of any statement */
      if (tkptr->InsideSsiStatement)
      {
         /**********************************/
         /* inside an "<!--#ssi" statement */
         /**********************************/

         for (cptr = sptr = tkptr->LineParsePtr; *sptr; sptr++)
         {
            while (*sptr && *sptr != '#' && *sptr != '-' && *sptr != '\\')
               sptr++;
            if (!*sptr) break;
            /* if "escape the next character" */
            if (*sptr == '\\') continue;
            /* if start of another statement then end of previous */
            if (*sptr == '#') break;
            /* must be a '-' */
            if (!memcmp (sptr, "-->", 3))
            {
               /* end of "<!--#ssi" tag, i.e. "-->" */
               if (Debug) fprintf (stdout, "<!--#SSI -->\n");
               tkptr->InsideSsiStatement = false;
               tkptr->SuppressLine = true;
               tkptr->LineParsePtr = sptr + 3;
               /* continue parsing the line */
               SysDclAst (&SsiParseLine, rqptr);
               return;
            }
            /* just another '-' character */
         }
      }
      else
      {
         /**************************************/
         /* not inside an "<!--#ssi" statement */
         /**************************************/

         for (cptr = sptr = tkptr->LineParsePtr; *sptr; sptr++)
         {
            while (*sptr && *sptr != '#') sptr++;
            if (!*sptr) break;
            if (sptr-cptr >= 4 && !memcmp (sptr-4, "<!--", 4)) break;
         }
      }

      if (*sptr)
      {
         /*****************/
         /* SSI statement */
         /*****************/
   
         if (Debug) fprintf (stdout, "sptr |%s|\n", sptr);

         tkptr->LineParsePtr = sptr;
         tkptr->StatementLineNumber = tkptr->LineNumber;

         ch = tolower(sptr[1]);
         if (ch == 'i' && strsame (sptr, "#if", 3) ||
             ch == 'e' && strsame (sptr, "#elif", 5) ||
             ch == 'e' && strsame (sptr, "#else", 5) ||
             ch == 'e' && strsame (sptr, "#endif", 6) ||
             ch == 'o' && strsame (sptr, "#orif", 5) ||
             ch == 's' && strsame (sptr, "#set", 4) ||
             ch == 't' && strsame (sptr, "#trace", 6) ||
             *((unsigned short*)sptr) == '##')
         {
            /* flow control line, etc., suppress resultant "blank" lines */
            tkptr->SuppressLine = true;
         }
         else
         if (ch == 's' && strsame (sptr, "#ssi", 4))
         {
            /* found the start of an "<!--#ssi" statement */
            if (Debug) fprintf (stdout, "<!--#SSI\n");
            if (tkptr->InsideSsiStatement)
            {
                /* already inside an "<!--#ssi" statement! */
                SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_STATEMENT), FI_LI);
                SsiEnd (rqptr);
                return;
            }

            tkptr->InsideSsiStatement = tkptr->SuppressLine = true;
            tkptr->StatementPtr = NULL;
            tkptr->LineParsePtr += 4;

            /* OK, continue parsing now we're inside the "<!---#ssi" */
            SysDclAst (&SsiParseLine, rqptr);
            return;
         }
         else
         {
            if (sptr-4-cptr > 0)
            {
               /* buffer the part line, then AST process the statement */
               SsiWriteBuffered (rqptr, &SsiStatement, cptr, sptr-4-cptr);
               return;
            }
            /* continue on to the process the statement */
         }

         SsiStatement (rqptr);
         return;
      }
      else
      {
         /*********************************************/
         /* no statement, send line (or part thereof) */
         /*********************************************/

         if (tkptr->SuppressLine)
         {
            /* flow control line, suppress "blank" lines */
            SysDclAst (&SsiNextRecord, rqptr);
            return;
         }

         /* end-of-line, ensure it has correct carriage control */
         if (sptr > tkptr->FileRab.rab$l_ubf)
         {
            if (sptr[-1] != '\n')
            {
               *sptr++ = '\n';
               *sptr = '\0';
            }
         }
         else
         {
            /* blank (empty) line */
            *sptr++ = '\n';
            *sptr = '\0';
         }

         SsiWriteBuffered (rqptr, &SsiNextRecord, cptr, sptr-cptr);
         return;
      }
   }
   else
   {
      /* continue to parse statement */
      SsiStatement (rqptr);
   }
}

/*****************************************************************************/
/*
'tkptr->LineParsePtr' points to the start of a pre-processor statement.  
Parse that statement and execute it, or provide an error message.
*/ 

SsiStatement (struct RequestStruct *rqptr)

{
   register char  ch;
   register char  *cptr, *sptr, *zptr;
   register struct SsiTaskStruct  *tkptr;

   boolean  ok;
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiStatement()\n|%s|\n",
               rqptr->SsiTaskPtr->LineParsePtr);

   tkptr = rqptr->SsiTaskPtr;

   if (tkptr->StatementPtr == NULL)
   {
      tkptr->StatementPtr = tkptr->Statement;
      tkptr->StatementLength = 0;
   }

   /* point at the statement buffer and set a pointer to the end of buffer */
   sptr = tkptr->StatementPtr;
   zptr = tkptr->Statement + sizeof(tkptr->Statement);

   /* continued lines introduce an implied space */
   if (sptr > tkptr->Statement && sptr < zptr) *sptr++ = ' ';

   cptr = tkptr->LineParsePtr;

   if (tkptr->InsideSsiStatement)
   {
      /**********************************/
      /* inside an "<!--#ssi" statement */
      /**********************************/

      /* get any leading '#' so as not to trigger the end '#' detection */
      if (sptr == tkptr->Statement && *cptr == '#' && sptr < zptr)
      {
         *sptr++ = *cptr++;
         /* cater for SSI comments */
         if (*cptr == '#' && sptr < zptr) *sptr++ = *cptr++;
      }

      while (*cptr && sptr < zptr)
      {
         /* get more of the statement while looking for the end */
         while (*cptr && *cptr != '#' && *cptr != '\\' && *cptr != '-' &&
                sptr < zptr)
            *sptr++ = *cptr++;
         if (!*cptr || sptr >= zptr) break;
         /* if escaped character */
         if (*cptr == '\\')
         {
            cptr++;
            if (*cptr) *sptr++ = *cptr++;
            continue;
         }
         /* if beginning of another statement means end of this one */
         if (*cptr == '#')
         {
            /* note the character position */
            tkptr->LineParsePtr = cptr;
            break;
         }
         /* if end of "<!--#ssi" tag */
         if (*cptr == '-' && !memcmp (cptr, "-->", 3))
         {
            if (Debug) fprintf (stdout, "<!--#SSI -->\n");
            rqptr->SsiTaskPtr->InsideSsiStatement = false;
            /* note the character position */
            tkptr->LineParsePtr = cptr + 3;
            break;
         }
         /* just another '-' character */
         *sptr++ = *cptr++;
      }
   }
   else
   {
      /**************************************/
      /* not inside an "<!--#ssi" statement */
      /**************************************/

      while (*cptr && sptr < zptr)
      {
         /* get more of the statement while looking for the end */
         while (*cptr && *cptr != '-' && sptr < zptr) *sptr++ = *cptr++;
         if (!*cptr || sptr >= zptr) break;
         /* if end-of-statement */
         if (!memcmp (cptr, "-->", 3))
         {
            /* note the character position */
            tkptr->LineParsePtr = cptr + 3;
            break;
         }
         /* just another '-' */
         if (*cptr && sptr < zptr) *sptr++ = *cptr++;
      }
   }

   if (sptr >= zptr)
   {
      SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_STATEMENT), FI_LI);
      SsiEnd (rqptr);
      return;
   }

   if (!*cptr)
   {
      /* got to the end of the line before seeing a statement terminator */
      *(tkptr->StatementPtr = sptr) = '\0';
      SsiNextRecord (rqptr);
      return;
   }

   /*********************/
   /* process statement */
   /*********************/

   /* trim trailing spaces */
   if (sptr > tkptr->Statement)
   {
      sptr--;
      while (sptr > tkptr->Statement && ISLWS(*sptr)) sptr--;
      if (sptr > tkptr->Statement && !ISLWS(*sptr)) sptr++;
   }

   *sptr = '\0';
   tkptr->StatementLength = sptr - tkptr->Statement;
   if (Debug)
      fprintf (stdout, "tkptr->Statement %d |%s|\n",
               tkptr->StatementLength, tkptr->Statement);

   /* indicate that there is no current statement being buffered */
   tkptr->StatementPtr = NULL;

   cptr = tkptr->Statement;
   ch = tolower(cptr[1]);
   if (Debug) fprintf (stdout, "ch: %d cptr |%s|\n", ch, cptr);

   if (tkptr->FlowControlIsExecuting[tkptr->FlowControlIndex])
   {
      /*************************************************/
      /* flow-control is currently allowing processing */
      /*************************************************/

      if (*((unsigned short*)cptr) == '##')
         SsiDoComment (rqptr);
      else
      if (ch == 'a' && strsame (cptr, "#accesses", 9))
         SsiDoAccesses (rqptr);
      else
      if (ch == 'c' && strsame (cptr, "#config", 7))
         SsiDoConfig (rqptr);
      else
      if (ch == 'd' && strsame (cptr, "#dcl", 4))
         SsiDoDcl (rqptr);
      else
      if (ch == 'd' && strsame (cptr, "#dir", 4))
         SsiDoDir (rqptr);
      else
      if (ch == 'e' && strsame (cptr, "#echo", 5))
         SsiDoEcho (rqptr);
      else
      if (ch == 'e' && strsame (cptr, "#elif", 5))
         SsiDoElif (rqptr);
      else
      if (ch == 'e' && strsame (cptr, "#else", 5))
         SsiDoElse (rqptr);
      else
      if (ch == 'e' && strsame (cptr, "#endif", 6))
         SsiDoEndif (rqptr);
      else
      if (ch == 'e' && strsame (cptr, "#exec", 5))
         SsiDoDcl (rqptr);
      else
      if (ch == 'f' && strsame (cptr, "#fcreated", 9))
         SsiDoFCreated (rqptr);
      else
      if (ch == 'f' && strsame (cptr, "#flastmod", 9))
         SsiDoFLastMod (rqptr);
      else
      if (ch == 'f' && strsame (cptr, "#fsize", 6))
         SsiDoFSize (rqptr);
      else
      if (ch == 'i' && strsame (cptr, "#if", 3))
         SsiDoIf (rqptr);
      else
      if (ch == 'i' && strsame (cptr, "#include", 8))
         SsiDoInclude (rqptr);
      else
      if (ch == 'i' && strsame (cptr, "#index", 6))
         SsiDoDir (rqptr);
      else
      if (ch == 'o' && strsame (cptr, "#orif", 5))
         SsiDoOrif (rqptr);
      else
      if (ch == 'p' && strsame (cptr, "#printenv", 9))
         SsiDoPrintEnv (rqptr);
      else
      if (ch == 's' && strsame (cptr, "#set", 4))
         SsiDoSet (rqptr);
      else
      if (ch == 's' && strsame (cptr, "#stop", 5))
         SsiDoStop (rqptr);
      else
      if (ch == 't' && strsame (cptr, "#trace", 6))
         SsiDoTrace (rqptr);
      else
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_STATEMENT_UNKNOWN), FI_LI);
         SsiEnd (rqptr);
         return;
      }
   }
   else
   {
      /********************************************/
      /* flow control is currently not processing */
      /********************************************/

      /* looking only for "#if", "#orif", "#elif", "#else" or "#endif" */
      if (ch == 'i' && strsame (cptr, "#if", 3))
         SsiDoIf (rqptr);
      else
      if (ch == 'o' && strsame (cptr, "#orif", 5))
         SsiDoOrif (rqptr);
      else
      if (ch == 'e' && strsame (cptr, "#elif", 5))
         SsiDoElif (rqptr);
      else
      if (ch == 'e' && strsame (cptr, "#else", 5))
         SsiDoElse (rqptr);
      else
      if (ch == 'e' && strsame (cptr, "#endif", 6))
         SsiDoEndif (rqptr);
      else
         SysDclAst (&SsiParseLine, rqptr);
   }
}

/*****************************************************************************/
/*
Number of times this document has been accessed.

This operation is fairly expensive in terms of I/O, and because it is atomic 
introduces a fair degree of granularity.  Its expense means the "#accesses" 
functionality should be used sparingly.  This functionality can be disabled. 
*/ 

SsiDoAccesses (struct RequestStruct *rqptr)

{
   static char  *Ordination [] =
      { "th", "st", "nd", "rd", "th", "th", "th", "th", "th", "th" };

   static $DESCRIPTOR (AccessesFaoDsc, "!AZ!AZ!AZ!AZ");
   static $DESCRIPTOR (StringDsc, "");

   register char  *dptr;

   boolean  DoOrdinal,
            SupplySinceTime;
   int  status,
        TensUnits;
   unsigned short  Length;
   unsigned long  AccessCount;
   unsigned long  SinceBinTime [2];
   char  *OrdinalPtr;
   char  CountString [32],
         SinceText [256],
         String [1024],
         TimeFormat [256],
         TimeString [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiDoAccesses() |%s|\n",
               rqptr->SsiTaskPtr->Statement);

   if (rqptr->SsiTaskPtr->TraceState) SsiTraceStatement (rqptr);

   if (!Config.SsiAccessesEnabled)
   {
      SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_ACCESS_DISABLED), FI_LI);
      SsiEnd (rqptr);
      return;
   }

   SinceText[0] = TimeFormat[0] = TimeString[0] = '\0';
   DoOrdinal = SupplySinceTime = false;

   dptr = rqptr->SsiTaskPtr->Statement;
   while (*dptr && !ISLWS(*dptr)) dptr++;

   while (*dptr)
   {
      if (Debug) fprintf (stdout, "dptr |%s|\n", dptr);

      if (rqptr->SsiTaskPtr->SsiProblemEncountered) break;

      while (*dptr && ISLWS(*dptr)) dptr++;
      if (!*dptr) break;

      if (strsame (dptr, "ORDINAL", 7))
      {
         dptr += 7;
         DoOrdinal = true;
      }
      else
      if (strsame (dptr, "SINCE=", 6))
      {
         dptr += SsiGetTagValue (rqptr, dptr, SinceText, sizeof(SinceText));
         SupplySinceTime = true;
      }
      else
      if (strsame (dptr, "TIMEFMT=", 8))
         dptr += SsiGetTagValue (rqptr, dptr, TimeFormat, sizeof(TimeFormat));
      else
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_TAG_UNKNOWN), FI_LI);
         SsiEnd (rqptr);
         return;
      }
   }

   if (rqptr->SsiTaskPtr->SsiProblemEncountered)
   {
      SsiEnd (rqptr);
      return;
   }

   if (VMSnok (SsiAccessCount (rqptr, &AccessCount, &SinceBinTime)))
   {
      SsiEnd (rqptr);
      return;
   }

   if (DoOrdinal)
   {
      TensUnits = AccessCount % 100;
      if (TensUnits >= 11 && TensUnits <= 19)
         OrdinalPtr = "th";
      else
         OrdinalPtr = Ordination[TensUnits%10];
   }
   else
      OrdinalPtr = "";

   if (SupplySinceTime)
   {
      if (!SsiTimeString (rqptr, &SinceBinTime, TimeFormat,
                          TimeString, sizeof(TimeString)))
      {
         SsiEnd (rqptr);
         return;
      }
   }

   CommaNumber (32, AccessCount, CountString);

   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String)-1;
   status = sys$fao (&AccessesFaoDsc, &Length, &StringDsc,
                     CountString, OrdinalPtr, SinceText, TimeString);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      SsiProblem (rqptr, SsiSysGetMsg(status), FI_LI);
      SsiEnd (rqptr);
      return;
   }
   String[Length] = '\0';

   SsiWriteBuffered (rqptr, &SsiParseLine, String, Length);
}

/*****************************************************************************/
/*
Comment hidden from the final document because it's inside an SSI statement!
*/

SsiDoComment (struct RequestStruct *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiDoComment()\n");

   if (rqptr->SsiTaskPtr->TraceState) SsiTraceStatement (rqptr);

   rqptr->SsiTaskPtr->SuppressLine = true;

   SysDclAst (&SsiParseLine, rqptr);
}

/*****************************************************************************/
/*
Config sets the default behaviour for a specific action for the rest of the 
document.
*/ 

SsiDoConfig (struct RequestStruct *rqptr)

{
   register char  *cptr, *dptr;
   register struct SsiTaskStruct  *tkptr;

   int  status;
   char  TagValue [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiDoConfig() |%s|\n", rqptr->SsiTaskPtr->Statement);

   tkptr = rqptr->SsiTaskPtr;

   if (tkptr->TraceState) SsiTraceStatement (rqptr);

   dptr = tkptr->Statement;
   while (*dptr && !ISLWS(*dptr)) dptr++;
   while (*dptr && ISLWS(*dptr)) dptr++;

   if (strsame (dptr, "ERRMSG", 6))
   {
      dptr += SsiGetTagValue (rqptr, dptr, TagValue, sizeof(TagValue));
      cptr = VmGetHeap (rqptr, strlen(TagValue)+1);
      strcpy (tkptr->ErrMsgPtr = cptr, TagValue);
   }
   else
   if (strsame (dptr, "TIMEFMT", 7))
   {
      dptr += SsiGetTagValue (rqptr, dptr, TagValue, sizeof(TagValue));
      cptr = VmGetHeap (rqptr, strlen(TagValue)+1);
      strcpy (tkptr->TimeFmtPtr = cptr, TagValue);
   }
   else
   if (strsame (dptr, "SIZEFMT", 7))
   {
      dptr += SsiGetTagValue (rqptr, dptr, TagValue, sizeof(TagValue));
      if (!(strsame (TagValue, "abbrev", -1) ||
            strsame (TagValue, "bytes", -1) ||
            strsame (TagValue, "blocks", -1)))
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_TAG_VALUE), FI_LI);
         SsiEnd (rqptr);
         return;
      }
      cptr = VmGetHeap (rqptr, strlen(TagValue)+1);
      strcpy (tkptr->SizeFmtPtr = cptr, TagValue);
   }
   else
   {
      SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_TAG_UNKNOWN), FI_LI);
      SsiEnd (rqptr);
      return;
   }

   /* declare an AST to execute the required function */
   SysDclAst (&SsiParseLine, rqptr);
}

/*****************************************************************************/
/*
Process a DCL/EXEC statement.  The array at the beginning of this function
provides  the allowed DCL statements and their DCL command equivalents.  Only
innocuous  DCL commands are allowed (hopefully! e.g. SHOW, WRITE SYS$OUTPUT)
for the  average document.  For documents owned by SYSTEM and NOT WORLD
WRITEABLE (for obvious reasons) the execution of any DCL command or command
procedure is allowed allowing maximum flexibility for the privileged document
author.
*/ 

SsiDoDcl (struct RequestStruct *rqptr)

{
   /*
      First element is allowed DCL command.
      Second is actual DCL verb/command/syntax.
      Third, if non-empty, idicates a file specification is required.
      Fourth, if non-empty, indicates it is for privileged documents only!
   */
   static char  *SupportedDcl [] =
   {
      "cmd", "", "", "P",                    /* privileged only! */
      "dir", "directory ", "", "",           /* any document */
      "exec", "", "", "P",                   /* privileged only! */
      "file", "@", "Y", "P",                 /* privileged only! */
      "run", "run ", "Y", "P",               /* privileged only! */
      "say", "write sys$output", "", "",     /* any document */
      "show", "show", "", "",                /* any document */
      "vdir", "directory ", "", "",          /* any document */
      "virtual", "@", "Y", "P",              /* privileged only! */
      "vrun", "run ", "Y", "P",              /* privileged only! */
      "", "", "", "P"  /* must be terminated by empty/privileged command! */
   };

   register int  idx;
   register char  *cptr, *dptr, *sptr, *zptr;
   register struct SsiTaskStruct  *tkptr;

   int  status;
   char  DclCommand [1024],
         TagValue [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiDoDcl() |%s|\n", rqptr->SsiTaskPtr->Statement);

   tkptr = rqptr->SsiTaskPtr;

   if (tkptr->TraceState) SsiTraceStatement (rqptr);

   dptr = rqptr->SsiTaskPtr->Statement;
   while (*dptr && !ISLWS(*dptr)) dptr++;
   while (*dptr && ISLWS(*dptr)) dptr++;

   for (idx = 0; *SupportedDcl[idx]; idx += 4)
   {
      if (Debug)
         fprintf (stdout, "|%s|%s|\n", SupportedDcl[idx], SupportedDcl[idx+1]);
      /* compare the user-supplied DCL command to the list of allowed */
      cptr = SupportedDcl[idx];
      sptr = dptr;
      while (*cptr && *sptr != '=' && toupper(*cptr) == toupper(*sptr))
         { cptr++; sptr++; }
      if (!*cptr && *sptr == '=') break;
   }

   if (!SupportedDcl[idx][0] &&
       !strsame (dptr = rqptr->SsiTaskPtr->Statement, "#exec ", 6))
   {
      /************************/
      /* unsupported command! */
      /************************/

      SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_DCL_UNSUPPORTED), FI_LI);
      SsiEnd (rqptr);
      return;
   }

   /*************************/
   /* supported DCL command */
   /*************************/

   if (SupportedDcl[idx+3][0])
   {
      /******************************/
      /* "privileged" DCL requested */
      /******************************/

      if (!Config.SsiExecEnabled)
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_DCL_DISABLED), FI_LI);
         SsiEnd (rqptr);
         return;
      }

      /* SYSTEM is UIC [1,4] */
      if (tkptr->FileXabPro.xab$w_grp != 1 ||
          tkptr->FileXabPro.xab$w_mbm != 4)
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_DCL_NOT_SYSTEM), FI_LI);
         SsiEnd (rqptr);
         return;
      }

      /* protect word: wwggooss, protect bits: dewr, 1 denies access */
      if (!(tkptr->FileXabPro.xab$w_pro & 0x2000))
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_DCL_NOT_WORLD), FI_LI);
         SsiEnd (rqptr);
         return;
      }
   }

   /****************************************************/
   /* create the DCL command from array and parameters */
   /****************************************************/

   cptr = SupportedDcl[idx+1];
   sptr = DclCommand;
   while (*cptr) *sptr++ = *cptr++;
   *sptr = '\0';

   if (SupportedDcl[idx+2][0])
   {
      /* file specification involved */
      dptr += SsiGetFileSpec (rqptr, dptr, TagValue);
      for (cptr = TagValue; *cptr; *sptr++ = *cptr++);
      *sptr = '\0';
   }
   else
   {
      /* some DCL verb/parameters or another */
      dptr += SsiGetTagValue (rqptr, dptr, TagValue, sizeof(TagValue));
      if (TagValue[0] && sptr > DclCommand) *sptr++ = ' ';
      for (cptr = TagValue; *cptr; *sptr++ = *cptr++);
      *sptr = '\0';
   }

   if (Debug) fprintf (stdout, "DclCommand |%s|\n", DclCommand);

   /* by default HTML-forbidden characters in DCL output are escaped */
   rqptr->NetWriteBufferedEscapeHtml = true;

   while (*dptr)
   {
      if (Debug) fprintf (stdout, "dptr |%s|\n", dptr);

      if (rqptr->SsiTaskPtr->SsiProblemEncountered) break;

      while (*dptr && ISLWS(*dptr)) dptr++;
      if (!*dptr) break;

      if (strsame (dptr, "PAR=", 4))
      {
         /* parameters to the command procedure, directory, etc. */
         dptr += SsiGetTagValue (rqptr, dptr, TagValue, sizeof(TagValue));
         if (TagValue[0]) *sptr++ = ' ';
         for (cptr = TagValue; *cptr; *sptr++ = *cptr++);
         *sptr = '\0';
      }
      else
      if (strsame (dptr, "TYPE=", 5))
      {
         dptr += SsiGetTagValue (rqptr, dptr, TagValue, sizeof(TagValue));
         if (TagValue[0] && strsame (TagValue, "text/html", -1))
            rqptr->NetWriteBufferedEscapeHtml = tkptr->TraceState;
         else
            rqptr->NetWriteBufferedEscapeHtml = true;
      }
      else
      if (*dptr)
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_TAG_UNKNOWN), FI_LI);
         SsiEnd (rqptr);
         return;
      }
   }

   if (rqptr->SsiTaskPtr->SsiProblemEncountered)
   {
      SsiEnd (rqptr);
      return;
   }

   DclBegin (rqptr, &SsiParseLine, DclCommand, NULL);
}

/*****************************************************************************/
/*
Generate an "Index of" directory listing by calling DirBegin() task.
*/

SsiDoDir (struct RequestStruct *rqptr)

{
   register char  *cptr, *dptr;

   int  status;
   char  DirSpec [256],
         TagValue [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiDoDir() |%s|\n", rqptr->SsiTaskPtr->Statement);

   if (rqptr->SsiTaskPtr->TraceState) SsiTraceStatement (rqptr);

   TagValue[0] = '\0';

   dptr = rqptr->SsiTaskPtr->Statement;
   while (*dptr && !ISLWS(*dptr)) dptr++;

   while (*dptr)
   {
      if (Debug) fprintf (stdout, "dptr |%s|\n", dptr);

      if (rqptr->SsiTaskPtr->SsiProblemEncountered) break;

      while (*dptr && ISLWS(*dptr)) dptr++;
      if (!*dptr) break;

      if (strsame (dptr, "FILE=", 5) || strsame (dptr, "VIRTUAL=", 8))
         dptr += SsiGetFileSpec (rqptr, dptr, DirSpec);
      else
      if (strsame (dptr, "PAR=", 4))
         dptr += SsiGetTagValue (rqptr, dptr, TagValue, sizeof(TagValue));
      else
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_TAG_UNKNOWN), FI_LI);
         SsiEnd (rqptr);
         return;
      }
   }

   if (rqptr->SsiTaskPtr->SsiProblemEncountered)
   {
      SsiEnd (rqptr);
      return;
   }

   if (Debug) fprintf (stdout, "|%s|%s|\n", DirSpec, TagValue);
   if (TagValue[0])
   {
      cptr = VmGetHeap (rqptr, strlen(TagValue)+1);
      strcpy (cptr, TagValue);
   }
   else
      cptr = "";

   DirBegin (rqptr, &SsiParseLine, DirSpec, cptr, "");
}

/*****************************************************************************/
/*
Output a server or user-assigned variable.
*/

boolean SsiDoEcho (struct RequestStruct *rqptr)

{
   register char  *cptr, *dptr, *sptr, *zptr;
   register struct SsiTaskStruct  *tkptr;

   int  status;
   char  String [256],
         TimeFormat [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiDoEcho() |%s|\n", rqptr->SsiTaskPtr->Statement);

   tkptr = rqptr->SsiTaskPtr;

   if (tkptr->TraceState) SsiTraceStatement (rqptr);

   TimeFormat[0] = '\0';

   dptr = rqptr->SsiTaskPtr->Statement;
   while (*dptr && !ISLWS(*dptr)) dptr++;
   while (*dptr && ISLWS(*dptr)) dptr++;

   while (*dptr)
   {
      if (Debug) fprintf (stdout, "dptr |%s|\n", dptr);

      if (rqptr->SsiTaskPtr->SsiProblemEncountered) break;
 
      while (*dptr && ISLWS(*dptr)) dptr++;
      if (!*dptr) break;

      if (*dptr == '\"')
      {
         /* format is <!--#echo "blah-blah" --> */
         dptr += SsiGetTagValue (rqptr, dptr, cptr = String, sizeof(String));
      }
      else
      if (strsame (dptr, "VALUE=", 6) ||
          strsame (dptr, "VAR=", 4))
      {
         /* format is <!--#echo var="" --> */
         dptr += SsiGetTagValue (rqptr, dptr, cptr = String, sizeof(String));
      }
      else
      {
         /* format is <!--#echo NAME[=format] --> */
         zptr = (sptr = String) + sizeof(String);
         while (*dptr && !ISLWS(*dptr) && *dptr != '=' && sptr < zptr)
            *sptr++ = toupper(*dptr++);
         if (sptr >= zptr)
         {
            SsiProblem (rqptr, MsgFor(rqptr,MSG_GENERAL_OVERFLOW), FI_LI);
            SsiEnd (rqptr);
            return;
         }
         *sptr = '\0';
         if (Debug) fprintf (stdout, "String |%s|\n", String);

         if (*dptr == '=')
            dptr += SsiGetTagValue (rqptr, dptr,
                                    TimeFormat, sizeof(TimeFormat));
         else
            TimeFormat[0] = '\0';

         cptr = SsiGetVar (rqptr, String, TimeFormat, false);
      }
      if (rqptr->SsiTaskPtr->SsiProblemEncountered) break;
      SsiWriteBuffered (rqptr, NULL, cptr, strlen(cptr));
   }

   if (rqptr->SsiTaskPtr->SsiProblemEncountered)
   {
      SsiEnd (rqptr);
      return;
   }

   SysDclAst (&SsiParseLine, rqptr);
}

/*****************************************************************************/
/*
Output the specified file's creation date and time.
*/

boolean SsiDoFCreated (struct RequestStruct *rqptr)

{
   register char  *dptr, *sptr, *zptr;

   char  FileName [256],
         TimeFormat [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiDoFCreated() |%s|\n",
               rqptr->SsiTaskPtr->Statement);

   if (rqptr->SsiTaskPtr->TraceState) SsiTraceStatement (rqptr);

   FileName[0] = TimeFormat[0] = '\0';

   dptr = rqptr->SsiTaskPtr->Statement;
   while (*dptr && !ISLWS(*dptr)) dptr++;

   while (*dptr)
   {
      if (Debug) fprintf (stdout, "dptr |%s|\n", dptr);

      if (rqptr->SsiTaskPtr->SsiProblemEncountered) break;

      while (*dptr && ISLWS(*dptr)) dptr++;
      if (!*dptr) break;

      if (strsame (dptr, "FILE=", 5) || strsame (dptr, "VIRTUAL=", 8))
         dptr += SsiGetFileSpec (rqptr, dptr, FileName);
      else
      if (strsame (dptr, "FMT=", 4))
         dptr += SsiGetTagValue (rqptr, dptr, TimeFormat, sizeof(TimeFormat));
      else
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_TAG_UNKNOWN), FI_LI);
         SsiEnd (rqptr);
         return;
      }
   }

   if (rqptr->SsiTaskPtr->SsiProblemEncountered)
   {
      SsiEnd (rqptr);
      return;
   }

   SsiFileDetails (rqptr, &SsiParseLine, FileName, FILE_FCREATED, TimeFormat);
}

/*****************************************************************************/
/*
Flow control statement. If the current flow control is executing then it now
disallowed. If the current flow control structure (including "#if"s, "#orif"s
and other "#elif"s) have not allowed execution then evaluate the conditional
and allow execution if true.
*/

SsiDoElif (struct RequestStruct *rqptr)

{
   register struct SsiTaskStruct  *tkptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiDoElif() |%s|\n", rqptr->SsiTaskPtr->Statement);

   tkptr = rqptr->SsiTaskPtr;

   tkptr->SuppressLine = true;

   /* if previous flow control was not an "#if" or "#elif" */
   if (tkptr->FlowControlState[tkptr->FlowControlIndex] != SSI_STATE_IF &&
       tkptr->FlowControlState[tkptr->FlowControlIndex] != SSI_STATE_ELIF)
   {
      SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_FLOW_CONTROL), FI_LI);
      SsiEnd (rqptr);
      return;
   }

   tkptr->FlowControlState[tkptr->FlowControlIndex] = SSI_STATE_ELIF;

   if (tkptr->FlowControlHasExecuted[rqptr->SsiTaskPtr->FlowControlIndex])
   {
      /* if flow control has already output just continue parsing */
      tkptr->FlowControlIsExecuting[tkptr->FlowControlIndex] = false;
      SysDclAst (&SsiParseLine, rqptr);
      return;
   }

   /* if the parent level is executing and it evaluates true then execute */
   if (tkptr->FlowControlIsExecuting[tkptr->FlowControlIndex-1])
   {
      if (tkptr->TraceState) SsiTraceStatement (rqptr);
      tkptr->FlowControlHasExecuted[tkptr->FlowControlIndex] =
      tkptr->FlowControlIsExecuting[tkptr->FlowControlIndex] =
         SsiEvaluate (rqptr);
   }

   /* declare an AST to execute the next function */
   SysDclAst (&SsiParseLine, rqptr);
}

/*****************************************************************************/
/*
Flow control statement.  If the current flow control structure (including
"#if"s, "#orif"s and/or "#elif"s) have not allowed execution then this will.
*/

SsiDoElse (struct RequestStruct *rqptr)

{
   register struct SsiTaskStruct  *tkptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiDoElse()\n");

   tkptr = rqptr->SsiTaskPtr;

   tkptr->SuppressLine = true;

   /* if previous flow control was not an "#if" or "#elif" */
   if (tkptr->FlowControlState[tkptr->FlowControlIndex] != SSI_STATE_IF &&
       tkptr->FlowControlState[tkptr->FlowControlIndex] != SSI_STATE_ELIF)
   {
      SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_FLOW_CONTROL), FI_LI);
      SsiEnd (rqptr);
      return;
   }

   tkptr->FlowControlState[tkptr->FlowControlIndex] = SSI_STATE_ELSE;

   /* if there has been no output so far */
   if (tkptr->FlowControlIsExecuting[tkptr->FlowControlIndex-1] &&
       !tkptr->FlowControlHasExecuted[tkptr->FlowControlIndex])
   {
      if (tkptr->TraceState) SsiTraceStatement (rqptr);
      tkptr->FlowControlHasExecuted[tkptr->FlowControlIndex] =
      tkptr->FlowControlIsExecuting[tkptr->FlowControlIndex] = true;
   }
   else
      tkptr->FlowControlIsExecuting[tkptr->FlowControlIndex] = false;

   /* declare an AST to execute the next function */
   SysDclAst (&SsiParseLine, rqptr);
}

/*****************************************************************************/
/*
Flow control statement.
*/

SsiDoEndif (struct RequestStruct *rqptr)

{
   register struct SsiTaskStruct  *tkptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiDoEndif()\n");

   tkptr = rqptr->SsiTaskPtr;

   if (tkptr->TraceState) SsiTraceStatement (rqptr);

   tkptr->SuppressLine = true;

   if (tkptr->FlowControlIndex)
   {
      tkptr->FlowControlIndex--;
      /* declare an AST to execute the next function */
      SysDclAst (&SsiParseLine, rqptr);
      return;
   }

   SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_FLOW_CONTROL), FI_LI);
   SsiEnd (rqptr);
}

/*****************************************************************************/
/*
Output the specified file's last modification date and time.
*/

SsiDoFLastMod (struct RequestStruct *rqptr)

{
   register char  *dptr, *sptr, *zptr;

   char  FileName [256],
         TimeFormat [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiDoFLastMod() |%s|\n",
               rqptr->SsiTaskPtr->Statement);

   if (rqptr->SsiTaskPtr->TraceState) SsiTraceStatement (rqptr);

   FileName[0] = TimeFormat[0] = '\0';

   dptr = rqptr->SsiTaskPtr->Statement;
   while (*dptr && !ISLWS(*dptr)) dptr++;

   while (*dptr)
   {
      if (Debug) fprintf (stdout, "dptr |%s|\n", dptr);

      if (rqptr->SsiTaskPtr->SsiProblemEncountered) break;

      while (*dptr && ISLWS(*dptr)) dptr++;
      if (!*dptr) break;

      if (strsame (dptr, "FILE=", 5) || strsame (dptr, "VIRTUAL=", 8))
         dptr += SsiGetFileSpec (rqptr, dptr, FileName);
      else
      if (strsame (dptr, "FMT=", 4))
         dptr += SsiGetTagValue (rqptr, dptr, TimeFormat, sizeof(TimeFormat));
      else
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_TAG_UNKNOWN), FI_LI);
         SsiEnd (rqptr);
         return;
      }
   }

   if (rqptr->SsiTaskPtr->SsiProblemEncountered)
   {
      SsiEnd (rqptr);
      return;
   }

   SsiFileDetails (rqptr, &SsiParseLine, FileName, FILE_FLASTMOD, TimeFormat);
}

/*****************************************************************************/
/*
Output the specified file's size.
*/

SsiDoFSize (struct RequestStruct *rqptr)

{
   register char  *dptr;

   char  FileSize [32],
         FileName [256],
         SizeFormat [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiDoFSize() |%s|\n", rqptr->SsiTaskPtr->Statement);

   if (rqptr->SsiTaskPtr->TraceState) SsiTraceStatement (rqptr);

   FileName[0] = SizeFormat[0] = '\0';

   dptr = rqptr->SsiTaskPtr->Statement;
   while (*dptr && !ISLWS(*dptr)) dptr++;

   while (*dptr)
   {
      if (Debug) fprintf (stdout, "dptr |%s|\n", dptr);

      if (rqptr->SsiTaskPtr->SsiProblemEncountered) break;

      while (*dptr && ISLWS(*dptr)) dptr++;
      if (!*dptr) break;

      if (strsame (dptr, "FILE=", 5) || strsame (dptr, "VIRTUAL=", 8))
         dptr += SsiGetFileSpec (rqptr, dptr, FileName);
      else
      if (strsame (dptr, "FMT=", 4))
         dptr += SsiGetTagValue (rqptr, dptr, SizeFormat, sizeof(SizeFormat));
      else
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_TAG_UNKNOWN), FI_LI);
         SsiEnd (rqptr);
         return;
      }
   }

   if (rqptr->SsiTaskPtr->SsiProblemEncountered)
   {
      SsiEnd (rqptr);
      return;
   }

   SsiFileDetails (rqptr, &SsiParseLine, FileName, FILE_FSIZE, SizeFormat);
}

/*****************************************************************************/
/*
Flow control statement.  If the parent level is executing evalutate the
conditional and allow execution if true.
*/

SsiDoIf (struct RequestStruct *rqptr)

{
   register struct SsiTaskStruct  *tkptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiDoIf()\n");

   tkptr = rqptr->SsiTaskPtr;

   tkptr->SuppressLine = true;

   tkptr->FlowControlIndex++;
   if (tkptr->FlowControlIndex > SSI_MAX_FLOW_CONTROL)
   {
      tkptr->FlowControlIndex = 0;
      SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_FLOW_CONTROL), FI_LI);
      SsiEnd (rqptr);
      return;
   }

   tkptr->FlowControlState[tkptr->FlowControlIndex] = SSI_STATE_IF;

   /* if the parent level is executing and it evaluates true then execute */
   if (tkptr->FlowControlIsExecuting[tkptr->FlowControlIndex-1])
   {
      if (tkptr->TraceState) SsiTraceStatement (rqptr);
      tkptr->FlowControlHasExecuted[tkptr->FlowControlIndex] =
      tkptr->FlowControlIsExecuting[tkptr->FlowControlIndex] =
         SsiEvaluate (rqptr);
   }
   else
      tkptr->FlowControlHasExecuted[tkptr->FlowControlIndex] =
      tkptr->FlowControlIsExecuting[tkptr->FlowControlIndex] = false;

   /* declare an AST to execute the next function */
   SysDclAst (&SsiParseLine, rqptr);
}

/*****************************************************************************/
/*
Include the contents of the specified file by calling FileBegin() task.
*/

SsiDoInclude (struct RequestStruct *rqptr)

{
   register char  *cptr, *dptr;

   boolean  PreTagFileContents;
   int  status;
   char  FileFormat [256],
         FileName [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiDoInclude() |%s|\n",
               rqptr->SsiTaskPtr->Statement);

   if (rqptr->SsiTaskPtr->TraceState) SsiTraceStatement (rqptr);

   FileName[0] = FileFormat[0] = '\0';

   dptr = rqptr->SsiTaskPtr->Statement;
   while (*dptr && !ISLWS(*dptr)) dptr++;

   while (*dptr)
   {
      if (Debug) fprintf (stdout, "dptr |%s|\n", dptr);

      if (rqptr->SsiTaskPtr->SsiProblemEncountered) break;

      while (*dptr && ISLWS(*dptr)) dptr++;
      if (!*dptr) break;

      if (strsame (dptr, "FILE=", 5) || strsame (dptr, "VIRTUAL=", 8))
         dptr += SsiGetFileSpec (rqptr, dptr, FileName);
      else
      if (strsame (dptr, "TYPE=", 5))
         dptr += SsiGetTagValue (rqptr, dptr, FileFormat, sizeof(FileFormat));
      else
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_TAG_UNKNOWN), FI_LI);
         SsiEnd (rqptr);
         return;
      }               
   }

   if (rqptr->SsiTaskPtr->SsiProblemEncountered)
   {
      SsiEnd (rqptr);
      return;
   }

   if (Debug) fprintf (stdout, "|%s|%s|\n", FileName, FileFormat);

   /* find the file extension and set the content type */
   for (cptr = FileName; *cptr && *cptr != ']'; cptr++);
   while (*cptr && *cptr != '.') cptr++;
   cptr = ConfigContentType (NULL, cptr);

   if (!strsame (cptr, "text/", 5))
   {
      SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_INCLUDE_NOT_TEXT), FI_LI);
      SsiEnd (rqptr);
      return;
   }

   if (strsame (cptr, ConfigContentTypeSsi, -1) &&
       !strsame (FileFormat, "text/plain", -1))
      SsiBegin (rqptr, &SsiParseLine, &SsiIncludeError, FileName);
   else
   {
      if (strsame (FileFormat, "text/plain", -1))
         PreTagFileContents = true;
      else
         PreTagFileContents = false;

      FileBegin (rqptr, &SsiParseLine, &SsiIncludeError, FileName,
                 cptr, false, PreTagFileContents, false);
   }
}

/*****************************************************************************/
/*
*/

SsiIncludeError (struct RequestStruct *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiIncludeError() |%s|\n",
               rqptr->SsiTaskPtr->Statement);

   SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_INCLUDE_ACCESS), FI_LI);
   SsiEnd (rqptr);
}

/*****************************************************************************/
/*
Flow control statement. If the preceding control statement was an "#if" or
"#elif" and it did not evaluate true then this allows another chance to
execute this segment. It is essentially an OR on the last evalution. The
evaluation here is not performed if the previous allowed execution.
*/

SsiDoOrif (struct RequestStruct *rqptr)

{
   register struct SsiTaskStruct  *tkptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiDoOrif()\n");

   tkptr = rqptr->SsiTaskPtr;

   tkptr->SuppressLine = true;

   /* if previous flow control was not an "#if" or "#elif" */
   if (tkptr->FlowControlState[tkptr->FlowControlIndex] != SSI_STATE_IF &&
       tkptr->FlowControlState[tkptr->FlowControlIndex] != SSI_STATE_ELIF)
   {
      SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_FLOW_CONTROL), FI_LI);
      SsiEnd (rqptr);
      return;
   }

   /* if the parent level is executing and the current is not then evaluate */
   if (tkptr->FlowControlIsExecuting[tkptr->FlowControlIndex-1] &&
       !tkptr->FlowControlIsExecuting[tkptr->FlowControlIndex])
   {
      if (tkptr->TraceState) SsiTraceStatement (rqptr);
      tkptr->FlowControlHasExecuted[tkptr->FlowControlIndex] =
      tkptr->FlowControlIsExecuting[tkptr->FlowControlIndex] =
         SsiEvaluate (rqptr);
   }

   /* declare an AST to execute the next function */
   SysDclAst (&SsiParseLine, rqptr);
}

/*****************************************************************************/
/*
Print all server and user-assigned variables. Usually used for no more than
document debugging. Variable names and values have HTML- forbidden characters
escaped.
*/

SsiDoPrintEnv (struct RequestStruct *rqptr)

{
   register char  *cptr, *sptr;
   register struct ListEntryStruct  *leptr;
   register struct SsiVarStruct  *varptr;

   boolean  EscapeHtmlBuffer;
   int  status,
        Length;
   char  VarName [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiDoPrintEnv()\n");

   if (rqptr->SsiTaskPtr->TraceState) SsiTraceStatement (rqptr);

   SsiWriteBuffered (rqptr, NULL, "<PRE>", 5);
   EscapeHtmlBuffer = rqptr->NetWriteBufferedEscapeHtml;
   rqptr->NetWriteBufferedEscapeHtml = true;

   /* SSI variables */
   cptr = SsiGetServerVar (rqptr, sptr = "CREATED", NULL);
   SsiPrintEnvVar (rqptr, sptr, cptr);
   cptr = SsiGetServerVar (rqptr, sptr = "DATE_GMT", NULL);
   SsiPrintEnvVar (rqptr, sptr, cptr);
   cptr = SsiGetServerVar (rqptr, sptr = "DATE_LOCAL", NULL);
   SsiPrintEnvVar (rqptr, sptr, cptr);
   cptr = SsiGetServerVar (rqptr, sptr = "DOCUMENT_NAME", NULL);
   SsiPrintEnvVar (rqptr, sptr, cptr);
   cptr = SsiGetServerVar (rqptr, sptr = "DOCUMENT_URI", NULL);
   SsiPrintEnvVar (rqptr, sptr, cptr);
   cptr = SsiGetServerVar (rqptr, sptr = "FILE_NAME", NULL);
   SsiPrintEnvVar (rqptr, sptr, cptr);
   cptr = SsiGetServerVar (rqptr, sptr = "LAST_MODIFIED", NULL);
   SsiPrintEnvVar (rqptr, sptr, cptr);
   cptr = SsiGetServerVar (rqptr, sptr = "QUERY_STRING_UNESCAPED", NULL);
   SsiPrintEnvVar (rqptr, sptr, cptr);

   /* CGI variables */
   SsiWriteBuffered (rqptr, NULL, "\n", 1);
   cptr = rqptr->SsiTaskPtr->CgiBufferPtr;
   for (;;)
   {
      if (!(Length = *(short*)cptr)) break;
      SsiPrintEnvVar (rqptr, cptr+sizeof(short)+DclCgiVariablePrefixLength,
                      NULL);
      cptr += Length + sizeof(short);
   }

   /* user variables */
   for (leptr = rqptr->SsiTaskPtr->SsiVarList.HeadPtr;
        leptr != NULL;
        leptr = leptr->NextPtr)
   {
      /* separate the first user variable using a blank line */
      if (leptr == rqptr->SsiTaskPtr->SsiVarList.HeadPtr)
         SsiWriteBuffered (rqptr, NULL, "\n", 1);
      varptr = (struct SsiVarStruct*)leptr;
      SsiPrintEnvVar (rqptr, varptr->NamePtr, varptr->ValuePtr);
   }

   rqptr->NetWriteBufferedEscapeHtml = EscapeHtmlBuffer;
   SsiWriteBuffered (rqptr, NULL, "</PRE>", 6);

   SysDclAst (&SsiParseLine, rqptr);
}

/*****************************************************************************/
/*
Simply do the "printenv" of a single variable for SsiDoPrintEnv().
*/

SsiPrintEnvVar
(
struct RequestStruct *rqptr,
char* VarName,
char* VarValue
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiPrintEnvVar() |%s|%s|\n", VarName, VarValue);

   SsiWriteBuffered (rqptr, NULL, VarName, strlen(VarName));
   if (VarValue != NULL)
   {
      SsiWriteBuffered (rqptr, NULL, "=", 1);
      SsiWriteBuffered (rqptr, NULL, VarValue, strlen(VarValue));
   }
   SsiWriteBuffered (rqptr, NULL, "\n", 1);
}

/*****************************************************************************/
/*
Set a user variable. Server variables cannot be set. Existing variable names
are of course set to the new value. New variables are created ex nihlo and
added to a simple linked list.
*/

SsiDoSet (struct RequestStruct *rqptr)

{
   register char  *cptr, *dptr;
   register struct ListEntryStruct  *leptr;
   register struct SsiVarStruct  *varptr;

   boolean  PreTagFileContents;
   int  status,
        Size,
        VarNameLength,
        VarValueLength;
   char  VarName [256],
         VarValue [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiDoSet() |%s|\n", rqptr->SsiTaskPtr->Statement);

   rqptr->SsiTaskPtr->SuppressLine = true;

   if (rqptr->SsiTaskPtr->TraceState) SsiTraceStatement (rqptr);

   dptr = rqptr->SsiTaskPtr->Statement;
   while (*dptr && !ISLWS(*dptr)) dptr++;

   VarName[0] = VarValue[0]= '\0';
   VarNameLength = VarValueLength = 0;

   while (*dptr)
   {
      if (Debug) fprintf (stdout, "dptr |%s|\n", dptr);

      if (rqptr->SsiTaskPtr->SsiProblemEncountered)
      {
         SsiEnd (rqptr);
         return;
      }

      while (*dptr && ISLWS(*dptr)) dptr++;
      if (!*dptr) break;

      if (strsame (dptr, "VAR=", 4))
      {
         dptr += SsiGetTagValue (rqptr, dptr, VarName, sizeof(VarName));
         VarNameLength = strlen(VarName);
         continue;
      }

      if (strsame (dptr, "VALUE=", 6))
      {
         dptr += SsiGetTagValue (rqptr, dptr, VarValue, sizeof(VarValue));
         VarValueLength = strlen(VarValue);
      }
      else
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_TAG_UNKNOWN), FI_LI);
         SsiEnd (rqptr);
         return;
      }               

      if (Debug) fprintf (stdout, "|%s|%s|\n", VarName, VarValue);

      if (rqptr->SsiTaskPtr->SsiProblemEncountered)
      {
         SsiEnd (rqptr);
         return;
      }

      if (!VarName[0])
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_VARIABLE), FI_LI);
         SsiEnd (rqptr);
         return;
      }
      for (cptr = VarName; *cptr; cptr++)
         if (!isalnum(*cptr) && *cptr != '_') break;
      if (*cptr)
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_VARIABLE), FI_LI);
         SsiEnd (rqptr);
         return;
      }

      if (SsiGetServerVar (rqptr, VarName, NULL) != NULL)
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_VARIABLE), FI_LI);
         SsiEnd (rqptr);
         return;
      }

      /* allow for terminating nulls then round to a set amount */
      Size = sizeof(struct SsiVarStruct) + VarNameLength + VarValueLength + 2;
      if (Size < 128) Size = 128; else
         if (Size < 256) Size = 256; else
            if (Size < 1024) Size = 1024; else
               if (Size < 4096) Size = 4096;
      if (Debug) fprintf (stdout, "Size: %d\n", Size);

      /* look for an existing user variable */
      varptr = NULL;
      for (leptr = rqptr->SsiTaskPtr->SsiVarList.HeadPtr;
           leptr != NULL;
           leptr = leptr->NextPtr)
      {
         varptr = (struct SsiVarStruct*)leptr;
         if (Debug)
            fprintf (stdout, "Name Value |%s|%s|\n",
                     varptr->NamePtr, varptr->ValuePtr);
         /* break if variable name found in list */
         if (toupper(*VarName) == toupper(*varptr->NamePtr) &&
             strsame (VarName, varptr->NamePtr, -1))
            break;
         varptr = NULL;
      }
      if (Debug) fprintf (stdout, "varptr: %d\n", varptr);

      if (varptr == NULL)
      {
         /* new user variable */
         varptr = VmGetHeap (rqptr, Size);
         ListAddTail (&rqptr->SsiTaskPtr->SsiVarList, varptr);

         varptr->Size = Size;
         varptr->NamePtr = varptr->Data;
         memcpy (varptr->NamePtr, VarName, VarNameLength+1);
         varptr->NameLength = VarNameLength;
         varptr->ValuePtr = varptr->NamePtr + VarNameLength+1;
         memcpy (varptr->ValuePtr, VarValue, VarValueLength+1);
         varptr->ValueLength = VarValueLength;
      }
      else
      {
         /* found an existing user variable */
         if (varptr->Size < Size)
         {
            /* size needs to be increased */
            ListRemove (&rqptr->SsiTaskPtr->SsiVarList, varptr);
            VmFreeFromHeap (rqptr, varptr);
            varptr = VmGetHeap (rqptr, Size);
            ListAddTail (&rqptr->SsiTaskPtr->SsiVarList, varptr);

            varptr->Size = Size;
            varptr->NamePtr = varptr->Data;
            memcpy (varptr->NamePtr, VarName, VarNameLength+1);
            varptr->NameLength = VarNameLength;
            varptr->ValuePtr = varptr->NamePtr + VarNameLength+1;
            memcpy (varptr->ValuePtr, VarValue, VarValueLength+1);
            varptr->ValueLength = VarValueLength;
         }
         else
         {
            /* variable memory stayed the same */
            memcpy (varptr->ValuePtr, VarValue, VarValueLength+1);
            varptr->ValueLength = VarValueLength;
         }
      }

      if (rqptr->SsiTaskPtr->TraceState)
         SsiTraceSetVar (rqptr, VarName, VarValue);

      /* ready for the next round (if any) */
      VarValue[0]= '\0';
      VarValueLength = 0;
   }

   SysDclAst (&SsiParseLine, rqptr);
}

/*****************************************************************************/
/*
Stop processing the document here!
*/

SsiDoStop (struct RequestStruct *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiDoStop()\n");

   if (rqptr->SsiTaskPtr->TraceState) SsiTraceStatement (rqptr);

   SsiEnd (rqptr);
}

/*****************************************************************************/
/*
Turn the SSI trace on or off.
*/

boolean SsiDoTrace (struct RequestStruct *rqptr)

{
   register char  *cptr, *dptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiDoTrace() |%s|\n", rqptr->SsiTaskPtr->Statement);

   dptr = rqptr->SsiTaskPtr->Statement;
   while (*dptr && !ISLWS(*dptr)) dptr++;
   while (*dptr && ISLWS(*dptr)) dptr++;
   for (cptr = dptr; *cptr && !ISLWS(*cptr); cptr++);
   *cptr = '\0';

   if (strsame (dptr, "ON", -1))
   {
      if (!rqptr->SsiTaskPtr->TraceState)
      {
         SsiWriteBuffered (rqptr, NULL, "<PRE>", 5);
         rqptr->NetWriteBufferedEscapeHtml = true;
         rqptr->SsiTaskPtr->TraceState = true;
      }
      SsiTraceStatement (rqptr);
   }
   else
   if (strsame (dptr, "OFF", -1))
   {
      if (rqptr->SsiTaskPtr->TraceState)
      {
         SsiTraceStatement (rqptr);
         rqptr->NetWriteBufferedEscapeHtml = false;
         SsiWriteBuffered (rqptr, NULL, "</PRE>", 6);
         rqptr->SsiTaskPtr->TraceState = false;
      }
   }
   else
   if (rqptr->SsiTaskPtr->TraceState)
      SsiTraceStatement (rqptr);

   SysDclAst (&SsiParseLine, rqptr);
}

/*****************************************************************************/
/*
Display the current SSI file record.
Use NetWriteBuffered() so the line is always displayed.
*/

SsiTraceLine (struct RequestStruct *rqptr)

{
   static char  Nesting [48];
   static $DESCRIPTOR (NestingFaoDsc,
          "<FONT COLOR=\"#0000cc\"><B>[!3ZL:!UL]");
   static $DESCRIPTOR (NestingDsc, Nesting);

   boolean  EscapeHtmlBuffer;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiTraceLine()\n");

   sys$fao (&NestingFaoDsc, &Length, &NestingDsc,
            rqptr->SsiTaskPtr->LineNumber,
            rqptr->SsiTaskPtr->FlowControlIndex);

   if (rqptr->SsiTaskPtr->TraceHasBeenOutput &&
       !rqptr->SsiTaskPtr->PageHasBeenOutput)
      NetWriteBuffered (rqptr, NULL, "\n", 1);

   EscapeHtmlBuffer = rqptr->NetWriteBufferedEscapeHtml;
   rqptr->NetWriteBufferedEscapeHtml = false;
   NetWriteBuffered (rqptr, NULL, Nesting, Length);
   rqptr->NetWriteBufferedEscapeHtml = true;
   NetWriteBuffered (rqptr, NULL,
      rqptr->SsiTaskPtr->FileRab.rab$l_ubf,
      rqptr->SsiTaskPtr->FileRab.rab$w_rsz);
   rqptr->NetWriteBufferedEscapeHtml = false;
   NetWriteBuffered (rqptr, NULL, "</B></FONT>\n", 12);
   rqptr->NetWriteBufferedEscapeHtml = EscapeHtmlBuffer;
}

/*****************************************************************************/
/*
Display the current SSI statement.
Use NetWriteBuffered() so the variable is always displayed.
*/

SsiTraceStatement (struct RequestStruct *rqptr)

{
   boolean  EscapeHtmlBuffer;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiTraceStatement()\n");

   rqptr->SsiTaskPtr->TraceHasBeenOutput = true;
   EscapeHtmlBuffer = rqptr->NetWriteBufferedEscapeHtml;
   rqptr->NetWriteBufferedEscapeHtml = false;
   NetWriteBuffered (rqptr, NULL, "<FONT COLOR=\"#cc0000\"><B>[", 26);
   rqptr->NetWriteBufferedEscapeHtml = true;
   NetWriteBuffered (rqptr, NULL,
      rqptr->SsiTaskPtr->Statement, rqptr->SsiTaskPtr->StatementLength);
   rqptr->NetWriteBufferedEscapeHtml = false;
   NetWriteBuffered (rqptr, NULL, "]</B></FONT>", 12);
   rqptr->NetWriteBufferedEscapeHtml = EscapeHtmlBuffer;
}

/*****************************************************************************/
/*
Display the name and value of a variable as it is retrieved.
Use NetWriteBuffered() so the variable is always displayed.
*/

SsiTraceGetVar
(
struct RequestStruct *rqptr,
char *VarName,
char *VarValue
)
{
   boolean  EscapeHtmlBuffer;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiTraceGetVar()\n");

   rqptr->SsiTaskPtr->TraceHasBeenOutput = true;
   EscapeHtmlBuffer = rqptr->NetWriteBufferedEscapeHtml;
   rqptr->NetWriteBufferedEscapeHtml = false;
   NetWriteBuffered (rqptr, NULL, "<FONT COLOR=\"#cc00cc\"><B>[", 26);
   rqptr->NetWriteBufferedEscapeHtml = true;
   NetWriteBuffered (rqptr, NULL, VarName, strlen(VarName));
   NetWriteBuffered (rqptr, NULL, ":", 1);
   NetWriteBuffered (rqptr, NULL, VarValue, strlen(VarValue));
   rqptr->NetWriteBufferedEscapeHtml = false;
   NetWriteBuffered (rqptr, NULL, "]</B></FONT>", 12);
   rqptr->NetWriteBufferedEscapeHtml = EscapeHtmlBuffer;
}

/*****************************************************************************/
/*
Display the name and value of a variable as it is stored.
Use NetWriteBuffered() so the variable is always displayed.
*/

SsiTraceSetVar
(
struct RequestStruct *rqptr,
char *VarName,
char *VarValue
)
{
   boolean  EscapeHtmlBuffer;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiTraceSetVar()\n");

   rqptr->SsiTaskPtr->TraceHasBeenOutput = true;
   EscapeHtmlBuffer = rqptr->NetWriteBufferedEscapeHtml;
   rqptr->NetWriteBufferedEscapeHtml = false;
   NetWriteBuffered (rqptr, NULL, "<FONT COLOR=\"#cc00cc\"><B>[", 26);
   rqptr->NetWriteBufferedEscapeHtml = true;
   NetWriteBuffered (rqptr, NULL, VarName, strlen(VarName));
   NetWriteBuffered (rqptr, NULL, "=", 1);
   NetWriteBuffered (rqptr, NULL, VarValue, strlen(VarValue));
   rqptr->NetWriteBufferedEscapeHtml = false;
   NetWriteBuffered (rqptr, NULL, "]</B></FONT>", 12);
   rqptr->NetWriteBufferedEscapeHtml = EscapeHtmlBuffer;
}

/*****************************************************************************/
/*
First search the server-assigned variables, then the user-assigned variables.
If found return a pointer to a symbol's value string.  If no such symbol found
return a NULL.
*/

char* SsiGetVar
(
struct RequestStruct *rqptr,
char *VarName,
char *Format,
boolean CheckOnly
)
{
   char  *ValuePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiGetVar() |%s|\n", VarName);

   ValuePtr = SsiGetServerVar (rqptr, VarName, Format);
   if (rqptr->SsiTaskPtr->SsiProblemEncountered) return (ValuePtr);

   /* if it's a server assigned variable then return */
   if (ValuePtr != NULL)
   {
      if (rqptr->SsiTaskPtr->TraceState)
         SsiTraceGetVar (rqptr, VarName, ValuePtr);
      return (ValuePtr);
   }

   ValuePtr = SsiGetUserVar (rqptr, VarName);
   if (rqptr->SsiTaskPtr->SsiProblemEncountered) return (ValuePtr);

   /* if it's a user assigned variable then return */
   if (ValuePtr != NULL)
   {
      if (rqptr->SsiTaskPtr->TraceState)
         SsiTraceGetVar (rqptr, VarName, ValuePtr);
      return (ValuePtr);
   }

   if (CheckOnly) return (ValuePtr);

   /* variable not found */
   return (ValuePtr = MsgFor(rqptr,MSG_SSI_VARIABLE_NOT_FOUND));
}

/*****************************************************************************/
/*
Return a pointer to the value of a server-assigned variable, NULL if no such
variable.  The calling routine might want to check for an error message being
generated by bad format or time from SsiTimeString().
*/

char* SsiGetServerVar
(
struct RequestStruct *rqptr,
char *VarName,
char *Format
)
{
#define SIZEOF_TIME_STRING 64

   register char  *cptr, *sptr;
   register char  ch;
   register struct SsiTaskStruct  *tkptr;

   int  status,
        Length;
   unsigned long  BinTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiGetServerVar() |%s|\n", VarName);

   tkptr = rqptr->SsiTaskPtr;

   cptr = VarName;
   ch = toupper(cptr[0]);
   if (ch == 'D' && strsame (cptr, "DATE_LOCAL", -1))
   {
      cptr = VmGetHeap (rqptr, SIZEOF_TIME_STRING);
      SsiTimeString (rqptr, NULL, Format, cptr, SIZEOF_TIME_STRING);
      return (cptr);
   }
   else
   if (ch == 'D' && strsame (cptr, "DATE_GMT", -1))
   {
      sys$gettim (&BinTime);
      TimeAdjustGMT (true, &BinTime);
      cptr = VmGetHeap (rqptr, SIZEOF_TIME_STRING+5);
      SsiTimeString (rqptr, &BinTime, Format, cptr, SIZEOF_TIME_STRING);
      strcat (cptr, " GMT");
      return (cptr);
   }
   else
   if (ch == 'D' && strsame (cptr, "DOCUMENT_NAME", -1))
      return (tkptr->FileName);
   else
   if (ch == 'D' && strsame (cptr, "DOCUMENT_URI", -1))
      return (rqptr->PathInfoPtr);
   else
   if (ch == 'C' && strsame (cptr, "CREATED", -1))
   {
      cptr = VmGetHeap (rqptr, SIZEOF_TIME_STRING);
      SsiTimeString (rqptr, &tkptr->FileXabDat.xab$q_cdt, Format,
                     cptr, SIZEOF_TIME_STRING);
      return (cptr);
   }
   else
   if (ch == 'F' && strsame (cptr, "FILE_NAME", -1))
      return (tkptr->FileName);
   else
   if (ch == 'L' && strsame (cptr, "LAST_MODIFIED", -1))
   {
      cptr = VmGetHeap (rqptr, SIZEOF_TIME_STRING);
      SsiTimeString (rqptr, &tkptr->FileXabDat.xab$q_rdt, Format,
                     cptr, SIZEOF_TIME_STRING);
      return (cptr);
   }
   else
   if (ch == 'Q' && strsame (cptr, "QUERY_STRING_UNESCAPED", -1))
      return (rqptr->QueryStringPtr);
   else
   {
      cptr = rqptr->SsiTaskPtr->CgiBufferPtr;
      for (;;)
      {
         if (!(Length = *(short*)cptr)) break;
         for (sptr = cptr+sizeof(short)+DclCgiVariablePrefixLength;
              *sptr && *sptr != '=';
              sptr++);
         *sptr = '\0';
         if (strsame (cptr+sizeof(short)+DclCgiVariablePrefixLength,
             VarName, -1))
         {
            *sptr = '=';
            break;
         }
         *sptr = '=';
         cptr += Length + sizeof(short);
      }
      /* if found */
      if (Length) return (sptr+1);
   }

   /* not found */
   return (NULL);
}

/*****************************************************************************/
/*
Search the user-assigned variables. If found return a pointer to a symbol's
value string. If no such symbol found return a NULL.
*/

char* SsiGetUserVar
(
struct RequestStruct *rqptr,
char *VarName
)
{
   register struct ListEntryStruct  *leptr;
   register struct SsiVarStruct  *varptr;

   char  *StringPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiGetUserVar() |%s|\n", VarName);

   /* look through the user-assigned list of variables */
   for (leptr = rqptr->SsiTaskPtr->SsiVarList.HeadPtr;
        leptr != NULL;
        leptr = leptr->NextPtr)
   {
      varptr = (struct SsiVarStruct*)leptr;
      /* return if variable name found in list */
      if (toupper(VarName[0]) == toupper(*varptr->NamePtr) &&
          strsame (VarName, varptr->NamePtr, -1))
      {
         if (Debug) fprintf (stdout, "|%s|\n", varptr->ValuePtr);
         return (varptr->ValuePtr);
      }
   }

   /* no such variable found */
   if (Debug) fprintf (stdout, "NULL\n");
   return (NULL);
}

/*****************************************************************************/
/*
Used by flow-control statements that do an evaluation to make a decision.
*/

boolean SsiEvaluate (struct RequestStruct *rqptr)

{
   register char  *dptr;
   register struct SsiTaskStruct  *tkptr;

   boolean  NegateResult,
            Result;
   int  NumberOne,
        NumberTwo;
   char  ValueOne [256],
         ValueTwo [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiEvaluate() |%s|\n", rqptr->SsiTaskPtr->Statement);

   tkptr = rqptr->SsiTaskPtr;

   Result = false;

   dptr = rqptr->SsiTaskPtr->Statement;
   while (*dptr && !ISLWS(*dptr)) dptr++;

   while (*dptr)
   {
      if (Debug) fprintf (stdout, "dptr |%s|\n", dptr);

      if (rqptr->SsiTaskPtr->SsiProblemEncountered) return (false);

      while (*dptr && ISLWS(*dptr)) dptr++;
      if (!*dptr) break;

      if (*dptr == '!')
      {
         dptr++;
         NegateResult = true;
      }
      else
         NegateResult = false;

      if (strsame (dptr, "VALUE=", 6) ||
          strsame (dptr, "VAR=", 4) ||
          strsame (dptr, "PAR=", 4))
      {
         dptr += SsiGetTagValue (rqptr, dptr, ValueOne, sizeof(ValueOne));
         if (tkptr->SsiProblemEncountered) return (false);
         if (isdigit(ValueOne[0]))
            Result = atoi(ValueOne);
         else
            Result = ValueOne[0];
         if (NegateResult) Result = !Result;
         continue;
      }

      if (strsame (dptr, "SRCH=", 5))
      {
         dptr += SsiGetTagValue (rqptr, dptr, ValueTwo, sizeof(ValueTwo));
         if (tkptr->SsiProblemEncountered) return (false);
         Result = SearchTextString (ValueOne, ValueTwo, false, NULL);
      }
      else
      if (strsame (dptr, "EQS=", 4))
      {
         dptr += SsiGetTagValue (rqptr, dptr, ValueTwo, sizeof(ValueTwo));
         if (tkptr->SsiProblemEncountered) return (false);
         Result = strsame (ValueOne, ValueTwo, -1);
      }
      else
      if (strsame (dptr, "EQ=", 3))
      {
         dptr += SsiGetTagValue (rqptr, dptr, ValueTwo, sizeof(ValueTwo));
         if (tkptr->SsiProblemEncountered) return (false);
         NumberOne = NumberTwo = 0;
         NumberOne = atoi(ValueOne);
         NumberTwo = atoi(ValueTwo);
         Result = (NumberOne == NumberTwo);
      }
      else
      if (strsame (dptr, "GT=", 3))
      {
         dptr += SsiGetTagValue (rqptr, dptr, ValueTwo, sizeof(ValueTwo));
         if (tkptr->SsiProblemEncountered) return (false);
         NumberOne = NumberTwo = 0;
         NumberOne = atoi(ValueOne);
         NumberTwo = atoi(ValueTwo);
         Result = (NumberOne > NumberTwo);
      }
      else
      if (strsame (dptr, "LT=", 3))
      {
         dptr += SsiGetTagValue (rqptr, dptr, ValueTwo, sizeof(ValueTwo));
         if (tkptr->SsiProblemEncountered) return (false);
         NumberOne = NumberTwo = 0;
         NumberOne = atoi(ValueOne);
         NumberTwo = atoi(ValueTwo);
         Result = (NumberOne < NumberTwo);
      }
      else
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_TAG_UNKNOWN), FI_LI);
         return (false);
      }               

      if (NegateResult) Result = !Result;
      if (!Result) break;
   }

   if (Debug) fprintf (stdout, "Result: %d\n", Result);
   return (Result);
}

/*****************************************************************************/
/*
Using the locale formatting capabilities of function strftime(), output the 
time represented by the specified VMS quadword, binary time.  If 
'BinaryTimePtr' is NULL then default to the current time.  Returns number of 
characters placed into 'TimeString', or zero if an error.
*/ 

int SsiTimeString
(
struct RequestStruct *rqptr,
unsigned long *BinaryTimePtr,
char *TimeFmtPtr,
char *TimeString,
int SizeOfTimeString
)
{
   int  Length;
   unsigned long  BinaryTime [2];
   struct tm  UnixTime;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiTimeString() |%s|\n", TimeFmtPtr);

   if (TimeFmtPtr == NULL)
      TimeFmtPtr = rqptr->SsiTaskPtr->TimeFmtPtr;
   else
   if (!TimeFmtPtr[0]) 
      TimeFmtPtr = rqptr->SsiTaskPtr->TimeFmtPtr;
   if (BinaryTimePtr == NULL) sys$gettim (BinaryTimePtr = &BinaryTime);
   TimeVmsToUnix (BinaryTimePtr, &UnixTime);

   if (!(Length =
         strftime (TimeString, SizeOfTimeString, TimeFmtPtr, &UnixTime)))
   {
      SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_DATE_TIME), FI_LI);
      return (0);
   }
   return (Length);
}

/*****************************************************************************/
/*
Get the value of a tag parameter (e.g. tag_name="value"). Allows variable
substitution into tag values, a la Apache. Tag values can have variable values
substituted into them using a leading "{" and trailing '}' character sequence
with the variable name between. Otherwise reserved characters may be escaped
using a leading backslash. If comma-separated numbers are appended to a
substitution variable these become starting and ending indices, extracting
that range from the variable (a single number sets the count from zero).
Returns the number of characters scanned to get the value; note that this is
not necessarily the same as the number of characters in the variable value!
*/ 

int SsiGetTagValue
(
struct RequestStruct *rqptr,
char *String,
char *Value,
int SizeOfValue
)
{
   register boolean  Quoted;
   register char  *cptr, *sptr, *vptr, *vzptr, *zptr;

   boolean  Negate;
   int  ExtractCount,
        StartIndex;
   char  VarName [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiGetTagValue() |%s|\n", String);

   if (!*(cptr = String)) return (0);

   zptr = (sptr = Value) + SizeOfValue;
   for (cptr = String; *cptr && *cptr != '=' && *cptr != '\"'; cptr++);
   if (*cptr == '=') cptr++;
   Quoted = false;
   if (*cptr == '\"')
   {
      cptr++;
      Quoted = true;
   }

   while (((*cptr && Quoted && *cptr != '\"') ||
           (*cptr && !Quoted && !ISLWS(*cptr))) &&
          sptr < zptr)
   {
      if (*cptr != '{')
      {
         /*********************/
         /* literal character */
         /*********************/

         /* escape character? */
         if (*cptr == '\\') cptr++;
         if (*cptr && sptr < zptr) *sptr++ = *cptr++;
         continue;
      }

      /*************************/
      /* variable substitution */
      /*************************/

      cptr++;
      vzptr = (vptr = VarName) + sizeof(VarName);
      while (*cptr && (isalnum(*cptr) || *cptr == '_') && vptr < vzptr)
         *vptr++ = *cptr++;
      *vptr = '\0';

      if (*cptr == ',')
      {
         cptr++;
         StartIndex = 0;
         ExtractCount = 999999999;

         if (isdigit(*cptr))
         {
            /* substring */
            StartIndex = atoi(cptr);
            while (isdigit(*cptr)) cptr++;
            if (*cptr == ',') cptr++;
            if (isdigit(*cptr))
            {
               /* two numbers provide a start index and an extract count */
               ExtractCount = atoi(cptr);
               while (isdigit(*cptr)) cptr++;
               if (*cptr == ',') cptr++;
            }
            else
            {
               /* one number extracts from the start of the string */
               ExtractCount = StartIndex;
               StartIndex = 0;
            }
         }
         if (Debug) fprintf (stdout, "%d %d\n", StartIndex, ExtractCount);

         if (isalpha(*cptr))
         {
            /* "function" on variable */
            if (strsame (cptr, "length", 6))
            {
               static $DESCRIPTOR (LengthFaoDsc, "!UL\0");
               int  Length;
               char  String [32];
               $DESCRIPTOR (StringDsc, String);

               vptr = SsiGetVar (rqptr, VarName, NULL, false);
               if (rqptr->SsiTaskPtr->SsiProblemEncountered)
                  return (cptr - String);

               Length = 0;
               while (StartIndex-- && *vptr) vptr++;
               while (ExtractCount-- && *vptr) { vptr++; Length++; }
               sys$fao (&LengthFaoDsc, 0, &StringDsc, Length);
               vptr = String;
               while (*vptr && sptr < zptr) *sptr++ = *vptr++;
               while (*cptr && isalpha(*cptr)) cptr++;
            }
            else
            if (strsame (cptr, "exists", 6))
            {
               vptr = SsiGetVar (rqptr, VarName, NULL, true);
               if (vptr == NULL)
                  vptr = "";
               else
                  vptr = "true";
               while (*vptr && sptr < zptr) *sptr++ = *vptr++;
               while (*cptr && isalpha(*cptr)) cptr++;
            }
            else
            {
               SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_VARIABLE), FI_LI);
               return (cptr - String);
            }
         }
         else
         {
            vptr = SsiGetVar (rqptr, VarName, NULL, false);
            if (rqptr->SsiTaskPtr->SsiProblemEncountered)
               return (cptr - String);
            while (StartIndex-- && *vptr) vptr++;
            while (ExtractCount-- && *vptr && sptr < zptr)
               *sptr++ = *vptr++;
         }
      }
      else
      {
         /* get all of variable */
         vptr = SsiGetVar (rqptr, VarName, NULL, false);
         if (rqptr->SsiTaskPtr->SsiProblemEncountered)
            return (cptr - String);
         while (*vptr && sptr < zptr) *sptr++ = *vptr++;
      }

      if (*cptr != '}')
      {
         SsiProblem (rqptr, MsgFor(rqptr,MSG_SSI_VARIABLE), FI_LI);
         return (cptr - String);
      }
      cptr++;
   }

   if (sptr >= zptr)
   {
      SsiProblem (rqptr, MsgFor(rqptr,MSG_GENERAL_OVERFLOW), FI_LI);
      return (cptr - String);
   }

   *sptr = '\0';
   if (*cptr == '\"') cptr++;

   if (Debug) fprintf (stdout, "Value |%s|\n", Value);
   return (cptr - String);
}

/*****************************************************************************/
/*
Get a 'FILE="file_name"' or a 'VIRTUAL="file_name"'.  Maximum number of 
characters allowed in value is 256.   Returns the number of characters scanned 
to get the value.
*/ 

int SsiGetFileSpec
(
struct RequestStruct *rqptr,
char *String,
char *FileName
)
{
   register char  *fptr, *sptr, *zptr;

   char  VirtualPath [256],
         VirtualFileName [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiGetFileSpec() |%s|\n", String);

   sptr = String;

   if (toupper(*sptr) == 'V')
   {
      /* virtual path to file */
      while (*sptr && *sptr != '=') sptr++;
      if (*sptr) sptr++;
      if (*sptr == '\"') sptr++;
      zptr = (fptr = VirtualPath) + sizeof(VirtualPath);
      while (*sptr && *sptr != ' ' && *sptr != '\"' && fptr < zptr)
      {
         if (*sptr == '\\') sptr++;
         *fptr++ = *sptr++;
      }
      *fptr = '\0';
      if (*sptr == '\"') sptr++;
      if (Debug) fprintf (stdout, "VirtualPath |%s|\n", VirtualPath);

      MapUrl_VirtualPath (rqptr->PathInfoPtr, VirtualPath,
                          VirtualFileName, sizeof(VirtualFileName));

      FileName[0] = '\0';
      zptr = MapUrl_Map (VirtualFileName, FileName, NULL, NULL, rqptr);
      if (!zptr[0])
      {
         SsiProblem (rqptr, zptr+1, FI_LI);
         return (sptr - String);
      }
   }
   else
   {
      while (*sptr && *sptr != '=') sptr++;
      if (*sptr) sptr++;
      if (*sptr == '\"') sptr++;
      zptr = (fptr = FileName) + 256;
      while (*sptr && *sptr != ' ' && *sptr != '\"' && fptr < zptr)
      {
         if (*sptr == '\\') sptr++;
         *fptr++ = *sptr++;
      }
      *fptr = '\0';
      if (*sptr == '\"') sptr++;
   }

   if (Debug) fprintf (stdout, "FileName |%s|\n", FileName);
   return (sptr - String);
}

/*****************************************************************************/
/*
Retrieve and display the specified file's specified attribute (size,
modification time, etc.)  This function completes synchronously, that is it
waits for the single QIO on the file details to complete before assembling the
details for the listing, because of this the function introduces slight
granularity.  It is a candidate for future improvement.

This function uses the ACP-QIO interface detailed in the "OpenVMS I/O User's 
Reference Manual".
*/ 

boolean SsiFileDetails
(
struct RequestStruct *rqptr,
void *AstFunctionPtr,
char *FileName,
int DisplayThis,
char *FormatPtr
)
{
   static $DESCRIPTOR (AbbrevOneByteFaoDsc, "!UL byte");
   static $DESCRIPTOR (AbbrevBytesFaoDsc, "!UL bytes");
   static $DESCRIPTOR (AbbrevOnekByteFaoDsc, "!UL kbyte");
   static $DESCRIPTOR (AbbrevkBytesFaoDsc, "!UL kbytes");
   static $DESCRIPTOR (AbbrevOneMByteFaoDsc, "!UL Mbyte");
   static $DESCRIPTOR (AbbrevMBytesFaoDsc, "!UL Mbytes");
   static $DESCRIPTOR (OneBlockFaoDsc, "!UL block");
   static $DESCRIPTOR (BlocksFaoDsc, "!UL blocks");
   static $DESCRIPTOR (BytesFaoDsc, "!AZ bytes");
   static $DESCRIPTOR (NumberFaoDsc, "!UL");

   register char  *cptr, *sptr;

   boolean  VmsUserHasAccess;

   int  status,
        FileNameLength,
        NumBytes;

   unsigned short  AcpChannel,
                   Length;
   unsigned long  AllocatedVbn,
                  Bytes,
                  EndOfFileVbn;
   unsigned long  AtrCdt [2],
                  AtrRdt [2];

   char  ExpandedFileName [256],
         Scratch [256],
         String [256],
         TimeString [256];

   $DESCRIPTOR (DeviceDsc, "");
   $DESCRIPTOR (StringDsc, String);
   $DESCRIPTOR (ScratchDsc, Scratch);


   struct FAB  FileFab;
   struct NAM  FileNam;

   struct {
      unsigned long  OfNoInterest1;
      unsigned short  AllocatedVbnHi;
      unsigned short  AllocatedVbnLo;
      unsigned short  EndOfFileVbnHi;
      unsigned short  EndOfFileVbnLo;
      unsigned short  FirstFreeByte;
      unsigned short  OfNoInterest2;
      unsigned long  OfNoInterestLots [4];
   } AtrRecord;

   struct fibdef  FileFib;
   struct atrdef  FileAtr [] =
   {
      { sizeof(AtrRecord), ATR$C_RECATTR, &AtrRecord },
      { sizeof(AtrCdt), ATR$C_CREDATE, &AtrCdt },
      { sizeof(AtrRdt), ATR$C_REVDATE, &AtrRdt },
      { 0, 0, 0 }
   };

   struct {
      unsigned short  Length;
      unsigned short  Unused;
      unsigned long  Address;
   } FileNameAcpDsc,
     FileFibAcpDsc,
     FileAtrAcpDsc;

   struct {
      unsigned short  Status;
      unsigned short  Unused1;
      unsigned long  Unused2;
   } AcpIOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiFileDetails() |%d|%s|\n", DisplayThis, FileName);

   FileNameLength = strlen(FileName);

   /* check if the user has access to this file! */
   if (rqptr->AuthVmsUserProfileLength)
   {
      status = AuthCheckVmsUserAccess (rqptr, FileName, FileNameLength);
      if (status == RMS$_PRV)
         VmsUserHasAccess = false;
      else
      if (VMSok (status))
         VmsUserHasAccess = true;
      else
      {
         /* error reported by access check */
         SsiEnd (rqptr);
         return;
      }
   }
   else
      VmsUserHasAccess = false;

   FileFab = cc$rms_fab;
   FileFab.fab$l_fna = FileName;
   FileFab.fab$b_fns = strlen(FileName);
   FileFab.fab$l_fop = FAB$M_NAM;
   FileFab.fab$l_nam = &FileNam;
   FileNam = cc$rms_nam;
   FileNam.nam$l_esa = ExpandedFileName;
   FileNam.nam$b_ess = sizeof(ExpandedFileName)-1;

   if (VmsUserHasAccess) EnableSysPrv ();

   status = sys$parse (&FileFab, 0, 0);

   if (VmsUserHasAccess) DisableSysPrv ();

   if (VMSok (status))
   {
      /* assign a channel to the disk device containing the file */
      DeviceDsc.dsc$w_length = FileNam.nam$b_dev;
      DeviceDsc.dsc$a_pointer = FileNam.nam$l_dev;
      status = sys$assign (&DeviceDsc, &AcpChannel, 0, 0, 0);
      if (Debug) fprintf (stdout, "sys$assign() %%X%08.08X\n", status);
   }

   if (VMSok (status))
   {
      /*******************************************/
      /* queue an ACP I/O to get file attributes */
      /*******************************************/

      /* set up the File Information Block for the ACP interface */
      memset (&FileFib, 0, sizeof(struct fibdef));
      FileFibAcpDsc.Length = sizeof(FileFib);
      FileFibAcpDsc.Address = &FileFib;
#ifdef __DECC
      memcpy (&FileFib.fib$w_did, &FileNam.nam$w_did, 6);
#else
      memcpy (&FileFib.fib$r_did_overlay.fib$w_did, &FileNam.nam$w_did, 6);
#endif

      FileNameAcpDsc.Length = FileNam.nam$b_name +
                              FileNam.nam$b_type +
                              FileNam.nam$b_ver;
      FileNameAcpDsc.Address = FileNam.nam$l_name;

      if (VmsUserHasAccess) EnableSysPrv ();

      status = sys$qiow (0, AcpChannel, IO$_ACCESS, &AcpIOsb, 0, 0, 
                         &FileFibAcpDsc, &FileNameAcpDsc, 0, 0,
                         &FileAtr, 0);

      if (VmsUserHasAccess) DisableSysPrv ();

      sys$dassgn (AcpChannel);

      if (Debug)
         fprintf (stdout, "sys$qio() %%X%08.08X IOsb: %%X%08.08X\n",
                 status, AcpIOsb.Status);

      if (VMSok (status)) status = AcpIOsb.Status;
   }

   /* ensure parse internal data structures are released */
   FileFab.fab$l_fna = "a:[b]c.d;";
   FileFab.fab$b_fns = 9;
   FileFab.fab$b_dns = 0;
   FileNam.nam$b_nop = NAM$M_SYNCHK;
   sys$parse (&FileFab, 0, 0);

   if (VMSnok (status)) 
   {
      SsiProblem (rqptr, SsiSysGetMsg(status), FI_LI);
      SsiEnd (rqptr);
      return;
   }

   if (DisplayThis == FILE_FCREATED)
   {
      /*********************/
      /* date/time created */
      /*********************/

      if (!SsiTimeString (rqptr, &AtrCdt, FormatPtr, String, sizeof(String)))
      {
         SsiEnd (rqptr);
         return;
      }
      Length = strlen(String);
   }
   else
   if (DisplayThis == FILE_FLASTMOD)
   {
      /*********************/
      /* date/time revised */
      /*********************/

      if (!SsiTimeString (rqptr, &AtrRdt, FormatPtr, String, sizeof(String)))
      {
         SsiEnd (rqptr);
         return;
      }
      Length = strlen(String);
   }
   else
   if (DisplayThis == FILE_FSIZE)
   {
      /*************/
      /* file size */
      /*************/

      AllocatedVbn = AtrRecord.AllocatedVbnLo +
                     (AtrRecord.AllocatedVbnHi << 16);
      EndOfFileVbn = AtrRecord.EndOfFileVbnLo +
                     (AtrRecord.EndOfFileVbnHi << 16);

      if (Debug)
         fprintf (stdout,
         "AllocatedVbn: %d EndOfFileVbn: %d FirstFreeByte %d\n",
         AllocatedVbn, EndOfFileVbn, AtrRecord.FirstFreeByte);

      if (EndOfFileVbn <= 1)
         Bytes = AtrRecord.FirstFreeByte;
      else
         Bytes = ((EndOfFileVbn-1) << 9) + AtrRecord.FirstFreeByte;
      if (Debug) fprintf (stdout, "Bytes %d\n", Bytes);

      if (!FormatPtr[0]) FormatPtr = rqptr->SsiTaskPtr->SizeFmtPtr;
      if (toupper(FormatPtr[0]) == 'A')  /* "abbrev" */
      {
         if (Bytes < 1024)
         {
            if (Bytes == 1)
               sys$fao (&AbbrevOneByteFaoDsc, &Length, &StringDsc, Bytes);
            else
               sys$fao (&AbbrevBytesFaoDsc, &Length, &StringDsc, Bytes);
         }
         else
         if (Bytes < 1048576)
         {
            if ((NumBytes = Bytes / 1024) == 1)
               sys$fao (&AbbrevOnekByteFaoDsc, &Length, &StringDsc, NumBytes);
            else
               sys$fao (&AbbrevkBytesFaoDsc, &Length, &StringDsc, NumBytes);
         }
         else
         {
            if ((NumBytes = Bytes / 1048576) == 1)
               sys$fao (&AbbrevOneMByteFaoDsc, &Length, &StringDsc, NumBytes);
            else
               sys$fao (&AbbrevMBytesFaoDsc, &Length, &StringDsc, NumBytes);
         }
         String[Length] = '\0';
      }
      else
      if (toupper(FormatPtr[0]) == 'B' &&
          toupper(FormatPtr[1]) == 'Y')  /* "bytes" */
      {
         sys$fao (&NumberFaoDsc, &Length, &ScratchDsc, Bytes);
         Scratch[Length] = '\0';
         sptr = String;
         cptr = Scratch;
         while (Length--)
         {
            *sptr++ = *cptr++;
            if (Length && !(Length % 3)) *sptr++ = ',';
         }
         *sptr = '\0';
         for (cptr = " bytes"; *cptr; *sptr++ = *cptr++);
         *sptr = '\0';
         Length = sptr - String;
      }
      else
      if (toupper(FormatPtr[0]) == 'B' &&
          toupper(FormatPtr[1]) == 'L')  /* "blocks" */
      {
         if (EndOfFileVbn == 1)
            sys$fao (&OneBlockFaoDsc, &Length, &StringDsc, EndOfFileVbn);
         else
            sys$fao (&BlocksFaoDsc, &Length, &StringDsc, EndOfFileVbn);
         String[Length] = '\0';
      }
   }

   SsiWriteBuffered (rqptr, AstFunctionPtr, String, Length);
   return (true);
}

/*****************************************************************************/
/*
Keep track of how many times an SSI file is accessed.  Do this by creating 
another file in the same directory, same name, extension modified by appending 
dollar symbol, containing a single longword record with the binary number of 
times the document has been accessed.  This can be reset by merely deleting 
the access count file.  Provide the access count and creation date of the 
access count file. 

This operation is fairly expensive in terms of I/O, and because it is atomic 
(all occurs during an AST routine, and opens-creates/writes/closes before 
returning to other processing) introduces a fair degree of granularity.  Its 
expense means the "#accesses" functionality should be used sparingly.  This 
functionality can be disabled.
*/ 

SsiAccessCount
(
struct RequestStruct *rqptr,
unsigned long *AccessCountPtr,
unsigned long *SinceBinTimePtr
)
{
   register char  *cptr, *sptr, *zptr;

   int  status,
        CreateStatus;
   char  FileName [256];
   struct FAB  FileFab;
   struct RAB  FileRab;
   struct XABDAT  FileXabDat;
   struct XABPRO  FileXabPro;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiAccessCount |%s|\n",
               rqptr->SsiTaskPtr->FileName);

   zptr = (sptr = FileName) + sizeof(FileName);
   for (cptr = rqptr->SsiTaskPtr->FileName;
        *cptr && *cptr != ';' && sptr < zptr;
        *sptr++ = *cptr++);
   if (sptr < zptr) *sptr++ = '$';
   if (sptr >= zptr)
   {
      SsiProblem (rqptr, MsgFor(rqptr,MSG_GENERAL_OVERFLOW), FI_LI);
      return (STS$K_ERROR);
   }
   *sptr = '\0';
   if (Debug) fprintf (stdout, "FileName: %s\n", FileName);

   FileFab = cc$rms_fab;
   FileFab.fab$l_fna = FileName;  
   FileFab.fab$b_fns = sptr-FileName;
   FileFab.fab$l_fop = FAB$M_CIF | FAB$M_SQO;
   FileFab.fab$b_fac = FAB$M_GET | FAB$M_PUT | FAB$M_UPD;
   FileFab.fab$w_mrs = sizeof(unsigned long);
   FileFab.fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT | FAB$M_SHRUPD;
   FileFab.fab$b_org = FAB$C_SEQ;
   FileFab.fab$b_rfm = FAB$C_FIX;

   /* initialize the date extended attribute block */
   FileFab.fab$l_xab = &FileXabDat;
   FileXabDat = cc$rms_xabdat;
   FileXabDat.xab$l_nxt = &FileXabPro;

   /* initialize the protection extended attribute block */
   FileXabPro = cc$rms_xabpro;
   FileXabPro.xab$w_pro = 0xaa00;  /* W:RE,G:RE,O:RWED,S:RWED */

   /* "temporary" file, automatic delete on closing it */
   if (rqptr->DeleteOnClose) FileFab.fab$l_fop |= FAB$M_DLT;

   /* turn on SYSPRV to allow access to the counter file */
   EnableSysPrv ();
   status = CreateStatus = sys$create (&FileFab, 0, 0);
   DisableSysPrv ();

   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "sys$create() %%X%08.08X\n", status);
      SsiProblem (rqptr, SsiSysGetMsg(status), FI_LI);
      return (status);
   }

   FileRab = cc$rms_rab;
   FileRab.rab$l_fab = &FileFab;
   FileRab.rab$l_rbf = AccessCountPtr;
   FileRab.rab$w_rsz = sizeof(*AccessCountPtr);

   if (VMSnok (status = sys$connect (&FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      SsiProblem (rqptr, SsiSysGetMsg(status), FI_LI);
   }

   /* copy the creation date (since date) of the access count file */
   memcpy (SinceBinTimePtr, &FileXabDat.xab$q_cdt, 8);

   if (VMSok (status))
   {
      if (CreateStatus == RMS$_CREATED)
      {
         /***************************************/
         /* count file did not previously exist */
         /***************************************/
   
         *AccessCountPtr = 1;
         if (Debug) fprintf (stdout, "*AccessCountPtr: %d\n", *AccessCountPtr);

         if (VMSnok (status = sys$put (&FileRab, 0, 0)))
         {
            if (Debug) fprintf (stdout, "sys$put() %%X%08.08X\n", status);
            SsiProblem (rqptr, SsiSysGetMsg(status), FI_LI);
         }
      }
      else
      {
         /**********************/
         /* count file existed */
         /**********************/

         FileRab.rab$l_ubf = AccessCountPtr;
         FileRab.rab$w_usz = sizeof(*AccessCountPtr);
         if (VMSnok (status = sys$get (&FileRab, 0, 0))) 
         {
            if (Debug) fprintf (stdout, "sys$get() %%X%08.08X\n", status);
            SsiProblem (rqptr, SsiSysGetMsg(status), FI_LI);
         }
         else
         {
            *AccessCountPtr += 1;
            if (Debug)
               fprintf (stdout, "*AccessCountPtr: %d\n", *AccessCountPtr);
         }

         if (VMSnok (status = sys$update (&FileRab, 0, 0)))
         {
            if (Debug) fprintf (stdout, "sys$update() %%X%08.08X\n", status);
            SsiProblem (rqptr, SsiSysGetMsg(status), FI_LI);
         }
      }
   }

   if (rqptr->DeleteOnClose)
   {
      /* use SYSPRV to ensure deletion-on-close of file */
      EnableSysPrv ();
      sys$close (&FileFab, 0, 0);
      DisableSysPrv ();
   }
   else
      sys$close (&FileFab, 0, 0);

   return (status);
}

/*****************************************************************************/
/*
Get just the text of the VMS message.
*/
 
char* SsiSysGetMsg (int StatusValue)
 
{
   static char  Message [256];
   short int  Length;
   $DESCRIPTOR (MessageDsc, Message);
 
   sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0);
   Message[Length] = '\0';
   if (Debug) fprintf (stdout, "SysGetMsg() |%s|\n", Message);
   return (Message);
}
 
/*****************************************************************************/
/*
Generate a general error, with explanation about the pre-processor error.
*/ 

int SsiProblem
(
struct RequestStruct *rqptr,
char *Explanation,
char *SourceFileName,
int SourceLineNumber
)
{
   static $DESCRIPTOR (ErrorMessageFaoDsc,
"<H1><FONT COLOR=\"#ff0000\"><U>!AZ</U></FONT></H1>\n\
<!!-- module: !AZ line: !UL -->\n\
!AZ &nbsp;(!AZ !UL)!AZ!AZ!AZ\n");

   register unsigned long  *vecptr;
   register struct SsiTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  String [1024];
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SsiProblem()\n");

   tkptr = rqptr->SsiTaskPtr;

   tkptr->SsiProblemEncountered = true;

   vecptr = FaoVector;
   if (tkptr->ErrMsgPtr != NULL && tkptr->ErrMsgPtr[0])
      *vecptr++ = tkptr->ErrMsgPtr;
   else
      *vecptr++ = MsgFor(rqptr,MSG_SSI_ERROR);
   *vecptr++ = "SSI";
   *vecptr++ = SourceLineNumber;
   *vecptr++ = Explanation;
   *vecptr++ = MsgFor(rqptr,MSG_SSI_LINE);
   *vecptr++ = tkptr->StatementLineNumber;
   if (tkptr->Statement[0])
   {
      *vecptr++ = " &nbsp;...&nbsp; \\<TT>";
      *vecptr++ = tkptr->Statement;
      *vecptr++ = "</TT>\\\n";
   }
   else
   {
      *vecptr++ = "";
      *vecptr++ = "";
      *vecptr++ = "";
   }

   status = sys$faol (&ErrorMessageFaoDsc, &Length, &StringDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$fao()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SsiEnd (rqptr);
      return;
   }
   String[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);

   /* error reports get through, come-what-may! */
   NetWriteBuffered (rqptr, NULL, String, Length);
}

/*****************************************************************************/
/*
Just wrap the network write function so that output can be suppressed when flow
control is skipping content.
*/ 

int SsiWriteBuffered
(
struct RequestStruct *rqptr,
void  *AstFunctionPtr,
char *DataPtr,
int DataLength
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SsiWriteBuffered() [%d]=%d\n",
               rqptr->SsiTaskPtr->FlowControlIndex,
               rqptr->SsiTaskPtr->FlowControlState
                  [rqptr->SsiTaskPtr->FlowControlIndex]);

   /* output unless flow control dictates otherwise */
   if (rqptr->SsiTaskPtr->FlowControlIsExecuting
          [rqptr->SsiTaskPtr->FlowControlIndex])
   {
      NetWriteBuffered (rqptr, AstFunctionPtr, DataPtr, DataLength);
      if (DataLength)
         if (DataPtr[DataLength-1] == '\n')
            rqptr->SsiTaskPtr->PageHasBeenOutput = true;
   }
   else
      SysDclAst (AstFunctionPtr, rqptr);
}

/*****************************************************************************/

