/*****************************************************************************/
/*
                                  StmLf.c

This module implements a full multi-threaded, AST-driven, asynchronous file 
conversion from variable to stream-LF record format, by copying it into a
next-highest version.  When called it allocates a dynamic structure and creates
its own I/O-event-driven thread.  Errors are reported to the HTTPd process log,
and do not affect any other activities (including request processing) within
the server.  Will purge the pre-existing version, however if multiple versions
already existed only deletes the one copied from!

Why this module's functionality?  The WASD VMS HTTPd is significantly more
efficient in transfering stream-format than variable-format files.  The reason?
Stream files can be read using block I/O, variable must be processed
record-by-record and buffered into a larger block before network I/O, very much
more expensive.  This module allows detected variable-format files to be
converted to stream-LF "on-the-fly".  This happens in a thread separate to the
detecting and initiating request, so the original variable-format file is used
to concurrently supply that particular request, with the converted file
probably available by the time the next request for it occurs.  In addition,
the WASD HTTPd can determine the "Content-Length:" of stream-LF files and hence
can not only supply this item to the client but as a consequence participate
in a persistent connection request.


VERSION HISTORY
---------------
29-AUG-98  MGD  confine conversions to specified paths
17-AUG-97  MGD  SYSUAF-authenticated users security-profile
01-AUG-96  MGD  initial, v3.3
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>

/* VMS related header files */
#include <descrip.h>
#include <rms.h>
#include <rmsdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application header files */
#include "wasd.h"
#include "httpd.h"
#include "stmlf.h"

/**
#define STMLF_NOPURGE 1
**/

#define MaxPurgeAttempts 30

/*************/
/* structure */
/*************/

struct StreamLfStruct
{
   boolean  ConversionOk,
            DstFileCreated;
   int  PurgeAttemptCount;
   char  Buffer [4096],
         DstExpandedFileName [256],
         DstRelatedFileName [256],
         FileName [256],
         SrcExpandedFileName [256];
   struct FAB  SrcFileFab;
   struct NAM  SrcFileNam;
   struct RAB  SrcFileRab;
   struct FAB  DstFileFab;
   struct NAM  DstFileNam;
   struct RAB  DstFileRab;
};

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern char  Utility[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;

/*****************************************************************************/
/*
*/ 

int StmLfBegin
(
char *FilePath,
char *FileName,
int FileNameLength,
boolean VmsUserHasAccess
)
{
   register char  *cptr, *sptr;

   int  status;
   struct StreamLfStruct  *StmPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "StmLfBegin() |%s|%s|\n", FilePath, FileName);

   if ((sptr = Config.StreamLfPathsPtr) == NULL) return;
   while (*sptr)
   {
      if (Debug) fprintf (stdout, "sptr |%s|\n", sptr);
      cptr = FilePath;
      while (*cptr)
      {
         if (*sptr == ',') break;
         if (*sptr == '*')
         {
            while (*sptr && *sptr == '*' && *sptr != ',') sptr++;
            while (*cptr && *cptr != *sptr) cptr++;
         }
         if (tolower(*cptr) != tolower(*sptr)) break;
         if (*cptr) cptr++;
         if (*sptr) sptr++;
      }
      if (!*cptr && (!*sptr || *sptr == ',')) break;
      while (*sptr && *sptr != ',') sptr++;
      if (*sptr) sptr++;
   }
   /* if it was NOT found then return without conversion */
   if (!(!*cptr && (!*sptr || *sptr == ','))) return;

   /* allocate dynamic memory for the conversion thread */
   StmPtr = (struct StreamLfStruct*) VmGet (sizeof(struct StreamLfStruct));

   /* reset this boolean so that anything but source EOF is not success */
   StmPtr->ConversionOk = StmPtr->DstFileCreated = false;
   /* copy the filename into thread-persistent storage */
   memcpy (StmPtr->FileName, FileName, FileNameLength+1);
   /* set a limit on the number of attempts to purge the source file */
   StmPtr->PurgeAttemptCount = MaxPurgeAttempts;

   StmPtr->SrcFileFab = cc$rms_fab;
   StmPtr->SrcFileFab.fab$l_fna = StmPtr->FileName;  
   StmPtr->SrcFileFab.fab$b_fns = FileNameLength;
   StmPtr->SrcFileFab.fab$b_fac = FAB$M_GET;
   StmPtr->SrcFileFab.fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT;

   StmPtr->SrcFileFab.fab$l_nam = &StmPtr->SrcFileNam;  
   StmPtr->SrcFileNam = cc$rms_nam;
   StmPtr->SrcFileNam.nam$l_esa = StmPtr->SrcExpandedFileName;  
   StmPtr->SrcFileNam.nam$b_ess = sizeof(StmPtr->SrcExpandedFileName)-1;

   if (VmsUserHasAccess) EnableSysPrv ();

   status = sys$open (&StmPtr->SrcFileFab, 0, 0);
   if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);

   if (VmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      StmLfLog (StmPtr, "sys$open()", status);
      StmLfEnd (StmPtr);
      return;
   }

   StmPtr->SrcFileNam.nam$l_ver[StmPtr->SrcFileNam.nam$b_ver] = '\0';

   StmPtr->SrcFileRab = cc$rms_rab;
   StmPtr->SrcFileRab.rab$l_fab = &StmPtr->SrcFileFab;
   StmPtr->SrcFileRab.rab$b_mbf = 2;
   StmPtr->SrcFileRab.rab$l_rop = RAB$M_RAH | RAB$M_ASY;
   StmPtr->SrcFileRab.rab$l_ubf = StmPtr->Buffer;
   StmPtr->SrcFileRab.rab$w_usz = sizeof(StmPtr->Buffer);

   /* set the RAB context to contain the conversion thread pointer */
   StmPtr->SrcFileRab.rab$l_ctx = StmPtr;

   status = sys$connect (&StmPtr->SrcFileRab, 0, 0);
   if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
   if (VMSnok (status))
   {
      StmLfLog (StmPtr, "sys$connect() src", status);
      StmLfEnd (StmPtr);
      return;
   }

   StmPtr->DstFileFab = cc$rms_fab;
   StmPtr->DstFileFab.fab$l_fna = StmPtr->FileName;  
   for (cptr = StmPtr->FileName; *cptr && *cptr != ';'; cptr++);
   StmPtr->DstFileFab.fab$b_fns = cptr - StmPtr->FileName;
   StmPtr->DstFileFab.fab$l_fop = FAB$M_SQO;
   StmPtr->DstFileFab.fab$b_rfm = FAB$C_STMLF;

   StmPtr->DstFileFab.fab$l_nam = &StmPtr->DstFileNam;  
   StmPtr->DstFileNam = cc$rms_nam;
   StmPtr->DstFileNam.nam$l_esa = StmPtr->DstExpandedFileName;  
   StmPtr->DstFileNam.nam$b_ess = sizeof(StmPtr->DstExpandedFileName)-1;
   StmPtr->DstFileNam.nam$l_rsa = StmPtr->DstRelatedFileName;  
   StmPtr->DstFileNam.nam$b_rss = sizeof(StmPtr->DstRelatedFileName)-1;

   /* use SYSPRV to ensure creation of file (provided protection is S:RWED) */
   EnableSysPrv ();

   status = sys$create (&StmPtr->DstFileFab, 0, 0);
   if (Debug) fprintf (stdout, "sys$create() %%X%08.08X\n", status);

   DisableSysPrv ();

   /* check the status of the file create */
   if (VMSnok (status))
   {
      StmLfLog (StmPtr, "sys$create()", status);
      StmLfEnd (StmPtr);
      return;
   }

   StmPtr->DstFileCreated = true;

   StmPtr->DstFileNam.nam$l_ver[StmPtr->DstFileNam.nam$b_ver] = '\0'; 
   if (Debug) fprintf (stdout, "|%s|\n", StmPtr->DstFileNam.nam$l_dev);

   StmPtr->DstFileRab = cc$rms_rab;
   StmPtr->DstFileRab.rab$l_fab = &StmPtr->DstFileFab;
   StmPtr->DstFileRab.rab$b_mbf = 2;
   StmPtr->DstFileRab.rab$l_rop = RAB$M_WBH | RAB$M_ASY;
   StmPtr->DstFileRab.rab$l_rbf = StmPtr->Buffer;

   /* set the RAB context to contain the conversion thread pointer */
   StmPtr->DstFileRab.rab$l_ctx = StmPtr;

   status = sys$connect (&StmPtr->DstFileRab, 0, 0);
   if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
   if (VMSnok (status))
   {
      StmLfLog (StmPtr, "sys$connect() dst", status);
      StmLfEnd (StmPtr);
      return;
   }

   /* get the first reacord */
   status = sys$get (&StmPtr->SrcFileRab,
                     &StmLfNextRecordAst, &StmLfNextRecordAst);
   if (Debug) fprintf (stdout, "sys$get() %%X%08.08X\n", status);
}

/****************************************************************************/
/*
Close source and destination files.  If successfully converted attempt to
purge the version it was copied from.  If the file is still locked due to
a still continuing transfer to a (the) client then set a timer and try again
in one second.  Do this a number of times before giving up!  If not
successfully copied then delete any new version created in the attempt.
*/

StmLfEnd (struct StreamLfStruct *StmPtr)

{
   /* one second between purge attempts */
   static unsigned long  PurgeDeltaBinTime [2] = { -10000000, -1 };

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "StmLfEnd()\n");

   if (StmPtr->SrcFileFab.fab$w_ifi) sys$close (&StmPtr->SrcFileFab, 0, 0);

   if (StmPtr->ConversionOk)
   {
      Accounting.StreamLfConversionCount++;

      if (StmPtr->DstFileFab.fab$w_ifi) sys$close (&StmPtr->DstFileFab, 0, 0);

#ifndef STMLF_NOPURGE
      StmPtr->SrcFileFab.fab$l_fop = FAB$M_NAM;

      /* use SYSPRV to ensure erasure of file */
      EnableSysPrv ();

      status = sys$erase (&StmPtr->SrcFileFab, 0, 0);
      if (Debug) fprintf (stdout, "SRC sys$erase() %%X%08.08X\n", status);

      DisableSysPrv ();

      if (VMSnok (status))
      {
         if (status == RMS$_FLK)
         {
            if (StmPtr->PurgeAttemptCount--)
            {
               status = sys$setimr (0, &PurgeDeltaBinTime,
                                    &StmLfEnd, StmPtr, 0);
               if (VMSok (status)) return;
               StmLfLog (StmPtr, "sys$setimr()", status);
            }
            else
               StmLfLog (StmPtr, "sys$erase()", status);
         }
         else
            StmLfLog (StmPtr, "sys$erase() src", status);
      }
#endif

      StmLfLog (StmPtr, "converted", 0);
   }
   else
   if (StmPtr->DstFileCreated)
   {
      sys$close (&StmPtr->DstFileFab, 0, 0);
      StmPtr->DstFileFab.fab$l_fop = FAB$M_NAM;

      /* use SYSPRV to ensure erasure of file */
      EnableSysPrv ();

      status = sys$erase (&StmPtr->DstFileFab, 0, 0);
      if (Debug) fprintf (stdout, "DST sys$erase() %%X%08.08X\n", status);

      DisableSysPrv ();

      if (VMSnok (status)) StmLfLog (StmPtr, "sys$erase() dst", status);
   }

   VmFree (StmPtr);
}

/****************************************************************************/
/*
*/

StmLfNextRecord (struct RAB *RabPtr)

{
   int  status;
   struct StreamLfStruct  *StmPtr;

   /*********/
   /* begin */
   /*********/

/**
   if (Debug)
      fprintf (stdout,
         "StmLfNextRecord() sts: %%X%08.08X stv: %%X%08.08X\n",
         RabPtr->rab$l_sts,
         RabPtr->rab$l_stv);
**/

   StmPtr = RabPtr->rab$l_ctx;

   if (VMSnok (StmPtr->DstFileRab.rab$l_sts))
   {
      /** StmLfLog (StmPtr, "sys$put()", StmPtr->DstFileRab.rab$l_sts); **/
      StmLfEnd (StmPtr);
      return;
   }

   status = sys$get (&StmPtr->SrcFileRab,
                     &StmLfNextRecordAst, &StmLfNextRecordAst);
   /** if (Debug) fprintf (stdout, "sys$get() %%X%08.08X\n", status); **/
}

/****************************************************************************/
/*
*/

StmLfNextRecordAst (struct RAB *RabPtr)

{
   int  status;
   struct StreamLfStruct  *StmPtr;

   /*********/
   /* begin */
   /*********/

/**
   if (Debug)
      fprintf (stdout,
         "StmLfNextRecordAST() sts: %%X%08.08X stv: %%X%08.08X rsz: %d\n",
         RabPtr->rab$l_sts,
         RabPtr->rab$l_stv,
         RabPtr->rab$w_rsz);
**/

   StmPtr = RabPtr->rab$l_ctx;

   if (VMSnok (StmPtr->SrcFileRab.rab$l_sts))
   {
      if (StmPtr->SrcFileRab.rab$l_sts == RMS$_EOF)
      {
         if (Debug) fprintf (stdout, "RMS$_EOF\n");
         /* only place this boolean gets set */
         StmPtr->ConversionOk = true;
         StmLfEnd (StmPtr);
         return;
      }
      StmLfLog (StmPtr, "sys$get()", StmPtr->SrcFileRab.rab$l_sts);
      StmLfEnd (StmPtr);
      return;
   }

   StmPtr->DstFileRab.rab$l_rbf = StmPtr->SrcFileRab.rab$l_ubf;
   StmPtr->DstFileRab.rab$w_rsz = StmPtr->SrcFileRab.rab$w_rsz;
   status = sys$put (&StmPtr->DstFileRab,
                     &StmLfNextRecord, &StmLfNextRecord);
   /** if (Debug) fprintf (stdout, "sys$put() %%X%08.08X\n", status); **/
}

/****************************************************************************/
/*
*/

StmLfLog
(
struct StreamLfStruct *StmPtr,
char *Explanation,
int StatusValue
)
{
   int  status;
   char  *FileNamePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "StmLfLog()\n");

   if (StmPtr == NULL)
      FileNamePtr = "";
   else
      FileNamePtr = StmPtr->FileName;

   if (StatusValue)
      fprintf (stdout, "%%%s-E-STREAMLF, %%X%08.08X %s %s\n",
               Utility, StatusValue, Explanation, FileNamePtr);
   else
      fprintf (stdout, "%%%s-I-STREAMLF, %s %s\n",
               Utility, Explanation, FileNamePtr);
}

/****************************************************************************/

