/*****************************************************************************
/*
                                 Support.c

Miscellaneous support functions for HTTPd.


VERSION HISTORY
---------------
10-AUG-98  MGD  refined CopyToHtml() slightly,
                bugfix; TimeSetTimezone() unsigned to signed longs
10-JUL-98  MGD  a little Y2K insurance
21-JUN-98  MGD  changed format of request logical
02-APR-98  MGD  after email about non-conforming date formats back to RFC1123
12-MAR-98  MGD  added FormatProtection()
08-JAN-97  MGD  TimeSetGmt() now can use SYS$TIMEZONE_DIFFERENTIAL
05-OCT-97  MGD  additional list processing functions
28-SEP-97  MGD  accounting structure has grown beyond 255 bytes
09-AUG-97  MGD  message database, added SearchTextString(),
                NameOfDirectoryFile()
12-JUL-97  MGD  EnableSysPrv(), DisableSysPrv() to rationalize this requirement
07-JUL-97  MGD  attempt to reduce dynamic memory fragmentation within C RTL
                using HEAP_MIN_CHUNK_SIZE functionality,
                prevent request logical redefinition if keep-alive timeout 
16-JUN-97  MGD  generic linked list functions
12-MAR-97  MGD  HTTP header generation
01-FEB-97  MGD  HTTPd version 4
01-OCT-96  MGD  report menu
18-JUN-96  MGD  bugfix; HTTPD$GMT conversion to VMS delta time in TimeSetGmt()
06-APR-96  MGD  persistent connections ("keep-alive")
01-DEC-95  MGD  HTTPd version 3.0
27-SEP-95  MGD  provide GMT functions
20-DEC-94  MGD  initial development
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <time.h>

/* VMS related header files */
#include <descrip.h>
#include <jpidef.h>
#include <libdef.h>
#include <libdtdef.h>
#include <lnmdef.h>
#include <prvdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application related header files */
#include "wasd.h"
#include "error.h"
#include "httpd.h"
#include "msg.h"
#include "net.h"
#include "support.h"

/******************/
/* global storage */
/******************/

#define RFC_1123_DATE yup

int  HeapOverhead = sizeof(struct HeapStruct*) +
                    sizeof(struct HeapStruct*) +
                    sizeof(unsigned int);

boolean TimeAheadOfGmt;
char  TimeGmtString [48],
      TimeGmtVmsString [50];
unsigned long  TimeGmtDeltaBinary [2];

char  HttpProtocol [] = "HTTP/1.0";

char  HtmlSgmlDoctype [] =
      "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2//EN\">\n";

char  *DayName [] =
   { "", "Monday", "Tuesday", "Wednesday",
     "Thursday", "Friday", "Saturday", "Sunday" };

char  *MonthName [] =
   { "", "January", "February", "March", "April", "May", "June",
         "July", "August", "September", "October", "November", "December" };

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern boolean  AccountingZeroOnStartup;
extern boolean  MonitorEnabled;
extern int ServerPort;
extern char  SoftwareID[];
extern struct AccountingStruct Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;
extern struct ServiceStruct  *ServiceListHead;

/*****************************************************************************/
/*
Declare an AST, exit if there is any problem ... must have out ASTs!
*/ 

void SysDclAst
(
void *Address,
unsigned long Parameter
)
{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SysDclAst()\n");

   if (VMSok (status = sys$dclast (Address,Parameter, 0))) return;
   ErrorExitVmsStatus (status, "sys$dclast()", __FILE__, __LINE__);
}

/*****************************************************************************/
/*
Generic list handling function.  Add entry to head of list.
*/ 

void* ListAddHead
(
register struct ListHeadStruct *lptr,
register struct ListEntryStruct *eptr
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout, "ListAddHead() %d %d\n", lptr, eptr);
      ListDebug (lptr);
   }

   if (lptr->HeadPtr == NULL)
   {
      /* empty list */
      lptr->HeadPtr = lptr->TailPtr = eptr;
      eptr->PrevPtr = eptr->NextPtr = NULL;
      lptr->EntryCount++;
      if (Debug) ListDebug (lptr);
      return (eptr);
   }
   else
   {
      /* non-empty list */
      eptr->PrevPtr = NULL;
      eptr->NextPtr = lptr->HeadPtr;
      eptr->NextPtr->PrevPtr = lptr->HeadPtr = eptr;
      lptr->EntryCount++;
      if (Debug) ListDebug (lptr);
      return (eptr);
   }
}

/*****************************************************************************/
/*
Generic list handling function.  Add entry to tail of list.
*/ 

void* ListAddTail
(
register struct ListHeadStruct *lptr,
register struct ListEntryStruct *eptr
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout, "ListAddTail() %d %d\n", lptr, eptr);
      ListDebug (lptr);
   }

   if (lptr->HeadPtr == NULL)
   {
      /* empty list */
      lptr->HeadPtr = lptr->TailPtr = eptr;
      eptr->PrevPtr = eptr->NextPtr = NULL;
      lptr->EntryCount++;
      if (Debug) ListDebug (lptr);
      return (eptr);
   }
   else
   {
      /* non-empty list */
      eptr->NextPtr = NULL;
      eptr->PrevPtr = lptr->TailPtr;
      eptr->PrevPtr->NextPtr = lptr->TailPtr = eptr;
      lptr->EntryCount++;
      if (Debug) ListDebug (lptr);
      return (eptr);
   }
}

/*****************************************************************************/
/*
Generic list handling function.  Add entry 2 "in front of" entry 1.
*/ 

void* ListAddBefore
(
register struct ListHeadStruct *lptr,
register struct ListEntryStruct *e1ptr,
register struct ListEntryStruct *e2ptr
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout, "ListAddBefore() %d %d\n", lptr, e1ptr, e2ptr);
      ListDebug (lptr);
   }

   if (lptr->HeadPtr == e1ptr)
   {
      /* at head of list */
      e1ptr->PrevPtr = e2ptr;
      e2ptr->PrevPtr = NULL;
      e2ptr->NextPtr = e1ptr;
      lptr->HeadPtr = e2ptr;
      return (e2ptr);
   }
   else
   {
      /* not at head of list */
      if (e1ptr->PrevPtr != NULL)
      {
         e2ptr->PrevPtr = e1ptr->PrevPtr;
         e2ptr->PrevPtr->NextPtr = e2ptr;
      }
      e1ptr->PrevPtr = e2ptr;
      e2ptr->NextPtr = e1ptr;
      return (e2ptr);
   }

   if (Debug) ListDebug (lptr);
}

/*****************************************************************************/
/*
Generic list handling function.  Remove entry from list.  Check first that
entry looks legitimate, return NULL if not!
*/ 

void* ListRemove
(
register struct ListHeadStruct *lptr,
register struct ListEntryStruct *eptr
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout, "ListRemove() %d %d\n", lptr, eptr);
      ListDebug (lptr);
   }

   if (eptr->PrevPtr == NULL)
   {                                                                        
      /* at head of list */
      if (eptr->NextPtr == NULL)
      {
         /* only entry in list */
         if (lptr->HeadPtr != eptr || lptr->TailPtr != eptr) return (NULL);
         lptr->HeadPtr = lptr->TailPtr = NULL;
         if (lptr->EntryCount) lptr->EntryCount--;
         if (Debug) ListDebug (lptr);
         return (eptr);
      }
      else
      {
         /* remove from head of list */
         if (lptr->HeadPtr != eptr) return (NULL);
         eptr->NextPtr->PrevPtr = NULL;
         lptr->HeadPtr = eptr->NextPtr;
         if (lptr->EntryCount) lptr->EntryCount--;
         if (Debug) ListDebug (lptr);
         return (eptr);
      }
   }
   else
   {
      /* not at head of list */
      if (eptr->NextPtr == NULL)
      {
         /* at tail of list */
         if (lptr->TailPtr != eptr) return (NULL);
         eptr->PrevPtr->NextPtr = NULL;
         lptr->TailPtr = eptr->PrevPtr;
         if (lptr->EntryCount) lptr->EntryCount--;
         if (Debug) ListDebug (lptr);
         return (eptr);
      }
      else
      {
         /* somewhere in the middle! */
         if (eptr->PrevPtr->NextPtr != eptr ||
             eptr->NextPtr->PrevPtr != eptr) return (NULL);
         eptr->PrevPtr->NextPtr = eptr->NextPtr;
         eptr->NextPtr->PrevPtr = eptr->PrevPtr;
         if (lptr->EntryCount) lptr->EntryCount--;
         if (Debug) ListDebug (lptr);
         return (eptr);
      }
   }
}

/*****************************************************************************/
/*
Generic list handling function.  If not already there move it to the head.
*/ 

#ifdef __DECC
#pragma inline(ListMoveHead)
#endif

void* ListMoveHead
(
struct ListHeadStruct *lptr,
struct ListEntryStruct *eptr
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ListMoveHead() %d %d %d\n",
               lptr, lptr->HeadPtr, eptr);

   /* move entry to the head of the cache list */
   if (lptr->HeadPtr != eptr)
   {
      ListRemove (lptr, eptr);
      ListAddHead (lptr, eptr);
   }

   return (eptr);
}

/*****************************************************************************/
/*
Generic list handling function.  If not already there move it to the tail.
*/ 

#ifdef __DECC
#pragma inline(ListMoveTail)
#endif

void* ListMoveTail
(
struct ListHeadStruct *lptr,
struct ListEntryStruct *eptr
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ListMoveTail() %d %d %d\n",
               lptr, lptr->HeadPtr, eptr);

   /* move entry to the head of the cache list */
   if (lptr->TailPtr != eptr)
   {
      ListRemove (lptr, eptr);
      ListAddTail (lptr, eptr);
   }

   return (eptr);
}

/*****************************************************************************/
/*
Generic list handling function.  Check if a specific entry is in specific list.
*/ 

void* ListCheck
(
register struct ListHeadStruct *lptr,
register struct ListEntryStruct *eptr
)
{
   register struct ListEntryStruct  *teptr;

   /*********/
   /* begin */
   /*********/

   fprintf (stdout, "ListCheck() %d %d %d %d\n",
            lptr, lptr->HeadPtr, lptr->TailPtr, lptr->EntryCount);

   for (teptr = lptr->HeadPtr; teptr != NULL; teptr = teptr->NextPtr)
      if (teptr == eptr) return (eptr);
   return (NULL);
}

/*****************************************************************************/
/*
list the entries in the list.
*/ 

ListDebug (register struct ListHeadStruct* lptr)

{
   register struct ListEntryStruct  *eptr;

   /*********/
   /* begin */
   /*********/

   fprintf (stdout, "ListDebug() %d %d %d %d\n",
            lptr, lptr->HeadPtr, lptr->TailPtr, lptr->EntryCount);

   for (eptr = lptr->HeadPtr; eptr != NULL; eptr = eptr->NextPtr)
      fprintf (stdout, "%d <- %d -> %d\n", eptr->PrevPtr, eptr, eptr->NextPtr);
}

/*****************************************************************************/
/*
Create an HTTP format local time string in the storage pointed at by
'TimeString', e.g. "Fri, 25 Aug 1995 17:32:40" (RFC1123). If 'BinTimePtr' is
null the time string represents the current time. If it points to a quadword,
VMS time value the string represents that time. 'TimeString' must point to
storage large enough for 31 characters.
*/

int HttpLocalTimeString
(
char *TimeString,
unsigned long *BinTimePtr
)
{
   /* alternate formats for time string */
#ifdef RFC_1123_DATE
   /* day, dd mmm yyyy hh:mm */
   static $DESCRIPTOR (HttpTimeFaoDsc, "!3AZ, !2ZW !3AZ !4ZW !2ZW:!2ZW:!2ZW");
#else
   /* weekday, dd-mmm-yyyy hh:mm */
   static $DESCRIPTOR (HttpTimeFaoDsc, "!3AZ, !2ZW-!3AZ-!4ZW !2ZW:!2ZW:!2ZW");
#endif

   static $DESCRIPTOR (TimeStringDsc, "");

   int  status;
   unsigned long  BinTime [2];
   unsigned short  Length;
   unsigned short  NumTime [7];
   unsigned long  DayOfWeek;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpLocalTimeString()\n");

   if (BinTimePtr == NULL)
      sys$gettim (&BinTime);
   else
   {
      BinTime[0] = BinTimePtr[0];
      BinTime[1] = BinTimePtr[1];
   }

   status = sys$numtim (&NumTime, &BinTime);
   if (Debug)
      fprintf (stdout, "sys$numtim() %%X%08.08X %d %d %d %d %d %d %d\n",
               status, NumTime[0], NumTime[1], NumTime[2],
               NumTime[3], NumTime[4], NumTime[5], NumTime[6]);

   if (VMSnok (status = lib$day_of_week (&BinTime, &DayOfWeek)))
      return (status);
   if (Debug)
      fprintf (stdout, "lib$day_of_week() %%X%08.08X is %d\n",
               status, DayOfWeek);

   /* set the descriptor address and size of the resultant time string */
   TimeStringDsc.dsc$w_length = 25;
   TimeStringDsc.dsc$a_pointer = TimeString;

   status = sys$fao (&HttpTimeFaoDsc, &Length, &TimeStringDsc,
                     DayName[DayOfWeek], NumTime[2], MonthName[NumTime[1]],
                     NumTime[0], NumTime[3], NumTime[4], NumTime[5]);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
      TimeString[0] = '\0';
   }
   else
      TimeString[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", TimeString);

   return (status);
}

/*****************************************************************************/
/*
Create an HTTP format Greenwich Mean Time (UTC) time string in the storage 
pointed at by 'TimeString', e.g. "Fri, 25 Aug 1995 17:32:40 GMT" (RFC1123). 
This must be at least 30 characters capacity.  If 'BinTimePtr' is null the 
time string represents the current time.  If it points to a quadword, VMS time 
value the string represents that time.  'TimeString' must point to storage 
large enough for 31 characters.
*/

int HttpGmTimeString
(
char *TimeString,
unsigned long *BinTimePtr
)
{
   /* alternate formats for time string */
#ifdef RFC_1123_DATE
   /* day, dd mmm yyyy hh:mm */
   static $DESCRIPTOR (HttpTimeFaoDsc,
          "!3AZ, !2ZW !3AZ !4ZW !2ZW:!2ZW:!2ZW GMT");
#else
   /* weekday, dd-mmm-yyyy hh:mm */
   static $DESCRIPTOR (HttpTimeFaoDsc,
          "!3AZ, !2ZW-!3AZ-!4ZW !2ZW:!2ZW:!2ZW GMT");
#endif

   static $DESCRIPTOR (TimeStringDsc, "");

   int  status;
   unsigned long  BinTime [2],
                  GmTime [2];
   unsigned short  Length;
   unsigned short  NumTime [7];
   unsigned long  DayOfWeek;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpGmTimeString()\n");

   if (BinTimePtr == NULL)
      sys$gettim (&GmTime);
   else
   {
      GmTime[0] = BinTimePtr[0];
      GmTime[1] = BinTimePtr[1];
   }

   if (VMSnok (status = TimeAdjustGMT (true, &GmTime)))
      return (status);

   status = sys$numtim (&NumTime, &GmTime);
   if (Debug)
      fprintf (stdout, "sys$numtim() %%X%08.08X %d %d %d %d %d %d %d\n",
               status, NumTime[0], NumTime[1], NumTime[2],
               NumTime[3], NumTime[4], NumTime[5], NumTime[6]);

   if (VMSnok (status = lib$day_of_week (&GmTime, &DayOfWeek)))
      return (status);
   if (Debug)
      fprintf (stdout, "lib$day_of_week() %%X%08.08X is %d\n",
               status, DayOfWeek);

   /* set the descriptor address and size of the resultant time string */
   TimeStringDsc.dsc$w_length = 29;
   TimeStringDsc.dsc$a_pointer = TimeString;

   status = sys$fao (&HttpTimeFaoDsc, &Length, &TimeStringDsc,
                     DayName[DayOfWeek], NumTime[2], MonthName[NumTime[1]],
                     NumTime[0], NumTime[3], NumTime[4], NumTime[5]);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
      TimeString[0] = '\0';
   }
   else
      TimeString[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", TimeString);

   return (status);
}

/*****************************************************************************/
/*
Given a string such as "Fri, 25 Aug 1995 17:32:40 GMT" (RFC1123), or "Friday,
25-Aug-1995 17:32:40 GMT" (RFC 1036), create an internal, local, binary time
with the current GMT offset.  See complementary function HttpGmTimeString().
*/ 

int HttpGmTime
(
char *TimeString,
unsigned long *BinTimePtr
)
{
   static char  *MonthName [] =
      { "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

   register char  *tptr;

   int  status;
   unsigned short  Length;
   unsigned short  NumTime [7] = { 0,0,0,0,0,0,0 };
   unsigned long  DayOfWeek;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpGmTime() |%s|\n", TimeString);

   tptr = TimeString;
   /* hunt straight for the comma after the weekday name! */
   while (*tptr && *tptr != ',') tptr++;
   if (*tptr) tptr++;
   /* span white space between weekday name and date */
   while (*tptr && ISLWS(*tptr)) tptr++;
   /** if (Debug) fprintf (stdout, "tptr |%s|\n", tptr); **/
   if (!*tptr) return (STS$K_ERROR);

   /* get the date and then skip to month name */
   if (isdigit(*tptr)) NumTime[2] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   while (*tptr && (*tptr == '-' || ISLWS(*tptr))) tptr++;
   /** if (Debug) fprintf (stdout, "tptr |%s|\n", tptr); **/
   if (!*tptr) return (STS$K_ERROR);

   /* get the month number from the name and skip to the year */
   for (NumTime[1] = 1; NumTime[1] <= 12; NumTime[1]++)
      if (strsame (tptr, MonthName[NumTime[1]], 3)) break;
   if (NumTime[1] > 12) return (STS$K_ERROR);
   while (*tptr && isalpha(*tptr)) tptr++;
   while (*tptr && (*tptr == '-' || ISLWS(*tptr))) tptr++;
   /** if (Debug) fprintf (stdout, "tptr |%s|\n", tptr); **/
   if (!*tptr) return (STS$K_ERROR);

   /* get the year and then skip to the hour */
   if (isdigit(*tptr))
   {
      NumTime[0] = atoi (tptr);
      if (NumTime[0] <= 99)
      {
         /* for older browsers supplying 2 digit years, some Y2K insurance! */
         if (NumTime[0] >= 70)
            NumTime[0] += 1900;
         else
            NumTime[0] += 2000;
      }
   }
   while (*tptr && isdigit(*tptr)) tptr++;
   while (*tptr && ISLWS(*tptr)) tptr++;
   /** if (Debug) fprintf (stdout, "tptr |%s|\n", tptr); **/
   if (!*tptr) return (STS$K_ERROR);

   /* get the hour, minute and second */
   if (isdigit(*tptr)) NumTime[3] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (isdigit(*tptr)) NumTime[4] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (isdigit(*tptr)) NumTime[5] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (!*tptr) return (STS$K_ERROR);

   /* the only thing remaining should be the "GMT" */
   while (*tptr && ISLWS(*tptr)) tptr++;
   /** if (Debug) fprintf (stdout, "tptr |%s|\n", tptr); **/
   if (!strsame (tptr, "GMT", 3)) return (STS$K_ERROR);

   /*******************************************/
   /* convert what looks like legitimate GMT! */
   /*******************************************/

   if (Debug)
      fprintf (stdout, "NumTime[] %d %d %d %d %d %d %d\n",
               NumTime[0], NumTime[1], NumTime[2], NumTime[3], 
               NumTime[4], NumTime[5], NumTime[6]); 
   status = lib$cvt_vectim (&NumTime, BinTimePtr);
   if (VMSnok (status)) return (status);
   if (Debug) fprintf (stdout, "lib$cvt_vectim() %%X%08.08X\n", status);

   return (TimeAdjustGMT (false, BinTimePtr));
}

/*****************************************************************************/
/*
Determine the offset from GMT (UTC) using either the HTTPD$GMT or
SYS$TIMEZONE_DIFFERENTIAL logicals. If HTTPD$GMT is not defined time should be
set from the timezone differential logical. Function RequestBegin() calls
this every hour to recheck GMT offset and detect daylight saving or other
timezone changes.
*/

int TimeSetGMT ()

{
   static boolean  UseTimezoneDifferential = false;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeSetGMT()\n");

   if (!UseTimezoneDifferential)
   {
      status = TimeSetHttpdGmt();
      /* return if error and that error was not that the name did not exist */
      if (VMSok (status) || status != SS$_NOLOGNAM) return (status);
   }

   UseTimezoneDifferential = true;
   return (TimeSetTimezone());
}

/*****************************************************************************/
/*
The SYS$TIMEZONE_DIFFERENTIAL logical contains the number of seconds offset
from GMT (UTC) as a positive (ahead) or negative (behind) number.  Set the
'TimeGmtString' global storage to a "+hh:mm" or "-hh:mm" string equivalent, and
the 'TimeAheadOfGmt' global boolean and 'TimeGmtVmsString' delta-time global
string.
*/

int TimeSetTimezone ()

{
   static unsigned short  Length;
   static $DESCRIPTOR (TimezoneLogicalNameDsc, "SYS$TIMEZONE_DIFFERENTIAL");
   static $DESCRIPTOR (LnmSystemDsc, "LNM$SYSTEM");
   static $DESCRIPTOR (TimeGmtStringFaoDsc, "!AZ!2ZL:!2ZL");
   static $DESCRIPTOR (TimeGmtVmsStringFaoDsc, "0 !2ZL:!2ZL");
   static $DESCRIPTOR (TimeGmtStringDsc, TimeGmtString);
   static struct {
      short int  buf_len;
      short int  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   } LnmItems [] =
   {
      { sizeof(TimeGmtString)-1, LNM$_STRING, TimeGmtString, &Length },
      { 0,0,0,0 }
   };

   int  status;
   long  Hours,
         Minutes,
         Seconds;
   char  *SignPtr;
   $DESCRIPTOR (TimeGmtVmsStringDsc, TimeGmtVmsString);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeSetTimezone()\n");

   status = sys$trnlnm (0, &LnmSystemDsc, &TimezoneLogicalNameDsc, 0, &LnmItems);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   TimeGmtString[Length] = '\0';
   if (Debug) fprintf (stdout, "TimeGmtString |%s|\n", TimeGmtString);
   Seconds = atol(TimeGmtString);
   if (Seconds < 0)
   {
      TimeAheadOfGmt = false;
      Seconds = -Seconds;
      SignPtr = "-";
   }
   else
   {
      TimeAheadOfGmt = true;
      SignPtr = "+";
   }
   Hours = Seconds / 3600;
   Minutes = (Seconds - Hours * 3600) / 60;
   if (Debug)
      fprintf (stdout, "%d %s%d:%d\n", Seconds, SignPtr, Hours, Minutes);
   sys$fao (&TimeGmtStringFaoDsc, &Length, &TimeGmtStringDsc,
            SignPtr, Hours, Minutes);
   TimeGmtString[Length] = '\0';
   if (Debug) fprintf (stdout, "TimeGmtString |%s|\n", TimeGmtString);

   sys$fao (&TimeGmtVmsStringFaoDsc, &Length, &TimeGmtVmsStringDsc,
            Hours, Minutes);
   TimeGmtVmsString[Length] = '\0';
   if (Debug) fprintf (stdout, "TimeGmtVmsString |%s|\n", TimeGmtVmsString);

   TimeGmtVmsStringDsc.dsc$w_length = Length;

   if (VMSnok (status =
       sys$bintim (&TimeGmtVmsStringDsc, &TimeGmtDeltaBinary)))
      return (status);
   if (TimeGmtDeltaBinary[0] || TimeGmtDeltaBinary[1])
      return (status);
   /* time must have been zero, make it one, one-hundreth of a second */
   TimeGmtDeltaBinary[0] = -100000;
   TimeGmtDeltaBinary[1] = -1;
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Translate the logical HTTPD$GMT (defined to be something like "+10:30" or "-
01:15") and convert it into a delta time structure and store in 
'TimeGmtDeltaBinary'.  Store whether it is in advance or behind GMT in boolean 
'TimeAheadOfGmt'.  Store the logical string in 'TimeGmtString'.
*/

int TimeSetHttpdGmt ()

{
   static unsigned short  Length;
   static $DESCRIPTOR (GmtLogicalNameDsc, "HTTPD$GMT");
   static $DESCRIPTOR (LnmFileDevDsc, "LNM$FILE_DEV");
   static struct {
      short int  buf_len;
      short int  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   } LnmItems [] =
   {
      { sizeof(TimeGmtString)-1, LNM$_STRING, TimeGmtString, &Length },
      { 0,0,0,0 }
   };

   register char  *cptr, *sptr;

   int  status;
   $DESCRIPTOR (TimeGmtVmsStringDsc, TimeGmtVmsString);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeSetHttpdGmt()\n");

   status = sys$trnlnm (0, &LnmFileDevDsc, &GmtLogicalNameDsc, 0, &LnmItems);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   TimeGmtString[Length] = '\0';
   if (Debug) fprintf (stdout, "TimeGmtString |%s|\n", TimeGmtString);

   if (TimeGmtString[0] == '$') return (SS$_NORMAL);

   if (*(cptr = TimeGmtString) == '-')
      TimeAheadOfGmt = false;
   else
      TimeAheadOfGmt = true;
   if (*cptr == '+' || *cptr == '-') cptr++;
   sptr = TimeGmtVmsString;
   *sptr++ = '0';
   *sptr++ = ' ';
   while (*cptr) *sptr++ = *cptr++;
   *sptr = '\0';
   if (Debug) fprintf (stdout, "TimeGmtVmsString |%s|\n", TimeGmtVmsString);

   TimeGmtVmsStringDsc.dsc$w_length = sptr - TimeGmtVmsString;

   if (VMSnok (status =
       sys$bintim (&TimeGmtVmsStringDsc, &TimeGmtDeltaBinary)))
      return (status);
   if (TimeGmtDeltaBinary[0] || TimeGmtDeltaBinary[1])
      return (status);
   /* time must have been zero, make it one, one-hundreth of a second */
   TimeGmtDeltaBinary[0] = -100000;
   TimeGmtDeltaBinary[1] = -1;
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
The GMT is generated by calculating using an offset using 
'TimeGmtDeltaOffset' and boolean 'TimeAheadOfGmt'.  Adjust either to or from 
GMT.
*/ 

int TimeAdjustGMT
(
boolean ToGmTime,
unsigned long *BinTimePtr
)
{
   int  status;
   unsigned long  AdjustedTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeAdjustGMT() ToGmTime: %d\n", ToGmTime);

   if ((ToGmTime && TimeAheadOfGmt) || (!ToGmTime && !TimeAheadOfGmt))
   {
      /* to GMT from local and ahead of GMT, or to local from GMT and behind */
      status = lib$sub_times (BinTimePtr, &TimeGmtDeltaBinary, &AdjustedTime);
      if (Debug) fprintf (stdout, "lib$sub_times() %%X%08.08X\n", status);
   }
   else
   {
      /* to GMT from local and behind GMT, or to local from GMT and ahead */
      status = lib$add_times (BinTimePtr, &TimeGmtDeltaBinary, &AdjustedTime);
      if (Debug) fprintf (stdout, "lib$add_times() %%X%08.08X\n", status);
   }

   if (Debug)
   {
      unsigned short  Length;
      char  String [64];
      $DESCRIPTOR (AdjustedTimeFaoDsc, "AdjustedTime: |!%D|\n");
      $DESCRIPTOR (StringDsc, String);

      sys$fao (&AdjustedTimeFaoDsc, &Length, &StringDsc, &AdjustedTime);
      String[Length] = '\0';
      fputs (String, stdout);
   }

   BinTimePtr[0] = AdjustedTime[0];
   BinTimePtr[1] = AdjustedTime[1];

   return (status);
}

/*****************************************************************************/
/*
Return a pointer to the date and time for the supplied binary time.
If a NULL pointer to the binary time then return current day of week.
As it returns a pointer to static area this will need to be copied out if the
contents must be preserved between calls.  'Precision' is the number of
characters in the time, '0' (or 23) for full hundreths, 20 to include seconds,
17 for minutes only.
*/

char* DateTime
(
unsigned long *BinTimePtr,
int Precision
)
{
   static char  DateTimeString [48];
   static $DESCRIPTOR (DateTimeFaoDsc, "!#%D\0");

   int  status;
   unsigned long  BinTime[2];
   $DESCRIPTOR (DateTimeStringDsc, DateTimeString);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DateTime()\n");

   if (BinTimePtr == NULL) sys$gettim (BinTimePtr = &BinTime);
   if (Precision <= 0 || Precision > 23) Precision = 23;
   if (!BinTimePtr[0] && !BinTimePtr[1])
      DateTimeString[0] = '\0';
   else
   if (VMSnok (sys$fao (&DateTimeFaoDsc, 0, &DateTimeStringDsc,
               Precision, BinTimePtr)))
      strcpy (DateTimeString, "*ERROR*");
   if (DateTimeString[0] == ' ') DateTimeString[0] = '0';
   return (DateTimeString);
}

/*****************************************************************************/
/*
Return a pointer to the day name, date and time for the supplied binary time.
If the supplied pointer is NULL then it returns the current date and time.
As it returns a pointer to static area this will need to be copied out if the
contents must be preserved between calls.  'Precision' is the number of
characters in the time, '0' (or 23) for full hundreths, 20 to include seconds,
17 for minutes only.
*/

char* DayDateTime
(
unsigned long *BinTimePtr,
int Precision
)
{
   static char  DayDateTimeString [36];
   static $DESCRIPTOR (DayDateTimeFaoDsc, "!AZ, !#%D\0");

   int  status;
   unsigned long  DayOfWeek;
   unsigned long  BinTime[2];
   $DESCRIPTOR (DayDateTimeStringDsc, DayDateTimeString);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DayDateTime()\n");

   if (BinTimePtr == NULL) sys$gettim (BinTimePtr = &BinTime);
   if (Precision <= 0 || Precision > 23) Precision = 23;
   if (!BinTimePtr[0] && !BinTimePtr[1])
      DayDateTimeString[0] = '\0';
   else
   if (VMSnok (lib$day_of_week (BinTimePtr, &DayOfWeek)))
      strcpy (DayDateTimeString, "*ERROR*");
   else
   if (VMSnok (sys$fao (&DayDateTimeFaoDsc, 0, &DayDateTimeStringDsc,
               DayName[DayOfWeek], Precision, BinTimePtr)))
      strcpy (DayDateTimeString, "*ERROR*");
   return (DayDateTimeString);
}

/*****************************************************************************/
/*
Return a pointer to the name of the day of the week for the supplied binary
time.  If a NULL pointer to the binary time then return current day of week.
*/

char* DayOfWeekName (unsigned long *BinTimePtr)

{
   int  status;
   unsigned long  DayOfWeek;
   unsigned long  BinTime[2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DayOfWeekName()\n");

   if (BinTimePtr == NULL) sys$gettim (BinTimePtr = &BinTime);
   if (VMSnok (status = lib$day_of_week (BinTimePtr, &DayOfWeek)))
      DayOfWeek = 0;
   if (Debug)
      fprintf (stdout, "lib$day_of_week() %%X%08.08X is %d\n",
               status, DayOfWeek);
   return (DayName[DayOfWeek]);
}

/*****************************************************************************/
/*
If the content length has changed, or if it has been modified since the
specified date and time then return a normal status indicating that the object
has been modified.  If not modified then creat a 304 HTTP header and return a
LIB$_NEGTIM status to indicate the file should not be sent.  With VMS'
fractions of a second, add one second to the specified 'since' time to ensure a
reliable comparison.

Implements defacto HTTP/1.0 persistent connections.  Currently we only provide
"keep-alive" response if we're sending a file in binary mode and know its
precise length. An accurate "Content-Length:" field is vital for the client's
correct reception of the transmitted data.  The only other time a "keep-alive"
response can be provided is HERE, where a 304 header is returned (it has a
content length of precisely zero!)
*/ 
 
int HttpIfModifiedSince
(
struct RequestStruct *rqptr,
unsigned long *LastModifiedBinaryTimePtr,
int LastContentLength
)
{
   static unsigned long  OneSecondDelta [2] = { -10000000, -1 };

   int  status;
   unsigned long  AdjustedBinTime[2],
                  ScratchBinTime [2];
   char  *KeepAlivePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "HttpIfModifiedSince() %d %d\n",
               LastContentLength, rqptr->IfModifiedSinceLength);

   if (rqptr->IfModifiedSinceLength >= 0 &&
       LastContentLength >= 0 &&
       rqptr->IfModifiedSinceLength != LastContentLength)
      return (SS$_NORMAL);

   /* add one second to the modified time, ensures a negative time */
   if (VMSnok (status =
       lib$add_times (&rqptr->IfModifiedSinceBinaryTime,
                      &OneSecondDelta, &AdjustedBinTime)))
   {
      rqptr->ErrorTextPtr = "sys$add_times()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }
   if (Debug) fprintf (stdout, "sys$add_times() %%X%08.08X\n", status);

   /* if a positive time results the file has been modified */
   if (VMSok (status =
       lib$sub_times (LastModifiedBinaryTimePtr,
                      &AdjustedBinTime, &ScratchBinTime)))
      return (status);

   if (Debug) fprintf (stdout, "sys$sub_times() %%X%08.08X\n", status);
   if (status != LIB$_NEGTIM)
   {
      rqptr->ErrorTextPtr = "sys$sub_times()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (status);
   }

   /*****************/
   /* not modified! */
   /*****************/

   rqptr->ResponseStatusCode = 304;

   /* return a "keep-alive" response field if configured and was requested */
   if (Config.KeepAliveTimeoutSeconds && rqptr->KeepAliveRequest)
   {
      rqptr->KeepAliveResponse = true;
      rqptr->KeepAliveCount++;
      KeepAlivePtr = KeepAliveHttpHeader;
   }
   else
      KeepAlivePtr = "";

   /* note the zero content length! */
   if ((rqptr->ResponseHeaderPtr =
        HttpHeader (rqptr, rqptr->ResponseStatusCode, NULL, 0,
                    NULL, KeepAlivePtr)) == NULL)
      return (STS$K_ERROR);

   return (LIB$_NEGTIM);
}

/*****************************************************************************/
/*
Returns a pointer to a VmGetHeap()ed string containing an HTTP response header.
'OtherHeaderPtr' is a catch-all parameter.  This string is directly included as
part of the HTTP header and so should contain correct carriage control, e.g.
"Keep-Alive:\r\n".
*/

char* HttpHeader
(
struct RequestStruct *rqptr,
int ResponseStatusCode,
char *ContentTypePtr,
int ContentLength,
unsigned long *ModifiedBinaryTimePtr,
char *OtherHeaderPtr
)
{
   static char  ContentTypeString [128] = "Content-Type: ",
                ExpiresString [64] = "Expires: ",
                ModifiedString [64] = "Last-Modified: ";

   static $DESCRIPTOR (ContentLengthFaoDsc, "Content-Length: !UL\r\n\0");

   static $DESCRIPTOR (HeaderFaoDsc,
"!AZ !UL !AZ\r\n\
Server: !AZ\r\n\
Date: !AZ\r\n\
!AZ\
!AZ\
!AZ\
!AZ\
!AZ\
\r\n");

   static $DESCRIPTOR (StringDsc, "");

   register char  *cptr, *sptr, *zptr;
   register unsigned long  *vecptr;

   int  status,
        ResponseStatusGroup;
   unsigned short  Length;
   unsigned long  FaoVector [16];
   char  *ExpiresStringPtr,
         *HeaderPtr,
         *ModifiedStringPtr;
   char  ContentLengthString [32],
         Modified [48],
         String [2048];

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      if (ContentTypePtr == NULL) ContentTypePtr = "";
      fprintf (stdout, "HttpHeader() %d |%s| %d\n",
               ResponseStatusCode, ContentTypePtr, ContentLength);
      if (!ContentTypePtr[0]) ContentTypePtr = NULL;
   }

   if (ContentTypePtr == NULL)
      ContentTypePtr = "";
   else
   {
      zptr = (sptr = ContentTypeString) + sizeof(ContentTypeString);
      sptr += 14;
      for (cptr = ContentTypePtr; *cptr && sptr < zptr; *sptr++ = *cptr++);
      if (sptr < zptr) *sptr++ = '\r';
      if (sptr < zptr) *sptr++ = '\n';
      if (sptr >= zptr)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         return (NULL);
      }
      *sptr = '\0';
      ContentTypePtr = ContentTypeString;
   }

   if (ContentLength == -1)
      ContentLengthString[0] = '\0';
   else
   {
      StringDsc.dsc$a_pointer = ContentLengthString;
      StringDsc.dsc$w_length = sizeof(ContentLengthString) - 1;
      status = sys$fao (&ContentLengthFaoDsc, 0, &StringDsc, ContentLength);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         return (NULL);
      }
   }

   if (ModifiedBinaryTimePtr == NULL)
      ModifiedStringPtr = "";
   else
   {
      if (VMSnok (status = HttpGmTimeString (Modified, ModifiedBinaryTimePtr)))
         return (NULL);

      sptr = (ModifiedStringPtr = ModifiedString) + 15;
      for (cptr = Modified; *cptr; *sptr++ = *cptr++);
      *sptr++ = '\r';
      *sptr++ = '\n';
      *sptr = '\0';
   }

   if (rqptr->ResponsePreExpired)
   {
      if (ModifiedBinaryTimePtr == NULL)
      {
         sptr = (ModifiedStringPtr = ModifiedString) + 15;
         for (cptr = rqptr->GmDateTime; *cptr; *sptr++ = *cptr++);
         *sptr++ = '\r';
         *sptr++ = '\n';
         *sptr = '\0';
      }
      /* including the terminating null */
      memcpy ((ExpiresStringPtr = ExpiresString) + 9,
              "Thu, 01-Jan-1970 00:00:01 GMT\r\n", 32);
   }
   else
      ExpiresStringPtr = "";

   if (!ResponseStatusCode) ResponseStatusCode = 200;
   if (OtherHeaderPtr == NULL) OtherHeaderPtr = "";

   vecptr = FaoVector;

   *vecptr++ = HttpProtocol;
   *vecptr++ = ResponseStatusCode;
   if ((ResponseStatusGroup = ResponseStatusCode / 100) == 1)
      *vecptr++ = "Informational";
   else
   if (ResponseStatusGroup == 2)
      *vecptr++ = "Success";
   else
   if (ResponseStatusCode == 304)
      *vecptr++ = "Not Modified";
   else
   if (ResponseStatusGroup == 3)
      *vecptr++ = "Redirection";
   else
   if (ResponseStatusGroup == 4)
      *vecptr++ = "Client Error";
   else
   if (ResponseStatusGroup == 5)
      *vecptr++ = "Server Error";
   else
      *vecptr++ = "?";

   *vecptr++ = SoftwareID;
   *vecptr++ = rqptr->GmDateTime;
   *vecptr++ = ContentTypePtr;
   *vecptr++ = ContentLengthString;
   *vecptr++ = ModifiedStringPtr;
   *vecptr++ = ExpiresStringPtr;
   *vecptr++ = OtherHeaderPtr;

   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String) - 1;

   status = sys$faol (&HeaderFaoDsc, &Length, &StringDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return (NULL);
   }
   String[Length] = '\0';

   HeaderPtr = VmGetHeap (rqptr, Length+1);
   memcpy (HeaderPtr, String, Length+1);
   rqptr->ResponseHeaderLength = Length;

   if (Debug) fprintf (stdout, "|%s|\n", String);

   return (HeaderPtr);
}

/*****************************************************************************/
/*
Append a string onto a HTTP header located in HTTPd heap allocated memory. 
This header string must already exist (created by HttpHeader200() or
equivalent).  The new string is appended to the current header, just before the
header terminating empty line.  Any string added must contain correct HTTP
header carriage control.
*/

char* HttpHeaderAppend
(
struct RequestStruct *rqptr,
char *HeaderPtr,
char *StringPtr,
int StringLength
)
{
   char  *cptr;
   int  HeaderLength;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpHeaderAppend()\n|%s|\n", HeaderPtr);

   if (HeaderPtr == NULL) return (NULL);

   HeaderLength = rqptr->ResponseHeaderLength;
   if (StringLength <= 0) StringLength = strlen(StringPtr);

   HeaderPtr = VmReallocHeap (rqptr, HeaderPtr, HeaderLength+StringLength);
   /* point to just before the header terminating empty line */
   cptr = HeaderPtr + HeaderLength - 3;
   memcpy (cptr, StringPtr, StringLength);
   /* add the new header terminating empty line */
   cptr += StringLength;
   *cptr++ = '\r';
   *cptr++ = '\n';
   *cptr = '\0';

   rqptr->ResponseHeaderLength = HeaderLength + StringLength - 1;

   if (Debug)
      fprintf (stdout, "%d |%s|\n", rqptr->ResponseHeaderLength, HeaderPtr);

   return (HeaderPtr);
}

/*****************************************************************************/
/*
A heap string is one located in HTTPd heap allocated memory.  Append the
supplied, null-terminated string to that pointed to by the heap string pointer. 
If this is NULL then allocate new memory, if not reallocate sufficient memory
to append the new string.  Return a pointer the string.
*/

char* HeapStringAppend
(
struct RequestStruct *rqptr,
char *HeapStringPtr,
char *StringPtr,
int StringLength
)
{
   int  HeapStringLength;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HeapStringAppend()\n|%s|\n", HeapStringPtr);

   if (StringLength <= 0) StringLength = strlen(StringPtr);

   if (HeapStringPtr == NULL)
   {
      HeapStringPtr = VmGetHeap (rqptr, StringLength+1);
      memcpy (HeapStringPtr, StringPtr, StringLength+1);
      if (Debug) fprintf (stdout, "|%s|\n", HeapStringPtr);
      return (HeapStringPtr);
   }
   else
   {
      HeapStringLength = strlen(HeapStringPtr);

      HeapStringPtr = VmReallocHeap (rqptr, HeapStringPtr,
                                     HeapStringLength+StringLength+1);
      /* copy just past the original end-of-string (incl. terminating null) */
      memcpy (HeapStringPtr+HeapStringLength, StringPtr, StringLength+1);
      if (Debug) fprintf (stdout, "|%s|\n", HeapStringPtr);
      return (HeapStringPtr);
   }
}

/*****************************************************************************/
/*
A heap string is one located in HTTPd heap allocated memory.  Prepend the
supplied, null-terminated string to that pointed to by the heap string pointer. 
If this is NULL then allocate new memory, if not reallocate sufficient memory
to append the new string.  Return a pointer the string.
*/

char* HeapStringPrepend
(
struct RequestStruct *rqptr,
char *HeapStringPtr,
char *StringPtr,
int StringLength
)
{
   int  HeapStringLength;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HeapStringPrepend()\n|%s|\n", HeapStringPtr);

   if (StringLength <= 0) StringLength = strlen(StringPtr);

   if (HeapStringPtr == NULL)
   {
      HeapStringPtr = VmGetHeap (rqptr, StringLength+1);
      memcpy (HeapStringPtr, StringPtr, StringLength+1);
      if (Debug) fprintf (stdout, "|%s|\n", HeapStringPtr);
      return (HeapStringPtr);
   }
   else
   {
      HeapStringLength = strlen(HeapStringPtr);
      HeapStringPtr = VmReallocHeap (rqptr, HeapStringPtr,
                                     HeapStringLength+StringLength+1);
      /* shift the existing string along in memory creating a gap */
      memcpy (HeapStringPtr+HeapStringLength,
              HeapStringPtr,
              HeapStringLength+1);  /* includes terminating null */
      /* copy just past the original end-of-string */
      memcpy (HeapStringPtr, StringPtr, StringLength);
      if (Debug) fprintf (stdout, "|%s|\n", HeapStringPtr);
      return (HeapStringPtr);
   }
}

/*****************************************************************************/
/*
Returns 1 if time 1 is later than time 2, 0 if the same, and -1 if time 1 is
earlier than time 2.
*/

int CompareVmsBinTimes
(
unsigned long *BinTime1Ptr,
unsigned long *BinTime2Ptr
)
{
   int  status;
   unsigned long  BinTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CompareVmsBinTimes()\n");

   status = lib$sub_times (BinTime1Ptr, BinTime2Ptr, BinTime);
   if (Debug) fprintf (stdout, "lib$sub_times() %%X%08.08X\n", status);
   if (status == LIB$_NEGTIM)
   {
      if (Debug) fprintf (stdout, "1 < 2\n");
      return (-1);
   }
   else
   if (VMSok (status))
   {
      if (BinTime1Ptr[0] == BinTime2Ptr[0] && BinTime1Ptr[1] == BinTime2Ptr[1])
      {
         if (Debug) fprintf (stdout, "1 == 2\n");
         return (0);
      }
      else
      {
         if (Debug) fprintf (stdout, "1 > 2\n");
         return (1); 
      }
   }
   else
      ErrorExitVmsStatus (status, "lib$sub_times", FI_LI);
}

/*****************************************************************************/
/*
If we can't get SYSPRV when we need it, or more importantly, if we can't turn
it off when we're finished with it the exit the server!
*/ 

EnableSysPrv ()

{
   static unsigned long  SysPrvMask [2] = { PRV$M_SYSPRV, 0 };

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "EnableSysPrv()\n");

   if (VMSok (status = sys$setprv (1, &SysPrvMask, 0, 0))) return;
   ErrorExitVmsStatus (status, "enabling SYSPRV", FI_LI);
}

/*****************************************************************************/
/*
If we can't get SYSPRV when we need it, or more importantly, if we can't turn
it off when we're finished with it the exit the server!
*/ 

DisableSysPrv ()

{
   static unsigned long  SysPrvMask [2] = { PRV$M_SYSPRV, 0 };

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DisableSysPrv()\n");

   if (VMSok (status = sys$setprv (0, &SysPrvMask, 0, 0))) return;
   ErrorExitVmsStatus (status, "disabling SYSPRV", FI_LI);
}

/*****************************************************************************/
/*
Returns a pointer to a VmGetHeap()ed string of <META ...> tags for an
internally generated HTML document.
*/

char* HtmlMetaInfo
(
struct RequestStruct *rqptr,
char *VmsInfoPtr
)
{
   static char  Port [16];

   register char  *cptr, *sptr, *zptr;

   char  *MetaPtr;
   char  String [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HtmlMetaInfo()\n");

   if (!Port[0])
   {
      $DESCRIPTOR (PortFaoDsc, "!UL\0");
      $DESCRIPTOR (PortDsc, Port);

      sys$fao (&PortFaoDsc, 0, &PortDsc, rqptr->ServicePtr->ServerPort);
   }

   zptr = (sptr = String) + sizeof(String);

   for (cptr = "<META NAME=\"generator\" CONTENT=\"";
        *cptr && sptr < zptr;
        *sptr++ = *cptr++);
   for (cptr = SoftwareID; *cptr && sptr < zptr; *sptr++ = *cptr++);
   for (cptr = "\">\n"; *cptr && sptr < zptr; *sptr++ = *cptr++);

   for (cptr = "<META NAME=\"date\" CONTENT=\"";
        *cptr && sptr < zptr;
        *sptr++ = *cptr++);
   for (cptr = rqptr->GmDateTime; *cptr && sptr < zptr; *sptr++ = *cptr++);
   for (cptr = "\">\n"; *cptr && sptr < zptr; *sptr++ = *cptr++);

   if (rqptr->RemoteUser[0])
   {
      for (cptr = "<META NAME=\"author\" CONTENT=\"";
           *cptr && sptr < zptr;
           *sptr++ = *cptr++);
      for (cptr = rqptr->RemoteUser; *cptr && sptr < zptr; *sptr++ = *cptr++);
      if (sptr < zptr) *sptr++ = '@';
   }
   else
   {
      for (cptr = "<META NAME=\"host\" CONTENT=\"";
           *cptr && sptr < zptr;
           *sptr++ = *cptr++);
   }
   for (cptr = rqptr->ServicePtr->ServerHostName;
        *cptr && sptr < zptr;
        *sptr++ = *cptr++);
   if (sptr < zptr) *sptr++ = ':';
   for (cptr = Port; *cptr && sptr < zptr; *sptr++ = *cptr++);
   for (cptr = "\">\n"; *cptr && sptr < zptr; *sptr++ = *cptr++);


   if (VmsInfoPtr != NULL)
      if (VmsInfoPtr[0])
         if (Config.IncludeCommentedInfo)
         {
            for (cptr = "<META NAME=\"VMS\" CONTENT=\"";
                 *cptr && sptr < zptr;
                 *sptr++ = *cptr++);
            for (cptr = VmsInfoPtr; *cptr && sptr < zptr; *sptr++ = *cptr++);
            for (cptr = "\">\n"; *cptr && sptr < zptr; *sptr++ = *cptr++);
         }

   if (sptr >= zptr) return ("<!-- META overflow ERROR -->\n");
   *sptr++ = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);

   MetaPtr = VmGetHeap (rqptr, sptr-String);
   memcpy (MetaPtr, String, sptr-String);
   return (MetaPtr);
}

/*****************************************************************************/
/*
Return success if the specified file name exists in the specified directory, 
error otherwise.  This function completes synchronously.
*/ 

int FileExists
(
char *Directory,
char *FileName
)
{
   int  status;
   char  ExpandedFileName [256];
   struct FAB  SearchFab;
   struct NAM  SearchNam;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "FileExists() |%s|%s|\n", Directory, FileName);

   SearchFab = cc$rms_fab;
   SearchFab.fab$l_dna = Directory;
   SearchFab.fab$b_dns = strlen(Directory);
   SearchFab.fab$l_fna = FileName;
   SearchFab.fab$b_fns = strlen(FileName);
   SearchFab.fab$l_fop = FAB$M_NAM;
   SearchFab.fab$l_nam = &SearchNam;
   SearchNam = cc$rms_nam;
   SearchNam.nam$l_esa = ExpandedFileName;
   SearchNam.nam$b_ess = sizeof(ExpandedFileName)-1;

   status = sys$parse (&SearchFab, 0, 0);
   if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);

   if (VMSok (status))
   {
      status = sys$search (&SearchFab, 0, 0);
      if (Debug) fprintf (stdout, "sys$search() %%X%08.08X\n", status);
   }

   /* release parse and search internal data structures */
   SearchFab.fab$l_fna = "a:[b]c.d;";
   SearchFab.fab$b_fns = 9;
   SearchFab.fab$b_dns = 0;
   SearchNam.nam$b_nop = NAM$M_SYNCHK;
   sys$parse (&SearchFab, 0, 0);

   return (status);
}

/*****************************************************************************/
/*
Given a file or directory specification in 'FileName' generate the file name
of the directory file.

(e.g. "DEVICE:[DIR1]DIR2.DIR" from "DEVICE:[DIR1.DIR2]FILE.EXT",
      "DEVICE:[DIR1]DIR2.DIR" from "DEVICE:[DIR1.DIR2]",
 and  "DEVICE:[000000]DIR1.DIR" from "DEVICE:[DIR1]", etc.)

Attempts to improve the perfomance of this function by storing the previous
file specification processed and the previous directory file name result.  If
this file specification directory part is the same as the previous directory
part then just return the previous result!
*/ 
 
int NameOfDirectoryFile
(
char *FileName,
int FileNameLength,
char *DirFileNamePtr,
int *DirFileLengthPtr
)
{
   static int  DirFileNameBufferLength = 0;
   static char  FileNameBuffer [256],
                DirFileNameBuffer [256] = "";

   register char  *cptr, *fptr, *sptr;

   int  status;
   char  ExpandedFileName [256];
   struct FAB  FileFab;
   struct NAM  FileNam;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "NameOfDirectoryFile() |%s|%s|%s|\n",
               FileName, FileNameBuffer, DirFileNameBuffer);

   if (!FileNameLength) FileNameLength = strlen(FileName);

   DirFileNamePtr[*DirFileLengthPtr = 0] = '\0';

   /* check if this has the same directory components as the last one */
   sptr = FileNameBuffer;
   cptr = FileName;
   while (toupper(*cptr) == toupper(*sptr) && *sptr != ']' && *cptr != ']')
   {
      *sptr++;
      *cptr++;
   }
   if (*cptr == ']' && *sptr == ']' && *(sptr-1) != '.' && *(cptr-1) != '.')
   {
      /* it does! return the result of the last one */
      strcpy (DirFileNamePtr, DirFileNameBuffer);
      *DirFileLengthPtr = DirFileNameBufferLength;
      if (Debug) fprintf (stdout, "buffer! |%s|\n", DirFileNamePtr);
      return (SS$_NORMAL);
   }

   FileFab = cc$rms_fab;
   FileFab.fab$l_fna = FileName;
   FileFab.fab$b_fns = FileNameLength;
   FileFab.fab$l_nam = &FileNam;

   FileNam = cc$rms_nam;
   FileNam.nam$l_esa = ExpandedFileName;
   FileNam.nam$b_ess = sizeof(ExpandedFileName)-1;
   /* parse without disk I/O, syntax check only, BUT WITH NOCONCEAL! */
   FileNam.nam$b_nop = NAM$M_SYNCHK | NAM$M_NOCONCEAL;

   if (VMSnok (status = sys$parse (&FileFab, 0, 0)))
      return (status);

   FileNam.nam$l_ver[FileNam.nam$b_ver] = '\0';
   if (Debug) fprintf (stdout, "ExpandedFileName |%s|\n", ExpandedFileName);

   fptr = FileNam.nam$l_name - 1;
   while (fptr > ExpandedFileName && *fptr != '[' && *fptr != '.') fptr--;
   if (fptr > ExpandedFileName && fptr[-1] == ']')
   {
      /* concealed, logical device */
      fptr -= 2;
   }

   sptr = DirFileNameBuffer;
   if (!memcmp (fptr, "[000000]", 8) || !memcmp (fptr, "[000000.]", 9))
   {
      for (cptr = FileNam.nam$l_dev; cptr < fptr; *sptr++ = *cptr++);
      for (cptr = "[000000]000000.DIR"; *cptr; *sptr++ = *cptr++);
      *sptr = '\0';
   }
   else
   if (*fptr == '[')
   {
      for (cptr = FileNam.nam$l_dev; cptr < fptr; *sptr++ = *cptr++);
      for (cptr = "[000000]"; *cptr; *sptr++ = *cptr++);
      if (fptr[0] == '.' && fptr[1] == ']')
         fptr += 3;
      else
         fptr++;
      while (*fptr && *fptr != '.' && *fptr != ']') *sptr++ = *fptr++;
      for (cptr = ".DIR"; *cptr; *sptr++ = *cptr++);
      *sptr = '\0';
   }
   else
   {
      for (cptr = FileNam.nam$l_dev; cptr < fptr; *sptr++ = *cptr++);
      *sptr++ = ']';
      if (fptr[0] == '.' && fptr[1] == ']')
         fptr += 3;
      else
         fptr++;
      while (*fptr && *fptr != '.' && *fptr != ']') *sptr++ = *fptr++;
      for (cptr = ".DIR"; *cptr; *sptr++ = *cptr++);
      *sptr = '\0';
   }

   /* buffer this (soon to be the previous) file name */
   strcpy (FileNameBuffer, FileName);

   /* copy out the generated directory file name */
   strcpy (DirFileNamePtr, DirFileNameBuffer);
   *DirFileLengthPtr = DirFileNameBufferLength = sptr - DirFileNameBuffer;

   if (Debug)
      fprintf (stdout, "DirFileNamePtr %d |%s|\n",
               *DirFileLengthPtr, DirFileNamePtr);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Zero server accounting structure and service counters.  Redefine monitor
logicals to reflect this.
*/ 

ZeroAccounting ()

{
   register struct ServiceStruct  *svptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ZeroAccounting()\n");

   for (svptr = ServiceListHead; svptr != NULL; svptr = svptr->NextPtr)
      svptr->ConnectCount = 0;

   memset ((char*)&Accounting + sizeof(Accounting.ZeroedCount),
           0,
           sizeof(Accounting) - sizeof(Accounting.ZeroedCount));

   /* minimum duration must be set very high */
   Accounting.ResponseDurationMin = 0xffffffff;

   Accounting.ZeroedCount++;

   if (MonitorEnabled) DefineMonitorLogicals (NULL);
}

/*****************************************************************************/
/*
There are enough exit points from the function that actually does the logical
name processing that this function wraps it with the appropriate privilege
changes to make sure we don't miss any!
*/ 

DefineMonitorLogicals (struct RequestStruct *rqptr)

{
   static unsigned long  SysNamMask [2] = { PRV$M_SYSNAM, 0 };

   int  status,
        SetPrvStatus;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DefineMonitorLogicals()\n");

   /* turn on SYSNAM to allow access to system table */
   if (VMSnok (status = sys$setprv (1, &SysNamMask, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", status);
      return (status);
   }

   status = DefineMonitorLogicalsHere (rqptr);

   /* turn off SYSNAM */
   if (VMSnok (SetPrvStatus = sys$setprv (0, &SysNamMask, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", SetPrvStatus);
      /* do NOT keep processing if there is a problem turning off SYSNAM! */
      ErrorExitVmsStatus (SetPrvStatus, "disabling SYSNAM", FI_LI);
   }

   return (status);
}

/*****************************************************************************/
/*
Define two/three logicals used by the HTTPd monitor utility.  The value of the 
count logical is the binary contents of the server counter data structure.  
The value of the request logical is some information on the latest request to 
complete.  The PID logical is defined only once, the first time this function 
is called.
*/ 

DefineMonitorLogicalsHere (struct RequestStruct *rqptr)

{
   static unsigned long  JpiPid;
   static char  CountLogicalName1 [32],
                CountLogicalName2 [32],
                RequestLogicalName [32],
                RequestLogicalValue [256];

   static $DESCRIPTOR (CountLogicalName1Dsc, CountLogicalName1);
   static $DESCRIPTOR (CountLogicalName2Dsc, CountLogicalName2);
   static $DESCRIPTOR (LnmSystemDsc, "LNM$SYSTEM");
   static $DESCRIPTOR (RequestLogicalNameDsc, RequestLogicalName);
   static $DESCRIPTOR (RequestLogicalValueDsc, RequestLogicalValue);
   /*
      "day time\0client-host\0status\0rx\0tx\0duration\0" +
      "request-scheme//server-host:port\0method resource"
   */
   static $DESCRIPTOR (RequestLogicalValueFaoDsc,
"!2ZL !2ZL:!2ZL:!2ZL!AD!UL!AD!UL!AD!UL!AD!UL.!3ZL!AD\
!AZ//!AZ:!AZ!AD!AZ!AD!AZ !AZ");

   static struct VmsItem LnmCountItem1 [] =
          {
             { 255, LNM$_STRING, &Accounting, 0 },
             { 0,0,0,0 }
          };
   static struct VmsItem LnmCountItem2 [] =
          {
             { sizeof(Accounting)-255, LNM$_STRING, ((char*)&Accounting)+255, 0 },
             { 0,0,0,0 }
          };
   static struct VmsItem LnmRequestItem [] =
          {
             { 0, LNM$_STRING, RequestLogicalValue, 0 },
             { 0,0,0,0 }
          };

   register unsigned long  *vecptr;

   boolean  KeepAliveTimeout;
   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char *ResourcePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DefineMonitorLogicalsHere()\n");

   if (!JpiPid)
   {
      /**********************/
      /* first time through */
      /**********************/

      /* local storage */
      char  PidLogicalName [16];
      $DESCRIPTOR (PidLogicalNameDsc, PidLogicalName);
      $DESCRIPTOR (PidLogicalNameFaoDsc, "HTTPD!UL$PID");
      $DESCRIPTOR (CountLogicalName1FaoDsc, "HTTPD!UL$COUNT1");
      $DESCRIPTOR (CountLogicalName2FaoDsc, "HTTPD!UL$COUNT2");
      $DESCRIPTOR (RequestLogicalNameFaoDsc, "HTTPD!UL$REQUEST");
      struct VmsItem  JpiPidItem [] =
      {
         { sizeof(JpiPid), JPI$_PID, &JpiPid, 0 },
         {0,0,0,0}
      };
      struct VmsItem  LnmPidItem [] =
      {
         { sizeof(JpiPid), LNM$_STRING, &JpiPid, 0 },
         { 0,0,0,0 }
      };

      if (Debug)
         fprintf (stdout, "sizeof(Accounting): %d\n", sizeof(Accounting));

      /* define a logical name containing the PID of the server process */
      if (VMSnok (status = sys$getjpiw (0, 0, 0, &JpiPidItem, 0, 0, 0)))
         return (status);
      sys$fao (&PidLogicalNameFaoDsc, &Length, &PidLogicalNameDsc,
               ServerPort);
      PidLogicalName[PidLogicalNameDsc.dsc$w_length = Length] = '\0';
      if (Debug) fprintf (stdout, "PidLogicalName |%s|\n", PidLogicalName);
      if (VMSnok (status =
          sys$crelnm (0, &LnmSystemDsc, &PidLogicalNameDsc, 0, &LnmPidItem)))
         return (status);

      /* generate approriate names for the count and request logicals */
      sys$fao (&CountLogicalName1FaoDsc, &Length, &CountLogicalName1Dsc,
               ServerPort);
      CountLogicalName1[CountLogicalName1Dsc.dsc$w_length = Length] = '\0';
      if (Debug) fprintf (stdout, "CountLogicalName1 |%s|\n", CountLogicalName1);
      sys$fao (&CountLogicalName2FaoDsc, &Length, &CountLogicalName2Dsc,
               ServerPort);
      CountLogicalName2[CountLogicalName2Dsc.dsc$w_length = Length] = '\0';
      if (Debug) fprintf (stdout, "CountLogicalName2 |%s|\n", CountLogicalName2);
      sys$fao (&RequestLogicalNameFaoDsc, &Length, &RequestLogicalNameDsc,
               ServerPort);
      RequestLogicalName[RequestLogicalNameDsc.dsc$w_length = Length] = '\0';
      if (Debug)
         fprintf (stdout, "RequestLogicalName |%s|\n", RequestLogicalName);

      /* attempt to read the values of any previous count logical name */
      if (VMSok (status =
          sys$trnlnm (0, &LnmSystemDsc, &CountLogicalName1Dsc, 0,
                      &LnmCountItem1)))
      if (VMSok (status =
          sys$trnlnm (0, &LnmSystemDsc, &CountLogicalName2Dsc, 0,
                      &LnmCountItem2)))
         Accounting.StartupCount++;
      else
      {
         if (status != SS$_NOLOGNAM)
            return (status);
      }
   }

   /*****************************************/
   /* (re)define count and request logicals */
   /*****************************************/

   if (rqptr == NULL)
   {
      KeepAliveTimeout = false;
      /* reset the request logical to all empty strings */
      RequestLogicalValue[0] = '\0';
      LnmRequestItem[0].buf_len = 1;
   }
   else
   if (!(KeepAliveTimeout = rqptr->KeepAliveTimeout))
   {
      if (rqptr->ResourcePtr == NULL)
         ResourcePtr = "";
      else
         ResourcePtr = rqptr->ResourcePtr;

      vecptr = FaoVector;
      *vecptr++ = rqptr->NumericTime[2];
      *vecptr++ = rqptr->NumericTime[3];
      *vecptr++ = rqptr->NumericTime[4];
      *vecptr++ = rqptr->NumericTime[5];
      *vecptr++ = 1;
      *vecptr++ = "\0";
      *vecptr++ = rqptr->ResponseStatusCode;
      *vecptr++ = 1;
      *vecptr++ = "\0";
      *vecptr++ = rqptr->BytesRx;
      *vecptr++ = 1;
      *vecptr++ = "\0";
      *vecptr++ = rqptr->BytesTx;
      *vecptr++ = 1;
      *vecptr++ = "\0";
      /* duration is in microseconds, provide this seconds and milliseconds */
      *vecptr++ = rqptr->ResponseDuration / 1000000;
      *vecptr++ = (rqptr->ResponseDuration /1000) % 1000;
      *vecptr++ = 1;
      *vecptr++ = "\0";
      *vecptr++ = rqptr->RequestSchemeNamePtr;
      *vecptr++ = rqptr->ServicePtr->ServerHostName;
      *vecptr++ = rqptr->ServicePtr->ServerPortString;
      *vecptr++ = 1;
      *vecptr++ = "\0";
      *vecptr++ = rqptr->ClientHostName;
      *vecptr++ = 1;
      *vecptr++ = "\0";
      *vecptr++ = rqptr->HttpMethodName;
      *vecptr++ = ResourcePtr;

      status = sys$faol (&RequestLogicalValueFaoDsc, &Length,
                         &RequestLogicalValueDsc, &FaoVector);

      if (Debug) fprintf (stdout, "sys$faol() %%X%08.08X\n", status);
      RequestLogicalValue[LnmRequestItem[0].buf_len = Length] = '\0';
      if (Debug)
         fprintf (stdout, "RequestLogicalValue |%s|\n", RequestLogicalValue);
   }

   if (VMSnok (status =
       sys$crelnm (0, &LnmSystemDsc, &CountLogicalName1Dsc, 0, &LnmCountItem1)))
      return (status);

   if (VMSnok (status =
       sys$crelnm (0, &LnmSystemDsc, &CountLogicalName2Dsc, 0, &LnmCountItem2)))
      return (status);

   if (!KeepAliveTimeout)
   {
      if (VMSnok (status =
          sys$crelnm (0, &LnmSystemDsc, &RequestLogicalNameDsc, 0,
                      &LnmRequestItem)))
         return (status);
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Convert a VMS quadword, binary time to a Unix-style time component structure.
*/

boolean TimeVmsToUnix
(
unsigned long *BinTimePtr,
struct tm *TmPtr
)
{
   static unsigned long  LibDayOfWeek = LIB$K_DAY_OF_WEEK;
   static unsigned long  LibDayOfYear = LIB$K_DAY_OF_YEAR;

   int  status;
   unsigned long  DayOfWeek,
                  DayOfYear;
   unsigned short  NumTime [7];
   unsigned long  BinTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeVmsToUnix()\n");

   if (BinTimePtr == NULL) sys$gettim (BinTimePtr = &BinTime);

   if (VMSnok (status = sys$numtim (&NumTime, BinTimePtr)))
      return (status);

   lib$cvt_from_internal_time (&LibDayOfWeek, &DayOfWeek, BinTimePtr);
   lib$cvt_from_internal_time (&LibDayOfYear, &DayOfYear, BinTimePtr);

   TmPtr->tm_sec = NumTime[5];
   TmPtr->tm_min = NumTime[4];
   TmPtr->tm_hour = NumTime[3];
   TmPtr->tm_mday = NumTime[2];
   TmPtr->tm_mon = NumTime[1] - 1;
   TmPtr->tm_year = NumTime[0] - 1900;
   TmPtr->tm_wday = DayOfWeek;
   TmPtr->tm_yday = DayOfYear;
   TmPtr->tm_isdst = 0;

   if (Debug) fprintf (stdout, "%s", asctime(TmPtr));

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Convert the 32/64 bit integer (depending on architecture) pointed to into an
ASCII number string containing commas.  The destination string should contain
capacity of a minimum 16 characters for 32 bits, or 32 characters for 64 bits.
Note that the "!SQ" directive is used, for VMS 6.2 the "!UQ" && "!UJ" loop
infinitely for signed values.  Returns the length of the generated string.
*/

int CommaNumber
(
int Bits,
unsigned long Value,
char *String
)
{
   static char  Scratch [32];
   static $DESCRIPTOR (ScratchDsc, Scratch);
   static $DESCRIPTOR (Value32FaoDsc, "!UL");
   static $DESCRIPTOR (Value64FaoDsc, "!@SJ");

   register int  cnt;
   register char  *cptr, *sptr;

   int  status;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CommaNumber()\n");

   if (Bits > 32)
      status = sys$fao (&Value64FaoDsc, &Length, &ScratchDsc, Value);
   else
      status = sys$fao (&Value32FaoDsc, &Length, &ScratchDsc, Value);
   if (VMSnok (status))
   {
      strcpy (String, "*ERROR*");
      return (7);
   }
   Scratch[Length] = '\0';
   if (((Length-1) / 3) < 1)
   {
      cptr = Scratch;
      sptr = String;
      while (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
      return (sptr - String);
   }
   else
   if (!(cnt = Length % 3))
      cnt = 3;
   
   cptr = Scratch;
   sptr = String;
   while (*cptr)
   {
      if (!cnt--)
      {
         *sptr++ = ',';
         cnt = 2;
      }
      *sptr++ = *cptr++;
   }
   *sptr = '\0';
   return (sptr - String);
}

/*****************************************************************************/
/*
If not compiling with DEC C then we need to create our own strftime().  Make 
it a minimal implementation, no attention paid to the format string, etc.
*/ 

#ifndef __DECC

int strftime
(
char *TimeStringPtr,
int SizeOfTimeString,
char *TimeFormatPtr,
struct tm *TmPtr
)
{
   static $DESCRIPTOR (TimeFaoDsc, "!3AZ, !2ZL !3AZ !2ZL, !2ZL:!2ZL:!2ZL");

   unsigned short  Length;
   $DESCRIPTOR (TimeStringPtrDsc, TimeStringPtr);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "strftime() well ... sort of\n");

   TimeStringPtrDsc.dsc$a_pointer = TimeStringPtr;
   TimeStringPtrDsc.dsc$w_length = SizeOfTimeString - 1;

   Length = 0;
   sys$fao (&TimeFaoDsc, &Length, &TimeStringPtrDsc,
            DayName[TmPtr->tm_wday], TmPtr->tm_mday, MonthName[TmPtr->tm_mon],
            TmPtr->tm_year, TmPtr->tm_hour, TmPtr->tm_min, TmPtr->tm_sec);
   TimeStringPtr[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", TimeStringPtr);
   return (Length);
}

#endif

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text in HTML.  For example the '<', '&', etc.  Convert these 
to the corresponding HTML character entities.  Returns the number of characters
added to the buffer or -1 to indicate buffer overflow.  Copies a maximum of
'Count' characters, or all id it is -1.
*/ 

int CopyToHtml
( 
char *Destination,
int SizeOfDestination,
char *Source,
int Count
)
{
    register char  *cptr, *sptr, *zptr;
    register int  ccnt;

    cptr = Source;
    if (SizeOfDestination <= 0) return (-1);
    zptr = (sptr = Destination) + SizeOfDestination - 1;
    ccnt = Count;
    while (--ccnt && *cptr)
    {
       switch (*cptr)
       {
          case '<' :
             if (sptr+4 >= zptr) return (-1);
             memcpy (sptr, "&lt;", 4); sptr += 4; cptr++; break;
          case '>' :
             if (sptr+4 >= zptr) return (-1);
             memcpy (sptr, "&gt;", 4); sptr += 4; cptr++; break;
          case '&' :
             if (sptr+5 >= zptr) return (-1);
             memcpy (sptr, "&amp;", 5); sptr += 5; cptr++; break;
          case '\"' :
             if (sptr+6 >= zptr) return (-1);
             memcpy (sptr, "&quot;", 6); sptr += 6; cptr++; break;
          default :
             if (sptr >= zptr) return (-1);
             *sptr++ = *cptr++;
       }
    }
    *sptr = '\0';
    if (sptr >= zptr) return (-1);
    return (sptr-Destination);
}

/*****************************************************************************/
/*
A null terminated string is parsed for the next "fieldname=fieldvalue[&]"
pair.  If an error is encountered generated an error message and return NULL,
otherwise return a pointer to the next pair or end-of-string.
*/

char* ParseQueryField
(
struct RequestStruct *rqptr,
char *QueryString,
char *FieldName,
int SizeOfFieldName,
char *FieldValue,
int SizeOfFieldValue,
char *SourceCodeFile,
int SourceCodeLine
)
{
   register char  *cptr, *qptr, *sptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ParseQueryField() |%s|\n", QueryString);

   qptr = QueryString;
   zptr = (sptr = FieldName) + SizeOfFieldName;
   while (*qptr && *qptr != '=')
   {
      while (*qptr == '\r' || *qptr == '\n') qptr++;
      while (*qptr && *qptr != '=' &&
             *qptr != '\r' && *qptr != '\n' && sptr < zptr) *sptr++ = *qptr++;
   }
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, SourceCodeFile, SourceCodeLine);
      return (NULL);
   }
   *sptr = '\0';
   if (Debug) fprintf (stdout, "FieldName |%s|\n", FieldName);

   if (*qptr++ != '=')
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_URL_ENC),
                    SourceCodeFile, SourceCodeLine);
      return (NULL);
   }

   zptr = (sptr = FieldValue) + SizeOfFieldValue;
   while (*qptr && *qptr != '&')
   {
      while (*qptr == '\r' || *qptr == '\n') qptr++;
      while (*qptr && *qptr != '&' &&
             *qptr != '\r' && *qptr != '\n' && sptr < zptr) *sptr++ = *qptr++;
   }
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, SourceCodeFile, SourceCodeLine);
      return (NULL);
   }
   *sptr = '\0';
   if (Debug) fprintf (stdout, "FieldValue |%s|\n", FieldValue);

   if (*qptr && *qptr != '&')
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_URL_ENC),
                    SourceCodeFile, SourceCodeLine);
      return (NULL);
   }
   if (*qptr) qptr++;

   if (UrlDecodeString (FieldValue) < 0)
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_REQUEST_URL_ENC),
                    SourceCodeFile, SourceCodeLine);
      return (NULL);
   }

   return (qptr);
}

/****************************************************************************/
/*
generate a standard VMS protection string from the mask. 'String' must be at
least 28 bytes (better use 32 :^)  Returns the length of the string.
*/ 
 
int FormatProtection
(
unsigned short pmask,
char *String
)
{
   register char  *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "FormatProtection() %%X%04.04X\n", pmask);

   sptr = String;

   if (!(pmask & 0x0001)) *sptr++ = 'R';
   if (!(pmask & 0x0002)) *sptr++ = 'W';
   if (!(pmask & 0x0004)) *sptr++ = 'E';
   if (!(pmask & 0x0008)) *sptr++ = 'D';
   *sptr++ = ',';
   if (!(pmask & 0x0010)) *sptr++ = 'R';
   if (!(pmask & 0x0020)) *sptr++ = 'W';
   if (!(pmask & 0x0040)) *sptr++ = 'E';
   if (!(pmask & 0x0080)) *sptr++ = 'D';
   *sptr++ = ',';
   if (!(pmask & 0x0100)) *sptr++ = 'R';
   if (!(pmask & 0x0200)) *sptr++ = 'W';
   if (!(pmask & 0x0400)) *sptr++ = 'E';
   if (!(pmask & 0x0800)) *sptr++ = 'D';
   *sptr++ = ',';
   if (!(pmask & 0x1000)) *sptr++ = 'R';
   if (!(pmask & 0x2000)) *sptr++ = 'W';
   if (!(pmask & 0x4000)) *sptr++ = 'E';
   if (!(pmask & 0x8000)) *sptr++ = 'D';

   *sptr = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);
   return (sptr - String);
}
 
/****************************************************************************/
/*
URL-decodes a string in place (can do this because URL-decoded text is always
the same or less than the length of the original).  Returns the number of
characters in the decoded string, or -1 to indicate a URL-encoding error.
*/ 
 
int UrlDecodeString (char *String)

{
   char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UrlDecodeString() |%s|\n", String);

   cptr = sptr = String;
   while (*cptr)
   {
      switch (*cptr)
      {
         case '=' :
         case '&' :
            /* URL-forbidden characters */
            return (-1);

         case '+' :
            *sptr++ = ' ';
            cptr++;
            break;

         case '%' :
            cptr++;
            if (*cptr >= '0' && *cptr <= '9')
               *sptr = (*cptr - '0') << 4;
            else
            if (toupper(*cptr) >= 'A' && toupper(*cptr) <= 'F')
               *sptr = (toupper(*cptr) - '7') << 4;
            else
               return (-1);
            cptr++;
            if (*cptr >= '0' && *cptr <= '9')
               *sptr |= *cptr - '0';
            else
            if (toupper(*cptr) >= 'A' && toupper(*cptr) <= 'F')
               *sptr |= toupper(*cptr) - '7';
            else
               return (-1);
            sptr++;
            cptr++;
            break;

         default :
            *sptr++ = *cptr++;
      }
   }
   *sptr = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);
   return (sptr - String);
}
 
/*****************************************************************************/
/*
String search allowing wildcard "*" (matching any multiple characters) and "%" 
(matching any single character).  Returns NULL if not found or a pointer to
start of matched string.
*/ 

char* SearchTextString
( 
register char *InThat,
register char *This,
register boolean CaseSensitive,
int *MatchedLengthPtr
)
{
/* wildcards implied at both ends of the search string */
#define IMPLIED_WILDCARDS 0

   register char  *cptr, *sptr, *inptr;
   char  *RestartCptr,
         *RestartInptr,
         *MatchPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SearchTextString()\n|%s|%s|\n", This, InThat);

   if (MatchedLengthPtr != NULL) *MatchedLengthPtr = 0;
   if (!*(cptr = This)) return (NULL);
   inptr = MatchPtr = InThat;

#if IMPLIED_WILDCARDS
   /* skip leading text up to first matching character (if any!) */
   if (*cptr != '*' && *cptr != '%')
   {
      if (CaseSensitive)
         while (*inptr && *inptr != *cptr) inptr++;
      else
         while (*inptr && toupper(*inptr) != toupper(*cptr)) inptr++;
      if (Debug && !*inptr) fprintf (stdout, "1. NOT matched!\n");
      if (!*inptr) return (NULL);
      cptr++;
      MatchPtr = inptr++;
   }
#endif /* IMPLIED_WILDCARDS */

   for (;;)
   {
      if (CaseSensitive)
      {
         while (*cptr && *inptr && *cptr == *inptr)
         {
            cptr++;
            inptr++;
         }
      }
      else
      {
         while (*cptr && *inptr && toupper(*cptr) == toupper(*inptr))
         {
            cptr++;
            inptr++;
         }
      }

#if IMPLIED_WILDCARDS
      if (!*cptr)
      {
         if (Debug) fprintf (stdout, "1. matched!\n");
         if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
         return (MatchPtr);
      }
#else
      if (!*cptr && !*inptr)
      {
         if (Debug) fprintf (stdout, "2. matched!\n");
         if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
         return (MatchPtr);
      }
      else
      if (*cptr != '*' && *cptr != '%')
         return (NULL);
#endif /* IMPLIED_WILDCARDS */

      if (Debug && !*inptr) fprintf (stdout, "3. NOT matched!\n");
      if (*cptr != '*' && !*inptr) return (NULL);

      if (*cptr != '*' && *cptr != '%')
      {
         cptr = This;
         MatchPtr = ++inptr;
         continue;
      }

      if (*cptr == '%')
      {
         /* single char wildcard processing */
         if (!*inptr) break;
         cptr++;
         inptr++;
         continue;
      }

      /* asterisk wildcard matching */
      while (*cptr == '*') cptr++;

      /* an asterisk wildcard at end matches all following */
      if (!*cptr)
      {
         if (Debug) fprintf (stdout, "4. matched!\n");
         while (*inptr) inptr++;
         if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
         return (MatchPtr);
      }

      /* note the current position in the string (first after the wildcard) */
      RestartCptr = cptr;
      for (;;)
      {
         /* find first char in InThat matching char after wildcard */
         if (CaseSensitive)
            while (*inptr && *cptr != *inptr) inptr++;
         else
            while (*inptr && toupper(*cptr) != toupper(*inptr)) inptr++;
         /* if did not find matching char in InThat being searched */
         if (Debug && !*inptr) fprintf (stdout, "5. NOT matched!\n");
         if (!*inptr) return (NULL);
         /* note the current position in InThat being searched */
         RestartInptr = inptr;
         /* try to match the remainder of the string and InThat */
         if (CaseSensitive)
         {
            while (*cptr && *inptr && *cptr == *inptr)
            {
               cptr++;
               inptr++;
            }
         }
         else
         {
            while (*cptr && *inptr && toupper(*cptr) == toupper(*inptr))
            {
               cptr++;
               inptr++;
            }
         }
         /* if reached the end of both string and InThat - match! */
#if IMPLIED_WILDCARDS
         if (!*cptr)
         {
            if (Debug) fprintf (stdout, "6. matched!\n");
            if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
            return (MatchPtr);
         }
#else
         if (!*cptr && !*inptr)
         {
            if (Debug) fprintf (stdout, "7. matched!\n");
            if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
            return (MatchPtr);
         }
#endif /* IMPLIED_WILDCARDS */
         /* break to the external loop if we encounter another wildcard */
         if (*cptr == '*' || *cptr == '%') break;
         /* lets have another go */
         cptr = RestartCptr;
         /* starting the character following the previous attempt */
         inptr = MatchPtr = RestartInptr + 1;
      }
   }
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 
 
boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}
 
/****************************************************************************/

