/*****************************************************************************/
/*
                                  Upd.c

This module implements a full multi-threaded, AST-driven, asynchronous file
and directory update facility.

It provides for directory navigation, allowing subdirectories to be created,
moved to, listed and deleted.  Files within directories may be selected for
viewing, update, copying and deletion.  The module honours the directory
access controls implemented by the directory module (e.g. ".WWW_HIDDEN",
etc., see Dir.c).

It allows the creation or update of "text/*" files via an editing page.  The
<TEXTAREA></TEXTAREA> tag widget is used as the edit window.

The file editing page also has custom functionality for providing for the
partial administration of HTTPd server configuration files.

This module behaves as if it was an external script :^).  This allows the
generic script processing functionality of the the request module to provide
all necessary parsing of the script and file components of the path, as well
as the all-important authorization checking of these.  The script component
of the path is detected before an external script is launched, and the request
redirected to this module.  This has certain efficiencies as well as allowing
a "generic" module to provide some of the functionality for server
configuration.  This module does its own query string parsing.

An essential component of this modules functionality is the ability to
generate a redirection.  This allows the parent directory and name to be
reconstructed as a path and then processed as if a new request.  Although
resulting in a higher overhead this considerably simplifies the handling of
these requests (which comprise only a very small proportion of those handled
by the server anyway :^)


VERSION HISTORY
---------------
12-MAR-98  MGD  add file protection selector
24-FEB-98  MGD  request scheme ("http:" or "https:") for hard-wired "http:"
18-OCT-97  MGD  facility to wildcard file deletes
17-AUG-97  MGD  message database,
                internationalized file date/times,
                SYSUAF-authenticated users security-profile
07-JUL-97  MGD  apparently MSIE 3.0ff allows file upload
30-JUN-97  MGD  bugfix; rename failed without SYSPRV (wonders will never)
12-APR-97  MGD  extended tree facility to providing general directory tree
25-MAR-97  MGD  added tree and filter capabilities
01-FEB-97  MGD  new for HTTPd version 4 (adapted from the UPD script)
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <atrdef.h>
#include <fibdef.h>
#include <iodef.h>
#include <libdef.h>
#include <libdtdef.h>
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application related header */
#include "wasd.h"
#include "auth.h"
#include "config.h"
#include "dir.h"
#include "error.h"
#include "httpd.h"
#include "mapurl.h"
#include "msg.h"
#include "put.h"
#include "support.h"
#include "upd.h"
#include "vm.h"

/****************/
/* global stuff */
/****************/

#define NavigateSize 7
/* this could be relocated without change by using a mapping rule */
#define UpdHelpPath "/httpd/-/UpdHelp.html"

/*
The preview note is introduced at the start of a file being previewed.
See PUT.C module for how it is further processed.
Experimentation (Navigator 3 and MSIE 3) has shown it can contain text
but no page or line layup tags (e.g. <BR>).
Here is a choice between an animated GIF and text.
*/
#define PreviewNotePlain "-PREVIEW-\n\n"

#ifndef UPD_PREVIEW_TEXT

#define PreviewNoteHtml \
"&lt;IMG SRC=&quot;/httpd/-/UpdPreview.gif&quot; \
ALT=&quot;-PREVIEW- &quot;&gt;&lt;BR&gt;"
#define PreviewNoteHtmlNewLine "&#10;"
#define PreviewNoteMenuNewLine "&lt;BR&gt;"

#else

#define PreviewNoteHtml \
"&lt;FONT SIZE=1 COLOR=&quot;#ff0000&quot;&gt;\
&lt;SUP&gt;&lt;BLINK&gt;&lt;B&gt;-PREVIEW-&lt;/B&gt;&lt;/BLINK&gt;&lt;/SUP&gt;\
&lt;/FONT&gt;&amp;nbsp;&amp;nbsp;"
#define PreviewNoteHtmlNewLine "&#10;"
#define PreviewNoteMenuNewLine "&lt;BR&gt;"

#endif /* UPD_PREVIEW_TEXT */

$DESCRIPTOR (UpdProtectionListFaoDsc,
"<SELECT NAME=protection>\n\
<OPTION VALUE=\"aa00\" SELECTED>!AZ\n\
<OPTION VALUE=\"fa00\">!AZ\n\
<OPTION VALUE=\"ff00\">!AZ\n\
</SELECT>\n");

/* must be the size of the above descriptor string plus enough for the !AZs */
char  UpdProtectionList [256];
$DESCRIPTOR (UpdProtectionListDsc, UpdProtectionList);

char  UpdLayout [64];

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern boolean  NaturalLanguageEnglish;
extern char  ClientHostName[];
extern char  ClientInternetAddress[];
extern char*  DayName[];
extern char  DirBrowsableFileName[];
extern char  DirHiddenFileName[];
extern char  DirNopFileName[];
extern char  DirNosFileName[];
extern char  DirNopsFileName[];
extern char  DirNoWildFileName[];
extern char  ErrorSanityCheck[];
extern char  HtmlSgmlDoctype[];
extern char  ServerHostPort[];
extern char  SoftwareID[];
extern char  Utility[];
extern struct AccountingStruct  Accounting;
extern struct ConfigStruct  Config;
extern struct MsgStruct  Msgs;

/*****************************************************************************/
/*
Begin processing an update request by allocating an update task structure,
processing the query string, and initiating the required function.
*/

UpdBegin
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   static $DESCRIPTOR (LocationCopyFileFaoDsc,
          "!AZ!AZ?s-copy=copy&as=!AZ&protection=!4XL\0");
   static $DESCRIPTOR (LocationDirProtectFaoDsc,
          "!AZ!AZ!AZ.dir?s-dirprotect=protect&protection=!4XL\0");
   static $DESCRIPTOR (LocationEditFileFaoDsc, "!AZ!AZ\0");
   static $DESCRIPTOR (LocationEditFileAsFaoDsc, "!AZ!AZ?as=!AZ\0");
   static $DESCRIPTOR (LocationFileProtectFaoDsc,
          "!AZ!AZ!AZ?s-fileprotect=protect&protection=!4XL\0");
   static $DESCRIPTOR (LocationFilterFaoDsc, "!AZ!AZ!AZ\0");
   static $DESCRIPTOR (LocationGotoSubDirFaoDsc, "!AZ!AZ/\0");
   static $DESCRIPTOR (LocationListSubDirFaoDsc,
          "!AZ//!AZ!AZ!AZ?httpd=index&expired=yes!AZ\0");
   static $DESCRIPTOR (LocationTreeFaoDsc, "!AZ!AZ!AZ?s-tree=tree\0");
   static $DESCRIPTOR (LocationViewFileFaoDsc, "!AZ//!AZ!AZ\0");

   register unsigned long  *vecptr;
   register char  *cptr, *qptr, *sptr, *zptr;
   register struct UpdTaskStruct  *tkptr;

   boolean  Navigate,
            NameIsPeriod,
            SubmitGoto,
            SubmitList,
            SubmitTree,
            SubmitMkdir,
            SubmitDirProtect,
            SubmitRmdir,
            SubmitView,
            SubmitEdit,
            SubmitCreate,
            SubmitFilter,
            SubmitRename,
            SubmitCopy,
            SubmitFileProtect,
            SubmitDelete;
   int  status;
   unsigned long  FaoVector [32];
   unsigned short  Length;
   char  AsName [256],
         FieldName [128],
         FieldValue [256],
         Location [512],
         Name [128],
         ParentDir [256];
   $DESCRIPTOR (LocationDsc, Location);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdBegin()\n");

   if (!rqptr->AccountingDone)
      rqptr->AccountingDone = ++Accounting.DoUpdateCount;

   if (rqptr->ErrorMessagePtr != NULL)
   {
      /* previous error, cause threaded processing to unravel */
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   /* set up the task structure (only ever one per request!) */
   rqptr->UpdTaskPtr = tkptr = (struct UpdTaskStruct*)
      VmGetHeap (rqptr, sizeof(struct UpdTaskStruct));
   tkptr->NextTaskFunction = NextTaskFunction;

   tkptr->ProtectionMask = PUT_DEFAULT_FILE_PROTECTION;
   tkptr->ParseInUse = false;

   if (strsame (rqptr->ScriptName, HttpdInternalScriptTree,
                sizeof(HttpdInternalScriptTree)-1))      
   {
      /***********/
      /* tree of */
      /***********/

      tkptr->UpdateTree = false;

      /* get the directory path from request path */
      sptr = tkptr->FilePath;
      for (cptr = rqptr->PathInfoPtr; *cptr; *sptr++ = *cptr++);
      *sptr = '\0';
      while (sptr > tkptr->FilePath && *sptr != '/') sptr--;
      if (*sptr == '/') *++sptr = '\0';

      /* get the directory from request file specification */
      sptr = tkptr->FileName;
      for (cptr = rqptr->RequestFileName; *cptr; *sptr++ = *cptr++);
      *sptr = '\0';
      for (cptr = sptr; cptr > tkptr->FileName && *cptr != ']'; cptr--);
      if (*cptr == ']') cptr++;
      tkptr->FileNameLength = cptr - tkptr->FileName;

      if (!tkptr->FileNameLength)
      {
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_DIRECTORY), FI_LI);
         UpdEnd (rqptr);
         return;
      }

      /* get any file name, type, version from request specification */
      sptr = tkptr->FileNamePart;
      if (*cptr)
      {
         *sptr++ = *cptr;
         *cptr++ = '\0';
      }
      while (*cptr && *cptr != ';') *sptr++ = *cptr++;
      if (*cptr == ';' && *(cptr+1))
         while (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';

      if (Debug)
         fprintf (stdout, "|%s|%s|\n", tkptr->FileName, tkptr->FileNamePart);
      
      rqptr->ResponsePreExpired = Config.DirPreExpired;
      /* scan the query string looking for "expired=[yes|no|true|false|1|0]" */
      for (cptr = rqptr->QueryStringPtr; *cptr; cptr++)
      {
         /* experience shows both make it easier! */
         if (tolower(*cptr) == 'e' && 
             (strsame (cptr, "expired=", 8)) || strsame (cptr, "expire=", 7))
         {
            if (cptr[7] == '=')
               cptr += 8;
            else
               cptr += 7;
            /* "true", "yes", "1", "false", "no" or "0" */
            if (tolower(*cptr) == 't' || tolower(*cptr) == 'y' || *cptr == '1')
               rqptr->ResponsePreExpired = true;
            else
            if (tolower(*cptr) == 'f' || tolower(*cptr) == 'n' || *cptr == '0')
               rqptr->ResponsePreExpired = false;
            break;
         }
         while (*cptr && *cptr != '&') cptr++;
         if (*cptr) cptr++;
      }

      UpdTreeBegin (rqptr);

      return;
   }

   /**********************/
   /* parse query string */
   /**********************/

   AsName[0] = Name[0] = ParentDir[0] = tkptr->CxR[0] = '\0';
   SubmitGoto = SubmitList = SubmitTree =
      SubmitMkdir = SubmitDirProtect = SubmitRmdir =
      SubmitView = SubmitEdit = SubmitCreate = SubmitFilter =
      SubmitRename = SubmitCopy = SubmitFileProtect = SubmitDelete = false;

   if (rqptr->HttpMethod == HTTP_METHOD_GET ||
       rqptr->HttpMethod == HTTP_METHOD_HEAD)
      qptr = rqptr->QueryStringPtr;
   else
      qptr = rqptr->ContentBufferPtr;
   while (*qptr)
   {
      qptr = ParseQueryField (rqptr, qptr,
                              FieldName, sizeof(FieldName),
                              FieldValue, sizeof(FieldValue),
                              FI_LI);
      if (qptr == NULL)
      {
         /* error occured */
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }

      /********************/
      /* get field values */
      /********************/

      if (strsame (FieldName, "as", -1))
      {
         cptr = FieldValue;
         zptr = (sptr = AsName) + sizeof(AsName);
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         if (sptr >= zptr)
         {
            ErrorGeneralOverflow (rqptr, FI_LI);
            UpdEnd (rqptr);
            return;
         }
         *sptr = '\0';
      }
      else
      if (strsame (FieldName, "cxr", -1))
      {
         cptr = FieldValue;
         zptr = (sptr = tkptr->CxR) + sizeof(tkptr->CxR);
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         if (sptr >= zptr)
         {
            ErrorGeneralOverflow (rqptr, FI_LI);
            UpdEnd (rqptr);
            return;
         }
         *sptr = '\0';
      }
      else
      if (strsame (FieldName, "name", -1))
      {
         cptr = FieldValue;
         zptr = (sptr = Name) + sizeof(Name);
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         if (sptr >= zptr)
         {
            ErrorGeneralOverflow (rqptr, FI_LI);
            UpdEnd (rqptr);
            return;
         }
         *sptr = '\0';
      }
      else
      if (strsame (FieldName, "parent", -1))
      {
         cptr = FieldValue;
         zptr = (sptr = ParentDir) + sizeof(ParentDir);
         while (*cptr && sptr < zptr) *sptr++ = *cptr++;
         if (sptr >= zptr)
         {
            ErrorGeneralOverflow (rqptr, FI_LI);
            UpdEnd (rqptr);
            return;
         }
         *sptr = '\0';
      }
      else
      if (strsame (FieldName, "protection", -1))
      {
         if (isxdigit(FieldValue[0]))
            tkptr->ProtectionMask = strtol (FieldValue, NULL, 16);
      }
      else
      if (tolower(FieldName[0]) == 's' && FieldName[1] == '-')
      {
         if (strsame (FieldName, "s-goto", -1))
            SubmitGoto = true;
         else
         if (strsame (FieldName, "s-list", -1))
            SubmitList = true;
         else
         if (strsame (FieldName, "s-tree", -1))
            SubmitTree = true;
         else
         if (strsame (FieldName, "s-mkdir", -1))
            SubmitMkdir = true;
         else
         if (strsame (FieldName, "s-dirprotect", -1))
            SubmitDirProtect = true;
         else
         if (strsame (FieldName, "s-rmdir", -1))
            SubmitRmdir = true;
         else
         if (strsame (FieldName, "s-view", -1))
            SubmitView = true;
         else
         if (strsame (FieldName, "s-edit", -1))
            SubmitEdit = true;
         else
         if (strsame (FieldName, "s-create", -1))
            SubmitCreate = true;
         else
         if (strsame (FieldName, "s-filter", -1))
            SubmitFilter = true;
         else
         if (strsame (FieldName, "s-rename", -1))
            SubmitRename = true;
         else
         if (strsame (FieldName, "s-copy", -1))
            SubmitCopy = true;
         else
         if (strsame (FieldName, "s-fileprotect", -1))
            SubmitFileProtect = true;
         else
         if (strsame (FieldName, "s-delete", -1))
            SubmitDelete = true;
         else
         {
            rqptr->ResponseStatusCode = 403;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_QUERY_FIELD), FI_LI);
            UpdEnd (rqptr);
            return;
         }
      }
      else
      {
         rqptr->ResponseStatusCode = 403;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_QUERY_FIELD), FI_LI);
         UpdEnd (rqptr);
         return;
      }
   }

   /**************************/
   /* end parse query string */
   /**************************/

   /* ensure more than spaces have been entered into the file name field */
   for (cptr = sptr = Name; *cptr && ISLWS(*cptr); cptr++);
   while (*cptr && !ISLWS(*cptr)) *sptr++ = *cptr++;
   *sptr = '\0';
   if (Name[0] == '.' && Name[1] == '\0')
   {
      NameIsPeriod = true;
      Name[0] = '\0';
   }
   else
      NameIsPeriod = false;

   /* reduce size of 'FilePath' to allow for possible '/' concatenation */
   if (MapUrl_VirtualPath (ParentDir, Name,
                           tkptr->FilePath, sizeof(tkptr->FilePath)-1) < 0)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      UpdEnd (rqptr);
      return;
   }

   if (AsName[0])
   {
      /* reduce size of 'AsPath' to allow for possible '/' concatenation */
      if (MapUrl_VirtualPath (ParentDir, AsName,
                              tkptr->AsPath, sizeof(tkptr->AsPath)-1) < 0)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         UpdEnd (rqptr);
         return;
      }
   }

   if (SubmitDelete)
   {
      /***************/
      /* delete file */
      /***************/

      if (!Name[0] && !AsName[0])
      {
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_FILENAME), FI_LI);
         UpdEnd (rqptr);
         return;
      }
      UpdConfirmDelete (rqptr);
      return;
   }

   if (SubmitMkdir)
   {
      /***********************/
      /* create subdirectory */
      /***********************/

      if (!AsName[0])
      {
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_DIRECTORY), FI_LI);
         UpdEnd (rqptr);
         return;
      }
      strcat (tkptr->AsPath, "/");
      UpdConfirmCreate (rqptr);
      return;
   }

   if (SubmitRmdir)
   {
      /***********************/
      /* delete subdirectory */
      /***********************/

      if (!Name[0])
      {
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_DIRECTORY), FI_LI);
         UpdEnd (rqptr);
         return;
      }
      strcat (tkptr->FilePath, "/");
      UpdConfirmDelete (rqptr);
      return;
   }

   if (ParentDir[0] || SubmitFilter)
   {
      /******************************/
      /* redirect (via "Location:") */
      /******************************/

      if (SubmitFilter)
      {
         if (!Name[0])
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_FILENAME), FI_LI);
            UpdEnd (rqptr);
            return;
         }
         if (!AsName[0])
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_FILTER), FI_LI);
            UpdEnd (rqptr);
            return;
         }
         if (AsName[0] == '/')
            status = sys$fao (&LocationFilterFaoDsc, &Length, &LocationDsc,
                              "", AsName, tkptr->FilePath);
         else
            status = sys$fao (&LocationFilterFaoDsc, &Length, &LocationDsc,
                              "/", AsName, tkptr->FilePath);
      }
      else
      if (SubmitCopy)
      {
         if (!Name[0])
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_SRC_FILENAME), FI_LI);
            UpdEnd (rqptr);
            return;
         }
         if (!AsName[0])
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_DST_FILENAME), FI_LI);
            UpdEnd (rqptr);
            return;
         }
         status = sys$fao (&LocationCopyFileFaoDsc, &Length, &LocationDsc,
                           rqptr->ScriptName, tkptr->FilePath, tkptr->AsPath,
                           tkptr->ProtectionMask);
      }
      else
      if (SubmitCreate)
      {
         if (!AsName[0])
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_FILENAME), FI_LI);
            UpdEnd (rqptr);
            return;
         }
         status = sys$fao (&LocationEditFileFaoDsc, &Length, &LocationDsc,
                           rqptr->ScriptName, tkptr->AsPath);
      }
      else
      if (SubmitEdit)
      {
         if (!Name[0])
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_FILENAME), FI_LI);
            UpdEnd (rqptr);
            return;
         }
         if (AsName[0])
            status = sys$fao (&LocationEditFileAsFaoDsc, &Length, &LocationDsc,
                            rqptr->ScriptName, tkptr->FilePath, tkptr->AsPath);
         else
            status = sys$fao (&LocationEditFileFaoDsc, &Length, &LocationDsc,
                              rqptr->ScriptName, tkptr->FilePath);
      }
      else
      if (SubmitDirProtect)
      {
         if (!Name[0])
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_DIRECTORY), FI_LI);
            UpdEnd (rqptr);
            return;
         }
         status = sys$fao (&LocationDirProtectFaoDsc, &Length, &LocationDsc,
                            rqptr->ScriptName, ParentDir, Name,
                            tkptr->ProtectionMask);
      }
      else
      if (SubmitFileProtect)
      {
         if (!Name[0])
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_FILENAME), FI_LI);
            UpdEnd (rqptr);
            return;
         }
         status = sys$fao (&LocationFileProtectFaoDsc, &Length, &LocationDsc,
                            rqptr->ScriptName, ParentDir, Name,
                            tkptr->ProtectionMask);
      }
      else
      if (SubmitGoto)
      {
         if (!Name[0])
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_DIRECTORY), FI_LI);
            UpdEnd (rqptr);
            return;
         }
         status = sys$fao (&LocationGotoSubDirFaoDsc, &Length, &LocationDsc,
                           rqptr->ScriptName, tkptr->FilePath);
      }
      else
      if (SubmitList)
      {
         if (!Name[0] && !NameIsPeriod)
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_DIRECTORY), FI_LI);
            UpdEnd (rqptr);
            return;
         }

         /* update directory listing layout including protection */
         if (!UpdLayout[0])
         {
            zptr = (sptr = UpdLayout) + sizeof(UpdLayout);
            for (cptr = "&layout="; *cptr; *sptr++ = *cptr++);

            if (Config.DefaultDirLayout[0])
               cptr = Config.DefaultDirLayout;
            else
               cptr = DEFAULT_DIR_LAYOUT;

            while (*cptr)
            {
               if (sptr >= zptr) break;
               if (*cptr == ':')
               {
                  if (sptr < zptr) *sptr++ = *cptr++;
                  if (*cptr && sptr < zptr) *sptr++ = *cptr++;
                  continue;
               }
               if (toupper(*cptr) == 'P')
               {
                  cptr++;
                  while (*cptr == '_') cptr++;
                  continue;
               }
               if (toupper(*cptr) == 'S')
               {
                  if (sptr < zptr) *sptr++ = *cptr++;
                  if (*cptr == ':')
                  {
                     *sptr++ = *cptr++;
                     if (*cptr && sptr < zptr) *sptr++ = *cptr++;
                  }
                  if (sptr < zptr) *sptr++ = '_';
                  if (sptr < zptr) *sptr++ = '_';
                  if (sptr < zptr) *sptr++ = 'P';
                  while (*cptr == '_' && sptr < zptr) *sptr++ = *cptr++;
                  continue;
               }
               if (sptr < zptr) *sptr++ = *cptr++;
            }
            if (sptr >= zptr)
            {
               UpdLayout[0] = '\0';
               ErrorGeneralOverflow (rqptr, FI_LI);
               UpdEnd (rqptr);
               return;
            }
            *sptr = '\0';
            if (Debug) fprintf (stdout, "UpdLayout |%s|\n", UpdLayout);
         }

         if (Name[0])
            status = sys$fao (&LocationListSubDirFaoDsc, &Length, &LocationDsc,
                              rqptr->RequestSchemeNamePtr,
                              rqptr->ServicePtr->ServerHostPort,
                              tkptr->FilePath, "/*.*", UpdLayout);
         else
            status = sys$fao (&LocationListSubDirFaoDsc, &Length, &LocationDsc,
                              rqptr->RequestSchemeNamePtr,
                              rqptr->ServicePtr->ServerHostPort,
                              tkptr->FilePath, "*.*", UpdLayout);
      }
      else
      if (SubmitRename)
      {
         if (!Name[0])
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_CUR_FILENAME), FI_LI);
            UpdEnd (rqptr);
            return;
         }
         if (!AsName[0])
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_NEW_FILENAME), FI_LI);
            UpdEnd (rqptr);
            return;
         }
         for (cptr = AsName; *cptr && *cptr != '/'; cptr++);
         if (*cptr == '/')
         {
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_RENAME_SAME), FI_LI);
            UpdEnd (rqptr);
            return;
         }

         UpdConfirmRename (rqptr, ParentDir, Name, AsName);
         return;
      }
      else
      if (SubmitTree)
      {
         if (!Name[0] && !NameIsPeriod)
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_DIRECTORY), FI_LI);
            UpdEnd (rqptr);
            return;
         }
         if (Name[0])
            status = sys$fao (&LocationTreeFaoDsc, &Length, &LocationDsc,
                              rqptr->ScriptName, tkptr->FilePath, "/");
         else
            status = sys$fao (&LocationTreeFaoDsc, &Length, &LocationDsc,
                              rqptr->ScriptName, tkptr->FilePath, "");
      }
      else
      if (SubmitView)
      {
         if (!Name[0])
         {
            rqptr->ResponseStatusCode = 400;
            ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_FILENAME), FI_LI);
            UpdEnd (rqptr);
            return;
         }
         status = sys$fao (&LocationViewFileFaoDsc, &Length, &LocationDsc,
                           rqptr->RequestSchemeNamePtr,
                           rqptr->ServicePtr->ServerHostPort,
                           tkptr->FilePath);
      }
      else
      {
         rqptr->ResponseStatusCode = 400;
         ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_ACTION), FI_LI);
         UpdEnd (rqptr);
         return;
      }

      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         UpdEnd (rqptr);
         return;
      }
      if (Debug) fprintf (stdout, "Location |%s|\n", Location);

      rqptr->LocationPtr = VmGetHeap (rqptr, Length);
      memcpy (rqptr->LocationPtr, Location, Length);
      UpdEnd (rqptr);
      return;
   }

   /********************************/
   /* navigate, edit, copy, rename */
   /********************************/

   cptr = rqptr->PathInfoPtr;
   zptr = (sptr = tkptr->FilePath) + sizeof(tkptr->FilePath);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   *sptr = '\0';

   /* check if the path ends in a slash (i.e. a directory specification) */
   if (cptr > rqptr->PathInfoPtr && cptr[-1] == '/')
      Navigate = true;
   else
      Navigate = false;

   cptr = rqptr->RequestFileName;
   zptr = (sptr = tkptr->FileName) + sizeof(tkptr->FileName);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   *sptr = '\0';
   if (Debug) fprintf (stdout, "tkptr->FileName |%s|\n", tkptr->FileName);

   if (Navigate)
   {
      /* ensure the file "name" ends at the directory */
      while (sptr > tkptr->FileName && *sptr != ']') sptr--;
      if (*sptr == ']') *++sptr = '\0';
   }

   tkptr->FileNameLength = sptr - tkptr->FileName;

   if (SubmitFileProtect || SubmitDirProtect)
   {
      /*********************/
      /* change protection */
      /*********************/

      UpdProtection (rqptr);
      UpdEnd (rqptr);
      return;
   }

   if (SubmitRename)
   {
      /**********/
      /* rename */
      /**********/

      if (!Name[0] || !AsName[0])
      {
         ErrorInternal (rqptr, 0, ErrorSanityCheck, FI_LI);
         UpdEnd (rqptr);
         return;
      }
      if (rqptr->HttpMethod == HTTP_METHOD_POST)
         UpdRenameFile (rqptr, Name, AsName);
      else
      {
         ErrorInternal (rqptr, 0, ErrorSanityCheck, FI_LI);
         UpdEnd (rqptr);
      }
      return;
   }

   if (SubmitTree)
   {
      /********/
      /* tree */
      /********/

      tkptr->UpdateTree = true;
      rqptr->ResponsePreExpired = PRE_EXPIRE_UPD_TREE_OF;
      UpdTreeBegin (rqptr);
      return;
   }

   if (SubmitCopy)
   {
      /********/
      /* copy */
      /********/

      UpdCopyFileBegin (rqptr);
      return;
   }

   if (Navigate)
   {
      /******************/
      /* let's navigate */
      /******************/

      UpdNavigateBegin (rqptr);
      return;
   }

   /*****************/
   /* edit the file */
   /*****************/

   UpdEditFileBegin (rqptr);
}

/*****************************************************************************/
/*
*/ 

UpdEnd (struct RequestStruct *rqptr)

{
   register struct UpdTaskStruct  *tkptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdEnd()\n");

   tkptr = rqptr->UpdTaskPtr;

   if (tkptr->ParseInUse)
   {
      /* ensure parse internal data structures are released */
      tkptr->SearchFab.fab$l_fna = "a:[b]c.d;";
      tkptr->SearchFab.fab$b_fns = 9;
      tkptr->SearchFab.fab$b_dns = 0;
      tkptr->SearchNam.nam$b_nop = NAM$M_SYNCHK;
      sys$parse (&tkptr->SearchFab, 0, 0);
   }

   if (tkptr->FileFab.fab$w_ifi)  sys$close (&tkptr->FileFab, 0, 0);

   SysDclAst (tkptr->NextTaskFunction, rqptr);
}

/*****************************************************************************/
/*
Begin by setting up a search for all directories.
*/

UpdNavigateBegin (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ !AZ</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ !AZ</H1>\n\
\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
\
<TR><TH COLSPAN=2>\n\
<FONT SIZE=+2><A HREF=\"!AZ\">!AZ</A></FONT>\n\
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;\n\
[<A HREF=\"!AZ\">!AZ</A>]\n\
</TH></TR>\n\
\
<TR><TH><FONT SIZE=+1>!AZ</FONT></TH>\
<TH><FONT SIZE=+1>!AZ</FONT></TH></TR>\n\
\
<TR>\n\
\
<TD VALIGN=top>\n\
<FORM METHOD=GET ACTION=\"!AZ/\">\n\
<INPUT TYPE=hidden NAME=parent VALUE=\"!AZ\">\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD VALIGN=top ALIGN=left>\n\
<SELECT SIZE=!UL NAME=name>\n\
<OPTION VALUE=\".\" SELECTED>./\n");

   register unsigned long  *vecptr;
   register char  *cptr, *sptr, *zptr;
   register struct UpdTaskStruct  *tkptr;

   boolean  ok;
   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [2048];
   void  *AstFunctionPtr;
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdNavigateBegin()\n");

   tkptr = rqptr->UpdTaskPtr;

   /***************************/
   /* initialize if necessary */
   /***************************/

   if (!UpdProtectionList[0])
   {
      strcpy (cptr = Buffer, MsgFor (rqptr, MSG_UPD_PROTECTION_LIST));

      vecptr = FaoVector;
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
      *vecptr++ = cptr;

      status = sys$faol (&UpdProtectionListFaoDsc, &Length,
                         &UpdProtectionListDsc, &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         UpdEnd (rqptr);
         return;
      }
      UpdProtectionList[Length] = '\0';
   }

   /*****************************************/
   /* check VMS-authenticated user's access */
   /*****************************************/

   if (rqptr->AuthVmsUserProfileLength)
   {
      status = AuthCheckVmsUserAccess (rqptr, tkptr->FileName,
                                       tkptr->FileNameLength);
      if (status == RMS$_PRV)
         tkptr->AuthVmsUserHasAccess = false;
      else
      if (VMSok (status))
         tkptr->AuthVmsUserHasAccess = true;
      else
      {
         /* error reported by access check */
         UpdEnd (rqptr);
         return;
      }
   }
   else
      tkptr->AuthVmsUserHasAccess = false;

   /************/
   /* navigate */
   /************/

   /* first check if the directory can be accessed */

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   if (Config.DirAccessSelective)
      status = FileExists (tkptr->FileName, DirBrowsableFileName);
   else
   {
      if (VMSok (status = FileExists (tkptr->FileName, DirHiddenFileName)))
         status = RMS$_FNF;
      else
      if (VMSok (status = FileExists (tkptr->FileName, DirNoWildFileName)))
         status = RMS$_WLD;
      else
         status = SS$_NORMAL;
   }

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      /* give some "disinformation"  ;^)  */
      if (status == RMS$_FNF) status = RMS$_DNF;
      rqptr->ErrorTextPtr = tkptr->FilePath;
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }

   /* check if we can see its subdirectories, if not then straight to files */

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   if (VMSnok (status = FileExists (tkptr->FileName, DirNosFileName)))
      status = FileExists (tkptr->FileName, DirNopsFileName);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok(status))
   {
      AstFunctionPtr = &UpdNavigateSearchDirs;

      /***************************/
      /* set up directory search */
      /***************************/

      tkptr->ParseInUse = true;

      /* set up ready to search for all files ending in ".DIR" (directories) */
      tkptr->SearchFab = cc$rms_fab;
      /* set the FAB user context to the client thread pointer */
      tkptr->SearchFab.fab$l_ctx = rqptr;
      tkptr->SearchFab.fab$l_dna = "*.DIR;";
      tkptr->SearchFab.fab$b_dns = 6;
      tkptr->SearchFab.fab$l_fna = tkptr->FileName;
      tkptr->SearchFab.fab$b_fns = tkptr->FileNameLength;
      tkptr->SearchFab.fab$l_fop = FAB$M_NAM;
      tkptr->SearchFab.fab$l_nam = &tkptr->SearchNam;

      tkptr->SearchNam = cc$rms_nam;
      tkptr->SearchNam.nam$l_esa = tkptr->ExpandedFileName;
      tkptr->SearchNam.nam$b_ess = sizeof(tkptr->ExpandedFileName)-1;
      tkptr->SearchNam.nam$l_rsa = tkptr->ResultantFileName;
      tkptr->SearchNam.nam$b_rss = sizeof(tkptr->ResultantFileName)-1;

      if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

      status = sys$parse (&tkptr->SearchFab, 0, 0);
      if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);

      if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

      if (VMSnok (status))
      {
         rqptr->ErrorTextPtr = tkptr->FilePath;
         rqptr->ErrorHiddenTextPtr = tkptr->FileName;
         ErrorVmsStatus (rqptr, status, FI_LI);
         UpdEnd (rqptr);
         return;
      }
   }
   else
      AstFunctionPtr = &UpdNavigateBeginFiles;

   /***************************/
   /* some browser specifics! */
   /***************************/

   tkptr->BrowserSupportsFileUpload = false;
   if (rqptr->HttpUserAgentPtr != NULL)
   {
      /* Netscape Navigator */
      if (strsame (rqptr->HttpUserAgentPtr, "Mozilla", 7))
         tkptr->BrowserSupportsFileUpload = true;
      /* MS Internet Explorer is "mozilla" compatible :^) */
      if (strstr (rqptr->HttpUserAgentPtr, "MSIE") != NULL)
         tkptr->BrowserSupportsFileUpload = true;
   }

   /**************/
   /* begin page */
   /**************/

   cptr = MsgFor(rqptr,MSG_UPD_NAVIGATE);
   zptr = (sptr = tkptr->MsgString) + sizeof(tkptr->MsgString);
   while (*cptr && sptr < zptr)
   {
      /* don't need any extraneous linefeeds from this long string */
      if (*cptr == '\n')
         cptr++;
      else
         *sptr++ = *cptr++;
   }
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   *sptr = '\0';

   rqptr->ResponsePreExpired = PRE_EXPIRE_UPD;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      UpdEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, tkptr->FileName);

   /* "Update" */
   *vecptr++ = cptr = tkptr->MsgString;

   *vecptr++ = rqptr->ServicePtr->ServerHostPort;

   /* "Update" again */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   *vecptr++ = rqptr->ServicePtr->ServerHostPort;

   *vecptr++ = tkptr->FilePath;
   *vecptr++ = tkptr->FilePath;

   *vecptr++ = UpdHelpPath;

   /* ["help"] */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* "subdirectories" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* "files" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* save the current position for use in later parts of the page */
   tkptr->MsgStringPtr = cptr;

   *vecptr++ = rqptr->ScriptName;
   *vecptr++ = rqptr->PathInfoPtr;
   *vecptr++ = NavigateSize;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   Buffer[Length] = '\0';

   tkptr->FileCount = 0;

   NetWriteBuffered (rqptr, AstFunctionPtr, Buffer, Length);
}

/*****************************************************************************/
/*
(AST) function to invoke another sys$search() call when listing directories.
*/ 

UpdNavigateSearchDirs (struct RequestStruct *rqptr)

{
   register struct UpdTaskStruct  *tkptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NavigateSearchDirs()\n");

   /* get the pointer to the task structure */
   tkptr = rqptr->UpdTaskPtr;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   /* call RMS directory search routine, AST completion to DirDirectories() */
   sys$search (&tkptr->SearchFab, &UpdNavigateDirs, &UpdNavigateDirs);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();
}

/*****************************************************************************/
/*
AST completion routine called each time sys$search() completes.  It will 
either point to another file name found or have "no more files found" status 
(or an error!).
*/ 

UpdNavigateDirs (struct FAB *FabPtr)

{
   register char  *cptr, *sptr;
   register struct RequestStruct  *rqptr;
   register struct UpdTaskStruct  *tkptr;

   int  status;
   char  c;
   char  Buffer [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
      "UpdNavigateDirs() sts: %%X%08.08X stv: %%X%08.08X\n",
      FabPtr->fab$l_sts, FabPtr->fab$l_stv);

   /* retrieve the pointer to the client thread from the FAB user context */
   rqptr = FabPtr->fab$l_ctx;
   /* get the pointer to the task structure */
   tkptr = rqptr->UpdTaskPtr;

   if (VMSnok (status = tkptr->SearchFab.fab$l_sts))
   {
      /* if its a search list treat directory not found as if file not found */
      if ((tkptr->SearchNam.nam$l_fnb & NAM$M_SEARCH_LIST) &&
          status == RMS$_DNF)
         status = RMS$_FNF;

      if (status == RMS$_FNF || status == RMS$_NMF)
      {
         /* end of directory search, list files */
         tkptr->ParseInUse = false;
         UpdNavigateBeginFiles (rqptr);
         return;
      }

      /* sys$search() error */
      rqptr->ErrorTextPtr = tkptr->FilePath;
      rqptr->ErrorHiddenTextPtr = tkptr->SearchNam.nam$l_dev;
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }

   /* terminate following the last character in the version number */
   tkptr->SearchNam.nam$l_ver[tkptr->SearchNam.nam$b_ver] = '\0';
   if (Debug) fprintf (stdout, "Directory |%s|\n", tkptr->ResultantFileName);

   if (!memcmp (tkptr->SearchNam.nam$l_name, "000000.", 7))
   {
      /* not interested in master file directories :^) */
      UpdNavigateSearchDirs (rqptr);
      return;
   }

   *(tkptr->SearchNam.nam$l_name-1) = '.';
   *tkptr->SearchNam.nam$l_type = ']';
   c = *(tkptr->SearchNam.nam$l_type+1);
   *(tkptr->SearchNam.nam$l_type+1) = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", tkptr->SearchNam.nam$l_rsa);

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   if (Config.DirAccessSelective)
      status = FileExists (tkptr->SearchNam.nam$l_rsa, DirBrowsableFileName);
   else
   {
      if (VMSok (status =
          FileExists (tkptr->SearchNam.nam$l_rsa, DirHiddenFileName)))
         status = STS$K_ERROR;
      else
      if (status == RMS$_FNF)
         status = SS$_NORMAL;
   }

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   *(tkptr->SearchNam.nam$l_name-1) = ']';
   *tkptr->SearchNam.nam$l_type = '.';
   *(tkptr->SearchNam.nam$l_type+1) = c;

   if (VMSnok (status))
   {
      /* no, it shouldn't/couldn't be accessed */
      UpdNavigateSearchDirs (rqptr);
      return;
   }

   tkptr->FileCount++;

   sptr = Buffer;
   strcpy (sptr, "<OPTION VALUE=\"");
   sptr += 15;
   for (cptr = tkptr->SearchNam.nam$l_name;
        *cptr && *cptr != '.';
        *sptr++ = tolower(*cptr++));
   *sptr++ = '\"';
   *sptr++ = '>';
   for (cptr = tkptr->SearchNam.nam$l_name;
        *cptr && *cptr != '.';
        *sptr++ = tolower(*cptr++));
   *sptr++ = '\n';
   *sptr = '\0';

   NetWriteBuffered (rqptr, &UpdNavigateSearchDirs, Buffer, sptr - Buffer);
}

/*****************************************************************************/
/*
End directory search.  Begin file search.
*/

UpdNavigateBeginFiles (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (NavigateFilesFaoDsc,
"</SELECT>\n\
<BR><SUP>1.</SUP> !AZ\n\
<BR><SUP>2.</SUP> !AZ\n\
<BR><INPUT TYPE=text NAME=as SIZE=15>\n\
!AZ\
<BR><INPUT TYPE=reset VALUE=\" !AZ \">\n\
</TD><TD ALIGN=left VALIGN=top>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR ALIGN=left><TD><INPUT TYPE=submit NAME=s-goto VALUE=\" !AZ \"></TD>\
<TD><SUP>1.</SUP></TD></TR>\n\
<TR ALIGN=left><TD><INPUT TYPE=submit NAME=s-list VALUE=\" !AZ \"></TD>\
<TD><SUP>1.</SUP></TD></TR>\n\
<TR ALIGN=left><TD><INPUT TYPE=submit NAME=s-tree VALUE=\" !AZ \"></TD>\
<TD><SUP>1.</SUP></TD></TR>\n\
<TR ALIGN=left><TD><INPUT TYPE=submit NAME=s-mkdir VALUE=\" !AZ \"></TD>\
<TD><SUP>2.</SUP></TD></TR>\n\
<TR ALIGN=left><TD><INPUT TYPE=submit NAME=s-dirprotect VALUE=\" !AZ \"></TD>\
<TD><SUP>1./2.</SUP></TD></TR>\n\
<TR ALIGN=left><TD><INPUT TYPE=submit NAME=s-rmdir VALUE=\" !AZ \"></TD>\
<TD><SUP>1.</SUP></TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
</FORM>\n\
</TD>\n\
\
<TD VALIGN=top>\n\
<FORM METHOD=GET ACTION=\"!AZ/\">\n\
<INPUT TYPE=hidden NAME=parent VALUE=\"!AZ\">\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD VALIGN=top ALIGN=left>\n\
<SELECT SIZE=!UL NAME=name>\n");

   register unsigned long  *vecptr;
   register char  *cptr, *sptr, *zptr;
   register struct UpdTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [2048];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdNavigateBeginFiles()\n");

   tkptr = rqptr->UpdTaskPtr;

   tkptr->ParseInUse = true;

   /* set up ready to search for the highest version of all files */
   tkptr->SearchFab = cc$rms_fab;
   /* set the FAB user context to the client thread pointer */
   tkptr->SearchFab.fab$l_ctx = rqptr;
   tkptr->SearchFab.fab$l_dna = "*.*;0";
   tkptr->SearchFab.fab$b_dns = 5;
   tkptr->SearchFab.fab$l_fna = tkptr->FileName;
   tkptr->SearchFab.fab$b_fns = tkptr->FileNameLength;
   tkptr->SearchFab.fab$l_fop = FAB$M_NAM;
   tkptr->SearchFab.fab$l_nam = &tkptr->SearchNam;

   tkptr->SearchNam = cc$rms_nam;
   tkptr->SearchNam.nam$l_esa = tkptr->ExpandedFileName;
   tkptr->SearchNam.nam$b_ess = sizeof(tkptr->ExpandedFileName)-1;
   tkptr->SearchNam.nam$l_rsa = tkptr->ResultantFileName;
   tkptr->SearchNam.nam$b_rss = sizeof(tkptr->ResultantFileName)-1;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   status = sys$parse (&tkptr->SearchFab, 0, 0);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
      rqptr->ErrorTextPtr = tkptr->FilePath;
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }

   /**************/
   /* begin page */
   /**************/

   vecptr = FaoVector;

   /* retrieve the previously saved position in the string */
   cptr = tkptr->MsgStringPtr;

   if (tkptr->FileCount)
   {
      /* "select from list" */
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
      /* skip over "none available" */
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) cptr++;
   }
   else
   {
      /* skip over "select from list" */
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) cptr++;
      /* "none available" */
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
   }

   /* "enter name" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* file protection */
   *vecptr++ = UpdProtectionList;

   /* reset */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* goto */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* list */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* tree */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* create */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* protection */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* delete */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* save the current position for use in later parts of the page */
   tkptr->MsgStringPtr = cptr;

   *vecptr++ = rqptr->ScriptName;
   *vecptr++ = tkptr->FilePath;
   *vecptr++ = NavigateSize;

   status = sys$faol (&NavigateFilesFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, &UpdNavigateSearchFiles, Buffer, Length);

   tkptr->FileCount = 0;
}

/*****************************************************************************/
/*
(AST) function to invoke another sys$search() call when listing Files.
*/ 

UpdNavigateSearchFiles (struct RequestStruct *rqptr)

{
   register struct UpdTaskStruct  *tkptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdNavigateSearchFiles()\n");

   /* get the pointer to the task structure */
   tkptr = rqptr->UpdTaskPtr;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   /* call RMS directory search routine, AST completion to DirFiles() */
   sys$search (&tkptr->SearchFab, &UpdNavigateFiles, &UpdNavigateFiles);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();
}

/*****************************************************************************/
/*
AST completion routine called each time sys$search() completes.  It will 
either point to another file name found or have "no more files found" status 
(or an error!).
*/ 

UpdNavigateFiles (struct FAB *FabPtr)

{
   register char  *cptr, *sptr;
   register struct RequestStruct  *rqptr;
   register struct UpdTaskStruct  *tkptr;

   int  status;
   char  Buffer [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
      "UpdNavigateFiles() sts: %%X%08.08X stv: %%X%08.08X\n",
      FabPtr->fab$l_sts, FabPtr->fab$l_stv);

   /* retrieve the pointer to the client thread from the FAB user context */
   rqptr = FabPtr->fab$l_ctx;
   /* get the pointer to the task structure */
   tkptr = rqptr->UpdTaskPtr;

   if (VMSnok (status = tkptr->SearchFab.fab$l_sts))
   {
      if (status == RMS$_FNF || status == RMS$_NMF)
      {
         /* end of file search */
         tkptr->ParseInUse = false;
         UpdNavigateEnd (rqptr);
         return;
      }

      /* sys$search() error */
      rqptr->ErrorTextPtr = tkptr->FilePath;
      rqptr->ErrorHiddenTextPtr = tkptr->SearchNam.nam$l_dev;
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }

   /* terminate following the last character in the version number */
   tkptr->SearchNam.nam$l_ver[tkptr->SearchNam.nam$b_ver] = '\0';
   if (Debug) fprintf (stdout, "File |%s|\n", tkptr->ResultantFileName);

   if (!memcmp (tkptr->SearchNam.nam$l_type, ".DIR;", 5))
   {
      /* already have directories */
      UpdNavigateSearchFiles (rqptr);
      return;
   }

   if (*tkptr->SearchNam.nam$l_name == '.')
   {
      /* "hidden" file, let it stay hidden */
      UpdNavigateSearchFiles (rqptr);
      return;
   }

   /* specially check permissions for each file! */
   if (rqptr->AuthVmsUserProfileLength)
   {
      /* check if SYSUAF-authenticated user has access */
      status = AuthCheckVmsUserAccess (rqptr, tkptr->SearchNam.nam$l_rsa,
                                              tkptr->SearchNam.nam$b_rsl);
      if (status == RMS$_PRV)
      {
         /* does not have access */
         UpdNavigateSearchFiles (rqptr);
         return;
      }
      else
      if (VMSnok (status))
      {
         /* error reported by access check */
         UpdEnd (rqptr);
         return;
      }
   }
   else
   {
      /* check if HTTPd server account has access */
      status = AuthCheckVmsUserAccess (NULL, tkptr->SearchNam.nam$l_rsa,
                                             tkptr->SearchNam.nam$b_rsl);
      if (status == RMS$_PRV)
      {
         /* does not have access */
         UpdNavigateSearchFiles (rqptr);
         return;
      }
      else
      if (VMSnok (status))
      {
         /* error reported by access check (reported here due to NULL) */
         rqptr->ErrorTextPtr = "AuthCheckVmsUserAccess()";
         rqptr->ErrorHiddenTextPtr = tkptr->SearchNam.nam$l_rsa;
         ErrorVmsStatus (rqptr, status, FI_LI);
         UpdEnd (rqptr);
         return;
      }
   }

   tkptr->FileCount++;

   sptr = Buffer;
   strcpy (sptr, "<OPTION VALUE=\"");
   sptr += 15;
   for (cptr = tkptr->SearchNam.nam$l_name;
        *cptr && *cptr != ';';
        *sptr++ = tolower(*cptr++));
   *sptr++ = '\"';
   *sptr++ = '>';
   for (cptr = tkptr->SearchNam.nam$l_name;
        *cptr && *cptr != ';';
        *sptr++ = tolower(*cptr++));
   *sptr++ = '\n';
   *sptr = '\0';

   NetWriteBuffered (rqptr, &UpdNavigateSearchFiles, Buffer, sptr - Buffer);
}

/*****************************************************************************/
/*
End file search and end of navigation functions.
*/

UpdNavigateEnd (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (NavigateEndFaoDsc,
"</SELECT>\n\
<BR><SUP>1.</SUP> !AZ\n\
<BR><SUP>2.</SUP> !AZ\n\
<BR><INPUT TYPE=text NAME=as SIZE=20>\n\
!AZ\
<BR><INPUT TYPE=reset VALUE=\" !AZ \">\n\
</TD><TD ALIGN=left VALIGN=top>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR ALIGN=left><TD><INPUT TYPE=submit NAME=s-view VALUE=\" !AZ \"></TD>\
<TD><SUP>1.</SUP></TD></TR>\n\
<TR ALIGN=left><TD><INPUT TYPE=submit NAME=s-edit VALUE=\" !AZ \"></TD>\
<TD><SUP>1.[&amp;2.]</SUP></TD></TR>\n\
<TR ALIGN=left><TD><INPUT TYPE=submit NAME=s-create VALUE=\" !AZ \"></TD>\
<TD><SUP>2.</SUP></TD></TR>\n\
<TR ALIGN=left><TD><INPUT TYPE=submit NAME=s-filter VALUE=\" !AZ \"></TD>\
<TD><SUP>1.&amp;2.</SUP></TD></TR>\n\
<TR ALIGN=left><TD><INPUT TYPE=submit NAME=s-rename VALUE=\" !AZ \"></TD>\
<TD><SUP>1.&amp;2.</SUP></TD></TR>\n\
<TR ALIGN=left><TD><INPUT TYPE=submit NAME=s-copy VALUE=\" !AZ \"></TD>\
<TD><SUP>1.&amp;2.</SUP></TD></TR>\n\
<TR ALIGN=left><TD><INPUT TYPE=submit NAME=s-fileprotect VALUE=\" !AZ \"></TD>\
<TD><SUP>1./2.</SUP></TD></TR>\n\
<TR ALIGN=left><TD><INPUT TYPE=submit NAME=s-delete VALUE=\" !AZ \"></TD>\
<TD><SUP>1./2.</SUP></TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
</FORM>\n\
\
<P>!AZ\
</TD>\n\
\
</TR>\n\
</TABLE>\n\
\
</BODY>\n\
</HTML>\n");

   static $DESCRIPTOR (FileUploadFaoDsc,
"<FORM METHOD=POST ACTION=\"!AZ\" ENCTYPE=\"multipart/form-data\">\n\
<INPUT TYPE=submit VALUE=\" !AZ \">\n\
!AZ <INPUT TYPE=text NAME=uploadfilename SIZE=20>\n\
<BR>!AZ\
<BR>!AZ <INPUT TYPE=file NAME=name SIZE=20>\n\
</FORM>\n\0");

   static $DESCRIPTOR (NoFileUploadFaoDsc,
"<FORM>\n\
<INPUT TYPE=reset VALUE=\" Upload \">\n\
as <INPUT TYPE=text NAME=uploadfilename SIZE=20 \
VALUE=\"NOT SUPPORTED by your browser :^( \">\n\
<BR>!AZ\
<BR><INPUT TYPE=text NAME=name SIZE=20 \
VALUE=\"NOT SUPPORTED by your browser :^( \">\n\
local file <INPUT TYPE=reset VALUE=\" Browse... \">\n\
</FORM>\n\0");

   register unsigned long  *vecptr;
   register char  *cptr;
   register struct UpdTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [16];
   char  Buffer [2048],
         FileUpload [512];
   $DESCRIPTOR (BufferDsc, Buffer);
   $DESCRIPTOR (FileUploadDsc, FileUpload);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdNavigateEnd()\n");

   tkptr = rqptr->UpdTaskPtr;

   /* retrieve the previously saved position in the string */
   cptr = tkptr->MsgStringPtr;

   if (tkptr->BrowserSupportsFileUpload)
   {
      vecptr = FaoVector;
      *vecptr++ = tkptr->FilePath;

      /* "Upload" */
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';

      /* "as" */
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';

      /* file protection */
      *vecptr++ = UpdProtectionList;

      /* "local file" */
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';

      status = sys$faol (&FileUploadFaoDsc, 0, &FileUploadDsc, &FaoVector);
   }
   else
   {
      /* step over the unused portions of the language string, "upload" */
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) cptr++;
      /* "as" */
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) cptr++;
      /* "local file" */
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) cptr++;

      /* file protection */
      vecptr = FaoVector;
      *vecptr++ = UpdProtectionList;

      status = sys$fao (&NoFileUploadFaoDsc, 0, &FileUploadDsc);
   }

   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$fao()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   if (tkptr->FileCount)
   {
      /* "select from list" */
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
      /* skip over "none available" */
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) cptr++;
   }
   else
   {
      /* skip over "select from list" */
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) cptr++;
      /* "none available" */
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
   }

   /* "enter name/path" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* file protection */
   *vecptr++ = UpdProtectionList;

   /* "reset" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* view */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* edit */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* create */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* filter */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* rename */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* copy */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* protection */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* delete */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   *vecptr++ = FileUpload;

   status = sys$faol (&NavigateEndFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   Buffer[Length] = '\0';

   NetWriteBuffered (rqptr, &UpdEnd, Buffer, Length);
   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
Provide a directory tree display.  Provides two types of tree.  The first, an
update tree, where tree node selection opens a new directory update page. 
Second, a general directory tree, where selecting a tree node behaves as any
other directory URL (i.e. if a wildcard name.type;version is provided a listing
is produced, if none then any home page existing in the directory is returned,
else a directory listing, any query string supplied is reproduced in the tree
link paths).
*/

UpdTreeBegin (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (UpdateFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>Update !AZ</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>Update !AZ</H1>\n\
<HR SIZE=2><P>\n\
<PRE>  !#AZ<A HREF=\"!AZ!AZ!AZ!AZ!AZ\"><B>!AZ</B></A>/\n");

   static $DESCRIPTOR (TreeOfFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ !AZ!AZ</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ <TT>!AZ!AZ</TT></H1>\n\
<HR SIZE=2><P>\n\
<PRE>  !#AZ<A HREF=\"!AZ!AZ!AZ!AZ!AZ\"><B>!AZ</B></A>/\n");

   register unsigned long  *vecptr;
   register char  *cptr, *sptr, *zptr;
   register struct UpdTaskStruct  *tkptr;
   register struct UpdTreeStruct  *tnptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [2048],
         TreeNodeName [64];
   void  *AstFunctionPtr;
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdTreeBegin()\n");

   tkptr = rqptr->UpdTaskPtr;

   /*****************************************/
   /* check VMS-authenticated user's access */
   /*****************************************/

   if (rqptr->AuthVmsUserProfileLength)
   {
      status = AuthCheckVmsUserAccess (rqptr, tkptr->FileName,
                                       tkptr->FileNameLength);
      if (status == RMS$_PRV)
         tkptr->AuthVmsUserHasAccess = false;
      else
      if (VMSok (status))
         tkptr->AuthVmsUserHasAccess = true;
      else
      {
         /* error reported by access check */
         UpdEnd (rqptr);
         return;
      }
   }
   else
      tkptr->AuthVmsUserHasAccess = false;

   /************/
   /* traverse */
   /************/

   /* first check if the directory can be accessed */

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   if (Config.DirAccessSelective)
      status = FileExists (tkptr->FileName, DirBrowsableFileName);
   else
   {
      if (VMSok (status = FileExists (tkptr->FileName, DirHiddenFileName)))
         status = STS$K_ERROR;
      else
      if (status == RMS$_FNF)
         status = SS$_NORMAL;
   }

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      /* give some "disinformation"  ;^)  */
      if (status == RMS$_FNF) status = RMS$_DNF;
      rqptr->ErrorTextPtr = tkptr->FilePath;
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }

   /* allocate heap memory for the initial tree structure */
   if (Debug) fprintf (stdout, "tnptr from HEAP\n");
   tnptr = (struct UpdTreeStruct*)
      VmGetHeap (rqptr, sizeof(struct UpdTreeStruct));
   tnptr->PrevTreeNodePtr = tnptr->NextTreeNodePtr = NULL;
   tkptr->TreeHeadPtr = tkptr->TreeNodePtr = tnptr;

   /* check if we can see its subdirectories, if not then end */
   if (FileExists (tkptr->FileName, DirNosFileName) == RMS$_FNF &&
       FileExists (tkptr->FileName, DirNopsFileName) == RMS$_FNF)
   {
      AstFunctionPtr = &UpdTreeListDirs;

      /* tranfer the VMS access information to the tree node */
      tnptr->AuthVmsUserHasAccess = tkptr->AuthVmsUserHasAccess;

      /***************************/
      /* set up directory search */
      /***************************/

      /* set up ready to search for all files ending in ".DIR" (directories) */
      tnptr->SearchFab = cc$rms_fab;
      /* set the FAB user context to the client thread pointer */
      tnptr->SearchFab.fab$l_ctx = rqptr;
      tnptr->SearchFab.fab$l_dna = "*.DIR;";
      tnptr->SearchFab.fab$b_dns = 6;
      tnptr->SearchFab.fab$l_fna = tkptr->FileName;
      tnptr->SearchFab.fab$b_fns = tkptr->FileNameLength;
      tnptr->SearchFab.fab$l_fop = FAB$M_NAM;
      tnptr->SearchFab.fab$l_nam = &tnptr->SearchNam;

      tnptr->SearchNam = cc$rms_nam;
      tnptr->SearchNam.nam$l_esa = tnptr->ExpandedFileName;
      tnptr->SearchNam.nam$b_ess = sizeof(tnptr->ExpandedFileName)-1;
      tnptr->SearchNam.nam$l_rsa = tnptr->ResultantFileName;
      tnptr->SearchNam.nam$b_rss = sizeof(tnptr->ResultantFileName)-1;

      if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

      status = sys$parse (&tnptr->SearchFab, 0, 0);

      if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

      if (VMSnok (status))
      {
         if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
         rqptr->ErrorTextPtr = tkptr->FilePath;
         rqptr->ErrorHiddenTextPtr = tnptr->FileName;
         ErrorVmsStatus (rqptr, status, FI_LI);
         UpdEnd (rqptr);
         return;
      }
   }
   else
      AstFunctionPtr = &UpdTreeEnd;

   /**************/
   /* begin page */
   /**************/

   for (cptr = tkptr->FilePath; *cptr; cptr++);
   while (cptr > tkptr->FilePath && *cptr != '/') cptr--;
   if (cptr > tkptr->FilePath && *cptr == '/' && !cptr[1]) cptr--;
   while (cptr > tkptr->FilePath && *cptr != '/') cptr--;
   if (*cptr == '/') cptr++;
   for (sptr = TreeNodeName; *cptr && *cptr != '/'; *sptr++ = tolower(*cptr++));
   *sptr = '\0';

   tkptr->FilePathSize = strlen(tkptr->FilePath) - strlen(TreeNodeName) - 1;

   /* "tree"s can have pre-expiry controlled from the query string */
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      UpdEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, tkptr->FileName);

   if (tkptr->UpdateTree)
   {
      *vecptr++ = rqptr->ServicePtr->ServerHostPort;
      *vecptr++ = rqptr->ServicePtr->ServerHostPort;
   }
   else
   {
      *vecptr++ = cptr = MsgFor(rqptr,MSG_DIR_TREE_OF);
      *vecptr++ = tkptr->FilePath;
      *vecptr++ = tkptr->FileNamePart;
      *vecptr++ = cptr;
      *vecptr++ = tkptr->FilePath;
      *vecptr++ = tkptr->FileNamePart;
   }

   *vecptr++ = tkptr->FilePathSize;
   *vecptr++ = tkptr->FilePath;
   if (tkptr->UpdateTree)
   {
      *vecptr++ = rqptr->ScriptName;
      *vecptr++ = tkptr->FilePath;
      *vecptr++ = "";
      *vecptr++ = "";
      *vecptr++ = "";
   }
   else
   {
      *vecptr++ = "";
      *vecptr++ = tkptr->FilePath;
      *vecptr++ = tkptr->FileNamePart;
      if (rqptr->QueryStringPtr[0])
         *vecptr++ = "?";
      else
         *vecptr++ = "";
      *vecptr++ = rqptr->QueryStringPtr;
   }
   *vecptr++ = TreeNodeName;

   if (tkptr->UpdateTree)
      status = sys$faol (&UpdateFaoDsc, &Length, &BufferDsc, &FaoVector);
   else
      status = sys$faol (&TreeOfFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   Buffer[Length] = '\0';

   tkptr->TreeLevel = 0;

   NetWriteBuffered (rqptr, AstFunctionPtr, Buffer, Length);
}

/*****************************************************************************/
/*
(AST) function to invoke another sys$search() call when listing directories.
*/ 

UpdTreeListDirs (struct RequestStruct *rqptr)

{
   register struct UpdTreeStruct  *tnptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdTreeListDirs()\n");

   /* get the pointer to the tree structure */
   tnptr = rqptr->UpdTaskPtr->TreeNodePtr;

   if (tnptr->AuthVmsUserHasAccess) EnableSysPrv ();

   /* call RMS directory search routine, AST completion to DirDirectories() */
   sys$search (&tnptr->SearchFab, &UpdTreeDirs, &UpdTreeDirs);

   if (tnptr->AuthVmsUserHasAccess) DisableSysPrv ();
}

/*****************************************************************************/
/*
AST completion routine called each time sys$search() completes.  It will 
either point to another dir name found or have "no more files found" status 
(or an error!).
*/ 

UpdTreeDirs (struct FAB *FabPtr)

{
   register char  *cptr, *sptr, *zptr;
   register struct RequestStruct  *rqptr;
   register struct UpdTaskStruct  *tkptr;
   register struct UpdTreeStruct  *tnptr,
                                  *tmptnptr;

   boolean  VmsUserHasAccess;
   int  status,
        cnt;
   char  c;
   char  Buffer [2048];
   struct UpdTreeStruct  *NextTreeNodePtr,
                         *PrevTreeNodePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
      "UpdTreeDirs() sts: %%X%08.08X stv: %%X%08.08X\n",
      FabPtr->fab$l_sts, FabPtr->fab$l_stv);

   /* retrieve the pointer to the client thread from the FAB user context */
   rqptr = FabPtr->fab$l_ctx;
   /* get the pointer to the task structure */
   tkptr = rqptr->UpdTaskPtr;
   /* get the pointer to the tree structure */
   tnptr = tkptr->TreeNodePtr;

   if (VMSnok (status = tnptr->SearchFab.fab$l_sts))
   {
      if (status == RMS$_FNF || status == RMS$_NMF)
      {
         /* end of directory search, nest back one level or end */
         tkptr->TreeLevel--;
         if (tnptr->PrevTreeNodePtr != NULL)
         {
            tkptr->TreeNodePtr = tnptr->PrevTreeNodePtr;
            UpdTreeListDirs (rqptr);
         }
         else
            UpdTreeEnd (rqptr);
         return;
      }

      /* sys$search() error */
      rqptr->ErrorTextPtr = tkptr->FilePath;
      rqptr->ErrorHiddenTextPtr = tnptr->SearchNam.nam$l_dev;
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdTreeEnd (rqptr);
      return;
   }

   /* terminate following the last character in the version number */
   tnptr->SearchNam.nam$l_ver[tnptr->SearchNam.nam$b_ver] = '\0';
   if (Debug) fprintf (stdout, "Directory |%s|\n", tnptr->ResultantFileName);

   if (!memcmp (tnptr->SearchNam.nam$l_name, "000000.", 7))
   {
      /* not interested in master file directories :^) */
      UpdTreeListDirs (rqptr);
      return;
   }

   /* check permissions for each directory */
   if (rqptr->AuthVmsUserProfileLength)
   {
      /* check if SYSUAF-authenticated user has access */
      status = AuthCheckVmsUserAccess (rqptr,
                                       tnptr->SearchNam.nam$l_rsa,
                                       tnptr->SearchNam.nam$b_rsl);
   }
   else
   {
      /* check if HTTPd server account has access */
      status = AuthCheckVmsUserAccess (NULL,
                                       tnptr->SearchNam.nam$l_rsa,
                                       tnptr->SearchNam.nam$b_rsl);
   }
   if (VMSnok (status))
   {
      if (status == RMS$_PRV && Config.DirNoPrivIgnore)
      {
         /* does not have access, as if it didn't exist! */
         UpdTreeListDirs (rqptr);
         return;
      }

      /* error reported by access check (reported here due to NULL) */
      rqptr->ErrorTextPtr = "AuthCheckVmsUserAccess()";
      rqptr->ErrorHiddenTextPtr = tkptr->SearchNam.nam$l_rsa;
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdTreeEnd (rqptr);
      return;
   }

   /************/
   /* continue */
   /************/

   /* first check if the directory can be accessed */

   *(tnptr->SearchNam.nam$l_name-1) = '.';
   *tnptr->SearchNam.nam$l_type = ']';
   c = *(tnptr->SearchNam.nam$l_type+1);
   *(tnptr->SearchNam.nam$l_type+1) = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", tnptr->SearchNam.nam$l_rsa);

   if (VmsUserHasAccess) EnableSysPrv ();

   if (Config.DirAccessSelective)
      status = FileExists (tnptr->SearchNam.nam$l_rsa, DirBrowsableFileName);
   else
   {
      if (VMSok (status =
          FileExists (tnptr->SearchNam.nam$l_rsa, DirHiddenFileName)))
         status = STS$K_ERROR;
      else
      if (status == RMS$_FNF)
         status = SS$_NORMAL;
   }

   if (VmsUserHasAccess) DisableSysPrv ();

   *(tnptr->SearchNam.nam$l_name-1) = ']';
   *tnptr->SearchNam.nam$l_type = '.';
   *(tnptr->SearchNam.nam$l_type+1) = c;

   if (VMSnok (status))
   {
      /* no, it can't be accessed */
      UpdTreeListDirs (rqptr);
      return;
   }

   /*********************/
   /* new level of tree */
   /*********************/

   if (tnptr->NextTreeNodePtr == NULL)
   {
      /* allocate heap memory for the new (nested) tree task structure */
      if (Debug) fprintf (stdout, "tnptr from HEAP\n");
      tnptr = (struct UpdTreeStruct*)
         VmGetHeap (rqptr, sizeof(struct UpdTreeStruct));
      tnptr->PrevTreeNodePtr = tkptr->TreeNodePtr;
      tnptr->PrevTreeNodePtr->NextTreeNodePtr = tnptr;
      tnptr->NextTreeNodePtr = NULL;
      
      /* tranfer the VMS access information to the tree node */
      tnptr->AuthVmsUserHasAccess = VmsUserHasAccess;
   }
   else
   {
      /* reuse previously allocated structure */
      if (Debug) fprintf (stdout, "REUSE tnptr\n");
      tnptr = tnptr->NextTreeNodePtr;
      NextTreeNodePtr = tnptr->NextTreeNodePtr;
      PrevTreeNodePtr = tnptr->PrevTreeNodePtr;
      memset (tnptr, 0, sizeof(struct UpdTreeStruct));
      tnptr->NextTreeNodePtr = NextTreeNodePtr;
      tnptr->PrevTreeNodePtr = PrevTreeNodePtr;

      /* tranfer the VMS access information to the tree node */
      tnptr->AuthVmsUserHasAccess = VmsUserHasAccess;
   }
   tkptr->TreeNodePtr = tnptr;

   tkptr->TreeLevel++;

   sptr = tnptr->FileName;
   for (cptr = tnptr->PrevTreeNodePtr->SearchNam.nam$l_rsa;
        cptr < tnptr->PrevTreeNodePtr->SearchNam.nam$l_name;
        *sptr++ = *cptr++);
   *(sptr-1) = '.';
   while (cptr < tnptr->PrevTreeNodePtr->SearchNam.nam$l_type)
      *sptr++ = *cptr++;
   *sptr++ = ']';
   *sptr = '\0';
   tnptr->FileNameLength = sptr - tnptr->FileName;
   if (Debug)
      fprintf (stdout, "%d |%s|\n", tnptr->FileNameLength, tnptr->FileName);

   /***************************/
   /* set up directory search */
   /***************************/

   /* set up ready to search for all files ending in ".DIR" (directories) */
   tnptr->SearchFab = cc$rms_fab;
   /* set the FAB user context to the client thread pointer */
   tnptr->SearchFab.fab$l_ctx = rqptr;
   tnptr->SearchFab.fab$l_dna = "*.DIR;";
   tnptr->SearchFab.fab$b_dns = 6;
   tnptr->SearchFab.fab$l_fna = tnptr->FileName;
   tnptr->SearchFab.fab$b_fns = tnptr->FileNameLength;
   tnptr->SearchFab.fab$l_fop = FAB$M_NAM;
   tnptr->SearchFab.fab$l_nam = &tnptr->SearchNam;

   tnptr->SearchNam = cc$rms_nam;
   tnptr->SearchNam.nam$l_esa = tnptr->ExpandedFileName;
   tnptr->SearchNam.nam$b_ess = sizeof(tnptr->ExpandedFileName)-1;
   tnptr->SearchNam.nam$l_rsa = tnptr->ResultantFileName;
   tnptr->SearchNam.nam$b_rss = sizeof(tnptr->ResultantFileName)-1;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   status = sys$parse (&tnptr->SearchFab, 0, 0);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
      rqptr->ErrorTextPtr = tkptr->FilePath;
      rqptr->ErrorHiddenTextPtr = tnptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }

   /**************************/
   /* display this directory */
   /**************************/

   zptr = (sptr = Buffer) + sizeof(Buffer);

   /* indent tree diagram using spaces, '|' and '-' */
   for (cnt = tkptr->FilePathSize+2; cnt && sptr < zptr; cnt--) *sptr++ = ' ';
   for (cnt = tkptr->TreeLevel-1; cnt && sptr < zptr; cnt--)
      for (cptr = "|   "; *cptr && sptr < zptr; *sptr++ = *cptr++);
   for (cptr = "|---"; *cptr && sptr < zptr; *sptr++ = *cptr++);

   /* start HTML anchor */
   for (cptr = "<A HREF=\""; *cptr && sptr < zptr; *sptr++ = *cptr++);

   /* if an update tree the add the update script name */
   if (tkptr->UpdateTree)
      for (cptr = rqptr->ScriptName; *cptr && sptr < zptr; *sptr++ = *cptr++);

   /* add the initial directory path to the link path */
   for (cptr = tkptr->FilePath; 
        *cptr && sptr < zptr;
        *sptr++ = tolower(*cptr++));

   /* add the name of this directory to the link path */
   tmptnptr = tkptr->TreeHeadPtr;
   while (tmptnptr != tnptr)
   {
      for (cptr = tmptnptr->SearchNam.nam$l_name;
           *cptr && *cptr != '.' && sptr < zptr;
           *sptr++ = tolower(*cptr++));
      if (sptr < zptr) *sptr++ = '/';
      tmptnptr = tmptnptr->NextTreeNodePtr;
   }

   if (!tkptr->UpdateTree)
   {
      /* directory tree, add any file name.type;version and any query string */
      for (cptr = tkptr->FileNamePart;
           *cptr && sptr < zptr;
           *sptr++ = *cptr++);
      if (rqptr->QueryStringPtr[0])
      {
         if (sptr < zptr) *sptr++ = '?';
         for (cptr = rqptr->QueryStringPtr;
              *cptr && sptr < zptr;
              *sptr++ = *cptr++);
      }
   }

   /* complete the anchor link path and and the link description */
   for (cptr = "\"><B>"; *cptr && sptr < zptr; *sptr++ = *cptr++);
   for (cptr = tnptr->PrevTreeNodePtr->SearchNam.nam$l_name;
        *cptr && *cptr != '.' && sptr < zptr;
        *sptr++ = tolower(*cptr++));
   for (cptr = "</B></A>\n"; *cptr && sptr < zptr; *sptr++ = *cptr++);

   /* check for buffer overflow */
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      strcpy (Buffer, "<FONT COLOR=\"#ff0000\">[Error]</FONT>\n");
      for (sptr = Buffer; *sptr; sptr++);
   }
   
   *sptr = '\0';

   NetWriteBuffered (rqptr, &UpdTreeListDirs, Buffer, sptr - Buffer);
}

/*****************************************************************************/
/*
*/ 

UpdTreeEnd (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (TreeEndFaoDsc,
"</PRE>\n\
<P><HR SIZE=2>\n\
</BODY>\n\
</HTML>\n");

   register struct UpdTaskStruct  *tkptr;
   register struct UpdTreeStruct  *tnptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdTreeEnd()\n");

   /* get the pointer to the task structure */
   tkptr = rqptr->UpdTaskPtr;
   /* get the pointer to the tree structure */
   tnptr = tkptr->TreeNodePtr;

   while (tnptr->PrevTreeNodePtr != NULL)
   {
      /* ensure parse internal data structures are released */
      tnptr->SearchFab.fab$l_fna = "a:[b]c.d;";
      tnptr->SearchFab.fab$b_fns = 9;
      tnptr->SearchFab.fab$b_dns = 0;
      tnptr->SearchNam.nam$b_nop = NAM$M_SYNCHK;
      sys$parse (&tnptr->SearchFab, 0, 0);
      /* get any previous node */
      tnptr = tnptr->PrevTreeNodePtr;
   }

   NetWriteBuffered (rqptr, &UpdEnd,
                    TreeEndFaoDsc.dsc$a_pointer, TreeEndFaoDsc.dsc$w_length);
   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
Generate a file editing page.  This comprises an HTTP response header.  Open
the file.  This checks whether it is an update (file exists) or create (file
does not exist).  Generate an HTML page and the start of a text area widget.
Then initiate the file records being read, to be included within the
<TEXTAREA></TEXTAREA> tags.
*/

UpdEditFileBegin (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (BytesFaoDsc, " (!UL bytes)\0");

   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ !AZ</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ !AZ</H1>\n\
!AZ\
!AZ\
!AZ\
<P>\n\
<FORM METHOD=POST ACTION=\"!AZ\">\n\
<INPUT TYPE=submit VALUE=\" !AZ \">&nbsp;\n\
<INPUT TYPE=submit NAME=previewonly VALUE=\" !AZ \">&nbsp;\n\
<INPUT TYPE=hidden NAME=previewnote VALUE=\"!AZ!AZ\">\n\
<INPUT TYPE=reset VALUE=\" !AZ \">\n\
!AZ\
<BR>\n\
<TEXTAREA NAME=document COLS=!UL ROWS=!UL>\
!AZ");

   static $DESCRIPTOR (DayDateFaoDsc, "!AZ, !20%D\0");

   static $DESCRIPTOR (OutputFormatDsc, "|!WC, !DB-!MAAU-!Y4|!H04:!M0:!S0|");

   static $DESCRIPTOR (NewFileFaoDsc, "<FONT COLOR=\"#ff0000\">!AZ</FONT>\0");

   static $DESCRIPTOR (SourceFaoDsc,
"<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH ALIGN=RIGHT>!AZ:</TH>\
<TD ALIGN=LEFT><A HREF=\"!AZ\">!AZ</A>!AZ</TD>\n\
<TD>&nbsp;</TD><TH>[<A HREF=\"!AZ\">!AZ</A>]</TH></TR>\n\
<TR><TH ALIGN=RIGHT>!AZ:</TH><TD ALIGN=LEFT>!AZ</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (SourceSaveAsFaoDsc,
"<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH ALIGN=RIGHT>!AZ:</TH>\
<TD ALIGN=LEFT><A HREF=\"!AZ\">!AZ</A>!AZ</TD>\n\
<TH ALIGN=RIGHT>[<A HREF=\"!AZ\">!AZ</A>]</TH></TR>\n\
<TR><TH ALIGN=RIGHT>!AZ:</TH><TD ALIGN=LEFT>!AZ</TD></TR>\n\
<TR><TH ALIGN=RIGHT>!AZ:</TH>\
<TD ALIGN=LEFT><A HREF=\"!AZ\">!AZ</A></TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (ConfigSourceFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Source: &quot;File&quot;</TH></TR>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH ALIGN=RIGHT>File:</TH>\
<TD ALIGN=LEFT>!AZ</TD>\
<TD>&nbsp;</TD><TH>[<A HREF=\"!AZ\">View</A>]</TH></TR>\n\
<TR><TH ALIGN=RIGHT>Revised:</TH><TD ALIGN=LEFT>!AZ</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (RuleCheckFaoDsc,
"<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TH>Check Path Against Rules</TH></TR>\n\
<TR><TD>\n\
<FORM METHOD=GET ACTION=\"!AZ/\">\n\
Using <INPUT TYPE=radio NAME=server VALUE=yes CHECKED>server or \
<INPUT TYPE=radio NAME=server VALUE=no>file rules ...\n\
<BR><INPUT TYPE=text NAME=path SIZE=50>\n\
<INPUT TYPE=submit VALUE=\" check \">\n\
<INPUT TYPE=reset VALUE=\" reset \">\n\
</FORM>\n\
</TD></TR>\n\
</TABLE>\n");

   static $DESCRIPTOR (HtmlTemplateFaoDsc,
"&lt;!!DOCTYPE HTML PUBLIC &quot;-//W3C//DTD HTML 3.2//EN&quot;&gt;\n\
&lt;HTML&gt;\n\
&lt;HEAD&gt;\n\
&lt;META NAME=&quot;generator&quot; CONTENT=&quot;!AZ&quot;&gt;\n\
&lt;META NAME=&quot;author&quot; CONTENT=&quot;!AZ@!AZ&quot;&gt;\n\
&lt;META NAME=&quot;date&quot; CONTENT=&quot;!AZ&quot;&gt;\n\
&lt;TITLE&gt;[title]&lt;/TITLE&gt;\n\
&lt;/HEAD&gt;\n\
&lt;BODY&gt;\n\
&lt;H1&gt;[title]&lt;/H1&gt;\n\
&lt;HR&gt;\n\
[content]\n\
&lt;HR&gt;\n\
[author/revision]\n\
&lt;/BODY&gt;\n\
&lt;/HTML&gt;\n");

   register unsigned long  *vecptr;
   register char  *cptr, *sptr, *zptr;
   register struct UpdTaskStruct  *tkptr;

   boolean  ConfigurationEdit,
            RuleCheckAvailable,
            SpecialCaseConfig,
            SpecialCaseMessages,
            SpecialCasePaths,
            SpecialCaseRules;
   int  status,
        ByteCount,
        RevDayOfWeek;
   unsigned short  Length;
   unsigned long  DateLength;
   unsigned long  FaoVector [32];
   char  *TitlePtr;
   char  Buffer [4096],
         Bytes [32],
         DayDate [128],
         HtmlTemplate [1024],
         PostFilePath [256],
         RuleCheck [1024],
         Source [1024];
   void  *AstFunctionPtr;
   $DESCRIPTOR (BufferDsc, Buffer);
   $DESCRIPTOR (BytesDsc, Bytes);
   $DESCRIPTOR (DayDateDsc, DayDate);
   $DESCRIPTOR (HtmlTemplateDsc, HtmlTemplate);
   $DESCRIPTOR (RuleCheckDsc, RuleCheck);
   $DESCRIPTOR (SourceDsc, Source);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdEditFileBegin() |%s|\n", tkptr->FileName);

   tkptr = rqptr->UpdTaskPtr;

   /*****************/
   /* special cases */
   /*****************/

   TitlePtr = "";
   ConfigurationEdit = RuleCheckAvailable = SpecialCaseMessages =
      SpecialCasePaths = SpecialCaseConfig = SpecialCaseRules = false;

   if ((SpecialCaseConfig =
        strsame (rqptr->PathInfoPtr, HttpdInternalReviseConfig, -1)) ||
       (SpecialCaseMessages =
        strsame (rqptr->PathInfoPtr, HttpdInternalReviseMessages, -1)) ||
       (SpecialCasePaths =
        strsame (rqptr->PathInfoPtr, HttpdInternalRevisePaths, -1)) ||
       (SpecialCaseRules =
        strsame (rqptr->PathInfoPtr, HttpdInternalReviseRules, -1)))
   {
      if (SpecialCaseConfig)
      {
         cptr = ConfigFileName;
         TitlePtr = "<H2>Edit Server Configuration File</H2>\n";
         ConfigurationEdit = true;
      }
      else
      if (SpecialCaseMessages)
      {
         cptr = MsgFileName;
         TitlePtr = "<H2>Edit Message File</H2>\n";
         ConfigurationEdit = true;
      }
      else
      if (SpecialCasePaths)
      {
         cptr = AuthConfigFileName;
         TitlePtr = "<H2>Edit Path Authorization File</H2>\n";
         ConfigurationEdit = true;
      }
      else
      if (SpecialCaseRules)
      {
         cptr = MapFileName;
         TitlePtr = "<H2>Edit Mapping Rule File</H2>\n";
         ConfigurationEdit = RuleCheckAvailable = true;
      }
      else
      {
         ErrorInternal (rqptr, 0, ErrorSanityCheck, FI_LI);
         UpdEnd (rqptr);
         return;
      }

      /*****************************************/
      /* check VMS-authenticated user's access */
      /*****************************************/

      if (rqptr->AuthVmsUserProfileLength)
      {
         status = AuthCheckVmsUserAccess (rqptr, cptr, 0);
         if (status == RMS$_PRV)
            tkptr->AuthVmsUserHasAccess = false;
         else
         if (VMSok (status))
            tkptr->AuthVmsUserHasAccess = true;
         else
         {
            /* error reported by access check */
            UpdEnd (rqptr);
            return;
         }
      }
      else
         tkptr->AuthVmsUserHasAccess = false;

      /*******************/
      /* parse file name */
      /*******************/

      tkptr->FileFab = cc$rms_fab;
      tkptr->FileFab.fab$l_fna = cptr;
      tkptr->FileFab.fab$b_fns = strlen(cptr);
      tkptr->FileFab.fab$l_nam = &tkptr->FileNam;
      /* no I/O, syntax check only */
      tkptr->FileNam.nam$b_nop = NAM$M_SYNCHK;

      tkptr->FileNam = cc$rms_nam;
      tkptr->FileNam.nam$l_esa = tkptr->ExpandedFileName;
      tkptr->FileNam.nam$b_ess = sizeof(tkptr->ExpandedFileName)-1;

      if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

      status = sys$parse (&tkptr->FileFab, 0, 0);

      if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

      if (VMSnok (status))
      {
         if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
         rqptr->ErrorTextPtr = cptr;
         ErrorVmsStatus (rqptr, status, FI_LI);
         UpdEnd (rqptr);
         return;
      }

      tkptr->ExpandedFileName[tkptr->FileNam.nam$b_esl] = '\0';
      if (Debug)
         fprintf (stdout, "ExpandedFileName |%s|\n", tkptr->ExpandedFileName);

      sptr = tkptr->FileName;
      for (cptr = tkptr->ExpandedFileName; 
           *cptr && *cptr != ';';
           *sptr++ = *cptr++);
      *sptr = '\0';
      tkptr->FileNameLength = sptr - tkptr->FileName;

      cptr = MapVmsPath (tkptr->FileName);
      if (!cptr[0] && cptr[0])
      {
         rqptr->ResponseStatusCode = 401;
         ErrorGeneral (rqptr, cptr+1, FI_LI);
         UpdEnd (rqptr);
         return;
      }
      zptr = (sptr = tkptr->FilePath) + sizeof(tkptr->FilePath);
      while (*cptr && sptr < zptr) *sptr++ = *cptr++;
      if (sptr >= zptr)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         UpdEnd (rqptr);
         return;
      }
      *sptr = '\0';
   }
   else
   {
      /*****************************************/
      /* check VMS-authenticated user's access */
      /*****************************************/

      if (rqptr->AuthVmsUserProfileLength)
      {
         status = AuthCheckVmsUserAccess (rqptr, tkptr->FileName,
                                          tkptr->FileNameLength);
         if (status == RMS$_PRV)
            tkptr->AuthVmsUserHasAccess = false;
         else
         if (VMSok (status))
            tkptr->AuthVmsUserHasAccess = true;
         else
         {
            /* error reported by access check */
            UpdEnd (rqptr);
            return;
         }
      }
      else
         tkptr->AuthVmsUserHasAccess = false;

   }

   /************/
   /* continue */
   /************/

   /* find the file extension and set the content type */
   for (cptr = tkptr->FileName + tkptr->FileNameLength;
        cptr > tkptr->FileName && *cptr != '.';
        cptr--);
   tkptr->ContentTypePtr = ConfigContentType (NULL, cptr);

   if (!strsame (tkptr->ContentTypePtr, "text/", 5))
   {
      rqptr->ResponseStatusCode = 400;
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_NOT_TEXT_FILE), FI_LI);
      UpdEnd (rqptr);
      return;
   }

   if (tkptr->AsPath[0])
   {
      /* find the file extension and set content type for destination file */
      for (cptr = tkptr->AsPath; *cptr; cptr++);
      while (cptr > tkptr->AsPath && *cptr != '.') cptr--;
      tkptr->ContentTypePtr = ConfigContentType (NULL, cptr);
   }

   /*************/
   /* open file */
   /*************/

   tkptr->FileFab = cc$rms_fab;
   tkptr->FileFab.fab$b_fac = FAB$M_GET;
   tkptr->FileFab.fab$l_fna = tkptr->FileName;
   tkptr->FileFab.fab$b_fns = tkptr->FileNameLength;
   tkptr->FileFab.fab$l_nam = &tkptr->FileNam;
   tkptr->FileFab.fab$b_shr = FAB$M_SHRGET;
   tkptr->FileFab.fab$l_xab = &tkptr->FileXabDat;

   tkptr->FileNam = cc$rms_nam;
   tkptr->FileNam.nam$l_esa = tkptr->ExpandedFileName;
   tkptr->FileNam.nam$b_ess = sizeof(tkptr->ExpandedFileName)-1;

   /* initialize the date extended attribute block */
   tkptr->FileXabDat = cc$rms_xabdat;
   tkptr->FileXabDat.xab$l_nxt = &tkptr->FileXabFhc;

   /* initialize the file header extended attribute block */
   tkptr->FileXabFhc = cc$rms_xabfhc;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   status = sys$open (&tkptr->FileFab, 0, 0);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status) && status != RMS$_FNF)
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   if (status == RMS$_FNF)
      tkptr->FileExists = false;
   else
      tkptr->FileExists = true;

   if (tkptr->FileExists)
   {
      tkptr->ExpandedFileName[tkptr->FileNam.nam$b_rsl] = '\0';
      if (Debug)
         fprintf (stdout, "ExpandedFileName |%s|\n", tkptr->ExpandedFileName);

      /* record access block */
      tkptr->FileRab = cc$rms_rab;
      tkptr->FileRab.rab$l_ctx = rqptr;
      tkptr->FileRab.rab$l_fab = &tkptr->FileFab;
      /* 2 buffers, transfer 6 blocks */
      tkptr->FileRab.rab$b_mbc = 6;
      tkptr->FileRab.rab$b_mbf = 2;
      /* read ahead performance option */
      tkptr->FileRab.rab$l_rop = RAB$M_RAH;
      tkptr->FileRab.rab$l_ubf = tkptr->ReadBuffer;
      /* allow for two extra characters, a newline and a terminating null */
      tkptr->FileRab.rab$w_usz = sizeof(tkptr->ReadBuffer)-2;

      if (VMSnok (status = sys$connect (&tkptr->FileRab, 0, 0)))
      {
         if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
         rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
         rqptr->ErrorHiddenTextPtr = tkptr->FileName;
         ErrorVmsStatus (rqptr, status, FI_LI);
         UpdEnd (rqptr);
         return;
      }
   }

   /**************/
   /* begin page */
   /**************/

   if (!ConfigurationEdit)
   {
      cptr = MsgFor(rqptr,MSG_UPD_EDIT);
      zptr = (sptr = tkptr->MsgString) + sizeof(tkptr->MsgString);
      while (*cptr && sptr < zptr)
      {
         /* don't need any extraneous linefeeds from this long string */
         if (*cptr == '\n')
            cptr++;
         else
            *sptr++ = *cptr++;
      }
      if (sptr >= zptr)
      {
         ErrorGeneralOverflow (rqptr, FI_LI);
         UpdEnd (rqptr);
         return;
      }
      *sptr = '\0';

      /* "Start at the very beginning, very good place to start ..." */
      tkptr->MsgStringPtr = tkptr->MsgString;
   }

   if (tkptr->FileExists)
   {
      if (ConfigurationEdit || NaturalLanguageEnglish)
      {
         lib$day_of_week (&tkptr->FileXabDat.xab$q_rdt, &RevDayOfWeek);
         sys$fao (&DayDateFaoDsc, 0, &DayDateDsc,
                  DayName[RevDayOfWeek], &tkptr->FileXabDat.xab$q_rdt);
      }
      else
      {
         static unsigned long  LibDateTimeContext = 0,
                               LibOutputFormat = LIB$K_OUTPUT_FORMAT;

         if (!LibDateTimeContext)
         {
            /* initialize this particular date/time format */
            if (VMSnok (status =
               lib$init_date_time_context (&LibDateTimeContext,
                                           &LibOutputFormat,
                                           &OutputFormatDsc)))
            {
               rqptr->ErrorTextPtr = "lib$init_date_time_context()";
               ErrorVmsStatus (rqptr, status, FI_LI);
               UpdEnd (rqptr);
               return;
            }
         }

         if (VMSnok (status =
             lib$format_date_time (&DayDateDsc, &tkptr->FileXabDat.xab$q_rdt,
                                   &LibDateTimeContext, &DateLength, 0)))
         {
            if (status != LIB$_ENGLUSED)
            {
               rqptr->ErrorTextPtr = "lib$format_date_time()";
               ErrorVmsStatus (rqptr, status, FI_LI);
               UpdEnd (rqptr);
               return;
            }
         }
         DayDate[DateLength] = '\0';
      }

      /* true for non-VAR record formats, almost true for those :^) */
      if (tkptr->FileXabFhc.xab$l_ebk)
         ByteCount = ((tkptr->FileXabFhc.xab$l_ebk - 1) * 512) +
                     tkptr->FileXabFhc.xab$w_ffb;
      else
         ByteCount = tkptr->FileXabFhc.xab$w_ffb;
      sys$fao (&BytesFaoDsc, 0, &BytesDsc, ByteCount);

      if (!ConfigurationEdit)
      {
         /* retrieve the previously saved position in the string */
         cptr = tkptr->MsgStringPtr;
         /* step over "New Document" */
         while (*cptr && *cptr != '|') cptr++;
         if (*cptr) cptr++;
        /* save the current position for use in later parts of the page */
        tkptr->MsgStringPtr = cptr;
     }
   }
   else
   {
      if (ConfigurationEdit)
         sptr = "New File";
      else
      {
         /* "New Document" */
         sptr = cptr = tkptr->MsgStringPtr;
         while (*cptr && *cptr != '|') cptr++;
         if (*cptr) *cptr++ = '\0';
         /* save the current position for use in later parts of the page */
         tkptr->MsgStringPtr = cptr;
      }

      if (VMSnok (status = sys$fao (&NewFileFaoDsc, 0, &DayDateDsc, sptr)))
         strcpy (DayDate, "New File");

      Bytes[0] = '\0';
   }

   /* remove any explicit version from the path to be POSTed */
   if (tkptr->AsPath[0])
      cptr = tkptr->AsPath;
   else
      cptr = tkptr->FilePath;
   zptr = (sptr = PostFilePath) + sizeof(PostFilePath);
   while (*cptr && *cptr != ';' && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   *sptr = '\0';

   if (ConfigurationEdit)
   {
      /**************************************************/
      /* editing a configuration file from server admin */
      /**************************************************/

      vecptr = FaoVector;

      *vecptr++ = tkptr->FileName;
      *vecptr++ = tkptr->FilePath;
      *vecptr++ = DayDate;

      if (tkptr->AsPath[0])
         *vecptr++ = "Save As";
      else
      if (tkptr->FileExists)
         *vecptr++ = "Update";
      else
         *vecptr++ = "Create";

      status = sys$faol (&ConfigSourceFaoDsc, &Length, &SourceDsc, &FaoVector);

      /* the administration menus, etc. are always in English (for now!) */
      strcpy (tkptr->MsgStringPtr = tkptr->MsgString,
      "Update|Save As|Update|Create|Preview|Undo Editing|Change Edit Window");
   }
   else
   {
      /******************************/
      /* editing any other document */
      /******************************/

      /* retrieve the previously saved position in the string */
      cptr = tkptr->MsgStringPtr;

      vecptr = FaoVector;

      /* "Document" */
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';

      *vecptr++ = tkptr->FilePath;
      *vecptr++ = tkptr->FilePath;
      *vecptr++ = Bytes;

      *vecptr++ = UpdHelpPath;

      /* "Help" */
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';

      /* "Revised" */
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';

      *vecptr++ = DayDate;

      /* save the current position for use in later parts of the page */
      tkptr->MsgStringPtr = cptr;

      if (tkptr->AsPath[0])
      {
         *vecptr++ = tkptr->AsPath;
         *vecptr++ = tkptr->AsPath;
         status = sys$faol (&SourceSaveAsFaoDsc, &Length, &SourceDsc,
                            &FaoVector);
      }
      else
         status = sys$faol (&SourceFaoDsc, &Length, &SourceDsc, &FaoVector);
   }
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   Source[Length] = '\0';

   if (RuleCheckAvailable)
   {
      /*
         The rule check form is only available to server configuration
         file editing.  It does not appear in any normal user-level page.
      */
      vecptr = FaoVector;
      *vecptr++ = HttpdInternalReportRules;

      status = sys$faol (&RuleCheckFaoDsc, &Length, &RuleCheckDsc, &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         UpdEnd (rqptr);
         return;
      }
      RuleCheck[Length] = '\0';
   }
   else
      RuleCheck[0] = '\0';

   if (!tkptr->CxR[0])
   {
#ifdef X11_80X24
      if (rqptr->HttpUserAgentPtr == NULL)
         strcpy (tkptr->CxR, "80x12");
      else
      {
         /* larger window if X Window System browser (assume workstation) */
         if (strstr (rqptr->HttpUserAgentPtr, "X11") != NULL)
            strcpy (tkptr->CxR, "80x24");
         else
            strcpy (tkptr->CxR, "80x12");
      }
#else
      strcpy (tkptr->CxR, "80x12");
#endif
   }

   if (!tkptr->FileExists && strsame (tkptr->ContentTypePtr, "text/html", 9))
   {
      /* HTML template */
      vecptr = FaoVector;

      *vecptr++ = SoftwareID;
      if (rqptr->RemoteUser[0])
         *vecptr++ = rqptr->RemoteUser;
      else
         *vecptr++ = "?";
      *vecptr++ = rqptr->ServicePtr->ServerHostPort;
      *vecptr++ = rqptr->GmDateTime;

      status = sys$faol (&HtmlTemplateFaoDsc, &Length, &HtmlTemplateDsc,
                         &FaoVector);
      if (VMSnok (status) || status == SS$_BUFFEROVF)
      {
         rqptr->ErrorTextPtr = "sys$faol()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         UpdEnd (rqptr);
         return;
      }
      HtmlTemplate[Length] = '\0';
   }
   else
      HtmlTemplate[0] = '\0';

   cptr = tkptr->CxR;
   tkptr->Cols = atoi(cptr);
   while (*cptr && tolower(*cptr) != 'x') cptr++;
   if (*cptr) cptr++;
   tkptr->Rows = atoi(cptr);

   if (tkptr->Cols <= 0)
      tkptr->Cols = 80;
   else
   if (tkptr->Cols >= 132)
      tkptr->Cols = 132;
   if (tkptr->Rows <= 0)
      tkptr->Rows = 12;
   else
   if (tkptr->Rows >= 48)
      tkptr->Rows = 48;

   /*
      The file edit page is deliberately no pre-expired.
      To do so results in the page being reloaded each time it's displayed.
   */
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      UpdEnd (rqptr);
      return;
   }

   /* retrieve the previously saved position in the string */
   cptr = tkptr->MsgStringPtr;

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, tkptr->FileName);

   /* ordinary editing or server administration file editing again */
   if (ConfigurationEdit)
   {
      *vecptr++ = "HTTPd";
      *vecptr++ = ServerHostPort;
      *vecptr++ = "HTTPd";
      *vecptr++ = ServerHostPort;
   }
   else
   {
      *vecptr++ = cptr;
      *vecptr++ = rqptr->ServicePtr->ServerHostPort;
      *vecptr++ = cptr;
      *vecptr++ = rqptr->ServicePtr->ServerHostPort;
   }

   /* step over the "Update" */
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   *vecptr++ = TitlePtr;
   *vecptr++ = RuleCheck;
   *vecptr++ = Source;

   *vecptr++ = PostFilePath;

   if (tkptr->AsPath[0])
   {
      /* "save as" */
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
      /* step over "Update" */
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) cptr++;
      /* step over "Create" */
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) cptr++;
   }
   else
   if (tkptr->FileExists)
   {
      /* step over "save as" */
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) cptr++;
      /* "Update" */
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
      /* step over "Create" */
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) cptr++;
   }
   else
   {
      /* step over "save as" */
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) cptr++;
      /* step over "Update" */
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) cptr++;
      /* "Create" */
      *vecptr++ = cptr;
      while (*cptr && *cptr != '|') cptr++;
      if (*cptr) *cptr++ = '\0';
   }

   /* "Preview" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   if (strsame (tkptr->ContentTypePtr, "text/html", -1))
   {
      *vecptr++ = PreviewNoteHtml;
      *vecptr++ = PreviewNoteHtmlNewLine;
   }
   else
   if (strsame (tkptr->ContentTypePtr, "text/x-menu", -1))
   {
      *vecptr++ = PreviewNoteHtml;
      *vecptr++ = PreviewNoteMenuNewLine;
   }
   else
   {
      *vecptr++ = PreviewNotePlain;
      *vecptr++ = "";
   }

   /* Undo Editing */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   /* save the current position for use in later parts of the page */
   tkptr->MsgStringPtr = cptr;

   /* file protection */
   *vecptr++ = UpdProtectionList;

   *vecptr++ = tkptr->Cols;
   *vecptr++ = tkptr->Rows;

   *vecptr++ = HtmlTemplate;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   Buffer[Length] = '\0';

   if (tkptr->FileExists)
      AstFunctionPtr = &UpdEditFileNextRecord;
   else
      AstFunctionPtr = &UpdEditFileEnd;
   NetWriteBuffered (rqptr, AstFunctionPtr, Buffer, Length);
}

/*****************************************************************************/
/*
Queue a read of the next record from the file.  When the read completes call 
UpdEditFileNextRecordAST() function to HTML-escape and include in the text
area.
*/ 

UpdEditFileNextRecord (struct RequestStruct *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdEditFileNextRecord()\n");

   sys$get (&rqptr->UpdTaskPtr->FileRab,
            &UpdEditFileNextRecordAST,
            &UpdEditFileNextRecordAST);
}

/*****************************************************************************/
/*
A record has been read.  Ensure it is newline terminated.  Escape any
HTML-forbidden characters (it is being included within <TEXTAREA></TEXTAREA>
tags!).  Buffer it.
*/ 

UpdEditFileNextRecordAST (struct RAB *RabPtr)

{
   register char  *cptr, *sptr, *zptr;
   register int  rsz;
   register struct RequestStruct  *rqptr;
   register struct UpdTaskStruct  *tkptr;

   int  status,
        Length;
   char  Buffer [2048];

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout,
      "UpdEditFileNextRecordAST() sts: %%X%08.08X stv: %%X%08.08X rsz: %d\n",
      RabPtr->rab$l_sts, RabPtr->rab$l_stv,  RabPtr->rab$w_rsz);
   }

   rqptr = RabPtr->rab$l_ctx;
   tkptr = rqptr->UpdTaskPtr;

   if (VMSnok (tkptr->FileRab.rab$l_sts))
   {
      if (tkptr->FileRab.rab$l_sts == RMS$_EOF)
      {
         if (Debug) fprintf (stdout, "RMS$_EOF\n");
         UpdEditFileEnd (rqptr);
         return;
      }

      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, tkptr->FileRab.rab$l_sts, FI_LI);
      UpdEnd (rqptr);
      return;
   }

   rsz = tkptr->FileRab.rab$w_rsz;
   if (rsz)
   {
      /* add newline if last character in line is not a newline! */
      if (tkptr->FileRab.rab$l_ubf[rsz-1] != '\n')
         tkptr->FileRab.rab$l_ubf[rsz++] = '\n';
   }
   else
   {
      /* must be a blank line (empty record), add a newline */
      tkptr->FileRab.rab$l_ubf[rsz++] = '\n';
   }
   tkptr->FileRab.rab$l_ubf[rsz] = '\0';
   if (Debug) fprintf (stdout, "rab$l_ubf |%s|\n", tkptr->FileRab.rab$l_ubf);

   cptr = tkptr->FileRab.rab$l_ubf;
   zptr = (sptr = Buffer) + sizeof(Buffer);
   while (rsz-- && sptr < zptr)
   {
      switch (*cptr)
      {
          case '<' :
             if (sptr+4 >= zptr) { sptr = zptr; break; }
             memcpy (sptr, "&lt;", 4); sptr += 4; cptr++; break;
          case '>' :
             if (sptr+4 >= zptr) { sptr = zptr; break; }
             memcpy (sptr, "&gt;", 4); sptr += 4; cptr++; break;
          case '&' :
             if (sptr+5 >= zptr) { sptr = zptr; break; }
             memcpy (sptr, "&amp;", 5); sptr += 5; cptr++; break;
          case '\"' :
             if (sptr+6 >= zptr) { sptr = zptr; break; }
             memcpy (sptr, "&quot;", 6); sptr += 6; cptr++; break;
          default :
             *sptr++ = *cptr++;
       }
   }
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   *sptr = '\0';
   if (Debug) fprintf (stdout, "Buffer |%s|\n", Buffer);

   NetWriteBuffered (rqptr, &UpdEditFileNextRecord, Buffer, sptr-Buffer);
}

/*****************************************************************************/
/*
End-of-file detected.  End text area, add form submit botton, change text
edit area form, etc., complete editing page HTML.
*/

UpdEditFileEnd (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (CurrentCxRFaoDsc, "!ULx!UL\0");

   static $DESCRIPTOR (EditFileFaoDsc,
"</TEXTAREA>\n\
</FORM>\n\
<P><FORM METHOD=GET ACTION=\"!AZ!AZ\">\n\
<INPUT TYPE=submit VALUE=\" !AZ \">\n\
<SELECT NAME=cxr>\n\
<OPTION VALUE=!AZ SELECTED>!AZ\n\
<OPTION VALUE=132x48>132x48\n\
<OPTION VALUE=132x24>132x24\n\
<OPTION VALUE=132x12>132x12\n\
<OPTION VALUE=80x48>80x48\n\
<OPTION VALUE=80x24>80x24\n\
<OPTION VALUE=80x12>80x12\n\
<OPTION VALUE=40x48>40x48\n\
<OPTION VALUE=40x24>40x24\n\
<OPTION VALUE=40x12>40x12\n\
</SELECT>\n\
</FORM>\n\
</BODY>\n\
</HTML>\n");

   register char  *cptr;
   register unsigned long  *vecptr;
   register struct UpdTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [8];
   char  Buffer [2048],
         CurrentCxR [32];
   $DESCRIPTOR (BufferDsc, Buffer);
   $DESCRIPTOR (CurrentCxRDsc, CurrentCxR);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdEditFileEnd()\n");

   tkptr = rqptr->UpdTaskPtr;

   vecptr = FaoVector;
   *vecptr++ = tkptr->Cols;
   *vecptr++ = tkptr->Rows;
   sys$faol (&CurrentCxRFaoDsc, 0, &CurrentCxRDsc, &FaoVector);

   vecptr = FaoVector;

   *vecptr++ = rqptr->ScriptName;
   *vecptr++ = tkptr->FilePath;

   /* "Change Edit Window" */
   *vecptr++ = cptr = tkptr->MsgStringPtr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   *vecptr++ = CurrentCxR;
   *vecptr++ = CurrentCxR;

   status = sys$faol (&EditFileFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, &UpdEnd, Buffer, Length);

   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
*/

UpdRenameFile
(
struct RequestStruct *rqptr,
char *OldName,
char *NewName
)
{
   static unsigned long  RenameFlags  = 1;

   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ 200</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ!!</H1>\n\
<P>!AZ\n\
<P>!AZ &nbsp;<TT>!AZ</TT>&nbsp; !AZ &nbsp;<TT>!AZ</TT>\n\
</BODY>\n\
</HTML>\n");

   register char  *cptr;
   register unsigned long  *vecptr;
   register struct UpdTaskStruct  *tkptr;

   int  status,
        RenameCount,
        SetPrvStatus;
   unsigned short  Length;
   unsigned long  Context;
   unsigned long  FaoVector [16];
   char  Buffer [2048],
         NewFileName [256],
         OldFileName [256];
   $DESCRIPTOR (BufferDsc, Buffer);
   $DESCRIPTOR (OldFileNameDsc, OldFileName);
   $DESCRIPTOR (NewFileNameDsc, NewFileName);

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "UpdRenameFile() |%s|%s|%s|\n",
               rqptr->UpdTaskPtr->FileName, OldName, NewName);

   tkptr = rqptr->UpdTaskPtr;

   if (!rqptr->RequestFileName[0])
   {
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_PARENT_DIRECTORY), FI_LI);
      UpdEnd (rqptr);
      return;
   }

   OldFileNameDsc.dsc$w_length =
      sprintf (OldFileName, "%s%s", rqptr->RequestFileName, OldName);
   NewFileNameDsc.dsc$w_length =
      sprintf (NewFileName, "%s%s", rqptr->RequestFileName, NewName);

   if (strsame (OldName, NewName, -1))
   {
      ErrorGeneral (rqptr, MsgFor(rqptr,MSG_UPD_RENAME_SAME), FI_LI);
      UpdEnd (rqptr);
      return;
   }

   /* check access to parent directory */
   if (VMSnok (status =
       AuthCheckWriteAccess (rqptr, rqptr->RequestFileName, 0)))
   {
      rqptr->ErrorTextPtr = rqptr->RequestFileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }

   /* use SYSPRV to ensure deletion of file (provided protection is S:RWED) */
   EnableSysPrv ();

   RenameCount = Context = 0;
   while (VMSok (status =
          lib$rename_file (&OldFileNameDsc, &NewFileNameDsc,
                           0, 0, &RenameFlags, 0, 0, 0, 0, 0, 0, &Context)))
      RenameCount++;

   DisableSysPrv ();

   if (!RenameCount)
   {
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      if (status == RMS$_FNF)
      {
         rqptr->ErrorTextPtr = OldName;
         ErrorVmsStatus (rqptr, status, FI_LI);
      }
      else
      if (status == RMS$_SYN)
      {
         rqptr->ErrorTextPtr = NewName;
         ErrorVmsStatus (rqptr, status, FI_LI);
      }
      else
      {
         rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
         ErrorVmsStatus (rqptr, status, FI_LI);
      }
      UpdEnd (rqptr);
      return;
   }

   rqptr->ResponsePreExpired = PRE_EXPIRE_UPD;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      UpdEnd (rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = MsgFor(rqptr,MSG_STATUS_SUCCESS);
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_SUCCESS);
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_REPORTED_BY_SERVER);

   *vecptr++ = MsgFor(rqptr,MSG_GENERAL_FILE);
   *vecptr++ = OldName;
   *vecptr++ = MsgFor(rqptr,MSG_UPD_RENAMED);
   *vecptr++ = NewName;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, &UpdEnd, Buffer, Length);
}

/*****************************************************************************/
/*
Copy a file by reading each 512 byte block and converting these into
4 x 128 byte, hexadecimal-encoded, form fields that can be recreated into a
file when POSTed to this server (see "hexencoded" in PUT.c module).
*/

UpdCopyFileBegin (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ !AZ</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ !AZ</H1>\n\
<FORM METHOD=POST ACTION=\"!AZ\">\n\
<INPUT TYPE=hidden NAME=protection VALUE=\"!4XL\">\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH ALIGN=right>!AZ</TH>\n\
<TD><A HREF=\"!AZ\"><TT>!AZ</TT></A></TD></TR>\n\
<TR><TH ALIGN=right>!AZ</TH>\n\
<TD><A HREF=\"!AZ\"><TT>!AZ</TT></A></TD></TR>\n\
</TABLE>\n\
</TD><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD>\n\
<INPUT TYPE=submit VALUE=\" !AZ \">\n\
</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n");

   register unsigned long  *vecptr;
   register char  *cptr, *sptr, *zptr;
   register struct UpdTaskStruct  *tkptr;

   int  status,
        ByteCount,
        RevDayOfWeek;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  *FileTypePtr;
   char  Buffer [2048],
         PostFilePath [256],
         Scratch [256];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdCopyFileBegin()\n");

   tkptr = rqptr->UpdTaskPtr;

   /*****************************************/
   /* check VMS-authenticated user's access */
   /*****************************************/

   if (rqptr->AuthVmsUserProfileLength)
   {
      status = AuthCheckVmsUserAccess (rqptr, rqptr->RequestFileName, 0);
      if (status == RMS$_PRV)
         tkptr->AuthVmsUserHasAccess = false;
      else
      if (VMSok (status))
         tkptr->AuthVmsUserHasAccess = true;
      else
      {
         /* error reported by access check */
         UpdEnd (rqptr);
         return;
      }
   }
   else
      tkptr->AuthVmsUserHasAccess = false;

   /********/
   /* open */
   /********/

   tkptr->FileFab = cc$rms_fab;
   tkptr->FileFab.fab$b_fac = FAB$M_GET | FAB$M_BIO; 
   tkptr->FileFab.fab$l_fna = tkptr->FileName;
   tkptr->FileFab.fab$b_fns = tkptr->FileNameLength;
   tkptr->FileFab.fab$l_nam = &tkptr->FileNam;
   tkptr->FileFab.fab$b_shr = FAB$M_SHRGET;

   tkptr->FileNam = cc$rms_nam;
   tkptr->FileNam.nam$l_esa = tkptr->ExpandedFileName;
   tkptr->FileNam.nam$b_ess = sizeof(tkptr->ExpandedFileName)-1;

   if (tkptr->AuthVmsUserHasAccess) EnableSysPrv ();

   status = sys$open (&tkptr->FileFab, 0, 0);

   if (tkptr->AuthVmsUserHasAccess) DisableSysPrv ();

   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }

   tkptr->ExpandedFileName[tkptr->FileNam.nam$b_rsl] = '\0';
   if (Debug)
      fprintf (stdout, "ExpandedFileName |%s|\n", tkptr->ExpandedFileName);

   /* record access block */
   tkptr->FileRab = cc$rms_rab;
   tkptr->FileRab.rab$l_ctx = rqptr;
   tkptr->FileRab.rab$l_fab = &tkptr->FileFab;
   tkptr->FileRab.rab$l_bkt = 0;
   tkptr->FileRab.rab$l_rop = RAB$M_RAH | RAB$M_BIO | RAB$M_ASY;
   tkptr->FileRab.rab$l_ubf = tkptr->ReadBuffer;
   /* MUST be 512 bytes (one block) */
   tkptr->FileRab.rab$w_usz = sizeof(tkptr->ReadBuffer);

   if (VMSnok (status = sys$connect (&tkptr->FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }

   /**************/
   /* begin page */
   /**************/

   /* find the file extension and set the content type */
   for (cptr = tkptr->FileName + tkptr->FileNameLength;
        cptr > tkptr->FileName && *cptr != '.';
        cptr--);
   tkptr->ContentTypePtr = ConfigContentType (NULL, cptr);

   /* remove any explicit version from the path to be POSTed */
   cptr = tkptr->FilePath;
   zptr = (sptr = PostFilePath) + sizeof(PostFilePath);
   while (*cptr && *cptr != ';' && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   *sptr = '\0';

   rqptr->ResponsePreExpired = PRE_EXPIRE_UPD;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      UpdEnd (rqptr);
      return;
   }

   cptr = MsgFor(rqptr,MSG_UPD_COPY);
   zptr = (sptr = Scratch) + sizeof(Scratch);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   *sptr = '\0';

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   /* "Update" */
   *vecptr++ = cptr = Scratch;

   *vecptr++ = rqptr->ServicePtr->ServerHostPort;

   /* "Update" again */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   *vecptr++ = rqptr->ServicePtr->ServerHostPort;

   /* action */
   *vecptr++ = tkptr->AsPath;

   /* protection */
   *vecptr++ = tkptr->ProtectionMask;

   /* "Copy" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   *vecptr++ = tkptr->FilePath;
   *vecptr++ = tkptr->FilePath;

   /* "to" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   *vecptr++ = tkptr->AsPath;
   *vecptr++ = tkptr->AsPath;

   /* "Confirm" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, &UpdCopyFileNextBlock, Buffer, Length);
}

/*****************************************************************************/
/*
Queue a read of the next record from the file.  When the read completes call 
UpdCopyFileNextBlockAST() function to HTML-escape and include in the text
area.
*/ 

UpdCopyFileNextBlock (struct RequestStruct *rqptr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdCopyFileNextBlock()\n");

   rqptr->UpdTaskPtr->FileRab.rab$l_bkt++;

   sys$read (&rqptr->UpdTaskPtr->FileRab,
             &UpdCopyFileNextBlockAST,
             &UpdCopyFileNextBlockAST);
}

/*****************************************************************************/
/*
A record has been read.  Turn the record into 128 byte, hexdecimal-encoded
form hidden fields.
*/ 

UpdCopyFileNextBlockAST (struct RAB *RabPtr)

{
   static char  HexDigit [] = "0123456789ABCDEF";
   static char  Bucket [64];
   static $DESCRIPTOR (BucketDsc, Bucket);
   static $DESCRIPTOR (BucketFaoDsc,
          "<INPUT TYPE=hidden NAME=hexencoded_!UL_!UL VALUE=\"\0");

   register int  cnt;
   register char  *cptr, *sptr;
   register struct RequestStruct  *rqptr;
   register struct UpdTaskStruct  *tkptr;

   int  status;
   char  Buffer [2048];

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout,
      "UpdCopyFileNextBlockAST() sts: %%X%08.08X stv: %%X%08.08X rsz: %d\n",
      RabPtr->rab$l_sts, RabPtr->rab$l_stv,  RabPtr->rab$w_rsz);
   }

   rqptr = RabPtr->rab$l_ctx;
   tkptr = rqptr->UpdTaskPtr;

   if (VMSnok (tkptr->FileRab.rab$l_sts))
   {
      if (tkptr->FileRab.rab$l_sts == RMS$_EOF)
      {
         if (Debug) fprintf (stdout, "RMS$_EOF\n");
         UpdCopyFileEnd (rqptr);
         return;
      }

      rqptr->ErrorTextPtr = MapVmsPath (tkptr->FileName);
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, tkptr->FileRab.rab$l_sts, FI_LI);
      UpdEnd (rqptr);
      return;
   }

   sptr = Buffer;
   for (cnt = 0; cnt < RabPtr->rab$w_rsz; cnt++)
   {
       if (!(cnt % 128))
       {
          if (cnt) for (cptr = "\">\n"; *cptr; *sptr++ = *cptr++);
          sys$fao (&BucketFaoDsc, 0, &BucketDsc,
                   tkptr->FileRab.rab$l_bkt, cnt);          
          for (cptr = Bucket; *cptr; *sptr++ = *cptr++);
          cptr = tkptr->FileRab.rab$l_ubf + cnt;
      }
      *sptr++ = HexDigit[((unsigned char)*cptr & 0xf0) >> 4];
      *sptr++ = HexDigit[(unsigned char)*cptr & 0x0f];
      cptr++;
   }
   for (cptr = "\">\n"; *cptr; *sptr++ = *cptr++);
   *sptr = '\0';

   NetWriteBuffered (rqptr, &UpdCopyFileNextBlock, Buffer, sptr-Buffer);
}

/*****************************************************************************/
/*
End-of-file detected.
*/

UpdCopyFileEnd (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (CopyFileEndFaoDsc,
"</FORM>\n\
</BODY>\n\
</HTML>\n");

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdCopyFileEnd()\n");

   NetWriteBuffered (rqptr, &UpdEnd,
                    CopyFileEndFaoDsc.dsc$a_pointer,
                    CopyFileEndFaoDsc.dsc$w_length);

   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
Output HTML allowing confirmation of directory creation.
*/

UpdConfirmCreate (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ !AZ</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ !AZ</H1>\n\
<FORM METHOD=POST ACTION=\"!AZ\">\n\
<INPUT TYPE=hidden NAME=protection VALUE=\"!4XL\">\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH ALIGN=right>!AZ</TH>\n\
<TD><A HREF=\"!AZ\"><TT>!AZ</TT></A></TD></TR>\n\
</TABLE>\n\
</TD><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD>\n\
<INPUT TYPE=submit VALUE=\" !AZ \">\n\
</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
</FORM>\n\
</BODY>\n\
</HTML>\n");

   register char  *cptr, *sptr, *zptr;
   register unsigned long  *vecptr;
   register struct UpdTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [2048],
         Scratch [256];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdConfirmCreate()\n");

   tkptr = rqptr->UpdTaskPtr;

   rqptr->ResponsePreExpired = PRE_EXPIRE_UPD;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      UpdEnd (rqptr);
      return;
   }

   cptr = MsgFor(rqptr,MSG_UPD_CREATE);
   zptr = (sptr = Scratch) + sizeof(Scratch);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   *sptr = '\0';

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   /* "Update" */
   *vecptr++ = cptr = Scratch;

   *vecptr++ = rqptr->ServicePtr->ServerHostPort;

   /* "Update" again */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   *vecptr++ = rqptr->ServicePtr->ServerHostPort;

   /* form action */
   if (tkptr->AsPath[0])
      *vecptr++ = tkptr->AsPath;
   else
      *vecptr++ = tkptr->FilePath;

   *vecptr++ = tkptr->ProtectionMask;

   /* "Create" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   if (tkptr->AsPath[0])
   {
      *vecptr++ = tkptr->AsPath;
      *vecptr++ = tkptr->AsPath;
   }
   else
   {
      *vecptr++ = tkptr->FilePath;
      *vecptr++ = tkptr->FilePath;
   }

   /* "Confirm" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, &UpdEnd, Buffer, Length);

   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
Output HTML allowing confirmation of directory deletion.
*/

UpdConfirmDelete (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ !AZ</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ !AZ</H1>\n\
<FORM METHOD=POST ACTION=\"!AZ;*\">\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH ALIGN=right><FONT COLOR=\"#ff00000\">!AZ</FONT></TH>\n\
<TD><A HREF=\"!AZ\"><TT>!AZ</TT></A></TD></TR>\n\
</TABLE>\n\
</TD><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD>\n\
<INPUT TYPE=submit VALUE=\" !AZ \">\n\
</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
</FORM>\n\
</BODY>\n\
</HTML>\n");

   register char  *cptr, *sptr, *zptr;
   register unsigned long  *vecptr;
   register struct UpdTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [2048],
         Scratch [256];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdConfirmDelete()\n");

   tkptr = rqptr->UpdTaskPtr;

   rqptr->ResponsePreExpired = PRE_EXPIRE_UPD;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      UpdEnd (rqptr);
      return;
   }

   cptr = MsgFor(rqptr,MSG_UPD_DELETE);
   zptr = (sptr = Scratch) + sizeof(Scratch);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   *sptr = '\0';

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   /* "Update" */
   *vecptr++ = cptr = Scratch;

   *vecptr++ = rqptr->ServicePtr->ServerHostPort;

   /* "Update" again */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   *vecptr++ = rqptr->ServicePtr->ServerHostPort;

   if (tkptr->AsPath[0])
      *vecptr++ = tkptr->AsPath;
   else
      *vecptr++ = tkptr->FilePath;

   /* "delete" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   if (tkptr->AsPath[0])
   {
      *vecptr++ = tkptr->AsPath;
      *vecptr++ = tkptr->AsPath;
   }
   else
   {
      *vecptr++ = tkptr->FilePath;
      *vecptr++ = tkptr->FilePath;
   }

   /* "Confirm" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, &UpdEnd, Buffer, Length);

   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
Output HTML allowing confirmation of action (file/directory creation/deletion).
*/

UpdConfirmRename
(
struct RequestStruct *rqptr,
char *ParentDir,
char *OldFileName,
char *NewFileName
)
{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ !AZ</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ !AZ</H1>\n\
<FORM METHOD=POST ACTION=\"!AZ!AZ\">\n\
<INPUT TYPE=hidden NAME=s-rename VALUE=\"rename\">\n\
<INPUT TYPE=hidden NAME=name VALUE=\"!AZ\">\n\
<INPUT TYPE=hidden NAME=as VALUE=\"!AZ\">\n\
<P><TABLE CELLPADDING=2 CELLSPACING=1 BORDER=1>\n\
<TR><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TH ALIGN=right>!AZ</TH>\n\
<TD><A HREF=\"!AZ!AZ\"><TT>!AZ!AZ</TT></A></TD></TR>\n\
<TR><TH ALIGN=right>!AZ</TH>\n\
<TD><A HREF=\"!AZ!AZ\"><TT>!AZ!AZ</TT></A></TD></TR>\n\
</TABLE>\n\
</TD><TD>\n\
<TABLE CELLPADDING=2 CELLSPACING=1 BORDER=0>\n\
<TR><TD>\n\
<INPUT TYPE=submit VALUE=\" !AZ \">\n\
</TD></TR>\n\
</TABLE>\n\
</TD></TR>\n\
</TABLE>\n\
</FORM>\n\
</BODY>\n\
</HTML>\n");

   register char  *cptr, *sptr, *zptr;
   register unsigned long  *vecptr;
   register struct UpdTaskStruct  *tkptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [32];
   char  Buffer [2048],
         Scratch [256];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "UpdConfirmRename() |%s|%s|%s|\n",
               ParentDir, OldFileName, NewFileName);

   tkptr = rqptr->UpdTaskPtr;

   rqptr->ResponsePreExpired = PRE_EXPIRE_UPD;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      UpdEnd (rqptr);
      return;
   }

   cptr = MsgFor(rqptr,MSG_UPD_RENAME);
   zptr = (sptr = Scratch) + sizeof(Scratch);
   while (*cptr && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      ErrorGeneralOverflow (rqptr, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   *sptr = '\0';

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   /* "Update" */
   *vecptr++ = cptr = Scratch;

   *vecptr++ = rqptr->ServicePtr->ServerHostPort;

   /* "Update" again */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   *vecptr++ = rqptr->ServicePtr->ServerHostPort;

   *vecptr++ = rqptr->ScriptName;
   *vecptr++ = ParentDir;
   *vecptr++ = OldFileName,
   *vecptr++ = NewFileName,

   /* "Rename" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   *vecptr++ = ParentDir;
   *vecptr++ = OldFileName,
   *vecptr++ = ParentDir;
   *vecptr++ = OldFileName,

   /* "to" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   *vecptr++ = ParentDir;
   *vecptr++ = NewFileName,
   *vecptr++ = ParentDir;
   *vecptr++ = NewFileName,

   /* "Confirm" */
   *vecptr++ = cptr;
   while (*cptr && *cptr != '|') cptr++;
   if (*cptr) *cptr++ = '\0';

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      UpdEnd (rqptr);
      return;
   }
   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, &UpdEnd, Buffer, Length);

   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
This function uses the ACP-QIO interface detailed in the "OpenVMS I/O User's 
Reference Manual".
*/ 

UpdProtection (struct RequestStruct *rqptr)

{
   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>!AZ 200</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>!AZ!!</H1>\n\
<P>!AZ\n\
<P>!AZ &nbsp;<TT>!AZ</TT>&nbsp; !AZ &nbsp;<TT>(!AZ)</TT>\n\
</BODY>\n\
</HTML>\n");

   static unsigned short  EnsureSystemAccessMask = 0xfff0; /* S:RWED */
   static $DESCRIPTOR (DeviceDsc,  "");

   register char  *cptr, *sptr;
   register unsigned long  *vecptr;
   register struct UpdTaskStruct  *tkptr;

   boolean  VmsUserHasAccess;
   int  status;
   unsigned short  AcpChannel,
                   Length;
   unsigned short  ProtectionMask;
   unsigned long  FaoVector [32];
   char  Buffer [2048],
         ExpandedFileName [256],
         ProtectionString [32];
   $DESCRIPTOR (BufferDsc, Buffer);

   struct FAB  FileFab;
   struct NAM  FileNam;

   struct fibdef  FileFib;
   struct atrdef  FileAtr [] =
   {
      { sizeof(ProtectionMask), ATR$C_FPRO, &ProtectionMask },
      { 0, 0, 0 }
   };

   struct {
      unsigned short  Length;
      unsigned short  Unused;
      unsigned long  Address;
   } FileFibAcpDsc,
     FileNameAcpDsc,
     FileAtrAcpDsc;
     
   struct {
      unsigned short  Status;
      unsigned short  Unused1;
      unsigned long  Unused2;
   } AcpIOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UpdProtection()\n");

   tkptr = rqptr->UpdTaskPtr;

   ProtectionMask = tkptr->ProtectionMask & EnsureSystemAccessMask;

   if (Debug)
      fprintf (stdout, "ProtectionMask: %%X%04.04X\n", ProtectionMask);

   /* check if the user has access to this file! */
   if (rqptr->AuthVmsUserProfileLength)
   {
      status = AuthCheckVmsUserAccess (rqptr, tkptr->FileName,
                                       tkptr->FileNameLength);
      if (status == RMS$_PRV)
         VmsUserHasAccess = false;
      else
      if (VMSok (status))
         VmsUserHasAccess = true;
      else
      {
         /* error reported by access check */
         return;
      }
   }
   else
      VmsUserHasAccess = false;

   FileFab = cc$rms_fab;
   FileFab.fab$l_fna = tkptr->FileName;
   FileFab.fab$b_fns = tkptr->FileNameLength;
   FileFab.fab$l_fop = FAB$M_NAM;
   FileFab.fab$l_nam = &FileNam;
   FileNam = cc$rms_nam;
   FileNam.nam$l_esa = ExpandedFileName;
   FileNam.nam$b_ess = sizeof(ExpandedFileName)-1;

   if (VmsUserHasAccess) EnableSysPrv ();

   status = sys$parse (&FileFab, 0, 0);

   if (VmsUserHasAccess) DisableSysPrv ();

   if (VMSok (status))
   {
      if (Debug) fprintf (stdout, "ExpandedFileName |%s|\n", ExpandedFileName);

      /* assign a channel to the disk device containing the file */
      DeviceDsc.dsc$w_length = FileNam.nam$b_dev;
      DeviceDsc.dsc$a_pointer = FileNam.nam$l_dev;
      status = sys$assign (&DeviceDsc, &AcpChannel, 0, 0, 0);
      if (Debug) fprintf (stdout, "sys$assign() %%X%08.08X\n", status);
   }

   if (VMSok (status))
   {
      /**********************************************/
      /* queue an ACP I/O to change file attributes */
      /**********************************************/

      /* set up the File Information Block for the ACP interface */
      memset (&FileFib, 0, sizeof(struct fibdef));
      FileFibAcpDsc.Length = sizeof(FileFib);
      FileFibAcpDsc.Address = &FileFib;
#ifdef __DECC
      memcpy (&FileFib.fib$w_did, &FileNam.nam$w_did, 6);
#else
      memcpy (&FileFib.fib$r_did_overlay.fib$w_did, &FileNam.nam$w_did, 6);
#endif

      FileNameAcpDsc.Length = FileNam.nam$b_name +
                              FileNam.nam$b_type +
                              FileNam.nam$b_ver;
      FileNameAcpDsc.Address = FileNam.nam$l_name;

      if (VmsUserHasAccess) EnableSysPrv ();

      status = sys$qiow (0, AcpChannel, IO$_ACCESS | IO$_MODIFY, &AcpIOsb,
                         0, 0, &FileFibAcpDsc, &FileNameAcpDsc,
                         0, 0, &FileAtr, 0);

      if (VmsUserHasAccess) DisableSysPrv ();

      sys$dassgn (AcpChannel);

      if (Debug)
         fprintf (stdout, "sys$qio() %%X%08.08X IOsb: %%X%08.08X\n",
                 status, AcpIOsb.Status);

      if (VMSok (status)) status = AcpIOsb.Status;
   }

   /* ensure parse internal data structures are released */
   FileFab.fab$l_fna = "a:[b]c.d;";
   FileFab.fab$b_fns = 9;
   FileFab.fab$b_dns = 0;
   FileNam.nam$b_nop = NAM$M_SYNCHK;
   sys$parse (&FileFab, 0, 0);

   if (VMSnok (status)) 
   {
      *FileNam.nam$l_ver = '\0';
      rqptr->ErrorTextPtr = tkptr->FilePath;
      rqptr->ErrorHiddenTextPtr = tkptr->FileName;
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   /***********/
   /* success */
   /***********/

   FormatProtection (tkptr->ProtectionMask, ProtectionString);

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = MsgFor(rqptr,MSG_STATUS_SUCCESS);
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_SUCCESS);
   *vecptr++ = MsgFor(rqptr,MSG_STATUS_REPORTED_BY_SERVER);
   *vecptr++ = MsgFor(rqptr,MSG_GENERAL_FILE);
   *vecptr++ = tkptr->FilePath;
   *vecptr++ = MsgFor(rqptr,MSG_UPD_PROTECTION);
   *vecptr++ = ProtectionString;

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      return;
   }

   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, NULL, Buffer, Length);
   rqptr->ResponseStatusCode = 200;
}

/*****************************************************************************/

