/*****************************************************************************
/*
                                   VM.c

Virtual memory support functions for HTTPd using the LIB$*_VM_* routines.

The reason for not using generic memory routines such as calloc() and malloc()
for frequently requested and then disposed of dynamic memory is of course
efficiency. The theory goes like this ... tailoring VMS memory zones against
the expected request profile for it's use will result in faster and more
efficient memory management. An added, and not inconsequential bonus, is the
additional integrity checking the library routines can provide.

A general zone is created for non-specific allocations.

A zone for fixed sized memory blocks is created for the request structures.

An individual zone is created for each request's heap.  Hopefully the overhead
of creating zone will be offset by the lower overhead probably required for
managing a smaller and short-lived collection of chunks, and the documentation
specifically states that reseting or deleting a memory zone is a more
efficient method of disposing of virtual memory than individually freeing each
chunk.

A separate zone is created for allocating cache memory.

MEMORY MANAGEMENT IS CONSIDERED CRUCIAL TO HTTPD FUNCTIONING. IF AN ALLOCATION
OR FREEING GENERATES AN ERROR (E.G. INSUFFICIENT VIRTUAL, BAD BLOCK, ETC.) THEN
THIS IS CONSIDERED SERIOUS (ESPECIALLY THE LATTER, INDICATING STRUCTURE
CORRUPTION) AND THE SERVER EXITS REPORTING STATUS INFORMATION.

Hence calls to memory allocation and freeing do not need to check for success
as only successful operations will return to them!


VERSION HISTORY
---------------
16-JUL-98  MGD  observation indicates VmGeneralInit() initial up to 1024
07-DEC-97  MGD  using LIB$*_VM_* routines for virtual memory manipulation
                (and unbundled from support.c for v5.0)
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include <errno.h>

/* VMS related header files */
#include <descrip.h>
#include <libdef.h>
#include <libvmdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* application related header files */
#include "wasd.h"
#include "error.h"
#include "Vm.h"
#include "support.h"

/******************/
/* global storage */
/******************/

/* align on an octaword boundary (the default anyway!) */
#define VM_ALIGNMENT 8

/* for zone with realloc functions use the first longword to store the size */
#define VM_OFFSET sizeof(unsigned long)

/*
   Where appropriate allocate an extra few bytes for carriage-control
   in buffers, null string terminators, programming errors ;^), etc.
*/
#define VM_ELBOW_ROOM 8

unsigned long  VmCacheVmZoneId,
               VmGeneralVmZoneId,
               VmRequestVmZoneId;

/********************/
/* external storage */
/********************/

#ifdef DBUG
extern boolean Debug;
#else
#define Debug 0 
#endif

extern char  HtmlSgmlDoctype[];
extern char  HttpProtocol[];
extern char  ServerHostPort[];
extern char  SoftwareID[];
extern char  Utility[];

/*****************************************************************************/
/*
Initialize the virtual memory zone general memory will be allocated from.
*/ 

VmGeneralInit
(
int TotalKBytesMax,
int ChunkInBytes
)
{
   static $DESCRIPTOR (ZoneNameDsc, "HTTPd General");
   static unsigned long  Algorithm = LIB$K_VM_QUICK_FIT,
                         AlgorithmArg = 64,
                         Flags = LIB$M_VM_BOUNDARY_TAGS |
                                 LIB$M_VM_GET_FILL0 |
                                 LIB$M_VM_TAIL_LARGE,
                         ExtendSize = 256,
                         InitialSize = 1024,
                         BlockSize = 64;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "VmGeneralInit()\n");

   /* create the cache virtual memory zone */
   status = lib$create_vm_zone (&VmGeneralVmZoneId,
                                &Algorithm, &AlgorithmArg, &Flags,
                                &ExtendSize, &InitialSize, &BlockSize,
                                0, 0, 0, &ZoneNameDsc, 0, 0);
   if (Debug)
      fprintf (stdout, "lib$create_vm_zone() %%X%08.08X id: %08.08X\n",
               status, VmGeneralVmZoneId);
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "lib$create_vm_zone()", FI_LI);
}

/*****************************************************************************/
/*
Allocate memory for general use.
*/ 

char* VmGet (unsigned long ChunkSize)

{
   int  status;
   unsigned long  BaseAddress;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "VmGet() %d\n", ChunkSize);

   ChunkSize += VM_OFFSET + VM_ELBOW_ROOM;
   if (VMSok (status =
       lib$get_vm (&ChunkSize, &BaseAddress, &VmGeneralVmZoneId)))
   {
      *(unsigned long*)BaseAddress = ChunkSize-VM_OFFSET;
      return ((unsigned char*)BaseAddress+VM_OFFSET);
   }
   ErrorExitVmsStatus (status, "lib$get_vm()", FI_LI);
}

/*****************************************************************************/
/*
Expand (or even contract) an individual a general-use chunk.  See VmGet().
*/ 

char* VmRealloc
(
char *ChunkPtr,
unsigned long ChunkSize
)
{
   int  status,
        CurrentChunkSize;
   unsigned long  BaseAddress,
                  FreeBaseAddress;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "VmRealloc() %d %d\n", ChunkPtr, ChunkSize);

   if (ChunkPtr == NULL) return (VmGet (ChunkSize));

   /* if this chunk satisfies the reallocation then just return it */
   CurrentChunkSize = *(unsigned long*)(ChunkPtr-VM_OFFSET);
   if (CurrentChunkSize >= ChunkSize) return (ChunkPtr);

   /* allocate a new, larger chunk */
   ChunkSize += VM_OFFSET + VM_ELBOW_ROOM;
   if (VMSok (status =
       lib$get_vm (&ChunkSize, &BaseAddress, &VmGeneralVmZoneId)))
   {
      /* copy the existing chunk into the new chunk */
      memcpy ((unsigned char*)BaseAddress+VM_OFFSET,
              ChunkPtr, CurrentChunkSize);

      /* free the previous chunk */
      FreeBaseAddress = ChunkPtr-VM_OFFSET;
      if (VMSok (status =
          lib$free_vm (0, &FreeBaseAddress, &VmGeneralVmZoneId)))
      {
         *(unsigned long*)BaseAddress = ChunkSize-VM_OFFSET;
         return ((unsigned char*)BaseAddress+VM_OFFSET);
      }
      ErrorExitVmsStatus (status, "lib$free_vm()", FI_LI);
   }
   ErrorExitVmsStatus (status, "lib$get_vm()", FI_LI);
}

/*****************************************************************************/
/*
Release memory allocated for general use.
*/ 

VmFree (char *ChunkPtr)
                      
{
   int  status;
   unsigned long  BaseAddress;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "VmFree()\n");

   BaseAddress = ChunkPtr-VM_OFFSET;
   if (VMSok (status = lib$free_vm (0, &BaseAddress, &VmGeneralVmZoneId)))
      return;
   ErrorExitVmsStatus (status, "lib$free_vm()", FI_LI);
}

/*****************************************************************************/
/*
Initialize the virtual memory zone for the request structures.
*/ 

VmRequestInit (int MaxConnections)

{
/*
Should be able to use the "fixed" algorithm here but when it's tried the cache
and request heap zones "disappear" from view (lib$find_vm())... although they
can still used!  So for now, and it's not probably a big optimization to miss
out on :^)
*/
#define REQUEST_FIXED 0
#if REQUEST_FIXED

   static $DESCRIPTOR (ZoneNameDsc, "HTTPd Request");
   static unsigned long  Algorithm = LIB$K_VM_FIXED,
                         AlgorithmArg = sizeof(struct RequestStruct),
                         Flags = LIB$M_VM_EXTEND_AREA |
                                 LIB$M_VM_GET_FILL0,
                         InitialSize;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "VmRequestInit() %d\n", MaxConnections);

   /* allow enough for all connections plus a little for the overhead */
   InitialSize = ((MaxConnections * sizeof(struct RequestStruct)) / 512) +
                 (MaxConnections / 5) + 1;

   /* create the request structures' zone */
   status = lib$create_vm_zone (&VmRequestVmZoneId,
                                &Algorithm, &AlgorithmArg, &Flags,
                                0, &InitialSize, 0,
                                0, 0, 0, &ZoneNameDsc, 0, 0);
   if (Debug)
      fprintf (stdout, "lib$create_vm_zone() %%X%08.08X id: %08.08X\n",
               status, VmRequestVmZoneId);
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "lib$create_vm_zone()", FI_LI);

#else /* REQUEST_FIXED */

   static $DESCRIPTOR (ZoneNameDsc, "HTTPd Request");
   static unsigned long  Algorithm = LIB$K_VM_QUICK_FIT,
                         AlgorithmArg = 32,
                         Flags = LIB$M_VM_BOUNDARY_TAGS |
                                 LIB$M_VM_GET_FILL0 |
                                 LIB$M_VM_TAIL_LARGE,
                         InitialSize,
                         BlockSize = 64;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "VmRequestInit() %d\n", MaxConnections);

   InitialSize = (MaxConnections *
                  sizeof(struct RequestStruct)) / 512 + 1;

   /* create the cache virtual memory zone */
   status = lib$create_vm_zone (&VmRequestVmZoneId,
                                &Algorithm, &AlgorithmArg, &Flags,
                                0, &InitialSize, &BlockSize,
                                0, 0, 0, &ZoneNameDsc, 0, 0);
   if (Debug)
      fprintf (stdout, "lib$create_vm_zone() %%X%08.08X id: %08.08X\n",
               status, VmRequestVmZoneId);
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "lib$create_vm_zone()", FI_LI);

#endif /* REQUEST_FIXED */
}

/*****************************************************************************/
/*
Allocate a request structure with associated virtual memory zone ready for
heap allocation.
*/ 

struct RequestStruct* VmGetRequest ()

{
   static $DESCRIPTOR (ZoneNameDsc, "HTTPd Heap");
   static unsigned long  Algorithm = LIB$K_VM_QUICK_FIT,
                         AlgorithmArg = 16,
                         Flags = LIB$M_VM_BOUNDARY_TAGS |
                                 LIB$M_VM_GET_FILL0 |
                                 LIB$M_VM_TAIL_LARGE,
                         ExtendSize = 32,
                         InitialSize = 32,
                         BlockSize = 64;

   static unsigned long RequestStructSize = sizeof(struct RequestStruct);

   register struct RequestStruct  *rqptr;

   int  status;
   unsigned long  BaseAddress;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "VmGetRequest()\n");

   if (VMSok (status =
       lib$get_vm (&RequestStructSize, &BaseAddress, &VmRequestVmZoneId)))
   {
      if (Debug) fprintf (stdout, "lib$get_vm() %%X%08.08X\n", status);
      rqptr = (struct RequestStruct*)BaseAddress;

      /* now create a virtual memory zone for the request's heap */
      rqptr->VmHeapZoneId = 0;
      status = lib$create_vm_zone (&rqptr->VmHeapZoneId,
                                   &Algorithm, &AlgorithmArg, &Flags,
                                   &ExtendSize, &InitialSize, &BlockSize,
                                   0, 0, 0, &ZoneNameDsc, 0, 0);
      if (Debug)
         fprintf (stdout, "lib$create_vm_zone() %%X%08.08X %08.08X\n",
                  status, rqptr->VmHeapZoneId);
      if (VMSnok (status))
      {
         VmFreeRequest (rqptr);
         return (NULL);
      }

      return (rqptr);
   }
   ErrorExitVmsStatus (status, "lib$get_vm()", FI_LI);
}

/*****************************************************************************/
/*
Delete any virtual memory zone created for the request's heap, then return the
request structure to the request virtual memory pool.
*/ 

VmFreeRequest (struct RequestStruct *rqptr)

{
   static unsigned long  RequestStructSize = sizeof(struct RequestStruct);

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "VmFreeRequest() %d %08.08X\n",
               rqptr, rqptr->VmHeapZoneId);

   if (rqptr->VmHeapZoneId)
   {
      /* delete the request's heap's virtual memory zone */
      if (VMSnok (status = lib$delete_vm_zone (&rqptr->VmHeapZoneId)))
         ErrorExitVmsStatus (status, "lib$delete_vm_zone()", FI_LI);
   }

   /* free the request structure's virtual memory */

#ifdef REQUEST_FIXED

   if (VMSok (status =
       lib$free_vm (&RequestStructSize, &rqptr, &VmRequestVmZoneId)))
      return;
   ErrorExitVmsStatus (status, "lib$free_vm()", FI_LI);

#else

   if (VMSok (status =
       lib$free_vm (0, &rqptr, &VmRequestVmZoneId)))
      return;
   ErrorExitVmsStatus (status, "lib$free_vm()", FI_LI);

#endif
}

/*****************************************************************************/
/*
Allocate dynamic memory to an individual thread's heap.  Return a pointer to 
the start of new *usable* memory area if successful, return a NULL if not.
*/ 

char* VmGetHeap
(
struct RequestStruct *rqptr,
unsigned long ChunkSize
)
{
   int  status;
   unsigned long  BaseAddress;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "VmGetHeap() %d %d\n", rqptr, ChunkSize);

   ChunkSize += VM_OFFSET + VM_ELBOW_ROOM;
   if (VMSok (status =
       lib$get_vm (&ChunkSize, &BaseAddress, &rqptr->VmHeapZoneId)))
   {
      *(unsigned long*)BaseAddress = ChunkSize-VM_OFFSET;
      return ((unsigned char*)BaseAddress+VM_OFFSET);
   }
   ErrorExitVmsStatus (status, "lib$get_vm()", FI_LI);
}

/*****************************************************************************/
/*
Expand (or even contract) an individual chunk.  See VmGetHeap().
*/ 

char* VmReallocHeap
(
struct RequestStruct *rqptr,
char *ChunkPtr,
unsigned long ChunkSize
)
{
   int  status,
        CurrentChunkSize;
   unsigned long  BaseAddress,
                  FreeBaseAddress;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "VmReallocHeap() %d %d\n", ChunkPtr, ChunkSize);

   if (ChunkPtr == NULL) return (VmGetHeap (rqptr, ChunkSize));

   /* if this chunk satisfies the reallocation then just return it */
   CurrentChunkSize = *(unsigned long*)(ChunkPtr-VM_OFFSET);
   if (CurrentChunkSize >= ChunkSize) return (ChunkPtr);

   /* allocate a new, larger chunk */
   ChunkSize += VM_OFFSET + VM_ELBOW_ROOM;
   if (VMSok (status =
       lib$get_vm (&ChunkSize, &BaseAddress, &rqptr->VmHeapZoneId)))
   {
      /* copy the existing chunk into the new chunk */
      memcpy ((unsigned char*)BaseAddress+VM_OFFSET,
              ChunkPtr, CurrentChunkSize);

      /* free the previous chunk */
      FreeBaseAddress = ChunkPtr-VM_OFFSET;
      if (VMSok (status =
          lib$free_vm (0, &FreeBaseAddress, &rqptr->VmHeapZoneId)))
      {
         *(unsigned long*)BaseAddress = ChunkSize-VM_OFFSET;
         return ((unsigned char*)BaseAddress+VM_OFFSET);
      }
      ErrorExitVmsStatus (status, "lib$free_vm()", FI_LI);
   }
   ErrorExitVmsStatus (status, "lib$get_vm()", FI_LI);
}

/*****************************************************************************/
/*
Release back into the virtual memory zone one chunk of request heap memory.
*/ 

VmFreeFromHeap
(
struct RequestStruct *rqptr,
char *ChunkPtr
)
{
   int  status;
   unsigned long  FreeBaseAddress;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "VmFreeFromHeap() %d %08.08X %d\n",
               rqptr, rqptr->VmHeapZoneId, ChunkPtr);

   /* free the previous chunk */
   FreeBaseAddress = ChunkPtr-VM_OFFSET;
   if (VMSok (status =
       lib$free_vm (0, &FreeBaseAddress, &rqptr->VmHeapZoneId)))
      return;
   ErrorExitVmsStatus (status, "lib$free_vm()", FI_LI);
}

/*****************************************************************************/
/*
Release back into the virtual memory zone the individually allocated chunks of
memory (the zone is deleted on request structure release).
*/ 

VmFreeHeap (struct RequestStruct *rqptr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "VmFreeHeap() %d %08.08X\n",
               rqptr, rqptr->VmHeapZoneId);

   if (VMSok (status = lib$reset_vm_zone (&rqptr->VmHeapZoneId)))
      return;
   ErrorExitVmsStatus (status, "lib$reset_vm_zone()", FI_LI);
}

/*****************************************************************************/
/*
Initialize the virtual memory zone cache memory will be allocated from.
*/ 

VmCacheInit
(
int TotalKBytesMax,
int ChunkInBytes
)
{
   static $DESCRIPTOR (ZoneNameDsc, "HTTPd Cache");
   static unsigned long  Algorithm = LIB$K_VM_QUICK_FIT,
                         AlgorithmArg = 128,
                         Flags = LIB$M_VM_BOUNDARY_TAGS |
                                 LIB$M_VM_GET_FILL0 |
                                 LIB$M_VM_TAIL_LARGE,
                         ExtendSize,
                         InitialSize,
                         BlockSize = 512;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "VmCacheInit() %d %d\n", TotalKBytesMax, ChunkInBytes);

   if ((InitialSize = ExtendSize = TotalKBytesMax) <= 0) 
      InitialSize = ExtendSize = 32;

   /* create the cache virtual memory zone */
   status = lib$create_vm_zone (&VmCacheVmZoneId,
                                &Algorithm, &AlgorithmArg, &Flags,
                                &ExtendSize, &InitialSize, &BlockSize,
                                0, 0, 0, &ZoneNameDsc, 0, 0);
   if (Debug)
      fprintf (stdout, "lib$create_vm_zone() %%X%08.08X id: %08.08X\n",
               status, VmCacheVmZoneId);
   if (VMSnok (status))
      ErrorExitVmsStatus (status, "lib$create_vm_zone()", FI_LI);
}

/*****************************************************************************/
/*
*/ 

char* VmGetCache (unsigned long ChunkSize)

{
   int  status;
   unsigned long  BaseAddress;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "VmGetCache() %d\n", ChunkSize);

   if (VMSok (status =
       lib$get_vm (&ChunkSize, &BaseAddress, &VmCacheVmZoneId)))
      return ((char*)BaseAddress);
   ErrorExitVmsStatus (status, "lib$get_vm()", FI_LI);
}

/*****************************************************************************/
/*
Release back into the virtual memory zone the individually allocated chunks of
cache memory.
*/ 

VmFreeCache (char *ChunkPtr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "VmFreeCache()\n");

   if (VMSok (status = lib$free_vm (0, &ChunkPtr, &VmCacheVmZoneId)))
      return;
   ErrorExitVmsStatus (status, "lib$free_vm()", FI_LI);
}

/*****************************************************************************/
/*
Return a report on processes' virtual memory usage.  This function blocks while
executing.
*/ 

int VmReport
(
struct RequestStruct *rqptr,
void *NextTaskFunction
)
{
   static unsigned long  ShowVmCode123 = 0,
                         ShowVmCode567 = 4,
                         ShowVmZoneDetail = 3;

   static $DESCRIPTOR (ResponseFaoDsc,
"!AZ\
<HTML>\n\
<HEAD>\n\
!AZ\
<TITLE>HTTPd !AZ ... Virtual Memory Report</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1><NOBR>HTTPd !AZ</NOBR></H1>\n\
<H2>Virtual Memory Report</H2>\n\
!AZ\n\
<PRE>");

   static $DESCRIPTOR (EndPageFaoDsc,
"</PRE>\n\
</BODY>\n\
</HTML>\n");

   register int  cnt;
   register unsigned long  *vecptr;

   int  status,
        Count;
   unsigned short  Length;
   unsigned long  Context,
                  ZoneId;
   unsigned long  FaoVector [32];
   char  Buffer [4096];
   $DESCRIPTOR (BufferDsc, Buffer);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CacheReport()\n");

   rqptr->ResponseStatusCode = 200;
   rqptr->ResponsePreExpired = PRE_EXPIRE_ADMIN;
   if ((rqptr->ResponseHeaderPtr = HttpHeader200Html (rqptr)) == NULL)
   {
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   vecptr = FaoVector;

   *vecptr++ = HtmlSgmlDoctype;
   *vecptr++ = HtmlMetaInfo (rqptr, NULL);

   *vecptr++ = ServerHostPort;
   *vecptr++ = ServerHostPort;
   *vecptr++ = DayDateTime (&rqptr->BinaryTime, 20);

   status = sys$faol (&ResponseFaoDsc, &Length, &BufferDsc, &FaoVector);
   if (VMSnok (status) || status == SS$_BUFFEROVF)
   {
      rqptr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }
   Buffer[Length] = '\0';
   NetWriteBuffered (rqptr, 0, Buffer, Length);

   NetWriteBuffered (rqptr, 0, "\n<HR WIDTH=100%>\n", 17);

   if (VMSnok (lib$show_vm (&ShowVmCode123, &VmWrite, rqptr)))
   {
      rqptr->ErrorTextPtr = "lib$show_vm()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   if (VMSnok (lib$show_vm (&ShowVmCode567, &VmWrite, rqptr)))
   {
      rqptr->ErrorTextPtr = "lib$show_vm()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   NetWriteBuffered (rqptr, 0, "\n", 1);

   Context = 0;
   while ((status = lib$find_vm_zone (&Context, &ZoneId)) == SS$_NORMAL)
   {
      NetWriteBuffered (rqptr, 0, "<HR WIDTH=90%>\n", 15);

      if (VMSnok (status =
          lib$show_vm_zone (&ZoneId, &ShowVmZoneDetail, &VmWrite, rqptr)))
      {
         rqptr->ErrorTextPtr = "lib$show_vm_zone()";
         ErrorVmsStatus (rqptr, status, FI_LI);
         SysDclAst (NextTaskFunction, rqptr);
         return;
      }
   }
   if (status != LIB$_NOTFOU)
   {
      rqptr->ErrorTextPtr = "lib$find_vm_zone()";
      ErrorVmsStatus (rqptr, status, FI_LI);
      SysDclAst (NextTaskFunction, rqptr);
      return;
   }

   NetWriteBuffered (rqptr, 0, "<HR WIDTH=100%>\n", 16);

   NetWriteBuffered (rqptr, NextTaskFunction,
                     EndPageFaoDsc.dsc$a_pointer,
                     EndPageFaoDsc.dsc$w_length);
}

/*****************************************************************************/
/*
Action routine for lib$show_vm*() routines.  Simply write the contents of the
paremeter descriptor to the client plus a newline character.
*/ 

int VmWrite
(
struct dsc$descriptor *DscPtr,
struct RequestStruct *rqptr
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "VmWrite()\n");

   NetWriteBuffered (rqptr, 0, DscPtr->dsc$a_pointer, DscPtr->dsc$w_length);
   NetWriteBuffered (rqptr, 0, "\n", 1);
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/ 

int VmDebug (unsigned long ZoneId)

{
#ifdef DBUG

   static unsigned long  ShowVmZoneDetail = 3;

   int status;
   unsigned long Context;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "VmDebug() %08.08X\n----------\n", ZoneId);

   if (ZoneId)
   {
      if (VMSnok (status =
          lib$show_vm_zone (&ZoneId, &ShowVmZoneDetail, 0, 0)))
         exit (status);
   }
   else
   {
      Context = 0;
      while ((status = lib$find_vm_zone (&Context, &ZoneId)) == SS$_NORMAL)
      {
         if (VMSnok (status =
             lib$show_vm_zone (&ZoneId, &ShowVmZoneDetail, 0, 0)))
            exit (status);
      }
   }
   if (Debug) fprintf (stdout, "----------\n");

#endif
}


/*****************************************************************************/
