/*****************************************************************************/
/*
                                  fetch.c

Script-based pseudo-proxy HTTP request.

(partially written in feeling the way for a full caching proxy server
implementation for WASD)

This program can also be used from within other scripts to fetch a document
from a server and display it as part of the script's output, in various
combinations of header, body, and plain-text, as well just check the document
exists.  Output can also be placed directly into a file.  When used in this way
the program creates a local CLI symbol for the contents of each response
header field (line) named FETCH_field_name, plus one named FETCH_STATUS with
the response status (first) line.  These should be deleted using
DELETE/SYMBOL/LOCAL/ALL before repeated use.


PSEUDO-PROXY
------------
MUST be mapped as "/fetch" to "/cgi-bin/fetch" or the script won't work as a
pseudo-proxy.  This has been done quite deliberately to make this
functionality only available by deliberate act of the server administrator.
The reason should be obvious.  Want your server to be acting as a client
getting any old page from any old server any old user wants?

Rules required in HTTPD$MAP:

  script /fetch/* /cgi-bin/fetch/*
  !(or can be a CGI-plus script)
  !script /fetch/* /cgiplus-bin/fetch/*
  !(substitute asterisks for the following plusses!!!!)
  pass /+.+/+ /+.+/+

URL supplied to server:

  http://local.host.domain/fetch/remote.host.domain/path/file

Using conditional rule mapping this could be used to allow a server known to
the outside world to "serve" pages from selected servers within an intranet,
the well-known server acting as a "firewall" of types.  For example:

  script /fetch/* /cgi-bin/fetch/*
  pass /server.inside.domain/* /inside.server.domain/for-outside/*


INCLUDE A REQUEST IN OTHER SCRIPT OUTPUT
----------------------------------------

Requires a foriegn verb to be assigned.

  $ FETCH="$HT_EXE:FETCH"

To fetch a document from some place:

  $ FETCH "http://host.domain/path/to/file.html"

To fetch an HTML document displayed as it's plain-text source:

  $ FETCH /ESCAPE "http://host.domain/path/to/file.html"

To fetch a request and show the complete response (header and body):

  $ FETCH /HEADER "http://host.domain/path/to/file.html"

Just the request header:

  $ FETCH /HEADER /NOBODY "http://host.domain/path/to/file.html"

To check whether a particular document is available (HEADs the document).
Exits with a success status is the request was successful, with no message and
an error status 2 (STS$K_ERROR) if the server was accessable and the request
was not successful, other error status may be returned (e.g. if the server was
not accessable, etc.)

  $ FETCH /CHECK "http://host.domain/path/to/file.html"

Request document straight into a file:

  $ FETCH /OUTPUT=file.html "http://host.domain/path/to/file.html"


QUALIFIERS
----------
(qualifiers are only required for command-line use in other scripts)
/CHECK          just check whether the document can be accessed
/ESCAPE         escape any HTML-forbidden characters (<, > and &)
/HEADER         show full response header
/METHOD=        HTTP method to be used (e.g. GET (default), HEAD)
/NOBODY         do not output the response body
/OUTPUT=        file name or "TERMINAL" or "SYS$OUTPUT"
/USER_AGENT=    browser user-agent string used when making request


BUILD_DETAILS
-------------
See BUILD_FETCH.COM
(Supports both UCX and NETLIB networking)


VERSION HISTORY
---------------
21-JUN-98  MGD  v1.1.0, revision of CgiVar()
07-MAR-98  MGD  v1.0.0, initial
*/
/*****************************************************************************/

/* if not NETLIB then UCX! */
#ifndef IP_NETLIB
#  undef IP_UCX
#  define IP_UCX 1
#endif

/* standard C header files */
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <descrip.h>
#include <dvidef.h>
#include <iodef.h>
#include <libclidef.h>
#include <ssdef.h>
#include <stsdef.h>

/* Internet-related header files */
#ifdef IP_UCX
#   include <socket.h>
#   include <in.h>
#   include <netdb.h>
#   include <inet.h>
#   include <ucx$inetdef.h>
#endif /* IP_UCX */

#ifdef IP_NETLIB
#   include "sys$library:netlibdef.h"
#endif /* IP_NETLIB */

#define boolean int
#define true 1
#define false 0

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

/******************/
/* global storage */
/******************/

char  Utility [] = "FETCH";

boolean  Debug,
         CheckOnly,
         EscapeHtml,
         IsCgiPlus,
         PseudoProxy,
         ResponseBody,
         ResponseHeader,
         ToTerminal;

int  BufferChunk = 16384,
     BufferCount,
     BufferRemaining,
     BufferSize,
     RemotePort;

char  *BufferPtr,
      *BufferCurrentPtr,
      *CgiPlusEofPtr,
      *CgiQueryStringPtr,
      *CgiScriptNamePtr,
      *FetchUrlPtr,
      *HttpMethodPtr,
      *OutputPtr,
      *UserAgentPtr;

char  RemoteHost [256],
      Request [2048],
      ResponseContentType [256];

struct AnIOsb {
   unsigned short  Status;
   unsigned short  Count;
   unsigned long  Unused;
}  IOsb;


#ifdef IP_UCX

$DESCRIPTOR (InetDeviceDsc, "UCX$DEVICE");

unsigned short  RemoteChannel;

struct sockaddr_in  RemoteSocketName;

struct hostent  *HostEntryPtr;

int  OptionEnabled = 1;

struct {
   unsigned int  Length;
   char  *Address;
} RemoteSocketNameItem =
   { sizeof(RemoteSocketName), &RemoteSocketName };
  
struct {
   unsigned short  Length;
   unsigned short  Parameter;
   char  *Address;
} ReuseAddress =
   { sizeof(OptionEnabled), UCX$C_REUSEADDR, &OptionEnabled },
  ReuseAddressSocketOption =
   { sizeof(ReuseAddress), UCX$C_SOCKOPT, &ReuseAddress };

unsigned short  TcpSocket [2] =
   { UCX$C_TCP, INET_PROTYP$C_STREAM };

#endif /* IP_UCX */


#ifdef IP_NETLIB

unsigned long  RemoteNetLibSocket;

$DESCRIPTOR (RemoteHostDsc, RemoteHost);
$DESCRIPTOR (DataDsc, "");

unsigned long  NetLibAddressListSize = 16,
               NetLibAddressListCount,
               NetLibSocketType = NETLIB_K_TYPE_STREAM,
               NetLibSocketFamily = NETLIB_K_AF_INET,
               NetLibLevel = NETLIB_K_LEVEL_SOCKET,
               NetLibReUseOption = NETLIB_K_OPTION_REUSEADDR,
               NetLibShutdownRecv = NETLIB_K_SHUTDOWN_RECEIVER,
               NetLibShutdownBoth = NETLIB_K_SHUTDOWN_BOTH,
               NetLibOptionOn = 1,
               NetLibSizeofINADDRDEF = sizeof(struct INADDRDEF);
               NetLibSizeofSINDEF = sizeof(struct SINDEF);
               NetLibSizeofOptionOn = sizeof(unsigned long);

#endif /* IP_NETLIB */

/* required function prototypes */
char* CgiVar (char*);

/*****************************************************************************/
/*
*/ 

main
(
int argc,
char *argv[]
)
{
   register char  *cptr;

   int  status,
        acnt;

   /*********/
   /* begin */
   /*********/

   HttpMethodPtr = "GET";
   OutputPtr = "";
   ResponseBody = true;

   /* doing it this way, parameters must be space separated! */
   for (acnt = 1; acnt < argc; acnt++)
   {
      if (Debug) fprintf (stdout, "argv[%d] |%s|\n", acnt, argv[acnt]);
      if (strsame (argv[acnt], "/NOBODY", 6))
      {
         ResponseBody = false;
         continue;
      }
      if (strsame (argv[acnt], "/CHECK", 4))
      {
         CheckOnly = true;
         continue;
      }
      if (strsame (argv[acnt], "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (strsame (argv[acnt], "/ESCAPE_HTML", 4))
      {
         EscapeHtml = true;
         continue;
      }
      if (strsame (argv[acnt], "/HEADER", 4))
      {
         ResponseHeader = true;
         continue;
      }
      if (strsame (argv[acnt], "/METHOD=", 4))
      {
         for (cptr = argv[acnt]; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         HttpMethodPtr = cptr;
         continue;
      }
      if (strsame (argv[acnt], "/OUTPUT=", 4))
      {
         for (cptr = argv[acnt]; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         OutputPtr = cptr;
         continue;
      }
      if (strsame (argv[acnt], "/USER_AGENT=", 4))
      {
         for (cptr = argv[acnt]; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         UserAgentPtr = cptr;
         continue;
      }
      if (argv[acnt][0] == '/')
      {
         fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
                  Utility, argv[acnt]+1);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      /* non-qualifier considered a URL to fetch */
      if (FetchUrlPtr != NULL)
      {
         fprintf (stdout, "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n",
                  Utility, argv[acnt]);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      FetchUrlPtr = argv[acnt];
   }

   /***********/
   /* process */
   /***********/

   /* when to a terminal do no reopen() stdout into binary mode */
   if (strsame (OutputPtr, "TERMINAL", -1) ||
       strsame (OutputPtr, "SYS$OUTPUT", -1))
      ToTerminal = true;

   if (getenv("FETCH$DBUG") != NULL) Debug = true;

   if (Debug)
      system ("show sym *");
   else
   if (!*OutputPtr && !ToTerminal)
   {
      /* reopen output stream so that the '\r' and '\n' are not filtered */
#ifdef __DECC
      if ((stdout = freopen ("SYS$OUTPUT", "w", stdout, "ctx=bin")) == NULL)
         exit (vaxc$errno);
#else
      if ((stdout = freopen ("SYS$OUTPUT", "w", stdout, "rfm=udf")) == NULL)
         exit (vaxc$errno);
#endif
   }

   IsCgiPlus = ((CgiPlusEofPtr = getenv("CGIPLUSEOF")) != NULL);

   do {

      /* with CGIplus this call will block waiting for the next request */
      CgiVar ("");

      CgiScriptNamePtr = cgivar("WWW_SCRIPT_NAME");

      if (*CgiScriptNamePtr)
      {
         if (!strsame (CgiScriptNamePtr, "/fetch", -1))
         {
            fprintf (stdout, "%%%s-E-SCRIPT, must be mapped to \"/fetch\"\n",
                     Utility);
            exit (STS$K_ERROR | STS$M_INHIB_MSG);
         }
         PseudoProxy = true;
      }

      /* URL not supplied on command line, get it from the script path */
      if (FetchUrlPtr == NULL)
      {
         FetchUrlPtr = cgivar("WWW_PATH_INFO");
         CgiQueryStringPtr = cgivar("WWW_QUERY_STRING");
         UserAgentPtr = cgivar("WWW_HTTP_USER_AGENT");
      }

      if (FetchUrlPtr == NULL)
      {
         fprintf (stdout, "%%%s-E-URL, no URL supplied\n", Utility);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }

      if (CheckOnly) HttpMethodPtr = "HEAD";

      FetchResponse ();

      if (CheckOnly)
      {
         if ((cptr = BufferPtr) == NULL)
            exit (STS$K_ERROR | STS$M_INHIB_MSG);
         if (Debug) fprintf (stdout, "|%s|\n", BufferPtr);
         /* this just creates the informational symbols */
         ProcessResponse ();
         /* check the response status code for success */
         while (*cptr && !isspace(*cptr)) cptr++;
         while (*cptr && isspace(*cptr)) cptr++;
         if (*cptr == '2') exit (SS$_NORMAL);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }

      if (*OutputPtr && !ToTerminal)
         if ((stdout = freopen (OutputPtr, "w", stdout)) == NULL)
            exit (vaxc$errno);

      /* if being directly used as a script then use as pseudo-proxy */
      if (PseudoProxy)
         MassageLinks ();
      else
         ProcessResponse ();

      if (IsCgiPlus)
      {
         fprintf (stdout, "%s\n", CgiPlusEofPtr);
         fflush (stdout);
         CgiQueryStringPtr = FetchUrlPtr = NULL;
         free (BufferPtr);
         BufferPtr = NULL;
         BufferSize = 0;
      }

   } while (IsCgiPlus);

   exit (SS$_NORMAL);
}

/****************************************************************************/
/*
Determine the host name and port and then create a request to send to it. 
Connect and send that request, reading the complete response into a buffer of
dynamically allocated memory.
*/

FetchResponse ()

{
   register char  *cptr, *rptr, *sptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FetchResponse() |%s|\n", FetchUrlPtr);

   Request[0] = RemoteHost[0] = '\0';

   rptr = Request;
   for (cptr = HttpMethodPtr; *cptr; *rptr++ = *cptr++);
   *rptr++ = ' ';

   cptr = FetchUrlPtr;
   /* step over introduced "http:" or URL "http://" */
   if (strsame (cptr, "http:", 5)) cptr += 5;
   if (strsame (cptr, "/http:", 6)) cptr += 6;
   /* step over any (other) leading slashes */
   while (*cptr == '/') cptr++;

   sptr = RemoteHost;
   while (*cptr && *cptr != '/' && *cptr != ':')
       *sptr++ = *cptr++;
   *sptr = '\0';
   if (*cptr == ':')
   {
      cptr++;
      RemotePort = atoi(cptr);
      while (*cptr && *cptr != '/') cptr++;
   }
   else
      RemotePort = 80;
   if (Debug) fprintf (stdout, "|%s|%d|%s|\n", RemoteHost, RemotePort, cptr);

   /* rest of the path, which is actually the desired request path */
   if (!*cptr) *rptr++ = '/';
   while (*cptr) *rptr++ = *cptr++;

   /* any non-command-line query string */
   if ((cptr = CgiQueryStringPtr) != NULL)
   {
      if (*cptr)
      {
         if (*cptr) *rptr++ = '?';
         while (*cptr) *rptr++ = *cptr++;
      }
   }

   /* add the HTTP protocol to the request */
   for (cptr = " HTTP/1.0\n"; *cptr; *rptr++ = *cptr++);

   if ((cptr = cgivar("WWW_HTTP_IF_MODIFIED_SINCE")) != NULL)
   {
      if (*cptr)
      {
         for (sptr = "If-Modified-Since: "; *sptr; *rptr++ = *sptr++);
         while (*cptr) *rptr++ = *cptr++;
         *rptr++ = '\n';
      }
   }

   if ((cptr = UserAgentPtr) != NULL)
   {
      if (*cptr)
      {
         for (sptr = "User-Agent: "; *sptr; *rptr++ = *sptr++);
         while (*cptr) *rptr++ = *cptr++;
         *rptr++ = '\n';
      }
   }

   if ((cptr = cgivar("WWW_HTTP_PRAGMA")) != NULL)
   {
      if (*cptr)
      {
         for (sptr = "Pragma: "; *sptr; *rptr++ = *sptr++);
         while (*cptr) *rptr++ = *cptr++;
         *rptr++ = '\n';
      }
   }

   /* end of request header blank line */
   *rptr++ = '\n';
   *rptr = '\0';

   if (Debug) fprintf (stdout, "|%s|\n", Request);

   /*************************/
   /* open proxy connection */
   /*************************/

   /* create the proxy socket */

#ifdef IP_UCX

   /* assign a channel to the internet template device */
   if (VMSnok (status = sys$assign (&InetDeviceDsc, &RemoteChannel, 0, 0)))
      exit (status);

   /* make the channel a TCP, connection-oriented socket */

   if (Debug) fprintf (stdout, "sys$qio() IO$_SETMODE\n");
   status = sys$qiow (0, RemoteChannel, IO$_SETMODE, &IOsb, 0, 0,
                      &TcpSocket, 0, 0, 0, &ReuseAddressSocketOption, 0);

   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status %%X%08.08X\n",
               status, IOsb.Status);
   if (VMSok (status) && VMSnok (IOsb.Status))
      status = IOsb.Status;
   if (VMSnok (status)) exit (status);

   if ((HostEntryPtr = gethostbyname (RemoteHost)) == NULL)
   {
      if (CheckOnly) exit (vaxc$errno | STS$M_INHIB_MSG);
      fprintf (stdout, "%%%s-E-CONNECT, %s\n", Utility, RemoteHost);
      exit (vaxc$errno);
   }

   RemoteSocketName.sin_family = AF_INET;
   RemoteSocketName.sin_port = htons (RemotePort);
   memcpy (&RemoteSocketName.sin_addr.s_addr, HostEntryPtr->h_addr, 4);

   if (Debug) fprintf (stdout, "sys$qiow() IO$_ACCESS\n");
   status = sys$qiow (0, RemoteChannel, IO$_ACCESS, &IOsb, 0, 0,
                      0, 0, &RemoteSocketNameItem, 0, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status %%X%08.08X\n",
               status, IOsb.Status);

   if (VMSok (status) && VMSnok (IOsb.Status))
      status = IOsb.Status;
   if (VMSnok (status))
   {
      if (CheckOnly) exit (status | STS$M_INHIB_MSG);
      fprintf (stdout, "%%%s-E-CONNECT, %s\n", Utility, RemoteHost);
      exit (status);
   }

#endif /* IP_UCX */

#ifdef IP_NETLIB

   if (Debug) fprintf (stdout, "netlib_socket()\n");
   status = netlib_socket (&RemoteNetLibSocket, &NetLibSocketType,
                           &NetLibSocketFamily);
   if (Debug) fprintf (stdout, "netlib_socket() %%X%08.08X\n", status);
   if (VMSnok (status)) exit (status);

   RemoteHostDsc.dsc$a_pointer = RemoteHost;
   RemoteHostDsc.dsc$w_length = strlen(RemoteHost);

   status = netlib_connect_by_name (&RemoteNetLibSocket,
                                    &RemoteHostDsc,
                                    &RemotePort,
                                    &IOsb, 0, 0);
   if (Debug)
      fprintf (stdout,
         "netlib_connect_by_name() %%X%08.08X IOsb %%X%08.08X\n",
         status, IOsb.Status);

   if (VMSok (status) && VMSnok (IOsb.Status))
      status = IOsb.Status;
   if (VMSnok (status))
   {
      if (CheckOnly) exit (status | STS$M_INHIB_MSG);
      fprintf (stdout, "%%%s-E-CONNECT, %s\n", Utility, RemoteHost);
      exit (status);
   }

#endif /* IP_NETLIB */

   /****************/
   /* send request */
   /****************/

   if (Debug) fprintf (stdout, "request |%s|\n", Request);

#ifdef IP_UCX

   status = sys$qiow (0, RemoteChannel, IO$_WRITEVBLK, 0, 0, 0,
                      Request, rptr-Request, 0, 0, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status %%X%08.08X\n",
               status, IOsb.Status);

#endif /* IP_UCX */

#ifdef IP_NETLIB

   DataDsc.dsc$a_pointer = Request;
   DataDsc.dsc$w_length = rptr-Request;

   status = netlib_write (&RemoteNetLibSocket, &DataDsc, 0, 0, &IOsb, 0, 0);
   if (Debug)
      fprintf (stdout, "netlib_write() %%X%08.08X IOsb.Status %%X%08.08X\n",
               status, IOsb.Status);

#endif /* IP_NETLIB */

   if (VMSok (status) && VMSnok (IOsb.Status))
      status = IOsb.Status;
   if (VMSnok (status))
   {
      if (CheckOnly) exit (status | STS$M_INHIB_MSG);
      fprintf (stdout, "%%%s-E-REQUEST, %s\n%s\n", Utility, RemoteHost, Request);
      exit (status);
   }

   /*********************/
   /* read the response */
   /*********************/

   BufferCount = BufferSize = BufferRemaining = 0;
   BufferCurrentPtr = BufferPtr = NULL;

   for (;;)
   {
      if (BufferCount == BufferSize)
      {
         BufferSize += (BufferRemaining = BufferChunk);
         if ((BufferPtr = realloc (BufferPtr, BufferSize+1)) == NULL)
            exit (vaxc$errno);
         BufferCurrentPtr = BufferPtr + BufferSize - BufferRemaining;
      }

#ifdef IP_UCX

      sys$qiow (0, RemoteChannel, IO$_READVBLK, &IOsb, 0, 0,
                BufferCurrentPtr, BufferRemaining, 0, 0, 0, 0);
      if (Debug)
         fprintf (stdout, "sys$qiow() IOsb.Status %%X%08.08X %d bytes\n",
                  IOsb.Status, IOsb.Count);

#endif /* IP_UCX */

#ifdef IP_NETLIB

      DataDsc.dsc$a_pointer = BufferCurrentPtr;
      DataDsc.dsc$w_length = BufferRemaining;

      status = netlib_read (&RemoteNetLibSocket, &DataDsc,
                            0, 0, 0, 0, &IOsb, 0, 0);
      if (Debug)
         fprintf (stdout, "netlib_read() %%X%08.08X IOsb.Status %%X%08.08X\n",
                  status, IOsb.Status);

#endif /* IP_NETLIB */

      if (VMSok (status) && VMSnok (IOsb.Status))
         status = IOsb.Status;
      if (VMSnok (IOsb.Status)) break;

      BufferCount += IOsb.Count;
      BufferCurrentPtr += IOsb.Count;
      BufferRemaining -= IOsb.Count;

      if (Debug)
         fprintf (stdout, "Remaining: %d Count: %d\n",
                  BufferRemaining, BufferCount);
   }

   /****************/
   /* close socket */
   /****************/

#ifdef IP_UCX

   status = sys$qiow (0, RemoteChannel, IO$_DEACCESS | IO$M_SHUTDOWN,
                      &IOsb, 0, 0, 0, 0, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb.Status %%X%08.08X\n",
               status, IOsb.Status);

   status = sys$dassgn (RemoteChannel);
   if (Debug) fprintf (stdout, "sys$dassgn() %%X%08.08X\n", status);

#endif /* IP_UCX */

#ifdef IP_NETLIB

   status = netlib_shutdown (&RemoteNetLibSocket, &NetLibShutdownBoth,
                             &IOsb, 0, 0);
   if (Debug)
      fprintf (stdout, "netlib_shutdown %%X%08.08X IOsb %%X%08.08X\n",
               status, IOsb.Status);

   status = netlib_close (&RemoteNetLibSocket);
   if (Debug) fprintf (stdout, "netlib_close() %%X%08.08X\n", status);

#endif /* IP_NETLIB */

   BufferPtr[BufferCount] = '\0';
}

/****************************************************************************/
/*
Output the response.  Can output only the header, only the body, both, and
also escape the HTML-forbidden characters "<", ">", and "&" so an HTML file
can be presented as plain text inside another HTML document.
*/

ProcessResponse ()

{
   register int  cnt;
   register char  *bptr, *cptr, *sptr;

   char  SymbolName [256],
         SymbolValue [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessResponse()\n");

   cptr = BufferPtr;
   cnt = BufferCount;

   /* process the response header */
   ResponseContentType[0] = '\0';
   while (cnt)
   {
      /* create a symbol representing the response header line */
      sptr = SymbolName;
      for (bptr = Utility; *bptr; *sptr++ = *bptr++);
      *sptr++ = '_';
      if (!memcmp (cptr, "HTTP/", 5))
      {
         for (bptr = "STATUS"; *bptr; *sptr++ = *bptr++);
         *sptr = '\0';
         bptr = cptr;
      }
      else
      {
         for (bptr = cptr;
              *bptr && *bptr != ':' &&
                 *bptr != ' ' && *bptr != '\t' &&
                 *bptr != '\r' && *bptr != '\n';
              bptr++)
         {
            if (!isalnum(*bptr))
               *sptr++ = '_';
            else
               *sptr++ = *bptr;
         }
         *sptr = '\0';
      }
      if (*bptr == ':') bptr++;
      while (*bptr && (*bptr == ' ' || *bptr == '\t')) bptr++;
      sptr = SymbolValue;
      while (*bptr && *bptr != '\r' && *bptr != '\n')
         *sptr++ = *bptr++;
      *sptr = '\0';
      SetLocalSymbol (SymbolName, SymbolValue);

      if (toupper (cptr[0]) == 'C' && tolower(cptr[1]) == 'o' &&
          strsame (cptr, "Content-Type:", 13))
      {
         cptr += 13;
         cnt -= 13;
         while (cnt && isspace(*cptr))
         {
            cptr++;
            cnt--;
         }
         sptr = ResponseContentType;
         while (cnt && *cptr != '\r' && *cptr != '\n')
         {
            *sptr++ = *cptr++;
            cnt--;
         }
         *sptr = '\0';
      }

      /* skip to end-of-line */
      while (cnt && *cptr != '\r' && *cptr != '\n')
      {
         cptr++;
         cnt--;
      }
      if (cnt && *cptr == '\r')
      {
         cptr++;
         cnt--;
      }
      if (cnt && *cptr == '\n')
      {
         cptr++;
         cnt--;
      }

      /* break if end of request header */
      if (cnt && *cptr == '\n')
      {
         cptr++;
         cnt--;
         break;
      }
      if (cnt >= 2 && cptr[0] == '\r' && cptr[1] == '\n')
      {
         cptr += 2;
         cnt -= 2;
         break;
      }
   }
   if (Debug)
      fprintf (stdout, "ResponseContentType |%s|\n", ResponseContentType);

   if (CheckOnly) return;

   if (ResponseHeader)
   {
      if (cptr > BufferPtr)
         if (EscapeHtml)
            fprintf (stdout, "%*.*s",
                     cptr-BufferPtr, cptr-BufferPtr, BufferPtr);
         else
            fwrite (BufferPtr, cptr-BufferPtr, 1, stdout);
   }

   if (ResponseBody)
   {
      BufferCurrentPtr = cptr;

      if (EscapeHtml && strsame (ResponseContentType, "text/html", -1))
      {
         while (cnt)
         {
            switch (*cptr)
            {
               case '<' :
                  if (cptr > BufferCurrentPtr)
                     fprintf (stdout, "%*.*s",
                              cptr-BufferCurrentPtr, cptr-BufferCurrentPtr, 
                              BufferCurrentPtr);
                  BufferCurrentPtr = cptr+1;
                  fprintf (stdout, "%s", "&lt;");
                  break;            
               case '>' :
                  if (cptr > BufferCurrentPtr)
                     fprintf (stdout, "%*.*s",
                              cptr-BufferCurrentPtr, cptr-BufferCurrentPtr, 
                              BufferCurrentPtr);
                  BufferCurrentPtr = cptr+1;
                  fprintf (stdout, "%s", "&gt;");
                  break;            
               case '&' :
                  if (cptr > BufferCurrentPtr)
                     fprintf (stdout, "%*.*s",
                              cptr-BufferCurrentPtr, cptr-BufferCurrentPtr, 
                              BufferCurrentPtr);
                  BufferCurrentPtr = cptr+1;
                  fprintf (stdout, "%s", "&amp;");
                  break;            
            }
            cptr++;
            cnt--;
         }
         /* write anything left */
         if (cptr > BufferCurrentPtr)
            fprintf (stdout, "%*.*s",
                     cptr-BufferCurrentPtr, cptr-BufferCurrentPtr, 
                     BufferCurrentPtr);
      }
      else
      {
         /* write response as-is */
         if (cnt) fwrite (cptr, cnt, 1, stdout);
      }
   }
}

/****************************************************************************/
/*
Check if the document is HTML text, if not then just return it direct to the
client.  If HTML the adjust checked-for links to use the pseudo-proxy by
inserting the script name and if necessary source host name into the URL.
Checks <A..>, <BODY..>, <FRAME..>, <FORM..> and <IMG..> tags for links.
*/

MassageLinks ()

{
   register int  cnt;
   register char  *cptr, *sptr;

   boolean  LinkPath;
   char  *ContentLengthPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MassageLinks()\n");

   ContentLengthPtr = NULL;

   cptr = BufferPtr;
   cnt = BufferCount;

   while (cnt)
   {
      if (toupper (cptr[0]) == 'C' && tolower(cptr[1]) == 'o' &&
          strsame (cptr, "Content-Type:", 13))
      {
         sptr = cptr + 13;
         if (Debug) fprintf (stdout, "|%13.13s|\n", cptr);
         while (*sptr && isspace(*sptr)) *sptr++;
         if (Debug) fprintf (stdout, "|%20.20s|\n", sptr);
         if (!strsame (sptr, "text/html", 9))
         {
            /* not HTML, just output original data */
            fwrite (BufferPtr, BufferCount, 1, stdout);
            return;
         }
      }

      if (toupper (cptr[0]) == 'C' && tolower(cptr[1]) == 'o' &&
          strsame (cptr, "Content-Length:", 15))
      {
         ContentLengthPtr = cptr;
         if (Debug) fprintf (stdout, "|%13.13s|\n", cptr);
      }

      /* skip to end-of-line */
      while (cnt && *cptr != '\r' && *cptr != '\n')
      {
         cptr++;
         cnt--;
      }
      if (cnt && *cptr == '\r')
      {
         cptr++;
         cnt--;
      }
      if (cnt && *cptr == '\n')
      {
         cptr++;
         cnt--;
      }

      /* break if end of request header */
      if (cnt && *cptr == '\n')
         break;
      if (cnt >= 2 && cptr[0] == '\r' && cptr[1] == '\n')
         break;
   }

   /* munge the content-length seeing we're probably going to alter it */
   if (ContentLengthPtr != NULL)
      memcpy (ContentLengthPtr, "C_ntent-L_ngth:", 15);

   BufferCurrentPtr = BufferPtr;

   /* output the request header */
   fwrite (BufferCurrentPtr, cptr - BufferCurrentPtr, 1, stdout);
   BufferCurrentPtr = cptr;

   LinkPath = false;

   while (cnt)
   {
      if (cptr[0] == '<' && toupper(cptr[1]) == 'A' && isspace(cptr[2]))
      {
         /*********/
         /* <A..> */
         /*********/

         if (Debug) fprintf (stdout, "<A |%80.80s|\n", cptr);
         while (cnt)
         {
            if (*cptr == '>') break;

            if (toupper(cptr[0]) == 'H' && toupper(cptr[1]) == 'R' &&
                toupper(cptr[2]) == 'E' && toupper(cptr[3]) == 'F' &&
                cptr[4] == '=')
            {
               cptr += 5;
               cnt -= 5;
               break;
            }
            cptr++;
            cnt--;
         }
         if (Debug) fprintf (stdout, "HREF= |%80.80s|\n", cptr);
         LinkPath = true;
      }
      else
      if (cptr[0] == '<' && toupper(cptr[1]) == 'I' &&
          toupper(cptr[2]) == 'M' && toupper(cptr[3]) == 'G' &&
          isspace(cptr[4]))
      {
         /***********/
         /* <IMG..> */
         /***********/

         if (Debug) fprintf (stdout, "<IMG |%80.80s|\n", cptr);
         while (cnt)
         {
            if (*cptr == '>') break;

            if (toupper(cptr[0]) == 'S' && toupper(cptr[1]) == 'R' &&
                toupper(cptr[2]) == 'C' && cptr[3] == '=')
            {
               cptr += 4;
               cnt -= 4;
               break;
            }
            cptr++;
            cnt--;
         }
         if (Debug) fprintf (stdout, "SRC= |%80.80s|\n", cptr);
         LinkPath = true;
      }
      else
      if (cptr[0] == '<' && toupper(cptr[1]) == 'F' &&
          toupper(cptr[2]) == 'R' && toupper(cptr[3]) == 'A' &&
          toupper(cptr[4]) == 'M' && toupper(cptr[5]) == 'E' &&
          isspace(cptr[6]))
      {
         /*************/
         /* <FRAME..> */
         /*************/

         if (Debug) fprintf (stdout, "<FRAME |%80.80s|\n", cptr);
         while (cnt)
         {
            if (*cptr == '>') break;

            if (toupper(cptr[0]) == 'S' && toupper(cptr[1]) == 'R' &&
                toupper(cptr[2]) == 'C' && cptr[3] == '=')
            {
               cptr += 4;
               cnt -= 4;
               break;
            }
            cptr++;
            cnt--;
         }
         if (Debug) fprintf (stdout, "SRC= |%80.80s|\n", cptr);
         LinkPath = true;
      }
      else
      if (cptr[0] == '<' && toupper(cptr[1]) == 'B' &&
          toupper(cptr[2]) == 'O' && toupper(cptr[3]) == 'D' &&
          toupper(cptr[4]) == 'Y' && isspace(cptr[5]))      
      {
         /************/
         /* <BODY..> */
         /************/

         if (Debug) fprintf (stdout, "<BODY |%80.80s|\n", cptr);
         while (cnt)
         {
            if (*cptr == '>') break;

            if (toupper(cptr[0]) == 'B' && toupper(cptr[1]) == 'A' &&
                toupper(cptr[2]) == 'C' && toupper(cptr[3]) == 'K' &&
                toupper(cptr[4]) == 'G' && toupper(cptr[5]) == 'R' &&
                toupper(cptr[6]) == 'O' && toupper(cptr[7]) == 'U' &&
                toupper(cptr[8]) == 'N' && toupper(cptr[9]) == 'D' &&
                cptr[10] == '=')
            {
               cptr += 11;
               cnt -= 11;
               break;
            }
            cptr++;
            cnt--;
         }
         if (Debug) fprintf (stdout, "SRC= |%80.80s|\n", cptr);
         LinkPath = true;
      }
      else
      if (cptr[0] == '<' && toupper(cptr[1]) == 'F' &&
          toupper(cptr[2]) == 'O' && toupper(cptr[3]) == 'R' &&
          toupper(cptr[4]) == 'M' && isspace(cptr[5]))      
      {
         /************/
         /* <FORM..> */
         /************/

         if (Debug) fprintf (stdout, "<FORM |%80.80s|\n", cptr);
         while (cnt)
         {
            if (*cptr == '>') break;

            if (toupper(cptr[0]) == 'A' && toupper(cptr[1]) == 'C' &&
                toupper(cptr[2]) == 'T' && toupper(cptr[3]) == 'I' &&
                toupper(cptr[4]) == 'O' && toupper(cptr[5]) == 'N' &&
                cptr[6] == '=')
            {
               cptr += 7;
               cnt -= 7;
               break;
            }
            cptr++;
            cnt--;
         }
         if (Debug) fprintf (stdout, "ACTION= |%80.80s|\n", cptr);
         LinkPath = true;
      }

      if (*cptr == '>')
      {
         cptr++;
         cnt--;
         continue;
      }

      if (LinkPath)
      {
         LinkPath = false;

         while (*cptr && *cptr != '>' && (isspace(*cptr) || *cptr == '\"'))
         {
            cptr++;
            cnt--;
            continue;
         }
         if (Debug) fprintf (stdout, "link |%80.80s|\n", cptr);

         if (*cptr == '>')
         {
            cptr++;
            cnt--;
            continue;
         }

         if (*cptr == '/' ||
             strsame (cptr, "http://", 7))
         {
            if (cptr > BufferCurrentPtr)
               fwrite (BufferCurrentPtr, cptr - BufferCurrentPtr, 1, stdout);
            BufferCurrentPtr = cptr;

            fwrite (CgiScriptNamePtr, strlen(CgiScriptNamePtr), 1, stdout);
            fwrite ("/", 1, 1, stdout);
            if (strsame (cptr, "http://", 7))
               BufferCurrentPtr = cptr + 7;
            else
               fwrite (RemoteHost, strlen(RemoteHost), 1, stdout);
         }
      }

      cptr++;
      cnt--;
   }

   if (cptr > BufferCurrentPtr)
      fwrite (BufferCurrentPtr, cptr - BufferCurrentPtr, 1, stdout);
}

/*****************************************************************************/
/*
*/

int SetLocalSymbol
(
char  *SymbolName,
char  *SymbolValue
)
{
   static int  LocalSymbol = LIB$K_CLI_LOCAL_SYM;
   static $DESCRIPTOR (SymbolNameDsc, "");
   static $DESCRIPTOR (SymbolValueDsc, "");

   int  status;

   /*********/
   /* begin */
   /*********/

   SymbolNameDsc.dsc$w_length = strlen(SymbolName);
   SymbolNameDsc.dsc$a_pointer = SymbolName;

   SymbolValueDsc.dsc$w_length = strlen(SymbolValue);
   SymbolValueDsc.dsc$a_pointer = SymbolValue;

   status = lib$set_symbol (&SymbolNameDsc, &SymbolValueDsc, &LocalSymbol);
   if (Debug) fprintf (stdout, "lib$set_symbol() %%X%08.08X\n", status);
   return (status);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/
/*
This function provides a simple, uniform mechanism for a C Language script to
access CGI variable values transparently in both the standard CGI and the
CGIplus environments.  Requires the global storage 'CgiPlusEofPtr' to be
non-NULL to indicate a CGIplus environment.

It is completely self-contained. Simply call the function with the CGI variable
name value to be accessed and if that CGI variable exists a pointer to the
value will be returned.  Non-existant variables return CGIPLUS_NONE (see
below).  The values pointed to should NOT be modified in any way (copy the
value if this is required). 

Supplying an empty variable name (i.e. "") will cause the function to block
until the arrival of a request, upon which it just returns a CGIPLUS_NONE
pointer, allowing start-of-request synchronization.  This will also release
allocated memory and reset in preparation for reading the next request, an
alternative to calling with a NULL parameter at the end of request processing
(see below).

For the CGIplus environment supplying a wilcard variable name (i.e. "*")
returns each CGIplus variable string (line from CGIPLUSIN) in turn, with no
more being indicated by a CGIPLUS_NONE pointer.

At the end of a CGIplus request call the function with a NULL parameter to
release allocated memory and reset ready for processing the next request's
variables.  It is not necessary to do this if calling with an empty parameter
to synchronize on the start of a request (see above).

The CGIplus functionality works by initially reading all lines from
CGIPLUSIN until the terminating blank line is encountered. These lines are
stored as a series of contiguous null-terminated strings, terminated by an
empty string, in allocated memory. These strings are searched for the required
variable name and a pointer to its value returned if found.

The value CGIPLUS_NONE shold be set according to the requirements of the
application the function is included within.  If calling functions can handle
NULL pointers being returned then set the value to NULL.  If non-NULL pointers
are prefered then set the value to an empty string.

21-JUN-98  MGD  version 2 of this function attempts to increase efficiency by
                reducing the number of realloc()s required and introducing the
                length of each variable improving search behaviour
01-JUL-97  MGD  version 1
*/

char* CgiVar (char *VarName)

{
#ifndef CGIPLUS_DEBUG
#   define CGIPLUS_DEBUG 0
#endif
/** #  define CGIPLUS_NONE NULL **/
#define CGIPLUS_NONE ""
#define CGIPLUS_LINE_SIZE 1024
#define CGIPLUS_CHUNK 2048

   static int  VarPtrSize = 0;
   static char  *NextPtr = NULL,
                *VarPtr = NULL;

   register char  *cptr, *sptr, *zptr;

   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

#if CGIPLUS_DEBUG
   fprintf (stdout, "CgiVar() |%s|\n", VarName);
#endif

   if (VarName == NULL || !VarName[0])
   {
      /* end of request, release memory and reinitialize */
      if (VarPtr != NULL)
      {
         free (VarPtr);
         NextPtr = VarPtr = NULL;
      }
      if (VarName == NULL) return (CGIPLUS_NONE);
   }

   if (CgiPlusEofPtr == NULL)
   {
      /* standard CGI environment, get variable from environment */
      if (!VarName[0]) return (CGIPLUS_NONE);
      if ((cptr = getenv(VarName)) == NULL)
         return (CGIPLUS_NONE);
      else
         return (cptr);
   }

   if (VarPtr == NULL)
   {
      /* read lines containing CGIplus variables from <CGIPLUSIN> */
      int  CurrentVarLength;
      char  Line [CGIPLUS_LINE_SIZE];
      char  *LengthPtr;
      FILE  *CgiPlusIn;
                                                 
      CurrentVarLength = VarPtrSize = 0;
      zptr = NULL;
      if ((CgiPlusIn = fopen (getenv("CGIPLUSIN"), "r")) == NULL)
         exit (vaxc$errno);
      while (fgets (Line, sizeof(Line), CgiPlusIn) != NULL)
      {
         /* ignore any non-alphabetic line */
         if (Line[0] != '\n' && !isalpha(Line[0])) continue;
         /* point to the next name=value storage in the variable buffer */
         LengthPtr = VarPtr + CurrentVarLength;
         sptr = LengthPtr + sizeof(unsigned short);
         for (cptr = Line; *cptr; *sptr++ = *cptr++)
         {
            if (sptr < zptr) continue;
            /* out of storage (or first variable) get more (or some) memory */
            if ((VarPtr = realloc (VarPtr, VarPtrSize+CGIPLUS_CHUNK)) == NULL)
               exit (vaxc$errno);
            VarPtrSize += CGIPLUS_CHUNK;
            /* recalculate pointers */
            LengthPtr = VarPtr + CurrentVarLength;
            sptr = LengthPtr + sizeof(unsigned short) + (cptr - Line);
            zptr = VarPtr + VarPtrSize;
         }
         /* remove the trailing newline */
         *--sptr = '\0';
         /* insert the length of this name=value pair immediately before it */
         *((unsigned short*)LengthPtr) = Length = cptr - Line;
         /* adjust the value of the current variable storage space used */
         CurrentVarLength += Length + sizeof(unsigned short);
         /* first empty line signals the end of CGIplus variables */
         if (Line[0] == '\n') break;
      }
      fclose (CgiPlusIn);
      if (VarPtr == NULL) return (CGIPLUS_NONE);

#if CGIPLUS_DEBUG
      fprintf (stdout, "VarPtr: %d VarPtrSize: %d CurrentVarLength: %d\n",
               VarPtr, VarPtrSize, CurrentVarLength);
      sptr = VarPtr;
      for (sptr = VarPtr;
           Length = *((unsigned short*)sptr);
           sptr += sizeof(unsigned short) + Length)
         fprintf (stdout, "%3d |%s|\n", Length, sptr+sizeof(unsigned short));
#endif
   }

   /* if just waiting for next request return now it's arrived */
   if (!VarName[0]) return (CGIPLUS_NONE);

   if (VarName[0] == '*')
   {
      /* return each stored line one at a time */
      if (NextPtr == NULL) NextPtr = VarPtr;
      if ((Length = *((unsigned short*)(sptr = NextPtr))) == 0)
      {
         NextPtr = NULL;
#if CGIPLUS_DEBUG
         fprintf (stdout, "|%s|=CGIPLUS_NONE\n", VarName);
#endif
         return (CGIPLUS_NONE);
      }
      sptr += sizeof(unsigned short);
      NextPtr = sptr + Length;
#if CGIPLUS_DEBUG
      fprintf (stdout, "|%s=%s|\n", VarName, sptr);
#endif
      return (sptr);
   }

   /* search CGIplus variable strings for required variable */
   for (sptr = VarPtr;
        Length = *((unsigned short*)sptr);
        sptr = zptr + Length)
   {
      /* simple comparison between supplied and in-string variable names */
      zptr = sptr = sptr + sizeof(unsigned short);
      cptr = VarName;
      while (*cptr && *sptr && *sptr != '=')
      {
         if (toupper(*cptr) != toupper(*sptr)) break;
         cptr++;
         sptr++;
      }
      if (*cptr || *sptr != '=') continue;
      /* found, return a pointer to the value */
#if CGIPLUS_DEBUG
      fprintf (stdout, "|%s=%s|\n", VarName, sptr+1);
#endif
      return (sptr+1);
   }
   /* not found */
#if CGIPLUS_DEBUG
   fprintf (stdout, "|%s|=CGIPLUS_NONE\n", VarName);
#endif
   return (CGIPLUS_NONE);
}

/****************************************************************************/

