/*****************************************************************************/
/*
                                 glist.c

CGI-compliant script to list graphics.  Call with the path to the directory 
and it returns a form allowing selection of what graphics are to be displayed.  
Each directory of graphics is displayed with the a list of subdirectories, 
the path to each graphic in the directory, and the image generated by the 
graphic. 

Page colouration may be specified via the appropriate command-line qualifiers
(or corresponding logical name). Defaults for any not specified.  Specifiy as
/WHATEVER="" to NOT specify the corresponding colour (i.e. leave it to the
browser). See "Qualifiers" section below, and also about the logical name
"GLIST$PARAM".

An example of changing the page colour to white and the banner to red!

  /PBGCOLOR="#ffffff" /PHBGCOLOR="#ff0000"


CGI VARIABLE NAMES
------------------
WWW_FORM_GIF            non-null includes GIF images
WWW_FORM_JPG            non-null includes JPEG images
WWW_FORM_JPG_INLINE     non-null includes JPEG as in-line images
WWW_FORM_LIST           non-null list the directory now
WWW_FORM_XBM            non-null includes X-bitmap images
WWW_PATH_INFO           the URL path component
WWW_PATH_TRANSLATED     the VMS equivalent of the URL path component
WWW_QUERY_STRING        any of the form components
WWW_SCRIPT_NAME         the name of the script being executed


LOGICAL NAMES
-------------
GLIST$DBUG              turns on all "if (Debug)" statements
GLIST$PARAM             equivalent to (overrides) the command line
                        parameters/qualifiers (define as a system-wide logical)


QUALIFIERS
----------
/BUTTONS=       string containing button labels/paths
/DBUG           turns on all "if (Debug)" statements
/PBACKGROUND=   <body> background image path
/PBGCOLOR=      <body> background colour
/PBBGCOLOR=     button background color
/PHBGCOLOR=     page heading background color
/PHLOCAL=       local information to be included in header
/PHTEXT=        page heading text colour
/PLAYOUT=       1 is coloured header & buttons, 2 is text & horizontal rules
/PLINK=         <body> link colour
/PTEXT=         <body> text colour
/PVLINK=        <body> visited link colour


BUILD
-----
See BUILD_GLIST.COM


VERSION HISTORY
---------------
24-JUL-98  MGD  v1.1.1, suppress table background colours if empty
20-MAY-98  MGD  v1.1.0, cosmetic changes
09-FEB-96  MGD  v1.0.0, quick hack
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "GLIST AXP-1.1.1";
#else
   char SoftwareID [] = "GLIST VAX-1.1.1";
#endif

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <stat.h>

#include <atrdef.h>
#include <descrip.h>
#include <dvidef.h>
#include <fibdef.h>
#include <iodef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

#define boolean int
#define true 1
#define false 0

/* macro provides NULL pointer if CGI variable does not exist */
#define GetCgiVarIfExists(CharPointer,CgiVariableName) \
   CharPointer = getenv(CgiVariableName)

/* macro provides pointer to empty string even if CGI variable does not exist */
#define GetCgiVar(CharPointer,CgiVariableName) \
   if ((CharPointer = getenv(CgiVariableName)) == NULL) \
       CharPointer = ""; \
   if (Debug) fprintf (stdout, "%s |%s|\n", CgiVariableName, CharPointer);

#define DEFAULT_BUTTONS ""

/* this macro just plugs in some script-specific code into ButtonBar() */
#define SCRIPT_SPECIFIC_BUTTON_CODE /* no script-internal buttons */

#define DEFAULT_PS_BGCOLOR        "#ffffff"
#define DEFAULT_PS_TEXT           "#000000"
#define DEFAULT_PS_LINK           "#0000ff"
#define DEFAULT_PS_VLINK          "#0000ff"
#define DEFAULT_PS_HEADBGCOLOR    "#cccccc"
#define DEFAULT_PS_HEADBORDER     "0"
#define DEFAULT_PS_HEADTEXT       "#000000"
#define DEFAULT_PS_BUTTONBGCOLOR  "#ffffff"
#define DEFAULT_PS_BUTTONBORDER   "1"

#define PS_BACKGROUND     0
#define PS_BGCOLOR        1
#define PS_TEXT           2
#define PS_LINK           3
#define PS_VLINK          4
#define PS_HEADBGCOLOR    5
#define PS_HEADTEXT       6
#define PS_HEADBORDER     7
#define PS_BUTTONBGCOLOR  8
#define PS_BUTTONBORDER   9
#define PS_BODYTAG       10
#define PS_LAYOUT        11
#define PS_HEADLOCAL     12
#define PS_HEADPADDING   13

char  *PageScheme [16];

char  ErrorInternal [] = "Internal error.",
      Utility [] = "GLIST";

char  Http200Header [] =
"HTTP/1.0 200 Document follows.\r\n\
Content-Type: text/html\r\n\
\r\n";

char  Http404Header [] =
"HTTP/1.0 404 Error report follows.\r\n\
Content-Type: text/html\r\n\
\r\n";

boolean  Debug,
         HttpHasBeenOutput,
         IncludeGif,
         IncludeJpg,
         IncludeXbm,
         JpgInline;

char  *ButtonPtr = DEFAULT_BUTTONS,
      *CgiFormGifPtr,
      *CgiFormJpgPtr,
      *CgiFormJpgInlinePtr,
      *CgiFormListPtr,
      *CgiFormXbmPtr,
      *CgiPathInfoPtr,
      *CgiPathTranslatedPtr,
      *CgiQueryStringPtr,
      *CgiScriptNamePtr;
      
/* required function prototypes */
char* CommaNumber (unsigned long);

/*****************************************************************************/
/*
*/

main
(
int argc,
char *argv[]
)
{
   int  acnt;

   /*********/
   /* begin */
   /*********/

   if (getenv ("GLIST$DBUG") != NULL) Debug = true;

   GetParameters ();

   if (Debug)
      system ("show sym *");
   else
   {
      /* reopen output stream so that the '\r' and '\n' are not filtered */
#ifdef __DECC
      if ((stdout = freopen ("SYS$OUTPUT", "w", stdout, "ctx=bin")) == NULL)
         exit (vaxc$errno);
#else
      if ((stdout = freopen ("SYS$OUTPUT", "w", stdout, "rfm=udf")) == NULL)
         exit (vaxc$errno);
#endif
   }

   SetPageScheme ();

   /*********************/
   /* get CGI variables */
   /*********************/

   GetCgiVar (CgiPathInfoPtr, "WWW_PATH_INFO");
   GetCgiVar (CgiPathTranslatedPtr, "WWW_PATH_TRANSLATED");
   GetCgiVar (CgiQueryStringPtr, "WWW_QUERY_STRING");
   GetCgiVar (CgiScriptNamePtr, "WWW_SCRIPT_NAME");
   GetCgiVar (CgiFormListPtr, "WWW_FORM_LIST");
   GetCgiVar (CgiFormGifPtr, "WWW_FORM_GIF");
   GetCgiVar (CgiFormJpgPtr, "WWW_FORM_JPG");
   GetCgiVar (CgiFormJpgInlinePtr, "WWW_FORM_JPG_INLINE");
   GetCgiVar (CgiFormXbmPtr, "WWW_FORM_XBM");

   /***********************/
   /* execute the request */
   /***********************/

   IncludeGif = IncludeJpg = IncludeXbm = false;
   if (CgiFormGifPtr[0]) IncludeGif = true;
   if (CgiFormJpgPtr[0]) IncludeJpg = true;
   if (CgiFormJpgInlinePtr[0]) JpgInline = true;
   if (CgiFormXbmPtr[0]) IncludeXbm = true;
   if (!(IncludeGif || IncludeJpg || IncludeXbm))
      IncludeGif = IncludeXbm = true;

   if (CgiFormListPtr[0])
      ProcessFileSpec ();
   else
      ProvideForm ();

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Get "command-line" parameters, whether from the command-line or from a
configuration logical containing the equivalent.
*/

GetParameters ()

{
   static char  CommandLine [256];
   static unsigned long  Flags = 0;

   register char  *aptr, *cptr, *clptr, *sptr;

   int  status;
   unsigned short  Length;
   char  ch;
   $DESCRIPTOR (CommandLineDsc, CommandLine);

   /*********/
   /* begin */
   /*********/

   if ((clptr = getenv ("GLIST$PARAM")) == NULL)
   {
      /* get the entire command line following the verb */
      if (VMSnok (status =
          lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
         exit (status);
      (clptr = CommandLine)[Length] = '\0';
   }

   aptr = NULL;
   ch = *clptr;
   for (;;)
   {
      if (aptr != NULL) *aptr = '\0';
      if (!ch) break;

      *clptr = ch;
      if (Debug) fprintf (stdout, "clptr |%s|\n", clptr);
      while (*clptr && isspace(*clptr)) *clptr++ = '\0';
      aptr = clptr;
      if (*clptr == '/') clptr++;
      while (*clptr && !isspace (*clptr) && *clptr != '/')
      {
         if (*clptr != '\"')
         {
            clptr++;
            continue;
         }
         cptr = clptr;
         clptr++;
         while (*clptr)
         {
            if (*clptr == '\"')
               if (*(clptr+1) == '\"')
                  clptr++;
               else
                  break;
            *cptr++ = *clptr++;
         }
         *cptr = '\0';
         if (*clptr) clptr++;
      }
      ch = *clptr;
      if (*clptr) *clptr = '\0';
      if (Debug) fprintf (stdout, "aptr |%s|\n", aptr);
      if (!*aptr) continue;

      if (strsame (aptr, "/BUTTONS=", 4))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         ButtonPtr = cptr;
         continue;
      }
      if (strsame (aptr, "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (GetPageParameter (aptr)) continue;

      if (*aptr != '/')
      {
         fprintf (stdout, "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n",
                  Utility, aptr);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      else
      {
         fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
                  Utility, aptr+1);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
   }
}

/*****************************************************************************/
/*
Get command-line parameters associated with page scheme.
*/

boolean GetPageParameter (char *aptr)

{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (strsame (aptr, "/PBACKGROUND=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BACKGROUND] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBGCOLOR=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBBGCOLOR=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BUTTONBGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBBORDER=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BUTTONBORDER] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHBGCOLOR=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADBGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHBORDER=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADBORDER] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHTEXT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADTEXT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PLAYOUT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_LAYOUT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PLINK=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_LINK] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHLOCAL=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADLOCAL] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PTEXT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_TEXT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PVLINK=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_VLINK] = cptr;
      return (true);
   }
   return (false);
}

/*****************************************************************************/
/*
Set the page layout and colouration.
*/

SetPageScheme ()

{
   int  size;
   char  *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SetPageScheme()\n");

   if (PageScheme[PS_LAYOUT] == NULL)
      PageScheme[PS_LAYOUT] = "1";

   if (PageScheme[PS_BACKGROUND] == NULL)
      PageScheme[PS_BACKGROUND] = "";

   if (PageScheme[PS_HEADLOCAL] == NULL)
      PageScheme[PS_HEADLOCAL] = "";

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      if (PageScheme[PS_BGCOLOR] == NULL) PageScheme[PS_BGCOLOR] = "";
      if (PageScheme[PS_TEXT] == NULL) PageScheme[PS_TEXT] = "";
      if (PageScheme[PS_LINK] == NULL) PageScheme[PS_LINK] = "";
      if (PageScheme[PS_VLINK] == NULL) PageScheme[PS_VLINK] = "";
      if (PageScheme[PS_HEADBGCOLOR] == NULL) PageScheme[PS_HEADBGCOLOR] = "";
      if (PageScheme[PS_HEADBORDER] == NULL) PageScheme[PS_HEADBORDER] = "";
      if (PageScheme[PS_HEADTEXT] == NULL) PageScheme[PS_HEADTEXT] = "";
      if (PageScheme[PS_BUTTONBGCOLOR] == NULL) PageScheme[PS_BUTTONBGCOLOR] = "";
      if (PageScheme[PS_BUTTONBORDER] == NULL) PageScheme[PS_BUTTONBORDER] = "";
   }
   else
   {
      if (PageScheme[PS_BGCOLOR] == NULL)
         PageScheme[PS_BGCOLOR] = DEFAULT_PS_BGCOLOR;
      if (PageScheme[PS_TEXT] == NULL)
         PageScheme[PS_TEXT] = DEFAULT_PS_TEXT;
      if (PageScheme[PS_LINK] == NULL)
         PageScheme[PS_LINK] = DEFAULT_PS_LINK;
      if (PageScheme[PS_VLINK] == NULL)
         PageScheme[PS_VLINK] = DEFAULT_PS_VLINK;
      if (PageScheme[PS_HEADBGCOLOR] == NULL)
         PageScheme[PS_HEADBGCOLOR] = DEFAULT_PS_HEADBGCOLOR;
      if (PageScheme[PS_HEADBORDER] == NULL)
         PageScheme[PS_HEADBORDER] = DEFAULT_PS_HEADBORDER;
      if (PageScheme[PS_HEADTEXT] == NULL)
         PageScheme[PS_HEADTEXT] = DEFAULT_PS_HEADTEXT;
      if (PageScheme[PS_BUTTONBGCOLOR] == NULL)
         PageScheme[PS_BUTTONBGCOLOR] = DEFAULT_PS_BUTTONBGCOLOR;
      if (PageScheme[PS_BUTTONBORDER] == NULL)
         PageScheme[PS_BUTTONBORDER] = DEFAULT_PS_BUTTONBORDER;
   }

   /* <BODY> tag attributes */
   size = strlen(PageScheme[PS_BACKGROUND]) +
          strlen(PageScheme[PS_BGCOLOR]) +
          strlen(PageScheme[PS_TEXT]) +
          strlen(PageScheme[PS_LINK]) +
          strlen(PageScheme[PS_VLINK]);
   if (size)
   {
      if ((sptr = calloc (1, size+64)) == NULL) exit (vaxc$errno);
      PageScheme[PS_BODYTAG] = sptr;
      if (PageScheme[PS_BACKGROUND][0])
         sptr += sprintf (sptr, " BACKGROUND=\"%s\"", PageScheme[PS_BACKGROUND]);
      if (PageScheme[PS_BGCOLOR][0])
         sptr += sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_BGCOLOR]);
      if (PageScheme[PS_TEXT][0])
         sptr += sprintf (sptr, " TEXT=\"%s\"", PageScheme[PS_TEXT]);
      if (PageScheme[PS_LINK][0])
         sptr += sprintf (sptr, " LINK=\"%s\"", PageScheme[PS_LINK]);
      if (PageScheme[PS_VLINK][0])
         sptr += sprintf (sptr, " VLINK=\"%s\"", PageScheme[PS_VLINK]);
   }
   else
      PageScheme[PS_BODYTAG] = "";

   if (PageScheme[PS_HEADBGCOLOR][0])
   {
      if ((sptr = calloc (1, strlen(PageScheme[PS_HEADBGCOLOR])+16)) == NULL)
         exit (vaxc$errno);
      sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_HEADBGCOLOR]);
      PageScheme[PS_HEADBGCOLOR] = sptr;
      PageScheme[PS_HEADPADDING] = "10";
   }
   else
      PageScheme[PS_HEADPADDING] = "0";

   if (PageScheme[PS_BUTTONBGCOLOR][0])
   {
      if ((sptr = calloc (1, strlen(PageScheme[PS_BUTTONBGCOLOR])+16)) == NULL)
         exit (vaxc$errno);
      sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_BUTTONBGCOLOR]);
      PageScheme[PS_BUTTONBGCOLOR] = sptr;
   }
}

/*****************************************************************************/
/*
Provides a divider for top and bottom of the content of the page. This can be
a coloured bar (using <TABLE>) or a horizontal rule depending on the page
layout. "Buttons" providing script-internal and/or additional user-specified
links ('ButtonPtr' string) can be placed with(in) this bar. All button labels
are derived from 'ButtonPtr', with script-internal buttons using link-paths
set up via 'ButtonInternal[]' array, or any user-specified path depending on
requirement. An empty path (i.e. PathPtr[0] == '\0') obviously does not have a
link created, it just displays the button label. For a button-bar at the top
of the document use 1, bottom of the document use 2, and for just a bar with
no buttons at all use 0.
*/

ButtonBar (int Top1Bottom2)

{
#define MAX_BUTTON_COUNT 8

   static int  ButtonCount = -1;
   static char  *ButtonInternal [MAX_BUTTON_COUNT],
                *ButtonLabel [MAX_BUTTON_COUNT],
                *ButtonPath [MAX_BUTTON_COUNT];

   int  idx;
   char  *PathPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ButtonBar() %d\n", Top1Bottom2);

   if (ButtonCount == -1)
   {
      register char  *cptr, *sptr;

      if (Debug) fprintf (stdout, "|%s|\n", ButtonPtr);
      cptr = ButtonPtr;
      for (ButtonCount = 0;
           ButtonCount < MAX_BUTTON_COUNT && *cptr;
           ButtonCount++)
      {
         for (sptr = cptr; *sptr && *sptr != '=' && *sptr != ';'; sptr++)
            if (*sptr == '\\') memcpy (sptr, sptr+1, strlen(sptr));
         if (*sptr == '=') *sptr++ = '\0';
         ButtonLabel[ButtonCount] = cptr;
         cptr = sptr;
         for (sptr = cptr; *sptr && *sptr != ';'; sptr++)
            if (*sptr == '\\') memcpy (sptr, sptr+1, strlen(sptr));
         if (*sptr) *sptr++ = '\0';
         ButtonPath[ButtonCount] = cptr;
         cptr = sptr;
      }
   }

   if (Top1Bottom2)
   {
      /***********************************/
      /* set up script-specified buttons */
      /***********************************/

      SCRIPT_SPECIFIC_BUTTON_CODE
   }

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      /************/
      /* format 2 */
      /************/

      if (Top1Bottom2 == 2 || !Top1Bottom2)
      {
         fprintf (stdout, "<HR ALIGN=left SIZE=2 WIDTH=95%%>\n");
         if (!Top1Bottom2) return;
      }

      fprintf (stdout, "<FONT SIZE=-1><NOBR>\n");
      for (idx = 0; idx < ButtonCount; idx++)
      {
         if (ButtonInternal[idx] == NULL)
            PathPtr = ButtonPath[idx];
         else
            PathPtr = ButtonInternal[idx];
         if (idx) fprintf (stdout, "&nbsp;");
         if (PathPtr[0])
            fprintf (stdout, "[<A HREF=\"%s\">%s</A>]\n",
                     PathPtr, ButtonLabel[idx]);
         else
            fprintf (stdout, "[%s]\n", ButtonLabel[idx]);
      }
      fprintf (stdout, "</NOBR></FONT>\n");

      if (Top1Bottom2 == 1)
         fprintf (stdout, "<HR ALIGN=left SIZE=2 WIDTH=95%%>\n");
   }
   else
   {
      /************/
      /* format 1 */
      /************/

      fprintf (stdout,
"<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=0>\
<TR><TD HEIGHT=2></TD></TR>\
</TABLE>\n\
<TABLE BORDER=%s CELLPADDING=%s CELLSPACING=0 WIDTH=100%%>\n\
<TR><TD%s>\n",
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADBGCOLOR]);

      if (ButtonCount == 0 || !Top1Bottom2)
         fprintf (stdout, "&nbsp;\n");
      else
      {
         fprintf (stdout,
"<TABLE BORDER=%s CELLPADDING=1 CELLSPACING=0>\n",
            PageScheme[PS_BUTTONBORDER]);

         for (idx = 0; idx < ButtonCount; idx++)
         {
            if (ButtonInternal[idx] == NULL)
               PathPtr = ButtonPath[idx];
            else
               PathPtr = ButtonInternal[idx];
            if (PathPtr[0])
               fprintf (stdout,
"<TD ALIGN=center%s><FONT SIZE=-1>\
<NOBR>&nbsp;&nbsp;<A HREF=\"%s\">%s</A>&nbsp;&nbsp;</NOBR></FONT></TD>\n",
                  PageScheme[PS_BUTTONBGCOLOR], PathPtr, ButtonLabel[idx]);
            else
               fprintf (stdout,
"<TD ALIGN=center%s><FONT SIZE=-1>\
<NOBR>&nbsp;&nbsp;%s&nbsp;&nbsp;</NOBR></FONT></TD>\n",
                  PageScheme[PS_BUTTONBGCOLOR], ButtonLabel[idx]);
         }

         fprintf (stdout, "</TR></TABLE>\n");
      }

      fprintf (stdout,
"</TD></TR>\n\
</TABLE>\n\
<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=0>\
<TR><TD HEIGHT=2></TD></TR>\
</TABLE>\n");
   }
}

/****************************************************************************/
/*
*/ 

ProvideForm ()

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProvideForm()\n");

   fprintf (stdout,
"%s\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<TITLE>Glist ... Graphics Browser</TITLE>\n\
</HEAD>\n\
<BODY%s>\n",
      Http200Header,
      SoftwareID,
      PageScheme[PS_BODYTAG]);

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      fprintf (stdout,
"%s<FONT SIZE=+2><B>\n\
Graphics Browser\n\
</B></FONT>\n\
<BR>\n\
<FONT SIZE=-1>\n\
&nbsp;<SUP>*</SUP><U>Glist</U>\n\
</FONT>\n\
<HR ALIGN=left SIZE=2 WIDTH=95%%>\n\
<P>\n",
         PageScheme[PS_HEADLOCAL]);
   }
   else
   {
      fprintf (stdout,
"<TABLE BORDER=%s CELLPADDING=%s CELLSPACING=0 WIDTH=100%%%s>\n\
<TR><TD>\n\
<FONT COLOR=\"%s\" SIZE=+2><B>\n\
Graphics Browser\n\
</B></FONT>\n\
<BR>\n\
<FONT COLOR=\"%s\" SIZE=-1>\n\
&nbsp;<SUP>*</SUP><U>Glist</U>\n\
</FONT>\n\
</TD>%s</TR>\n\
</TABLE>\n",
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADPADDING],
         PageScheme[PS_HEADBGCOLOR],
         PageScheme[PS_HEADTEXT],
         PageScheme[PS_HEADTEXT],
         PageScheme[PS_HEADLOCAL]);
   }

   fprintf (stdout,
"<BLOCKQUOTE>\n\
<FORM ACTION=\"%s%s\">\n\
Listing of graphics in <TT>%s</TT>\n\
<P><INPUT TYPE=checkbox NAME=gif VALUE=yes CHECKED> GIF\n\
<BR><INPUT TYPE=checkbox NAME=jpg VALUE=yes CHECKED> JPEG &nbsp;&nbsp;\
<INPUT TYPE=checkbox NAME=jpg_inline VALUE=yes CHECKED> \
<FONT SIZE=-1>browser supports in-line JPEG</FONT>\n\
<BR><INPUT TYPE=checkbox NAME=xbm VALUE=yes CHECKED> X-bitmap\n\
<INPUT TYPE=hidden NAME=list VALUE=now>\n\
<P><INPUT TYPE=submit VALUE=\" list \"> <INPUT TYPE=reset VALUE=\" reset \">\n\
</FORM>\n\
</BLOCKQUOTE>\n\
<P>\n",
      CgiScriptNamePtr, CgiPathInfoPtr, CgiPathInfoPtr);

   ButtonBar (2);

   fprintf (stdout, "</BODY>\n</HTML>\n");
}

/****************************************************************************/
/*
*/ 

ProcessFileSpec ()

{
   register char  *cptr, *sptr, *zptr;

   boolean  FileOk,
            JpgGraphic;
   int  status,
        Bytes,
        CgiPathTranslatedLength,
        FileCount = 0,
        SubdirectoryCount = 0,
        TotalGif = 0,
        TotalJpg = 0,
        TotalXbm = 0;
   char  *GraphicTypePtr,
         *QuestionMarkPtr;
   char  DirectoryPath [256],
         ExpandedFileSpec [256],
         FileName [256],
         Name [256],
         NameAndType [256];
   struct FAB  FileFab;
   struct NAM  FileNam;
   struct RAB  FileRab;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessFileSpec()\n");

   if (CgiQueryStringPtr[0])
      QuestionMarkPtr = "?";
   else
      QuestionMarkPtr = "";

   /* length is immediately after the closing directory bracket */
   for (cptr = CgiPathTranslatedPtr; *cptr; cptr++);
   while (cptr > CgiPathTranslatedPtr && *cptr != ']') cptr--;
   if (*cptr == ']') cptr++;
   CgiPathTranslatedLength = cptr - CgiPathTranslatedPtr;

   /* find the directory (i.e. the "/dir1/dir2/" from "/dir1/dir2/*.*") */
   zptr = (sptr = DirectoryPath) + sizeof(DirectoryPath)-1;
   for (cptr = CgiPathInfoPtr; *cptr && sptr < zptr; *sptr++ = *cptr++);
   if (sptr >= zptr) exit (ErrorGeneral (ErrorInternal, __FILE__, __LINE__));
   *sptr = '\0';
   while (sptr > DirectoryPath && *sptr != '/') sptr--;
   if (sptr > DirectoryPath && *sptr == '/') *++sptr = '\0';

   /***************************/
   /* list any subdirectories */
   /***************************/

   /* initialize the file access block */
   FileFab = cc$rms_fab;
   FileFab.fab$l_dna = "*.*";
   FileFab.fab$b_dns = 3;
   FileFab.fab$l_fna = CgiPathTranslatedPtr;
   FileFab.fab$b_fns = CgiPathTranslatedLength;
   FileFab.fab$l_fop = FAB$V_NAM;
   FileFab.fab$l_nam = &FileNam;

   /* initialize the file name block */
   FileNam = cc$rms_nam;
   FileNam.nam$l_esa = ExpandedFileSpec;
   FileNam.nam$b_ess = sizeof(ExpandedFileSpec)-1;
   FileNam.nam$l_rsa = FileName;
   FileNam.nam$b_rss = sizeof(FileName)-1;

   if (VMSnok (status = sys$parse (&FileFab, 0, 0)))
      exit (ErrorVmsStatus (status, CgiPathInfoPtr, CgiPathTranslatedPtr,
            __FILE__, __LINE__));

   /* output the HTTP header and page title, etc. */
   FileNam.nam$l_name[-1] = '\0';
   fprintf (stdout,
"%s\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"VMS\" CONTENT=\"%s]\">\n\
<TITLE>Glist ... Graphics Browser</TITLE>\n\
</HEAD>\n\
<BODY%s>\n",
      Http200Header,
      SoftwareID,
      (char*)FileNam.nam$l_dev,
      PageScheme[PS_BODYTAG]);

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      fprintf (stdout,
"%s<FONT SIZE=+2><B>\n\
Graphics Browser\n\
</B></FONT>\n\
<BR>\n\
<FONT COLOR=\"%s\" SIZE=-1>\n\
&nbsp;<SUP>*</SUP><U>Glist</U>\n\
</FONT>\n\
<HR ALIGN=left SIZE=2 WIDTH=95%%>\n\
<P>\n",
         PageScheme[PS_HEADLOCAL]);
   }
   else
   {
      fprintf (stdout,
"<TABLE BORDER=%s CELLPADDING=%s CELLSPACING=0 WIDTH=100%%%s>\n\
<TR><TD>\n\
<FONT COLOR=\"%s\" SIZE=+2><B>\n\
Graphics Browser\n\
</B></FONT>\n\
<BR>\n\
<FONT COLOR=\"%s\" SIZE=-1>\n\
&nbsp;<SUP>*</SUP><U>Glist</U>\n\
</FONT>\n\
</TD>%s</TR>\n\
</TABLE>\n",
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADPADDING],
         PageScheme[PS_HEADBGCOLOR],
         PageScheme[PS_HEADTEXT],
         PageScheme[PS_HEADTEXT],
         PageScheme[PS_HEADLOCAL]);
   }

   fprintf (stdout, "<BLOCKQUOTE>\n");

   HttpHasBeenOutput = true;
   FileNam.nam$l_name[-1] = ']';

   while (VMSok (status = sys$search (&FileFab, 0, 0)))
   {
      if (Debug)
      {
         FileNam.nam$l_ver[0] = '\0';
         fprintf (stdout, "FileName |%s|\n", FileName);
         FileNam.nam$l_ver[0] = ';';
      }

      /* numeric equivalent of "DIR;" */
      if (*(unsigned long*)(FileNam.nam$l_type+1) != 0x3b524944)
         continue;

      if (!SubdirectoryCount++)
         fprintf (stdout,
"<TABLE BORDER=0 CELLPADDING=2 CELLSPACING=2>\n\
<TR><TH ALIGN=left>Subdirectories</TD></TR>\n<TR><TD><UL>\n");

      *(char*)FileNam.nam$l_type = '\0';
      zptr = (sptr = Name) + sizeof(Name)-1;
      for (cptr = FileNam.nam$l_name;
           *cptr && sptr < zptr;
           *sptr++ = tolower(*cptr++));
      *(char*)FileNam.nam$l_type = '.';
      if (sptr >= zptr) exit (ErrorGeneral (ErrorInternal, __FILE__, __LINE__));
      *sptr = '\0';

      fprintf (stdout, "<LI><A HREF=\"%s%s%s/*.*%s%s\">%s%s</A>\n",
               CgiScriptNamePtr, DirectoryPath, Name,
               QuestionMarkPtr, CgiQueryStringPtr,
               DirectoryPath, Name);
   }

   if (SubdirectoryCount) fprintf (stdout, "</UL></TD></TR>\n</TABLE>\n");

   if (status != RMS$_FNF && status != RMS$_NMF)
   {
      *(char*)FileNam.nam$l_ver = '\0';
      exit (ErrorVmsStatus (status, FileName, FileName, __FILE__, __LINE__));
   }

   /*********************/
   /* list any graphics */
   /*********************/

   /* initialize the file access block */
   FileFab = cc$rms_fab;
   FileFab.fab$l_dna = "*.*";
   FileFab.fab$b_dns = 3;
   FileFab.fab$l_fna = CgiPathTranslatedPtr;
   FileFab.fab$b_fns = CgiPathTranslatedLength;
   FileFab.fab$l_fop = FAB$V_NAM;
   FileFab.fab$l_nam = &FileNam;

   /* initialize the file name block */
   FileNam = cc$rms_nam;
   FileNam.nam$l_esa = ExpandedFileSpec;
   FileNam.nam$b_ess = sizeof(ExpandedFileSpec)-1;
   FileNam.nam$l_rsa = FileName;
   FileNam.nam$b_rss = sizeof(FileName)-1;

   if (VMSnok (status = sys$parse (&FileFab, 0, 0)))
      exit (ErrorVmsStatus (status, CgiPathInfoPtr, CgiPathTranslatedPtr,
            __FILE__, __LINE__));

   while (VMSok (status = sys$search (&FileFab, 0, 0)))
   {
      if (Debug)
      {
         FileNam.nam$l_ver[0] = '\0';
         FileNam.nam$l_ver[0] = '\0';
         fprintf (stdout, "FileName |%s|\n", FileName);
         FileNam.nam$l_ver[0] = ';';
      }

      /* numeric equivalent of "DIR;" */
      if (*(unsigned long*)(FileNam.nam$l_type+1) == 0x3b524944)
         continue;

      FileOk = JpgGraphic = false;
      /* numeric equivalent of "GIF;" */
      if (IncludeGif && *(unsigned long*)(FileNam.nam$l_type+1) == 0x3b464947)
         { FileOk = true; TotalGif++; GraphicTypePtr = "GIF"; }
      if (IncludeJpg)
      {
         /* numeric equivalent of "JPG;" */
         if (*(unsigned long*)(FileNam.nam$l_type+1) == 0x3b47504a)
            { FileOk = JpgGraphic = true; TotalJpg++; GraphicTypePtr = "JPEG"; }
         /* numeric equivalent of "JPEG" */
         if (*(unsigned long*)(FileNam.nam$l_type+1) == 0x4745504a)
            { FileOk = JpgGraphic = true; TotalJpg++; GraphicTypePtr = "JPEG"; }
      }
      /* numeric equivalent of "XBM;" */
      if (IncludeXbm && *(unsigned long*)(FileNam.nam$l_type+1) == 0x3b4d4258)
         { FileOk = true; TotalXbm++; GraphicTypePtr = "X-bitmap"; }

      if (!FileOk) continue;

      if (VMSnok (status = FileSize (&FileNam, &Bytes)))
      {
         *(char*)FileNam.nam$l_ver = '\0';
         exit (ErrorVmsStatus (status, FileName, FileName,
                               __FILE__, __LINE__));
      }

      if (!(FileCount % 10))
      {
         if (FileCount) fprintf (stdout, "</TABLE>\n<P>\n");
         fprintf (stdout,
"<TABLE BORDER=0 CELLPADDING=2 CELLSPACING=2>\n\
<TR><TH ALIGN=left><U>Name</U></TH>\
<TH ALIGN=right><U>Size</U></TH>\
<TH></TH><TH ALIGN=left><U>Path</U></TH>\
<TH ALIGN=left><U>Image</U></TH></TR>\n");
      }
      FileCount++;

      *(char*)FileNam.nam$l_ver = '\0';
      zptr = (sptr = NameAndType) + sizeof(NameAndType)-1;
      for (cptr = FileNam.nam$l_name;
           *cptr && sptr < zptr;
           *sptr++ = tolower(*cptr++));
      *(char*)FileNam.nam$l_ver = ';';
      if (sptr >= zptr) exit (ErrorGeneral (ErrorInternal, __FILE__, __LINE__));
      *sptr = '\0';

      *(char*)FileNam.nam$l_type = '\0';
      fprintf (stdout,
"<TR><TD VALIGN=top>%s</TD><TD ALIGN=right VALIGN=top>%s</TD>\
<TD>&nbsp;</TD><TD VALIGN=top><TT>%s%s</TT></TD>",
               FileNam.nam$l_name, CommaNumber(Bytes),
               DirectoryPath, NameAndType);
      *(char*)FileNam.nam$l_type = '.';

      if (JpgGraphic && !JpgInline)
         fprintf (stdout,
"<TD>&nbsp;<A HREF=\"%s%s\">[JPG]</A></TD></TR>\n",
                  DirectoryPath, NameAndType);
      else
         fprintf (stdout,
"<TD>&nbsp;<A HREF=\"%s%s\"><IMG SRC=\"%s%s\" ALIGN=TOP></A></TD>\n",
                  DirectoryPath, NameAndType,
                  DirectoryPath, NameAndType);
   }

   if (FileCount) fprintf (stdout, "</TABLE>\n");

   if (status != RMS$_FNF && status != RMS$_NMF)
   {
      *(char*)FileNam.nam$l_ver = '\0';
      exit (ErrorVmsStatus (status, FileName, FileName, __FILE__, __LINE__));
   }

   if (FileCount)
   {
      if (PageScheme[PS_LAYOUT][0] == '2')
      {
         fprintf (stdout,
"<P>\n\
<FONT SIZE=-1><U>\
<B>GIF:</B> %d&nbsp;&nbsp;\
<B>JPEG:</B> %d&nbsp;&nbsp;\
<B>X-bitmap:</B> %d\
</U></FONT>\n",
         TotalGif, TotalJpg, TotalXbm);
      }
      else
      {
         fprintf (stdout,
"<P>\n\
<TABLE BORDER=%s CELLPADDING=2 CELLSPACING=0%s>\n\
<TR><TD>\n\
&nbsp;<FONT SIZE=-1 COLOR=\"%s\">\n\
<B>GIF:</B> %d&nbsp;&nbsp;\
<B>JPEG:</B> %d&nbsp;&nbsp;\
<B>X-bitmap:</B> %d\
&nbsp;</FONT>\n\
</TD></TR>\n\
</TABLE>\n",
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADBGCOLOR],
         PageScheme[PS_HEADTEXT],
         TotalGif, TotalJpg, TotalXbm);
      }
   }
   else
      fprintf (stdout, "No graphics found!\n");

   fprintf (stdout, "</BLOCKQUOTE>\n");

   ButtonBar (2);

   fprintf (stdout, "</BODY>\n</HTML>\n");
}

/*****************************************************************************/
/*
Convert the integer into an ASCII number string containing commas. Returns a
pointer to the string.
*/

char* CommaNumber (unsigned long Value)

{
   static char  Scratch [32],
                String [32];
   static $DESCRIPTOR (ScratchDsc, Scratch);
   static $DESCRIPTOR (ValueFaoDsc, "!UL");

   register int  cnt;
   register char  *cptr, *sptr;

   int  status;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CommaNumber()\n");

   if (VMSnok (status = sys$fao (&ValueFaoDsc, &Length, &ScratchDsc, Value)))
      return ("*ERROR*");
   Scratch[Length] = '\0';
   if (((Length-1) / 3) < 1)
   {
      cptr = Scratch;
      sptr = String;
      while (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
      return (String);
   }
   else
   if (!(cnt = Length % 3))
      cnt = 3;
   
   cptr = Scratch;
   sptr = String;
   while (*cptr)
   {
      if (!cnt--)
      {
         *sptr++ = ',';
         cnt = 2;
      }
      *sptr++ = *cptr++;
   }
   *sptr = '\0';
   return (String);
}

/*****************************************************************************/
/*
This function uses the ACP-QIO interface detailed in the "OpenVMS I/O User's 
Reference Manual".
*/ 

int FileSize
(
struct NAM *FileNamPtr,
int *BytesPtr
)
{
   static $DESCRIPTOR (DeviceDsc, "");

   static struct {
      unsigned long  OfNoInterest1;
      unsigned short  AllocatedVbnHi;
      unsigned short  AllocatedVbnLo;
      unsigned short  EndOfFileVbnHi;
      unsigned short  EndOfFileVbnLo;
      unsigned short  FirstFreeByte;
      unsigned short  OfNoInterest2;
      unsigned long  OfNoInterestLots [4];
   } AtrRecord;

   static struct fibdef  FileFib;

   static struct atrdef  FileAtr [] =
   {
      { sizeof(AtrRecord), ATR$C_RECATTR, &AtrRecord },
      { 0, 0, 0 }
   };

   static struct {
      unsigned short  Length;
      unsigned short  Unused;
      unsigned long  Address;
   } FileNameAcpDsc,
     FileFibAcpDsc,
     FileAtrAcpDsc;

   static struct {
      unsigned short  Status;
      unsigned short  Unused1;
      unsigned long  Unused2;
   } AcpIOsb;

   int  status;
   unsigned short  AcpChannel;
   unsigned long  AllocatedVbn,
                  EndOfFileVbn;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FileSize()\n");

   /* assign a channel to the disk device containing the file */
   DeviceDsc.dsc$w_length = FileNamPtr->nam$b_dev;
   DeviceDsc.dsc$a_pointer = FileNamPtr->nam$l_dev;
   status = sys$assign (&DeviceDsc, &AcpChannel, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$assign() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   /* set up the File Information Block for the ACP interface */
   memset (&FileFib, 0, sizeof(struct fibdef));
   FileFibAcpDsc.Length = sizeof(FileFib);
   FileFibAcpDsc.Address = &FileFib;
   /* quick work around, different syntax for this structure with DEC C? */
#ifdef __DECC
   FileFib.fib$w_did[0] = FileNamPtr->nam$w_did[0];
   FileFib.fib$w_did[1] = FileNamPtr->nam$w_did[1];
   FileFib.fib$w_did[2] = FileNamPtr->nam$w_did[2];
#else
   FileFib.fib$r_did_overlay.fib$w_did[0] = FileNamPtr->nam$w_did[0];
   FileFib.fib$r_did_overlay.fib$w_did[1] = FileNamPtr->nam$w_did[1];
   FileFib.fib$r_did_overlay.fib$w_did[2] = FileNamPtr->nam$w_did[2];
#endif

   FileNameAcpDsc.Address = FileNamPtr->nam$l_name;
   FileNameAcpDsc.Length = FileNamPtr->nam$b_name + FileNamPtr->nam$b_type +
                           FileNamPtr->nam$b_ver;

   status = sys$qiow (0, AcpChannel, IO$_ACCESS, &AcpIOsb, 0, 0, 
                      &FileFibAcpDsc, &FileNameAcpDsc, 0, 0,
                      &FileAtr, 0);

   /* immediately deassign the channel in case we return on an error */
   sys$dassgn (AcpChannel);

   if (Debug)
      fprintf (stdout, "sys$qio() %%X%08.08X IOsb: %%X%08.08X\n",
              status, AcpIOsb.Status);

   if (VMSok (status)) status = AcpIOsb.Status;
   if (VMSnok (status)) return (status);

   AllocatedVbn = AtrRecord.AllocatedVbnLo + (AtrRecord.AllocatedVbnHi << 16);
   EndOfFileVbn = AtrRecord.EndOfFileVbnLo + (AtrRecord.EndOfFileVbnHi << 16);

   if (Debug)
      fprintf (stdout, "AllocatedVbn: %d EndOfFileVbn: %d FirstFreeByte %d\n",
               AllocatedVbn, EndOfFileVbn, AtrRecord.FirstFreeByte);

   if (EndOfFileVbn <= 1)
      *BytesPtr = AtrRecord.FirstFreeByte;
   else
      *BytesPtr = ((EndOfFileVbn - 1) << 9) + AtrRecord.FirstFreeByte;
   if (Debug) fprintf (stdout, "Bytes %d\n", *BytesPtr);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, stdout);
   fprintf (stdout,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n",
   SoftwareID, cptr, SourceLineNumber, Text);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      exit (status);

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, stdout);
   fprintf (stdout,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n",
   SoftwareID, cptr, SourceLineNumber, Message, Text, StatusValue, HiddenText);

   return (SS$_NORMAL);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/

