/*****************************************************************************/
/*
                                 Query.c


A CGI-compliant script to search plain-text and HTML files.  Accepts
query-based searches (e.g. "?find+this+phrase") or form-based fields briefly
discussed below.  By default searches the URL-supplied path, or will map the
path supplied with the "path=" query form field.  A VMS directory elipsis (i.e.
"...") may be supplied in the path to result in a directory tree search.

Page layout and colouration may be specified via the appropriate command-line
qualifiers (or corresponding logical/symbol name). Defaults apply for any not
specified.  See "Qualifiers" section below, and also about the logical name or
symbol "QUERY$PARAM".

An example of changing the page colour to white and the banner to red!

  /PBGCOLOR="#ffffff" /PHBGCOLOR="#ff0000"

Don't like explicitly setting a browser's colours?  A colour may be disabled by
setting it to empty.  The following example disables all colours.

  /PBGCOLOR/PBBGCOLOR/PHBGCOLOR/PHTEXT/PLINK/PTEXT/PVLINK

The script can format a page in either of two layouts.

  1. Tables are used to create a coloured header and button bar (DEFAULT).
     Default colours are white page with grey heading and button outlines.
  2. Textual header, horizontal rules and a textual button bar.
     No default colours.

Select other than the default using the following:

  /PLAYOUT=2

Local information may be included in the header. For layout 1 this should be
integrated with the <TABLE> formatted header and to the right of the header
information. Text, an image logo, just about anything could be included. This
is a example of providing a textual form of a local logo:

  /PHLOCAL="<TD ALIGN=right VALIGN=top><FONT SIZE=-1 COLOR=#ffffff>"-
  "<B>WASD HyperText Services</B></FONT></TD>"

This is an example of providing a local graphical logo:

  /PHLOCAL="<TD ALIGN=right VALIGN=top><IMG SRC=/local/logo.gif></TD>"

Such local information with layout 2 is included immediately before the header
information at the top of the page.

Button labels are customizable (potentially to non-English language). They
comprise a label, equate symbol and URL-style path suitable for creating a
link. Multiple buttons are separated using the semicolon. Note that any such
button customization must provide escaped HTML-forbidden characters in the
button label and URI-forbidden characters in the path! The backslash
character, "\", escapes characters, including the button-delimitting "=" and
";". There are defaults, see DEFAULT_BUTTONS.

Here is an example of changing the button labels:

  /BUTTON="About=/query/-/aboutquery.html"

Additional buttons may be created by adding "label=path;" elements to the
button string. In this way an additional information page could be referenced
as follows:

  /BUTTON="About=/query/-/aboutquery.html;Other Information=/info/"

DIFFICULTY FITTING ALL THESE QUALIFIERS INTO A COMMAND LINE OR LOGICAL?
Use an existing, or create a new, DCL wrapper procedure for the script (same
name and directory) and build up a DCL symbol to provide the information. Up
to 1024 characters can be included in this way. For example:

  $ QUERY$PARAM = "/BUTTON=""About=/query/-/aboutquery.html"""
  $ QUERY$PARAM = QUERY$PARAM + "/PBGCOLOR/PLINK/PVLINK"
  $ QUERY$PARAM = QUERY$PARAM + "/PHLOCAL=""<TD VALIGN=top>...</TD>"""
  $ RUN HT_EXE:QUERY


LOGICAL NAMES
-------------
QUERY$DBUG          turns on all "if (Debug)" statements
QUERY$PARAM         equivalent to (overrides) the command line
                    parameters/qualifiers (define as a system-wide logical)


QUALIFIERS
----------
/ABOUT=         synonym for /HELP
/BUTTONS=       string containing button labels/paths
/DBUG           turns on all "if (Debug)" statements
/HELP=          URL for help on searching
/HTML=          list of comma separated HTML file types
/TEXT=          list of comma separated TEXT file types
/DBUG           turns on all "if (Debug)" statements
/PBACKGROUND=   <body> background image path
/PBGCOLOR=      <body> background colour
/PBBGCOLOR=     button background color
/PBBORDER=      width of button border
/PHBGCOLOR=     heading background color
/PHBORDER=      width of heading and button-bar border
/PHLOCAL=       local information to be included in header
/PHTEXT=        heading text colour
/PLAYOUT=       1 is coloured header & buttons, 2 is text & horizontal rules
/PLINK=         <body> link colour
/PTEXT=         <body> text colour
/PVLINK=        <body> visited link colour


HTML FORM ELEMENTS
------------------
case=                  case sensitive search (Y or N)
exact=                 exact number of records (for extract utility, Y or N)
extract=               number of line to pass to extract utility
hits=                  show all hits or document only (D or A)
path=                  form supplied path (otherwise URL path)
search=                text to search for
what=                  (forgotten)


BUILD DETAILS
-------------
See BUILD_QUERY.COM procedure.


VERSION HISTORY (update SoftwareID as well!)
---------------
24-JUL-98  MGD  v2.5.1, suppress table background colours if empty
20-MAY-98  MGD  v2.5.0, general maintenance,
                        cosmetic changes
02-APR-98  MGD  v2.4.0, report file locked against access a non-fatal error,
                        added form-based path to override path-info,
                        refined (some might say corrected ;^) HTML tag parsing
19-FEB-98  MGD  v2.3.2, bugfix; "&hits=document" nesting,
                        modified "?about=search" redirection
19-AUG-97  MGD  v2.3.1, MapUrl() to MapUrl_Map() for conditional mapping
23-MAY-97  MGD  v2.3.0, wildcard search,
                        un-escape character entities (e.g. "&lt;") before match
19-SEP-95  MGD  v2.2.1, replace <CR><LF> carriage-control with single newline,
                        still acceptable for HTTP, and slightly more efficient
24-MAY-95  MGD  v2.2.0, minor changes for AXP compatibility
21-APR-95  MGD  v2.1.1, added 'FormWhat'
27-MAR-95  MGD  v2.1.0, modifications to CGI interface
05-DEC-94  MGD  v2.0.0, major revision, URL mapping, CGI-like interface
10-JUN-94  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "QUERY AXP-2.5.1";
#else
   char SoftwareID [] = "QUERY VAX-2.5.1";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS-related header files */
#include <descrip.h>
#include <libdef.h>
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

/* macro provides NULL pointer if CGI variable does not exist */
#define GetCgiVarIfExists(CharPointer,CgiVariableName) \
   CharPointer = getenv(CgiVariableName)

/* macro provides pointer to empty string even if CGI variable does not exist */
#define GetCgiVar(CharPointer,CgiVariableName) \
   if ((CharPointer = getenv(CgiVariableName)) == NULL) \
       CharPointer = ""; \
   if (Debug) fprintf (stdout, "%s |%s|\n", CgiVariableName, CharPointer);

#define DEFAULT_EXTRACT_RECORDS 25

#define MAX_RECORD_SIZE 8192

/* the first comma delimits an empty file type (e.g. README.;) */
#define DEFAULT_TEXT_TYPES \
"ADA,ASCII,BAS,C,CNF,COB,CONF,COM,CPP,DIS,FOR,JAVA,H,LIS,LOG,MAR,\
PAS,PRO,RELEASE_NOTES,SDML,TEX,TEXT,TXT"

#define DEFAULT_HTML_TYPES "HTM,HTML,HTMLX,SHTML"

#define DEFAULT_ABOUT_PATH "/query/-/aboutquery.html"

#define DEFAULT_BUTTONS "About"

/* this macro just plugs in some script-specific code into ButtonBar() */
#define SCRIPT_SPECIFIC_BUTTON_CODE \
   ButtonInternal[0] = HelpPathPtr;

#define DEFAULT_PS_BGCOLOR        "#ffffff"
#define DEFAULT_PS_TEXT           "#000000"
#define DEFAULT_PS_LINK           "#0000ff"
#define DEFAULT_PS_VLINK          "#0000ff"
#define DEFAULT_PS_HEADBGCOLOR    "#cccccc"
#define DEFAULT_PS_HEADBORDER     "0"
#define DEFAULT_PS_HEADTEXT       "#000000"
#define DEFAULT_PS_BUTTONBGCOLOR  "#ffffff"
#define DEFAULT_PS_BUTTONBORDER   "1"

#define PS_BACKGROUND     0
#define PS_BGCOLOR        1
#define PS_TEXT           2
#define PS_LINK           3
#define PS_VLINK          4
#define PS_HEADBGCOLOR    5
#define PS_HEADTEXT       6
#define PS_HEADBORDER     7
#define PS_BUTTONBGCOLOR  8
#define PS_BUTTONBORDER   9
#define PS_BODYTAG       10
#define PS_LAYOUT        11
#define PS_HEADLOCAL     12
#define PS_HEADPADDING   13

char  *PageScheme [16];

char  ExtractScriptName [] = "extract";

char  Utility [] = "QUERY";

char  Http200Header [] =
"HTTP/1.0 200 Success\n\
Content-Type: text/html\n\
\n";

char  Http404Header [] =
"HTTP/1.0 404 Error\n\
Content-Type: text/html\n\
\n";

boolean  Debug,
         ExactNumberOfRecords,
         HttpHasBeenOutput,
         ListStarted;

int  ExtractNumberOfRecords,
     KeyCount;

char  FormPath [256],
      KeyName [16];
      SearchString [256];

char  *ButtonPtr = DEFAULT_BUTTONS,
      *CgiPathInfoPtr,
      *CgiPathTranslatedPtr,
      *CgiRequestMethodPtr,
      *CgiFormAboutPtr,
      *CgiFormCasePtr,
      *CgiFormExactPtr,
      *CgiFormExtractPtr,
      *CgiFormHitsPtr,
      *CgiFormPathPtr,
      *CgiFormSearchPtr,
      *CgiFormWhatPtr,
      *CgiKeyCountPtr,
      *CgiKeyValuePtr,
      *HelpPathPtr = DEFAULT_ABOUT_PATH,
      *HtmlFileTypesPtr = DEFAULT_HTML_TYPES,
      *TextFileTypesPtr = DEFAULT_TEXT_TYPES;
      
/* required prototypes */
char* MapUrl_Map (char*, char*, char*, char*, void*);
char* CopyTextIntoHtml (char*, char*, int);
char* CopyTextIntoUri (char*, char*, int);
char* SearchTextString (char*, char*, boolean, int*);

/*****************************************************************************/
/*
*/

main ()

{
   register char  *cptr;

   boolean  CaseSensitive = false,
            DocumentOnly = false;
   int  status,
        Count,
        CgiPathTranslatedLength;

   /*********/
   /* begin */
   /*********/

   if (getenv ("QUERY$DBUG") != NULL) Debug = true;

   GetParameters ();

   if (Debug)
      system ("show sym *");
   else
   {
      /* reopen output stream so that the '\r' and '\n' are not filtered */
#ifdef __DECC
      if ((stdout = freopen ("SYS$OUTPUT", "w", stdout, "ctx=bin")) == NULL)
         exit (vaxc$errno);
#else
      if ((stdout = freopen ("SYS$OUTPUT", "w", stdout, "rfm=udf")) == NULL)
         exit (vaxc$errno);
#endif
   }

   SetPageScheme ();

   /*************************/
   /* get the CGI variables */
   /*************************/

   GetCgiVar (CgiRequestMethodPtr, "WWW_REQUEST_METHOD");
   /* numeric equivalent of "GET\0" (just a bit more efficient, that's all!) */
   if (*(unsigned long*)CgiRequestMethodPtr != 0x00544547)
   {
      fprintf (stdout, "HTTP/1.0 501 Error\n\n");
      exit (SS$_NORMAL);
   }

   GetCgiVar (CgiPathInfoPtr, "WWW_PATH_INFO");
   GetCgiVar (CgiPathTranslatedPtr, "WWW_PATH_TRANSLATED");

   GetCgiVar (CgiFormAboutPtr, "WWW_FORM_ABOUT");
   GetCgiVar (CgiFormCasePtr, "WWW_FORM_CASE");
   GetCgiVar (CgiFormExactPtr, "WWW_FORM_EXACT");
   GetCgiVar (CgiFormExtractPtr, "WWW_FORM_EXTRACT");
   GetCgiVar (CgiFormHitsPtr, "WWW_FORM_HITS");
   GetCgiVar (CgiFormPathPtr, "WWW_FORM_PATH");
   GetCgiVar (CgiFormSearchPtr, "WWW_FORM_SEARCH");
   GetCgiVar (CgiFormWhatPtr, "WWW_FORM_WHAT");
   GetCgiVar (CgiKeyCountPtr, "WWW_KEY_COUNT");

   SearchString[0] = '\0';
   KeyCount = atoi(CgiKeyCountPtr);
   for (Count = 1; Count <= KeyCount; Count++)
   {
      sprintf (KeyName, "WWW_KEY_%d", Count);
      GetCgiVar (CgiKeyValuePtr, KeyName);
      if (SearchString[0]) strcat (SearchString, " ");
      strcat (SearchString, CgiKeyValuePtr);
   }
   if (SearchString[0]) CgiFormSearchPtr = SearchString;

   /***********/
   /* process */
   /***********/

   if (strsame (CgiFormAboutPtr, "SEARCH", 6))
   {
      fprintf (stdout, "Location: %s\n\n", HelpPathPtr);
      exit (SS$_NORMAL);
   }

   if (CgiFormPathPtr[0])
   {
      cptr = MapUrl_Map (CgiFormPathPtr, FormPath, NULL, NULL, NULL);
      if (!*cptr)
      {
         ErrorGeneral (cptr+1, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }      
      CgiPathTranslatedPtr = FormPath;
   }
   CgiPathTranslatedLength = strlen (CgiPathTranslatedPtr);

   if (!CgiPathTranslatedPtr[0])
   {
      ErrorGeneral ("Search path not supplied.", __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
   if (!CgiFormSearchPtr[0])
   {
      ErrorGeneral ("Search string not supplied.", __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
   if (CgiFormExtractPtr[0] && !isdigit(CgiFormExtractPtr[0]))
   {
      ErrorGeneral ("Invalid number of lines to extract.",
                    __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   ExtractNumberOfRecords = atoi(CgiFormExtractPtr);
   if (!ExtractNumberOfRecords)
      ExtractNumberOfRecords = DEFAULT_EXTRACT_RECORDS;

   if (toupper(CgiFormExactPtr[0]) == 'Y')
      ExactNumberOfRecords = true;
   else
      ExactNumberOfRecords = false;

   if (toupper(CgiFormCasePtr[0]) == 'Y')
      CaseSensitive = true;
   else
      CaseSensitive = false;

   if (toupper(CgiFormHitsPtr[0]) == 'D')
      DocumentOnly = true;
   else
      DocumentOnly = false;

   SearchFiles (CgiPathTranslatedPtr, CgiPathTranslatedLength,
                CgiFormSearchPtr, CaseSensitive, DocumentOnly);

   fclose (stdout);

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Get "command-line" parameters, whether from the command-line or from a
configuration logical containing the equivalent.
*/

GetParameters ()

{
   static char  CommandLine [256];
   static unsigned long  Flags = 0;

   register char  *aptr, *cptr, *clptr, *sptr;

   int  status;
   unsigned short  Length;
   char  ch;
   $DESCRIPTOR (CommandLineDsc, CommandLine);

   /*********/
   /* begin */
   /*********/

   if ((clptr = getenv ("QUERY$PARAM")) == NULL)
   {
      /* get the entire command line following the verb */
      if (VMSnok (status =
          lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
         exit (status);
      (clptr = CommandLine)[Length] = '\0';
   }

   aptr = NULL;
   ch = *clptr;
   for (;;)
   {
      if (aptr != NULL) *aptr = '\0';
      if (!ch) break;

      *clptr = ch;
      if (Debug) fprintf (stdout, "clptr |%s|\n", clptr);
      while (*clptr && isspace(*clptr)) *clptr++ = '\0';
      aptr = clptr;
      if (*clptr == '/') clptr++;
      while (*clptr && !isspace (*clptr) && *clptr != '/')
      {
         if (*clptr != '\"')
         {
            clptr++;
            continue;
         }
         cptr = clptr;
         clptr++;
         while (*clptr)
         {
            if (*clptr == '\"')
               if (*(clptr+1) == '\"')
                  clptr++;
               else
                  break;
            *cptr++ = *clptr++;
         }
         *cptr = '\0';
         if (*clptr) clptr++;
      }
      ch = *clptr;
      if (*clptr) *clptr = '\0';
      if (Debug) fprintf (stdout, "aptr |%s|\n", aptr);
      if (!*aptr) continue;

      if (strsame (aptr, "/ABOUT=", 4))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         HelpPathPtr = cptr;
         continue;
      }
      if (strsame (aptr, "/BUTTONS=", 4))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         ButtonPtr = cptr;
         continue;
      }
      if (strsame (aptr, "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (strsame (aptr, "/HTML=", 4))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         HtmlFileTypesPtr = cptr;
         continue;
      } 
      if (strsame (aptr, "/TEXT=", 4))
      {
         for (cptr = aptr; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         TextFileTypesPtr = cptr;
         continue;
      }
      if (GetPageParameter (aptr)) continue;

      if (*aptr != '/')
      {
         fprintf (stdout, "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n",
                  Utility, aptr);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      else
      {
         fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
                  Utility, aptr+1);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
   }
}

/*****************************************************************************/
/*
Get command-line parameters associated with page scheme.
*/

boolean GetPageParameter (char *aptr)

{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (strsame (aptr, "/PBACKGROUND=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BACKGROUND] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBGCOLOR=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBBGCOLOR=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BUTTONBGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PBBORDER=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_BUTTONBORDER] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHBGCOLOR=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADBGCOLOR] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHBORDER=", 5))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADBORDER] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHTEXT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADTEXT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PLAYOUT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_LAYOUT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PLINK=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_LINK] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PHLOCAL=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_HEADLOCAL] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PTEXT=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_TEXT] = cptr;
      return (true);
   }
   if (strsame (aptr, "/PVLINK=", 4))
   {
      for (cptr = aptr; *cptr && *cptr != '='; cptr++);
      if (*cptr) cptr++;
      PageScheme[PS_VLINK] = cptr;
      return (true);
   }
   return (false);
}

/*****************************************************************************/
/*
Set the page layout and colouration.
*/

SetPageScheme ()

{
   int  size;
   char  *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SetPageScheme()\n");

   if (PageScheme[PS_LAYOUT] == NULL)
      PageScheme[PS_LAYOUT] = "1";

   if (PageScheme[PS_BACKGROUND] == NULL)
      PageScheme[PS_BACKGROUND] = "";

   if (PageScheme[PS_HEADLOCAL] == NULL)
      PageScheme[PS_HEADLOCAL] = "";

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      if (PageScheme[PS_BGCOLOR] == NULL) PageScheme[PS_BGCOLOR] = "";
      if (PageScheme[PS_TEXT] == NULL) PageScheme[PS_TEXT] = "";
      if (PageScheme[PS_LINK] == NULL) PageScheme[PS_LINK] = "";
      if (PageScheme[PS_VLINK] == NULL) PageScheme[PS_VLINK] = "";
      if (PageScheme[PS_HEADBGCOLOR] == NULL) PageScheme[PS_HEADBGCOLOR] = "";
      if (PageScheme[PS_HEADBORDER] == NULL) PageScheme[PS_HEADBORDER] = "";
      if (PageScheme[PS_HEADTEXT] == NULL) PageScheme[PS_HEADTEXT] = "";
      if (PageScheme[PS_BUTTONBGCOLOR] == NULL) PageScheme[PS_BUTTONBGCOLOR] = "";
      if (PageScheme[PS_BUTTONBORDER] == NULL) PageScheme[PS_BUTTONBORDER] = "";
   }
   else
   {
      if (PageScheme[PS_BGCOLOR] == NULL)
         PageScheme[PS_BGCOLOR] = DEFAULT_PS_BGCOLOR;
      if (PageScheme[PS_TEXT] == NULL)
         PageScheme[PS_TEXT] = DEFAULT_PS_TEXT;
      if (PageScheme[PS_LINK] == NULL)
         PageScheme[PS_LINK] = DEFAULT_PS_LINK;
      if (PageScheme[PS_VLINK] == NULL)
         PageScheme[PS_VLINK] = DEFAULT_PS_VLINK;
      if (PageScheme[PS_HEADBGCOLOR] == NULL)
         PageScheme[PS_HEADBGCOLOR] = DEFAULT_PS_HEADBGCOLOR;
      if (PageScheme[PS_HEADBORDER] == NULL)
         PageScheme[PS_HEADBORDER] = DEFAULT_PS_HEADBORDER;
      if (PageScheme[PS_HEADTEXT] == NULL)
         PageScheme[PS_HEADTEXT] = DEFAULT_PS_HEADTEXT;
      if (PageScheme[PS_BUTTONBGCOLOR] == NULL)
         PageScheme[PS_BUTTONBGCOLOR] = DEFAULT_PS_BUTTONBGCOLOR;
      if (PageScheme[PS_BUTTONBORDER] == NULL)
         PageScheme[PS_BUTTONBORDER] = DEFAULT_PS_BUTTONBORDER;
   }

   /* <BODY> tag attributes */
   size = strlen(PageScheme[PS_BACKGROUND]) +
          strlen(PageScheme[PS_BGCOLOR]) +
          strlen(PageScheme[PS_TEXT]) +
          strlen(PageScheme[PS_LINK]) +
          strlen(PageScheme[PS_VLINK]);
   if (size)
   {
      if ((sptr = calloc (1, size+64)) == NULL) exit (vaxc$errno);
      PageScheme[PS_BODYTAG] = sptr;
      if (PageScheme[PS_BACKGROUND][0])
         sptr += sprintf (sptr, " BACKGROUND=\"%s\"", PageScheme[PS_BACKGROUND]);
      if (PageScheme[PS_BGCOLOR][0])
         sptr += sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_BGCOLOR]);
      if (PageScheme[PS_TEXT][0])
         sptr += sprintf (sptr, " TEXT=\"%s\"", PageScheme[PS_TEXT]);
      if (PageScheme[PS_LINK][0])
         sptr += sprintf (sptr, " LINK=\"%s\"", PageScheme[PS_LINK]);
      if (PageScheme[PS_VLINK][0])
         sptr += sprintf (sptr, " VLINK=\"%s\"", PageScheme[PS_VLINK]);
   }
   else
      PageScheme[PS_BODYTAG] = "";

   if (PageScheme[PS_HEADBGCOLOR][0])
   {
      if ((sptr = calloc (1, strlen(PageScheme[PS_HEADBGCOLOR])+16)) == NULL)
         exit (vaxc$errno);
      sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_HEADBGCOLOR]);
      PageScheme[PS_HEADBGCOLOR] = sptr;
      PageScheme[PS_HEADPADDING] = "10";
   }
   else
      PageScheme[PS_HEADPADDING] = "0";

   if (PageScheme[PS_BUTTONBGCOLOR][0])
   {
      if ((sptr = calloc (1, strlen(PageScheme[PS_BUTTONBGCOLOR])+16)) == NULL)
         exit (vaxc$errno);
      sprintf (sptr, " BGCOLOR=\"%s\"", PageScheme[PS_BUTTONBGCOLOR]);
      PageScheme[PS_BUTTONBGCOLOR] = sptr;
   }
}

/*****************************************************************************/
/*
Provides a divider for top and bottom of the content of the page. This can be
a coloured bar (using <TABLE>) or a horizontal rule depending on the page
layout. "Buttons" providing script-internal and/or additional user-specified
links ('ButtonPtr' string) can be placed with(in) this bar. All button labels
are derived from 'ButtonPtr', with script-internal buttons using link-paths
set up via 'ButtonInternal[]' array, or any user-specified path depending on
requirement. An empty path (i.e. PathPtr[0] == '\0') obviously does not have a
link created, it just displays the button label. For a button-bar at the top
of the document use 1, bottom of the document use 2, and for just a bar with
no buttons at all use 0.
*/

ButtonBar (int Top1Bottom2)

{
#define MAX_BUTTON_COUNT 8

   static int  ButtonCount = -1;
   static char  *ButtonInternal [MAX_BUTTON_COUNT],
                *ButtonLabel [MAX_BUTTON_COUNT],
                *ButtonPath [MAX_BUTTON_COUNT];

   int  idx;
   char  *PathPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ButtonBar() %d\n", Top1Bottom2);

   if (ButtonCount == -1)
   {
      register char  *cptr, *sptr;

      if (Debug) fprintf (stdout, "|%s|\n", ButtonPtr);
      cptr = ButtonPtr;
      for (ButtonCount = 0;
           ButtonCount < MAX_BUTTON_COUNT && *cptr;
           ButtonCount++)
      {
         for (sptr = cptr; *sptr && *sptr != '=' && *sptr != ';'; sptr++)
            if (*sptr == '\\') memcpy (sptr, sptr+1, strlen(sptr));
         if (*sptr == '=') *sptr++ = '\0';
         ButtonLabel[ButtonCount] = cptr;
         cptr = sptr;
         for (sptr = cptr; *sptr && *sptr != ';'; sptr++)
            if (*sptr == '\\') memcpy (sptr, sptr+1, strlen(sptr));
         if (*sptr) *sptr++ = '\0';
         ButtonPath[ButtonCount] = cptr;
         cptr = sptr;
      }
   }

   if (Top1Bottom2)
   {
      /***********************************/
      /* set up script-specified buttons */
      /***********************************/

      SCRIPT_SPECIFIC_BUTTON_CODE
   }

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      /************/
      /* format 2 */
      /************/

      if (Top1Bottom2 == 2 || !Top1Bottom2)
      {
         fprintf (stdout, "<HR ALIGN=left SIZE=2 WIDTH=95%%>\n");
         if (!Top1Bottom2) return;
      }

      fprintf (stdout, "<FONT SIZE=-1><NOBR>\n");
      for (idx = 0; idx < ButtonCount; idx++)
      {
         if (ButtonInternal[idx] == NULL)
            PathPtr = ButtonPath[idx];
         else
            PathPtr = ButtonInternal[idx];
         if (idx) fprintf (stdout, "&nbsp;");
         if (PathPtr[0])
            fprintf (stdout, "[<A HREF=\"%s\">%s</A>]\n",
                     PathPtr, ButtonLabel[idx]);
         else
            fprintf (stdout, "[%s]\n", ButtonLabel[idx]);
      }
      fprintf (stdout, "</NOBR></FONT>\n");

      if (Top1Bottom2 == 1)
         fprintf (stdout, "<HR ALIGN=left SIZE=2 WIDTH=95%%>\n");
   }
   else
   {
      /************/
      /* format 1 */
      /************/

      fprintf (stdout,
"<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=0>\
<TR><TD HEIGHT=2></TD></TR>\
</TABLE>\n\
<TABLE BORDER=%s CELLPADDING=%s CELLSPACING=0 WIDTH=100%%>\n\
<TR><TD%s>\n",
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADBGCOLOR]);

      if (ButtonCount == 0 || !Top1Bottom2)
         fprintf (stdout, "&nbsp;\n");
      else
      {
         fprintf (stdout,
"<TABLE BORDER=%s CELLPADDING=1 CELLSPACING=0>\n",
            PageScheme[PS_BUTTONBORDER]);

         for (idx = 0; idx < ButtonCount; idx++)
         {
            if (ButtonInternal[idx] == NULL)
               PathPtr = ButtonPath[idx];
            else
               PathPtr = ButtonInternal[idx];
            if (PathPtr[0])
               fprintf (stdout,
"<TD ALIGN=center%s><FONT SIZE=-1>\
<NOBR>&nbsp;&nbsp;<A HREF=\"%s\">%s</A>&nbsp;&nbsp;</NOBR></FONT></TD>\n",
                  PageScheme[PS_BUTTONBGCOLOR], PathPtr, ButtonLabel[idx]);
            else
               fprintf (stdout,
"<TD ALIGN=center%s><FONT SIZE=-1>\
<NOBR>&nbsp;&nbsp;%s&nbsp;&nbsp;</NOBR></FONT></TD>\n",
                  PageScheme[PS_BUTTONBGCOLOR], ButtonLabel[idx]);
         }

         fprintf (stdout, "</TR></TABLE>\n");
      }

      fprintf (stdout,
"</TD></TR>\n\
</TABLE>\n\
<TABLE BORDER=0 CELLPADDING=0 CELLSPACING=0>\
<TR><TD HEIGHT=2></TD></TR>\
</TABLE>\n");
   }
}

/*****************************************************************************/
/*
Search files in specification for specified string.  Unless a file type 
(extension) matches an HTML type then consider the file to be a plain-text 
file.  The two need to searched differently because of the HTML markup tags 
compsrising an HTML file.
*/ 

SearchFiles
(
char *FileSpec,
int FileSpecLength,
char *SearchString,
boolean CaseSensitive,
boolean DocumentOnly
)
{
   static int  TimerElapsed = 1,
               TimerCpu = 2,
               TimerBio = 3,
               TimerDio = 4;
   static $DESCRIPTOR (ElapsedTimeFaoDsc, "!%T");
   static $DESCRIPTOR (StatisticsFaoDsc,
"<B>Elapsed:</B> !AZ &nbsp;<B>CPU:</B> !2ZL:!2ZL.!2ZL \
&nbsp;<B>I/O:</B> !UL &nbsp;<B>Disk:</B> !UL \
&nbsp;<B>Records (lines):</B> !UL");

   register char  *sptr;

   int  status,
        BioCount,
        CpuTime,
        DioCount,
        FileCount = 0,
        FileHitCount = 0,
        NotSearchedFileCount = 0,
        TotalHitCount = 0,
        RecordCount = 0,
        SearchStringLength;
   unsigned long  ElapsedTime [2];
   unsigned short  Length;
   char  *FilePtr,
         *FileHitPtr,
         *TotalHitPtr;
   char  DocumentName [2048],
         ElapsedString [32],
         ExpandedFileSpec [256],
         FileName [256],
         NotSearchedString [32],
         Statistics [256],
         String [512],
         UriSearchString [256];
   struct FAB  SearchFab;
   struct NAM  SearchNam;
   $DESCRIPTOR (ElapsedStringDsc, ElapsedString);
   $DESCRIPTOR (StatisticsDsc, Statistics);

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SearchFiles() |%s|%s|\n", FileSpec, UriSearchString);

   ListStarted = false;

   lib$init_timer (0);

   if (SearchString[0])
   {
      SearchStringLength = strlen(SearchString);
      CopyTextIntoUri (UriSearchString, SearchString, -1);
   }
   else
   {
      SearchStringLength = 0;
      UriSearchString[0] = '\0';
   }

   sptr = String;
   if (CgiFormWhatPtr[0])
      sptr += sprintf (sptr, "Search &nbsp;%s&nbsp; for &nbsp;&quot;",
                       CgiFormWhatPtr);
   else
      sptr += sprintf (sptr, "Search &nbsp;%s&nbsp; for &nbsp;&quot;",
                       CgiPathInfoPtr);
   sptr = CopyTextIntoHtml (sptr, SearchString, -1);
   strcpy (sptr, "&quot;");

   fprintf (stdout,
"%s\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"text-types\" CONTENT=\"%s\">\n\
<META NAME=\"HTML-types\" CONTENT=\"%s\">\n\
<TITLE>%s</TITLE>\n\
</HEAD>\n\
<BODY%s>\n",
      Http200Header,
      SoftwareID,
      TextFileTypesPtr,
      HtmlFileTypesPtr,
      String,
      PageScheme[PS_BODYTAG]);

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      fprintf (stdout,
"%s<FONT SIZE=+2><B>\n\
%s\n\
</B></FONT>\n\
<BR>\
<FONT SIZE=-1>\n\
&nbsp;<SUP>*</SUP><U>WASDquery</U>\n\
</FONT>\n\
<P>\n",
         PageScheme[PS_HEADLOCAL],
         String);
   }
   else
   {
      fprintf (stdout,
"<TABLE BORDER=%s CELLPADDING=%s CELLSPACING=0 WIDTH=100%%%s>\n\
<TR><TD>\n\
<FONT COLOR=\"%s\" SIZE=+2><B>\n\
%s\n\
</B></FONT>\n\
<BR>\
<FONT COLOR=\"%s\" SIZE=-1>\n\
&nbsp;<SUP>*</SUP><U>WASDquery</U>\n\
</FONT>\n\
</TD>%s</TR>\n\
</TABLE>\n",
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADPADDING],
         PageScheme[PS_HEADBGCOLOR],
         PageScheme[PS_HEADTEXT],
         String,
         PageScheme[PS_HEADTEXT],
         PageScheme[PS_HEADLOCAL]);
   }

   ButtonBar (1);

   fflush (stdout);
   HttpHasBeenOutput = true;

   SearchFab = cc$rms_fab;
   SearchFab.fab$l_fna = FileSpec;
   SearchFab.fab$b_fns = FileSpecLength;
   SearchFab.fab$l_fop = FAB$M_NAM;
   SearchFab.fab$l_nam = &SearchNam;
   SearchNam = cc$rms_nam;
   SearchNam.nam$l_esa = ExpandedFileSpec;
   SearchNam.nam$b_ess = sizeof(ExpandedFileSpec)-1;
   SearchNam.nam$l_rsa = FileName;
   SearchNam.nam$b_rss = sizeof(FileName)-1;

   if (VMSnok (status = sys$parse (&SearchFab, 0, 0)))
   {
      ErrorVmsStatus (status, CgiPathInfoPtr, FileSpec, __FILE__, __LINE__);
      return (status);
   }

   if (Debug) fprintf (stdout, "ExpandedFileSpec |%s|\n", ExpandedFileSpec);

   /***************/
   /* file search */
   /***************/

   while (VMSok (status = sys$search (&SearchFab, 0, 0)))
   {
      *SearchNam.nam$l_ver = '\0';
      if (Debug) fprintf (stdout, "FileName |%s|\n", FileName);

      if (SameFileType (HtmlFileTypesPtr, SearchNam.nam$l_type))
      {
         FileCount++;
         /* if HTML document's <title> can't be resolved then use file name */
         sprintf (DocumentName, "(file name) %s", SearchNam.nam$l_name);

         status = SearchHtmlFile (FileName, SearchNam.nam$l_ver-FileName,
                                  DocumentName,
                                  UriSearchString,
                                  SearchString, SearchStringLength,
                                  &RecordCount, &FileHitCount, &TotalHitCount,
                                  CaseSensitive, DocumentOnly);
         if (VMSnok (status))
         {
            ErrorVmsStatus (status, MapUrl_Map(NULL,FileName,NULL,NULL,NULL),
                            FileName, __FILE__, __LINE__);
            return (SS$_NORMAL);
         }
         else;
      }
      else
      if (SameFileType (TextFileTypesPtr, SearchNam.nam$l_type))
      {
         FileCount++;
         DocumentName[0] = '\0';

         status =
         SearchTextFile (FileName, SearchNam.nam$l_ver-FileName,
                         DocumentName,
                         UriSearchString,
                         SearchString, SearchStringLength,
                         &RecordCount, &FileHitCount, &TotalHitCount,
                         CaseSensitive, DocumentOnly,
                         ExtractNumberOfRecords, ExactNumberOfRecords);
         if (VMSnok (status))
         {
            ErrorVmsStatus (status, MapUrl_Map(NULL,FileName,NULL,NULL,NULL),
                            FileName, __FILE__, __LINE__);
            return (SS$_NORMAL);
         }
         else;
      }
      else
         NotSearchedFileCount++;

      *SearchNam.nam$l_ver = ';';
   }

   if (Debug) fprintf (stdout, "sys$search() %%X%08.08X\n", status);

   /*******************/
   /* end file search */
   /*******************/

   /* if its a search list treat directory not found as if file not found */
   if ((SearchNam.nam$l_fnb & NAM$M_SEARCH_LIST) && status == RMS$_DNF)
      status = RMS$_FNF;
   if (status == RMS$_FNF || status == RMS$_NMF) status = SS$_NORMAL;
   if (VMSnok (status))
   {
       ErrorVmsStatus (status, CgiPathInfoPtr, FileSpec, __FILE__, __LINE__);
       return (SS$_NORMAL);
   }

   /*********************/
   /* results of search */
   /*********************/

   lib$stat_timer (&TimerElapsed, &ElapsedTime, 0);
   lib$stat_timer (&TimerCpu, &CpuTime, 0);
   lib$stat_timer (&TimerBio, &BioCount, 0);
   lib$stat_timer (&TimerDio, &DioCount, 0);

   sys$fao (&ElapsedTimeFaoDsc, &Length, &ElapsedStringDsc, &ElapsedTime);
   ElapsedString[Length] = '\0';
   sys$fao (&StatisticsFaoDsc, &Length, &StatisticsDsc,
            ElapsedString+3, CpuTime/6000, CpuTime/100, CpuTime%100,
            BioCount, DioCount, RecordCount);
   Statistics[Length] = '\0';

   if (!NotSearchedFileCount)
      NotSearchedString[0] = '\0';
   else
   if (NotSearchedFileCount == 1)
      strcpy (NotSearchedString, " (1 not)");
   else
      sprintf (NotSearchedString, " (%d not)", NotSearchedFileCount);

   if (ListStarted) fputs ("</OL>\n", stdout);

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      fprintf (stdout,
"<P>\n\
<BLOCKQUOTE>\n\
<FONT SIZE=-1><NOBR>\n&nbsp;<U>");
   }
   else
   {
      fprintf (stdout,
"<P>\n\
<BLOCKQUOTE>\n\
<TABLE BORDER=%s CELLPADDING=2 CELLSPACING=0>\n\
<TR><TD%s>\n\
<FONT SIZE=-1 COLOR=\"%s\"><NOBR>\n&nbsp;",
         PageScheme[PS_HEADBORDER],
         PageScheme[PS_HEADBGCOLOR],
         PageScheme[PS_HEADTEXT]);
   }

   if (TotalHitCount)
   {
      if (FileCount == 1) FilePtr = "file"; else FilePtr = "files";
      if (FileHitCount == 1) FileHitPtr = "file"; else FileHitPtr = "files";
      if (TotalHitCount == 1) TotalHitPtr = "hit"; else TotalHitPtr = "hits";
      if (DocumentOnly)
      {
         fprintf (stdout,
"<B>%d %s searched%s with %d %s hit.</B>",
         FileCount, FilePtr, NotSearchedString, FileHitCount, FileHitPtr);
      }
      else
      {
         fprintf (stdout,
"<B>%d %s searched%s with %d %s hit, for a total of %d %s.</B>",
         FileCount, FilePtr, NotSearchedString, FileHitCount, FileHitPtr,
         TotalHitCount, TotalHitPtr);
      }
   }

   /*
       This cannot be an 'else' statement.
       Something wierd after change of carriage-control from '\r\n' to '\n',
       and, unfortunately, after update to DEC C v5.0 (so I don't know which
       caused it!)
       If it is an else the module access violates!
   */
   if (!TotalHitCount)
   {
      if (FileCount)
      {
         if (FileCount == 1) FilePtr = "file"; else FilePtr = "files";
         fprintf (stdout, "<B>%d %s searched%s, string not found.</B>",
                  FileCount, FilePtr, NotSearchedString);
      }
      else
         fprintf (stdout, "<B>No files found!</B>");
   }

   if (PageScheme[PS_LAYOUT][0] == '2') fprintf (stdout, "</U>");

   fprintf (stdout, "\n<BR>&nbsp;%s\n", Statistics);

   if (VMSnok (status))
   {
       ErrorVmsStatus (status, CgiPathInfoPtr, FileName, __FILE__, __LINE__);
       return (SS$_NORMAL);
   }

   if (PageScheme[PS_LAYOUT][0] == '2')
   {
      fprintf (stdout,
"</NOBR></FONT>\n\
</BLOCKQUOTE>\n");
   }
   else
   {
      fprintf (stdout,
"</NOBR></FONT>\n\
</TD></TR>\n\
</TABLE>\n\
</BLOCKQUOTE>\n");
   }

   ButtonBar (2);

   fprintf (stdout, "</BODY>\n</HTML>\n");

   return (status);
}

/*****************************************************************************/
/*
This function accepts a comma-separated list of file types (extensions, e.g.
"TXT,TEXT,COM,C,PAS,FOR") and a VMS file type (e.g. ".TXT;", ".TXT", "TXT").
It returns true if the file type is in the list, false if not.
*/

boolean SameFileType
(
char *TypeList,
char *FileType
)
{
   register char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SameFileType() |%s|%s|\n", TypeList, FileType);

   if (*FileType == '.') FileType++;
   cptr = TypeList;
   while (*cptr)
   {
      sptr = FileType;
      while (*sptr && *sptr != ';' && *cptr && *cptr != ',' &&
             toupper(*sptr) == toupper(*cptr))
      {
         cptr++;
         sptr++;
      }
      if ((!*sptr || *sptr == ';') && (!*cptr || *cptr == ',')) return (true);
      while (*cptr && *cptr != ',') cptr++;
      if (*cptr) cptr++;
   }
   return (false);
}

/*****************************************************************************/
/*
String search allowing wildcard "*" (matching any multiple characters) and "%" 
(matching any single character).  Returns NULL if not found or a pointer to
start of matched string.
*/ 

char* SearchTextString
( 
register char *InThat,
register char *This,
register boolean CaseSensitive,
int *MatchedLengthPtr
)
{
/* wildcards implied at both ends of the search string */
#define IMPLIED_WILDCARDS 1

   register char  *cptr, *sptr, *inptr;
   char  *RestartCptr,
         *RestartInptr,
         *MatchPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SearchTextString()\n|%s|%s|\n", This, InThat);

   if (MatchedLengthPtr != NULL) *MatchedLengthPtr = 0;
   if (!*(cptr = This)) return (NULL);
   inptr = MatchPtr = InThat;

#if IMPLIED_WILDCARDS
   /* skip leading text up to first matching character (if any!) */
   if (*cptr != '*' && *cptr != '%')
   {
      if (CaseSensitive)
         while (*inptr && *inptr != *cptr) inptr++;
      else
         while (*inptr && toupper(*inptr) != toupper(*cptr)) inptr++;
      if (Debug && !*inptr) fprintf (stdout, "1. NOT matched!\n");
      if (!*inptr) return (NULL);
      cptr++;
      MatchPtr = inptr++;
   }
#endif /* IMPLIED_WILDCARDS */

   for (;;)
   {
      if (CaseSensitive)
      {
         while (*cptr && *inptr && *cptr == *inptr)
         {
            cptr++;
            inptr++;
         }
      }
      else
      {
         while (*cptr && *inptr && toupper(*cptr) == toupper(*inptr))
         {
            cptr++;
            inptr++;
         }
      }

#if IMPLIED_WILDCARDS
      if (!*cptr)
      {
         if (Debug) fprintf (stdout, "1. matched!\n");
         if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
         return (MatchPtr);
      }
#else
      if (!*cptr && !*inptr)
      {
         if (Debug) fprintf (stdout, "2. matched!\n");
         if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
         return (MatchPtr);
      }
      else
      if (*cptr != '*' && *cptr != '%')
         return (NULL);
#endif /* IMPLIED_WILDCARDS */

      if (Debug && !*inptr) fprintf (stdout, "3. NOT matched!\n");
      if (!*inptr) return (NULL);

      if (*cptr != '*' && *cptr != '%')
      {
         cptr = This;
         MatchPtr = ++inptr;
         continue;
      }

      if (*cptr == '%')
      {
         /* single char wildcard processing */
         if (!*inptr) break;
         cptr++;
         inptr++;
         continue;
      }

      /* asterisk wildcard matching */
      while (*cptr == '*') cptr++;

      /* an asterisk wildcard at end matches all following */
      if (!*cptr)
      {
         if (Debug) fprintf (stdout, "4. matched!\n");
         while (*inptr) inptr++;
         if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
         return (MatchPtr);
      }

      /* note the current position in the string (first after the wildcard) */
      RestartCptr = cptr;
      for (;;)
      {
         /* find first char in InThat matching char after wildcard */
         if (CaseSensitive)
            while (*inptr && *cptr != *inptr) inptr++;
         else
            while (*inptr && toupper(*cptr) != toupper(*inptr)) inptr++;
         /* if did not find matching char in InThat being searched */
         if (Debug && !*inptr) fprintf (stdout, "5. NOT matched!\n");
         if (!*inptr) return (NULL);
         /* note the current position in InThat being searched */
         RestartInptr = inptr;
         /* try to match the remainder of the string and InThat */
         if (CaseSensitive)
         {
            while (*cptr && *inptr && *cptr == *inptr)
            {
               cptr++;
               inptr++;
            }
         }
         else
         {
            while (*cptr && *inptr && toupper(*cptr) == toupper(*inptr))
            {
               cptr++;
               inptr++;
            }
         }
         /* if reached the end of both string and InThat - match! */
#if IMPLIED_WILDCARDS
         if (!*cptr)
         {
            if (Debug) fprintf (stdout, "6. matched!\n");
            if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
            return (MatchPtr);
         }
#else
         if (!*cptr && !*inptr)
         {
            if (Debug) fprintf (stdout, "7. matched!\n");
            if (MatchedLengthPtr != NULL) *MatchedLengthPtr = inptr - MatchPtr;
            return (MatchPtr);
         }
#endif /* IMPLIED_WILDCARDS */
         /* break to the external loop if we encounter another wildcard */
         if (*cptr == '*' || *cptr == '%') break;
         /* lets have another go */
         cptr = RestartCptr;
         /* starting the character following the previous attempt */
         inptr = MatchPtr = RestartInptr + 1;
      }
   }
}

/*****************************************************************************/

#ifdef ORIGINAL_SEARCH

/*
Case sensitive or case insensitive search of the string pointed to by 'sptr' 
for the string pointed to by 'SearchString'.
*/ 

char* SearchTextString
( 
register char *tptr,
register char *SearchString,
register boolean CaseSensitive
)
{
   register char  *cptr, *sptr;

   sptr = SearchString;
   if (CaseSensitive)
   {
      while (*tptr)
      {
         if (*tptr++ != *sptr) continue;
         /* first character of search string matched record character */
         sptr++;
         cptr = tptr;
         while (*cptr && *sptr)
         {
            if (*cptr != *sptr) break;
            cptr++;
            sptr++;
         }
         if (!*sptr)
         {
            tptr--;
            break;
         }
         sptr = SearchString;
      }
   }
   else
   {
      while (*tptr)
      {
         if (toupper(*tptr++) != toupper(*sptr)) continue;
         /* first character of search string matched record character */
         sptr++;
         cptr = tptr;
         while (*cptr && *sptr)
         {
            if (toupper(*cptr) != toupper(*sptr)) break;
            cptr++;
            sptr++;
         }
         if (!*sptr)
         {
            tptr--;
            break;
         }
         sptr = SearchString;
      }
   }
   return (tptr);
}

#endif /* ORIGINAL_SEARCH */

/*****************************************************************************/
/*
Search an HTML marked up file.  Simply count the number of '<' and '>' 
characters, which should be balanced, and when not inside an HTML markup tag 
search the text.  As HTML files cannot easily have text extracted from within 
them without the results being unpredictable simply return the document as 
having the search string hit.  The HTML <title> tag (if present) is used as 
the document name.
*/ 

SearchHtmlFile
(
char *FileName,
int FileNameLength,
char *DocumentName,
char *UriSearchString,
char *SearchString,
int SearchStringLength,
int *RecordCountPtr,
int *FileHitCountPtr,
int *TotalHitCountPtr,
boolean CaseSensitive,
boolean DocumentOnly
)
{
   register int  TagCharCount = 0;
   register char  *cptr, *dptr, *rptr, *sptr, *tptr;

   boolean  InsideComment = false,
            RetrievingTitle = false;
   int  status,
        HitCount = 0,
        MatchedLength,
        RecordNumber = 0;
   char  ch;
   char  Record [MAX_RECORD_SIZE+1],
         String [2048],
         Text [2048];
   struct FAB  FileFAB;
   struct RAB  FileRAB;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SearchHtmlFile() |%s|%s|\n", FileName, SearchString);

   FileFAB = cc$rms_fab;
   FileFAB.fab$b_fac = FAB$M_GET;
   FileFAB.fab$l_fna = FileName;  
   FileFAB.fab$b_fns = FileNameLength;
   FileFAB.fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT;

   status = sys$open (&FileFAB, 0, 0);
   if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);

   if (VMSnok (status))
   {
      if (!ListStarted)
      {
         ListStarted = true;
         fputs ("<P>\n<OL>\n", stdout);
      }
      fprintf (stdout,
"<P><NOBR><B>ERROR</B> opening file (%%X%08.08X) \
<TT>%s</TT> <!-- %s --></NOBR><P>\n",
         status, MapUrl_Map(NULL,FileName,NULL,NULL,NULL), FileName);
      return (SS$_NORMAL);
   }

   FileRAB = cc$rms_rab;
   FileRAB.rab$l_fab = &FileFAB;
   /* 2 buffers and read ahead performance option */
   FileRAB.rab$b_mbf = 2;
   FileRAB.rab$l_rop = RAB$M_RAH;
   FileRAB.rab$l_ubf = Record;
   FileRAB.rab$w_usz = sizeof(Record)-1;

   if (VMSnok (status = sys$connect (&FileRAB, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&FileFAB, 0, 0);
      return (status);
   }

   /**********************/
   /* search all records */
   /**********************/

   while (VMSok (status = sys$get (&FileRAB, 0, 0)))
   {
      RecordNumber++;
      if (!FileRAB.rab$w_rsz) continue;
      Record[FileRAB.rab$w_rsz] = '\0';
      if (Debug) fprintf (stdout, "Record |%s|\n", Record);

      /* terminate on any carriage control that may be in the record */
      for (rptr = Record; *rptr && *rptr != '\r' && *rptr != '\n'; rptr++);
      *rptr = '\0';
      /* continue if empty line */
      if (!Record[0]) continue;

      /**************************************/
      /* retrieve text not inside HTML tags */
      /**************************************/

      tptr = Text;
      rptr = Record;
      while (*rptr)
      {
         if (InsideComment)
         {
            if (rptr[0] == '-' && rptr[1] == '-' && rptr[2] == '>')
            {
               InsideComment = false;
               rptr += 3;
            }
            else
               rptr++;
            continue;
         }
         if (*rptr == '<' && *((unsigned long*)rptr) == '<!--')
         {
            InsideComment = true;
            rptr += 4;
            continue;
         }

         /* less-thans are forbidden inside tags! */
         if ((TagCharCount & 1) && rptr[0] == '<') break;

         /* consider adjacent white-space when determining what is a tag */
         if ((rptr[0] == '<' && rptr[1] && !isspace(rptr[1])) ||
             ((TagCharCount & 1) && rptr[0] == '>'))
         {
            TagCharCount++;
            rptr++;
            if (TagCharCount & 1)
            {
               /* checks to detect start and end of title */
               if (toupper(*rptr) == 'T')
               {
                  if (strsame (rptr, "TITLE>", 6))
                  {
                     rptr += 5;
                     dptr = DocumentName;
                     RetrievingTitle = true;
                  }
                  else;
               }
               if (RetrievingTitle && *rptr == '/')
               {
                  if (strsame (rptr, "/TITLE>", 7))
                  {
                     rptr += 6;
                     *dptr = '\0';
                     RetrievingTitle = false;
                  }
               }
            }
         }
         else
         {
            if (RetrievingTitle)
            {
               if (dptr < DocumentName+255)
                  *dptr++ = *rptr++;
               else
                  rptr++;
            }
            else
            if (TagCharCount & 1)
               rptr++;
            else
            {
               if (*rptr == '&')
               {
                  if (strsame (rptr, "&lt;", 4))
                  {
                     *tptr++ = '<';
                     rptr += 4;
                  }
                  else
                  if (strsame (rptr, "&gt;", 4))
                  {
                     *tptr++ = '>';
                     rptr += 4;
                  }
                  else
                  if (strsame (rptr, "&amp;", 5))
                  {
                     *tptr++ = '&';
                     rptr += 5;
                  }
                  else
                  if (strsame (rptr, "&quot;", 6))
                  {
                     *tptr++ = '\"';
                     rptr += 6;
                  }
                  else
                  if (strsame (rptr, "&nbsp;", 6))
                  {
                     *tptr++ = ' ';
                     rptr += 6;
                  }
                  else
                  if (*(rptr+1) == '#')
                  {
                     for (cptr = rptr+2; *cptr && isdigit(*cptr); cptr++);
                     if (*cptr == ';')
                     {
                        ch = atoi(rptr+2) & 0xff;
                        *tptr++ = ch;
                        rptr = cptr + 1;
                     }
                     else
                        *tptr++ = *rptr++;
                  }
                  else
                     *tptr++ = *rptr++;
               }
               else
                  *tptr++ = *rptr++;
            }
         }

      }  /* while (*rptr) */
 
      *tptr = '\0';
      if (!Text[0]) continue;

      tptr = SearchTextString (Text, SearchString,
                               CaseSensitive, &MatchedLength);
      if (tptr != NULL)
      {
         /********/
         /* hit! */
         /********/

         if (Debug) fprintf (stdout, "Hit |%s|\n", tptr);
         if (!HitCount++)
         {
            /******************************/
            /* first hit, document anchor */
            /******************************/

            if (!ListStarted)
            {
               ListStarted = true;
               fputs ("<P>\n<OL>\n", stdout);
            }

            fprintf (stdout,
"<NOBR><B><LI><A HREF=\"%s\">%s</A></B> \
&nbsp;&nbsp;[<FONT SIZE=-1>HTML</FONT>]</NOBR>\n",
               MapUrl_Map(NULL,FileName,NULL,NULL,NULL), DocumentName);
            if (!DocumentOnly) fprintf (stdout, "<UL>\n");
         }
         *TotalHitCountPtr += 1;
         if (DocumentOnly) break;

         /*******************************/
         /* display line hit occured in */
         /*******************************/

         sptr = String;
         strcpy (sptr, "<LI>");
         sptr += 4;
         /* copy the text up to the first character of the search string */
         sptr = CopyTextIntoHtml (sptr, Text, tptr-Text);
         /* matched string, highlighted */
         strcpy (sptr, "<U>");
         sptr += 3;
         sptr = CopyTextIntoHtml (sptr, tptr, MatchedLength);
         strcpy (sptr, "</U>");
         sptr += 4;
         /* rest of Text after the matched search string */
         sptr = CopyTextIntoHtml (sptr, tptr+MatchedLength, -1);
         *sptr++ = '\r'; *sptr++ = '\n'; *sptr = '\0';

         if (Debug) fprintf (stdout, "String |%s|\n", String);
         fputs (String, stdout);
      }
   }

   /***************/
   /* end of file */
   /***************/

   if (status == RMS$_EOF) status = SS$_NORMAL;
   sys$close (&FileFAB, 0, 0);

   if (VMSnok (status))
   {
      if (!ListStarted)
      {
         ListStarted = true;
         fputs ("<P>\n<OL>\n", stdout);
      }
      fprintf (stdout,
"<P><NOBR><B>ERROR</B> reading file (%%X%08.08X) \
<TT>%s</TT> <!-- %s --></NOBR><P>\n",
         status, MapUrl_Map(NULL,FileName,NULL,NULL,NULL), FileName);
   }

   if (HitCount)
   {
      if (!DocumentOnly) fputs ("</UL>\n", stdout);
      *FileHitCountPtr += 1;
   }
   if (TagCharCount & 1)
   {
      /* must have encountered an opening '<' without a closing '>' */
      if (!ListStarted)
      {
         ListStarted = true;
         fputs ("<P>\n<OL>\n", stdout);
      }
      cptr = MapUrl_Map (NULL, FileName, NULL, NULL, NULL);
      fprintf (stdout,
"<P><NOBR><B>HTML problem</B>, unbalanced <B>&lt;&gt;</B> \
in <TT><A HREF=\"%s\">%s</A></TT> <!-- %s --></NOBR><P>\n",
         cptr, cptr, FileName);
   }

   *RecordCountPtr += RecordNumber;

   return (status);
}

/*****************************************************************************/
/*
Search each record of what is presumed to be a plain-text file for the 
supplied string.  When the first hit occurs output an HTML anchor for 
retrieving the entire file.  For each hit output an HTML anchor to extract a 
specified range of record (lines) from the file.  This search only checks for 
at least one match in a line before considering it a hit.
*/

SearchTextFile
(
char *FileName,
int FileNameLength,
char *DocumentName,
char *UriSearchString,
char *SearchString,
int SearchStringLength,
int *RecordCountPtr,
int *FileHitCountPtr,
int *TotalHitCountPtr,
boolean CaseSensitive,
boolean DocumentOnly,
int ExtractNumberOfRecords,
boolean ExactNumberOfRecords
)
{
   register char  *rptr, *sptr;
   register int  idx,
                 len,
                 LastSectionRecordNumber = 1,
                 RecordNumber = 0;

   int  status,
        HitCount = 0,
        MatchedLength;
   char  *CaseSensitivePtr = "";
   char  Record [MAX_RECORD_SIZE+1],
         String [2048];
   struct FAB  FileFAB;
   struct RAB  FileRAB;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SearchTextFile() |%s|%s|\n", FileName, SearchString);

   FileFAB = cc$rms_fab;
   FileFAB.fab$b_fac = FAB$M_GET;
   FileFAB.fab$l_fna = FileName;  
   FileFAB.fab$b_fns = FileNameLength;
   FileFAB.fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT;

   status = sys$open (&FileFAB, 0, 0);
   if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);

   if (VMSnok (status))
   {
      if (!ListStarted)
      {
         ListStarted = true;
         fputs ("<P>\n<OL>\n", stdout);
      }
      fprintf (stdout,
"<P><NOBR><B>ERROR</B> opening file (%%X%08.08X) \
<TT>%s</TT> <!-- %s --></NOBR><P>\n",
         status, MapUrl_Map(NULL,FileName,NULL,NULL,NULL), FileName);
      return (SS$_NORMAL);
   }

   FileRAB = cc$rms_rab;
   FileRAB.rab$l_fab = &FileFAB;
   /* 2 buffers and read ahead performance option */
   FileRAB.rab$b_mbf = 2;
   FileRAB.rab$l_rop = RAB$M_RAH;
   FileRAB.rab$l_ubf = Record;
   FileRAB.rab$w_usz = sizeof(Record)-1;

   if (VMSnok (status = sys$connect (&FileRAB, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&FileFAB, 0, 0);
      return (status);
   }

   if (CaseSensitive) CaseSensitivePtr = "&case=yes";

   /**********************/
   /* search all records */
   /**********************/

   while (VMSok (status = sys$get (&FileRAB, 0, 0)))
   {
      RecordNumber++;
      Record[FileRAB.rab$w_rsz] = '\0';

      /* terminate on any carriage control that may be in the record */
      for (rptr = Record; *rptr && *rptr != '\r' && *rptr != '\n'; rptr++);
      *rptr = '\0';

      /* if necessary generate document name from first non-blank line */
      if (!DocumentName[0]) GenerateDocumentName (Record, DocumentName);

      if (*(rptr = Record))
         while (*rptr && isspace(*rptr)) rptr++;
      if (!*rptr)
      {
         /* the line contained none, or only white-space characters */
         LastSectionRecordNumber = RecordNumber + 1;
         continue;
      }

      rptr = SearchTextString (Record, SearchString,
                               CaseSensitive, &MatchedLength);
      if (rptr != NULL)
      {
         /********/
         /* hit! */
         /********/

         if (Debug) fprintf (stdout, "Hit |%s|\n", rptr);

         if (!HitCount++)
         {
            /*************************************/
            /* first hit, entire document anchor */
            /*************************************/

            if (!ListStarted)
            {
               ListStarted = true;
               fputs ("<P>\n<OL>\n", stdout);
            }

            if (DocumentOnly)
            {
               fprintf (stdout,
"<NOBR><B><LI><A HREF=\"/%s%s?highlight=%s%s\">%s</A></B> \
&nbsp;&nbsp;[<FONT SIZE=-1>TEXT</FONT>]</NOBR>\n",
                  ExtractScriptName, MapUrl_Map(NULL,FileName,NULL,NULL,NULL),
                  UriSearchString, CaseSensitivePtr, DocumentName);

               /**********************************************/
               /* document only, break from search loop now! */
               /**********************************************/

               *TotalHitCountPtr += 1;
               break;
            }
            else
            {
               fprintf (stdout,
"<NOBR><B><LI><A HREF=\"/%s%s?highlight=%s%s\">%s</A></B> \
&nbsp;&nbsp;[<FONT SIZE=-1>TEXT</FONT>]</NOBR>\n\
<OL>\n",
                  ExtractScriptName, MapUrl_Map(NULL,FileName,NULL,NULL,NULL),
                  UriSearchString, CaseSensitivePtr, DocumentName);
               *TotalHitCountPtr += 1;
            }
         }

         /***********************/
         /* file extract anchor */
         /***********************/

         sptr = String;
         strcpy (sptr, "<LI>");
         sptr += 4;
         /* copy the record up to the first character of the search string */
         sptr = CopyTextIntoHtml (sptr, Record, rptr-Record);

         /* add the HTML anchor */
         len = sprintf (sptr,
            "<A HREF=\"/%s%s?highlight=%s",
            ExtractScriptName, MapUrl_Map(NULL,FileName,NULL,NULL,NULL),
            UriSearchString);
         sptr += len;

         if (ExactNumberOfRecords)
            len = sprintf (sptr, "&start=%d&end=%d&exact=true\">",
                  RecordNumber, RecordNumber+ExtractNumberOfRecords-1);
         else
            len = sprintf (sptr, "&start=%d&end=%d\">",
                  LastSectionRecordNumber,
                  RecordNumber+ExtractNumberOfRecords-1);
         sptr += len;

         /* matched string, highlighted within the anchor */
         sptr = CopyTextIntoHtml (sptr, rptr, MatchedLength);
         strcpy (sptr, "</A>");
         sptr += 4;

         /* rest of record after the matched search string */
         sptr = CopyTextIntoHtml (sptr, rptr+MatchedLength, -1);
         *sptr++ = '\n'; *sptr = '\0';

         if (Debug) fprintf (stdout, "String |%s|\n", String);
         fputs (String, stdout);
      }
   }

   /***************/
   /* end of file */
   /***************/

   if (status == RMS$_EOF) status = SS$_NORMAL;
   sys$close (&FileFAB, 0, 0);

   if (VMSnok (status))
   {
      if (!ListStarted)
      {
         ListStarted = true;
         fputs ("<P>\n<OL>\n", stdout);
      }

      fprintf (stdout,
"<P><NOBR><B>ERROR</B> reading file (%%X%08.08X) \
<TT>%s</TT> <!-- %s --></NOBR><P>\n",
         status, MapUrl_Map(NULL,FileName,NULL,NULL,NULL), FileName);
   }

   if (HitCount)
   {
      if (!DocumentOnly) fputs ("</OL>\n", stdout);
      *FileHitCountPtr += 1;
   }

   *RecordCountPtr += RecordNumber;

   return (status);
}

/*****************************************************************************/
/*
This function serves to generate a document name string (for use in the HTML
<TITLE> tag) for a plain-text file from the first line containing alpha-
numeric characters.  Copy the string pointed to by 'sptr' into the string 
pointed to by 'NamePtr', compressing white-space.
*/ 

GenerateDocumentName
(
register char *sptr,
register char *nptr
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GenerateDocumentName() |%s|\n", sptr);

   /* skip leading non-alphanumerics */
   while (*sptr && !isalnum(*sptr)) sptr++;
   while (*sptr)
   {
      /* copy alphanumeric element */
      while (*sptr && !isspace(*sptr) && isalnum(*sptr)) *nptr++ = *sptr++;
      if (!*sptr) break;
      /* skip intervening/trailing non-alphanumerics */
      while (*sptr && !isalnum(*sptr)) sptr++;
      if (!*sptr) break;
      /* add a single space between alphanumeric elements */
      *nptr++ = ' ';
   }
   *nptr = '\0';
}

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text in HTML.  For example the '<', '&', etc.  Convert these 
to the corresponding HTML character entities.
*/ 

char* CopyTextIntoHtml
( 
register char *bptr,
register char *tptr,
register int  ccnt
)
{
    while (ccnt-- && *tptr)
    {
       switch (*tptr)
       {
          case '\t' : while ((int)(bptr-1) % 8) *bptr++ = ' '; tptr++; break;
          case '<' : strcpy (bptr, "&lt;"); bptr += 4; tptr++; break;
          case '>' : strcpy (bptr, "&gt;"); bptr += 4; tptr++; break;
          case '&' : strcpy (bptr, "&amp;"); bptr += 5; tptr++; break;
          case '\"' : strcpy (bptr, "&quot;"); bptr += 6; tptr++; break;
          default : if (isprint(*tptr)) *bptr++ = *tptr++; else tptr++;
       }
    }
    *bptr = '\0';
    return (bptr);
}

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text text in an HTTP URL.  For example the '?', '+', '&', etc.  
Convert these to "%nn" hexadecimal escaped characters.
*/ 

char* CopyTextIntoUri
( 
register char *bptr,
register char *tptr,
register int  ccnt
)
{
    while (ccnt-- && *tptr)
    {
       switch (*tptr)
       {
          case '\t' : strcpy (bptr, "%09"); bptr += 3; tptr++; break;
          case ' ' : strcpy (bptr, "%20"); bptr += 3; tptr++; break;
          case '#' : strcpy (bptr, "%23"); bptr += 3; tptr++; break;
          case '+' : strcpy (bptr, "%2B"); bptr += 3; tptr++; break;
          case '/' : strcpy (bptr, "%2F"); bptr += 3; tptr++; break;
          case '=' : strcpy (bptr, "%3D"); bptr += 3; tptr++; break;
          case '?' : strcpy (bptr, "%3F"); bptr += 3; tptr++; break;
          default : if (isprint(*tptr)) *bptr++ = *tptr++; else tptr++;
       }
    }
    *bptr = '\0';
    return (bptr);
}

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text in an HTTP URL back into plain-text.  For example the 
'?', '+', '&', etc., are converted from "%nn" hexadecimal escaped characters. 
*/ 

char* CopyTextFromUri
(
register char *tptr,
register char *uptr,
register int  ccnt
)
{
   register char  c;
   register char  *sptr;

   while (ccnt-- && *uptr)
   {
      if (*uptr == '%')
      {
         /* an escaped character ("%xx" where xx is a hexadecimal number) */
         uptr++;
         sptr = uptr;
         if (*uptr) { ccnt--; uptr++; }
         if (*uptr) { ccnt--; uptr++; }
         c = *uptr;
         *uptr = '\0';
         *tptr++ = (unsigned char) strtol (sptr, NULL, 16);
         *uptr = c;
      }
      else
         *tptr++ = *uptr++;
   }
   *tptr = '\0';

   return (tptr);
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (HttpHasBeenOutput)
   {
      fprintf (stdout,
"<P><HR SIZE=3>\n\
<H1>ERROR!</H1>\n\
<!-- %s, %s, %d -->\n\
<P>Reported by server.\n\
<P>%s\n\
</BODY>\n\
</HTML>\n",
      SoftwareID, cptr, SourceLineNumber, Text);
   }
   else
   {
      fprintf (stdout,
"%s\n\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"module\" CONTENT=\"%s\">\n\
<META NAME=\"line\" CONTENT=\"%d\">\n\
<TITLE>Error 404</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n\
</BODY>\n\
</HTML>\n",
      Http404Header, SoftwareID, cptr, SourceLineNumber, Text);
   }

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      strcpy (Message, "&quot;sys$getmsg() failed&quot;");

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (HttpHasBeenOutput)
   {
      fprintf (stdout,
"<P><HR SIZE=3>\n\
<H1>ERROR!</H1>\n\
<!-- %s, %s, %d -->\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n\
</BODY>\n\
</HTML>\n",
      SoftwareID, cptr, SourceLineNumber,
      Message, Text, StatusValue, HiddenText);
   }
   else
   {
      fprintf (stdout,
"%s\n\
<HTML>\n\
<HEAD>\n\
<META NAME=\"generator\" CONTENT=\"%s\">\n\
<META NAME=\"module\" CONTENT=\"%s\">\n\
<META NAME=\"line\" CONTENT=\"%d\">\n\
<TITLE>Error 404</TITLE>\n\
</HEAD>\n\
<BODY>\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n\
</BODY>\n\
</HTML>\n",
      Http404Header, SoftwareID, cptr, SourceLineNumber,
      Message, Text, StatusValue, HiddenText);
   }

   return (SS$_NORMAL);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/

