/*****************************************************************************/
/*
                                sdm2htm.c

The SDM2HTM utility converts SDML (VAX DOCUMENT source) into HTML (Hyper-Text
Markup Language).  Not bullet-proof!  No apologies; this is a quick throw-
together to get our SDML documentation available to the hypertext environment.
(NOTE at 23-JAN-95; even with HyperReader working reasonably well and 
providing access to native Bookreader documents this utility still has a 
valuable function in providing HTML-formatted versions, which look and work 
better under Web browsers) 

This program cannot be understood without a knowledge of VAX DOCUMENT and 
HTML, and without reference to descriptions of DOCUMENT markup tags.

Uses recursive function calls extensively ...

The need for recursion is illustrated by the following SDML fragment:
"<CHAPTER>(The <UNDERLINE>(Need For <EMPHASIS>(Recursion\BOLD)))\symbol)

Uses fixed size buffers for all processing.  This means that there is fixed 
capacity for such constructs as a tag's parameter parentheses ( e.g. 
<tag>(...))  The size of these buffers is generous, but still finite.  Checks 
for buffer overflow are included. 

This utility could be improved by making these buffers dynamically allocated, 
and expand as string sizes demanded.  Sigh!

Many common GLOBAL tags are handled elegantly.  This utility does not pretend 
to provide a complete SDML conversion.  It is easily extensible.  It does not
do any sort of syntax checking.  It relies on the prior use of DOCUMENT to 
ensure that all all syntax problems are detected and corrected.  It assumes 
that the source is syntax-problem-free. 

The file names generated comprise three components.  First the source file 
name, separated from the other two by an underscore.  Second the <CHAPTER> 
number as two digits, 01 to 99.  Third, the sub-chapter (<EXAMPLE>) files as 
two digits, 00 to 99.  Hence file names look like ... NAME_0100.HTML
DOCUMENT <FRONT_MATTER> tags occur in files with digits 0000 to 0099.

A "Table Of Contents" file is created (NAME_0000).
A <TITLE_PAGE> tag becomes a "Title Page" HTML file (usually NAME_0001).
A <COPYRIGHT_PAGE> tag becomes a "Copyright Page" HTML file (usually NAME_0002).
A <PREFACE> tag becomes a "Preface" HTML file (usually NAME_0003).
Each <CHAPTER> becomes a separate HTML file (NAME_nn00). 
<EXAMPLE> tags become separate HTML files (NAME_nnnn).
<TABLE> tags become separate HTML files (NAME_nnnn).


HTML/DOCUMENT SPECIFIC OUTPUT
-----------------------------

HTML control is enabled from within <COMMENT>(...) tags.  As this is embedded 
in comment tags DOCUMENT nevers sees it, allowing HTML directives to be passed 
to this utility.  These directives are:

<COMMENT>(HTML/OFF) turns off HTML generation, allowing printed/Bookreader
specific text to be generated.

<COMMENT>(HTML/ON) turns HTML generation back on.

<COMMENT>(HTML=...) allows HTML specific markup text to be provided for Hyper-
Text documents, but is off course hidden from VAX DOCUMENT and is not 
included in printed/online text. 

<COMMENT>(HTML/POPUP=...) ... <COMMENT>(HTML/ENDPOPUP) creates a separate HTML 
file that is linked to using the text following the /POPUP=.


QUALIFIERS
----------
/[NO]COMMENTS           SDML <COMMENT>s are included (default: NO)
/BODY=                  string for <BODY> tag attributes
/DBUG                   turns on all "if (Debug)" statements
/DIRECTORY=             directory into which to place the generated HTML files
/[NO]FLAG_UNKNOWN_TAGS  report all unknown tags (default: NO)
/[NO]FRAMED             create an HTML <FRAMESET> framed document (default: NO)
/[NO]HTML               output HTML (default), /NOHTML allows checking
/INDEX=[file-name]      create a "home page" also with the name_0000.HTML
/[NO]UNKNOWN_TAGS       include commented-out  unknown flags (default: NO)
/[NO]VERBOSE            display progress messages (default)


BUILD DETAILS
-------------
See BUILD_SDM2HTM.COM


VERSION HISTORY (update SoftwareID as well!)
---------------
23-JAN-98  MGD  v1.7.0, "home page" copy of document name_0000.HTML,
                        "unmarkup" text for <TITLE></TITLE>
28-AUG-97  MGD  v1.6.1, minor fiddle to [contents] link in framed documents
03-AUG-97  MGD  v1.6.0, frames capability,
                        bugfix; "[next]" link truncated all links following
29-JUL-97  MGD  v1.5.0, /BODY qualifier
27-MAY-97  MGD  v1.4.0, added HTML DOCTYPE and META information
08-JUL-96  MGD  v1.3.0, output file record format changed from VAR to STMLF;
                        bugfix in concluding a popup-file;
                        changes to table formatting for better compliance
24-MAY-95  MGD  v1.2.0, minor changes for AXP compatibility
20-APR-95  MGD  v1.1.2, added <H1>book-title</H1> to each chapter
23-JAN-95  MGD  v1.1.1, remove "[" and "]" anchor descriptions
20-OCT-94  MGD  v1.1.0, ignore tags outside of <TITLE_PAGE>, <COPYRIGHT_PAGE>,
                        <PREFACE>, <CHAPTER> context ... shouldn't be there!
08-AUG-94  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID[] = "SDM2HTM AXP-1.7.0";
#else
   char SoftwareID[] = "SDM2HTM VAX-1.7.0";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <jpidef.h>
#include <libdtdef.h>
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>
#include <syidef.h>

#include "[-.httpd]copyright.h"

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))
 
#define OutputLineSize 8192
#define BufferSize 8192
#define InLineSize 1024
#define SymbolSize 32
#define TagNameSize 32
#define MaxHeadTag 6

#define HtmlDocType "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2//EN\">\n"

#define DefaultIndexPageName "INDEX.HTML"

/**************************/
/* data type declarations */
/**************************/

struct SdmlFileData
{
   struct FAB  FileFab;
   struct RAB  FileRab;
   struct NAM  FileNam;
   char  FileName [256];
   char  ExpandedFileName [256];
   char  InLine [InLineSize];
   int  LineNumber;
};

struct HtmlFileData
{
   struct FAB  FileFab;
   struct RAB  FileRab;
   char  FileName [256];
   char  NameOfFile [256];
};

struct ReferenceData
{
   struct ReferenceData  *NextPtr;
   int  ReferenceNumber;
   char  NameOfFile [64];
   char  Symbol [SymbolSize];
   char  Number [32];
   char  Title [256];
};

struct ErrorData
{
   char  *TagNamePtr;
   int  TagLineNumber;
   char *SdmlFileNamePtr;
   int  SourceLineNumber;
};

/******************/
/* global storage */
/******************/

char  Utility[] = "SDM2HTM";

boolean  Debug,
         DoFramed,
         DoGenerateHtml,
         DoVerbose,
         IncludeComments,
         IncludeSourceLineNumber,
         IncludeUnknownTags,
         FlagUnknownTags,
         HtmlOn;

int  FrontMatterCount,
     InsideComment,
     InsideFormalExample,
     InsideFormalTable,
     InsideLiteral,
     InsidePreformattedText,
     InsideTagCount,
     Pass,
     ReferenceNumber,
     PopUpNumber,
     TableOfContentsLevel,
     TableOfContentsCount,
     TitlePageCount,
     TotalChapters,
     UnknownTagCount;

/* section numbers range from 0 (chapter) to 6 (head6) */
int  SectionNumber[MaxHeadTag+1];

char  CommandLine [256],
      ChapterTitle [256],
      ContentsFileName [256],
      DocumentTitle [256],
      HtmlBodyTag [256],
      HtmlDirectory [256],
      HtmlName [256],
      IndexPageName [256],
      OutputLine [OutputLineSize],
      SdmlFileSpec [256];

char  *FrameTargetPage,
      *FrameTargetSelf,
      *FrameTargetToc,
      *FrameTargetTop,
      *OutputLinePtr;

char  EndListTag [64][8];
int  EndListTagIndex = 0;

struct FAB  ContentsFileFab;
struct RAB  ContentsFileRab;

struct HtmlFileData  HtmlFile;
struct SdmlFileData  SdmlFile;

struct ReferenceData  *RefPtr = NULL,
                      *LastPtr = NULL;

/***********************/
/* required prototypes */
/***********************/

char* AbsorbTagParameter (struct SdmlFileData*, char*);
struct ReferenceData* FindReference (char*);
char* HtmlCopy (struct ErrorData*, char*, int*, char*, int);
char* MakeReference (struct HtmlFileData*, struct ErrorData*,
                     char*, int*, char*);
char *MetaInformation (struct SdmlFileData *SdmlFilePtr);
char* NavigationButtons (boolean);
char* ProcessSdmlLines (struct SdmlFileData*, struct HtmlFileData*, char*);
char* ProcessTag (struct SdmlFileData*, struct HtmlFileData*,
                  struct ErrorData*, char*, char*, int);
char* ProcessTagParameter (struct SdmlFileData*, struct HtmlFileData*,
                           struct ErrorData*, char*, char*, int);
char* RawHtml (struct SdmlFileData*, struct HtmlFileData*,
               struct ErrorData*, char*, char*, int);
char* ReadSdmlLine (struct SdmlFileData*);
char* SkipUnknownTag (struct SdmlFileData*, struct ErrorData*,
                      char*, char*, int);
char* StringCopy (struct ErrorData*, char*, int*, char*, int);
char* SysGetMsg (int);
char* UnmarkupTitle (char*);

/*****************************************************************************/
/*
*/

int main ()

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (VMSnok (status = ParseCommandLine ()))
      exit (status);

   exit (SourceFile (SdmlFileSpec));
}

/*****************************************************************************/
/*
Using 'SdmlFileSpec' find matching file(s), process SDML text into HTML text.
*/ 

SourceFile (char *FileSpec)

{
   register char  *cptr;
   int  status,
        FileCount = 0;
   unsigned short  Length;
   char  c;
   char  DclCommand [256],
         FileName [256];
   struct FAB  ParseFab;
   struct NAM  ParseNam;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SourceFile() |%s|\n", FileSpec);

   ParseFab = cc$rms_fab;
   ParseFab.fab$l_dna = ".SDML";
   ParseFab.fab$b_dns = 5;
   ParseFab.fab$l_fna = FileSpec;
   ParseFab.fab$b_fns = strlen(FileSpec);
   ParseFab.fab$l_fop = FAB$M_NAM;
   ParseFab.fab$l_nam = &ParseNam;
   ParseNam = cc$rms_nam;
   ParseNam.nam$l_esa = FileName;
   ParseNam.nam$b_ess = sizeof(FileName)-1;

   if (VMSnok (status = sys$parse (&ParseFab, 0, 0)))
      return (status);

   if (ParseNam.nam$l_fnb & NAM$M_WILDCARD)
   {
      fprintf (stdout,
      "%%%s-E-WLD, wildcards not allowed for input specification\n",
      Utility);
      return (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   *ParseNam.nam$l_ver = '\0';
   if (Debug) fprintf (stdout, "FileName |%s|\n", FileName);

   if (!HtmlDirectory[0])
   {
      c = *ParseNam.nam$l_name;
      *ParseNam.nam$l_name = '\0';
      strcpy (HtmlDirectory, ParseNam.nam$l_dev);
      *ParseNam.nam$l_name = c;
   }
   *ParseNam.nam$l_type = '\0';
   strcpy (HtmlName, ParseNam.nam$l_name);
   *ParseNam.nam$l_type = '.';
   if (Debug) fprintf (stdout, "|%s|%s|\n", HtmlDirectory, HtmlName);

   *ParseNam.nam$l_ver = '\0';
   strcpy (SdmlFile.FileName, FileName);
   if (DoVerbose) fprintf (stdout, "%%%s-I-FILE, %s\n", Utility, FileName);
   *ParseNam.nam$l_ver = ';';

   SdmlFile.InLine[0] = HtmlFile.FileName[0] = HtmlFile.NameOfFile[0] = '\0';
   HtmlFile.FileFab = cc$rms_fab;
   HtmlFile.FileRab = cc$rms_rab;

   if (DoFramed)
   {
      FrameTargetPage = " TARGET=\"frame_page\"";
      FrameTargetSelf = " TARGET=\"_self\"";
      FrameTargetToc = " TARGET=\"frame_toc\"";
      FrameTargetTop = "TARGET=\"_top\" ";
   }
   else
      FrameTargetPage = FrameTargetSelf = FrameTargetTop = FrameTargetTop = "";

   /**************/
   /* first pass */
   /**************/

   /* count chapters, gets title, etc., */
   Pass = 1;
   if (DoVerbose) fprintf (stdout, "%%%s-I-PASS, 1\n", Utility);

   strcpy (DocumentTitle, "*** UNTITLED ***"); 
   TotalChapters = 0;

   FrontMatterCount =
   InsideComment =
   InsideFormalExample =
   InsideFormalTable =
   InsideLiteral =
   InsidePreformattedText =
   InsideTagCount =
   PopUpNumber =
   ReferenceNumber =
   SectionNumber[0] =
   TableOfContentsLevel =
   UnknownTagCount = 0;

   HtmlOn = true;

   if (VMSnok (status = ProcessSdmlFile (&SdmlFile, &HtmlFile)))
      return (status);

   TotalChapters = SectionNumber[0];

   /***************/
   /* second pass */
   /***************/

   /* outputs document */
   Pass = 2;
   if (DoVerbose) fprintf (stdout, "%%%s-I-PASS, 2\n", Utility);

   FrontMatterCount =
   InsideComment =
   InsideFormalExample =
   InsideFormalTable =
   InsideLiteral =
   InsidePreformattedText =
   InsideTagCount =
   PopUpNumber =
   ReferenceNumber =
   SectionNumber[0] =
   TableOfContentsLevel =
   UnknownTagCount = 0;

   HtmlOn = true;

   BeginContents (&SdmlFile);

   status = ProcessSdmlFile (&SdmlFile, &HtmlFile);

   EndHtmlFile (&SdmlFile, &HtmlFile);
   EndContents ();

   if (DoFramed) WriteFrameFile (&SdmlFile);

   if (IndexPageName[0]) WriteIndexPage ();

   if (DoVerbose && UnknownTagCount)
      fprintf (stdout, "%%%s-W-TAG, unknown tag(s) encountered %d time(s)\n",
               Utility, UnknownTagCount);

   return (status);
}

/*****************************************************************************/
/*
Recursively called function.

Open input SDML text file, process into HTML text file.
*/ 

int ProcessSdmlFile
(
struct SdmlFileData *SdmlFilePtr,
struct HtmlFileData *HtmlFilePtr
)
{
   register char  *cptr;
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ProcessSdmlFile() |%s|\n", SdmlFilePtr->FileName);

   SdmlFilePtr->FileFab = cc$rms_fab;
   SdmlFilePtr->FileFab.fab$b_fac = FAB$M_GET;
   SdmlFilePtr->FileFab.fab$l_fna = SdmlFilePtr->FileName;  
   SdmlFilePtr->FileFab.fab$b_fns = strlen(SdmlFilePtr->FileName);
   SdmlFilePtr->FileFab.fab$b_shr = FAB$M_SHRGET;
   SdmlFilePtr->FileFab.fab$l_fop = FAB$M_NAM;
   SdmlFilePtr->FileFab.fab$l_nam = &SdmlFilePtr->FileNam;
   SdmlFilePtr->FileNam = cc$rms_nam;
   SdmlFilePtr->FileNam.nam$l_esa = SdmlFilePtr->ExpandedFileName;
   SdmlFilePtr->FileNam.nam$b_ess = sizeof(SdmlFilePtr->ExpandedFileName)-1;

   if (VMSnok (status = sys$open (&SdmlFilePtr->FileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      fprintf (stdout, "%%%s-E-SDMLFILE, opening %s\n-%s\n",
               Utility, SdmlFilePtr->FileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   SdmlFilePtr->FileRab = cc$rms_rab;
   SdmlFilePtr->FileRab.rab$l_fab = &SdmlFilePtr->FileFab;
   /* 2 buffers and read ahead performance option */
   SdmlFilePtr->FileRab.rab$b_mbf = 2;
   SdmlFilePtr->FileRab.rab$l_rop = RAB$M_RAH;

   if (VMSnok (status = sys$connect (&SdmlFilePtr->FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&SdmlFilePtr->FileFab, 0, 0);
      fprintf (stdout, "%%%s-E-SDMLFILE, opening %s\n-%s\n",
               Utility, SdmlFilePtr->FileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   if (DoVerbose)
   {
      for (cptr = SdmlFilePtr->ExpandedFileName; *cptr && *cptr != ';'; cptr++);
      *cptr = '\0';
      fprintf (stdout, "[Processing %s]\n", SdmlFilePtr->ExpandedFileName);
   }

   SdmlFilePtr->FileRab.rab$l_ubf = SdmlFilePtr->InLine;
   SdmlFilePtr->FileRab.rab$w_usz = sizeof(SdmlFilePtr->InLine)-1;

   SdmlFilePtr->LineNumber = 0;

   if (ReadSdmlLine (SdmlFilePtr) == NULL)
      return (SS$_NORMAL);

   ProcessSdmlLines (SdmlFilePtr, HtmlFilePtr, SdmlFilePtr->InLine);

   sys$close (&SdmlFilePtr->FileFab, 0, 0);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Recursively called function.

Read lines (records) in the SDML text file, processing into HTML text.  This 
line processing can be terminated by end-of-file or by a tag returning a 
pseudo-tag indicating that a recursive call to this function should be 
concluded.  In this case a single SDML source file may not yet be exhausted.

If an opening '<' of a tag is encountered the function ProcessTag() is called 
to process this tag.  If it is tag with no parameters then a conversion into 
SDML will be made and returned in 'Scratch'.  If it has parameter (e.g. 
<tag>(...)), then it will in turn call function ProcessTagParameter(), which 
may in turn encounter tags within the parameter itself (e.g. "<UNDERLINE>(this 
is some <EMPHASIS>(example\BOLD) text)").  Eventually this function will be 
returned to with text built up into 'Scratch', which is then copied to the 
HTML output buffer.
*/ 

char* ProcessSdmlLines
(
struct SdmlFileData *SdmlFilePtr,
struct HtmlFileData *HtmlFilePtr,
register char *sdptr
)
{
   static int  RecursionLevel = 0;

   int  status,
        Capacity;
   char  Scratch [BufferSize];
   struct ErrorData  ErrorInfo;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessSdmlLines(%d)\n", ++RecursionLevel);

   ErrorInfo.SdmlFileNamePtr = SdmlFilePtr->FileName;

   FlushOutputLine (HtmlFilePtr);
   Capacity = sizeof(OutputLine);

   /*******************/
   /* read lines loop */
   /*******************/

   for (;;)
   {
      /*********************/
      /* process line loop */
      /*********************/

      while (*sdptr)
      {
         if (*sdptr == '<')
         {
            ErrorInfo.TagNamePtr = sdptr;
            ErrorInfo.TagLineNumber = SdmlFilePtr->LineNumber;

            sdptr = ProcessTag (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                sdptr, Scratch, sizeof(Scratch));

            /***********************************************************/
            /* detect pseudo-tag to exit from a recusive function call */
            /***********************************************************/

            if (strsame (Scratch, "<PSEUDO-END>", -1))
            {
               FlushOutputLine (HtmlFilePtr);
               RecursionLevel--;
               return (sdptr);
            }

            Capacity = sizeof(OutputLine) - 1 -
                       ((int)OutputLinePtr - (int)OutputLine);
            OutputLinePtr = StringCopy (NULL, OutputLinePtr, &Capacity,
                                        Scratch, -1);
         }
         else
         {
            if (HtmlOn && (!InsideComment || IncludeComments)) 
               OutputLinePtr = HtmlCopy (NULL, OutputLinePtr, &Capacity,
                                         sdptr, 1);
            /* this increment must be outside the if() statement above */
            sdptr++;
         }
      }

      /*************************/
      /* end process line loop */
      /*************************/

      FlushOutputLine (HtmlFilePtr);
      Capacity = sizeof(OutputLine);

      if (ReadSdmlLine (SdmlFilePtr) == NULL)
      {
         /***************/
         /* end-of-file */
         /***************/

         FlushOutputLine (HtmlFilePtr);
         RecursionLevel--;
         return (sdptr);
      }

      sdptr = SdmlFilePtr->InLine;

      /* add a new line to the output buffer */
      if (InsidePreformattedText && !*sdptr)
         OutputLinePtr = HtmlCopy (NULL, OutputLinePtr, &Capacity, "\n", -1);
   }
}

/*****************************************************************************/
/*
Recursively called function.

Some SDML tags have parameters.  One, or more, strings between parentheses, 
e.g. "<tag>(...)".  If multiple parameters then each is separated by a 
backslash, e.g. "\".  This function obtains one parameter from between the 
tag's "(...)".  If multiple parameters exists then each may be obtained by 
calling this function multiple times.  The terminating character, a ")" or "\" 
is returned being pointed at by 'sdptr' so that progress may be determined by 
the calling routine.

Calling this function with parameter 'htptr' set to NULL effectively absorbs 
any tag parameter.
*/ 

char* ProcessTagParameter
(
struct SdmlFileData  *SdmlFilePtr,
struct HtmlFileData  *HtmlFilePtr,
struct ErrorData *ErrorInfoPtr,
register char *sdptr,
register char *htptr,
int Capacity
)
{
   static int  RecursionLevel = 0;

   int  status,
        ParenthesesCount = 0;
   char  Buffer [BufferSize];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessTagParameter(%d)\n", ++RecursionLevel);

   /* allow for the terminating null */
   if (Capacity) Capacity--;

   /*******************/
   /* read lines loop */
   /*******************/

   for (;;)
   {
      /*********************/
      /* process line loop */
      /*********************/

      while (*sdptr)
      {
         if (!InsideComment && *sdptr == '<')
         {
            /********************************************/
            /* start of nested tag, process recursively */
            /********************************************/

            sdptr = ProcessTag (SdmlFilePtr, HtmlFilePtr, ErrorInfoPtr,
                                sdptr, Buffer, sizeof(Buffer));
            if (htptr != NULL)
            {
               ErrorInfoPtr->SourceLineNumber = __LINE__;
               htptr = StringCopy (ErrorInfoPtr, htptr, &Capacity, Buffer, -1);
            }
         }
         else
         if (*sdptr == '\\')
         {
            /**************************/
            /* start of new parameter */
            /**************************/

            RecursionLevel--;
            return (sdptr);
         }
         else
         if (*sdptr == '(')
         {
            /*****************************************/
            /* opening parenthesis (not tag-related) */
            /*****************************************/

            ParenthesesCount++;
            if (htptr != NULL) 
            {
               ErrorInfoPtr->SourceLineNumber = __LINE__;
               htptr = StringCopy (ErrorInfoPtr, htptr, &Capacity, sdptr, 1);
            }
            sdptr++;
         }
         else
         if (*sdptr == ')')
         {
            if (!ParenthesesCount)
            {
               /********************/
               /* end of parameter */
               /********************/

               InsideTagCount--;
               RecursionLevel--;
               /* return pointing at the tag parameter termination character */
               return (sdptr);
            }
            else
            {
               /*****************************************/
               /* closing parenthesis (not tag-related) */
               /*****************************************/

               ParenthesesCount--;
               if (htptr != NULL)
               {
                  ErrorInfoPtr->SourceLineNumber = __LINE__;
                  htptr = StringCopy (ErrorInfoPtr, htptr, &Capacity, sdptr, 1);
               }
               sdptr++;
            }
         }
         else
         {
            /***************************/
            /* copy a single character */
            /***************************/

            if (htptr != NULL)
            {
               ErrorInfoPtr->SourceLineNumber = __LINE__;
               htptr = HtmlCopy (ErrorInfoPtr, htptr, &Capacity, sdptr, 1);
            }
            sdptr++;
         }
      }

      if (ReadSdmlLine (SdmlFilePtr) == NULL)
      {
         /***************/
         /* end-of-file */
         /***************/

         RecursionLevel--;
         return (sdptr);
      }

      sdptr = SdmlFilePtr->InLine;

      /* add a new line to the output buffer */
      if (htptr != NULL)
      {
         ErrorInfoPtr->SourceLineNumber = __LINE__;
         htptr = StringCopy (ErrorInfoPtr, htptr, &Capacity, "\n", 1);
      }
   }
}

/*****************************************************************************/
/*
Recursively called function.

Sorry about its length!

The opening '<' of an SDML tag has been encountered.  It is pointed to by 
'sdptr'.  Get the tag name.  By string comparison find it, and process it. If 
not a known tag then absorb it if a tag without parameters, e.g. "<tag>", or 
absorb all between the parentheses if a tag with parameters, e.g. 
"<tag>(...)".

The 'htptr' that the tag is processed into is not directly for output into the 
HTML output file.  When functions ProcessSdmlLines() and 
ProcessTagParameters() encounter an opening '<' of a tag this function, 
ProcessTag(), is called with 'htptr' pointing at a fixed sized buffer.  When 
function ProcessTag() finishes processing the tag and returns the function 
that called it resumes with that buffer containing the
*/ 
 
char* ProcessTag
(
struct SdmlFileData *SdmlFilePtr,
struct HtmlFileData *HtmlFilePtr,
struct ErrorData *ErrorInfoPtr,
register char *sdptr,
register char *htptr,
int Capacity
)
{
   static int  RecursionLevel = 0;

   register int  Count;
   register char  *cptr, *sptr;
   int  status;
   char  HeadDigit;
   char  Number [32],
         Buffer [BufferSize],
         Scratch [BufferSize],
         Symbol [SymbolSize],
         TagName [TagNameSize];
   struct HtmlFileData  PopUpFile;
   struct SdmlFileData  IncludeFile;
   struct ErrorData  ErrorInfo;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessTag(%d)\n", ++RecursionLevel);

   /* allow for the terminating null */
   if (Capacity) Capacity--;

   *htptr = Buffer[0] = Scratch[0] = Symbol[0] = '\0';

   /****************************/
   /* handle literal tag first */
   /****************************/

   if (InsideLiteral)
   {
      if (strsame (sdptr, "<ENDLITERAL>", 12))
      {
         InsideLiteral--;
         sdptr += 12;
      }
      else
      {
         ErrorInfoPtr->SourceLineNumber = __LINE__;
         htptr = HtmlCopy (ErrorInfoPtr, htptr, &Capacity, sdptr++, 1);
      }

      RecursionLevel--;
      return (sdptr);
   }

   /**********************************************/
   /* handle HTML control sequences specifically */
   /**********************************************/

   if (strsame (sdptr, "<COMMENT>(", 10))
   {
      ErrorInfo.TagNamePtr = "<COMMENT>";
      ErrorInfo.TagLineNumber = SdmlFilePtr->LineNumber;
      ErrorInfo.SdmlFileNamePtr = SdmlFilePtr->FileName;

      if (strsame (sdptr+10, "HTML=", 5))
      {
         sdptr = RawHtml (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                          sdptr+15, Buffer, sizeof(Buffer));
         if (*sdptr == ')') sdptr++;
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         RecursionLevel--;
         return (sdptr);
      }

      if (strsame (sdptr+10, "HTML/ON)", 8))
      {
         HtmlOn = true;
         sdptr += 18;
         RecursionLevel--;
         return (sdptr);
      }

      if (strsame (sdptr+10, "HTML/OFF)", 9))
      {
         HtmlOn = false;
         sdptr += 19;
         RecursionLevel--;
         return (sdptr);
      }

      if (strsame (sdptr+10, "HTML/POPUP=", 11))
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+21, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, Symbol, sizeof(Symbol));
         if (*sdptr == ')') sdptr++;

         PopUpFile.FileFab = cc$rms_fab;
         PopUpFile.FileRab = cc$rms_rab;
         GenerateHtmlFileName (&PopUpFile, SectionNumber[0], ++PopUpNumber);
      
         /* create an HTML anchor point in the document for the example file */
         if (!Buffer[0]) strcpy (Buffer, "(no title)");
         sprintf (Scratch,
"\n\
<P>\n\
<A HREF=\"%s\"%s>%s</A>\n\
<P>\n",
         PopUpFile.NameOfFile, FrameTargetSelf, Buffer);
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Scratch, -1);

         /* flush any remaining output into the current HTML file */
         FlushOutputLine (HtmlFilePtr);

         CreateHtmlFile (&PopUpFile, SdmlFilePtr);

         sprintf (OutputLinePtr,
"%s\
<HTML>\n\
<HEAD>\n\
%s\
<TITLE>%s</TITLE>\n\
<HEAD>\n\
%s\
<A NAME=\"%d\">\n\
<H2>%s</H2>\n\
</A>\n",
         HtmlDocType, MetaInformation(SdmlFilePtr),
         UnmarkupTitle(ChapterTitle), HtmlBodyTag, ++ReferenceNumber, Buffer);
         while (*OutputLinePtr) OutputLinePtr++;

         NoteReference (HtmlFilePtr, Symbol, "", Buffer);

         /* recursive call to process the SDML text into the example file */
         sdptr = ProcessSdmlLines (SdmlFilePtr, &PopUpFile, sdptr);

         strcpy (OutputLinePtr, "</BODY>\n</HTML>");
         OutputLinePtr += 7;

         /* flush output into the current HTML file */
         FlushOutputLine (&PopUpFile);

         CloseHtmlFile (&PopUpFile);

         RecursionLevel--;
         return (sdptr);
      }

      if (strsame (sdptr+10, "HTML/ENDPOPUP)", 14))
      {
         /*
            This psuedo-tag is detected and actioned in function
            ProcessSdmlLines() to terminate the recursive call made
            to it by "HTML/POPUP=".
         */
         sdptr += 24;
         strcpy (htptr, "<PSEUDO-END>");
         htptr += 12;
         RecursionLevel--;
         return (sdptr);
      }
   }
   if (!HtmlOn)
   {
      /* absorb the '<' character */
      sdptr++;
      RecursionLevel--;
      return (sdptr);
   }

   /********************/
   /* get the tag name */
   /********************/

   /* put the tag information into the error data structure */
   ErrorInfo.TagNamePtr = sdptr;
   ErrorInfo.TagLineNumber = SdmlFilePtr->LineNumber;
   ErrorInfo.SdmlFileNamePtr = SdmlFilePtr->FileName;
   ErrorInfo.SourceLineNumber = __LINE__;

   Count = sizeof(TagName)-1;
   cptr = TagName;
   while (*sdptr && *sdptr != '>')
   {
      *cptr++ = *sdptr++;
      if (!Count--) BufferOverflow (&ErrorInfo);
   }
   *cptr++ = *sdptr++;
   *cptr = '\0';
   if (Debug) fprintf (stdout, "TagName |%s|\n", TagName);

   /* put the tag name into the error data structure */
   ErrorInfo.TagNamePtr = TagName;

   /***********************************/
   /* handle comment tag specifically */
   /***********************************/

   if (strsame (TagName, "<ENDCOMMENT>", -1))
   {
      if (InsideComment) InsideComment--;
      if (IncludeComments && !InsideComment)
      {
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, " -->", -1);
      }
      RecursionLevel--;
      return (sdptr);
   }

   if (InsideComment)
   {
      if (IncludeComments)
      {
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, TagName, -1);
      }
      RecursionLevel--;
      return (sdptr);
   }

   if (strsame (TagName, "<COMMENT>", -1))
   {
      if (*sdptr == '(')
      {
         if (IncludeComments && !InsideComment++)
         {
            ErrorInfo.SourceLineNumber = __LINE__;
            htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<!-- ", -1);
         }
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Scratch, sizeof(Scratch));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, NULL, 0);
         if (*sdptr == ')') sdptr++;
         if (InsideComment) InsideComment--;
         if (IncludeComments)
         {
            ErrorInfo.SourceLineNumber = __LINE__;
            htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Scratch, -1);
            if (!InsideComment)
            {
               ErrorInfo.SourceLineNumber = __LINE__;
               htptr = StringCopy (&ErrorInfo, htptr, &Capacity, " -->", -1);
            }
         }
      }
      else
      {
         if (IncludeComments && !InsideComment)
         {
            ErrorInfo.SourceLineNumber = __LINE__;
            htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<!-- ", -1);
         }
         InsideComment++;
      }
      RecursionLevel--;
      return (sdptr);
   }

   /*********************************/
   /* identify and process SDML tag */
   /*********************************/

   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ABSTRACT>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "\n<BLOCKQUOTE>", -1);
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, NULL, 0);
         if (*sdptr == ')') sdptr++;
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                             "</BLOCKQUOTE>\n", -1);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDABSTRACT>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</BLOCKQUOTE>\n", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<AMPERSAND>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "&amp;", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<BACKSLASH>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "\\", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<BOX>", -1))
   {
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, NULL, 0);
         if (*sdptr == ')') sdptr++;
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<TT>", -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</TT>", -1);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<CENTER_LINE>", -1))
   {
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, Scratch, sizeof(Scratch));
         if (*sdptr == ')') sdptr++;
      }
      ErrorInfo.SourceLineNumber = __LINE__;
      if (strsame (Scratch, "BIGSKIP", -1) ||
          strsame (Scratch, "SMALLSKIP", -1))
      {
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<P><CENTER>", -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</CENTER>", -1);
      }
      else
      {
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<CENTER>", -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</CENTER>", -1);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<CHAPTER>", -1))
   {
      /* finalize any previous HTML file (from previous chapters, etc.) */
      EndHtmlFile (SdmlFilePtr, HtmlFilePtr);

      /* increment the chapter number */
      SectionNumber[0]++;
      /* set all head counts back to zero */
      for (Count = 1; Count <= MaxHeadTag; SectionNumber[Count++] = 0);
      /* no popup sections yet (formal examples, tables, etc.) */
      PopUpNumber = 0;

      GenerateHtmlFileName (HtmlFilePtr, SectionNumber[0], PopUpNumber);
      CreateHtmlFile (HtmlFilePtr, SdmlFilePtr);

      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, HtmlDocType, -1);
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                          "<HTML>\n<HEAD>\n", -1);
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                          MetaInformation(SdmlFilePtr), -1);

      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                              sdptr+1, ChapterTitle, sizeof(ChapterTitle));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, Symbol, sizeof(Symbol));
         if (*sdptr == ')') sdptr++;
      }

      sprintf (Number, "%d", SectionNumber[0]);

      sprintf (Scratch, "<TITLE>%s</TITLE>\n", UnmarkupTitle(ChapterTitle));
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Scratch, -1);

      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</HEAD>\n", -1);
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, HtmlBodyTag, -1);

      sprintf (Scratch, "<H1>%s</H1>\n", DocumentTitle);
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Scratch, -1);

      sptr = NavigationButtons (false);
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, sptr, -1);

      sprintf (Scratch,
"\n\
<A NAME=\"%d\">\n\
<H1>%s - %s</H1>\n\
</A>\n",
      ++ReferenceNumber, Number, ChapterTitle);
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Scratch, -1);

      NoteReference (HtmlFilePtr, Symbol, Number, ChapterTitle);

      WriteContentsItem (HtmlFilePtr, 0, Symbol);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<CODE_EXAMPLE>", -1) ||
       strsame (TagName, "<DISPLAY>", -1))
   {
      if (!InsidePreformattedText++)
      {
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "\n<PRE>", -1);
      }
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, NULL, 0);
         if (*sdptr == ')') sdptr++;
         if (strsame (Buffer, "KEEP", -1) ||
             strsame (Buffer, "WIDE", -1) ||
             strsame (Buffer, "WIDE/MAXIMUM", -1) ||
             strsame (Buffer, "MAXIMUM", -1))
         {
            /* prevent a newline being generated immediately after "<PRE>" */
            while (*sdptr && isspace(*sdptr)) sdptr++;
            if (!*sdptr)
            {
               ReadSdmlLine (SdmlFilePtr);
               sdptr = SdmlFilePtr->InLine;
            }
         }
         else
         {
            if (InsidePreformattedText) InsidePreformattedText--;
            ErrorInfo.SourceLineNumber = __LINE__;
            htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
            if (!InsidePreformattedText)
            {
               ErrorInfo.SourceLineNumber = __LINE__;
               htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                                   "</PRE>\n", -1);
            } 
         } 
      }
      else
      {
         /* prevent a newline being generated immediately after the "<PRE>" */
         while (*sdptr && isspace(*sdptr)) sdptr++;
         if (!*sdptr)
         {
            ReadSdmlLine (SdmlFilePtr);
            sdptr = SdmlFilePtr->InLine;
         }
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDCODE_EXAMPLE>", -1) ||
       strsame (TagName, "<ENDDISPLAY>", -1))
   {
      if (InsidePreformattedText) InsidePreformattedText--;
      if (!InsidePreformattedText)
      {
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</PRE>\n", -1);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<CONTENTS_FILE>", -1))
   {
      if (*sdptr == '(') sdptr = AbsorbTagParameter (SdmlFilePtr, sdptr+1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<CP>", -1))
   {
      /* just absorb this parameter-less tag */
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<COPYRIGHT_PAGE>", -1))
   {
      EndHtmlFile (SdmlFilePtr, HtmlFilePtr);

      strcpy (ChapterTitle, "Copyright Page");
      GenerateHtmlFileName (HtmlFilePtr, 0, ++FrontMatterCount);
      CreateHtmlFile (HtmlFilePtr, SdmlFilePtr);

      sprintf (Scratch, "<TITLE>%s</TITLE>\n<H1>%s</H1>\n",
               UnmarkupTitle(ChapterTitle), DocumentTitle);
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Scratch, -1);

      /* in this case 'Symbol' will be generated by NoteReference() */
      ReferenceNumber++;
      NoteReference (HtmlFilePtr, Symbol, "", ChapterTitle);
      WriteContentsItem (HtmlFilePtr, 0, Symbol);

      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, HtmlDocType, -1);
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                          "<HTML>\n<HEAD>\n", -1);
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                          MetaInformation(SdmlFilePtr), -1);
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</HEAD>\n", -1);
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, HtmlBodyTag, -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDCOPYRIGHT_PAGE>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                          "</BODY>\n</HTML>\n", -1);
      EndHtmlFile (SdmlFilePtr, HtmlFilePtr);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<CPAREN>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, ")", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<DOUBLEQUOTE>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "\"", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ELLIPSIS>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      if (InsidePreformattedText)
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                             "\n.\n.\n.\n", -1);
      else
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                             "<BR>.\n<BR>.\n<BR>.\n<BR>", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<EMPHASIS>", -1))
   {
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, Scratch, sizeof(Scratch));
         if (*sdptr == ')') sdptr++;
      }
      ErrorInfo.SourceLineNumber = __LINE__;
      if (strsame (Scratch, "BOLD", -1))
      {
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<B>", -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</B>", -1);
      }
      else
      if (strsame (Scratch, "SMALLCAPS", -1))
      {
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<TT>", -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</TT>", -1);
      }
      else
      {
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<I>", -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</I>", -1);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<EXAMPLE>", -1))
   {
      if (*sdptr == '(')
      {
         /* its got a caption, its a formal example, into a file of its own */
         InsideFormalExample++;

         /*
            This is a reasonably complex tag to implement because it
            generates a separate HTML output file. It makes use of a
            recursive call to function ProcessSdmlLines().
         */

         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, Symbol, sizeof(Symbol));
         if (*sdptr == ')') sdptr++;

         GenerateHtmlFileName (&PopUpFile, SectionNumber[0], ++PopUpNumber);
      
         /* create an HTML anchor point in the document for the example file */
         if (!Buffer[0]) strcpy (Buffer, "(no title)");
         sprintf (Scratch, "\n<P>\n<A HREF=\"%s\"%s>[Example: %s]</A>\n<P>\n",
                  PopUpFile.NameOfFile, FrameTargetSelf, Buffer);
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Scratch, -1);

         /* flush any remaining output into the current HTML file */
         FlushOutputLine (HtmlFilePtr);

         CreateHtmlFile (&PopUpFile, SdmlFilePtr);

         sprintf (OutputLinePtr,
"%s\
<HTML>\n\
<HEAD>\n\
%s\
<TITLE>%s</TITLE>\n\
</HEAD>\n\
%s\
<H1>%s</H1>\n\
\n\
<A NAME=\"%d\">\n\
<H2>%s</H2>\n\
</A>\n",
         HtmlDocType, MetaInformation(SdmlFilePtr),
         UnmarkupTitle(Buffer), HtmlBodyTag, ChapterTitle,
         ++ReferenceNumber, Buffer);
         while (*OutputLinePtr) OutputLinePtr++;

         sprintf (Scratch, "Example: %s", Buffer);
         NoteReference (HtmlFilePtr, Symbol, "", Scratch);

         /* recursive call to process the SDML text into the example file */
         sdptr = ProcessSdmlLines (SdmlFilePtr, &PopUpFile, sdptr);

         strcpy (OutputLinePtr, "</BODY>\n</HTML>");
         OutputLinePtr += 7;

         /* flush output into the current HTML file */
         FlushOutputLine (&PopUpFile);

         CloseHtmlFile (&PopUpFile);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDEXAMPLE>", -1))
   {
      if (InsideFormalExample)
      {
         InsideFormalExample--;
         /*
            This psuedo-tag is detected and actioned in function
            ProcessSdmlLines() to terminate the recursive call made
            to it by <EXAMPLE> processing.
         */
         strcpy (htptr, "<PSEUDO-END>");
         htptr += 12;
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<EXAMPLE_ATTRIBUTES>", -1))
   {
      if (*sdptr == '(') sdptr = AbsorbTagParameter (SdmlFilePtr, sdptr+1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<FRONT_MATTER>", -1))
   {
      if (*sdptr == '(') sdptr = AbsorbTagParameter (SdmlFilePtr, sdptr+1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDFRONT_MATTER>", -1))
   {
      /* just absorb this parameter-less tag */
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<HEAD", 5) ||
       strsame (TagName, "<CHEAD>", -1))
   {
      if (isdigit (TagName[5]))
      {
         HeadDigit = atol (TagName+5);
         if (HeadDigit > MaxHeadTag) HeadDigit = 0;
      }
      else
         HeadDigit = 0;

      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, Symbol, sizeof(Symbol));
         if (*sdptr == ')') sdptr++;
      }

      if (HeadDigit)
      {
         /* numbered heading */
         sptr = Number;
         /* increment the current heading level ('HeadDigit' always >= 2) */
         SectionNumber[HeadDigit]++;
         /* output the heading numbering into the heading string */
         for (Count = 0; Count <= HeadDigit; Count++)
         {
            if (Count) *sptr++ = '.';
            sprintf (sptr, "%d", SectionNumber[Count]);
            while (*sptr) sptr++;
         }

         sprintf (Scratch,
"\n\
<A NAME=\"%d\">\n\
<H%d><BR><U>%s - %s</U></H%d>\n\
</A>\n",
         ++ReferenceNumber, HeadDigit+1, Number, Buffer, HeadDigit+1);
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Scratch, -1);

         /* create a reference for the numbered heading */
         NoteReference (HtmlFilePtr, Symbol, Number, Buffer);

         /* create a table of contents entry for the numbered heading */
         WriteContentsItem (HtmlFilePtr, HeadDigit, Symbol);

         /* reset the remainder of the heading counters to zero */
         for (Count = HeadDigit+1;
              Count <= MaxHeadTag;
              SectionNumber[Count++] = 0);
      }
      else
      {
         /* non-numbered heading */
         if (strsame (TagName, "<CHEAD>", -1))
         {
            sprintf (Scratch,
"\n\
<A NAME=\"%d\">\n\
<H3><CENTER>%s</CENTER></H3>\n\
</A>\n",
            ++ReferenceNumber, Buffer);
         }
         else
         {
            sprintf (Scratch,
"\n\
<A NAME=\"%d\">\n\
<H3>%s</H3>\n\
</A>\n",
            ++ReferenceNumber, Buffer);
         }
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Scratch, -1);

         /* create a reference for the non-numbered heading */
         NoteReference (HtmlFilePtr, Symbol, "", Buffer);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<INCLUDE>", -1) ||
       strsame (TagName, "<ELEMENT>", -1) ||
       strsame (TagName, "<EXAMPLE_FILE>", -1) ||
       strsame (TagName, "<TABLE_FILE>", -1))
   {
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, NULL, 0);
         if (*sdptr == ')') sdptr++;

         memset (&IncludeFile, 0, sizeof(struct SdmlFileData));
         strcpy (IncludeFile.FileName, Buffer);

         if (DoVerbose)
            fprintf (stdout, "[Include %s]\n", IncludeFile.FileName);

         ProcessSdmlFile (&IncludeFile, HtmlFilePtr);

         if (DoVerbose)
            fprintf (stdout, "[Resuming %s]\n", SdmlFilePtr->FileName);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<LE>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<LI>", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<HELLIPSIS>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "...", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<INDEX_FILE>", -1))
   {
      if (*sdptr == '(') sdptr = AbsorbTagParameter (SdmlFilePtr, sdptr+1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<KEEP>", -1))
   {
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, NULL, 0);
         if (*sdptr == ')') sdptr++;
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<LINE>", -1))
   {
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, NULL, 0);
         if (*sdptr == ')') sdptr++;
      }
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "\n<BR>", -1);
      if (strsame (Buffer, "SMALLSKIP", -1))
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "\n<BR>", -1);
      if (strsame (Buffer, "BIGSKIP", -1))
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "\n<BR>", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<LITERAL>", -1))
   {
      InsideLiteral++;
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));

         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         ErrorInfo.SourceLineNumber = __LINE__;
         while (*sdptr == '\\')
         {
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, Buffer, sizeof(Buffer));
            htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         }
         if (*sdptr == ')') sdptr++;
         InsideLiteral--;
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<LIST>", -1))
   {
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, NULL, 0);
         if (*sdptr == ')') sdptr++;
      }
      ErrorInfo.SourceLineNumber = __LINE__;
      if (strsame (Buffer, "NUMBERED", -1))
      {
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<OL>\n", -1);
         strcpy (EndListTag[EndListTagIndex++], "</OL>\n");
      }
      else
      if (strsame (Buffer, "ALPHABETIC", -1))
      {
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<OL>\n", -1);
         strcpy (EndListTag[EndListTagIndex++], "</OL>\n");
      }
      else
      {
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<UL>\n", -1);
         strcpy (EndListTag[EndListTagIndex++], "</UL>\n");
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDLIST>", -1))
   {
      if (EndListTagIndex)
      {
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                             EndListTag[--EndListTagIndex], -1);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<PAGE>", -1))
   {
      if (*sdptr == '(') sdptr = AbsorbTagParameter (SdmlFilePtr, sdptr+1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<NOTE>", -1))
   {
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, NULL, 0);
         if (*sdptr == ')') sdptr++;
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<H2>", -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                             "</H2>\n<BLOCKQUOTE>", -1);
      }
      else
      {
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                    "<BLOCKQUOTE><CENTER><B>- NOTE -</B></CENTER><BR>\n", -1);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDNOTE>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</BLOCKQUOTE>\n", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ONLINE_CHUNK>", -1))
   {
      /* just absorb this parameter-less tag */
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ONLINE_POPUP>", -1))
   {
      if (*sdptr == '(') sdptr = AbsorbTagParameter (SdmlFilePtr, sdptr+1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDONLINE_POPUP>", -1))
   {
      /* just absorb this parameter-less tag */
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ONLINE_TITLE>", -1))
   {
      if (*sdptr == '(') sdptr = AbsorbTagParameter (SdmlFilePtr, sdptr+1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<OPAREN>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "(", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<P>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "\n<P>\n", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<PREFACE>", -1))
   {
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, NULL, 0);
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, Symbol, sizeof(Symbol));
         if (*sdptr == ')') sdptr++;
      }

      EndHtmlFile (SdmlFilePtr, HtmlFilePtr);

      strcpy (ChapterTitle, "Preface");
      GenerateHtmlFileName (HtmlFilePtr, 0, ++FrontMatterCount);
      CreateHtmlFile (HtmlFilePtr, SdmlFilePtr);

      sprintf (Scratch, "<TITLE>%s</TITLE>\n<H1>%s</H1>\n",
               UnmarkupTitle(ChapterTitle), DocumentTitle);
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Scratch, -1);

      ReferenceNumber++;
      NoteReference (HtmlFilePtr, Symbol, "", ChapterTitle);
      WriteContentsItem (HtmlFilePtr, 0, Symbol);

      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, HtmlDocType, -1);
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                          "<HTML>\n<HEAD>\n", -1);
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                          MetaInformation(SdmlFilePtr), -1);
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</HEAD>\n", -1);
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, HtmlBodyTag, -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDPREFACE>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                          "</BODY>\n</HTML>\n", -1);
      EndHtmlFile (SdmlFilePtr, HtmlFilePtr);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<PROFILE>", -1))
   {
      /* just absorb this parameter-less tag */
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDPROFILE>", -1))
   {
      /* just absorb this parameter-less tag */
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<QUOTE>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "&quot;", -1);
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, NULL, 0);
         if (*sdptr == ')') sdptr++;
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "&quot;", -1);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDQUOTE>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "&quot;", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<REFERENCE>", -1))
   {
      Buffer[0] = '\0';
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Symbol, sizeof(Symbol));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, NULL, 0);
         if (*sdptr == ')') sdptr++;
      }
      if (Symbol[0])
         htptr = MakeReference (HtmlFilePtr, &ErrorInfo,
                                htptr, &Capacity, Symbol);
   }
   else 
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<REVISION_INFO>", -1))
   {
      Scratch[0] = Buffer[0] = '\0';
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Scratch, sizeof(Scratch));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, Buffer, sizeof(Buffer));
         if (*sdptr == ')') sdptr++;

         if (!Scratch[0]) strcpy (Scratch, "Revision Information");
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "\n<H2>", -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Scratch, -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</H2>\n<P>\n", -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
      }
   }
   else 
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<RULE>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "\n<HR>\n", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<SAMPLE_TEXT>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "\n<BLOCKQUOTE>", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDSAMPLE_TEXT>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</BLOCKQUOTE>\n", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<TABLE>", -1))
   {
      if (*sdptr == '(')
      {
         /* its got a caption, its a formal table, into a file of its own */
         InsideFormalTable++;

         /*
            This is a reasonably complex tag to implement because it
            generates a separate HTML output file. It makes use of a
            recursive call to function ProcessSdmlLines().  It also
            flushes any HTML text in the output buffer (hence the global
            variable 'OutputLinePtr') before and after processing the table.
         */

         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, Symbol, sizeof(Symbol));
         if (*sdptr == ')') sdptr++;

         GenerateHtmlFileName (&PopUpFile, SectionNumber[0], ++PopUpNumber);
      
         /* create an HTML anchor point in the document for the table file */
         if (!Buffer[0]) strcpy (Buffer, "(no title)");
         sprintf (Scratch, "\n<P>\n<A HREF=\"%s\"%s>[Table: %s]</A>\n<P>\n",
                  PopUpFile.NameOfFile, FrameTargetSelf, Buffer);
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Scratch, -1);

         /* flush any remaining output into the current HTML file */
         FlushOutputLine (HtmlFilePtr);

         CreateHtmlFile (&PopUpFile, SdmlFilePtr);

         sprintf (OutputLinePtr,
"%s\
<HTML>\n\
<HEAD>\n\
%s\
<TITLE>%s</TITLE>\n\
</HEAD>\n\
%s\
<H1>%s</H1>\n\
\n\
<CENTER><TABLE CELLPADDING=2 BORDER=1>\n\
<A NAME=\"%d\">\n\
<CAPTION>Table: %s</CAPTION>\n\
</A>\n",
         HtmlDocType, MetaInformation(SdmlFilePtr),
         UnmarkupTitle(Buffer), HtmlBodyTag, ChapterTitle,
         ++ReferenceNumber, Buffer);
         while (*OutputLinePtr) OutputLinePtr++;

         /* recursive call to process the SDML text into the table file */
         sdptr = ProcessSdmlLines (SdmlFilePtr, &PopUpFile, sdptr);

         strcpy (OutputLinePtr, "</TABLE></CENTER>\n</BODY>\n</HTML>\n");
         OutputLinePtr += 15;

         /* flush output into the current HTML file */
         FlushOutputLine (&PopUpFile);

         CloseHtmlFile (&PopUpFile);
      }
      else
      {
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                 "\n<BR><CENTER><TABLE CELLPADDING=2 BORDER=1>\n", -1);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDTABLE>", -1))
   {
      if (InsideFormalTable)
      {
         InsideFormalTable--;
         /*
            This psuedo-tag is detected and actioned in function
            ProcessSdmlLines() to terminate the recursive call made
            to it by <TABLE> processing.
         */
         strcpy (htptr, "<PSEUDO-END>");
         htptr += 12;
      }
      else
      {
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                             "</TABLE></CENTER><BR>\n\n", -1);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<TABLE_HEADS>", -1))
   {
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<TH>", -1);
         ErrorInfo.SourceLineNumber = __LINE__;
         while (*sdptr == '\\')
         {
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, Buffer, sizeof(Buffer));
            htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<TH>", -1);
            htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         }
         if (*sdptr == ')') sdptr++;
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</TH>", -1);
      }
   }
   else 
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<TABLE_KEY>", -1))
   {
      /* just absorb this parameter-less tag */
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDTABLE_KEY>", -1))
   {
      /* just absorb this parameter-less tag */
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<TABLE_KEYREF>", -1))
   {
      /* just absorb this parameter-less tag */
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<TABLE_ATTRIBUTES>", -1))
   {
      if (*sdptr == '(') sdptr = AbsorbTagParameter (SdmlFilePtr, sdptr+1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<TABLE_ROW>", -1))
   {
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<TR><TD>", -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         ErrorInfo.SourceLineNumber = __LINE__;
         while (*sdptr == '\\')
         {
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, Buffer, sizeof(Buffer));
            htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<TD>", -1);
            htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         }
         if (*sdptr == ')') sdptr++;
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</TR>\n", -1);
      }
   }
   else 
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<TABLE_ROW_BREAK>", -1))
   {
      if (*sdptr == '(') sdptr = AbsorbTagParameter (SdmlFilePtr, sdptr+1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<TABLE_SPACE>", -1))
   {
      if (*sdptr == '(') sdptr = AbsorbTagParameter (SdmlFilePtr, sdptr+1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<TABLE_SETUP>", -1))
   {
      if (*sdptr == '(') sdptr = AbsorbTagParameter (SdmlFilePtr, sdptr+1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<TABLE_UNIT>", -1))
   {
      /* just absorb this parameter-less tag */
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDTABLE_UNIT>", -1))
   {
      /* just absorb this parameter-less tag */
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<TABLE_UNIT_HEADS>", -1))
   {
      if (*sdptr == '(') sdptr = AbsorbTagParameter (SdmlFilePtr, sdptr+1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<TITLE>", -1))
   {
      /* note: <title>() tags can have up to 3 '\'-separated lines in them */
      if (*sdptr == '(')
      {
         cptr = DocumentTitle;
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         for (sptr = Buffer; *sptr; *cptr++ = *sptr++);
         while (*sdptr == '\\')
         {
            *cptr++ = ' ';
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, Buffer, sizeof(Buffer));
            for (sptr = Buffer; *sptr; *cptr++ = *sptr++);
         }
         *cptr = '\0';
         if (*sdptr == ')') sdptr++;
      }
      sprintf (Scratch, "<TITLE>%s</TITLE>\n<H1>%s</H1>\n",
               UnmarkupTitle(DocumentTitle), DocumentTitle);
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Scratch, -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<TITLE_PAGE>", -1))
   {
      EndHtmlFile (SdmlFilePtr, HtmlFilePtr);

      TitlePageCount = ++FrontMatterCount;
      GenerateHtmlFileName (HtmlFilePtr, 0, TitlePageCount);
      CreateHtmlFile (HtmlFilePtr, SdmlFilePtr);

      /* in this case 'Symbol' will be generated by NoteReference() */
      NoteReference (HtmlFilePtr, Symbol, "", "Title Page");
      WriteContentsItem (HtmlFilePtr, 0, Symbol);

      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, HtmlDocType, -1);
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                          "<HTML>\n<HEAD>\n", -1);
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                          MetaInformation(SdmlFilePtr), -1);
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</HEAD>\n", -1);
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, HtmlBodyTag, -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDTITLE_PAGE>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity,
                          "</BODY>\n</HTML>\n", -1);
      EndHtmlFile (SdmlFilePtr, HtmlFilePtr);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<UNDERLINE>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<U>", -1);
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, Buffer, sizeof(Buffer));
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, NULL, 0);
         if (*sdptr == ')') sdptr++;
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Buffer, -1);
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</U>", -1);
      }
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<ENDUNDERLINE>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "</U>", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<VBAR>", -1))
   {
      ErrorInfo.SourceLineNumber = __LINE__;
      htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "|", -1);
   }
   else
   /*------------------------------------------------------------------------*/
   if (strsame (TagName, "<X>", -1) ||
       strsame (TagName, "<XS>", -1) ||
       strsame (TagName, "<XSUBENTRY>", -1))
   {
      if (*sdptr == '(')
      {
         sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                      sdptr+1, NULL, 0);
         while (*sdptr == '\\')
            sdptr = ProcessTagParameter (SdmlFilePtr, HtmlFilePtr, &ErrorInfo,
                                         sdptr+1, NULL, 0);
         if (*sdptr == ')') sdptr++;
      }
   }
   else
   /*------------------------------------------------------------------------*/
   {
      /*************************/
      /* unrecognised SDML tag */
      /*************************/

      if (Pass == 2)
      {
         UnknownTagCount++;
         if (FlagUnknownTags)
         {
            fprintf (stdout, "%%%s-W-TAG, unknown %s in line %d\n",
                     Utility, TagName, SdmlFilePtr->LineNumber);
         }
      }
      if (!InsideComment++)
      {
         if (IncludeUnknownTags)
         {
            ErrorInfo.SourceLineNumber = __LINE__;
            htptr = StringCopy (&ErrorInfo, htptr, &Capacity, "<!-- ", -1);
         }
      }
      if (IncludeUnknownTags)
      {
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = HtmlCopy (&ErrorInfo, htptr, &Capacity, TagName, -1);
      }
      if (*sdptr == '(')
      {
         sdptr = SkipUnknownTag (SdmlFilePtr, &ErrorInfo,
                                 sdptr+1, Scratch, sizeof(Scratch));
         if (IncludeUnknownTags)
         {
            ErrorInfo.SourceLineNumber = __LINE__;
            htptr = StringCopy (&ErrorInfo, htptr, &Capacity, Scratch, -1);
         }
         else
            sdptr++;
      }
      if (InsideComment) InsideComment--;
      if (!InsideComment && IncludeUnknownTags)
      {
         ErrorInfo.SourceLineNumber = __LINE__;
         htptr = StringCopy (&ErrorInfo, htptr, &Capacity, " -->", -1);
      }
   }

   RecursionLevel--;
   return (sdptr);
}

/*****************************************************************************/
/*
*/ 
 
EndHtmlFile
(
struct SdmlFileData *SdmlFilePtr,
struct HtmlFileData *HtmlFilePtr
)
{
   register char  *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "EndHtmlFile() |%s|\n", HtmlFilePtr->FileName);

   if (HtmlFilePtr->FileName[0])
   {
      FlushOutputLine (HtmlFilePtr);
      if (SectionNumber[0])
      {
         /* there has been a previous chapter being output, finalize */
         sptr = NavigationButtons (true);
         WriteHtml (HtmlFilePtr, sptr);
         WriteHtml (HtmlFilePtr, "</BODY>\n</HTML>");
      }
      CloseHtmlFile (HtmlFilePtr);
      HtmlFilePtr->FileName[0] = '\0';
   }
}

/*****************************************************************************/
/*
Raw HTML can be introduced into output text from the SDML source text by using 
the construct "<COMMENT>(HTML=...)".
*/ 

char* RawHtml
(
struct SdmlFileData *SdmlFilePtr,
struct HtmlFileData *HtmlFilePtr,
struct ErrorData *ErrorInfoPtr,
register char *sdptr,
register char *htptr,
int Capacity
)
{
   register char  *cptr;

   int  status,
        ParenthesisCount = 0;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "RawHtml()\n");

   /* allow for the terminating null */
   if (Capacity) Capacity--;

   for (;;)
   {
      while (!*sdptr)
      {
         *htptr++ = '\n';
         if (!Capacity--) BufferOverflow (ErrorInfoPtr);
         if (ReadSdmlLine (SdmlFilePtr) == NULL)
         {
            *htptr = '\0';
            return (sdptr);
         }
         sdptr = SdmlFilePtr->InLine;
      }

      if (*sdptr == ')')
      {
         if (!ParenthesisCount)
         {
            *htptr = '\0';
            return (sdptr);
         }
         ParenthesisCount--;
         if (Debug) fprintf (stdout, "Par..Count %d\n", ParenthesisCount);
         *htptr++ = *sdptr++;
         if (!Capacity--) BufferOverflow (ErrorInfoPtr);
         continue;
      }

      if (*sdptr == '(')
      {
         ParenthesisCount++;
         if (Debug) fprintf (stdout, "Par..Count %d\n", ParenthesisCount);
         *htptr++ = *sdptr++;
         if (!Capacity--) BufferOverflow (ErrorInfoPtr);
         continue;
      }

      if (*sdptr == '<')
      {
         *htptr++ = *sdptr++;
         if (!Capacity--) BufferOverflow (ErrorInfoPtr);
         if (toupper(*sdptr) == 'A')
         {
            *htptr++ = *sdptr++;
            if (!Capacity--) BufferOverflow (ErrorInfoPtr);
            while (*sdptr && isspace(*sdptr))
            {
               *htptr++ = *sdptr++;
               if (!Capacity--) BufferOverflow (ErrorInfoPtr);
            }
            if (strsame (sdptr, "HREF=\"", 6))
            {
               cptr = FrameTargetTop;
               while (*cptr)
               {
                  *htptr++ = *cptr++;
                  if (!Capacity--) BufferOverflow (ErrorInfoPtr);
               }
            }
         }
         continue;
      }

      *htptr++ = *sdptr++;
      if (!Capacity--) BufferOverflow (ErrorInfoPtr);
   }
}

/*****************************************************************************/
/*
Recursively called function.

Skip all text between the parentheses of a "<tag>(...)" construct.
*/ 

char* AbsorbTagParameter
(
struct SdmlFileData *SdmlFilePtr,
register char  *sdptr
)
{
   register int  status,
                 ParenthesisCount = 0;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AbsorbTagParameter()\n");

   for (;;)
   {
      if (!*sdptr)
      {
         if (ReadSdmlLine (SdmlFilePtr) == NULL)
            return (sdptr);
         sdptr = SdmlFilePtr->InLine;
      }
      if (*sdptr == ')')
      {
         if (!ParenthesisCount) return (sdptr+1);
         ParenthesisCount--;
         if (Debug) fprintf (stdout, "Par..Count %d\n", ParenthesisCount);
      }
      if (*sdptr == '(')
      {
         ParenthesisCount++;
         if (Debug) fprintf (stdout, "Par..Count %d\n", ParenthesisCount);
      }
      sdptr++;
   }
}

/*****************************************************************************/
/*
Recursively called function.

Get all text between the parentheses of a "<tag>(...)" construct.
*/ 

char* SkipUnknownTag
(
struct SdmlFileData *SdmlFilePtr,
struct ErrorData *ErrorInfoPtr,
register char  *sdptr,
register char  *htptr,
int Capacity
)
{
   register int  status,
                 ParenthesisCount = 0;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SkipUnknownTag()\n");

   for (;;)
   {
      if (!*sdptr)
      {
         if (ReadSdmlLine (SdmlFilePtr) == NULL)
            return (sdptr);
         sdptr = SdmlFilePtr->InLine;
      }
      if (*sdptr == ')')
      {
         if (!ParenthesisCount) return (sdptr+1);
         ParenthesisCount--;
         if (Debug) fprintf (stdout, "Par..Count %d\n", ParenthesisCount);
      }
      if (*sdptr == '(')
      {
         ParenthesisCount++;
         if (Debug) fprintf (stdout, "Par..Count %d\n", ParenthesisCount);
      }
      ErrorInfoPtr->SourceLineNumber = __LINE__;
      htptr = HtmlCopy (ErrorInfoPtr, htptr, &Capacity, sdptr++, 1);
   }
}

/*****************************************************************************/
/*
Put HTML anchors "[contents]", "[next]", "[previous]", etc., at the start or 
end of a chapter.  Writes directly into the 'OutputLine'.
*/ 

char* NavigationButtons (boolean AtEnd)

{
   static char  Scratch [1024];

   register char  *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NavigationButtons()\n");

   sptr = Scratch;

   if (AtEnd)
   {
      strcpy (sptr, "\n<P>\n<HR>\n");
      while (*sptr) sptr++;
   }

   if (SectionNumber[0] < TotalChapters)
   {
      sprintf (sptr, "[<A HREF=\"%s_%02.02d%02.02d.HTML\"%s>next</A>]",
               HtmlName, SectionNumber[0]+1, 0, FrameTargetSelf);     
      while (*sptr) sptr++;
   }
   else
   {
      strcpy (sptr, "[next]");
      sptr += 6;
   }

   if (SectionNumber[0] > 1)
   {
      sprintf (sptr, " [<A HREF=\"%s_%02.02d%02.02d.HTML\"%s>previous</A>]",
               HtmlName, SectionNumber[0]-1, 0, FrameTargetSelf);
      while (*sptr) sptr++;
   }
   else
   {
      strcpy (sptr, " [previous]");
      sptr += 11;
   }

   sprintf (sptr, " [<A HREF=\"%s_%02.02d%02.02d.HTML\"%s>contents</A>]",
            HtmlName, 0, TableOfContentsCount, FrameTargetTop);
   while (*sptr) sptr++;

   if (DoFramed)
   {
      sprintf (sptr, " [<A HREF=\"%s_%02.02d%02.02d.HTML\"%s>full-page</A>]",
               HtmlName, SectionNumber[0], 0, FrameTargetTop);
      while (*sptr) sptr++;
   }

   if (AtEnd)
   {
      *sptr++ = '\n';
      *sptr = '\0';
   }
   else
   {
      strcpy (sptr, "\n<HR>\n");
      while (*sptr) sptr++;
   }

   return (Scratch);
}

/*****************************************************************************/
/*
Create an entry in the linked-list of cross-reference data.

'Symbol' (usually from SDML tag parameter information but will create a unique 
symbol if requied) is used as an index into the list (i.e. can be searched 
on). 

'Number' is the chapter/heading number (e.g. "1.2.3 - Section Numbering").  If 
supplied the above style of reference text heading is generated in the anchor, 
if not then the anchor would be unnumbered (e.g. "Section Numbering").

'Title' is the text appearing in the HTML document anchor (e.g. "Section 
Numbering").
*/ 

NoteReference
(
struct HtmlFileData *HtmlFilePtr,
register char *Symbol,
register char *Number,
register char *Title
)
{
   register struct ReferenceData  *rptr;
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "NoteReference() |%s|%s|\n", Symbol, Title);

   if ((rptr = malloc (sizeof (struct ReferenceData))) == NULL)
   {
      status = vaxc$errno;
      fprintf (stdout, "%%%s-E-MALLOC, allocating reference memory\n-%s\n",
               Utility, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }
   if (RefPtr == NULL)
      RefPtr = LastPtr = rptr;
   else
   {
      LastPtr->NextPtr = rptr;
      LastPtr = rptr;
   }
   rptr->ReferenceNumber = ReferenceNumber;
   /* ensure the title does not overwrite the size of the buffer */
   if (Symbol[0])
      Symbol[SymbolSize-1] = '\0';
   else
      sprintf (Symbol, "symbol_%d", ReferenceNumber);
   strcpy (rptr->Symbol, Symbol);
   /* ensure the number does not overwrite the size of the buffer */
   strncpy (rptr->Number, Number, sizeof(rptr->Number)-1);
   rptr->Number[sizeof(rptr->Number)-1] = '\0';
   /* ensure the title does not overwrite the size of the buffer */
   strncpy (rptr->Title, Title, sizeof(rptr->Title)-1);
   rptr->Title[sizeof(rptr->Title)-1] = '\0';
   strcpy (rptr->NameOfFile, HtmlFilePtr->NameOfFile);
   rptr->NextPtr = NULL;
}

/*****************************************************************************/
/*
Look for an entry ('Symbol') in the linked-list of cross-reference data.  If 
found return a pointer to it, if not return NULL.
*/ 

struct ReferenceData*  FindReference  (register char *Symbol)

{
   register int  Count = 0;
   register struct ReferenceData  *rptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FindReference() |%s|\n", Symbol);

   rptr = RefPtr;
   while (rptr != NULL)
   {
      Count++;
      if (toupper(Symbol[0]) == toupper(rptr->Symbol[0]))
         if (strsame (Symbol, rptr->Symbol, -1))
            break;
      rptr = rptr->NextPtr;
   }
   return (rptr);
}

/*****************************************************************************/
/*
Create an anchor in the HTML text.  Locate 'Symbol' in the linked-list of 
reference data and use the file name, reference number and title for the 
reference.
*/ 

char* MakeReference
(
struct HtmlFileData *HtmlFilePtr,
struct ErrorData *ErrorInfoPtr,
register char *htptr,
int *CapacityPtr,
register char *Symbol
)
{
   register struct ReferenceData  *rptr;

   char  Scratch [BufferSize];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MakeReference() |%s|\n", Symbol);

   if ((rptr = FindReference (Symbol)) == NULL) return (htptr);

   if (rptr->Number[0])
      sprintf (Scratch, "<A HREF=\"%s#%d\"%s>%s - %s</A>",
               rptr->NameOfFile, rptr->ReferenceNumber, FrameTargetSelf,
               rptr->Number, rptr->Title);
   else
      sprintf (Scratch, "<A HREF=\"%s#%d\"%s>%s</A>",
               rptr->NameOfFile, rptr->ReferenceNumber, FrameTargetSelf,
               rptr->Title);

   ErrorInfoPtr->SourceLineNumber = __LINE__;
   htptr = StringCopy (ErrorInfoPtr, htptr, CapacityPtr, Scratch, -1);

   return (htptr);
}

/*****************************************************************************/
/*
Begin the "table of contents" HTML file.
*/ 

int BeginContents (struct SdmlFileData *SdmlFilePtr)

{
   register char  *sptr;
   int  status;
   char  Buffer [BufferSize];

   /*********/
   /* begin */
   /*********/

   strcpy (ChapterTitle, "Table of Contents");
   OpenContentsFile (SdmlFilePtr);

   if (DoFramed)
   {
      sprintf (Buffer,
"%s\
<HTML>\n\
<HEAD>\n\
%s\
<TITLE>%s</TITLE>\n\
</HEAD>\n\
%s\
<H2>%s</H2>\n\
<FONT SIZE=-1>\n\
<P>\n",
      HtmlDocType, MetaInformation(SdmlFilePtr),
      UnmarkupTitle(ChapterTitle), HtmlBodyTag, ChapterTitle);
   }
   else
   {
      sprintf (Buffer,
"%s\
<HTML>\n\
<HEAD>\n\
%s\
<TITLE>%s</TITLE>\n\
</HEAD>\n\
%s\
<H1>%s</H1>\n\
<H2>%s</H2>\n\
<P>\n",
      HtmlDocType, MetaInformation(SdmlFilePtr),
      UnmarkupTitle(ChapterTitle), HtmlBodyTag, DocumentTitle, ChapterTitle);
   }

   WriteToRab (&ContentsFileRab, Buffer);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Add an item to the "table of contents" HTML file.  The 'Level' parameter 
indicates whether this is a level 1, level 2, etc., heading.  The item becomes 
an anchor pointing to a chapter HTML file, or into a heading within a chapter 
section.  All "<Hn>" HTML tags increase this count by one.
*/ 

WriteContentsItem
(
struct HtmlFileData *HtmlFilePtr,
int Level,
char *Symbol
)
{
   register int  Count;
   register char  *sptr;
   register struct ReferenceData  *rptr;
   char  Fragment [32],
         String [BufferSize];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "WriteContentsItem() |%d|%s|\n", Level, Symbol);

   if (Pass == 1) return;

   Level++;
   sptr = String;

   while (TableOfContentsLevel < Level)
   {
      /********************************/
      /* heading number has increased */
      /********************************/

      /* nest the HTML text to the right according to the heading level */
      Count = TableOfContentsLevel++ * 4;
      while (Count--) *sptr++ = ' ';
      /* table of contents are implemented as unsigned lists for each level */
      strcpy (sptr, "<DL>\n");
      sptr += 5;
   }

   while (TableOfContentsLevel > Level)
   {
      /********************************/
      /* heading number has decreased */
      /********************************/

      /* nest the HTML text to the right according to the heading level */
      Count = --TableOfContentsLevel * 4;
      while (Count--) *sptr++ = ' ';
      /* end of unsigned list for this level */
      strcpy (sptr, "</DL>\n");
      sptr += 6;
   }

   /******************/
   /* contents entry */
   /******************/

   /* nest the HTML text to the right according to the heading level */
   Count = Level * 4;
   while (Count--) *sptr++ = ' ';

   strcpy (sptr, "<DT>");
   sptr += 4;

   if ((rptr = FindReference (Symbol)) == NULL)
      strcpy (sptr, "ERROR: REFERENCE NOT FOUND");
   else
   {
      if (Level > 1)
         sprintf (Fragment, "#%d", rptr->ReferenceNumber);
      else
         Fragment[0] = '\0';

      if (rptr->Number[0])
         sprintf (sptr, "<NOBR>%s - <A HREF=\"%s%s\"%s>%s</A></NOBR>\n",
                  rptr->Number, HtmlFilePtr->NameOfFile, Fragment,
                  FrameTargetPage, rptr->Title);
      else
         sprintf (sptr, "<NOBR><A HREF=\"%s%s\"%s><NOBR>%s</A></NOBR>\n",
                  HtmlFilePtr->NameOfFile, Fragment,
                  FrameTargetPage, rptr->Title);
   }

   WriteToRab (&ContentsFileRab, String);
}

/*****************************************************************************/
/*
End the "table of contents" HTML file.
*/ 

EndContents ()

{
   register int  Count;
   register char  *sptr;
   char  String [BufferSize];

   /*********/
   /* begin */
   /*********/

   sptr = String;

   while (TableOfContentsLevel)
   {
      /* nest the HTML text to the right according to the heading level */
      Count = --TableOfContentsLevel * 4;
      while (Count--) *sptr++ = ' ';
      strcpy (sptr, "</DL>\n");
      sptr += 6;
   }

   if (DoFramed)
   {
      sprintf (sptr, "</FONT>\n");
      sptr += 8;
   }

   sprintf (sptr,
"<P>\n\
<FONT SIZE=-1>This HTML document was produced from SDML (DEC DOCUMENT source) \
by the %s utility (%s)</FONT>\n\
</BODY>\n\
</HTML>",
   Utility, SoftwareID);

   WriteToRab (&ContentsFileRab, String);

   sys$close (&ContentsFileFab, 0, 0);
}

/*****************************************************************************/
/*
Open the "table of contents" HTML file.
*/ 

int OpenContentsFile (struct SdmlFileData *SdmlFilePtr)

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "OpenContentsFile()");

   if (!DoGenerateHtml || Pass == 1) return (SS$_NORMAL);

   if (DoFramed)
      TableOfContentsCount = ++FrontMatterCount;
   else
      TableOfContentsCount = FrontMatterCount;

   sprintf (ContentsFileName, "%s%s_00%02.02d.HTML",
            HtmlDirectory, HtmlName, TableOfContentsCount);

   if (DoVerbose) fprintf (stdout, "[Created %s]\n", ContentsFileName);

   ContentsFileFab = cc$rms_fab;
   ContentsFileFab.fab$b_fac = FAB$M_PUT;
   ContentsFileFab.fab$l_fna = ContentsFileName;  
   ContentsFileFab.fab$b_fns = strlen(ContentsFileName);
   /* deferred write performance option */
   ContentsFileFab.fab$l_fop = FAB$M_DFW | FAB$M_SQO;
   ContentsFileFab.fab$b_rat = FAB$M_CR;
   ContentsFileFab.fab$b_rfm = FAB$C_STMLF;

   if (VMSnok (status = sys$create (&ContentsFileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$create %%X%08.08X\n", status);
      fprintf (stdout, "%%%s-E-TOCFILE, opening %s\n-%s\n",
               Utility, SdmlFilePtr->FileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   ContentsFileRab = cc$rms_rab;
   ContentsFileRab.rab$l_fab = &ContentsFileFab;
   /* 2 buffers, write behind performance option */
   ContentsFileRab.rab$b_mbf = 2;
   ContentsFileRab.rab$l_rop = RAB$M_WBH;

   if (VMSnok (status = sys$connect (&ContentsFileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect %%X%08.08X\n", status);
      sys$close (&ContentsFileFab, 0, 0);
      fprintf (stdout, "%%%s-E-TOCFILE, opening %s\n-%s\n",
               Utility, SdmlFilePtr->FileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   return (status);
}

/*****************************************************************************/
/*
*/ 

int WriteFrameFile (struct SdmlFileData *SdmlFilePtr)

{
   int  status,
        BufferLength;
   char  Buffer [2048],
         FrameFileName [256];
   struct FAB  FramedFileFab;
   struct RAB  FramedFileRab;


   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "WriteFramedFile()");

   if (!DoGenerateHtml || Pass == 1) return (SS$_NORMAL);

   BufferLength = sprintf (Buffer,
"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2//EN\">\n\
<HTML>\n\
<HEAD>\n\
%s\
<TITLE>%s</TITLE>\n\
</HEAD>\n\
<FRAMESET BORDER=5 ROWS=\"100%%\" COLS=\"35%%,65%%\">\n\
   <FRAME RESIZE NAME=\"frame_toc\" SRC=\"%s_00%02.02d.html\">\n\
   <FRAME RESIZE NAME=\"frame_page\" SRC=\"%s_00%02.02d.html\">\n\
</FRAMESET>\n\
</HTML>\n",
      MetaInformation(SdmlFilePtr),
      UnmarkupTitle(DocumentTitle),
      HtmlName, TableOfContentsCount,
      HtmlName, TitlePageCount);

   sprintf (FrameFileName, "%s%s_0000.HTML", HtmlDirectory, HtmlName);

   if (DoVerbose) fprintf (stdout, "[Created %s]\n", FrameFileName);

   FramedFileFab = cc$rms_fab;
   FramedFileFab.fab$b_fac = FAB$M_PUT;
   FramedFileFab.fab$l_fna = FrameFileName;  
   FramedFileFab.fab$b_fns = strlen(FrameFileName);
   /* deferred write performance option */
   FramedFileFab.fab$l_fop = FAB$M_DFW | FAB$M_SQO;
   FramedFileFab.fab$b_rat = FAB$M_CR;
   FramedFileFab.fab$b_rfm = FAB$C_STMLF;

   if (VMSnok (status = sys$create (&FramedFileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$create %%X%08.08X\n", status);
      fprintf (stdout, "%%%s-E-FRAMEDFILE, creating %s\n-%s\n",
               Utility, SdmlFilePtr->FileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   FramedFileRab = cc$rms_rab;
   FramedFileRab.rab$l_fab = &FramedFileFab;
   /* 2 buffers, write behind performance option */
   FramedFileRab.rab$b_mbf = 2;
   FramedFileRab.rab$l_rop = RAB$M_WBH;

   if (VMSnok (status = sys$connect (&FramedFileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect %%X%08.08X\n", status);
      sys$close (&FramedFileFab, 0, 0);
      fprintf (stdout, "%%%s-E-FRAMEDFILE, opening %s\n-%s\n",
               Utility, SdmlFilePtr->FileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   FramedFileRab.rab$w_rsz = BufferLength;
   FramedFileRab.rab$l_rbf = Buffer;
   if (VMSnok (status = sys$put (&FramedFileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$put %%X%08.08X\n", status);
      fprintf (stdout, "%%%s-E-FRAMEDFILE, writing %s\n-%s\n",
               Utility, SdmlFilePtr->FileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   status = sys$close (&FramedFileFab, 0, 0);

   return (status);
}

/*****************************************************************************/
/*
*/ 

int WriteIndexPage (struct SdmlFileData *SdmlFilePtr)

{
   int  status,
        BufferLength;
   char  Buffer [2048],
         IndexPageFileName [256],
         ZeroInLine [256],
         ZeroFileName [256],
         ZeroExpandedFileName [256];
   struct FAB  IndexPageFileFab;
   struct RAB  IndexPageFileRab;
   struct FAB  ZeroFileFab;
   struct RAB  ZeroFileRab;
   struct NAM  ZeroFileNam;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "WriteIndexPage()");

   if (!DoGenerateHtml || Pass == 1) return (SS$_NORMAL);

   sprintf (ZeroFileName, "%s%s_0000.HTML", HtmlDirectory, HtmlName);

   sprintf (IndexPageFileName, "%s%s", HtmlDirectory, IndexPageName);

   ZeroFileFab = cc$rms_fab;
   ZeroFileFab.fab$b_fac = FAB$M_GET;
   ZeroFileFab.fab$l_fna = ZeroFileName;  
   ZeroFileFab.fab$b_fns = strlen(ZeroFileName);
   ZeroFileFab.fab$b_shr = FAB$M_SHRGET;
   ZeroFileFab.fab$l_fop = FAB$M_NAM;
   ZeroFileFab.fab$l_nam = &ZeroFileNam;
   ZeroFileNam = cc$rms_nam;
   ZeroFileNam.nam$l_esa = ZeroExpandedFileName;
   ZeroFileNam.nam$b_ess = sizeof(ZeroExpandedFileName)-1;

   if (VMSnok (status = sys$open (&ZeroFileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      fprintf (stdout, "%%%s-E-INDEXPAGE, opening %s\n-%s\n",
               Utility, ZeroFileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   ZeroFileRab = cc$rms_rab;
   ZeroFileRab.rab$l_fab = &ZeroFileFab;
   /* 2 buffers and read ahead performance option */
   ZeroFileRab.rab$b_mbf = 2;
   ZeroFileRab.rab$l_rop = RAB$M_RAH;

   if (VMSnok (status = sys$connect (&ZeroFileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&ZeroFileFab, 0, 0);
      fprintf (stdout, "%%%s-E-INDEXPAGE, opening %s\n-%s\n",
               Utility, ZeroFileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   if (DoVerbose) fprintf (stdout, "[Created %s]\n", IndexPageFileName);

   IndexPageFileFab = cc$rms_fab;
   IndexPageFileFab.fab$b_fac = FAB$M_PUT;
   IndexPageFileFab.fab$l_fna = IndexPageFileName;  
   IndexPageFileFab.fab$b_fns = strlen(IndexPageFileName);
   /* deferred write performance option */
   IndexPageFileFab.fab$l_fop = FAB$M_DFW | FAB$M_SQO;
   IndexPageFileFab.fab$b_rat = FAB$M_CR;
   IndexPageFileFab.fab$b_rfm = FAB$C_STMLF;

   if (VMSnok (status = sys$create (&IndexPageFileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$create %%X%08.08X\n", status);
      fprintf (stdout, "%%%s-E-INDEXPAGE, creating %s\n-%s\n",
               Utility, IndexPageFileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   IndexPageFileRab = cc$rms_rab;
   IndexPageFileRab.rab$l_fab = &IndexPageFileFab;
   /* 2 buffers, write behind performance option */
   IndexPageFileRab.rab$b_mbf = 2;
   IndexPageFileRab.rab$l_rop = RAB$M_WBH;

   if (VMSnok (status = sys$connect (&IndexPageFileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect %%X%08.08X\n", status);
      sys$close (&IndexPageFileFab, 0, 0);
      fprintf (stdout, "%%%s-E-INDEXPAGE, creating %s\n-%s\n",
               Utility, IndexPageFileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   ZeroFileRab.rab$w_usz = sizeof(ZeroInLine)-1;
   ZeroFileRab.rab$l_ubf = ZeroInLine;

   while (VMSok (status = sys$get (&ZeroFileRab, 0, 0)))
   {
      ZeroInLine[ZeroFileRab.rab$w_rsz] = '\0';
      if (Debug) fprintf (stdout, "|%s|\n", ZeroInLine);
      IndexPageFileRab.rab$w_rsz = ZeroFileRab.rab$w_rsz;
      IndexPageFileRab.rab$l_rbf = ZeroInLine;

      if (VMSnok (status = sys$put (&IndexPageFileRab, 0, 0)))
      {
         if (Debug) fprintf (stdout, "sys$put %%X%08.08X\n", status);
         fprintf (stdout, "%%%s-E-INDEXPAGE, writing %s\n-%s\n",
                  Utility, ZeroFileName, SysGetMsg(status)+1);
         exit (status | STS$M_INHIB_MSG);
      }
   }

   if (status == RMS$_EOF) status = SS$_NORMAL;
   if (VMSnok (status))
   {
      if (Debug) fprintf (stdout, "%%%08.08X\n", status);
      fprintf (stdout, "%%%s-E-INDEXPAGE, reading %s\n-%s\n",
               Utility, ZeroFileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   status = sys$close (&ZeroFileFab, 0, 0);

   status = sys$close (&IndexPageFileFab, 0, 0);

   return (status);
}

/*****************************************************************************/
/*
The file name comprises the RMS file 'name' component plus a two digit section
number (the 'chapter' or "<H1>" number),  plus a two digit subsection number.
*/ 

GenerateHtmlFileName
(
struct HtmlFileData *HtmlFilePtr,
int SectionCount,
int SubSectionCount
)
{
   register char  *cptr;

   sprintf (HtmlFilePtr->NameOfFile, "%s_%02.02d%02.02d.HTML",
            HtmlName, SectionCount, SubSectionCount);
   sprintf (HtmlFilePtr->FileName, "%s%s",
            HtmlDirectory, HtmlFilePtr->NameOfFile);
   for (cptr = HtmlFilePtr->NameOfFile; *cptr; *cptr++) *cptr = tolower(*cptr);
   if (Debug)
      fprintf (stdout, "GenerateHtmlFileName |%s|%s|\n",
               HtmlFilePtr->NameOfFile, HtmlFilePtr->FileName);
}

/*****************************************************************************/
/*
Meta tags providing utility name, source files and date.
*/

char *MetaInformation (struct SdmlFileData *SdmlFilePtr)

{
   static $DESCRIPTOR (MetaFaoDsc,
"<META NAME=\"generator\" CONTENT=\"!AZ\">\n\
<META NAME=\"source\" CONTENT=\"!AZ\">\n\
<META NAME=\"date\" CONTENT=\"!AZ\">\n\
<!!--\n\
!AZ\
-->\n");

   static char  String [BufferSize];

   register char  *cptr;
   int  status;
   short  Length;
   unsigned long  UnixTime;
   char  UnixDateTime [32];
   struct tm  *UnixTmPtr;
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "MetaInformation() |%s|\n",
               SdmlFilePtr->ExpandedFileName);

   time (&UnixTime);
   UnixTmPtr = localtime (&UnixTime);
   if (!strftime (UnixDateTime, sizeof(UnixDateTime),
                  "%a, %d %b %Y %T", UnixTmPtr))
      strcpy (UnixDateTime, "[error]");
   if (Debug) fprintf (stdout, "UnixDateTime |%s|\n", UnixDateTime);

   if (VMSnok (status =
       sys$fao (&MetaFaoDsc, &Length, &StringDsc,
                SoftwareID,
                SdmlFilePtr->ExpandedFileName,
                UnixDateTime,
                CopyRightMessageBrief)))
   {
      fprintf (stdout, "%%%s-E-METAINFO, generating META information\n-%s\n",
               Utility, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   String[Length] = '\0';

   return (String);
}

/*****************************************************************************/
/*
Open an HTML output file.
*/ 

int CreateHtmlFile
(
struct HtmlFileData *HtmlFilePtr,
struct SdmlFileData *SdmlFilePtr
)
{
   int  status;
   struct NAM  HtmlFileNAM;
   char  ExpandedFileName [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CreateHtmlFile()\n");

   if (!DoGenerateHtml || Pass == 1) return (SS$_NORMAL);

   HtmlFilePtr->FileFab = cc$rms_fab;
   HtmlFilePtr->FileFab.fab$b_fac = FAB$M_PUT;
   HtmlFilePtr->FileFab.fab$l_fna = HtmlFilePtr->FileName;  
   HtmlFilePtr->FileFab.fab$b_fns = strlen(HtmlFilePtr->FileName);
   HtmlFilePtr->FileFab.fab$l_fop = FAB$M_NAM;
   HtmlFilePtr->FileFab.fab$l_nam = &HtmlFileNAM;
   /* deferred write performance option */
   HtmlFilePtr->FileFab.fab$l_fop = FAB$M_DFW | FAB$M_SQO;
   HtmlFilePtr->FileFab.fab$b_rat = FAB$M_CR;
   HtmlFilePtr->FileFab.fab$b_rfm = FAB$C_STMLF;

   HtmlFileNAM = cc$rms_nam;
   HtmlFileNAM.nam$l_esa = ExpandedFileName;
   HtmlFileNAM.nam$b_ess = sizeof(ExpandedFileName)-1;

   if (VMSnok (status = sys$create (&HtmlFilePtr->FileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$create %%X%08.08X\n", status);
      fprintf (stdout, "%%%s-E-HTMLFILE, creating %s\n-%s\n",
               Utility, HtmlFilePtr->FileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   HtmlFilePtr->FileRab = cc$rms_rab;
   HtmlFilePtr->FileRab.rab$l_fab = &HtmlFilePtr->FileFab;
   /* 2 buffers, write behind performance option */
   HtmlFilePtr->FileRab.rab$b_mbf = 2;
   HtmlFilePtr->FileRab.rab$l_rop = RAB$M_WBH;

   if (VMSnok (status = sys$connect (&HtmlFilePtr->FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect %%X%08.08X\n", status);
      sys$close (&HtmlFilePtr->FileFab, 0, 0);
      fprintf (stdout, "%%%s-E-HTMLFILE, creating %s\n-%s\n",
               Utility, HtmlFilePtr->FileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }

   if (DoVerbose) fprintf (stdout, "[Created %s]\n", HtmlFilePtr->FileName);
   return (status);
}

/*****************************************************************************/
/*
*/ 
 
CloseHtmlFile (struct HtmlFileData *HtmlFilePtr)

{
   if (Debug) fprintf (stdout, "CloseHtmlFile()\n");

   if ((unsigned short)HtmlFilePtr->FileFab.fab$w_ifi)
   {
      FlushOutputLine (HtmlFilePtr);
      sys$close (&HtmlFilePtr->FileFab, 0, 0);
      HtmlFilePtr->FileName[0] = '\0';
   }
}

/*****************************************************************************/
/*
*/
 
FlushOutputLine (struct HtmlFileData *HtmlFilePtr)

{
   if (Debug) fprintf (stdout, "FlushOutputLine()\n");

   if (OutputLine[0]) WriteHtml (HtmlFilePtr, OutputLine);
   *(OutputLinePtr = OutputLine) = '\0';
}

/*****************************************************************************/
/*
Write a string (one of more records/lines) to an HTML output file.
*/
 
int WriteHtml
(
struct HtmlFileData *HtmlFilePtr,
register char *rptr
)
{
   if (Debug) fprintf (stdout, "WriteHtml()\n");

   /* if a file (title page, chapter, etc.) is not open just ignore */
   if (!(unsigned short)HtmlFilePtr->FileFab.fab$w_ifi) return (SS$_NORMAL);

   return (WriteToRab (&HtmlFilePtr->FileRab, rptr));
}

/*****************************************************************************/
/*
Write a null-terminated string to the file associated with the RAB parameter.
Embedded new-line characters ('\n') terminate a line and generate a single
record.  The string following the last new-line character generates a record
regardless, therefore a trailing new-line generates a blank record.
*/
 
int WriteToRab
(
struct RAB *RabPtr,
register char *rptr
)
{
   register int  status;
   register char  *cptr;

   if (Debug) fprintf (stdout, "WriteToRab()\n");

   if (!DoGenerateHtml || Pass == 1)
   {
      if (Debug) fprintf (stdout, "|%s|\n", rptr);
      return (SS$_NORMAL);
   }

   while (*rptr)
   {
      cptr = rptr;
      while (*rptr && *rptr != '\n') rptr++;
      if (Debug) fprintf (stdout, "|%*s|\n", rptr-cptr, cptr);
      RabPtr->rab$w_rsz = rptr - cptr;
      RabPtr->rab$l_rbf = cptr;
      if (VMSnok (status = sys$put (RabPtr, 0, 0)))
         exit (status);
      if (*rptr) rptr++;
   }
   return (status);
}

/*****************************************************************************/
/*
Read a record (line) from the source SDML file.  Returns pointer to buffer if 
not end-of-file, NULL if end-of-file (a'la fgets()).
*/ 

char* ReadSdmlLine (struct SdmlFileData *SdmlFilePtr)

{
   register int  status;

   SdmlFilePtr->InLine[0] = '\0';
   if (VMSnok (status = sys$get (&SdmlFilePtr->FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "ReadSdmlLine() %%%08.08X\n", status);
      if (status == RMS$_EOF) return (NULL);
      fprintf (stdout, "%%%s-E-READSDML, reading %s\n-%s\n",
               Utility, SdmlFilePtr->FileName, SysGetMsg(status)+1);
      exit (status | STS$M_INHIB_MSG);
   }
   SdmlFilePtr->InLine[SdmlFilePtr->FileRab.rab$w_rsz] = '\0';
   SdmlFilePtr->LineNumber++;
   if (Debug) fprintf (stdout, "ReadSdmlLine()\n|%s|\n", SdmlFilePtr->InLine);
   return (SdmlFilePtr->InLine);
}

/*****************************************************************************/
/*
Uses static storage ... careful!  Assume all document '<' and '>' are entity-
escaped and any here only represent markup!
*/ 

char* UnmarkupTitle (char *String)

{
   static char  OutString [256];
   register char  *cptr, *sptr;

   sptr = OutString;
   cptr = String;
   while (*cptr)
   {
      if (*cptr == '<')
      {
         while (*cptr && *cptr != '>') cptr++;
         if (*cptr) cptr++;
      }
      else
         *sptr++ = *cptr++;
   }
   *sptr = '\0';
   return (OutString);
}

/*****************************************************************************/
/*
Copy one string to another, with capacity checking.  If parameter 'ccnt' is 
less than 0 then all characters are copied, if 1 then one character copied, 
etc.
*/ 

char* StringCopy
( 
struct ErrorData *ErrorInfoPtr,
register char *sptr,
int *CapacityPtr,
register char *tptr,
register int ccnt
)
{
    int  Capacity;

    Capacity = *CapacityPtr;

    while (ccnt-- && *tptr)
    {
       if (isprint(*tptr) || *tptr == '\n')
       {
          *sptr++ = *tptr++;
          if (!Capacity--) BufferOverflow (ErrorInfoPtr);
       }
       else
          tptr++;
    }
    *sptr = '\0';
    *CapacityPtr = Capacity;

    return (sptr);
}

/*****************************************************************************/
/*
Copy text from one string to another, with capacity checking, converting 
characters forbidden to appear as plain-text in HTML.  For example the '<', 
'&', etc.  Convert these to the corresponding HTML character entities.  If 
parameter 'ccnt' is less than 0 then all characters are copied, if 1 then one 
character copied, etc.
*/ 

char* HtmlCopy
( 
struct ErrorData *ErrorInfoPtr,
register char *sptr,
int *CapacityPtr,
register char *tptr,
register int ccnt
)
{
    int  Capacity;

    Capacity = *CapacityPtr;

    while (ccnt-- && *tptr)
    {
       switch (*tptr)
       {
          case '<' :
             if ((Capacity -= 4) < 0) BufferOverflow (ErrorInfoPtr);
             strcpy (sptr, "&lt;");
             sptr += 4;
             tptr++;
             break;
          case '>' :
             if ((Capacity -= 4) < 0) BufferOverflow (ErrorInfoPtr);
             strcpy (sptr, "&gt;");
             sptr += 4;
             tptr++;
             break;
          case '&' :
             if ((Capacity -= 5) < 0) BufferOverflow (ErrorInfoPtr);
             strcpy (sptr, "&amp;");
             sptr += 5;
             tptr++;
             break;
          default :
             if (isprint(*tptr) || *tptr == '\t' || *tptr == '\n')
             {
                *sptr++ = *tptr++;
                if (!Capacity--) BufferOverflow (ErrorInfoPtr);
             }
             else
                tptr++;
       }
    }
    *sptr = '\0';
    *CapacityPtr = Capacity;
    return (sptr);
}

/*****************************************************************************/
/*
Report buffer overflow error.
*/
 
BufferOverflow (struct ErrorData *ErrorInfoPtr) 

{
   if (ErrorInfoPtr == NULL)
      fprintf (stdout, "%%%s-E-BUFOVF, buffer overflow\n", Utility);
   else
   {
      if (IncludeSourceLineNumber)
      {
         fprintf (stdout,
         "%%%s-E-BUFOVF, buffer overflow (%s line %d)\n",
         Utility, SoftwareID, ErrorInfoPtr->SourceLineNumber);
      }
      else
         fprintf (stdout, "%%%s-E-BUFOVF, buffer overflow\n", Utility);

      fprintf (stdout,
      "-%s-I-TAG, \"%s\"\n-%s-I-LOCN, line %d of %s\n",
      Utility, ErrorInfoPtr->TagNamePtr,
      Utility, ErrorInfoPtr->TagLineNumber, ErrorInfoPtr->SdmlFileNamePtr);
   }
   exit (STS$K_ERROR | STS$M_INHIB_MSG);
}

/*****************************************************************************/
/*
*/
 
char* SysGetMsg (int StatusValue)
 
{
   static char  Message [256];
   short int  Length;
   $DESCRIPTOR (MessageDsc, Message);
 
   sys$getmsg (StatusValue, &Length, &MessageDsc, 0, 0);
   Message[Length] = '\0';
   if (Debug) fprintf (stdout, "SysGetMsg() |%s|\n", Message);
   return (Message);
}
 
/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 
 
boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Get the 
entire command line following the verb that activated the image.  The command 
line is returned in uppercase, space compressed (i.e. maximum of one space 
between text elements, trimmed of leading and trailing spaces).  Returns a 
warning status if there were no parameters/qualifiers on the command line.
The variable CommandLine is global.
*/ 
 
int ParseCommandLine ()
 
{
   int  status;
   unsigned short  Length;
   unsigned long  Flags = 0;
   struct dsc$descriptor_s 
          CommandLineDsc = { sizeof(CommandLine)-1, DSC$K_DTYPE_T,
                             DSC$K_CLASS_S, CommandLine };
 
   /* get the entire command line following the verb */
   if (VMSnok (status = lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
      return (status);
   CommandLine[Length] = '\0';
 
   if (ParseCommand (CommandLine))
      return (SS$_NORMAL);
   else
      return (STS$K_ERROR | STS$M_INHIB_MSG);
}
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Quoted 
strings are always indicated by being parsed to include a single leading 
quote.
*/ 
 
boolean ParseCommand (char *CommandLine)
 
{
   register int  QuoteCount = 0;
   register char  *cptr, *eptr;
   boolean  CommandLineOK = true;
   char  Entity [256] = "";
 
   /* set up any argument defaults */
   ParseCommandEntity (NULL);
 
   cptr = CommandLine;
   eptr = Entity;
 
   for (;;)
   {
      if (*cptr == '\"')
      {
         QuoteCount++;
         *eptr++ = *cptr++;
         continue;
      }
 
      if (QuoteCount & 1 && *cptr)
      {
         /* inside quoted text, copy all characters as literals */
         *eptr++ = *cptr++;
         continue;
      }
 
      if (*cptr == '/' || isspace (*cptr) || !*cptr)
      {
         if (isspace (*cptr))
         {
            /* span the white space */
            while (*cptr && isspace (*cptr)) cptr++;
            if (*cptr == '=')
            {
               /* part of a qualifier, continue to get the value */
               *eptr++ = *cptr++;
               /* span any intervening white space */
               while (*cptr && isspace (*cptr)) cptr++;
               continue;
            }
         }
 
         if (Entity[0])
         {
            *eptr = '\0';
            if (!ParseCommandEntity (Entity)) CommandLineOK = false;
         }
 
         /* if end of command line then break from loop */
         if (!*cptr) break;
 
         /* start of new entity */
         eptr = Entity;
         /* if start of qualifier ensure slash is copied */
         if (*cptr == '/') *eptr++ = *cptr++;
 
         continue;
      }
 
      /* any other character, just copy, ensure upper case */
      *eptr++ = toupper(*cptr++);
   }
 
   return (CommandLineOK);
}
 
/*****************************************************************************/
/*
Get a string value from a qualifier, e.g. '/EXAMPLE=TEST'.
*/
 
boolean ParseCommandString
(
char *Entity,
char *String,
boolean Qualifier,
boolean ReportErrors,
boolean EnsureUpperCase
)
{
   register int  QuoteCount = 0;
   register char  *eptr, *sptr;
 
   if (Debug) fprintf (stdout, "ParseCommandString()\nEntity: '%s'\n", Entity);
 
   eptr = Entity;
 
   if (Qualifier)
   {
      /* scan down to equate symbol */
      while (*eptr && *eptr != '=') eptr++;
      if (*eptr) eptr++;
      if (!*eptr)
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
            Utility, Entity+1);
         }
         return (false);
      }
   }
 
   sptr = String;
   while (*eptr)
   {
      if (*eptr == '\"')
      {
         if (QuoteCount & 1)
         {
            /* are inside quotes, check for escaped quotes ("") */
            if (*++eptr != '\"')
            {
               /* now outside quotes */
               QuoteCount++;
            }
            /* drop thru to character copy */
         }
         else
         {
            /* now inside quotes */
            QuoteCount++;
            eptr++;
            continue;
         }
      }
 
      if (EnsureUpperCase)
         *sptr++ = toupper(*eptr++);
      else
         *sptr++ = *eptr++;
   }
   *sptr = '\0';
 
   if (Debug) fprintf (stdout, "String: '%s'\n", String);
 
   return (true);
}
 
/*****************************************************************************/
/*
Get an integer value from a qualifier, e.g. '/EXAMPLE=99'.
*/
 
boolean ParseCommandInteger
(
char *Entity,
int *IntegerPtr,
int Base,
boolean ReportErrors
)
{
   register char  *eptr;
   char  *sptr;
 
   if (Debug)
      fprintf (stdout, "ParseCommandInteger() '%s' Base: %d\n", Entity, Base);
 
   for (eptr = Entity; *eptr && *eptr != '='; eptr++);
   if (*eptr) eptr++;
   if (*eptr)
   {
      *IntegerPtr = strtol (eptr, &sptr, Base);
      if (sptr > eptr && !*sptr)
         return (true);
      else
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-BADVALUE, '%s' is an invalid keyword value\n",
            Utility, eptr);
         }
         return (false);
      }
   }
   else
   {
      if (ReportErrors)
      {
         fprintf (stdout,
         "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
         Utility, Entity+1);
      }
      return (false);
   }
}
 
/*****************************************************************************/
/*
A single command line "entity" has been parsed, check if its recognised.  This 
function is the one modified for the individual requirements of each program.
*/
 
boolean ParseCommandEntity (char *Entity)
 
{
   if (Entity == NULL)
   {
      /* set up any argument defaults */
      Debug = FlagUnknownTags = IncludeComments =
      IncludeSourceLineNumber = IncludeUnknownTags = false;
      DoVerbose = DoGenerateHtml = true;
      HtmlDirectory[0] = SdmlFileSpec[0] = '\0';
      strcpy (HtmlBodyTag, "<BODY>\n");
      return (true);
   }
 
   if (Debug) fprintf (stdout, "ParseCommandEntity() Entity: '%s'\n", Entity);
 
   if (Entity[0] == '/')
   {
      if (strsame (Entity, "/COMMENTS", 4))
         return (IncludeComments = true);
      if (strsame (Entity, "/NOCOMMENTS", 6))
      {
         IncludeComments = false;
         return (true);
      }

      if (strsame (Entity, "/BODY=", 4))
      {
         strcpy (HtmlBodyTag, "<BODY ");
         if ((ParseCommandString (Entity, HtmlBodyTag+6,
                 true, true, false)))
         {
            strcat (HtmlBodyTag, ">\n");
            return (true);
         }
         else
            return (false);
      }

      /* turns on all "if (Debug)" statements */
      if (strsame (Entity, "/DBUG", -1))
         return (Debug = true);

      if (strsame (Entity, "/DIRECTORY=", 4))
         return (ParseCommandString (Entity, HtmlDirectory, true, true, true));

      if (strsame (Entity, "/FLAG_UNKNOWN_TAGS", 4))
         return (FlagUnknownTags = true);
      if (strsame (Entity, "/NOFLAG_UNKNOWN_TAGS", 6))
      {
         FlagUnknownTags = false;
         return (true);
      }

      if (strsame (Entity, "/FRAMED", 4))
         return (DoFramed = true);
      if (strsame (Entity, "/NOFRAMED", 6))
      {
         DoFramed = false;
         return (true);
      }

      if (strsame (Entity, "/HTML", 4))
         return (DoGenerateHtml = true);
      if (strsame (Entity, "/NOHTML", 6))
      {
         DoGenerateHtml = false;
         return (true);
      }

      if (strsame (Entity, "/INDEX=", 4))
      {
         IndexPageName[0] = '\0';
         ParseCommandString (Entity, IndexPageName, true, false, true);
         if (!IndexPageName[0])
            strcpy (IndexPageName, DefaultIndexPageName);
         return (true);
      }

      if (strsame (Entity, "/SOURCE", 4))
         return (IncludeSourceLineNumber = true);

      if (strsame (Entity, "/UNKNOWN_TAGS", 4))
         return (IncludeUnknownTags = true);
      if (strsame (Entity, "/NOUNKNOWN_TAGS", 6))
      {
         IncludeUnknownTags = false;
         return (true);
      }

      if (strsame (Entity, "/VERBOSE", 4))
         return (DoVerbose = true);
      if (strsame (Entity, "/NOVERBOSE", 6))
      {
         DoVerbose = false;
         return (true);
      }

      fprintf (stdout,
      "%%%s-E-IVQUAL, unrecognised qualifier\n \\%s\\\n", Utility, Entity+1);
      return (false);
   }
 
   if (!SdmlFileSpec[0])
      return (ParseCommandString (Entity, SdmlFileSpec, false, true, true));

   fprintf (stdout,
   "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n", Utility, Entity);
   return (false);
}
   
/*****************************************************************************/

