/*****************************************************************************/
/*
                                HTTPdMon.c

This utility displays information about a WASD HTTPd server.  The information 
is derived from JPI data about the process, and interpreted from four logicals
defined and refreshed by the HTTPd server. 

   HTTPDnn$COUNT1     and
   HTTPDnn$COUNT2     (binary) server accumulators for various events, etc. 
   HTTPDnn$PID        (binary) process ID of the HTTPd server
   HTTPDnn$REQUEST    summary of the most recently completed request

The /ALERT and /ALERT=ALL parameters ring the bell every time a new request is
reported (the connect count increases).  The /ALERT=HOST rings the bell
every time the host name in the request changes.  The /LOOKUP qualifier
(default) attempts to resolve a dotted-decimal host address in the request (if
the server's [DNSlookup] configuration directive is disabled).  This operation
can introduce some latency as name resolution occurs.  A trailing "#" indicates
this is an address that has been resolved this way.  If it cannot be resolved a
"?" is appended to the numeric address.


QUALIFIERS
----------
/ALERT[=ALL|HOST]       ring bell for all accesses or just a change in host
/DBUG                   turns on all "if (Debug)" statements
/HELP                   display brief usage information
/INTERVAL=              number of seconds between display refresh
/[NO]LOOKUP             resolve dotted-decimal host addresses to names
/PORT=                  the IP port number of the server to monitor
/REFRESH=               synonym for /INTERVAL


BUILD DETAILS
-------------
See HTTPDMON_BUILD.COM


VERSION HISTORY (update SoftwareID as well!)
---------------
27-AUG-98  MGD  v1.7.0, HTTPD v5.2, (no real changes)
21-JUN-98  MGD  v1.6.0, HTTPD v5.1, alert, and lookup host address to name
14-FEB-98  MGD  v1.5.0, HTTPd v5.0, DECnet, SSL and minor changes
15-OCT-97  MGD  v1.4.0, HTTPd v4.5, cache
21-AUG-97  MGD  v1.3.0, HTTPd v4.4, accounting structure > 255 bytes, duration
23-JUL-97  MGD  v1.2.1, HTTPd v4.3, (no real changes)
14-JUN-97  MGD  v1.2.0, HTTPd v4.2, CGIplus,
                        bugfix; arithmetic trap on AXP systems
01-FEB-97  MGD  v1.1.0, HTTPd v4.0, (no real changes)
01-DEC-95  MGD  v1.0.0, HTTPd v3.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "HTTPDMON v1.7.0 AXP";
#else
   char SoftwareID [] = "HTTPDMON v1.7.0 VAX";
#endif

char  ForHTTPdVersion [] = "(HTTPd v5.2)";

/* standard C header files */
#include <stdio.h>
#include <ctype.h>

/* VMS related header files */
#include <ssdef.h>
#include <stsdef.h>
#include <descrip.h>
#include <iodef.h>
#include <jpidef.h>
#include <libclidef.h>
#include <libdtdef.h>
#include <lnmdef.h>
#include <psldef.h>
#include <prvdef.h>

/* this header file contains the accounting structure definition */
#include "../httpd/wasd.h"

/* Internet-related header files */

/*
#define IP_NONE 1 
*/

#ifdef IP_NONE
#  undef IP_UCX
#  undef IP_NETLIB
#else
/* if not NETLIB then UCX! */
#  ifndef IP_NETLIB
#    undef IP_UCX
#    define IP_UCX 1
#  endif
#endif

#ifdef IP_UCX
#   include <socket.h>
#   include <in.h>
#   include <netdb.h>
#   include <inet.h>
/*#   include <ucx$inetdef.h>*/
#endif

#ifdef IP_NETLIB
#   include "netlibdef.h"
#endif

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

#define DEFAULT_INTERVAL_SECONDS 2

char  Utility [] = "HTTPDMON";

#define ControlZ '\x1a'

boolean  ControlY,Debug,
         DoAlertAll,
         DoAlertHost,
         DoLookupHost,
         DoShowHelp;

int  CurrentSubprocessCount,
     IntervalSeconds = DEFAULT_INTERVAL_SECONDS,
     PrevConnectCount,
     ServerPort = 80;

char  CommandLine [256],
      PrevHostName [256],
      ServerProcessName [16],
      Screen [8192];

char  *ScrPtr;

struct AccountingStruct  Accounting;

/* ANSI terminal control sequences */
char ANSIcls [] = "\x1b[0;0H\x1b[J",
     ANSIhome [] = "\x1b[0;0H",
     ANSIceol [] = "\x1b[K",
     ANSIceos [] = "\x1b[J",
     ANSInormal [] = "\x1b[0m",
     ANSIbold [] = "\x1b[1m",
     ANSIblink [] = "\x1b[5m",
     ANSIreverse [] = "\x1b[7m";

/* required prototypes */
ControlY_AST ();
char *LookupHostName (char*);
char* TimeString (); 
 
/*****************************************************************************/
/*
*/

int main ()

{
   int  status,
        MonitorStatus;

   /*********/
   /* begin */
   /*********/

#ifndef IP_NONE
   DoLookupHost = true;
#endif

   GetParameters ();

   if (DoShowHelp) exit (ShowHelp ());

#ifdef IP_NONE
   if (DoLookupHost)
   {
      fprintf (stdout, "%%%s-E-RESOLVE, host resolution not compiled in!\n",
               Utility);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }
#endif

   if (VMSnok (status = OnControlY (&ControlY_AST)))
      exit (status);

   MonitorStatus = MonitorHttpd ();

   if (VMSnok (status = OnControlY (0)))
      exit (status);

   exit (MonitorStatus);
}

/*****************************************************************************/
/*
Get "command-line" parameters, whether from the command-line or from a
configuration logical containing the equivalent.
*/

GetParameters ()

{
   static char  CommandLine [256];
   static unsigned long  Flags = 0;

   register char  *aptr, *cptr, *clptr, *sptr;

   int  status;
   unsigned short  Length;
   char  ch;
   $DESCRIPTOR (CommandLineDsc, CommandLine);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetParameters()\n");

   if ((clptr = getenv ("HTTPDMON$PARAM")) == NULL)
   {
      /* get the entire command line following the verb */
      if (VMSnok (status =
          lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
         exit (status);
      (clptr = CommandLine)[Length] = '\0';
   }

   aptr = NULL;
   ch = *clptr;
   for (;;)
   {
      if (aptr != NULL) *aptr = '\0';
      if (!ch) break;

      *clptr = ch;
      if (Debug) fprintf (stdout, "clptr |%s|\n", clptr);
      while (*clptr && isspace(*clptr)) *clptr++ = '\0';
      aptr = clptr;
      if (*clptr == '/') clptr++;
      while (*clptr && !isspace (*clptr) && *clptr != '/')
      {
         if (*clptr != '\"')
         {
            clptr++;
            continue;
         }
         cptr = clptr;
         clptr++;
         while (*clptr)
         {
            if (*clptr == '\"')
               if (*(clptr+1) == '\"')
                  clptr++;
               else
                  break;
            *cptr++ = *clptr++;
         }
         *cptr = '\0';
         if (*clptr) clptr++;
      }
      ch = *clptr;
      if (*clptr) *clptr = '\0';
      if (Debug) fprintf (stdout, "aptr |%s|\n", aptr);
      if (!*aptr) continue;

      if (strsame (aptr, "/ALERT=", 4))
      {
         sptr = aptr;
         while (*aptr && *aptr != '=') aptr++;
         if (*aptr) aptr++;
         if (!*aptr || strsame (aptr, "ALL", 3))
         {
            DoAlertAll = true;
            DoAlertHost = false;
         }
         else
         if (strsame (aptr, "HOST", 4))
         {
            DoAlertAll = false;
            DoAlertHost = true;
         }
         else
         {
            fprintf (stdout, "%%%s-E-INVPARM, invalid parameter\n \\%s\\\n",
                     Utility, sptr);
            exit (STS$K_ERROR | STS$M_INHIB_MSG);
         }
         continue;
      }
      if (strsame (aptr, "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (strsame (aptr, "/HELP", 4))
      {
         DoShowHelp = true;
         continue;
      }
      if (strsame (aptr, "/INTERVAL=", 4) ||
          strsame (aptr, "/REFRESH=", 4))
      {
         sptr = aptr;
         while (*aptr && *aptr != '=') aptr++;
         if (*aptr) aptr++;
         if (!isdigit(*aptr))
         {
            fprintf (stdout, "%%%s-E-INVPARM, invalid parameter\n \\%s\\\n",
                     Utility, sptr);
            exit (STS$K_ERROR | STS$M_INHIB_MSG);
         }
         IntervalSeconds = atoi(aptr);
         continue;
      }
      if (strsame (aptr, "/PORT=", 4))
      {
         sptr = aptr;
         while (*aptr && *aptr != '=') aptr++;
         if (*aptr) aptr++;
         if (!isdigit(*aptr))
         {
            fprintf (stdout, "%%%s-E-INVPARM, invalid parameter\n \\%s\\\n",
                     Utility, sptr);
            exit (STS$K_ERROR | STS$M_INHIB_MSG);
         }
         ServerPort = atoi(aptr);
         continue;
      }
      if (strsame (aptr, "/LOOKUP", 4))
      {
         DoLookupHost = true;
         continue;
      }
      if (strsame (aptr, "/NOLOOKUP", 6))
      {
         DoLookupHost = false;
         continue;
      }
      if (*aptr != '/')
      {
         fprintf (stdout, "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n",
                  Utility, aptr);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      else
      {
         fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
                  Utility, aptr+1);
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
   }
}

/*****************************************************************************/
/*
Assign a channel to the terminal device.  Create a buffered screenful of 
information about the HTTPd server and output it in one IO.
*/

int MonitorHttpd ()

{
   int  status,
        PortLength;
   unsigned short  TTChannel;
   char  ReadBuffer;
   char  Line [128],
         Port [16];
   char  *TimeStringPtr;
   
   $DESCRIPTOR (TTDsc, "TT:");
   struct {
      unsigned short  Status;
      unsigned short  Offset;
      unsigned short  Terminator;
      unsigned short  TerminatorSize;
   } IOsb;

   /*********/
   /* begin */
   /*********/

   if (VMSnok (status = sys$assign (&TTDsc, &TTChannel, 0, 0, 0)))
      return (status);

   if (VMSnok (status =
       sys$qiow (0, TTChannel, IO$_WRITELBLK, 0, 0, 0,
                 ANSIcls, sizeof(ANSIcls)-1, 0, 0, 0, 0)))
      return (status);

   sprintf (ServerProcessName, "HTTPd:%d", ServerPort);
   PortLength = sprintf (Port, " Port: %d", ServerPort) / 2;

   for (;;)
   {
      if (ControlY) break;

      TimeStringPtr = TimeString ();

      ScrPtr = Screen;
      ScrPtr += sprintf (ScrPtr,
         "%s%s%s%s%*s %s %s %*s%s%s%s%s\r\n",
         ANSIhome,
         ANSIbold, Port, ANSInormal,
         71 - PortLength - sizeof(SoftwareID)-1 -
              sizeof(ForHTTPdVersion)-1 - strlen(TimeStringPtr), "",
         SoftwareID, ForHTTPdVersion,
         71 - PortLength - sizeof(SoftwareID)-1 -
              sizeof(ForHTTPdVersion)-1 - strlen(TimeStringPtr), "",
         ANSIbold, TimeStringPtr, ANSInormal,
         ANSIceol);

      /* we need the startup count and last exit status from count date */
      if (VMSnok (status = GetAccounting ()))
         return (status);
      if (VMSnok (status = AddProcess ()))
         return (status);
      if (VMSnok (status = AddAccounting ()))
         return (status);
      if (VMSnok (status = AddRequest ()))
         return (status);

      strcpy (ScrPtr, ANSIceos);
      ScrPtr += sizeof(ANSIceos)-1;

      if (VMSnok (status =
          sys$qiow (0, TTChannel, IO$_WRITELBLK, 0, 0, 0,
                    Screen, ScrPtr-Screen, 0, 0, 0, 0)))
         return (status);

      status = sys$qiow (0, TTChannel, IO$_READLBLK | IO$M_TIMED,
                         &IOsb, 0, 0,
                         &ReadBuffer, 1, IntervalSeconds, 0, 0, 0);

      if (status == SS$_TIMEOUT) continue;
      if (VMSnok (status)) return (status);
      if (IOsb.Terminator == ControlZ) return (SS$_NORMAL);
   }

   if (VMSnok (status =
       sys$qiow (0, TTChannel, IO$_WRITELBLK, 0, 0, 0,
                 Screen, ScrPtr-Screen, 0, 0, 0, 0)))
      return (status);

   return (status);
}

/*****************************************************************************/
/*
Add the HTTPd server process information to the screen buffer using the 
process ID from the PID logical and a a sys$getjpi() call.  The PID logical 
contains binary data in the form of a longword process ID.
*/ 

int AddProcess ()

{
   static char  JpiPrcNam [16],
                JpiUserName [13],
                LogicalName [32],
                UpTime [32];
   static unsigned long  JpiAstCnt,
                         JpiAstLm,
                         JpiBioCnt,
                         JpiBytLm,
                         JpiBytCnt,
                         JpiBioLm,
                         JpiCpuTime,
                         JpiDioCnt,
                         JpiDioLm,
                         JpiEnqCnt,
                         JpiEnqLm,
                         JpiFilCnt,
                         JpiFilLm,
                         JpiPageFlts,
                         JpiPagFilCnt,
                         JpiPgFlQuota,
                         JpiPid,
                         JpiPrcCnt,
                         JpiPrcLm,
                         JpiTqCnt,
                         JpiTqLm,
                         JpiVirtPeak,
                         JpiWsSize,
                         JpiWsPeak,
                         PageFileUsedPercent,
                         Pid;
   static unsigned long  ConnectTime [2],
                         CurrentTime [2],
                         JpiLoginTime [2];
   static char  LastExitStatus [16] = "";
   static $DESCRIPTOR (LnmSystemDsc, "LNM$SYSTEM");
   static $DESCRIPTOR (LogicalNameDsc, LogicalName);
   static $DESCRIPTOR (UpTimeFaoDsc, "!%D");
   static $DESCRIPTOR (UpTimeDsc, UpTime);
   static struct
   {
      unsigned short  buf_len;
      unsigned short  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   }
      JpiItems [] =
   {
      { sizeof(JpiAstCnt), JPI$_ASTCNT, &JpiAstCnt, 0 },
      { sizeof(JpiAstLm), JPI$_ASTLM, &JpiAstLm, 0 },
      { sizeof(JpiBioCnt), JPI$_BIOCNT, &JpiBioCnt, 0 },
      { sizeof(JpiBioLm), JPI$_BIOLM, &JpiBioLm, 0 },
      { sizeof(JpiBytCnt), JPI$_BYTCNT, &JpiBytCnt, 0 },
      { sizeof(JpiBytLm), JPI$_BYTLM, &JpiBytLm, 0 },
      { sizeof(JpiCpuTime), JPI$_CPUTIM, &JpiCpuTime, 0 },
      { sizeof(JpiDioCnt), JPI$_DIOCNT, &JpiDioCnt, 0 },
      { sizeof(JpiDioLm), JPI$_DIOLM, &JpiDioLm, 0 },
      { sizeof(JpiEnqCnt), JPI$_ENQCNT, &JpiEnqCnt, 0 },
      { sizeof(JpiEnqLm), JPI$_ENQLM, &JpiEnqLm, 0 },
      { sizeof(JpiFilCnt), JPI$_FILCNT, &JpiFilCnt, 0 },
      { sizeof(JpiFilLm), JPI$_FILLM, &JpiFilLm, 0 },
      { sizeof(JpiLoginTime), JPI$_LOGINTIM, &JpiLoginTime, 0 },
      { sizeof(JpiPageFlts), JPI$_PAGEFLTS, &JpiPageFlts, 0 },
      { sizeof(JpiPagFilCnt), JPI$_PAGFILCNT, &JpiPagFilCnt, 0 },
      { sizeof(JpiPgFlQuota), JPI$_PGFLQUOTA, &JpiPgFlQuota, 0 },
      { sizeof(JpiPid), JPI$_PID, &JpiPid, 0 },
      { sizeof(JpiPrcCnt), JPI$_PRCCNT, &JpiPrcCnt, 0 },
      { sizeof(JpiPrcLm), JPI$_PRCLM, &JpiPrcLm, 0 },
      { sizeof(JpiPrcNam), JPI$_PRCNAM, &JpiPrcNam, 0 },
      { sizeof(JpiTqCnt), JPI$_TQCNT, &JpiTqCnt, 0 },
      { sizeof(JpiTqLm), JPI$_TQLM, &JpiTqLm, 0 },
      { sizeof(JpiUserName), JPI$_USERNAME, &JpiUserName, 0 },
      { sizeof(JpiVirtPeak), JPI$_VIRTPEAK, &JpiVirtPeak, 0 },
      { sizeof(JpiWsSize), JPI$_WSSIZE, &JpiWsSize, 0 },
      { sizeof(JpiWsPeak), JPI$_WSPEAK, &JpiWsPeak, 0 },
      {0,0,0,0}
   },
      LnmItem [] =
   {
      { sizeof(Pid), LNM$_STRING, &Pid, 0 },
      {0,0,0,0}
   };

   register char  *cptr;

   int  status;
   unsigned short  Length;
   char  *StatePtr,
         *UpTimePtr;
   char  AstString [32],
         BioString [32],
         BytString [32],
         DioString [32],
         EnqString [32],
         FilString [32],
         PrcString [32],
         TqString [32];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AddProcess()\n");

   if (!LogicalName[0])
      LogicalNameDsc.dsc$w_length =
         sprintf (LogicalName, "HTTPD%d$PID", ServerPort);

   Pid = 0;
   status = sys$trnlnm (0, &LnmSystemDsc, &LogicalNameDsc, 0, &LnmItem);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (VMSnok (status) && status != SS$_NOLOGNAM) return (status);
   if (Debug) fprintf (stdout, "Pid: %08.08X\n", Pid);

   if (Pid)
      status = sys$getjpiw (0, &Pid, 0, &JpiItems, 0, 0, 0);
   else
      status = SS$_NONEXPR;
   if (Debug) fprintf (stdout, "sys$getjpi() %%X%08.08X\n", status);
   if (VMSnok (status) || status == SS$_NONEXPR)
   {
      if (status == SS$_NONEXPR)
         StatePtr = "-NOT EXECUTING-";
      else
      if (status == SS$_SUSPENDED)
         StatePtr = "-MWAIT-";
      else
         return (status);

      /* we divide by this so it can't be zero, make it very small! */
      JpiPgFlQuota = 1;
   }
   else
      StatePtr = "";

   CurrentSubprocessCount = JpiPrcCnt;

   JpiUserName[12] = '\0';
   for (cptr = JpiUserName; *cptr && *cptr != ' '; cptr++);
   *cptr = '\0';
   if (Debug) fprintf (stdout, "JpiUserName |%s|\n", JpiUserName);

   JpiPrcNam[15] = '\0';
   for (cptr = JpiPrcNam; *cptr && *cptr != ' '; cptr++);
   *cptr = '\0';
   if (Debug) fprintf (stdout, "JpiPrcNam |%s|\n", JpiPrcNam);

   sys$gettim (&CurrentTime);
   lib$sub_times (&CurrentTime, &JpiLoginTime, &ConnectTime);
   sys$fao (&UpTimeFaoDsc, &Length, &UpTimeDsc, &ConnectTime);
   UpTime[Length] = '\0';
   for (UpTimePtr = UpTime; isspace(*UpTimePtr); UpTimePtr++);

   if (Accounting.StartupCount > 1)
      sprintf (LastExitStatus, "%%X%08.08X", Accounting.LastExitStatus);

   sprintf (AstString, "%4d/%-4d", JpiAstLm - JpiAstCnt, JpiAstLm);
   sprintf (BioString, "%4d/%-4d", JpiBioLm - JpiBioCnt, JpiBioLm);
   sprintf (BytString, "%6d/%-6d", JpiBytLm - JpiBytCnt, JpiBytLm);
   sprintf (DioString, "%4d/%-4d", JpiDioLm - JpiDioCnt, JpiDioLm);
   sprintf (EnqString, "%4d/%-4d", JpiEnqLm - JpiEnqCnt, JpiEnqLm);
   sprintf (FilString, "%4d/%-4d", JpiFilLm - JpiFilCnt, JpiFilLm);
   sprintf (PrcString, "%6d/%-6d", JpiPrcCnt, JpiPrcLm);
   sprintf (TqString, "%4d/%-4d", JpiTqLm - JpiTqCnt, JpiTqLm);

   if (status != SS$_SUSPENDED)
      PageFileUsedPercent = 100 - ((JpiPagFilCnt * 100) / JpiPgFlQuota);

   ScrPtr += sprintf (ScrPtr,
"%s\r\n\
 %sProcess:%s %s  %sPID:%s %08.08X  %sUser:%s %s  %s%s%s\
%s\r\n\
      %sUp:%s %s  %sCPU:%s %d %02.02d:%02.02d:%02.02d.%02.02d\
  %sRestart:%s %d %s\
%s\r\n\
 %sPg.Flts:%s %d  %sPg.Used:%s %d%%  \
%sWsSize:%s %d (%dkB)  %sWsPeak:%s %d (%dkB)\
%s\r\n\
     %sAST:%s %9.9s  %sBIO:%s %9.9s  %sBYT:%s %13.13s  %sDIO:%s %9.9s\
%s\r\n\
     %sENQ:%s %9.9s  %sFIL:%s %9.9s  %sPRC:%s %13.13s   %sTQ:%s %9.9s\
%s\r\n",
      ANSIceol,
      ANSIbold, ANSInormal,
      JpiPrcNam,
      ANSIbold, ANSInormal,
      Pid,
      ANSIbold, ANSInormal,
      JpiUserName,
      ANSIbold, StatePtr, ANSInormal,
      ANSIceol,
      ANSIbold, ANSInormal,
      UpTimePtr,
      ANSIbold, ANSInormal,
      JpiCpuTime / 8640000,                     /* CPU day */
      (JpiCpuTime % 8640000) / 360000,          /* CPU hour */
      (JpiCpuTime % 360000) / 6000,             /* CPU minute */
      (JpiCpuTime % 6000 ) / 100,               /* CPU second */
      JpiCpuTime % 100,                         /* CPU 10mS */
      ANSIbold, ANSInormal,
      Accounting.StartupCount,
      LastExitStatus,
      ANSIceol,
      ANSIbold, ANSInormal,
      JpiPageFlts,
      ANSIbold, ANSInormal,
      PageFileUsedPercent,
      ANSIbold, ANSInormal,
      JpiWsSize, JpiWsSize / 2,
      ANSIbold, ANSInormal,
      JpiWsPeak, JpiWsPeak / 2,
      ANSIceol,
      ANSIbold, ANSInormal,
      AstString,
      ANSIbold, ANSInormal,
      BioString,
      ANSIbold, ANSInormal,
      BytString,
      ANSIbold, ANSInormal,
      DioString,
      ANSIceol,
      ANSIbold, ANSInormal,
      EnqString,
      ANSIbold, ANSInormal,
      FilString,
      ANSIbold, ANSInormal,
      PrcString,
      ANSIbold, ANSInormal,
      TqString,
      ANSIceol);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Get the HTTPd server counter information from its logical definition.
*/

int GetAccounting ()

{
   static unsigned short  Length;
   static char  LogicalName1 [32],
                LogicalName2 [32];
   static $DESCRIPTOR (LnmSystemDsc, "LNM$SYSTEM");
   static $DESCRIPTOR (LogicalName1Dsc, LogicalName1);
   static $DESCRIPTOR (LogicalName2Dsc, LogicalName2);

   static struct {
      short BufferLength;
      short ItemCode;
      void  *BufferPtr;
      void  *LengthPtr;
   }
     LnmItem1 [] =
   {
      { 255, LNM$_STRING, &Accounting, 0 },
      {0,0,0,0}
   },
     LnmItem2 [] =
   {
      { sizeof(Accounting)-255, LNM$_STRING, ((char*)&Accounting)+255, 0 },
      {0,0,0,0}
   };

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetAccounting()\n");

   if (!LogicalName1[0])
   {
      LogicalName1Dsc.dsc$w_length =
         sprintf (LogicalName1, "HTTPD%d$COUNT1", ServerPort);
      LogicalName2Dsc.dsc$w_length =
         sprintf (LogicalName2, "HTTPD%d$COUNT2", ServerPort);
   }

   status = sys$trnlnm (0, &LnmSystemDsc, &LogicalName1Dsc, 0, &LnmItem1);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (VMSnok (status) && status != SS$_NOLOGNAM) return (status);

   status = sys$trnlnm (0, &LnmSystemDsc, &LogicalName2Dsc, 0, &LnmItem2);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (VMSnok (status) && status != SS$_NOLOGNAM) return (status);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Add the HTTPd server counter information to the screen buffer from the count 
logical.  The information in this logical is binary data in the form of the 
HTTPd count data structure.  This is used to extract each field member from 
the data.
*/

int AddAccounting ()

{
   int  status,
        SslPercent;
   char  QuadBytesRx [32],
         QuadBytesTx [32];
   char  *AlertPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AddAccounting()\n");

   AlertPtr = "";
   if (DoAlertAll)
      if (PrevConnectCount && Accounting.ConnectCount > PrevConnectCount)
         AlertPtr = "\x07";
   PrevConnectCount = Accounting.ConnectCount;

   CommaNumber (64, &Accounting.QuadBytesRx[0], QuadBytesRx); 
   CommaNumber (64, &Accounting.QuadBytesTx[0], QuadBytesTx); 

   if (Accounting.ConnectSslCount)
      SslPercent = (Accounting.ConnectSslCount*100) / Accounting.ConnectCount;
   else
      SslPercent = 0;

   ScrPtr += sprintf (ScrPtr,
"%s\r\n\
 %s%sConnect:%s %d  %sSSL:%s %d (%d%%)  %sAcc/Rej/Bsy:%s %d / %d / %d\
  %sCur/Peak:%s %d / %d\
%s\r\n\
   %sParse:%s %d  %sError:%s %d  %sForbidden:%s %d\
  %sRedirect-Rem/Loc:%s %d / %d\
%s\r\n\
  %sDELETE:%s %d  %sGET:%s %d  %sHEAD:%s %d  %sPOST:%s %d  %sPUT:%s %d\
%s\r\n\
   %sAdmin:%s %d  %sCache:%s %d / %d (%d)  %sDECnet-CGI/OSU:%s %d - %d / %d\
%s\r\n\
     %sDCL:%s %d  %sCGI/plus:%s %d / %d (%d)  %sSpawn/Cur:%s %d / %d\
  %sDir:%s %d\
%s\r\n\
    %sFile:%s %d (%d)  %sIsMap:%s %d  %sMenu:%s %d\
  %sPut:%s %d  %sSSI:%s %d  %sUpd:%s %d\
%s\r\n\
%s\r\n\
     %s1xx:%s %d  %s2xx:%s %d  %s3xx:%s %d  %s4xx:%s %d  %s5xx:%s %d\
  (%d errors)\
%s\r\n\
      %sRx:%s %s  %sTx:%s %s  (bytes)\
%s\r\n",
      ANSIceol,
      AlertPtr,
      ANSIbold, ANSInormal,
      Accounting.ConnectCount,
      ANSIbold, ANSInormal,
      Accounting.ConnectSslCount,
      SslPercent,
      ANSIbold, ANSInormal,
      Accounting.ConnectAcceptedCount,
      Accounting.ConnectRejectedCount,
      Accounting.ConnectTooBusyCount,
      ANSIbold, ANSInormal,
      Accounting.ConnectCurrent,
      Accounting.ConnectPeak,
      ANSIceol,
      ANSIbold, ANSInormal,
      Accounting.RequestParseCount,
      ANSIbold, ANSInormal,
      Accounting.RequestErrorCount,
      ANSIbold, ANSInormal,
      Accounting.RequestForbiddenCount,
      ANSIbold, ANSInormal,
      Accounting.RedirectRemoteCount,
      Accounting.RedirectLocalCount,
      ANSIceol,
      ANSIbold, ANSInormal,
      Accounting.MethodDeleteCount,
      ANSIbold, ANSInormal,
      Accounting.MethodGetCount,
      ANSIbold, ANSInormal,
      Accounting.MethodHeadCount,
      ANSIbold, ANSInormal,
      Accounting.MethodPostCount,
      ANSIbold, ANSInormal,
      Accounting.MethodPutCount,
      ANSIceol,
      ANSIbold, ANSInormal,
      Accounting.DoServerAdminCount,
      ANSIbold, ANSInormal,
      Accounting.CacheLoadCount,
      Accounting.CacheHitCount,
      Accounting.CacheHitNotModifiedCount,
      ANSIbold, ANSInormal,
      Accounting.DoDECnetCount,
      Accounting.DoDECnetCgiCount,
      Accounting.DoDECnetOsuCount,
      ANSIceol,
      ANSIbold, ANSInormal,
      Accounting.DoDclCommandCount,
      ANSIbold, ANSInormal,
      Accounting.DoScriptCount,
      Accounting.DoCgiPlusScriptCount,
      Accounting.DclCgiPlusReusedCount,
      ANSIbold, ANSInormal,
      Accounting.DclSpawnCount,
      CurrentSubprocessCount,
      ANSIbold, ANSInormal,
      Accounting.DoDirectoryCount,
      ANSIceol,
      ANSIbold, ANSInormal,
      Accounting.DoFileCount,
      Accounting.DoFileNotModifiedCount,
      ANSIbold, ANSInormal,
      Accounting.DoIsMapCount,
      ANSIbold, ANSInormal,
      Accounting.DoMenuCount,
      ANSIbold, ANSInormal,
      Accounting.DoPutCount,
      ANSIbold, ANSInormal,
      Accounting.DoSsiCount,
      ANSIbold, ANSInormal,
      Accounting.DoUpdateCount,
      ANSIceol,
      ANSIceol,
      ANSIbold, ANSInormal,
      Accounting.ResponseStatusCodeCountArray[1],
      ANSIbold, ANSInormal,
      Accounting.ResponseStatusCodeCountArray[2],
      ANSIbold, ANSInormal,
      Accounting.ResponseStatusCodeCountArray[3],
      ANSIbold, ANSInormal,
      Accounting.ResponseStatusCodeCountArray[4],
      ANSIbold, ANSInormal,
      Accounting.ResponseStatusCodeCountArray[5],
      Accounting.ResponseStatusCodeCountArray[0],
      ANSIceol,
      ANSIbold, ANSInormal,
      QuadBytesRx,
      ANSIbold, ANSInormal,
      QuadBytesTx,
      ANSIceol);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Add the information about the latest request to the screen buffer from the 
request logical.  Each plain-text string in this logical is terminated by a 
null character.
*/ 

int AddRequest ()

{
   static unsigned short  Length;
   static char  LogicalName [32],
                LogicalValue [256];
   static $DESCRIPTOR (LnmSystemDsc, "LNM$SYSTEM");
   static $DESCRIPTOR (LogicalNameDsc, LogicalName);
   struct {
      short BufferLength;
      short ItemCode;
      void  *BufferPtr;
      void  *LengthPtr;
   } LnmItem [] =
   {
      { sizeof(LogicalValue)-1, LNM$_STRING, LogicalValue, &Length },
      {0,0,0,0}
   };

   register int  cnt;
   register char  *cptr, *hptr, *sptr;

   int  status;
   char  HostName [256];
   char  *AlertPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AddRequest()\n");

   if (!LogicalName[0])
      LogicalNameDsc.dsc$w_length =
         sprintf (LogicalName, "HTTPD%d$REQUEST", ServerPort);

   Length = 0;
   status = sys$trnlnm (0, &LnmSystemDsc, &LogicalNameDsc, 0, &LnmItem);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (VMSnok (status) && status != SS$_NOLOGNAM) return (status);
   LogicalValue[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", LogicalValue);

   sptr = ScrPtr;
   cptr = LogicalValue;
   cnt = Length;

   sptr += sprintf (sptr, "%s\r\n    %sTime:%s ",
                    ANSIceol, ANSIbold, ANSInormal);
   while (*cptr && cnt--) *sptr++ = *cptr++;
   if (cnt) { cnt--; cptr++; }

   sptr += sprintf (sptr, "  %sStatus:%s ", ANSIbold, ANSInormal);
   while (*cptr && cnt--) *sptr++ = *cptr++;
   if (cnt) { cnt--; cptr++; }

   sptr += sprintf (sptr, "  %sRx:%s ", ANSIbold, ANSInormal);
   while (*cptr && cnt--) *sptr++ = *cptr++;
   if (cnt) { cnt--; cptr++; }

   sptr += sprintf (sptr, "  %sTx:%s ", ANSIbold, ANSInormal);
   while (*cptr && cnt--) *sptr++ = *cptr++;
   if (cnt) { cnt--; cptr++; }

   sptr += sprintf (sptr, "  %sDuration:%s ", ANSIbold, ANSInormal);
   while (*cptr && cnt--) *sptr++ = *cptr++;
   if (cnt) { cnt--; cptr++; }

   sptr += sprintf (sptr, "%s\r\n %sService:%s ",
                    ANSIceol, ANSIbold, ANSInormal);
   while (*cptr && cnt--) *sptr++ = *cptr++;
   if (cnt) { cnt--; cptr++; }

   AlertPtr = "";
   if (DoAlertHost)
   {
      if (PrevHostName[0])
      {
         if (strcmp (cptr, PrevHostName))
         {
            AlertPtr = "\x07";
            strcpy (PrevHostName, cptr);
         }
      }
      else
         strcpy (PrevHostName, cptr);
   }

   sptr += sprintf (sptr, "%s\r\n    %s%sHost:%s ",
                    ANSIceol, AlertPtr, ANSIbold, ANSInormal);
#ifdef IP_NONE
   while (*cptr && cnt--) *sptr++ = *cptr++;
   if (cnt) { cnt--; cptr++; }
#else
   if (DoLookupHost && isdigit(*cptr))
   {
      hptr = HostName;
      while (*cptr && cnt--) *hptr++ = *cptr++;
      *hptr = '\0';
      if (cnt) { cnt--; cptr++; }
      hptr = LookupHostName (HostName);
      if (*hptr)
      {
         while (*hptr) *sptr++ = *hptr++;
         *sptr++ = ' ';
         *sptr++ = '(';
         for (hptr = HostName; *hptr; *sptr++ = *hptr++);
         *sptr++ = ')';
      }
      else
         for (hptr = HostName; *hptr; *sptr++ = *hptr++);
   }
   else
   {
      while (*cptr && cnt--) *sptr++ = *cptr++;
      if (cnt) { cnt--; cptr++; }
   }
#endif

   sptr += sprintf (sptr, "%s\r\n %sRequest:%s ",
                    ANSIceol, ANSIbold, ANSInormal);
   while (*cptr && cnt--) *sptr++ = *cptr++;
   if (cnt) { cnt--; cptr++; }
   if (Length == 255)
      sptr += sprintf (sptr, "%s...%s", ANSIbold, ANSInormal);

   sptr += sprintf (sptr, "%s", ANSIceol);

   ScrPtr = sptr;

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
UCX version host name lookup.
*/

#ifdef IP_UCX

char* LookupHostName (char *HostDotDecimal)

{
   static char  HostName [256],
                PrevHostDotDecimal [256];

   register char  *cptr, *sptr;

   int  HostIpNumber;
   struct hostent  *HostEntryPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "LookupHostName() |%s|\n", HostDotDecimal);

   /* no need to look it up again if it's the same one! */
   if (!strcmp (HostDotDecimal, PrevHostDotDecimal)) return (HostName);
   strcpy (PrevHostDotDecimal, HostDotDecimal);

   /* get the host address details */
   if ((HostIpNumber = inet_addr (HostDotDecimal)) == -1)
   {
      sptr = HostName;
      for (cptr = HostDotDecimal; *cptr; *sptr++ = *cptr++);
      strcpy (sptr, " [error]");
      return (HostName);
   }
   if (Debug) fprintf (stdout, "%08.08X\n", HostIpNumber);

   if (Debug) fprintf (stdout, "gethostbyaddr()\n");
   if ((HostEntryPtr = gethostbyaddr (&HostIpNumber, 4, AF_INET)) == NULL)
   {
      HostName[0] = '\0';
      return (HostName);
   }

   sptr = HostName;
   /* looks better if its all in lower case (host name is sometimes upper) */
   for (cptr = HostEntryPtr->h_name; *cptr; cptr++) *sptr++ = tolower(*cptr);
   *sptr = '\0';
   if (Debug) fprintf (stdout, "HostName |%s|\n", HostName);
   return (HostName);
}

#endif /* IP_UCX */

/*****************************************************************************/
/*
NETLIB version host name lookup.
*/

#ifdef IP_NETLIB

char* LookupHostName (char *HostDotDecimal)

{
   static unsigned long  NetLibSocketType = NETLIB_K_TYPE_STREAM,
                         NetLibSocketFamily = NETLIB_K_AF_INET,
                         NetLibSizeofINADDRDEF = sizeof(struct INADDRDEF);

   struct SINDEF  HostNetLibName;

   static unsigned long  NetLibSocket;

   static $DESCRIPTOR (HostDotDecimalDsc, "");
   static $DESCRIPTOR (HostNameDsc, "");

   static char  HostName [256],
                PrevHostDotDecimal [256];

   register char  *cptr, *sptr;

   int  Length;

   int  status,
        HostIpNumber;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "LookupHostName() |%s|\n", HostDotDecimal);

   if (!NetLibSocket)
   {
      /* just need any old socket to be able to use the service! */
      status = netlib_socket (&NetLibSocket, &NetLibSocketType,
                              &NetLibSocketFamily);
      if (Debug) fprintf (stdout, "netlib_socket() %%X%08.08X\n", status);
      if (VMSnok (status)) exit (status);
   }

   /* no need to look it up again if it's the same one! */
   if (!strcmp (HostDotDecimal, PrevHostDotDecimal)) return (HostName);
   strcpy (PrevHostDotDecimal, HostDotDecimal);

   HostDotDecimalDsc.dsc$a_pointer = HostDotDecimal;
   HostDotDecimalDsc.dsc$w_length = strlen(HostDotDecimal);

   status = netlib_strtoaddr (&HostDotDecimalDsc,
                              &HostNetLibName.sin_x_addr.inaddr_l_addr);
   if (Debug) fprintf (stdout, "netlib_strtoaddr() %%X%08.08X\n", status);
   if (VMSnok (status))
   {
      sptr = HostName;
      for (cptr = HostDotDecimal; *cptr; *sptr++ = *cptr++);
      strcpy (sptr, " [error]");
      return (HostName);
   }

   HostNameDsc.dsc$w_length = sizeof(HostName)-1;
   HostNameDsc.dsc$a_pointer = &HostName;

   Length = 0;
   status = netlib_address_to_name (&NetLibSocket, 0,
               &HostNetLibName.sin_x_addr.inaddr_l_addr,
               &NetLibSizeofINADDRDEF, &HostNameDsc, &Length, 0, 0, 0);
   if (Debug)
      fprintf (stdout, "netlib_address_to_name() %%X%08.08X\n", status);
   if (!Length)
   {
      HostName[0] = '\0';
      return (HostName);
   }

   HostName[Length] = '\0';
   if (Debug) fprintf (stdout, "HostName |%s|\n", HostName);
   return (HostName);
}

#endif /* IP_NETLIB */

/*****************************************************************************/
/*
*/

int OnControlY (void *FunctionAddress)

{
   static boolean  Disabled = false;
   static unsigned long  Mask = LIB$M_CLI_CTRLY,
                         OldMask;
   static unsigned short  TTChannel = 0;

   int  status;
   $DESCRIPTOR (TTDsc, "TT:");

   /*********/
   /* begin */
   /*********/

   if (FunctionAddress)
   {
      if (!TTChannel)
         if (VMSnok (status = sys$assign (&TTDsc, &TTChannel, 0, 0, 0)))
            return (status);

      if (VMSnok (status =
          sys$qiow (0, TTChannel, IO$_SETMODE | IO$M_CTRLYAST, 0, 0, 0,
                    FunctionAddress, 0, PSL$C_USER, 0, 0, 0)))
         return (status);

      if (VMSnok (status =
          sys$qiow (0, TTChannel, IO$_SETMODE | IO$M_CTRLCAST, 0, 0, 0,
                    FunctionAddress, 0, PSL$C_USER, 0, 0, 0)))
         return (status);

      if (!Disabled)
      {
         Disabled = true;
         return (lib$disable_ctrl (&Mask, &OldMask));
      }
      else
         return (status);
   }
   else
   {
      sys$cancel (TTChannel);
      return (lib$enable_ctrl (&OldMask, 0));
   }
}

/*****************************************************************************/
/*
*/

ControlY_AST ()

{
   ControlY = true;
   sys$wake (0, 0);
   OnControlY (&ControlY_AST);
}

/*****************************************************************************/
/*
*/

char* TimeString ()

{
   static int  LibDayOfWeek = LIB$K_DAY_OF_WEEK;
   static char  *WeekDays [] =
   {"","Monday","Tuesday","Wednesday","Thursday","Friday","Saturday","Sunday"};
   static char  TimeString [35];
   static $DESCRIPTOR (DayDateTimeDsc, "!AZ, !%D");
   static $DESCRIPTOR (TimeStringDsc, TimeString);

   unsigned long  BinTime [2];
   int  status,
        DayOfWeek;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   sys$gettim (&BinTime);
   lib$cvt_from_internal_time (&LibDayOfWeek, &DayOfWeek, &BinTime);
   sys$fao (&DayDateTimeDsc, &Length, &TimeStringDsc,
            WeekDays[DayOfWeek], &BinTime);
   TimeString[Length-3] = '\0';
   return (TimeString);
}

/*****************************************************************************/
/*
Convert the 32/64 bit integer (depending on architecture) pointed to into an
ASCII number string containing commas.  The destination string should contain
capacity of a minimum 16 characters for 32 bits, or 32 characters for 64 bits.
*/

int CommaNumber
(
int Bits,
unsigned long Value,
char *String
)
{
   static $DESCRIPTOR (Value32FaoDsc, "!UL");
   static $DESCRIPTOR (Value64FaoDsc, "!@UJ");

   register int  cnt;
   register char  *cptr, *sptr;

   int  status;
   char  Scratch [32];
   unsigned short  Length;
   $DESCRIPTOR (ScratchDsc, Scratch);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CommaNumber()\n");

   if (Bits > 32)
      status = sys$fao (&Value64FaoDsc, &Length, &ScratchDsc, Value);
   else
      status = sys$fao (&Value32FaoDsc, &Length, &ScratchDsc, Value);
   if (VMSnok (status))
   {
      strcpy (String, "*ERROR*");
      return;
   }
   Scratch[Length] = '\0';
   if (((Length-1) / 3) < 1)
   {
      strcpy (String, Scratch);
      return;
   }
   else
   if (!(cnt = Length % 3))
      cnt = 3;
   
   cptr = Scratch;
   sptr = String;
   while (*cptr)
   {
      if (!cnt--)
      {
         *sptr++ = ',';
         cnt = 2;
      }
      *sptr++ = *cptr++;
   }
   *sptr = '\0';
}

/****************************************************************************/
/*
*/
 
int ShowHelp ()
 
{
   fprintf (stdout,
"%%%s-I-HELP, usage for the WASD HTTPd Monitor (%s)\n\
\n\
Provides a display of the current status of the HyperText Transfer Protocol\n\
Daemon (HTTPd) process.  Must be executed on the system that has the process\n\
running on it.  Defaults to the server associated with IP port number 80.\n\
Displays process information, server counters and latest request information.\n\
By default attempts to resolve request dotted-decimal host address to host\n\
name.  The alert qualifier activates the terminal bell for either an increase\n\
in server connect count (all) or change in request host.\n\
\n\
$ HTTPDMON [qualifiers...]\n\
\n\
/ALERT[=ALL|HOST] /HELP /INTERVAL=integer /[NO]LOOKUP /PORT=integer\n\
/REFRESH=integer\n\
\n\
Usage examples:\n\
\n\
$ HTTPDMON\n\
$ HTTPDMON /PORT=8080\n\
$ HTTPDMON /INTERVAL=5\n\
$ HTTPDMON /PORT=8080 /ALERT\n\
$ HTTPDMON /NOLOOKUP\n\
\n",
   Utility, SoftwareID);
 
   return (SS$_NORMAL);
}
 
/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 
 
boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}
 
/*****************************************************************************/

