/*****************************************************************************/
/*
                                 StreamLF.c

A utility to copy files, converting to/from STREAM_LF record format.  Primary
use will be for NFS-accessable files where VMS by default creates and uses
VARIABLE format but to be updateable (writeable) via NFS they need to be
STREAM_LF.  This utility provides a next-highest version in the required
format.

The default behaviour is to convert all files to stream-LF with confirmation
and logging.


QUALIFIERS
----------
/[NO]CONFIRM    confirm any copy before commencing
/FROM           convert from STREAM_LF to VARIABLE
/HELP           display a screen of information
/IDENTIFY       list all the files with non-stream formats
/[NO]LOG        log all decisions/copies
/PURGE          delete previous (original record format) version ... CAUTION!
/TO             convert from <format> (usually VARIABLE) to STREAM_LF


VERSION HISTORY
---------------
03-MAY-96  MGD  initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "STREAMLF AXP-1.0.0";
#else
   char SoftwareID [] = "STREAMLF VAX-1.0.0";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <atrdef.h>
#include <descrip.h>
#include <fibdef.h>
#include <iodef.h>
#include <rms.h>
#include <rmsdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* extracted from FATDEF.H */
#define FAT$C_UNDEFINED 0               /* undefined record type */
#define FAT$C_FIXED 1                   /* fixed record type */
#define FAT$C_VARIABLE 2                /* variable length */
#define FAT$C_VFC 3                     /* variable + fixed control */
#define FAT$C_STREAM 4                  /* RMS-11 stream format  */
#define FAT$C_STREAMLF 5                /* LF-terminated stream format */
#define FAT$C_STREAMCR 6                /* CR-terminated stream format */
#define FAT$C_SEQUENTIAL 0              /* sequential organization */
#define FAT$C_RELATIVE 1                /* relative organization */
#define FAT$C_INDEXED 2                 /* indexed organization */
#define FAT$C_DIRECT 3                  /* direct organization */
#define FAT$M_FORTRANCC 0x1
#define FAT$M_IMPLIEDCC 0x2
#define FAT$M_PRINTCC 0x4
#define FAT$M_NOSPAN 0x8
#define FAT$M_MSBRCW 0x10

#define boolean int
#define true 1
#define false 0

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

char  Utility [] = "STREAMLF";

boolean  Debug,
         DoConfirm,
         DoFromStreamLF,
         DoHelp,
         DoIdentify,
         DoLog,
         DoPurge,
         DoToStreamLF;

char  CommandLine [256],
      FileSpec [256];

/*****************************************************************************/
/*
*/

main
(
int argc,
char *argv[]
)
{
   int  acnt,
        status;

   /*********/
   /* begin */
   /*********/

   if (getenv ("STREAMLF$DBUG") != NULL) Debug = true;

   if (VMSnok (status = ParseCommandLine ()))
      exit (status);

   if (DoHelp)
   {
      ShowHelp ();
      exit (SS$_NORMAL);
   }

   SearchFileSpec (FileSpec);

   exit (SS$_NORMAL);
}

/****************************************************************************/
/*
*/ 

SearchFileSpec (char *FileSpec)

{
   int  status,
        FileCount,
        FileNameLength,
        Length;
   unsigned char  RecordType;
   char  FileName [256],
         ExpandedFileSpec [256];
   struct FAB  SearchFileFab;
   struct NAM  SearchFileNam;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SearchFileSpec() |%s|\n", FileSpec);

   /* initialize the file access block */
   SearchFileFab = cc$rms_fab;
   SearchFileFab.fab$l_dna = "*.TXT";
   SearchFileFab.fab$b_dns = 5;
   SearchFileFab.fab$l_fna = FileSpec;
   SearchFileFab.fab$b_fns = strlen(FileSpec);
   SearchFileFab.fab$l_fop = FAB$V_NAM;
   SearchFileFab.fab$l_nam = &SearchFileNam;

   /* initialize the file name block */
   SearchFileNam = cc$rms_nam;
   SearchFileNam.nam$l_esa = ExpandedFileSpec;
   SearchFileNam.nam$b_ess = sizeof(ExpandedFileSpec)-1;
   SearchFileNam.nam$l_rsa = FileName;
   SearchFileNam.nam$b_rss = sizeof(FileName)-1;

   if (VMSnok (status = sys$parse (&SearchFileFab, 0, 0)))
      exit (status);

   while (VMSok (status = sys$search (&SearchFileFab, 0, 0)))
   {
      SearchFileNam.nam$l_ver[SearchFileNam.nam$b_ver] = '\0';
      if (Debug) fprintf (stdout, "FileName |%s|\n", FileName);

      FileNameLength = (SearchFileNam.nam$l_ver - FileName) +
                       SearchFileNam.nam$b_ver;

      GetFileInfo (&SearchFileNam, &RecordType);

      if (DoIdentify)
      {
         if ((RecordType == FAT$C_STREAMLF) ||
             (RecordType == FAT$C_STREAM) ||
             (RecordType == FAT$C_STREAMCR)) continue;

         if (RecordType == FAT$C_VARIABLE)
            fprintf (stdout, "%%%s-I-VARIABLE, %s\n", Utility, FileName);
         else
         if (RecordType == FAT$C_VFC)
            fprintf (stdout, "%%%s-I-VFC, %s\n", Utility, FileName);
         else
         if (RecordType == FAT$C_FIXED)
            fprintf (stdout, "%%%s-W-FIXED, %s\n", Utility, FileName);
         else
         if (RecordType == FAT$C_UNDEFINED)
            fprintf (stdout, "%%%s-W-UNDEFINED, %s\n", Utility, FileName);
         else
         {
            fprintf (stdout, "%%%s-E-INTERNAL, error\n", Utility);
            exit (STS$K_ERROR | STS$M_INHIB_MSG);
         }
      }
      else
      if (DoToStreamLF)
      {
         if ((RecordType == FAT$C_STREAMLF) ||
             (RecordType == FAT$C_STREAM) ||
             (RecordType == FAT$C_STREAMCR))
         {
            if (DoLog)
               fprintf (stdout, "%%%s-I-ISSTREAM, %s\n",
                        Utility, FileName);
         }
         else
         if (RecordType == FAT$C_VARIABLE)
         {
            if (DoLog)
               fprintf (stdout, "%%%s-I-VARIABLE, %s\n", Utility, FileName);
            CopyFile (FileName, FileNameLength, FAB$C_STMLF);
         }
         else
         if (RecordType == FAT$C_VFC)
         {
            if (DoLog)
               fprintf (stdout, "%%%s-I-VFC, %s\n", Utility, FileName);
            CopyFile (FileName, FileNameLength, FAB$C_STMLF);
         }
         else
         if (RecordType == FAT$C_FIXED)
         {
            if (DoLog)
               fprintf (stdout, "%%%s-W-FIXED, %s\n", Utility, FileName);
         }
         else
         if (RecordType == FAT$C_UNDEFINED)
         {
            if (DoLog)
               fprintf (stdout, "%%%s-W-UNDEFINED, %s\n", Utility, FileName);
         }
         else
         {
            fprintf (stdout, "%%%s-E-INTERNAL, error\n", Utility);
            exit (STS$K_ERROR | STS$M_INHIB_MSG);
         }
      }
      else
      if (DoFromStreamLF)
      {
         if (RecordType == FAT$C_STREAMLF)
            CopyFile (FileName, FileNameLength, FAB$C_VAR);
         else
            if (DoLog)
               fprintf (stdout, "%%%s-W-NOTSTREAMLF, %s\n", Utility, FileName);
      }

      FileCount++;
   }

   if (VMSnok (status) && status != RMS$_NMF) exit (status);
}

/*****************************************************************************/
/*
*/ 

int CopyFile
(
char *FileName,
int FileNameLength,
int DstRecordFormat
)
{
   register char  *cptr;

   int  status;
   char  Buffer [32768],
         Confirmation [32],
         DstExpandedFileName [256],
         DstRelatedFileName [256],
         SrcExpandedFileName [256];
   struct FAB  SrcFileFab;
   struct NAM  SrcFileNam;
   struct RAB  SrcFileRab;
   struct FAB  DstFileFab;
   struct NAM  DstFileNam;
   struct RAB  DstFileRab;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CopyFile() |%s|\n", FileName);

   if (DoConfirm)
   {
      if (DoToStreamLF)
         fprintf (stdout, "%s\nCopy with conversion to STREAM_LF? [N]: ",
                  FileName);
      else
         fprintf (stdout, "%s\nCopy with conversion to VARIABLE? [N]: ",
                  FileName);
      fgets (Confirmation, sizeof(Confirmation), stdin);
      if (toupper(Confirmation[0]) != 'Y') return;
   }

   SrcFileFab = cc$rms_fab;
   SrcFileFab.fab$l_fna = FileName;  
   SrcFileFab.fab$b_fns = FileNameLength;

   if (DoPurge)
   {
      SrcFileFab.fab$l_nam = &SrcFileNam;  
      SrcFileNam = cc$rms_nam;
      SrcFileNam.nam$l_esa = SrcExpandedFileName;  
      SrcFileNam.nam$b_ess = sizeof(SrcExpandedFileName)-1;
   }

   status = sys$open (&SrcFileFab, 0, 0);
   if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
   if (VMSnok (status)) exit (status);

   if (DoPurge) SrcFileNam.nam$l_ver[SrcFileNam.nam$b_ver] = '\0';

   SrcFileRab = cc$rms_rab;
   SrcFileRab.rab$l_fab = &SrcFileFab;
   SrcFileRab.rab$l_ubf = Buffer;
   SrcFileRab.rab$w_usz = sizeof(Buffer);

   status = sys$connect (&SrcFileRab, 0, 0);
   if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
   if (VMSnok (status)) exit (status);

   DstFileFab = cc$rms_fab;
   DstFileFab.fab$l_fna = FileName;  
   for (cptr = FileName; *cptr && *cptr != ';'; cptr++);
   DstFileFab.fab$b_fns = cptr - FileName;
   DstFileFab.fab$l_fop = FAB$M_SQO;
   DstFileFab.fab$b_rfm = DstRecordFormat;

   DstFileFab.fab$l_nam = &DstFileNam;  
   DstFileNam = cc$rms_nam;
   DstFileNam.nam$l_esa = DstExpandedFileName;  
   DstFileNam.nam$b_ess = sizeof(DstExpandedFileName)-1;
   DstFileNam.nam$l_rsa = DstRelatedFileName;  
   DstFileNam.nam$b_rss = sizeof(DstRelatedFileName)-1;

   status = sys$create (&DstFileFab, 0, 0);
   if (Debug) fprintf (stdout, "sys$create() %%X%08.08X\n", status);
   if (VMSnok (status)) exit (status);

   DstFileNam.nam$l_ver[DstFileNam.nam$b_ver] = '\0'; 

   DstFileRab = cc$rms_rab;
   DstFileRab.rab$l_fab = &DstFileFab;
   DstFileRab.rab$l_rbf = Buffer;

   status = sys$connect (&DstFileRab, 0, 0);
   if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
   if (VMSnok (status)) exit (status);

   while (VMSok (status = sys$get (&SrcFileRab , 0, 0)))
   {
      /** if (Debug) fprintf (stdout, "sys$get() %%X%08.08X\n", status); **/
      DstFileRab.rab$w_rsz = SrcFileRab.rab$w_rsz;
      if (VMSnok (status = sys$put (&DstFileRab, 0, 0)))
         exit (status);
      /** if (Debug) fprintf (stdout, "sys$put() %%X%08.08X\n", status); **/
   }

   if (status != RMS$_EOF) exit (status);

   sys$close (&DstFileFab, 0, 0);
   sys$close (&SrcFileFab, 0, 0);

   if (DoLog)
   {
      if (DoToStreamLF)
         fprintf (stdout, "%%%s-I-STREAMLF, %s\n", Utility, DstRelatedFileName);
      else
         fprintf (stdout, "%%%s-I-VARIABLE, %s\n", Utility, DstRelatedFileName);
   }

   if (DoPurge)
   {
      if (VMSnok (status = sys$erase (&SrcFileFab, 0, 0)))
         exit (status);
      if (DoLog) fprintf (stdout, "%%%s-I-PURGE, %s\n", Utility, FileName);
   }
}

/*****************************************************************************/
/*
This function uses the ACP-QIO interface detailed in the "OpenVMS I/O User's 
Reference Manual".
*/ 

int GetFileInfo
(
struct NAM *FileNamPtr,
unsigned char *RecordTypePtr
)
{
   static $DESCRIPTOR (DeviceDsc, "");
   static struct fibdef  FileFib;

   static struct {
      unsigned char  RecordType;
      unsigned char  RecordAttributes;
      unsigned char  OfNoInterest1 [30];
   } AtrRecord;

   static struct atrdef FileAtr [] =
   {
      { sizeof(AtrRecord), ATR$C_RECATTR, &AtrRecord },
      { 0, 0, 0 }
   };

   static struct {
      unsigned short  Length;
      unsigned short  Unused;
      unsigned long  Address;
   } FileNameAcpDsc,
     FileFibAcpDsc;

   static struct {
      unsigned short  Status;
      unsigned short  Unused1;
      unsigned long  Unused2;
   } AcpIOsb;

   int  status;
   unsigned short  AcpChannel;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetFileInfo()\n");

   /* assign a channel to the disk device containing the file */
   DeviceDsc.dsc$w_length = FileNamPtr->nam$b_dev;
   DeviceDsc.dsc$a_pointer = FileNamPtr->nam$l_dev;
   if (Debug)
      fprintf (stdout, "device |%*.*s|\n",
               FileNamPtr->nam$b_dev, FileNamPtr->nam$b_dev,
               FileNamPtr->nam$l_dev);

   status = sys$assign (&DeviceDsc, &AcpChannel, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$assign() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   /* set up the File Information Block for the ACP interface */
   memset (&FileFib, 0, sizeof(struct fibdef));
   FileFibAcpDsc.Length = sizeof(FileFib);
   FileFibAcpDsc.Address = &FileFib;
   /* quick work around, different syntax for this structure with DEC C? */
#ifdef __DECC
   FileFib.fib$w_did[0] = FileNamPtr->nam$w_did[0];
   FileFib.fib$w_did[1] = FileNamPtr->nam$w_did[1];
   FileFib.fib$w_did[2] = FileNamPtr->nam$w_did[2];
#else
   FileFib.fib$r_did_overlay.fib$w_did[0] = FileNamPtr->nam$w_did[0];
   FileFib.fib$r_did_overlay.fib$w_did[1] = FileNamPtr->nam$w_did[1];
   FileFib.fib$r_did_overlay.fib$w_did[2] = FileNamPtr->nam$w_did[2];
#endif

   FileNameAcpDsc.Address = FileNamPtr->nam$l_name;
   FileNameAcpDsc.Length = FileNamPtr->nam$b_name +
                           FileNamPtr->nam$b_type +
                           FileNamPtr->nam$b_ver;

   status = sys$qiow (0, AcpChannel, IO$_ACCESS, &AcpIOsb, 0, 0, 
                      &FileFibAcpDsc, &FileNameAcpDsc, 0, 0,
                      &FileAtr, 0);

   sys$dassgn (AcpChannel);

   if (Debug)
      fprintf (stdout, "sys$qio() %%X%08.08X IOsb: %%X%08.08X\n",
              status, AcpIOsb.Status);

   if (VMSok (status)) status = AcpIOsb.Status;

   if (Debug)
      fprintf (stdout, "Type: %02.02X Attributes: %02.02X\n",
               AtrRecord.RecordType, AtrRecord.RecordAttributes);

   *RecordTypePtr = AtrRecord.RecordType;

   return (status);
}

/*****************************************************************************/
/*
*/

int ShowHelp ()

{
   fprintf (stdout,
"%%%s-I-HELP, usage for StreamLF Utility (%s)\n\
\n\
A utility to copy files, converting to/from STREAM_LF record format.\n\
\n\
$ STREAMLF [file-specification] [qualifier...]\n\
\n\
/[NO]CONFIRM /FROM /HELP /IDENTIFY /[NO]LOG /PURGE /TO\n\
\n\
Usage examples:\n\
\n\
$ STREAMLF *.TXT                !prompts to convert any non-stream-LF\n\
$ STREAMLF *.TXT /NOCONFIRM     !converts any non-stream-LF (caution)!\n\
$ STREAMLF *.TXT /IDENTIFY      !identifies the current format\n\
$ STREAMLF *.TXT /TO            !converts any non-stream-LF (caution)!\n\
$ STREAMLF *.TXT /FROM          !converts from stream-LF to variable\n\
$ STREAMLF *.TXT /PURGE         !converts, deletes original (caution)!\n\
\n",
   Utility, SoftwareID);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 
 
boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Get the 
entire command line following the verb that activated the image.  The command 
line is returned in uppercase, space compressed (i.e. maximum of one space 
between text elements, trimmed of leading and trailing spaces).  Returns a 
warning status if there were no parameters/qualifiers on the command line.
The variable CommandLine is global.
*/ 
 
int ParseCommandLine ()
 
{
   int  status;
   unsigned short  Length;
   unsigned long  Flags = 0;
   struct dsc$descriptor_s 
          CommandLineDsc = { sizeof(CommandLine)-1, DSC$K_DTYPE_T,
                             DSC$K_CLASS_S, CommandLine };
 
   /* get the entire command line following the verb */
   if (VMSnok (status = lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
      return (status);
   CommandLine[Length] = '\0';
 
   if (ParseCommand (CommandLine))
      return (SS$_NORMAL);
   else
      return (STS$K_ERROR | STS$M_INHIB_MSG);
}
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Quoted 
strings are always indicated by being parsed to include a single leading 
quote.
*/ 
 
boolean ParseCommand (char *CommandLine)
 
{
   register int  QuoteCount = 0;
   register char  *cptr, *eptr;
   boolean  CommandLineOK = true;
   char  Entity [256] = "";
 
   /* set up any argument defaults */
   ParseCommandEntity (NULL);
 
   cptr = CommandLine;
   eptr = Entity;
 
   for (;;)
   {
      if (*cptr == '\"')
      {
         QuoteCount++;
         *eptr++ = *cptr++;
         continue;
      }
 
      if (QuoteCount & 1 && *cptr)
      {
         /* inside quoted text, copy all characters as literals */
         *eptr++ = *cptr++;
         continue;
      }
 
      if (*cptr == '/' || isspace (*cptr) || !*cptr)
      {
         if (isspace (*cptr))
         {
            /* span the white space */
            while (*cptr && isspace (*cptr)) cptr++;
            if (*cptr == '=')
            {
               /* part of a qualifier, continue to get the value */
               *eptr++ = *cptr++;
               /* span any intervening white space */
               while (*cptr && isspace (*cptr)) cptr++;
               continue;
            }
         }
 
         if (Entity[0])
         {
            *eptr = '\0';
            if (!ParseCommandEntity (Entity)) CommandLineOK = false;
         }
 
         /* if end of command line then break from loop */
         if (!*cptr) break;
 
         /* start of new entity */
         eptr = Entity;
         /* if start of qualifier ensure slash is copied */
         if (*cptr == '/') *eptr++ = *cptr++;
 
         continue;
      }
 
      /* any other character, just copy, ensure upper case */
      *eptr++ = toupper(*cptr++);
   }
 
   return (CommandLineOK);
}
 
/*****************************************************************************/
/*
Get a string value from a qualifier, e.g. '/EXAMPLE=TEST'.
*/
 
boolean ParseCommandString
(
char *Entity,
char *String,
boolean Qualifier,
boolean ReportErrors,
boolean EnsureUpperCase
)
{
   register int  QuoteCount = 0;
   register char  *eptr, *sptr;
 
   if (Debug) fprintf (stdout, "ParseCommandString()\nEntity: '%s'\n", Entity);
 
   eptr = Entity;
 
   if (Qualifier)
   {
      /* scan down to equate symbol */
      while (*eptr && *eptr != '=') eptr++;
      if (*eptr) eptr++;
      if (!*eptr)
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
            Utility, Entity+1);
         }
         return (false);
      }
   }
 
   sptr = String;
   while (*eptr)
   {
      if (*eptr == '\"')
      {
         if (QuoteCount & 1)
         {
            /* are inside quotes, check for escaped quotes ("") */
            if (*++eptr != '\"')
            {
               /* now outside quotes */
               QuoteCount++;
            }
            /* drop thru to character copy */
         }
         else
         {
            /* now inside quotes */
            QuoteCount++;
            eptr++;
            continue;
         }
      }
 
      if (EnsureUpperCase)
         *sptr++ = toupper(*eptr++);
      else
         *sptr++ = *eptr++;
   }
   *sptr = '\0';
 
   if (Debug) fprintf (stdout, "String: '%s'\n", String);
 
   return (true);
}
 
/*****************************************************************************/
/*
Get an integer value from a qualifier, e.g. '/EXAMPLE=99'.
*/
 
boolean ParseCommandInteger
(
char *Entity,
int *IntegerPtr,
int Base,
boolean ReportErrors
)
{
   register char  *eptr;
   char  *sptr;
 
   if (Debug)
      fprintf (stdout, "ParseCommandInteger() '%s' Base: %d\n", Entity, Base);
 
   for (eptr = Entity; *eptr && *eptr != '='; eptr++);
   if (*eptr) eptr++;
   if (*eptr)
   {
      *IntegerPtr = strtol (eptr, &sptr, Base);
      if (sptr > eptr && !*sptr)
         return (true);
      else
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-BADVALUE, '%s' is an invalid keyword value\n",
            Utility, eptr);
         }
         return (false);
      }
   }
   else
   {
      if (ReportErrors)
      {
         fprintf (stdout,
         "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
         Utility, Entity+1);
      }
      return (false);
   }
}
 
/*****************************************************************************/
/*
A single command line "entity" has been parsed, check if its recognised.  This 
function is the one modified for the individual requirements of each program.
*/
 
boolean ParseCommandEntity (char *Entity)
 
{
   if (Entity == NULL)
   {
      /* set up any argument defaults */
      DoToStreamLF = DoConfirm = DoLog = true;
      return (true);
   }
 
   if (Debug) fprintf (stdout, "ParseCommandEntity() Entity: '%s'\n", Entity);
 
   if (Entity[0] == '/')
   {
      if (strsame (Entity, "/CONFIRM", 4))
         return (DoConfirm = true);
      if (strsame (Entity, "/NOCONFIRM", 6))
      {
         DoConfirm = false;
         return (true);
      }

      /* turns on all "if (Debug)" statements */
      if (strsame (Entity, "/DBUG", -1))
         return (Debug = true);

      if (strsame (Entity, "/FROM", 5))
      {
         DoIdentify = DoToStreamLF = false;
         return (DoFromStreamLF = true);
      }

      if (strsame (Entity, "/HELP", 4))
         return (DoHelp = true);

      if (strsame (Entity, "/IDENTIFY", 3))
      {
         DoFromStreamLF = DoToStreamLF = false;
         return (DoIdentify = true);
      }

      if (strsame (Entity, "/PURGE", 4))
         return (DoPurge = true);

      if (strsame (Entity, "/TO", 3))
      {
         DoIdentify = DoFromStreamLF = DoConfirm = false;
         return (DoToStreamLF = true);
      }

      if (strsame (Entity, "/LOG", 4))
         return (DoLog = true);
      if (strsame (Entity, "/NOLOG", 6))
      {
         DoLog = false;
         return (true);
      }

      fprintf (stdout,
      "%%%s-E-IVQUAL, unrecognised qualifier\n \\%s\\\n", Utility, Entity+1);
      return (false);
   }
 
   if (!FileSpec[0])
      return (ParseCommandString (Entity, FileSpec, false, true, true));

   fprintf (stdout,
   "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n", Utility, Entity);
   return (false);
}
   
/****************************************************************************/
