/*****************************************************************************/
/*
                                WWWrkout.c

"Workout" utility for a WWW server.  Stresses a server both in number of 
simultaneous connections maintained and in the number of back-to-back 
sequential connection requests and data transfers.  This is probably not the
best test tool but it's better than nothing.  To really exercise a server it
should be used from multiple systems simultaneously.

Sets up and maintains a specified number of simultaneous connections (or until 
server limit is reached).  Is AST-driven, so it will generally push throughput
to whatever the server can supply. When the transfer is complete it closes that
connection at this end and uses the closed socket to initiate  another.  If
enabled (it is by default), the utility attempts to reflect the  real-world by
varying the data transfer rate for each connection, through  setting the number
of bytes read during each read loop differently for each  connection.  All data
transfered is discarded. If enabled will occasionally break a connection at
random times during transfer to exercise server recovery, etc.

The data transfer rate for each connection is displayed at connection close.  
It is by default the effective transfer rate, that is the rate from opening 
the connection to closing it, and so includes request processing time, etc., 
or if the /NOEFFECTIVE qualifier is emplaoyed measure the document data 
transfer rate only.

The document paths should be specified one per line in a plain text file.  
Preferably each document path and/or type specified should be different to the 
others, to exercise the server and file system cache.  Any number of paths may 
be specified in the file.  If the file is exhausted before the specified 
number of connections have been established the file contents are recycled 
from the first path.  If path or a file of paths is not specified the utility 
just continually requests the welcome document. 


QUALIFIERS
----------
/[NO]BREAK              break a connection occasionally (test server recovery)
/BUFFER=                maximum number bytes to be read from server each time
/COUNT=                 total number of connections and requests to be done
/DBUG                   turns on all "if (Debug)" statements
/[NO]EFFECTIVE          measures data transfer rate from request to close
/FILEOF=                file name containing paths to documents
/[NO]HEAD               mix HEAD and GET methods at pseudo-random
/HELP                   display brief usage information
/[NO]OUTPUT=            file name for output, no output suppresses only the
                        report for each connection, totals and errors still
/PATH=                  path to document to be retrieved
/PORT=                  IP port number of HTTP server
/SERVER=                HTTP server host name
/SIMULTANEOUS=          number of simultaneous connections at any one time
/[NO]VARY               varies the size of the read buffer for each connection


BUILD DETAILS
-------------
See BUILD_WWWRKOUT.COM


VERSION HISTORY (update SoftwareID as well!)
---------------
15-FEB-98  MGD  v1.5.1, allow comments in source files
29-NOV-97  MGD  v1.5.0, add HEAD method
08-OCT-97  MGD  v1.4.1, float time now 0.000001 not 0.0
11-SEP-97  MGD  v1.4.0, quit after a maximum connection errors
04-AUG-97  MGD  v1.3.1, bugfix; UCX gethostname() error detection
29-JUL-97  MGD  v1.3.0, redesign for being AST-driven and with NETLIB support
25-JUN-97  MGD  v1.2.0, broken connections to test server recovery
06-JAN-96  MGD  v1.1.0, Multinet compatibility
05-MAY-95  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "WWWRKOUT v1.5.1 AXP";
#else
   char SoftwareID [] = "WWWRKOUT v1.5.1 VAX";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <libdef.h>
#include <libdtdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* Internet-related header files */

#ifndef IP_NETLIB
#  define IP_UCX 1
#endif

#ifdef IP_UCX
#   include <socket.h>
#   include <in.h>
#   include <netdb.h>
#   include <inet.h>
#   include <ucx$inetdef.h>
#endif

#ifdef IP_NETLIB
#   include "netlibdef.h"
#endif

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

#define MaxConnections 256
#define MaxReadBufferSize 8192
#define DefaultServerPort 80
#define DefaultSpecifiedCount 100
#define DefaultSpecifiedSimultaneous 10
#define DefaultReadBufferSize 1024

/* lib$cvtf_from_internal_time() complains about about a zero elapsed time! */
#define LIB$_DELTIMREQ 1410052

struct AnIOsb
{
   unsigned short  Status;
   unsigned short  Count;
   unsigned long  Unused;
};

struct ConnectionStruct
{
   struct ConnectionStruct  *NextPtr;
#ifdef IP_UCX
   unsigned short  Channel;
#endif
#ifdef IP_NETLIB
   unsigned long  NetLibSocket;
#endif
   int  ByteCount,
        ConnectionNumber,
        ReadBufferSize;
   unsigned long  StatTimerContext;
   char  *MethodPtr;
   char  Path [256],
         ReadBuffer [MaxReadBufferSize],
         Request [512];
   struct AnIOsb  IOsb;
};

#define MAX_CONNECTION_ERROR_COUNT 20

char  Utility [] = "WWWRKOUT";

unsigned long  StatTimerElapsedTime = 1,
               CvtTimeFloatSeconds = LIB$K_DELTA_SECONDS_F;

boolean  BrokenConnections,
         Debug,
         DoShowHelp,
         HeadMethod,
         EffectiveTransferRate,
         NoOutput,
         VaryBufferSize;

int  BrokenConnectionCount,
     ConnectionErrorCount,
     ConnectionCount,
     ReadBufferSize,
     GetMethodCount,
     HeadMethodCount,
     HttpReadErrorCount,
     HttpRequestErrorCount,
     OutstandingConnections,
     ServerPort,
     SpecifiedCount,
     SpecifiedSimultaneous,
     StatusCodeErrorCount,
     TotalBytesReceived,
     TotalBytesSent;

int  StatusCodeCount [10];

FILE  *Output;

unsigned long  TotalStatTimerContext;

char  CommandLine [256],
      OutFileName [256],
      PathFileName [256],
      ServerHostName [256],
      SpecifiedPath [256];

struct ConnectionStruct  *ConnectionListPtr,
                         *LastConnectionPtr;

#ifdef IP_UCX

$DESCRIPTOR (InetDeviceDsc, "UCX$DEVICE");

int  OptionEnabled = 1;

struct hostent  *ServerHostEntryPtr;
struct sockaddr_in  ServerSocketName;

struct {
   unsigned int  Length;
   char  *Address;
} ServerSocketNameItem =
   { sizeof(ServerSocketName), &ServerSocketName };

struct {
   unsigned short  Length;
   unsigned short  Parameter;
   char  *Address;
} ReuseAddress =
   { sizeof(OptionEnabled), UCX$C_REUSEADDR, &OptionEnabled },
  ReuseAddressSocketOption =
   { sizeof(ReuseAddress), UCX$C_SOCKOPT, &ReuseAddress };

unsigned short  TcpSocket [2] =
   { UCX$C_TCP, INET_PROTYP$C_STREAM };

#endif /* IP_UCX */


#ifdef IP_NETLIB

unsigned long  NetLibSocketType = NETLIB_K_TYPE_STREAM,
               NetLibSocketFamily = NETLIB_K_AF_INET,
               NetLibLevel = NETLIB_K_LEVEL_SOCKET,
               NetLibReUseOption = NETLIB_K_OPTION_REUSEADDR,
               NetLibShutdown = NETLIB_K_SHUTDOWN_RECEIVER,
               NetLibOptionOn = 1,
               NetLibSizeofOptionOn = sizeof(unsigned long);

unsigned short  ServerPortWord;

$DESCRIPTOR (ServerHostNameDsc, ServerHostName);

#endif /* IP_NETLIB */

/* required function prototypes */
ConnectToServer (struct ConnectionStruct*);
ConnectToServerAst (struct ConnectionStruct*);
ReadResponseAst (struct ConnectionStruct*);
WriteRequestAst (struct ConnectionStruct*);
char* ErrorMessage (int);

/*****************************************************************************/
/*
*/

int main ()

{
   register char  *cptr;

   int  status;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSnok (status = ParseCommandLine ()))
      exit (status);

   if (DoShowHelp) exit (ShowHelp ());

   if (ReadBufferSize > MaxReadBufferSize)
   {
      fprintf (stdout, "%%%s-E-BUFFER, maximum buffer size is %d bytes\n",
               Utility, MaxReadBufferSize);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   if (!ServerHostName[0])
   {
      /* assume the local host is the server */
#ifdef IP_UCX
      if (gethostname (ServerHostName, sizeof(ServerHostName)))
         status = SS$_NOSUCHNODE;
#endif
#ifdef IP_NETLIB
      status = netlib_get_hostname (&ServerHostNameDsc, &Length);
      ServerHostName[Length] = '\0';
#endif
      if (VMSnok (status))
      {
         fprintf (stdout, "%%%s-E-SERVER, \"%s\" %s\n",
                  Utility, ServerHostName, ErrorMessage(status));
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      if (Debug) fprintf (stdout, "ServerHostName |%s|\n", ServerHostName);
   }
   else
   {
      for (cptr = ServerHostName; *cptr && *cptr != ':'; cptr++);
      if (*cptr)
      {
         *cptr++ = '\0';
         if (isdigit(*cptr))
            ServerPort = atoi(cptr);
      }
   }

#ifdef IP_UCX
   /* get the server host address details */
   if ((ServerHostEntryPtr = gethostbyname (ServerHostName)) == NULL)
   {
      fprintf (stdout, "%%%s-E-SERVER, \"%s\" %s\n",
               Utility, ServerHostName, ErrorMessage(status = vaxc$errno));
      exit (status | STS$M_INHIB_MSG);
   }
   ServerSocketName.sin_family = ServerHostEntryPtr->h_addrtype;
   ServerSocketName.sin_port = htons (ServerPort);
   ServerSocketName.sin_addr = *((struct in_addr *)ServerHostEntryPtr->h_addr);

   strcpy (ServerHostName, ServerHostEntryPtr->h_name);
#endif

#ifdef IP_NETLIB
   ServerPortWord = ServerPort;
   ServerHostNameDsc.dsc$w_length = strlen(ServerHostName);
#endif

   /* looks better if its all in lower case (host name is sometimes upper) */
   for (cptr = ServerHostName; *cptr; cptr++) *cptr = tolower(*cptr);
   if (Debug) fprintf (stdout, "ServerHostName |%s|\n", ServerHostName);

   if (OutFileName[0])
   {
      fclose (stdout);
      if ((stdout = fopen (OutFileName, "w")) == NULL)
         exit (vaxc$errno);
   }
   else
   if (!NoOutput)
      Output = stdout;

   ReportParameters ();

   GenerateConnections ();

   ReportTotals ();

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/
 
ReportParameters ()

{
   fprintf (stdout,
      "\n$ WWWRKOUT /SERVER=%s /PORT=%d /SIMULTANEOUS=%d /COUNT=%d /BUFFER=%d",
       ServerHostName, ServerPort, SpecifiedSimultaneous,
       SpecifiedCount, ReadBufferSize);
   if (VaryBufferSize)
      fprintf (stdout, " /VARY");
   else
      fprintf (stdout, " /NOVARY");
   if (BrokenConnections)
      fprintf (stdout, " /BREAK");
   else
      fprintf (stdout, " /NOBREAK");
   if (EffectiveTransferRate)
      fprintf (stdout, " /EFFECTIVE");
   else
      fprintf (stdout, " /NOEFFECTIVE");
   fprintf (stdout, "\n");
}

/*****************************************************************************/
/*
*/
 
ReportTotals ()

{
   int  status,
        Code;
   unsigned long  ElapsedBinTime [2];
   unsigned short  Length;
   float  ElapsedFloatTime;
   char  ElapsedTime [32];
   $DESCRIPTOR (ElapsedTimeFaoDsc, "!%T");
   $DESCRIPTOR (ElapsedTimeDsc, ElapsedTime);

   /*********/
   /* begin */
   /*********/

   if (VMSnok (status =
       lib$stat_timer (&StatTimerElapsedTime,
                       &ElapsedBinTime,
                       &TotalStatTimerContext)))
      exit (status);

   if (VMSnok (status =
       lib$cvtf_from_internal_time (&CvtTimeFloatSeconds,
                                    &ElapsedFloatTime,
                                    &ElapsedBinTime)))
      exit (status);

   if (VMSnok (status =
       sys$fao (&ElapsedTimeFaoDsc, &Length, &ElapsedTimeDsc,
                &ElapsedBinTime)))
      exit (status);
   ElapsedTime[Length] = '\0';

   fprintf (stdout,
"\n\
Summary;   Elapsed: %s  Connections: %d  Deliberately Broken: %d\n\
Methods;   GET: %d  HEAD: %d\n\
Bytes;     Tx: %d  Rx: %d (%.0f/S)\n\
Failures;  Connect %d  Request %d  Read %d\n",
   ElapsedTime, ConnectionCount, BrokenConnectionCount,
   GetMethodCount, HeadMethodCount,
   TotalBytesSent, TotalBytesReceived,
   (float)TotalBytesReceived / ElapsedFloatTime,
   ConnectionErrorCount, HttpRequestErrorCount, HttpReadErrorCount);

   fprintf (stdout, "Status;    Errors: %d", StatusCodeErrorCount);
   for (Code = 0; Code <= 9; Code++)
      if (StatusCodeCount[Code])
         fprintf (stdout, "  %dXX: %d", Code, StatusCodeCount[Code]);
   fprintf (stdout, "\n");
}

/*****************************************************************************/
/*
Loop through the specified number of connections (or attempts thereof).  
Maintain, as best it can be, the specified number of simultaneous connections.  
If a connection attempt fails just quit there for that particular loop, leave 
a bit of time (as reads occur) to allow the server to recover.
*/
 
int GenerateConnections ()

{
   register struct ConnectionStruct  *cxptr;

   int  status,
        Count;

   /*********/
   /* begin */
   /*********/

   /* dynamically allocate the specified number of connection structures */
   for (Count = 0; Count < SpecifiedSimultaneous; Count++)
   {
      if ((cxptr = calloc (1, sizeof(struct ConnectionStruct))) == NULL)
         exit (vaxc$errno);
      if (Debug) fprintf (stdout, "cxptr: %d\n", cxptr);
      cxptr->NextPtr = NULL;
      if (ConnectionListPtr == NULL)
         ConnectionListPtr = cxptr;
      else
         LastConnectionPtr->NextPtr = cxptr;
      LastConnectionPtr = cxptr;
   }

   /* initialize statistics timer */
   if (VMSnok (status = lib$init_timer (&TotalStatTimerContext)))
      exit (status);

   /******************************************************/
   /* loop for the total number of connections specified */
   /******************************************************/

   ConnectionErrorCount = OutstandingConnections = ConnectionCount =
      GetMethodCount = HeadMethodCount = 0;

   sys$setast (0);

   for (cxptr = ConnectionListPtr; cxptr != NULL; cxptr = cxptr->NextPtr)
       sys$dclast (ConnectToServer, cxptr, 0);

   sys$setast (1);

   sys$hiber ();
}

/*****************************************************************************/
/*
Loop through the specified number of connections (or attempts thereof).  
Maintain, as best it can be, the specified number of simultaneous connections.  
If a connection attempt fails just quit there for that particular loop, leave 
a bit of time (as reads occur) to allow the server to recover.
*/
 
int ConnectToServer (struct ConnectionStruct *cxptr)

{
   int  status;
   unsigned short  NumTime [7];
   unsigned long  BinTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConnectToServer() cxptr: %d\n", cxptr);

   if (ConnectionCount >= SpecifiedCount)
   {
      if (!OutstandingConnections) sys$wake (0, 0);
      return;
   }
   ConnectionCount++;
   OutstandingConnections++;

   memset (cxptr, 0, sizeof(struct ConnectionStruct));
   cxptr->ConnectionNumber = ConnectionCount;
   cxptr->ByteCount = 0;

   if (VaryBufferSize)
   {
      /* adjust the buffer size using a pseudo-random value */
      sys$gettim (&BinTime);
      sys$numtim (&NumTime, &BinTime);
      cxptr->ReadBufferSize = ReadBufferSize / ((NumTime[6] & 3) + 1);
   }
   else
      cxptr->ReadBufferSize = ReadBufferSize;

#ifdef IP_UCX
      cxptr->Channel = 0;
#endif
#ifdef IP_NETLIB
      cxptr->NetLibSocket = 0;
#endif

   /* initialize timer */
   if (VMSnok (status = lib$init_timer (&cxptr->StatTimerContext)))
      exit (status);

#ifdef IP_UCX
   /* assign a channel to the internet template device */
   if (VMSnok (status = sys$assign (&InetDeviceDsc, &cxptr->Channel, 0, 0)))
   {
      fprintf (stdout, "assign() error; #%d, %s\n",
               ConnectionCount, ErrorMessage(status));
      cxptr->Channel = 0;
      exit (status);
   }

   /* make the channel a TCP, connection-oriented socket */

   if (Debug) fprintf (stdout, "sys$qiow() IO$_SETMODE\n");
   status = sys$qiow (0, cxptr->Channel, IO$_SETMODE, &cxptr->IOsb, 0, 0,
                      &TcpSocket, 0, 0, 0, &ReuseAddressSocketOption, 0);

   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb %%X%08.08X\n",
               status, cxptr->IOsb.Status);
   if (VMSok (status) && VMSnok (cxptr->IOsb.Status))
      status = cxptr->IOsb.Status;
   if (VMSnok (status)) exit (status);

   status = sys$qio (0, cxptr->Channel, IO$_ACCESS, &cxptr->IOsb,
                     &ConnectToServerAst, cxptr,
                     0, 0, &ServerSocketNameItem, 0, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb %%X%08.08X\n",
               status, cxptr->IOsb.Status);
#endif

#ifdef IP_NETLIB
   if (Debug) fprintf (stdout, "netlib_socket()\n");
   status = netlib_socket (&cxptr->NetLibSocket, &NetLibSocketType,
                           &NetLibSocketFamily);
   if (Debug) fprintf (stdout, "netlib_socket() %%X%08.08X\n", status);
   if (VMSnok (status)) exit (status);

   status = netlib_setsockopt (&cxptr->NetLibSocket, &NetLibLevel,
                               &NetLibReUseOption,
                               &NetLibOptionOn, &NetLibSizeofOptionOn,
                               &cxptr->IOsb, 0, 0);
   if (Debug)
      fprintf (stdout, "netlib_setsockopt() %%X%08.08X IOsb %%X%08.08X\n",
               status, cxptr->IOsb.Status);
   if (VMSok (status) && VMSnok (cxptr->IOsb.Status))
      status = cxptr->IOsb.Status;
   if (VMSnok (status)) exit (status);

   if (Debug) fprintf (stdout, "netlib_connect()\n");
   status = netlib_connect_by_name (&cxptr->NetLibSocket,
                                    &ServerHostNameDsc, &ServerPortWord,
                                    &cxptr->IOsb, ConnectToServerAst, cxptr);
   if (Debug) fprintf (stdout, "netlib_connect_by_name() %%X%08.08X\n", status);
   if (VMSnok (status)) exit (status);
#endif
}

/*****************************************************************************/
/*
Connection to server has completed, either successfully or unsuccessfully!
*/
 
int ConnectToServerAst (struct ConnectionStruct *cxptr)

{
#ifdef IP_NETLIB
   static $DESCRIPTOR (DataDsc, "");
#endif

   int  status,
        RequestLength;
   unsigned short  CurrentNumTime [7];
   unsigned long  CurrentBinTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ConnectToServerAst() IOsb %%X%08.08X\n",
               cxptr->IOsb.Status);

   status = cxptr->IOsb.Status;

   if (VMSnok (status))
   {
      /*****************/
      /* connect error */
      /*****************/

      fprintf (Output, "connect error; #%d %s\n",
               cxptr->ConnectionNumber, ErrorMessage(status));
      HttpRequestErrorCount++;
      if (ConnectionErrorCount++ > MAX_CONNECTION_ERROR_COUNT)
      {
         ReportTotals ();
         exit (SS$_NORMAL);
      }

      /* free timer */
      if (VMSnok (status = lib$free_timer (&cxptr->StatTimerContext)))
         exit (status);

      /* close current then re-initiate a new connection */
      CloseConnection (cxptr);
      ConnectToServer (cxptr);

      return;
   }

   /****************/
   /* send request */
   /****************/

   sys$gettim (&CurrentBinTime);
   sys$numtim (&CurrentNumTime, &CurrentBinTime);

   /* create and send a moderate size HTTP request */
   GetPath (cxptr->Path, sizeof(cxptr->Path));

   if (HeadMethod && !(CurrentNumTime[6] % 10))
   {
      HeadMethodCount++;
      cxptr->MethodPtr = "HEAD";
      RequestLength = sprintf (cxptr->Request,
"HEAD %s HTTP/1.0\r\n\
User-Agent: %s\r\n\
Accept: */*\r\n\
\r\n",
         cxptr->Path, SoftwareID);
      if (Debug) fprintf (stdout, "cxptr->Request |%s|\n", cxptr->Request);
   }
   else
   {
      GetMethodCount++;
      cxptr->MethodPtr = "GET";
      RequestLength = sprintf (cxptr->Request,
"GET %s HTTP/1.0\r\n\
User-Agent: %s\r\n\
Accept: */*\r\n\
\r\n",
         cxptr->Path, SoftwareID);
      if (Debug) fprintf (stdout, "cxptr->Request |%s|\n", cxptr->Request);
   }

   TotalBytesSent += RequestLength;

#ifdef IP_UCX
   status = sys$qio (0, cxptr->Channel,
                     IO$_WRITEVBLK, &cxptr->IOsb, WriteRequestAst, cxptr,
                     cxptr->Request, RequestLength, 0, 0, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qio() %%X%08.08X IOsb: %%X%08.08X\n",
               status, cxptr->IOsb.Status);
#endif

#ifdef IP_NETLIB
   DataDsc.dsc$a_pointer = cxptr->Request;
   DataDsc.dsc$w_length = RequestLength;

   status = netlib_write (&cxptr->NetLibSocket, &DataDsc, 0, 0,
                          &cxptr->IOsb, WriteRequestAst, cxptr);

   if (Debug)
      fprintf (stdout, "netlib_write() %%X%08.08X IOsb: %%X%08.08X\n",
               status, cxptr->IOsb.Status);
#endif
}

/*****************************************************************************/
/*
*/
 
int WriteRequestAst (struct ConnectionStruct *cxptr)

{
#ifdef IP_NETLIB
   static $DESCRIPTOR (DataDsc, "");
#endif

   int  status,
        BytesPerSecond;
   unsigned short  CurrentNumTime [7];
   unsigned long  CurrentBinTime [2],
                  ElapsedBinTime [2];
   float  ElapsedFloatTime;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "WriteRequestAst() IOsb: %%X%08.08X\n",
               cxptr->IOsb.Status);

   status = cxptr->IOsb.Status;

   if (VMSnok (status))
   {
      /***************/
      /* write error */
      /***************/

      fprintf (Output, "write error; #%d \"%s\", %s\n",
               cxptr->ConnectionNumber, cxptr->Path, ErrorMessage(status));
      HttpRequestErrorCount++;

      /* free timer */
      if (VMSnok (status = lib$free_timer (&cxptr->StatTimerContext)))
         exit (status);

      /* close current then re-initiate a new connection */
      CloseConnection (cxptr);
      ConnectToServer (cxptr);

      return;
   }

   /*****************/
   /* read response */
   /*****************/

#ifdef IP_UCX
      status = sys$qio (0, cxptr->Channel,
                         IO$_READVBLK, &cxptr->IOsb,
                         ReadResponseAst, cxptr,
                         cxptr->ReadBuffer, cxptr->ReadBufferSize, 0, 0, 0, 0);

      if (Debug)
         fprintf (stdout, "sys$qio() %%X%08.08X IOsb: %%X%08.08X\n",
                  status, cxptr->IOsb.Status);
#endif

#ifdef IP_NETLIB
      DataDsc.dsc$a_pointer = cxptr->ReadBuffer;
      DataDsc.dsc$w_length = cxptr->ReadBufferSize;

      status = netlib_read (&cxptr->NetLibSocket, &DataDsc,
                            0, 0, 0, 0,
                            &cxptr->IOsb, ReadResponseAst, cxptr);

      if (Debug)
         fprintf (stdout, "netlib_read() %%X%08.08X IOsb: %%X%08.08X\n",
                  status, cxptr->IOsb.Status);
#endif
}

/*****************************************************************************/
/*
A read from the server has completed.  Check it's status.  If complete
(disconnected) then report.  If not complete the queue to read more.
*/
 
int ReadResponseAst (struct ConnectionStruct *cxptr)

{
#ifdef IP_NETLIB
   static $DESCRIPTOR (DataDsc, "");
#endif

   static boolean  OneConnectionBroken;

   register char  *cptr, *sptr, *zptr;

   int  status,
        BytesPerSecond;
   unsigned short  CurrentNumTime [7];
   unsigned long  CurrentBinTime [2],
                  ElapsedBinTime [2];
   float  ElapsedFloatTime;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ReadResponseAst() IOsb: %%X%08.08X Bytes: %d\n",
               cxptr->IOsb.Status, cxptr->IOsb.Count);

   status = cxptr->IOsb.Status;

   if (VMSnok (status))
   {
      /*********/
      /* error */
      /*********/

      if (status == SS$_LINKDISCON)
      {
         /****************************/
         /* server closed connection */
         /****************************/

         if (VMSnok (status =
             lib$stat_timer (&StatTimerElapsedTime,
                             &ElapsedBinTime,
                             &cxptr->StatTimerContext)))
            exit (status);

         if (VMSnok (status =
             lib$cvtf_from_internal_time (&CvtTimeFloatSeconds,
                                          &ElapsedFloatTime,
                                          &ElapsedBinTime)))
         {
            if (status == LIB$_DELTIMREQ)
               ElapsedFloatTime = 0.000001;
            else
               exit (status);
         }

         BytesPerSecond = (int)((float)cxptr->ByteCount / ElapsedFloatTime);

         fprintf (Output,
         "end;    #%d \"%s %s\", %d bytes in %.1f seconds (%d/S, size: %d)\n",
            cxptr->ConnectionNumber, cxptr->MethodPtr, cxptr->Path,
            cxptr->ByteCount,
            ElapsedFloatTime, BytesPerSecond, cxptr->ReadBufferSize);
      }
      else
      {
         /**************/
         /* read error */
         /**************/

         fprintf (Output, "read error; #%d \"%s\", after %d bytes, %s\n",
                  cxptr->ConnectionNumber, cxptr->Path,
                  cxptr->ByteCount, ErrorMessage(status));
         HttpReadErrorCount++;
      }

      TotalBytesReceived += cxptr->ByteCount;

      /* free timer */
      if (VMSnok (status = lib$free_timer (&cxptr->StatTimerContext)))
         exit (status);

      /* close current then re-initiate a new connection */
      CloseConnection (cxptr);
      ConnectToServer (cxptr);

      return;
   }

   sys$gettim (&CurrentBinTime);
   sys$numtim (&CurrentNumTime, &CurrentBinTime);

   if (BrokenConnections && CurrentNumTime[6] == 5 && !OneConnectionBroken)
   {
      /*********************************/
      /* deliberately break connection */
      /*********************************/

      if (VMSnok (status =
          lib$stat_timer (&StatTimerElapsedTime,
                          &ElapsedBinTime,
                          &cxptr->StatTimerContext)))
         exit (status);

      if (VMSnok (status =
          lib$cvtf_from_internal_time (&CvtTimeFloatSeconds,
                                       &ElapsedFloatTime,
                                       &ElapsedBinTime)))
      {
         if (status == LIB$_DELTIMREQ)
            ElapsedFloatTime = 0.000001;
         else
            exit (status);
      }

      fprintf (Output,
"end;    #%d \"%s %s\", DELIBERATELY BROKEN after %.1f seconds and %d bytes\n",
         cxptr->ConnectionNumber, cxptr->MethodPtr, cxptr->Path,
         ElapsedFloatTime, cxptr->ByteCount);

      OneConnectionBroken = true;
      BrokenConnectionCount++;

      /* free timer */
      if (VMSnok (status = lib$free_timer (&cxptr->StatTimerContext)))
         exit (status);

      /* close current then re-initiate a new connection */
      CloseConnection (cxptr);
      ConnectToServer (cxptr);

      return;
   }

   if (BrokenConnections && CurrentNumTime[6] && OneConnectionBroken)
      OneConnectionBroken = false;

   if (!cxptr->ByteCount)
   {
      /************************/
      /* HTTP response header */
      /************************/

      if (VMSnok (status =
          lib$stat_timer (&StatTimerElapsedTime,
                          &ElapsedBinTime,
                          &cxptr->StatTimerContext)))
         exit (status);

      if (VMSnok (status =
          lib$cvtf_from_internal_time (&CvtTimeFloatSeconds,
                                       &ElapsedFloatTime,
                                       &ElapsedBinTime)))
      {
         if (status == LIB$_DELTIMREQ)
            ElapsedFloatTime = 0.000001;
         else
            exit (status);
      }

      /* (assumes first line is all in buffer!) */
      zptr = (cptr = cxptr->ReadBuffer) + cxptr->IOsb.Count;
      while (cptr < zptr && *cptr != '\r' && *cptr != '\n') cptr++;
      if (cptr < zptr)
      {
         *cptr = '\0';

         /* skip over "HTTP/n.n" and spaces to get 3 digit status code */
         cptr = cxptr->ReadBuffer;
         while (*cptr && !isspace(*cptr)) cptr++;
         while (isspace(*cptr)) cptr++;
         /* increment the counter corresponding to the first digit */
         if (isdigit(*cptr))
            StatusCodeCount[*cptr-'0']++;
         else
            StatusCodeErrorCount++;

         fprintf (Output, "header; #%d \"%s %s\", in %.1f seconds \"%s\"\n",
                  cxptr->ConnectionNumber, cxptr->MethodPtr, cxptr->Path,
                  ElapsedFloatTime, cxptr->ReadBuffer);
      }
      else
      {
         fprintf (Output,
"header error (buffer too small); #%d \"%s\", in %.1f seconds\n",
         cxptr->ConnectionNumber, cxptr->Path, ElapsedFloatTime);
      }
   }

   cxptr->ByteCount += cxptr->IOsb.Count;

   /**********************/
   /* read more response */
   /**********************/

#ifdef IP_UCX
      status = sys$qio (0, cxptr->Channel,
                         IO$_READVBLK, &cxptr->IOsb,
                         ReadResponseAst, cxptr,
                         cxptr->ReadBuffer, cxptr->ReadBufferSize, 0, 0, 0, 0);

      if (Debug)
         fprintf (stdout, "sys$qio() %%X%08.08X IOsb: %%X%08.08X\n",
                  status, cxptr->IOsb.Status);
#endif

#ifdef IP_NETLIB
      DataDsc.dsc$a_pointer = cxptr->ReadBuffer;
      DataDsc.dsc$w_length = cxptr->ReadBufferSize;

      status = netlib_read (&cxptr->NetLibSocket, &DataDsc,
                            0, 0, 0, 0,
                            &cxptr->IOsb, ReadResponseAst, cxptr);

      if (Debug)
         fprintf (stdout, "netlib_read() %%X%08.08X IOsb: %%X%08.08X\n",
                  status, cxptr->IOsb.Status);
#endif
}

/****************************************************************************/
/*
*/

CloseConnection (struct ConnectionStruct *cxptr)

#ifdef IP_UCX
{
   int  status;
   struct  AnIOsb  IOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "NetCloseConnection() %d\n", cxptr->Channel);

   status = sys$qiow (0, cxptr->Channel, IO$_DEACCESS | IO$M_SHUTDOWN,
                      &IOsb, 0, 0, 0, 0, 0, 0, 0, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%X%08.08X IOsb %%X%08.08X\n",
               status, IOsb.Status);

   status = sys$dassgn (cxptr->Channel);
   if (Debug) fprintf (stdout, "sys$dassgn() %%X%08.08X\n", status);

   cxptr->Channel = 0;
   if (OutstandingConnections) OutstandingConnections--;
}
#endif

#ifdef IP_NETLIB
{
   int  status;
   struct  AnIOsb  IOsb;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "NetCloseConnection() cxptr->NetLibSocket: %d\n",
               cxptr->NetLibSocket);

   status = netlib_shutdown (&cxptr->NetLibSocket, &NetLibShutdown,
                             &IOsb, 0, 0);
   if (Debug)
      fprintf (stdout, "netlib_shutdown %%X%08.08X IOsb %%X%08.08X\n",
               status, IOsb.Status);

   status = netlib_close (&cxptr->NetLibSocket);
   if (Debug) fprintf (stdout, "netlib_close() %%X%08.08X\n", status);

   cxptr->NetLibSocket = 0;
   if (OutstandingConnections) OutstandingConnections--;
}
#endif

/*****************************************************************************/
/*
*/
 
int GetPath
(
char *PathPtr,
int SizeOfPath
)
{
   static FILE  *PathFile = NULL;

   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetPath()\n");

   if (SpecifiedPath[0])
   {
      strcpy (PathPtr, SpecifiedPath);
      return;
   }

   if (!PathFileName[0])
   {
      /* if no file name has been specified get the server welcome document */
      PathPtr[0] = '/';
      PathPtr[1] = '\0';
      return;
   }

   if (PathFile == NULL)
      if ((PathFile = fopen (PathFileName, "r")) == NULL)
         exit (vaxc$errno);

   *PathPtr = '\0';
   for (;;)
   {
      if (fgets (PathPtr, 256, PathFile) == NULL)
      {
         rewind (PathFile);
         if (fgets (PathPtr, SizeOfPath, PathFile) == NULL)
         exit (vaxc$errno);
      }
      if (*PathPtr != '#' && *PathPtr != '!') break;
   }

   for (cptr = PathPtr; *cptr && *cptr != '\n'; cptr++);
   *cptr = '\0';
   if (Debug) fprintf (stdout, "PathPtr |%s|\n", PathPtr);
}

/*****************************************************************************/
/*
*/
 
int ShowHelp ()
 
{
   fprintf (stdout,
"%%%s-I-HELP, usage for the WWWRKOUT utility (%s)\n\
\n\
Test utility for HTTP server.  Sets up and maintains a specified number of\n\
concurrent connections.  Reads a buffer of data from each connection in\n\
turn until the document requested is exhausted.  Then closes that connection\n\
and initiates another.  The document paths should be specified one per line\n\
in a plain text file, each path and/or type specified should be different to\n\
the others.   Any number of paths may be specified in the file with the\n\
paths continually recycled from the first.  If a file of paths is not\n\
specified the utility just continually requests the welcome document.\n\
\n\
$ WWWRKOUT [server_host_name[:port]] [document_path] [qualifiers ...]\n\
\n\
/[NO]BREAK(d) /BUFFER=integer /COUNT=integer /[NO]EFFECTIVE(d) /[NO]HEAD(d)\n\
/HELP /[NO]OUTPUT=file_name /PATH=document_path /PORT=integer\n\
/SERVER=host_name /SIMULTANEOUS=integer /FILEOF=file_name /[NO]VARY(d)\n\
\n\
Usage examples:\n\
$ WWWRKOUT\n\
$ WWWRKOUT WWW.SERVER.HOST:8080\n\
$ WWWRKOUT /SERVER=WWW.SERVER.HOST /PORT=8080 /FILEOF=PATHS.TXT\n\
$ WWWRKOUT WWW.SERVER.HOST /FILEOF=PATHS.TXT /COUNT=500 /SIMUL=20 /BUFFER=512\n\
\n",
   Utility, SoftwareID);
 
   return (SS$_NORMAL);
}
 
/*****************************************************************************/
/*
*/

char* ErrorMessage (int VmsStatus)

{
   static char  Message [256];

   short int  Length;
   $DESCRIPTOR (MessageDsc, Message);

   /*********/
   /* begin */
   /*********/

   Message[0] = '\0';
   if (VmsStatus)
   {
      /* text component only */
      sys$getmsg (VmsStatus, &Length, &MessageDsc, 1, 0);
      Message[Length] = '\0';
   }
   if (Message[0])
      return (Message);
   else
      return ("(internal error)");
}
 
/*****************************************************************************/
/*
*/

char* SysGetMsg (int StatusValue)

{
   static char  Message [256];
   register int  status;
   short int  Length;
   $DESCRIPTOR (MessageDsc, Message);

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 0, 0))) 
      Message[Length] = '\0';
   else
      sprintf (Message, "%%sys$getmsg() %%X%08.08X", status);

   return (Message);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 
 
boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Get the 
entire command line following the verb that activated the image.  The command 
line is returned in uppercase, space compressed (i.e. maximum of one space 
between text elements, trimmed of leading and trailing spaces).  Returns a 
warning status if there were no parameters/qualifiers on the command line.
The variable CommandLine is global.
*/ 
 
int ParseCommandLine ()
 
{
   int  status;
   unsigned short  Length;
   unsigned long  Flags = 0;
   struct dsc$descriptor_s 
          CommandLineDsc = { sizeof(CommandLine)-1, DSC$K_DTYPE_T,
                             DSC$K_CLASS_S, CommandLine };
 
   /* get the entire command line following the verb */
   if (VMSnok (status = lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
      return (status);
   CommandLine[Length] = '\0';
 
   if (ParseCommand (CommandLine))
      return (SS$_NORMAL);
   else
      return (STS$K_ERROR | STS$M_INHIB_MSG);
}
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Quoted 
strings are always indicated by being parsed to include a single leading 
quote.
*/ 
 
boolean ParseCommand (char *CommandLine)
 
{
   register int  QuoteCount = 0;
   register char  *cptr, *eptr;
   boolean  CommandLineOK = true;
   char  Entity [256] = "";
 
   /* set up any argument defaults */
   ParseCommandEntity (NULL);
 
   cptr = CommandLine;
   eptr = Entity;
 
   for (;;)
   {
      if (*cptr == '\"')
      {
         QuoteCount++;
         *eptr++ = *cptr++;
         continue;
      }
 
      if (QuoteCount & 1 && *cptr)
      {
         /* inside quoted text, copy all characters as literals */
         *eptr++ = *cptr++;
         continue;
      }
 
      if (*cptr == '/' || isspace (*cptr) || !*cptr)
      {
         if (isspace (*cptr))
         {
            /* span the white space */
            while (*cptr && isspace (*cptr)) cptr++;
            if (*cptr == '=')
            {
               /* part of a qualifier, continue to get the value */
               *eptr++ = *cptr++;
               /* span any intervening white space */
               while (*cptr && isspace (*cptr)) cptr++;
               continue;
            }
         }
 
         if (Entity[0])
         {
            *eptr = '\0';
            if (!ParseCommandEntity (Entity)) CommandLineOK = false;
         }
 
         /* if end of command line then break from loop */
         if (!*cptr) break;
 
         /* start of new entity */
         eptr = Entity;
         /* if start of qualifier ensure slash is copied */
         if (*cptr == '/') *eptr++ = *cptr++;
 
         continue;
      }
 
      /* any other character, just copy, ensure upper case */
      *eptr++ = toupper(*cptr++);
   }
 
   return (CommandLineOK);
}
 
/*****************************************************************************/
/*
Get a string value from a qualifier, e.g. '/EXAMPLE=TEST'.
*/
 
boolean ParseCommandString
(
char *Entity,
char *String,
boolean Qualifier,
boolean ReportErrors,
boolean EnsureUpperCase
)
{
   register int  QuoteCount = 0;
   register char  *eptr, *sptr;
 
   if (Debug) fprintf (stdout, "ParseCommandString()\nEntity: '%s'\n", Entity);
 
   eptr = Entity;
 
   if (Qualifier)
   {
      /* scan down to equate symbol */
      while (*eptr && *eptr != '=') eptr++;
      if (*eptr) eptr++;
      if (!*eptr)
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
            Utility, Entity+1);
         }
         return (false);
      }
   }
 
   sptr = String;
   while (*eptr)
   {
      if (*eptr == '\"')
      {
         if (QuoteCount & 1)
         {
            /* are inside quotes, check for escaped quotes ("") */
            if (*++eptr != '\"')
            {
               /* now outside quotes */
               QuoteCount++;
            }
            /* drop thru to character copy */
         }
         else
         {
            /* now inside quotes */
            QuoteCount++;
            eptr++;
            continue;
         }
      }
 
      if (EnsureUpperCase)
         *sptr++ = toupper(*eptr++);
      else
         *sptr++ = *eptr++;
   }
   *sptr = '\0';
 
   if (Debug) fprintf (stdout, "String: '%s'\n", String);
 
   return (true);
}
 
/*****************************************************************************/
/*
Get an integer value from a qualifier, e.g. '/EXAMPLE=99'.
*/
 
boolean ParseCommandInteger
(
char *Entity,
int *IntegerPtr,
int Base,
boolean ReportErrors
)
{
   register char  *eptr;
   char  *sptr;
 
   if (Debug)
      fprintf (stdout, "ParseCommandInteger() '%s' Base: %d\n", Entity, Base);
 
   for (eptr = Entity; *eptr && *eptr != '='; eptr++);
   if (*eptr) eptr++;
   if (*eptr)
   {
      *IntegerPtr = strtol (eptr, &sptr, Base);
      if (sptr > eptr && !*sptr)
         return (true);
      else
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-BADVALUE, '%s' is an invalid keyword value\n",
            Utility, eptr);
         }
         return (false);
      }
   }
   else
   {
      if (ReportErrors)
      {
         fprintf (stdout,
         "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
         Utility, Entity+1);
      }
      return (false);
   }
}
 
/*****************************************************************************/
/*
A single command line "entity" has been parsed, check if its recognised.  This 
function is the one modified for the individual requirements of each program.
*/
 
boolean ParseCommandEntity (char *Entity)
 
{
   if (Entity == NULL)
   {
      /* set up any argument defaults */
      Debug = DoShowHelp = false;
      BrokenConnections = EffectiveTransferRate = HeadMethod =
         VaryBufferSize = true;
      ReadBufferSize = DefaultReadBufferSize;
      ServerPort = DefaultServerPort;
      SpecifiedSimultaneous = DefaultSpecifiedSimultaneous;
      SpecifiedCount = DefaultSpecifiedCount;
      OutFileName[0] = PathFileName[0] = ServerHostName[0] = '\0';
      return (true);
   }
 
   if (Debug) fprintf (stdout, "ParseCommandEntity() Entity: '%s'\n", Entity);
 
   if (Entity[0] == '/')
   {
      if (strsame (Entity, "/BREAK", 4))
         return (BrokenConnections = true);
      if (strsame (Entity, "/NOBREAK", 6))
      {
         BrokenConnections = false;
         return (true);
      }

      if (strsame (Entity, "/BUFFER=", 4))
         return (ParseCommandInteger (Entity, &ReadBufferSize, 10, true));

      if (strsame (Entity, "/COUNT=", 4))
         return (ParseCommandInteger (Entity, &SpecifiedCount, 10, true));

      /* turns on all "if (Debug)" statements */
      if (strsame (Entity, "/DBUG", -1))
         return (Debug = true);

      if (strsame (Entity, "/EFFECTIVE", 4))
         return (EffectiveTransferRate = true);
      if (strsame (Entity, "/NOEFFECTIVE", 6))
      {
         EffectiveTransferRate = false;
         return (true);
      }

      if (strsame (Entity, "/HEAD", 4))
         return (HeadMethod = true);
      if (strsame (Entity, "/NOHEAD", 6))
      {
         HeadMethod = false;
         return (true);
      }

      if (strsame (Entity, "/HELP", 4))
         return (DoShowHelp = true);

      if (strsame (Entity, "/PATH=", 4))
         return (ParseCommandString (Entity, SpecifiedPath, true, true, true));

      if (strsame (Entity, "/PORT=", 4))
         return (ParseCommandInteger (Entity, &ServerPort, 10, true));

      if (strsame (Entity, "/SERVER=", 4))
         return (ParseCommandString (Entity, ServerHostName, true, true, true));

      if (strsame (Entity, "/SIMULTANEOUS=", 4))
         return (ParseCommandInteger (Entity, &SpecifiedSimultaneous, 10, true));

      if (strsame (Entity, "/FILEOF=", 4))
         return (ParseCommandString (Entity, PathFileName, true, true, true));

      if (strsame (Entity, "/OUTPUT=", 4))
      {
         NoOutput = false;
         return (ParseCommandString (Entity, OutFileName, true, true, true));
      }
      if (strsame (Entity, "/NOOUTPUT", 6))
         return (NoOutput = true);

      if (strsame (Entity, "/VARY", 4))
         return (VaryBufferSize = true);
      if (strsame (Entity, "/NOVARY", 6))
      {
         VaryBufferSize = false;
         return (true);
      }

      fprintf (stdout,
      "%%%s-E-IVQUAL, unrecognised qualifier\n \\%s\\\n", Utility, Entity+1);
      return (false);
   }
 
   if (!ServerHostName[0])
   {
      ParseCommandString (Entity, ServerHostName, false, false, false);
      return (true);
   }
   if (!SpecifiedPath[0])
   {
      ParseCommandString (Entity, SpecifiedPath, false, false, false);
      return (true);
   }

   fprintf (stdout,
   "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n", Utility, Entity);
   return (false);
}
   
/*****************************************************************************/

