/*****************************************************************************/
/*
                                 dBIV.c


CGI-compliant script to query dBaseIV database files under VMS.

Database data structures are based on information from appendices of the 
"dBASEIV Language Reference" manual. 

Can provide tables layed out within a fixed-font, or using HTML <TABLE>s.


CGI VARIABLE NAMES
------------------
WWW_PATH_INFO           the URL path component
WWW_SCRIPT_NAME         the name of the script being executed
WWW_SERVER_NAME         the server TCP/IP host name
WWW_SERVER_PORT         the server TCP/IP port number

WWW_FORM_BUILD          whole page, or for 'I'nclusion in another document
WWW_FORM_COLUMNHEADER   fixed format table, column headings every 'n' records 
WWW_FORM_DATABASE       file specification for database file (overrides path)
WWW_FORM_DBIV           non-empty indicates it came from DataBaseNotSpecified()
WWW_FORM_DESCRIPTION    string used as explanation on built page
WWW_FORM_DO             specifies the function (e.g. "query", "build", etc.)
WWW_FORM_FIELDS         all or only selected fields (query and construct role)
WWW_FORM_FORMAT         report to be generated using "fixed" or "html" tables
WWW_FORM_LAYOUT         report to be "table" or "list"
WWW_FORM_MEMOFIELD      merely supplies the field name for use in title
WWW_FORM_MEMOWIDTH      specifies the width of the memo text displayed
WWW_FORM_RECORD         for single record number and memo block number access
WWW_FORM_TITLE          string used as page title (defaults to file name)

To differentiate between script-used CGI variable names and database variable 
names being passed as CGI form variables ("WWW_FORM_..."), the latter are 
passed using names beginning with a leading underscore (i.e. generates a 
double underscore in the resulting CGI variable/symbol name: WWW_FORM__name). 


BUILD
-----
See BUILD_DBIV.COM


VERSION HISTORY
---------------
08-OCT-95  MGD  v1.1.0, query on numeric range in a field
04-OCT-95  MGD  v1.0.1, bugfix, stopped free()ing environment variables!
08-SEP-95  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "DBIV AXP-1.1.0";
#else
   char SoftwareID [] = "DBIV VAX-1.1.0";
#endif

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include <descrip.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

#define boolean int
#define true 1
#define false 0

/* macro provides NULL pointer if CGI variable does not exist */
#define GetCgiVarIfExists(CharPointer,CgiVariableName) \
   CharPointer = getenv(CgiVariableName)

/* macro provides pointer to empty string even if CGI variable does not exist */
#define GetCgiVar(CharPointer,CgiVariableName) \
   if ((CharPointer = getenv(CgiVariableName)) == NULL) \
       CharPointer = ""; \
   if (Debug) fprintf (stdout, "%s |%s|\n", CgiVariableName, CharPointer);

/* magic number (why?) */
#define BLOCK_CNT 18

#define MaxSizeOfFieldName 11

/* dBASEIV database file-header */
struct DbHeader
{
   unsigned char  ValidByte;
   unsigned char  LastModifiedYY;
   unsigned char  LastModifiedMM;
   unsigned char  LastModifiedDD;
   unsigned int  NumberOfRecords;
   unsigned short  SizeOfHeader;
   unsigned short  SizeOfRecord;
   unsigned char  Reserved1 [2];
   unsigned char  IncompleteTransactionFlag;
   unsigned char  EncryptionFlag;
   unsigned char  MultiUserEnvironment [12];
   unsigned char  ProductionMdx;
   unsigned char  Reserved2 [3];
};

/* dBASEIV field-descriptor (fixed 32 bytes) */
struct DbField
{
   unsigned char  FieldName [MaxSizeOfFieldName];
   unsigned char  FieldType;
   unsigned char  Reserved1 [4];
   unsigned char  FieldWidth;
   unsigned char  FieldDecimalCount;
   unsigned char  Reserved2 [2];
   unsigned char  WorkAreaID;
   unsigned char  Reserved3 [10];
   unsigned char  ProductionMdxFlag;
};

/* structures determined from dump of .DBT file (no guarantees!) */

struct MemoHeader
{
   unsigned char  Unknown1 [20];
   unsigned int  BlockSize;
};

struct MemoBlock
{
   unsigned char  Unknown1 [8];
   unsigned char  Data [];
};

typedef struct DbField *DbFieldPtr;

char  Utility [] = "DBIV";

char  Http200Header [] =
"HTTP/1.0 200 Document follows.\r\n\
Content-Type: text/html\r\n\
\r\n";

char  Http404Header [] =
"HTTP/1.0 404 Error report follows.\r\n\
Content-Type: text/html\r\n\
\r\n";

char  DashedLine[] =
"----------------------------------------\
----------------------------------------\
----------------------------------------\
----------------------------------------\
----------------------------------------";
char  CircumflexLine[] =
"~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~";

boolean  AllFields,
         UseHtmlSelectOption,
         Debug,
         DoBuild,
         DoConstruct,
         DoMemo,
         DoQuery,
         DoSelect,
         HttpHasBeenOutput,
         ListQueryFields = true,
         ReportFormatIsFixedFont,
         ReportLayoutIsTable,
         SelectQuery,
         WildCardDataBase;

boolean  *SelectedArrayPtr;

int  ColumnHeaderRecordCount,
     DataBaseCount,
     LargestFieldWidth,
     NumberOfFields,
     RecordCount,
     RecordOutputCount,
     SelectedCount,
     ThisRecordNumber;

char  DataBaseSpec [256],
      DataBaseFileName [256],
      IconImg [256],
      UriPageTitle [256];

char  *CgiFormDataBasePtr,
      *CgiFormDoPtr,
      *CgiFormFieldsPtr,
      *CgiFormLayoutPtr,
      *CgiFormFormatPtr,
      *CgiPathInfoPtr,
      *CgiScriptNamePtr,
      *CgiServerNamePtr,
      *CgiServerPortPtr,
      *DataBaseNamePtr,
      *HelpUrlPtr,
      *IconUrlPtr,
      *PageTitlePtr;
      
char  **QueryArrayPtr;

struct DbHeader  DataBaseHeader;
DbFieldPtr  *FieldArrayPtr;

FILE  *HttpOut;

struct FAB  DbFileFab;
struct NAM  DbFileNam;

char* GetMemoField (int);
boolean QueryAllowed (char);

/*****************************************************************************/
/*
*/

main
(
int argc,
char *argv[]
)
{
   register char  *cptr, *sptr;

   int  acnt;
   char  Scratch [256];
   char  *ScratchPtr;

   /*********/
   /* begin */
   /*********/

   /* open another output stream so that the '\r' and '\n' are not filtered */
#ifdef __DECC
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "ctx=bin")) == NULL)
      exit (vaxc$errno);
#else
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "rfm=udf")) == NULL)
      exit (vaxc$errno);
#endif

   /***********************************/
   /* get the command line parameters */
   /***********************************/

   /* doing it this way, parameters must be space separated! */
   HelpUrlPtr = IconUrlPtr = "";
   for (acnt = 1; acnt < argc; acnt++)
   {
      if (Debug) fprintf (stdout, "argv[%d] |%s|\n", acnt, argv[acnt]);
      if (strsame (argv[acnt], "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (strsame (argv[acnt], "/HELP=", 6))
      {
         HelpUrlPtr = argv[acnt]+6;
         continue;
      }
      if (strsame (argv[acnt], "/ICON=", 6))
      {
         IconUrlPtr = argv[acnt]+6;
         continue;
      }
      fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
               Utility, argv[acnt]+1);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   /*********************/
   /* get CGI variables */
   /*********************/

   /*
      Get the database file specification from path info.
      CGI variable 'database' overrides anything in the path.
   */
   GetCgiVar (CgiPathInfoPtr, "WWW_PATH_INFO");
   GetCgiVar (CgiFormDataBasePtr, "WWW_FORM_DATABASE");
   if (CgiFormDataBasePtr[0])
      cptr = CgiFormDataBasePtr;
   else
   {
      cptr = CgiPathInfoPtr;
      if (*cptr == '/') cptr++;
   }
   sptr = DataBaseSpec;
   while (*cptr)
   {
      if (*cptr == '*' || *cptr == '%') WildCardDataBase = true;
      *sptr++ = toupper(*cptr++);
   }
   *sptr = '\0';
   if (sptr > DataBaseSpec)
      if (sptr[-1] == ']') WildCardDataBase = true;

   GetCgiVar (CgiFormLayoutPtr, "WWW_FORM_LAYOUT");
   if (toupper (CgiFormLayoutPtr[0]) == 'T')  /* 'T'able */
      ReportLayoutIsTable = true;
   else
   if (toupper (CgiFormLayoutPtr[0]) == 'L')  /* 'L'ist */
      ReportLayoutIsTable = false;
   else
      ReportLayoutIsTable = true;

   GetCgiVar (CgiFormFormatPtr, "WWW_FORM_FORMAT");
   if (toupper (CgiFormFormatPtr[0]) == 'F')  /* 'F'ixed */
      ReportFormatIsFixedFont = true;
   else
   if (toupper (CgiFormFormatPtr[0]) == 'H')  /* 'H'tml */
      ReportFormatIsFixedFont = false;
   else
   {
      ReportFormatIsFixedFont = true;
      CgiFormFormatPtr = "fixed";
   }

   GetCgiVar (PageTitlePtr, "WWW_FORM_TITLE");
   if (!PageTitlePtr[0])
   {
      /* if a title has not been explicitly provided, use the specification */
      cptr = DataBaseSpec;
      while (*cptr && *cptr != ']') cptr++;
      if (*cptr) cptr++;
      if (*cptr == '[')
      {
         while (*cptr && *cptr != ']') cptr++;
         if (*cptr) cptr++;
      }
      if (*cptr)
         PageTitlePtr = cptr;
      else
         PageTitlePtr = DataBaseFileName;
   }
   CopyIntoUri (UriPageTitle, PageTitlePtr, -1);

   GetCgiVar (ScratchPtr, "WWW_FORM_RECORD");
   if (ScratchPtr[0])
   {
      if (WildCardDataBase)
      {
         ErrorGeneral ("Cannot specify record number with wildcard database!",
                       __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
      ThisRecordNumber = atoi (ScratchPtr);
      if (!ThisRecordNumber)
      {
         sprintf (Scratch, "Invalid record: <TT>%s</TT>", ScratchPtr);
         ErrorGeneral (Scratch, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
      ListQueryFields = ReportLayoutIsTable = false;
   }

   GetCgiVar (CgiFormFieldsPtr, "WWW_FORM_FIELDS");
   if (toupper(CgiFormFieldsPtr[0]) == 'A') AllFields = true;

   GetCgiVar (CgiScriptNamePtr, "WWW_SCRIPT_NAME");

   GetCgiVar (CgiFormDoPtr, "WWW_FORM_DO");

   /***********************/
   /* execute the request */
   /***********************/

   if (IconUrlPtr[0])
      sprintf (IconImg, "<IMG SRC=\"%s?do=icon\" ALT=\"Hyper-dBASEIV, \">",
               CgiScriptNamePtr);
   else
      strcpy (IconImg, "Hyper-dBASEIV, ");

   if (toupper(CgiFormDoPtr[0]) == 'H')  /* 'H'elp */
   {
      /* redirect this request */
      GetCgiVar (CgiServerNamePtr, "WWW_SERVER_NAME");
      GetCgiVar (CgiServerPortPtr, "WWW_SERVER_PORT");
      if (!CgiServerPortPtr[0]) CgiServerPortPtr = "80";
      /*
         This allows a '#'-delimited local part of a URL to be included
         for any "do=help", specifying a location within the help document,
         using the syntax "do=help%%23local-part", etc.
      */
      if (strlen (CgiFormDoPtr) < 4) CgiFormDoPtr = "help";
      fprintf (HttpOut,
"HTTP/1.0 302 Redirection.\n\
Location: http://%s:%s%s%s\n\
\n",
      CgiServerNamePtr, CgiServerPortPtr, HelpUrlPtr, CgiFormDoPtr+4);
      exit (SS$_NORMAL);
   }

   if (toupper(CgiFormDoPtr[0]) == 'I')  /* 'I'con */
   {
      /* redirect this request */
      GetCgiVar (CgiServerNamePtr, "WWW_SERVER_NAME");
      GetCgiVar (CgiServerPortPtr, "WWW_SERVER_PORT");
      if (!CgiServerPortPtr[0]) CgiServerPortPtr = "80";
      fprintf (HttpOut,
"HTTP/1.0 302 Redirection.\n\
Location: http://%s:%s%s\n\
\n",
      CgiServerNamePtr, CgiServerPortPtr, IconUrlPtr);
      exit (SS$_NORMAL);
   }

   if (!DataBaseSpec[0])
      DataBaseNotSpecified ();
   else
   if (!CgiFormDoPtr[0] && WildCardDataBase)
      IndexWildCardDataBase ();
   else
   if (!CgiFormDoPtr[0] || toupper(CgiFormDoPtr[0]) == 'S')  /* 'S'elect */
   {
      DoSelect = true;
      if (strlen(CgiFormDoPtr) >= 6)
         if (CgiFormDoPtr[6] == ':' && toupper(CgiFormDoPtr[7]) == 'Q')
            SelectQuery = true;
      ProcessDataBaseSpec ();
   }
   else
   if (toupper(CgiFormDoPtr[0]) == 'Q')  /* 'Q'uery */
   {
      DoQuery = true;
      ProcessDataBaseSpec ();
   }
   else
   if (toupper(CgiFormDoPtr[0]) == 'B')  /* 'B'uild */
   {
      DoBuild = UseHtmlSelectOption = true;
      if (strlen(CgiFormDoPtr) >= 5)
      {
         if (CgiFormDoPtr[5] == ':' && toupper(CgiFormDoPtr[6]) == 'S')
            UseHtmlSelectOption = true;
         if (CgiFormDoPtr[5] == ':' && toupper(CgiFormDoPtr[6]) == 'R')
            UseHtmlSelectOption = false;
      }
      ProcessDataBaseSpec ();
   }
   else
   if (toupper(CgiFormDoPtr[0]) == 'C')  /* 'C'onstruct */
   {
      DoConstruct = true;
      ProcessDataBaseSpec ();
   }
   else
   if (toupper(CgiFormDoPtr[0]) == 'M')  /* 'M'emo */
   {
      DoMemo = true;
      ProcessDataBaseSpec ();
   }
   else
   {
      sprintf (Scratch, "Unknown function: \"%s\"", CgiFormDoPtr);
      ErrorGeneral (Scratch, __FILE__, __LINE__);
   }

   exit (SS$_NORMAL);
}

/****************************************************************************/
/*
Parse the database specification.  This make the various components of any 
filename available.  If wildcarded the search will return multiple databases, 
if not the just the one!
*/ 

ProcessDataBaseSpec ()

{
   int  status;
   char  ExpandedFileSpec [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessDataBaseSpec()\n");

   /* initialize the file access block */
   DbFileFab = cc$rms_fab;
   DbFileFab.fab$l_fna = DataBaseSpec;
   DbFileFab.fab$b_fns = strlen(DataBaseSpec);
   DbFileFab.fab$l_dna = "*.DBF;";
   DbFileFab.fab$b_dns = 6;
   DbFileFab.fab$l_fop = FAB$V_NAM;
   DbFileFab.fab$l_nam = &DbFileNam;

   /* initialize the file name block */
   DbFileNam = cc$rms_nam;
   DbFileNam.nam$l_esa = ExpandedFileSpec;
   DbFileNam.nam$b_ess = sizeof(ExpandedFileSpec)-1;
   DbFileNam.nam$l_rsa = DataBaseFileName;
   DbFileNam.nam$b_rss = sizeof(DataBaseFileName)-1;

   if (VMSnok (status = sys$parse (&DbFileFab, 0, 0)))
   {
      ErrorVmsStatus (status, DataBaseSpec, DataBaseSpec, __FILE__, __LINE__);
      return;
   }

   while (VMSok (status = sys$search (&DbFileFab, 0, 0)))
   {
      /* ignore any other than .DBF (numeric ".DBF") */
      if (*(unsigned long*)DbFileNam.nam$l_type != 0x4642442e) continue;

      DataBaseCount++;
      DataBaseNamePtr = DbFileNam.nam$l_name;
      *(char*)DbFileNam.nam$l_ver = '\0';
      if (DoMemo)
         ProcessMemo ();
      else
         ProcessDataBase ();
      *(char*)DbFileNam.nam$l_ver = ';';
   }
   if (status != RMS$_NMF)
   {
      *(char*)DbFileNam.nam$l_ver = '\0';
      ErrorVmsStatus (status, DataBaseFileName, DataBaseFileName,
                      __FILE__, __LINE__);
      return;
   }
}

/*****************************************************************************/
/*
Open the specified .DBF file.  Read the database header, extracting field 
(variable) information.  Call appropriate function to process the field 
information.  Read the data records.  Call appropriate function to process the 
data.
*/ 

ProcessDataBase ()

{
   register int  idx;
   register char  *dsptr, *sptr;

   char  String [256];
   char  *DataStringPtrBuffer;
   FILE  *DbFile;
         
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessDataBase() |%s|\n", DataBaseFileName);

   DbFile = fopen (DataBaseFileName, "rb", "ctx=stm",
                   "shr=put", "shr=get", "shr=upd");
   if (DbFile == NULL)
   {
       ErrorVmsStatus (vaxc$errno, PageTitlePtr, DataBaseFileName,
                       __FILE__, __LINE__);
       exit (SS$_NORMAL);
   }

   /***************************/
   /* process database header */
   /***************************/

   if (!fread (&DataBaseHeader, sizeof(DataBaseHeader), 1, DbFile))
   {
       ErrorVmsStatus (vaxc$errno, PageTitlePtr, DataBaseFileName,
                       __FILE__, __LINE__);
       exit (SS$_NORMAL);
   }

   if (!DataBaseHeader.LastModifiedDD ||
       !DataBaseHeader.LastModifiedMM ||
       !DataBaseHeader.LastModifiedYY)
   {
      sprintf (String, "Database <TT>%s</TT> appears invalid!",
               DataBaseFileName);
      ErrorGeneral (String, __FILE__, __LINE__);
      exit (SS$_NORMAL); 
   }
   if (DataBaseHeader.EncryptionFlag)
   {
      sprintf (String,
               "Database <TT>%s</TT> is <I>encrypted</I>, cannot access!",
               DataBaseFileName);
      ErrorGeneral (String, __FILE__, __LINE__);
      exit (SS$_NORMAL); 
   }
   if (DataBaseHeader.IncompleteTransactionFlag)
   {
      sprintf (String,
"Database <TT>%s</TT> indicates <I>incomplete transaction</I>, cannot access!",
               DataBaseFileName);
      ErrorGeneral (String, __FILE__, __LINE__);
      exit (SS$_NORMAL); 
   }

   NumberOfFields = (DataBaseHeader.SizeOfHeader-sizeof(struct DbHeader))/
                     sizeof(struct DbField);
   FieldArrayPtr = (DbFieldPtr*)malloc (NumberOfFields * sizeof(DbFieldPtr));
   if (FieldArrayPtr == NULL)
   {
      ErrorVmsStatus (vaxc$errno, PageTitlePtr, DataBaseFileName,
                      __FILE__, __LINE__);
      exit (SS$_NORMAL);;
   }
   SelectedArrayPtr = (boolean*)malloc (NumberOfFields * sizeof(boolean));
   if (SelectedArrayPtr == NULL)
   {
      ErrorVmsStatus (vaxc$errno, PageTitlePtr, DataBaseFileName,
                      __FILE__, __LINE__);
      exit (SS$_NORMAL);;
   }
   QueryArrayPtr = (char*)malloc (NumberOfFields * sizeof(char*));
   if (QueryArrayPtr == NULL)
   {
      ErrorVmsStatus (vaxc$errno, PageTitlePtr, DataBaseFileName,
                      __FILE__, __LINE__);
      exit (SS$_NORMAL);;
   }

   /***************************************/
   /* process field information in header */
   /***************************************/

   if (ThisRecordNumber && ThisRecordNumber > DataBaseHeader.NumberOfRecords)
   {
      sprintf (String, "Record number out of range: <TT>%d</TT>",
               ThisRecordNumber);
      ErrorGeneral (String, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   SelectedCount = 0;
   strcpy (String, "WWW_FORM__");

   for (idx = 0; idx < NumberOfFields; idx++)
   {
      FieldArrayPtr[idx] = (struct DbField*)malloc (sizeof(struct DbField));
      if (FieldArrayPtr[idx] == NULL)
      {
         ErrorVmsStatus (vaxc$errno, PageTitlePtr, DataBaseFileName,
                         __FILE__, __LINE__);
         exit (SS$_NORMAL);;
      }
      fread (FieldArrayPtr[idx], sizeof(struct DbField), 1, DbFile);

      /*
         Append the variable name to the leading "WWW_FORM__".
         Use this to check if there is a correspoding "WWW_" CGI symbol.
         To differentiate between script-used CGI variable names and
         database variable names, the latter are passed using names
         beginning with a leading underscore.
      */
      CopyDbaseString (String+sizeof("WWW_FORM__")-1,
                       FieldArrayPtr[idx]->FieldName, MaxSizeOfFieldName);
      GetCgiVar (sptr, String);
      if (sptr[0])
      {
         /*
            If the query string associated with the field has a single
            asterisk then this matches any data.  Indicate this by setting
            the corresponding element in the query array to NULL, otherwise
            point that element at the string.  This will be used in
            ProcessDataBaseRecord() to selectively display records by 
            matching against the pointed-to query (wildcarded) string.
         */
         SelectedArrayPtr[idx] = true;
         SelectedCount++;
         if (sptr[0] == '*' && !sptr[1])
            QueryArrayPtr[idx] = NULL;
         else
            QueryArrayPtr[idx] = sptr;
      }
      else
      {
         /* either no such CGI variable name, or value was empty string */
         SelectedArrayPtr[idx] = false;
         QueryArrayPtr[idx] = NULL;
         if (AllFields)
         {
             SelectedArrayPtr[idx] = true;
            SelectedCount++;
         }
      }
      if (FieldArrayPtr[idx]->FieldWidth > LargestFieldWidth)
          LargestFieldWidth = FieldArrayPtr[idx]->FieldWidth;
   }

   /*************************/
   /* field name processing */
   /*************************/

   if (DoSelect || DoBuild || DoConstruct)
   {
      if (DoSelect || DoBuild)
         FieldSelectionForm ();
      else
      if (DoConstruct)
         ConstructQueryForm ();

      fclose (DbFile);

      for (idx = 0; idx < NumberOfFields; idx++) free (FieldArrayPtr[idx]);
      free (FieldArrayPtr);
      free (QueryArrayPtr);
      free (SelectedArrayPtr);

      return;
   }

   /************************/
   /* process data records */
   /************************/

   if (DoQuery)
      ProcessDataBaseRecord (NULL);

   dsptr = (char *)malloc (DataBaseHeader.SizeOfRecord*BLOCK_CNT);
   if (dsptr == NULL)
   {
      ErrorVmsStatus (vaxc$errno, PageTitlePtr, DataBaseFileName,
                      __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
   DataStringPtrBuffer = dsptr;

   /* skip over the first character (why?) */
   fread (dsptr, 1, 1, DbFile);

   RecordCount = 0;
   while (RecordCount < DataBaseHeader.NumberOfRecords)
   {
      if (!(RecordCount++ % BLOCK_CNT))
      {
          dsptr = DataStringPtrBuffer;
          fread (dsptr, DataBaseHeader.SizeOfRecord, BLOCK_CNT, DbFile);
      }

      if (ThisRecordNumber && RecordCount > ThisRecordNumber)
         break;

      /* end of database */
      if (*dsptr == 0x1a) break;

      /* a deleted record is marked with a '*' */
      if (*dsptr != '*')
      {
         if (!ThisRecordNumber || (RecordCount == ThisRecordNumber))
         {
            if (DoQuery)
               if (!ProcessDataBaseRecord (dsptr+1)) break;
         }
      }

      dsptr += DataBaseHeader.SizeOfRecord;
   }

   if (DoQuery)
      ProcessDataBaseRecord (NULL);

   /*********************************/
   /* end of data record processing */
   /*********************************/

   fclose(DbFile);

   for (idx = 0; idx < NumberOfFields; idx++) free (FieldArrayPtr[idx]);
   free (FieldArrayPtr);
   free (QueryArrayPtr);
   free (SelectedArrayPtr);
   free (DataStringPtrBuffer);
}

/*****************************************************************************/
/*
This overly-long function performs two main jobs.  Pre- and Post- Processing 
of a database file, indicated by a NULL being passed for the data record.  
This generally means generating appropriate HTML for the page.  The second 
involves the processing and optional display of a single database record.  
This function supports the filtering of the records displayed by comparision 
with any field query strings supplied with the request. 

The function builds a string of all the required fields of the record into 
'OutLine', which is output (if required) at the end of the function.

If query strings are supplied each is matched against the data in 
corresponding fields.  If any one field does not match the record is 
immediately discarded.  If all query records match (or there were no query 
fields) the constructed line is output.
*/ 

boolean ProcessDataBaseRecord (register char *dsptr)

{
   static int  PrevDataBaseCount;
   static char  OutLine [8192];
   static char  *OutLinePtr;

   register int  idx;

   boolean  Output;
   int  MemoBlockNumber,
        Count,
        FieldWidth;
   unsigned long  BinTime [2];
   unsigned short  NumTime [7];
   char  FieldName [MaxSizeOfFieldName+1],
         Scratch [4096],
         String [512];
   char  *ScratchPtr,
         *CgiFormDescriptionPtr;
   double  DbFloat;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessDataBaseRecord()\n");

   if (dsptr == NULL)
   {
      if (DataBaseCount != PrevDataBaseCount)
      {
         /****************/
         /* new-database */
         /****************/

         PrevDataBaseCount = DataBaseCount;

         if (DataBaseCount == 1)
         {
            /******************/
            /* first database */
            /******************/

            if (!(ThisRecordNumber || SelectedCount))
            {
               ErrorGeneral ("No fields selected!", __FILE__, __LINE__);
               exit (SS$_NORMAL);
            }

            sys$gettim (&BinTime);
            sys$numtim (&NumTime, &BinTime);

            fprintf (HttpOut,
"%s\
<!-- SoftwareID: %s -->\n\
<!-- %s -->\n\
<TITLE>Hyper-dBASEIV, %s</TITLE>\n\
<H1>%s%s</H1>\n\
Report generated: <TT>%02.02d/%02.02d/%02.02d %02.02d:%02.02d</TT><BR>\n\
Database last modified : <TT>%02d/%02d/%02d</TT>\n\
<BR>Number of records : <TT>%d</TT>\n",
            Http200Header,
            SoftwareID,
            DataBaseSpec, 
            PageTitlePtr,
            IconImg, PageTitlePtr,
            NumTime[2], NumTime[1], NumTime[0] % 100, NumTime[3], NumTime[4],
            DataBaseHeader.LastModifiedDD,
            DataBaseHeader.LastModifiedMM,
            DataBaseHeader.LastModifiedYY,
            DataBaseHeader.NumberOfRecords);
            fflush (HttpOut);

            HttpHasBeenOutput = true;

            GetCgiVar (CgiFormDescriptionPtr, "WWW_FORM_DESCRIPTION");

            if (CgiFormDescriptionPtr[0])
               fprintf (HttpOut, "<P>\n%s\n", CgiFormDescriptionPtr);

            if (ListQueryFields) QueryFieldsList ();

            fputs ("<P><HR>\n", HttpOut);
         }

         GetCgiVar (ScratchPtr, "WWW_FORM_COLUMNHEADER");
         if (ScratchPtr[0]) ColumnHeaderRecordCount = atoi (ScratchPtr);
         if (!ColumnHeaderRecordCount) ColumnHeaderRecordCount = 20;

         RecordOutputCount = 0;

         if (ReportFormatIsFixedFont)
         {
            fprintf (HttpOut, "<P> Database: <TT>%s</TT> <!-- %s -->\n<PRE>",
                     DataBaseNamePtr, DataBaseFileName);
         }
         else
         {
            fprintf (HttpOut,
"<P>\n\
Database: %s <!-- %s -->\n\
<P>\n\
<TABLE BORDER>\n",
            DataBaseNamePtr, DataBaseFileName);
         }
      }
      else
      {
         /*******************/
         /* end-of-database */
         /*******************/

         if (ReportFormatIsFixedFont)
         {
            if (ReportLayoutIsTable)
            {
               if (RecordOutputCount)
                  FixedFormatColumnHeader (false);
               else
                  fputs ("(no records located)\n", HttpOut);
            }
            fputs ("</PRE>\n", HttpOut);
         }
         else
         {
            /* HTML table format */
            fputs ("</TABLE>\n", HttpOut);
         }
         fprintf (HttpOut, "<P><HR>\n");
         fflush (HttpOut);
      }

      return (true);
   }

   /***********************/
   /* process data record */
   /***********************/

   Output = true;
   OutLinePtr = OutLine;

   if (ReportLayoutIsTable)
   {
      Count = sprintf (Scratch, "%d", RecordOutputCount+1);
      if (ReportFormatIsFixedFont)
      {
         OutLinePtr += sprintf (OutLinePtr,
"%.*s\
<A HREF=\"%s/%s?do=query&format=%s&layout=list&record=%d&fields=all&title=%s\">\
%s</A> ",
         4-Count, "   ", CgiScriptNamePtr, DataBaseFileName, CgiFormFormatPtr,
         RecordCount, UriPageTitle, Scratch);
      }
      else
      {
         /* format is using HTML <table> */
         OutLinePtr += sprintf (OutLinePtr,
"<TD>\
<A HREF=\"%s/%s?do=query&format=%s&layout=list&record=%d&fields=all&title=%s\">\
%s</A> ",
         CgiScriptNamePtr, DataBaseFileName, CgiFormFormatPtr,
         RecordCount, UriPageTitle, Scratch);
      }
   }
   else
   {
      /* report layout is list */
      if (RecordOutputCount)
      {
         if (ReportFormatIsFixedFont)
             OutLinePtr += sprintf (OutLinePtr, "\n");
         else
             OutLinePtr += sprintf (OutLinePtr,
                "</TABLE>\n<BR>\n<TABLE BORDER>\n");
      }
   }

   FieldWidth = 0;
   for (idx = 0; idx < NumberOfFields; idx++)
   {
      dsptr += FieldWidth;
      FieldWidth = FieldArrayPtr[idx]->FieldWidth;

      if (!SelectedArrayPtr[idx]) continue;

      CopyDbaseString (FieldName, FieldArrayPtr[idx]->FieldName,
                       MaxSizeOfFieldName);

      switch (FieldArrayPtr[idx]->FieldType)
      {
          /**********************/
          /* character variable */
          /**********************/

          case 'C':

             CopyDbaseString (String, dsptr, FieldWidth);

             if (QueryArrayPtr[idx] != NULL)
                if (!(Output = QueryField (String, QueryArrayPtr[idx])))
                   break;

             if (ReportLayoutIsTable)
             {
                if (ReportFormatIsFixedFont)
                {
                   sprintf (Scratch, "|%-*.*s", FieldWidth, FieldWidth, String);
                   OutLinePtr += CopyIntoHtml (OutLinePtr, Scratch, -1);
                }
                else
                {
                   CopyIntoHtml (Scratch, String, -1);
                   OutLinePtr += sprintf (OutLinePtr, "<TD>%s", Scratch);
                }
             }
             else
             {
                if (ReportFormatIsFixedFont)
                {
                   sprintf (Scratch, "%-*.*s |%-*.*s|\n",
                            MaxSizeOfFieldName, MaxSizeOfFieldName, FieldName,
                            FieldWidth, FieldWidth, String);
                   OutLinePtr += CopyIntoHtml (OutLinePtr, Scratch, -1);
                }
                else
                {
                   CopyIntoHtml (Scratch, String, -1);
                   OutLinePtr += sprintf (OutLinePtr, "<TD>%s<TD>%s<TR>\n",
                                 FieldName, Scratch);
                }
             }

             break;

          /*****************/
          /* date variable */
          /*****************/

          case 'D':

             CopyDbaseString (String, dsptr, FieldWidth);

             /* create a "dd/mm/yy"-style date in scratch */
             if (String[0])
                sprintf (Scratch, "%2.2s/%2.2s/%2.2s",
                         String+6, String+4, String+2);
             else
                Scratch[0] = '\0';

             if (QueryArrayPtr[idx] != NULL)
                if (!(Output = QueryField (Scratch, QueryArrayPtr[idx])))
                   break;

             if (ReportLayoutIsTable)
             {
                if (ReportFormatIsFixedFont)
                   OutLinePtr += sprintf (OutLinePtr, "|%-*.*s",
                                          FieldWidth, FieldWidth, Scratch);
                else
                   OutLinePtr += sprintf (OutLinePtr, "<TD>%s", Scratch);
             }
             else
             {
                if (ReportFormatIsFixedFont)
                   OutLinePtr += sprintf (OutLinePtr, "%-*.*s |%-*.*s|\n",
                                 MaxSizeOfFieldName, MaxSizeOfFieldName,
                                 FieldName, FieldWidth, FieldWidth, Scratch);
                else
                   OutLinePtr += sprintf (OutLinePtr, "<TD>%s<TD>%s<TR>\n",
                                 FieldName, Scratch);
             }

             break;

          /*****************************/
          /* floating/numeric variable */
          /*****************************/

          case 'F':
          case 'N':

             CopyDbaseString (String, dsptr, FieldWidth);

             if (strlen(String) > 0)
             {
                if (sscanf (String, "%f", &DbFloat) != 1)
                   strcpy (Scratch, "*error*");
                else
                   sprintf (Scratch, "%f", DbFloat);
             }

             if (ReportLayoutIsTable)
             {
                if (ReportFormatIsFixedFont)
                   OutLinePtr += sprintf (OutLinePtr, "|%-*.*s",
                                          FieldWidth, FieldWidth, Scratch);
                else
                   OutLinePtr += sprintf (OutLinePtr, "<TD>%s", Scratch);
             }
             else
             {
                if (ReportFormatIsFixedFont)
                   OutLinePtr += sprintf (OutLinePtr, "%-*.*s |%-*.*s|\n",
                                 MaxSizeOfFieldName, MaxSizeOfFieldName,
                                 FieldName, FieldWidth, FieldWidth, Scratch);
                else
                   OutLinePtr += sprintf (OutLinePtr, "<TD>%s<TD>%s<TR>\n",
                                 FieldName, Scratch);
             }

             break;

          /********************/
          /* logical variable */
          /********************/

          case 'L':

             switch (*dsptr)
             {
                case 'Y':
                case 'y':
                case 'T':
                case 't':

                   if (QueryArrayPtr[idx] != NULL)
                   {
                      if (QueryArrayPtr[idx][0] != 'Y' &&
                          QueryArrayPtr[idx][0] != 'y' &&
                          QueryArrayPtr[idx][0] != 'T' &&
                          QueryArrayPtr[idx][0] != 't')
                      {
                         /* the match has failed */
                         Output = false;
                       }
                   }
                   break;

                case 'N':
                case 'n':
                case 'F':
                case 'f':

                   if (QueryArrayPtr[idx] != NULL)
                   {
                      if (QueryArrayPtr[idx][0] != 'N' &&
                          QueryArrayPtr[idx][0] != 'n' &&
                          QueryArrayPtr[idx][0] != 'F' &&
                          QueryArrayPtr[idx][0] != 'f')
                      {
                         /* the match has failed */
                         Output = false;
                      }
                   }
                   break;
                }

                /* if a match has failed then exit for the 'case:' */
                if (!Output) break;

                if (ReportLayoutIsTable)
                {
                   if (ReportFormatIsFixedFont)
                      OutLinePtr += sprintf (OutLinePtr, "|%c", *dsptr);
                   else
                      OutLinePtr += sprintf (OutLinePtr, "<TD>%c", *dsptr);
                }
                else
                {
                   if (ReportFormatIsFixedFont)
                      OutLinePtr += sprintf (OutLinePtr, "%-*s |%c|\n",
                                    MaxSizeOfFieldName, FieldName, *dsptr);
                   else
                      OutLinePtr += sprintf (OutLinePtr, "<TD>%s<TD>%c<TR>\n",
                                    FieldName, *dsptr);
                }

             break;

          /*****************/
          /* memo variable */
          /*****************/

          case 'M':
          {
             /* local storage */
             register char  *cptr, *fptr, *sptr;
             int  MemoTextWidth;
             char  *MemoTextPtr;

             if (LargestFieldWidth < 40)
                 MemoTextWidth = 40;
             else
             if (LargestFieldWidth > 80)
                 MemoTextWidth = 80;
             else
                 MemoTextWidth = LargestFieldWidth;

             CopyDbaseString (String, dsptr, FieldWidth);
             for (cptr = String; *cptr && !isdigit(*cptr); cptr++);
             if (isdigit(*cptr))
                MemoBlockNumber = atoi (cptr);
             else
                MemoBlockNumber = 0;

             if (MemoBlockNumber)
             {
                /*****************************/
                /* has associated memo block */
                /*****************************/

                cptr = MemoTextPtr = GetMemoField (MemoBlockNumber);
                /* ensure each string is separated by only one space */
                Count = CompressWhiteSpace (cptr);

                if (QueryArrayPtr[idx] != NULL)
                   if (!(Output = QueryField (cptr, QueryArrayPtr[idx])))
                      break;

                if (ReportLayoutIsTable)
                {
                   if (ReportFormatIsFixedFont)
                   {
                      OutLinePtr += sprintf (OutLinePtr,
"|<A HREF=\"%s/%s?do=memo&record=%d&title=%s\
&MemoField=%s&MemoWidth=%d\">memo</A>",
                      CgiScriptNamePtr, DataBaseFileName, MemoBlockNumber,
                      UriPageTitle, FieldName, MemoTextWidth);
                   }
                   else
                   {
                      /* format is using HTML <table> */
                      OutLinePtr += sprintf (OutLinePtr,
"<TD><A HREF=\"%s/%s?do=memo&record=%d&title=%s\
&MemoField=%s&MemoWidth=%d\">memo</A>",
                      CgiScriptNamePtr, DataBaseFileName, MemoBlockNumber,
                      UriPageTitle, FieldName, MemoTextWidth);
                   }
                }
                else
                {
                   /* report layout is list */

                   /* wrap the string into lines ensuring its within width */
                   WrapAt (cptr, MemoTextWidth);

                   /* add width-limited memo text lines into output line */
                   fptr = FieldName;
                   Count = 0;
                   while (*cptr)
                   {
                      sptr = cptr;
                      while (*cptr && *cptr != '\n') cptr++;
                      if (*cptr) *cptr++ = '\0';
                      if (Count++) fptr = "";
                      if (ReportFormatIsFixedFont)
                      {
                         sprintf (Scratch, "%-*.*s |%-*.*s|\n",
                            MaxSizeOfFieldName, MaxSizeOfFieldName, fptr,
                            MemoTextWidth, MemoTextWidth, sptr);
                         /* copy over HTML-sanitized memo text */
                         OutLinePtr += CopyIntoHtml (OutLinePtr, Scratch, -1);
                      }
                      else
                      {
                         /* format is using HTML <table> */
                         if (fptr[0]) OutLinePtr +=
                            sprintf (OutLinePtr, "<TD>%s<TD>\n", fptr);
                         /* copy over HTML-sanitized memo text */
                         OutLinePtr += CopyIntoHtml (OutLinePtr, sptr, -1);
                         OutLinePtr += sprintf (OutLinePtr, "<BR>\n");
                      }
                   }
                   if (!ReportFormatIsFixedFont)
                      OutLinePtr += sprintf (OutLinePtr, "<TR>\n");
                }
             }
             else
             {
                /*****************/
                /* no memo block */
                /*****************/

                if (ReportLayoutIsTable)
                   if (ReportFormatIsFixedFont)
                      OutLinePtr += sprintf (OutLinePtr, "|    ");
                   else
                      OutLinePtr += sprintf (OutLinePtr, "<TD>");
                else
                   if (ReportFormatIsFixedFont)
                      OutLinePtr += sprintf (OutLinePtr, "%-*.*s |    |\n",
                         MaxSizeOfFieldName, MaxSizeOfFieldName, FieldName);
                   else
                      OutLinePtr += sprintf (OutLinePtr, "<TD>%s<TD><TR>",
                                    FieldName);
             }

             break;
          }

          /********************/
          /* unknown variable */
          /********************/

          default :

             if (ReportLayoutIsTable)
             {
                if (ReportFormatIsFixedFont)
                   OutLinePtr += sprintf (OutLinePtr, "|%*.*s",
                      FieldWidth, FieldWidth, CircumflexLine);
                else
                   OutLinePtr += sprintf (OutLinePtr,
                      "<TD><I>(unknown field type)</I>");
             }
             else
             {
                if (ReportFormatIsFixedFont)
                {
                   OutLinePtr += sprintf (OutLinePtr, "%-*.*s |%-*.*s|\n",
                      MaxSizeOfFieldName, MaxSizeOfFieldName,
                      FieldName, FieldWidth, FieldWidth,
                      CircumflexLine);
                }
                else
                   OutLinePtr += sprintf (OutLinePtr,
                      "<TD>%s<TD><I>(unknown field type)</I><TR>\n",
                      FieldName);
            }
      }

      /* if a match has failed then exit for the 'for(;;)' loop */
      if (!Output) break;
   }

   /***************************/
   /* end process data record */
   /***************************/

   if (Output)
   {
      /* output this record */
      if (ReportLayoutIsTable)
      {
         if (ReportFormatIsFixedFont)
         {
            if (!(RecordOutputCount % ColumnHeaderRecordCount))
            {
               FixedFormatColumnHeader (false);
               FixedFormatColumnHeader (true);
               FixedFormatColumnHeader (false);
            }
            strcpy (OutLinePtr, "|\n");
            fputs (OutLine, HttpOut);
         }
         else
         {
            /* format is using HTML <table> */
            if (!(RecordOutputCount % ColumnHeaderRecordCount))
               HtmlTableColumnHeader ();
            strcpy (OutLinePtr, "<TR>\n");
            fputs (OutLine, HttpOut);
         }
      }
      else
      {
         /* layout is list */
         if (ReportFormatIsFixedFont)
            fputs (OutLine, HttpOut);
         else
         {
            /* format is using HTML <table> */
            strcpy (OutLinePtr, "<TR>\n");
            fputs (OutLine, HttpOut);
         }
      }

      RecordOutputCount++;
   }

   return (true);
}

/*****************************************************************************/
/*
Request was a hypertext link to a memo field.
*/ 

ProcessMemo (int BlockNumber)

{
   int  MemoTextWidth;
   char  HtmlMemoText [4096];
   char  *CgiMemoFieldNamePtr,
         *MemoPtr,
         *ScratchPtr;
                
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessMemo() |%s|\n", DataBaseFileName);

   GetCgiVar (CgiMemoFieldNamePtr, "WWW_FORM_MEMOFIELD");
   if (!CgiMemoFieldNamePtr[0]) CgiMemoFieldNamePtr = "(unknown)";

   GetCgiVar (ScratchPtr, "WWW_FORM_MEMOWIDTH");
   if (isdigit(ScratchPtr[0]))
      MemoTextWidth = atoi (ScratchPtr);
   else
      MemoTextWidth = 0;
   if (MemoTextWidth < 40)
      MemoTextWidth = 40;
   else
   if (MemoTextWidth > 80)
      MemoTextWidth = 80;

   MemoPtr = GetMemoField (ThisRecordNumber);
   /* compress white-space and ensure it wraps appropriately */
   CompressWhiteSpace (MemoPtr);
   WrapAt (MemoPtr, MemoTextWidth);
   /* copy over HTML-sanitized memo text, into fixed length string :^( */
   CopyIntoHtml (HtmlMemoText, MemoPtr, -1);

   fprintf (HttpOut,
"%s\
<!-- SoftwareID: %s -->\n\
<!-- %s -->\n\
<TITLE>Hyper-dBASEIV, %s</TITLE>\n\
<H1>%s%s</H1>\n\
<P> Memo Field: <TT>%s</TT>\n\
<P><HR>\n\
<PRE>%s\n\
</PRE>\n\
<P><HR>\n",
   Http200Header,
   SoftwareID,
   DataBaseFileName, 
   PageTitlePtr,
   IconImg, PageTitlePtr,
   CgiMemoFieldNamePtr,
   HtmlMemoText);
}

/*****************************************************************************/
/*
Memo files and structures are not well (or at all) documented in the reference
manual.  Worked this out from what there was ... and dumps of memo files.  Hope
its right!  The memo field filename is derived from the database filename, by 
substituting the .DBT of the memo for the .DBF of the database, then restoring 
it!  Return a pointer to a string containing the memo field data (or error 
message).
*/ 

char* GetMemoField (int BlockNumber)

{
   static int  BlockCount,
               PrevDataBaseCount;
   static FILE  *MemoFile;
   static struct MemoHeader  MemoHead;
   static struct MemoBlock  *MemoBlockPtr;
         
   register char  *cptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "GetMemoField() |%s|%d|\n",
               DataBaseFileName, BlockNumber);

   if (DataBaseCount != PrevDataBaseCount)
   {
      PrevDataBaseCount = DataBaseCount;

      /***************************************/
      /* cleanup after any previous database */
      /***************************************/

      if (MemoFile != NULL)
      {
         fclose (MemoFile);
         free (MemoBlockPtr);
      }

      /*****************************/
      /* prepare for this database */
      /*****************************/

      /* sneakily :^) change the extension to that required of the memo file */
      strcpy (DbFileNam.nam$l_type, ".DBT");
      if (Debug) fprintf (stdout, "MemoFileName |%s|\n", DataBaseFileName);

      MemoFile = fopen (DataBaseFileName, "rb", "ctx=stm",
                 "shr=get", "shr=put", "shr=upd");
      if (MemoFile == NULL)
      {
          ErrorVmsStatus (vaxc$errno, DataBaseFileName, DataBaseFileName,
                          __FILE__, __LINE__);
          exit (SS$_NORMAL);
      }

      if (!fread (&MemoHead, sizeof(struct MemoHeader), 1, MemoFile))
      {
          ErrorVmsStatus (vaxc$errno, DataBaseFileName, DataBaseFileName,
                          __FILE__, __LINE__);
          exit (SS$_NORMAL);
      }

      /* restore the original file extension */
      strcpy (DbFileNam.nam$l_type, ".DBF");

      if (Debug) fprintf (stdout, "BlockSize: %d\n", MemoHead.BlockSize);
      MemoBlockPtr = (struct MemoBlock*)malloc (MemoHead.BlockSize);
      if (MemoBlockPtr == NULL)
      {
         ErrorVmsStatus (vaxc$errno, "<I>memo</I> field processing.",
                         DataBaseFileName, __FILE__, __LINE__);
         exit (SS$_NORMAL);;
      }

      BlockCount = 0;
   }

   /************************************/
   /* find and process this memo field */
   /************************************/

   /* 
      If any previous memo field retrieval left the file partially read,
      then depending on whether the required block is before or after
      the point previously read, rewind the file to the start.
      Because of the fixed block size of memo fields it should be
      possible to fseek() to the correct block, rather than the less
      efficient way chosen here, however with VMS record-structured
      files this is often impractical, so ...
   */
   if (!BlockCount || BlockNumber <= BlockCount)
   {
      rewind (MemoFile);
      /* skip over the first block (memo header) */
      if (!fread (MemoBlockPtr, MemoHead.BlockSize, 1, MemoFile))
      {
         ErrorVmsStatus (vaxc$errno, "<I>memo</I> field processing.",
                         DataBaseFileName, __FILE__, __LINE__);
         exit (SS$_NORMAL);;
      }
      BlockCount = 0;
   }

   while (fread (MemoBlockPtr, MemoHead.BlockSize, 1, MemoFile))
   {
      if (++BlockCount >= BlockNumber) break;
      if (Debug) fprintf (stdout, "BlockCount %d\n", BlockCount);
   }

   if (BlockCount != BlockNumber) return ("ERROR: Invalid memo block.");

   /* 
      Memo text is carriage return terminated, with embedded line feeds!
      Find the end-of-memo, careful about corrupted/incomplete memo block.
   */
   zptr = MemoBlockPtr + MemoHead.BlockSize; 
   for (cptr = MemoBlockPtr->Data; *cptr != '\r' && cptr < zptr; cptr++);
   if (cptr >= zptr) return ("ERROR: Invalid memo block.");
   *cptr = '\0';
   return (MemoBlockPtr->Data);
}

/*****************************************************************************/
/*
For the HTML <TABLE> formatted table generate column headers.
*/ 

HtmlTableColumnHeader ()

{
   register int  fw, idx;

   char  Name [MaxSizeOfFieldName];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HtmlTableColumnHeader()\n");

   fputs ("<TH>", HttpOut);
   for (idx = 0; idx < NumberOfFields; idx++)
   {
      if (SelectedArrayPtr[idx])
      {
         CopyDbaseString (Name, FieldArrayPtr[idx]->FieldName);
         fprintf (HttpOut, "<TH>%s", Name);
      }
   }
   fputs ("<TR>\n", HttpOut);
}

/*****************************************************************************/
/*
For the fixed-format tabular record output generate column headings.  When the 
boolean is false creates a line to box in a field.  When called with boolean 
true generates the field names inside field-width-wide columns.
*/ 

FixedFormatColumnHeader (boolean DoFieldNames)

{
   register int  fw, idx;

   char  Name [MaxSizeOfFieldName];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FixedFormatColumnHeader()\n");

   fputs ("     ", HttpOut);
   for (idx = 0; idx < NumberOfFields; idx++)
   {
      if (SelectedArrayPtr[idx])
      {
         if (FieldArrayPtr[idx]->FieldType == 'M')
            fw = 4;
         else
            fw = FieldArrayPtr[idx]->FieldWidth;
         if (DoFieldNames)
         {
            CopyDbaseString (Name, FieldArrayPtr[idx]->FieldName,
                             MaxSizeOfFieldName);
            fprintf (HttpOut, "|%-*.*s", fw, fw, Name);
         }
         else
            fprintf (HttpOut, "+%*.*s", fw, fw, DashedLine);
      }
   }
   if (DoFieldNames)
      fputs ("|\n", HttpOut);
   else
      fputs ("+\n", HttpOut);
}

/*****************************************************************************/
/*
Generate an HTML list of any fields being queried.
*/ 

QueryFieldsList ()

{
   register int  idx, Count;

   char  Name [MaxSizeOfFieldName],
         Scratch [512];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "QueryFieldsList()\n");

   Count = 0;
   for (idx = 0; idx < NumberOfFields; idx++)
   {
      if (QueryArrayPtr[idx] != NULL)
      {
         if (!Count++)
            fputs ("<P>\n<B>Query Fields:</B>\n<UL>\n", HttpOut);
         CopyDbaseString (Name, FieldArrayPtr[idx]->FieldName,
                          MaxSizeOfFieldName);
         CopyIntoHtml (Scratch, QueryArrayPtr[idx], -1);
         fprintf (HttpOut, "<LI> %s : ``<TT>%s</TT>''", Name, Scratch);
      }
   }
   if (Count)
      fputs ("</UL>\n", HttpOut);
   else
      fputs ("<P>\n<B>Query Fields:</B> <I>(none)</I>\n", HttpOut);
}

/*****************************************************************************/
/*
Operates in two modes, form selection and form build.  Form selection allows 
the user to select the fields to be displayed (or enter query strings against 
the field name).  This is submitted as a query.  The form build mode allows a 
similar selection of fields to be displayed and/or queried but when submitted 
returns a dedicated form that can be used and/or saved as a customized query 
interface to the particular database.
*/ 

FieldSelectionForm ()

{
   register int  idx;

   char  String [256];
   char  *DoPtr,
         *SubmitPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FieldSelectionForm()\n");

   if (DoBuild)
   {
      DoPtr = "construct";
      SubmitPtr = "Build Query Form";
   }
   else
   {
      DoPtr = "query";
      SubmitPtr = "Submit Query";
   }

   fprintf (HttpOut,
"%s\
<!-- SoftwareID: %s -->\n\
<!-- %s -->\n\
<TITLE>Hyper-dBASEIV, %s</TITLE>\n\
<H1>%s%s</H1>\n\
Database last modified : <TT>%02d/%02d/%02d</TT>\n\
<BR>Number of records : <TT>%d</TT>\n\
<P> Other pages:\n",
   Http200Header,
   SoftwareID,
   DataBaseFileName,
   PageTitlePtr,
   IconImg, PageTitlePtr,
   DataBaseHeader.LastModifiedDD,
   DataBaseHeader.LastModifiedMM,
   DataBaseHeader.LastModifiedYY,
   DataBaseHeader.NumberOfRecords);

   HttpHasBeenOutput = true;

   if (DoBuild)
   {
      fprintf (HttpOut,
"<A HREF=\"%s?do=help\">Help</A> <I>(\
<A HREF=\"%s?do=help%%23BuildPage\">this page</A>)</I>.\n",
      CgiScriptNamePtr, CgiScriptNamePtr);
   }
   else
   if (SelectQuery)
   {
      fprintf (HttpOut,
"<A HREF=\"%s?do=help\">Help</A> <I>(\
<A HREF=\"%s?do=help%%23SelectQueryPage\">this page</A>,\n\
<A HREF=\"%s?do=help%%23QueryStrings\">Query Strings</A>)</I>.\n",
      CgiScriptNamePtr, CgiScriptNamePtr, CgiScriptNamePtr);
   }
   else
   {
      fprintf (HttpOut,
"<A HREF=\"%s?do=help\">Help</A> <I>(\
<A HREF=\"%s?do=help%%23SelectPage\">this page</A>)</I>;\n\
<A HREF=\"%s%s?do=select:query\">Query All Fields</A>;\n\
<A HREF=\"%s%s?do=build\">Build Query Form</A>.\n",
      CgiScriptNamePtr,
      CgiScriptNamePtr,
      CgiScriptNamePtr, CgiPathInfoPtr,
      CgiScriptNamePtr, CgiPathInfoPtr);
   }

   fprintf (HttpOut,
"<P><HR>\n\
<FORM ACTION=\"%s/%s\">\n\
<INPUT TYPE=hidden NAME=do VALUE=%s>\n\
<INPUT TYPE=hidden NAME=title VALUE=\"%s\">\n\
<INPUT TYPE=submit VALUE=\"%s\">\n\
<INPUT TYPE=reset>\n",
   CgiScriptNamePtr, DataBaseFileName,
   DoPtr,
   PageTitlePtr,
   SubmitPtr);

   if (DoBuild)
   {
      fprintf (HttpOut,
"<P>\n\
<INPUT TYPE=checkbox NAME=build VALUE=inclusion> \
Build form <I>for inclusion</I> in another document\n\
<P>\n\
Database specification ...\n\
<BR><INPUT TYPE=text NAME=alternative SIZE=50 VALUE=\"%s\">\n\
<P>\n\
Title of form <I>(optional)</I> ...\n\
<BR><INPUT TYPE=text NAME=title VALUE=\"%s\" SIZE=50>\n\
<P>\n\
Description/Explanation to be included at top of form <I>(optional)</I> ...\n\
<BR><TEXTAREA NAME=description ROWS=4 COLS=50></TEXTAREA>\n\
<P>\n\
Display all fields: \n\
<INPUT TYPE=radio NAME=Fields VALUE=\"\" CHECKED> <I>never</I> \n\
<INPUT TYPE=radio NAME=Fields VALUE=always> <I>always</I> \n\
<INPUT TYPE=radio NAME=Fields VALUE=WhenSelected> \
<I>allow user selection</I> \n",
      DataBaseFileName,
      PageTitlePtr);

      if (!UseHtmlSelectOption)
         fprintf (HttpOut,
"<P>\n\
Key: &lt;&gt; <I> Exclude </I> &lt;&gt; <I> Always </I> \
&lt;&gt; <I> Select </I> &lt;&gt; <I> Query</I>\n\
<BR>\n");
      else
         fputs ("<BR>\n", HttpOut);
   }
   else
   {
      fprintf (HttpOut,
"<P>\n\
Format: <INPUT TYPE=radio NAME=layout VALUE=table CHECKED> <I>table</I>\n\
<INPUT TYPE=radio NAME=layout VALUE=list> <I>list</I>\n\
<BR>\n\
Layout: <INPUT TYPE=radio NAME=format VALUE=fixed CHECKED> <I>fixed</I>\n\
<INPUT TYPE=radio NAME=format VALUE=html> <I>html</I>\n\
<BR>\n\
Report: <INPUT TYPE=radio NAME=fields VALUE=selected CHECKED> \
<I>selected fields</I>\n\
<INPUT TYPE=radio NAME=fields VALUE=all> <I>all fields</I>\n\
<BR>\n");
   }

   /**************************/
   /* field (variable) names */
   /**************************/

   for (idx = 0; idx < NumberOfFields; idx++)
   {
      CopyDbaseString (String, FieldArrayPtr[idx]->FieldName,
                       MaxSizeOfFieldName);

      if (DoBuild)
      {
         if (UseHtmlSelectOption)
         {
            fprintf (HttpOut,
"<BR>\n\
<SELECT NAME=_%s> \
<OPTION VALUE=\"\"> \
<OPTION VALUE=A> ALWAYS \
<OPTION VALUE=S> SELECT ",
                    String);
            if (QueryAllowed (FieldArrayPtr[idx]->FieldType))
               fprintf (HttpOut, "<OPTION VALUE=Q> QUERY ");
            fprintf (HttpOut, "</SELECT> %s \n", String);
         }
         else
         {
            fprintf (HttpOut,
"<BR>\n\
<INPUT TYPE=radio NAME=_%s VALUE=\"\" CHECKED> \
<INPUT TYPE=radio NAME=_%s VALUE=A> \
<INPUT TYPE=radio NAME=_%s VALUE=S> ",
                     String, String, String);
            if (QueryAllowed (FieldArrayPtr[idx]->FieldType))
               fprintf (HttpOut, "<INPUT TYPE=radio NAME=_%s VALUE=Q> %s ",
                        String, String);
            else
               fprintf (HttpOut, " < > %s ", String);
         }
         DescribeField (FieldArrayPtr[idx]);
      }
      else
      if (SelectQuery)
      {
         if (QueryAllowed (FieldArrayPtr[idx]->FieldType))
         {
            fputs ("<BR>\n", HttpOut);
            QueryInput (FieldArrayPtr[idx], "");
         }
         else
            fprintf (HttpOut,
               "<BR><INPUT TYPE=checkbox NAME=_%s VALUE=\"*\"> %s ",
               String, String);
      }
      else
      {
         fprintf (HttpOut,
            "<BR><INPUT TYPE=checkbox NAME=_%s VALUE=\"*\"> %s ",
            String, String);
         DescribeField (FieldArrayPtr[idx]);
      }
   }

   /*****************/
   /* complete form */
   /*****************/

   if (!DoBuild)
   {
      fprintf (HttpOut,
"<BR>\n\
<BR>\n\
Comments to be included in report header <I>(optional)</I> ...\n\
<BR><TEXTAREA NAME=description ROWS=4 COLS=50></TEXTAREA>\n");
   }

   fprintf (HttpOut,
"<BR>\n\
<BR>\n\
<INPUT TYPE=submit VALUE=\"%s\">\n\
<INPUT TYPE=reset>\n\
</FORM>\n\
<P><HR>\n",
   SubmitPtr);

   return (true);
}

/*****************************************************************************/
/*
Using the fields selected via FieldSelectionForm() in 'build' mode and 
submitted as a 'do=construct', construct a page with the appropriate selection 
boxes, query text input fields, etc., to provide a customized query form.  
This form can be used as-is (for testing, functionality, etc.) then saved in 
HTML format by the using, for inclusion in hypertext documents, etc.
*/ 

ConstructQueryForm ()

{
   register int  idx;
   register char  *cptr;

   boolean  ForInclusion;
   int  FieldCount;
   unsigned long  BinTime [2];
   unsigned short  NumTime [7];
   char  Scratch [256],
         String [256];
   char  *DataBasePtr,
         *CgiFormAlternativePtr,
         *CgiFormBuildPtr,
         *CgiFormDescriptionPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConstructQueryForm()\n");

   if (!SelectedCount)
   {
      ErrorGeneral ("No fields selected!", __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   GetCgiVar (CgiFormAlternativePtr, "WWW_FORM_ALTERNATIVE");
   GetCgiVar (CgiFormBuildPtr, "WWW_FORM_BUILD");
   if (toupper(CgiFormBuildPtr[0]) != 'I')
      ForInclusion = false;
   else
      ForInclusion = true;
   GetCgiVar (CgiFormDescriptionPtr, "WWW_FORM_DESCRIPTION");

   sys$gettim (&BinTime);
   sys$numtim (&NumTime, &BinTime);

   fprintf (HttpOut,
"%s\
<!-- Created by: %s -->\n\
<!-- For dBASEIV file: %s -->\n\
<!-- Date/Time: %02.02d/%02.02d/%02.02d %02.02d:%02.02d -->\n",
   Http200Header,
   SoftwareID,
   DataBaseFileName,
   NumTime[2], NumTime[1], NumTime[0] % 100, NumTime[3], NumTime[4]);

   if (!ForInclusion)
   {
      fprintf (HttpOut,
"<TITLE>Hyper-dBASEIV, %s</TITLE>\n\
<H1>%s%s</H1>\n",
      PageTitlePtr,
      IconImg, PageTitlePtr);
   }

   if (CgiFormDescriptionPtr[0])
      fprintf (HttpOut, "%s\n", CgiFormDescriptionPtr);

   if (!ForInclusion)
   {
      fprintf (HttpOut,
"<P> Other pages:\n\
<A HREF=\"%s?do=help\">Help</A> <I>(\
<A HREF=\"%s?do=help%%23QueryStrings\">Query Strings</A>)</I>.\n",
      CgiScriptNamePtr, CgiScriptNamePtr);
   }

   if (CgiFormAlternativePtr[0])
      DataBasePtr = CgiFormAlternativePtr;
   else
      DataBasePtr = DataBaseSpec;

   fprintf (HttpOut,
"<P><HR>\n\
<FORM ACTION=\"%s/%s\">\n\
<INPUT TYPE=hidden NAME=do VALUE=query>\n\
<INPUT TYPE=hidden NAME=title VALUE=\"%s\">\n\
<INPUT TYPE=submit>\n\
<INPUT TYPE=reset>\n\
<P>\n\
Format: <INPUT TYPE=radio NAME=layout VALUE=table CHECKED> <I>table</I>\n\
<INPUT TYPE=radio NAME=layout VALUE=list> <I>list</I>\n\
<BR>\n\
Layout: <INPUT TYPE=radio NAME=format VALUE=fixed CHECKED> <I>fixed</I>\n\
<INPUT TYPE=radio NAME=format VALUE=html> <I>html</I>\n",
   CgiScriptNamePtr, DataBasePtr,
   PageTitlePtr);

   HttpHasBeenOutput = true;

   if (toupper(CgiFormFieldsPtr[0]) == 'A')
   {
      /* display all fields 'A'lways */
      fprintf (HttpOut, "<INPUT TYPE=hidden NAME=fields VALUE=all>\n");
   }
   else
   if (toupper(CgiFormFieldsPtr[0]) == 'W')
   {
      /* allow the user to select to display all fields 'W'hen selected */
      fprintf (HttpOut,
"<BR>\n\
Report: <INPUT TYPE=radio NAME=fields VALUE=selected CHECKED> \
<I>selected fields</I>\n\
<INPUT TYPE=radio NAME=fields VALUE=all> <I>all fields</I>\n");
   }

   for (idx = FieldCount = 0; idx < NumberOfFields; idx++)
   {
      if (!SelectedArrayPtr[idx]) continue;

      CopyDbaseString (String, FieldArrayPtr[idx]->FieldName,
                       MaxSizeOfFieldName);

      sprintf (Scratch, "WWW_FORM__%s", String);
      GetCgiVar (cptr, Scratch);
      if (toupper(cptr[0]) == 'A')  /* 'A'lways */
      {
         fprintf (HttpOut, "<INPUT TYPE=hidden NAME=_%s VALUE=\"*\">\n",
                  String);
      }
      else
      if (toupper(cptr[0]) == 'Q')  /* 'Q'uery */
      {
         if (!FieldCount++) fputs ("<BR>\n", HttpOut);
         fputs ("<BR>", HttpOut);
         QueryInput (FieldArrayPtr[idx], "*");
      }
      else
      if (toupper(cptr[0]) == 'S')  /* 'S'elect */
      {
         if (!FieldCount++) fputs ("<BR>\n", HttpOut);
         fprintf (HttpOut,
            "<BR><INPUT TYPE=checkbox NAME=_%s VALUE=\"*\"> %s\n",
            String, String);
      }
      /* else ... do not generate anything for this field! */
   }

   fprintf (HttpOut,
"<BR>\n\
<BR>Comments to be included in report header <I>(optional)</I> ...\n\
<BR><TEXTAREA NAME=description ROWS=4 COLS=50></TEXTAREA>\n\
<BR><BR>\n\
<INPUT TYPE=submit>\n\
<INPUT TYPE=reset>\n\
</FORM>\n\
<P><HR>\n");

   return (true);
}

/****************************************************************************/
/*
Appends information on the field's type and size to the page being built up.
*/ 

DescribeField (DbFieldPtr FieldPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DescribeField()\n");

   switch (FieldPtr->FieldType)
   {
      case 'C' :
         fprintf (HttpOut, "<I>(character, %d <!-- %d -->)</I>\n",
                  FieldPtr->FieldWidth, FieldPtr->FieldDecimalCount);
         break;
      case 'D' :
         fprintf (HttpOut, "<I>(date, %d <!-- %d -->)</I>\n",
                  FieldPtr->FieldWidth, FieldPtr->FieldDecimalCount);
         break;
      case 'F' : 
         fprintf (HttpOut, "<I>(floating, %d <!-- %d -->)</I>\n",
                  FieldPtr->FieldWidth, FieldPtr->FieldDecimalCount);
         break;
      case 'L' :
         fprintf (HttpOut, "<I>(logical) <!-- %d %d --></I>\n",
                  FieldPtr->FieldWidth, FieldPtr->FieldDecimalCount);
         break;
      case 'M' :
         fprintf (HttpOut, "<I>(memo) <!-- %d %d --></I>\n",
                  FieldPtr->FieldWidth, FieldPtr->FieldDecimalCount);
         break;
      case 'N' :
         fprintf (HttpOut, "<I>(numeric, %d <!-- %d -->)</I>\n",
                  FieldPtr->FieldWidth, FieldPtr->FieldDecimalCount);
         break;
      default :
         fprintf (HttpOut, "<I>(unknown: ``%c'', %d <!-- %d -->)</I>\n",
                  FieldPtr->FieldType, FieldPtr->FieldWidth,
                  FieldPtr->FieldDecimalCount);
   }
}

/****************************************************************************/
/*
Adds a text input field to the page being built up, with sizes appropriate to 
the field, allowing a query string to be entered against the field name.
*/ 

QueryInput
(
DbFieldPtr FieldPtr,
char *DefaultValue
)
{
   int  Width;
   char  String [256];
   char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "QueryInput()\n");

   switch (FieldPtr->FieldType)
   {
      case 'C' :
         cptr = "<I>(string)</I>";
         break;
      case 'D' :
         cptr = "<I>(date; <TT>dd/mm/yy</TT>)</I>";
         break;
      case 'F' :
         cptr = "<I>(float)</I>";
         break;
      case 'L' :
         cptr = "<I>(boolean; <TT>y/n,t/f</TT>)</I>";
         break;
      case 'M' :
         cptr = "<I>(string; <TT>memo</TT>)</I>";
         break;
      case 'N' :
         cptr = "<I>(numeric)</I>";
         break;
      default :
         cptr = "<I>(unknown!)</I>";
   }

   CopyDbaseString (String, FieldPtr->FieldName, MaxSizeOfFieldName);

   if (FieldPtr->FieldType == 'M')
      Width = 30;
   else
      Width = FieldPtr->FieldWidth;
   if (Width < 1)
      Width = 1;
   else
   if (Width > 50)
      Width = 50;

   fprintf (HttpOut,
"<INPUT TYPE=text SIZE=%d MAXLENGTH=%d NAME=_%s VALUE=\"%s\"> %s %s\n",
   Width, Width*2, String, DefaultValue, String, cptr);
}

/****************************************************************************/
/*
Return true if a query is allowed against this variable type.
*/ 

boolean QueryAllowed (register char FieldType)

{
   switch (FieldType)
   {
      case 'C' :
      case 'D' :
      case 'L' :
      case 'M' :
         return (true);
      default :
         return (false);
   }
}

/*****************************************************************************/
/*
The was no database specification supplied with the request, prompt for one.  
Has a check to prevent infinite looping in this type of request (i.e. if a 
user clicks on the "access" button without entering a specification!)
*/ 

DataBaseNotSpecified ()

{
   char  *ScratchPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DataBaseNotSpecified()\n");

   /* prevent user "looping" by not supplying a specification again! */
   GetCgiVar (ScratchPtr, "WWW_FORM_DBIV");
   if (ScratchPtr[0])
   {
      ErrorGeneral ("No database specification supplied!", __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   fprintf (HttpOut,
"%s\
<!-- SoftwareID: %s -->\n\
<TITLE>Hyper-dBASEIV, Specify Database</TITLE>\n\
<H1>%sSpecify Database</H1>\n\
<P> Other pages:\n\
<A HREF=\"%s?do=help\">Help</A>.\n\
<P><HR>\n\
<P> Provide the VMS file path to the required <TT>dBASEIV</TT> database\n\
file (<TT>.DBF</TT> default extension), or provide a VMS directory\n\
specification to generate an index to all <TT>.DBF</TT> database files.\n\
<P>\n\
<FORM ACTION=%s>\n\
<INPUT TYPE=hidden NAME=dBIV VALUE=yes>\n\
Specification: <INPUT TYPE=text NAME=database SIZE=40>\n\
<INPUT TYPE=submit VALUE=\"Access\">\n\
<INPUT TYPE=reset VALUE=\"Reset\">\n\
</FORM>\n\
<P><HR>\n",
   Http200Header,
   SoftwareID,
   IconImg,
   CgiScriptNamePtr,
   CgiScriptNamePtr);
}

/****************************************************************************/
/*
Provide anchors for all the .DBF files according to the specification provided 
in 'DataBaseFileName'.
*/ 

IndexWildCardDataBase ()

{
   int  status,
        FileCount = 0;
   char  ExpandedFileSpec [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "IndexWildCardDataBase()\n");

   /* initialize the file access block */
   DbFileFab = cc$rms_fab;
   DbFileFab.fab$l_fna = DataBaseSpec;
   DbFileFab.fab$b_fns = strlen(DataBaseSpec);
   DbFileFab.fab$l_dna = "*.DBF;";
   DbFileFab.fab$b_dns = 6;
   DbFileFab.fab$l_fop = FAB$V_NAM;
   DbFileFab.fab$l_nam = &DbFileNam;

   /* initialize the file name block */
   DbFileNam = cc$rms_nam;
   DbFileNam.nam$l_esa = ExpandedFileSpec;
   DbFileNam.nam$b_ess = sizeof(ExpandedFileSpec)-1;
   DbFileNam.nam$l_rsa = DataBaseFileName;
   DbFileNam.nam$b_rss = sizeof(DataBaseFileName)-1;

   if (VMSnok (status = sys$parse (&DbFileFab, 0, 0)))
   {
      ErrorVmsStatus (status, PageTitlePtr, DataBaseSpec, __FILE__, __LINE__);
      return;
   }

   fprintf (HttpOut,
"%s\
<!-- SoftwareID: %s -->\n\
<TITLE>Hyper-dBASEIV, Index of <TT>%s</TT></TITLE>\n\
<H1>%sIndex of <TT>%s</TT></H1>\n\
<P> Other pages:\n\
<A HREF=\"%s?do=help\">Help</A>.\n\
<P><HR>\n",
   Http200Header,
   SoftwareID,
   DataBaseSpec,
   IconImg, DataBaseSpec,
   CgiScriptNamePtr);

   while (VMSok (status = sys$search (&DbFileFab, 0, 0)))
   {
      /* ignore any other than .DBF (numeric ".DBF") */
      if (*(unsigned long*)DbFileNam.nam$l_type != 0x4642442e) continue;

      if (!FileCount++) fputs ("<OL>\n", HttpOut);

      *(char*)DbFileNam.nam$l_ver = '\0';
      fprintf (HttpOut, "<LI><A HREF=\"%s/%s?do=select\">%s</A>\n",
               CgiScriptNamePtr, DataBaseFileName, (char*)DbFileNam.nam$l_name);
      *(char*)DbFileNam.nam$l_ver = ';';
   }
   if (status != RMS$_FNF && status != RMS$_NMF)
   {
      ErrorVmsStatus (status, PageTitlePtr, DataBaseFileName,
                      __FILE__, __LINE__);
      return;
   }

   if (FileCount)
      fputs ("</OL>\n<P><HR>\n", HttpOut);
   else
      fputs ("<P>\nNo <TT>.DBF</TT> files found.\n<P><HR>\n", HttpOut);
}

/*****************************************************************************/
/*
Eliminate leading and trailing white-space, compress other down to a single 
space.  Non-printable characters are considered white-space.  Return the 
number of characters resulting.
*/ 

int CompressWhiteSpace (char *String)

{
   register char  *cptr, *sptr;

   cptr = sptr = String;
   while (isspace(*cptr)) cptr++;
   for (;;)
   {
      while (!isspace(*cptr) && isprint(*cptr)) *sptr++ = *cptr++;
      if (!*cptr) break;
      if (isspace (*cptr) || !isprint(*cptr))
      {
         while (*cptr && (isspace(*cptr) || !isprint(*cptr))) cptr++;
         if (!*cptr) break;
         *sptr++ = ' ';
      }
   }
   *sptr = '\0';
   return (sptr - String);
}

/*****************************************************************************/
/*
Assuming a space-compressed string, substitute line-feeds for spaces at 
appropriate points to ensure lines do not exceed the specified text width 
(unless there is no previous space to break the line).  Return the number of 
lines generated.
*/ 

WrapAt
(
register char *cptr,
register int TextWidth
)
{
   register int  ccnt, lcnt;
   register char  *sptr;

   /*********/
   /* begin */
   /*********/

   sptr = cptr;
   ccnt = 0;
   lcnt = 1;
   while (*cptr)
   {
      ccnt++;
      if (*cptr == ' ') sptr = cptr;
      if (ccnt > TextWidth)
      {
         ccnt = cptr - sptr;
         if (*sptr == ' ') { *sptr = '\n'; lcnt++; }
         sptr = cptr;
      }
      cptr++;
   }
   if (ccnt > TextWidth && *sptr == ' ') { *sptr = '\n'; lcnt++; }
   return (lcnt);
}

/*****************************************************************************/
/*
Copy DBASE-string (fixed-length, space filled) to C-string (trimmed and null-
terminated).  Returns the number of characters copied.
*/ 

int CopyDbaseString
(
register char *sptr,
register char *dptr,
register int ccnt
)
{
   register char  *zptr;
   char  *FirstPtr;

   /*********/
   /* begin */
   /*********/

   FirstPtr = zptr = sptr;
   while (*dptr && ccnt--)
      if ((*sptr++ = *dptr++) != ' ') zptr = sptr;
   *zptr = '\0';
   return (zptr-FirstPtr);
}

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text in HTML.  For example the '<', '&', etc.  Convert these 
to the corresponding HTML character entities.  Returns number of characters 
copied into HTML string.
*/ 

int CopyIntoHtml
( 
register char *hptr,
register char *sptr,
register int ccnt
)
{
   char  *FirstPtr;

   /*********/
   /* begin */
   /*********/

   FirstPtr = hptr;
   while (*sptr && ccnt--)
   {
      switch (*sptr)
      {
         case '<' : strcpy (hptr, "&lt;"); hptr += 4; sptr++; break;
         case '>' : strcpy (hptr, "&gt;"); hptr += 4; sptr++; break;
         case '&' : strcpy (hptr, "&amp;"); hptr += 5; sptr++; break;
         case '\"' : strcpy (hptr, "&quot;"); hptr += 6; sptr++; break;
         case '\r' : *hptr++ = *sptr++; break;
         case '\n' : *hptr++ = *sptr++; break;
         default : if (isprint(*sptr)) *hptr++ = *sptr++; else sptr++;
      }
   }
   *hptr = '\0';
   return (hptr-FirstPtr);
}

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text text in an HTTP URL.  For example the '?', '+', '&', etc.  
Convert these to "%nn" hexadecimal escaped characters.
*/ 

int CopyIntoUri
( 
register char *uptr,
register char *sptr,
register int  ccnt
)
{
   char  *FirstPtr;

   /*********/
   /* begin */
   /*********/

   FirstPtr = uptr;
   while (*sptr && ccnt--)
   {  
      switch (*sptr)
      {
         case '\t' : strcpy (uptr, "%09"); uptr += 3; sptr++; break;
         case ' ' : strcpy (uptr, "%20"); uptr += 3; sptr++; break;
         case '!' : strcpy (uptr, "%21"); uptr += 3; sptr++; break;
         case '\"' : strcpy (uptr, "%22"); uptr += 3; sptr++; break;
         case '#' : strcpy (uptr, "%23"); uptr += 3; sptr++; break;
         case '$' : strcpy (uptr, "%24"); uptr += 3; sptr++; break;
         case '%' : strcpy (uptr, "%25"); uptr += 3; sptr++; break;
         case '&' : strcpy (uptr, "%26"); uptr += 3; sptr++; break;
         case '\'' : strcpy (uptr, "%27"); uptr += 3; sptr++; break;
         case '(' : strcpy (uptr, "%28"); uptr += 3; sptr++; break;
         case ')' : strcpy (uptr, "%29"); uptr += 3; sptr++; break;
         case '*' : strcpy (uptr, "%2a"); uptr += 3; sptr++; break;
         case '+' : strcpy (uptr, "%2b"); uptr += 3; sptr++; break;
         case ',' : strcpy (uptr, "%2c"); uptr += 3; sptr++; break;
         case '-' : strcpy (uptr, "%2d"); uptr += 3; sptr++; break;
         case '.' : strcpy (uptr, "%2e"); uptr += 3; sptr++; break;
         case '/' : strcpy (uptr, "%2f"); uptr += 3; sptr++; break;
         case ':' : strcpy (uptr, "%3a"); uptr += 3; sptr++; break;
         case ';' : strcpy (uptr, "%3b"); uptr += 3; sptr++; break;
         case '<' : strcpy (uptr, "%3c"); uptr += 3; sptr++; break;
         case '=' : strcpy (uptr, "%3d"); uptr += 3; sptr++; break;
         case '>' : strcpy (uptr, "%3e"); uptr += 3; sptr++; break;
         case '?' : strcpy (uptr, "%3f"); uptr += 3; sptr++; break;
         case '[' : strcpy (uptr, "%5b"); uptr += 3; sptr++; break;
         case '\\' : strcpy (uptr, "%5c"); uptr += 3; sptr++; break;
         case ']' : strcpy (uptr, "%5d"); uptr += 3; sptr++; break;
         case '^' : strcpy (uptr, "%5e"); uptr += 3; sptr++; break;
         case '_' : strcpy (uptr, "%5f"); uptr += 3; sptr++; break;
         case '{' : strcpy (uptr, "%7b"); uptr += 3; sptr++; break;
         case '|' : strcpy (uptr, "%7c"); uptr += 3; sptr++; break;
         case '}' : strcpy (uptr, "%7d"); uptr += 3; sptr++; break;
         case '~' : strcpy (uptr, "%7e"); uptr += 3; sptr++; break;
         default : if (isprint(*sptr)) *uptr++ = *sptr++; else sptr++;
      }
   }
   *uptr = '\0';
   return (uptr-FirstPtr);
}

/****************************************************************************/
/*
Query this field, either as a wildcarded pattern match, or as a numeric range 
check.  The numeric range-check is enabled if the first character of the query 
field is a "<", "=", or ">" character.  If a required pattern match begins 
with any of these characters then escape this by making the first character a 
blackslash.

The field is scanned for the FIRST number (integer or floating point).  
Multiple numbers, mixed with any other characters, may be in the field but 
just the first one is extracted.  If the field does not contain a (leading) 
number then the range check immediately fails.

The range-checking is quite elementary. Permitted equality symbols are "<", 
"<=", "=", ">=", ">", and "<>" (not equal-to).  Terms are implicitly .AND.ed, 
until a comma is encountered (implicit .OR.), at which point the result so far 
is checked and if true evaluation ceases and true returned, if not then 
another set of terms are evaluated.  Some examples: 

The query string "<25" returns true if the field is less than 25.

The query string ">10<=25" returns true if the field is greater than 10 and 
less than or equal to 25.

The query string "<20,>30" returns true if the field is less than 20 or
greater than 30.

The query string ">10<=25,>=60<100" returns true if the field is greater than 
10 and less than or equal to 25, or if it is greater than or equal to 60 and 
less than100. 
*/

boolean QueryField
(
char *FieldString,
char *QueryString
)
{
   register char  *eptr, *qptr;

   boolean  Ok,
            SoFarOk;
   float  FieldValue,
          Value;
   char  String [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "QueryField()\n|%s|%s|\n", FieldString, QueryString);

   qptr = QueryString;

   /**************************/
   /* wildcard pattern match */
   /**************************/

   /* if the query doesn't begin with equality symbol */
   if (*qptr != '<' && *qptr != '=' && *qptr != '>')
      return (PatternMatch (FieldString, QueryString));
   /* the backslash escapes a wildcard match beginning with equality symbol */
   if (*qptr == '\\')
      return (PatternMatch (FieldString, QueryString+1));

   /******************/
   /* range checking */
   /******************/

   /* if the field does not contain a number then you can't range check it! */
   if (sscanf (FieldString, "%f", &FieldValue) < 1)
      return (false);

   /* start off with a true value that can initially be .AND.ed with */
   Ok = true;
   while (*qptr)
   {
      while (isspace(*qptr)) qptr++;
      if (!*qptr) break;

      if (*qptr == ',')
      {
         /* decision time, an .OR., if within range so far return true */
         if (SoFarOk) return (true);
         /* new true value that can initially be .AND.ed with */
         Ok = true;
         /* skip over (one or more consecutive) comma(s) */
         while (*qptr == ',') qptr++;
         if (!*qptr) break;
         continue;
      }

      while (*qptr && !isdigit(*qptr) && 
             *qptr != '<' && *qptr != '=' && *qptr != '>') qptr++;
      if (!*qptr) break;

      if (*qptr == '<' || *qptr == '=' || *qptr == '>')
      {
         eptr = qptr;
         while (*qptr == '<' || *qptr == '=' || *qptr == '>') qptr++;
      }
      else
      {
         if (sscanf (qptr, "%f", &Value) < 1) return (false);

         if (Debug)
            fprintf (stdout, "|%f|%2.2s|%f|\n", FieldValue, eptr, Value);

         if (eptr[0] == '<' && eptr[1] != '=')
            SoFarOk = Ok = (Ok && (FieldValue < Value));
         else
         if (eptr[0] == '<' && eptr[1] == '=')
            SoFarOk = Ok = (Ok && (FieldValue <= Value));
         else
         if (eptr[0] == '=')
            SoFarOk = Ok = (Ok && (FieldValue == Value));
         else
         if (eptr[0] == '>' && eptr[1] == '=')
            SoFarOk = Ok = (Ok && (FieldValue >= Value));
         else
         if (eptr[0] == '>' && eptr[1] != '=')
            SoFarOk = Ok = (Ok && (FieldValue > Value));
         else
         if (eptr[0] == '<' && eptr[1] == '>')
            SoFarOk = Ok = (Ok && (FieldValue != Value));

         /* skip over the number */
         while (*qptr == '+' || *qptr == '-') qptr++;
         while (isdigit(*qptr) || *qptr == '.') qptr++;

         /* any number without an equality symbol will now be 'equal-to' */
         eptr == "=";
      }
   }

   return (Ok);
}

/****************************************************************************/
/*
String search allowing wildcard "*" (matching any multiple characters) and "%" 
(matching any single character).  Returns true or false.
*/ 

boolean PatternMatch
(
char *InThat,
char *This
)
{
   register char  *inptr,
                  *thptr,
                  *RestartThptr,
                  *RestartInptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PatternMatch()\n|%s|%s|\n", This, InThat);

   thptr = This;
   inptr = InThat;
   for (;;)
   {
      while (*thptr && *inptr && toupper(*thptr) == toupper(*inptr))
      {
         thptr++;
         inptr++;
      }
      if (!*thptr && !*inptr) return (true);
      if (*thptr != '*' && *thptr != '%') return (false);

      if (*thptr == '%')
      {
         /* single char wildcard processing */
         if (!*inptr) return (false);
         thptr++;
         inptr++;
         continue;
      }
      /* asterisk wildcard matching */
      while (*thptr == '*') thptr++;
      /* an asterisk wildcard at end matches all following */
      if (!*thptr) return (true);
      /* note the current position in the string (first after the wildcard) */
      RestartThptr = thptr;
      for (;;)
      {
         /* find first char in InThat matching char after wildcard */
         while (*inptr && toupper(*thptr) != toupper(*inptr)) inptr++;
         /* if did not find matching char in InThat being searched */
         if (!*inptr) return (false);
         /* note the current position in InThat being searched */
         RestartInptr = inptr;
         /* try to match the remainder of the string and InThat */
         while (*thptr && *inptr && toupper(*thptr) == toupper(*inptr))
         {
            thptr++;
            inptr++;
         }
         /* if reached the end of both string and InThat - match! */
         if (!*thptr && !*inptr) return (true);
         /* break to the external loop if we encounter another wildcard */
         if (*thptr == '*' || *thptr == '%') break;
         /* lets have another go */
         thptr = RestartThptr;
         /* starting the character following the previous attempt */
         inptr = RestartInptr + 1;
      }
   }
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n",
   SoftwareID, cptr, SourceLineNumber, Text);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      exit (status);

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n",
   SoftwareID, cptr, SourceLineNumber, Message, Text, StatusValue, HiddenText);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/

