/*****************************************************************************/
/*
                                Auth.c

This implementation of WWW "Basic" authentication uses the SYSUAF to validate 
a remote username/password in the "local" authentication "realm".  It is 
designed to do this only, and is not a full implementation of authentication 
control. 

A linked-list, binary tree (using the the LIB$..._TREE routines) is used to 
store authentication records.  When a request with an "Authorization:" header 
line encounters a point where authentication is required this binary tree is 
checked for an existing record.  If one does not exist a new one is entered 
into the binary tree, with an empty password field.  The password in the 
authorization line is checked against the password in the record.  If it 
matches the request is authenticated.  If not the SYSUAF password is checked.  
If the hashed password matches, the plain password is copied into the record 
and used for future authentications.  If not, the authentication fails. 

Due to the use of the SYSUAF for password validation it is necessary to 
incorporate authentication failure recording and evasion to prevent password 
attacks via the HTTPD server.  When a given username exceeds a defined limit 
all attempts at authentication fail without reference to the SYSUAF.  This 
failure count record must be reset manually by systems staff. 

Note that at present only one "realm" is supported - "local".


VERSION HISTORY
---------------
01-DEC-95  MGD  HTTPd version 3
01-APR-95  MGD  initial development for addition to multi-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <descrip.h>
#include <iodef.h>
#include <libdef.h>
#include <prvdef.h>
#include <rms.h>
#include <rmsdef.h>
#include <ssdef.h>
#include <stsdef.h>

#include <uaidef.h>
/* not defined in VAX C 3.2 <uaidef.h> */
#define UAI$M_RESTRICTED 0x8

/* application related header file */
#include "httpd.h"

/**************************/
/* structure declarations */
/**************************/

struct AuthRecordStruct
{
   struct AuthRecordStruct  *LeftLink;
   struct AuthRecordStruct  *RightLink;
   unsigned short  Reserved;
   char  Realm [17];
   char  UserName [17];
   char  Password [17];
   unsigned long  AccessCount;
   unsigned long  FailureCount;
};

/******************/
/* global storage */
/******************/

int  AuthFailureLimit = 20;
int  AuthDisplayRecordCount;
struct AuthRecordStruct  *AuthTreeHead = 0;

/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern unsigned short  ControlMbxChannel;
extern char  ErrorMemoryAllocation[];
extern char  ErrorStringSize[];
extern char  SoftwareID[];
extern char  Utility[];
extern struct ConfigStruct  Config;

/****************************/
/* functions in this module */
/****************************/

Authenticate (struct RequestStruct*);
AuthAllList ();
AuthFailList ();
AuthFailResetCount ();
char* AuthPrintableDecode (char*, char*, int);
int AuthTreeAllocateRecord (struct AuthRecordStruct*,
                            struct AuthRecordStruct**,
                            unsigned long);
int AuthTreeCompareRecord (struct AuthRecordStruct*,
                           struct AuthRecordStruct*,
                           unsigned long);
int AuthTreeDisplayRecord (struct AuthRecordStruct*, boolean);
int AuthVerifyUafPassword (char*, char*);

/*************************************/
/* prototypes for external functions */
/*************************************/

ControlMbxOutput (char*, int);
ErrorExitVmsStatus (int, char*, char*, int);
ErrorVmsStatus (struct RequestStruct*, int, char*, int);

/*****************************************************************************/
/*
Parse the "Authorization:" request header line.  Decode the username and 
password.  Look for an authentication record with the realm and username in 
the linked-list, binary tree.  If one doesn't exists create a new one.  Check 
the supplied password against the one in the record.  If it matches then 
authentication has succeeded.  If not, the check the supplied password against 
the database.  If it matches then copy the supplied password into the 
authentication record and the authentication succeeds.  If it doesn't match 
then the authentication fails.
*/ 

boolean Authenticate (struct RequestStruct *RequestPtr)

{
   static int  NoDuplicatesFlag = 0;

   register char  *cptr, *sptr, *zptr;

   int  status;
   char  EncodedString [256],
         UserNamePassword [256];
   struct AuthRecordStruct  AuthRecord;
   struct AuthRecordStruct  *AuthRecordPtr,
                            *TreeNodePtr;
   
   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "Authenticate() |%s|\n",
               RequestPtr->HttpAuthorizationPtr);

   /***********************************/
   /* get the HTTP authorization line */
   /***********************************/

   if (RequestPtr->HttpAuthorizationPtr == NULL)
   {
      RequestPtr->RemoteUser[0] = RequestPtr->RemoteUserPassword[0] = '\0';
      ErrorInternal (RequestPtr, STS$K_ERROR, "When authentication user.",
                     __FILE__, __LINE__);
      return (STS$K_ERROR);
   }

   cptr = RequestPtr->HttpAuthorizationPtr;

   zptr = (sptr = RequestPtr->AuthType) + sizeof(RequestPtr->AuthType);
   while (*cptr && !isspace(*cptr) && sptr < zptr) *sptr++ = toupper(*cptr++);
   if (sptr >= zptr)
   {
      RequestPtr->RemoteUser[0] = RequestPtr->RemoteUserPassword[0] = '\0';
      RequestPtr->ResponseStatusCode = 500;
      ErrorGeneral (ErrorStringSize, __FILE__, __LINE__);
      return (STS$K_ERROR);
   }
   *sptr = '\0';
   if (strcmp (RequestPtr->AuthType, "BASIC"))
   {
      RequestPtr->RemoteUser[0] = RequestPtr->RemoteUserPassword[0] = '\0';
      RequestPtr->ResponseStatusCode = 501;
      ErrorGeneral ("Only <TT>BASIC</TT> authentication acceptable",
                    __FILE__, __LINE__);
      return (STS$K_ERROR);
   }

   /* skip over white-space between scheme and encoded string */
   while (*cptr && isspace(*cptr)) cptr++;

   /* get the RFC1421-encoded "username:password" string */
   zptr = (sptr = EncodedString) + sizeof(EncodedString);
   while (*cptr && !isspace(*cptr) && sptr < zptr) *sptr++ = *cptr++;
   if (sptr >= zptr)
   {
      RequestPtr->RemoteUser[0] = RequestPtr->RemoteUserPassword[0] = '\0';
      RequestPtr->ResponseStatusCode = 500;
      ErrorGeneral (ErrorStringSize, __FILE__, __LINE__);
      return (STS$K_ERROR);
   }
   *sptr = '\0';

   /************************************************/
   /* decode and extract the username and password */
   /************************************************/

   sptr = AuthPrintableDecode (EncodedString, UserNamePassword,
                               sizeof(UserNamePassword));
   if (sptr[0])
   {
      /* an error report string has been returned by the decode function */
      RequestPtr->RemoteUser[0] = RequestPtr->RemoteUserPassword[0] = '\0';
      RequestPtr->ResponseStatusCode = 500;
      ErrorGeneral (RequestPtr, sptr, __FILE__, __LINE__);
      return (STS$K_ERROR);
   }
                                                                   
   cptr = UserNamePassword;

   /* get the remote username from the decoded "username:password" string */
   zptr = (sptr = RequestPtr->RemoteUser) + sizeof(RequestPtr->RemoteUser);
   while (*cptr && *cptr != ':' && sptr < zptr) *sptr++ = toupper(*cptr++);
   if (sptr >= zptr)
   {
      RequestPtr->RemoteUser[0] = RequestPtr->RemoteUserPassword[0] = '\0';
      RequestPtr->ResponseStatusCode = 500;
      ErrorGeneral (RequestPtr, ErrorStringSize, __FILE__, __LINE__);
      return (STS$K_ERROR);
   }
   *sptr = '\0';
   if (Debug) fprintf (stdout, "RemoteUser |%s|\n", RequestPtr->RemoteUser);
                                             
   /* get the password from the decoded "username:password" string */
   if (*cptr == ':') cptr++;
   zptr = (sptr = RequestPtr->RemoteUserPassword) +
          sizeof(RequestPtr->RemoteUserPassword);
   while (*cptr && sptr < zptr) *sptr++ = toupper(*cptr++);
   if (sptr >= zptr)
   {
      RequestPtr->RemoteUser[0] = RequestPtr->RemoteUserPassword[0] = '\0';
      RequestPtr->ResponseStatusCode = 500;
      ErrorGeneral (RequestPtr, ErrorStringSize, __FILE__, __LINE__);
      return (STS$K_ERROR);
   }
   *sptr = '\0';
/*
   if (Debug)
      fprintf (stdout, "Password |%s|\n", RequestPtr->RemoteUserPassword);
*/

   /*********************/
   /* lookup the record */
   /*********************/

   strcpy (AuthRecord.Realm, RequestPtr->AuthRealm);
   strcpy (AuthRecord.UserName, RequestPtr->RemoteUser);
   strcpy (AuthRecord.Password, RequestPtr->RemoteUserPassword);

   status = lib$lookup_tree (&AuthTreeHead, &AuthRecord,
                             &AuthTreeCompareRecord, &TreeNodePtr);
   if (Debug) fprintf (stdout, "lib$lookup_tree() %%X%08.08X\n", status);

   if (VMSok (status))
   {
      /*******************/
      /* existing record */
      /*******************/

      if (Debug)
         fprintf (stdout, "AuthRecord |%s|%s| access: %d failure: %d\n",
                  TreeNodePtr->UserName, TreeNodePtr->Realm,
                  TreeNodePtr->AccessCount, TreeNodePtr->FailureCount);

      if (TreeNodePtr->FailureCount > AuthFailureLimit)
      {
         /* automatically fail those above the allowed limit */
         TreeNodePtr->FailureCount++;
         RequestPtr->RemoteUser[0] = RequestPtr->RemoteUserPassword[0] = '\0';
         if (Debug) fprintf (stdout, "NOT authenticated!\n");
         return (SS$_NORMAL); 
      }

      if (!strcmp (RequestPtr->RemoteUserPassword, TreeNodePtr->Password))
      {
         /* authenticated */
         TreeNodePtr->FailureCount = 0;
         TreeNodePtr->AccessCount++;
         if (Debug) fprintf (stdout, "authenticated!\n");
         return (SS$_NORMAL); 
      }
   }
   else
   if (status == LIB$_KEYNOTFOU)
   {
      /****************************************/
      /* record not found in tree, create one */
      /****************************************/

      if ((AuthRecordPtr = calloc (1, sizeof (struct AuthRecordStruct)))
          == NULL)
      {
         RequestPtr->RemoteUser[0] = RequestPtr->RemoteUserPassword[0] = '\0';
         RequestPtr->ResponseStatusCode = 500;
         ErrorGeneral (ErrorMemoryAllocation, __FILE__, __LINE__);
         return (STS$K_ERROR);
      }
      
      strcpy (AuthRecordPtr->Realm, RequestPtr->AuthRealm);
      strcpy (AuthRecordPtr->UserName, RequestPtr->RemoteUser);
      AuthRecordPtr->Password[0] = '\0';
      AuthRecordPtr->AccessCount = AuthRecordPtr->FailureCount = 0;

      if (VMSnok (status =
          lib$insert_tree (&AuthTreeHead, AuthRecordPtr,
                           &NoDuplicatesFlag, &AuthTreeCompareRecord,
                           &AuthTreeAllocateRecord, &TreeNodePtr, 0)))
         free (AuthRecordPtr);
      if (Debug) fprintf (stdout, "lib$insert_tree() %%X%08.08X\n", status);
   }

   if (VMSnok (status))
   {
      RequestPtr->RemoteUser[0] = RequestPtr->RemoteUserPassword[0] = '\0';
      RequestPtr->ErrorTextPtr = "when authenticating user";
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      return (status);
   }

   /********************/
   /* (re)set password */
   /********************/

   if (VMSok (status = 
       AuthVerifyUafPassword (RequestPtr->RemoteUser,
                              RequestPtr->RemoteUserPassword)))
      strcpy (AuthRecordPtr->Password, RequestPtr->RemoteUserPassword);
   else
   if (status == SS$_INVLOGIN)
      AuthRecordPtr->Password[0] = '\0';
   else
   {
      RequestPtr->RemoteUser[0] = RequestPtr->RemoteUserPassword[0] = '\0';
      RequestPtr->ErrorTextPtr = "when authenticating user";
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      return (status);
   }

   /******************/
   /* check password */
   /******************/

   if (Debug)
      fprintf (stdout, "AuthRecord |%s|%s| access: %d failure: %d\n",
               TreeNodePtr->UserName, TreeNodePtr->Realm,
               TreeNodePtr->AccessCount, TreeNodePtr->FailureCount);

   if (!strcmp (RequestPtr->RemoteUserPassword, TreeNodePtr->Password))
   {
      /* authenticated */
      TreeNodePtr->FailureCount = 0;
      TreeNodePtr->AccessCount++;
      if (Debug) fprintf (stdout, "authenticated!\n");
   }
   else
   {
      /* not authenticated */
      TreeNodePtr->FailureCount++;
      RequestPtr->RemoteUser[0] = RequestPtr->RemoteUserPassword[0] = '\0';
      if (Debug) fprintf (stdout, "NOT authenticated!\n");
   }

   return (SS$_NORMAL); 
}

/*****************************************************************************/
/*
Called by lib$insert_tree() and lib$lookup_tree().  Returns negative number if
user node is lexicographically less than the tree node, positive if greater,
or zero if exactly the same.
*/ 

AuthTreeCompareRecord
(
struct AuthRecordStruct *UserNodePtr,
struct AuthRecordStruct *TreeNodePtr,
unsigned long Unused
)
{
   int  Result;
   
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthTreeCompareRecord()\n");

   if (Result = strcmp (UserNodePtr->Realm, TreeNodePtr->Realm))
      return (Result);
   return (strcmp (UserNodePtr->UserName, TreeNodePtr->UserName));
}

/*****************************************************************************/
/*
Called by lib$insert_tree().  Insert the address in 'UserNodePtr' into the
location pointed at by 'NewTreeNodePtrAddress' (inside the binary tree
structure presumably!)
*/ 

AuthTreeAllocateRecord
(
struct AuthRecordStruct *UserNodePtr,
struct AuthRecordStruct **NewTreeNodePtrAddress,
unsigned long Unused
)
{
   if (Debug) fprintf (stdout, "AuthTreeAllocateRecord()\n");

   *NewTreeNodePtrAddress = UserNodePtr;

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Called by lib$traverse_tree().  Display the details of either all records, or 
only those with failure counts, in the linked-list binary tree.  Output the 
formatted details via the control mailbox.
*/

AuthTreeDisplayRecord
(                      
struct AuthRecordStruct *NodePtr,
boolean OnlyFailureRecords
)
{
   static $DESCRIPTOR (FailureRecordFaoDsc, "!16AZ !16AZ !8UL !8UL");

   unsigned short  Length;
   char  String [1024];
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthTreeDisplayRecord()\n");

   if (Debug)
      fprintf (stdout, "AuthRecord |%s|%s| access: %d failure: %d\n",
               NodePtr->Realm, NodePtr->UserName,
               NodePtr->AccessCount, NodePtr->FailureCount);

   if (OnlyFailureRecords && !NodePtr->FailureCount) return (SS$_NORMAL);

   sys$fao (&FailureRecordFaoDsc, &Length, &StringDsc,
            NodePtr->Realm, NodePtr->UserName,
            NodePtr->AccessCount, NodePtr->FailureCount);
   String[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);
   ControlMbxOutput (String, Length);

   AuthDisplayRecordCount++;
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Display all records in the authentication linked-list, binary tree.
*/

AuthAllList ()

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthAllList()\n");

   AuthDisplayRecordCount = 0;
   status = lib$traverse_tree (&AuthTreeHead, &AuthTreeDisplayRecord, false);
   if (Debug) fprintf (stdout, "lib$traverse_tree() %%X%08.08X\n", status);
   return (AuthDisplayRecordCount);
}

/*****************************************************************************/
/*
Display records with a non-zero failure count in the authentication linked-
list, binary tree.
*/ 

AuthFailList ()

{
   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthFailList()\n");

   AuthDisplayRecordCount = 0;
   status = lib$traverse_tree (&AuthTreeHead, &AuthTreeDisplayRecord, true);
   if (Debug) fprintf (stdout, "lib$traverse_tree() %%X%08.08X\n", status);
   return (AuthDisplayRecordCount);
}

/*****************************************************************************/
/*
Reset the specified record's failure count to zero.  Output any messages via 
the control mailbox.
*/

AuthFailResetCount
(
char *Realm,
char *Name
)
{
   static $DESCRIPTOR (RecordNotFoundFaoDsc, "Record not found: \"!AZ:!AZ\"");
   static $DESCRIPTOR (VmsStatusFaoDsc, "Error status: %X!XL");

   int  status;
   unsigned short  Length;
   char  String [1024];
   $DESCRIPTOR (StringDsc, String);
   struct AuthRecordStruct  AuthRecord;
   struct AuthRecordStruct  *TreeNodePtr;
   
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthFailResetCount()\n");

   strcpy (AuthRecord.Realm, Realm);
   strcpy (AuthRecord.UserName, Name);

   status = lib$lookup_tree (&AuthTreeHead, &AuthRecord,
                             &AuthTreeCompareRecord, &TreeNodePtr);
   if (Debug) fprintf (stdout, "lib$lookup_tree() %%X%08.08X\n", status);

   if (VMSok (status))
   {
      /* record exists */
      if (Debug)
         fprintf (stdout, "AuthRecord |%s|%s| access: %d failure: %d\n",
                  TreeNodePtr->UserName, TreeNodePtr->Realm,
                  TreeNodePtr->AccessCount, TreeNodePtr->FailureCount);

      TreeNodePtr->FailureCount = 0;
   }
   else
   if (status == LIB$_KEYNOTFOU)
   {
      /* record not found in tree */
      sys$fao (&RecordNotFoundFaoDsc, &Length, &StringDsc, Realm, Name);
      String[Length] = '\0';
      if (Debug) fprintf (stdout, "|%s|\n", String);
      ControlMbxOutput (String, Length);
   }
   else
   {
      sys$fao (&VmsStatusFaoDsc, &Length, &StringDsc, status);
      String[Length] = '\0';
      if (Debug) fprintf (stdout, "|%s|\n", String);
      ControlMbxOutput (String, Length);
   }

   return (status);
}

/*****************************************************************************/
/*
Get the specified user's flags, quadword password, hash salt and encryption
algorithm from UAF.  If any of specified bits in the flags are set (e.g.
"disusered") then return an invalid login status.  Using the salt and
encryption algorithm hash the supplied password and compare it to the UAF
hashed password.  If the same return a normal status, if not return an invalid
login status.  This function can also return VMS error status.  It is up to the
calling code to interpret the return values and behave accordingly.
*/ 

int AuthVerifyUafPassword
(
char *UserName,
char *Password
)
{
   static unsigned long  SysPrvMask [2] = { PRV$M_SYSPRV, 0 };
   static unsigned long  DisallowFlags =
          UAI$M_DISACNT | UAI$M_PWD_EXPIRED | UAI$M_PWD2_EXPIRED |
          UAI$M_CAPTIVE | UAI$M_RESTRICTED;

   int  status,
        SetPrvStatus;
   unsigned long  HashedPwd [2],
                  UaiFlags,
                  UaiPwd [2];
   unsigned short  UaiSalt;
   unsigned char  UaiEncrypt;
   struct {
      short BufferLength;
      short ItemCode;
      void  *BufferPtr;
      void  *LengthPtr;
   } items [] = 
   {
      { sizeof(UaiFlags), UAI$_FLAGS, &UaiFlags, 0 },
      { sizeof(UaiPwd), UAI$_PWD, &UaiPwd, 0 },
      { sizeof(UaiEncrypt), UAI$_ENCRYPT, &UaiEncrypt, 0 },
      { sizeof(UaiSalt), UAI$_SALT, &UaiSalt, 0 },
      { 0, 0, 0, 0 }
   };
   $DESCRIPTOR (UserNameDsc, UserName);
   $DESCRIPTOR (PasswordDsc, Password);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthVerifyUafPassword()\n");

   if (!Config.AuthLocalEnabled) return (SS$_INVLOGIN);

   /* turn on SYSPRV to allow access to SYSUAF records */
   if (VMSnok (SetPrvStatus = sys$setprv (1, &SysPrvMask, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", SetPrvStatus);
      return (SetPrvStatus);
   }

   UserNameDsc.dsc$w_length = strlen(UserName);
   status = sys$getuai (0, 0, &UserNameDsc, &items, 0, 0, 0);
   if (Debug) fprintf (stdout, "sys$getuai() %%X%08.08X\n", status);

   /* turn off SYSPRV */
   if (VMSnok (SetPrvStatus = sys$setprv (0, &SysPrvMask, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", SetPrvStatus);
      /* do NOT keep processing if there is a problem turning off SYSPRV! */
      ErrorExitVmsStatus (SetPrvStatus, "reducing privileges",
                          __FILE__, __LINE__);
   }

   if (status == RMS$_RNF) return (SS$_INVLOGIN);
   if (VMSnok (status)) return (status);

   /* automatically disallow if any of these flags are set! */
   if (UaiFlags & DisallowFlags)
   {
      if (Debug) fprintf (stdout, "UaiFlags: %%X%08.08X\n", UaiFlags);
      return (SS$_INVLOGIN);
   }

   PasswordDsc.dsc$w_length = strlen(Password);
   status = sys$hash_password (&PasswordDsc, UaiEncrypt,
                               UaiSalt, &UserNameDsc, &HashedPwd);
   if (Debug) fprintf (stdout, "sys$hash_password() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   if (HashedPwd[0] == UaiPwd[0] && HashedPwd[1] == UaiPwd[1])
      return (SS$_NORMAL);
   else
      return (SS$_INVLOGIN);
}

/*****************************************************************************/
/*
Decodes a string encoded using "Printable Encoding" as described in RFC1421.
*/

char* AuthPrintableDecode
(
char *Encoded,
char *Plain,
int SizeOfPlain
)
{
   static char  ErrorIncorrect [] = "Incorrect character in quantum.";
   static char  ErrorIncomplete [] = "Incomplete quantum.";
   static char  ErrorStringSize [] = "Plain text string too small.";

   static unsigned char  ValueTable [128] =
   {
      255, 255, 255, 255, 255, 255, 255, 255,    /*   7 */
      255, 255, 255, 255, 255, 255, 255, 255,    /*  15 */
      255, 255, 255, 255, 255, 255, 255, 255,    /*  23 */
      255, 255, 255, 255, 255, 255, 255, 255,    /*  31 */
      255, 255, 255, 255, 255, 255, 255, 255,    /*  39 */
      255, 255, 255,  62, 255, 255, 255,  63,    /*  47 */
       52,  53,  54,  55,  56,  57,  58,  59,    /*  55 */
       60,  61, 255, 255, 255, 255, 255, 255,    /*  63 */
      255,   0,   1,   2,   3,   4,   5,   6,    /*  71 */
        7,   8,   9,  10,  11,  12,  13,  14,    /*  79 */
       15,  16,  17,  18,  19,  20,  21,  22,    /*  87 */
       23,  24,  25, 255, 255, 255, 255, 255,    /*  95 */
      255,  26,  27,  28,  29,  30,  31,  32,    /* 103 */
       33,  34,  35,  36,  37,  38,  39,  40,    /* 111 */
       41,  42,  43,  44,  45,  46,  47,  48,    /* 119 */
       49,  50,  51, 255, 255, 255, 255, 255     /* 127 */
   };

   register int  ccnt, pcnt;
   register unsigned long  reg;
   register char  val;
   register char  *eptr, *pptr; 

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthPrintableDecode()\n");
/*
   if (Debug) fprintf (stdout, "Encoded |%s|\n", Encoded);
*/

   eptr = Encoded;
   pptr = Plain;
   pcnt = SizeOfPlain-1;

   while (*eptr)
   {
      reg = 0;
      if (*eptr)
      {
         if (*eptr == '=') return (ErrorIncorrect);
         if ((val = ValueTable [*eptr++ & 0x7f]) == 255)
             return (ErrorIncorrect);
         reg |= val << 18;
      }
      if (*eptr)
      {
         if (*eptr == '=') return (ErrorIncorrect);
         ccnt = 1;
         if ((val = ValueTable [*eptr++ & 0x7f]) == 255)
            return (ErrorIncorrect);
         reg |= val << 12;
      }
      if (*eptr)
      {
         if (*eptr == '=')
            eptr++;
         else
         {
            ccnt++;
            if ((val = ValueTable [*eptr++ & 0x7f]) == 255)
               return (ErrorIncorrect);
            reg |= val << 6;
         }
      }
      if (*eptr)
      {
         if (*eptr == '=')
            eptr++;
         else
         {
            ccnt++;
            if ((val = ValueTable [*eptr++ & 0x7f]) == 255)
               return (ErrorIncorrect);
            reg |= val;
         }
      }
      else
         return (ErrorIncomplete);
      if (Debug) fprintf (stdout, "reg: %08.08X\n", reg);
      if (!pcnt--) return (ErrorStringSize);
      if (ccnt--) *pptr++ = (reg >> 16) & 0xff;
      if (!pcnt--) return (ErrorStringSize);
      if (ccnt--) *pptr++ = (reg >> 8) & 0xff;
      if (!pcnt--) return (ErrorStringSize);
      if (ccnt--) *pptr++ = reg & 0xff;
   }

   *pptr = '\0';
/*
   if (Debug) fprintf (stdout, "Plain |%s|\n", Plain);
*/
   return ("");
}

/*****************************************************************************/
/*
Encodes a plain-text string using "Printable Encoding" as described in RFC1421.

This function is not actually used at all in this daemon!  It is included here
as a tutorial on how the string is encoded at the browser end!
*/

char* AuthPrintableEncode
(
unsigned char *Plain,
unsigned char *Encoded,
int SizeOfEncoded
)
{
   static char  ErrorStringSize [] = "Encoded text string too small.";
   static unsigned char  ValueTable [] =
   "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

   register int  ccnt, ecnt;
   register unsigned long  reg;
   register unsigned char  *eptr, *pptr; 

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AuthPrintableEncode()\nPlain |%s|\n", Plain);

   pptr = Plain;
   eptr = Encoded;
   ecnt = SizeOfEncoded-1;

   while (*pptr)
   {
      ccnt = reg = 0;
      if (*pptr) { reg |= *pptr++ << 16; ccnt++; }
      if (*pptr) { reg |= *pptr++ << 8; ccnt++; }
      if (*pptr) { reg |= *pptr++; ccnt++; }
      if (Debug) fprintf (stdout, "reg: %08.08X\n", reg);
      *eptr++ = ValueTable[(reg>>18)&0x3f];
      if (!ecnt--) return (ErrorStringSize);
      *eptr++ = ValueTable[(reg>>12)&0x3f];
      if (!ecnt--) return (ErrorStringSize);
      if (ccnt == 1)
      {
         if (!ecnt--) return (ErrorStringSize);
         *eptr++ = '=';
         if (!ecnt--) return (ErrorStringSize);
         *eptr++ = '=';
      }
      else
      {
         *eptr++ = ValueTable[(reg>>6)&0x3f];
         if (ccnt == 2)
         {
            if (!ecnt--) return (ErrorStringSize);
            *eptr++ = '=';
         }
         else
         {
            if (!ecnt--) return (ErrorStringSize);
            *eptr++ = ValueTable[reg&0x3f];
         }
      }
   }

   *eptr = '\0';
   if (Debug) fprintf (stdout, "Encoded |%s|\n", Encoded);
   return ("");
}

/****************************************************************************/
