/*****************************************************************************/
/*
                                 Config.c

This module performs two functions.

  1.  It reads the HTTPD configuration file, converting the directives
      into internal representations.

  2.  Provides functions, accessed by other modules, returning the
      original directive information as appropriate.

Configuration can only be performed at server startup.  To reset the 
configuration the server must be restarted.

HFRD HTTPd Configuration is a relevant subset of the CERN configuration 
directives (sort-of, well it started that way!)  Although there are some 
differences (beware!) the CERN documentation can be consulted for a general 
discussion of the purpose and functioning of these directives. 

Directives are:

  o  Accept            <host/domain>
  o  AddType           <suffix> <content-type> <encoding> -|</script> ...
                       [<description>]
  o  AddIcon           <URL> <alternative-text> <content-type-template>
  o  AddDirIcon        <URL> <alternative-text>
  o  AddBlankIcon      <URL> <alternative-text>
  o  AddUnknownIcon    <URL> <alternative-text>
  o  AddParentIcon     <URL> <alternative-text>
  o  AuthLocal         OFF | ON
  o  Busy              <integer>
  o  CommentedInfo     OFF | ON
  o  DirAccess         OFF | ON | SELECTIVE
  o  DirLayout         <string>
  o  DirOwner          OFF | ON
  o  DirReadMe         OFF | TOP | BOTTOM
  o  DirReadMeFile     <file.suffix>
  o  DirWildcard       OFF | ON
  o  DNSLookup         OFF | ON
  o  InputTimeout      <integer>
  o  Logging           OFF | ON
  o  Monitor           OFF | ON
  o  OutputTimeout     <integer>
  o  Port              <integer>
  o  Recommend         OFF | ON
  o  Reject            <host/domain>
  o  Search            <absolute-path-to-search-script>
  o  Shtml             OFF | ON
  o  ShtmlAccesses     OFF | ON
  o  ShtmlExec         OFF | ON
  o  Welcome           <file.suffix>


VERSION HISTORY
---------------
01-DEC-95  MGD  HTTPd version 3
27-SEP-95  MGD  added auto-scripting funtionality
07-AUG-95  MGD  added support for VMS-style directory structure and file sizes
16-JUN-95  MGD  file contents description in 'AddType' (also see dir.c module)
20-DEC-94  MGD  developed for multi-threaded HTTP daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application related header */
#include "httpd.h"

/*****************/
/* module macros */
/*****************/

#define DefaultContentType "x-internal/unknown"
#define DefaultBinaryEncoding true

/******************/
/* global storage */
/******************/

/* report warning messages by default */
boolean  ConfigReportProblems = true;
struct ConfigStruct  Config;

/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern char  SoftwareID[];
extern char  Utility[];

/****************************/
/* functions in this module */
/****************************/

int Configure ();
boolean ConfigAcceptClientHostName (char*);
ConfigAddType (char*, int);
ConfigContentType (struct RequestStruct*, char*);
char* ConfigHomePage (int);
ConfigAddIcon (char*, int);
char* ConfigIconFor (struct RequestStruct*, char*);
char* ConfigReadMeFile (int);
ConfigSetAccept (char*, int);
ConfigSetDirReadMe (char*, int);
ConfigSetDirReadMeFile (char*, int);
ConfigSetDirWildcard (char*, int);
ConfigSetLogging (char*, int);
ConfigSetMonitor (char*, int);
ConfigSetPort (char*, int);
ConfigSetReject (char*, int);
ConfigSetDnsLookup (char*, int);
ConfigSetInputTimeout (char*, int);
ConfigSetOutputTimeout (char*, int);
ConfigSetShtml (char*, int);
ConfigSetShtmlAccesses (char*, int);
ConfigSetShtmlExec (char*, int);
ConfigSetWelcome (char*, int);

/**********************/
/* external functions */
/**********************/

ErrorExitVmsStatus (int, char*, char*, int);

/*****************************************************************************/
/*
Open the configuration file defined by the logical name HTTPD$CONFIG.  Read 
each line, passing to the appropriate interpretation function according to the 
directive on the line.  Close the file.  Is either called before AST-driven 
processing or called at AST delivery level which makes processing atomic.
*/

int Configure ()

{
   static char  Line [256];
   static struct FAB  ConfigFileFab;
   static struct RAB  ConfigFileRab;

   register char  *cptr, *lptr;

   boolean  DebugBuffer;
   int  status,
        LineNumber;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "Configure()\n");

   /* not interested anymore in seeing debug information for config load! */
   DebugBuffer = Debug;
   Debug = false;

   Config.ProblemCount = 0;

   /************************/
   /* open the config file */
   /************************/

   if (Debug) fprintf (stdout, "ConfigFileName |%s|\n", ConfigFileName);

   ConfigFileFab = cc$rms_fab;
   ConfigFileFab.fab$b_fac = FAB$M_GET;
   ConfigFileFab.fab$l_fna = ConfigFileName;
   ConfigFileFab.fab$b_fns = strlen(ConfigFileName);
   ConfigFileFab.fab$b_shr = FAB$M_SHRGET;

   if (VMSnok (status = sys$open (&ConfigFileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      ErrorExitVmsStatus (status, ConfigFileName, __FILE__, __LINE__);
   }

   /* record access block */
   ConfigFileRab = cc$rms_rab;
   ConfigFileRab.rab$l_fab = &ConfigFileFab;
   /* 2 buffers */
   ConfigFileRab.rab$b_mbf = 2;
   /* read ahead performance option */
   ConfigFileRab.rab$l_rop = RAB$M_RAH;
   ConfigFileRab.rab$l_ubf = Line;
   ConfigFileRab.rab$w_usz = sizeof(Line)-1;

   if (VMSnok (status = sys$connect (&ConfigFileRab, 0, 0)))
   {
      sys$close (&ConfigFileFab, 0, 0);
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      ErrorExitVmsStatus (status, ConfigFileName, __FILE__, __LINE__);
   }

   LineNumber = 0;

   /************************/
   /* read the config file */
   /************************/

   Config.DefaultPost = "/ht_root/script/post";
   Config.HtmlContentType = "text/html";
   Config.IsMapContentType = "text/x-ismap";
   Config.MenuContentType = "text/x-menu";
   Config.PlainTextContentType = "text/plain";
   Config.ShtmlContentType = "text/x-shtml";
   Config.AcceptHostsPtr = Config.RejectHostsPtr = 
      Config.ContentTypeListHeadPtr = Config.IconListHeadPtr = NULL;
   Config.ContentTypeStructOverhead = sizeof(struct ContentTypeStruct);
   Config.IconStructOverhead = sizeof(struct IconStruct);

   while (VMSok (status = sys$get (&ConfigFileRab, 0, 0)))
   {
      LineNumber++;
      Line[ConfigFileRab.rab$w_rsz] = '\0';
      if (Debug) fprintf (stdout, "Line %d |%s|\n", LineNumber, Line);

      lptr = Line;
      while (*lptr && isspace(*lptr)) lptr++;
      if (!*lptr || *lptr == '#') continue;

      /* note the start, skip across the keyword and any intervening space */
      cptr = lptr;
      while (*lptr && !isspace(*lptr)) lptr++;
      while (*lptr && isspace(*lptr)) lptr++;

      if (strsame (cptr, "Accept", 6))
         ConfigSetAccept (lptr, LineNumber);
      else
      if (strsame (cptr, "AddType", 7))
         ConfigAddType (lptr, LineNumber);
      else
      if (strsame (cptr, "AddIcon", 7))
         ConfigAddIcon (lptr, LineNumber);
      else
      if (strsame (cptr, "AddBlankIcon", 12))
      {
         strcat (Line, " x-internal/blank");
         ConfigAddIcon (lptr, LineNumber);
      }
      else
      if (strsame (cptr, "AddUnknownIcon", 14))
      {
         strcat (Line, " x-internal/unknown");
         ConfigAddIcon (lptr, LineNumber);
      }
      else
      if (strsame (cptr, "AddDirIcon", 10))
      {
         strcat (Line, " x-internal/dir");
         ConfigAddIcon (lptr, LineNumber);
      }
      else
      if (strsame (cptr, "AddParentIcon", 13))
      {
         strcat (Line, " x-internal/parent");
         ConfigAddIcon (lptr, LineNumber);
      }
      else
      if (strsame (cptr, "AuthLocal", 9))
         ConfigSetAuthLocal (lptr, LineNumber);
      else
      if (strsame (cptr, "Busy", 4))
         ConfigSetBusy (lptr, LineNumber);
      else
      if (strsame (cptr, "CommentedInfo", 13))
         ConfigSetCommentedInfo (lptr, LineNumber);
      else
      if (strsame (cptr, "DirAccess", 9))
         ConfigSetDirAccess (lptr, LineNumber);
      else
      if (strsame (cptr, "DirLayout", 9))
         ConfigSetDirLayout (lptr, LineNumber);
      else
      if (strsame (cptr, "DirOwner", 8))
         ConfigSetDirOwner (lptr, LineNumber);
      else
      if (strsame (cptr, "DirReadMeFile", 13))
         ConfigSetDirReadMeFile (lptr, LineNumber);
      else
      /* "ReadMe" MUST come after the other ReadMes for obvious reasons */
      if (strsame (cptr, "DirReadMe", 9))
         ConfigSetDirReadMe (lptr, LineNumber);
      else
      if (strsame (cptr, "DirWildcard", 11))
         ConfigSetDirWildcard (lptr, LineNumber);
      else
      if (strsame (cptr, "InputTimeout", 12))
         ConfigSetInputTimeout (lptr, LineNumber);
      else
      if (strsame (cptr, "Logging", 7))
         ConfigSetLogging (lptr, LineNumber);
      else
      if (strsame (cptr, "Monitor", 7))
         ConfigSetMonitor (lptr, LineNumber);
      else
      if (strsame (cptr, "OutputTimeout", 13))
         ConfigSetOutputTimeout (lptr, LineNumber);
      else
      if (strsame (cptr, "Port", 4))
         ConfigSetPort (lptr, LineNumber);
      else
      if (strsame (cptr, "Reject", 6))
         ConfigSetReject (lptr, LineNumber);
      else
      if (strsame (cptr, "DNSLookup", 9))
         ConfigSetDnsLookup (lptr, LineNumber);
      else
      if (strsame (cptr, "Recommend", 9))
         ConfigSetRecommend (lptr, LineNumber);
      else
      if (strsame (cptr, "Search", 6))
         ConfigSetSearch (lptr, LineNumber);
      else
      if (strsame (cptr, "ShtmlAccesses", 13))
         ConfigSetShtmlAccesses (lptr, LineNumber);
      else
      if (strsame (cptr, "ShtmlExec", 9))
         ConfigSetShtmlExec (lptr, LineNumber);
      else
      /* "Shtml" MUST come after the other SHTMLs for obvious reasons */
      if (strsame (cptr, "Shtml", 5))
         ConfigSetShtml (lptr, LineNumber);
      else
      if (strsame (cptr, "Welcome", 7))
         ConfigSetWelcome (lptr, LineNumber);
      else
      {
         Config.ProblemCount++;
         if (ConfigReportProblems)
            fprintf (stdout, "%%%s-W-CONFIG, unknown directive at line %d\n",
                     Utility, LineNumber);
      }
   }

   if (status == RMS$_EOF) status = SS$_NORMAL;

   /*************************/
   /* close the config file */
   /*************************/

   sys$close (&ConfigFileFab, 0, 0); 

   Debug = DebugBuffer;

   return (status);
}

/*****************************************************************************/
/*
Add a mime content type corresponding to the file suffix (extension, type).

Format: "AddType <suffix> <content-type> <encoding> <-|/script> [<description>]"

Examples: "AddType .html text/html 8bit - HTML"
          "AddType .txt text/plain 7bit - plain text"
          "AddType .hlb text/x-script binary /conan VMS help library"
          "AddType .gif image/gif binary - GIF image"

The directive is stored as an singly-linked list of structures of dynamically 
allocated memory containing a reformatted version of the directive string.  
Function ConfigContentType() searches this list to get pointers to the content 
type, encoding method, any auto-script, and description, into the client 
request data structure.
*/ 

ConfigAddType
(
register char *lptr,
int LineNumber
)
{
   register char  *cptr, *sptr;
   register struct ContentTypeStruct  *clptr;

   int  status,
        Count = 0;
   char  *DescriptionPtr = "",
         *EncodingPtr = "",
         *ContentTypePtr = "",
         *AutoScriptNamePtr = "",
         *SuffixPtr = "";

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigAddType()\n");

   /* suffix (file type/extension) */
   SuffixPtr = lptr;
   while (*lptr && !isspace(*lptr))
   {
      *lptr = toupper(*lptr);
      lptr++;
   }
   Count += lptr - SuffixPtr;
   if (*lptr) *lptr++ = '\0';

   /* "mime" content type (representation) */
   while (*lptr && isspace(*lptr)) lptr++;
   ContentTypePtr = lptr;
   while (*lptr && !isspace(*lptr)) lptr++;
   Count += lptr - ContentTypePtr;
   if (*lptr) *lptr++ = '\0';

   /* encoding (binary or text) */
   while (*lptr && isspace(*lptr)) lptr++;
   EncodingPtr = lptr;
   while (*lptr && !isspace(*lptr)) lptr++;
   if (*lptr) *lptr++ = '\0';

   /* (optional) script activation */
   while (*lptr && isspace(*lptr)) lptr++;
   AutoScriptNamePtr = lptr;
   while (*lptr && !isspace(*lptr)) lptr++;
   Count += lptr - AutoScriptNamePtr;
   if (*lptr) *lptr++ = '\0';

   /* (optional) description, to end of line */
   while (*lptr && isspace(*lptr)) lptr++;
   DescriptionPtr = lptr;
   while (*lptr) lptr++;
   Count += lptr - DescriptionPtr;

   if (!(*SuffixPtr && *ContentTypePtr && *EncodingPtr))
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"AddType\" problem at line %d\n",
                  Utility, LineNumber);
      return;
   }

   /* plus structure overhead, plus 4 x null terminators */
   Count += Config.ContentTypeStructOverhead + 4;
   if ((clptr = (struct ContentTypeStruct*)malloc(Count)) == NULL)
      ErrorExitVmsStatus (vaxc$errno, "memory allocation", __FILE__, __LINE__);

   if (Config.ContentTypeListHeadPtr == NULL)
      Config.ContentTypeListHeadPtr = clptr; 
   else
      Config.ContentTypeListTailPtr->NextPtr = clptr; 
   Config.ContentTypeListTailPtr = clptr;
   clptr->NextPtr = NULL;

   sptr = (char*)clptr + Config.ContentTypeStructOverhead;

   clptr->SuffixPtr = sptr;
   cptr = SuffixPtr;
   /* ensure suffix is in upper-case so we don't have to do it when matching */
   while (*cptr) *sptr++ = toupper(*cptr++);
   *sptr++ = '\0';

   clptr->ContentTypePtr = sptr;
   cptr = ContentTypePtr;
   while (*cptr) *sptr++ = *cptr++;
   *sptr++ = '\0';

   clptr->AutoScriptNamePtr = sptr;
   cptr = AutoScriptNamePtr;
   while (*cptr) *sptr++ = *cptr++;
   *sptr++ = '\0';

   clptr->ContentDescriptionPtr = sptr;
   cptr = DescriptionPtr;
   while (*cptr) *sptr++ = *cptr++;
   *sptr = '\0';

   if (toupper(*EncodingPtr) == 'B')
      clptr->BinaryEncoding = true;
   else
      clptr->BinaryEncoding = false;
}

/*****************************************************************************/
/*
This function searches (linear, so put the most frequently hit suffixes at 
the start) the list of content type directives, pointing the appropriate 
content type and encoding method in the client request data structure.  
Attempts to improve efficiency by buffering a pointer the last content-type 
matched.  This yields benefits only where repeated calls are made to this 
function by the one thread, e.g. during directory listings, where there are 
often series of the same file type.
*/ 

ConfigContentType
(
struct RequestStruct *RequestPtr,
char *Suffix
)
{
   register char  *cptr, *sptr;
   register struct ContentTypeStruct  *clptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigContentType() |%s|\n", Suffix);

   if ((clptr = (struct ContentTypeStruct*)RequestPtr->ConfigPrevContentTypePtr)
       == NULL)
      clptr = Config.ContentTypeListHeadPtr;
    
   while (clptr != NULL)
   {
      /* 'clptr->SuffixPtr' has already been converted to upper case! */
      cptr = clptr->SuffixPtr;
      /** if (Debug) fprintf (stdout, "|%s|\n", cptr); **/
      sptr = Suffix;
      while (*cptr && *sptr && *sptr != ';' && *cptr == toupper(*sptr))
         { cptr++; sptr++; }

      if (*cptr || (*sptr && *sptr != ';'))
      {
         if (RequestPtr->ConfigPrevContentTypePtr == NULL)
         {
            clptr = clptr->NextPtr;
            continue;
         }
         clptr = Config.ContentTypeListHeadPtr;
         RequestPtr->ConfigPrevContentTypePtr = NULL;
         continue;
      }

      /**********/
      /* match! */
      /**********/

      RequestPtr->ContentTypePtr = clptr->ContentTypePtr;
      RequestPtr->AutoScriptNamePtr = clptr->AutoScriptNamePtr;
      RequestPtr->ContentDescriptionPtr = clptr->ContentDescriptionPtr;
      RequestPtr->BinaryEncoding = clptr->BinaryEncoding;

      /* note the location of this icon information */
      RequestPtr->ConfigPrevContentTypePtr = (unsigned char*)clptr;

      if (!Debug) return;

      fprintf (stdout, "|%s|%s|%s|%d|\n",
               RequestPtr->ContentTypePtr, RequestPtr->AutoScriptNamePtr,
               RequestPtr->ContentDescriptionPtr, RequestPtr->BinaryEncoding);
      return;
   }

   /*************/
   /* no match! */
   /*************/

   RequestPtr->ContentTypePtr = DefaultContentType;
   RequestPtr->AutoScriptNamePtr = RequestPtr->ContentDescriptionPtr = "";
   RequestPtr->BinaryEncoding = DefaultBinaryEncoding;
}

/*****************************************************************************/
/*
Add a URL for an icon corresponding to the mime content type.

Format:   "AddIcon <URL> <alternative-text> <content-type-template>"

Examples: "AddIcon /icon/httpd/text.xbm [TXT] text/plain"
          "AddIcon /icon/httpd/doc.xbm [HTM] text/html"
          "AddIcon /icon/httpd/image.xbm [IMG] image/jpeg"

The template can contain a wildcard in place of a string in either side of the 
template slash.  In this case the wildcard matches any string on that side.

The directive is stored as an list of structures of dynamically allocated 
memory containing a reformatted version of the directive string, including a 
complete HTML anchor for the URL.  Function ConfigIconFor() searches this 
list to return a pointer to the anchor.
*/ 

ConfigAddIcon
(
register char *lptr,
int LineNumber
)
{
   register struct IconStruct  *ilptr;

   int  Count = 0;
   char  *AltTextPtr,
         *IconPathPtr,
         *TemplatePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigAddIcon()\n");

   IconPathPtr = lptr;
   while (*lptr && !isspace(*lptr)) lptr++;
   Count = lptr - IconPathPtr;
   if (*lptr) *lptr++ = '\0';

   /* alternative text */
   while (*lptr && isspace(*lptr)) lptr++;
   AltTextPtr = lptr;
   while (*lptr && !isspace(*lptr))
   {
      /* underscores represent spaces! (especially for the "blank" icon) */
      if (*lptr == '_') *lptr = ' ';
      lptr++;
   }
   Count += lptr - AltTextPtr;
   if (*lptr) *lptr++ = '\0';

   /* template */
   while (*lptr && isspace(*lptr)) lptr++;
   TemplatePtr = lptr;
   while (*lptr && !isspace(*lptr))
   {
      /* force to upper case to save a toupper() in ConfigIconFor() */
      *lptr = toupper(*lptr);
      lptr++;
   }
   Count += lptr - TemplatePtr;
   if (*lptr) *lptr++ = '\0';

   if (!(*IconPathPtr && *AltTextPtr && *TemplatePtr))
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"AddIcon\" problem at line %d\n",
                  Utility, LineNumber);
      return;
   }

   /* plus structure overhead, plus the HTML IMG element used below */
   Count += Config.IconStructOverhead + 31;
   if ((ilptr = (struct IconStruct*)malloc(Count)) == NULL)
      ErrorExitVmsStatus (vaxc$errno, "memory allocation", __FILE__, __LINE__);

   if (Config.IconListHeadPtr == NULL)
      Config.IconListHeadPtr = ilptr; 
   else
      Config.IconListTailPtr->NextPtr = ilptr; 
   Config.IconListTailPtr = ilptr;
   ilptr->NextPtr = NULL;

   sprintf ((char*)ilptr + Config.IconStructOverhead,
            "%s <IMG ALIGN=top SRC=\"%s\" ALT=\"%s\">",
            TemplatePtr, IconPathPtr, AltTextPtr);
   if (Debug)
      fprintf (stdout, "|%s|\n", (char*)ilptr + Config.IconStructOverhead);
}

/*****************************************************************************/
/*
This function searches the list of icons for an one corresponding to the 
supplied mime content type.  The directive content type and associated HTML 
anchor (containing the original configuration-supplied URL) is stored as an 
list of structures dynamically allocated memory containing a string.  Attempts 
to improve efficiency by buffering a pointer the last content-type matched.  
This yields benefits only where repeated calls are made to this function by 
the one thread, e.g. during directory listings, where there are often series 
of the same file type.
*/

char* ConfigIconFor
(
struct RequestStruct *RequestPtr,
char *ContentType
)
{
   register char  *cptr, *sptr;
   register struct IconStruct  *ilptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigIconFor() |%s|\n", ContentType);

   if ((ilptr = (struct IconStruct*)RequestPtr->ConfigPrevIconPtr) == NULL)
      ilptr = Config.IconListHeadPtr;

   while (ilptr != NULL)
   {
      cptr = (char*)ilptr + Config.IconStructOverhead;
      /** if (Debug) fprintf (stdout, "|%s|\n", cptr); **/
      sptr = ContentType;
      while (*cptr && *cptr != '*' && *sptr && *cptr == toupper(*sptr))
         { cptr++; sptr++; }
      if (*cptr == ' ' && !*sptr)
      {
         /* hit!  note the location of this icon information */
         RequestPtr->ConfigPrevIconPtr = (unsigned char*)ilptr;
         /* return a pointer to the HTML IMG element for the icon */
         if (*cptr) cptr++;
         return (cptr);
      }
      if (*cptr == '*')
      {
         /* template contained a wildcard on the left side of the slash */
         while (*cptr && *cptr != '/' && !isspace(*cptr)) cptr++;
         if (*cptr) cptr++;
         /* find the right side of the content type */
         while (*sptr && *sptr != '/') sptr++;
         if (*sptr) sptr++;
         /* compare the right side of the slash */
         while (*cptr && *cptr != '*' && *sptr && *cptr == toupper(*sptr))
            { cptr++; sptr++; }
         if (*cptr == '*')
         {
            while (*cptr == '*') cptr++;
            while (*sptr) sptr++;
         }
         if (*cptr == ' ' && !*sptr)
         {
            /* hit!  note the location of this icon information */
            RequestPtr->ConfigPrevIconPtr = (unsigned char*)ilptr;
            /* return a pointer to the HTML IMG element for the icon */
            if (*cptr) cptr++;
            return (cptr);
         }
      }

      if (RequestPtr->ConfigPrevIconPtr == NULL)
      {
         ilptr = ilptr->NextPtr;
         continue;
      }
      ilptr = Config.IconListHeadPtr;
      RequestPtr->ConfigPrevIconPtr = NULL;
   }
   /* nothing found, essentially an error! */
   return ("");
}

/*****************************************************************************/
/*
*/

ConfigSetAccept
(
register char *lptr,
int LineNumber
)
{
   register char  *sptr;

   int  Comma;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetAccept()\n");

   if (!*lptr)
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"Accept\" problem at line %d\n",
                  Utility, LineNumber);
      return;
   }

   for (sptr = lptr; *sptr; *sptr++);
   if (Config.AcceptHostsPtr == NULL)
      Comma = 0;
   else
      Comma = 1;
   if ((Config.AcceptHostsPtr = realloc (Config.AcceptHostsPtr,
                                Config.AcceptHostsLength+sptr-lptr+1+Comma))
       == NULL)
      ErrorExitVmsStatus (vaxc$errno, "memory reallocation",
                          __FILE__, __LINE__);
   if (Comma) Config.AcceptHostsPtr[Config.AcceptHostsLength++] = ',';
   memcpy (Config.AcceptHostsPtr+Config.AcceptHostsLength, lptr, sptr-lptr+1);
   Config.AcceptHostsLength += sptr-lptr;
   if (Debug)
      fprintf (stdout, "Config.AcceptHostsPtr |%s|\n", Config.AcceptHostsPtr);
}

/*****************************************************************************/
/*
*/

ConfigSetReject
(
register char *lptr,
int LineNumber
)
{
   register char  *sptr;

   int  Comma;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetReject()\n");

   if (!*lptr)
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"Reject\" problem at line %d\n",
                  Utility, LineNumber);
      return;
   }

   for (sptr = lptr; *sptr; *sptr++);
   if (Config.RejectHostsPtr == NULL)
      Comma = 0;
   else
      Comma = 1;
   if ((Config.RejectHostsPtr = realloc (Config.RejectHostsPtr,
                                Config.RejectHostsLength+sptr-lptr+1+Comma))
       == NULL)
      ErrorExitVmsStatus (vaxc$errno, "memory reallocation",
                          __FILE__, __LINE__);
   if (Comma) Config.RejectHostsPtr[Config.RejectHostsLength++] = ',';
   memcpy (Config.RejectHostsPtr+Config.RejectHostsLength, lptr, sptr-lptr+1);
   Config.RejectHostsLength += sptr-lptr;
   if (Debug)
      fprintf (stdout, "Config.RejectHostsPtr |%s|\n", Config.RejectHostsPtr);
}

/*****************************************************************************/
/*
Accept or reject the client connection.  Lists of hosts must be a comma-
separated string.  A wildcard '*' in a domain specification matches any string 
that occurs between the dots.
Example: "host1.network1.domain1,*.network2.domain1,*.*.domain2". 
*/ 

boolean ConfigAcceptClientHostName (char *ClientHostName)

{
   register char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "ConfigAcceptConnection() |%s|\n", ClientHostName);

   if (Config.AcceptHostsPtr == NULL && Config.RejectHostsPtr == NULL)
      return (true);

   if (!ClientHostName[0] || ClientHostName[0] == '?') return (false);

   /******************/
   /* hosts accepted */
   /******************/

   if (Config.AcceptHostsPtr != NULL)
   {
      sptr = Config.AcceptHostsPtr;
      while (*sptr)
      {
         /** if (Debug) fprintf (stdout, "sptr |%s|\n", sptr); **/
         cptr = ClientHostName;
         while (*cptr)
         {
            if (*sptr == ',') break;
            if (*sptr == '*')
            {
               while (*sptr && *sptr != '.' && *sptr != ',') sptr++;
               while (*cptr && *cptr != '.') cptr++;
            }
            if (tolower(*cptr) != tolower(*sptr)) break;
            if (*cptr) cptr++;
            if (*sptr) sptr++;
         }
         if (!*cptr && (!*sptr || *sptr == ',')) break;
         while (*sptr && *sptr != ',') sptr++;
         if (*sptr) sptr++;
      }
      /* if it was NOT found then reject the connection request */
      if (!(!*cptr && (!*sptr || *sptr == ',')))
         return (false);
   }

   /******************/
   /* hosts rejected */
   /******************/

   if (Config.RejectHostsPtr != NULL)
   {
      sptr = Config.RejectHostsPtr;
      while (*sptr)
      {
         /** if (Debug) fprintf (stdout, "sptr |%s|\n", sptr); **/
         cptr = ClientHostName;
         while (*cptr)
         {
            if (*sptr == ',') break;
            if (*sptr == '*')
            {
               while (*sptr && *sptr != '.' && *sptr != ',') sptr++;
               while (*cptr && *cptr != '.') cptr++;
            }
            if (tolower(*cptr) != tolower(*sptr)) break;
            if (*cptr) cptr++;
            if (*sptr) sptr++;
         }
         if (!*cptr && (!*sptr || *sptr == ',')) break;
         while (*sptr && *sptr != ',') sptr++;
         if (*sptr) sptr++;
      }
      /* if it WAS found then reject the connection request */
      if (!*cptr && (!*sptr || *sptr == ','))
         return (false);
   }

   /* accept the connection */
   return (true);
}

/*****************************************************************************/
/*
Set local (SYSUAF) authentication boolean.  Allowed values OFF or ON.
*/

ConfigSetAuthLocal
(
register char *lptr,
int LineNumber
)
{
   if (Debug) fprintf (stdout, "ConfigSetAuthLocal()\n");

   if (strsame (lptr, "OFF", 3))
      Config.AuthLocalEnabled = false;
   else
   if (strsame (lptr, "ON", 2))
      Config.AuthLocalEnabled = true;
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"AuthLocal\" problem at line %d\n",
                  Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
*/

ConfigSetBusy
(
register char *lptr,
int LineNumber
)
{
   if (Debug) fprintf (stdout, "ConfigSetBusy()\n");

   if (isdigit (*lptr))
      Config.Busy = atoi (lptr);
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"Busy\" problem at line %d\n",
                  Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
Set error recommend boolean.  Allowed values OFF or ON.
*/

ConfigSetCommentedInfo
(
register char *lptr,
int LineNumber
)
{
   if (Debug) fprintf (stdout, "ConfigSetCommentedInfo()\n");

   if (strsame (lptr, "OFF", 3))
      Config.IncludeCommentedInfo = false;
   else
   if (strsame (lptr, "ON", 2))
      Config.IncludeCommentedInfo = true;
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout,
            "%%%s-W-CONFIG, \"CommentedInfo\" problem at line %d\n",
            Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
Set directory access booleans.  Allowed values OFF, ON, SELECTIVE.
*/

ConfigSetDirAccess
(
register char *lptr,
int LineNumber
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetDirAccess()\n");

   if (strsame (lptr, "OFF", 3))
      Config.DirAccess = Config.DirAccessSelective = false;
   else
   if (strsame (lptr, "ON", 3))
   {
      Config.DirAccess = true;
      Config.DirAccessSelective = false;
   }
   else
   if (strsame (lptr, "SEL", 3))
      Config.DirAccess = Config.DirAccessSelective = true;
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"DirAccess\" problem at line %d\n",
                  Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
*/

ConfigSetDirLayout
(
register char *lptr,
int LineNumber
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetDirLayout()\n");

   if (!*lptr)
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"DirLayout\" problem at line %d\n",
               Utility, LineNumber);
      return;
   }

   strncpy (Config.DefaultDirLayout, lptr, sizeof(Config.DefaultDirLayout)-1);
   Config.DefaultDirLayout[sizeof(Config.DefaultDirLayout)-1] = '\0';
}

/*****************************************************************************/
/*
Set directory owner enabled booleans.  Allowed values OFF, ON.
*/

ConfigSetDirOwner
(
register char *lptr,
int LineNumber
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetDirOwner()\n");

   if (strsame (lptr, "OFF", 3))
      Config.DirOwnerEnabled = false;
   else
   if (strsame (lptr, "ON", 3))
      Config.DirOwnerEnabled = true;
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"DirOwner\" problem at line %d\n",
                  Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
Set directory README file booleans.  Allowed values OFF, TOP, BOTTOM.
*/

ConfigSetDirReadMe
(
register char *lptr,
int LineNumber
)
{
   if (Debug) fprintf (stdout, "ConfigSetDirReadMe()\n");

   if (strsame (lptr, "BOT", 3))
   {
      Config.DirReadMeTop = false;
      Config.DirReadMeBottom = true;
   }
   else
   if (strsame (lptr, "OFF", 3))
      Config.DirReadMeBottom = Config.DirReadMeTop = false;
   else
   if (strsame (lptr, "TOP", 3))
   {
      Config.DirReadMeTop = true;
      Config.DirReadMeBottom = false;
   }
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"DirReadMe\" problem at line %d\n",
                  Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
Add a "read-me" file name.

Format:   "ReadMeFile file.suffix"

Examples: "ReadMeFile README.HTML"
          "ReadMeFile README.TXT"
          "ReadMeFile README."

The directive is stored as an array of pointers to dynamically allocated 
memory containing the "read-me" file name.  Function ConfigReadMeFile() gets 
these by index number.
*/

ConfigSetDirReadMeFile
(
register char *lptr,
int LineNumber
)
{
   int  status,
        Count;
   char  *NamePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetDirReadMeFile()\n");

   if (Config.ReadMeFileCount >= MaxConfigReadMeFiles)
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, maximum read-me files at line %d\n",
                  Utility, __LINE__);
      return;
   }

   NamePtr = lptr;
   while (*lptr && !isspace(*lptr))
   {
      *lptr = toupper(*lptr);
      lptr++;
   }
   Count = lptr - NamePtr;
   *lptr = '\0';

   if (!*NamePtr)
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout,
            "%%%s-W-CONFIG, \"DirReadMeFile\" problem at line %d\n",
            Utility, LineNumber);
      return;
   }

   if (Debug) fprintf (stdout, "malloc()ing %d bytes\n", Count+1);
   if ((Config.ReadMeFileArray[Config.ReadMeFileCount] = malloc (Count+1))
       == NULL)
      ErrorExitVmsStatus (vaxc$errno, "memory allocation", __FILE__, __LINE__);

   strcpy (Config.ReadMeFileArray[Config.ReadMeFileCount], NamePtr);
   if (Debug)
      fprintf (stdout, "|%s|\n",
               Config.ReadMeFileArray[Config.ReadMeFileCount]);
   Config.ReadMeFileCount++;
}

/*****************************************************************************/
/*
Return a pointer to the "read-me" file name stored against the index number of 
the array.  If none exists return a null string.  The calling function would 
use this by stepping through the index numbers 0 to n, until the null string 
was returned indicating the possible "read-me" file names were exhausted.
*/

char* ConfigReadMeFile (int Number)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigReadMeFile |%d|\n", Number);

   if (Number >= 0 && Number < Config.ReadMeFileCount)
      return (Config.ReadMeFileArray[Number]);
   else
      return ("");
}

/*****************************************************************************/
/*
Set directory owner enabled booleans.  Allowed values OFF, ON.
*/

ConfigSetDirWildcard
(
register char *lptr,
int LineNumber
)
{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetDirWildcard()\n");

   if (strsame (lptr, "OFF", 3))
      Config.DirWildcardEnabled = false;
   else
   if (strsame (lptr, "ON", 3))
      Config.DirWildcardEnabled = true;
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"DirWildcard\" problem at line %d\n",
                  Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
Set host name resolution boolean.  Allowed values OFF or ON.
*/

ConfigSetDnsLookup
(
register char *lptr,
int LineNumber
)
{
   if (Debug) fprintf (stdout, "ConfigSetDnsLookup()\n");

   if (strsame (lptr, "OFF", 3))
      Config.DnsLookup = false;
   else
   if (strsame (lptr, "ON", 2))
      Config.DnsLookup = true;
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"DnsLookup\" problem at line %d\n",
                  Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
Set log boolean.  Allowed values OFF or ON.
*/

ConfigSetLogging
(
register char *lptr,
int LineNumber
)
{
   if (Debug) fprintf (stdout, "ConfigSetLogging()\n");

   if (strsame (lptr, "OFF", 3))
      Config.LoggingEnabled = false;
   else
   if (strsame (lptr, "ON", 2))
      Config.LoggingEnabled = true;
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"Logging\" problem at line %d\n",
                  Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
Set log boolean.  Allowed values OFF or ON.
*/

ConfigSetMonitor
(
register char *lptr,
int LineNumber
)
{
   if (Debug) fprintf (stdout, "ConfigSetMonitor()\n");

   if (strsame (lptr, "OFF", 3))
      Config.MonitorEnabled = false;
   else
   if (strsame (lptr, "ON", 2))
      Config.MonitorEnabled = true;
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"Monitor\" problem at line %d\n",
                  Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
*/

ConfigSetPort
(
register char *lptr,
int LineNumber
)
{
   if (Debug) fprintf (stdout, "ConfigSetPort()\n");

   if (isdigit (*lptr))
      Config.ServerPort = atoi (lptr);
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"Port\" problem at line %d\n",
                  Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
Set error recommend boolean.  Allowed values OFF or ON.
*/

ConfigSetRecommend
(
register char *lptr,
int LineNumber
)
{
   if (Debug) fprintf (stdout, "ConfigSetRecommend()\n");

   if (strsame (lptr, "OFF", 3))
      Config.ErrorRecommend = false;
   else
   if (strsame (lptr, "ON", 2))
      Config.ErrorRecommend = true;
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"Recommend\" problem at line %d\n",
                  Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
*/

ConfigSetInputTimeout
(
register char *lptr,
int LineNumber
)
{
   if (Debug) fprintf (stdout, "ConfigSetInputTimeout()\n");

   if (isdigit (*lptr))
      Config.InputTimeoutMinutes = atoi (lptr);
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout,
            "%%%s-W-CONFIG, \"InputTimeout\" problem at line %d\n",
            Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
*/

ConfigSetOutputTimeout
(
register char *lptr,
int LineNumber
)
{
   if (Debug) fprintf (stdout, "ConfigSetOutputTimeout()\n");

   if (isdigit (*lptr))
      Config.OutputTimeoutMinutes = atoi (lptr);
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout,
            "%%%s-W-CONFIG, \"OutputTimeout\" problem at line %d\n",
            Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
*/

ConfigSetSearch
(
register char *lptr,
int LineNumber
)
{
   register char  *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetSearch()\n");

   if (*lptr)
   {
      sptr = Config.KeywordSearch;
      while (*lptr && !isspace(*lptr)) *sptr++ = *lptr++;
      *sptr = '\0';
   }
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"Search\" problem at line %d\n",
                  Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
Set shtml enabled boolean.  Allowed values OFF or ON.
*/

ConfigSetShtml
(
register char *lptr,
int LineNumber
)
{
   if (Debug) fprintf (stdout, "ConfigSetShtml()\n");

   if (strsame (lptr, "OFF", 3))
      Config.ShtmlEnabled = false;
   else
   if (strsame (lptr, "ON", 2))
      Config.ShtmlEnabled = true;
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"Shtml\" problem at line %d\n",
                  Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
Set shtml accesses boolean.  Allowed values OFF or ON.
*/

ConfigSetShtmlAccesses
(
register char *lptr,
int LineNumber
)
{
   if (Debug) fprintf (stdout, "ConfigSetShtmlAccesses()\n");

   if (strsame (lptr, "OFF", 3))
      Config.ShtmlAccessesEnabled = false;
   else
   if (strsame (lptr, "ON", 2))
      Config.ShtmlAccessesEnabled = true;
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout,
            "%%%s-W-CONFIG, \"ShtmlAccesses\" problem at line %d\n",
            Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
Set shtml exec enabled boolean.  Allowed values OFF or ON.
*/

ConfigSetShtmlExec
(
register char *lptr,
int LineNumber
)
{
   if (Debug) fprintf (stdout, "ConfigSetShtmlExec()\n");

   if (strsame (lptr, "OFF", 3))
      Config.ShtmlExecEnabled = false;
   else
   if (strsame (lptr, "ON", 2))
      Config.ShtmlExecEnabled = true;
   else
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"ShtmlExec\" problem at line %d\n",
                  Utility, LineNumber);
   }
}

/*****************************************************************************/
/*
Add a welcome (home) page file name.

Format:   "Welcome file.suffix"

Examples: "Welcome HOME.HTML"
          "Welcome HOME.HTM"
          "Welcome HOME.MENU"

The directive is stored as an array of pointers to dynamically allocated 
memory containing the home page file name.  Function ConfigHomePage() gets 
these by index number.
*/ 

ConfigSetWelcome
(
register char *lptr,
int LineNumber
)
{
   int  status,
        Count;
   char  *WelcomePtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigSetWelcome()\n");

   if (Config.HomePageCount >= MaxConfigHomePages)
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, maximum welcome files at line %d\n",
                  Utility, __LINE__);
      return;
   }

   WelcomePtr = lptr;
   while (*lptr && !isspace(*lptr))
   {
      *lptr = toupper(*lptr);
      lptr++;
   }
   Count = lptr - WelcomePtr;
   *lptr = '\0';

   if (!*WelcomePtr)
   {
      Config.ProblemCount++;
      if (ConfigReportProblems)
         fprintf (stdout, "%%%s-W-CONFIG, \"DirWelcome\" problem at line %d\n",
                  Utility, LineNumber);
      return;
   }

   if (Debug) fprintf (stdout, "malloc()ing %d bytes\n", Count+1);
   if ((Config.HomePageArray[Config.HomePageCount] = malloc (Count+1)) == NULL)
      ErrorExitVmsStatus (vaxc$errno, "memory allocation", __FILE__, __LINE__);

   strcpy (Config.HomePageArray[Config.HomePageCount], WelcomePtr);
   if (Debug)
      fprintf (stdout, "|%s|\n", Config.HomePageArray[Config.HomePageCount]);
   Config.HomePageCount++;
}

/*****************************************************************************/
/*
Return a pointer to the home page (welcome) file name stored against the index 
number of the array.  If none exists return a null string.  The calling 
function would use this by stepping through the index numbers 0 to n, until 
the null string was returned indicating the possible home page file names were 
exhausted.
*/

char* ConfigHomePage (int Number)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ConfigHomePage |%d|\n", Number);

   if (Number >= 0 && Number < Config.HomePageCount)
      return (Config.HomePageArray[Number]);
   else
      return ("");
}

/*****************************************************************************/

