/*****************************************************************************/
/*
                                 IsMap.c

Image-mapping module, integrated into the HTTPd (at no great cost), removing 
the overhead of image mapping via a script.  It relies on the function 
ConcludeProcessing() in httpd.c to do the actual redirection, either locally 
or to generate a response providing an absolute path. 

All of the "smarts" for this application have been plagiarized from the NCSA 
imagemap.c application (v1.8).  The routines directly lifted from that program 
are grouped together at the end of this code module.  Due acknowlegement to 
the original authors and maintainers.  Any copyright over sections of that 
code is also acknowleged: 

  ** mapper 1.2
  ** 7/26/93 Kevin Hughes, kevinh@pulua.hcc.hawaii.edu
  ** "macmartinized" polygon code copyright 1992 by Eric Haines, erich@eye.com

Up until this module was put together HFRD was reliant on a pre-compiled 
version of the CERN htimage.exe program used as a script.  Having this gives a 
certain independence, functionality integrated with the HTTP server, 
documented code (have a look at imagemap.c!), and better error reporting. 

This application allows the image configuration file to be specified in either 
CERN and NCSA format (or a mix of the two for that matter).  It is slightly 
more efficient to use the NCSA format, and this is suggested anyway, but the 
CERN format is provided for backward compatibility.  The region type keywords 
must supply at least 3 characters (although four is generally peferable).  
Comments can be prefixed by '#' or '!' characters, and blank lines are 
ignored. 

As an extension to both of the NCSA and CERN mapping applications, this HTTPd 
can accept and process server-side relative paths.   Hence it not only 
processes full URL's like "http://host/area/document.html", local absolute 
URL's like "/area/document.html", but also local relative URLs such as 
"../area/document.html", etc.  This provides for greater ease of mobility for 
mapped documents. 


NCSA FORMAT
-----------
default url
circle url x,y x,y              # centre-point then edge-point
point url x,y
polygon url x,y x,y [x,y ...]
rectangle url x,y x,y


CERN FORMAT
-----------
default url
circle (x,y) r url              # centre-point then radius
point (x,y) url
polygon (x,y) (x,y) [(x,y) ...] url
rectangle (x,y) (x,y) url


VERSION HISTORY
---------------
01-DEC-95  MGD  new for HTTPd version 3
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <math.h>

/* VMS related header files */
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application-related header files */
#include "httpd.h"

#define IsMapBufferSize 1024
#define MaxPathSize 512

/* required by the NCSA mapping functions, used by this module */
#define MAXVERTS 100
#define X 0
#define Y 1

/******************/
/* global storage */
/******************/

/* these are reported via IsMapExplainError() */
char  IsMapErrorCoordinate [] = "Coordinate specification problem",
      IsMapErrorConfused [] = "Confused",
      IsMapErrorDefault [] = "No <I>default</I> path specified",
      IsMapErrorIncomplete [] = "Incomplete specification",
      IsMapErrorIncorrect [] = "Incorrect number of coordinates",
      IsMapErrorExceeds [] = "Number of coordinates exceeds internal limit",
      IsMapErrorUnknownRegion [] = "Region specification problem";

/* this is reported via ErrorGeneral() */
char  IsMapErrorUnacceptable [] =
         "Client (browser) has supplied an unacceptable coordinate.";
                               
/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern char  SoftwareID[];
extern char  ErrorStringSize[];
extern struct AccountingStruct  Accounting;

/****************************/
/* functions in this module */
/****************************/

IsMapBegin (struct RequestStruct*);
IsMapEnd (struct RequestStruct*);
IsMapNextRecordAST (struct RAB*);
IsMapParseLine (struct RequestStruct*);
IsMapExplainError (struct RequestStruct*, char*, char*, int);

/**********************/
/* external functions */
/**********************/

ConcludeProcessing (struct RequestStruct*);
ErrorGeneral (struct RequestStruct*, char*, char*, int);
ErrorVmsStatus (struct RequestStruct*, int, char*, int);
unsigned char* HeapAlloc (struct RequestStruct*, int);
NextTask (struct RequestStruct*);
QioNetWrite (struct RequestStruct*, void*, char*, int);

/*****************************************************************************/
/*
Open the specified file.  If there is a problem opening it then report it and 
dispose of the thread.

Once open and connected the image mapping becomes AST-driven.

'RequestPtr->FileName' contains the VMS file specification.
*/

IsMapBegin (struct RequestStruct *RequestPtr)

{
   register char  *cptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "IsMapBegin() |%s|\n", RequestPtr->FileName);

   /* make sure any "If-Modified-Since:" request header line is ignored */
   RequestPtr->IfModifiedSinceBinaryTime[0] =
      RequestPtr->IfModifiedSinceBinaryTime[1] = 0;

   if (RequestPtr->AdjustAccounting)
   {
      Accounting.DoIsMapCount++;
      RequestPtr->AdjustAccounting = 0;
   }

   /* set up default error report information (just in case!) */
   RequestPtr->ErrorTextPtr = RequestPtr->PathInfoPtr;
   RequestPtr->ErrorHiddenTextPtr = RequestPtr->FileName;

   RequestPtr->FileFab = cc$rms_fab;
   RequestPtr->FileFab.fab$b_fac = FAB$M_GET;
   RequestPtr->FileFab.fab$l_fna = RequestPtr->FileName;  
   RequestPtr->FileFab.fab$b_fns = strlen(RequestPtr->FileName);
   RequestPtr->FileFab.fab$b_shr = FAB$M_SHRGET;

   if (VMSnok (status = sys$open (&RequestPtr->FileFab, 0, 0)))
   {
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      IsMapEnd (RequestPtr);
      return;
   }

   /* allocate heap memory for the file record buffer */
   if (RequestPtr->IsMapBufferPtr == NULL)
   {
      /* allow one byte for terminating null */
      if ((RequestPtr->IsMapBufferPtr =
          HeapAlloc (RequestPtr, IsMapBufferSize+1)) == NULL)
      {
         ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
         IsMapEnd (RequestPtr);
         return (SS$_NORMAL);
      }
   }

   RequestPtr->FileRab = cc$rms_rab;
   RequestPtr->FileRab.rab$l_fab = &RequestPtr->FileFab;
   /* 2 buffers and read ahead performance option */
   RequestPtr->FileRab.rab$b_mbf = 2;
   RequestPtr->FileRab.rab$l_rop = RAB$M_RAH;
   RequestPtr->FileRab.rab$l_ubf = RequestPtr->IsMapBufferPtr;
   RequestPtr->FileRab.rab$w_usz = IsMapBufferSize;

   if (VMSnok (status = sys$connect (&RequestPtr->FileRab, 0, 0)))
   {
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      IsMapEnd (RequestPtr);
      return;
   }

   /********************/
   /* begin processing */
   /********************/

   /* get the coordinate of the mouse-click */
   cptr = RequestPtr->QueryStringPtr;
   while (isspace(*cptr)) cptr++;
   if (Debug) fprintf (stdout, "cptr |%s|\n", cptr);
   if (isdigit(*cptr))
   {
      RequestPtr->IsMapClickCoord[X] = (double)atoi(cptr);
      while (*cptr && isdigit(*cptr)) cptr++;
      while (isspace(*cptr) || *cptr == ',') cptr++;
      if (isdigit(*cptr))
         RequestPtr->IsMapClickCoord[Y] = (double)atoi(cptr);
      else
      {
         RequestPtr->ResponseStatusCode = 400;
         ErrorGeneral (RequestPtr, IsMapErrorUnacceptable,
                       __FILE__, __LINE__);
         IsMapEnd (RequestPtr);
         return;
      }
   }
   else
   {
      RequestPtr->ResponseStatusCode = 400;
      ErrorGeneral (RequestPtr, IsMapErrorUnacceptable,
                    __FILE__, __LINE__);
      IsMapEnd (RequestPtr);
      return;
   }
   if (Debug)
      fprintf (stdout, "IsMapClickCoord |%f|%f|\n",
               RequestPtr->IsMapClickCoord[X], RequestPtr->IsMapClickCoord[Y]);

   /* allocate heap memory to accomodate the largest allowable path */
   if ((RequestPtr->LocationPtr = HeapAlloc (RequestPtr, MaxPathSize+1))
       == NULL)
   {
      ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
      IsMapEnd (RequestPtr);
      return;
   }
   /* make it an empty string to indicate no default path specified (yet) */
   RequestPtr->LocationPtr[0] = '\0';

   RequestPtr->IsMapCharNumber = RequestPtr->IsMapLineNumber = 0;
   RequestPtr->IsMapClosestPoint = HUGE_VAL;

   /* set then RAB user context storage to the client thread pointer */
   RequestPtr->FileRab.rab$l_ctx = RequestPtr;

   /* queue the first record read */
   sys$get (&RequestPtr->FileRab, &IsMapNextRecordAST, &IsMapNextRecordAST);

   /*
      Return to the calling routine now.  Subsequent processing is
      event-driven.  Routine completion ASTs drive the pre-processing.
   */
   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Ensure the source file is closed.  Conclude the connection, which will cause 
any local redirection to be effected, or send a redirection header to the 
client.
*/ 

IsMapEnd (struct RequestStruct *RequestPtr)

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "IsMapEnd()\n");

   if (RequestPtr->FileFab.fab$w_ifi)
   {
      sys$close (&RequestPtr->FileFab, 0, 0);
      RequestPtr->FileFab.fab$w_ifi = 0;
   }

   ConcludeProcessing (RequestPtr);
}

/*****************************************************************************/
/*
The asynchronous read of the next image-mapping configuration file record has 
completed.  Call a function to parse the configuration line.  Either end the 
task or queue another read depending on the results of the parse.
*/

IsMapNextRecordAST (struct RAB *RabPtr)

{
   int  status;
   struct RequestStruct  *RequestPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout,
               "IsMapNextRecordAST() sts: %%X%08.08X stv: %%X%08.08X\n",
               RabPtr->rab$l_sts,
               RabPtr->rab$l_stv);

   /* get the pointer to the thread from the RAB user context storage */
   RequestPtr = RabPtr->rab$l_ctx;

   if (VMSnok (RequestPtr->FileRab.rab$l_sts))
   {
      if (RequestPtr->FileRab.rab$l_sts == RMS$_EOF)
      {
         /***************/
         /* end of file */
         /***************/

         if (Debug)
            fprintf (stdout, "LocationPtr |%s|\n", RequestPtr->LocationPtr);

         if (!RequestPtr->LocationPtr[0] && RequestPtr->ErrorMessagePtr == NULL)
            IsMapExplainError (RequestPtr, IsMapErrorDefault,
                               __FILE__, __LINE__);

         IsMapEnd (RequestPtr);
         return;
      }

      /**********************/
      /* error reading file */
      /**********************/

      ErrorVmsStatus (RequestPtr, RequestPtr->FileRab.rab$l_sts,
                      __FILE__, __LINE__);
      IsMapEnd (RequestPtr);
      return;
   }

   /*****************************/
   /* process the record (line) */
   /*****************************/

   RequestPtr->IsMapLineNumber++;

   /* terminate the line */
   RequestPtr->FileRab.rab$l_ubf[RequestPtr->FileRab.rab$w_rsz] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", RequestPtr->FileRab.rab$l_ubf);

   IsMapParseLine (RequestPtr);

   if (Debug) fprintf (stdout, "LocationPtr |%s|\n", RequestPtr->LocationPtr);

   if (!RequestPtr->IsMapLineNumber || RequestPtr->ErrorMessagePtr != NULL)
   {
      /*
         Stop processing immediately because:
         The line number has reset to zero (when a successful mapping occured).
         An error has been reported.
      */
      IsMapEnd (RequestPtr);
   }
   else
   {
      /* queue the next record read */
      sys$get (&RequestPtr->FileRab, &IsMapNextRecordAST, &IsMapNextRecordAST);
   }

   /*
      Return to the AST-interrupted routine now.  Continued processing
      is event-driven.  Routine completion ASTs drive the image-mapping.
   */
}

/*****************************************************************************/
/*
Parse the configuration file record (line) for mapping directives in either 
the NCSA or CERN format.  If a mapping successful reset the line number 
storage to zero to indicate this.
*/ 

IsMapParseLine (struct RequestStruct* RequestPtr)

{
   register int  ccnt;
   register char  *bptr, *cptr, *sptr, *zptr;

   boolean  CernCompliant;
   int  status,
        PathSize;
   double  Distance;
   double  CoordArray [MAXVERTS][2];
   char  Path [MaxPathSize+1],
         RegionKeyword [256];

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "IsMapParseLine() |%s|\n",
               RequestPtr->FileRab.rab$l_ubf);

   CernCompliant = false;
   PathSize = 0;

   bptr = RequestPtr->FileRab.rab$l_ubf;

   while (isspace(*bptr)) bptr++;
   if (!*bptr || *bptr == '#' || *bptr == '!') return;

   /**********************/
   /* get region keyword */
   /**********************/

   zptr = (sptr = RegionKeyword) + sizeof(RegionKeyword);
   while (*bptr && !isspace(*bptr) && sptr < zptr)
      *sptr++ = tolower(*bptr++);
   if (sptr >= zptr)
   {
      RequestPtr->ResponseStatusCode = 500;
      ErrorGeneral (RequestPtr, ErrorStringSize, __FILE__, __LINE__);
      return;
   }
   *sptr = '\0';
   if (Debug) fprintf (stdout, "RegionKeyword |%s|\n", RegionKeyword);

   if (!(!memcmp (RegionKeyword, "cir", 3) ||
         !memcmp (RegionKeyword, "poi", 3) ||
         !memcmp (RegionKeyword, "pol", 3) ||
         !memcmp (RegionKeyword, "rec", 3) ||
         !memcmp (RegionKeyword, "def", 3)))
   {
      IsMapExplainError (RequestPtr, IsMapErrorUnknownRegion,
                         __FILE__, __LINE__);
      return;
   }

   while (isspace(*bptr)) bptr++;
   if (!*bptr || *bptr == '#' || *bptr == '!')
   {
      RequestPtr->IsMapCharNumber =
         bptr - (char*)RequestPtr->FileRab.rab$l_ubf + 1;
      IsMapExplainError (RequestPtr, IsMapErrorIncomplete,
                         __FILE__, __LINE__);
      return;
   }

   /************/
   /* get path */
   /************/

   if (isalpha(*bptr) || *bptr == '/' || *bptr == '.')
   {
      /******************/
      /* NCSA compliant */
      /******************/

      if (Debug) fprintf (stdout, "NCSA compliant\n");
      zptr = (sptr = Path) + sizeof(Path);
      while (*bptr && !isspace(*bptr) && sptr < zptr) *sptr++ = *bptr++;
      if (sptr >= zptr)
      {
         RequestPtr->ResponseStatusCode = 500;
         ErrorGeneral (RequestPtr, ErrorStringSize, __FILE__, __LINE__);
         return;
      }
      *sptr++ = '\0';
      PathSize = sptr - Path;
      if (Debug) fprintf (stdout, "Path |%s|\n", Path);
   }
   else
   if (isdigit(*bptr) || *bptr == '(')
   {
      /******************/
      /* CERN compliant */
      /******************/

      /*
          The CERN configuration file has the path following the
          coordinates.  Skip over any coordinates here looking
          for the path, terminate after them, get the path, resume.
          Turn any coordinate-grouping parentheses into spaces.
      */
      if (Debug) fprintf (stdout, "CERN compliant\n");
      CernCompliant = true;

      /* skip over everything looking like coordinates */
      cptr = bptr;
      while (*cptr && (isspace(*cptr) || isdigit(*cptr) ||
             *cptr == '(' || *cptr == ',' || *cptr == ')'))
      {
         if (*cptr == '(' || *cptr == ')')
            *cptr++ = ' ';
         else
            cptr++;
      }

      if (cptr[0] && isspace(cptr[-1]))
         cptr[-1] = '\0';
      else
      {
         RequestPtr->IsMapCharNumber =
            cptr - (char*)RequestPtr->FileRab.rab$l_ubf + 1;
         IsMapExplainError (RequestPtr, IsMapErrorConfused, __FILE__, __LINE__);
         return;
      }

      zptr = (sptr = Path) + sizeof(Path);
      while (*cptr && !isspace(*cptr) && sptr < zptr) *sptr++ = *cptr++;
      *sptr++ = '\0';
      PathSize = sptr - Path;
      if (Debug) fprintf (stdout, "Path |%s|\n", Path);
   }
   else
   {
      /***********************/
      /* specification error */
      /***********************/

      RequestPtr->IsMapCharNumber =
         bptr - (char*)RequestPtr->FileRab.rab$l_ubf + 1;
      IsMapExplainError (RequestPtr, IsMapErrorConfused, __FILE__, __LINE__);
      return;
   }

   /*****************/
   /* default path? */
   /*****************/

   if (RegionKeyword[0] == 'd')
   {
      /* know it will fit, path size is the same and already checked! */
      memcpy (RequestPtr->LocationPtr, Path, PathSize);

      /* not interested in any more of the line */
      return;
   }

   /*************************/
   /* process coordinate(s) */
   /*************************/

   ccnt = 0;
   while (*bptr)
   {
      if (Debug) fprintf (stdout, "bptr |%s|\n", bptr);
      while (isspace(*bptr)) bptr++;
      if (!*bptr || *bptr == '#' || *bptr == '!') break;

      if (ccnt >= MAXVERTS-1)
      {
         IsMapExplainError (RequestPtr, IsMapErrorExceeds, __FILE__, __LINE__);
         return;
      }

      if (!isdigit(*bptr))
      {
         /**************/
         /* should be! */
         /**************/

         RequestPtr->IsMapCharNumber =
            bptr - (char*)RequestPtr->FileRab.rab$l_ubf + 1;
         IsMapExplainError (RequestPtr, IsMapErrorConfused, __FILE__, __LINE__);
         return;
      }

      /********************/
      /* get X coordinate */
      /********************/

      if (Debug) fprintf (stdout, "bptr |%s|\n", bptr);
      CoordArray[ccnt][X] = (double)atoi(bptr);
      while (*bptr && isdigit(*bptr)) bptr++;

      /*************************************/
      /* find comma-separated Y coordinate */
      /*************************************/

      while (isspace(*bptr) || *bptr == ',') bptr++;
      if (Debug) fprintf (stdout, "bptr |%s|\n", bptr);

      if (isdigit(*bptr))
      {
         /********************/
         /* get Y coordinate */
         /********************/

         CoordArray[ccnt][Y] = (double)atoi(bptr);
         while (*bptr && isdigit(*bptr)) bptr++;
      }
      else
      {
         /*******************/
         /* no Y coordinate */
         /*******************/

         if (CernCompliant && RegionKeyword[0] == 'c' && ccnt)
         {
            /*
               CERN image mapping "circle" is "(X,Y) radius".
               Fudge it by creating an "X,Y" out of the radius.
            */
            CoordArray[ccnt][Y] = CoordArray[ccnt-1][Y] +
                                  CoordArray[ccnt][X];
            CoordArray[ccnt][X] = CoordArray[ccnt-1][X];
         }
         else
         {
            RequestPtr->IsMapCharNumber =
               bptr - (char*)RequestPtr->FileRab.rab$l_ubf + 1;
            IsMapExplainError (RequestPtr, IsMapErrorCoordinate,
                               __FILE__, __LINE__);
            return;
         }
      }
      if (Debug)
         fprintf (stdout, "CoordArray[%d] |%f|%f|\n",
                  ccnt, CoordArray[ccnt][X], CoordArray[ccnt][Y]);

      ccnt++;
   }

   if (!ccnt)
   {
      /* no coordinates have been supplied */
      RequestPtr->IsMapCharNumber =
         bptr - (char*)RequestPtr->FileRab.rab$l_ubf + 1;
      IsMapExplainError (RequestPtr, IsMapErrorIncomplete,
                         __FILE__, __LINE__);
      return;
   }

   /********************/
   /* lets try it out! */
   /********************/

   if (Debug) fprintf (stdout, "mapping |%s|\n", RegionKeyword);
   CoordArray[ccnt][X] = -1;

   if (RegionKeyword[0] == 'r')
   {
      if (ccnt != 2)
         IsMapExplainError (RequestPtr, IsMapErrorIncorrect,
                            __FILE__, __LINE__);
      else
      if (pointinrect (RequestPtr->IsMapClickCoord, CoordArray))
      {
         /* know it will fit, capacity is the same and already checked! */
         memcpy (RequestPtr->LocationPtr, Path, PathSize);
         /* indicate the mapping has been successful */
         RequestPtr->IsMapLineNumber = 0;
      }
      return;
   }

   if (RegionKeyword[0] == 'c')
   {
      if (ccnt != 2)
         IsMapExplainError (RequestPtr, IsMapErrorIncorrect,
                            __FILE__, __LINE__);
      else
      if (pointincircle (RequestPtr->IsMapClickCoord, CoordArray))
      {
         /* know it will fit, capacity is the same and already checked! */
         memcpy (RequestPtr->LocationPtr, Path, PathSize);
         /* indicate the mapping has been successful */
         RequestPtr->IsMapLineNumber = 0;
      }
      return;
   }

   if (!memcmp (RegionKeyword, "pol", 3))
   {
      if (ccnt < 3)
         IsMapExplainError (RequestPtr, IsMapErrorIncorrect,
                            __FILE__, __LINE__);
      else
      if (pointinpoly (RequestPtr->IsMapClickCoord, CoordArray))
      {
         /* know it will fit, capacity is the same and already checked! */
         memcpy (RequestPtr->LocationPtr, Path, PathSize);
         /* indicate the mapping has been successful */
         RequestPtr->IsMapLineNumber = 0;
      }
      return;
   }

   if (!memcmp (RegionKeyword, "poi", 3))
   {
      /* the essential functionality of this is also from NCSA imagemap.c */
      if (ccnt != 1)
         IsMapExplainError (RequestPtr, IsMapErrorIncorrect,
                            __FILE__, __LINE__);
      else
      {
         Distance = (RequestPtr->IsMapClickCoord[X] - CoordArray[0][X]) *
                    (RequestPtr->IsMapClickCoord[X] - CoordArray[0][X]) +
                    (RequestPtr->IsMapClickCoord[Y] - CoordArray[0][Y]) *
                    (RequestPtr->IsMapClickCoord[Y] - CoordArray[0][Y]);
         if (Distance < RequestPtr->IsMapClosestPoint)
         {
            RequestPtr->IsMapClosestPoint = Distance;
            /* know it will fit, capacity is the same and already checked! */
            memcpy (RequestPtr->LocationPtr, Path, PathSize);
         }
      }
      return;
   }
}

/*****************************************************************************/
/*
Generate a general error, with explanation about the image mapping error.
*/ 

int IsMapExplainError
(
struct RequestStruct *RequestPtr,
char *Explanation,
char *SourceFileName,
int SourceLineNumber
)
{
   static $DESCRIPTOR (ErrorMessageFaoDsc,
"!AZ.\n\
<P><I>(document requires correction)<I>\n");

   static $DESCRIPTOR (ErrorMessageLineFaoDsc,
"!AZ; at line !UL.\n\
<P><I>(document requires correction)<I>\n");

   static $DESCRIPTOR (ErrorMessageLineCharFaoDsc,
"!AZ; at line !UL, character !UL.\n\
<P><I>(document requires correction)<I>\n");

   unsigned short  Length;
   char  String [1024];
   $DESCRIPTOR (StringDsc, String);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "IsMapExplainError()\n");

   if (!RequestPtr->ResponseStatusCode)
      RequestPtr->ResponseStatusCode = 501;

   if (RequestPtr->IsMapLineNumber)
      if (RequestPtr->IsMapCharNumber)
         sys$fao (&ErrorMessageLineCharFaoDsc, &Length, &StringDsc,
                  Explanation, RequestPtr->IsMapLineNumber,
                  RequestPtr->IsMapCharNumber);
      else
         sys$fao (&ErrorMessageLineFaoDsc, &Length, &StringDsc,
                  Explanation, RequestPtr->IsMapLineNumber);
   else
      sys$fao (&ErrorMessageFaoDsc, &Length, &StringDsc,
               Explanation);

   String[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);

   ErrorGeneral (RequestPtr, String, SourceFileName, SourceLineNumber);
}

/*****************************************************************************/
/*
NCSA imagemap.c routines.
*/

int pointinrect(double point[2], double coords[MAXVERTS][2])
{
        return ((point[X] >= coords[0][X] && point[X] <= coords[1][X]) &&
        (point[Y] >= coords[0][Y] && point[Y] <= coords[1][Y]));
}

int pointincircle(double point[2], double coords[MAXVERTS][2])
{
        int radius1, radius2;

        radius1 = ((coords[0][Y] - coords[1][Y]) * (coords[0][Y] -
        coords[1][Y])) + ((coords[0][X] - coords[1][X]) * (coords[0][X] -
        coords[1][X]));
        radius2 = ((coords[0][Y] - point[Y]) * (coords[0][Y] - point[Y])) +
        ((coords[0][X] - point[X]) * (coords[0][X] - point[X]));
        return (radius2 <= radius1);
}

int pointinpoly(double point[2], double pgon[MAXVERTS][2])
{
        int i, numverts, inside_flag, xflag0;
        int crossings;
        double *p, *stop;
        double tx, ty, y;

        for (i = 0; pgon[i][X] != -1 && i < MAXVERTS; i++)
                ;
        numverts = i;
        crossings = 0;

        tx = point[X];
        ty = point[Y];
        y = pgon[numverts - 1][Y];

        p = (double *) pgon + 1;
        if ((y >= ty) != (*p >= ty)) {
                if ((xflag0 = (pgon[numverts - 1][X] >= tx)) ==
                (*(double *) pgon >= tx)) {
                        if (xflag0)
                                crossings++;
                }
                else {
                        crossings += (pgon[numverts - 1][X] - (y - ty) *
                        (*(double *) pgon - pgon[numverts - 1][X]) /
                        (*p - y)) >= tx;
                }
        }

        stop = pgon[numverts];

        for (y = *p, p += 2; p < stop; y = *p, p += 2) {
                if (y >= ty) {
                        while ((p < stop) && (*p >= ty))
                                p += 2;
                        if (p >= stop)
                                break;
                        if ((xflag0 = (*(p - 3) >= tx)) == (*(p - 1) >= tx)) {
                                if (xflag0)
                                        crossings++;
                        }
                        else {
                                crossings += (*(p - 3) - (*(p - 2) - ty) *
                                (*(p - 1) - *(p - 3)) / (*p - *(p - 2))) >= tx;
                        }
                }
                else {
                        while ((p < stop) && (*p < ty))
                                p += 2;
                        if (p >= stop)
                                break;
                        if ((xflag0 = (*(p - 3) >= tx)) == (*(p - 1) >= tx)) {
                                if (xflag0)
                                        crossings++;
                        }
                        else {
                                crossings += (*(p - 3) - (*(p - 2) - ty) *
                                (*(p - 1) - *(p - 3)) / (*p - *(p - 2))) >= tx;
                        }
                }
        }
        inside_flag = crossings & 0x01;
        return (inside_flag);
}

/*****************************************************************************/

