/*****************************************************************************/
/*
                                 MapUrl.c

Functions supporting mapping of URLs to VMS file specifications and VMS 
specifications to URLs, uses the HTTPd rule mapping file.  The number of rules 
it can store is not fixed, but is limited by available memory and the 
practical considerations of processing large, linear rule databases. 

This module is designed to be completely self-contained, only the global 
storage listed below is required for linking.  This is to allow other 
programs, particularly CGI scripts, to use the URL mapping and reverse-mapping 
functionality if they require it.  That is the primary reason for the separate 
rule-mapping database file with the HFRD VMS HTTPd.


VERSION HISTORY
---------------
15-FEB-95  MGD  bugfix, redirect rule
01-DEC-95  MGD  HTTPd version 3
24-MAR-95  MGD  bugfix, end-of-string sometimes not detected when rule matching
20-DEC-94  MGD  revised for multi-threaded HTTP daemon
20-JUN-94  MGD  single-threaded daemon
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application header */
#include "httpd.h"

/**********/
/* macros */
/**********/

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

/*******************/
/* data structures */
/*******************/

#ifdef __ALPHA
#   pragma member_alignment __save
#   pragma member_alignment
#endif

/* rule database is a singly-linked list of these structures */
struct MappingRuleStruct
{
   struct MappingRuleStruct  *NextPtr;
   int  MapFileLineNumber;
   char  RuleType;
   char  *TemplatePtr;
   char  *ResultPtr;
   /* template and result null-terminated strings stored straight after */
};

#ifdef __ALPHA
#   pragma member_alignment __restore
#endif

/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern char  SoftwareID[];
extern char  Utility[];

/****************************/
/* functions in this module */
/****************************/

char* MapUrl (char*, char*, char*, char*);
int MapUrl_UrlToVms (char*, char*);
int MapUrl_VmsToUrl (char*, char*);
int MapUrl_VirtualPath (char*, char*, char*, int);

/*****************************************************************************/
/*
This overly-long function (sorry) is not AST reentrant, but when executed 
within the context of the HTTPd, is executed either without the possibility of 
AST interruption, or during AST processing and so provides atomic 
functionality.  Pointers to any strings returned must be used immediately 
(before leaving AST context), or copied into storage maintained by the calling 
routine. 

Maps from URL format to VMS file specification, and REVERSE-MAPS from VMS file 
specification to URL format.  Maps scripts.

Always returns a pointer to char.  An error message is detected by being  
returned as a pointer to a string beginning with a null-character!  The 
message begins from character one. 

Call with 'PathPtr' pointing at a string containing the URL path and 'VmsPtr' 
pointing to the storage to contain the mapped VMS equivalent.

Call with 'VmsPtr' pointing to a VMS file specification (dev:[dir]file.ext) 
and 'PathPtr' pointing to an empty string, used as storage to contain the 
REVERSE-MAPPED file specification.

Call with 'PathPtr' pointing at a string containing the URL path, 'VmsPtr' 
pointing to storage to contain the mapped VMS equivalent, 'ScriptPtr' pointing 
to storage for the URL format script path and 'ScriptVmsPtr' pointing to 
storage to contain the VMS specification of the script procedure/image.  If 
there are two consecutive forward slashes following a script specification the 
path derived from the characters following the script are not mapped to a VMS 
specification, it is returned as is.  Allows "raw" paths to be provided to 
scripts.

Current Rules:

EXEC       template   result
FAIL       template
MAP        template   result
PASS       template   [result]
REDIRECT   template   result
SCRIPT     template   result
*/ 

char* MapUrl
(
char *PathPtr,
char *VmsPtr,
char *ScriptPtr,
char *ScriptVmsPtr
)
{
   static char  AccessDeniedDefault [] = 
                "\0Access denied (forbidden by default).",
                AccessDeniedInternal [] =
                "\0Access denied (INTERNAL processing problem!)",
                AccessDeniedNoRules [] = 
                "\0Access denied (NO RULES LOADED!)",
                AccessDeniedRule [] = 
                "\0Access denied (forbidden by rule).";

   /* this is a "counted string", byte of value one, then a null character */
   static unsigned char  AsciiNull [2] = { 1, 0 };
   static $DESCRIPTOR (MapErrorLineFaoDsc, "!AC!AZ, line !UL");
   static $DESCRIPTOR (MapErrorVmsStatusFaoDsc, "!AC!AZ, status: %X!8XL");

   static int  MappingRuleStructOverhead = sizeof(struct MappingRuleStruct);
   static struct MappingRuleStruct  *MappingRulesHeadPtr = NULL,
                                    *MappingRulesTailPtr = NULL;
   static char  DerivedPathBuffer [512],
                PathBuffer [512],
                VmsBuffer [512],
                ErrorString [64];
   static $DESCRIPTOR (ErrorStringDsc, ErrorString);

   register char  PathChar,
                  RuleChar;
   register char  *cptr, *pptr, *sptr, *zptr;
   register struct MappingRuleStruct  *mrptr;

   boolean  MapPathToVms;
   int  status,
        MapFileLineNumber,
        TotalLength;
   unsigned short  Length;
   char  Scratch [512],
         WildBuffer [512];

   /*********/
   /* begin */
   /*********/

   if (Debug)
   {
      fprintf (stdout, "MapUrl()");
      if (PathPtr != NULL) fprintf (stdout, " PathPtr: |%s|", PathPtr);
      if (VmsPtr != NULL) fprintf (stdout, " VmsPtr: |%s|", VmsPtr);
      if (ScriptPtr != NULL) fprintf (stdout, " (script)");
      fputc ('\n', stdout);
   }

   if (PathPtr == NULL && VmsPtr == NULL && ScriptPtr == NULL)
   {
      /**************************/
      /* read new mapping rules */
      /**************************/

      /* free the current rule database */
      mrptr = MappingRulesHeadPtr;
      while (mrptr != NULL)
      {
         cptr = (char*)mrptr;
         mrptr = mrptr->NextPtr;
         free (cptr);
      }
      MappingRulesHeadPtr = MappingRulesTailPtr = NULL;
   }

   if (MappingRulesHeadPtr == NULL)
   {
      /*************************/
      /* open the mapping file */
      /*************************/

      /* local storage */
      boolean  RequiresResult;
      int  TemplateLength,
           TemplateWildcardCount,
           ResultWildcardCount;
      char  *ResultPtr;
      char  Line [256];
      struct FAB  MapFileFab;
      struct RAB  MapFileRab;

      if (Debug) fprintf (stdout, "MapFileName |%s|\n", MapFileName);

      MapFileFab = cc$rms_fab;
      MapFileFab.fab$b_fac = FAB$M_GET;
      MapFileFab.fab$l_fna = MapFileName;
      MapFileFab.fab$b_fns = strlen(MapFileName);
      MapFileFab.fab$b_shr = FAB$M_SHRGET;

      if (VMSnok (status = sys$open (&MapFileFab, 0, 0)))
      {
         if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
         sys$fao (&MapErrorVmsStatusFaoDsc, &Length, &ErrorStringDsc,
                  AsciiNull, "Error opening mapping file", status);
         ErrorString[Length] = '\0';
         if (Debug) fprintf (stdout, "ErrorString |%s|\n", ErrorString+1);
         return (ErrorString);
      }

      /* record access block */
      MapFileRab = cc$rms_rab;
      MapFileRab.rab$l_fab = &MapFileFab;
      /* 2 buffers, transfer 6 blocks (3Kbytes) per physical IO */
      MapFileRab.rab$b_mbc = 6;
      MapFileRab.rab$b_mbf = 2;
      /* read ahead performance option */
      MapFileRab.rab$l_rop = RAB$M_RAH;
      MapFileRab.rab$l_ubf = Line;
      MapFileRab.rab$w_usz = sizeof(Line)-1;

      if (VMSnok (status = sys$connect (&MapFileRab, 0, 0)))
      {
         sys$close (&MapFileFab, 0, 0);
         if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
         sys$fao (&MapErrorVmsStatusFaoDsc, &Length, &ErrorStringDsc,
                  AsciiNull, "Error opening mapping file", status);
         ErrorString[Length] = '\0';
         if (Debug) fprintf (stdout, "ErrorString |%s|\n", ErrorString+1);
         return (ErrorString);
      }

      /*************************/
      /* read the mapping file */
      /*************************/

      MapFileLineNumber = 0;

      while (VMSok (status = sys$get (&MapFileRab, 0, 0)))
      {
         MapFileLineNumber++;
         Line[MapFileRab.rab$w_rsz] = '\0';
         if (Debug) fprintf (stdout, "Line |%s|\n", Line);

         cptr = Line;
         while (isspace(*cptr)) cptr++;
         /* if a blank or comment line, then ignore */
         if (!*cptr || *cptr == '!' || *cptr == '#') continue;
         /* get the first character of the rule, then skip over the rule */
         RuleChar = toupper(*cptr);
         while (*cptr && !isspace(*cptr)) cptr++;
         while (isspace(*cptr)) cptr++;
         /* copy template into scratch area */
         sptr = Scratch;
         while (*cptr && !isspace(*cptr)) *sptr++ = *cptr++;
         *sptr++ = '\0';
         TemplateLength = sptr - Scratch;
         /* skip intervening white-space */
         while (isspace(*cptr)) cptr++;
         /* copy (optional) result into scratch area, */
         ResultPtr = sptr;
         while (*cptr && !isspace(*cptr)) *sptr++ = *cptr++;
         *sptr++ = '\0';
         TotalLength = sptr - Scratch;

         if (Debug)
            fprintf (stdout, "|%c|%s|%s|\n", RuleChar, Scratch, ResultPtr);

         /******************************/
         /* basic consistency checking */
         /******************************/

         TemplateWildcardCount = 0;
         for (cptr = Scratch; *cptr; cptr++)
            if (*cptr == '*') TemplateWildcardCount++;

         ResultWildcardCount = 0;
         for (cptr = ResultPtr; *cptr; cptr++)
            if (*cptr == '*') ResultWildcardCount++;

         RequiresResult = false;
         switch (RuleChar)
         {
            case 'E' :
            case 'S' :
               if (ResultWildcardCount != 1 && TemplateWildcardCount != 1)
               {
                  fprintf (stdout,
                  "%%%s-W-RULE, wildcard mapping problem at line %d\n \\%s\\\n",
                     Utility, MapFileLineNumber, Line);
                  continue;
               }
               RequiresResult = true;
               break;

            case 'F' :
            case 'P' :
               break;

            case 'M' :
            case 'R' :
               RequiresResult = true;
               break;

            default :
               fprintf (stdout,
                  "%%%s-W-RULE, unknown rule at line %d\n \\%s\\\n",
                  Utility, MapFileLineNumber, Line);
               continue;
         }

         if (!Scratch[0])
         {
            /* there must be a "template" to map from, so this is an error */
            fprintf (stdout,
               "%%%s-W-RULE, no template specified at line %d\n \\%s\\\n",
               Utility, MapFileLineNumber, Line);
            continue;
         }

         if (RequiresResult && !ResultPtr[0])
         {
            /* there must be a "result" to map to, so this is an error */
            fprintf (stdout,
               "%%%s-W-RULE, result required at line %d\n \\%s\\\n",
               Utility, MapFileLineNumber, Line);
            continue;
         }

         if (ResultWildcardCount > TemplateWildcardCount)
         {
            /* cannot wildcard map more elements than in template */
            fprintf (stdout,
               "%%%s-W-RULE, wildcard mapping problem at line %d\n \\%s\\\n",
               Utility, MapFileLineNumber, Line);
            continue;
         }

         /************************************/
         /* enter the rule into the database */
         /************************************/

         if ((mrptr = (struct MappingRuleStruct*)
                       malloc (MappingRuleStructOverhead+TotalLength)) == NULL)
         {
            /* no point in continuing if we can't get memory for essentials */
            status = vaxc$errno;
            fprintf (stdout,
               "%%%s-E-RULE, fatal error while processing line %d\n",
               Utility, MapFileLineNumber);
            exit (status);
         }

         memcpy ((char*)mrptr+MappingRuleStructOverhead, Scratch, TotalLength);
         mrptr->NextPtr = NULL;
         mrptr->MapFileLineNumber = MapFileLineNumber;
         mrptr->RuleType = RuleChar;
         mrptr->TemplatePtr = (char*)mrptr + MappingRuleStructOverhead;
         mrptr->ResultPtr = (char*)mrptr + MappingRuleStructOverhead +
                            TemplateLength;
         if (MappingRulesHeadPtr == NULL)
            MappingRulesHeadPtr = mrptr;
         else
            MappingRulesTailPtr->NextPtr = mrptr;
         MappingRulesTailPtr = mrptr;
      }

      /**************************/
      /* close the mapping file */
      /**************************/

      sys$close (&MapFileFab, 0, 0);
      if (VMSnok (status) && status != RMS$_EOF)
      {
         if (Debug) fprintf (stdout, "sys$get() %%X%08.08X\n", status);

         /* free the current rule database */
         mrptr = MappingRulesHeadPtr;
         while (mrptr != NULL)
         {
            cptr = (char*)mrptr;
            mrptr = mrptr->NextPtr;
            free (cptr);
         }
         MappingRulesHeadPtr = MappingRulesTailPtr = NULL;

         sys$fao (&MapErrorVmsStatusFaoDsc, &Length, &ErrorStringDsc,
                  AsciiNull, "Error reading mapping file", status);
         ErrorString[Length] = '\0';
         if (Debug) fprintf (stdout, "ErrorString |%s|\n", ErrorString+1);
         return (ErrorString);
      }

      /* if just reading new rules then return now */
      if (PathPtr == NULL && VmsPtr == NULL && ScriptPtr == NULL)
         return ("\0\0");
   }

   /*****************************************/
   /* map from the URL or VMS to VMS or URL */
   /*****************************************/

   if (PathPtr == NULL) *(PathPtr = PathBuffer) = '\0';
   if (VmsPtr == NULL) *(VmsPtr = VmsBuffer) = '\0';
   /* making 'ScriptPtr' non-empty means "exec" and "script" are ignored */
   if (ScriptPtr == NULL) ScriptPtr = "?";

   if (PathPtr[0])
   {
      /* if the URL is not an empty string then force the conversion to VMS */
      MapPathToVms = true;
      *VmsPtr = '\0';
   }
   else
   {
      /* URL was an empty string, convert from VMS to URL */
      MapPathToVms = false;
      /* generate a URL-style version of the VMS specification */
      MapUrl_VmsToUrl (PathPtr, VmsPtr);
   }

   /***********************/
   /* loop thru the rules */
   /***********************/

   if ((mrptr = MappingRulesHeadPtr) == NULL) return (AccessDeniedNoRules);

   /*
      This loop may be restarted by the "exec" or "script" rule after mapping
      the script component to further map the derived path.  This is done by
      setting up 'PathPtr' to the derived path and then merely repointing
      'mrptr' to the first rule in the list and 'continue'ing the loop.
   */

   for (/* above! */; mrptr != NULL; mrptr = mrptr->NextPtr)
   {
      if (Debug)
         fprintf (stdout, "Rule |%c|%s|%s|\n",
                  mrptr->RuleType, mrptr->TemplatePtr, mrptr->ResultPtr);

      /* when mapping from VMS to URL then ignore all BUT "pass" rules */
      if (!MapPathToVms && mrptr->RuleType != 'P') continue;

      /* if 'ScriptPtr' is not to be used then ignore "exec" and "script" */
      if (ScriptPtr[0] && (mrptr->RuleType == 'E' || mrptr->RuleType == 'S'))
         continue;

      /*****************************************************************/
      /* compare the URL with the template/result in the mapping entry */
      /*****************************************************************/

      cptr = mrptr->TemplatePtr;
      /* if reverse-mapping VMS to URL use the result if available */
      if (!MapPathToVms && mrptr->ResultPtr[0]) cptr = mrptr->ResultPtr;

      /* 'WildBuffer' contains the null-separated wildcard-matched portions */
      zptr = (sptr = WildBuffer) + sizeof(WildBuffer)-1;
      PathChar = *(pptr = PathPtr);
      RuleChar = *cptr;
      /** if (Debug) fprintf (stdout, "|%s|%s|\n", pptr, cptr); **/

      while (PathChar && RuleChar &&
             (tolower(PathChar) == tolower(RuleChar) || RuleChar == '*'))
      {
         if (RuleChar != '*')
         {
            PathChar = *++pptr;
            RuleChar = *++cptr;
            /** if (Debug) fprintf (stdout, "|%s|%s|\n", pptr, cptr); **/
            continue;
         }
         /* asterisk wildcard, multiple consecutive treated as single */
         while ((RuleChar = *cptr) == '*') cptr++;
         if (PathChar) *sptr++ = PathChar;
         if (sptr >= zptr) return (AccessDeniedInternal);
         while ((PathChar = *++pptr) &&
                tolower(PathChar) != tolower(RuleChar) &&
                sptr < zptr)
            *sptr++ = PathChar;
         if (sptr >= zptr) return (AccessDeniedInternal);
         *sptr++ = '\0';
         if (sptr >= zptr) return (AccessDeniedInternal);
         /** if (Debug) fprintf (stdout, "|%s|%s|\n", pptr, cptr); **/
      }

      /* account for trailing asterisk on rule template */
      if (RuleChar == '*')
         while ((RuleChar = *++cptr) == '*');

      /* if not matched then straight to next rule */
      if (PathChar || RuleChar) continue;

      /* final, terminating null for 'WildBuffer' */
      *sptr = '\0';

      /****************/
      /* matched them */
      /****************/

      if (MapPathToVms)
      {
         /***************************/
         /* mapping from URL to VMS */
         /***************************/

         switch (mrptr->RuleType)
         {
         case 'E' :

            /* script directory mapping rule (different to "script") */

            /* get the virtual path to the script (CGI: "WWW_SCRIPT_NAME") */
            cptr = WildBuffer;
            sptr = mrptr->TemplatePtr;
            zptr = (pptr = ScriptPtr) + 255;
            while (*sptr && *sptr != '*' && pptr < zptr) *pptr++ = *sptr++;
            if (pptr >= zptr) return (AccessDeniedInternal);
            /* get the wildcard buffer up to the first slash (script name) */
            while (*cptr && *cptr != '/' && pptr < zptr) *pptr++ = *cptr++;
            if (pptr >= zptr) return (AccessDeniedInternal);
            *pptr = '\0';

            /* now get the URL-format path to the physical script */
            cptr = WildBuffer;
            sptr = mrptr->ResultPtr;
            zptr = (pptr = Scratch) + 255;
            /* get fixed portion of rule as script directory location */
            while (*sptr && *sptr != '*' && pptr < zptr) *pptr++ = *sptr++;
            if (pptr >= zptr) return (AccessDeniedInternal);
            /* get the wildcard buffer up to the first slash (script name) */
            while (*cptr && *cptr != '/' && pptr < zptr) *pptr++ = *cptr++;
            if (pptr >= zptr) return (AccessDeniedInternal);
            *pptr = '\0';

            /* convert the URL-format script to a VMS-format specification */
            MapUrl_UrlToVms (Scratch, ScriptVmsPtr);

            /* get all including and after the first slash as the new path */
            zptr = (pptr = DerivedPathBuffer) + sizeof(DerivedPathBuffer)-1;
            while (*cptr && pptr < zptr) *pptr++ = *cptr++;
            if (pptr >= zptr) return (AccessDeniedInternal);
            *pptr++ = '\0';
            TotalLength = pptr - DerivedPathBuffer;

            if (Debug)
               fprintf (stdout, "EXEC |%s|%s|%s|\n",
                        ScriptPtr, ScriptVmsPtr, DerivedPathBuffer);

            if (!DerivedPathBuffer[0]) return ("\0\0");

            /* two consecutive slashes means do not map derived path */
            if (*((unsigned short*)DerivedPathBuffer) == 0x2f2f)
            {
               /* remove the leading slash, then return */
               memcpy (DerivedPathBuffer, DerivedPathBuffer+1, TotalLength-1);
               return (DerivedPathBuffer);
            }

            /* restarting at first rule map the path derived from the script */
            *VmsPtr = '\0';
            PathPtr = DerivedPathBuffer;
            mrptr = MappingRulesHeadPtr;
            continue;

         case 'F' :

            if (Debug) fprintf (stdout, "FAIL |%s|\n", PathPtr);
            return (AccessDeniedRule);

         case 'M' :

            /* map the result onto the original (even for "redirect") */

            cptr = WildBuffer;
            /* place this in a buffer so not to overwrite original path */
            zptr = (pptr = PathPtr = PathBuffer) + sizeof(PathBuffer)-1;
            sptr = mrptr->ResultPtr;
            /* scan through the result string */
            while (*sptr)
            {
               while (*sptr && *sptr != '*' && pptr < zptr) *pptr++ = *sptr++;
               if (pptr >= zptr) return (AccessDeniedInternal);
               if (!*sptr) break;
               /* a wildcard asterisk, substitute from original path */
               while (*++sptr == '*');
               while (*cptr && pptr < zptr) *pptr++ = *cptr++;
               if (pptr >= zptr) return (AccessDeniedInternal);
               cptr++;
            }
            *pptr = '\0';

            if (mrptr->RuleType == 'M')
            {
               /* continue with the substituted URL mapping */
               if (Debug) fprintf (stdout, "MAP |%s|\n", PathPtr);
               continue;
            }

            if (Debug) fprintf (stdout, "REDIRECT |%s|\n", PathPtr);
            return (PathPtr);

         case 'P' :
         case 'R' :

            /* if there is no result to map just pass back the original path */
            if (!*(sptr = mrptr->ResultPtr)) 
            {
               /* convert the URL-style to a VMS-style specification */
               MapUrl_UrlToVms (PathPtr, VmsPtr);

               if (Debug) fprintf (stdout, "PASS |%s|%s|\n", PathPtr, VmsPtr);
               return (PathPtr);
            }

            cptr = WildBuffer;
            zptr = (pptr = PathBuffer) + sizeof(PathBuffer)-1;
            /* scan through the result string */
            while (*sptr)
            {
               while (*sptr && *sptr != '*' && pptr < zptr) *pptr++ = *sptr++;
               if (pptr >= zptr) return (AccessDeniedInternal);
               if (!*sptr) break;
               /* a wildcard asterisk, substitute from original path */
               while (*++sptr == '*');
               while (*cptr && pptr < zptr) *pptr++ = *cptr++;
               if (pptr >= zptr) return (AccessDeniedInternal);
               cptr++;
            }
            *pptr = '\0';

            if (mrptr->RuleType == 'P')
            {
               /* convert the URL-style to a VMS-style specification */
               MapUrl_UrlToVms (PathBuffer, VmsPtr);
               if (Debug) fprintf (stdout, "PASS |%s|%s|\n", PathPtr, VmsPtr);
               return (PathPtr);
            }
            else
            {
               if (Debug) fprintf (stdout, "PASS |%s|%s|\n", PathPtr, VmsPtr);
               return (PathBuffer);
            }


         case 'S' :

            /* script file name mapping rule (different to "exec") */

            /* get the virtual path to the script (CGI: "WWW_SCRIPT_NAME") */
            sptr = mrptr->TemplatePtr;
            zptr = (pptr = ScriptPtr) + 255;
            while (*sptr && *sptr != '*' && pptr < zptr) *pptr++ = *sptr++;
            if (pptr >= zptr) return (AccessDeniedInternal);
            if (pptr > ScriptPtr)
               if (pptr[-1] == '/')
                  pptr--;
            *pptr = '\0';

            /* now get the URL-format path to the physical script */
            zptr = (pptr = Scratch) + 255;
            sptr = mrptr->ResultPtr;
            /* 0x2a2f is the numeric equivalent of "/*" */
            while (*sptr && *sptr != '*' &&
                   *(unsigned short*)sptr != 0x2a2f && pptr < zptr)
               *pptr++ = *sptr++;
            if (pptr >= zptr) return (AccessDeniedInternal);
            *pptr = '\0';

            /* convert the URL-format script to a VMS-format specification */
            MapUrl_UrlToVms (Scratch, ScriptVmsPtr);

            /* get wildcard matched second section of path as new path */
            zptr = (pptr = DerivedPathBuffer) + sizeof(DerivedPathBuffer)-1;
            if (*sptr == '/') *pptr++ = '/';
            cptr = WildBuffer;
            while (*cptr && pptr < zptr) *pptr++ = *cptr++;
            if (pptr >= zptr) return (AccessDeniedInternal);
            *pptr++ = '\0';
            TotalLength = pptr - DerivedPathBuffer;

            if (Debug)
               fprintf (stdout, "SCRIPT |%s|%s|%s|\n",
                        ScriptPtr, ScriptVmsPtr, DerivedPathBuffer);

            if (!DerivedPathBuffer[0]) return ("\0\0");

            /* two consecutive slashes means do not map derived path */
            if (*((unsigned short*)DerivedPathBuffer) == 0x2f2f)
            {
               /* remove the leading slash, then return */
               memcpy (DerivedPathBuffer, DerivedPathBuffer+1, TotalLength-1);
               return (DerivedPathBuffer);
            }

            /* restarting at first rule map the path derived from the script */
            *VmsPtr = '\0';
            PathPtr = DerivedPathBuffer;
            mrptr = MappingRulesHeadPtr;
            continue;
         }
      }
      else
      {
         /***************************/
         /* mapping from VMS to URL */
         /***************************/

         /* REVERSE maps a VMS "result" to a "template" :^) */

         cptr = WildBuffer;
         pptr = PathPtr;
         sptr = mrptr->TemplatePtr;
         /* scan through the template string */
         while (*sptr)
         {
            while (*sptr && *sptr != '*') *pptr++ = *sptr++;
            if (!*sptr) break;
            /* a wildcard asterisk, substitute from result path */
            while (*++sptr == '*');
            if (*cptr)
            {
               while (*cptr) *pptr++ = *cptr++;
               cptr++;
            }
         }
         *pptr = '\0';

         if (Debug) fprintf (stdout, "PASS |%s|\n", PathPtr);
         return (PathPtr);
      }
   }

   /***********************/
   /* a mapping not found */
   /***********************/

   return (AccessDeniedDefault);
}

/*****************************************************************************/
/*
Convert a URL-style specification into a RMS-style specification.  For 
example: "/disk/dir1/dir2/file.txt" into "disk:[dir1.dir2]file.txt".  Non-RMS 
characters (non-alphanumberic and non-"$_-" are converted to "$'.  Where a 
specification has multiple ".", all but the final one of the file component is 
converted to a "$".
*/ 
 
MapUrl_UrlToVms
(
char *PathPtr,
char *VmsPtr
)
{
   register int  ncnt, ccnt;
   register char  *cptr, *pptr, *sptr;

   char  PathComponents [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MapUrl_UrlToVms() |%s|\n", PathPtr);

   if (!PathPtr[0])
   {
      VmsPtr[0] = VmsPtr[1] = '\0';
      return;
   }

   /******************************************/
   /* parse out each "/" delimited component */
   /******************************************/

   /* each component becomes a null-terminated string in a character array */
   sptr = PathComponents;
   pptr = PathPtr;
   ccnt = 0;
   while (*pptr)
   {
      if (*pptr == '/') pptr++;
      /* copy, converting all non-RMS characters into exclamation marks */
      cptr = sptr;
      while (*pptr && *pptr != '/')
      {
         if (isalnum (*pptr))
            *sptr++ = toupper(*pptr++);
         else
         if (pptr[0] == '.' && pptr[1] == '.' && pptr[2] == '.')
         {
            *sptr++ = *pptr++;
            *sptr++ = *pptr++;
            *sptr++ = *pptr++;
         }
         else
         if (*pptr == '$' || *pptr == '_' || *pptr == '-' ||
             *pptr == '*' || *pptr == '%' || *pptr == ';')
            *sptr++ = *pptr++;
         else
         {
            *sptr++ = '!';
            pptr++;
         }
      }
      *sptr++ = '\0';
      ccnt++;
      if (Debug) fprintf (stdout, "%d |%s|\n", ccnt, cptr);
   }

   /********************************/
   /* build VMS file specification */
   /********************************/

   if (ccnt == 1)
   {
      /* single component */
      cptr = PathComponents;
      sptr = VmsPtr;
      while (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
   }
   else
   if (ccnt == 2)
   {
      /***********************/
      /* top-level directory */
      /***********************/

      cptr = PathComponents;
      sptr = VmsPtr;
      /* start with the first component, the "device" */
      while (*cptr) *sptr++ = *cptr++;
      cptr++;
      /* append a Master File Directory component */
      strcpy (sptr, ":[000000]");
      while (*sptr) sptr++;
      /* add the second component, the "file" */
      while (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
   }
   else
   if (ccnt >= 3)
   {
      /****************/
      /* subdirectory */
      /****************/

      cptr = PathComponents;
      sptr = VmsPtr;
      /* start with the first component, the "device" */
      while (*cptr) *sptr++ = *cptr++;
      cptr++;
      *sptr++ = ':';
      *sptr++ = '[';
      if (cptr[0] == '.' && cptr[1] == '.' && cptr[2] == '.')
      {
         strcpy (sptr, "000000");
         sptr += 6;
      }
      for (ncnt = 2; ncnt < ccnt; ncnt++)
      {
         /* add the next component, a "directory" */
         if (cptr[0] == '.' && cptr[1] == '.' && cptr[2] == '.')
            while (*cptr) *sptr++ = *cptr++;
         else
         {
            if (ncnt > 2) *sptr++ = '.';
            while (*cptr) *sptr++ = *cptr++;
         }
         cptr++;
      }
      *sptr++ = ']';
      /* add the last component, the "file" */
      while (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
   }

   /*******************************************/
   /* convert intermediate non-RMS characters */
   /*******************************************/

   /* convert the last of the file name "!" to a ".", all others to "$" */
   sptr--;
   while (sptr > VmsPtr && *sptr != '!' && *sptr != ']') sptr--;
   if (*sptr == '!') *sptr = '.';
   while (sptr >= VmsPtr)
      if (*sptr == '!')
         *sptr-- = '$';
      else
         *sptr--;

   if (!VmsPtr[0]) VmsPtr[1] = '\0';

   if (Debug) fprintf (stdout, "VmsPtr |%s|\n", VmsPtr);
}

/*****************************************************************************/
/*
Convert a VMS file specification into a URL-style specification.  For example:
"DEVICE:[DIR1.DIR2]FILE.TXT" into "/device/dir1/dir2/file.txt".
*/ 
 
MapUrl_VmsToUrl
(
char *PathPtr,
char *VmsPtr
)
{
   register char  *pptr, *vptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "MapUrl_VmsToUrl() |%s|\n", VmsPtr);

   vptr = VmsPtr;
   pptr = PathPtr;
   *pptr++ = '/';
   /* copy the device and directory components */
   while (*vptr)
   {
      if (*vptr == ':' && *(vptr+1) == '[')
      {
         vptr++;
         vptr++;
         /* remove any reference to a Master File Directory */
         if (strncmp (vptr, "000000", 6) == 0)
            vptr += 6;
         else
            *pptr++ = '/';
      }
      if (*vptr == '.')
      {
         if (vptr[1] == '.' && vptr[2] == '.')
         {
            *pptr++ = '/';
            *pptr++ = *vptr++;
            *pptr++ = *vptr++;
            *pptr++ = *vptr++;
         }
         else
         {
            vptr++;
            *pptr++ = '/';
         }
      }
      else
      if (*vptr == ']')
      {
         vptr++;
         *pptr++ = '/';
         break;
      }
      else
         *pptr++ = tolower(*vptr++);
   }
   /* copy the file component */
   while (*vptr) *pptr++ = tolower(*vptr++);
   *pptr++ = '\0';
   *pptr = '\0';

   if (Debug) fprintf (stdout, "PathPtr |%s|\n", PathPtr);
}

/*****************************************************************************/
/*
Given the current document's URL path and the specified document's URL path, 
generate a resultant URL path based on whether the specified document's path 
is absolute or relative the the current document's path.  Insufficient space to
contain the result path will not crash the function and the result will be set
to an empty string.
*/ 

int MapUrl_VirtualPath
(
char *CurrentPath,
char *DocumentPath,
char *ResultPath,
int SizeOfResultPath
)
{
   register char  *cptr, *dptr, *sptr, *tptr, *zptr;

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "MapUrl_VirtualPath() |%s|%s|\n",
               CurrentPath, DocumentPath);

   if (*(dptr = DocumentPath) == '/' || strstr (DocumentPath, "//") != NULL) 
   {
      /*****************/
      /* absolute path */
      /*****************/

      strcpy (ResultPath, dptr);
      zptr = (sptr = ResultPath) + SizeOfResultPath-1;
      while (*dptr && sptr < zptr) *sptr++ = *dptr++;
      if (sptr == zptr) sptr = ResultPath;
      *sptr = '\0';
      if (Debug) fprintf (stdout, "ResultPath |%s|\n", ResultPath);
      return;
   }

   /* step over any "relative to this directory" syntax ("./") */
   if (dptr[0] == '.' && dptr[1] == '/') dptr += 2;
   if (*dptr != '.')
   {
      /*****************/
      /* inferior path */
      /*****************/

      zptr = (sptr = tptr = ResultPath) + SizeOfResultPath-1;
      cptr = CurrentPath;
      while (*cptr && sptr < zptr)
      {
         if (*cptr == '/') tptr = sptr+1;
         *sptr++ = *cptr++;
      }
      sptr = tptr;
      while (*dptr && sptr < zptr) *sptr++ = *dptr++;
      if (sptr == zptr) sptr = ResultPath;
      *sptr = '\0';
      if (Debug) fprintf (stdout, "ResultPath |%s|\n", ResultPath);
      return;
   }

   /*****************/
   /* relative path */
   /*****************/

   zptr = (sptr = tptr = ResultPath) + SizeOfResultPath-1;
   cptr = CurrentPath;
   while (*cptr && sptr < zptr)
   {
      if (*cptr == '/') tptr = sptr;
      *sptr++ = *cptr++;
   }
   sptr = tptr;
   /* loop, stepping back one level for each "../" in the document path */
   while (dptr[0] == '.' && dptr[1] == '.' && dptr[2] == '/')
   {
      dptr += 3;
      if (sptr > ResultPath) sptr--;
      while (sptr > ResultPath && *sptr != '/') sptr--;
   }
   if (sptr > ResultPath)
      sptr++;
   else
      *sptr++ = '/';
   while (*dptr && sptr < zptr) *sptr++ = *dptr++;
   if (sptr == zptr) sptr = ResultPath;
   *sptr = '\0';
   if (Debug) fprintf (stdout, "ResultPath |%s|\n", ResultPath);
}

/*****************************************************************************/

