/*****************************************************************************
/*
                                 Support.c

Miscellaneous support functions for HTTPd.


VERSION HISTORY
---------------
01-DEC-95  MGD  HTTPd version 3.0
27-SEP-95  MGD  provide GMT functions
20-DEC-94  MGD  initial development
*/
/*****************************************************************************/

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <time.h>

/* VMS related header files */
#include <descrip.h>
#include <jpidef.h>
#include <libdef.h>
#include <libdtdef.h>
#include <lnmdef.h>
#include <prvdef.h>
#include <ssdef.h>
#include <stsdef.h>

#include "httpd.h"

/**********/
/* macros */
/**********/

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) (!((x) & STS$M_SUCCESS))
 
int  HeapOverhead = sizeof(struct HeapStruct*) +
                    sizeof(struct HeapStruct*) +
                    sizeof(unsigned int);

/******************/
/* global storage */
/******************/

boolean TimeAheadOfGmt;
char  TimeGmtString [32],
      TimeGmtVmsString [34];
unsigned long  TimeGmtDeltaBinary [2];

/********************/
/* external storage */
/********************/

extern boolean  Debug;
extern int ServerPort;
extern char  ServerHostName[];
extern char  SoftwareID[];
extern struct AccountingStruct Accounting;

/****************************/
/* functions in this module */
/****************************/

int CopyTextIntoHtml (char*, char*, int);
int DefineMonitorLogicals (struct RequestStruct*);
int DefineMonitorLogicalsHere (struct RequestStruct*);
unsigned char* HeapAlloc (struct RequestStruct*, int);
HeapDebug (struct RequestStruct*);
HeapFree (struct RequestStruct*);
unsigned char* HeapRealloc (struct RequestStruct*, unsigned char*, int);
HttpGmTime (char*, unsigned long*);
HttpGmTimeString (char*, unsigned long*);
int IfModifiedSince (struct RequestStruct*, unsigned long*, unsigned long*);
ServerReport (struct RequestStruct*);
int TimeAdjustGMT (boolean, unsigned long*);
int TimeSetGMT ();
int TimeVmsToUnix (unsigned long*, struct tm*);

/**********************/
/* external functions */
/**********************/

int BufferOutput (struct RequestStruct*, void*, char*, int);
ConcludeProcessing (struct RequestStruct*);
ErrorHeapAlloc (struct RequestStruct*, char*, int);
ErrorExitVmsStatus (int, char*, char*, int);
ErrorVmsStatus (struct RequestStruct*, int, char*, int);
QioNetWrite (struct RequestStruct*, void*, char*, int);

/*****************************************************************************/
/*
Allocate dynamic memory to an individual thread's heap.  Return a pointer to 
the start of new *usable* memory area if successful, return a NULL if not.

The heap is maintained as a doubly-linked list of separate calloc()ed chunks
of memory.  The size of each memory area is contained in an unsigned int
immediately preceding each returned pointer.  This can be used to provide the
length of dynamically stored, null-terminated string (without strlen()ing it,
remember to subtract one to exclude the null terminator), or to determine if a
given area is large enough to be re-used, etc.
*/ 

unsigned char* HeapAlloc
(
struct RequestStruct *RequestPtr,
int ChunkSize
)
{
   register struct HeapStruct  *NewPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HeapAlloc() %d\n", ChunkSize);

   if ((NewPtr = (struct HeapStruct*)calloc (1, ChunkSize + HeapOverhead))
       == NULL)
   {
      Accounting.HeapCallocFailCount++;
      return (NULL);
   }

   if (Debug) fprintf (stdout, "NewPtr: %d\n", NewPtr);
   RequestPtr->HeapSize += ChunkSize + HeapOverhead;

   if (RequestPtr->HeapHeadPtr == NULL)
   {
      RequestPtr->HeapHeadPtr = NewPtr;
      NewPtr->PrevPtr = NULL;
   }
   else
   {
      RequestPtr->HeapTailPtr->NextPtr = NewPtr;
      NewPtr->PrevPtr = RequestPtr->HeapTailPtr;
   }
   NewPtr->NextPtr = NULL;
   NewPtr->ChunkSize = ChunkSize;
   RequestPtr->HeapTailPtr = NewPtr;

   if (Debug) HeapDebug (RequestPtr);

   return ((unsigned char*)NewPtr + HeapOverhead);
}

/*****************************************************************************/
/*
Expand (or even contract) an individual chunk.  See HeapAlloc().
*/ 

unsigned char* HeapRealloc
(
struct RequestStruct *RequestPtr,
unsigned char *ChunkPtr,
int ChunkSize
)
{
   register struct HeapStruct  *NewPtr,
                               *NextPtr,
                               *OldPtr,
                               *PrevPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HeapRealloc() %d %d\n", ChunkPtr, ChunkSize);

   if (ChunkPtr == NULL) return (HeapAlloc (RequestPtr, ChunkSize));

   /* point at the start of the chunk, not the usable area which was passed */
   OldPtr = (struct HeapStruct*)(ChunkPtr - HeapOverhead);

   if (OldPtr->ChunkSize >= ChunkSize) return (ChunkPtr);

   PrevPtr = OldPtr->PrevPtr;
   NextPtr = OldPtr->NextPtr;

   RequestPtr->HeapSize -= OldPtr->ChunkSize + HeapOverhead;

   if ((NewPtr = (struct HeapStruct*)
        realloc (OldPtr, ChunkSize + HeapOverhead)) == NULL)
   {
      Accounting.HeapReallocFailCount++;
      /* remove this former chunk from the linked list */
      if (PrevPtr == NULL)
         RequestPtr->HeapHeadPtr = NextPtr;
      else
         PrevPtr->NextPtr = NextPtr;
      if (NextPtr == NULL)
         RequestPtr->HeapTailPtr = PrevPtr;
      else
         NextPtr->PrevPtr = PrevPtr;
      return (NULL);
   }

   if (Debug) fprintf (stdout, "NewPtr: %d\n", NewPtr);
   RequestPtr->HeapSize += ChunkSize + HeapOverhead;

   if (NewPtr != OldPtr)
   {
      if (PrevPtr != NULL) PrevPtr->NextPtr = NewPtr;
      if (NextPtr != NULL) NextPtr->PrevPtr = NewPtr;
      NewPtr->PrevPtr = PrevPtr;
      NewPtr->NextPtr = NextPtr;
      if (RequestPtr->HeapHeadPtr == OldPtr) RequestPtr->HeapHeadPtr = NewPtr;
      if (RequestPtr->HeapTailPtr == OldPtr) RequestPtr->HeapTailPtr = NewPtr;
   }

   NewPtr->ChunkSize = ChunkSize;

   if (Debug) HeapDebug (RequestPtr);

   return ((unsigned char*)NewPtr + HeapOverhead);
}

/*****************************************************************************/
/*
Dispose of the thread's heap of dynamic memory by traversing the linked list
and disposing of each memory chunk.
*/ 

HeapFree (struct RequestStruct *RequestPtr)

{
   register struct HeapStruct  *cptr, *hptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HeapFree() %d bytes\n", RequestPtr->HeapSize);

   hptr = RequestPtr->HeapHeadPtr;
   while (hptr)
   {
      cptr = hptr;
      hptr = hptr->NextPtr;
      if (Debug)
         fprintf (stdout, "free() %d|%d|%d|%d|%d|%30.30s|\n",
                  cptr, cptr->PrevPtr, cptr->NextPtr, cptr->ChunkSize,
                  (unsigned char*)cptr+HeapOverhead,
                  (unsigned char*)cptr+HeapOverhead);
      free (cptr);
   }
   RequestPtr->HeapHeadPtr = RequestPtr->HeapTailPtr = NULL;
   RequestPtr->HeapSize = 0;
}

/*****************************************************************************/
/*
Display the thread's heap of dynamic memory by traversing the linked list
and displaying the contents of each memory chunk.
*/ 

HeapDebug (struct RequestStruct *RequestPtr)

{
   register struct HeapStruct  *hptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HeapDebug() %d bytes\n", RequestPtr->HeapSize);

   hptr = RequestPtr->HeapHeadPtr;
   while (hptr)
   {
      fprintf (stdout, "%d|%d|%d|%d|%d|%30.30s|\n",
               hptr, hptr->PrevPtr, hptr->NextPtr, hptr->ChunkSize,
               (unsigned char*)hptr+HeapOverhead,
               (unsigned char*)hptr+HeapOverhead);
      hptr = hptr->NextPtr;
   }
}

/*****************************************************************************/
/*
If the object has been modified since the specified date and time then return 
a normal status indicating that the object has been modified.  If not modified 
then send a "304 Not Modified" HTTP header and return a LIB$_NEGTIM status to 
indicate the file should not be sent.  With VMS' fractions of a second, add 
one second to the specified 'since' time to ensure a reliable comparison. 
*/ 
 
int IfModifiedSince
(
struct RequestStruct *RequestPtr,
unsigned long *BinTimePtr,
unsigned long *SinceBinTimePtr
)
{
   static unsigned long  OneSecondDelta [2] = { -10000000, -1 };
   static $DESCRIPTOR (StringDsc, "");

   static $DESCRIPTOR (Http304HeaderFaoDsc,
"!AZ 304 Not modified.\r\n\
Server: !AZ\r\n\
Date: !AZ\r\n\
\r\n");

   int  status;
   unsigned short  Length;
   unsigned long  AdjustedBinTime[2],
                  ScratchBinTime [2];
   char  String [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "IfModifiedSince()\n");

   /* add one second to the modified time, ensures a negative time */
   if (VMSnok (status =
       lib$add_times (SinceBinTimePtr, &OneSecondDelta, &AdjustedBinTime)))
   {
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      return (status);
   }
   if (Debug) fprintf (stdout, "sys$add_times() %%X%08.08X\n", status);

   /* if a positive time results the file has been modified */
   if (VMSok (status =
       lib$sub_times (BinTimePtr, &AdjustedBinTime, &ScratchBinTime)))
      return (status);

   if (Debug) fprintf (stdout, "sys$sub_times() %%X%08.08X\n", status);
   if (status != LIB$_NEGTIM)
   {
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      return (status);
   }

   RequestPtr->ResponseStatusCode = 304;

   StringDsc.dsc$a_pointer = String;
   StringDsc.dsc$w_length = sizeof(String)-1;
   sys$fao (&Http304HeaderFaoDsc, &Length, &StringDsc,
            HttpProtocol, SoftwareID, RequestPtr->GmDateTime);
   String[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", String);

   /* allocate heap memory for the response header */
   if ((RequestPtr->ResponseHeaderPtr =
        HeapAlloc (RequestPtr, Length)) == NULL)
   {
      ErrorHeapAlloc (RequestPtr, __FILE__, __LINE__);
      return (SS$_INSFMEM);
   }
   memcpy (RequestPtr->ResponseHeaderPtr, String, Length);

   if (VMSnok (status = QioNetWrite (RequestPtr, 0, String, Length)))
   {
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      return (status);
   }

   return (LIB$_NEGTIM);
}

/*****************************************************************************/
/*
Create an HTTP format Greenwich Mean Time (UTC) time string in the storage 
pointed at by 'TimeString', e.g. "Fri, 25 Aug 1995 17:32:40 GMT" (RFC 1123). 
This must be at least 30 characters capacity.  If 'BinTimePtr' is null the 
time string represents the current time.  If it points to a quadword, VMS time 
value the string represents that time.  'TimeString' must point to storage 
large enough for 31 characters.
*/

int HttpGmTimeString
(
char *TimeString,
unsigned long *BinTimePtr
)
{
   static char  *DayNames [] =
      { "", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun" };
   static char  *MonthName [] =
      { "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

   static $DESCRIPTOR (HttpTimeFaoDsc, "!AZ, !2ZW !AZ !4ZW !2ZW:!2ZW:!2ZW GMT");
   static $DESCRIPTOR (TimeStringDsc, "");

   int  status;
   unsigned long  BinTime [2],
                  GmTime [2];
   unsigned short  Length;
   unsigned short  NumTime [7];
   unsigned long  DayOfWeek;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpGmTimeString()\n");

   if (BinTimePtr == NULL)
      sys$gettim (&GmTime);
   else
   {
      GmTime[0] = BinTimePtr[0];
      GmTime[1] = BinTimePtr[1];
   }
   if (VMSnok (status = TimeAdjustGMT (true, &GmTime)))
      return (status);

   status = sys$numtim (&NumTime, &GmTime);
   if (Debug)
      fprintf (stdout, "sys$numtim() %%X%08.08X %d %d %d %d %d %d %d\n",
               status, NumTime[0], NumTime[1], NumTime[2],
               NumTime[3], NumTime[4], NumTime[5], NumTime[6]);

   if (VMSnok (status = lib$day_of_week (&GmTime, &DayOfWeek)))
      return (status);
   if (Debug)
      fprintf (stdout, "lib$day_of_week() %%X%08.08X is %d\n",
               status, DayOfWeek);

   /* set the descriptor address and size of the resultant time string */
   TimeStringDsc.dsc$w_length = 30;
   TimeStringDsc.dsc$a_pointer = TimeString;

   if (VMSnok (status =
       sys$fao (&HttpTimeFaoDsc, &Length, &TimeStringDsc,
                DayNames[DayOfWeek], NumTime[2], MonthName[NumTime[1]],
                NumTime[0], NumTime[3], NumTime[4], NumTime[5])))
   {
      if (Debug) fprintf (stdout, "sys$fao() %%X%08.08X\n", status);
      TimeString[0] = '\0';
   }
   else
      TimeString[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", TimeString);

   return (status);
}

/*****************************************************************************/
/*
Given a string such as "Fri, 25 Aug 1995 17:32:40 GMT" (RFC 1123), or "Friday,
25-Aug-1995 17:32:40 GMT" (RFC 1036), create an internal, local, binary time
with the current GMT offset.  See complementary function HttpGmTimeString().
*/ 

int HttpGmTime
(
char *TimeString,
unsigned long *BinTimePtr
)
{
   static char  *MonthName [] =
      { "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

   register char  *tptr;

   int  status;
   unsigned short  Length;
   unsigned short  NumTime [7] = { 0,0,0,0,0,0,0 };
   unsigned long  DayOfWeek;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "HttpGmTime() |%s|\n", TimeString);

   tptr = TimeString;
   /* hunt straight for the comma after the weekday name! */
   while (*tptr && *tptr != ',') tptr++;
   if (*tptr) tptr++;
   /* span white space between weekday name and date */
   while (*tptr && isspace(*tptr)) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the date and then skip to month name */
   if (isdigit(*tptr)) NumTime[2] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   while (*tptr && (*tptr == '-' || isspace(*tptr))) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the month number from the name and skip to the year */
   for (NumTime[1] = 1; NumTime[1] <= 12; NumTime[1]++)
      if (strsame (tptr, MonthName[NumTime[1]], 3)) break;
   if (NumTime[1] > 12) return (STS$K_ERROR);
   while (*tptr && isalpha(*tptr)) tptr++;
   while (*tptr && (*tptr == '-' || isspace(*tptr))) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the year and then skip to the hour */
   if (isdigit(*tptr))
   {
      NumTime[0] = atoi (tptr);
      if (NumTime[0] < 100) NumTime[0] += 1900;
   }
   while (*tptr && isdigit(*tptr)) tptr++;
   while (*tptr && isspace(*tptr)) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!*tptr) return (STS$K_ERROR);

   /* get the hour, minute and second */
   if (isdigit(*tptr)) NumTime[3] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (isdigit(*tptr)) NumTime[4] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (isdigit(*tptr)) NumTime[5] = atoi (tptr);
   while (*tptr && isdigit(*tptr)) tptr++;
   if (*tptr == ':') tptr++;
   if (!*tptr) return (STS$K_ERROR);

   /* the only thing remaining should be the "GMT" */
   while (*tptr && isspace(*tptr)) tptr++;
   if (Debug) fprintf (stdout, "tptr |%s|\n", tptr);
   if (!strsame (tptr, "GMT", 3)) return (STS$K_ERROR);

   /*******************************************/
   /* convert what looks like legitimate GMT! */
   /*******************************************/

   if (Debug)
      fprintf (stdout, "NumTime[] %d %d %d %d %d %d %d\n",
               NumTime[0], NumTime[1], NumTime[2], NumTime[3], 
               NumTime[4], NumTime[5], NumTime[6]); 
   status = lib$cvt_vectim (&NumTime, BinTimePtr);
   if (VMSnok (status)) return (status);
   if (Debug) fprintf (stdout, "lib$cvt_vectim() %%X%08.08X\n", status);

   return (TimeAdjustGMT (false, BinTimePtr));
}

/*****************************************************************************/
/*
Translate the logical HTTPD$GMT (defined to be something like "+10:30" or "-
01:15") and convert it into a delta time structure and store in 
'TimeGmtDeltaBinary'.  Store whether it is in advance or behind GMT in boolean 
'TimeAheadOfGmt'.  Store the logical string in 'TimeGmtString'.
*/

int TimeSetGMT ()

{
   register char  *cptr, *sptr;

   int  status;
   unsigned short  Length;
   $DESCRIPTOR (GmtLogicalNameDsc, "HTTPD$GMT");
   $DESCRIPTOR (LnmFileDevDsc, "LNM$FILE_DEV");
   $DESCRIPTOR (TimeGmtVmsStringDsc, "");
   struct {
      short int  buf_len;
      short int  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   } LnmItems [] =
   {
      { sizeof(TimeGmtString)-1, LNM$_STRING, TimeGmtString, &Length },
      { 0,0,0,0 }
   };

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeSetGMT()\n");

   status = sys$trnlnm (0, &LnmFileDevDsc, &GmtLogicalNameDsc, 0, &LnmItems);
   if (Debug) fprintf (stdout, "sys$trnlnm() %%X%08.08X\n", status);
   if (VMSnok (status)) return (status);

   TimeGmtString[Length] = '\0';
   if (Debug) fprintf (stdout, "TimeGmtString |%s|\n", TimeGmtString);
   sptr = TimeGmtVmsString;
   if (*(cptr = TimeGmtString) == '-')
      TimeAheadOfGmt = false;
   else
      TimeAheadOfGmt = true;
   *sptr++ = *cptr++;
   *sptr++ = '0';
   *sptr++ = ' ';
   while (*cptr && *cptr != ' ') *sptr++ = *cptr++;
   *sptr++ = '\0';
   if (Debug) fprintf (stdout, "TimeGmtVmsString |%s|\n", TimeGmtVmsString);

   /* set the descriptor address and size of the offset time string */
   if (TimeGmtVmsString[0] == '+' || TimeGmtVmsString[0] == '-')
   {
      TimeGmtVmsStringDsc.dsc$w_length = Length-1;
      TimeGmtVmsStringDsc.dsc$a_pointer = TimeGmtVmsString+1;
   }
   else
   {
      TimeGmtVmsStringDsc.dsc$w_length = Length;
      TimeGmtVmsStringDsc.dsc$a_pointer = TimeGmtVmsString;
   }

   return (sys$bintim (&TimeGmtVmsStringDsc, &TimeGmtDeltaBinary));
}

/*****************************************************************************/
/*
The GMT is generated by calculating using an offset using 
'TimeGmtDeltaOffset' and boolean 'TimeAheadOfGmt'.  Adjust either to or from 
GMT.
*/ 

int TimeAdjustGMT
(
boolean ToGmTime,
unsigned long *BinTimePtr
)
{
   int  status;
   unsigned long  AdjustedTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeAdjustGMT() ToGmTime: %d\n", ToGmTime);

   if ((ToGmTime && TimeAheadOfGmt) || (!ToGmTime && !TimeAheadOfGmt))
   {
      /* to GMT from local and ahead of GMT, or to local from GMT and behind */
      status = lib$sub_times (BinTimePtr, &TimeGmtDeltaBinary, &AdjustedTime);
      if (Debug) fprintf (stdout, "lib$sub_times() %%X%08.08X\n", status);
   }
   else
   {
      /* to GMT from local and behind GMT, or to local from GMT and ahead */
      status = lib$add_times (BinTimePtr, &TimeGmtDeltaBinary, &AdjustedTime);
      if (Debug) fprintf (stdout, "lib$add_times() %%X%08.08X\n", status);
   }

   if (Debug)
   {
      unsigned short  Length;
      char  String [64];
      $DESCRIPTOR (AdjustedTimeFaoDsc, "AdjustedTime: |!%D|\n");
      $DESCRIPTOR (StringDsc, String);

      sys$fao (&AdjustedTimeFaoDsc, &Length, &StringDsc, &AdjustedTime);
      String[Length] = '\0';
      fputs (String, stdout);
   }

   BinTimePtr[0] = AdjustedTime[0];
   BinTimePtr[1] = AdjustedTime[1];

   return (status);
}

/*****************************************************************************/
/*
There are enough exit points from the function that actually does the logical
name processing that this function wraps it with the appropriate privilege
changes to make sure we don't miss any!
*/ 

DefineMonitorLogicals (struct RequestStruct *RequestPtr)

{
   static unsigned long  SysNamMask [2] = { PRV$M_SYSNAM, 0 };

   int  status,
        SetPrvStatus;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DefineMonitorLogicals()\n");

   /* turn on SYSNAM to allow access to system table */
   if (VMSnok (status = sys$setprv (1, &SysNamMask, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", status);
      return (status);
   }

   status = DefineMonitorLogicalsHere (RequestPtr);

   /* turn off SYSNAM */
   if (VMSnok (SetPrvStatus = sys$setprv (0, &SysNamMask, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$setprv() %%X%08.08X\n", SetPrvStatus);
      /* do NOT keep processing if there is a problem turning off SYSNAM! */
      ErrorExitVmsStatus (SetPrvStatus, "disabling SYSNAM", __FILE__, __LINE__);
   }

   return (status);
}

/*****************************************************************************/
/*
Define two/three logicals used by the HTTPd monitor utility.  The value of the 
count logical is the binary contents of the server counter data structure.  
The value of the request logical is some information on the latest request to 
complete.  The PID logical is defined only once, the first time this function 
is called.
*/ 

DefineMonitorLogicalsHere (struct RequestStruct *RequestPtr)

{
   static unsigned long  JpiPid;
   static char  CountLogicalName [32],
                RequestLogicalName [32],
                RequestLogicalValue [256];

   static $DESCRIPTOR (CountLogicalNameDsc, CountLogicalName);
   static $DESCRIPTOR (LnmSystemDsc, "LNM$SYSTEM");
   static $DESCRIPTOR (RequestLogicalNameDsc, RequestLogicalName);
   static $DESCRIPTOR (RequestLogicalValueDsc, RequestLogicalValue);
   /* day time HTTP-status client-host method path?query */
   static $DESCRIPTOR (RequestLogicalValueFaoDsc,
          "!2ZL !2ZL:!2ZL:!2ZL!AD!AZ!AD!UL!AD!UL!AD!UL!AD!AZ !AZ!AZ!AZ!AZ");

   static struct VmsItem LnmCountItem [] =
          {
             { sizeof(Accounting), LNM$_STRING, &Accounting, 0 },
             { 0,0,0,0 }
          };
   static struct VmsItem LnmRequestItem [] =
          {
             { 0, LNM$_STRING, RequestLogicalValue, 0 },
             { 0,0,0,0 }
          };

   register unsigned long  *vecptr;

   int  status;
   unsigned short  Length;
   unsigned long  FaoVector [64];
   char *PathInfoPtr,
        *QueryStringPtr,
        *QuestionMarkPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DefineMonitorLogicalsHere()\n");

   if (!JpiPid)
   {
      /**********************/
      /* first time through */
      /**********************/

      /* local storage */
      char  PidLogicalName [16];
      $DESCRIPTOR (PidLogicalNameDsc, PidLogicalName);
      $DESCRIPTOR (PidLogicalNameFaoDsc, "HTTPD!UL$PID");
      $DESCRIPTOR (CountLogicalNameFaoDsc, "HTTPD!UL$COUNT");
      $DESCRIPTOR (RequestLogicalNameFaoDsc, "HTTPD!UL$REQUEST");
      struct VmsItem  JpiPidItem [] =
      {
         { sizeof(JpiPid), JPI$_PID, &JpiPid, 0 },
         {0,0,0,0}
      };
      struct VmsItem  LnmPidItem [] =
      {
         { sizeof(JpiPid), LNM$_STRING, &JpiPid, 0 },
         { 0,0,0,0 }
      };

      /* define a logical name containing the PID of the server process */
      if (VMSnok (status = sys$getjpiw (0, 0, 0, &JpiPidItem, 0, 0, 0)))
         return (status);
      sys$fao (&PidLogicalNameFaoDsc, &Length, &PidLogicalNameDsc, ServerPort);
      PidLogicalName[PidLogicalNameDsc.dsc$w_length = Length] = '\0';
      if (Debug) fprintf (stdout, "PidLogicalName |%s|\n", PidLogicalName);
      if (VMSnok (status =
          sys$crelnm (0, &LnmSystemDsc, &PidLogicalNameDsc, 0, &LnmPidItem)))
         return (status);

      /* generate approriate names for the count and request logicals */
      sys$fao (&CountLogicalNameFaoDsc, &Length, &CountLogicalNameDsc,
               ServerPort);
      CountLogicalName[CountLogicalNameDsc.dsc$w_length = Length] = '\0';
      if (Debug) fprintf (stdout, "CountLogicalName |%s|\n", CountLogicalName);
      sys$fao (&RequestLogicalNameFaoDsc, &Length, &RequestLogicalNameDsc,
               ServerPort);
      RequestLogicalName[RequestLogicalNameDsc.dsc$w_length = Length] = '\0';
      if (Debug)
         fprintf (stdout, "RequestLogicalName |%s|\n", RequestLogicalName);

      /* attempt to read the values of any previous count logical name */
      if (VMSok (status =
         sys$trnlnm (0, &LnmSystemDsc, &CountLogicalNameDsc, 0, &LnmCountItem)))
         Accounting.StartupCount++;
      else
      {
         if (status != SS$_NOLOGNAM)
            return (status);
      }
   }

   /*****************************************/
   /* (re)define count and request logicals */
   /*****************************************/

   if (RequestPtr == NULL)
   {
      /* reset the request logical to all empty strings */
      RequestLogicalValue[0] = '\0';
      LnmRequestItem[0].buf_len = 1;
   }
   else
   {
      if (RequestPtr->PathInfoPtr == NULL)
         PathInfoPtr = "";
      else
         PathInfoPtr = RequestPtr->PathInfoPtr;

      if (RequestPtr->QueryStringPtr == NULL)
      {
         QueryStringPtr = "";
         QuestionMarkPtr = "";
      }
      else
      {
         if ((QueryStringPtr = RequestPtr->QueryStringPtr)[0])
            QuestionMarkPtr = "?";
         else
            QuestionMarkPtr = "";
      }

      vecptr = FaoVector;
      *vecptr++ = RequestPtr->NumericTime[2];
      *vecptr++ = RequestPtr->NumericTime[3];
      *vecptr++ = RequestPtr->NumericTime[4];
      *vecptr++ = RequestPtr->NumericTime[5];
      *vecptr++ = 1;
      *vecptr++ = (unsigned long)"\0";
      *vecptr++ = RequestPtr->ClientHostName;
      *vecptr++ = 1;
      *vecptr++ = (unsigned long)"\0";
      *vecptr++ = RequestPtr->ResponseStatusCode;
      *vecptr++ = 1;
      *vecptr++ = (unsigned long)"\0";
      *vecptr++ = RequestPtr->BytesRx;
      *vecptr++ = 1;
      *vecptr++ = (unsigned long)"\0";
      *vecptr++ = RequestPtr->BytesTx;
      *vecptr++ = 1;
      *vecptr++ = (unsigned long)"\0";
      *vecptr++ = (unsigned long)RequestPtr->Method;
      *vecptr++ = (unsigned long)RequestPtr->ScriptName;
      *vecptr++ = (unsigned long)PathInfoPtr;
      *vecptr++ = (unsigned long)QuestionMarkPtr;
      *vecptr++ = (unsigned long)QueryStringPtr;

      status = sys$faol (&RequestLogicalValueFaoDsc, &Length,
                         &RequestLogicalValueDsc, &FaoVector);

      if (Debug) fprintf (stdout, "sys$faol() %%X%08.08X\n", status);
      RequestLogicalValue[LnmRequestItem[0].buf_len = Length] = '\0';
      if (Debug)
         fprintf (stdout, "RequestLogicalValue |%s|\n", RequestLogicalValue);
   }

   if (VMSnok (status =
       sys$crelnm (0, &LnmSystemDsc, &CountLogicalNameDsc, 0, &LnmCountItem)))
      return (status);

   if (VMSnok (status =
       sys$crelnm (0, &LnmSystemDsc, &RequestLogicalNameDsc, 0,
                   &LnmRequestItem)))
      return (status);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Convert a VMS quadword, binary time to a Unix-style time component structure.
*/

boolean TimeVmsToUnix
(
unsigned long *BinTimePtr,
struct tm *TmPtr
)
{
   static unsigned long  LibDayOfWeek = LIB$K_DAY_OF_WEEK;
   static unsigned long  LibDayOfYear = LIB$K_DAY_OF_YEAR;

   int  status;
   unsigned long  DayOfWeek,
                  DayOfYear;
   unsigned short  NumTime [7];
   unsigned long  BinTime [2];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TimeVmsToUnix()\n");

   if (BinTimePtr == NULL) sys$gettim (BinTimePtr = &BinTime);

   if (VMSnok (status = sys$numtim (&NumTime, BinTimePtr)))
      return (status);

   lib$cvt_from_internal_time (&LibDayOfWeek, &DayOfWeek, BinTimePtr);
   lib$cvt_from_internal_time (&LibDayOfYear, &DayOfYear, BinTimePtr);

   TmPtr->tm_sec = NumTime[5];
   TmPtr->tm_min = NumTime[4];
   TmPtr->tm_hour = NumTime[3];
   TmPtr->tm_mday = NumTime[2];
   TmPtr->tm_mon = NumTime[1] - 1;
   TmPtr->tm_year = NumTime[0] - 1900;
   TmPtr->tm_wday = DayOfWeek;
   TmPtr->tm_yday = DayOfYear;
   TmPtr->tm_isdst = 0;

   if (Debug) fprintf (stdout, "%s", asctime(TmPtr));

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
Return a report on the HTTPd server's performance.  A kludge to more easily 
support the HEAD method: do most of the work anyway, just use an FAO directive 
that only outputs the HTTP header!
*/ 

ServerReport (struct RequestStruct *RequestPtr)

{
/*
   VAX VMS does not seem to like "!@UQ" and must be made "!@UJ"!!!
   The VAX version not supporting this doesn't seem to be documented,
   but at run-time returns a %X00000014 status (%SYSTEM-F-BADPARAM).
   Implication: VAX version can only report a maximum of 2^32-1 bytes, or
   4,294,967,295 before overflow, AXP 2^64-1 (calculator gave an ERROR :^)
*/

   static $DESCRIPTOR (ServerReportHeadFaoDsc,
"!AZ 200 Report follows.\r\n\
Server: !AZ\r\n\
Date: !AZ\r\n\
Content-Type: text/html\r\n\
\r\n");

   static $DESCRIPTOR (ServerReportFaoDsc,
"!AZ 200 Report follows.\r\n\
Server: !AZ\r\n\
Date: !AZ\r\n\
Content-Type: text/html\r\n\
\r\n\
<TITLE>HTTPd !AZ:!UL</TITLE>\n\
<H1>HTTPd !AZ:!UL</H1>\n\
<I>Generated: !AZ, !20%D</I>\n\
<HR>\n\
<DL>\n\
\
<DT><B>Version</B>\n\
\
<DD><TT>!AZ</TT>\n\
\
<DT><B>Server</B>\n\
\
<DD>\
Process: <TT>!AZ</TT>\
 PID: <TT>!8XL</TT>\
 User: <TT>!AZ</TT>\n\
<BR>Startup: <TT>!UL</TT> Exit Status: <TT>!AZ</TT> (Zeroed: <TT>!UL</TT>)\n\
<BR>Up-time: <TT>!AZ</TT>\
 CPU-time: <TT>!UL !2ZL:!2ZL:!2ZL.!2ZL</TT>\n\
<BR>Buf.IO: <TT>!UL</TT>\
 Dir.IO: <TT>!UL</TT>\
 Pg.Faults: <TT>!UL</TT>\n\
 Pg.Used: <TT>!UL%</TT>\
<BR>WsSize: <TT>!UL</TT> (<TT>!ULKB</TT>)\
 WsPeak: <TT>!UL</TT> (<TT>!ULKB</TT>)\n\
 PeakVirt: <TT>!UL</TT> (<TT>!ULKB</TT>)\
<BR>Files: <TT>!UL</TT>/<TT>!UL</TT>\
 Subprocesses: <TT>!UL</TT>/<TT>!UL</TT>\n\
\
<DT><B>Connect</B>\n\
\
<DD>\
Total: <TT>!UL</TT>\
 Accepted: <TT>!UL</TT>\
 Rejected: <TT>!UL</TT>\
 Too Busy: <TT>!UL</TT>\
 Current: <TT>!UL</TT>\
 Peak: <TT>!UL</TT>\n\
\
<DT><B>Request</B>\n\
\
<DD>\
Error: <TT>!UL</TT>\
 Parsed: <TT>!UL</TT>\
 <I>(<TT>!UL</TT> redirect)</I>\
 Forbidden-By-Rule: <TT>!UL</TT>\
 RMS Error: <TT>!UL</TT>\n\
<BR>GET: <TT>!UL</TT>\
 HEAD: <TT>!UL</TT>\
 POST: <TT>!UL</TT>\n\
<BR>File: <TT>!UL</TT> <I>(<TT>!UL</TT> not modified)</I>\
 Menu: <TT>!UL</TT> <I>(<TT>!UL</TT> not modified)</I>\
 Directory: <TT>!UL</TT>\
 sHTML: <TT>!UL</TT>\
 IsMap: <TT>!UL</TT>\
 Internal: <TT>!UL</TT>\n\
\
<DT><B>DCL</B>\n\
\
<DD>\
Script: <TT>!UL</TT> <I>(<TT>!UL</TT> auto-script)</I>\
 Subprocess: <TT>!UL</TT> <I>(<TT>!UL</TT> multiple)</I>\n\
\
<DT><B>Response</B>\n\
\
<DD>\n\
1xx: <TT>!UL</TT>\
 2xx: <TT>!UL</TT>\
 3xx: <TT>!UL</TT>\
 4xx: <TT>!UL</TT>\
 5xx: <TT>!UL</TT>\
 <I>(<TT>!UL</TT> errors)</I>\
 Redirect: <TT>!UL</TT>\n\
\
<DT><B>Bytes</B>\n\
\
<DD>\
Request: <TT>!@UJ</TT>\
 Response: <TT>!@UJ</TT>\
 <I>(excluding this request)</I>\n\
\
<DT><B>Failure</B>\n\
\
<DD>\
Connect <I>calloc()</I>: <TT>!UL</TT>\
 Heap <I>calloc()</I>: <TT>!UL</TT>\
 Heap <I>realloc()</I>: <TT>!UL</TT>\n\
</DL>\n\
<P><HR>\n");

   static char  *DayName [] =
      { "", "Monday", "Tuesday", "Wednesday", "Thursday",
            "Friday", "Saturday", "Sunday" };

   static char  JpiPrcNam [16],
                JpiUserName [13],
                UpTime [32];
   static unsigned long  JpiBufIO,
                         JpiCpuTime,
                         JpiDirIO,
                         JpiFilCnt,
                         JpiFilLm,
                         JpiPageFlts,
                         JpiPagFilCnt,
                         JpiPgFlQuota,
                         JpiPid,
                         JpiPrcCnt,
                         JpiPrcLm,
                         JpiVirtPeak,
                         JpiWsSize,
                         JpiWsPeak,
                         Pid;
   static unsigned long  ConnectTime [2],
                         CurrentTime [2],
                         JpiLoginTime [2];
   static char  LastExitStatus [16] = "n/a";
   static $DESCRIPTOR (LastExitStatusDsc, LastExitStatus);
   static $DESCRIPTOR (LastExitStatusFaoDsc, "%X!XL");
   static $DESCRIPTOR (UpTimeFaoDsc, "!%D");
   static $DESCRIPTOR (UpTimeDsc, UpTime);
   static struct
   {
      unsigned short  buf_len;
      unsigned short  item;
      unsigned char   *buf_addr;
      unsigned short  *ret_len;
   }
      JpiItems [] =
   {
      { sizeof(JpiCpuTime), JPI$_CPUTIM, &JpiCpuTime, 0 },
      { sizeof(JpiBufIO), JPI$_BUFIO, &JpiBufIO, 0 },
      { sizeof(JpiDirIO), JPI$_DIRIO, &JpiDirIO, 0 },
      { sizeof(JpiFilCnt), JPI$_FILCNT, &JpiFilCnt, 0 },
      { sizeof(JpiFilLm), JPI$_FILLM, &JpiFilLm, 0 },
      { sizeof(JpiLoginTime), JPI$_LOGINTIM, &JpiLoginTime, 0 },
      { sizeof(JpiPageFlts), JPI$_PAGEFLTS, &JpiPageFlts, 0 },
      { sizeof(JpiPagFilCnt), JPI$_PAGFILCNT, &JpiPagFilCnt, 0 },
      { sizeof(JpiPgFlQuota), JPI$_PGFLQUOTA, &JpiPgFlQuota, 0 },
      { sizeof(JpiPid), JPI$_PID, &JpiPid, 0 },
      { sizeof(JpiPrcCnt), JPI$_PRCCNT, &JpiPrcCnt, 0 },
      { sizeof(JpiPrcLm), JPI$_PRCLM, &JpiPrcLm, 0 },
      { sizeof(JpiPrcNam), JPI$_PRCNAM, &JpiPrcNam, 0 },
      { sizeof(JpiUserName), JPI$_USERNAME, &JpiUserName, 0 },
      { sizeof(JpiVirtPeak), JPI$_VIRTPEAK, &JpiVirtPeak, 0 },
      { sizeof(JpiWsSize), JPI$_WSSIZE, &JpiWsSize, 0 },
      { sizeof(JpiWsPeak), JPI$_WSPEAK, &JpiWsPeak, 0 },
      {0,0,0,0}
   };

   register unsigned long  *vecptr;
   register char  *cptr;

   int  status;
   unsigned short  Length;
   unsigned long  DayOfWeek;
   unsigned long  BinTime [2],
                  FaoVector [128];
   $DESCRIPTOR (OutputBufferDsc, "");

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ServerReport()\n");

   sys$gettim (&BinTime);
   if (VMSnok (status = lib$day_of_week (&BinTime, &DayOfWeek)))
      return (status);
   if (Debug)
      fprintf (stdout, "lib$day_of_week() %%X%08.08X is %d\n",
               status, DayOfWeek);

   if (VMSnok (status = sys$getjpiw (0, &Pid, 0, &JpiItems, 0, 0, 0)))
   {
      RequestPtr->ErrorTextPtr = "sys$getjpi()";
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return;
   }

   JpiUserName[12] = '\0';
   for (cptr = JpiUserName; *cptr && *cptr != ' '; cptr++);
   *cptr = '\0';
   if (Debug) fprintf (stdout, "JpiUserName |%s|\n", JpiUserName);

   JpiPrcNam[15] = '\0';
   for (cptr = JpiPrcNam; *cptr && *cptr != ' '; cptr++);
   *cptr = '\0';
   if (Debug) fprintf (stdout, "JpiPrcNam |%s|\n", JpiPrcNam);

   sys$gettim (&CurrentTime);
   lib$sub_times (&CurrentTime, &JpiLoginTime, &ConnectTime);
   sys$fao (&UpTimeFaoDsc, &Length, &UpTimeDsc, &ConnectTime);
   UpTime[Length] = '\0';

   if (Accounting.StartupCount > 1)
   {
      sys$fao (&LastExitStatusFaoDsc, &Length, &LastExitStatusDsc, 
               Accounting.LastExitStatus);
      LastExitStatus[Length] = '\0';
   }

   vecptr = FaoVector;
   *vecptr++ = (unsigned long)HttpProtocol;
   *vecptr++ = (unsigned long)SoftwareID;
   *vecptr++ = (unsigned long)RequestPtr->GmDateTime;
   *vecptr++ = (unsigned long)ServerHostName;
   *vecptr++ = ServerPort;
   *vecptr++ = (unsigned long)ServerHostName;
   *vecptr++ = ServerPort;
   *vecptr++ = (unsigned long)DayName[DayOfWeek];
   *vecptr++ = &BinTime;
   *vecptr++ = (unsigned long)SoftwareID;
   *vecptr++ = (unsigned long)JpiPrcNam;
   *vecptr++ = JpiPid;
   *vecptr++ = (unsigned long)JpiUserName;
   *vecptr++ = Accounting.StartupCount;
   *vecptr++ = (unsigned long)LastExitStatus;
   *vecptr++ = Accounting.ZeroedCount;
   for (cptr = UpTime; isspace(*cptr); cptr++);
   *vecptr++ = (unsigned long)cptr;
   *vecptr++ = JpiCpuTime / 8640000;             /* CPU day */
   *vecptr++ = (JpiCpuTime % 8640000) / 360000;  /* CPU hour */
   *vecptr++ = (JpiCpuTime % 360000) / 6000;     /* CPU minute */
   *vecptr++ = (JpiCpuTime % 6000 ) / 100;       /* CPU second */
   *vecptr++ = JpiCpuTime % 100;                 /* CPU 10 milliseconds */
   *vecptr++ = JpiBufIO;
   *vecptr++ = JpiDirIO;
   *vecptr++ = JpiPageFlts;
   *vecptr++ = 100 - ((JpiPagFilCnt * 100) / JpiPgFlQuota);
   *vecptr++ = JpiWsSize;
   *vecptr++ = JpiWsSize / 2;
   *vecptr++ = JpiWsPeak;
   *vecptr++ = JpiWsPeak / 2;
   *vecptr++ = JpiVirtPeak;
   *vecptr++ = JpiVirtPeak / 2;
   *vecptr++ = JpiFilLm - JpiFilCnt;
   *vecptr++ = JpiFilLm;
   *vecptr++ = JpiPrcCnt;
   *vecptr++ = JpiPrcLm;
   *vecptr++ = Accounting.ConnectCount;
   *vecptr++ = Accounting.ConnectAcceptedCount;
   *vecptr++ = Accounting.ConnectRejectedCount;
   *vecptr++ = Accounting.ConnectTooBusyCount;
   *vecptr++ = Accounting.ConnectCurrent;
   *vecptr++ = Accounting.ConnectPeak;
   *vecptr++ = Accounting.RequestErrorCount;
   *vecptr++ = Accounting.RequestParseCount;
   *vecptr++ = Accounting.RedirectLocalCount;
   *vecptr++ = Accounting.RequestForbiddenCount;
   *vecptr++ = Accounting.RequestRmsErrorCount;
   *vecptr++ = Accounting.MethodGetCount;
   *vecptr++ = Accounting.MethodHeadCount;
   *vecptr++ = Accounting.MethodPostCount;
   *vecptr++ = Accounting.DoFileCount;
   *vecptr++ = Accounting.DoFileNotModifiedCount;
   *vecptr++ = Accounting.DoMenuCount;
   *vecptr++ = Accounting.DoMenuNotModifiedCount;
   *vecptr++ = Accounting.DoDirectoryCount;
   *vecptr++ = Accounting.DoShtmlCount;
   *vecptr++ = Accounting.DoIsMapCount;
   *vecptr++ = Accounting.DoInternalCount;
   *vecptr++ = Accounting.DoScriptCount;
   *vecptr++ = Accounting.DoAutoScriptCount;
   *vecptr++ = Accounting.DclExecutedCount;
   *vecptr++ = Accounting.DclExecutedMoreThanOnceCount;
   *vecptr++ = Accounting.ResponseStatusCodeCountArray[1];
   /* a little fudge!  allow for this report's reponse */
   *vecptr++ = Accounting.ResponseStatusCodeCountArray[2] + 1;
   *vecptr++ = Accounting.ResponseStatusCodeCountArray[3];
   *vecptr++ = Accounting.ResponseStatusCodeCountArray[4];
   *vecptr++ = Accounting.ResponseStatusCodeCountArray[5];
   *vecptr++ = Accounting.ResponseStatusCodeCountArray[0];
   *vecptr++ = Accounting.RedirectRemoteCount;
   *vecptr++ = (unsigned long)&Accounting.QuadBytesRx[0];
   *vecptr++ = (unsigned long)&Accounting.QuadBytesTx[0];
   *vecptr++ = Accounting.ConnectCallocFailCount;
   *vecptr++ = Accounting.HeapCallocFailCount;
   *vecptr++ = Accounting.HeapReallocFailCount;

   /* initialize the output buffer space */
   if (RequestPtr->OutputBufferPtr == NULL)
   {
      if (VMSnok (BufferOutput (RequestPtr, 0, NULL, 0)))
      {
         ConcludeProcessing (RequestPtr);
         return;
      }
   }
   OutputBufferDsc.dsc$a_pointer = RequestPtr->OutputBufferPtr;
   OutputBufferDsc.dsc$w_length = RequestPtr->OutputBufferSize;

   if (RequestPtr->MethodHead)
      status = sys$faol (&ServerReportHeadFaoDsc, &Length,
                         &OutputBufferDsc, &FaoVector);
   else
      status = sys$faol (&ServerReportFaoDsc, &Length,
                         &OutputBufferDsc, &FaoVector);

   if (VMSnok (status))
   {
      RequestPtr->ErrorTextPtr = "sys$faol()";
      ErrorVmsStatus (RequestPtr, status, __FILE__, __LINE__);
      ConcludeProcessing (RequestPtr);
      return;
   }

   QioNetWrite (RequestPtr, &ConcludeProcessing,
                RequestPtr->OutputBufferPtr, Length);

   RequestPtr->ResponseStatusCode = 200;
}

/*****************************************************************************/
/*
If not compiling with DEC C then we need to create our own strftime().  Make 
it a minimal implementation, no attention paid to the format string, etc.
*/ 

#ifndef __DECC

int strftime
(
char *TimeStringPtr,
int SizeOfTimeString,
char *TimeFormatPtr,
struct tm *TmPtr
)
{
   static char  *DayName [] =
   { "Mon","Tue","Wed","Thu","Fri","Sat","Sun" };
   static char  *MonName [] =
   { "Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec" };

   static $DESCRIPTOR (TimeFaoDsc, "!AZ, !2ZL !AZ !2ZL, !2ZL:!2ZL:!2ZL");

   unsigned short  Length;
   $DESCRIPTOR (TimeStringPtrDsc, TimeStringPtr);

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "strftime() well ... sort of\n");

   TimeStringPtrDsc.dsc$a_pointer = TimeStringPtr;
   TimeStringPtrDsc.dsc$w_length = SizeOfTimeString - 1;

   Length = 0;
   sys$fao (&TimeFaoDsc, &Length, &TimeStringPtrDsc,
            DayName[TmPtr->tm_wday], TmPtr->tm_mday, MonName[TmPtr->tm_mon],
            TmPtr->tm_year, TmPtr->tm_hour, TmPtr->tm_min, TmPtr->tm_sec);
   TimeStringPtr[Length] = '\0';
   if (Debug) fprintf (stdout, "|%s|\n", TimeStringPtr);
   return (Length);
}

#endif

/*****************************************************************************/
/*
Copy text from one string to another, converting characters forbidden to 
appear as plain-text in HTML.  For example the '<', '&', etc.  Convert these 
to the corresponding HTML character entities.  Returns the number of characters
added to the buffer.
*/ 

int CopyTextIntoHtml
( 
register char *bptr,
register char *tptr,
register int  ccnt
)
{
    char *Buffer;

    Buffer = bptr;
    while (ccnt-- && *tptr)
    {
       switch (*tptr)
       {
          case '<' : strcpy (bptr, "&lt;"); bptr += 4; tptr++; break;
          case '>' : strcpy (bptr, "&gt;"); bptr += 4; tptr++; break;
          case '&' : strcpy (bptr, "&amp;"); bptr += 5; tptr++; break;
          case '\"' : strcpy (bptr, "&quot;"); bptr += 6; tptr++; break;
          default : if (isprint(*tptr)) *bptr++ = *tptr++; else tptr++;
       }
    }
    *bptr = '\0';
    return (bptr-Buffer);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 
 
boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}
 
/****************************************************************************/

