/*****************************************************************************/
/*
                                 HTTPd302.c

Listens at the well-known HTTP port (80) for incoming requests.  When received 
it reads sufficient of the request to get the URL, using it to construct a 302 
(redirection) response which it immediately returns to the client.  This 
causes the client to re-request from the specified host.  The client will 
cause all following requests to be sent to that host.

Redirects all HTTP requests to a host (or a list of hosts) specified on the 
command line, or as an environment variable.  This allows all hosts on which 
it runs to act as redirectors to one or more central HTTP servers, without the 
client (browser) needing to know precisely which host should be contacted, or 
which can respond.  This also provides a sort of "failover", where one (or 
more) of multiple servers can "go down" without permanently affecting overall 
service. 

A comma-separated list of HTTP server host names can be supplied either on the 
command line, as argument one, or as an environment variable.  The full host 
name is retrieved from the network database, so local systems can be specified 
by a non-fully qualified host name, e.g. "datlv,dajav,alpha,beta".

For VMS systems the enviroment variable can be a process-level symbol or 
logical name (DEC C behaviour of getenv()), or usefully, a system-level 
logical.  As this is read at each request received, the list of servers could 
be changed dynamically by redefining the logical name, e.g.

   DEFINE /SYSTEM HTTPD302$SERVERS "node1,node2,node3"

From first to last host, the utility attempts to connect to the HTTP port.  
The first successful has its host name returned to client as the server in a 
302 (redirection) response.  If none responds then an HTTP error status and 
message is returned. 

Initially developed on a Linux-based (Unix) PC.
Designed to be usable on AXP and VAX VMS systems.


VERSION HISTORY
---------------
10-JUN-95  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __VMS
#  ifdef __ALPHA
      char SoftwareID [] = "HTTPD302 AXP-1.0.0";
#  else
      char SoftwareID [] = "HTTPD302 VAX-1.0.0";
#  endif
#else
   char SoftwareID [] = "HTTPD302 v1.0.0";
#endif

#include <stdio.h>
#include <ctype.h>
#include <errno.h>

#ifdef __VMS
#  include <in.h>
#  include <iodef.h>
#  include <netdb.h>
#  include <socket.h>
#  include <ssdef.h>
/*
These values have been determined from <ucx$inetdef.h> and allow the module to
be compiled on a Multinet TCP/IP system.
*/
#   define UCX$C_TCP_PROBE_IDLE 128
#   define UCX$C_TCPOPT 6
#else
#  include <sys/socket.h>
#  include <netinet/in.h>
#  include <netdb.h>
#endif

#define boolean int
#define true 1
#define false 0

#define MAX_LISTEN 5
#define REQUEST_BUFFER_SIZE 1024
#define WELL_KNOWN_HTTP_PORT 80

#ifdef __VMS
    /* vaxc$errno will be an extern int */
#   define ERROR_VALUE vaxc$errno
#else
#   define ERROR_VALUE errno
#endif

char Utility [] = "HTTPD302";

boolean  Debug;

/*****************************************************************************/
/*
*/

main
(
int argc,
char *argv[]
)
{
   static int  OptionOn = 1;

   register int  cnt;
   register char  *cptr;

   int  BufferCount,
        Count,
        Length,
        LocalSocket,
        RemoteSocket;
   /* allow sufficient extra space in the buffer to reconstruct the request */
   char  Buffer [REQUEST_BUFFER_SIZE + 256],
         LocalHostName [128],
         RemoteHostName [128],
         ServerHostName [256];
   char  *RedirectionHostNamePtr;
   struct  sockaddr_in  RemoteAddress,
                        LocalAddress;
   struct  hostent  LocalHostEntry,
                    RemoteHostEntry,
                    *HostEntryPtr;

   /*********/
   /* begin */
   /*********/

#ifdef __VMS
   if (getenv ("HTTPD302$DEBUG") != NULL)
      Debug = true;
#else
   if (getenv ("HTTPD302_DEBUG") != NULL)
      Debug = true;
#endif

   if (gethostname (LocalHostName, sizeof(LocalHostName)))
      FatalError ("GTHSTNM", "gethostname() failed", ERROR_VALUE);
   if (Debug) fprintf (stdout, "LocalHostName |%s|\n", LocalHostName);

   if ((HostEntryPtr = gethostbyname (LocalHostName)) == NULL)
      FatalError ("GTHSTBYNM", "gethostbyname() failed", ERROR_VALUE);
   LocalHostEntry = *HostEntryPtr;
   strcpy (LocalHostName, LocalHostEntry.h_name);
   if (Debug) fprintf (stdout, "LocalHostName |%s|\n", LocalHostName);

   if ((LocalSocket = socket (AF_INET, SOCK_STREAM, 0)) < 0)
      FatalError ("SOCKET", "socket() failed", ERROR_VALUE);

   if (setsockopt (LocalSocket, SOL_SOCKET, SO_REUSEADDR,
                   &OptionOn, sizeof(OptionOn)) < 0)
      FatalError ("STSCKPT", "setsockopt() failed", ERROR_VALUE);

   LocalAddress.sin_family = AF_INET;
   LocalAddress.sin_port = htons (WELL_KNOWN_HTTP_PORT);
   LocalAddress.sin_addr.s_addr = INADDR_ANY;
   LocalAddress.sin_addr = *((struct in_addr *)LocalHostEntry.h_addr);

   if (bind (LocalSocket, (struct sockaddr*)&LocalAddress,
                            sizeof(LocalAddress)) == -1)
      FatalError ("BIND", "bind() failed", ERROR_VALUE);

   if (listen (LocalSocket, MAX_LISTEN) == -1)
      FatalError ("LISTEN", "listen() failed", ERROR_VALUE);

   /*****************/
   /* infinite loop */
   /*****************/

   for (;;)
   {
      if (Debug) fprintf (stdout, "now accept() ing\n");
      if ((RemoteSocket =
           accept (LocalSocket, (struct sockaddr*)&RemoteAddress, &Length))
           == -1)
         FatalError ("ACCEPT", "accept() failed", ERROR_VALUE);

      /***********************/
      /* accepted connection */
      /***********************/

      if ((HostEntryPtr =
           gethostbyaddr ((char*)&RemoteAddress.sin_addr,
                          sizeof(RemoteAddress.sin_addr),
                          RemoteAddress.sin_family)) == NULL)
      {
         if (errno != ENOENT)
            FatalError ("GTHSTBYADDR", "gethostbyaddr() failed", ERROR_VALUE);
         RemoteHostName[0] = '\0';
      }
      else
      {
         RemoteHostEntry = *HostEntryPtr;
         strcpy (RemoteHostName, RemoteHostEntry.h_name);
      }
      if (Debug) fprintf (stdout, "RemoteHostName |%s|\n", RemoteHostName);

      if (!(BufferCount = FindHttpServer (argc, argv, Buffer, ServerHostName)))
      {
         /****************************/
         /* viable HTTP server found */
         /****************************/

         /* read data from client */
         Count = 0;
         for (;;)
         {
            if ((BufferCount =
                 recv (RemoteSocket, Buffer, REQUEST_BUFFER_SIZE, 0))
                 <= 0) break;
            Buffer[BufferCount] = '\0';
            if (Debug)
               fprintf (stdout, "Buffer %d |%s|\n", BufferCount, Buffer);
            if (BufferCount =
                ProcessHttpRequest (++Count, Buffer, ServerHostName))
               break;
         }
         if (Debug) fprintf (stdout, "errno: %d\n", errno);
      }

      /* send redirection or error (no servers available) response */
      send (RemoteSocket, Buffer, BufferCount, 0);

      shutdown (RemoteSocket, 2);
      close (RemoteSocket);
   }
}

/*****************************************************************************/
/*
A comma-separated list of HTTP server host names can be supplied either on the 
command line, as argument one, or as an environment variable.  The full host 
name is retrieved from the network database, so local systems can be specified 
by a non-fully qualified host name, e.g. "datlv,dajav,alpha,beta".  From first 
to last host attempt to connect to it on the HTTP port.  The first successful 
has its host name returned to the calling routine, and zero as the return 
value.  If none are responding then return an HTTP error status and message 
in the buffer (to be sent to the client), and the length of the message as its 
return value.
*/ 

int FindHttpServer
(
int argc,
char *argv[],
char *Buffer,
char *HostName
)
{
   register char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "FindHttpServer() |%s|\n", HostName);

   if (argc > 1)
      cptr = argv[1];
   else
   {
#ifdef __VMS
      if ((cptr = (getenv ("HTTPD302$SERVERS"))) == NULL)
         cptr = "";
#else
      if ((cptr = getenv ("HTTPD302_SERVERS"))) == NULL)
         cptr = "";
#endif
   }

   /* scan through the comma-separated list, attempting to connect to each */
   while (*cptr)
   {
      sptr = HostName;
      while (*cptr && *cptr != ',') *sptr++ = *cptr++;
      *sptr = '\0';
      if (*cptr) cptr++;
      if (!HostName[0]) continue;
      if (ProbeHttpServer (HostName)) return (0);
   }

   /* no servers currently available, generate an error message */
   return (
      sprintf (Buffer,
"HTTP/1.0 404 No HTTP server system available!\r\n\
Content-Type: text/html\r\n\
\r\n\
<!-- SoftwareID: %s -->\n\
<TITLE>ERROR!</TITLE>\n\
<H1>ERROR!</H1>\n\
<P>Reported by the <I>redirection</I> server.\n\
<H2>There are currently no server systems available!</H2>\n\
<P>Please try again later.\n",
      SoftwareID)
   );
}

/*****************************************************************************/
/*
Attempt to connect to the supplied host name at the HTTP port.  If successful 
return true, if unsuccessful return false.  Close the connection in either 
case.  This is only a "probe" to see if it responding as an HTTP server.
*/ 

boolean ProbeHttpServer (char *HostName)

{
   register char  *cptr, *sptr;

   int  HostSocket;
   struct  sockaddr_in HostAddress;
   struct  hostent  *HostEntryPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProbeHttpServer() |%s|\n", HostName);

   if ((HostEntryPtr = gethostbyname (HostName)) == NULL)
   {
      if (Debug)
         fprintf (stdout, "gethostbyname() %d %s\n", errno, strerror(errno));
      return (false);
   }

   HostAddress.sin_family = AF_INET;
   HostAddress.sin_port = htons (WELL_KNOWN_HTTP_PORT);
   HostAddress.sin_addr.s_addr = INADDR_ANY;
   HostAddress.sin_addr = *((struct in_addr *)HostEntryPtr->h_addr);

   /* ensure its all lower-case (VMS/UCX sometimes returns uppercase) */
   sptr = HostName;
   for (cptr = HostEntryPtr->h_name; *cptr; *sptr++ = tolower(*cptr++));
   if (Debug) fprintf (stdout, "|%s|\n", HostName);

   if ((HostSocket = socket (AF_INET, SOCK_STREAM, 0)) < 0)
      FatalError ("SOCKET", "socket() failed", ERROR_VALUE);

#ifdef __VMS
{
   /*
      Couldn't come up with a less VMS way of connect()ing with a short
      timeout period.  So, set a connection timeout period using sys$qio().
   */

   static struct timeval  TimeOut = { 3, 0 };

   int  status;
   unsigned short  HostSocketChannel;
   struct {
      unsigned short  Status;
      unsigned char  Unused [6];
   } IOsb;
   struct {
      unsigned short  Length;
      unsigned short  Name;
      unsigned long  Address;
   } 
      ProbeIdleDsc =
      { sizeof(TimeOut), UCX$C_TCP_PROBE_IDLE, &TimeOut },
      TimeOutDsc =
      { sizeof(ProbeIdleDsc), UCX$C_TCPOPT, &ProbeIdleDsc };

   HostSocketChannel = vaxc$get_sdc (HostSocket);
   status = sys$qiow (0, HostSocketChannel, IO$_SETMODE, &IOsb, 0, 0,
                      0, 0, 0, 0, &TimeOutDsc, 0);
   if (Debug)
      fprintf (stdout, "sys$qiow() %%x%08.08X IOsb: %%X%08.08X\n",
               status, IOsb.Status);
}
#endif /* __VMS */

   if ((connect (HostSocket, (struct sockaddr*)&HostAddress,
                 sizeof(HostAddress))) == -1)
   {
      if (Debug) fprintf (stdout, "connect() %d %s\n", errno, strerror(errno));
      close (HostSocket);
      return (false);
   }

   if (Debug) fprintf (stdout, "connect() ok\n");
   shutdown (HostSocket, 2);
   close (HostSocket);
   return (true);
}

/*****************************************************************************/
/*
This moderately complex function can be called more than once if necessary, 
building up a string in the static 'Request' buffer.  Once the HTTP end-of-
request blank line is received the buffer is scanned for the first line, and 
the request URL is extracted from that.  This is used to create an HTTP 
302 (redirection) response (in the original buffer) to return to the client.  
If the response has been generated the length of that response is returned, if 
more data is required from the client zero is returned.
*/ 

int ProcessHttpRequest
(
int Count,
char *Buffer,
char *RedirectionHostName
)
{
   static char  Request [REQUEST_BUFFER_SIZE];
   static char  *RequestPtr;

   register char  *bptr, *rptr, *sptr;

   char  Scratch [1024];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessHttpRequest() |%s|\n", Buffer);

   if (Count == 1) *(RequestPtr = Request) = '\0';
   rptr = RequestPtr;
   bptr = Buffer;
   while (*bptr && *bptr != '\r'  && *bptr != '\n') *rptr++ = *bptr++;
   if (*bptr)
   {
      if (bptr[0] == '\n' || (bptr[0] = '\r' && bptr[1] == '\n'))
      {
         *rptr = '\0';
         if (Debug) fprintf (stdout, "Request |%s|\n", Request);
         rptr = Request;
         /* scan across HTTP method (e.g. GET) */
         while (*rptr && !isspace(*rptr)) rptr++;
         /* scan across (single) white-space between method and URI */
         while (isspace(*rptr)) rptr++;
         /* copy request string */
         for (sptr = Scratch; *rptr && !isspace(*rptr); *sptr++ = *rptr++);
         *sptr = '\0';
         /* construct a "302" response in the original buffer */
         bptr = Buffer + sprintf (Buffer,
"HTTP/1.0 302 Use this host!\r\n\
Location: http://%s",
            RedirectionHostName);
         for (sptr = Scratch; *sptr; *bptr++ = *sptr++);
         *bptr++ = '\r';
         *bptr++ = '\n';
         *bptr++ = '\r';
         *bptr++ = '\n';
         *bptr = '\0';
         if (Debug) fprintf (stdout, "Buffer |%s|\n", Buffer);
         return (bptr - Buffer);
      }
   }
   return (0);
}

/*****************************************************************************/
/*
*/

FatalError
(
char *Identifier,
char *Description,
int ErrorValue
)
{
   /*********/
   /* begin */
   /*********/

#ifdef __VMS
   fprintf (stdout, "%%%s-E-%s, %s\n", Utility, Identifier, Description);
   if (ErrorValue)
      exit (ErrorValue);
   else
      exit (2 | 0x10000000);
#else
   if (ErrorValue)
      fprintf (stdout, "%s: %s: %s\n",
               Utility, Description, strerror(ErrorValue));
   else
      fprintf (stdout, "%s: %s\n", Utility, Description);
   exit (1);
#endif
}

/*****************************************************************************/

