/*****************************************************************************/
/*
                                HyperSpi.c

A CGI-compliant script.

This application generates VMS System Performance Information HTML documents.  
Coupled with 'HyperSpi$agent.c', the data-collection application, it can 
profile, complete with graphics, fundamental system performance indicators as 
CPU usage, memory usage, IO values.

It operates in two distinct modes, text and graphic.  In text mode it returns 
an HTML stream to the browser comprising a hypertext page, with selection 
menu, node performance presentation page, data listing or dump page.  In 
graphic mode it returns a GIF image to the browser, first processing the 
specified data into an in-memory bitmap graph, then sending this image to the 
client via an internal GIF processor. 

A node preformance presentation page may have one or more HTML <IMG...> tags 
within it.  Each of these tags will separately cause the browser to invoke the 
application to generate a graphic from the data specified in the SRC= URL. 

If the /NODES= qualifier specifies a comma-separated list of node names these 
are used in the menus for listing the available nodes.  If this is not 
supplied the data directory is searched for current-day data files, those 
found have the respective node name extacted and these are used to list the 
nodes. 


CGI VARIABLES
-------------
FORM_BUFFERED_IO        if non-null then buffered IO requested
FORM_CPU        if non-null then cpu usage requested
FORM_DO         what to do(!): "DUMP", "GRAPH", "LIST", "PAGE", "MENU"
FORM_DAY        the (first) day of the data file(s)
FORM_DIRECT_IO  if non-null then direct IO requested
FORM_EXTRACT    if non-null provide a link to allow graph extraction
FORM_HELP       redirect to help URL
FORM_HOUR       the (first) hour of the data
FORM_INCLUDE    additional category: PEAK, TOTAL (only with "user-mode-cpu"),
                                     HARD-FAULTS (only with "faults")
FORM_LIST_NODE  name of node to process data (see also FORM_NODE)
FORM_MINUTE     the (first) minute of the data
FORM_MONTH      the (first) month of the data file(s)
FORM_MSCP_IO    if non-null then MSCP IO requested
FORM_NODE       name of node to process data (used before FORM_LIST_NODE)
FORM_PERIOD     standard period (e.g. "until" now, "business", "today",
                "yesterday", or since a number of days)
FORM_TODAY      the last day of the data file(s)
FORM_TOMINUTE   the last minute of the data
FORM_TOHOUR     the last hour of the data
FORM_TOMONTH    the last month of the data file(s)
FORM_TOYEAR     the last year of the data file(s)
FORM_USER_MODE_CPU      if non-null then user-mode-cpu usage requested
FORM_WHAT       any data category can be placed in this comma-separated list
FORM_XMAG       number of times X axis should be enlarged
FORM_YEAR       the (first) year of the data file(s)
FORM_YMAG       number of times Y axis should be enlarged


QUALIFIERS
----------
/DBUG           turns on all "if (Debug)" statements
/DIRECTORY=     directory containing any node description HTML files
/HELP=          help HTML file URL
/ICON=          SPI icon URL
/NODES=         comma-separated list of node names collecting SPI data


LOGICAL NAMES
-------------
Required system-level logical HYPERSPI$DATA to locate the data files.


SOURCE FILES
------------
HyperSpi.h
HyperSpi.c
PlotSpi.c
GifSpi.c


BUILD DETAILS
-------------
See BUILD_HYPERSPI.COM procedure.


VERSION HISTORY (update SoftwareID as well!)
---------------
19-SEP-95  MGD  v1.0.1, replace <CR><LF> carriage-control with single newline,
                        still acceptable for HTTP, and slightly more efficient
20-JUN-95  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "HYPERSPI AXP-1.0.1";
#else
   char SoftwareID [] = "HYPERSPI VAX-1.0.1";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <unixio.h>

/* VMS related header files */
#include <descrip.h>
#include <libdef.h>
#include <libdtdef.h>
#include <rmsdef.h>
#include <rms.h>
#include <ssdef.h>
#include <stsdef.h>

/* application header file */
#include "HyperSpi.h"

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))
 
/* for use by functions in PlotSpi.c */
#define PLOT_ON 0
#define PLOT_OFF 1
#define PLOT_XOR 2

#define MaxSamplePeriodDays 31
#define MaxNodeCountBeforeSelectBox 5

char  Utility [] = "HYPERSPI";

char  Http200Header [] =
"HTTP/1.0 200 Document follows.\n\
Content-Type: text/html\n\
\n";

char  Http302Header [] =
"HTTP/1.0 302 Redirection.\n\
Location: http://%s%s\n\
\n";

char  Http404Header [] =
"HTTP/1.0 404 Error report follows.\n\
Content-Type: text/html\n\
\n";

boolean  Debug,
         DoDump,
         DoGraph,
         DoList,
         DoComprehensiveMenu,
         DoSummaryMenu,
         DoPresentSummaryPage,
         ExtractGraph,
         GifTransparentBackground = true,
         HttpHasBeenOutput,
         IncludePeak,
         IncludeTotal,
         ProvidePercentCPU,
         ProvidePercentUserModeCPU,
         ProvideBufferedIO,
         ProvidePeakBufferedIO,
         ProvideDirectIO,
         ProvidePeakDirectIO,
         ProvideMemory,
         ProvideMscpIO,
         ProvidePeakMscpIO,
         ProvidePageFaults,
         ProvidePeakPageFaults,
         ProvideHardPageFaults,
         ProvidePeakHardPageFaults,
         ProvideProcesses;

int  AvePercentCPU,
     AvePercentUserModeCPU,
     AveBufferedIO,
     AveDirectIO,
     AveMscpIO,
     AvePageFaults,
     AveHardPageFaults,
     AveNumberOfProcesses,
     AveSystemMemoryPercentInUse,
     AvePageSpacePercentInUse,
     CurrentJulianDate,
     CurrentMinuteFromStartOfPeriod,
     DataFileNameLength,
     DataFileSpecLength,
     DataFilesFoundCount,
     DataFilesProcessedCount,
     DataRecordsReadCount,
     DataRecordsProcessedCount,
     DataMinute,
     DataHour,
     DataDay,
     DataMonth,
     DataYear,
     FromJulianDate,
     FromDay,
     FromHour,
     FromMinute,
     FromMonth,
     FromYear,
     HaveDataY,
     MaxAveBufferedIO,
     MaxAveDirectIO,
     MaxAveMscpIO,
     MaxAvePageFaults,
     MaxAveHardPageFaults,
     MaxNumberOfProcesses,
     MaxPageSpacePercentInUse,
     MaxSystemMemoryPercentInUse,
     NodeCount,
     NumberOfCPUs,
     NumberOfDays,
     NumberOfDaysIntoData,
     NumberOfHours,
     NumberOfMinutesIntoData,
     NumberOfProcesses,
     PageSpaceMBytes,
     PageSpacePercentInUse,
     PeakPercentUserModeCPU,
     PeakBufferedIO,
     PeakDirectIO,
     PeakMscpIO,
     PeakPageFaults,
     PeakPercentCPU,
     PeakHardPageFaults,
     PercentCPU,
     PercentUserModeCPU,
     RecordSampleRate,
     SizeOfMarginX = 10,
     SizeOfMarginY = 20,
     SizeOfPlotX,
     SizeOfPlotY,
     StartMinuteOnFirstDay,
     StartMinuteOfData,
     SystemMemoryMBytes,
     SystemMemoryPercentInUse,
     ToJulianDate,
     ToMinute,
     ToHour,
     ToDay,
     ToMonth,
     ToYear,
     XMag,
     YMag;
     
float  ScalingFactorY;

char  CgiRequestMethod [16],
      CgiPathInfo [256],
      CgiPathTranslated [256],
      CgiScriptName [256],
      CgiServerName [256],
      DataFileName [256],
      DataFileSpec [256],
      DataNode [32],
      DateString [32],
      FileSpecification [256],
      FormDo [32],
      FormInclude [32],
      FormHelp [32],
      FormNode [32],
      FormPeriod [32],
      FormWhat [256],
      HelpUrl [256],
      HyperSpiDirectory [256],
      HyperSpiUrl [256],
      IconImg [256],
      IconUrl [256],
      ToDateString [32];

char  SpiNodeNames [16][16] = { "" };

unsigned long  CurrentBinTime [2];
unsigned short  CurrentNumTime [7];

FILE  *HttpOut;

/* this structure is declared in HyperSpi.h */
struct HyperSpiData  SpiRecord;

/* external storage (declared in PlotSpi.c) */
extern boolean  DoPlotDiagnostic;
extern int  PixelPlotOperation,
            PlotScaleY;

/* external storage (declared in GifSpi.c) */
extern char  Http200GifHeader[];

/* function prototypes for graphing (from PlotSpi.c) */
PlotAllocate (int XSize, int YSize);
PlotAxes (int Operation);
PlotPixel (int Operation, int AtX, int AtY, int XOrigin, int YOrigin);
PlotLineX (int Operation, int FromX, int ToX, int AtY,
           int XOrigin, int YOrigin);
PlotLineY (int Operation, int FromY, int ToY, int AtX,
           int XOrigin, int YOrigin);

/* required function prototypes */
DumpRecord();
ListProcessedRecord();
GraphRecordCPU();
GraphRecordMemory();
GraphRecordProcesses();
GraphRecordRange();
SelectNodeNameByDataFileName();
SummarizeRecord();
char* UniqueNumber();

/*****************************************************************************/
/*
*/

main (int argc, char *argv[])

{
   register int  acnt, idx;
   register char  *cptr, *sptr;

   int  status;
   char  String [256];

   /*********/
   /* begin */
   /*********/

   /* open another output stream so that the "\n" are not filtered */
#ifdef __DECC
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "ctx=bin")) == NULL)
      exit (vaxc$errno);
#else
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "rfm=udf")) == NULL)
      exit (vaxc$errno);
#endif

   /***********************************/
   /* get the command line parameters */
   /***********************************/

   /* using this mechanism parameters must be space-separated! */
   for (acnt = 1; acnt < argc; acnt++)
   {
      if (Debug) fprintf (stdout, "argv[%d] |%s|\n", acnt, argv[acnt]);
      if (strsame (argv[acnt], "/DBUG", -1))
      {
         Debug = true;
         continue;
      }
      if (strsame (argv[acnt], "/DIRECTORY=", 3))
      {
         for (cptr = argv[acnt]+3; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = HyperSpiDirectory;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         continue;
      }
      if (strsame (argv[acnt], "/HELP=", 3))
      {
         for (cptr = argv[acnt]+3; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = HelpUrl;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         continue;
      }
      if (strsame (argv[acnt], "/ICON=", 3))
      {
         for (cptr = argv[acnt]+3; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         sptr = IconUrl;
         while (*cptr) *sptr++ = *cptr++;
         *sptr = '\0';
         sprintf (IconImg, "<IMG ALIGN=top SRC=\"%s\" ALT=\"HyperSPI\"> ",
                  IconUrl);
         continue;
      }
      if (strsame (argv[acnt], "/NODES=", 3))
      {
         for (cptr = argv[acnt]+3; *cptr && *cptr != '='; cptr++);
         if (*cptr) cptr++;
         /* quick and nasty, no checks on array bounds (caveat emptor) */
         idx = 0;
         while (*cptr)
         {
            sptr = SpiNodeNames[idx++];
            while (*cptr && *cptr != ',') *sptr++ = toupper(*cptr++);
            *sptr = '\0';
            if (*cptr) cptr++;
         }
         SpiNodeNames[idx][0] = '\0';
         continue;
      }
      fprintf (stdout, "%%%s-E-IVQUAL, unrecognized qualifier\n \\%s\\\n",
               Utility, argv[acnt]+1);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   /*********************/
   /* get CGI variables */
   /*********************/

   GetCgiVariable ("WWW_REQUEST_METHOD", CgiRequestMethod,
                   sizeof(CgiRequestMethod));

   /* numeric equivalent of "GET\0" (just a bit more efficient, that's all!) */
   if (*(unsigned long*)CgiRequestMethod != 0x00544547) exit (SS$_NORMAL);

   GetCgiVariable ("WWW_SCRIPT_NAME", CgiScriptName, sizeof(CgiScriptName));
   GetCgiVariable ("WWW_SERVER_NAME", CgiServerName, sizeof(CgiServerName));

   /***********/
   /* process */
   /***********/

   sys$gettim (&CurrentBinTime);
   sys$numtim (&CurrentNumTime, &CurrentBinTime);

   GetCgiVariable ("WWW_FORM_HELP", FormHelp, sizeof(FormHelp));
   if (FormHelp[0])
   {
      GetCgiVariable ("WWW_SERVER_NAME", CgiServerName, sizeof(CgiServerName));
      fprintf (HttpOut, Http302Header, CgiServerName, HelpUrl);
      exit (SS$_NORMAL);
   }

   GetRequest ();
   ValidateRequest ();

   if (DoSummaryMenu || DoComprehensiveMenu)
   {
      SummaryMenu ();
      exit (SS$_NORMAL);
   }

   /* create file specification for processing data */
   DataFileSpecLength =
      sprintf (DataFileSpec,
               "%sHYPERSPI_%s_%s_%%%%%%%%%%%%.DAT",
               HyperSpiDataDirectory, HyperSpiDataVersion, FormNode);
   if (Debug) fprintf (stdout, "DataFileSpec |%s|\n", DataFileSpec);

   if (DoPresentSummaryPage)
      PresentSummaryPage ();
   else   
   if (DoDump)
      DumpData ();
   else   
   if (DoGraph)
      GraphData ();
   else   
   if (DoList)
      ListProcessedData ();

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Get the request CGI variables.
*/

GetRequest ()

{
   register char  *cptr;

   char  Scratch [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetRequest()\n");

   /* determine what has been requested */

   GetCgiVariable ("WWW_FORM_WHAT", FormWhat, sizeof(FormWhat));
   for (cptr = FormWhat; *cptr; cptr++) *cptr = toupper(*cptr);

   if (strsame (FormWhat, "CPU", -1)) ProvidePercentCPU = true;
   if (strstr (FormWhat, ",CPU") != NULL) ProvidePercentCPU = true;
   GetCgiVariable ("WWW_FORM_CPU", Scratch, sizeof(Scratch));
   if (Scratch[0]) ProvidePercentCPU = true;

   if (strstr (FormWhat, "USER_MODE") != NULL)
      ProvidePercentUserModeCPU= true;
   GetCgiVariable ("WWW_FORM_USER_MODE_CPU", Scratch, sizeof(Scratch));
   if (Scratch[0]) ProvidePercentUserModeCPU= true;

   if (strstr (FormWhat, "BUFFERED") != NULL) ProvideBufferedIO = true;
   GetCgiVariable ("WWW_FORM_BUFFERED_IO", Scratch, sizeof(Scratch));
   if (Scratch[0]) ProvideBufferedIO = true;

   if (strstr (FormWhat, "PEAK_BUFFERED") != NULL) ProvidePeakBufferedIO = true;
   GetCgiVariable ("WWW_FORM_PEAK_BUFFERED_IO", Scratch, sizeof(Scratch));
   if (Scratch[0]) ProvidePeakBufferedIO = true;

   if (strstr (FormWhat, "DIRECT") != NULL) ProvideDirectIO = true;
   GetCgiVariable ("WWW_FORM_DIRECT_IO", Scratch, sizeof(Scratch));
   if (Scratch[0]) ProvideDirectIO = true;

   if (strstr (FormWhat, "PEAK_DIRECT") != NULL) ProvidePeakDirectIO = true;
   GetCgiVariable ("WWW_FORM_PEAK_DIRECT_IO", Scratch, sizeof(Scratch));
   if (Scratch[0]) ProvidePeakDirectIO = true;

   if (strstr (FormWhat, "MSCP") != NULL) ProvideMscpIO = true;
   GetCgiVariable ("WWW_FORM_MSCP_IO", Scratch, sizeof(Scratch));
   if (Scratch[0]) ProvideMscpIO = true;

   if (strstr (FormWhat, "PEAK_MSCP") != NULL) ProvidePeakMscpIO = true;
   GetCgiVariable ("WWW_FORM_PEAK_MSCP_IO", Scratch, sizeof(Scratch));
   if (Scratch[0]) ProvidePeakMscpIO = true;

   if (strstr (FormWhat, "FAULTS") != NULL) ProvidePageFaults = true;
   GetCgiVariable ("WWW_FORM_FAULTS", Scratch, sizeof(Scratch));
   if (Scratch[0]) ProvidePageFaults = true;

   if (strstr (FormWhat, "PEAK_FAULTS") != NULL)
      ProvidePeakHardPageFaults = true;
   GetCgiVariable ("WWW_FORM_PEAK_FAULTS", Scratch, sizeof(Scratch));
   if (Scratch[0]) ProvidePeakPageFaults = true;

   if (strstr (FormWhat, "HARD_FAULTS") != NULL) ProvideHardPageFaults = true;
   GetCgiVariable ("WWW_FORM_HARD_FAULTS", Scratch, sizeof(Scratch));
   if (Scratch[0]) ProvideHardPageFaults = true;

   if (strstr (FormWhat, "PEAK_HARD_FAULTS") != NULL)
      ProvidePeakHardPageFaults = true;
   GetCgiVariable ("WWW_FORM_PEAK_HARD_FAULTS", Scratch, sizeof(Scratch));
   if (Scratch[0]) ProvidePeakHardPageFaults = true;

   if (strstr (FormWhat, "MEMORY") != NULL) ProvideMemory = true;
   GetCgiVariable ("WWW_FORM_MEMORY", Scratch, sizeof(Scratch));
   if (Scratch[0]) ProvideMemory = true;

   if (strstr (FormWhat, "PROCESSES") != NULL) ProvideProcesses = true;
   GetCgiVariable ("WWW_FORM_PROCESSES", Scratch, sizeof(Scratch));
   if (Scratch[0]) ProvideProcesses = true;

   GetCgiVariable ("WWW_FORM_INCLUDE", FormInclude, sizeof(FormInclude));
   for (cptr = FormInclude; *cptr; cptr++) *cptr = toupper(*cptr);
   if (strstr (FormInclude, "PEAK") != NULL) IncludePeak = true;
   if (strstr (FormInclude, "TOTAL") != NULL) IncludeTotal = true;

   /* (re)build the 'what=' CGI variable for possible later use */
   *(cptr = FormWhat) = '\0';
   if (ProvidePercentCPU)
   {
      while (*cptr) cptr++;
      if (FormWhat[0]) *cptr++ = ',';
      strcpy (cptr, "cpu");
   }
   if (ProvidePercentUserModeCPU)
   {
      while (*cptr) cptr++;
      if (FormWhat[0]) *cptr++ = ',';
      strcpy (cptr, "user_mode_cpu");
   }
   if (ProvideMemory)
   {
      while (*cptr) cptr++;
      if (FormWhat[0]) *cptr++ = ',';
      strcpy (cptr, "memory");
   }
   if (ProvideBufferedIO)
   {
      while (*cptr) cptr++;
      if (FormWhat[0]) *cptr++ = ',';
      strcpy (cptr, "buffered_IO");
   }
   if (ProvidePeakBufferedIO)
   {
      while (*cptr) cptr++;
      if (FormWhat[0]) *cptr++ = ',';
      strcpy (cptr, "peak_buffered_IO");
   }
   if (ProvideDirectIO)
   {
      while (*cptr) cptr++;
      if (FormWhat[0]) *cptr++ = ',';
      strcpy (cptr, "direct_IO");
   }
   if (ProvidePeakDirectIO)
   {
      while (*cptr) cptr++;
      if (FormWhat[0]) *cptr++ = ',';
      strcpy (cptr, "peak_direct_IO");
   }
   if (ProvideMscpIO)
   {
      while (*cptr) cptr++;
      if (FormWhat[0]) *cptr++ = ',';
      strcpy (cptr, "mscp_IO");
   }
   if (ProvidePeakMscpIO)
   {
      while (*cptr) cptr++;
      if (FormWhat[0]) *cptr++ = ',';
      strcpy (cptr, "peak_mscp_IO");
   }
   if (ProvidePageFaults)
   {
      while (*cptr) cptr++;
      if (FormWhat[0]) *cptr++ = ',';
      strcpy (cptr, "faults");
   }
   if (ProvidePeakPageFaults)
   {
      while (*cptr) cptr++;
      if (FormWhat[0]) *cptr++ = ',';
      strcpy (cptr, "peak_faults");
   }
   if (ProvideHardPageFaults)
   {
      while (*cptr) cptr++;
      if (FormWhat[0]) *cptr++ = ',';
      strcpy (cptr, "hard_faults");
   }
   if (ProvidePeakHardPageFaults)
   {
      while (*cptr) cptr++;
      if (FormWhat[0]) *cptr++ = ',';
      strcpy (cptr, "peak_hard_faults");
   }
   if (ProvideProcesses)
   {
      while (*cptr) cptr++;
      if (FormWhat[0]) *cptr++ = ',';
      strcpy (cptr, "processes");
   }

   /********/
   /* from */
   /********/

   GetCgiVariable ("WWW_FORM_MINUTE", Scratch, sizeof(Scratch));
   if (isdigit(Scratch[0])) FromMinute = atoi (Scratch);

   GetCgiVariable ("WWW_FORM_HOUR", Scratch, sizeof(Scratch));
   if (isdigit(Scratch[0])) FromHour = atoi (Scratch);

   GetCgiVariable ("WWW_FORM_DAY", Scratch, sizeof(Scratch));
   if (isdigit(Scratch[0])) FromDay = atoi (Scratch);

   GetCgiVariable ("WWW_FORM_MONTH", Scratch, sizeof(Scratch));
   if (isdigit(Scratch[0])) FromMonth = atoi (Scratch);

   GetCgiVariable ("WWW_FORM_YEAR", Scratch, sizeof(Scratch));
   if (isdigit(Scratch[0])) FromYear = atoi (Scratch);

   /********/
   /* to */
   /********/

   GetCgiVariable ("WWW_FORM_TOMINUTE", Scratch, sizeof(Scratch));
   if (isdigit(Scratch[0])) ToMinute = atoi (Scratch);

   GetCgiVariable ("WWW_FORM_TOHOUR", Scratch, sizeof(Scratch));
   if (isdigit(Scratch[0])) ToHour = atoi (Scratch);

   GetCgiVariable ("WWW_FORM_TODAY", Scratch, sizeof(Scratch));
   if (isdigit(Scratch[0])) ToDay = atoi (Scratch);

   GetCgiVariable ("WWW_FORM_TOMONTH", Scratch, sizeof(Scratch));
   if (isdigit(Scratch[0])) ToMonth = atoi (Scratch);

   GetCgiVariable ("WWW_FORM_TOYEAR", Scratch, sizeof(Scratch));
   if (isdigit(Scratch[0])) ToYear = atoi (Scratch);

   /*****************/
   /* miscellaneous */
   /*****************/

   GetCgiVariable ("WWW_FORM_DO", FormDo, sizeof(FormDo));

   GetCgiVariable ("WWW_FORM_NODE", FormNode, sizeof(FormNode));
   if (!FormNode[0])
      GetCgiVariable ("WWW_FORM_LIST_NODE", FormNode, sizeof(FormNode));
   if (FormNode[0])
   {
      /* ensure the node name is in upper case */
      for (cptr = FormNode; *cptr; cptr++) *cptr = toupper(*cptr);
   }

   GetCgiVariable ("WWW_FORM_EXTRACT", Scratch, sizeof(Scratch));
   if (Scratch[0]) ExtractGraph = true;

   GetCgiVariable ("WWW_FORM_PERIOD", FormPeriod, sizeof(FormPeriod));

   GetCgiVariable ("WWW_FORM_XMAG", Scratch, sizeof(Scratch));
   if (isdigit(Scratch[0]))
      XMag = atoi (Scratch);
   else
      XMag = 1;
   if (XMag > 4) XMag = 4;
   if (XMag < 1) XMag = 1;

   GetCgiVariable ("WWW_FORM_YMAG", Scratch, sizeof(Scratch));
   if (isdigit(Scratch[0]))
      YMag = atoi (Scratch);
   else
      YMag = 1;
   if (YMag > 4) YMag = 4;
   if (YMag < 1) YMag = 1;
}

/*****************************************************************************/
/*
Process the request parameters (e.g. time, node name).  Verify the parameters 
are within constraints, particular time.  Process the time components into 
values the application can use.
*/

ValidateRequest ()

{
   static long  LibJulianDate = LIB$K_JULIAN_DATE;
   static MonthName [13] = { "???", "JAN", "FEB", "MAR", "APR", "MAY", "JUN",
                                    "JUL", "AUG", "SEP", "OCT", "NOV", "DEC" };

   int  status;
   unsigned long  BinTime [2],
                  DeltaTime [2];
   unsigned short  NumTime [7];
   char  Scratch [256];
   $DESCRIPTOR (TempDsc, "");

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ValidateRequest()\n");

   if (!FormDo[0])
      DoSummaryMenu = true;
   else
   if (toupper(FormDo[0]) == 'G')
      DoGraph = true;
   else
   if (toupper(FormDo[0]) == 'P')
      DoPresentSummaryPage = true;
   else
   if (toupper(FormDo[0]) == 'C')
      DoComprehensiveMenu = true;
   else
   if (toupper(FormDo[0]) == 'M')
      DoSummaryMenu = true;
   else
   if (toupper(FormDo[0]) == 'D')
      DoDump = true;
   else
   if (toupper(FormDo[0]) == 'L')
      DoList = true;
   else
   {
      sprintf (Scratch, "Cannot do \"%s\"", FormDo);
      ErrorGeneral (Scratch, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   if ((DoPresentSummaryPage || DoGraph || DoList || DoDump) && !FormNode[0])
   {
      ErrorGeneral ("Node name not specified.", __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   /********************/
   /* standard periods */
   /********************/

   if (FormPeriod[0] && toupper(FormPeriod[0]) != 'O')
   {
      /* a standard time period has been specified, other than "other" */
      FromHour = 0;
      FromMinute = 0;
      ToHour = 23;
      ToMinute = 59;
      FromDay = ToDay =  CurrentNumTime[2];
      FromMonth = ToMonth =  CurrentNumTime[1];
      FromYear = ToYear =  CurrentNumTime[0] % 100;

      if (toupper(FormPeriod[0]) == 'U')
      {
         /* "until_now" (7am to this hour) */
         FromHour = 7;
         ToHour = CurrentNumTime[3];
      }
      else
      if (toupper(FormPeriod[0]) == 'B')
      {
         /* "business" hours (7am to 7pm) */
         FromHour = 7;
         ToHour = 18;
      }
      else
      if (toupper(FormPeriod[0]) == 'T')
      {
         /* "today" */
         ToHour = 23;
      }
      else
      if (toupper(FormPeriod[0]) == 'S' ||
          toupper(FormPeriod[0]) == 'Y')
      {
         /* "since_yesterday" until now, or all of "yesterday" */
         TempDsc.dsc$a_pointer = "1 00:00:00.00";
         TempDsc.dsc$w_length = 13;
         if (VMSnok (status = sys$bintim (&TempDsc, &DeltaTime)))
         {
            ErrorVmsStatus (status, FormPeriod, FormPeriod, __FILE__, __LINE__);
            exit (SS$_NORMAL);
         }
         lib$sub_times (&CurrentBinTime, &DeltaTime, &BinTime);
         sys$numtim (&NumTime, &BinTime);
         FromDay = NumTime[2];
         FromMonth = NumTime[1];
         FromYear = NumTime[0] % 100;
         if (toupper(FormPeriod[0]) == 'S')
            ToHour = CurrentNumTime[3];
         else
         {
            ToDay = NumTime[2];
            ToMonth = NumTime[1];
            ToYear = NumTime[0] % 100;
         }
      }
      else
      if (isdigit(FormPeriod[0]))
      {
         TempDsc.dsc$a_pointer = Scratch;
         TempDsc.dsc$w_length =
            sprintf (Scratch, "%s 00:00:00.00", FormPeriod);
         if (VMSnok (status = sys$bintim (&TempDsc, &DeltaTime)))
         {
            ErrorVmsStatus (status, FormPeriod, FormPeriod, __FILE__, __LINE__);
            exit (SS$_NORMAL);
         }
         lib$sub_times (&CurrentBinTime, &DeltaTime, &BinTime);
         sys$numtim (&NumTime, &BinTime);
         FromDay = NumTime[2];
         FromMonth = NumTime[1];
         FromYear = NumTime[0] % 100;
      }
      else
      {
         ErrorGeneral (
"Periods are \"until_now\", \"business\", \"today\", \
\"since_yesterday\", \"yesterday\", \
or <I>number-of-days-previous</I>.",
            __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
   }

   /*******************/
   /* time components */
   /*******************/

   /* non-specified date components default to those of the current day */
   if (!FromDay) FromDay = CurrentNumTime[2];
   if (!FromMonth) FromMonth = CurrentNumTime[1];
   if (!FromYear) FromYear = CurrentNumTime[0] % 100;
   if (!ToDay) ToDay =  CurrentNumTime[2];
   if (!ToMonth) ToMonth =  CurrentNumTime[1];
   if (!ToYear) ToYear =  CurrentNumTime[0] % 100;

   /* bit of a sanity check (prevents 'MonthName[x]' access violating, etc.) */
   if (FromMonth < 1 || FromMonth > 12) FromMonth = 0;
   if (FromYear > 99) FromYear %= 100;
   if (ToMonth < 1 || ToMonth > 12) ToMonth = 0;
   if (ToYear > 99) ToYear %= 100;

   /* ensure the commencement date/time is acceptable */
   TempDsc.dsc$a_pointer = DateString;
   TempDsc.dsc$w_length = 
      sprintf (DateString, "%d-%s-19%d %02.02d:%02.02d",
               FromDay, MonthName[FromMonth], FromYear, FromHour, FromMinute);
   if (Debug) fprintf (stdout, "DateString |%s|\n", DateString);
   if (VMSnok (status = sys$bintim (&TempDsc, &BinTime)))
   {
      sprintf (Scratch, "%02.02d:%02.02d %02.02d/%02.02d/%02.02d",
               FromHour, FromMinute, FromDay, FromMonth, FromYear);
      ErrorVmsStatus (status, Scratch, DateString, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
   /* get the commencement julian date (number of days in epoch) */
   lib$cvt_from_internal_time (&LibJulianDate, &FromJulianDate, &BinTime);

   /* ensure the conclusion date/time is acceptable */
   TempDsc.dsc$a_pointer = ToDateString;
   TempDsc.dsc$w_length = 
      sprintf (ToDateString, "%d-%s-19%d %02.02d:%02.02d",
               ToDay, MonthName[ToMonth], ToYear, ToHour, ToMinute);
   if (Debug) fprintf (stdout, "ToDateString |%s|\n", ToDateString);
   if (VMSnok (status = sys$bintim (&TempDsc, &BinTime)))
   {
      sprintf (Scratch, "%02.02d:%02.02d %02.02d/%02.02d/%02.02d",
               ToHour, ToMinute, ToDay, ToMonth, ToYear);
      ErrorVmsStatus (status, Scratch, ToDateString, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
   /* get the conclusion julian date (number of days in epoch) */
   lib$cvt_from_internal_time (&LibJulianDate, &ToJulianDate, &BinTime);

   /***************************/
   /* calculate period values */
   /***************************/

   /* using julian dates, get number of days and hours sample period covers */
   NumberOfHours = ((ToJulianDate - FromJulianDate) * 24) + ToHour - FromHour;
   NumberOfDays = (NumberOfHours / 24);

   if (DoPresentSummaryPage || DoGraph || DoList || DoDump)
   {
      if (NumberOfHours < 0 || (NumberOfHours == 0 && ToMinute <= FromMinute))
      {
         sprintf (Scratch,
"Beginning of period (%02.02d:%02.02d %02.02d/%02.02d/%02.02d) \
does not precede end (%02.02d:%02.02d %02.02d/%02.02d/%02.02d)!",
         FromHour, FromMinute, FromDay, FromMonth, FromYear,
         ToHour, ToMinute, ToDay, ToMonth, ToYear);
         ErrorGeneral (Scratch, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
      if (NumberOfHours < 1)
      {
         ErrorGeneral ("Little point to such a short period!",
                       __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
      if (NumberOfDays > MaxSamplePeriodDays)
      {
         sprintf (Scratch, "Maximum sample period is %d days.",
                  MaxSamplePeriodDays);
         ErrorGeneral (Scratch, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
   }

   /* rate at which the data (and any graphic) must be X-axis compressed */
   if (NumberOfHours <= 12)
      RecordSampleRate = 1;
   else
   if (NumberOfHours <= 24)
      RecordSampleRate = 2;
   else
      RecordSampleRate = (NumberOfDays + 1) * 2;

   /* add one to number of hours, for graphing purposes, re-calculate days */
   NumberOfHours++;
   NumberOfDays = (NumberOfHours / 24);

   /* the number of minutes from midnight data begins being processed */
   StartMinuteOnFirstDay = (FromHour * 60) + FromMinute;

   /* the current minute from the start of the specified period */
   lib$cvt_from_internal_time (&LibJulianDate, &CurrentJulianDate,
                               &CurrentBinTime);
   CurrentMinuteFromStartOfPeriod =
      ((((CurrentJulianDate - FromJulianDate) * 24) +
        (CurrentNumTime[3] - FromHour)) * 60) +
       CurrentNumTime[4];

   if (Debug)
      fprintf (stdout,
"FromJulianDate: %d ToJulianDate: %d\n\
NumberOfDays: %d NumberOfHours: %d RecordSampleRate: %d\n",
      FromJulianDate, ToJulianDate,
      NumberOfDays, NumberOfHours, RecordSampleRate);
}

/*****************************************************************************/
/*
Provide either a standard or comprehensive menu allowing System Performance 
Information system, category and period to be specified.
*/

SummaryMenu ()

{
   char  *CheckedPtr,
         *TodayCheckedPtr,
         *UntilNowCheckedPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SummaryMenu()\n");

   if (!DoComprehensiveMenu) CheckedPtr = " CHECKED";

   fprintf (HttpOut,
"%s\
<HTML>\n\
<!-- SoftwareID: %s -->\n\
<TITLE>HyperSpi, Performance Summary</TITLE>\n\
<H1>%sPerformance Summary</H1>\n\
<P>\n\
<HR>\n\
<P>\n\
<FORM ACTION=\"%s\">\n",
   Http200Header, 
   SoftwareID,
   IconImg, CgiScriptName);

   SelectNodeNames ();

   if (DoComprehensiveMenu)
   {
      fputs (
"or ... node <INPUT TYPE=text NAME=node SIZE=6 MAXLENGTH=6><BR>",
      HttpOut);
   }

   if (CurrentNumTime[3] >= 7)
   {
      UntilNowCheckedPtr = " CHECKED";
      TodayCheckedPtr = "";
   }
   else
   {
      UntilNowCheckedPtr = "";
      TodayCheckedPtr = " CHECKED";
   }

   fprintf (HttpOut,
"<P>\n\
<INPUT TYPE=radio NAME=period VALUE=until_now%s>\
 since 7am <I>(until now)</I><BR>\
<INPUT TYPE=radio NAME=period VALUE=today%s> today<BR>\n\
<INPUT TYPE=radio NAME=period VALUE=since_yesterday>\
 since yesterday <I>(until now)</I><BR>\n\
<INPUT TYPE=radio NAME=period VALUE=yesterday> yesterday<BR>\n\
<INPUT TYPE=radio NAME=period VALUE=7> previous week \n\
<INPUT TYPE=radio NAME=period VALUE=14> two weeks \n\
<INPUT TYPE=radio NAME=period VALUE=28> four weeks<BR>\n\
<INPUT TYPE=radio NAME=period VALUE=other>\
 from\n\
<INPUT TYPE=text SIZE=2 MAXLENGTH=2 NAME=hour VALUE=\"00\">:\n\
<INPUT TYPE=text SIZE=2 MAXLENGTH=2 NAME=minute VALUE=\"00\">\n\
<INPUT TYPE=text SIZE=2 MAXLENGTH=2 NAME=day VALUE=\"%02.02d\">/\n\
<INPUT TYPE=text SIZE=2 MAXLENGTH=2 NAME=month VALUE=\"%02.02d\">/\n\
<INPUT TYPE=text SIZE=2 MAXLENGTH=2 NAME=year VALUE=\"%02.02d\"> \
 to\n\
<INPUT TYPE=text SIZE=2 MAXLENGTH=2 NAME=tohour VALUE=\"%02.02d\">:\n\
<INPUT TYPE=text SIZE=2 MAXLENGTH=2 NAME=tominute VALUE=\"59\">\n\
<INPUT TYPE=text SIZE=2 MAXLENGTH=2 NAME=today VALUE=\"%02.02d\">/\n\
<INPUT TYPE=text SIZE=2 MAXLENGTH=2 NAME=tomonth VALUE=\"%02.02d\">/\n\
<INPUT TYPE=text SIZE=2 MAXLENGTH=2 NAME=toyear VALUE=\"%02.02d\">\n\
<P>\n\
<INPUT TYPE=checkbox NAME=cpu VALUE=yes CHECKED> CPU<BR>\n\
<INPUT TYPE=checkbox NAME=memory VALUE=yes%s> memory<BR>\n\
<INPUT TYPE=checkbox NAME=processes VALUE=yes> processes<BR>\n",
   UntilNowCheckedPtr, TodayCheckedPtr,
   CurrentNumTime[2], CurrentNumTime[1], CurrentNumTime[0] % 100,
   CurrentNumTime[3],
   CurrentNumTime[2], CurrentNumTime[1], CurrentNumTime[0] % 100,
   CheckedPtr);

   if (DoComprehensiveMenu)
   {
      fputs (
"<INPUT TYPE=checkbox NAME=faults VALUE=yes> paging (soft) \
 <INPUT TYPE=checkbox NAME=peak_faults VALUE=yes> peak<BR>\n",
      HttpOut);
   }

   fprintf (HttpOut,
"<INPUT TYPE=checkbox NAME=hard_faults VALUE=yes%s> paging",
   CheckedPtr);
   if (DoComprehensiveMenu)
   {
      fputs (
" (hard)\
 <INPUT TYPE=checkbox NAME=peak_hard_faults VALUE=yes> peak",
      HttpOut);
   }
   fputs ("<BR>\n", HttpOut);

   fputs (
"<INPUT TYPE=checkbox NAME=direct_IO VALUE=yes> disk IO",
   HttpOut);
   if (DoComprehensiveMenu)
   {
      fputs (
" <INPUT TYPE=checkbox NAME=peak_direct_IO VALUE=yes> peak",
      HttpOut);
   }
   fputs ("<BR>\n", HttpOut);

   fputs (
"<INPUT TYPE=checkbox NAME=buffered_IO VALUE=yes> other IO",
   HttpOut);
   if (DoComprehensiveMenu)
   {
      fputs (
" <INPUT TYPE=checkbox NAME=peak_buffered_IO VALUE=yes> peak",
      HttpOut);
   }
   fputs ("<BR>\n", HttpOut);

   fputs (
"<INPUT TYPE=checkbox NAME=mscp_IO VALUE=yes> MSCP IO",
   HttpOut);
   if (DoComprehensiveMenu)
   {
      fputs (
" <INPUT TYPE=checkbox NAME=peak_mscp_IO VALUE=yes> peak",
      HttpOut);
   }
   fputs ("<BR>\n", HttpOut);

   if (DoComprehensiveMenu)
   {
      fputs (
"<INPUT TYPE=checkbox NAME=include VALUE=peak>\
 <I>include peak plot (where applicable)</I>\n\
<P>\n\
<INPUT TYPE=radio NAME=do VALUE=page CHECKED>graph\n\
 <I>(X,Y magnification:\
<SELECT NAME=Xmag SIZE=1>\n\
<OPTION VALUE=1 SELECTED>1 \n\
<OPTION VALUE=2>2 \n\
<OPTION VALUE=4>4 \n\
</SELECT>,\
<SELECT NAME=Ymag SIZE=1>\n\
<OPTION VALUE=1 SELECTED> 1 \n\
<OPTION VALUE=2> 2 \n\
<OPTION VALUE=4> 4 \n\
</SELECT>)</I><BR>\n\
<INPUT TYPE=radio NAME=do VALUE=list> list<BR>\n\
<INPUT TYPE=radio NAME=do VALUE=dump> dump<BR>\n\
<INPUT TYPE=hidden NAME=extract VALUE=yes>\n\
<P>\n",
      HttpOut);
   }
   else
   {
      fputs (
"<INPUT TYPE=hidden NAME=do VALUE=page>\n",
      HttpOut);
   }

   fputs (
"<P>\n\
<INPUT TYPE=submit VALUE=process> \n",
   HttpOut);

   if (!DoComprehensiveMenu)
   { 
      fputs (
"<I>(<INPUT TYPE=checkbox NAME=Ymag VALUE=2> double-height graph)</I> \n",
      HttpOut);
   }

   fprintf (HttpOut,
"<INPUT TYPE=reset VALUE=reset>\n\
</FORM>\n\
<P>\n\
<HR>\n\
<A HREF=\"%s\"><I>Help</I></A>\n\
</HTML>\n",
   HelpUrl);
}

/*****************************************************************************/
/*
Output a list of node names that data can be selected from.  Get these either 
from a command-line-supplied list or from data files present for the current 
day.
*/ 

SelectNodeNames ()

{
   register int  idx;

   char  *CheckedPtr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SelectNodeNames()\n");

   if (SpiNodeNames[0][0])
   {
      NodeCount = 0;
      for (idx = 0; SpiNodeNames[idx][0]; idx++) NodeCount++;
      if (NodeCount <= MaxNodeCountBeforeSelectBox)
      {
         CheckedPtr = " CHECKED";
         for (idx = 0; SpiNodeNames[idx][0]; idx++)
         {
            fprintf (HttpOut,
               "<INPUT TYPE=radio NAME=list_node VALUE=\"%s\"%s> %s<BR>\n",
               SpiNodeNames[idx], CheckedPtr, SpiNodeNames[idx]);
            CheckedPtr = "";
         }
      }
      else
      {
         CheckedPtr = " SELECTED";
         fprintf (HttpOut, "<SELECT NAME=list_node SIZE=%d>\n",
                  MaxNodeCountBeforeSelectBox);
         for (idx = 0; SpiNodeNames[idx][0]; idx++)
         {
            fprintf (HttpOut, "<OPTION VALUE=\"%s\"%s> %s\n",
                     SpiNodeNames[idx], CheckedPtr, SpiNodeNames[idx]);
            CheckedPtr = "";
         }
         fputs ("</SELECT>\n", HttpOut);
      }
   }
   else
   {
      /* create file specification for getting node names (today's files) */
      DataFileSpecLength =
         sprintf (DataFileSpec,
                  "%sHYPERSPI_%s_*_%02.02d%02.02d%02.02d.DAT",
                  HyperSpiDataDirectory, HyperSpiDataVersion,
                  FromDay, FromMonth, FromYear);
      if (Debug) fprintf (stdout, "DataFileSpec |%s|\n", DataFileSpec);

      /* count the number of node data files */
      ProcessDataFiles (NULL, false);
      NodeCount = DataFilesFoundCount;

      if (NodeCount <= MaxNodeCountBeforeSelectBox)
         ProcessDataFiles (&SelectNodeNameByDataFileName, false);
      else
      {
         fprintf (HttpOut, "<SELECT NAME=list_node SIZE=%d>\n",
                  MaxNodeCountBeforeSelectBox);
         ProcessDataFiles (&SelectNodeNameByDataFileName, false);
         fputs ("</SELECT>\n", HttpOut);
      }
   }
}

/*****************************************************************************/
/*
Called by pointer to function each time a matching data file is returned by 
sys$search(0 in function FindDataFiles().  Output the node name associated 
with the data file.
*/ 

SelectNodeNameByDataFileName ()

{
   static char  *CheckedPtr = " CHECKED";
   static char  *SelectedPtr = " SELECTED";

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "SelectNodeNameByFileName() |%s|\n", DataFileName);

   if (NodeCount < MaxNodeCountBeforeSelectBox)
   {
      fprintf (HttpOut,
         "<INPUT TYPE=radio NAME=list_node VALUE=\"%s\"%s> %s<BR>\n",
         DataNode, CheckedPtr, DataNode);
      CheckedPtr = "";
   }
   else
   {
      fprintf (HttpOut, "<OPTION VALUE=\"%s\"%s> %s\n",
               DataNode, SelectedPtr, DataNode);
      SelectedPtr = "";
   }
}

/*****************************************************************************/
/*
Generates an HTML page containing information about the selected node, and the 
any additional performance information and graphs.
*/

PresentSummaryPage ()

{
   int  PeriodHours;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "PresentSummaryPage()\n");

   /*
      Data processing for commentary can take some time.
      Let the client know something's happening by returning the HTTP
      header as soon as possible.
   */
   fprintf (HttpOut,
"%s\
<HTML>\n\
<!-- SoftwareID: %s -->\n\
<TITLE>HyperSpi Performance Summary: %s</TITLE>\n\
<H1>%sPerformance Summary: %s</H1>\n\
<P>\n\
Period: ",
   Http200Header,
   SoftwareID,
   FormNode, IconImg, FormNode);
   fflush (HttpOut);

   if ((!NumberOfDays && NumberOfHours == 24) || NumberOfDays == 1)
      fputs ("1 day ", HttpOut);
   else
   if (NumberOfDays > 1)
      fprintf (HttpOut, "%d days ", NumberOfDays);

   if ((PeriodHours = (NumberOfHours % 24)) == 1)
      fputs ("1 hour ", HttpOut);
   else
   if (PeriodHours > 1)
      fprintf (HttpOut, "%d hours ", PeriodHours);

   fprintf (HttpOut,
" from %02.02d:%02.02d <I>%02.02d/%02.02d/%02.02d </I> \
to %02.02d:%02.02d <I>%02.02d/%02.02d/%02.02d </I> \
(generated %02.02d:%02.02d <I>%02.02d/%02.02d/%02.02d</I>)\n",
   FromHour, FromMinute, FromDay, FromMonth, FromYear,
   ToHour, ToMinute, ToDay, ToMonth, ToYear,
   CurrentNumTime[3], CurrentNumTime[4],
   CurrentNumTime[2], CurrentNumTime[1], CurrentNumTime[0] % 100);

   /* summarize the data and calculate required averages */
   SummarizeData ();

   if (!DataRecordsProcessedCount)
   {
      fputs (
"<P>\n\
<HR>\n\
<P>\n\
<I>No data available.</I>\n\
<P>\n\
<HR>\n\
</HTML>\n",
             HttpOut);
      return;
   }

   IncludeFile (FormNode);

   fprintf (HttpOut,
"<P>\n\
CPUs: <TT>%d</TT>\n\
<BR>Memory: <TT>%dMb</TT> physical, <TT>%dMb</TT> page space \
<I>(<TT>%dMb</TT> total)</I>.\n\
<BR>Processes: <TT>%d</TT> average, <TT>%d</TT> peak.\n\
<BR><I>(X axes are marked at hour intervals)</I>\n\
<P>\n\
<HR>\n\
<P>\n",
   NumberOfCPUs,
   SystemMemoryMBytes, PageSpaceMBytes, SystemMemoryMBytes + PageSpaceMBytes,
   AveNumberOfProcesses, MaxNumberOfProcesses);

   if (ProvidePercentCPU) PresentCPU ();
   if (ProvideMemory) PresentMemory ();
   if (ProvideProcesses) PresentProcesses ();
   if (ProvidePageFaults) PresentPageFaults ();
   if (ProvidePeakPageFaults) PresentPeakPageFaults ();
   if (ProvideHardPageFaults) PresentHardPageFaults ();
   if (ProvidePeakHardPageFaults) PresentPeakHardPageFaults ();
   if (ProvideBufferedIO) PresentBufferedIO ();
   if (ProvidePeakBufferedIO) PresentPeakBufferedIO ();
   if (ProvideDirectIO) PresentDirectIO ();
   if (ProvidePeakDirectIO) PresentPeakDirectIO ();
   if (ProvideMscpIO) PresentMscpIO ();
   if (ProvidePeakMscpIO) PresentPeakMscpIO ();

   fprintf (HttpOut,
"<P>\n\
<HR>\n\
<A HREF=\"%s\"><I>Help</I></A>\n\
</HTML>\n",
   HelpUrl);
   
}

/*****************************************************************************/
/*
Place an HTML link into the HTTP output stream to get a plot-data, GIF image, 
generated dynamically by this application, embedded in the current HTML 
document.  The 'UniqueNumber()' is used to place a unique component into the 
URL of the graph GIF URL, ensuring a cached version is not retrieved.  The 
'WhatPtr' is the name of the performance count to graph.  The optional 
'IncludePtr' allows selected graphs to contain additional information.
*/ 

GraphImageLink
(
boolean InlineImage,
char *WhatPtr,
char *IncludePtr
)
{
   char  *StartPtr,
         *EndPtr;

   if (Debug) fprintf (stdout, "GraphImageLink()\n");

   if (InlineImage)
   {
      StartPtr = "<IMG ALIGN=top ALT=\"[graph]\" SRC=";
      EndPtr = ">";
   }
   else
   {
      StartPtr = "<I>(<A HREF=";
      EndPtr = ">Extract graph.</A>)</I>";
   }

   fprintf (HttpOut,
"%s\"%s/?do=graph&unique=%s&%s=yes&include=%s\
&Xmag=%d&Ymag=%d&node=%s\
&minute=%d&hour=%d&day=%d&month=%d&year=%d\
&ToMinute=%d&ToHour=%d&ToDay=%d&ToMonth=%d&ToYear=%d\"%s\n",
   StartPtr,
   CgiScriptName, UniqueNumber(), WhatPtr, IncludePtr, 
   XMag, YMag, FormNode,
   FromMinute, FromHour, FromDay, FromMonth, FromYear,
   ToMinute, ToHour, ToDay, ToMonth, ToYear,
   EndPtr);
}

/*****************************************************************************/
/*
CPU usage.  Place textual information and a graphic GIF image link into the 
current HTML document being generated.
*/ 

PresentCPU ()

{
   if (Debug) fprintf (stdout, "PresentCPU()\n");

   fputs ("<H2>CPU Usage</H2>\n", HttpOut);
   GraphImageLink (true, "cpu", "peak");
   fprintf (HttpOut,
"<BR>Y axis is percentage. \
Histogram is average, plot is peak \
<I>(total of available %d CPU(s))</I>.\n\
<BR>Overall <TT>%d%%</TT> average, <TT>%d%%</TT> peak \
<I>(user-mode <TT>%d%%</TT> average, <TT>%d%%</TT> peak)</I>.\n",
   NumberOfCPUs,
   AvePercentCPU, PeakPercentCPU,
   AvePercentUserModeCPU, PeakPercentUserModeCPU);
   if (ExtractGraph) GraphImageLink (false, "cpu", "peak");
}

/*****************************************************************************/
/*
Page space and physical memory usage.  Place textual information and a graphic 
GIF image link into the current HTML document being generated.
*/ 

PresentMemory ()

{
   if (Debug) fprintf (stdout, "PresentMemory()\n");

   fputs ("<H2>Memory Usage</H2>\n", HttpOut);
   GraphImageLink (true, "memory", "");
   fprintf (HttpOut,
"<BR>Y axis is percentage. \
Histogram is page space (%dMb) usage, plot is physical memory (%dMb).\n\
<BR>Overall page space <TT>%d%%</TT> average, <TT>%d%%</TT> peak; \
physical memory <TT>%d%%</TT> average, <TT>%d%%</TT> peak.\n",
   PageSpaceMBytes, SystemMemoryMBytes,
   AvePageSpacePercentInUse, MaxPageSpacePercentInUse,
   AveSystemMemoryPercentInUse, MaxSystemMemoryPercentInUse);
   if (ExtractGraph) GraphImageLink (false, "memory", "");
}

/*****************************************************************************/
/*
Number of processes.  Place textual information and a graphic GIF image link 
into the current HTML document being generated.
*/ 

PresentProcesses ()

{
   if (Debug) fprintf (stdout, "PresentProcesses()\n");

   fputs ("<H2>Number of Processes</H2>\n", HttpOut);
   GraphImageLink (true, "processes", "");
   SetScaling (MaxNumberOfProcesses);
   fprintf (HttpOut,
"<BR>Y axis is number of processes (full-scale %d).\n\
<BR>Overall <TT>%d</TT> average <I>(maximum <TT>%d</TT>)</I>.\n",
   PlotScaleY, AveNumberOfProcesses, MaxNumberOfProcesses);
   if (ExtractGraph) GraphImageLink (false, "processes", "");
}

/*****************************************************************************/
/*
Buffered IO (non-disk, non-tape, i.e. network, terminal).  Place textual 
information and a graphic GIF image link into the current HTML document being 
generated.
*/ 

PresentBufferedIO ()

{
   char  *ExplainPtr;

   if (Debug) fprintf (stdout, "PresentBufferedIO()\n");

   fputs ("<H2>Buffered IO (network, terminal, etc.)</H2>\n", HttpOut);
   if (IncludePeak)
   {
      GraphImageLink (true, "buffered_IO", "peak");
      SetScaling (PeakBufferedIO);
      ExplainPtr = " Histogram is average, plot is peak.";
   }
   else
   {
      GraphImageLink (true, "buffered_IO", "");
      SetScaling (MaxAveBufferedIO);
      ExplainPtr = "";
   }
   fprintf (HttpOut,
"<BR>Y axis is IOs per-second (full-scale %d).%s\n\
<BR>Overall <TT>%d</TT> average <I>(maximum <TT>%d</TT>)</I>, \
peak <TT>%d</TT>.\n",
   PlotScaleY, ExplainPtr, AveBufferedIO, MaxAveBufferedIO, PeakBufferedIO);
   if (IncludePeak)
      if (ExtractGraph) GraphImageLink (false, "buffered_IO", "peak"); else;
   else
      if (ExtractGraph) GraphImageLink (false, "buffered_IO", "");
}

/*****************************************************************************/
/*
Peak buffered IO.  Place textual information and a graphic GIF image link 
into the current HTML document being generated.
*/ 

PresentPeakBufferedIO ()

{
   if (Debug) fprintf (stdout, "PresentPeakBufferedIO()\n");

   fputs ("<H2>Peak Buffered IO (network, terminal, etc.)</H2>\n", HttpOut);
   GraphImageLink (true, "peak_buffered_IO", "");
   SetScaling (PeakBufferedIO);
   fprintf (HttpOut,
"<BR>Y axis is IOs per-second (full-scale %d). \n\
<BR>Peak <TT>%d</TT> per-second.\n",
   PlotScaleY, PeakBufferedIO);
   if (ExtractGraph) GraphImageLink (false, "peak_buffered_IO", "");
}

/*****************************************************************************/
/*
Direct IO (disk, tape, etc).  Place textual information and a graphic GIF 
image link into the current HTML document being generated.
*/ 

PresentDirectIO ()

{
   char  *ExplainPtr;

   if (Debug) fprintf (stdout, "PresentDirectIO()\n");

   fputs ("<H2>Direct IO (disk, tape, etc.)</H2>\n", HttpOut);
   if (IncludePeak)
   {
      GraphImageLink (true, "direct_IO", "peak");
      SetScaling (PeakDirectIO);
      ExplainPtr = " Histogram is average, plot is peak.";
   }
   else
   {
      GraphImageLink (true, "direct_IO", "");
      SetScaling (MaxAveDirectIO);
      ExplainPtr = "";
   }
   fprintf (HttpOut,
"<BR>Y axis is IOs per-second (full-scale %d).%s\n\
<BR>Overall <TT>%d</TT> average <I>(maximum <TT>%d</TT>)</I>, \
peak <TT>%d</TT>.\n",
   PlotScaleY, ExplainPtr, AveDirectIO, MaxAveDirectIO, PeakDirectIO);
   if (IncludePeak)
      if (ExtractGraph) GraphImageLink (false, "direct_IO", "peak"); else;
   else
      if (ExtractGraph) GraphImageLink (false, "direct_IO", "");
}

/*****************************************************************************/
/*
Peak direct IO.  Place textual information and a graphic GIF image link 
into the current HTML document being generated.
*/ 

PresentPeakDirectIO ()

{
   if (Debug) fprintf (stdout, "PresentPeakDirectIO()\n");

   fputs ("<H2>Peak Direct IO (disk, tape, etc.)</H2>\n", HttpOut);
   GraphImageLink (true, "peak_direct_IO", "");
   SetScaling (PeakDirectIO);
   fprintf (HttpOut,
"<BR>Y axis is IOs per-second (full-scale %d). \n\
<BR>Peak <TT>%d</TT> per-second.\n",
   PlotScaleY, PeakDirectIO);
   if (ExtractGraph) GraphImageLink (false, "peak_direct_IO", "");
}

/*****************************************************************************/
/*
MSCP IO.  Place textual information and a graphic GIF image link into the 
current HTML document being generated.
*/ 

PresentMscpIO ()

{
   char  *ExplainPtr;

   if (Debug) fprintf (stdout, "PresentMscpIO()\n");

   fputs ("<H2>MSCP IO (served disk and tape)</H2>\n", HttpOut);
   if (IncludePeak)
   {
      GraphImageLink (true, "mscp_IO", "peak");
      SetScaling (PeakMscpIO);
      ExplainPtr = " Histogram is average, plot is peak.";
   }
   else
   {
      GraphImageLink (true, "mscp_IO", "");
      SetScaling (MaxAveMscpIO);
      ExplainPtr = "";
   }
   fprintf (HttpOut,
"<BR>Y axis is IOs per-second (full-scale %d).%s\n\
<BR>Overall <TT>%d</TT> average <I>(maximum <TT>%d</TT>)</I>, \
peak <TT>%d</TT>.\n",
   PlotScaleY, ExplainPtr, AveMscpIO, MaxAveMscpIO, PeakMscpIO);
   if (IncludePeak)
      if (ExtractGraph) GraphImageLink (false, "mscp_IO", "peak"); else;
   else
      if (ExtractGraph) GraphImageLink (false, "mscp_IO", "");
}

/*****************************************************************************/
/*
Peak mscp IO.  Place textual information and a graphic GIF image link 
into the current HTML document being generated.
*/ 

PresentPeakMscpIO ()

{
   if (Debug) fprintf (stdout, "PresentPeakMscpIO()\n");

   fputs ("<H2>Peak MSCP IO (served disk and tape)</H2>\n", HttpOut);
   GraphImageLink (true, "peak_mscp_IO", "");
   SetScaling (PeakMscpIO);
   fprintf (HttpOut,
"<BR>Y axis is IOs per-second (full-scale %d). \n\
<BR>Peak <TT>%d</TT> per-second.\n",
   PlotScaleY, PeakMscpIO);
   if (ExtractGraph) GraphImageLink (false, "peak_mscp_IO", "");
}

/*****************************************************************************/
/*
Page faulting IO.  Place textual information and a graphic GIF image link into 
the current HTML document being generated.
*/ 

PresentPageFaults ()

{
   char  *ExplainPtr;

   if (Debug) fprintf (stdout, "PresentPageFaults()\n");

   fputs ("<H2>Paging (Soft)</H2>\n", HttpOut);
   if (IncludePeak)
   {
      GraphImageLink (true, "faults", "peak");
      SetScaling (PeakPageFaults);
      ExplainPtr = " Histogram is average, plot is peak.";
   }
   else
   {
      GraphImageLink (true, "faults", "");
      SetScaling (MaxAvePageFaults);
      ExplainPtr = "";
   }
   fprintf (HttpOut,
"<BR>Y axis is page faults per-second (full-scale %d).%s\n\
<BR>Overall <TT>%d</TT> average <I>(maximum <TT>%d</TT>)</I>, \
peak <TT>%d</TT>.\n",
   PlotScaleY, ExplainPtr, AvePageFaults, MaxAvePageFaults, PeakPageFaults);
   if (IncludePeak)
      if (ExtractGraph) GraphImageLink (false, "faults", "peak"); else;
   else
      if (ExtractGraph) GraphImageLink (false, "faults", "");
}

/*****************************************************************************/
/*
Peak paging to disk.  Place textual information and a graphic GIF image link 
into the current HTML document being generated.
*/ 

PresentPeakPageFaults ()

{
   if (Debug) fprintf (stdout, "PresentPeakPageFaults()\n");

   fputs ("<H2>Peak Paging (Soft)</H2>\n", HttpOut);
   GraphImageLink (true, "peak_faults", "");
   SetScaling (PeakPageFaults);
   fprintf (HttpOut,
"<BR>Y axis is page faults per-second (full-scale %d). \n\
<BR>Peak <TT>%d</TT> per-second.\n",
   PlotScaleY, PeakPageFaults);
   if (ExtractGraph) GraphImageLink (false, "peak_faults", "");
}

/*****************************************************************************/
/*
Page faulting IO.  Place textual information and a graphic GIF image link into 
the current HTML document being generated.
*/ 

PresentHardPageFaults ()

{
   char  *ExplainPtr;

   if (Debug) fprintf (stdout, "PresentHardPageFaults()\n");

   fputs ("<H2>Paging To Disk</H2>\n", HttpOut);
   if (IncludePeak)
   {
      GraphImageLink (true, "hard_faults", "peak");
      SetScaling (PeakHardPageFaults);
      ExplainPtr = " Histogram is average, plot is peak.";
   }
   else
   {
      GraphImageLink (true, "hard_faults", "");
      SetScaling (MaxAveHardPageFaults);
      ExplainPtr = "";
   }
   fprintf (HttpOut,
"<BR>Y axis is IOs per-second (full-scale %d).%s\n\
<BR>Overall <TT>%d</TT> average <I>(maximum <TT>%d</TT>)</I>, \
peak <TT>%d</TT>.\n",
   PlotScaleY, ExplainPtr, AveHardPageFaults, MaxAveHardPageFaults,
   PeakHardPageFaults);
   if (IncludePeak)
      if (ExtractGraph) GraphImageLink (false, "hard_faults", "peak"); else;
   else
      if (ExtractGraph) GraphImageLink (false, "hard_faults", "");
}

/*****************************************************************************/
/*
Peak paging to disk.  Place textual information and a graphic GIF image link 
into the current HTML document being generated.
*/ 

PresentPeakHardPageFaults ()

{
   if (Debug) fprintf (stdout, "PresentPeakHardPageFaults()\n");

   fputs ("<H2>Peak Paging To Disk</H2>\n", HttpOut);
   GraphImageLink (true, "peak_hard_faults", "");
   SetScaling (PeakHardPageFaults);
   fprintf (HttpOut,
"<BR>Y axis is IOs per-second (full-scale %d). \n\
<BR>Peak <TT>%d</TT> per-second.\n",
   PlotScaleY, PeakHardPageFaults);
   if (ExtractGraph) GraphImageLink (false, "peak_hard_faults", "");
}

/*****************************************************************************/
/*
Read all records for the specified node in the specified time range, 
calculating the average and maximum for each of the data categories.  Assumes 
it is only to be called once as it does not initialize any of the storage.
*/ 

SummarizeData ()

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SummarizeData()\n");

   ProcessDataFiles (&SummarizeRecord, true);

   /* the averaging must be by the actual sample rate */
   DataRecordsProcessedCount /= RecordSampleRate;

   if (!DataRecordsProcessedCount) return;

   AvePercentCPU /= DataRecordsProcessedCount;
   AvePercentUserModeCPU /= DataRecordsProcessedCount;
   AveSystemMemoryPercentInUse /= DataRecordsProcessedCount;
   AveNumberOfProcesses /= DataRecordsProcessedCount;
   AvePageSpacePercentInUse /= DataRecordsProcessedCount;
   AvePageFaults /= DataRecordsProcessedCount;
   AveHardPageFaults /= DataRecordsProcessedCount;
   AveBufferedIO /= DataRecordsProcessedCount;
   AveDirectIO /= DataRecordsProcessedCount;
   AveMscpIO /= DataRecordsProcessedCount;
}

/*****************************************************************************/
/*
Set the various accumulators according to the data in the current 'SpiRecord'.
*/

SummarizeRecord ()

{
   static unsigned long  RecordCount = 0,
                         NumberOfProcesses = 0,
                         SystemMemoryPercentInUse = 0,
                         PageSpacePercentInUse = 0,
                         PercentCPU = 0,
                         PercentUserModeCPU = 0,
                         BufferedIO = 0,
                         DirectIO = 0,
                         MscpIO = 0,
                         PageFaults = 0,
                         HardPageFaults = 0;

   register unsigned long  tmp;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "SummarizeRecord()\n");

   /* usually (though not unconditionally) these remain constant! */
   if (SpiRecord.NumberOfCPUs > NumberOfCPUs)
      NumberOfCPUs = SpiRecord.NumberOfCPUs;
   if (SpiRecord.SystemMemoryMBytes > SystemMemoryMBytes)
      SystemMemoryMBytes = SpiRecord.SystemMemoryMBytes;
   if (SpiRecord.PageSpaceMBytes > PageSpaceMBytes)
      PageSpaceMBytes = SpiRecord.PageSpaceMBytes;

   /*
      The percentage CPU values VERY OCCASIONALLY get slightly above
      100%.  I attribute this behaviour to slight inconsistancies
      between obtaining system times and actually obtaining the CPU
      usage data, i.e. to the delta-time used to calculate the percentage.
      I chose to "massage" the data at the display end rather than the
      collection/recording end so that this behaviour could be monitored
      by using the "dump" facility to examine the actual data.
   */
   if (SpiRecord.PercentCPU > 100) SpiRecord.PercentCPU = 100;
   if (SpiRecord.PeakPercentCPU > 100) SpiRecord.PeakPercentCPU = 100;
   if (SpiRecord.PercentUserModeCPU > 100)
      SpiRecord.PercentUserModeCPU = 100;
   if (SpiRecord.PeakPercentUserModeCPU > 100)
      SpiRecord.PeakPercentUserModeCPU = 100;

   /* peaks are always taken as absolutes! */
   if (SpiRecord.PeakPercentCPU > PeakPercentCPU)
      PeakPercentCPU = SpiRecord.PeakPercentCPU;
   if (SpiRecord.PeakPercentUserModeCPU > PeakPercentUserModeCPU)
      PeakPercentUserModeCPU = SpiRecord.PeakPercentUserModeCPU;
   if (SpiRecord.PeakBufferedIO > PeakBufferedIO)
      PeakBufferedIO = SpiRecord.PeakBufferedIO;
   if (SpiRecord.PeakDirectIO > PeakDirectIO)
      PeakDirectIO = SpiRecord.PeakDirectIO;
   if (SpiRecord.PeakMscpIO > PeakMscpIO)
      PeakMscpIO = SpiRecord.PeakMscpIO;
   if (SpiRecord.PeakPageFaults > PeakPageFaults)
      PeakPageFaults = SpiRecord.PeakPageFaults;
   if (SpiRecord.PeakHardPageFaults > PeakHardPageFaults)
      PeakHardPageFaults = SpiRecord.PeakHardPageFaults;

   if (RecordSampleRate > 1)
   {
      /* X axis compression, i.e. less than one record per plot point */
      PercentCPU += SpiRecord.PercentCPU;
      PercentUserModeCPU += SpiRecord.PercentUserModeCPU;
      NumberOfProcesses += SpiRecord.NumberOfProcesses;
      SystemMemoryPercentInUse += SpiRecord.SystemMemoryPercentInUse;
      PageSpacePercentInUse += SpiRecord.PageSpacePercentInUse;
      BufferedIO += SpiRecord.BufferedIO;
      DirectIO += SpiRecord.DirectIO;
      MscpIO += SpiRecord.MscpIO;
      PageFaults += SpiRecord.PageFaults;
      HardPageFaults += SpiRecord.HardPageFaults;

      if (++RecordCount < RecordSampleRate) return;

      /* average by dividing the accumlated values by the record sample rate */
      SpiRecord.PercentCPU = PercentCPU / RecordCount;
      SpiRecord.PercentUserModeCPU = PercentUserModeCPU / RecordCount;
      SpiRecord.NumberOfProcesses = NumberOfProcesses / RecordCount;
      SpiRecord.SystemMemoryPercentInUse =
         SystemMemoryPercentInUse / RecordCount;
      SpiRecord.PageSpacePercentInUse = PageSpacePercentInUse / RecordCount;
      SpiRecord.BufferedIO = BufferedIO / RecordCount;
      SpiRecord.DirectIO = DirectIO / RecordCount;
      SpiRecord.MscpIO = MscpIO / RecordCount;
      SpiRecord.PageFaults = PageFaults / RecordCount;
      SpiRecord.HardPageFaults = HardPageFaults / RecordCount;

      /* reset the accumulators to zero */
      RecordCount =
      PercentCPU = PercentUserModeCPU = NumberOfProcesses =
      SystemMemoryPercentInUse = PageSpacePercentInUse =
      BufferedIO = DirectIO = MscpIO =
      PageFaults = HardPageFaults = 0;
   }

   /* this will need to be divided by the number of records processed */
   AvePercentCPU += SpiRecord.PercentCPU;

   /* this will need to be divided by the number of records processed */
   AvePercentUserModeCPU += SpiRecord.PercentUserModeCPU;

   /* this will need to be divided by the number of records processed */
   AveNumberOfProcesses += SpiRecord.NumberOfProcesses;
   if (SpiRecord.NumberOfProcesses > MaxNumberOfProcesses)
      MaxNumberOfProcesses = SpiRecord.NumberOfProcesses;

   /* this will need to be divided by the number of records processed */
   AveSystemMemoryPercentInUse += SpiRecord.SystemMemoryPercentInUse;
   if (SpiRecord.SystemMemoryPercentInUse > MaxSystemMemoryPercentInUse)
      MaxSystemMemoryPercentInUse = SpiRecord.SystemMemoryPercentInUse;

   /* this will need to be divided by the number of records processed */
   AvePageSpacePercentInUse += SpiRecord.PageSpacePercentInUse;
   if (SpiRecord.PageSpacePercentInUse > MaxPageSpacePercentInUse)
      MaxPageSpacePercentInUse = SpiRecord.PageSpacePercentInUse;

   AveBufferedIO += (tmp = SpiRecord.BufferedIO / 60);
   if (tmp > MaxAveBufferedIO) MaxAveBufferedIO = tmp;

   AveDirectIO += (tmp = SpiRecord.DirectIO / 60);
   if (tmp > MaxAveDirectIO) MaxAveDirectIO = tmp;

   AveMscpIO += (tmp = SpiRecord.MscpIO / 60);
   if (tmp > MaxAveMscpIO) MaxAveMscpIO = tmp;

   AvePageFaults += (tmp = SpiRecord.PageFaults / 60);
   if (tmp > MaxAvePageFaults) MaxAvePageFaults = tmp;

   AveHardPageFaults += (tmp = SpiRecord.HardPageFaults / 60);
   if (tmp > MaxAveHardPageFaults) MaxAveHardPageFaults = tmp;
}

/*****************************************************************************/
/*
Set the Y axis scaling factor and maximum value represented on the X axis 
according to the specified value (maximum out of the to-be plotted data).
*/

SetScaling (int  Maximum)

{
   if (Debug) fprintf (stdout, "SetScaling() %d\n", Maximum);

   if (Maximum <= 10)
   {
      ScalingFactorY = 10.0;
      PlotScaleY = 10;
   }
   else
   if (Maximum <= 25)
   {
      ScalingFactorY = 4.0;
      PlotScaleY = 25;
   }
   else
   if (Maximum <= 50)
   {
      ScalingFactorY = 2.0;
      PlotScaleY = 50;
   }
   else
   if (Maximum <= 100)
   {
      ScalingFactorY = 1.0;
      PlotScaleY = 100;
   }
   else
   if (Maximum <= 250)
   {
      ScalingFactorY = 0.4;
      PlotScaleY = 250;
   }
   else
   if (Maximum <= 500)
   {
      ScalingFactorY = 0.2;
      PlotScaleY = 500;
   }
   else
   if (Maximum <= 1000)
   {
      ScalingFactorY = 0.1;
      PlotScaleY = 1000;
   }
   else
   if (Maximum <= 2500)
   {
      ScalingFactorY = 0.04;
      PlotScaleY = 2500;
   }
   else
   if (Maximum <= 5000)
   {
      ScalingFactorY = 0.02;
      PlotScaleY = 5000;
   }
   else
   if (Maximum <= 10000)
   {
      ScalingFactorY = 0.01;
      PlotScaleY = 10000;
   }
   else
   if (Maximum <= 25000)
   {
      ScalingFactorY = 0.004;
      PlotScaleY = 25000;
   }
   else
   if (Maximum <= 50000)
   {
      ScalingFactorY = 0.002;
      PlotScaleY = 50000;
   }
   else
   if (Maximum <= 100000)
   {
      ScalingFactorY = 0.001;
      PlotScaleY = 100000;
   }
   else
   {
      ScalingFactorY = 0.0001;
      PlotScaleY = 1000000;
   }

   if (Debug) fprintf (stdout, "ScalingFactorY: %f\n", ScalingFactorY);
}

/*****************************************************************************/
/*
The application is being called to plot a graph of the data of the specified 
node in the specified time range, and generate a GIF image of that graph and 
return it to the browser.
*/

GraphData ()

{
   register int  Xcnt, Ycnt, AtX, AtY;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphData()\n");

   /*
      Data processing for graphs can take some time, often two passes.
      Let the client know something's happening by returning the HTTP
      header as soon as possible.
   */
   fputs (Http200GifHeader, HttpOut);
   fflush (HttpOut);

   SizeOfPlotX = NumberOfHours * 60 * XMag / RecordSampleRate;
   SizeOfPlotY = 100 * YMag;
   if (Debug) fprintf (stdout, "size %d,%d\n", SizeOfPlotX, SizeOfPlotY);

   HaveDataY = SizeOfMarginY / 6;

   PlotAllocate (SizeOfPlotX + SizeOfMarginX*3,
                 SizeOfPlotY + SizeOfMarginY + SizeOfMarginY/2);

   if (ProvidePercentCPU || ProvidePercentUserModeCPU)
   {
      /* no scaling to be done for the X axis so plot it now */
      PlotAxes (0);
      /* process specified data calling 'GraphRecordCPU()' for each record */
      ProcessDataFiles (&GraphRecordCPU, true);
   }
   else
   if (ProvideMemory)
   {
      /* no scaling to be done for the X axis so plot it now */
      PlotAxes (0);
      /* process specified data calling 'GraphRecordMemory()' for each record */
      ProcessDataFiles (&GraphRecordMemory, true);
   }
   else
   if (ProvideBufferedIO || ProvidePeakBufferedIO ||
       ProvideDirectIO ||  ProvidePeakDirectIO ||
       ProvideMscpIO || ProvidePeakMscpIO ||
       ProvidePageFaults || ProvidePeakPageFaults  ||
       ProvideHardPageFaults || ProvidePeakHardPageFaults ||
       ProvideProcesses)
   {
      /* summarize the data to determine the required Y axis */
      SummarizeData ();

      /* set the scaling factor for the Y axis */
      if (ProvideBufferedIO)
         if (IncludePeak)
            SetScaling (PeakBufferedIO);
         else
            SetScaling (MaxAveBufferedIO);
      else 
      if (ProvidePeakBufferedIO)
         SetScaling (PeakBufferedIO);
      else 
      if (ProvideDirectIO)
         if (IncludePeak)
            SetScaling (PeakDirectIO);
         else
            SetScaling (MaxAveDirectIO);
      else 
      if (ProvidePeakDirectIO)
         SetScaling (PeakDirectIO);
      else 
      if (ProvideMscpIO)
         if (IncludePeak)
            SetScaling (PeakMscpIO);
         else
            SetScaling (MaxAveMscpIO);
      else 
      if (ProvidePeakMscpIO)
         SetScaling (PeakMscpIO);
      else 
      if (ProvidePageFaults)
         if (IncludePeak)
            SetScaling (PeakPageFaults);
         else
            SetScaling (MaxAvePageFaults);
      else 
      if (ProvidePeakPageFaults)
         SetScaling (PeakPageFaults);
      else 
      if (ProvideHardPageFaults)
         if (IncludePeak)
            SetScaling (PeakHardPageFaults);
         else
            SetScaling (MaxAveHardPageFaults);
      else 
      if (ProvidePeakHardPageFaults)
         SetScaling (PeakHardPageFaults);
      else 
      if (ProvideProcesses)
         SetScaling (MaxNumberOfProcesses);

      /* plot the X and Y axes after summarizing the data and setting scaling */
      PlotAxes (0);

      if (ProvideProcesses)
      {
         /* process data calling 'GraphRecordProcesses()' for each record */
         ProcessDataFiles (&GraphRecordProcesses, true);
      }
      else
      {
         /* process data calling 'GraphRecordRange()' for each record */
         ProcessDataFiles (&GraphRecordRange, true);
      }
   }
   else
   {
      ErrorGeneral ("Internal error.", __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   /* if the current time is within the graphed X axis timescale */
   if (CurrentMinuteFromStartOfPeriod < NumberOfHours * 60)
   {
      /* place a horizontal mark at the end of the data available line */
      AtX = (CurrentMinuteFromStartOfPeriod / RecordSampleRate) * XMag;
      AtY = SizeOfMarginY / 4;
      for (Xcnt = 0; Xcnt < XMag; Xcnt++)
         for (Ycnt = -3; Ycnt < 3 + YMag; Ycnt++)
            PlotPixel (0, AtX+Xcnt, HaveDataY+Ycnt, SizeOfMarginX, 0);
   }

   /* generate and return the GIF image to the browser */
   GifImage ("");
}

/*****************************************************************************/
/*
Plot a CPU record.  Either total or user-mode usage, as a line from zero to 
the percentage value uasage.  Optionally and additionally, plot peak total or 
user-mode usage, or total usage (for use with the user-mode line plot) as a 
single point.  Plot has a Y axis fixed to representPeak 100 percent.
*/ 

GraphRecordCPU ()

{
   static int  RecordCount = 0;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphRecordCPU()\n");

   /*
      The percentage CPU values VERY OCCASIONALLY get slightly above
      100%.  I attribute this behaviour to slight inconsistancies
      between obtaining system times and actually obtaining the CPU
      usage data, i.e. to the delta-time used to calculate the percentage.
      I chose to "massage" the data at the display end rather than the
      collection/recording end so that this behaviour could be monitored
      by using the "dump" facility to examine the actual data.
   */
   if (SpiRecord.PercentCPU > 100) SpiRecord.PercentCPU = 100;
   if (SpiRecord.PeakPercentCPU > 100) SpiRecord.PeakPercentCPU = 100;
   if (SpiRecord.PercentUserModeCPU > 100) SpiRecord.PercentUserModeCPU = 100;
   if (SpiRecord.PeakPercentUserModeCPU > 100)
      SpiRecord.PeakPercentUserModeCPU = 100;

   if (RecordSampleRate > 1)
   {
      /* X axis compression, i.e. less than one record per plot point */
      PercentCPU += SpiRecord.PercentCPU;
      PercentUserModeCPU += SpiRecord.PercentUserModeCPU;
      /* get the maximum of the peak values read */
      if (SpiRecord.PeakPercentCPU > PeakPercentCPU)
          PeakPercentCPU = SpiRecord.PeakPercentCPU;
      if (SpiRecord.PeakPercentUserModeCPU > PeakPercentUserModeCPU)
          PeakPercentUserModeCPU = SpiRecord.PeakPercentUserModeCPU;

      if (++RecordCount < RecordSampleRate) return;

      /* get the average of the values read */
      SpiRecord.PercentCPU = PercentCPU / RecordCount;
      SpiRecord.PercentUserModeCPU = PercentUserModeCPU / RecordCount;
      /* maximum of the peak values read */
      SpiRecord.PeakPercentCPU = PeakPercentCPU;
      SpiRecord.PeakPercentUserModeCPU = PeakPercentUserModeCPU;

      PercentCPU = PercentUserModeCPU = PeakPercentCPU =
         PeakPercentUserModeCPU = RecordCount = 0;
   }

   /*********************/
   /* plot the value(s) */
   /*********************/

   {
      /* push these registers onto the stack only if necessary! */
      register int  Xcnt, Ycnt, ToY, AtX, AtY;

      AtX = (NumberOfMinutesIntoData / RecordSampleRate) * XMag;

      /* plot the fact we have data for this minute */
      for (Xcnt = 0; Xcnt < XMag; Xcnt++)
         for (Ycnt = 0; Ycnt < YMag; Ycnt++)
            PlotPixel (0, AtX+Xcnt, HaveDataY+Ycnt, SizeOfMarginX, 0);

      if (ProvidePercentCPU)
         ToY = SpiRecord.PercentCPU * YMag;
      else
      if (ProvidePercentUserModeCPU)
         ToY = SpiRecord.PercentUserModeCPU * YMag;

      /* plot as line from zero to whatever the value is */
      for (Xcnt = 0; Xcnt < XMag; Xcnt++)
         PlotLineY (0, 0, ToY, AtX+Xcnt, SizeOfMarginX, SizeOfMarginY);

      /* any extra data to plot? (is plotted in single pixels) */

      if (IncludePeak)
      {
         /* CPU peak usage has been requested to be included */
         if (ProvidePercentCPU)
            AtY = SpiRecord.PeakPercentCPU * YMag;
         else
         if (ProvidePercentUserModeCPU)
            AtY = SpiRecord.PeakPercentUserModeCPU * YMag;
         else;
      }
      else
      if (IncludeTotal)
      {
         /* CPU total usage has been requested to be included */
         AtY = SpiRecord.PercentCPU * YMag;
      }
      else
      {
         /* no, nothing extra to plot! */
         return;
      }

      /* plot the extra value as a point */
      for (Xcnt = 0; Xcnt < XMag; Xcnt++)
         for (Ycnt = 0; Ycnt < YMag; Ycnt++)
            PlotPixel (PLOT_XOR, AtX+Xcnt, AtY+Ycnt,
                       SizeOfMarginX, SizeOfMarginY);
   }
}

/*****************************************************************************/
/*
Plot page space usage as a line from zero to percentage used.  Plot physical 
memory as a point at the percentage used.  Plot has a Y axis fixed to 
represent 100 percent.
*/ 

GraphRecordMemory ()

{
   static int  RecordCount = 0;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphRecordMemory()\n");

   if (RecordSampleRate > 1)
   {
      /* X axis compression, i.e. less than one record per plot point */
      SystemMemoryPercentInUse += SpiRecord.SystemMemoryPercentInUse;
      PageSpacePercentInUse += SpiRecord.PageSpacePercentInUse;

      if (++RecordCount < RecordSampleRate) return;

      /* get the average of the values read */
      SpiRecord.SystemMemoryPercentInUse =
         SystemMemoryPercentInUse / RecordCount;
      SpiRecord.PageSpacePercentInUse =
         PageSpacePercentInUse / RecordCount;

      SystemMemoryPercentInUse = PageSpacePercentInUse = RecordCount = 0;
   }

   /*******************/
   /* plot the values */
   /*******************/

   {
      /* push these registers onto the stack only if necessary! */
      register int  Xcnt, Ycnt, ToY, AtX, AtY;

      AtX = (NumberOfMinutesIntoData / RecordSampleRate) * XMag;

      /* plot the fact we have data for this minute */
      for (Xcnt = 0; Xcnt < XMag; Xcnt++)
         for (Ycnt = 0; Ycnt < YMag; Ycnt++)
            PlotPixel (0, AtX+Xcnt, HaveDataY+Ycnt, SizeOfMarginX, 0);

      ToY = SpiRecord.PageSpacePercentInUse * YMag;
      /* plot as line from zero to whatever the value is */
      for (Xcnt = 0; Xcnt < XMag; Xcnt++)
         PlotLineY (0, 0, ToY, AtX+Xcnt, SizeOfMarginX, SizeOfMarginY);

      AtY = SpiRecord.SystemMemoryPercentInUse * YMag;
      /* plot a point */
      for (Xcnt = 0; Xcnt < XMag; Xcnt++)
         for (Ycnt = 0; Ycnt < YMag; Ycnt++)
            PlotPixel (PLOT_XOR, AtX+Xcnt, AtY+Ycnt,
                       SizeOfMarginX, SizeOfMarginY);
   }
}

/*****************************************************************************/
/*
Plot the number oif processes on the system as plot-points (forming a
semi-continuous line).  Requires a variable range for the Y axis.  That is data
that  can vary considerably in the maximum value represented on the Y axis. 
Employs  a previously set variable 'ScalingFactorY' to adjust the plot values
to fall  within the Y axis.  'ScalingFactorY' must be set after scanning all of
the  specified data to determine the maximum value.
*/ 

GraphRecordProcesses ()

{
   static int  RecordCount = 0;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphRecordProcesses()\n");

   if (RecordSampleRate > 1)
   {
      /* X axis compression, i.e. less than one record per plot point */
      NumberOfProcesses += SpiRecord.NumberOfProcesses;
      /* return if still averaging according to sample rate */
      if (++RecordCount < RecordSampleRate) return;
      /* get the average of the values read */
      SpiRecord.NumberOfProcesses = NumberOfProcesses / RecordCount;
      /* reset the accumulator */
      NumberOfProcesses = RecordCount = 0;
   }

   /*******************/
   /* plot the value */
   /*******************/

   {
      /* push these registers onto the stack only if necessary! */
      register int  Xcnt, Ycnt, AtX, AtY;

      AtX = (NumberOfMinutesIntoData / RecordSampleRate) * XMag;

      /* plot the fact we have data for this minute */
      for (Xcnt = 0; Xcnt < XMag; Xcnt++)
         for (Ycnt = 0; Ycnt < YMag; Ycnt++)
            PlotPixel (0, AtX+Xcnt, HaveDataY+Ycnt, SizeOfMarginX, 0);

      AtY = SpiRecord.NumberOfProcesses * ScalingFactorY * YMag;
      /* plot a point */
      for (Xcnt = 0; Xcnt < XMag; Xcnt++)
         for (Ycnt = 0; Ycnt < YMag; Ycnt++)
            PlotPixel (PLOT_XOR, AtX+Xcnt, AtY+Ycnt,
                       SizeOfMarginX, SizeOfMarginY);
   }
}

/*****************************************************************************/
/*
Plot values that require a variable range for the Y axis.  That is data that 
can vary considerably in the maximum value represented on the Y axis.  Employs 
a previously set variable 'ScalingFactorY' to adjust the plot values to fall 
within the Y axis.  'ScalingFactorY' must be set after scanning all of the 
specified data to determine the maximum value.
*/

GraphRecordRange ()

{
   static int  RecordCount = 0;
   static unsigned long  StaticPeakValue = 0,
                         StaticValue = 0;

   register unsigned long  PeakValue,
                           Value;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GraphRecordRange()\n");

   if (ProvideBufferedIO)
   {
      Value = SpiRecord.BufferedIO;
      PeakValue = SpiRecord.PeakBufferedIO;
   }
   else
   if (ProvidePeakBufferedIO)
   {
      Value = PeakValue = SpiRecord.PeakBufferedIO;
   }
   else
   if (ProvideDirectIO)
   {
      Value = SpiRecord.DirectIO;
      PeakValue = SpiRecord.PeakDirectIO;
   }
   else
   if (ProvidePeakDirectIO)
   {
      Value = PeakValue = SpiRecord.PeakDirectIO;
   }
   else
   if (ProvideMscpIO)
   {
      Value = SpiRecord.MscpIO;
      PeakValue = SpiRecord.PeakMscpIO;
   }
   else
   if (ProvidePeakMscpIO)
   {
      Value = PeakValue = SpiRecord.PeakMscpIO;
   }
   else
   if (ProvidePageFaults)
   {
      Value = SpiRecord.PageFaults;
      PeakValue = SpiRecord.PeakPageFaults;
   }
   else
   if (ProvidePeakPageFaults)
   {
      Value = PeakValue = SpiRecord.PeakPageFaults;
   }
   else
   if (ProvideHardPageFaults)
   {
      Value = SpiRecord.HardPageFaults;
      PeakValue = SpiRecord.PeakHardPageFaults;
   }
   else
   if (ProvidePeakHardPageFaults)
   {
      Value = PeakValue = SpiRecord.PeakHardPageFaults;
   }

   if (RecordSampleRate > 1)
   {
      /* X axis compression, i.e. less than one record per plot point */
      StaticValue += Value; 
      /* get the maximum of the peak values read */
      if (PeakValue > StaticPeakValue) StaticPeakValue = PeakValue; 

      if (++RecordCount < RecordSampleRate) return;

      /* get the average of the values read */
      Value = StaticValue / RecordCount;
      /* maximum of the peak values read */
      PeakValue = StaticPeakValue;

      StaticValue = StaticPeakValue = RecordCount = 0;
   }
   if (Debug) fprintf (stdout, "Value: %d PeakValue: %d\n", Value, PeakValue);

   /*********************/
   /* plot the value(s) */
   /*********************/

   {
      /* push these registers onto the stack only if necessary! */
      register int  Xcnt, Ycnt, ToY, AtX, AtY;

      AtX = (NumberOfMinutesIntoData / RecordSampleRate) * XMag;

      /* plot the fact we have data for this minute */
      for (Xcnt = 0; Xcnt < XMag; Xcnt++)
         for (Ycnt = 0; Ycnt < YMag; Ycnt++)
            PlotPixel (0, AtX+Xcnt, HaveDataY+Ycnt, SizeOfMarginX, 0);

      if (ProvideBufferedIO ||
          ProvideDirectIO ||
          ProvideMscpIO ||
          ProvidePageFaults ||
          ProvideHardPageFaults)
         ToY = (int)((float)(Value / 60) * ScalingFactorY) * YMag;
      else
         ToY = (int)((float)PeakValue * ScalingFactorY) * YMag;

      /* plot as line from zero to whatever the value is */
      for (Xcnt = 0; Xcnt < XMag; Xcnt++)
         PlotLineY (0, 0, ToY, AtX+Xcnt, SizeOfMarginX, SizeOfMarginY);

      if (IncludePeak)
         AtY = (int)((float)PeakValue * ScalingFactorY) * YMag;
      else
      {
         /* no, nothing extra to plot! */
         return;
      }

      /* plot peak value as a point */
      for (Xcnt = 0; Xcnt < XMag; Xcnt++)
         for (Ycnt = 0; Ycnt < YMag; Ycnt++)
            PlotPixel (PLOT_XOR, AtX+Xcnt, AtY+Ycnt,
                       SizeOfMarginX, SizeOfMarginY);
   }
}

/*****************************************************************************/
/*
*/

ListProcessedData ()

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ListProcessedData()\n |%s|");

   fprintf (HttpOut,
"%s\
<HTML>\n\
<!-- SoftwareID: %s -->\n\
<TITLE>HyperSpi Processed Data</TITLE>\n\
<H1>%sProcessed Data</H1>\n",
   Http200Header, SoftwareID, IconImg);

   ProcessDataFiles (&ListProcessedRecord, true);

   if (DataFilesProcessedCount)
      fputs ("</PRE>\n", HttpOut);
   else
      fputs ("<P>\n<HR>\n<P>\n<I>No matching data files.</I>\n",
             HttpOut);
   fputs ("<P>\n<HR>\n</HTML>\n", HttpOut);
}

/*****************************************************************************/
/*
*/

ListProcessedRecord ()

{
   static int  PrevDay = -1,
               PrevHour = -1;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ListProcessedRecord()\n");

   if (DataDay != PrevDay)
   {
      if (PrevDay != -1) fputs ("</PRE>\n", HttpOut);
      PrevDay = DataDay;
      fprintf (HttpOut, 
"<P>\n\
<HR>\n\
<P>\n\
<H2>%s %02.02d/%02.02d/%02.02d<H2>\n\
<P>\n\
<PRE>",
      FormNode, DataDay, DataMonth, DataYear);
   }

   if (SpiRecord.Hour != PrevHour)
   {
      if (PrevHour != -1) fputs ("\n", HttpOut);
      PrevHour = SpiRecord.Hour;
      fprintf (HttpOut,
"hh:mm  CPU usr  mem pge  b-IO peak  d-IO peak  mscp peak  \
 flts  peak  hard peak\n\
-----  --- ---  --- ---  ---- ----  ---- ----  ---- ----  \
----- -----  ---- ----\n");
   }

   fprintf (HttpOut,
"%02.02d:%02.02d  %3d %3d  %3d %3d  \
%4d %4d  %4d %4d  %4d %4d  %5d %5d  %4d %4d\n",
   SpiRecord.Hour, SpiRecord.Minute,
   SpiRecord.PercentCPU, SpiRecord.PercentUserModeCPU,
   SpiRecord.SystemMemoryPercentInUse,
   SpiRecord.PageSpacePercentInUse,
   SpiRecord.BufferedIO / 60, SpiRecord.PeakBufferedIO,
   SpiRecord.DirectIO / 60, SpiRecord.PeakDirectIO,
   SpiRecord.MscpIO / 60, SpiRecord.PeakMscpIO,
   SpiRecord.PageFaults / 60, SpiRecord.PeakPageFaults,
   SpiRecord.HardPageFaults / 60, SpiRecord.PeakHardPageFaults);
}

/*****************************************************************************/
/*
*/

DumpData ()

{
   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DumpData()\n");

   fprintf (HttpOut,
"%s\
<HTML>\n\
<!-- SoftwareID: %s -->\n\
<TITLE>HyperSpi Data Dump</TITLE>\n\
<H1>%sData Dump</H1>\n",
   Http200Header, SoftwareID, IconImg);

   ProcessDataFiles (&DumpRecord, true);

   if (DataFilesProcessedCount)
      fputs ("</PRE>\n", HttpOut);
   else
      fputs ("<P>\n<HR>\n<P>\n<I>No matching data files.</I>\n",
             HttpOut);
   fputs ("<P>\n<HR>\n</HTML>\n", HttpOut);
}

/*****************************************************************************/
/*
*/

DumpRecord ()

{
   static int  PrevDay = -1,
               PrevHour = -1;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "DumpRecord()\n");

   if (DataDay != PrevDay)
   {
      if (PrevDay != -1) fputs ("</PRE>\n", HttpOut);
      PrevDay = DataDay;
      fprintf (HttpOut, 
"<P>\n\
<HR>\n\
<P>\n\
<H2>%s %02.02d/%02.02d/%02.02d<H2>\n\
<P>\n\
<PRE>%s\n",
      FormNode, DataDay, DataMonth, DataYear,
      DataFileName);
   }

   if (SpiRecord.Hour != PrevHour)
   {
      fputs ("\n", HttpOut);
      PrevHour = SpiRecord.Hour;
      fprintf (HttpOut,
"hh:mm  prc  CPU  pk  usr  pk  mem pge   buf-IO    peak  \
 dir-IO    peak  mscp-IO    peak   faults    peak   hard  peak\n\
-----  ---  --- ---  --- ---  --- ---  ------- -------  \
------- -------  ------- -------  ------- -------  ----- -----\n");
   }

   fprintf (HttpOut,
"%02.02d:%02.02d  %3d  %3d %3d  %3d %3d  %3d %3d  \
%7d %7d  %7d %7d  %7d %7d  %7d %7d  %5d %5d\n",
   SpiRecord.Hour, SpiRecord.Minute,
   SpiRecord.NumberOfProcesses,
   SpiRecord.PercentCPU, SpiRecord.PeakPercentCPU,
   SpiRecord.PercentUserModeCPU, SpiRecord.PeakPercentUserModeCPU,
   SpiRecord.SystemMemoryPercentInUse,
   SpiRecord.PageSpacePercentInUse,
   SpiRecord.BufferedIO, SpiRecord.PeakBufferedIO,
   SpiRecord.DirectIO, SpiRecord.PeakDirectIO,
   SpiRecord.MscpIO, SpiRecord.PeakMscpIO,
   SpiRecord.PageFaults, SpiRecord.PeakPageFaults,
   SpiRecord.HardPageFaults, SpiRecord.PeakHardPageFaults);
}

/*****************************************************************************/
/*
The first parameter to this function is the address (a pointer to) the 
function used to process each file name found.
*/ 
 
ProcessDataFiles
(
int (*ProcessFileFunction)(),
boolean OpenDataFile
)
{
   register char  *cptr, *sptr;

   int  status;
   char  ExpandedFileName [256],
         Scratch [256];
   struct FAB  SearchFab;
   struct RAB  SearchRab;
   struct NAM  SearchNam;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessDataFiles() |%s|\n", DataFileSpec);

   DataFilesFoundCount = DataFilesProcessedCount =
      DataRecordsReadCount = DataRecordsProcessedCount = 0;

   SearchFab = cc$rms_fab;
   SearchFab.fab$l_fna = DataFileSpec;
   SearchFab.fab$b_fns = DataFileSpecLength;
   SearchFab.fab$l_fop = FAB$M_NAM;
   SearchFab.fab$l_nam = &SearchNam;

   SearchNam = cc$rms_nam;
   SearchNam.nam$l_esa = ExpandedFileName;
   SearchNam.nam$b_ess = sizeof(ExpandedFileName)-1;
   SearchNam.nam$l_rsa = DataFileName;
   SearchNam.nam$b_rss = sizeof(DataFileName)-1;

   if (VMSnok (status = sys$parse (&SearchFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$parse() %%X%08.08X\n", status);
      ErrorVmsStatus (status, DataFileSpec, DataFileSpec, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   while (VMSok (status = sys$search (&SearchFab, 0, 0)))
   {
      DataFilesFoundCount++;

      SearchNam.nam$l_ver[SearchNam.nam$b_ver] = '\0';
      DataFileNameLength = SearchNam.nam$b_rsl;
      if (Debug) fprintf (stdout, "DataFileName |%s|\n", DataFileName);

      /*
         Pull the node name and time components from the data file name.
         Format: "HYPERSPI_v_node_ddmmyy.DAT"
         ("v" represents the data file version, a single digit number)
      */
      cptr = SearchNam.nam$l_name;
      /* skip "HYPERSPI_version_" */
      while (*cptr && *cptr != '_') cptr++;
      if (*cptr) cptr++;
      while (*cptr && *cptr != '_') cptr++;
      if (*cptr) cptr++;
      /* get the node name this data represents */
      sptr = DataNode;
      while (*cptr && *cptr != '_') *sptr++ = *cptr++;
      *sptr = '\0';
      if (*cptr) cptr++;
      /* get the day, month and year */
      sptr = Scratch;
      if (*cptr) *sptr++ = *cptr++;
      if (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
      DataDay = atoi (Scratch);
      sptr = Scratch;
      if (*cptr) *sptr++ = *cptr++;
      if (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
      DataMonth = atoi (Scratch);
      sptr = Scratch;
      if (*cptr) *sptr++ = *cptr++;
      if (*cptr) *sptr++ = *cptr++;
      *sptr = '\0';
      DataYear = atoi (Scratch);
      if (Debug)
         fprintf (stdout, "data-file-name |%s|%d|%d|%d|\n",
                  DataNode, DataDay, DataMonth, DataYear);

      /* filter on year, month and day */
      if (DataYear < FromYear || DataYear > ToYear) continue;
      if (DataYear == FromYear)
      {
         if (DataMonth < FromMonth) continue;
         if (DataMonth == FromMonth && DataDay < FromDay) continue;
      }
      if (DataYear == ToYear)
      {
         if (DataMonth > ToMonth) continue;
         if (DataMonth == ToMonth && DataDay > ToDay) continue;
      }

      DataFilesProcessedCount++;

      /* can be used to just count the number of matching files! */
      if (ProcessFileFunction == NULL) continue;

      if (OpenDataFile)
         ProcessDataFileRecords (ProcessFileFunction);
      else
         /* by pointer, call the function to process this file name */
         (*ProcessFileFunction) ();
   }
   if (Debug) fprintf (stdout, "sys$search() %%X%08.08X\n", status);

   if (SearchFab.fab$l_sts == RMS$_FNF || SearchFab.fab$l_sts == RMS$_NMF)
      return (SS$_NORMAL);

   ErrorVmsStatus (status, DataFileSpec, DataFileSpec, __FILE__, __LINE__);
   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Open the 'DataFileName', read each record and call a specified function to 
process it, then close the file.  The first parameter to this function is the 
address (a pointer to) the function used to process the SPI data record.
*/ 

ProcessDataFileRecords (int (*ProcessDataFunction)(struct HyperSpiData*))

{
   static long  LibJulianDate = LIB$K_JULIAN_DATE;
   static unsigned short  PrevNumTime [7] = {0,0,0,0,0,0,0};

   int  status;
   unsigned long  JulDate,
                  BinTime [2];
   struct FAB  DataFileFab;
   struct RAB  DataFileRab;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "ProcessDataFileRecords() |%s|\n", DataFileName);

   DataFileFab = cc$rms_fab;
   DataFileFab.fab$b_fac = FAB$M_GET;
   DataFileFab.fab$l_fna = DataFileName;  
   DataFileFab.fab$b_fns = DataFileNameLength;
   DataFileFab.fab$b_shr = FAB$M_SHRGET | FAB$M_SHRPUT;

   if (VMSnok (status = sys$open (&DataFileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      if (status == RMS$_FNF)
      {
         ErrorVmsStatus (status, DataNode, DataFileName, __FILE__, __LINE__);
         exit (SS$_NORMAL);
      }
      ErrorVmsStatus (status, DataNode, DataFileName, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   DataFileRab = cc$rms_rab;
   DataFileRab.rab$l_fab = &DataFileFab;
   /* 2 buffers and read ahead performance option */
   DataFileRab.rab$b_mbf = 2;
   DataFileRab.rab$l_rop = RAB$M_RAH;
   DataFileRab.rab$l_ubf = &SpiRecord;
   DataFileRab.rab$w_usz = sizeof(SpiRecord);

   if (VMSnok (status = sys$connect (&DataFileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&DataFileFab, 0, 0);
      ErrorVmsStatus (status, DataNode, DataFileName, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   while (VMSok (status = sys$get (&DataFileRab, 0, 0)))
   {
      DataRecordsReadCount++;

      /* filter on hour and minute */
      if (SpiRecord.Day == FromDay)
      {
         if (SpiRecord.Hour < FromHour) continue;
         if (SpiRecord.Hour == FromHour && SpiRecord.Minute < FromMinute)
            continue;
      }
      if (SpiRecord.Day == ToDay)
      {
         if (SpiRecord.Hour > ToHour) continue;
         if (SpiRecord.Hour == ToHour && SpiRecord.Minute > ToMinute) continue;
      }

      if (SpiRecord.Hour != PrevNumTime[3] ||
          SpiRecord.Day != PrevNumTime[2] ||
          SpiRecord.Month != PrevNumTime[1] ||
          SpiRecord.Year != PrevNumTime[0])
      {
         PrevNumTime[3] = SpiRecord.Hour;
         PrevNumTime[2] = SpiRecord.Day;
         PrevNumTime[1] = SpiRecord.Month;
         PrevNumTime[0] = SpiRecord.Year;

         lib$cvt_vectim (&PrevNumTime, &BinTime);
         lib$cvt_from_internal_time (&LibJulianDate, &JulDate, &BinTime);
         NumberOfDaysIntoData = JulDate - FromJulianDate;
         StartMinuteOfData = NumberOfDaysIntoData * 1440;
         if (Debug)
            fprintf (stdout,
               "NumberOfDaysIntoData: %d StartMinuteOfData: %d\n",
               NumberOfDaysIntoData, StartMinuteOfData);
      }
      NumberOfMinutesIntoData = StartMinuteOfData +
                                ((SpiRecord.Hour * 60) + SpiRecord.Minute) -
                                StartMinuteOnFirstDay;

      DataRecordsProcessedCount++;

      /* by pointer, call the function to process this SPI data record */
      (*ProcessDataFunction) (&SpiRecord);
   }

   sys$close (&DataFileFab, 0, 0);

   if (status == RMS$_EOF) status = SS$_NORMAL;
   if (VMSnok (status))
   {
      ErrorVmsStatus (status, DataNode, DataFileName, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }
}

/*****************************************************************************/
/*
The specified parameter is the name of the file, excluding directory and file 
type, just the name.  Construct a full HTML file name and open it, read each 
record just passing the contents into the HTTP output stream, the close the 
file.  If the file cannot be found the status RMS$_FNF is returned, all other 
errors are reported and the image exits.
*/ 

IncludeFile (char *Name)

{
   int  status;
   char  FileName [256],
         Line [256],
         Scratch [256];
   struct FAB  FileFab;
   struct RAB  FileRab;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "IncludeFile() |%s|\n", Name);

   FileFab.fab$b_fns = sprintf (FileName, "%s%s.HTML",
                                HyperSpiDirectory, Name);
   if (Debug) fprintf (stdout, "FileName |%s|\n", FileName);

   FileFab = cc$rms_fab;
   FileFab.fab$b_fac = FAB$M_GET;
   FileFab.fab$l_fna = FileName;  
   FileFab.fab$b_shr = FAB$M_SHRGET;

   if (VMSnok (status = sys$open (&FileFab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$open() %%X%08.08X\n", status);
      if (status == RMS$_FNF) return (status);
      ErrorVmsStatus (status, Name, FileName, __FILE__, __LINE__); 
      exit (SS$_NORMAL);
   }

   FileRab = cc$rms_rab;
   FileRab.rab$l_fab = &FileFab;
   /* 2 buffers and read ahead performance option */
   FileRab.rab$b_mbf = 2;
   FileRab.rab$l_rop = RAB$M_RAH;
   FileRab.rab$l_ubf = Line;
   FileRab.rab$w_usz = sizeof(Line)-1;

   if (VMSnok (status = sys$connect (&FileRab, 0, 0)))
   {
      if (Debug) fprintf (stdout, "sys$connect() %%X%08.08X\n", status);
      sys$close (&FileFab, 0, 0);
      ErrorVmsStatus (status, Name, FileName, __FILE__, __LINE__);
      exit (SS$_NORMAL);
   }

   while (VMSok (status = sys$get (&FileRab, 0, 0)))
   {
      Line[FileRab.rab$w_rsz++] = '\r';
      Line[FileRab.rab$w_rsz++] = '\n';
      Line[FileRab.rab$w_rsz] = '\0';
      if (Debug) fprintf (stdout, "Line |%s|\n", Line);
      fputs (Line, HttpOut);
   }

   sys$close (&FileFab, 0, 0);

   if (status == RMS$_EOF) return (status);

   ErrorVmsStatus (status, Name, FileName, __FILE__, __LINE__);
   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
Return a pointer to a unique number built up of time components.  Used to 
place a unique component into the URL of a graph GIF, ensuring a cached 
version is not retrieved.
*/

char* UniqueNumber ()

{
   static char  String [16];

   unsigned long  BinTime [2];
   unsigned short  NumTime [7];

   sys$gettim (&BinTime);
   sys$numtim (&NumTime, &BinTime);

   sprintf (String, "%d%d%d%d%d%d%d",
            NumTime[0] % 10, NumTime[1], NumTime[2], NumTime[3],
            NumTime[4], NumTime[5], NumTime[6]);
   return (String);
}

/*****************************************************************************/
/*
*/

boolean GetCgiVariable
(
char *VariableName,
char *VariableValue,
int VariableValueSize
)
{
   static $DESCRIPTOR (VariableNameDsc, "");
   static $DESCRIPTOR (VariableValueDsc, "");

   int  status;
   unsigned short  Length;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetCgiVariable() |%s|\n", VariableName);

   VariableNameDsc.dsc$w_length = strlen(VariableName);
   VariableNameDsc.dsc$a_pointer = VariableName;
   VariableValueDsc.dsc$w_length = VariableValueSize-1;
   VariableValueDsc.dsc$a_pointer = VariableValue;

   if (VMSok (status =
       lib$get_symbol (&VariableNameDsc, &VariableValueDsc, &Length, 0)))
   {
      VariableValue[Length] = '\0';
      if (Debug) fprintf (stdout, "|%s|\n", VariableValue);
      return (true);
   }
   else
   {
      if (Debug) fprintf (stdout, "lib$get_symbol() %%X%08.08X\n", status);
      VariableValue[0] = '\0';
      if (status == LIB$_NOSUCHSYM) return (false);
      exit (status);
   }
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n",
   SoftwareID, cptr, SourceLineNumber, Text);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      exit (status);

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n",
   SoftwareID, cptr, SourceLineNumber, Message, Text, StatusValue, HiddenText);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/


