/*****************************************************************************/
/*
                               calendar.c

CGI-compliant script to produce calendar.


CGI VARIABLE NAMES
------------------
WWW_PATH_INFO           the URL path component
WWW_PATH_TRANSLATED     the VMS equivalent of the URL path component
WWW_QUERY_STRING        any of the form components
WWW_SCRIPT_NAME         the name of the script being executed
WWW_FORM_ACROSS         number of months across page
WWW_FORM_FROMMONTH      beginning with month (1 == jan, 12 = dec)
WWW_FORM_TOMONTH        ending with month
WWW_FORM_OTHERYEAR      explicitly specified year from form
WWW_FORM_YEAR           year selected from selection box in form


BUILD
-----
See BUILD_CALENDAR.COM


VERSION HISTORY
---------------
20-FEB-96  MGD  v1.0.0, quick hack
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "CALENDAR AXP-1.0.0";
#else
   char SoftwareID [] = "CALENDAR VAX-1.0.0";
#endif

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include <descrip.h>
#include <libdef.h>
#include <libdtdef.h>
#include <ssdef.h>
#include <stsdef.h>

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

#define boolean int
#define true 1
#define false 0

/* macro provides NULL pointer if CGI variable does not exist */
#define GetCgiVarIfExists(CharPointer,CgiVariableName) \
   CharPointer = getenv(CgiVariableName)

/* macro provides pointer to empty string even if CGI variable does not exist */
#define GetCgiVar(CharPointer,CgiVariableName) \
   if ((CharPointer = getenv(CgiVariableName)) == NULL) \
       CharPointer = ""; \
   if (Debug) fprintf (stdout, "%s |%s|\n", CgiVariableName, CharPointer);

char  Utility [] = "CALENDAR";

char  Http200Header [] =
"HTTP/1.0 200 Document follows.\r\n\
Content-Type: text/html\r\n\
\r\n";

char  Http404Header [] =
"HTTP/1.0 404 Error report follows.\r\n\
Content-Type: text/html\r\n\
\r\n";

int  DaysInMonth [] = { 0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

char  *DayName [] =
      { "", "Sunday", "Monday", "Tuesday", "Wednesday",
            "Thursday", "Friday", "Saturday", };

char  *MonthName [] =
      { "", "January", "February", "March", "April", "May", "June",
            "July", "August", "September", "October", "November", "December" };

unsigned long  LibDayOfMonth = LIB$K_DAY_OF_MONTH,
               LibDayOfWeek = LIB$K_DAY_OF_WEEK;

boolean  Debug,
         HttpHasBeenOutput;

int  MonthsAcrossPage,
     MonthsOnLine;

char  *CgiFormYearPtr, 
      *CgiFormOtherYearPtr,
      *CgiScriptNamePtr;
      
#define MonthIncrement 23
#define MaxPageLines 8
#define PageLineLength 512
char  PageLines [MaxPageLines][PageLineLength];

FILE  *HttpOut;

/*****************************************************************************/
/*
*/

main ()

{
   /*********/
   /* begin */
   /*********/

   if (getenv ("CALENDAR$DBUG") != NULL)
   {
      Debug = true;
      system ("SHOW SYM WWW_*");
   }

   /* open another output stream so that the '\r' and '\n' are not filtered */
#ifdef __DECC
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "ctx=bin")) == NULL)
      exit (vaxc$errno);
#else
   if ((HttpOut = fopen ("SYS$OUTPUT", "w", "rfm=udf")) == NULL)
      exit (vaxc$errno);
#endif

   /*********************/
   /* get CGI variables */
   /*********************/

   GetCgiVar (CgiScriptNamePtr, "WWW_SCRIPT_NAME");

   /***********************/
   /* execute the request */
   /***********************/

   GetCgiVar (CgiFormOtherYearPtr, "WWW_FORM_OTHERYEAR");
   if (*CgiFormOtherYearPtr)
      Calendar ();
   else
   {
      GetCgiVar (CgiFormYearPtr, "WWW_FORM_YEAR");
      if (*CgiFormYearPtr)
         Calendar ();
      else
         CalendarForm ();
   }

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

CalendarForm ()

{
   int  LastYear,
        Year;
   unsigned long  BinTime [2];
   unsigned short  NumTime [7];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "CalendarForm()\n");

   sys$gettim (&BinTime);
   sys$numtim (&NumTime, &BinTime);

   fprintf (HttpOut,
"%s\
<!-- SoftwareID: %s -->\n\
<TITLE>Calendar</TITLE>\n\
<H1>Calendar</H1>\n\
<HR>\n\
<FORM ACTION=\"%s\">\n\
From <SELECT NAME=FromMonth>\n\
<OPTION VALUE=1 SELECTED> January\n\
<OPTION VALUE=2> February\n\
<OPTION VALUE=3> March\n\
<OPTION VALUE=4> April\n\
<OPTION VALUE=5> May\n\
<OPTION VALUE=6> June\n\
<OPTION VALUE=7> July\n\
<OPTION VALUE=8> August\n\
<OPTION VALUE=9> September\n\
<OPTION VALUE=10> October\n\
<OPTION VALUE=11> November\n\
<OPTION VALUE=12> December\n\
</SELECT>\n\
to <SELECT NAME=ToMonth>\n\
<OPTION VALUE=1> January\n\
<OPTION VALUE=2> February\n\
<OPTION VALUE=3> March\n\
<OPTION VALUE=4> April\n\
<OPTION VALUE=5> May\n\
<OPTION VALUE=6> June\n\
<OPTION VALUE=7> July\n\
<OPTION VALUE=8> August\n\
<OPTION VALUE=9> September\n\
<OPTION VALUE=10> October\n\
<OPTION VALUE=11> November\n\
<OPTION VALUE=12 SELECTED> December\n\
</SELECT>\n\
of <SELECT NAME=Year>\n",
      Http200Header, SoftwareID,
      CgiScriptNamePtr);

   LastYear = (Year = (NumTime[0] / 10) * 10) + 10;
   for (/* above */; Year < LastYear; Year++)
      if (Year == NumTime[0])
         fprintf (HttpOut, "<OPTION VALUE=%d SELECTED>%d\n", Year, Year);
      else
         fprintf (HttpOut, "<OPTION VALUE=%d>%d\n", Year, Year);

   fprintf (HttpOut,
"</SELECT>\n\
<I>( or <INPUT TYPE=text SIZE=4 NAME=OtherYear> )</I>\n\
<P><I>( <SELECT NAME=Across>\n\
<OPTION VALUE=1>1\n\
<OPTION VALUE=2>2\n\
<OPTION VALUE=3 SELECTED>3\n\
<OPTION VALUE=4>4\n\
<OPTION VALUE=5>5\n\
<OPTION VALUE=6>6\n\
<OPTION VALUE=7>7\n\
<OPTION VALUE=8>8\n\
<OPTION VALUE=9>9\n\
<OPTION VALUE=10>10\n\
<OPTION VALUE=11>11\n\
<OPTION VALUE=12 SELECTED>12\n\
</SELECT>\n\
months across page )</I>\n\
<P><INPUT TYPE=submit VALUE=\" generate \">\n\
<INPUT TYPE=reset VALUE=\" reset \">\n\
</FORM>\n\
<P><HR>\n",
      Http200Header, SoftwareID,
      CgiScriptNamePtr,
      NumTime[0],
      NumTime[1],
      NumTime[1]);
}

/****************************************************************************/
/*
*/ 

Calendar ()

{
   register char  *cptr;

   int  status,
        Day,
        FromMonth,
        Month,
        ToMonth,
        Year;
   unsigned long  BinTime [2],
                  DayOfWeek;
   unsigned short  NumTime [7],
                   WeekDayNumTime [7];
   char  Range [32];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "Calendar()\n");

   sys$gettim (&BinTime);
   sys$numtim (&NumTime, &BinTime);

   if (*CgiFormOtherYearPtr)
      Year = atoi(CgiFormOtherYearPtr);
   else
      Year = atoi(CgiFormYearPtr);
   if (Year < 1859 || Year > 9999)
      exit (ErrorGeneral ("Year out of range (1859 to 9999).",
                          __FILE__, __LINE__));

   GetCgiVar (cptr, "WWW_FORM_FROMMONTH");
   if (*cptr)
      FromMonth = atoi(cptr);
   else
      FromMonth = 1;
   if (FromMonth < 1 || FromMonth > 12)
      exit (ErrorGeneral ("<I>From</I> month out of range.",
                          __FILE__, __LINE__));

   GetCgiVar (cptr, "WWW_FORM_TOMONTH");
   if (*cptr)
      ToMonth = atoi(cptr);
   else
      ToMonth = 12;
   if (ToMonth < 1 || ToMonth > 12)
      exit (ErrorGeneral ("<I>To</I> month out of range.",
                          __FILE__, __LINE__));

   if (FromMonth > ToMonth)
      exit (ErrorGeneral (
            "<I>From</I> month greater than <I>to</I> out of range.",
            __FILE__, __LINE__));

   GetCgiVar (cptr, "WWW_FORM_ACROSS");
   if (*cptr)
      MonthsAcrossPage = atoi(cptr);
   else
      MonthsAcrossPage = 1;
   if (MonthsAcrossPage < 1 || MonthsAcrossPage > 12)
      exit (ErrorGeneral ("Months across page out of range.",
                          __FILE__, __LINE__));

   if (FromMonth > 1 || ToMonth < 12)
      sprintf (Range, " <I>(%s to %s)</I>",
               MonthName[FromMonth], MonthName[ToMonth]);
   else
      Range[0] = '\0';

   fprintf (HttpOut,
"%s\
<!-- SoftwareID: %s -->\n\
<TITLE>Calendar for %d</TITLE>\n\
<H1>Calendar %d%s</H1>\n\
<HR>\n\
<PRE>",
            Http200Header, SoftwareID,
            Year, Year, Range);
   HttpHasBeenOutput = true;
   fflush (HttpOut);

   NumTime[0] = Year;
   NumTime[1] = Month;
   NumTime[2] = 1;
   NumTime[3] = NumTime[4] = NumTime[5] = NumTime[6] = 0;

   if (!MonthsAcrossPage) MonthsAcrossPage = 3;

   for (Month = FromMonth; Month <= ToMonth; Month++)
       AddMonth (Year, Month);
   if (MonthsOnLine) OutputPageLines ();

   fprintf (HttpOut, "</PRE>\n<P><HR>\n");
}

/*****************************************************************************/
/*
*/

AddMonth
(
int Year,
int Month
)
{
   register int  ccnt, lcnt;
   register char  *cptr;

   int  status,
        Count,
        DaysInThisMonth;
   unsigned long  DayOfMonth,
                  DayOfWeek;
   unsigned long  BinTime [2];
   unsigned short  NumTime [7];
   char  Line [256];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "AddMonth() %d %d\n", Year, Month);

   if (!MonthsOnLine || MonthsOnLine >= MonthsAcrossPage)
   {
      if (MonthsOnLine >= MonthsAcrossPage) OutputPageLines ();

      for (lcnt = 0; lcnt < MaxPageLines; lcnt++)
      {
          if (Debug) fprintf (stdout, "lcnt: %d\n", lcnt);
          memset (PageLines[lcnt], ' ', PageLineLength);
          PageLines[lcnt][PageLineLength-1] = '\0';
      }
      MonthsOnLine = 0;
   }
   lcnt = 0;

   memset (NumTime, 0, sizeof(NumTime));
   NumTime[0] = Year;
   NumTime[1] = Month;

   DaysInThisMonth = DaysInMonth[Month];
   if (Month == 2)
   {
      /* if the time conversion returns an error then it's not a leap year! */
      NumTime[2] = 29;
      status = lib$cvt_vectim (&NumTime, &BinTime);
      if (VMSnok (status) && status != LIB$_IVTIME)
         exit (ErrorVmsStatus (status, "lib$cvt_vectim()", "",
               __FILE__, __LINE__));
      if (VMSok (status)) DaysInThisMonth = 29;
   }

   /* the the day of the week the month starts on */
   NumTime[2] = 1;
   if (VMSnok (status = lib$cvt_vectim (&NumTime, &BinTime)))
      exit (ErrorVmsStatus (status, "lib$cvt_vectim()", "",
            __FILE__, __LINE__));
   if (VMSnok (status =
       lib$cvt_from_internal_time (&LibDayOfWeek, &DayOfWeek, &BinTime)))
      exit (ErrorVmsStatus (status, "lib$cvt_from_internal_time()", "",
            __FILE__, __LINE__));

   /* we count our week days traditionally (Sunday to Saturday) */
   if (DayOfWeek == 7)
      DayOfWeek = 1;
   else
      DayOfWeek++;
   if (Debug)
       fprintf (stdout, "DayOfWeek: %d DaysInThisMonth: %d\n",
                DayOfWeek, DaysInThisMonth);

   /*********/
   /* month */
   /*********/

   cptr = PageLines[lcnt++] + MonthsOnLine * MonthIncrement;
   for (Count = 0; Count < MonthsOnLine; Count++) cptr += 7;
   ccnt = sprintf (cptr, "%*s<B>%s</B>",
                   (20 - strlen(MonthName[Month])) / 2, "", MonthName[Month]);
   cptr[ccnt] = ' ';

   /*************/
   /* day names */
   /*************/

   cptr = PageLines[lcnt++] + MonthsOnLine * MonthIncrement;
   for (Count = 0; Count < MonthsOnLine; Count++) cptr += 7;
   memcpy (cptr, "<B>", 3); 
   cptr += 3;
   for (Count = 1; Count <= 7; Count++)
   {
      if (Count > 1) *cptr++ = ' ';
      memcpy (cptr, DayName[Count], 2); 
      cptr += 2;
   }
   memcpy (cptr, "</B>", 4); 

   /**********************/
   /* leading empty days */
   /**********************/

   cptr = PageLines[lcnt++] + MonthsOnLine * MonthIncrement;
   for (Count = 1; Count < DayOfWeek; Count++)
   {
      memcpy (cptr, "   ", 3); 
      if (Count == 1)
         cptr += 2;
      else
         cptr += 3;
   }

   /********/
   /* days */
   /********/

   for (DayOfMonth = 1; DayOfMonth <= DaysInThisMonth; DayOfMonth++)
   {
      if (Count == 1)
         cptr += sprintf (cptr, "%2d", DayOfMonth);
      else
         cptr += sprintf (cptr, " %2d", DayOfMonth);
      *cptr = ' ';
      if (Count++ >= 7)
      {
         cptr = PageLines[lcnt++] + MonthsOnLine * MonthIncrement;
         Count = 1;
      }
   }

   MonthsOnLine++;
}

/*****************************************************************************/
/*
*/

OutputPageLines ()

{
   register int  lcnt;
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "OutputPageLines()\n");

   if (!MonthsOnLine) return;
   fprintf (HttpOut, "\n");
   for (lcnt = 0; lcnt < MaxPageLines; lcnt++)
   {
       if (Debug) fprintf (stdout, "lcnt: %d\n", lcnt);
       for (cptr = PageLines[lcnt]; *cptr && *cptr == ' '; cptr++);
       if (!*cptr) break;
       while (*cptr) cptr++;
       if (cptr > PageLines[lcnt]) cptr--;
       while (*cptr == ' ' && cptr > PageLines[lcnt]) cptr--;
       if (cptr > PageLines[lcnt]) cptr++;
       *cptr = '\0';
       fprintf (HttpOut, "%s\n", PageLines[lcnt]);
   }
   MonthsOnLine = 0;
}

/*****************************************************************************/
/*
*/

ErrorGeneral
(
char *Text,
char *SourceFileName,
int SourceLineNumber
)
{
   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s\n",
   SoftwareID, cptr, SourceLineNumber, Text);

   return (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/

ErrorVmsStatus
(
int StatusValue,
char *Text,
char *HiddenText,
char *SourceFileName,
int SourceLineNumber
)
{
   static char  Message [256];
   static $DESCRIPTOR (MessageDsc, Message);

   register char  *cptr;
   int  status;
   short int  Length;

   /*********/
   /* begin */
   /*********/

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 1, 0))) 
   {
      Message[Length] = '\0';
      Message[0] = toupper(Message[0]);
   }
   else
      exit (status);

   /* 
      The source file format provided by the "__FILE__" macro will be
      "device:[directory]name.type;ver".  Reduce that to "name.type".
   */
   for (cptr = SourceFileName; *cptr && *cptr != ';'; cptr++);
   if (*cptr)
   {
      while (*cptr != '.') cptr--;
      *cptr-- = '\0';
   }
   while (*cptr != ']') cptr--;
   cptr++;

   if (!HttpHasBeenOutput) fputs (Http404Header, HttpOut);
   fprintf (HttpOut,
"<!-- SoftwareID: %s Module: %s Line: %d -->\n\
<H1>ERROR!</H1>\n\
<P>Reported by server.\n\
<P>%s ... <TT>%s</TT>\n\
<!-- %%X%08.08X \"%s\" -->\n",
   SoftwareID, cptr, SourceLineNumber, Message, Text, StatusValue, HiddenText);

   return (SS$_NORMAL);
}

/****************************************************************************/
