/*****************************************************************************/
/*
                                WWWrkout.c

"Workout" utility for a WWW server.  Stresses a server both in number of 
simultaneous connections maintained and in the number of back-to-back 
sequential connection requests and data transfers. 

Sets up and maintains a specified number of simultaneous connections (or until 
server limit is reached).  Reads a buffer of data from each connection where 
data is waiting in turn (does not block, peeks to see if any is waiting) until 
the document request is complete and the connection closed by the server.  
Then closes that connection at this end and uses the closed socket to initiate 
another.  If enabled (it is by default), the utility attempts to reflect the 
real-world by varying the data transfer rate for each connection, through 
setting the number of bytes read during each read loop differently for each 
connection.  All data transfered is discarded. 

The data transfer rate for each connection is displayed at connection close.  
It is by default the effective transfer rate, that is the rate from opening 
the connection to closing it, and so includes request processing time, etc., 
or if the /NOEFFECTIVE qualifier is emplaoyed measure the document data 
transfer rate only.

The document paths should be specified one per line in a plain text file.  
Preferably each document path and/or type specified should be different to the 
others, to exercise the server and file system cache.  Any number of paths may 
be specified in the file.  If the file is exhausted before the specified 
number of connections have been established the file contents are recycled 
from the first path.  If path or a file of paths is not specified the utility 
just continually requests the welcome document. 


QUALIFIERS
----------
/DBUG                   turns on all "if (Debug)" statements
/BUFFER=                maximum number bytes to be read from server each time
/COUNT=                 total number of connections and requests to be done
/[NO]EFFECTIVE          measures data transfer rate from request to close
/FILEOF=                file name containing paths to documents
/HELP                   display brief usage information
/[NO]OUTPUT=            file name for output, no output suppresses only the
                        report for each connection, totals and errors still
/PATH=                  path to document to be retrieved
/PORT=                  IP port number of HTTP server
/SERVER=                HTTP server host name
/SIMULTANEOUS=          number of simultaneous connections at any one time
/[NO]VARY               varies the size of the read buffer for each connection


BUILD DETAILS
-------------
See BUILD_WWWRKOUT.COM


VERSION HISTORY (update SoftwareID as well!)
---------------
06-JAN-96  MGD  v1.1.0, Multinet compatibility
05-MAY-95  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "WWWRKOUT v1.1.0 AXP";
#else
   char SoftwareID [] = "WWWRKOUT v1.1.0 VAX";
#endif

/* standard C header files */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

/* VMS related header files */
#include <descrip.h>
#include <libdef.h>
#include <libdtdef.h>
#include <ssdef.h>
#include <stsdef.h>

/* Internet-related header files */

#ifndef IP_MULTINET
#  define IP_UCX 1
#endif

#ifdef IP_UCX
#   include <socket.h>
#   include <in.h>
#   include <netdb.h>
#   include <inet.h>
#endif

#ifdef IP_MULTINET
#   include "multinet_root:[multinet.include.sys]types.h"
#   include "multinet_root:[multinet.include.sys]socket.h"
#   include "multinet_root:[multinet.include.netinet]in.h"
#   include "multinet_root:[multinet.include]errno.h"
#   include "multinet_root:[multinet.include]netdb.h"
#endif

#define boolean int
#define true 1
#define false 0
 
#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

#define MaxConnections 256
#define MaxReadBufferSize 8192
#define DefaultServerPort 80
#define DefaultSpecifiedCount 100
#define DefaultSpecifiedSimultaneous 10
#define DefaultReadBufferSize 1024

struct ConnectionStruct
{
   struct ConnectionStruct  *NextPtr;
   int  BufferSize,
        ByteCount,
        ConnectionNumber,
        Socket;
   unsigned long  StatTimerContext;
   char  Path [256];
};

char  Utility [] = "WWWRKOUT";

unsigned long  StatTimerElapsedTime = 1,
               CvtTimeFloatSeconds = LIB$K_DELTA_SECONDS_F;

boolean  Debug,
         DoShowHelp,
         EffectiveTransferRate,
         NoOutput,
         VaryBufferSize;

int  Connections,
     ConnectionErrorCount,
     ConnectionCount,
     ReadBufferSize,
     HttpReadErrorCount,
     HttpRequestErrorCount,
     ServerPort,
     SpecifiedCount,
     SpecifiedSimultaneous,
     StatusCodeErrorCount,
     TotalBytesReceived,
     TotalBytesSent;

int  StatusCodeCount [10];

FILE  *Output;

unsigned long  TotalStatTimerContext;

char  CommandLine [256],
      OutFileName [256],
      PathFileName [256],
      ServerHostName [256],
      SpecifiedPath [256];

struct ConnectionStruct  *ConnectionListPtr,
                         *LastConnectionPtr;

struct hostent  *ServerHostEntryPtr;
struct sockaddr_in  ServerSocketName;

/* required function prototypes */
char* ErrorMessage (int, int);

/*****************************************************************************/
/*
*/

int main ()

{
   register char  *cptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (VMSnok (status = ParseCommandLine ()))
      exit (status);

   if (DoShowHelp) exit (ShowHelp ());

   if (ReadBufferSize > MaxReadBufferSize)
   {
      fprintf (stdout, "%%%s-E-BUFFER, maximum buffer size is %d bytes\n",
               Utility, MaxReadBufferSize);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   if (!ServerHostName[0])
   {
      /* assume the local host is the server */
      if ((status = gethostname (ServerHostName, sizeof(ServerHostName))) != 0)
      {
         fprintf (stdout, "%%%s-E-SERVER, \"%s\" %s\n",
                  Utility, ServerHostName, ErrorMessage(vaxc$errno,errno));
         exit (STS$K_ERROR | STS$M_INHIB_MSG);
      }
      if (Debug) fprintf (stdout, "ServerHostName |%s|\n", ServerHostName);
   }
   else
   {
      for (cptr = ServerHostName; *cptr && *cptr != ':'; cptr++);
      if (*cptr)
      {
         *cptr++ = '\0';
         if (isdigit(*cptr))
            ServerPort = atoi(cptr);
      }
   }

   /* get the server host address details */
   if ((ServerHostEntryPtr = gethostbyname (ServerHostName)) == NULL)
   {
      fprintf (stdout, "%%%s-E-SERVER, \"%s\" %s\n",
               Utility, ServerHostName, ErrorMessage(vaxc$errno,errno));
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }
   ServerSocketName.sin_family = ServerHostEntryPtr->h_addrtype;
   ServerSocketName.sin_port = htons (ServerPort);
   ServerSocketName.sin_addr = *((struct in_addr *)ServerHostEntryPtr->h_addr);

   strcpy (ServerHostName, ServerHostEntryPtr->h_name);
   /* looks better if its all in lower case (host name is sometimes upper) */
   for (cptr = ServerHostName; *cptr; cptr++) *cptr = tolower(*cptr);
   if (Debug) fprintf (stdout, "ServerHostName |%s|\n", ServerHostName);

   if (OutFileName[0])
   {
      fclose (stdout);
      if ((stdout = fopen (OutFileName, "w")) == NULL)
         exit (vaxc$errno);
   }
   else
   if (!NoOutput)
      Output = stdout;

   ReportParameters ();

   GenerateConnections ();

   ReportParameters ();

   ReportTotals ();

   exit (SS$_NORMAL);
}

/*****************************************************************************/
/*
*/
 
ReportParameters ()

{
   fprintf (stdout,
      "\n$ WWWRKOUT /SERVER=%s /PORT=%d /SIMULTANEOUS=%d /COUNT=%d /BUFFER=%d",
       ServerHostName, ServerPort, SpecifiedSimultaneous,
       SpecifiedCount, ReadBufferSize);
   if (VaryBufferSize)
      fprintf (stdout, " /VARY");
   else
      fprintf (stdout, " /NOVARY");
   if (EffectiveTransferRate)
      fprintf (stdout, " /EFFECTIVE");
   else
      fprintf (stdout, " /NOEFFECTIVE");
   fprintf (stdout, "\n");
}

/*****************************************************************************/
/*
*/
 
ReportTotals ()

{
   int  status,
        Code;
   unsigned long  ElapsedBinTime [2];
   unsigned short  Length;
   float  ElapsedFloatTime;
   char  ElapsedTime [32];
   $DESCRIPTOR (ElapsedTimeFaoDsc, "!%T");
   $DESCRIPTOR (ElapsedTimeDsc, ElapsedTime);

   /*********/
   /* begin */
   /*********/

   if (VMSnok (status =
       lib$stat_timer (&StatTimerElapsedTime,
                       &ElapsedBinTime,
                       &TotalStatTimerContext)))
      exit (status);
   
   if (VMSnok (status =
       lib$cvtf_from_internal_time (&CvtTimeFloatSeconds,
                                    &ElapsedFloatTime,
                                    &ElapsedBinTime)))
      exit (status);

   if (VMSnok (status =
       sys$fao (&ElapsedTimeFaoDsc, &Length, &ElapsedTimeDsc,
                &ElapsedBinTime)))
      exit (status);
   ElapsedTime[Length] = '\0';

   fprintf (stdout,
"\n\
Summary;   Elapsed: %s  Connections: %d\n\
Bytes;     Tx: %d  Rx: %d (%.0f/S)\n\
Failures;  Connect %d  Request %d  Read %d\n",
   ElapsedTime, ConnectionCount,
   TotalBytesSent, TotalBytesReceived,
   (float)TotalBytesReceived / ElapsedFloatTime,
   ConnectionErrorCount, HttpRequestErrorCount, HttpReadErrorCount);

   fprintf (stdout, "Status;    Errors: %d", StatusCodeErrorCount);
   for (Code = 0; Code <= 9; Code++)
      if (StatusCodeCount[Code])
         fprintf (stdout, "  %dXX: %d", Code, StatusCodeCount[Code]);
   fprintf (stdout, "\n");
}

/*****************************************************************************/
/*
Loop through the specified number of connections (or attempts thereof).  
Maintain, as best it can be, the specified number of simultaneous connections.  
If a connection attempt fails just quit there for that particular loop, leave 
a bit of time (as reads occur) to allow the server to recover.
*/
 
int GenerateConnections ()

{
   register struct ConnectionStruct  *cxptr;

   int  status,
        ConnectionNumber,
        RequestLength;
   unsigned long  BinTime [2];
   unsigned short  NumTime [7];
   char  Path [256],
         Request [512];

   /*********/
   /* begin */
   /*********/

   /* dynamically allocate the specified number of connection structures */
   for (ConnectionNumber = 0;
        ConnectionNumber < SpecifiedSimultaneous;
        ConnectionNumber++)
   {
      if ((cxptr = calloc (1, sizeof(struct ConnectionStruct))) == NULL)
         exit (vaxc$errno);
      if (Debug) fprintf (stdout, "cxptr: %d\n", cxptr);
      cxptr->Socket = 0;
      cxptr->NextPtr = NULL;
      if (ConnectionListPtr == NULL)
         ConnectionListPtr = cxptr;
      else
         LastConnectionPtr->NextPtr = cxptr;
      LastConnectionPtr = cxptr;
   }

   /* initialize statistics timer */
   if (VMSnok (status = lib$init_timer (&TotalStatTimerContext)))
      exit (status);

   /******************************************************/
   /* loop for the total number of connections specified */
   /******************************************************/

   ConnectionCount = 0;
   while (ConnectionCount < SpecifiedCount)
   {
      if (Debug) fprintf (stdout, "ConnectionCount %d\n", ConnectionCount);

      /*************************************/
      /* loop looking for free connections */
      /*************************************/

      for (cxptr = ConnectionListPtr; cxptr != NULL; cxptr = cxptr->NextPtr)
      {
         if (Debug) fprintf (stdout, "cxptr: %d\n", cxptr);

         if (cxptr->Socket) continue;

         /***************************/
         /* free, connect to server */
         /***************************/

         ConnectionCount++;

         /* initialize timer */
         if (VMSnok (status = lib$init_timer (&cxptr->StatTimerContext)))
            exit (status);

         /* connect to the server */
         if ((cxptr->Socket = socket (AF_INET, SOCK_STREAM, 0)) < 0)
         {
            fprintf (stdout, "socket() error; #%d, %s\n",
                     ConnectionCount, ErrorMessage(vaxc$errno,errno));
            cxptr->Socket = 0;
            ConnectionErrorCount++;
            break;
         }
         else
         if (connect (cxptr->Socket,
                      &ServerSocketName,
                      sizeof(ServerSocketName)))
         {
            fprintf (stdout, "connect() error; #%d, %s\n",
                     ConnectionCount, ErrorMessage(vaxc$errno,errno));
            close (cxptr->Socket);
            cxptr->Socket = 0;
            ConnectionErrorCount++;
            break;
         }

         /****************/
         /* send request */
         /****************/

         /* create and send a moderate size HTTP request */
         GetPath (Path, sizeof(Path));
         strcpy (cxptr->Path, Path);
         RequestLength = sprintf (Request,
"GET %s HTTP/1.0\r\n\
User-Agent: %s\r\n\
Accept: */*\r\n\
\r\n",
         Path, SoftwareID);
         if (Debug) fprintf (stdout, "Request |%s|\n", Request);

         if (send (cxptr->Socket, Request, RequestLength, 0) == -1)
         {
            fprintf (stdout, "send() error; #%d \"%s\", %s\n",
                     ConnectionCount, Path, ErrorMessage(vaxc$errno,errno));
            close (cxptr->Socket);
            cxptr->Socket = 0;
            HttpRequestErrorCount++;
            break;
         }

         cxptr->ConnectionNumber = ConnectionCount;
         cxptr->ByteCount = 0;
         TotalBytesSent += RequestLength;

         if (VaryBufferSize)
         {
            /* adjust the buffer size using a pseudo-random value */
            sys$gettim (&BinTime);
            sys$numtim (&NumTime, &BinTime);
            cxptr->BufferSize = ReadBufferSize / ((NumTime[6] & 7) + 1);
         }
         else
            cxptr->BufferSize = ReadBufferSize;
         if (Debug) fprintf (stdout, "BufferSize: %d\n", cxptr->BufferSize);

         if (ConnectionCount >= SpecifiedCount) break;
      }

      /**************************************/
      /* read a buffer from each connection */
      /**************************************/

      if (!TransferData ()) break;
   }

   /* read from the final lot of connections */
   while (TransferData ());
}

/*****************************************************************************/
/*
Looping through the array of sockets, check for data waiting to be read 
(using the peek kludge), if none do not block, continue trying with next 
socket, if data available read it.  If an error report it.  If end-of-file 
(connection closed by server) close the socket and reset the array values to 
zero (in part to indicate the socket is no longer in use).
*/ 
 
int TransferData ()

{
   register char  *cptr, *zptr;
   register struct ConnectionStruct *cxptr;

   int  status,
        BytesPerSecond,
        Count,
        NumberOfConnections,
        Quantity;
   unsigned long  ElapsedBinTime [2];
   float  ElapsedFloatTime;
   char  ReadBuffer [MaxReadBufferSize];

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "TransferData()\n");

   NumberOfConnections = 0;

   for (cxptr = ConnectionListPtr; cxptr != NULL; cxptr = cxptr->NextPtr)
   {
      if (!cxptr->Socket) continue;
      NumberOfConnections++;

      /* peek to see if there is any data available for this socket */
      if ((Count = recv (cxptr->Socket, ReadBuffer, 1, MSG_PEEK)) <= 0)
      {
         if (Debug)
            fprintf (stdout, "errno: %d |%s|\n",
                     errno, ErrorMessage(vaxc$errno,errno));

         /* if no data for this socket, don't wait, continue with next */
         if (errno == EWOULDBLOCK) continue;

         if (!Count || errno == ECONNRESET)
         {
            /****************************/
            /* server closed connection */
            /****************************/

            if (VMSnok (status =
                lib$stat_timer (&StatTimerElapsedTime,
                                &ElapsedBinTime,
                                &cxptr->StatTimerContext)))
               exit (status);

            if (VMSnok (status =
                lib$cvtf_from_internal_time (&CvtTimeFloatSeconds,
                                             &ElapsedFloatTime,
                                             &ElapsedBinTime)))
               exit (status);

            BytesPerSecond = (int)((float)cxptr->ByteCount / ElapsedFloatTime);

            fprintf (Output,
            "end;    #%d \"%s\", %d bytes in %.1f seconds (%d/S, size: %d)\n",
               cxptr->ConnectionNumber, cxptr->Path, cxptr->ByteCount,
               ElapsedFloatTime, BytesPerSecond, cxptr->BufferSize);
         }
         else
         {
            /**************/
            /* read error */
            /**************/

            fprintf (stdout, "read() error; #%d \"%s\", after %d bytes, %s\n",
                     cxptr->ConnectionNumber, cxptr->Path,
                     cxptr->ByteCount, ErrorMessage(vaxc$errno,errno));
            HttpReadErrorCount++;
         }
         close (cxptr->Socket);
         cxptr->Socket = 0;
         TotalBytesReceived += cxptr->ByteCount;

         continue;
      }

      if ((Count = recv (cxptr->Socket, ReadBuffer,
                         cxptr->BufferSize, 0)) <= 0)
      {
         if (Debug)
            fprintf (stdout, "errno: %d |%s|\n",
                     errno, ErrorMessage(vaxc$errno,errno));

         /* shouldn't happen after the peek recv() above, but if it does */
         fprintf (Output, "read() error; #%d \"%s\", after %d bytes, %s\n",
                  cxptr->ConnectionNumber, cxptr->Path,
                  cxptr->ByteCount, ErrorMessage(vaxc$errno,errno));
         HttpReadErrorCount++;

         close (cxptr->Socket);
         cxptr->Socket = 0;
         TotalBytesReceived += cxptr->ByteCount;
      }
      else
      {
         if (Debug) fprintf (stdout, "Count: %d\n", Count);

         if (!cxptr->ByteCount)
         {
            /************************/
            /* HTTP response header */
            /************************/

            if (VMSnok (status =
                lib$stat_timer (&StatTimerElapsedTime,
                                &ElapsedBinTime,
                                &cxptr->StatTimerContext)))
               exit (status);

            if (VMSnok (status =
                lib$cvtf_from_internal_time (&CvtTimeFloatSeconds,
                                             &ElapsedFloatTime,
                                             &ElapsedBinTime)))
               exit (status);

            if (!EffectiveTransferRate)
            {
               /* re-initialize timer to measure HTTP data transfer time */
               if (VMSnok (status = lib$init_timer (&cxptr->StatTimerContext)))
                  exit (status);
            }

            /* (assumes first line is all in buffer!) */
            zptr = (cptr = ReadBuffer) + Count;
            while (cptr < zptr && *cptr != '\r' && *cptr != '\n') cptr++;
            if (cptr < zptr)
            {
               *cptr = '\0';

               /* skip over "HTTP/n.n" and spaces to get 3 digit status code */
               cptr = ReadBuffer;
               while (*cptr && !isspace(*cptr)) cptr++;
               while (isspace(*cptr)) cptr++;
               /* increment the counter corresponding to the first digit */
               if (isdigit(*cptr))
                  StatusCodeCount[*cptr-'0']++;
               else
                  StatusCodeErrorCount++;

               fprintf (Output, "header; #%d \"%s\", in %.1f seconds \"%s\"\n",
                        cxptr->ConnectionNumber, cxptr->Path,
                        ElapsedFloatTime, ReadBuffer);
            }
            else
            {
               fprintf (Output,
"header error (buffer too small); #%d \"%s\", in %.1f seconds\n",
                  cxptr->ConnectionNumber, cxptr->Path, ElapsedFloatTime);
            }
         }

         cxptr->ByteCount += Count;
         continue;
      }
   }

   return (NumberOfConnections);
}

/*****************************************************************************/
/*
*/
 
int GetPath
(
char *PathPtr,
int SizeOfPath
)
{
   static FILE  *PathFile = NULL;

   register char  *cptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "GetPath()\n");

   if (SpecifiedPath[0])
   {
      strcpy (PathPtr, SpecifiedPath);
      return;
   }

   if (!PathFileName[0])
   {
      /* if no file name has been specified get the server welcome document */
      PathPtr[0] = '/';
      PathPtr[1] = '\0';
      return;
   }

   if (PathFile == NULL)
      if ((PathFile = fopen (PathFileName, "r")) == NULL)
         exit (vaxc$errno);

   if (fgets (PathPtr, 256, PathFile) == NULL)
   {
      rewind (PathFile);
      if (fgets (PathPtr, SizeOfPath, PathFile) == NULL)
         exit (vaxc$errno);
   }

   for (cptr = PathPtr; *cptr && *cptr != '\n'; cptr++);
   *cptr = '\0';
   if (Debug) fprintf (stdout, "PathPtr |%s|\n", PathPtr);
}

/*****************************************************************************/
/*
*/
 
int ShowHelp ()
 
{
   fprintf (stdout,
"%%%s-I-HELP, usage for the WWWRKOUT utility (%s)\n\
\n\
Test utility for HTTP server.  Sets up and maintains a specified number of\n\
simultaneous connections.  Reads a buffer of data from each connection in\n\
turn until the document requested is exhausted.  Then closes that connection\n\
and initiates another.  The document paths should be specified one per line\n\
in a plain text file, each path and/or type specified should be different to\n\
the others.   Any number of paths may be specified in the file with the\n\
paths continually recycled from the first.  If a file of paths is not\n\
specified the utility just continually requests the welcome document.\n\
\n\
$ WWWRKOUT [server_host_name[:port]] [document_path] [qualifiers ...]\n\
\n\
/BUFFER=integer /COUNT=integer /[NO]EFFECTIVE(d) /HELP /[NO]OUTPUT=file_name\n\
/PATH=document_path /PORT=integer /SERVER=host_name /SIMULTANEOUS=integer\n\
/FILEOF=file_name /[NO]VARY(d)\n\
\n\
Usage examples:\n\
\n\
$ WWWRKOUT\n\
$ WWWRKOUT WWW.SERVER.HOST:8080\n\
$ WWWRKOUT /SERVER=WWW.SERVER.HOST /PORT=8080 /FILEOF=PATHS.TXT\n\
$ WWWRKOUT WWW.SERVER.HOST /FILEOF=PATHS.TXT /COUNT=500 /SIMUL=20 /BUFFER=512\n\
\n",
   Utility, SoftwareID);
 
   return (SS$_NORMAL);
}
 
/*****************************************************************************/
/*
*/

char* ErrorMessage
(
int VmsStatus,
int ErrorNumber
)
{
   static char  Message [256];

   register char  *cptr;

   short int  Length;
   $DESCRIPTOR (MessageDsc, Message);

   /*********/
   /* begin */
   /*********/

   Message[0] = '\0';
   if (VmsStatus)
   {
      /* text component only */
      sys$getmsg (VmsStatus, &Length, &MessageDsc, 1, 0);
      Message[Length] = '\0';
   }
   if (ErrorNumber)
   {
      if ((cptr = strerror(ErrorNumber)) != NULL)
         strcpy (Message, cptr);
      else
      if (!VmsStatus)
         strcpy (Message, "(not translatable)");
   }
   if (Message[0])
      return (Message);
   else
      return ("(internal error)");
}
 
/*****************************************************************************/
/*
*/

char* SysGetMsg (int StatusValue)

{
   static char  Message [256];
   register int  status;
   short int  Length;
   $DESCRIPTOR (MessageDsc, Message);

   if (VMSok (status = sys$getmsg (StatusValue, &Length, &MessageDsc, 0, 0))) 
      Message[Length] = '\0';
   else
      sprintf (Message, "%%sys$getmsg() %%X%08.08X", status);

   return (Message);
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 
 
boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Get the 
entire command line following the verb that activated the image.  The command 
line is returned in uppercase, space compressed (i.e. maximum of one space 
between text elements, trimmed of leading and trailing spaces).  Returns a 
warning status if there were no parameters/qualifiers on the command line.
The variable CommandLine is global.
*/ 
 
int ParseCommandLine ()
 
{
   int  status;
   unsigned short  Length;
   unsigned long  Flags = 0;
   struct dsc$descriptor_s 
          CommandLineDsc = { sizeof(CommandLine)-1, DSC$K_DTYPE_T,
                             DSC$K_CLASS_S, CommandLine };
 
   /* get the entire command line following the verb */
   if (VMSnok (status = lib$get_foreign (&CommandLineDsc, 0, &Length, &Flags)))
      return (status);
   CommandLine[Length] = '\0';
 
   if (ParseCommand (CommandLine))
      return (SS$_NORMAL);
   else
      return (STS$K_ERROR | STS$M_INHIB_MSG);
}
 
/****************************************************************************/
/*
This function allows images activated by a "foreign verb" to behave in a way 
that approximates the CLI$ (Command Line Interpreter) utility calls.  Quoted 
strings are always indicated by being parsed to include a single leading 
quote.
*/ 
 
boolean ParseCommand (char *CommandLine)
 
{
   register int  QuoteCount = 0;
   register char  *cptr, *eptr;
   boolean  CommandLineOK = true;
   char  Entity [256] = "";
 
   /* set up any argument defaults */
   ParseCommandEntity (NULL);
 
   cptr = CommandLine;
   eptr = Entity;
 
   for (;;)
   {
      if (*cptr == '\"')
      {
         QuoteCount++;
         *eptr++ = *cptr++;
         continue;
      }
 
      if (QuoteCount & 1 && *cptr)
      {
         /* inside quoted text, copy all characters as literals */
         *eptr++ = *cptr++;
         continue;
      }
 
      if (*cptr == '/' || isspace (*cptr) || !*cptr)
      {
         if (isspace (*cptr))
         {
            /* span the white space */
            while (*cptr && isspace (*cptr)) cptr++;
            if (*cptr == '=')
            {
               /* part of a qualifier, continue to get the value */
               *eptr++ = *cptr++;
               /* span any intervening white space */
               while (*cptr && isspace (*cptr)) cptr++;
               continue;
            }
         }
 
         if (Entity[0])
         {
            *eptr = '\0';
            if (!ParseCommandEntity (Entity)) CommandLineOK = false;
         }
 
         /* if end of command line then break from loop */
         if (!*cptr) break;
 
         /* start of new entity */
         eptr = Entity;
         /* if start of qualifier ensure slash is copied */
         if (*cptr == '/') *eptr++ = *cptr++;
 
         continue;
      }
 
      /* any other character, just copy, ensure upper case */
      *eptr++ = toupper(*cptr++);
   }
 
   return (CommandLineOK);
}
 
/*****************************************************************************/
/*
Get a string value from a qualifier, e.g. '/EXAMPLE=TEST'.
*/
 
boolean ParseCommandString
(
char *Entity,
char *String,
boolean Qualifier,
boolean ReportErrors,
boolean EnsureUpperCase
)
{
   register int  QuoteCount = 0;
   register char  *eptr, *sptr;
 
   if (Debug) fprintf (stdout, "ParseCommandString()\nEntity: '%s'\n", Entity);
 
   eptr = Entity;
 
   if (Qualifier)
   {
      /* scan down to equate symbol */
      while (*eptr && *eptr != '=') eptr++;
      if (*eptr) eptr++;
      if (!*eptr)
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
            Utility, Entity+1);
         }
         return (false);
      }
   }
 
   sptr = String;
   while (*eptr)
   {
      if (*eptr == '\"')
      {
         if (QuoteCount & 1)
         {
            /* are inside quotes, check for escaped quotes ("") */
            if (*++eptr != '\"')
            {
               /* now outside quotes */
               QuoteCount++;
            }
            /* drop thru to character copy */
         }
         else
         {
            /* now inside quotes */
            QuoteCount++;
            eptr++;
            continue;
         }
      }
 
      if (EnsureUpperCase)
         *sptr++ = toupper(*eptr++);
      else
         *sptr++ = *eptr++;
   }
   *sptr = '\0';
 
   if (Debug) fprintf (stdout, "String: '%s'\n", String);
 
   return (true);
}
 
/*****************************************************************************/
/*
Get an integer value from a qualifier, e.g. '/EXAMPLE=99'.
*/
 
boolean ParseCommandInteger
(
char *Entity,
int *IntegerPtr,
int Base,
boolean ReportErrors
)
{
   register char  *eptr;
   char  *sptr;
 
   if (Debug)
      fprintf (stdout, "ParseCommandInteger() '%s' Base: %d\n", Entity, Base);
 
   for (eptr = Entity; *eptr && *eptr != '='; eptr++);
   if (*eptr) eptr++;
   if (*eptr)
   {
      *IntegerPtr = strtol (eptr, &sptr, Base);
      if (sptr > eptr && !*sptr)
         return (true);
      else
      {
         if (ReportErrors)
         {
            fprintf (stdout,
            "%%%s-E-BADVALUE, '%s' is an invalid keyword value\n",
            Utility, eptr);
         }
         return (false);
      }
   }
   else
   {
      if (ReportErrors)
      {
         fprintf (stdout,
         "%%%s-E-VALREQ, missing qualifier or keyword value\n \\%s\\\n",
         Utility, Entity+1);
      }
      return (false);
   }
}
 
/*****************************************************************************/
/*
A single command line "entity" has been parsed, check if its recognised.  This 
function is the one modified for the individual requirements of each program.
*/
 
boolean ParseCommandEntity (char *Entity)
 
{
   if (Entity == NULL)
   {
      /* set up any argument defaults */
      Debug = DoShowHelp = false;
      EffectiveTransferRate = VaryBufferSize = true;
      ReadBufferSize = DefaultReadBufferSize;
      ServerPort = DefaultServerPort;
      SpecifiedSimultaneous = DefaultSpecifiedSimultaneous;
      SpecifiedCount = DefaultSpecifiedCount;
      OutFileName[0] = PathFileName[0] = ServerHostName[0] = '\0';
      return (true);
   }
 
   if (Debug) fprintf (stdout, "ParseCommandEntity() Entity: '%s'\n", Entity);
 
   if (Entity[0] == '/')
   {
      if (strsame (Entity, "/BUFFER=", 4))
         return (ParseCommandInteger (Entity, &ReadBufferSize, 10, true));

      if (strsame (Entity, "/COUNT=", 4))
         return (ParseCommandInteger (Entity, &SpecifiedCount, 10, true));

      /* turns on all "if (Debug)" statements */
      if (strsame (Entity, "/DBUG", -1))
         return (Debug = true);

      if (strsame (Entity, "/EFFECTIVE", 4))
         return (EffectiveTransferRate = true);
      if (strsame (Entity, "/NOEFFECTIVE", 6))
      {
         EffectiveTransferRate = false;
         return (true);
      }

      if (strsame (Entity, "/HELP", 4))
         return (DoShowHelp = true);

      if (strsame (Entity, "/PATH=", 4))
         return (ParseCommandString (Entity, SpecifiedPath, true, true, true));

      if (strsame (Entity, "/PORT=", 4))
         return (ParseCommandInteger (Entity, &ServerPort, 10, true));

      if (strsame (Entity, "/SERVER=", 4))
         return (ParseCommandString (Entity, ServerHostName, true, true, true));

      if (strsame (Entity, "/SIMULTANEOUS=", 4))
         return (ParseCommandInteger (Entity, &SpecifiedSimultaneous, 10, true));

      if (strsame (Entity, "/FILEOF=", 4))
         return (ParseCommandString (Entity, PathFileName, true, true, true));

      if (strsame (Entity, "/OUTPUT=", 4))
      {
         NoOutput = false;
         return (ParseCommandString (Entity, OutFileName, true, true, true));
      }
      if (strsame (Entity, "/NOOUTPUT", 6))
         return (NoOutput = true);

      if (strsame (Entity, "/VARY", 4))
         return (VaryBufferSize = true);
      if (strsame (Entity, "/NOVARY", 6))
      {
         VaryBufferSize = false;
         return (true);
      }

      fprintf (stdout,
      "%%%s-E-IVQUAL, unrecognised qualifier\n \\%s\\\n", Utility, Entity+1);
      return (false);
   }
 
   if (!ServerHostName[0])
   {
      ParseCommandString (Entity, ServerHostName, false, false, false);
      return (true);
   }
   if (!SpecifiedPath[0])
   {
      ParseCommandString (Entity, SpecifiedPath, false, false, false);
      return (true);
   }

   fprintf (stdout,
   "%%%s-E-MAXPARM, too many parameters\n \\%s\\\n", Utility, Entity);
   return (false);
}
   
/*****************************************************************************/

