/*
 * sound handling for Spy vs BoB
 */

#ifdef NETAUDIO
#include        <stdio.h>
#include	<audio/audiolib.h>
#include	<audio/soundlib.h>

/* sound data */
#include	"crash.h"
#include	"ledge.h"
#include	"vator.h"
#include	"gift.h"
#include	"level.h"

#define	DEFAULT_VOLUME		60
#define VOLUME_INC              3
#define MIN_VOLUME              0
#define MAX_VOLUME              100
#define	VOL(src)		((1 << 16) * (src)->volume * (src)->volume_mult / 100)

#define	BG_SOUND		1
#define	CRASH_SOUND		2
#define	LEDGE_SOUND		3
#define	VATOR_SOUND		4
#define	GIFT_SOUND		5
#define	LEVEL_SOUND		6

#define	NUM_SOUNDS		7

#undef	DEBUG

/* how many copies of a sound can be playing at once */
#define	MAX_INSTANCES	20

static AuBool use_audio = AuFalse;
static AuBool play_audio = AuFalse;

AuServer   *aud;

#define	PLAY_OFF	0
#define	PLAY_ON		1
#define	PLAY_STOPPED	2

#define	SOUND_TYPE_BUCKET	1
#define	SOUND_TYPE_FILE		2

typedef struct _soundrec {
    int         sound_type;
    AuFlowID    flow[MAX_INSTANCES];
    unsigned int num[MAX_INSTANCES];
    int         playing[MAX_INSTANCES];
    AuEventHandlerRec *handler[MAX_INSTANCES];
    AuBucketID  id;
    AuBool      continuous;
    int         volume,
                volume_mult;
    int         cur_idx;
    char       *descr;
    char       *filename;
}           SndRec, *SndPtr;

SndRec      sounds[NUM_SOUNDS];

static void start_sound();

static void
init_audio(audioServerString)
    char       *audioServerString;
{

    if (use_audio && (aud =
	  AuOpenServer(audioServerString, 0, NULL, 0, NULL, NULL)) != NULL) {
	use_audio = AuTrue;
	play_audio = AuTrue;
    } else {
	use_audio = AuFalse;
	play_audio = AuFalse;
    }
}

static void
stop_sound(cursound, all)
    SndPtr      cursound;
    AuBool      all;
{
    int         i;

    for (i = 0; i < MAX_INSTANCES; i++) {
	if (cursound->playing[i] != PLAY_ON)
	    continue;
	cursound->playing[i] = PLAY_STOPPED;
	AuStopFlow(aud, cursound->flow[i], NULL);
	if (!all)
	    return;
    }

#ifdef DEBUG
    if (!all)
	fprintf(stderr, "no sound to stop\n");
#endif
}

static void
done_cb(aud, handler, ev, data)
    AuServer   *aud;
    AuEventHandlerRec *handler;
    AuEvent    *ev;
    AuPointer   data;
{
    int         i;
    SndPtr      snd = (SndPtr) data;
    int         was_playing;

    for (i = 0; i < MAX_INSTANCES; i++) {
	if (snd->handler[i] == handler) {
	    snd->handler[i] = NULL;
	    snd->flow[i] = 0;
	    was_playing = snd->playing[i];
	    snd->playing[i] = PLAY_OFF;
	    /* restart any loopers */
	    if (snd->continuous && (was_playing == PLAY_ON)) {
		start_sound(snd);
	    }
	    return;
	}
    }

#ifdef DEBUG
    fprintf(stderr, "HELP -- no match for ending sound\n");
#endif
}

void
start_sound_file(snd)
    SndPtr      snd;
{
    int         mult;
    extern unsigned int AuSoundPortDuration;
    unsigned int old_duration = AuSoundPortDuration;

    AuSoundPortDuration = 2;
    if (snd->handler[0] = AuSoundPlayFromFile(aud, snd->filename, AuNone,
	  VOL(snd), done_cb, snd, &snd->flow[0], &snd->num[0], NULL, NULL)) {
	snd->playing[0] = PLAY_ON;
    }
    AuSoundPortDuration = old_duration;
}

static void
start_sound(cursound)
    SndPtr      cursound;
{
    int         ni;
    int         multiplier;


    if (!play_audio)
	return;
    if (cursound->sound_type == SOUND_TYPE_FILE) {
	start_sound_file(cursound);
	return;
    }
    ni = cursound->cur_idx++;
    if (cursound->cur_idx >= MAX_INSTANCES) {
	cursound->cur_idx = 0;
    }
    if (cursound->playing[ni] != PLAY_ON) {
	if (cursound->handler[ni] = AuSoundPlayFromBucket(aud, cursound->id,
	    AuNone, VOL(cursound), done_cb, cursound, 1, &cursound->flow[ni],
					   &cursound->num[ni], NULL, NULL)) {
	    cursound->playing[ni] = PLAY_ON;
	}
    }
}

/* ARGSUSED */
static void
sync_cb(aud, handler, ev, data)
    AuServer   *aud;
    AuEventHandlerRec *handler;
    AuEvent    *ev;
    AuPointer   data;
{
    int        *d = (int *) data;

    *d = 1;
}

/* plays till it gets the AudioStopped event */
static void
start_sound_sync(cursound)
    SndPtr      cursound;
{
    AuFlowID    flow;
    int         ni;
    int         multiplier;
    AuBool      done = AuFalse;
    AuEvent     ev;

    if (!play_audio)
	return;
    ni = ++cursound->cur_idx;
    if (cursound->cur_idx >= MAX_INSTANCES) {
	cursound->cur_idx = 0;
    }
    cursound->playing[ni] = PLAY_ON;

    if (!AuSoundPlayFromBucket(aud, cursound->id,
		AuNone, VOL(cursound), sync_cb, &done, 1, &flow, &multiplier,
			       NULL, NULL))
	return;
    /* wait for end of sound */
    while (1) {
	AuNextEvent(aud, AuTrue, &ev);
	AuDispatchEvent(aud, &ev);
	if (done) {
	    cursound->playing[ni] = PLAY_OFF;
	    return;
	}
    }
}

static void
set_volume(cursound)
    SndPtr      cursound;
{
    AuElementParameters parms[MAX_INSTANCES];
    int         i,
                num;

    for (i = 0, num = 0; i < MAX_INSTANCES; i++) {
	if (cursound->playing[i] == PLAY_ON) {
	    parms[num].flow = cursound->flow[i];
	    parms[num].element_num = cursound->num[i];
	    parms[num].num_parameters = AuParmsMultiplyConstant;
	    parms[num].parameters[AuParmsMultiplyConstantConstant] =
		VOL(cursound);
	    num++;
	}
    }
    AuSetElementParameters(aud, num, parms, NULL);
}

static void
adjust_volume(delta, abs)
    int         delta;
    AuBool      abs;
{
    SndPtr      cursound;
    int         i;

    for (i = 1; i < NUM_SOUNDS; i++) {
	cursound = &sounds[i];
	if (abs)
	    cursound->volume = delta;
	else
	    cursound->volume += delta;
	if (cursound->volume < MIN_VOLUME)
	    cursound->volume = MIN_VOLUME;
	else if (cursound->volume > MAX_VOLUME)
	    cursound->volume = MAX_VOLUME;
	set_volume(cursound);
    }
}

void
gift_sound()
{
    start_sound(&sounds[GIFT_SOUND]);
}

void
crash_sound()
{
    if (play_audio)
	start_sound_sync(&sounds[CRASH_SOUND]);
    else			/* need delay in any case */
	sleep(1);
}

void
vator_sound()
{
    start_sound(&sounds[VATOR_SOUND]);
}

void
ledge_sound()
{
    start_sound(&sounds[LEDGE_SOUND]);
}

void
level_sound()
{
    start_sound(&sounds[LEVEL_SOUND]);
}

void
start_bg_music()
{
    SndPtr      snd = &sounds[BG_SOUND];

    start_sound_file(snd);
}

void
stop_bg_music()
{
    SndPtr      snd = &sounds[BG_SOUND];

    stop_sound(snd, AuFalse);
}

/* XXX this stops subsequent sounds, not those currently playing */
/* note that this means the BG music won't stop till its EOF */
/* may want to force a stop/start for it */
void
toggle_sound()
{
    if (use_audio)
	play_audio = !play_audio;
}

void
increase_volume()
{
    adjust_volume(VOLUME_INC, AuFalse);
}

void
decrease_volume()
{
    adjust_volume(-VOLUME_INC, AuFalse);
}

int
auservernum()
{
    return AuServerConnectionNumber(aud);
}

void
suck_audio_events()
{
    AuEvent     aev;

    if (_AuEventsQueued(aud, AuEventsQueuedAfterFlush)) {
	AuNextEvent(aud, AuTrue, &aev);
	AuDispatchEvent(aud, &aev);
    }
}

static void
load_memory_sound(s, data, idx)
    Sound	s;
    AuPointer   data;
    int         idx;
{
    bzero(&sounds[idx], sizeof(SndRec));
    sounds[idx].volume = DEFAULT_VOLUME;
    sounds[idx].volume_mult = 1;

    sounds[idx].id = AuSoundCreateBucketFromData(aud,
				s, data, AuAccessAllMasks, 0, NULL);
    sounds[idx].descr = SoundComment(s);
    sounds[idx].sound_type = SOUND_TYPE_BUCKET;
}

static void
init_file_sound(fname, idx)
    char       *fname;
    int         idx;
{
    bzero(&sounds[idx], sizeof(SndRec));
    sounds[idx].volume = DEFAULT_VOLUME;
    sounds[idx].volume_mult = 1;
    sounds[idx].sound_type = SOUND_TYPE_FILE;

    sounds[idx].filename = fname;
}

static char *bg_filename = "james_bond.snd";

void
audio_init(audioServerString, ua)
    char       *audioServerString;
    int         ua;
{
    Sound	s;
    use_audio = ua;
    init_audio(audioServerString);
    if (use_audio) {
        s = SoundCreate(SoundFileFormatNone, crashDataFormat,
        	crashNumTracks, crashSampleRate, crashNumSamples,
                crashComment);
	load_memory_sound(s, crashSamples, CRASH_SOUND);
        SoundDestroy(s);
        s = SoundCreate(SoundFileFormatNone, ledgeDataFormat,
        	ledgeNumTracks, ledgeSampleRate, ledgeNumSamples,
                ledgeComment);
	load_memory_sound(s, ledgeSamples, LEDGE_SOUND);
        SoundDestroy(s);
        s = SoundCreate(SoundFileFormatNone, vatorDataFormat,
        	vatorNumTracks, vatorSampleRate, vatorNumSamples,
                vatorComment);
	load_memory_sound(s, vatorSamples, VATOR_SOUND);
        SoundDestroy(s);
        s = SoundCreate(SoundFileFormatNone, giftDataFormat,
        	giftNumTracks, giftSampleRate, giftNumSamples,
                giftComment);
	load_memory_sound(s, giftSamples, GIFT_SOUND);
        SoundDestroy(s);
        s = SoundCreate(SoundFileFormatNone, levelDataFormat,
        	levelNumTracks, levelSampleRate, levelNumSamples,
                levelComment);
	load_memory_sound(s, levelSamples, LEVEL_SOUND);
        SoundDestroy(s);

	init_file_sound(bg_filename, BG_SOUND);

	/* make the crash extra loud */
	sounds[CRASH_SOUND].volume_mult = 3;
	set_volume(&sounds[CRASH_SOUND]);
	sounds[LEDGE_SOUND].volume_mult = 2;
	set_volume(&sounds[LEDGE_SOUND]);

	sounds[BG_SOUND].continuous = AuTrue;
    }
}

#endif				/* NETAUDIO */
