N /****************************************************************************/ /*									    */ 0 /*  FACILITY:	Generic Support Library					    */ /*									    */ > /*  MODULE:	Command Line Keyword and Option Processing		    */ /*									    */ G /*  AUTHOR:	Steve Branam, Network Product Support Group, Digital	    */ 6 /*		Equipment Corporation, Littleton, MA, USA.		    */ /*									    */ M /*  DESCRIPTION: This module contains routines to process command line	    */ N /*  arguments, given a dispatch table of keywords and handlers. This can be */N /*  used for main program command lines, or internal command lines parsed   */$ /*  in a similar manner.						    */ /*									    */ " /*  REVISION HISTORY:							    */ /*									    */ / /*  V0.1-00 24-AUG-1994 Steve Branam					    */  /*									    */   /*	Original version.						    */ /*									    */ / /*  V0.2-00 23-SEP-1994 Steve Branam					    */  /*									    */ - /*	Addded keyword translation code.				    */  /*									    */ N /****************************************************************************/   #include <stdio.h> #include <ctype.h> #include "cmdopt.h"   : static KEYWORD_DEFINITION			/* Option dispatch table,   */4 	    *mOptionTable;			/* used by options file	    */ 						/* processing.		    */  N /*************************************************************************++*/
 int ustrncmp( N /* Compares strings in upper-case, serving as a case-insensitive string	    */8 /* comparison; otherwise identical to strncmp.				    */       char    *aStr1, $ 	    /* (READ, BY ADDR):						    */+ 	    /* First string to compare.					    */        char    *aStr2, $ 	    /* (READ, BY ADDR):						    */+ 	    /* Second string to compare.				    */        int	    vLength # 	    /* (READ, BY VAL):						    */ 9 	    /* Maximum number of characters to compare.			    */   @ )	/* Returns status value indicating comparison results:		    */( 	/*    0	- Strings are equal.					    */4 	/*  < 0 - String aStr1 is less than aStr2.			    */7 	/*  > 0 - String aStr1 is greater than aStr2.			    */ G 	/*****************************************************************--*/    {      /*+									    */M     /*	Compare upper-case version of each char in strings until unequal    */ K     /*	chars ard found, end of string is found, or maximum number of	    */ '     /*	characters exceeded.						    */      /*- 								    */  B     for (; toupper(*aStr1) == toupper(*aStr2); aStr1++, aStr2++) {( 	if (*aStr1 == '\0' || --vLength == 0) {3 	    return 0;			    /* Strings are "equal".	    */  	}-     }					    /* String are not equal.	    */ -     return toupper(*aStr1) - toupper(*aStr2);  }   N /*************************************************************************++*/ static int cmdopt_cmp(L /* Compares an argument string to a command line option using a case-	    */N /* insensitive string comparison.  The argument matches the option if the   */K /* keyword portion is at least as long as the minimum option keyword	    */ G /* length, and matches the keyword for all specified characters.	    */        KEYWORD_DEFINITION 	    *aOption,$ 	    /* (READ, BY ADDR):						    */A 	    /* Command line option keyword definition to compare.	    */        char    *aArgStr$ 	    /* (READ, BY ADDR):						    */D 	    /* Argument string to compare. Options are assumed to be	    */G 	    /* specified in one of two forms, either just the option switch */ G 	    /* char followed by the keyword (for toggle options), or the    */ G 	    /* option switch char and keyword followed by an equal sign and */ < 	    /* the option value string (for value options).		    */  H )	/* Returns length of keyword portion of aArgstr (including switch   */G 	/* char) to indicate successful match, or 0 if argument does not    */  	/* match option.						    */ G 	/********************************************************************/    { ;     int	    kwlen;			    /* Length of argument keyword   */ + 					    /* portion (including switch    */  					    /* char).			    */        /*+									    */M     /*	Find length of argument keyword portion. If it is less than minimum */ M     /*	required length, no match. Otherwise, compare argument to option    */ -     /*	keyword to determine match.					    */      /*-									    */  D     if ((kwlen = kwstrlen(&aArgStr[1])) < kwdef_minlen(aOption)) {  / 	return 0;			    /* Too short, no match.	    */      } :     else if (ustrncmp(kwdef_keyword(aOption), &aArgStr[1],, 		max(kwlen, kwdef_minlen(aOption))) == 0) {; 	return kwlen + 1;			    /* Argument matches option.	    */      } 
     else {% 	return 0;			    /* No match.		    */      }  }   N /*************************************************************************++*/ static int kwdef_cmp( I /* Compares a string to a keyword using a case- insensitive string	    */ N /* comparison.  The string matches the keyword if it is at least as long as */N /* the minimum keyword length, and matches the keyword for all specified    */ /* characters.								    */       KEYWORD_DEFINITION
 	    *aKwDef, $ 	    /* (READ, BY ADDR):						    */0 	    /* Keyword definition to compare.				    */       char    *aKwStr,$ 	    /* (READ, BY ADDR):						    */, 	    /* Keyword string to compare.				    */       int	    vLength # 	    /* (READ, BY VAL):						    */ ) 	    /* Keyword string length.					    */   % )	/* Returns status flag:						    */ $ 	/*  1 - Successful match					    */1 	/*  0 - String does not match keyword.				    */ I 	/* ******************************************************************	*/    {      /*+									    */M     /*	If string length is less than minimum required length, no match.    */ M     /*	Otherwise, compare argument to option keyword to determine match.   */      /*									    */   +     if (vLength < kwdef_minlen(aKwDef)) {   / 	return 0;			    /* Too short, no match.	    */      } 4     else if (ustrncmp(kwdef_keyword(aKwDef), aKwStr,- 		max(vLength, kwdef_minlen(aKwDef))) == 0) { 2 	return 1;			    /* String matches keyword.	    */     } 
     else {% 	return 0;			    /* No match.		    */      }  }   N /*************************************************************************++*/
 int kwstrlen( N /* Finds the length of a keyword string. A keyword is a contiguous string   */> /* of alphanumerics and the underscore character '_'.			    */       char    *aKwStr $ 	    /* (READ, BY ADDR):						    */G 	    /* Keyword string to find length of. It is assumed to contain   */ G 	    /* no leading non-keyword characters (if it does, the length    */ ' 	    /* returned will be 0).					    */   H )	/* Returns length of string, or 0 if the first character is not a   */- 	/* valid keyword string character.				    */ G 	/********************************************************************/    { 2     int	    kwlen;				/* Length of keyword.	    */       for (kwlen = 0; 0 	isalnum(aKwStr[kwlen]) || aKwStr[kwlen] == '_';
 	kwlen++);     return kwlen;  }   N /*************************************************************************++*/ int kwstrleadlen( N /* Finds the length of leading non-keyword characters in a keyword string,  */N /* which may subsequently be treated as the offset to the beginning of the  */ /* keyword.								    */        char    *aKwStr $ 	    /* (READ, BY ADDR):						    */B 	    /* Keyword string that may contain leading non-keyword	    */ 	    /* characters.						    */   D )	/* Returns length of leading characters, or 0 if the string	    */@ 	/* contains no leading characters (or is a null string).	    */G 	/********************************************************************/    { 6     int	    leadlen;				/* Length of leading chars. */       for (leadlen = 0;  	!isalnum(aKwStr[leadlen])6 	&& aKwStr[leadlen] != '_' && aKwStr[leadlen] != '\0'; 	leadlen++);     return leadlen;  }   N /*************************************************************************++*/ int process_options(N /* Parses user command line arguments for options and dispatches option	    */N /* handler routines. Processing will terminate if any handler indicates	    */ /* failure.								    */*       int	    vArgc,# 	    /* (READ, BY VAL):						    *//< 	    /* Number of program argument strings in aArgv.		    */       char    *aArgv[],*$ 	    /* (READ, BY ADDR):						    */2 	    /* List of program argument strings.			    */       int	    vFirstOption,r# 	    /* (READ, BY VAL):						    */*G 	    /* Starting argument number for option. All preceding arguments */ 9 	    /* are assumed to be required, not options.			    */c       KEYWORD_DEFINITION 	    *aOptionTable$ 	    /* (READ, BY ADDR):						    */ 	    hG 	    /* Option definition table, containing option keywords (not	    */dG 	    /* including option switch character), handler routine ptrs,    */*G 	    /* and keyword translation codes.  The table is terminated by   */4G 	    /* an entry containing a NULL keyword ptr. The interface for a  */o& 	    /* handler routine is:					    */ 	    /*								    */v1 	    /*	    int handler(code, valstr, n)			    */  	    /*								    */eG 	    /* where code is the translation code for the matched keyword,  */*G 	    /* valstr is the ptr to the remainder of the argument string    */tF 	    /* following the matched argument characters, and n is the	    */G 	    /* option's argument position on the command line; however, a   */ G 	    /* handler is free to ignore (and therefore not declare) any    */*G 	    /* trailing subset of these parameters. The handler returns 1   */ G 	    /* if the argument is handled successfully, or 0 otherwise.	    */;G 	    /* This interface supports two styles of usage: common handlers *//G 	    /* may handle several different options, discriminating the	    */*E 	    /* matched option via the translation code; or individual	    *//A 	    /* handlers may be used for each option, ignoring the	    */ % 	    /* translation codes.					    *//  : )	/* Returns status indicating processing success:		    */B 	/*  1	- All options handled successfully (any unrecognized	    */+ 	/*	  arguments/options ignored).				    */S5 	/*  0	- An argument could not be processed.			    */rG 	/********************************************************************/*   {*.     int	    arg;				/* Argument number.	    */8     KEYWORD_DEFINITION				/* Current keyword def.	    */ 	    *kwdef;2     int	    kwlen;				/* Length of argument	    */ 						/* keyword portion	    */m$ 						/* (including switch char). */       /*+									    */M     /*	Save option table ptr in case we run into an option to process an   */ M     /*	options file, then attempt to process each command line argument    */t9     /*	following the required ones as an option.			    */      /*-									    */        mOptionTable = aOptionTable;2     for (arg = vFirstOption; arg < vArgc; arg++) {   	/*+								    */G 	/*  If argument starts with option switch character, go through	    */gG 	/*  option table trying to match it. On match, call option handler  */pG 	/*  and skip the rest of the option table; if the handler return    */iE 	/*  failure, abort. If no match, issue warning and ignore the	    */dG 	/*  argument. If it did not start with switch char, issue warning   */_ 	/*  and ignore it.						    *// 	/*-								    */ 						, 	if (*aArgv[arg] == CMDLINE_OPTION_SWITCH) {H 	    for (kwdef = aOptionTable; kwdef_keyword(kwdef) != NULL; kwdef++) {0 		if ((kwlen = cmdopt_cmp(kwdef, aArgv[arg]))) {  $ 						/* Match, call handler.	    */4 		    if (!(kwdef_handler(kwdef))(kwdef_code(kwdef)," 			    &aArgv[arg][kwlen], arg)) {, 			return 0;		/* Abort, handler failed!   */ 		    } , 		    break;			/* Skip rest of table.	    */ 		}q 	    }1 	    if (!kwlen) {			/* No option matches.	    */eB 		printf("ERROR: Unrecognized/ambiguous option %s\n", aArgv[arg]); 		return 0;g 	    } 	}* 	else {					/* Not an option argument.  */H 	    printf("ERROR: Unexpected command line argument %s\n", aArgv[arg]); 	    return 0; 	})     }						/* All options processed    */ )     return 1;					/* successfully.	    */  }u  N /*************************************************************************++*/ int process_keyword(L /* Parses keyword strings and dispatches to keyword handler routines.	    */G /* Processing will terminate if any handler indicates failure. 		    */a       char    *aKwStr,$ 	    /* (READ, BY ADDR):						    */E 	    /* Keyword string, with no leading, trailing, or enclosed	    */rG 	    /* whitespace. May be a single keyword, or a list of keywords,  */c' 	    /* separated by commas.					    */(       KEYWORD_DEFINITION 	    *aKwTable$ 	    /* (READ, BY ADDR):						    */E 	    /* Keyword definition table, containing keywords, handler	    */ G 	    /* routine ptrs, and keyword translation codes. The table is    */*G 	    /* terminated by an entry containing a NULL keyword ptr. The    */ 4 	    /* interface for a handler routine is:			    */ 	    /*								    */s( 	    /*	    int handler(code)					    */ 	    /*								    */oG 	    /* where code is the keyword translation code.  (there are no   */tG 	    /* arguments to the handler).  The handler returns 1 if the	    */ E 	    /* argument is handled successfully, or 0 otherwise. This	    */ G 	    /* interface supports two styles of usage: common handlers may  */wD 	    /* handle several different keywords, discriminating the	    */F 	    /* matched keyword via the translation code; or individual	    */B 	    /* handlers may be used for each keyword, ignoring the	    */% 	    /* translation codes.					    */c  : )	/* Returns status indicating processing success:		    */4 	/*  1	- All keywords handled successfully.			    */3 	/*  0	- A keyword could not be processed.			    */	G 	/********************************************************************/h   {m8     KEYWORD_DEFINITION				/* Current keyword def.	    */ 	    *kwdef;2     int	    kwlen;				/* Length of keyword.	    */4     int	    found;				/* Flag indicating match.   */<     char    *savestr = aKwStr;			/* Saved string ptr.	    */  $ 						/* For each keyword in list */:     while (*aKwStr != '\0') {			/* (or just one)...	    */  ' 	if ((kwlen = kwstrlen(aKwStr)) == 0) { 4 	    printf("ERROR: Missing keyword %s\n", savestr); 	    return 0; 	}* 	else {					/* Scan keyword table for   */ 						/* match.		    */+D 	    for (kwdef = aKwTable; kwdef_keyword(kwdef) != NULL; kwdef++) {2 		if ((found = kwdef_cmp(kwdef, aKwStr, kwlen))) {  $ 						/* Match, call handler with */! 						/* translation code.	    */R7 		    if (!(kwdef_handler(kwdef))(kwdef_code(kwdef))) {d, 			return 0;		/* Abort, handler failed!   */ 		    }l, 		    break;			/* Skip rest of table.	    */ 		}t 	    } 	    if (!found) {? 		printf("ERROR: Unrecognized/ambiguous keyword %s\n", aKwStr);t 		return 0;a 	    } 	}/ 	if (aKwStr[kwlen] == KEYWORD_LIST_SEPARATOR) { / 	    kwlen++;				/* Account for separator.   */* 	}2 	aKwStr += kwlen;			/* Advance past keyword.    */     }						f0     return 1;					/* All keywords processed   */ }						/* successfully.	    */  N /*************************************************************************++*/ int translate_keyword(H /* Parses a single keyword string and returns a translation code.	    */       char    *aKwStr,$ 	    /* (READ, BY ADDR):						    */E 	    /* Keyword string. If it contains any leading non-keyword	    */yG 	    /* characters, they will be ignored.  If it contains more than  */	B 	    /* one keyword, only the first one will be translated.	    */       KEYWORD_DEFINITION 	    *aKwTable$ 	    /* (READ, BY ADDR):						    */E 	    /* Keyword definition table, containing keywords, handler	    */(G 	    /* routine ptrs, and keyword translation codes. The table is    */*G 	    /* terminated by an entry containing a NULL keyword ptr. The    */lG 	    /* handler routine is called only if its ptr is non-NULL. The   */&4 	    /* interface for a handler routine is:			    */ 	    /*								    */n( 	    /*	    int handler(code)					    */ 	    /*								    */*G 	    /* where code is the keyword translation code.  (there are no   */aG 	    /* arguments to the handler).  The handler returns 1 if the	    */uE 	    /* argument is handled successfully, or 0 otherwise. This	    */iG 	    /* interface supports two styles of usage: common handlers may  */ D 	    /* handle several different keywords, discriminating the	    */F 	    /* matched keyword via the translation code; or individual	    */B 	    /* handlers may be used for each keyword, ignoring the	    */% 	    /* translation codes.					    */*  G )	/* Returns matching keyword translation code, or 0 if no match	    */ < 	/* found or the matching keyword handler returns 0.		    */G 	/********************************************************************/	   { 8     KEYWORD_DEFINITION				/* Current keyword def.	    */ 	    *kwdef;2     int	    kwlen;				/* Length of keyword.	    */<     char    *savestr = aKwStr;			/* Saved string ptr.	    */  *     if ((kwlen = kwstrlen(aKwStr)) == 0) {
 	return 0;     }c-     else {					/* Scan keyword table for   */  						/* match.		    */o@ 	for (kwdef = aKwTable; kwdef_keyword(kwdef) != NULL; kwdef++) {+ 	    if (kwdef_cmp(kwdef, aKwStr, kwlen)) {	  ! 						/* Match, if handler	    */r$ 						/* defined, call it with    */! 						/* translation code.	    */o" 		if (kwdef_handler(kwdef) != NULL6 		    && !(kwdef_handler(kwdef))(kwdef_code(kwdef))) {0 		    return 0;			/* Abort, handler failed!   */ 		}i( 		else {				/* Handler ok (or not	    */> 		    return kwdef_code(kwdef);	/* defined), return code.   */ 		}  	    } 	}" 	return 0;				/* No match.		    */     }h }l  N /*************************************************************************++*/ int process_options_file(tN /* Command line option handler to read an options file, where each line is  */N /* a single command line option, just as if it were parsed from the command */N /* line. Option files may be nested, just be careful of looped files or	    */N /* running out of resources due to excessive nesting. Comments lines and    */+ /* trailing comment are allowed.					    */	       int	    vKwCode,$ 	    /* (READ, BY ADDR):						    */G 	    /* Option keyword translation code, ignored by this routine.    */r       char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */ 7 	/*  1	- Successful processing of this option.			    */ C 	/*  0	- Option file name missing, or file cannot be opened.	    */	G 	/*****************************************************************--*/    {	7     FILE    *optfile;			    /* Options file ptr.	    */ 7     int	    rcount;			    /* Routine name count.	    */t=     char    option[512];		    /* Option string buffer.	    */*7     char    *optend;			    /* End-of-option ptr.	    */ 5     char    *optv;			    /* Option buffer ptr.	    */   + 					    /* Check for equal sign and	    */)+ 					    /* make sure there is a file.   */aE     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR && *aValStr != '\0') { / 	if ((optfile = fopen(aValStr, "r")) != NULL) {  	p( 					    /* For each line, locate	    */% 					    /* beginning of text.	    */f= 	    while (fgets(option, sizeof(option), optfile) != NULL) {d 		for (optv = option;d8 		    *optv != '\0' && *optv != CMDLINE_OPTION_SWITCH &&& 		    *optv != CMDLINE_OPTION_COMMENT; 		    optv++);+ 					    /* Ignore comment lines. Find   */ * 					    /* end of option field and	    */# 					    /* process option.		    */k( 		if (*optv != CMDLINE_OPTION_COMMENT) { 		    for (optend = optv;l6 			*optend > ' ' && *optend != CMDLINE_OPTION_COMMENT;
 			optend++);  		    *optend = '\0';g8 		    if (!process_options(1, &optv, 0, mOptionTable)) {
 			printf(9 			    "ERROR: Failure processing option \"%s\" in %s\n",  			    option, aValStr); 			fclose(optfile);r 			return 0; 		    }e 		}i 	    } 	    fclose(optfile);; 	    return 1; 	} 	else {e@ 	    printf("ERROR: Unable to open options file %s for input\n", 		aValStr);d 	    return 0; 	}     }[
     else {? 	printf("ERROR: %coptions option requires options file name\n",e 	    CMDLINE_OPTION_SWITCH);
 	return 0;     }* }*  