N /****************************************************************************/ /*									    */ ) /*  FACILITY:	Routine Analyzer					    */  /*									    */ / /*  MODULE:	Object Management Support				    */  /*									    */ G /*  AUTHOR:	Steve Branam, Network Product Support Group, Digital	    */ 6 /*		Equipment Corporation, Littleton, MA, USA.		    */ /*									    */ N /*  DESCRIPTION: This module contains all support routines for managing the */N /*  object types used by Routine Analyzer. They are: source files; routine  */N /*  definitions; and routine references. These objects are all descendents  */N /*  of the List Entry generic type, which means that they are linkable via  */N /*  a list entry header. This allows them to be stored in arbitrary ordered */N /*  and unordered lists, stacks, and queues. Support routines are supplied  */N /*  for creating and destroying objects from the heap, and for comparing,   */* /*  finding, and sorting them.						    */ /*									    */ " /*  REVISION HISTORY:							    */ /*									    */ / /*  V0.1-00 24-AUG-1994 Steve Branam					    */  /*									    */   /*	Original version.						    */ /*									    */ N /****************************************************************************/   #include "ranalyzer.h"  N /*************************************************************************++*/ char *def_ident(N /* Formats an indentification string for a definition entry. The string	    */H /* includes the routine name and the source file name (if known).	    */       DEFINITION
 	    *aDef% 	    /* (READ, BY ADDR):  					    */ / 	    /* Definition entry to identify.				    */   H )	/* Returns formatted string ptr.  WARNING: The returned ptr is the  */G 	/* address of a statically-defined string buffer, so the string	    */ / 	/* must be read from it immediately.				    */ G 	/*****************************************************************--*/    { + 					    /* Trace name string buffer.    */ .     static char	strbuf[MAX_ROUTINE_IDENT + 1];  .     sprintf(strbuf, "%s - %s", def_name(aDef), 	(isdefined_routine(aDef) ? . 	source_name(def_source(aDef)) : "External"));     return strbuf; }   N /*************************************************************************++*/ LANGUAGE_TRANSLATION *new_lang( E /* Creates and initializes a new language translation record.		    */        char    *aFileExt,$ 	    /* (READ, BY ADDR):						    *// 	    /* Source file extension string.				    */        SOURCE_LANGUAGE 
 	    vCode# 	    /* (READ, BY VAL):						    */ ( 	    /* Source language code.					    */  1 )	/* Returns ptr to initialized record.				    */ G 	/*****************************************************************--*/    { :     LANGUAGE_TRANSLATION		    /* Ptr to new record.	    */
 	    *record;   *     if ((record = (LANGUAGE_TRANSLATION *), 	    obj_alloc(sizeof(LANGUAGE_TRANSLATION),? 		sizeof(LANGUAGE_TRANSLATION), OBJ_NAME_LANGTRANS)) != NULL) { 
 	inc_nlang();  	record->code   = vCode;F 	record->fext   = new_str(aFileExt, MAX_FILE_NAME, OBJ_NAME_LANG_EXT); 	if (trace_obj_enabled()) { E 	    trace_new_obj(record, OBJ_NAME_LANGTRANS, aFileExt, num_lang());  	}     }      return record; }   N /*************************************************************************++*/  LANGUAGE_TRANSLATION *free_lang(N /* Frees a source language translation record. The file extension string is */C /* not freed; it is assumed to be picked up by another ptr.		    */        LANGUAGE_TRANSLATION 	    *aLangTrans% 	    /* (DELETE, BY ADDR):					    */ , 	    /* Translation block to free.				    */  " )	/* Returns NULL ptr.						    */G 	/*****************************************************************--*/    {      dec_nlang();     if (trace_obj_enabled()) {F 	trace_free_obj(aLangTrans, OBJ_NAME_LANGTRANS, lang_fext(aLangTrans), 	    num_lang());      } =     return obj_free(aLangTrans, sizeof(LANGUAGE_TRANSLATION),  	OBJ_NAME_LANGTRANS);  }   N /*************************************************************************++*/ SOURCEFILE *new_file( = /* Creates and initializes a new source file record.			    */        char    *aSourceName, $ 	    /* (READ, BY ADDR):						    */+ 	    /* Source file name string.					    */        int	    vTabSize# 	    /* (READ, BY VAL):						    */ ( 	    /* Source text tab size.					    */  1 )	/* Returns ptr to initialized record.				    */ G 	/*****************************************************************--*/    { 2     SOURCEFILE				    /* Ptr to new record.	    */
 	    *record;   >     if ((record = (SOURCEFILE *) obj_alloc(sizeof(SOURCEFILE),9 	    sizeof(SOURCEFILE), OBJ_NAME_SOURCEFILE)) != NULL) {  	inc_nfiles(); 	record->tabsize = vTabSize; 	record->seqnum	= num_files();3 	record->name	= new_str(aSourceName, MAX_FILE_NAME,  			    OBJ_NAME_FILENAME);A 	record->author	= new_str(PROGRAM_AUTHOR, sizeof(PROGRAM_AUTHOR),  			    OBJ_NAME_AUTHORNAME); 	if (trace_obj_enabled()) { < 	    trace_new_obj(record, OBJ_NAME_SOURCEFILE, aSourceName, 		num_files());  	}     }      return record; }   N /*************************************************************************++*/ SOURCEFILE *free_file(* /* Frees a source file record.						    */       SOURCEFILE 	    *aSourceFile % 	    /* (DELETE, BY ADDR):					    */ ' 	    /* Source file to free.					    */   " )	/* Returns NULL ptr.						    */G 	/*****************************************************************--*/    {      dec_nfiles();      if (trace_obj_enabled()) {1 	trace_free_obj(aSourceFile, OBJ_NAME_SOURCEFILE, , 	    source_name(aSourceFile), num_files());     } '     free_str(source_name(aSourceFile)); J     return obj_free(aSourceFile, sizeof(SOURCEFILE), OBJ_NAME_SOURCEFILE); }   N /*************************************************************************++*/ int compare_file( N /* Compares two file information records for ordering by file name. Entries */) /* are sorted alphabetically.						    */        SOURCEFILE 	    *aSourceFile1, $ 	    /* (READ, BY ADDR):						    */+ 	    /* First record to compare.					    */        SOURCEFILE 	    *aSourceFile2$ 	    /* (READ, BY ADDR):						    */+ 	    /* Second record to compare.				    */  	     @ )	/* Returns status value indicating comparison results:		    */+ 	/*    0	- File names are equal.					    */ @ 	/*  < 0 - File name of aSourceFile1 is less than name of	    */ 	/*	  aSourceFile2.						    */ C 	/*  > 0 - File name of aSourceFile1 is greater than name of	    */  	/*	  aSourceFile2.						    */ G 	/*****************************************************************--*/    { H     return strcmp(source_name(aSourceFile1), source_name(aSourceFile2)); }   N /*************************************************************************++*/ DEFINITION *new_def(C /* Creates and initializes a new routine definition record.		    */        char    *aName, $ 	    /* (READ, BY ADDR):						    */' 	    /* Routine name string.					    */        SOURCEFILE 	    *aSourceRecord $ 	    /* (READ, BY ADDR):						    */& 	    /* Source file record.					    */  1 )	/* Returns ptr to initialized record.				    */ G 	/*****************************************************************--*/    { 2     DEFINITION				    /* Ptr to new record.	    */
 	    *record;   >     if ((record = (DEFINITION *) obj_alloc(sizeof(DEFINITION),9 	    sizeof(DEFINITION), OBJ_NAME_DEFINITION)) != NULL) { 
 	inc_ndefs(); G 	record->name = new_str(aName, MAX_ROUTINE_NAME, OBJ_NAME_ROUTINENAME); ' 	set_def_source(record, aSourceRecord);  	if (trace_obj_enabled()) { C 	    trace_new_obj(record, OBJ_NAME_DEFINITION, aName, num_defs()); ! 	    if (aSourceRecord != NULL) { * 		printf("       Source file %s @ %lxh\n",1 		    source_name(aSourceRecord), aSourceRecord);  	    } 	    else { ? 		puts("       No source file (routine call, not definition)");  	    } 	}     }      return record; }   N /*************************************************************************++*/ DEFINITION *free_def( 0 /* Frees a routine definition record.					    */       DEFINITION
 	    *aDef% 	    /* (DELETE, BY ADDR):					    */ & 	    /* Definition to free.					    */  " )	/* Returns NULL ptr.						    */G 	/*****************************************************************--*/    {      dec_ndefs();     if (trace_obj_enabled()) {; 	trace_free_obj(aDef, OBJ_NAME_DEFINITION, def_ident(aDef),r 	    num_defs());c     }rC     return obj_free(aDef, sizeof(DEFINITION), OBJ_NAME_DEFINITION);S }	  N /*************************************************************************++*/ void trace_def( 4 /* Traces an action on a definition entry.				    */       DEFINITION 	    *aDef,/% 	    /* (READ, BY ADDR):  					    */e, 	    /* Definition entry to trace.				    */       char    *aAction% 	    /* (READ, BY ADDR):  					    */bD 	    /* Action string describing what is being done to entry.	    */  & )	/* No return value.      					    */G 	/*****************************************************************--*/i   {d     if (trace_obj_enabled()) {A 	printf("TRACE: %s %s %s @ %lxh\n", aAction, OBJ_NAME_DEFINITION,  	    def_ident(aDef), aDef);     }H }O  N /*************************************************************************++*/ int compare_def(N /* Compares two routine definition entries for ordering by routine, file    */N /* names, and line number (some languages allow multiple definition of the  */N /* same routine name within different scopes in the same module). Entries   */N /* are sorted alphabetically, with a special provision. If either or both   */N /* routine is currently undefined (i.e. the source file is not yet known),  */N /* they will be considered equal, since either 1) they are both forward	    */L /* references to an as yet undefined routine, or 2) one is the actual	    */L /* definition, while the other is just a reference, in which case the	    */N /* source information from the defined one is copied to the undefined one.  */       DEFINITION 	    *aDef1,% 	    /* (MODIFY, BY ADDR):					    */*D 	    /* First entry to compare. The source information may be	    */ 	    /* updated.							    */f       DEFINITION 	    *aDef2 % 	    /* (MODIFY, BY ADDR):					    */fE 	    /* Second entry to compare. The source information may be	    */e  	    /* updated.		   					    */ 	    *@ )	/* Returns status value indicating comparison results:		    */B 	/*    0	- Routine, file names, and line numbers are equal.	    */B 	/*  < 0 - Routine/file name/line of aDef1 less than aDef2.	    */E 	/*  > 0 - Routine/file name/line of aDef1 greater than aDef2.	    */ G 	/*****************************************************************--*/    {	7     int	    cmpstat;			    /* Comparison status.	    */        /*+									    */M     /*	Compare routine names. If they are unequal, no further comparison   */*M     /*	is needed. Otherwise, if both routines are defined, compare their   */ M     /*	files names and line numbers to determine final comparison status.  */l     /*-									    */  =     if ((cmpstat = ustrncmp(def_name(aDef1), def_name(aDef2), @ 	max(strlen(def_name(aDef1)), strlen(def_name(aDef2))))) == 0) {< 	if (isdefined_routine(aDef1) && isdefined_routine(aDef2)) {G 	    if ((cmpstat = compare_file(def_source(aDef1), def_source(aDef2)))a	 		== 0) {m0 		cmpstat = def_begin(aDef1) - def_begin(aDef2); 	    }
 	}				     	else {*   	    /*+								    */F 	    /*	One or both is undefined, assume either one is the actual   */F 	    /*	definition and the other is actually a reference to it, or  */F 	    /*	both are actually just references. In either case they are  */F 	    /*	equal in that they represent the same routine. Copy the	    */F 	    /*	source information from the defined one to the undefined    */6 	    /*	one so they will both be truly equal.			    */ 	    /*								    */* 		     	    cmpstat = 0;lG 	    if (isdefined_routine(aDef2)) { /* aDef2 is the defined one.    */A+ 		set_def_source(aDef1, def_source(aDef2));m 	    }4 	    else {			    /* Vice versa (or both undef).  */+ 		set_def_source(aDef2, def_source(aDef1));  	    }
 	}				         }*     return cmpstat;* }*  N /*************************************************************************++*/ int compare_def_file(rN /* Compares two routine definition entries for ordering by file name and    */M /* line number, i.e. by the order in which they occurred in the files.	    */ N /* Entries are sorted alphabetically, with the provision that undefined	    */0 /* routines sort before defined ones.					    */       DEFINITION 	    *aDef1,$ 	    /* (READ, BY ADDR):						    */* 	    /* First entry to compare.					    */       DEFINITION 	    *aDef2 $ 	    /* (READ, BY ADDR):						    */+ 	    /* Second entry to compare.					    */I 	    J@ )	/* Returns status value indicating comparison results:		    */4 	/*    0	- File and line numbers are equal.			    */7 	/*  < 0 - File/line of aDef1 less than aDef2.			    */M9 	/*  > 0 - File/line of aDef1 greater than aDef2.		    */eG 	/*****************************************************************--*/d   { 7     int	    cmpstat;			    /* Comparison status.	    */N       /*+									    */M     /*	If both routines are defined, compare file names and line numbers.  */*L     /*	Otherwise, undefined one is less than defined one (or both are	    */=     /*	undefined, and entries are considered equal).			    */      /*-									    */  ?     if (isdefined_routine(aDef1) && isdefined_routine(aDef2)) {tC 	if ((cmpstat = compare_file(def_source(aDef1), def_source(aDef2)))* 	    == 0) {3 	    cmpstat = def_begin(aDef1) - def_begin(aDef2);r 	}     }bN     else if (isdefined_routine(aDef2)) {    /* aDef2 is the defined one.    */ 	cmpstat = -1;     })B     else if (isdefined_routine(aDef1)) {    /* Vice versa.		    */
 	cmpstat = 1;l     }o,     else {				    /* Both undefined.		    */
 	cmpstat = 0;*     }*     return cmpstat;* }*  N /*************************************************************************++*/ DEFINITION *find_def(fN /* Searches the global routine list for a routine definition entry from the */N /* specified source file. If an entry is not found, creates and initializes */( /* a new one and returns it.						    */       char    *aName,2$ 	    /* (READ, BY ADDR):						    */. 	    /* Routine name string to find.				    */       SOURCEFILE 	    *aSourceRecorda$ 	    /* (READ, BY ADDR):						    */G 	    /* Source file record to find. If a NULL ptr is passed, any	    */FG 	    /* routine definition from any file will satisfy the request.   */>F 	    /* In any case, if a definition entry must be created, the	    */G 	    /* value passed as aSourceRecord will be used to initialize the */** 	    /* definition source file.					    */    @ )	/* Returns ptr to found (and possibly created) record.		    */G 	/*****************************************************************--*/+   {E4     DEFINITION				    /* Ptr to search entry.	    */ 	    *searchdef;3     DEFINITION				    /* Ptr to found entry.	    */A
 	    *curdef;	       /*+									    */M     /*	Create a temporary search entry and use it to search the definition */,M     /*	list for a match. If a match is found, discard the search entry and */pM     /*	return the found entry. Otherwise, add the search entry to the list */*K     /*	and return it as the "found" entry. This guarantees that this	    */o5     /*	routine will always "find" an entry.				    */T     /*-									    */  .     searchdef = new_def(aName, aSourceRecord);A     if ((curdef = find_ordered_entry(global_deflist(), searchdef,m 		    compare_def)) != NULL) {< 	free_def(searchdef);		    /* Found match, return it.	    */ 	return curdef;)     } 4     else {				    /* Match not found, use search  */, 	add_def(searchdef);		    /* entry.			    */ 	return searchdef;     }  }e  N /*************************************************************************++*/ REFERENCE *new_ref( B /* Creates and initializes a new routine reference record.		    */       long    vLine,# 	    /* (READ, BY VAL):						    */*B 	    /* Line number in source file where routine is called.	    */       DEFINITION 	    *aRefDef,$ 	    /* (READ, BY ADDR):						    */6 	    /* Refenced routine's definition record.			    */       DEFINITION 	    *aCallerDef$ 	    /* (READ, BY ADDR):						    */- 	    /* Caller's definition record.				    */*  1 )	/* Returns ptr to initialized record.				    */ G 	/*****************************************************************--*/d   {e1     REFERENCE				    /* Ptr to new record.	    */o
 	    *record;i  O     if ((record = (REFERENCE *) obj_alloc(sizeof(REFERENCE), sizeof(REFERENCE),*$ 	    OBJ_NAME_REFERENCE)) != NULL) {
 	inc_nrefs();e 	record->definition = aRefDef; 	record->line       = vLine;! 	record->caller     = aCallerDef;  	if (trace_obj_enabled()) { B 	    trace_new_obj(record, OBJ_NAME_REFERENCE, def_name(aRefDef),  		num_refs());A 	    printf("       Definition is @ %lxh, caller is %s @ %lxh\n",aB 		aRefDef, (aCallerDef == NULL ? "(none)" : def_name(aCallerDef)),, 		(aCallerDef == NULL ? NULL : aCallerDef)); 	}     }*     return record; }   N /*************************************************************************++*/ REFERENCE *free_ref(/ /* Frees a routine reference record.					    */*  
     REFERENCE*
 	    *aRef% 	    /* (DELETE, BY ADDR):					    */*% 	    /* Reference to free.					    */   " )	/* Returns NULL ptr.						    */G 	/*****************************************************************--*/    {p     dec_nrefs();     if (trace_obj_enabled()) {) 	trace_free_obj(aRef, OBJ_NAME_REFERENCE,n2 	    def_ident(ref_definition(aRef)), num_refs());     }oB     return obj_free(aRef, sizeof(REFERENCE), OBJ_NAME_DEFINITION); }   N /*************************************************************************++*/ void trace_ref(b3 /* Traces an action on a reference entry.				    */n  
     REFERENCE  	    *aRef,t% 	    /* (READ, BY ADDR):  					    */h+ 	    /* Reference entry to trace.				    */        char    *aAction% 	    /* (READ, BY ADDR):  					    */ D 	    /* Action string describing what is being done to entry.	    */  & )	/* No return value.      					    */G 	/*****************************************************************--*/    {f     if (trace_obj_enabled()) {@ 	printf("TRACE: %s %s %s @ %lxh\n", aAction, OBJ_NAME_REFERENCE,, 	    def_ident(ref_definition(aRef)), aRef);     }/ }p  N /*************************************************************************++*/ int compare_caller( N /* Compares two reference entries for ordering by caller routine. Entries   */N /* are sorted alphabetically, and by line number within the same caller.    */  
     REFERENCEt 	    *aRef1,$ 	    /* (READ, BY ADDR):						    */* 	    /* First entry to compare.					    */  
     REFERENCE  	    *aRef2 $ 	    /* (READ, BY ADDR):						    */+ 	    /* Second entry to compare. 				    */n 	    f@ )	/* Returns status value indicating comparison results:		    */( 	/*    0	- Callers are equal.					    */9 	/*  < 0 - Caller from aRef1 is less than aRef2.			    */d; 	/*  > 0 - Caller from aRef1 is greater than aRef2.		    *//G 	/*****************************************************************--*/s   {(7     int	    cmpstat;			    /* Comparison status.	    */        /*+									    */I     /*	If the caller names are the same, compare the line numbers.	    */(     /*-									    */  M     if ((cmpstat = compare_def(ref_caller(aRef1), ref_caller(aRef2))) == 0) {	1 	cmpstat = ref_offset(aRef1) - ref_offset(aRef2);n     }t     return cmpstat;e }t  N /*************************************************************************++*/ int compare_caller_name(M /* Compares two reference entries for equality by caller routine name.	    */n4 /* Entries are not sorted by this routine.				    */  
     REFERENCEi 	    *aRef1,$ 	    /* (READ, BY ADDR):						    */* 	    /* First entry to compare.					    */  
     REFERENCE  	    *aRef2 $ 	    /* (READ, BY ADDR):						    */+ 	    /* Second entry to compare. 				    */s 	    f@ )	/* Returns status value indicating comparison results:		    */( 	/*    0	- Callers are equal.					    */+ 	/*   -1 - Callers are not equal.				    */_G 	/*****************************************************************--*/    {*     /*+									    */+     /*	Compare the caller names.					    */+     /*-									    */  P     if (strcmp(def_name(ref_caller(aRef1)), def_name(ref_caller(aRef2))) != 0) { 	return -1;r     }b
     else {
 	return 0;     }r }i  N /*************************************************************************++*/ int compare_ref(L /* Compares two reference entries for equality by routine definition.	    */4 /* Entries are not sorted by this routine.				    */  
     REFERENCEa 	    *aRef1,$ 	    /* (READ, BY ADDR):						    */* 	    /* First entry to compare.					    */  
     REFERENCEp 	    *aRef2*$ 	    /* (READ, BY ADDR):						    */+ 	    /* Second entry to compare. 				    */  	    l@ )	/* Returns status value indicating comparison results:		    */+ 	/*    0	- Definitions are equal.				    */l/ 	/*   -1 - Definitions are not equal.				    */*G 	/*****************************************************************--*/    {      /*+									    */*     /*	Compare the definitions.					    */     /*-									    */  I     if (compare_def(ref_definition(aRef1), ref_definition(aRef2)) != 0) {h 	return -1;i     } 
     else {
 	return 0;     }t }r  N /*************************************************************************++*/ REFERENCE *find_ref_in_tree(@ /* Find the first reference to a routine in a call tree.		    */       DEFINITION 	    *aTreeRoot,$ 	    /* (READ, BY ADDR):						    */G 	    /* Routine definition entry that is root of this call tree.	    */i       char    *aName$ 	    /* (READ, BY ADDR):						    */- 	    /* Routine name to search for.				    */i  H )	/* Returns ptr to found reference, or NULL if no matching reference */ 	/* found.							    */tG 	/*****************************************************************--*/    {*6     REFERENCE				    /* Current reference being	    */4 	    *curref;			    /* checked at this level.	    */3     REFERENCE				    /* Reference found in a	    */y* 	    *foundref;			    /* subtree.			    */       /*+									    */M     /*	For each routine referenced at this level, see if it matches the    */ M     /*	name. If not, recursively search that routine's subtree for a match */fM     /*	and return the match if found. If no match is found at this level   */:F     /*	or any sublevels, no such reference exists in this tree.	    */     /*-									    */  B     for (curref = list_first(def_refs(aTreeRoot)); curref != NULL; 	curref = next_entry(curref)) {e< 	if (strcmp(def_name(ref_definition(curref)), aName) == 0) {: 	    return curref;		    /* Found it at this level!	    */ 	}F 	else if ((foundref = find_ref_in_tree(ref_definition(curref), aName)) 	    != NULL) { ; 	    return foundref;		    /* Found it in a subtree!	    */* 	}     }-9     return NULL;			    /* Did not find it anywhere.    */  }/  N /*************************************************************************++*/ void discard_dup_refs(L /* Discards any duplicate reference entries in any definition ref and	    */N /* caller lists. This leaves at most one reference for each routine in any  */ /* list.								    */  !     /* No arguments.							    */   ! )	/* No return value.						    */aG 	/*****************************************************************--*/    {u1     DEFINITION					/* Current routine entry.   */*
 	    *curdef;/+     REFERENCE					/* Current routine	    */o& 	    *curref;				/* reference.		    */4     LIST    curlist;				/* Current list being	    */ 						/* trimmed.		    */h  ?     for (curdef = list_first(global_deflist()); curdef != NULL;  	curdef = next_entry(curdef)) {t   	/*+								    */G 	/*  First, if the current routine is defined, trim its reference    */ G 	/*  list. To do this, move the reference list to a temporary list,  */*E 	/*  clearing the definition's list. For each reference in the	    */uG 	/*  temporary list, see if a matching entry already exists in the   */AF 	/*  definition's list. If so, this is a duplicate, discard it;	    */G 	/*  otherwise, add it to the definition's list. This regrows the    */	G 	/*  definition's list back with only one reference to each routine. */  	/*-								    */  ! 	if (isdefined_routine(curdef)) {	$ 	    trace_def(curdef, "Trim refs");+ 	    copy_list(def_refs(curdef), &curlist);t! 	    init_list(def_refs(curdef));*9 	    while ((curref = dequeue_entry(&curlist)) != NULL) {-# 		if (find_ref(curdef, curref)) {		t7 		    free_ref(curref);		/* Duplicate, discard it.   */r 		}R 		else {: 		    add_ref(curdef, curref);	/* New one, keep it.	    */$ 		    trace_ref(curref, "Keep ref"); 		}e 	    } 	}   	/*+								    */G 	/*  Now, regardless of whether or not the routine is defined, trim  */a8 	/*  its caller list, using the same procedure.			    */ 	/*-								    */  # 	trace_def(curdef, "Trim callers");($ 						/* Move list to temporary.  */* 	copy_list(def_callers(curdef), &curlist);  	init_list(def_callers(curdef));5 	while ((curref = dequeue_entry(&curlist)) != NULL) {}' 	    if (find_caller(curdef, curref)) {*3 		free_ref(curref);		/* Duplicate, discard it.   */* 	    } 	    else {E9 		add_caller(curdef, curref);	/* New one, keep it.	    */	# 		trace_ref(curref, "Keep caller");f 	    } 	}     }B }D  N /*************************************************************************++*/ void sort_file_order(*N /* Resorts the global definition list in order by file, then routine within */ /* file.								    */  !     /* No arguments.							    */   ! )	/* No return value.						    */rG 	/*****************************************************************--*/e   {F1     DEFINITION					/* Current routine entry.   */*
 	    *curdef;*2     LIST    templist;				/* Temporary list.	    */       /*									    */ M     /*	Perform an insertion sort on the global definition list.  To do	    */ M     /*	this, move the definition list to a temporary list, clearing the    */ M     /*	definition list. For each definition, dequeue it from the temporary */nL     /*	list and insert it back into the definition list according the	    */-     /*	file/routine/line ordering.					    */*     /*-									    */  +     copy_list(global_deflist(), &templist);n      init_list(global_deflist());9     while ((curdef = dequeue_entry(&templist)) != NULL) {r& 	trace_def(curdef, "Sort file order");B 	insert_ordered_entry(global_deflist(), curdef, compare_def_file);     }* }+  