N /****************************************************************************/ /*									    */ ) /*  FACILITY:	Routine Analyzer					    */  /*									    */ # /*  MODULE:	Main Module						    */  /*									    */ G /*  AUTHOR:	Steve Branam, Network Product Support Group, Digital	    */ 6 /*		Equipment Corporation, Littleton, MA, USA.		    */ /*									    */ N /*  DESCRIPTION: This is the main module for Routine Analyzer. It contains  */N /*  the main routine, command option handlers, and the main application	    */> /*  routines for processing product and source files.			    */ /*									    */ " /*  REVISION HISTORY:							    */ /*									    */ / /*  V1.0-00 27-JAN-1995 Steve Branam					    */  /*									    */ F /*	Modified help logic when insufficient arguments supplied and	    */ /*	cleaned up help.						    */  /*									    */ / /*  V0.1-00 24-AUG-1994 Steve Branam					    */  /*									    */   /*	Original version.						    */ /*									    */ N /****************************************************************************/  < #define MAIN_MODULE			    /* This is the main module.	    */ #include <ctype.h> #include "ranalyzer.h"    # extern language_element c_parser(); ' extern language_element bliss_parser(); % extern language_element dcl_parser(); & extern language_element text_parser();  N /*************************************************************************++*/ PARSER get_parser(N /* Returns the parser function appropriate for the source language, based   */* /* on the file name extension.						    */       char    *aSourceName, % 	    /* (READ, BY ADDR):  					    */ + 	    /* Source file name string.					    */        char    **aParserName & 	    /* (WRITE, BY ADDR):  					    */A 	    /* Parser name string ptr, set to parser name string.	    */   . )	/* Returns ptr to parser function.				    */G 	/*****************************************************************--*/    { ?     KEYWORD_DEFINITION			    /* Current keyword definition.  */  	    *curkwdef; ?     LANGUAGE_TRANSLATION		    /* Current language trans.	    */  	    *curtrans; 8     char    *extstr;			    /* File extension ptr.	    */  %     if (global_langtable() == NULL) { 1 	set_lang_table(list_entries(global_langlist())); $ 	for (curkwdef = global_langtable();( 	    (curtrans = remove_lang()) != NULL; 	    curkwdef++) {6 	    set_kwdef_keyword(curkwdef, lang_fext(curtrans));= 	    set_kwdef_minlen(curkwdef, strlen(lang_fext(curtrans))); 3 	    set_kwdef_code(curkwdef, lang_code(curtrans));  	    free_lang(curtrans);  	}     }          /*+									    */M     /*	Scan back from end of file name string for file extension. If not   */ M     /*	found, can't identify parser. Otherwise, locate end of extension    */ ?     /*	string and compare it to known file extensions.			    */      /*-									    */  4     for (extstr = &aSourceName[strlen(aSourceName)];8 	extstr >= aSourceName && *extstr != FILE_EXT_SEPARATOR; 	extstr--);      if (extstr < aSourceName) { I 	printf("ERROR: No file extension specified for file %s\n", aSourceName); 
 	return NULL;      } 
     else {
 	extstr++;9 	switch (translate_keyword(extstr, global_langtable())) { ? 	case LANGUAGE_UNKNOWN:		    /* No matches on file type.	    */  	    printf(? 	    "ERROR: Unable to identify source language for file %s\n",  		aSourceName);  	    return NULL;  	    break;  	case LANGUAGE_TEXT:3 	    *aParserName = "text (no language structure)";  	    return text_parser; 	    break;  	case LANGUAGE_DCL:  	    *aParserName = "DCL"; 	    return dcl_parser;  	    break;  	case LANGUAGE_CC: 	    *aParserName = "C"; 	    return c_parser;  	    break;  	case LANGUAGE_BLISS:  	    *aParserName = "BLISS"; 	    return bliss_parser;  	    break;  	}     }  }   N /*************************************************************************++*/ int analyze_file( N /* Analyze the code in a product source file, using the designated parser.  */N /* This adds routine definition entries for every routine defined in the    */M /* file, and for every unknown routine referenced in the file. Unknown	    */ N /* routines are expected to be defined later in the file, in another file,  */< /* or in another product (i.e. a run-time library).			    */       char    *aSourceName,   		/* (READ, BY ADDR):					    */' 		/* Source file name string.				    */        int	    vTabSize,   		/* (READ, BY ADDR):					    */$ 		/* Source file tab size.				    */       PARSER  aParser,  		/* (READ, BY ADDR):					    */@ 		/* Source language parse routine, called to return the next */* 		/* token from the source file.				    */       char    *aLangName  		/* (READ, BY ADDR):					    */$ 		/* Language name string.				    */  * )	/* Returns status indication:					    */. 	/*  1	- File analyzed successfully.				    */3 	/*  0   - File not analyzed successfully.			    */ G 	/*****************************************************************--*/    { 8     DEFINITION				    /* Current routine definition.  */ 	    *curdef = NULL;		  2     DEFINITION				    /* Referenced routine	    */* 	    *refdef;			    /* definition.		    */7     REFERENCE				    /* Current routine reference.   */ 
 	    *curref;      SOURCEFILE3 	    *curfile;			    /* Current source file.	    */ 9     FILE    *sourcefile;		    /* Source file ptr.		    */ N     char    element[MAX_ROUTINE_NAME + 1];  /* Language element buffer.	    */ 		=     language_element			    /* Type of language element.    */  	    element_type;?     long    sourceline;		    /* Line number in source file.  */  		     /*+									    */M     /*	Add a new source file entry to the source file list, then analyze   */ M     /*	the language elements in the source file, adding definition and	    */ 6     /*	reference entries for routines found.				    */     /*-									    */  $ 					    /* Open source file.	    */9     if ((sourcefile = fopen(aSourceName, "r")) == NULL) { ; 	printf("ERROR: Unable to open source file %s for input\n",  	    aSourceName);
 	return 0;     } 4     else {				    /* Create new source file entry */& 					    /* and add it to list.	    */+ 	curfile = new_file(aSourceName, vTabSize);  	add_file(curfile);		     % 	list_file_begin(curfile, aLangName);  	new_list_line(curfile);+ 					    /* Process next source language */  					    /* element.			    */ @ 	while ((element_type = (*aParser)(sourcefile, curfile, element,) 				    &sourceline)) != END_OF_SOURCE) {  	    switch(element_type) {  	    case PARSE_ERROR:   		    /*+							    */? 		    /*	Parser believes some source construct is invalid,   */ ( 		    /*	give up on this file.				    */ 		    /*-							    */ 	      		    fclose(sourcefile);  		    return 0;  		    break; 	    case ROUTINE_DEF_BEGIN:   		    /*+							    */? 		    /*	Start of a routine found: Get its definition entry  */ ? 		    /*	for this file and save begin line number. Add a	    */ ? 		    /*	source reference, passing this def as the caller as */ ? 		    /*	well as the definition so that it will be inserted  */ ) 		    /*	in alphabetical order.				    */  		    /*-							    */	 					     * 		    curdef = find_def(element, curfile);( 		    set_def_begin(curdef, sourceline);? 		    add_srcref(curfile, new_ref(sourceline, curdef, curdef)); & 		    list_def_begin(curfile, curdef); 		    break; 	    case ROUTINE_DEF_END:   		    /*+							    */? 		    /*	End of a routine found: Save end line number and    */ 2 		    /*	clear current definition context.		    */ 		    /*-							    */  & 		    set_def_end(curdef, sourceline);# 		    inc_source_routines(curfile); 6 		    inc_source_rlength(curfile, def_length(curdef));7 		    inc_source_calls(curfile, def_num_calls(curdef)); $ 		    list_def_end(curfile, curdef); 		    curdef = NULL;	      		    break; 	    case ROUTINE_REF:   		    /*+							    */? 		    /*	Routine reference found: Make sure this is inside a */ ? 		    /*	routine definition (as opposed to a static data	    */ ? 		    /*	initializer macro or some other construct that the  */ ? 		    /*	parser has misconstrued as a routine call). Find    */ ? 		    /*	the definition of the referenced routine, add a	    */*? 		    /*	reference entry for it to current routine being	    */Y= 		    /*	defined, and add a caller entry to it for the	    */	# 		    /*	current routine.				    */  		    /*-							    */   		    if (curdef != NULL) {   $ 			refdef = find_def(element, NULL);0 			curref = new_ref(sourceline, refdef, curdef); 			add_ref(curdef, curref);     $ 		        list_ref(curfile, curref);0 			curref = new_ref(sourceline, refdef, curdef); 			add_caller(refdef, curref); 		    }	 		    break; 	    } 	}   	/*+								    */G 	/*  Parser has reached the end of the source file, close it and	    */ ? 	/*  roll up all file counts to total product counters.		    */	 	/*-								    */   	fclose(sourcefile);. 	inc_total_comments(source_comments(curfile));2 	inc_total_statements(source_statements(curfile));( 	inc_total_mixed(source_mixed(curfile));( 	inc_total_empty(source_empty(curfile));. 	inc_total_routines(source_routines(curfile));, 	inc_total_rlength(source_rlength(curfile));( 	inc_total_calls(source_calls(curfile)); 	list_file_end(curfile);
 	return 1;     }o }e  N /*************************************************************************++*/ int analyze_product(N /* Processes a product definition file. This file contains a line for the   */N /* product name, then one line for each product source file.  Analyzes the  */N /* source code in each product source file, then produces analysis reports  */  /* for the product.							    */       char    *aProductNameh  		/* (READ, BY ADDR):					    */( 		/* Product file name string.				    */ 		* )	/* Returns status indication:					    */0 	/*  1	- Product analyzed successfully.			    */6 	/*  0   - Product not analyzed successfully.			    */G 	/*****************************************************************--*/s   {t:     FILE    *productfile;		    /* Product file ptr.	    */' 					    /* Product name buffer.	    */ 0     char    productname[MAX_PRODUCT_NAME + 1];  N     char    sourcebuf[MAX_FILE_NAME + 1];   /* Source file line buffer.	    */N     char    sourcename[MAX_FILE_NAME + 1];  /* Source file name buffer.	    */N     char    outputname[MAX_FILE_NAME + 1];  /* Output file name buffer.	    */=     PARSER  *parser;			    /* Parser function for source   */a 					    /* language.		    */m@     char    *parsername;		    /* Parser language name ptr.    */4     int	    tabsize;			    /* File tab size.		    */       /*+									    */M     /*	Analyze each source file listed in the product definition file.	    */_K     /*	Abort if the product file cannot be opened or an error occurs	    */oH     /*	during analysis. The appropriate parser must be determined	    */-     /*	individually for each file.					    */      /*-									    */  % 					    /* Open product file.	    */n;     if ((productfile = fopen(aProductName, "r")) == NULL) { < 	printf("ERROR: Unable to open product file %s for input\n", 	    aProductName);E
 	return 0;     })L     else if (fgets(productname, sizeof(productname), productfile) == NULL) {< 	printf("ERROR: Unable to read product name from file %s\n", 	    aProductName); 
 	return 0;     }k
     else {- 	productname[strlen(productname) - 1] = '\0';O 	set_product_name(productname);p" 	list_product_begin(aProductName);+ 					    /* Get next source file name.   */eC 	while (fgets(sourcebuf, sizeof(sourcebuf), productfile) != NULL) {A  @ 	    if (sscanf(sourcebuf, "%s %d", sourcename, &tabsize) < 2) { 		tabsize = TAB_SIZE;; 	    }+ 					    /* Get parser for file type and */m! 					    /* analyze file.		    */ ? 	    if ((parser = get_parser(sourcename, &parsername)) == NULL > 		|| !analyze_file(sourcename, tabsize, parser, parsername)) { 		fclose(productfile); 		return 0;l 	    } 	} 	fclose(productfile); & 	list_analysis_complete(aProductName);     }*       /*+									    */M     /*	Write alphabetical reports (ordered by routine name): defined and   */eM     /*	undefined routines; calls/callers for defined routines; and call    */ K     /*	trees for defined routines. Before the last report, duplicate	    */fM     /*	references must be trimmed from the routine definitions so that	    */tM     /*	minimal call trees can be generated (i.e. a routine will appear	    */ ;     /*	only once in a given level of a call tree).			    */      /*-									    */       if (rpt_html_enabled()) {  	assign_xreffiles(); 	assign_byfilefiles();     }	     if (rpt_defined_enabled())G 	report_defined(make_filename(outfile_prefix(), OUTFILE_SUFFIX_DEFLIST,/# 	    report_filext(), outputname));       if (rpt_undefined_enabled())K 	report_undefined(make_filename(outfile_prefix(), OUTFILE_SUFFIX_UNDEFLIST, # 	    report_filext(), outputname));a     if (rpt_calls_enabled())C 	report_calls(make_filename(outfile_prefix(), OUTFILE_SUFFIX_CALLS,u# 	    report_filext(), outputname));n<     discard_dup_refs();			    /* Trim duplicate refs.	    */ #if 0*     if (rpt_trees_enabled())L 	report_call_trees(make_filename(outfile_prefix(), OUTFILE_SUFFIX_CALLTREES,# 	    report_filext(), outputname));  #endif     if (rpt_xrefs_enabled())B 	report_xrefs(make_filename(outfile_prefix(), OUTFILE_SUFFIX_XREF,# 	    report_filext(), outputname));r       /*+									    */M     /*	Sort the file definitions by source file and beginning line number  */ M     /*	and write the defined routines in this order. Finally, report the   */n"     /*	product files.							    */     /*-									    */       sort_file_order();     if (rpt_byfile_enabled()) F 	report_by_file(make_filename(outfile_prefix(), OUTFILE_SUFFIX_BYFILE,# 	    report_filext(), outputname));w     if (rpt_files_enabled())C 	report_files(make_filename(outfile_prefix(), OUTFILE_SUFFIX_FILES, # 	    report_filext(), outputname));      if (rpt_source_enabled())f& 	report_source(OUTFILE_SUFFIX_SOURCE);  #     list_product_end(aProductName); 4     return 1;				    /* Successful completion	    */ }"  N /*************************************************************************++*/ int cmdopt_set(rJ /* Common command line option handler to set simple toggle options.	    */       int	    vToggleFlag	# 	    /* (READ, BY VAL):						    */m& 	    /* Toggle flag to set.					    */  G )	/* Returns 1 to indicate successful processing of this option.	    */ G 	/*****************************************************************--*/w   {(     set_option(vToggleFlag);
     return 1;e }l  N /*************************************************************************++*/ int cmdopt_clear(OL /* Common command line option handler to clear simple toggle options.	    */       int	    vToggleFlagg# 	    /* (READ, BY VAL):						    */ ( 	    /* Toggle flag to clear.					    */  G )	/* Returns 1 to indicate successful processing of this option.	    */ G 	/*****************************************************************--*/t   {n     clear_option(vToggleFlag);
     return 1;  }e  N /*************************************************************************++*/ int cmdopt_trace(l@ /* Command line option handler to set specified traces.			    */       int	    vOptCode, # 	    /* (READ, BY VAL):						    */ G 	    /* Option keyword translation code, ignored by this routine.    */f       char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */ 7 	/*  1	- Successful processing of this option.			    */ + 	/*  0	- Report keywords missing.				    */rG 	/*****************************************************************--*/_   {u' 					    /* Trace option keyword	    */i# 					    /* dispatch table.		    */g,     static KEYWORD_DEFINITION keywords[] = {0 	{"memalloc",	3, cmdopt_set,  TRACE_MEM_ENABLE},/ 	{"strings",	3, cmdopt_set,  TRACE_STR_ENABLE},N/ 	{"objects",	3, cmdopt_set,  TRACE_OBJ_ENABLE}, / 	{"reports",	3, cmdopt_set,  TRACE_RPT_ENABLE},n1 	{"plevel",	3, cmdopt_set,  TRACE_PLEVEL_ENABLE},r1 	{"blevel",	3, cmdopt_set,  TRACE_BLEVEL_ENABLE},*1 	{"parser",	3, cmdopt_set,  TRACE_PARSER_ENABLE},u. 	{NULL,		0, NULL}	    /* End of table.		    */     };  E     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR && *aValStr != '\0') {t+ 	if (!process_keyword(aValStr, keywords)) {f6 	    printf("ERROR: Unable to process %ctrace option", 		CMDLINE_OPTION_SWITCH);  	    return 0; 	} 	else {c 	    return 1; 	}     } 
     else {F 	printf("ERROR: %ctrace option requires trace type keyword or list\n", 	    CMDLINE_OPTION_SWITCH);
 	return 0;     }f }u  N /*************************************************************************++*/ int cmdopt_log((I /* Command line option handler to set log file for stdout logging.	    */        int	    vOptCode,/# 	    /* (READ, BY VAL):						    */cG 	    /* Option keyword translation code, ignored by this routine.    */l       char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */a7 	/*  1	- Successful processing of this option.			    */)@ 	/*  0	- Log file name missing, or file cannot be opened.	    */G 	/*****************************************************************--*/    {t     /*+									    */M     /*	If a file name value has been supplied, reopen stdout on that file. */*     /*-									    */  E     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR && *aValStr != '\0') {s- 	if (freopen(aValStr, "w", stdout) != NULL) {a 	    return 1; 	} 	else {tG 	    printf("ERROR: Unable to open log file %s for output\n", aValStr);s 	    return 0; 	}     }s
     else {7 	printf("ERROR: %clog option requires log file name\n",	 	    CMDLINE_OPTION_SWITCH);
 	return 0;     }E }   N /*************************************************************************++*/ int cmdopt_list(N /* Command line option handler to enable listing file. Attempts to create   */ /* listing file.							    */   !     /* No arguments.							    */*  % )	/* Returns status code:						    */ 7 	/*  1	- Successful processing of this option.			    */	/ 	/*  0	- Unable to open listing file.				    */ G 	/*****************************************************************--*/A   {EB     char    name[MAX_FILE_NAME + 1];		/* Listing file name.	    */       set_option(LIST_ENABLE);K     if (open_list_file(make_filename(outfile_prefix(), OUTFILE_SUFFIX_LIST,O( 	    OUTFILE_EXT_LIST, name)) == NULL) {D 	printf("ERROR: Unable to open listing file %s for output\n", name);
 	return 0;     } 
     else {
 	return 1;     }r }   N /*************************************************************************++*/ int cmdopt_outprefix( G /* Command line option handler to set report file output prefix.	    */        int	    vOptCode, # 	    /* (READ, BY VAL):						    */rG 	    /* Option keyword translation code, ignored by this routine.    */e       char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */c7 	/*  1	- Successful processing of this option.			    */p# 	/*  0	- Prefix missing.					    */rG 	/*****************************************************************--*/p   {tE     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR && *aValStr != '\0') {n 	set_outfile_prefix(aValStr);d
 	return 1;     } 
     else {= 	printf("ERROR: %coutprefix option requires prefix string\n",t 	    CMDLINE_OPTION_SWITCH);
 	return 0;     }( }o  N /*************************************************************************++*/ int cmdopt_description(cN /* Command line option handler to set the product description file name.    */       int	    vOptCode,I# 	    /* (READ, BY VAL):						    */fG 	    /* Option keyword translation code, ignored by this routine.    */r       char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */s7 	/*  1	- Successful processing of this option.			    */ - 	/*  0	- Option value out of range.				    */aG 	/*****************************************************************--*/u   {n1     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR) {d" 	set_product_description(aValStr);
 	return 1;     } >     printf("ERROR: %cdescription option requires file name\n", 	CMDLINE_OPTION_SWITCH);
     return 0;t }   N /*************************************************************************++*/ int cmdopt_separate(N /* Command line option handler to set list of routines for which separate   */N /* call trees MUST be generated if they call anything. The option value is  */N /* the name of a file containing a list of routine names, one per line.	    */       int	    vOptCode, # 	    /* (READ, BY VAL):						    */ G 	    /* Option keyword translation code, ignored by this routine.    */f       char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */u7 	/*  1	- Successful processing of this option.			    */eA 	/*  0	- List file name missing, or file cannot be opened.	    */eG 	/*****************************************************************--*/S   {_>     FILE    *sepfile;			    /* Sep routine list file ptr.   */7     int	    rcount;			    /* Routine name count.	    */lJ     char    rname[MAX_ROUTINE_NAME * 2];    /* Routine name buffer.	    */=     char    **rlist;			    /* Ptr to routine name list.    */e     E     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR && *aValStr != '\0') {n/ 	if ((sepfile = fopen(aValStr, "r")) != NULL) {o$ 						/* Count lines in file.	    */C 	    for (rcount = 0; fgets(rname, sizeof(rname), sepfile) != NULL;  		rcount++);+ 					    /* Allocate routine name list   */F$ 					    /* and terminate it.	    */3 	    rlist = malloc((rcount + 1) * sizeof(char *));e 	    rlist[rcount] = NULL;   	    /*+								    */F 	    /*	Reset to beginning of file. For each line in file, get the  */F 	    /*	routine name, stripping the ending newline. Allocate memory */F 	    /*	for it, copy the name to the memory, and put the memory ptr */+ 	    /*	in the routine name list.				    */* 	    /*-								    */  " 	    fseek(sepfile, 0, SEEK_SET);  	    while (rcount-- > 0) {s' 		fgets(rname, sizeof(rname), sepfile);n" 		rname[strlen(rname) - 1] = '\0';; 		rlist[rcount] = strcpy(malloc(strlen(rname) + 1), rname);  	    } 	    fclose(sepfile);cD 	    set_separate_list(rlist);	    /* Save list in global db.	    */ 	    return 1; 	    * 	} 	else {*I 	    printf("ERROR: Unable to open separate routine file %s for input\n",* 		aValStr);* 	    return 0; 	}     }*
     else {E 	printf("ERROR: %cseparate option requires routine list file name\n",d 	    CMDLINE_OPTION_SWITCH);
 	return 0;     }  }	  N /*************************************************************************++*/ int cmdopt_language(J /* Command line option handler to set list of source file extension	    */% /* language translations.						    */*       int	    vOptCode,t# 	    /* (READ, BY VAL):						    */;G 	    /* Option keyword translation code, ignored by this routine.    */*       char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */	7 	/*  1	- Successful processing of this option.			    */dE 	/*  0	- Language file name missing, file cannot be opened, or	    */R1 	/*	  invalid language definition found.			    */eG 	/*****************************************************************--*//   {--     static KEYWORD_DEFINITION languages[] = { " 	{"text",	3, NULL, LANGUAGE_TEXT},! 	{"dcl",		3, NULL, LANGUAGE_DCL},*$ 	{"bliss",	3, NULL, LANGUAGE_BLISS}, 	{"c",		1, NULL, LANGUAGE_CC},6 	{NULL,          0, NULL}	    /* End of table.		    */     };?     FILE    *langfile;			    /* Lang translation file ptr.   */cE     char    translation[512];		    /* Translation string buffer.   */E;     char    *fextbegin;			    /* File extension ptr.	    */{=     char    *fextend;			    /* File extension end ptr.	    */c<     SOURCE_LANGUAGE			    /* Language identification.	    */ 	    langcode;+ 					    /* Check for equal sign and	    */A+ 					    /* make sure there is a file.   */oE     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR && *aValStr != '\0') {T0 	if ((langfile = fopen(aValStr, "r")) != NULL) { 	l( 					    /* For each line, locate	    */% 					    /* beginning of text.	    */LH 	    while (fgets(translation, sizeof(translation), langfile) != NULL) {. 		translation[strlen(translation) - 1] = '\0'; 		for (fextbegin = translation;i2 		    *fextbegin != '\0' && isspace(*fextbegin) &&+ 		    *fextbegin != CMDLINE_OPTION_COMMENT;* 		    fextbegin++);*+ 					    /* Ignore comment lines. Find   */C+ 					    /* end of translation field and */r' 					    /* process translation.	    */O- 		if (*fextbegin != CMDLINE_OPTION_COMMENT) {  		    for (fextend = fextbegin;n: 			*fextend > ' ' && *fextend != CMDLINE_OPTION_SEPARATOR; 			fextend++);. 		    if (*fextend != CMDLINE_OPTION_SEPARATOR! 			|| !isalnum(*(fextend + 1))) { 
 			printf(A 		    "ERROR: Invalid language definition in language file %s\n",g 			    aValStr);
 			printf(8 		    "       Language name missing in definition %s\n", 			    fextbegin); 			fclose(langfile); 			return 0; 		    }* 		    else { 			*fextend++ = '\0'; - 			if ((langcode = translate_keyword(fextend,l) 			    languages)) == LANGUAGE_UNKNOWN) {* 			    printf(A 		    "ERROR: Invalid language definition in language file %s\n",\ 			    aValStr);7 			    printf("       Language %s unknown\n", fextend);  			    fclose(langfile); 			    return 0; 			}	 			else {o/ 			    add_lang(new_lang(fextbegin, langcode));  			} 		    }i 		}E 	    } 	    fclose(langfile); 	    return 1; 	} 	else {OA 	    printf("ERROR: Unable to open language file %s for input\n",* 		aValStr);* 	    return 0; 	}     }*
     else {A 	printf("ERROR: %clanguage option requires language file name\n",f 	    CMDLINE_OPTION_SWITCH);
 	return 0;     }	 }   N /*************************************************************************++*/ int cmdopt_urlprefix(u? /* Command line option handler to set HTML URL prefix.			    */i       int	    vOptCode,/# 	    /* (READ, BY VAL):						    */*G 	    /* Option keyword translation code, ignored by this routine.    */1       char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */m7 	/*  1	- Successful processing of this option.			    */l# 	/*  0	- Prefix missing.					    */;G 	/*****************************************************************--*/*   {*E     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR && *aValStr != '\0') {  	set_url_prefix(aValStr);l
 	return 1;     }l
     else {= 	printf("ERROR: %curlprefix option requires prefix string\n",	 	    CMDLINE_OPTION_SWITCH);
 	return 0;     }c },  N /*************************************************************************++*/ int cmdopt_callers( N /* Command line option handler to set maximum number of callers to allow    */7 /* inline expansion of routine call subtrees.				    */s       int	    vOptCode,0# 	    /* (READ, BY VAL):						    */*G 	    /* Option keyword translation code, ignored by this routine.    */        char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */n7 	/*  1	- Successful processing of this option.			    */;- 	/*  0	- Option value out of range.				    */*G 	/*****************************************************************--*/s   {i6     int	    maxcallers;			    /* Option value.		    */  1     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR) {I 	maxcallers = atoi(aValStr);F 	if (maxcallers >= MIN_MAX_CALLERS && maxcallers <= MAX_MAX_CALLERS) {! 	    set_max_callers(maxcallers);  	    return 1; 	}     }	H     printf("ERROR: %ccallers option requires value between %d and %d\n",: 	CMDLINE_OPTION_SWITCH, MIN_MAX_CALLERS, MAX_MAX_CALLERS);
     return 0;o }h  N /*************************************************************************++*/ int cmdopt_depth(*N /* Command line option handler to set maximum call tree expansion depth.    */       int	    vOptCode,p# 	    /* (READ, BY VAL):						    */ G 	    /* Option keyword translation code, ignored by this routine.    */,       char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */p7 	/*  1	- Successful processing of this option.			    */ - 	/*  0	- Option value out of range.				    */lG 	/*****************************************************************--*/s   {r4     int	    maxdepth;			    /* Option value.		    */  1     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR) {  	maxdepth = atoi(aValStr);1 	if (maxdepth > 0 && maxdepth < MAX_TREE_DEPTH) { " 	    set_max_tree_depth(maxdepth); 	    return 1; 	}     }vF     printf("ERROR: %cdepth option requires value between %d and %d\n",/ 	CMDLINE_OPTION_SWITCH, 1, MAX_TREE_DEPTH - 1);g
     return 0;. }   N /*************************************************************************++*/ int cmdopt_report(I /* Command line option handler to set specified report generation.	    */	       int	    vOptCode, # 	    /* (READ, BY VAL):						    */	G 	    /* Option keyword translation code, ignored by this routine.    */2       char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */ 7 	/*  1	- Successful processing of this option.			    */o+ 	/*  0	- Report keywords missing.				    */fG 	/*****************************************************************--*/+   {	( 					    /* Report option keyword	    */# 					    /* dispatch table.		    *//,     static KEYWORD_DEFINITION keywords[] = {6 	{"defined",	3, cmdopt_clear,    RPT_DEFINED_DISABLE},: 	{"undefined",	3, cmdopt_clear,    RPT_UNDEFINED_DISABLE},7 	{"calls",	3, cmdopt_clear,    RPT_CALLS_DISABLE},	    n7 	{"trees",	3, cmdopt_clear,    RPT_TREES_DISABLE},	    e7 	{"xrefs",	3, cmdopt_clear,    RPT_XREFS_DISABLE},	    n7 	{"files",	3, cmdopt_clear,    RPT_FILES_DISABLE},	    /9 	{"byfile",	3, cmdopt_clear,    RPT_BYFILE_DISABLE},	    -9 	{"source",	3, cmdopt_clear,    RPT_SOURCE_DISABLE},	    [. 	{NULL,		0, NULL}	    /* End of table.		    */     };  E     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR && *aValStr != '\0') {t     + 					    /* Disable all reports, then    */b+ 					    /* enable only those specified  */*; 	set_option( RPT_DEFINED_DISABLE |   /* by keyword.		    */  		     RPT_UNDEFINED_DISABLE | 		     RPT_CALLS_DISABLE | 		     RPT_TREES_DISABLE | 		     RPT_XREFS_DISABLE | 		     RPT_BYFILE_DISABLE |  		     RPT_FILES_DISABLE | 		     RPT_SOURCE_DISABLE);*+ 	if (!process_keyword(aValStr, keywords)) {*7 	    printf("ERROR: Unable to process %creport option",a 		CMDLINE_OPTION_SWITCH);e 	    return 0; 	} 	else {o 	    return 1; 	}     }l
     else {H 	printf("ERROR: %creport option requires report type keyword or list\n", 	    CMDLINE_OPTION_SWITCH);
 	return 0;     }b }h  N /*************************************************************************++*/ int cmdopt_noreport(N /* Command line option handler to suppress specified report generation.	    */       int	    vOptCode, # 	    /* (READ, BY VAL):						    */ G 	    /* Option keyword translation code, ignored by this routine.    *//       char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */[7 	/*  1	- Successful processing of this option.			    */N+ 	/*  0	- Report keywords missing.				    */GG 	/*****************************************************************--*/    {E+ 					    /* No-report option keyword	    */n# 					    /* dispatch table.		    */ ,     static KEYWORD_DEFINITION keywords[] = {4 	{"defined",	3, cmdopt_set,    RPT_DEFINED_DISABLE},8 	{"undefined",	3, cmdopt_set,    RPT_UNDEFINED_DISABLE},5 	{"calls",	3, cmdopt_set,    RPT_CALLS_DISABLE},	     5 	{"trees",	3, cmdopt_set,    RPT_TREES_DISABLE},	    /5 	{"xrefs",	3, cmdopt_set,    RPT_XREFS_DISABLE},	     5 	{"files",	3, cmdopt_set,    RPT_FILES_DISABLE},	     7 	{"byfile",	3, cmdopt_set,    RPT_BYFILE_DISABLE},	    !7 	{"source",	3, cmdopt_set,    RPT_SOURCE_DISABLE},	    ). 	{NULL,		0, NULL}	    /* End of table.		    */     };  E     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR && *aValStr != '\0') {r+ 	if (!process_keyword(aValStr, keywords)) {o9 	    printf("ERROR: Unable to process %cnoreport option",s 		CMDLINE_OPTION_SWITCH);! 	    return 0; 	} 	else {  	    return 1; 	}     }L
     else { 	printf(G 	    "ERROR: %cnoreport option requires report type keyword or list\n",  	    CMDLINE_OPTION_SWITCH);
 	return 0;     }/ }r  N /*************************************************************************++*/ int cmdopt_fmt_kwhandler( A /* Common keyword handler for format command line option.		    */e       report_output_format     	    vFormat# 	    /* (READ, BY VAL):						    */1  	    /* Format code.						    */  % )	/* Returns status flag:						    */i7 	/*  1	- Successful processing of this option.			    */a6 	/*  0	- Conflicting format option specified.			    */G 	/*****************************************************************--*/t   { ?     if (rpt_text_enabled() || report_format() != FORMAT_TEXT) {l6 	puts("ERROR: Conflicting format options specififed");
 	return 0;     }a
     else { 	switch (vFormat) {e 	case FORMAT_TEXT:@ 	    set_option(RPT_TEXT_ENABLE);	/* Text has special flag.   */ 	case FORMAT_SDML: 	case FORMAT_HTML:  	    set_report_format(vFormat); 	    break;f 	case FORMAT_RTF:) 	case FORMAT_WINHELP:E 	case FORMAT_VMSHELP:aM puts("Sorry, the RTF, WINHELP, and VMSHELP formats are not yet implemented");u	 return 0;  	    break;* 	}         return 1;t     }  }   N /*************************************************************************++*/ int cmdopt_format(C /* Command line option handler to set report output format.		    */*       int	    vOptCode,*# 	    /* (READ, BY VAL):						    */uG 	    /* Option keyword translation code, ignored by this routine.    */*       char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */ 7 	/*  1	- Successful processing of this option.			    */r* 	/*  0	- Format keyword missing.				    */G 	/*****************************************************************--*/    {o( 					    /* Format option keyword	    */# 					    /* dispatch table.		    */*,     static KEYWORD_DEFINITION keywords[] = {: 	{"text",        3, cmdopt_fmt_kwhandler,    FORMAT_TEXT},: 	{"sdml",        3, cmdopt_fmt_kwhandler,    FORMAT_SDML},: 	{"html",        3, cmdopt_fmt_kwhandler,    FORMAT_HTML},2 	{"rtf",		3, cmdopt_fmt_kwhandler,    FORMAT_RTF},9 	{"winhelp",	3, cmdopt_fmt_kwhandler,    FORMAT_WINHELP},*9 	{"vmshelp",	3, cmdopt_fmt_kwhandler,    FORMAT_VMSHELP},t. 	{NULL,		0, NULL}	    /* End of table.		    */     };  E     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR && *aValStr != '\0') {u+ 	if (!process_keyword(aValStr, keywords)) { 7 	    printf("ERROR: Unable to process %cformat option",o 		CMDLINE_OPTION_SWITCH);d 	    return 0; 	} 	else {  	    return 1; 	}     } 
     else {B 	printf("ERROR: %cformat option requires report format keyword\n", 	    CMDLINE_OPTION_SWITCH);
 	return 0;     }  }*  N /*************************************************************************++*/ int cmdopt_htmlbyfile(N /* Command line option handler to set maximum number of entries of HTML	    *// /* output to allow per by-file file.					    */.       int	    vOptCode,S# 	    /* (READ, BY VAL):						    */ G 	    /* Option keyword translation code, ignored by this routine.    */l       char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */D7 	/*  1	- Successful processing of this option.			    */ " 	/*  0	- Value missing.					    */G 	/*****************************************************************--*/t   {h1     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR) {a$ 	set_max_html_byfile(atoi(aValStr));
 	return 1;     } 9     printf("ERROR: %chtmlbyfile option requires value\n",d 	CMDLINE_OPTION_SWITCH);
     return 0;e }   N /*************************************************************************++*/ int cmdopt_htmlxref(N /* Command line option handler to set maximum number of entries of HTML	    */1 /* output to allow per call xref file.					    *//       int	    vOptCode,f# 	    /* (READ, BY VAL):						    */*G 	    /* Option keyword translation code, ignored by this routine.    */        char    *aValStr$ 	    /* (READ, BY ADDR):						    */< 	    /* Option value string, preceded by equal sign.		    */     % )	/* Returns status code:						    */H7 	/*  1	- Successful processing of this option.			    */ " 	/*  0	- Value missing.					    */G 	/*****************************************************************--*/,   {T1     if (*aValStr++ == CMDLINE_OPTION_SEPARATOR) {*" 	set_max_html_xref(atoi(aValStr));
 	return 1;     }*7     printf("ERROR: %chtmlxref option requires value\n",n 	CMDLINE_OPTION_SWITCH);
     return 0;  }*  N /*************************************************************************++*/ int cmdopt_author(N /* Command line option handler to show program author for posterity (gee,   */" /* I'm embarrassed!).							    */  !     /* No arguments.							    */.  = )	/* Returns 0 to indicate that this option is totally bogus!SG 	/*****************************************************************--*/m   {gL     printf("%s %s\n%s\n", PROGRAM_IDENT, PROGRAM_COPYRIGHT, PROGRAM_AUTHOR);
     return 0;  }	  N /*************************************************************************++*/ void show_help(N@ /* Prints help information to stdout and EXITS PROGRAM.			    */  !     /* No arguments.							    */c  & )	/* No return value.      					    */G 	/*****************************************************************--*/e   { :     printf("%s %s\n\n", PROGRAM_IDENT, PROGRAM_COPYRIGHT);     puts(PROGRAM_PARAMS);{0     printf(PROGRAM_HELP, CMDLINE_OPTION_SWITCH);	     puts(fQ "\nWhere: product_file is the product definition file listing the source files"); 	     puts(E "          to be analyzed.");U	     puts(E? "       options are the following (they may be abbreviated):");I4 #if 0 /* This help is too long and is incomplete! */     printf( M "          %chelp - Show this full help and exit.\n", CMDLINE_OPTION_SWITCH);(     printf(D? "          %clist - Create listing file <output_prefix>%s%s\n",|? 	CMDLINE_OPTION_SWITCH, OUTFILE_SUFFIX_LIST, OUTFILE_EXT_LIST);R     printf(BI "          %csilent - Disable stdout logging.\n", CMDLINE_OPTION_SWITCH);S     printf()F "          %cbrief - Brief stdout logging.\n", CMDLINE_OPTION_SWITCH);     printf(sT "          %coutprefix=output_prefix - Output file pathname and filename prefix;\n", 	CMDLINE_OPTION_SWITCH);	     puts(iO "             suffixes and extensions will be added to this prefix to create");E	     puts(T( "             full output file names.");     printf(*N "          %cdefinition - Log routine definitions.\n", CMDLINE_OPTION_SWITCH);     printf( L "          %creference - Log routine references.\n", CMDLINE_OPTION_SWITCH);     printf(  "          %cseparate=sep_file - Name of file containing routine names for which\n             separate trees must always be generated if they call anything.\n",	 	CMDLINE_OPTION_SWITCH);     printf(e "          %clanguage - Name of file containing language definitions of the\n            form <ext>=<language>, where <ext> is a file type extension, and\n            <language> is one of the keywords \"c\", \"bliss\", or \"text\".\n",* 	CMDLINE_OPTION_SWITCH);     printf(pl "          %cnoinline - Generate separate call trees for all routines that call\n              anything.\n", 	CMDLINE_OPTION_SWITCH);     printf(FE "          %ctext - Format reports as plain text files (default).\n",, 	CMDLINE_OPTION_SWITCH);     printf(Sk "          %csdml - Format reports as Standard Digital Markup Language for\n              VAX Document.\n",E 	CMDLINE_OPTION_SWITCH);     printf(tq "          %chtml - Format reports as Hyper Text Markup Language for World-\n              Wide Web browsers.\n",_ 	CMDLINE_OPTION_SWITCH);     printf(Uo "          %ccallers=N - Set max callers for inline subtrees to N\n              (range %d-%d, default %d).\n",!9 	CMDLINE_OPTION_SWITCH, MIN_MAX_CALLERS, MAX_MAX_CALLERS,  	DEF_MAX_CALLERS); #else ?     printf("          %chelp                       %cauthor\n",t/ 	CMDLINE_OPTION_SWITCH, CMDLINE_OPTION_SWITCH); I     printf("          %coptions=opt_file           %ctrace=debug_list\n",e/ 	CMDLINE_OPTION_SWITCH, CMDLINE_OPTION_SWITCH); =     printf("          %clog=log_file               %clist\n",*/ 	CMDLINE_OPTION_SWITCH, CMDLINE_OPTION_SWITCH);o>     printf("          %csilent                     %cbrief\n",/ 	CMDLINE_OPTION_SWITCH, CMDLINE_OPTION_SWITCH);tM     printf("          %coutprefix=outfile_prefix   %curlprefix=url_prefix\n",	/ 	CMDLINE_OPTION_SWITCH, CMDLINE_OPTION_SWITCH);*J     printf("          %cformat=report_format       %cseparate=sep_file\n",/ 	CMDLINE_OPTION_SWITCH, CMDLINE_OPTION_SWITCH);*N     printf("          %clanguage=lang_file         %cdescription=desc_file\n",/ 	CMDLINE_OPTION_SWITCH, CMDLINE_OPTION_SWITCH);EB     printf("          %cdefinition                 %creference\n",/ 	CMDLINE_OPTION_SWITCH, CMDLINE_OPTION_SWITCH); @     printf("          %ccallers=n                  %cdepth=n\n",/ 	CMDLINE_OPTION_SWITCH, CMDLINE_OPTION_SWITCH);MM     printf("          %creport=report_list         %cnoreport=report_list\n",c/ 	CMDLINE_OPTION_SWITCH, CMDLINE_OPTION_SWITCH);sC     printf("          %chtmlbyfile=n               %chtmlxref=n\n",d/ 	CMDLINE_OPTION_SWITCH, CMDLINE_OPTION_SWITCH);n$     printf("          %cnoinline\n", 	CMDLINE_OPTION_SWITCH); #endif 	*A     if (list_enabled()) {		    /* Close the listing file if    */h6 	fclose(list_file());		    /* one was created.		    */     }t     /     exit(0);				    /* End the program!		    */p }n  N /*************************************************************************++*/ main(/$ /* Program main routine.						    */       int	    vArgc,# 	    /* (READ, BY VAL):						    */ < 	    /* Number of program argument strings in aArgv.		    */       char    *aArgv[]$ 	    /* (READ, BY ADDR):						    */2 	    /* List of program argument strings.			    */  , )	/* Returns system success code.					    */G 	/*****************************************************************--*/    { + 					    /* Main program command line    */ + 					    /* argument options dispatch    */F 					    /* table.			    */ +     static KEYWORD_DEFINITION options[] = {{* 	{"options",     3, process_options_file}, 	{"help",	3, show_help}, 	{"trace",	3, cmdopt_trace},% 	{"log",         3, cmdopt_log},	    h% 	{"list",        3, cmdopt_list},    p5 	{"silent",      3, cmdopt_set,	LOG_SILENT_ENABLE},  L5 	{"brief",       3, cmdopt_set,	LOG_BRIEF_ENABLE},    & 	{"outprefix",   3, cmdopt_outprefix}, 	{"format",	3, cmdopt_format},( 	{"description", 3, cmdopt_description},0 	{"definition",  3, cmdopt_set,	LOG_DEF_ENABLE},0 	{"reference",   3, cmdopt_set,	LOG_REF_ENABLE},% 	{"separate",    3, cmdopt_separate}, " 	{"language",	3, cmdopt_language},5 	{"noinline",    3, cmdopt_set,	TREE_INLINE_DISABLE},D& 	{"urlprefix",   3, cmdopt_urlprefix},  	{"callers",	3, cmdopt_callers}, 	{"depth",	3, cmdopt_depth}, 	{"report",	3, cmdopt_report},% 	{"noreport",    3, cmdopt_noreport},h& 	{"htmlbyfile",	5, cmdopt_htmlbyfile}," 	{"htmlxref",	5, cmdopt_htmlxref}, 	{"author",	3, cmdopt_author},6 	{NULL,          0, NULL}	    /* End of table.		    */     };       /*+									    */M     /*	Make sure enough required arguments were specified, then process    */RI     /*	the optional arguments and analyze the product files. If no	    */sM     /*	arguments, show brief help. If first argument is an option, show    */ M     /*	full help regardless of which options were specified. Note that	    */*J     /*	showing full help terminates the program without any further	    */M     /*	processing of the command line, even if all the arguments are ok.   */l     /*-									    */          if (vArgc < 2) { 	puts(PROGRAM_PARAMS);4         printf(PROGRAM_HELP, CMDLINE_OPTION_SWITCH);     }*2     else if (*aArgv[1] == CMDLINE_OPTION_SWITCH) { 	    show_help();*     }n
     else {+ 					    /* Disable these reports by	    */  					    /* default.			    */M3 	set_option(RPT_CALLS_DISABLE | RPT_TREES_DISABLE);  	*" 	set_max_callers(DEF_MAX_CALLERS);* 	set_max_html_byfile(DEF_MAX_HTML_BYFILE);& 	set_max_html_xref(DEF_MAX_HTML_XREF);( 	set_max_tree_depth(MAX_TREE_DEPTH + 1);1 	if (process_options(vArgc, aArgv, 2, options)) {p. 	    add_lang(new_lang("DAT", LANGUAGE_TEXT));. 	    add_lang(new_lang("TXT", LANGUAGE_TEXT));- 	    add_lang(new_lang("COM", LANGUAGE_DCL));., 	    add_lang(new_lang("C",   LANGUAGE_CC));, 	    add_lang(new_lang("H",   LANGUAGE_CC));/ 	    add_lang(new_lang("BLI", LANGUAGE_BLISS));S/ 	    add_lang(new_lang("REQ", LANGUAGE_BLISS));m/ 	    add_lang(new_lang("R32", LANGUAGE_BLISS));r 	    analyze_product(aArgv[1]);q 	    if (list_enabled()) { 		fclose(list_file()); 	    } 	}     }* }*  