/*
 * DECnet object for servicing SSL.  We accept the inbound and speak a
 * simple protocol for multiplexing the 2 data streams (application and
 * ssl data) over this logical link.
 *
 * Each NSP message sent over the DECnet link has the following structure:
 *    struct rpc_msg { 
 *      char channel;
 *      char function;
 *      short length;
 *      char data[MAX_DATA];
 *    } msg;
 *
 * The channel field designates the virtual data stream this message applies
 * to and is one of:
 *   A - Application data (payload).
 *   R - Remote client connection that initiated the SSL connection.  Encrypted
 *       data is sent over this connection.
 *   G - General data, reserved for future use.
 *
 * The data streams are half-duplex read/write and have following functions:
 *   G - Get, requests that up to msg.length bytes of data be returned.  The
 *       data is returned in the next 'C' function response that matches the
 *       requesting channel.
 *   P - Put, requests that the first msg.length bytes of msg.data be appended
 *       to the designated stream.
 *   C - Confirms a get or put.  Every get and put will get a confirm response,
 *       you cannot initiate another function on a channel until the previous
 *       operation has been confirmed.
 *
 *  The 2 channels may interleave their operations, for example:
 *        Server msg           Client msg
 *         A, Get, 4092          ---->
 *                               <----  R, get, 4092
 *         R, Confirm, {hello}   ---->
 *                               <----  R, put, {srv hello}
 *         R, Confirm, 0         ---->
 *                               .		(SSL handshake completed)
 *                               .              (read first app data).
 *                               <----  A, confirm, {http data}
 *         A, Put, {http data}   ---->
 *                               <----  A, confirm, 0
 *
 *  The length field is not permitted to be larger that 4092 bytes.
 *
 * Author: Dave Jones
 * Date:   22-JUL-1996
 */
#include <stdlib.h>
#include <stdio.h>
#include <iodef.h>		/* VMS IO$_ definitions */
#include <descrip.h>		/* VMS string descriptors */
extern int SYS$QIOW(), SYS$ASSIGN();

struct rpc_msg {		/* Should have member alignment inhibited */
   char channel;		/* 'A'-app data. 'R'-remote client 'G'-global */
   char function;		/* 'G'-get, 'P'-put, 'C'-confirm, 'X'-close */
   unsigned short int length;	/* Amount of data returned or max to return */
   char data[4092];		/* variable data */
};
#define RPC_HDR_SIZE (sizeof(struct rpc_msg) - 4092)

static $DESCRIPTOR(sysnet, "SYS$NET");
typedef unsigned short io_channel;

struct io_status {
    unsigned short status;
    unsigned short count;
    unsigned long stsval;
};
/*****************************************************************************/
/* Decnet I/O routines.
 */
static int get ( io_channel chan, char *buffer, int maxlen, int *length )
{
    int status;
    struct io_status iosb;
    status = SYS$QIOW ( 0, chan, IO$_READVBLK, &iosb, 0, 0,
	buffer, maxlen, 0, 0, 0, 0 );
    if ( (status&1) == 1 ) status = iosb.status;
    if ( (status&1) == 1 ) *length = iosb.count;
    return status;
}

static int put ( io_channel chan, char *buffer, int length )
{
    int status;
    struct io_status iosb;
    status = SYS$QIOW ( 0, chan, IO$_WRITEVBLK, &iosb, 0, 0,
	buffer, length, 0, 0, 0, 0 );
    if ( (status&1) == 1 ) status = iosb.status;
    return status;
}
/***************************************************************************/
/* Handle operations on the 'A' channel.
 */
static int app_request ( io_channel chan, struct rpc_msg *msg, int length )
{
    int status, segment;
    status = 1;
    switch ( msg->function ) {
	case 'G':
	    /*
	     * Request remote data (channel R).
	     */
	    msg->channel = 'R';
	    status = put ( chan, (char *) msg, length );
	    if ( (status&1) == 0 ) break;
	    /*
	     * Read response, verify answer is remote confirm.
	     */
	    status = get ( chan, (char *) msg, sizeof(*msg), &segment );
	    if ( (status&1) == 0 ) break;
	    if ( (length < RPC_HDR_SIZE) || (msg->channel != 'R') ||
		(msg->function != 'C') ) {
		printf("Unexpected response to get: '%c' '%c' (%d)\n",
			msg->channel, msg->function, length );
		status = 20;
		break;
	    }
	    /*
	     * Send confirm (and data) back to channel A.
	     */
	    msg->channel = 'A';
	    msg->function = 'C';
	    status = put ( chan, (char *) msg, segment );
	    if ( (status&1) == 0 ) break;
	    break;

	case 'P':
	    /*
	     * Put data to remote.
	     */
	    msg->channel = 'R';
	    status = put ( chan, (char *) msg, length );
	    if ( (status&1) == 0 ) break;
	    /*
	     * Read confirm message.
	     */
	    status = get ( chan, (char *) msg, sizeof(*msg), &length );
	    if ( (status&1) == 0 ) break;
	    if ( (length < RPC_HDR_SIZE) || (msg->channel != 'R') ||
		(msg->function != 'C') ) {
		printf("Unexpected response to put: '%c' '%c' (%d)\n",
			msg->channel, msg->function, length );
		status = 20;
		break;
	    }
	    /*
	     * Confirm the send.
	     */
	    msg->channel = 'A';
	    msg->function = 'C';
	    msg->length = 0;
	    status = put ( chan, (char *) msg, RPC_HDR_SIZE );
	    break;
	case 'X':
	    /*
	     * Close down connection, error status will force exit.
	     */
	    status = 2160;
	    break;
	default:
		printf("Unexpected app request: '%c'\n", msg->function );
	    status = 20;
	    break;
    }
    return status;
}
/***************************************************************************/
/* Handle operations on the 'G' channel.
 */
static int general_request ( io_channel chan, struct rpc_msg *msg, int length )
{
    return 48;
}
/***************************************************************************/
int main ( int argc, char **argv )
{
    int status, length;
    io_channel chan;
    struct rpc_msg msg;
    /*
     * Confirm logical link with initiating client.
     */
    status = SYS$ASSIGN ( &sysnet, &chan, 0, 0, 0 );
    printf("status of assign to SYS$NET: %d\n", status );
    /*
     * Take commands from client until bad status.
     */
    while ( (status&1) == 1 ) {
	/*
	 * Read message.
	 */
	status = get ( chan, (char *) &msg, sizeof(msg), &length );
	if ( (status&1) == 0 ) {
	    printf("Error in main loop get: %d\n", status );
	    break;
	}
	if ( length < RPC_HDR_SIZE ) {
	    printf("Error in main loop get size: %d\n", length );
	    break;
	}
	/*
	 * Take action depending upon which data stream selected.
	 */
printf("client command: chan='%c' func='%c', len=%d\n", msg.channel,
msg.function,length );
	switch ( msg.channel ) {
	    case 'A':
		status = app_request ( chan, &msg, length );
		break;
	    case 'R':
		/*
		 * The SSL server, can't respond to requests for the
		 * the remote channel (we make requests for the R channel).
		 */
		printf("Invalid channel: '%c'\n", msg.channel );
		status = 20;
		break;
	    case 'G':
		status = general_request ( chan, &msg, length );
		break;
	    default:
		printf("Unknown channel code: '%c'\n", msg.channel );
		status = 20;
		break;
	}
    }
    return 1;
}
