/*------------------------------------------------------------
| psdat.c
|
| part of ps2text
|
| this module reads the ps2text.dat file.  The dat file is
| self-documenting.  Please see introductory comments there.
|
| Note that there are no global variables.
| Please keep it that way.
|
| Notice also that this code is completely portable to unix cc.
| Please keep it that way.
|
| Program written by David Parmenter, CUP ITG/E
|
| Modified:
|   Mark Goodrich, SDT Base Technologies & Environments - 11/24/93
+------------------------------------------------------------*/

#include <stdio.h>
#include <ctype.h>
#ifdef vms
#include <stdlib.h>
#endif
#include "ps2text.h"


/*------------------------------------------------------------
| because Unix doesn't seem to have strstr(), we have our own.
|
| it returns a pointer to the beginning of the matched segment in the
| searched string
+------------------------------------------------------------*/

char *
ps_strstr(s1,s2)
    char		*s1;		/* string to be searched */
    char		*s2;		/* string to look for */
{
    while (*s1)				/* loop to end of s1 */
	{
	if (*s1 == *s2)			/* optimize by testing first chars */
	    if (Strnequal(s1, s2))	/* now compare entire string */
		return(s1);		/* if it's a match, return address */
	s1++;				/* move forward */
	}

    return(NULL);			/* couldn't find it */
}


/*------------------------------------------------------------
| open up the data file
|
| exit() if there's an error otherwise
|
| we do nothing.
+------------------------------------------------------------*/
void
open_dat_file(dat_fp)
    FILE		**dat_fp;	/* dat file */
{
    char		ps_dat_name[MAX_BUF];

    /* platform specific code */

#ifdef VMS
    strcpy(ps_dat_name,"$PS2TEXT:");		/* PS2TEXT dir in VMS */
#else
    strcpy(ps_dat_name,getenv("HOME"));		/* top level shelff in Unix */
    if (ps_dat_name[strlen(ps_dat_name)-1] != '/')
	strcat(ps_dat_name, "/");
#endif

    strcat(ps_dat_name,"ps2text.dat");		/* same file name */

    if ((*dat_fp = fopen(ps_dat_name, "r" )) == NULL)
	Exit_error("I couldn't find $ps2text:ps2text.dat");
}


/*------------------------------------------------------------
| scan the dat file for the help endhelp flags.  Issue all text
| between the two flags.
+------------------------------------------------------------*/
int
issue_help()
{
    FILE		*dat_fp;		/* ps2text data file */
    char		string[MAX_BUF];	/* utility buffer */
    int			print_lines = FALSE;	/* turn on once we find help */


    open_dat_file(&dat_fp);			/* open up data file */

    for (;;)
	{
	fgets(string, MAX_BUF, dat_fp);		/* get one line */

	if (feof(dat_fp))			/* can't find help or endhelp */
	    Exit_error("I couldn't find help ... endhelp in ps2text.dat")

	else if (Strnequal(string, HELP))	/* begin printing help */
	    print_lines = TRUE;

	else if (Strnequal(string, ENDHELP))	/* stop printing help */
	    break;

	else if (print_lines)			/* print a help line */
		fprintf(stdout, "%s", string);
	}

    fclose(dat_fp);
    return(TRUE);
}


/*------------------------------------------------------------
| this macro is a jacket for get_entry()
| - gets the entry, and tests for success
| - allocates storage for the string pointer
| - copies in the string
|
| - note that get_entry() returns TRUE or FALSE
+------------------------------------------------------------*/

#define Get_string_entry(_entry, _storage)			\
	{							\
	_storage = NULL;					\
	if (! get_entry(dat_fp, _entry, &string[0]))		\
		return(FALSE);					\
	_storage = (char *) malloc(strlen(string)+1);		\
	if (_storage == NULL)					\
		Exit_error("I couldn't allocate enough memory");\
	strcpy(_storage, string);				\
	}

/*------------------------------------------------------------
| this macro is a jacket for get_entry().
| - gets the entry, and tests for success.
| - converts and then copies the int
|
| - note that get_entry() returns TRUE or FALSE
+------------------------------------------------------------*/

#define Get_int_entry(_entry, _storage)				\
	{							\
	_storage = 0;						\
	if (! get_entry(dat_fp, _entry, &string[0]))		\
		return(FALSE);					\
	_storage = (int) atoi(string);				\
	}

/*------------------------------------------------------------
| get_entry()
|
| - gets a line
| - looks for the entry name, and the value in the following form:
|	'entry_name	[entry_value]'
| - returns the value in e_v.
| - returns TRUE if we got a match, FALSE if we didn't.
+------------------------------------------------------------*/

int
get_entry(dat_fp, entry_name, entry_value)
    FILE		*dat_fp;		/* pointer to dat file */
    char		*entry_name;		/* entry name */
    char		entry_value[];		/* return string */
{
    char		string[MAX_BUF];	/* utility buffer */
    char		*c, *c2;		/* ch ptrs, for parsing */

    fgets(string, MAX_BUF, dat_fp);		/* get one line */

    if (string[0] == '!')			/* this is a comment line */
	return(FALSE);				/* we ignore it */

    if (! Strnequal(string, entry_name))	/* name doesn't match */
	return(FALSE);

    c = (char *) strchr(string,'[');		/* find the '[' */
    c2 = (char *) strchr(string,']');		/* find the ']' */

    if ((c == NULL) || (c2 == NULL))		/* look for errors */
	return(FALSE);

    *c2 = '\0';					/* null it */
    strcpy(entry_value, ++c);			/* skip '[', then set e_v */

    return(TRUE);				/* we found it! */
}


/*------------------------------------------------------------
| get_special_chars()
|
| reads dat file for any special characters, and loads them in
+------------------------------------------------------------*/

int
get_special_chars(dat_fp, creator_data)
    FILE		*dat_fp;		/* pointer to dat file */
    CREATOR		*creator_data;		/* pointer to creator_data block */
{
    char		string[MAX_BUF];	/* utility buffer */
    char		string2[MAX_BUF];	/* utility buffer */
    int			i = 0;			/* utility int */

    for (i = 0; i<255; i++)			/* initialize these */
	creator_data->special_chars[i] = NULL;

    fgets(string, MAX_BUF, dat_fp);		/* get next line */

    if (! Strnequal(string, SPECIALCHARS))
	return(FALSE);				/* data file is out of whack */

    for (;;)					/* loop until endspecialchars */
	{
	fgets(string, MAX_BUF, dat_fp);		/* get a line */

	if (Strnequal(string, ENDSPECIALCHARS))
	    return(TRUE);			/* succesful termination */

	if (string[0] != '!')			/* check for comment lines */
	    {
	    if (sscanf( string, " %d %s", &i, &string2[0]) != 2)
		return(FALSE);			/* something's wrong */

	    creator_data->special_chars[i] =	/* allocate for this string */
		(char *) malloc (strlen(string2) + 1);

	    strcpy(creator_data->special_chars[i], string2);	/* copy it */
	    }
	}
}


/*------------------------------------------------------------
| find_this_ps_creator()
|
| scans the dat file looking for the data pertaining to the
| current creator.
+------------------------------------------------------------*/
int
find_this_ps_creator(dat_fp, ps_creator)
    FILE		*dat_fp;		/* dat file */
    char		*ps_creator;		/* name of desired creator */
{
    char		value[MAX_BUF];		/* receives value from g_e() */

    for (;;)					/* start scanning */
	{
	if (get_entry(dat_fp, "creator", &value[0]))	/* check entry */
	    if (ps_strstr(ps_creator, value) != NULL)	/* if it matches */
		return(TRUE);				/* we're done */

	if (feof(dat_fp))			/* if we reached eof, we */
	    return(FALSE);			/* don't know this creator */
	}
}


/*------------------------------------------------------------
| This is the entry point whenever a new creator is needed
+------------------------------------------------------------*/

int get_creator_data(ps_fp, creator_data, creator_name)
    FILE		*ps_fp;			/* PS file */
    CREATOR		**creator_data;		/* pointer to block pointer */
    char		creator_name[];		/* name of creator, from PS file */
{
    CREATOR		*local_data;		/* local copy of creator_data */
    FILE		*dat_fp;		/* ps2text data file */
    char		string[MAX_BUF];	/* utility buffer */


    open_dat_file(&dat_fp);			/* open up data file */

    /* allocate a creator block */

    if ((local_data = (CREATOR *) malloc(sizeof(CREATOR))) == NULL)
	Exit_error("I couldn't allocate enough memory");

    /* find data for this creator */

    if (! find_this_ps_creator(dat_fp, creator_name))
	{
	fprintf(stderr, "\nerror: I don't recognize this kind of Postscript:\n\t%s\n", creator_name );
	fprintf(stderr, "warning: I'm attempting to continue\n\n");
	fclose(dat_fp);
	return(FALSE);
	}

    /* get the rest of the entry */

    Get_string_entry("begindocument", local_data->begindocument);
    Get_string_entry("enddocument", local_data->enddocument);
    Get_string_entry("beginpage", local_data->beginpage);
    Get_string_entry("endpage", local_data->endpage);
    Get_string_entry("delta_x", local_data->delta_x);
    Get_string_entry("delta_y", local_data->delta_y);
    Get_string_entry("delta_xy", local_data->delta_xy);
    Get_string_entry("absolute_x", local_data->absolute_x);
    Get_string_entry("absolute_y", local_data->absolute_y);
    Get_string_entry("absolute_xy", local_data->absolute_xy);
    Get_string_entry("text", local_data->text);
    Get_string_entry("rule", local_data->rule);

    /* kern and paragraph_test are ints */

    Get_int_entry("kern", local_data->kern);
    Get_int_entry("paragraph_test", local_data->paragraph_test);

    /* get special chars */

    if (! get_special_chars(dat_fp, local_data))
	{
	fprintf(stderr, "\nerror: occured while reading specialchars section of ps2text.dat\n");
	fclose(dat_fp);
	return(FALSE);
	}

    /* pass data block back out */

    local_data->flags = 0;	/* process text immediately */
    if (Strnequal(creator_name, "Windows"))
	local_data->flags |= POSTPROCESS;  /* process text as OPERATOR */

    *creator_data = local_data;

    /* close dat file and exit with success */

    fclose(dat_fp);
    return(TRUE);
}


/*------------------------------------------------------------
| free up the entire creator_data structure.
+------------------------------------------------------------*/
int
free_creator_block(creator_data)
    CREATOR		*creator_data;
{
    int			i = 0;			/* utility int */

    if (creator_data)
	{
	free(creator_data->begindocument);
	free(creator_data->enddocument);
	free(creator_data->beginpage);
	free(creator_data->endpage);
	free(creator_data->delta_x);
	free(creator_data->delta_y);
	free(creator_data->delta_xy);
	free(creator_data->absolute_x);
	free(creator_data->absolute_y);
	free(creator_data->absolute_xy);
	free(creator_data->text);

	for (i = 0; i<255; i++)		/* check these */
	    {
	    if (creator_data->special_chars[i] != NULL)
		free(creator_data->special_chars[i]);
	    }

	free(creator_data);
	}

    return(TRUE);
}

