/***************************************************************************
 *
 * FILE NAME		: xce
 *
 * AUTHOR		: Andrew Peebles
 *
 * DESCRIPTION		: color edit program, an experiment in colormaps
 *
 * VERSIONS		: %W%
 *
 ***************************************************************************/

/*
 * standard includes
 */
#include <stdio.h>
#include <ctype.h>

#include "patchlevel.h"

#include <X11/Xatom.h>
#include <X11/cursorfont.h>

#ifndef VMS
#include <Xm/mwm.h>
#endif /* VMs */

#include <Xm/Form.h>
#include <Xm/MessageB.h>
#include <Xm/Scale.h>
#include <Xm/PushB.h>
#include <Xm/Separator.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/Text.h>
#include <Xm/DrawingA.h>
#include <Xm/RowColumn.h>
#include <Xm/ToggleB.h>
#include <X11/Shell.h>

#include "pix.h"

Widget	top,
        form,
        red, green, blue, intensity,
        frame,
        label,
        text,
        from_text, from_window,
        quit, rgbBtn,
        sep1,
        image_dialog,
        image_window,
        palet_dialog,
        palet_window;

static
XmStringCharSet char_set = (XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

#define xmstr(str) XmStringCreateLtoR (str, char_set)

Display	*display;
XStandardColormap standard_colormap;
Colormap colormap;
int screen;
XColor *rgb, *PixelToRGB();
Pixel	pxs[5];
XImage *image = NULL;
GC gc = 0;
Pixel  *NewPixels = NULL;
int	ncolors;

main (argc, argv)
int	argc;
char	**argv;
{
  int status;
  Colormap *clist;

  top = XtInitialize ("main",
		      "XCe",
		      NULL,NULL,
		      &argc,argv);

  display = XtDisplay(top);
  screen = DefaultScreen(display);
  colormap = DefaultColormap (display,screen);

  CreateApplicationWidgets();

  CreateAndInstallPixmap (pix_bits,
			  pix_width, pix_height,
			  "pix");

  XtRealizeWidget (top);

  FixUp();

  XtMainLoop ();
}

FixUp()
/*
 * pin the sliders to the right side of the form for proper resizing
 */
{
  Arg	args[5];
  int	n;

  n = SetArgs (args,
	       XmNrightAttachment, XmATTACH_FORM,
	       NULL);
  XtSetValues (red, args, n);
  XtSetValues (green, args, n);
  XtSetValues (blue, args, n);
  XtSetValues (intensity, args, n);
}

CreateApplicationWidgets ()
/*
 * create the main screen
 */
{
  Arg	args[20];
  int	n;
  Pixel	pix_bg;
  char	rgb_value[80];
  void	drag_cb(), quit_cb(), window_cb(), help_cb();
  void  CreateRgbDialog();

  n = 0;
  form = XmCreateForm (top,
		       "form",
		       args, n);
  XtManageChild (form);

  n = SetArgs (args,
	       XmNtopAttachment, XmATTACH_FORM,
	       XmNleftAttachment, XmATTACH_FORM,
	       NULL);
  frame = XmCreateFrame (form,
			 "frame",
			 args, n);
  XtManageChild (frame);

  n = 0;
  label = XmCreateLabel (frame,
			 "pix",
			 args, n);
  XtManageChild (label);

  n = SetArgs (args,
	       XmNlabelType, XmPIXMAP,
	       XmNlabelPixmap, GetPixmap (label, "pix"),
	       NULL);
  XtSetValues (label, args, n);

  n = SetArgs (args,
	       XmNwidth, pix_width,
	       XmNheight, pix_height,
	       NULL);
  XtSetValues (frame, args, n);

/***/

  /*
   * get the pixel number from the object under test
   */
  n = SetArgs (args,
	       XmNbackground, &pix_bg,
	       NULL);
  XtGetValues (label, args, n);

  /*
   * obtain an XColor struct for that pixel in the default color map
   */
  rgb = PixelToRGB (display, colormap, pix_bg);

  /*
   * allocate some r/w cells in the default color map, return pixel
   * numbers to use
   */
  XAllocColorCells (display, colormap, False,
		    NULL, 0,
		    pxs, 1);

  /*
   * take our otherwise filled in XColor struct and set its pixel
   * number to one of our r/w cells
   */
  rgb->pixel = pxs[0];

  /*
   * make that pixel number the original objects color
   */
  XStoreColor(display,colormap,rgb);

  /*
   * change the objects pixel number to our new one, same color but
   * now we can change its color on the fly
   */
  n = SetArgs (args,
	       XmNbackground, pxs[0],
	       NULL);
  XtSetValues (label, args, n);

/***/

  n = SetArgs (args,
	       XmNtopAttachment, XmATTACH_FORM,
	       XmNleftAttachment, XmATTACH_WIDGET,
	       XmNleftWidget, frame,
	       XmNorientation, XmHORIZONTAL,
	       XmNmaximum, 255,
	       XmNvalue, (rgb->red>>8),
	       NULL);
  red = XmCreateScale (form,
		       "Red",
		       args, n);
  XtManageChild (red);

  n = SetArgs (args,
	       XmNtopAttachment, XmATTACH_WIDGET,
	       XmNtopWidget, red,
	       XmNleftAttachment, XmATTACH_WIDGET,
	       XmNleftWidget, frame,
	       XmNorientation, XmHORIZONTAL,
	       XmNmaximum, 255,
	       XmNvalue, (rgb->green>>8),
	       NULL);
  green = XmCreateScale (form,
		       "Green",
		       args, n);
  XtManageChild (green);

  n = SetArgs (args,
	       XmNtopAttachment, XmATTACH_WIDGET,
	       XmNtopWidget, green,
	       XmNleftAttachment, XmATTACH_WIDGET,
	       XmNleftWidget, frame,
	       XmNorientation, XmHORIZONTAL,
	       XmNmaximum, 255,
	       XmNvalue, (rgb->blue>>8),
	       NULL);
  blue = XmCreateScale (form,
		       "Blue",
		       args, n);
  XtManageChild (blue);

  n = SetArgs (args,
	       XmNtopAttachment, XmATTACH_WIDGET,
	       XmNtopWidget, blue,
	       XmNleftAttachment, XmATTACH_WIDGET,
	       XmNleftWidget, frame,
	       XmNorientation, XmHORIZONTAL,
	       XmNmaximum, 127,
	       XmNminimum, -127,
	       XmNvalue, 0,
	       NULL);
  intensity = XmCreateScale (form,
			     "Intensity",
			     args, n);
  XtManageChild (intensity);

  sprintf (rgb_value, "#%02x%02x%02x", 
	   (rgb->red>>8), 
	   (rgb->green>>8), 
	   (rgb->blue>>8));

  n = SetArgs (args,
	       XmNtopAttachment, XmATTACH_WIDGET,
	       XmNtopWidget, intensity,
	       XmNleftAttachment, XmATTACH_FORM,
	       XmNrightAttachment, XmATTACH_FORM,
	       
	       XmNeditMode, XmSINGLE_LINE_EDIT,
	       XmNeditable, False,
	       XmNvalue, rgb_value,
	       NULL);
  text = XmCreateText (form,
		       "entry",
		       args, n);
  XtManageChild (text);

  n = SetArgs (args,
	       XmNtopAttachment, XmATTACH_WIDGET,
	       XmNtopWidget, text,
	       XmNleftAttachment, XmATTACH_FORM,
	       NULL);
  quit = XmCreatePushButton (form,
			     "Quit",
			     args, n);
  XtManageChild (quit);

  n = SetArgs (args,
	       XmNtopAttachment, XmATTACH_WIDGET,
	       XmNtopWidget, text,
	       XmNrightAttachment, XmATTACH_FORM,
	       NULL);
  from_window = XmCreatePushButton (form,
				    "Window",
				    args, n);
  XtManageChild (from_window);

  n = SetArgs (args,
	       XmNtopAttachment, XmATTACH_WIDGET,
	       XmNtopWidget, text,
	       XmNrightAttachment, XmATTACH_WIDGET,
	       XmNrightWidget, from_window,
	       NULL);
  from_text = XmCreatePushButton (form,
				  "help",
				  args, n);
  XtManageChild (from_text);

  n = SetArgs (args,
	       XmNtopAttachment, XmATTACH_WIDGET,
	       XmNtopWidget, text,
	       XmNrightAttachment, XmATTACH_WIDGET,
	       XmNrightWidget, from_text,
	       NULL);
  rgbBtn = XmCreatePushButton (form,
			       "RGB",
			       args, n);
  XtManageChild (rgbBtn);


  XtAddCallback (red, XmNdragCallback, drag_cb, 0);
  XtAddCallback (red, XmNvalueChangedCallback, drag_cb, 0);
  XtAddCallback (green, XmNdragCallback, drag_cb, 1);
  XtAddCallback (green, XmNvalueChangedCallback, drag_cb, 1);
  XtAddCallback (blue, XmNdragCallback, drag_cb, 2);
  XtAddCallback (blue, XmNvalueChangedCallback, drag_cb, 2);
  
  XtAddCallback (intensity, XmNdragCallback, drag_cb, 3);
  XtAddCallback (intensity, XmNvalueChangedCallback, drag_cb, 3);

  XtAddCallback (quit, XmNactivateCallback, quit_cb, NULL);
  XtAddCallback (from_window, XmNactivateCallback, window_cb, NULL);
  XtAddCallback (from_text, XmNactivateCallback, help_cb, NULL);
  XtAddCallback (rgbBtn, XmNactivateCallback, CreateRgbDialog, NULL);
}

void
quit_cb (w, client, call)
Widget	w;
caddr_t	client, call;
{
  /*
   * We'll need to do some king of XFreeColors here ...
   */
  XFreeColors (display, colormap, pxs, 1, 8);
  if (NewPixels)
    XFreeColors (display, colormap, NewPixels, ncolors);
  if (image)
    XDestroyImage(image);
  exit(0);
}

int last_value = 0;
void
drag_cb (w, client, call)
Widget	w;
int	client;
XmScaleCallbackStruct *call;
{
  char	rgb_value[80];

  switch (client) {
  case 0:
    rgb->red = (unsigned short) (call->value<<8);
    XmScaleSetValue (intensity, 0);
    last_value = 0;
    break;
  case 1:
    rgb->green = (unsigned short) (call->value<<8);
    XmScaleSetValue (intensity, 0);
    last_value = 0;
    break;
  case 2:
    rgb->blue = (unsigned short) (call->value<<8);
    XmScaleSetValue (intensity, 0);
    last_value = 0;
    break;
  case 3:
    if (call->value > last_value) 
      call->value = 1;
    else if (call->value < last_value)
      call->value = -1;
    else
      call->value = 0;

    last_value += call->value;

    if ((int) (rgb->red>>8) + call->value > 0) {
      rgb->red = (rgb->red>>8) + call->value;
      if (rgb->red > 255) rgb->red = 255;
      rgb->red = rgb->red<<8;
    }
    else {
      rgb->red = 0;
    }
    XmScaleSetValue (red, rgb->red>>8);

    if ((int) (rgb->green>>8) + call->value > 0) {
      rgb->green = (rgb->green>>8) + call->value;
      if (rgb->green > 255) rgb->green = 255;
      rgb->green = rgb->green<<8;
    }
    else {
      rgb->green = 0;
    }
    XmScaleSetValue (green, rgb->green>>8);

    if ((int) (rgb->blue>>8) + call->value > 0) {
      rgb->blue = (rgb->blue>>8) + call->value;
      if (rgb->blue > 255) rgb->blue = 255;
      rgb->blue = rgb->blue<<8;
    }
    else {
      rgb->blue = 0;
    }
    XmScaleSetValue (blue, rgb->blue>>8);

    break;
  }
  XStoreColor (display, colormap, rgb);

  sprintf (rgb_value, "#%02x%02x%02x", 
	   (rgb->red>>8), 
	   (rgb->green>>8), 
	   (rgb->blue>>8));
  XmTextSetString (text, rgb_value);
}


void
window_cb(w, client, call)
Widget	w;
caddr_t	client, call;
{
  Arg	args[5];
  int	n;

  Window target_window, Select_Window();
  target_window = Select_Window (display);
  ObtainImage (display, target_window);

  /*
   * only allow this to happen once
   */
  n = SetArgs (args,
	       XmNsensitive, False,
	       NULL);
  XtSetValues (w, args, n);
}

ObtainImage (dpy, window)
/*
 * Some of this code comes from xwd and xwud.  Many thanks to those
 * authors.  I've removed most of the nice machine independent code
 * so this app is pretty much hard wired to 8 bit r/w colormap displays.
 */
Display	*dpy;
Window	window;
{
  XWindowAttributes win_info;
  int absx, absy, x, y;
  unsigned width, height;
  int dwidth, dheight;
  int bw;
  int nobdrs = True;
  int format = ZPixmap;
  Window new_window;
  GC gc;
  XGCValues values;
  XColor NewColors[256];
  Pixel  OldPixels[256];
  int	i,j;
  int	isize;

  XGetWindowAttributes(dpy, window, &win_info);
  absx = win_info.x + (nobdrs ? win_info.border_width : 0);
  absy = win_info.y + (nobdrs ? win_info.border_width : 0);
  width = win_info.width + (nobdrs ? 0 : (2 * win_info.border_width));
  height = win_info.height + (nobdrs ? 0 : (2 * win_info.border_width));
  dwidth = DisplayWidth (dpy, screen);
  dheight = DisplayHeight (dpy, screen);

  /* clip to window */
  if (absx < 0) width += absx, absx = 0;
  if (absy < 0) height += absy, absy = 0;
  if (absx + width > dwidth) width = dwidth - absx;
  if (absy + height > dheight) width = dheight - absy;

  bw = nobdrs ? 0 : win_info.border_width;
  x = absx - win_info.x - bw;
  y = absy - win_info.y - bw;
  image = XGetImage (dpy, window, x, y, width, height, AllPlanes, format);
  if (!image) {
    fprintf (stderr, "%s:  unable to get image at %dx%d+%d+%d\n",
	     "xce", width, height, x, y);
    exit (1);
  }
  
  /*
   * we got the image, now create a dialog window to display the
   * image and take care of exposures
   */
  CreateImageDialog();

  /*
   * now we're gonna create some r/w color cells.  March through
   * the pixel data and find all unique pixel values:
   */
  isize = Image_Size (image);

  ncolors = 0;
  for (i=0; i<isize; i++) {
    Boolean found = False;
    for (j=0; j<ncolors; j++) {
      if ((unsigned long) image->data[i] == NewColors[j].pixel) {
	found = True;
	break;
      }
    }
    if (!found) {
      OldPixels[ncolors] =
      NewColors[ncolors].pixel = (unsigned long) image->data[i];
      ++ncolors;
    }
  }

  /*
   * Query those new colors to get rgb values
   */
  XQueryColors (display, colormap, NewColors, ncolors);

  NewPixels = (Pixel *) XtMalloc (ncolors * sizeof(Pixel));
  XAllocColorCells (display, colormap, False, NULL, 0,
		    NewPixels, ncolors);
  for (i=0; i<ncolors; i++)
    NewColors[i].pixel = NewPixels[i];

  XStoreColors (display, colormap, NewColors, ncolors);

  /*
   * now change the pixel values in the image to point to our
   * new color map entries
   */
  for (i=0; i<isize; i++) {
    for (j=0; j<ncolors; j++) {
      if ((unsigned long) image->data[i] == OldPixels[j]) {
	image->data[i] = NewPixels[j];
	break;
      }
    }
  }

  /*
   * Now create a popup panel of colors for this image
   */
  CreatePalet (ncolors, NewPixels);

}

CreatePalet (num, pixvals)
/*
 * build the palet of unique colors for the image under test
 */
int num;
unsigned long pixvals[];
{
  Widget	shell, rc, tb;
  void		act_cb();
  Arg		args[15];
  int		n,i;

  n = SetArgs (args,
	       XmNdeleteResponse, XmUNMAP,
	       NULL);
  shell = XtAppCreateShell ("xce", "XCe",
			    applicationShellWidgetClass,
			    display, args, n);

/*  n=0;  */
  palet_dialog = XtCreatePopupShell ("color palet",
				     topLevelShellWidgetClass,
				     shell, args, n);
  n=0;
  rc = XmCreateRowColumn (palet_dialog,
			  "palet",
			  args, n);

  for (i=0; i<num; i++) {
    n=0;
    tb = XmCreatePushButton (rc,
			       "color",
			       args,n);
    XtManageChild(tb);
    n = SetArgs (args,
		 XmNlabelString, xmstr (""),
		 XmNwidth, pix_width,
		 XmNheight, pix_height,
		 NULL);
    XtSetValues (tb, args, n);
    n = SetArgs (args,
		 XmNbackground, pixvals[i],
		 NULL);
    XtSetValues (tb, args, n);
    XtAddCallback (tb, XmNactivateCallback, act_cb, pixvals[i]);
  }

  XtManageChild(rc);


  XtRealizeWidget (palet_dialog);
  XtPopup (palet_dialog, XtGrabNone);
}

void
act_cb (w, client, call)
/*
 * called when a user pushes a button on the palet
 */
Widget	w;
unsigned long client;
caddr_t call;
{
  Arg	args[5];
  int	n;
  char	rgb_value[80];

  n = SetArgs (args,
	       XmNbackground, client,
	       NULL);
  XtSetValues (label, args, n);

  rgb = PixelToRGB (display, colormap, client);
  rgb->pixel = client;

  XmScaleSetValue (red, (rgb->red>>8));
  XmScaleSetValue (blue, (rgb->blue>>8));
  XmScaleSetValue (green, (rgb->green>>8));
  XmScaleSetValue (intensity, 0);

  sprintf (rgb_value, "#%02x%02x%02x",
           (rgb->red>>8),
           (rgb->green>>8),
           (rgb->blue>>8));
  XmTextSetString (text, rgb_value);
}


Window 
Select_Window(dpy)
/*
 * again, from xwd.
 */
Display *dpy;
{
  int status;
  Cursor cursor;
  XEvent event;
  Window target_win = None;
  int buttons = 0;

  /* Make the target cursor */
  cursor = XCreateFontCursor(dpy, XC_crosshair);

  /* Grab the pointer using target cursor, letting it room all over */
  status = XGrabPointer(dpy, RootWindow(dpy, screen), False,
			ButtonPressMask|ButtonReleaseMask, GrabModeSync,
			GrabModeAsync, None, cursor, CurrentTime);
  if (status != GrabSuccess) {
    fprintf (stderr,"Can't grab the mouse.");
    return 0;
  }

  /* Let the user select a window... */
  while ((target_win == None) || (buttons != 0)) {
    /* allow one more event */
    XAllowEvents(dpy, SyncPointer, CurrentTime);
    XWindowEvent(dpy, RootWindow(dpy, screen),
		 ButtonPressMask|ButtonReleaseMask, &event);
    switch (event.type) {
    case ButtonPress:
      if (target_win == None) {
	target_win = event.xbutton.subwindow; /* window selected */
	if (target_win == None)
	  target_win = RootWindow(dpy, screen);
      }
      buttons++;
      break;
    case ButtonRelease:
      if (buttons > 0) /* there may have been some down before we started */
	buttons--;
       break;
    }
  } 

  XUngrabPointer(dpy, CurrentTime);      /* Done with pointer */
  XSync(display,False);
  return(target_win);
}


void image_expose_cb();
XtCallbackRec expose_cblist[] = {
  {image_expose_cb, NULL},
  {NULL, NULL}
};

CreateImageDialog()
/*
 * create a window for the image
 */
{
  Arg	args[15];
  int	n;

  Widget shell;

  shell = XtAppCreateShell ("xce","XCe",
			    applicationShellWidgetClass,
			    display, NULL, 0);

  n = 0;
  image_dialog = XtCreatePopupShell ("image",
				     topLevelShellWidgetClass,
				     shell, args, n);

  n = SetArgs (args,
	       XmNwidth, image->width,
	       XmNheight, image->height,
	       XmNexposeCallback, expose_cblist,
	       NULL);
  image_window = XmCreateDrawingArea (image_dialog,
				      "image",
				      args, n);
  XtManageChild (image_window);

  XtRealizeWidget (image_dialog);

  XtPopup (image_dialog, XtGrabNone);
}

void
image_expose_cb()
/*
 * handles redraws of the image
 */
{
  XGCValues values;

  if (!gc) {
    values.foreground = (unsigned long) BlackPixel (display, screen);
    values.background = (unsigned long) WhitePixel (display, screen);
    gc = XCreateGC (display, XtWindow(image_window), 
		    GCForeground|GCBackground, &values);
  }

  XPutImage (display, XtWindow(image_window), gc, image,
	     0,0,0,0,
	     image->width, image->height);
}

int Image_Size(image)
XImage *image;
{
  if (image->format != ZPixmap)
    return(image->bytes_per_line * image->height * image->depth);

  return(image->bytes_per_line * image->height);
}


static char help_message[] = "\
Welcome to Color Editor!\n\
\n\
The red, green, and blue labeled slider bars control the amount\n\
of those colors in the mix for the small screen displayed in the\n\
upper left corner.  The forth slider is used to ajust the intensity\n\
of the color displayed.  As the sliders are moved, the color of the screen\n\
changes according to the new vaules of the sliders.  A text string\n\
suitable for use in resources files is also displayed.\n\
\n\
The button labeled `rgb' will pop up a window which contains the default\n\
rgb database on your system.  You can choose colors from this table as a\n\
base for editing in the small color edit screen.\n\
\n\
You may edit the colors for an entire window.  Push the `Window' button\n\
and the cursor will change into cross hairs.  Move this cursor over a\n\
window you'd like to edit and click the left mouse button.  You will\n\
get a new window with the image of the first placed into it.  You will\n\
also get a palet of colors that make up the image.  Select a color and\n\
it will appear in the edit sceen, where it can be ajusted with the sliders.\n\
You may only do this once, since the color table is a limited resource.";

void
help_cb (w, client, call)
Widget	w;
caddr_t	client, call;
{
  Warning (w, help_message);
}

Warning (reference, message)
Widget  reference;
char    *message;
/*
 * post an error message
 */
{
  Widget        mbox;
  Widget        button;
  Arg           args[10];
  int           n;
  XmString      msg_string;

  msg_string = XmStringLtoRCreate (message, XmSTRING_DEFAULT_CHARSET);

  n=0;
  XtSetArg (args[n], XmNmessageString, msg_string); n++;
  XtSetArg (args[n], XmNdefaultPosition, True); n++;
  XtSetArg (args[n], XmNdialogTitle, XmStringCreateLtoR ("Simple Instructions",XmSTRING_DEFAULT_CHARSET)); n++;
  mbox = XmCreateWarningDialog (reference,
                              "helpWindow",
                              args,n);
  button = XmMessageBoxGetChild (mbox, XmDIALOG_CANCEL_BUTTON);
  XtUnmanageChild (button);
  button = XmMessageBoxGetChild (mbox, XmDIALOG_HELP_BUTTON);
  XtUnmanageChild (button);

  /*
   * bring it up
   */
  XtManageChild (mbox);
}

void
CreateRgbDialog ()
{
  Widget	rgb_dialog, rgb_window, rc, tb, shell;
  Arg	args[15];
  int	n;
  FILE	*fp, *fopen();
  unsigned long	r,g,b;
  unsigned long	lr,lg,lb;
  char	line[160];
  char	*tok;
  XColor rgb_color, tmp;
  void rgb_select_cb();

  shell = XtAppCreateShell ("xce", "XCe",
			    applicationShellWidgetClass,
                            display, NULL, 0);

  n = 0;
  rgb_dialog = XtCreatePopupShell ("rgb screen",
				   topLevelShellWidgetClass,
				   shell, args, n);
  
  n = SetArgs (args,
	       XmNradioAlwaysOne, True,
	       XmNradioBehavior, True,
	       NULL);
  rc = XmCreateRowColumn (rgb_dialog,
                          "rgb",
                          args, n);
  XtManageChild (rc);

#ifdef VMS
  if ((fp = fopen ("SYS$MANAGER:DECW$RGB.DAT","r")) == NULL) {
    Warning (top, "Can't open SYS$MANAGER:DECW$RGB.DAT for input.");
#else
  if ((fp = fopen ("/usr/lib/X11/rgb.txt","r")) == NULL) {
    Warning (top, "Can't open /usr/lib/X11/rgb.txt for input.");
#endif  /* VMS */
    return;
  }

  lr = lb = lg = 0;

  while (fgets (line, 160, fp)) {
    tok = (char *) strtok (line," ");
    r = (unsigned long) strtol (tok, NULL, 10);
    tok = (char *) strtok (NULL," ");
    g = (unsigned long) strtol (tok, NULL, 10);
    tok = (char *) strtok (NULL," \t");
    b = (unsigned long) strtol (tok, NULL, 10);
    
    tok = (char *) strtok (NULL,"\n");
    if (tok[0] == '\t')
      ++tok;

    if ((r == lr) && (b == lb) && (g == lg))
      continue;

    if ((r == g) && (g == b)) {
      continue;
    }

    lr = r;
    lg = g;
    lb = b;

    if (!XLookupColor (display, colormap, tok, &tmp, &rgb_color))
      continue;

    if (!XAllocColor (display, colormap, &rgb_color))
      continue;

    /*
     * Got a unique color spec and name.  Create a button in the
     * row column to display it.
     */
    n = SetArgs (args,
		 XmNforeground, rgb_color.pixel,
		 XmNlabelString, xmstr (tok),
		 NULL);
    tb = XmCreateToggleButton (rc,
			       "rgbButton",
			       args, n);
    XtManageChild (tb);
    XtAddCallback (tb, XmNarmCallback, rgb_select_cb, rgb_color.pixel);
  }

  XtRealizeWidget (rgb_dialog);
  XtPopup (rgb_dialog, XtGrabNone);
}

void
rgb_select_cb (w, client, call)
Widget	w;
unsigned long client;
caddr_t call;
{
  XColor new_rgb_spec;
  char  rgb_value[80];

  new_rgb_spec.pixel = client;
  XQueryColor (display, colormap, &new_rgb_spec);

  rgb->red = new_rgb_spec.red;
  rgb->green = new_rgb_spec.green;
  rgb->blue = new_rgb_spec.blue;

  XStoreColor (display, colormap, rgb);

  XmScaleSetValue (red, (rgb->red>>8));
  XmScaleSetValue (blue, (rgb->blue>>8));
  XmScaleSetValue (green, (rgb->green>>8));
  XmScaleSetValue (intensity, 0);

  sprintf (rgb_value, "#%02x%02x%02x",
           (rgb->red>>8),
           (rgb->green>>8),
           (rgb->blue>>8));
  XmTextSetString (text, rgb_value);
}
