/*
 *   xsnap -- take a snapshot of a portion of the screen.
 *
 *   Copyright 1989 Clauss Strauch
 *                  cbs@cad.cs.cmu.edu
 *
 *   Permission to use, copy, modify, and distribute this software and its
 *   documentation for any purpose and without fee is hereby granted.
 *   This software is provided "as is", without express or implied warranty.
 *   
 *   Modified by Hal Brand 04-May-1989 to use XImages instead of Pixmaps
 */

#include <stdio.h> 
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

/*  Leave arguments as globals, since there are so many of them.
 *  They'll only be referenced in process_args and main.
 */

char *display_string = NULL;         /* display we'll open */
int  border_width = 2;               /* border_width  of the snapshot window */
int  start_iconic = FALSE;           /* start snap window in iconic state?  */
char *color_name = NULL;             /* user-supplied name of border color */
char *window_geom_string = NULL;     /* geometry of snapshot window */
char *icon_geom_string = NULL;       /* icon geometry */
char *region_geom_string = NULL;     /* location of region to copy */
char *app_name = "xsnap";            /* name of application for window manager */
int grab_server = TRUE;              /* grab the server? */


/*
 *  create_event_window returns the ID of a InputOnly window that covers
 *  the given window.
 */

Window create_event_window(a_display,a_window, init_cursor)
     Display *a_display;
     Window a_window;
{
  XSetWindowAttributes xswa;
  unsigned long xswa_valuemask;
  Window root_win;
  Window event_window;
  unsigned int win_width, win_height;
  unsigned int  win_border_width, win_depth;
  int win_x, win_y;
  
  /* get the geometry of the window  */
  
  XGetGeometry(a_display, a_window, &root_win, &win_x, &win_y,
	       &win_width, &win_height, &win_border_width, &win_depth);
  
  /* make an input only window to get events from  */
  
  xswa.cursor = init_cursor;
  xswa.override_redirect = True;
  xswa.event_mask = ButtonPressMask | ButtonReleaseMask | Button1MotionMask;
  xswa_valuemask = CWCursor | CWOverrideRedirect | CWEventMask;
  event_window = XCreateWindow(a_display, a_window, win_x, win_y, win_width,
			       win_height, 0, 0, InputOnly, CopyFromParent, 
			       xswa_valuemask, &xswa);
  return(event_window);
}

/*
 *   draw box draws a box on the given window, with the given GC 
 *
 */

void draw_box(a_display, a_window, a_gc, x1, y1, x2, y2)
     GC a_gc;
     Window a_window;
     Display *a_display;
     int x1, y1, x2, y2;
{
  XSegment segments[4];
  segments[0].x1 = (short)x1;
  segments[0].y1 = (short)y1;
  segments[0].x2 = (short)x1;
  segments[0].y2 = (short)y2;
  
  segments[1].x1 = (short)x1;
  segments[1].y1 = (short)y1;
  segments[1].x2 = (short)x2;
  segments[1].y2 = (short)y1;
  
  segments[2].x1 = (short)x2;
  segments[2].y1 = (short)y2;
  segments[2].x2 = (short)x1;
  segments[2].y2 = (short)y2;
  
  segments[3].x1 = (short)x2;
  segments[3].y1 = (short)y2;
  segments[3].x2 = (short)x2;
  segments[3].y2 = (short)y1;
  
  XDrawSegments(a_display, a_window, a_gc, segments, 4);
}

/*
 *  get_region
 * takes as input:
 *    display
 *    window to get region from 
 *    pointers to x1, y1, width, height
 *  
 *   returns:  the position and width and height of a
 *             user selected region via the given pointers.
 *
 */


get_region(a_display, a_window,  x, y, width, height)
     Display *a_display;
     Window a_window;
     int *x, *y;
     unsigned int *height, *width;
{
  Window event_window;
  Cursor up_right_curs, up_left_curs;
  Cursor low_right_curs, low_left_curs;
  Cursor current_cursor;
  int done;
  int init_x, init_y;
  int last_x, last_y;
  XEvent event;
  GC xor_gc;
  XGCValues xor_gc_values;             /* for creating xor_gc */
  unsigned long xor_gc_valuemask;      /* valuemask for creating xor_gc */
  
  /* make the GC and cursors we'll need */
  
  up_right_curs = XCreateFontCursor(a_display, XC_ur_angle);
  
  up_left_curs = XCreateFontCursor(a_display, XC_ul_angle);
  
  low_right_curs = XCreateFontCursor(a_display, XC_lr_angle);
  
  low_left_curs = XCreateFontCursor(a_display, XC_ll_angle);
  
  
  
  xor_gc_valuemask = GCFunction | GCSubwindowMode  | GCForeground;
  xor_gc_values.function = GXxor;
  xor_gc_values.foreground = 0xfd;
  xor_gc_values.subwindow_mode = IncludeInferiors;
  xor_gc = XCreateGC(a_display, a_window, xor_gc_valuemask, &xor_gc_values);
  
  event_window = create_event_window(a_display, a_window,up_left_curs);
  XMapRaised(a_display, event_window);
  
  
  if (XGrabPointer(a_display, event_window, True, 
		   ButtonPressMask,
		   GrabModeAsync, GrabModeAsync, None, up_left_curs, 
		   CurrentTime) != 0)
    {
      fprintf(stderr, "Cannot grab pointer.");
      exit(1);
    }
  
  
  /* get the initial button  press */
  done = FALSE;
  while (! done)
    {
      XNextEvent(a_display, &event);
      switch(event.type)
	{
	case MappingNotify:
	  XRefreshKeyboardMapping((XMappingEvent *) &event);
	  break;
	case ButtonPress:
	  if (event.xbutton.button == 1)
	    {
	      init_x = event.xbutton.x;
	      init_y = event.xbutton.y;
	      
	      
	      done = TRUE;
	      break;
	    }
	}
    }
  
  
  /*  now we have the location of one corner of the box.   change the cursor,
   *  and have the user drag out the area.
   */
  current_cursor = low_right_curs;
  last_x = init_x;
  last_y = init_y;
  XChangeActivePointerGrab(a_display, ButtonReleaseMask | Button1MotionMask,
			   current_cursor, CurrentTime);
  done = FALSE;
  draw_box(a_display, a_window, xor_gc, init_x, init_y, last_x, last_y);
  while (! done)
    {
      XNextEvent(a_display, &event);
      switch(event.type)
	{
	case MappingNotify:
	  XRefreshKeyboardMapping((XMappingEvent *) &event);
	  break;
	case MotionNotify:
	  draw_box(a_display, a_window, xor_gc, 
		   init_x, init_y, last_x, last_y);  /* erase old */
	  last_x = event.xmotion.x;
	  last_y = event.xmotion.y;
	  draw_box(a_display, a_window, xor_gc, 
		   init_x, init_y, last_x, last_y); /* draw new  */
	  /*  Change cursor to correspond to position of pointer */
	  if ((init_x < last_x) && (init_y < last_y)
	      && (current_cursor != low_right_curs))
	    {
	      current_cursor = low_right_curs;
	      XChangeActivePointerGrab(a_display, 
				       ButtonReleaseMask | Button1MotionMask,
				       low_right_curs, CurrentTime);
	    }
	  else if ((last_x < init_x) && (last_y < init_y)
		   &&  (current_cursor != up_left_curs))
	    {
	      current_cursor = up_left_curs;
	      XChangeActivePointerGrab(a_display, 
				       ButtonReleaseMask | Button1MotionMask,
				       up_left_curs, CurrentTime);
	    }
	  
	  else if ((init_x < last_x) && (last_y < init_y)
		   && (current_cursor != up_right_curs))
	    {
	      current_cursor = up_right_curs;
	      XChangeActivePointerGrab(a_display, 
				       ButtonReleaseMask | Button1MotionMask,
				       up_right_curs, CurrentTime);
	      
	    }
	  else if ((last_x < init_x) && (init_y < last_y)
		   && (current_cursor != low_left_curs))
	    {
	      current_cursor = low_left_curs;
	      XChangeActivePointerGrab(a_display, 
				       ButtonReleaseMask | Button1MotionMask,
				       low_left_curs, CurrentTime);
	    }
	  
	  break;
	case ButtonRelease:
	  if (event.xbutton.button == 1)
	    {
	      done = TRUE;
	      /* erase last box drawn */
	      draw_box(a_display, a_window, xor_gc, 
		       init_x, init_y, last_x, last_y);
	    }
	  break;
	}
    }
  XFlush(a_display);   /*  gets rid of last box on screen  */
  if (init_x < last_x) *x = init_x;
  else *x = last_x;
  if (init_y < last_y) *y = init_y;
  else *y = last_y;
  *width = (unsigned int)abs(last_x - init_x);
  *height = (unsigned int)abs(last_y - init_y);
  /* clean up after ourself: */
  
  XDestroyWindow(a_display, event_window);
  XFreeGC(a_display, xor_gc);
  
  /* we'll let the caller ungrab the pointer */
  
}

/* 
 *  get_image_region 
 *
 *       input :
 *               a display, a window, x, y, width, height, interactive.
 *               if interactive, the user is prompted for a region,
 *               other wise the given region is copied to a Image.
 *       returns : an image containing a copy of a user-specified area
 *                 of the given window;
 *
 */

XImage *get_image_region(a_display, a_screen, a_window, a_gc, x, y, 
			 width, height, interactive)
     Display *a_display;
     GC a_gc;
     int a_screen;
     Window a_window;
     int *x, *y;
     unsigned int *width, *height;
     int interactive;
     
{
  int reg_x, reg_y;
  unsigned int reg_width, reg_height;
  XImage *image_returned;
  int return_flag;
  
  if (interactive){
    get_region(a_display, a_window, &reg_x, &reg_y,
	       &reg_width, &reg_height);
    *x = reg_x;
    *y = reg_y;
    *width = reg_width;
    *height = reg_height;
  }
  
  image_returned = XGetImage(a_display,DefaultRootWindow(a_display),
			     *x,*y,*width,*height,
			     XAllPlanes(),ZPixmap);
  
  if (interactive)  XUngrabPointer(a_display, CurrentTime);
  return(image_returned);
}


/* 
 *  X error handler for,  mainly for future use
 */

int my_X_error_handler(display, myerr)
     Display *display;
     XErrorEvent *myerr;
{
  char msg[80];  
  
  
  /* Print out the error string that X generates */
  
  XGetErrorText(display, myerr->error_code, msg,80);
  fprintf(stderr, "xsnap error: %s\n", msg);
  /* no way to continue, so we'll just bag it  */
  exit(1);
  return(1);
}

  

/*
 *  print a usage message to the user
 */

usage()
{
  printf("%s\n", "Usage:  xsnap [-options...]");
  printf("%s\n", "Where options are:");
  printf("%s\n", "[-display host:display]  Display to connect to.");
  printf("%s\n", "[-bd borderwidth]        Color of border");
  printf("%s\n", "[-bw borderwidth]        Width of border");
  printf("%s\n", "[-geometry geom]         Geometry of snapshot window");
  printf("%s\n", "[-icongeometry geom]     Location of icon.");
  printf("%s\n", "[-region geom]           Region of screen to be copied.");
  printf("%s\n", "[-iconic]                Start up in iconic state");
  printf("%s\n", "[-name name]             Passed to window manager for name of window.");
  printf("%s\n", "[-nograb]                Don't grab server during specification of region.");
  exit(1);
}


/*
 *        get the arguments.   arguments are matched to the first
 *        distinguishing substring.
 */
process_args(argc, argv)
     int argc;
     char **argv;
{
  int i;
  
  for (i = 1; i < argc; i++)
    {
      if (strncmp(argv[i], "-h", 2) == 0)
	{
	  usage();
	  continue;
	}
      if (strncmp(argv[i], "-display", 2) == 0)
	{
	  display_string = argv[++i];
	  continue;
	}
      if (strncmp(argv[i], "-bw", 3) == 0)
	{
	  border_width = atoi(argv[++i]);
	  continue;
	}
      if (strncmp(argv[i], "-bd", 3) == 0)
	{
	  color_name = argv[++i];
	  continue;
	}
      if (strncmp(argv[i], "-ge", 3) == 0)
	{
	  window_geom_string = argv[++i];
	  continue;
	}
      if (strncmp(argv[i], "-icong", 6) == 0)
	{
	  icon_geom_string = argv[++i];
	  continue;
	}
      if (strncmp(argv[i], "-r", 2) == 0)
	{
	  region_geom_string = argv[++i];
	  continue;
	}
      if (strncmp(argv[i], "-iconi", 6) == 0)
	{
	  start_iconic = TRUE;
	  continue;
	}
      if (strncmp(argv[i], "-na", 3) == 0)
	{
	  app_name = argv[++i];
	  continue;
	}
      if (strncmp(argv[i], "-no", 3) == 0)
	{
	  grab_server = FALSE;
	  continue;
	}
      usage();
    }
}

main(argc, argv)
     int argc;
     char **argv;
{
  Display *the_display;
  int the_screen;
  XWMHints wmhints;
  XEvent an_event;
  GC copy_gc;
  unsigned long copy_gc_valuemask;
  XGCValues copy_gc_values;
  long wmhints_mask;
  XSizeHints wm_size_hints;
  int x_return, y_return;           /* for XParseGeometry */
  int width_return, height_return;  /* ditto              */
  int geom_mask;                    /* bitmask for XParseGeometry fields */
  unsigned long border_color_pixel; /* pixel for border color */
  XColor color;
  Colormap cmap;
  Window window_to_snap;            /* always root window, now */
  Window snapshot;
  Cursor snap_cursor;
  XImage *snap_image;
  int reg_x, reg_y;
  int icon_x, icon_y;
  int snap_x, snap_y;
  unsigned int snap_width, snap_height;
  unsigned int reg_width, reg_height;
  int done;
  char buffer [4];                   /* key press buffer for XLookupString  */
  int string_length;                 /* length of returned string */
  
  process_args(argc, argv);
  XSetErrorHandler(my_X_error_handler);
  if ((the_display = XOpenDisplay(display_string)) == NULL)
    {
      
      fprintf(stderr, "Cannot open display:    %s\n", 
	      XDisplayName(display_string));
      exit(1);
    }
  XSetErrorHandler(my_X_error_handler);
  the_screen = DefaultScreen(the_display);
  window_to_snap = XRootWindow(the_display, the_screen);
  
  /* make copy GC */
  
  copy_gc_valuemask = GCSubwindowMode;
  copy_gc_values.subwindow_mode = IncludeInferiors;
  copy_gc = XCreateGC(the_display, window_to_snap, copy_gc_valuemask,
		      &copy_gc_values);
  
  if (grab_server)
    {
      XGrabServer(the_display);
      XSync(the_display, 0);
    }
  
  if (region_geom_string)
    {
      geom_mask = XParseGeometry(region_geom_string, &reg_x, &reg_y,
				 &reg_width, &reg_height);
      if ((geom_mask & XValue) && (geom_mask & YValue) &&
	  (geom_mask & WidthValue) && (geom_mask & HeightValue))
	/* must specify complete geometry for region */
	{
	  if (geom_mask & XNegative)
	    reg_x += DisplayWidth(the_display, the_screen) - reg_width;
	  if (geom_mask & YNegative)
	    reg_y += DisplayHeight(the_display, the_screen) - reg_height;
	  snap_image = get_image_region(the_display, the_screen, 
					window_to_snap, copy_gc,
					&reg_x, &reg_y, &reg_width, 
					&reg_height, FALSE);
	}
      else usage();
    }
  else 
    snap_image = get_image_region(the_display,the_screen,window_to_snap,
				  copy_gc,&reg_x,&reg_y,
				  &reg_width,&reg_height,TRUE);
  
  /* ungrab the server */
  
  XUngrabServer(the_display);
  
  /* process border color */
  
  cmap = DefaultColormap(the_display, the_screen);
  if ((color_name) && (XParseColor(the_display, cmap, color_name, &color)) &&
      (XAllocColor(the_display, cmap, &color)))
    border_color_pixel = color.pixel;
  else
    border_color_pixel = BlackPixel(the_display, the_screen);
  
  /*
   * get location of our window(from the window_geom_string),
   * and set up the size hints structure.
   */     

  wm_size_hints.flags = 0;
  width_return = reg_width;
  height_return = reg_height;
  if (window_geom_string)
    {
      geom_mask = XParseGeometry(window_geom_string, &x_return, 
				 &y_return, (unsigned int *) &width_return, 
				 (unsigned int *) &height_return);
      if (geom_mask & XValue)
	{
	  if (geom_mask & XNegative)
	    wm_size_hints.x = DisplayWidth(the_display, 
					   the_screen)
	      - width_return + x_return - 
		(border_width * 2);
	  else
	    wm_size_hints.x = x_return;
	}
      if (geom_mask & YValue)
	{
	  if (geom_mask & YNegative)
	    wm_size_hints.y =  
	      DisplayHeight(the_display, the_screen)
		- height_return + y_return  - 
		  (border_width * 2);
	  
	  else
	    wm_size_hints.y = y_return;
	}
      
    }
  
  if ((geom_mask & XValue) || (geom_mask & YValue))
    wm_size_hints.flags |= USPosition;
  
  if ((geom_mask & WidthValue) || (geom_mask & HeightValue))
    wm_size_hints.flags |= USSize;
  wm_size_hints.width = width_return;
  wm_size_hints.height = height_return;
  
  
  snapshot = XCreateSimpleWindow(the_display, 
				 XRootWindow(the_display, the_screen), 
				 wm_size_hints.x, wm_size_hints.y,
				 wm_size_hints.width, 
				 wm_size_hints.height, border_width,
				 border_color_pixel,
				 BlackPixel(the_display, the_screen));
  
  if (window_geom_string)
    XSetNormalHints(the_display, snapshot, &wm_size_hints);
  XStoreName(the_display, snapshot, app_name);
  wmhints_mask = 0;
  if (start_iconic)
    {
      wmhints_mask |= StateHint;
      wmhints.initial_state = IconicState;
    }
  x_return = y_return = 0;
  
  /* Icon geometry is sorta broken, since we don't supply an icon, and 
   * there isn't any way I know of to get the geometry  of the icon that
   * the window manager will provide.   So, we'll pretend that our icon 
   * is 32x32 (ugh).
   */
  
  if (icon_geom_string)
    {
      geom_mask = XParseGeometry(icon_geom_string, &x_return, 
				 &y_return, (unsigned int *) &width_return, 
				 (unsigned int *) &height_return);
      if (geom_mask & XValue)
	{
	  if (geom_mask & XNegative)
	    wmhints.icon_x = DisplayWidth(the_display, 
					  the_screen)
	      - 32 + x_return - (border_width * 2);
	  else
	    wmhints.icon_x = x_return;
	}
      if (geom_mask & YValue)
	{
	  if (geom_mask & YNegative)
	    wmhints.icon_y =  DisplayHeight(the_display, 
					    the_screen)
	      - 32 + y_return  - (border_width * 2);
	  
	  else
	    wmhints.icon_y = y_return;
	}
      wmhints_mask |= IconPositionHint;
    }
  
  
  if (wmhints_mask)
    {
      wmhints.flags = wmhints_mask;
      XSetWMHints(the_display, snapshot, &wmhints);
    }
  
  /* solicit expose events */
  
  XSelectInput(the_display, snapshot, ExposureMask | KeyPressMask);
  
  /* give it a different cursor */
  
  snap_cursor = XCreateFontCursor(the_display, XC_gumby);
  
  XDefineCursor(the_display, snapshot, snap_cursor);
  
  /* map it */
  
  XMapRaised(the_display, snapshot);
  
  /* process events */
  done = FALSE;
  
  while (! done)
    {
      XNextEvent(the_display, &an_event);
      switch (an_event.type)
	{
	case MappingNotify:
	  XRefreshKeyboardMapping((XMappingEvent *) &an_event);
	  break;
	case Expose:
	  if (an_event.xexpose.count==0)
	    XPutImage(the_display,snapshot,copy_gc,snap_image,
		      0,0,0,0,snap_image->width,
		      snap_image->height);
	  break;
	case KeyPress:
	  string_length = XLookupString((XKeyEvent *) &an_event, buffer, 
					sizeof(buffer),
					NULL, NULL);
	  if ((string_length == 1) && ((buffer[0] == 'q') ||
				       (buffer[0] == 'Q') ||
				       (buffer[0] == '\003')))
	    
	    /*  clean up  */
	    {
	      XDestroyWindow(the_display, snapshot);
	      XFreeGC(the_display, copy_gc);
	      XCloseDisplay(the_display);
	      done = TRUE;
	      exit(0);
	      break;
	    }
	}
    }
  
}
