/*
 * Program:	VMS TCP/IP routines for Netlib.
 *
 * Author:	Yehavi Bourvine, The Hebrew University of Jerusalem
 *		Internet: Yehavi@VMS.huji.ac.il
 *
 * Date:	2 August 1994
 * Last Edited:	7 September 1994
 *
 * Updated:     23 September 1996 Andy Harper
 *		-  Modify for netlib version 2.0 interface
 *		-  Added a user callable 'gethostname' routine
 *		-  Added logging (not in production code though!)
 *
 * Copyright 1994 by the University of Washington
 *
 *  Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appears in all copies and that both the
 * above copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the University of Washington not be
 * used in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.	This software is made available
 * "as is", and
 * THE UNIVERSITY OF WASHINGTON DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 * WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN
 * NO EVENT SHALL THE UNIVERSITY OF WASHINGTON BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT
 * (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */



#include <descrip.h>
#include "netlib_dir:netlibdef.h"

/*
   This network module includes logging but requires the main program to
   initialize it. Comment out this code if that is not available.
*/
/*
   extern FILE *logfile;
   extern int loglevel;
#  define LOG(n,fmt,a,b,c,d) {if (logfile && loglevel>=n) fprintf(logfile,fmt,a,b,c,d);}
*/
#  define LOG(n,fmt,a,b,c,d) {}

static TCPSTREAM *clientstream = NULL;	/* For the client calling us */



/* TCP/IP manipulate parameters
 * Accepts: function code
 *	    function-dependent value
 * Returns: function-dependent return value
 */

void *tcp_parameters (long function,void *value)
{
  return NIL;
}


/* TCP/IP server setup
 * Accepts: nothing
 * Returns: TCP/IP stream structure for the stdio channels created by master server
 */
TCPSTREAM *tcp_server_init()
{
  char hostname[MAILTMPLEN] = "UNKNOWN";
  unsigned int sock = 0;
  int status;
  				/* hostname connected to */
  struct dsc$descriptor HostDesc = { 0, DSC$K_DTYPE_T, DSC$K_CLASS_S, NULL };


				/* Find out remote hostname on socket */
				/* .... Left for the moment - don't need it */

				/* Initialize the channels created by master server */
  LOG(12,"tcp_server_init: entering\r\n",0,0,0,0);
  status = netlib_server_setup(&sock, 0, 0);
  LOG(12,"tcp_server_init: return from netlib_server_setup,status=%08x,sock=%08x\r\n",status,sock,0,0);
  if (!(status & 0x1))
    {
      char tmp[100];
      sprintf (tmp,"Unable to set up server, status=%08x",status);
      mm_log (tmp,ERROR);
      return NIL;
    }


  clientstream = (TCPSTREAM *) fs_get (sizeof (TCPSTREAM));
				/* copy official host name */
  clientstream->host = cpystr (hostname);
				/* copy local host name */
  clientstream->localhost = cpystr (mylocalhost ());
				/* init sockets */
  clientstream->tcpsi = clientstream->tcpso = sock;
  clientstream->ictr = 0;		/* init input counter */
  LOG(12,"tcp_server_init: exiting, stream addr =%08x\r\n",clientstream,0,0,0);
  return clientstream;		/* return success */
}

 
/* TCP/IP open
 * Accepts: host name
 *	    contact service name
 *	    contact port number
 * Returns: TCP/IP stream if success else NIL
 */

TCPSTREAM *tcp_open (char *host,char *service,long port)
{
  TCPSTREAM *stream = NIL;
  unsigned long sock;
  int status;
  char *s;
  char hostname[MAILTMPLEN];
  char tmp[MAILTMPLEN];
				/* hostname to connect to */
  struct dsc$descriptor HostDesc = { 0, DSC$K_DTYPE_T, DSC$K_CLASS_S, NULL };

				/* assign a local socket */
  if (!((status = netlib_socket (&sock)) & 0x1)) {
    sprintf (tmp,"Unable to assign to net, status=%08x",status);
    mm_log (tmp,ERROR);
    return NIL;
  }

				/* open connection */
  HostDesc.dsc$w_length = strlen (host);
  HostDesc.dsc$a_pointer = host;
  if (!((status = netlib_connect_by_name (&sock, &HostDesc, &port)) & 0x1)) {
    sprintf (tmp,"Can't connect to %.80s,%d: 0x%08x",host,port,status);
    mm_log (tmp,ERROR);
    return NIL;
  }

  strcpy(hostname,host);
				/* create TCP/IP stream */
  stream = (TCPSTREAM *) fs_get (sizeof (TCPSTREAM));
				/* copy official host name */
  stream->host = cpystr (hostname);
				/* copy local host name */
  stream->localhost = cpystr (mylocalhost ());
				/* init sockets */
  stream->tcpsi = stream->tcpso = sock;
  stream->ictr = 0;		/* init input counter */
  return stream;		/* return success */
}

/* TCP/IP authenticated open
 * Accepts: host name
 *	    service name
 * Returns: TCP/IP stream if success else NIL
 */

TCPSTREAM *tcp_aopen (char *host,char *service)
{
  return NIL;
}

/* TCP/IP receive line
 * Accepts: TCP/IP stream
 * Returns: text line string or NIL if failure
 */

char *tcp_getline (TCPSTREAM *stream)
{
  int n,m;
  char *st,*ret,*stp;
  char c = '\0';
  char d;
				/* make sure have data */
  LOG(12,"tcp_getline: Entering on stream %08x\r\n",stream,0,0,0);
  if (!tcp_getdata (stream))
    {
      LOG(12,"tcp_getline:  NIL return from tcp_getdata\r\n",0,0,0,0);
      return NIL;
    }
  st = stream->iptr;		/* save start of string */
  n = 0;			/* init string count */
  while (stream->ictr--) {	/* look for end of line */
    d = *stream->iptr++;	/* slurp another character */
    if ((c == '\015') && (d == '\012')) {
      ret = (char *) fs_get (n--);
      memcpy (ret,st,n);	/* copy into a free storage string */
      ret[n] = '\0';		/* tie off string with null */
      LOG(12,"tcp_getline: returning line size %d\r\n%S\r\n",strlen(ret),ret,0,0);
      return ret;
    }
    n++;			/* count another character searched */
    c = d;			/* remember previous character */
  }
				/* copy partial string from buffer */
  memcpy ((ret = stp = (char *) fs_get (n)),st,n);
				/* get more data from the net */
  LOG(12,"tcp_getline: testing tcp_getdata again\r\n",0,0,0,0);
  if (!tcp_getdata (stream))
    {
      LOG(12,"tcp_getline: NIL return from tcp_getdata\r\n",0,0,0,0);
      return NIL;
    }
				/* special case of newline broken by buffer */
  if ((c == '\015') && (*stream->iptr == '\012')) {
    stream->iptr++;		/* eat the line feed */
    stream->ictr--;
    ret[n - 1] = '\0';		/* tie off string with null */
  }
				/* else recurse to get remainder */
  else if (st = tcp_getline (stream)) {
    ret = (char *) fs_get (n + 1 + (m = strlen (st)));
    memcpy (ret,stp,n);		/* copy first part */
    memcpy (ret + n,st,m);	/* and second part */
    fs_give ((void **) &stp);	/* flush first part */
    fs_give ((void **) &st);	/* flush second part */
    ret[n + m] = '\0';		/* tie off string with null */
  }
  LOG(12,"tcp_getline: returning line size %d\r\n%s\r\n",strlen(ret),ret,0,0);
  return ret;
}

/* TCP/IP receive buffer
 * Accepts: TCP/IP stream
 *	    size in bytes
 *	    buffer to read into
 * Returns: T if success, NIL otherwise
 */

long tcp_getbuffer (TCPSTREAM *stream,unsigned long size,char *buffer)
{
  unsigned long n;
  char *bufptr = buffer;
  LOG(12,"tcp_getbuffer: Entering on stream %08x, size %d\r\n",stream,size,0,0);
  while (size > 0) {		/* until request satisfied */
    if (!tcp_getdata (stream))
      {
        LOG(12,"tcp_getbuffer: NIL return from tcp_getdata\r\n",0,0,0,0);
        return NIL;
      }
    n = min (size,stream->ictr);/* number of bytes to transfer */
				/* do the copy */
    memcpy (bufptr,stream->iptr,n);
    bufptr += n;		/* update pointer */
    stream->iptr +=n;
    size -= n;			/* update # of bytes to do */
    stream->ictr -=n;
  }
  bufptr[0] = '\0';		/* tie off string */
  LOG(12,"tcp_getbuffer: returning buffer, size %d\r\n%s\r\n",strlen(buffer),buffer,0,0);
  return T;
}


/* TCP/IP receive data
 * Accepts: TCP/IP stream
 * Returns: T if success, NIL otherwise
 */

long tcp_getdata (TCPSTREAM *stream)
{
  char tmp[MAILTMPLEN];
  int i,status;
  /* Note: the doc says we need here dynamic descriptor, but we need static
   * one... */
  struct dsc$descriptor BufDesc = {BUFLEN,DSC$K_DTYPE_T,DSC$K_CLASS_S,
				     stream->ibuf};
  static short iosb[4];
  LOG(12,"tcp_getdata: entering on stream %08x, socket=%08x\r\n",stream,stream->tcpsi,0,0);
  if (stream->tcpsi < 0) return NIL;
  while (stream->ictr < 1) {	/* if nothing in the buffer */
    if (!((status = netlib_read(&(stream->tcpsi), &BufDesc, 0,0,0,0, iosb)) & 0x1)) {
      LOG(12,"tcp_getdata: error from netlib_read, status=%08x\r\n",status,0,0,0);
      sprintf (tmp,"Error reading from TcpIp/NETLIB, status=%08x",status);
      mm_log (tmp,ERROR);
      return tcp_abort (stream);
    }
    if (iosb[1] > BUFLEN) i = BUFLEN;
    else i = iosb[1];
    if (i < 1)
      {
         LOG(12,"tcp_getdata: byte count < 1 (=%d), aborting stream\r\n",i,0,0,0);
         return tcp_abort (stream);
      }
    stream->ictr = i;		/* set new byte count */
    stream->iptr = stream->ibuf;/* point at TCP buffer */
  }
  LOG(12,"tcp_data: exiting successfully\r\n",0,0,0,0);
  return T;
}

/* TCP/IP send string as record
 * Accepts: TCP/IP stream
 *	    string pointer
 * Returns: T if success else NIL
 */

long tcp_soutr (TCPSTREAM *stream,char *string)
{
  return tcp_sout (stream,string,(unsigned long) strlen (string));
}


/* TCP/IP send string
 * Accepts: TCP/IP stream
 *	    string pointer
 *	    byte count
 * Returns: T if success else NIL
 */

long tcp_sout (TCPSTREAM *stream,char *string,unsigned long size)
{
  int status;
  struct dsc$descriptor_s BufDesc = {size,DSC$K_DTYPE_T,DSC$K_CLASS_S,string};

  LOG(12,"tcp_sout: entering on stream %08x (socket %08x) with string size %d\r\n%s\r\n",stream,stream->tcpso,strlen(string),string);
				/* 2 = Do not add \r\n */
  if ((status = netlib_write(&(stream->tcpso),&BufDesc)) & 0x1)
    {
      LOG(12,"netlib_write: netlib_write successful. Exiting with T\r\n",0,0,0,0);
      return T;
    }
  else
    {
      LOG(12,"netlib_write: netlib_write failed, status=%08x\r\n",status,0,0,0);
      return tcp_abort(stream);
    }
/*  return ((status = netlib_write (&(stream->tcpso),&BufDesc)) & 0x1) ? T :
    tcp_abort (stream);
*/
}

/* TCP/IP close
 * Accepts: TCP/IP stream
 */

void tcp_close (TCPSTREAM *stream)
{
  tcp_abort (stream);		/* nuke the stream */
				/* flush host names */
  fs_give ((void **) &stream->host);
  fs_give ((void **) &stream->localhost);
  fs_give ((void **) &stream);	/* flush the stream */
}


/* TCP/IP abort stream
 * Accepts: TCP/IP stream
 * Returns: NIL always
 */

long tcp_abort (TCPSTREAM *stream)
{
  if (stream->tcpsi >= 0) {	/* no-op if no socket */
				/* nuke the socket */
    netlib_close (&(stream->tcpsi));
    stream->tcpsi = stream->tcpso = -1;
  }
  return NIL;
}


/* TCP/IP get host name
 * Accepts: TCP/IP stream
 * Returns: host name for this stream
 */

char *tcp_host (TCPSTREAM *stream)
{
  return stream->host;		/* return host name */
}


/* TCP/IP get local host name
 * Accepts: TCP/IP stream
 * Returns: local host name
 */

char *tcp_localhost (TCPSTREAM *stream)
{
  return stream->localhost;	/* return local host name */
}

/* TCP/IP get server host name
 * Accepts: pointer to destination
 * Returns: string pointer if got results, else NIL
 */

char *tcp_clienthost (char *dst)
{
  char buffer[256];
  struct SINDEF sockaddr;
  struct INADDRDEF ipaddr;
  struct dsc$descriptor_s hostname = {sizeof(buffer), DSC$K_DTYPE_T, DSC$K_CLASS_S, buffer};
  unsigned int sockaddr_l, sockaddr_s = sizeof(sockaddr), ipaddr_s = sizeof(ipaddr);
  unsigned short int retlen_w;
  int tempsock, status;

  strcpy(dst,"UNKNOWN");

  if ((clientstream == NULL) || (clientstream->tcpsi == -1))
    {
      LOG(12,"tcp_clienthost: null client stream or aborted socket (stream %0x08x)\r\n",clientstream,0,0,0);
      return dst;		/* Not connected as a server yet (or an aborted connection)*/
    }


/* Get the socket information */
  if (!((status = netlib_getpeername(&clientstream->tcpsi, &sockaddr, &sockaddr_s, &sockaddr_l)) & 0x1))
    {
       LOG(12, "tcp_clienthost: netlib_getpeername failed, status = 0x%08x\r\n", status, 0,0,0);
       return dst;
    }


 /* Allocate a temporary socket and do a name lookup on the address */
  if ( ((status = netlib_socket(&tempsock))&0x1) &&
       ((status = netlib_address_to_name(&tempsock, 0, &sockaddr.sin_x_addr, &ipaddr_s, &hostname, &retlen_w))&0x1) &&
       ((status = netlib_close(&tempsock))&0x1)   )
    {
      buffer[retlen_w] = '\0';
      LOG(12,"tcp_clienthost: returned '%s'\r\n", buffer,0,0,0);
      strcpy(dst,buffer);
      return dst;
    }
  LOG(12,"tcp_clienthost: netlib_address_name failed, status =0x%08x\r\n", status,0,0,0)


/* Name lookup failed, try converting the address */
  if ( !((status=netlib_addrtostr(&sockaddr.sin_x_addr, &hostname, &retlen_w)) & 0x1))
    {
      LOG(12,"tcp_clienthost: netlib_addrstr failed, status = 0x%08d\r\n", status,0,0,0);      
    }
  else
   {
      buffer[retlen_w] = '\0';
      LOG(12,"tcp_clienthost: netlib_addrstr returned '%s'\r\n", buffer,0,0,0);
      strcpy(dst,"["); strcat(dst,buffer); strcat(dst, "]");
    }

  return dst;
}


/* Return my local host name
 * Returns: my local host name
 */

char *mylocalhost ()
{
  int status;
  char tmp[MAILTMPLEN];
  if (!myLocalHost) {		/* have local host yet? */
				/* receives local host name */
    struct dsc$descriptor LocalhostDesc = {0,DSC$K_DTYPE_T,DSC$K_CLASS_D,NULL};
    if (!((status = net_get_hostname (&LocalhostDesc)) & 0x1)) {
      sprintf (tmp,"Can't get local hostname, status=%08x",status);
      mm_log (tmp,ERROR);
      return "UNKNOWN";
    }
    strncpy (tmp,LocalhostDesc.dsc$a_pointer,LocalhostDesc.dsc$w_length);
    tmp[LocalhostDesc.dsc$w_length] = '\0';
    str$free1_dx (&LocalhostDesc);
    myLocalHost = cpystr (tmp);
  }
  return myLocalHost;
}


/*
 | return the local hostname in Buffer whose size is Size. This is a
 | replacement to the Gethostname() which does not exist in Netlib.
 */
get_local_host_name(buffer, size)
char	*buffer;
int	size;
{
  char *s = mylocalhost();

  if (strlen(s) > size-1)
    {strncpy( buffer, s, size-1); buffer[size-1] = '\0';}
  else
    strcpy(buffer, s);
}
