/*
 * Program:	VMS TCP/IP routines for Multinet.
 *
 * Author:	Yehavi Bourvine, The Hebrew University of Jerusalem
 *		Internet: Yehavi@VMS.huji.ac.il
 *
 * Date:	2 August 1994
 * Last Edited:	7 September 1994
 *
 * Copyright 1994 by the University of Washington
 *
 *  Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appears in all copies and that both the
 * above copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the University of Washington not be
 * used in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  This software is made available
 * "as is", and
 * THE UNIVERSITY OF WASHINGTON DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 * WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN
 * NO EVENT SHALL THE UNIVERSITY OF WASHINGTON BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT
 * (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <descrip.h>


/*
   This network module includes logging but requires the main program to
   initialize it. Comment out this code if that is not available.
*/
/*
extern FILE *logfile;
extern int loglevel;
#define LOG(n,fmt,a,b,c,d) {if (logfile && loglevel>=n) fprintf(logfile,fmt,a,b,c,d);}
*/

#  define LOG(n,fmt,a,b,c,d) {}
TCPSTREAM *clientstream = NULL;

				/* TCP timeout handler routine */
static tcptimeout_t tcptimeout = NIL;
				/* TCP timeouts, in seconds */
static long tcptimeout_read = 0;
static long tcptimeout_write = 0;

/* TCP/IP manipulate parameters
 * Accepts: function code
 *	    function-dependent value
 * Returns: function-dependent return value
 */

void *tcp_parameters (long function,void *value)
{
  switch ((int) function) {
  case SET_TIMEOUT:
    tcptimeout = (tcptimeout_t) value;
    break;
  case GET_TIMEOUT:
    value = (void *) tcptimeout;
    break;
  case SET_READTIMEOUT:
    tcptimeout_read = (long) value;
    break;
  case GET_READTIMEOUT:
    value = (void *) tcptimeout_read;
    break;
  case SET_WRITETIMEOUT:
    tcptimeout_write = (long) value;
    break;
  case GET_WRITETIMEOUT:
    value = (void *) tcptimeout_write;
    break;
  default:
    value = NIL;		/* error case */
    break;
  }
  return value;
}

/* TCP/IP server setup
 * Accepts: nothing
 * Returns: TCP/IP stream structure for the stdio channels created by master server
 */
TCPSTREAM *tcp_server_init()
{
  static struct {int size; char *ptr;} descrout = {10,"SYS$OUTPUT"};
  int SYS$ASSIGN ();
  char hostname[MAILTMPLEN] = "UNKNOWN";
  unsigned int sock = 0;
  int status;
  				/* hostname connected to */
  struct dsc$descriptor HostDesc = { 0, DSC$K_DTYPE_T, DSC$K_CLASS_S, NULL };


  LOG(12,"tcp_server_init: entered\r\n",0,0,0,0);  
				/* Find out remote hostname on socket */
				/* .... Left for the moment - don't need it */

				/* Initialize the channels created by master server */
  status = SYS$ASSIGN(&descrout, &sock, 0, 0);
  LOG(12,"tcp_server_init: assign channel (%d),status=%08d\r\n",sock,status,0,0);
  if (!(status & 0x1))
    {
      char tmp[256];
      sprintf (tmp,"Unable to set up server, status=%d",status);
      mm_log (tmp,ERROR);
      return NIL;
    }

  clientstream = (TCPSTREAM *) fs_get (sizeof (TCPSTREAM));
				/* copy official host name */
  clientstream->host = cpystr (hostname);
				/* copy local host name */
  clientstream->localhost = cpystr (mylocalhost ());
				/* init sockets */
  clientstream->tcpsi = clientstream->tcpso = sock;
  clientstream->ictr = 0;		/* init input counter */
  LOG(12,"tcp_server_init: server initialised. stream addr =%08d\r\n",clientstream,0,0,0);
  return clientstream;		/* return success */
}

 
/* TCP/IP open
 * Accepts: host name
 *	    contact service name
 *	    contact port number
 * Returns: TCP/IP stream if success else NIL
 */

TCPSTREAM *tcp_open (char *host,char *service,long port)
{
  TCPSTREAM *stream = NIL;
  int sock;
  char *s;
  struct sockaddr_in sin;
  struct hostent *host_name;
  char hostname[MAILTMPLEN];
  char tmp[MAILTMPLEN];
  struct protoent *pt = getprotobyname ("ip");
  struct servent *sv = service ? getservbyname (service,"tcp") : NIL;
  if (s = strchr (host,':')) {	/* port number specified? */
    *s++ = '\0';		/* yes, tie off port */
    port = strtol (s,&s,10);	/* parse port */
    if (s && *s) {
      sprintf (tmp,"Junk after port number: %.80s",s);
      mm_log (tmp,ERROR);
      return NIL;
    }
    sin.sin_port = htons (port);
  }
				/* copy port number in network format */
  else sin.sin_port = sv ? sv->s_port : htons (port);
  /* The domain literal form is used (rather than simply the dotted decimal
     as with other Unix programs) because it has to be a valid "host name"
     in mailsystem terminology. */
				/* look like domain literal? */
  if (host[0] == '[' && host[(strlen (host))-1] == ']') {
    strcpy (hostname,host+1);	/* yes, copy number part */
    hostname[(strlen (hostname))-1] = '\0';
    if ((sin.sin_addr.s_addr = inet_addr (hostname)) != -1) {
      sin.sin_family = AF_INET;	/* family is always Internet */
      strcpy (hostname,host);	/* hostname is user's argument */
    }
    else {
      sprintf (tmp,"Bad format domain-literal: %.80s",host);
      mm_log (tmp,ERROR);
      return NIL;
    }
  }

  else {			/* lookup host name, note that brain-dead Unix
				   requires lowercase! */
    strcpy (hostname,host);	/* in case host is in write-protected memory */
    if ((host_name = gethostbyname (lcase (hostname)))) {
				/* copy address type */
      sin.sin_family = host_name->h_addrtype;
				/* copy host name */
      strcpy (hostname,host_name->h_name);
				/* copy host addresses */
      memcpy (&sin.sin_addr,host_name->h_addr,host_name->h_length);
    }
    else {
      sprintf (tmp,"No such host as %.80s",host);
      mm_log (tmp,ERROR);
      return NIL;
    }
  }
				/* get a TCP stream */
  if ((sock = socket (sin.sin_family,SOCK_STREAM,pt ? pt->p_proto : 0)) < 0) {
    sprintf (tmp,"Unable to create TCP socket: %s",strerror (errno));
    mm_log (tmp,ERROR);
    return NIL;
  }
				/* open connection */
  if (connect (sock,(struct sockaddr *)&sin,sizeof (sin)) < 0) {
    sprintf (tmp,"Can't connect to %.80s,%d: %s",hostname,port,
	     strerror (errno));
    mm_log (tmp,ERROR);
    return NIL;
  }
				/* create TCP/IP stream */
  stream = (TCPSTREAM *) fs_get (sizeof (TCPSTREAM));
				/* copy official host name */
  stream->host = cpystr (hostname);
				/* get local name */
  gethostname (tmp,MAILTMPLEN-1);
  stream->localhost = cpystr ((host_name = gethostbyname (tmp)) ?
			      host_name->h_name : tmp);
				/* init sockets */
  stream->tcpsi = stream->tcpso = sock;
  stream->ictr = 0;		/* init input counter */
  return stream;		/* return success */
}

/* TCP/IP authenticated open
 * Accepts: host name
 *	    service name
 * Returns: TCP/IP stream if success else NIL
 */

TCPSTREAM *tcp_aopen (char *host,char *service)
{
	return NIL;
}

/* TCP/IP receive line
 * Accepts: TCP/IP stream
 * Returns: text line string or NIL if failure
 */

char *tcp_getline (TCPSTREAM *stream)
{
  int n,m;
  char *st,*ret,*stp;
  char c = '\0';
  char d;
				/* make sure have data */
  LOG(12,"tcp_getline: Entered, stream addr = %08d\r\n", stream,0,0,0);
  if (!tcp_getdata (stream))
    {
      LOG(12,"tcp_getline: tcp_getdata failed, exiting with NIL\r\n",0,0,0,0);
      return NIL;
    }
  LOG(12,"tcp_getline: starting to get line\r\n",0,0,0,0);
  st = stream->iptr;		/* save start of string */
  n = 0;			/* init string count */
  while (stream->ictr--) {	/* look for end of line */
    d = *stream->iptr++;	/* slurp another character */
    if ((c == '\015') && (d == '\012')) {
      LOG(12,"tcp_getline: Found EOL, getting memory buffer size %d\r\n",n,0,0,0);
      ret = (char *) fs_get (n--);
      memcpy (ret,st,n);	/* copy into a free storage string */
      ret[n] = '\0';		/* tie off string with null */
      LOG(12,"tcp_getline: exiting, Returning line (len=%d)\r\n%s\r\n",strlen(ret),ret,0,0);
      return ret;
    }
    n++;			/* count another character searched */
    c = d;			/* remember previous character */
  }
  LOG(12,"tcp_getline: end of buffer - no EOL\r\n",0,0,0,0);
				/* copy partial string from buffer */
  memcpy ((ret = stp = (char *) fs_get (n)),st,n);
				/* get more data from the net */
  if (!tcp_getdata (stream)) fs_give ((void **) &ret);
				/* special case of newline broken by buffer */
  else if ((c == '\015') && (*stream->iptr == '\012')) {
    stream->iptr++;		/* eat the line feed */
    stream->ictr--;
    ret[n - 1] = '\0';		/* tie off string with null */
  }
				/* else recurse to get remainder */
  else if (st = tcp_getline (stream)) {
    ret = (char *) fs_get (n + 1 + (m = strlen (st)));
    memcpy (ret,stp,n);		/* copy first part */
    memcpy (ret + n,st,m);	/* and second part */
    fs_give ((void **) &stp);	/* flush first part */
    fs_give ((void **) &st);	/* flush second part */
    ret[n + m] = '\0';		/* tie off string with null */
  }
  LOG(12,"tcp_getline: exiting, returning line\r\n%s\r\n",ret,0,0,0);
  return ret;
}

/* TCP/IP receive buffer
 * Accepts: TCP/IP stream
 *	    size in bytes
 *	    buffer to read into
 * Returns: T if success, NIL otherwise
 */

long tcp_getbuffer (TCPSTREAM *stream,unsigned long size,char *buffer)
{
  unsigned long n;
  char *bufptr = buffer;
  LOG(12,"tcp_getbuffer: entered, stream addr = %08d, size=%d\r\n",stream,size,0,0);
  while (size > 0) {		/* until request satisfied */
    LOG(12,"tcp_getbuffer: Checking for data in buffer (size=%d)\r\n",size,0,0,0);
    if (!tcp_getdata (stream))
      {
        LOG(12,"tcp_getbuffer: tcp_getdata failed, exiting with NIL\r\n",0,0,0,0);
        return NIL;
      }
    n = min (size,stream->ictr);/* number of bytes to transfer */
				/* do the copy */
    memcpy (bufptr,stream->iptr,n);
    bufptr += n;		/* update pointer */
    stream->iptr +=n;
    size -= n;			/* update # of bytes to do */
    stream->ictr -=n;
  }
  bufptr[0] = '\0';		/* tie off string */
  LOG(12,"tcp_getbuffer: exiting with buffer (T)\r\n",0,0,0,0);
  return T;
}

/* TCP/IP receive data
 * Accepts: TCP/IP stream
 * Returns: T if success, NIL otherwise
 */

long tcp_getdata (TCPSTREAM *stream)
{
  int i;
  fd_set fds,efds;
  struct timeval tmo;
  time_t t = time (0);
  LOG(12,"tcp_getdata: Entered on stream %08d\r\n",stream,0,0,0);
  tmo.tv_sec = tcptimeout_read;
  tmo.tv_usec = 0;
  FD_ZERO (&fds);		/* initialize selection vector */
  FD_ZERO (&efds);		/* handle errors too */
  LOG(12,"tcp_getdata: Checking stream channel %08d\r\n",stream->tcpsi,0,0,0);
  if (stream->tcpsi < 0)
    {
      LOG(12,"tcp_getdata: stream check failed, returning NIL\r\n",0,0,0,0);
      return NIL;
    }
  while (stream->ictr < 1) {	/* if nothing in the buffer */
    FD_SET (stream->tcpsi,&fds);/* set bit in selection vector */
    FD_SET(stream->tcpsi,&efds);/* set bit in error selection vector */
    errno = NIL;		/* block and read */
    LOG(12,"tcp_getdata: selecting\r\n",0,0,0,0);
    while (((i = select (getdtablesize (),&fds,0,&efds,tmo.tv_sec ? &tmo:0))<0)
	   && (errno == EINTR));
    if (!i) {			/* timeout? */
      if (tcptimeout && ((*tcptimeout) (time (0) - t))) continue;
      else
        {
          LOG(12,"tcp_getdata: read timedout, aborting stream\r\n",0,0,0,0);
          return tcp_abort (stream);
        }
    }
    if (i < 0) return tcp_abort (stream);
    LOG(12,"tcp_getdata: reading data\r\n",0,0,0,0);
    while (((i = socket_read (stream->tcpsi,stream->ibuf,BUFLEN)) < 0) &&
	   (errno == EINTR));
    if (i < 1)
      {
        LOG(12,"tcp_getdata: error during socket_read (status = %d)\r\n",i,0,0,0);
        return tcp_abort (stream);
      }
    stream->iptr = stream->ibuf;/* point at TCP buffer */
    stream->ictr = i;		/* set new byte count */
    LOG(12,"tcp_getdata: exiting (T)\r\n",0,0,0,0);
  }
  return T;
}

/* TCP/IP send string as record
 * Accepts: TCP/IP stream
 *	    string pointer
 * Returns: T if success else NIL
 */

long tcp_soutr (TCPSTREAM *stream,char *string)
{
  return tcp_sout (stream,string,(unsigned long) strlen (string));
}


/* TCP/IP send string
 * Accepts: TCP/IP stream
 *	    string pointer
 *	    byte count
 * Returns: T if success else NIL
 */

long tcp_sout (TCPSTREAM *stream,char *string,unsigned long size)
{
  int i;
  fd_set fds;
  struct timeval tmo;
  time_t t = time (0);
  LOG(12,"tcp_sout: Entered on stream %08d\r\n",stream,0,0,0);
  tmo.tv_sec = tcptimeout_write;
  tmo.tv_usec = 0;
  FD_ZERO (&fds);		/* initialize selection vector */
  LOG(12,"tcp_sout: Stream check on socket %d\r\n",stream->tcpso,0,0,0);
  if (stream->tcpso < 0) return NIL;
  while (size > 0) {		/* until request satisfied */
    FD_SET (stream->tcpso,&fds);/* set bit in selection vector */
    errno = NIL;		/* block and write */
    while (((i = select (getdtablesize (),0,&fds,0,tmo.tv_sec ? &tmo : 0)) < 0)
	   && (errno == EINTR));
    if (!i) {			/* timeout? */
      if (tcptimeout && ((*tcptimeout) (time (0) - t))) continue;
      else
        {
          LOG(12,"tcp_sout: select failed (timeout?) (i=%d)\r\n",i,0,0,0);
          return tcp_abort (stream);
        }
    }
    if (i < 0)
      {
        LOG(12,"tcp_sout: select failed (timeout?) (i=%d)\r\n",i,0,0,0);
        return tcp_abort (stream);
      }
    while (((i = socket_write (stream->tcpso,string,size)) < 0) &&
	   (errno == EINTR));
    if (i < 0)
      {
        LOG(12,"tcp_sout: socket_write failed (i=%d)\r\n",i,0,0,0);
        return tcp_abort (stream);
      }
    size -= i;			/* how much we sent */
    string += i;
  }
  LOG(12,"tcp_sout: exiting OK\r\n",0,0,0,0);
  return T;			/* all done */
}

/* TCP/IP close
 * Accepts: TCP/IP stream
 */

void tcp_close (TCPSTREAM *stream)
{
  tcp_abort (stream);		/* nuke the stream */
				/* flush host names */
  fs_give ((void **) &stream->host);
  fs_give ((void **) &stream->localhost);
  fs_give ((void **) &stream);	/* flush the stream */
}


/* TCP/IP abort stream
 * Accepts: TCP/IP stream
 * Returns: NIL always
 */

long tcp_abort (TCPSTREAM *stream)
{
  int i;
  if (stream->tcpsi >= 0) {	/* no-op if no socket */
				/* nuke the socket */
    socket_close (stream->tcpsi);
    if (stream->tcpsi != stream->tcpso) socket_close (stream->tcpso);
    stream->tcpsi = stream->tcpso = -1;
  }
  return NIL;
}


/* TCP/IP get host name
 * Accepts: TCP/IP stream
 * Returns: host name for this stream
 */

char *tcp_host (TCPSTREAM *stream)
{
  return stream->host;		/* return host name */
}


/* TCP/IP get local host name
 * Accepts: TCP/IP stream
 * Returns: local host name
 */

char *tcp_localhost (TCPSTREAM *stream)
{
  return stream->localhost;	/* return local host name */
}

/* TCP/IP get server host name
 * Accepts: pointer to destination
 * Returns: string pointer if got results, else NIL
 */

char *tcp_clienthost (char *dst)
{
  struct hostent *hn;
  struct sockaddr_in from;
  int fromlen = sizeof (from);
  if ((clientstream == NULL) || (clientstream->tcpsi == -1))
     {
      strcpy(dst,"UNKNOWN");
      return dst;
     }
  if (getpeername (clientstream->tcpsi,(struct sockaddr *) &from,&fromlen)) return "UNKNOWN";
  strcpy (dst,(hn = gethostbyaddr ((char *) &from.sin_addr,
				   sizeof (struct in_addr),from.sin_family)) ?
	  hn->h_name : inet_ntoa (from.sin_addr));
  return dst;
}


/* Return my local host name
 * Returns: my local host name
 */

char *mylocalhost ()
{
  char tmp[MAILTMPLEN];
  struct hostent *hn;
  if (!myLocalHost) {		/* have local host yet? */
    gethostname(tmp,MAILTMPLEN);/* get local host name */
    myLocalHost = cpystr ((hn = gethostbyname (tmp)) ? hn->h_name : tmp);
  }
  return myLocalHost;
}

/* Return my local host name
 * Returns: my local host name
 */

get_local_host_name (buffer,size)
char *buffer;
int size;
{
  char *s = mylocalhost();

  if (strlen(s) > size-1)
    {strncpy(buffer, s, size-1); buffer[size-1] = '\0';}
  else
    strcpy(buffer, s);
}
