#ifndef lint
static char sccsid[] = "@(#)gif2ras.c 1.1 89/01/17";
#endif
/*-
 * gifdecode.c is strongly based on gif2ras.c ( W. B. )
 *
 * gif2ras.c - Converts from a Compuserve GIF (tm) image to a Sun Raster image.
 *
 * Copyright (c) 1988, 1989 by Patrick J. Naughton
 *
 * Author: Patrick J. Naughton
 * naughton@wind.sun.com
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The author
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 * event will the author be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 * Comments and additions should be sent to the author:
 *
 *                     Patrick J. Naughton
 *                     Sun Microsystems
 *                     2550 Garcia Ave, MS 14-40
 *                     Mountain View, CA 94043
 *                     (415) 336-1080
 *
 * Revision History:
 * 01-Jan-89: Added error checking and removed NEXTSHORT.
 * 07-Sep-88: Added BytesPerScanline fix.
 * 30-Aug-88: Allow stdin/stdout. Restructured argument parser.
 * 27-Jul-88: Updated to use libpixrect to fix 386i byteswapping problems.
 * 11-Apr-88: Converted to C and changed to write Sun rasterfiles.
 * 19-Jan-88: GIFSLOW.PAS posted to comp.graphics by Jim Briebel,
 *            a Turbo Pascal 4.0 program to painfully slowly display
 *            GIF images on an EGA equipped IBM-PC.
 *
 * Description:
 *   This program takes a Compuserve "Graphics Interchange Format" or "GIF"
 * file as input and writes a file known as a Sun rasterfile.  This datafile
 * can be loaded by the NeWS "readcanvas" operator and is of the same format
 * as the files in /usr/NeWS/smi/*.  Under X11R2 there is a program called
 * xraster to display these files.
 *
 * Portability:
 *   To make this program convert to some image format other than Sun's
 * Rasterfile format simply seach for the tag "SS:" in the source and
 * replace these simple mechanisms with the appropriate ones for the
 * other output format.  I have marked all (six) Sun Specific pieces
 * of code with this comment.
 *
 * SS: compile with "cc -o gif2ras -O gif2ras.c -lpixrect"
 */

#include <stdio.h>
#include <stdlib.h>

typedef int boolean;
#define True (1)
#define False (0)

#define NEXTBYTE (*ptr++)
#define IMAGESEP 0x2c
#define INTERLACEMASK 0x40
#define COLORMAPMASK 0x80

FILE *fp;

int BitOffset = 0,		/* Bit Offset of next code */
    XC = 0, YC = 0,		/* Output X and Y coords of current pixel */
    Pass = 0,			/* Used by output routine if interlaced pic */
    OutCount = 0,		/* Decompressor output 'stack count' */
    RWidth, RHeight,		/* screen dimensions */
    Width, Height,		/* image dimensions */
    LeftOfs, TopOfs,		/* image offset */
    BitsPerPixel,		/* Bits per pixel, read from GIF header */
    BytesPerScanline,		/* bytes per scanline in output raster */
    ColorMapSize,		/* number of colors */
    CodeSize,			/* Code size, read from GIF header */
    InitCodeSize,		/* Starting code size, used during Clear */
    Code,			/* Value returned by ReadCode */
    MaxCode,			/* limiting value for current code size */
    ClearCode,			/* GIF clear code */
    EOFCode,			/* GIF end-of-information code */
    CurCode, OldCode, InCode,	/* Decompressor variables */
    FirstFree,			/* First free code, generated per GIF spec */
    FreeCode,			/* Decompressor, next free slot in hash table */
    FinChar,			/* Decompressor variable */
    BitMask,			/* AND mask for data size */
    ReadMask;			/* Code AND mask for current code size */

boolean Interlace, HasColormap;
boolean Verbose = False;
static int GColIndex[256];		/* vector for color reordering */
unsigned char *GIFred, *GIFgreen, *GIFblue; 

typedef (*Gfunc)();
Gfunc putpixel;

unsigned char *RawGIF;			/* The heap array to hold it, raw */
unsigned char *Raster;			/* The raster data stream, unblocked */

    /* The hash table used by the decompressor */
int Prefix[4096];
int Suffix[4096];

    /* An output array used by the decompressor */
int OutCode[1025];

   /* free allocated memory after an error */

GIFfreeAll()
  {
   if(RawGIF!= NULL)
     free(RawGIF);
   if(Raster != NULL)
     free(Raster);
   RawGIF = NULL;
   Raster = NULL;
   if(fp != NULL)
     fclose(fp);
  }

/* sort the colors with qsort */

int GIFsortColor(ind1, ind2)
  int *ind1, *ind2;
  {
   int r, g, b, c, r1, g1, b1, r2, g2, b2, ret, ret1, ret2;
   int i1, i2, *index1, *index2;
   
   i1 = *ind1;
   i2 = *ind2;
   r1 = (int)GIFred[i1];
   g1 = (int)GIFgreen[i1];
   b1 = (int)GIFblue[i1];
   ret1 = r1*r1+g1*g1+b1*b1;
   r2 = (int)GIFred[i2];
   g2 = (int)GIFgreen[i2];
   b2 = (int)GIFblue[i2];
   ret2 = r2*r2+g2*g2+b2*b2;
   ret = ret1 -ret2;
   return(ret);
  }

/*
 * allow reordering of colormap, if you don't want this
 * to happen remove the qsort line
 */


GIFinitColorDeco(red, green, blue)
unsigned char *red, *green, *blue;
  {
   int i, *ind, i1, dum[256];
   unsigned char dr[256], dg[256], db[256];
   
   ind = GColIndex;
   for(i= 0; i<ColorMapSize; i++)
      *ind++ = i;
   GIFred = red;
   GIFgreen = green;
   GIFblue = blue;
   qsort(GColIndex, ColorMapSize, sizeof(int), (int(*)())GIFsortColor);
   for(i=0; i<ColorMapSize; i++)
      {
       i1 = GColIndex[i];
       dr[i] = red[i1];
       dg[i] = green[i1];
       db[i] = blue[i1];
       dum[GColIndex[i]] = i; 
      }
   for(i=0; i<ColorMapSize; i++)
     {
      red[i] = dr[i];
      green[i] = dg[i];
      blue[i] = db[i];
      GColIndex[i] = dum[i];
     }
  } 

GIFdecode(FName, Red, Green, Blue, GCMapSize, GWidth, GHeight,
	GInitInput, GPutPixel, Gerror)
char *FName;
unsigned char *Red, *Green, *Blue;
int *GCMapSize, *GWidth, *GHeight;
Gfunc GInitInput, GPutPixel, Gerror;
{
char *inf = NULL;
char *outf = NULL;
char *id = "GIF87a";
int filesize;
register unsigned char ch, ch1;
register unsigned char *ptr, *ptr1;
register int i;

    setbuf(stderr, NULL);

    inf = FName;

/* init static var's */

    BitOffset = 0;
    XC = 0;
    YC = 0;
    Pass = 0;
    OutCount = 0;
    RawGIF = NULL;
    Raster = NULL;
    putpixel = GPutPixel;
    if (!(fp = fopen(inf, "r")))
	(*Gerror)("Could not open %s for read", inf);

    /* find the size of the file */
    fseek(fp, 0L, 2);
    filesize = ftell(fp);
    fseek(fp, 0L, 0);

    if (!(ptr = RawGIF = (unsigned char *) malloc(filesize)))
	(*Gerror)("not enough memory to read gif file.\n", NULL);

    if (!(Raster = (unsigned char *) malloc(filesize)))
	(*Gerror)("not enough memory to read gif file.\n", NULL);

    if (fread(ptr, filesize, 1, fp) != 1)
	(*Gerror)("GIF data read failed\n", NULL);

    if (strncmp(ptr, id, 6))
	(*Gerror)("%s is not a GIF file.\n", inf);
    ptr += 6;

/* Get variables from the GIF screen descriptor */

    ch = NEXTBYTE;
    RWidth = ch + 0x100 * NEXTBYTE;	/* screen dimensions... not used. */
    ch = NEXTBYTE;
    RHeight = ch + 0x100 * NEXTBYTE;

    ch = NEXTBYTE;
    HasColormap = ((ch & COLORMAPMASK) ? True : False);

    BitsPerPixel = (ch & 7) + 1;
    ColorMapSize = 1 << BitsPerPixel;
    *GCMapSize = ColorMapSize;
    BitMask = ColorMapSize - 1;

    ch = NEXTBYTE;		/* background color... not used. */

    if (NEXTBYTE)		/* supposed to be NULL */
	(*Gerror)("%s is a corrupt GIF file (nonull).\n", inf);

/* Read in global colormap. */

    if (HasColormap) {
	for (i = 0; i < ColorMapSize; i++) {
	    Red[i] = NEXTBYTE;
	    Green[i] = NEXTBYTE;
	    Blue[i] = NEXTBYTE;
	}

    }
    else (*Gerror)("%s does not have a colormap.\n", inf);
    
/* allow reordering of colors */

    GIFinitColorDeco(Red, Green, Blue);


/* Check for image seperator */

    if (NEXTBYTE != IMAGESEP)
	(*Gerror)("%s is a corrupt GIF file (nosep).\n", inf);

/* Now read in values from the image descriptor */

    ch = NEXTBYTE;
    LeftOfs = ch + 0x100 * NEXTBYTE;
    ch = NEXTBYTE;
    TopOfs = ch + 0x100 * NEXTBYTE;
    ch = NEXTBYTE;
    Width = ch + 0x100 * NEXTBYTE;
    ch = NEXTBYTE;
    Height = ch + 0x100 * NEXTBYTE;
    Interlace = ((NEXTBYTE & INTERLACEMASK) ? True : False);
    *GWidth = Width;
    *GHeight = Height;
    if((*GInitInput)(Width, Height) == False)
      (*Gerror)("Couldn't allocate enough memory for GIF import", NULL);

/* Note that I ignore the possible existence of a local color map.
 * I'm told there aren't many files around that use them, and the spec
 * says it's defined for future use.  This could lead to an error
 * reading some files. 
 */

/* Start reading the raster data. First we get the intial code size
 * and compute decompressor constant values, based on this code size.
 */

    CodeSize = NEXTBYTE;
    ClearCode = (1 << CodeSize);
    EOFCode = ClearCode + 1;
    FreeCode = FirstFree = ClearCode + 2;

/* The GIF spec has it that the code size is the code size used to
 * compute the above values is the code size given in the file, but the
 * code size used in compression/decompression is the code size given in
 * the file plus one. (thus the ++).
 */

    CodeSize++;
    InitCodeSize = CodeSize;
    MaxCode = (1 << CodeSize);
    ReadMask = MaxCode - 1;

/* Read the raster data.  Here we just transpose it from the GIF array
 * to the Raster array, turning it from a series of blocks into one long
 * data stream, which makes life much easier for ReadCode().
 */

    ptr1 = Raster;
    do {
	ch = ch1 = NEXTBYTE;
	while (ch--) *ptr1++ = NEXTBYTE;
	if ((Raster - ptr1) > filesize)
	    (*Gerror)("%s is a corrupt GIF file (unblock).\n", inf);
    } while(ch1);

    free(RawGIF);		/* We're done with the raw data now... */
    RawGIF = NULL;

/* Decompress the file, continuing until you see the GIF EOF code.
 * One obvious enhancement is to add checking for corrupt files here.
 */

    Code = ReadCode();
    while (Code != EOFCode) {

/* Clear code sets everything back to its initial value, then reads the
 * immediately subsequent code as uncompressed data.
 */

	if (Code == ClearCode) {
	    CodeSize = InitCodeSize;
	    MaxCode = (1 << CodeSize);
	    ReadMask = MaxCode - 1;
	    FreeCode = FirstFree;
	    CurCode = OldCode = Code = ReadCode();
	    FinChar = CurCode & BitMask;
	    AddToPixel(FinChar);
	}
	else {

/* If not a clear code, then must be data: save same as CurCode and InCode */

	    CurCode = InCode = Code;

/* If greater or equal to FreeCode, not in the hash table yet;
 * repeat the last character decoded
 */

	    if (CurCode >= FreeCode) {
		CurCode = OldCode;
		OutCode[OutCount++] = FinChar;
	    }

/* Unless this code is raw data, pursue the chain pointed to by CurCode
 * through the hash table to its end; each code in the chain puts its
 * associated output code on the output queue.
 */

	    while (CurCode > BitMask) {
		if (OutCount > 1024)
		    (*Gerror)("%s is a corrupt GIF file (OutCount).\n", inf);
		OutCode[OutCount++] = Suffix[CurCode];
		CurCode = Prefix[CurCode];
	    }

/* The last code in the chain is treated as raw data. */

	    FinChar = CurCode & BitMask;
	    OutCode[OutCount++] = FinChar;

/* Now we put the data out to the Output routine.
 * It's been stacked LIFO, so deal with it that way...
 */

	    for (i = OutCount - 1; i >= 0; i--)
		AddToPixel(OutCode[i]);
	    OutCount = 0;

/* Build the hash table on-the-fly. No table is stored in the file. */

	    Prefix[FreeCode] = OldCode;
	    Suffix[FreeCode] = FinChar;
	    OldCode = InCode;

/* Point to the next slot in the table.  If we exceed the current
 * MaxCode value, increment the code size unless it's already 12.  If it
 * is, do nothing: the next code decompressed better be CLEAR
 */

	    FreeCode++;
	    if (FreeCode >= MaxCode) {
		if (CodeSize < 12) {
		    CodeSize++;
		    MaxCode *= 2;
		    ReadMask = (1 << CodeSize) - 1;
		}
	    }
	}
	Code = ReadCode();
    }

    free(Raster);
    Raster = NULL;

    if (fp != stdin)
	fclose(fp);

}


/* Fetch the next code from the raster data stream.  The codes can be
 * any length from 3 to 12 bits, packed into 8-bit bytes, so we have to
 * maintain our location in the Raster array as a BIT Offset.  We compute
 * the byte Offset into the raster array by dividing this by 8, pick up
 * three bytes, compute the bit Offset into our 24-bit chunk, shift to
 * bring the desired code to the bottom, then mask it off and return it. 
 */
ReadCode()
{
int RawCode, ByteOffset;

    ByteOffset = BitOffset / 8;
    RawCode = Raster[ByteOffset] + (0x100 * Raster[ByteOffset + 1]);
    if (CodeSize >= 8)
	RawCode += (0x10000 * Raster[ByteOffset + 2]);
    RawCode >>= (BitOffset % 8);
    BitOffset += CodeSize;
    return(RawCode & ReadMask);
}


AddToPixel(Index)
unsigned char Index;
{
    (*putpixel)(XC,YC,(int)GColIndex[Index]);

/* Update the X-coordinate, and if it overflows, update the Y-coordinate */

    if (++XC == Width) {

/* If a non-interlaced picture, just increment YC to the next scan line. 
 * If it's interlaced, deal with the interlace as described in the GIF
 * spec.  Put the decoded scan line out to the screen if we haven't gone
 * past the bottom of it
 */

	XC = 0;
	if (!Interlace) YC++;
	else {
	    switch (Pass) {
		case 0:
		    YC += 8;
		    if (YC >= Height) {
			Pass++;
			YC = 4;
		    }
		break;
		case 1:
		    YC += 8;
		    if (YC >= Height) {
			Pass++;
			YC = 2;
		    }
		break;
		case 2:
		    YC += 4;
		    if (YC >= Height) {
			Pass++;
			YC = 1;
		    }
		break;
		case 3:
		    YC += 2;
		break;
		default:
		break;
	    }
	}
    }
}
