/****************************************************************/
/*								*/
/* xdxtk.c:	stuff that depends on the			*/
/*		X Toolkit					*/
/*								*/
/*		Design: Walter Benzing 1994			*/
/*		NO RIGHTS RESERVED				*/
/*								*/
/****************************************************************/

/* standard includes */
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>

/*includes for the X11 toolkit*/
#include <X11/Intrinsic.h>  /* Intrinsics Definitions */
#include <X11/StringDefs.h>
#include <X11/Shell.h>      /* Top Level Shell Widget */

/*my includes */
#include "xdim.h"

/*
 * Simple popup callback function
 */

void XT_Popitup(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XtPopup((Widget) client_data, XtGrabNone);
   }

void XT_ManageIt(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XtManageChild((Widget) client_data);
   }

void XT_PopnGrab(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XtPopup((Widget) client_data, XtGrabExclusive);
   }

void XT_Popdown(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XtPopdown((Widget) client_data);
   }

void XT_UnmanageIt(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XtUnmanageChild((Widget) client_data);
   }

/*
 * Redraw color window
 */

void XT_RedrawColors(Widget w, XtPointer client_data, XtPointer call_data)
   {
    Dimension width, height;
    Window win;
    XDimInfo *XDim;
    XDColorInfo *XDC;
    int i,j,x0,y0,x1,y1;
    double delx,dely;
    Pixel *myColor;

    win =XtWindow(w);
    if(win)
      {
       XDim = (XDimInfo*) client_data;
       XDC = &(XDim->colors);
       XtVaGetValues(w,
    		   XtNwidth, &width,
    		   XtNheight, &height,
    		   NULL);
       dely = ((double)height)/(XDC->rowNum);
       y0 = 0;
       myColor= XDC->colTrans;
       for(i=0; i<XDC->rowNum; i++)
	  {
           delx = ((double)width)/((XDC->colors)[i]);
           x0 = 0;
           y1 = (i+1)*dely;
           for(j=1; j<=(XDC->colors)[i]; j++)
	      {
	       x1 = j*delx;
	       HW_FillBox(x0,y0,x1,y1,*myColor,win,XDim);
	       myColor++;
	       x0 = x1;
	      } 
	   y0= y1;
	  }
      }
   }

/*
 * Redraw color row
 */

void XT_RedrawColorRow(Widget w, XtPointer client_data, XtPointer call_data)
   {
    Dimension width, height;
    Window win;
    XDimInfo *XDim;
    XDColorInfo *XDC;
    int i,j,x0,y0,x1,y1;
    double delx;
    Pixel *myColor;

    win =XtWindow(w);
    if(win)
      {
       XDim = (XDimInfo*) client_data;
       XDC = &(XDim->colors);
       XtVaGetValues(w,
    		   XtNwidth, &width,
    		   XtNheight, &height,
    		   NULL);
       y0 = 0;
       x0 = 0;
       y1 = y0 + height;
       if(XDC->single)
	  {
           x1 = x0+width;
	   HW_FillBox(x0,y0,x1,y1,(XDC->colTrans)[XDC->selected],win,XDim);
	  }
       else
	  {
	   j =0;
	   for(i=0; i<XDC->selected;i++)
	     j += (XDC->colors)[i];
           myColor = &((XDC->colTrans)[j]);
           delx = ((double)width)/((XDC->colors)[XDC->selected]);
           for(j=1; j<=(XDC->colors)[i]; j++)
	      {
	       x1 = j*delx;
	       HW_FillBox(x0,y0,x1,y1,*myColor,win,XDim);
	       myColor++;
	       x0 = x1;
	      } 
	  }
      }
   }

/*
 * Redraw deco window
 */

void XT_RedrawDeco(Widget w, int dec1, int dec2, double per,
	myPixel *deco, XDimInfo *XDim)
   {
    Dimension width, height;
    Window win;
    int i,x0,y0,x1,y1, delta, Sum;
    myPixel *dec;
    Pixel myColor, *trans;

    win =XtWindow(w);
    if(win)
      {
       XtVaGetValues(w,
    		XtNwidth, &width,
    		XtNheight, &height,
		NULL);
       y0 = (1.0-per)*height;
       y1 = height;
       Sum = 0;
       if((delta = dec2-dec1) <=0)
          delta =1;
       x0 = 0;
       x1 = 0;      
       trans = (XDim->colors).colTrans;
       dec = &deco[dec1];
       dec++;
       myColor = trans[*dec];
       for(i=dec1+1; i< dec2; i++)
	  {
           if(trans[*dec] != myColor)
             {
	      HW_FillBox(x0,y0,x1,y1,myColor,win,XDim);
              x0 = x1;
	      myColor = trans[*dec];                     
             }
           dec++;
           Sum += width;
           while(Sum >= delta)
	     {
              x1++;
              Sum -= delta;
             }
	  }
       HW_FillBox(x0,y0,x1,y1,myColor,win,XDim);
      }
   }

/*
 * Redraw full vector
 */

void XT_RedrawDecoVector(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    ViewportInfo *XDV;

    XDim = (XDimInfo*) client_data;
    if(XDim->port >= 0)
      {
       XDV = &((XDim->Views)[XDim->port]);
       XT_RedrawDeco(w, 0, MAX_DECO_NUMBER, 1.0, XDV->deco, XDim);
      }
   }


/*
 * Redraw zoomed vector
 */

void XT_RedrawZoomedDeco(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    ViewportInfo *XDV;

    XDim = (XDimInfo*) client_data;
    if(XDim->port >= 0)
      {
       XDV = &((XDim->Views)[XDim->port]);
       XT_RedrawDeco(w, XDV->sZoom, XDV->eZoom, 1.0, XDV->deco, XDim);
      }
   }

/*
 * make all windows know, that the palette was changed
 */

void XT_NotifyPaletteChanged(XDimInfo *XDim)
   {
    XDimColorControl *XDC;
    XDimColorSelect *XDS;
    XDColorInfo *XCI;
    XDimDeco *XDD;
    ViewportInfo *views;
    XDimViewport *vwidgets;
    int i;

    XDC = &((XDim->myWidgets).ColorPanel);
    XDS = &((XDim->myWidgets).ColorSelect);
    XCI = &(XDim->colors);
    XDD = &((XDim->myWidgets).DecoEdit);
    GR_AllocateColors(XDim);
    XT_RedrawColors(XDC->ColorArea, XDim, NULL);
    XT_RedrawColors(XDS->ColorArea, XDim, NULL);
    XCI->selected = 0;
    XT_RedrawColorRow(XDS->ColorSArea, XDim, NULL);
    GR_ResetAllDecoding(XDim);
    XT_RedrawDecoVector(XDD->DArea, XDim, NULL);
    XT_RedrawZoomedDeco(XDD->DZoom, XDim, NULL);
    if(XCI->fore >= XCI->wantedColors)
      XCI->fore = XCI->wantedColors -1;
    WD_ReInitDeco(XDim);
    views = XDim->Views;
    vwidgets = (XDim->myWidgets).Views;
    for(i=0; i<MAX_VIEWPORT_NUMBER; i++)
       {
        if((views[i]).pict)
          {
	   (views[i]).back = XCI->back;
	   (views[i]).fore = XCI->fore;
	   (views[i]).axisFore = XCI->axisFore;
	   (views[i]).axisBackX = XCI->axisBackX;
	   (views[i]).axisBackY = XCI->axisBackY;
	   (views[i]).axisBackZ = XCI->axisBackZ;
	   WD_SetBackground((vwidgets[i]).DArea, XCI);
           XT_Redraw((vwidgets[i]).DArea,(XtPointer)&(views[i]), NULL);
	  }
       }
   }

/*
 * Delete a row of colors
 */

void XT_DeleteColorRow(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    XDColorInfo *XDC;
    XDimColorSelect *XDS;
    XDHardwareInfo *HWI;
    int i, num, myColor;

    XDim = (XDimInfo*) client_data;
    XDC = &(XDim->colors);
    XDS = &((XDim->myWidgets).ColorSelect);
    HWI = &(XDim->hwInfo);
    if(XDC->rowNum >1)
      {
       myColor =0;
       for(i=0; i<XDC->selected;i++)
	 myColor += (XDC->colors)[i];
       num = (XDC->colors)[XDC->selected];
       for(i=myColor; i< myColor+num;i++)
	  {
           if((HWI->colStat)[i]) 
	      (HWI->colStat)[i] = HW_COLORCLAIMED;
	  }
       HW_FreeUnusedColors(XDim);
       XDC->wantedColors -= num;
       if(XDC->colS >= XDC->wantedColors)
         XDC->colS = 0;
       if(XDC->colE >= XDC->wantedColors)
         XDC->colE = XDC->wantedColors -1;
       if(XDC->back >= XDC->wantedColors)
         XDC->back = XDC->wantedColors -1;
       if(XDC->fore >= XDC->wantedColors)
         XDC->fore = 0;
       if(XDC->axisFore >= XDC->wantedColors)
         XDC->axisFore = XDC->fore;
       if(XDC->axisBackX >= XDC->wantedColors)
         XDC->axisBackX = (XDC->wantedColors*3)/4;
       if(XDC->axisBackY >= XDC->wantedColors)
         XDC->axisBackY = XDC->wantedColors/2;
       if(XDC->axisBackZ >= XDC->wantedColors)
         XDC->axisBackZ = XDC->wantedColors/4;
       for(i= myColor; i<XDC->wantedColors;i++)
          {
	   (XDC->colTrans)[i]= (XDC->colTrans)[i+num];
	   (HWI->colStat)[i]= (HWI->colStat)[i+num];
	  }
       for(i=XDC->wantedColors; i<XDC->wantedColors+num; i++)
          HWI->colStat[i] = HW_COLORFREE;
       for(i= XDC->selected; i<XDC->rowNum-1; i++)
	  {
	   (XDC->colors)[i] = (XDC->colors)[i+1];
	   (XDC->beginR)[i] = (XDC->beginR)[i+1];
	   (XDC->beginG)[i] = (XDC->beginG)[i+1];
	   (XDC->beginB)[i] = (XDC->beginB)[i+1];
	   (XDC->endR)[i] = (XDC->endR)[i+1];
	   (XDC->endG)[i] = (XDC->endG)[i+1];
	   (XDC->endB)[i] = (XDC->endB)[i+1];
	  }
       (XDC->rowNum)--;
      }
    XtPopdown(XDS->ColorShell);
    XT_NotifyPaletteChanged(XDim);
   }


/*
 * Find the viewport to the widget
 */

void XT_SetViewport(Widget w, XtPointer client_data, XEvent *event, Boolean *cdr)
   {
    extern XDimInfo XDim;
    int num;
    ViewportInfo *view1;
    XDimWidgets *XDW;

    if(event->type == FocusIn && XDim.RecSave != True)
      {
       num= *((int*)client_data);
       XDW = &(XDim.myWidgets);
       if(XDim.link2Window && XDim.hLink.link1 != num)
	 {
	  XDim.link2Window = False;
	  XDim.hLink.link2 = num;
	  num = XDim.port;
	 }
       else if(XDim.math && num!=XDim.port)
         {
          XDim.oper2 = num;
          XDim.math = False;
          WD_RedrawMath(&XDim);
         }
       else if(num!=XDim.port)
         {
          XDim.port = num;
          XtPopdown(XDW->DecoEdit.Shell);
         }
       view1 = &((XDim.Views)[num]);
       WD_RedrawLink(&(XDW->Link), view1, &XDim);
      }
   }

/*
 * callback to destroy a viewport
 */

void XT_DestroyViewport(Widget w, XtPointer client_data, XtPointer call_data)
   {
    extern XDimInfo XDim;
    ViewportInfo *view, *views;
    int del;
    
    del = *((int*)client_data);
    if(del >=0)
      {
       view = &((XDim.Views)[del]);
       if(view->pict == NULL)
          return;
       if(view->edit && del == XDim.aView.edPort)
         XDim.aView.edPort = -1;
       views = XDim.Views;
       free(view->pict);
       view->pict = NULL;
       if(view->Data)
         {
          free(view->Data);
          view->Data = NULL;
         }
       if(view->actData)
         {
          free(view->actData);
          view->actData = NULL;
         }
       if(view->charData)
         {
          free(view->charData);
          view->charData = NULL;
         }
       FI_RemoveWasteLinks(del, &XDim);
       for(XDim.port = MAX_VIEWPORT_NUMBER-1;XDim.port >=0; (XDim.port)--)
	  if((views[XDim.port]).pict) break;
       XtPopdown(XDim.myWidgets.DecoEdit.Shell);
       XtDestroyWidget(((XDim.myWidgets.Views)[del]).Shell);
      }
  }


/*
 * callback to destroy the edit matrix
 */

void XT_DestroyMatrix(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    XDimEditMatrix *XEM;
    
    XDim = (XDimInfo*)client_data;
    XEM = &((XDim->myWidgets).EditMatrix);
    XtDestroyWidget(XEM->Shell);
    free(XEM->Matrix);
    free(XEM->Values);
    XEM->Matrix = NULL;
    XEM->Values = NULL;
   }

/*
 * Histogramm Equalisation
 */

void XT_Histogramm(Widget w, XtPointer client_data, XtPointer call_data)
   {
    int hg[MAX_DECO_NUMBER],i,Size, index;
    XDimInfo *XDim;
    ViewportInfo *view;
    XDimDeco *XDD;
    myCoor *cp, min;
    myPixel *deco, colS;
    double Norm;

    XDim = (XDimInfo *) client_data;
    if(XDim->port >= 0)
      {   
       view = &((XDim->Views)[XDim->port]);
       XDD = &((XDim->myWidgets).DecoEdit);       
       if(view->actData || view->charData)
         {
	  if(view->changed & DATA_CHANGED)
            {
             if(CO_Data2Int(view, XDim))
               {
		WD_XDimError(XDim, NULL);
		return;
               }
            }
          view->changed |= TRANS_AND_CHANGE;
	  for(i=0;i<MAX_DECO_NUMBER;i++)
	     hg[i]=0;
	  Size = view->Width * view->Height;
	  Norm = (view->colE-view->colS)/(double)Size;
	  cp = view->Data;
	  min = view->dMin;
	  for(i=0; i<Size; i++)
	     {
	      index = *cp++-min;
	      hg[index]++;
	     }
	  for(i=1; i<MAX_DECO_NUMBER; i++)
	     hg[i] += hg[i-1];
	  deco = view->deco;
	  colS = view->colS;
	  for(i=0; i<MAX_DECO_NUMBER; i++)
	     *deco++ = (myPixel)(Norm*hg[i]+0.5)+colS;
	  XT_RedrawZoomedDeco(XDD->DZoom, XDim, NULL);
	  XT_RedrawDecoVector(XDD->DArea, XDim, NULL);
	  XT_Redraw((((XDim->myWidgets).Views)[XDim->port]).DArea,
		(XtPointer)(view), NULL);
	 }
       else
         {
          strcpy(XDim->error, "To calculate a HISTOGRAM, a data field must be loaded");
	  WD_XDimError(XDim, XDD->Shell);          
         }
      }
   }

/* Timer callbacks for rotation, zoom etc. in the viewports*/

void XT_StopTimer(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;

    XDim = (XDimInfo*) client_data;
    XDim->colors.tNext = False;
   }

/* Rot left */

void XT_ViewRotLeft(XtPointer client_data, XtIntervalId *IId)
   {
    XDimInfo *XDim;
    XDColorInfo *XDC;
    ViewportInfo *view;
    
    XDim = (XDimInfo*) client_data;
    if(XDim->port >=0)
      {
       XDC = &(XDim->colors);
       if(XDC->tNext)
         {
	  view = &((XDim->Views)[XDim->port]);
	  view->changed |= CHANGED;
	  view->rotXY -= view->dAngle;
	  if(view->rotXY <0.0)
	    view->rotXY += 2*PI;
	  view->sinXY = (myCoor)(CO_ONE*sin(view->rotXY));
	  view->cosXY = (myCoor)(CO_ONE*cos(view->rotXY));
	  XT_Redraw((((XDim->myWidgets).Views)[XDim->port]).DArea,
		(XtPointer)(view), NULL);
	  XtAppAddTimeOut(XDim->myWidgets.XDimContext, XDC->tInterval,
	  	XT_ViewRotLeft, XDim);
         }
      }
   }

void XT_StartRotLeft(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    
    XDim = (XDimInfo *) client_data;
    XDim->colors.tNext = True;
    XT_ViewRotLeft(client_data, (XtIntervalId *) NULL);
   }

/* Rot right */

void XT_ViewRotRight(XtPointer client_data, XtIntervalId *IId)
   {
    XDimInfo *XDim;
    XDColorInfo *XDC;
    ViewportInfo *view;
    
    XDim = (XDimInfo*) client_data;
    if(XDim->port >=0)
      {
       XDC = &(XDim->colors);
       if(XDC->tNext)
         {
	  view = &((XDim->Views)[XDim->port]);
	  view->changed |= CHANGED;
	  view->rotXY += view->dAngle;
	  if(view->rotXY >= 2*PI)
	    view->rotXY -= 2*PI;
	  view->sinXY = (myCoor)(CO_ONE*sin(view->rotXY));
	  view->cosXY = (myCoor)(CO_ONE*cos(view->rotXY));
	  XT_Redraw((((XDim->myWidgets).Views)[XDim->port]).DArea,
		(XtPointer)(view), NULL);
	  XtAppAddTimeOut(XDim->myWidgets.XDimContext, XDC->tInterval,
	  	XT_ViewRotRight, XDim);
         }
      }
   }

void XT_StartRotRight(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    
    XDim = (XDimInfo *) client_data;
    XDim->colors.tNext = True;
    XT_ViewRotRight(client_data, (XtIntervalId *) NULL);
   }

/* Rot up */

void XT_ViewRotUp(XtPointer client_data, XtIntervalId *IId)
   {
    XDimInfo *XDim;
    XDColorInfo *XDC;
    ViewportInfo *view;
    
    XDim = (XDimInfo*) client_data;
    if(XDim->port >=0)
      {
       XDC = &(XDim->colors);
       if(XDC->tNext)
         {
	  view = &((XDim->Views)[XDim->port]);
	  view->changed |= CHANGED;
	  view->rotZ += view->dAngle;
	  if(view->rotZ > PI)
	    view->rotZ = PI;
	  view->sinZ = (myCoor)(CO_ONE*sin(view->rotZ));
	  view->cosZ = (myCoor)(CO_ONE*cos(view->rotZ));
	  XT_Redraw((((XDim->myWidgets).Views)[XDim->port]).DArea,
		(XtPointer)(view), NULL);
	  XtAppAddTimeOut(XDim->myWidgets.XDimContext, XDC->tInterval,
	  	XT_ViewRotUp, XDim);
         }
      }
   }

void XT_StartRotUp(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    
    XDim = (XDimInfo *) client_data;
    XDim->colors.tNext = True;
    XT_ViewRotUp(client_data, (XtIntervalId *) NULL);
   }

/* Rot down */

void XT_ViewRotDown(XtPointer client_data, XtIntervalId *IId)
   {
    XDimInfo *XDim;
    XDColorInfo *XDC;
    ViewportInfo *view;
    
    XDim = (XDimInfo*) client_data;
    if(XDim->port >=0)
      {
       XDC = &(XDim->colors);
       if(XDC->tNext)
         {
	  view = &((XDim->Views)[XDim->port]);
	  view->changed |= CHANGED;
	  view->rotZ -= view->dAngle;
	  if(view->rotZ < 0.0)
	    view->rotZ = 0.0;
	  view->sinZ = (myCoor)(CO_ONE*sin(view->rotZ));
	  view->cosZ = (myCoor)(CO_ONE*cos(view->rotZ));
	  XT_Redraw((((XDim->myWidgets).Views)[XDim->port]).DArea,
		(XtPointer)(view), NULL);
	  XtAppAddTimeOut(XDim->myWidgets.XDimContext, XDC->tInterval,
	  	XT_ViewRotDown, XDim);
         }
      }
   }

void XT_StartRotDown(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    
    XDim = (XDimInfo *) client_data;
    XDim->colors.tNext = True;
    XT_ViewRotDown(client_data, (XtIntervalId *) NULL);
   }

/* Zoom XY + */

void XT_ViewZoomXYPlus(XtPointer client_data, XtIntervalId *IId)
   {
    XDimInfo *XDim;
    XDColorInfo *XDC;
    ViewportInfo *view;
    
    XDim = (XDimInfo*) client_data;
    if(XDim->port >=0)
      {
       XDC = &(XDim->colors);
       if(XDC->tNext)
         {
	  view = &((XDim->Views)[XDim->port]);
	  view->changed |= CHANGED;
	  view->zoomX *= view->zoom;
	  if(view->zoomX >= CO_ONE)
	    view->zoomX = CO_ONE;
	  else
	    view->zoomX++;
	  view->zoomY *= view->zoom;
	  if(view->zoomY >= CO_ONE)
	    view->zoomY = CO_ONE;
	  else
	    view->zoomY++;
	  view->kmin = Mmax3(view->zoomX, view->zoomY, view->zoomZ) << 1;
	  if(view->k && view->k < view->kmin)
	    view->k = view->kmin;
	  XT_Redraw((((XDim->myWidgets).Views)[XDim->port]).DArea,
		(XtPointer)(view), NULL);
	  XtAppAddTimeOut(XDim->myWidgets.XDimContext, XDC->tInterval,
	  	XT_ViewZoomXYPlus, XDim);
         }
      }
   }

void XT_StartZoomXYPlus(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    
    XDim = (XDimInfo *) client_data;
    XDim->colors.tNext = True;
    XT_ViewZoomXYPlus(client_data, (XtIntervalId *) NULL);
   }

/* Zoom XY - */

void XT_ViewZoomXYMinus(XtPointer client_data, XtIntervalId *IId)
   {
    XDimInfo *XDim;
    XDColorInfo *XDC;
    ViewportInfo *view;
    
    XDim = (XDimInfo*) client_data;
    if(XDim->port >=0)
      {
       XDC = &(XDim->colors);
       if(XDC->tNext)
         {
	  view = &((XDim->Views)[XDim->port]);
	  view->changed |= CHANGED;
	  view->zoomX /= view->zoom;
	  view->zoomY /= view->zoom;
	  view->kmin = Mmax3(view->zoomX, view->zoomY, view->zoomZ) << 1;
	  XT_Redraw((((XDim->myWidgets).Views)[XDim->port]).DArea,
		(XtPointer)(view), NULL);
	  XtAppAddTimeOut(XDim->myWidgets.XDimContext, XDC->tInterval,
	  	XT_ViewZoomXYMinus, XDim);
         }
      }
   }

void XT_StartZoomXYMinus(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    
    XDim = (XDimInfo *) client_data;
    XDim->colors.tNext = True;
    XT_ViewZoomXYMinus(client_data, (XtIntervalId *) NULL);
   }

/* Zoom Z + */

void XT_ViewZoomZPlus(XtPointer client_data, XtIntervalId *IId)
   {
    XDimInfo *XDim;
    XDColorInfo *XDC;
    ViewportInfo *view;
    
    XDim = (XDimInfo*) client_data;
    if(XDim->port >=0)
      {
       XDC = &(XDim->colors);
       if(XDC->tNext)
         {
	  view = &((XDim->Views)[XDim->port]);
	  view->changed |= CHANGED;
	  view->zoomZ *= view->zoom;
	  if(view->zoomZ > CO_ONE)
	    view->zoomZ = CO_ONE;
	  else
	    view->zoomZ++;
	  view->kmin = Mmax3(view->zoomX, view->zoomY, view->zoomZ) << 1;
	  if(view->k && view->k < view->kmin)
	    view->k = view->kmin;
	  XT_Redraw((((XDim->myWidgets).Views)[XDim->port]).DArea,
		(XtPointer)(view), NULL);
	  XtAppAddTimeOut(XDim->myWidgets.XDimContext, XDC->tInterval,
	  	XT_ViewZoomZPlus, XDim);
         }
      }
   }

void XT_StartZoomZPlus(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    
    XDim = (XDimInfo *) client_data;
    XDim->colors.tNext = True;
    XT_ViewZoomZPlus(client_data, (XtIntervalId *) NULL);
   }

/* Zoom Z - */

void XT_ViewZoomZMinus(XtPointer client_data, XtIntervalId *IId)
   {
    XDimInfo *XDim;
    XDColorInfo *XDC;
    ViewportInfo *view;
    
    XDim = (XDimInfo*) client_data;
    if(XDim->port >=0)
      {
       XDC = &(XDim->colors);
       if(XDC->tNext)
         {
	  view = &((XDim->Views)[XDim->port]);
	  view->changed |= CHANGED;
	  view->zoomZ /= view->zoom;
	  view->kmin = Mmax3(view->zoomX, view->zoomY, view->zoomZ) << 1;
	  XT_Redraw((((XDim->myWidgets).Views)[XDim->port]).DArea,
		(XtPointer)(view), NULL);
	  XtAppAddTimeOut(XDim->myWidgets.XDimContext, XDC->tInterval,
	  	XT_ViewZoomZMinus, XDim);
         }
      }
   }

void XT_StartZoomZMinus(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    
    XDim = (XDimInfo *) client_data;
    XDim->colors.tNext = True;
    XT_ViewZoomZMinus(client_data, (XtIntervalId *) NULL);
   }

/* Move Left */

void XT_ViewMoveLeft(XtPointer client_data, XtIntervalId *IId)
   {
    XDimInfo *XDim;
    XDColorInfo *XDC;
    ViewportInfo *view;
    
    XDim = (XDimInfo*) client_data;
    if(XDim->port >=0)
      {
       XDC = &(XDim->colors);
       if(XDC->tNext)
         {
	  view = &((XDim->Views)[XDim->port]);
	  view->changed |= CHANGED;
	  view->moveX -= view->delta;
	  if(view->moveX < -CO_ONE)
	    view->moveX = -CO_ONE;
	  XT_Redraw((((XDim->myWidgets).Views)[XDim->port]).DArea,
		(XtPointer)(view), NULL);
	  XtAppAddTimeOut(XDim->myWidgets.XDimContext, XDC->tInterval,
	  	XT_ViewMoveLeft, XDim);
         }
      }
   }

void XT_StartMoveLeft(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    
    XDim = (XDimInfo *) client_data;
    XDim->colors.tNext = True;
    XT_ViewMoveLeft(client_data, (XtIntervalId *) NULL);
   }

/* Move Right */

void XT_ViewMoveRight(XtPointer client_data, XtIntervalId *IId)
   {
    XDimInfo *XDim;
    XDColorInfo *XDC;
    ViewportInfo *view;
    
    XDim = (XDimInfo*) client_data;
    if(XDim->port >=0)
      {
       XDC = &(XDim->colors);
       if(XDC->tNext)
         {
	  view = &((XDim->Views)[XDim->port]);
	  view->changed |= CHANGED;
	  view->moveX += view->delta;
	  if(view->moveX+view->pwidth > CO_ONE)
	    view->moveX = CO_ONE-view->pwidth;
	  XT_Redraw((((XDim->myWidgets).Views)[XDim->port]).DArea,
		(XtPointer)(view), NULL);
	  XtAppAddTimeOut(XDim->myWidgets.XDimContext, XDC->tInterval,
	  	XT_ViewMoveRight, XDim);
         }
      }
   }

void XT_StartMoveRight(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    
    XDim = (XDimInfo *) client_data;
    XDim->colors.tNext = True;
    XT_ViewMoveRight(client_data, (XtIntervalId *) NULL);
   }

/* Move Down */

void XT_ViewMoveDown(XtPointer client_data, XtIntervalId *IId)
   {
    XDimInfo *XDim;
    XDColorInfo *XDC;
    ViewportInfo *view;
    
    XDim = (XDimInfo*) client_data;
    if(XDim->port >=0)
      {
       XDC = &(XDim->colors);
       if(XDC->tNext)
         {
	  view = &((XDim->Views)[XDim->port]);
	  view->changed |= CHANGED;
	  view->moveY -= view->delta;
	  if(view->moveY < -CO_ONE)
	    view->moveY = -CO_ONE;
	  XT_Redraw((((XDim->myWidgets).Views)[XDim->port]).DArea,
		(XtPointer)(view), NULL);
	  XtAppAddTimeOut(XDim->myWidgets.XDimContext, XDC->tInterval,
	  	XT_ViewMoveDown, XDim);
         }
      }
   }

void XT_StartMoveDown(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    
    XDim = (XDimInfo *) client_data;
    XDim->colors.tNext = True;
    XT_ViewMoveDown(client_data, (XtIntervalId *) NULL);
   }

/* Move Up */

void XT_ViewMoveUp(XtPointer client_data, XtIntervalId *IId)
   {
    XDimInfo *XDim;
    XDColorInfo *XDC;
    ViewportInfo *view;
    
    XDim = (XDimInfo*) client_data;
    if(XDim->port >=0)
      {
       XDC = &(XDim->colors);
       if(XDC->tNext)
         {
	  view = &((XDim->Views)[XDim->port]);
	  view->changed |= CHANGED;
	  view->moveY += view->delta;
	  if(view->moveY+view->pheight > CO_ONE)
	    view->moveY = CO_ONE-view->pheight;
	  XT_Redraw((((XDim->myWidgets).Views)[XDim->port]).DArea,
		(XtPointer)(view), NULL);
	  XtAppAddTimeOut(XDim->myWidgets.XDimContext, XDC->tInterval,
	  	XT_ViewMoveUp, XDim);
         }
      }
   }

void XT_StartMoveUp(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    
    XDim = (XDimInfo *) client_data;
    XDim->colors.tNext = True;
    XT_ViewMoveUp(client_data, (XtIntervalId *) NULL);
   }


/* Redraw Methods */

void XT_Redraw(Widget w, XtPointer client_data, XtPointer call_data)
   {
    ViewportInfo *view, *view2;
    Window win;
    int cut;
    Pixel *colTrans;
    extern XDimInfo XDim;
#ifdef REDRAW_TIME
    clock_t start,ende;
#endif /* REDRAW_TIME */

    win =XtWindow(w);
    if(!win)
      return;
    view = (ViewportInfo*)client_data;
#ifdef REDRAW_TIME
    start = clock();
#endif /* REDRAW_TIME */
    colTrans = XDim.colors.colTrans;
    if(view->changed)
      {
       if(view->changed & TRANS_CHANGED)
         GR_Trans(view, colTrans, XDim.colors.transCol, XDim.colors.export);
       if(view->changed & DATA_CHANGED)
         {
          if(CO_Data2Int(view, &XDim))
            {
	     WD_XDimError(&XDim, NULL);
	     return;
            }
         }
       if(view->Data)
	 {
	  (XDim.ClearFast)(win, view, &XDim);
          (XDim.FastRedraw)(win, view, NULL, &XDim);
         }
       else
	 {
	  if(view->cut == CUT_WINDOW)
            {
             cut = FI_GetLink("CutData", view, &XDim);
	     if(cut < 0)
               {
        	strcpy(XDim.mess, "Link to Cut Data not found");
        	WD_ViewportMessage(view, &XDim);
	       }
	     else
	       {
        	view2 = &((XDim.Views)[cut]);
        	if(view2->actData == NULL ||
			view->x1 > (double)(view2->Width-1) ||
			view->y1 > (double)(view2->Height-1) ||
			view->x2 > (double)(view2->Width-1) ||
			view->y2 > (double)(view2->Height-1))
		  {
        	   strcpy(XDim.mess,
		       "Cut Data Changed, could NOT create a crossection");
        	   WD_ViewportMessage(view, &XDim);
		  }
		else
		  {
		   HW_ClearPict(win, view, &XDim);
		   XT_RedrawCut2D(win, view, view2, &XDim);
		  }
	       }
	    }
	  else
	    {
	     strcpy( XDim.mess, "Use file menu to load data");
	     WD_ViewportMessage(view, &XDim);
	    }
	 }
      }    
#ifdef REDRAW_TIME
    ende = clock();
   printf("Fast start: %d\nend: %d\ndelta T: %e\n", start, ende,
   ((double)(ende-start))/CLOCKS_PER_SEC);
#endif /* REDRAW_TIME */
    if(!view->changed)
      {
#ifdef REDRAW_TIME
       start = clock();
#endif /* REDRAW_TIME */
       if(HW_Image(0,0,view->pwidth,view->pheight,view->pict,win,
		colTrans,&(XDim.hwInfo)))
	 {
	  strcpy(XDim.error, "Couldn't allocate a correct XImage structure");
	  WD_XDimError(&XDim, NULL);
	  return;
	 }
#ifdef REDRAW_TIME
       ende = clock();
       printf("Image conversion start: %d\nend: %d\ndelta T: %e\n\n", start, ende,
         ((double)(ende-start))/CLOCKS_PER_SEC);
#endif /* REDRAW_TIME */
      }
   }

void XT_NormalRedraw(Widget w, XtPointer client_data, XtPointer call_data)
   {
    ViewportInfo *view, *view1, *view2;
    Window win;
    int real, imag, cut;
    Pixel *colTrans;
    extern XDimInfo XDim;
#ifdef REDRAW_TIME
    clock_t start,ende;
#endif /* REDRAW_TIME */

    win =XtWindow(w);
    if(!win)
      return;
    view = (ViewportInfo*) client_data;
#ifdef REDRAW_TIME
    start = clock();
#endif /* REDRAW_TIME */
    colTrans = XDim.colors.colTrans;
    if(XDim.NormalRedraw== XT_RedrawComFilled ||
	XDim.NormalRedraw== XT_RedrawComHFilled)
      {
       real = FI_GetLink("real", view, &XDim);
       imag = FI_GetLink("imag", view, &XDim);
       if(real < 0 || imag < 0)
         {
          strcpy(XDim.mess, "Link to REAL or IMAGinary part not found");
          WD_ViewportMessage(view, &XDim);
	  return;
	 }
       view1 = &((XDim.Views)[imag]);
       view = &((XDim.Views)[real]);
      }
    else
      view1 = NULL;
    if(view1)
      {
       if(!view1->Data)
         {
	  strcpy( XDim.mess, "No imaginary data loaded for this Viewport");
          WD_ViewportMessage(view, &XDim);
          return;
         }
       if(view1->changed & TRANS_CHANGED)
         GR_Trans(view1, colTrans, XDim.colors.transCol, XDim.colors.export);
       if(view1->changed & DATA_CHANGED)
         {
          if(CO_Data2Int(view1, &XDim))
            {
	     WD_XDimError(&XDim, NULL);
	     return;
            }
         }
      }    
    if(view->changed)
      {
       if(view->changed & TRANS_CHANGED)
         GR_Trans(view, colTrans, XDim.colors.transCol, XDim.colors.export);
       if(view->changed & DATA_CHANGED)
         {
          if(CO_Data2Int(view, &XDim))
            {
	     WD_XDimError(&XDim, NULL);
	     return;
            }
         }
      }
   if(view->Grid)
     view->DrawGrid = True;
   if(view->Data)
     {
      (XDim.ClearNormal)(win, view, &XDim);
      (XDim.NormalRedraw)(win, view, view1, &XDim);
     }
   else
     {
      if(view->cut == CUT_WINDOW)
        {
         cut = FI_GetLink("CutData", view, &XDim);
	 if(cut < 0)
           {
            strcpy(XDim.mess, "Link to Cut Data not found");
            WD_ViewportMessage(view, &XDim);
	   }
	 else
	   {
            view2 = &((XDim.Views)[cut]);
            if(view2->actData == NULL || view->x1 > (double)(view2->Width-1) ||
		   view->y1 > (double)(view2->Height-1) ||
		   view->x2 > (double)(view2->Width-1) ||
		   view->y2 > (double)(view2->Height-1))
	      {
               strcpy(XDim.mess,
		   "Cut Data Changed, could NOT create a crossection");
               WD_ViewportMessage(view, &XDim);
	      }
	    else
	      {
	       HW_ClearPict(win, view, &XDim);
	       XT_RedrawCut2D(win, view, view2, &XDim);
	      }
	   }
        }
      else
        {
	 strcpy( XDim.mess, "Use file menu to load data");
	 WD_ViewportMessage(view, &XDim);
	}
     }
   view->DrawGrid = False;
#ifdef REDRAW_TIME
   ende = clock();
   printf("Normal start: %d\nend: %d\ndelta T: %e\n", start, ende,
   ((double)(ende-start))/CLOCKS_PER_SEC);
#endif /* REDRAW_TIME */
    if(!view->changed)
      {
#ifdef REDRAW_TIME
       start = clock();
#endif /* REDRAW_TIME */
       if(HW_Image(0,0,view->pwidth,view->pheight,view->pict,win,
		colTrans,&(XDim.hwInfo)))
	 {
	  strcpy(XDim.error, "Couldn't allocate a correct XImage structure");
	  WD_XDimError(&XDim, NULL);
	  return;
	 }
#ifdef REDRAW_TIME
       ende = clock();
       printf("Image conversion start: %d\nend: %d\ndelta T: %e\n\n", start, ende,
         ((double)(ende-start))/CLOCKS_PER_SEC);
#endif /* REDRAW_TIME */
      }
   }

void XT_ForceActRedraw(Widget w, XtPointer client_data, XtPointer call_data)
   {
    XDimInfo *XDim;
    int port;

    XDim = (XDimInfo*)client_data;
    port = XDim->port;
    XT_NormalRedraw(((XDim->myWidgets).Views[port]).DArea,
	(XtPointer) &((XDim->Views)[port]), NULL);
   }

/* Redraw Grid only */

void XT_RedrawGrid(Window win, ViewportInfo *view, ViewportInfo *view1,
		XDimInfo *XDim)
   {
    XDHardwareInfo *HWI;
    int i, Num;
    myCoor x0, y0, x1, y1, sinXY, cosXY, sinZ, cosZ, moveX, moveY;
    myCoor z0, z1, Fact, xd, yd,MaxX, MaxY, zoomZ, zoomX, zoomY, Sum, height;
    myCoor coMaxX, coMaxY, k;
    int xa,ya,xb,yb;
    
    HWI = &(XDim->hwInfo);
    sinXY = view->sinXY;
    cosXY = view->cosXY;
    sinZ = view->sinZ;
    cosZ = view->cosZ;
    moveX = view->moveX;
    moveY = view->moveY;
    zoomX = view->zoomX;
    zoomY = view->zoomY;
    zoomZ = view->zoomZ;
    Num = view->gNum;
    MaxX = view->Width;
    MaxY = view->Height;
    height = view->pheight;
    k = view->k;
    view->changed |= CHANGED;    
    HW_SetFore( ((XDim->colors).colTrans)[view->fore], HWI);   

    /* redraw y grid */

    if(MaxY > MaxX)
      {
       if(MaxY >1)
	 Fact = MaxY-1;
       else
	 Fact = 1;
       coMaxY = CO_ONE >> 1;
       coMaxX = ((MaxX-1)<<(CO_ONESHIFT-1))/Fact;
      }
    else
      {
       if(MaxX >1)
	 Fact = MaxX-1;
       else
	 Fact = 1;
       coMaxX = CO_ONE >> 1;
       coMaxY = ((MaxY-1)<<(CO_ONESHIFT-1))/Fact;
      }
    Sum = MaxX -1;
    y0 = -coMaxY;
    y1 = coMaxY;
    for(i=0; i<MaxX; i++)
       {
        Sum += Num;
        if(Sum >= MaxX)
          {
           Sum -= MaxX;
           x0 = (i<<CO_ONESHIFT)/Fact -coMaxX;
           x1 = x0;
           CO_TransformXY(xa,ya,xd,yd,x0,y0,0,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
           CO_TransformXY(xb,yb,xd,yd,x1,y1,0,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
           HW_DrawLine(xa, Mupdown(ya,height), xb, Mupdown(yb,height), win, HWI);
          }
       }
    /* Draw last line */
    x0 = coMaxX;
    x1 = x0;
    CO_TransformXY(xa,ya,xd,yd,x0,y0,0,sinXY,cosXY,sinZ,cosZ,
	zoomX,zoomY,zoomZ,moveX,moveY,k);
    CO_TransformXY(xb,yb,xd,yd,x1,y1,0,sinXY,cosXY,sinZ,cosZ,
	zoomX,zoomY,zoomZ,moveX,moveY,k);
    HW_DrawLine(xa, Mupdown(ya,height), xb, Mupdown(yb,height), win, HWI);

    /* redraw x grid */

    Sum = MaxY -1;
    x0 = -coMaxX;
    x1 = coMaxX;
    for(i=0; i<MaxY; i++)
       {
        Sum += Num;
        if(Sum >= MaxY)
          {
           Sum -= MaxY;
           y0 = (i<<CO_ONESHIFT)/Fact -coMaxY;
           y1 = y0;
           CO_TransformXY(xa,ya,xd,yd,x0,y0,0,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
           CO_TransformXY(xb,yb,xd,yd,x1,y1,0,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
           HW_DrawLine(xa, Mupdown(ya,height), xb, Mupdown(yb,height), win, HWI);
          }
       }
    /* Draw last line */
    y0 = coMaxY;
    y1 = y0;
    CO_TransformXY(xa,ya,xd,yd,x0,y0,0,sinXY,cosXY,sinZ,cosZ,
	 zoomX,zoomY,zoomZ,moveX,moveY,k);
    CO_TransformXY(xb,yb,xd,yd,x1,y1,0,sinXY,cosXY,sinZ,cosZ,
	 zoomX,zoomY,zoomZ,moveX,moveY,k);
    HW_DrawLine(xa, Mupdown(ya,height), xb, Mupdown(yb,height), win, HWI);

    /* Show (0,0) */
    z0 = view->dMin;
    z1 = z0+CO_ONE;
    CO_TransformXY(xa,ya,xd,yd,-coMaxX,-coMaxY,z0,sinXY,cosXY,sinZ,cosZ,
	 zoomX,zoomY,zoomZ,moveX,moveY,k);
    CO_TransformXY(xb,yb,xd,yd,-coMaxX,-coMaxY,z1,sinXY,cosXY,sinZ,cosZ,
	 zoomX,zoomY,zoomZ,moveX,moveY,k);
    HW_DrawLine(xa, Mupdown(ya,height), xb, Mupdown(yb,height), win, HWI);
   }

/* Redraw  Lines */

void XT_RedrawLines(Window win, ViewportInfo *view, ViewportInfo *view1,
		XDimInfo *XDim)
   {
    XDHardwareInfo *HWI;
    int i,j, gStep, del;
    myCoor y0, x1, y1, z1, coMaxX, sinXY, cosXY, sinZ, cosZ, moveX, moveY;
    myCoor Fact, xd, yd, coMaxY, MaxX, MaxY, zoomZ, zoomX, zoomY, height;
    myCoor x2,y2,z2,k,*dp, *sp;
    int xa,ya,xb,yb,xc,yc;
    
    HWI = &(XDim->hwInfo);
    sinXY = view->sinXY;
    cosXY = view->cosXY;
    sinZ = view->sinZ;
    cosZ = view->cosZ;
    moveX = view->moveX;
    moveY = view->moveY;
    zoomX = view->zoomX;
    zoomY = view->zoomY;
    zoomZ = view->zoomZ;
    gStep = view->gStep;
    MaxX = view->Width;
    MaxY = view->Height;
    height = view->pheight;
    k = view->k;
    view->changed |= CHANGED;    
    HW_SetFore( ((XDim->colors).colTrans)[view->fore], HWI);   

    /* redraw y grid */

    if(MaxY > MaxX)
      {
       if(MaxY >1)
	 Fact = MaxY-1;
       else
	 Fact = 1;
       coMaxY = CO_ONE >> 1;
       coMaxX = ((MaxX-1)<<(CO_ONESHIFT-1))/Fact;
      }
    else
      {
       if(MaxX >1)
	 Fact = MaxX-1;
       else
	 Fact = 1;
       coMaxX = CO_ONE >> 1;
       coMaxY = ((MaxY-1)<<(CO_ONESHIFT-1))/Fact;
      }
    del = view->Width*gStep;
    sp = view->Data;
    y2 = -coMaxY;
    for(i=gStep; i<MaxY; i += gStep)
       {
        dp = sp;
        sp += del;
        x1 = -coMaxX;
        y0 = y2;
        y1 = y0;
        x2 = x1;
        y2 = (i<<CO_ONESHIFT)/Fact -coMaxY;
        z2 = *(dp+del);
        z1 = *dp;
        dp += gStep;
	CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
	     zoomX,zoomY,zoomZ,moveX,moveY,k);
	CO_TransformXY(xc,yc,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
	     zoomX,zoomY,zoomZ,moveX,moveY,k);
	HW_DrawLine(xb, Mupdown(yb,height), xc, Mupdown(yc,height), win, HWI);
        for(j=gStep; j<MaxX; j += gStep)
           {
            xa = xb;
            ya = yb;
            x1 = (j<<CO_ONESHIFT)/Fact -coMaxX;
            x2 = x1;
            z2 = *(dp+del);
            z1 = *dp;
            dp += gStep;
	    CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		 zoomX,zoomY,zoomZ,moveX,moveY,k);
	    CO_TransformXY(xc,yc,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		 zoomX,zoomY,zoomZ,moveX,moveY,k);
	    HW_DrawLine(xa, Mupdown(ya,height), xb, Mupdown(yb,height), win, HWI);
	    HW_DrawLine(xb, Mupdown(yb,height), xc, Mupdown(yc,height), win, HWI);
           }
       }
    y0 = y2;
    y1 = y2;
    x1 = -coMaxX;
    dp = sp;
    z1 = *dp;
    dp += gStep;
    CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
	 zoomX,zoomY,zoomZ,moveX,moveY,k);
    for(j=gStep; j<MaxX; j += gStep)
       {
        xa = xb;
        ya = yb;
        x1 = (j<<CO_ONESHIFT)/Fact -coMaxX;
        z1 = *dp;
        dp += gStep;
	CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
	     zoomX,zoomY,zoomZ,moveX,moveY,k);
	HW_DrawLine(xa, Mupdown(ya,height), xb, Mupdown(yb,height), win, HWI);
       }
   }

/* Redraw  Hidden Lines */

void XT_RedrawHidden(Window win, ViewportInfo *view, ViewportInfo *view1,
		XDimInfo *XDim)
   {
    XDColorInfo *XDC;
    XDHardwareInfo *HWI;
    int i,j, gStep, del;
    double pi4, dAngle;
    myCoor x1, y1,coMaxX, coMaxY, sinXY, cosXY, sinZ, cosZ, moveX, moveY;
    myCoor z1, Fact, xd, yd,MaxX, MaxY, zoomZ, zoomX, zoomY, width, height;
    myCoor x2,y2,z2,k,*dp, *sp, *HMin, *HMax, Min, Max;
    myPixel Col, *pict;
    int xa,ya,xb,yb,xc,yc;
    
    HWI = &(XDim->hwInfo);
    XDC = &(XDim->colors);
    dAngle = 2.0*PI;
    pi4 = PI/4.0;
    sinXY = view->sinXY;
    cosXY = view->cosXY;
    sinZ = view->sinZ;
    cosZ = view->cosZ;
    moveX = view->moveX;
    moveY = view->moveY;
    zoomX = view->zoomX;
    zoomY = view->zoomY;
    zoomZ = view->zoomZ;
    gStep = view->gStep;
    width = view->pwidth;
    height = view->pheight;
    k = view->k;
    pict = view->pict;
    if(!XDC->export)
      view->changed = NOCHANGE;
    if(XDC->transCol || XDC->export)
      {
       Col = view->fore;
      }
    else
      {
       Col = (myPixel)((XDC->colTrans)[view->fore]);
      }
    /* reset min max */

    HMin = view->HMin;
    HMax = view->HMax;
    Min = CO_ONE+1;
    Max = -Min;
    for(i=0; i<CO_ONE; i++)
       {
        *HMin++ = Min;
        *HMax++ = Max; 
       }
    Min = view->dMin;
    Max = Min+CO_ONE;

    MaxX = view->Width;
    MaxY = view->Height;
    del = MaxX*gStep;
    if(gStep > 1)
      {
       MaxX = MaxX/gStep;
       MaxY = MaxY/gStep;
      }
    if(MaxY > MaxX)
      {
       if(MaxY >1)
	 Fact = MaxY-1;
       else
	 Fact = 1;
       coMaxY = CO_ONE >> 1;
       coMaxX = ((MaxX-1)<<(CO_ONESHIFT-1))/Fact;
      }
    else
      {
       if(MaxX >1)
	 Fact = MaxX-1;
       else
	 Fact = 1;
       coMaxX = CO_ONE >> 1;
       coMaxY = ((MaxY-1)<<(CO_ONESHIFT-1))/Fact;
      }

    /* redraw */
    HMin = view->HMin;
    HMax = view->HMax;
    if((dAngle -= pi4) <= view->rotXY)
      {
       sp = view->Data;
       dp = sp;
       y1 = -coMaxY;
       x1 = -coMaxX;
       z1 = *dp;
       dp += gStep;
       CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
	    zoomX,zoomY,zoomZ,moveX,moveY,k);
       for(j=1; j<MaxX; j++)
	  {
           xa = xb;
           ya = yb;
           x1 = (j<<CO_ONESHIFT)/Fact -coMaxX;
           z1 = *dp;
           dp += gStep;
	   CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
	  }
       y2 = -coMaxY;
       for(i=1; i<MaxY; i++)
	  {
           x1 = -coMaxX;
           y1 = y2;
           z1 = *sp;
	   sp += del;
	   dp = sp;
	   x2 = x1;
           y2 = (i<<CO_ONESHIFT)/Fact -coMaxY;
           z2 = *dp;
           dp += gStep;
	   CO_TransformXY(xa,ya,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
           for(j=1; j<MaxX; j++)
              {
               xa = xb;
               ya = yb;
               x1 = (j<<CO_ONESHIFT)/Fact -coMaxX;
               z1 = *(dp-del);
               x2 = x1;
               z2 = *dp;
               dp += gStep;
	       CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xc,yc,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       GR_myHiddenLine(Col, xc, yc, xb, yb,
			HMin, HMax, width, height, pict);
	       GR_myHiddenLine(Col, xb, yb, xa, ya,
			HMin, HMax, width, height, pict);
              }
	  }
       if(view->DrawGrid)
	 {
	  GR_HiddenGridImage(coMaxX,coMaxY,Min,Max,
		POS_TOP_X | POS_TOP_Y, view, XDim);
	 }
      }
    else if((dAngle -= pi4) <= view->rotXY)
      {
       sp = view->Data;
       dp = sp;
       y1 = -coMaxY;
       x1 = -coMaxX;
       z1 = *dp;
       dp += del;
       CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
	    zoomX,zoomY,zoomZ,moveX,moveY,k);
       for(j=1; j<MaxY; j++)
	  {
           xa = xb;
           ya = yb;
           y1 = (j<<CO_ONESHIFT)/Fact -coMaxY;
           z1 = *dp;
           dp += del;
	   CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
	  }
       x2 = -coMaxX;
       for(i=1; i<MaxX; i++)
	  {
           x1 = x2;
	   y1 = -coMaxY;
	   z1 = *sp;
	   sp += gStep;
	   dp = sp;
           x2 = (i<<CO_ONESHIFT)/Fact -coMaxX;
           y2 = y1;
           z2 = *dp;
           dp += del;
	   CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xa,ya,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
           for(j=1; j<MaxY; j++)
              {
               xa = xb;
               ya = yb;
               y1 = (j<<CO_ONESHIFT)/Fact -coMaxY;
               z1 = *(dp-gStep);
               y2 = y1;
               z2 = *dp;
               dp += del;
	       CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xc,yc,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       GR_myHiddenLine(Col, xc, yc, xb, yb,
			HMin, HMax, width, height, pict);
	       GR_myHiddenLine(Col, xb, yb, xa, ya,
			HMin, HMax, width, height, pict);
              }
	  }
       if(view->DrawGrid)
	 {
	  GR_HiddenGridImage(coMaxX, coMaxY, Min, Max,
		POS_TOP_X | POS_TOP_Y, view, XDim);
	 }
      }
    else if((dAngle -= pi4) <= view->rotXY)
      {
       sp = view->Data+ (MaxY-1)*del;
       dp = sp;
       y1 = coMaxY;
       x1 = -coMaxX;
       z1 = *dp;
       dp -= del;
       CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
	    zoomX,zoomY,zoomZ,moveX,moveY,k);
       for(j=1; j<MaxY; j++)
	  {
           xa = xb;
           ya = yb;
           y1 = coMaxY-(j<<CO_ONESHIFT)/Fact;
           z1 = *dp;
           dp -= del;
	   CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
	  }
       x2 = -coMaxX;
       for(i=1; i<MaxX; i++)
	  {
           x1 = x2;
	   y1 = coMaxY;
	   z1 = *sp;
	   sp += gStep;
	   dp = sp;
           x2 = (i<<CO_ONESHIFT)/Fact -coMaxX;
           y2 = y1;
           z2 = *dp;
           dp -= del;
	   CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xa,ya,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
           for(j=1; j<MaxY; j++)
              {
               xa = xb;
               ya = yb;
               y1 = coMaxY-(j<<CO_ONESHIFT)/Fact;
               z1 = *(dp-gStep);
               y2 = y1;
               z2 = *dp;
               dp -= del;
	       CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xc,yc,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       GR_myHiddenLine(Col, xc, yc, xb, yb,
			HMin, HMax, width, height, pict);
	       GR_myHiddenLine(Col, xb, yb, xa, ya,
			HMin, HMax, width, height, pict);
              }
	  }
       if(view->DrawGrid)
	 {
	  GR_HiddenGridImage(coMaxX, coMaxY, Min, Max,
		POS_TOP_X, view, XDim);
	 }
      }
    else if((dAngle -= pi4) <= view->rotXY)
      {
       sp = view->Data+ (MaxY-1)*del;
       dp = sp;
       y1 = coMaxY;
       x1 = -coMaxX;
       z1 = *dp;
       dp += gStep;
       CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
	    zoomX,zoomY,zoomZ,moveX,moveY,k);
       for(j=1; j<MaxX; j++)
	  {
           xa = xb;
           ya = yb;
           x1 = (j<<CO_ONESHIFT)/Fact -coMaxX;
           z1 = *dp;
           dp += gStep;
	   CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
	  }
       y2 = coMaxY;
       for(i=1; i<MaxY; i++)
	  {
           x1 = -coMaxX;
           y1 = y2;
           z1 = *sp;
	   sp -= del;
	   dp = sp;
	   x2 = x1;
           y2 = coMaxY-(i<<CO_ONESHIFT)/Fact;
           z2 = *dp;
           dp += gStep;
	   CO_TransformXY(xa,ya,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
           for(j=1; j<MaxX; j++)
              {
               xa = xb;
               ya = yb;
               x1 = (j<<CO_ONESHIFT)/Fact -coMaxX;
               z1 = *(dp+del);
               x2 = x1;
               z2 = *dp;
               dp += gStep;
	       CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xc,yc,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       GR_myHiddenLine(Col, xc, yc, xb, yb,
			HMin, HMax, width, height, pict);
	       GR_myHiddenLine(Col, xb, yb, xa, ya,
			HMin, HMax, width, height, pict);
              }
	  }
       if(view->DrawGrid)
	 {
	  GR_HiddenGridImage(coMaxX,-coMaxY,Min,Max,
		POS_TOP_X, view, XDim);
	 }
      }
    else if((dAngle -= pi4) <= view->rotXY)
      {
       sp = view->Data+(MaxY-1)*del+(MaxX-1)*gStep;
       dp = sp;
       y1 = coMaxY;
       x1 = coMaxX;
       z1 = *dp;
       dp -= gStep;
       CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
	    zoomX,zoomY,zoomZ,moveX,moveY,k);
       for(j=1; j<MaxX; j++)
	  {
           xa = xb;
           ya = yb;
           x1 = coMaxX-(j<<CO_ONESHIFT)/Fact;
           z1 = *dp;
           dp -= gStep;
	   CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
	  }
       y2 = coMaxY;
       for(i=1; i<MaxY; i++)
	  {
           x1 = coMaxX;
           y1 = y2;
           z1 = *sp;
	   sp -= del;
	   dp = sp;
	   x2 = x1;
           y2 = coMaxY-(i<<CO_ONESHIFT)/Fact;
           z2 = *dp;
           dp -= gStep;
	   CO_TransformXY(xa,ya,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
           for(j=1; j<MaxX; j++)
              {
               xa = xb;
               ya = yb;
               x1 = coMaxX-(j<<CO_ONESHIFT)/Fact;
               z1 = *(dp+del);
               x2 = x1;
               z2 = *dp;
               dp -= gStep;
	       CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xc,yc,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       GR_myHiddenLine(Col, xc, yc, xb, yb,
			HMin, HMax, width, height, pict);
	       GR_myHiddenLine(Col, xb, yb, xa, ya,
			HMin, HMax, width, height, pict);
              }
	  }
       if(view->DrawGrid)
	 {
	  GR_HiddenGridImage(-coMaxX,-coMaxY,Min,Max,
		0, view, XDim);
	 }
      }
    else if((dAngle -= pi4) <= view->rotXY)
      {
       sp = view->Data+(MaxY-1)*del+(MaxX-1)*gStep;
       dp = sp;
       y1 = coMaxY;
       x1 = coMaxX;
       z1 = *dp;
       dp -= del;
       CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
	    zoomX,zoomY,zoomZ,moveX,moveY,k);
       for(j=1; j<MaxY; j++)
	  {
           xa = xb;
           ya = yb;
           y1 = coMaxY-(j<<CO_ONESHIFT)/Fact;
           z1 = *dp;
           dp -= del;
	   CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
	  }
       x2 = coMaxX;
       for(i=1; i<MaxX; i++)
	  {
           x1 = x2;
	   y1 = coMaxY;
	   z1 = *sp;
	   sp -= gStep;
	   dp = sp;
           x2 = coMaxX-(i<<CO_ONESHIFT)/Fact;
           y2 = y1;
           z2 = *dp;
           dp -= del;
	   CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xa,ya,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
           for(j=1; j<MaxY; j++)
              {
               xa = xb;
               ya = yb;
               y1 = coMaxY-(j<<CO_ONESHIFT)/Fact;
               z1 = *(dp+gStep);
               y2 = y1;
               z2 = *dp;
               dp -= del;
	       CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xc,yc,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       GR_myHiddenLine(Col, xc, yc, xb, yb,
			HMin, HMax, width, height, pict);
	       GR_myHiddenLine(Col, xb, yb, xa, ya,
			HMin, HMax, width, height, pict);
              }
	  }
       if(view->DrawGrid)
	 {
	  GR_HiddenGridImage(-coMaxX,-coMaxY,Min,Max,
		0, view, XDim);
	 }
      }
    else if((dAngle -= pi4) <= view->rotXY)
      {
       sp = view->Data+ (MaxX-1)*gStep;
       dp = sp;
       y1 = -coMaxY;
       x1 = coMaxX;
       z1 = *dp;
       dp += del;
       CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
	    zoomX,zoomY,zoomZ,moveX,moveY,k);
       for(j=1; j<MaxY; j++)
	  {
           xa = xb;
           ya = yb;
           y1 = (j<<CO_ONESHIFT)/Fact -coMaxY;
           z1 = *dp;
           dp += del;
	   CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
	  }
       x2 = coMaxX;
       for(i=1; i<MaxX; i++)
	  {
           x1 = x2;
	   y1 = -coMaxY;
	   z1 = *sp;
	   sp -= gStep;
	   dp = sp;
           x2 = coMaxX-(i<<CO_ONESHIFT)/Fact;
           y2 = y1;
           z2 = *dp;
           dp += del;
	   CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xa,ya,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
           for(j=1; j<MaxY; j++)
              {
               xa = xb;
               ya = yb;
               y1 = (j<<CO_ONESHIFT)/Fact -coMaxY;
               z1 = *(dp+gStep);
               y2 = y1;
               z2 = *dp;
               dp += del;
	       CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xc,yc,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       GR_myHiddenLine(Col, xc, yc, xb, yb,
			HMin, HMax, width, height, pict);
	       GR_myHiddenLine(Col, xb, yb, xa, ya,
			HMin, HMax, width, height, pict);
              }
	  }
       if(view->DrawGrid)
	 {
	  GR_HiddenGridImage(-coMaxX,coMaxY,Min,Max,
		POS_TOP_Y, view, XDim);
	 }
      }
    else
      {
       sp = view->Data+ (MaxX-1)*gStep;
       dp = sp;
       y1 = -coMaxY;
       x1 = coMaxX;
       z1 = *dp;
       dp -= gStep;
       CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
	    zoomX,zoomY,zoomZ,moveX,moveY,k);
       for(j=1; j<MaxX; j ++)
	  {
           xa = xb;
           ya = yb;
           x1 = coMaxX-(j<<CO_ONESHIFT)/Fact;
           z1 = *dp;
           dp -= gStep;
	   CO_TransformXY(xb,yb,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
	  }
       y2 = -coMaxY;
       for(i=1; i<MaxY; i++)
	  {
           x1 = coMaxX;
           y1 = y2;
           z1 = *sp;
	   sp += del;
	   dp = sp;
	   x2 = x1;
           y2 = (i<<CO_ONESHIFT)/Fact -coMaxY;
           z2 = *dp;
           dp -= gStep;
	   CO_TransformXY(xa,ya,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   GR_myHiddenLine(Col, xa, ya, xb, yb,
		    HMin, HMax, width, height, pict);
           for(j=1; j<MaxX; j++)
              {
               xa = xb;
               ya = yb;
               x1 = coMaxX-(j<<CO_ONESHIFT)/Fact;
               z1 = *(dp-del);
               x2 = x1;
               z2 = *dp;
               dp -= gStep;
	       CO_TransformXY(xb,yb,xd,yd,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xc,yc,xd,yd,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       GR_myHiddenLine(Col, xc, yc, xb, yb,
			HMin, HMax, width, height, pict);
	       GR_myHiddenLine(Col, xb, yb, xa, ya,
			HMin, HMax, width, height, pict);
              }
	  }
       if(view->DrawGrid)
	 {
	  GR_HiddenGridImage(-coMaxX,coMaxY,Min,Max,
		POS_TOP_Y, view, XDim);
	 }
      }
   }

/* Redraw  Filled */

void XT_RedrawFilled(Window win, ViewportInfo *view, ViewportInfo *view1,
		XDimInfo *XDim)
   {
    XDColorInfo *XDC;
    XDHardwareInfo *HWI;
    int i,j, gStep, del;
    double pi2, dAngle;
    myCoor x1, y1, coMaxX, coMaxY, sinXY, cosXY, sinZ, cosZ, moveX, moveY;
    myCoor z1, Fact, xs, ys,MaxX, MaxY, zoomZ, zoomX, zoomY, width, height;
    myCoor x2,y2,z2,*dp, *sp, Max, Min,k;
    myPixel *pict, *trans;
    int xa,ya,za,xb,yb,zb,xc,yc,zc,xd,yd,zd;
    Boolean inter;
    
    XDC = &(XDim->colors);
    HWI = &(XDim->hwInfo);
    if(!XDC->export)
      view->changed = NOCHANGE;
    dAngle = 2.0*PI;
    pi2 = PI/2.0;
    inter = XDC->interCol;
    sinXY = view->sinXY;
    cosXY = view->cosXY;
    sinZ = view->sinZ;
    cosZ = view->cosZ;
    moveX = view->moveX;
    moveY = view->moveY;
    zoomX = view->zoomX;
    zoomY = view->zoomY;
    zoomZ = view->zoomZ;
    gStep = view->gStep;
    Min = view->dMin;
    Max = Min+CO_ONE;
    MaxX = view->Width;
    MaxY = view->Height;
    del = MaxX*gStep;
    if(gStep > 1)
      {
       MaxX = MaxX/gStep;
       MaxY = MaxY/gStep;
      }
    if(MaxY > MaxX)
      {
       if(MaxY >1)
	 Fact = MaxY-1;
       else
	 Fact = 1;
       coMaxY = CO_ONE >> 1;
       coMaxX = ((MaxX-1)<<(CO_ONESHIFT-1))/Fact;
      }
    else
      {
       if(MaxX >1)
	 Fact = MaxX-1;
       else
	 Fact = 1;
       coMaxX = CO_ONE >> 1;
       coMaxY = ((MaxY-1)<<(CO_ONESHIFT-1))/Fact;
      }
    width = view->pwidth;
    height = view->pheight;
    k = view->k;
    pict = view->pict;
    trans = view->trans;
    if((dAngle -= pi2) <= view->rotXY)
      {
       if(view->DrawGrid)
	 {
	  GR_ColoredGridImage(coMaxX,coMaxY,Min,Max,
		POS_TOP_X | POS_TOP_Y, view, XDim);
	 }
       sp = view->Data + (MaxY-1)*del+(MaxX-1)*gStep;
       dp = sp;
       y2 = coMaxY;
       for(i=1; i<MaxY; i++)
	  {
           x1 = coMaxX;
	   y1 = y2;
	   z1 = *sp;
	   sp -= del;
	   zc = z1-Min;
	   dp = sp;
	   x2 = coMaxX;
           y2 = coMaxY-(i<<CO_ONESHIFT)/Fact;
           z2 = *dp;
           dp -= gStep;
           zd = z2-Min;
	   CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
           for(j=1; j<MaxX; j++)
              {
               xa = xc;
               ya = yc;
               za = zc;
               xb = xd;
               yb = yd;
               zb = zd;
               x1 = coMaxX-(j<<CO_ONESHIFT)/Fact;
               z1 = *(dp+del);
               zc = z1-Min;
               x2 = x1;
               z2 = *dp;
               dp -= gStep;
               zd = z2-Min;
	       CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       if(inter)
	         {
		  GR_myTriangle(trans, xb, yb, zb, xc, yc, zc, xa, ya, za,
			   width, height, pict);
		  GR_myTriangle(trans, xd, yd, zd, xc, yc, zc, xb, yb, zb,
			   width, height, pict);
		 }
	       else
	         {
		  GR_Triangle(trans, xb, yb, xc, yc, xa, ya, za,
			   width, height, pict);
		  GR_Triangle(trans, xd, yd, xc, yc, xb, yb, zc,
			   width, height, pict);
		 }
              }
	  }
      }
    else if((dAngle -= pi2) <= view->rotXY)
      {
       if(view->DrawGrid)
	 {
	  GR_ColoredGridImage(coMaxX,-coMaxY,Min,Max,
		POS_TOP_X, view, XDim);
	 }
       sp = view->Data + (MaxX-1)*gStep;
       dp = sp;
       x2 = coMaxX;
       for(i=1; i<MaxX; i++)
	  {
           x1 = x2;
	   y1 = -coMaxY;
	   z1 = *sp;
	   sp -= gStep;
	   zc = z1-Min;
	   dp = sp;
	   x2 = coMaxX-(i<<CO_ONESHIFT)/Fact;
           y2 = -coMaxY;
           z2 = *dp;
           dp += del;
           zd = z2-Min;
	   CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
           for(j=1; j<MaxY; j++)
              {
               xa = xc;
               ya = yc;
               za = zc;
               xb = xd;
               yb = yd;
               zb = zd;
               y1 = (j<<CO_ONESHIFT)/Fact-coMaxY;
               z1 = *(dp+gStep);
               zc = z1-Min;
               y2 = y1;
               z2 = *dp;
               dp += del;
               zd = z2-Min;
	       CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       if(inter)
	         {
		  GR_myTriangle(trans, xb, yb, zb, xc, yc, zc, xa, ya, za,
			   width, height, pict);
		  GR_myTriangle(trans, xd, yd, zd, xc, yc, zc, xb, yb, zb,
			   width, height, pict);
		 }
	       else
	         {
		  GR_Triangle(trans, xb, yb, xc, yc, xa, ya, za,
			   width, height, pict);
		  GR_Triangle(trans, xd, yd, xc, yc, xb, yb, zc,
			   width, height, pict);
		 }
              }
	  }
      }
    else if((dAngle -= pi2) <= view->rotXY)
      {
       if(view->DrawGrid)
	 {
	  GR_ColoredGridImage(-coMaxX,-coMaxY,Min,Max,
		0, view, XDim);
	 }
       sp = view->Data ;
       dp = sp;
       y2 = -coMaxY;
       for(i=1; i<MaxY; i++)
	  {
           x1 = -coMaxX;
	   y1 = y2;
	   z1 = *sp;
	   sp += del;
	   zc = z1-Min;
	   dp = sp;
	   x2 = -coMaxX;
           y2 = (i<<CO_ONESHIFT)/Fact-coMaxY;
           z2 = *dp;
           dp += gStep;
           zd = z2-Min;
	   CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
           for(j=1; j<MaxX; j++)
              {
               xa = xc;
               ya = yc;
               za = zc;
               xb = xd;
               yb = yd;
               zb = zd;
               x1 = (j<<CO_ONESHIFT)/Fact-coMaxX;
               z1 = *(dp-del);
               zc = z1-Min;
               x2 = x1;
               z2 = *dp;
               dp += gStep;
               zd = z2-Min;
	       CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       if(inter)
	         {
		  GR_myTriangle(trans, xb, yb, zb, xc, yc, zc, xa, ya, za,
			   width, height, pict);
		  GR_myTriangle(trans, xd, yd, zd, xc, yc, zc, xb, yb, zb,
			   width, height, pict);
		 }
	       else
	         {
		  GR_Triangle(trans, xb, yb, xc, yc, xa, ya, za,
			   width, height, pict);
		  GR_Triangle(trans, xd, yd, xc, yc, xb, yb, zc,
			   width, height, pict);
		 }
              }
	  }
      }
    else
      {
       if(view->DrawGrid)
	 {
	  GR_ColoredGridImage(-coMaxX,coMaxY,Min,Max,
		POS_TOP_Y, view, XDim);
	 }
       sp = view->Data + (MaxY-1)*del;
       dp = sp;
       x2 = -coMaxX;
       for(i=1; i<MaxX; i++)
	  {
           x1 = x2;
	   y1 = coMaxY;
	   z1 = *sp;
	   sp += gStep;
	   zc = z1-Min;
	   dp = sp;
	   x2 = (i<<CO_ONESHIFT)/Fact-coMaxX;
           y2 = coMaxY;
           z2 = *dp;
           dp -= del;
           zd = z2-Min;
	   CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
           for(j=1; j<MaxY; j++)
              {
               xa = xc;
               ya = yc;
               za = zc;
               xb = xd;
               yb = yd;
               zb = zd;
               y1 = coMaxY-(j<<CO_ONESHIFT)/Fact;
               z1 = *(dp-gStep);
               zc = z1-Min;
               y2 = y1;
               z2 = *dp;
               dp -= del;
               zd = z2-Min;
	       CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       if(inter)
	         {
		  GR_myTriangle(trans, xb, yb, zb, xc, yc, zc, xa, ya, za,
			   width, height, pict);
		  GR_myTriangle(trans, xd, yd, zd, xc, yc, zc, xb, yb, zb,
			   width, height, pict);
		 }
	       else
	         {
		  GR_Triangle(trans, xb, yb, xc, yc, xa, ya, za,
			   width, height, pict);
		  GR_Triangle(trans, xd, yd, xc, yc, xb, yb, zc,
			   width, height, pict);
		 }
              }
	  }
      }
   }

/* Redraw  Hidden&Filled */

void XT_RedrawHFilled(Window win, ViewportInfo *view, ViewportInfo *view1,
		XDimInfo *XDim)
   {
    XT_RedrawFilled(win, view, NULL, XDim);
    view->DrawGrid = False;
    XT_RedrawHidden(win, view, NULL, XDim);
   }

/* Redraw  Complex Filled */

void XT_RedrawComFilled(Window win, ViewportInfo *view, ViewportInfo *view1,
		XDimInfo *XDim)
   {
    XDColorInfo *XDC;
    XDHardwareInfo *HWI;
    int i,j, gStep, del, delS;
    double pi2, dAngle;
    myCoor x1, y1, coMaxX, coMaxY, sinXY, cosXY, sinZ, cosZ, moveX, moveY;
    myCoor z1, Fact, xs, ys,MaxX, MaxY, zoomZ, zoomX, zoomY, width, height;
    myCoor x2,y2,z2,*dp, *sp, *dp1, *sp1, Min, Max, MinZ, MaxZ, k;
    myPixel *pict, *trans;
    int xa,ya,za,xb,yb,zb,xc,yc,zc,xd,yd,zd;
    Boolean inter;
    
    XDC = &(XDim->colors);
    if(!XDC->export)
      {
       view->changed = NOCHANGE;
       view1->changed = NOCHANGE;
      }
    HWI = &(XDim->hwInfo);
    dAngle = 2.0*PI;
    pi2 = PI/2.0;
    inter = XDC->interCol;
    sinXY = view->sinXY;
    cosXY = view->cosXY;
    sinZ = view->sinZ;
    cosZ = view->cosZ;
    moveX = view->moveX;
    moveY = view->moveY;
    zoomX = view->zoomX;
    zoomY = view->zoomY;
    zoomZ = view->zoomZ;
    Min = view1->dMin;
    Max = Min+CO_ONE;
    MinZ = view->dMin;
    MaxZ = MinZ+CO_ONE;
    k = view->k;
    gStep = view->gStep;
    MaxX = view->Width;
    MaxY = view->Height;
    if(MaxX != view1->Width || MaxY != view1->Height)
      {
       strcpy( XDim->mess,
	"Real and Imaginary part must have the same X-/Y-Dimensions");
       WD_ViewportMessage(view, XDim);
       return;
      }
    del = MaxX*gStep;
    if(gStep > 1)
      {
       MaxX = MaxX/gStep;
       MaxY = MaxY/gStep;
      }
    if(MaxY > MaxX)
      {
       if(MaxY >1)
	 Fact = MaxY-1;
       else
	 Fact = 1;
       coMaxY = CO_ONE >> 1;
       coMaxX = ((MaxX-1)<<(CO_ONESHIFT-1))/Fact;
      }
    else
      {
       if(MaxX >1)
	 Fact = MaxX-1;
       else
	 Fact = 1;
       coMaxX = CO_ONE >> 1;
       coMaxY = ((MaxY-1)<<(CO_ONESHIFT-1))/Fact;
      }
    width = view->pwidth;
    height = view->pheight;
    pict = view->pict;
    trans = view->trans;
    if((dAngle -= pi2) <= view->rotXY)
      {
       if(view->DrawGrid)
	 {
	  GR_ColoredGridImage(coMaxX,coMaxY,MinZ,MaxZ,
		POS_TOP_X | POS_TOP_Y, view, XDim);
	 }
       delS = (MaxY-1)*del+(MaxX-1)*gStep;
       sp = view->Data + delS;
       dp = sp;
       sp1 = view1->Data + delS;
       dp1 = sp1;
       y2 = coMaxY;
       for(i=1; i<MaxY; i ++)
	  {
           x1 = coMaxX;
	   y1 = y2;
	   z1 = *sp;
	   zc = *sp1-Min;
	   sp -= del;
	   dp = sp;
	   sp1 -= del;
	   dp1 = sp1;
	   x2 = coMaxX;
           y2 = coMaxY-(i<<CO_ONESHIFT)/Fact;
           z2 = *dp;
           dp -= gStep;
           zd = *dp1-Min;
           dp1 -= gStep;;
	   CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
           for(j=1; j<MaxX; j ++)
              {
               xa = xc;
               ya = yc;
               za = zc;
               xb = xd;
               yb = yd;
               zb = zd;
               x1 = coMaxX-(j<<CO_ONESHIFT)/Fact;
               z1 = *(dp+del);
               zc = *(dp1+del)-Min;
               x2 = x1;
               z2 = *dp;
               dp -= gStep;
               zd = *dp1-Min;
               dp1 -= gStep;
	       CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       if(inter)
	         {
		  GR_myTriangle(trans, xb, yb, zb, xc, yc, zc, xa, ya, za,
			   width, height, pict);
		  GR_myTriangle(trans, xd, yd, zd, xc, yc, zc, xb, yb, zb,
			   width, height, pict);
		 }
	       else
	         {
		  GR_Triangle(trans, xb, yb, xc, yc, xa, ya, za,
			   width, height, pict);
		  GR_Triangle(trans, xd, yd, xc, yc, xb, yb, zc,
			   width, height, pict);
		 }
              }
	  }
      }
    else if((dAngle -= pi2) <= view->rotXY)
      {
       if(view->DrawGrid)
	 {
	  GR_ColoredGridImage(coMaxX,-coMaxY,MinZ,MaxZ,
		POS_TOP_X, view, XDim);
	 }
       delS = (MaxX-1)*gStep;
       sp = view->Data + delS;
       dp = sp;
       sp1 = view1->Data + delS;
       dp1 = sp1;
       x2 = coMaxX;
       for(i=1; i<MaxX; i ++)
	  {
           x1 = x2;
	   y1 = -coMaxY;
	   z1 = *sp;
	   sp -= gStep;
	   dp = sp;
	   zc = *sp1-Min;
	   sp1 -= gStep;
	   dp1 = sp1;
	   x2 = coMaxX-(i<<CO_ONESHIFT)/Fact;
           y2 = -coMaxY;
           z2 = *dp;
           dp += del;
           zd = *dp1-Min;
           dp1 += del;
	   CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
           for(j=1; j<MaxY; j++)
              {
               xa = xc;
               ya = yc;
               za = zc;
               xb = xd;
               yb = yd;
               zb = zd;
               y1 = (j<<CO_ONESHIFT)/Fact-coMaxY;
               z1 = *(dp+gStep);
               zc = *(dp1+gStep)-Min;
               y2 = y1;
               z2 = *dp;
               dp += del;
               zd = *dp1-Min;
               dp1 += del;
	       CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       if(inter)
	         {
		  GR_myTriangle(trans, xb, yb, zb, xc, yc, zc, xa, ya, za,
			   width, height, pict);
		  GR_myTriangle(trans, xd, yd, zd, xc, yc, zc, xb, yb, zb,
			   width, height, pict);
		 }
	       else
	         {
		  GR_Triangle(trans, xb, yb, xc, yc, xa, ya, za,
			   width, height, pict);
		  GR_Triangle(trans, xd, yd, xc, yc, xb, yb, zc,
			   width, height, pict);
		 }
              }
	  }
      }
    else if((dAngle -= pi2) <= view->rotXY)
      {
       if(view->DrawGrid)
	 {
	  GR_ColoredGridImage(-coMaxX,-coMaxY,MinZ,MaxZ,
		0, view, XDim);
	 }
       sp = view->Data ;
       dp = sp;
       sp1 = view1->Data;
       dp1 = sp1;
       y2 = -coMaxY;
       for(i=1; i<MaxY; i++)
	  {
           x1 = -coMaxX;
	   y1 = y2;
	   z1 = *sp;
	   sp += del;
	   dp = sp;
	   zc = *sp1-Min;
	   sp1 += del;
	   dp1 = sp1;
	   x2 = -coMaxX;
           y2 = (i<<CO_ONESHIFT)/Fact-coMaxY;
           z2 = *dp;
           dp += gStep;
           zd = *dp1-Min;
           dp1 += gStep;
	   CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
           for(j=1; j<MaxX; j++)
              {
               xa = xc;
               ya = yc;
               za = zc;
               xb = xd;
               yb = yd;
               zb = zd;
               x1 = (j<<CO_ONESHIFT)/Fact-coMaxX;
               z1 = *(dp-del);
               zc = *(dp1-del)-Min;
               x2 = x1;
               z2 = *dp;
               dp += gStep;
               zd = *dp1-Min;
               dp1 += gStep;
	       CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       if(inter)
	         {
		  GR_myTriangle(trans, xb, yb, zb, xc, yc, zc, xa, ya, za,
			   width, height, pict);
		  GR_myTriangle(trans, xd, yd, zd, xc, yc, zc, xb, yb, zb,
			   width, height, pict);
		 }
	       else
	         {
		  GR_Triangle(trans, xb, yb, xc, yc, xa, ya, za,
			   width, height, pict);
		  GR_Triangle(trans, xd, yd, xc, yc, xb, yb, zc,
			   width, height, pict);
		 }
              }
	  }
      }
    else
      {
       if(view->DrawGrid)
	 {
	  GR_ColoredGridImage(-coMaxX,coMaxY,MinZ,MaxZ,
		POS_TOP_Y, view, XDim);
	 }
       delS = (MaxY-1)*del;
       sp = view->Data + delS;
       dp = sp;
       sp1 = view1->Data + delS;
       dp1 = sp1;
       x2 = -coMaxX;
       for(i=1; i<MaxX; i++)
	  {
           x1 = x2;
	   y1 = coMaxY;
	   z1 = *sp;
	   sp += gStep;
	   dp = sp;
	   zc = *sp1-Min;
	   sp1 += gStep;
	   dp1 = sp1;
	   x2 = (i<<CO_ONESHIFT)/Fact-coMaxX;
           y2 = coMaxY;
           z2 = *dp;
           dp -= del;
           zd = *dp1-Min;
           dp1 -= del;
	   CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
	   CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		zoomX,zoomY,zoomZ,moveX,moveY,k);
           for(j=1; j<MaxY; j++)
              {
               xa = xc;
               ya = yc;
               za = zc;
               xb = xd;
               yb = yd;
               zb = zd;
               y1 = coMaxY-(j<<CO_ONESHIFT)/Fact;
               z1 = *(dp-gStep);
               zc = *(dp1-gStep)-Min;
               y2 = y1;
               z2 = *dp;
               dp -= del;
               zd = *dp1-Min;
               dp1 -= del;
	       CO_TransformXY(xc,yc,xs,ys,x1,y1,z1,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       CO_TransformXY(xd,yd,xs,ys,x2,y2,z2,sinXY,cosXY,sinZ,cosZ,
		    zoomX,zoomY,zoomZ,moveX,moveY,k);
	       if(inter)
	         {
		  GR_myTriangle(trans, xb, yb, zb, xc, yc, zc, xa, ya, za,
			   width, height, pict);
		  GR_myTriangle(trans, xd, yd, zd, xc, yc, zc, xb, yb, zb,
			   width, height, pict);
		 }
	       else
	         {
		  GR_Triangle(trans, xb, yb, xc, yc, xa, ya, za,
			   width, height, pict);
		  GR_Triangle(trans, xd, yd, xc, yc, xb, yb, zc,
			   width, height, pict);
		 }
              }
	  }
      }
   }

/* Redraw  Complex&Filled */

void XT_RedrawComHFilled(Window win, ViewportInfo *view, ViewportInfo *view1,
		XDimInfo *XDim)
   {
    XT_RedrawComFilled(win, view, view1, XDim);
    view->DrawGrid = False;
    XT_RedrawHidden(win, view, NULL, XDim);
   }

/* Redraw Cut, view redraw viewport, view1 data viewport */

void XT_RedrawCut2D(Window win, ViewportInfo *view, ViewportInfo *view1,
		XDimInfo *XDim)
   {
    XDHardwareInfo *HWI;
    XDColorInfo *XDC;
    int Dx, Dy, next, Num;
    double fact,Min,Max,sinf,cosf,xn,yn,zn,x1,y1,z1,x2,y2,z2,Sub;
    myPixel *pict, Fore, Col, textCol;
    myCoor coMax, moveX, moveY, Fact;
    myCoor x0, y0, z0, MaxX, MaxY, zoomZ, zoomXY,delx;
    int xa,ya,xb,yb,xc,yc,xd,yd,xt,yt,dx,dy,i,Sum,mode;
    char text[MAX_DIM_LENGTH+16];
    
    HWI = &(XDim->hwInfo);
    XDC = &(XDim->colors);
    if(!XDC->export)
      view->changed = NOCHANGE;
    if(XDC->transCol || XDC->export)
      {
       Fore = view->fore;
       Col = view->axisFore;
       textCol = XDC->textFore;
      }
    else
      {
       Fore = (myPixel)((XDC->colTrans)[view->fore]);
       Col = (myPixel)((XDC->colTrans)[view->axisFore]);
       textCol = (myPixel)((XDC->colTrans)[XDC->textFore]);
      }
    x1 = view->x1;
    y1 = view->y1;
    z1 = view->z1;
    xn = x1;
    yn = y1;
    zn = z1;
    x2 = view->x2;
    y2 = view->y2;
    z2 = view->z2;
    cosf = (x2-x1);
    cosf *= cosf;
    sinf = (y2-y1);
    sinf *= sinf;
    fact = cosf+sinf;
    if(fact == 0.0)
      fact = 1.0;
    fact = 1.0/fact;
    sinf *= fact;
    cosf *= fact;
    coMax = CO_ONE >> 1;
    moveX = view->moveX;
    moveY = view->moveY;    
    zoomXY = (myCoor)sqrt(cosf*(double)(view->zoomX*view->zoomX)
	+sinf*(double)(view->zoomY*view->zoomY));
    zoomZ = view->zoomZ;
    pict = view->pict;
    Dx = view->pwidth;
    Dy = view->pheight;
    MaxX = view1->Width;
    MaxY = view1->Height;
    delx = Dx>>2;
    if(MaxY > MaxX)
      {
       if(MaxY >1)
	 Fact = MaxY-1;
       else
	 Fact = 1;
      }
    else
      {
       if(MaxX >1)
	 Fact = MaxX-1;
       else
	 Fact = 1;
      }
    Min = view1->Min;
    Max = view1->Max;
    if(Max > Min)
      fact = (CO_ONE-1)/(Max - Min);
    else
      fact = 1.0;
    if(Min <0.0 && Max > 0.0)
      Sub = 0.0;
    else
      Sub = Min;
    z0 = (myCoor)(fact * (z1 - Sub));
    CO_TransformHW(xb,yb,0,z0,zoomXY,zoomZ,moveX,moveY);
    xb -= delx;
    do
      {
       xa = xb;
       ya = yb;
       next=CO_NextStep(x2,y2,&xn,&yn,&zn, view1);
       x0 = (myCoor)((xn-x1)*CO_ONE)/Fact;
       y0 = (myCoor)((yn-y1)*CO_ONE)/Fact;
       z0 = (myCoor)(fact * (zn - Sub));
       x0 = (myCoor)sqrt((double)(x0*x0+y0*y0));
       CO_TransformHW(xb,yb,x0,z0,zoomXY,zoomZ,moveX,moveY);
       xb -=delx;
       HW_PutLine(Fore, xa, ya, xb, yb, Dx, Dy, pict, HWI);
      }
    while(next);
    if(view->DrawGrid)
      {
       if(Sub == 0.0)
         {
          /* Draw Zero Line */
          CO_TransformHW(xa,ya,0,0,zoomXY,zoomZ,moveX,moveY);
          xa -= delx;
          CO_TransformHW(xb,yb,x0,0,zoomXY,zoomZ,moveX,moveY);
          xb -= delx;
	  HW_PutLine(Col, xa, ya, xb, yb, Dx, Dy, pict, HWI);
         }
       z0 = (myCoor)(fact * (Min - Sub));
       CO_TransformHW(xa,ya,0,z0,zoomXY,zoomZ,moveX,moveY);
       xa -= delx;
       z0 = (myCoor)(fact * (Max - Sub));
       CO_TransformHW(xb,yb,x0,z0,zoomXY,zoomZ,moveX,moveY);
       xb -= delx;
       HW_PutLine(Col, xa, ya, xa, yb, Dx, Dy, pict, HWI);
       HW_PutLine(Col, xa, yb, xb, yb, Dx, Dy, pict, HWI);
       HW_PutLine(Col, xb, yb, xb, ya, Dx, Dy, pict, HWI);
       HW_PutLine(Col, xb, ya, xa, ya, Dx, Dy, pict, HWI);
       dx = 0.01*(xb-xa);
       if(dx < 3)
         dx = 3;
       dy = yb-ya;
       Num = view->ticsZ-1;
       if(Num <= 0)
         Num = 1;
       xc = xa-dx;
       Sum = 0;
       xt = xc-2*dx;
       mode = ALIGN_CENTER_Y | ALIGN_RIGHT;
       if(view1->baseZ == 0.0)
         fact = 1.0;
       else
         fact = view1->baseZ;
       z1 = Min/fact;
       fact = (Max-Min)/(Num*fact);
       for(i=0; i<= Num; i++)
          {
           yc = ya+Sum/Num;
           Sum += dy;
	   HW_PutLine(Col, xa, yc, xc, yc, Dx, Dy, pict, HWI);
           sprintf(text,"%1.2f",z1);
           z1 += fact;
           HW_PutText(text, xt, yc, Dx, Dy, textCol, mode, pict, HWI);  
          }
       if(view->DrawAxisText)
         {
          HW_TextExtent(text, &xd, &yd, HWI);
          if(view1->baseZ == 1.0)
            sprintf(text, "[%s]", view1->DimZ);
          else if(view1->baseZ > 0.01 && view1->baseZ < 100.0)
            sprintf(text, "[%1.2f %s]", view1->baseZ, view1->DimZ);
          else
            sprintf(text, "[%1.2e %s]", view1->baseZ, view1->DimZ);
          yt = (3*yb+ya) >> 2;
          xt = xt-(xd<<1);
          HW_PutText(text, xt, yt, Dx, Dy, textCol, mode, pict, HWI);  
         }
       yc = ya-dx;
       yt = ya-3*dx;
       yd = ya;
       dx = xb-xa;
       mode = ALIGN_TOP | ALIGN_CENTER_X;
       Num = view->ticsX-1;
       if(Num <= 0)
         Num = 1;
       Sum = 0;
       x1= 0.0;
       x2 = (view->x2-view->x1);
       y2 = (view->y2-view->y1);
       if(view1->baseX > view1->baseY)
         fact = view1->baseX;
       else
         fact = view1->baseY;
       if(fact == 0.0)
         fact = 1.0;
       fact = sqrt(x2*x2+y2*y2)/(Num*fact);
       for(i=0; i<= Num; i++)
          {
           xc = xa+Sum/Num;
           Sum += dx;
	   HW_PutLine(Col, xc, ya, xc, yc, Dx, Dy, pict, HWI);
           sprintf(text,"%1.2f",x1);
           x1 += fact;
           HW_PutText(text, xc, yt, Dx, Dy, textCol, mode, pict, HWI);  
          }
       if(view->DrawAxisText)
         {
          HW_TextExtent(text, &dx, &dy, HWI);
          if(view1->baseX == 1.0)
            sprintf(text, "[%s]", view1->DimX);
          else if(view1->baseX > 0.01 && view1->baseX < 100.0)
            sprintf(text, "[%1.2f %s]", view1->baseX, view1->DimX);
          else
            sprintf(text, "[%1.2e %s]", view1->baseX, view1->DimX);
          xt = (3*xb+xa) >> 2;
          yt = ya- 4*dy;
          HW_PutText(text, xt, yt, Dx, Dy, textCol, mode, pict, HWI);  
         }
      }
   }

void *XT_GetNamedRedraw(char *rname, XDimInfo *XDim)
   {
    void *RetFunc = NULL;
    
    if(!strcmp(rname, RED_GRID))
      RetFunc = (void*)XT_RedrawGrid ;
    else if(!strcmp(rname, RED_LINES))
      RetFunc = (void*)XT_RedrawLines;
    else if(!strcmp(rname, RED_HIDDEN))
      RetFunc = (void*)XT_RedrawHidden;
    else if(!strcmp(rname, RED_FILLED))
      RetFunc = (void*)XT_RedrawFilled;
    else if(!strcmp(rname, RED_HFILLED))
      RetFunc = (void*)XT_RedrawHFilled;
    else if(!strcmp(rname, RED_CUT))
      RetFunc = (void*)XT_RedrawCut2D;
    else if(!strcmp(rname, RED_COMP))
      RetFunc = (void*)XT_RedrawComFilled;
    else if(!strcmp(rname, RED_HCOMP))
      RetFunc = (void*)XT_RedrawComHFilled;
    else sprintf(XDim->error, "No such Redraw Method : %s !!", rname);

    return(RetFunc);
   }

int XT_GetRedrawIndex(void *RMethod, XDimInfo *XDim)
   {
    static int ret;
    
    if(RMethod == (funcCast)XT_RedrawGrid)
      ret = 0;
    else if(RMethod == (funcCast)XT_RedrawLines)
      ret = 1;
    else if(RMethod == (funcCast)XT_RedrawHidden)
      ret = 2;
    else if(RMethod == (funcCast)XT_RedrawFilled)
      ret = 3;
    else if(RMethod == (funcCast)XT_RedrawHFilled)
      ret = 4;
    else if(RMethod == (funcCast)XT_RedrawCut2D)
      ret = 5;
    else if(RMethod == (funcCast)XT_RedrawComFilled)
      ret = 6;
    else if(RMethod == (funcCast)XT_RedrawComHFilled)
      ret = 7;

    return(ret);
   }
