/*
**++
**  FACILITY:
**      NEWSCONTROL
**
**  ABSTRACT:
**      This module interprets Control: messages.
**
**  ENTRY POINTS:
**	int parse_control_item(argv,fn,switches)
**		char *argv[], *fn; int switches;
**
**  AUTHOR:
**      Geoff Huston
**
**  COPYRIGHT:
**      Copyright  1990,1991
**
**  VERSION
**	V6.1	23-07-91
**		Bill Fenner       wcf@ecl.psu.edu
**		> We got a few newgroup's last night, like "newgroup comp.sys.handhelds
** 		> unmoderated".  ANU/NEWS created the newsgroups _moderated_, it seems as though
**     		> it only checks for the presence of a second argument, not the text of it.
**    		Actually, it turns out that it checks for an argument with the string
**     		"moderated" anywhere in it.  Too bad "unmoderated" has "moderated" in it.
**     		[change made] to make the comparison exact, and also to log
**     		moderated/unmoderated status in the log file and in the subject of the
**     		mail message.
**	V6.1	26-Feb-1992
**		Mark Pizzolato	mark@infocomm.com
**		Allow the /ACCEPT command qualifier to override missing
**		"Approved" header lines.
**		This is determined by referencing the more general global
**		variable itm_approved (set in NEWSADD) instead of explicitly 
**		referencing the APPROVED header line.
**	V6.1	26-Jul-1992
**		Bill Fenner  wcf@ecl.psu.edu
**		Added support for descriptions in newgroup control messages
**		of the form:
**		For your newsgroups file:
**		news.group.name		Description
**		This is the format that David Lawrence uses for his control
**		messages, so all "official" USENET newsgroups will now get
**		their descriptions automatically.
**	V6.1	13-Sep-1992
**		Bill Fenner  wcf@ecl.psu.edu
**		Modified newgrp description code to skip blank lines early
**		in the loop -- somehow, the strncmp compared true on a
**		blank line, so you'd get newsgroups with descriptions of
**		"your newsgroups file:\n".  Also checked a bit more strictly
**		for the proper format.
**	V6.1b8  22-Dec-1993	mark.martinec@ijs.si
**		convert calls to RMS to use new sys_* macros for error handling 
**	V6.1b9  25-May-1994  Charles Bailey  bailey@genetics.upenn.edu
**	 - add dummy flk param to do_oitem calls
**      V6.1b9  24-Feb-1995	Bob Sloane sloane@kuhub.cc.ukans.edu
**       - added extended newsgroup names for control items:
**         control.type.hier, so that cancel, rmgroup, etc messages
**         can be put in groups like control.cancel.alt, or control.rmgroup.
**	V6.1b9   3-Mar-1995	mark.martinec@ijs.si
**	 - replace strncpy with util_idcpy to match the way it is done
**	   in NNTP_SERVER.C (is item ID size 60 or 59+'\0' characters ?)
**--
**/

#ifdef vaxc
#module NEWSCONTROL "V6.1"
#endif

#define _NEWSCONTROL_C
#define module_name "NEWSCONTROL"

#include "newsinclude.h"
#include "newsextern.h"

#if NNTP_CLIENT_ONLY

int parse_control_item(argv,fn,switches,newsgroups)
  char *argv[], *fn;
  int switches;
  char *newsgroups;
{
  return(1);
}

#else

static char mail_scratch[256] = "";

/*
 *  control_version
 *
 *  Mail the News version level back to poster
 */

static int control_version(argv,fn,switches)
  char *argv[], *fn;
  int switches;
{
  char vms_address[256], *mail_replyto;
  FILE *fpm;

  mail_replyto = argv[REPLY_TO];
  if (!mail_replyto || !*mail_replyto) mail_replyto = argv[FROM];
  if (!mail_replyto || !*mail_replyto) return(NORETURNADDRESS);

  strcpy(vms_address,add_transform(mail_replyto));
  if (!*mail_scratch) sprintf(mail_scratch,"SYS$SCRATCH:NEWS_MAIL_%X.TMP",getpid());
  if (!(fpm = fopen(mail_scratch,"w","rfm=var","rat=cr"))) return(NOSCRATCHFILE);

  fprintf(fpm,"Response-To-Control: %s message\n",argv[CONTROL]);
  fprintf(fpm,"Responding-System: %s\n",news_node);
  fprintf(fpm,"News-Software: %s\n",NEWS_VERSION);
  fclose(fpm);
  sprintf(err_oline,"Control: version - Forward to %s",vms_address);
  call_mail(mail_scratch,err_oline,USENET,err_oline);
  delete(mail_scratch);
  return(CTLOK);
}

/*
 *  control_senduuname
 *
 *  Mail the UUCP control file back to poster - mail error text for VMS
 */

static int control_senduuname(argv,fn,switches)
  char *argv[], *fn;
  int switches;
{
  char vms_address[256], *mail_replyto;
  FILE *fpm;

  mail_replyto = argv[REPLY_TO];
  if (!mail_replyto || !*mail_replyto) mail_replyto = argv[FROM];
  if (!mail_replyto || !*mail_replyto) return(NORETURNADDRESS);

  strcpy(vms_address,add_transform(mail_replyto));
  if (!*mail_scratch) sprintf(mail_scratch,"SYS$SCRATCH:NEWS_MAIL_%X.TMP",getpid());
  if (!(fpm = fopen(mail_scratch,"w","rfm=var","rat=cr"))) return(NOSCRATCHFILE);

  fprintf(fpm,"Response-To-Control: %s message\n",argv[CONTROL]);
  fprintf(fpm,"Responding-System: %s\n",news_node);
  fprintf(fpm,"News-Software: %s\n",NEWS_VERSION);
  fprintf(fpm,"This is a VAX/VMS NEWS, not configured with UUCP\n");
  fclose(fpm);
  sprintf(err_oline,"Control: senduuname - Forward to %s",vms_address);
  call_mail(mail_scratch,err_oline,USENET,err_oline);
  delete(mail_scratch);
  return(CTLOK);
}

/*
 *  control_sendsys
 *
 *  Mail the NEWS.SYS control file back to poster
 */

static int control_sendsys(argv,fn,switches)
  char *argv[], *fn;
  int switches;
{
  char il[512], vms_address[256], *mail_replyto;
  FILE *fpm, *fpr;

  mail_replyto = argv[REPLY_TO];
  if (!mail_replyto || !*mail_replyto) mail_replyto = argv[FROM];
  if (!mail_replyto || !*mail_replyto) return(NORETURNADDRESS);
  if (!strchr(mail_replyto,'@')) return(NORETURNADDRESS);

  strcpy(vms_address,add_transform(mail_replyto));
  if (!*mail_scratch) sprintf(mail_scratch,"SYS$SCRATCH:NEWS_MAIL_%X.TMP",getpid());
  if (!(fpm = fopen(mail_scratch,"w","rfm=var","rat=cr"))) return(NOSCRATCHFILE);

  fprintf(fpm,"Response-To-Control: %s message\n",argv[CONTROL]);
  fprintf(fpm,"Responding-System: %s\n",news_node);
  fprintf(fpm,"News-Software: %s\n",NEWS_VERSION);
  sysprv();
  if (!(fpr = fopen("NEWS_MANAGER:NEWS.SYS","r")))
    fprintf(fpm,"No-SYS: No NEWS.SYS file defined on this node\n");
  else {
    while (fgets(il,510,fpr)) if (*il != '#') fputs(il,fpm);
    fclose(fpr);
    }
  nosysprv();
  fclose(fpm);
  sprintf(err_oline,"Control: sendsys - forward to %s",vms_address);
  call_mail(mail_scratch,err_oline,USENET,err_oline);
  delete(mail_scratch);
  return(CTLOK);
}

/*
 * append_control
 *
 * Appends the text of a control message to the mail message sent to USENET.
 *
 * Author: William C. Fenner <wcf@ecl.psu.edu>
 */

static void append_control(from,to)
  char *from;
  FILE *to;
{
  FILE *f;
  char il[512];

  fprintf(to,"$! Control message follows:\n$!\n");
  if((f=fopen(from,"r"))==NULL) {
    fprintf(to,"$! * Could not open input file %s\n",from);
    return;
    }
  while (fgets(il,510,f)) fprintf(to,"$! %s",il);
  fclose(f);
  return;
  }

/*
 *  control_newgroup
 *
 *  Create new newsgroup control message - if auto execution enabled then
 *  new newsgroup is created, otherwise DCL is posted to USENET
 */

static int control_newgroup(argv,fn,switches)
  char *argv[], *fn;
  int switches;
{
  char *c, locname[132];
  char descrip[132];
  char xbuf[512];
  unsigned int g;
  int mod = 0;
  FILE *fpm = 0;
  FILE *fpa;

  if (!itm_approved) return(NONAPPROVEDCONTROL);

  c = argv[CONTROL] + 8;
  while (isspace(*c)) c++;
  if (!*c) return(NONEWSGROUPFIELD);

  strcpy(locname,c);
  c = locname;
  while (isgraph(*c)) c++;
  *c++ = '\0';

  *descrip='\0';
  if ((fpa = fopen(fn,"r")) != 0) {
	int flag=0, lnl = strlen(locname);

	while (fgets(xbuf,sizeof xbuf - 1,fpa)) {
		if (!strcmp(xbuf,"For your newsgroups file:\n")) {
			flag=1;
			continue;
		}
		if (flag) {
			char *p = xbuf;
			chop_str(p,'\n');
			if (!*p) continue;	/* skip blank lines early */
			if (!strncmp(p,locname,lnl)) {	/* if the newsgroup name is at the beginning of the line */
				while (*p && !isspace(*p)) p++;	/* skip ng name */
				while (*p && isspace(*p)) p++;	/* skip whitespace */
				if (*p) {	/* if there's a description, use it */
					strcpy(descrip,p);
					break;
				} else
					flag=0;	/* Malformed... */
			} else {
				while (*p && isspace(*p)) p++;
				if (*p) flag=0;		/* allow blank lines */
			}
		}
	}
	fclose(fpa);
  }

  chop_str(c,' ');
  mod = ((*c) && (strcmp(lower_case(c),"moderated")==0)) ? NEWS_M_MAILMODERATE : 0;
  util_cvrt(locname,locname);
  if (!*mail_scratch) sprintf(mail_scratch,"SYS$SCRATCH:NEWS_MAIL_%X.TMP",getpid());
  if ((!(g = ga_exact_name(locname))) && (switches & EXECUTE_CONTROL)) {
    int sc = auto_cre_grp;

    auto_cre_grp = 1;
    do_newg(locname,0,1);
    auto_cre_grp = sc;
    if ((g = ga_exact_name(locname)) && (fpm = fopen(mail_scratch,"w","rfm=var","rat=cr"))) {
      fprintf(fpm,"Control: newgroup - created %s\n",ga[g]->grp_name);
      if (*descrip) {
        fprintf(fpm,"Control: newgroup - modified %s/TITLE=\"%s\"\n",
		ga[g]->grp_name,descrip);
        strncpy(ga[g]->grp_topic,descrip,124);
        ga[g]->grp_topic[124]='\0';
      }
      update_newsgrp(g);
      }
    }
  if (g) {
    if (   (ga[g]->grp_flags & NEWS_M_LOCAL) 
        || ((ga[g]->grp_flags & NEWS_M_MAILMODERATE) != mod)
        || ga[g]->grp_life) {
      if (switches & EXECUTE_CONTROL) {
        if (ga[g]->grp_flags & NEWS_M_LOCAL) ga[g]->grp_flags &= ~NEWS_M_LOCAL;
        if (mod != (ga[g]->grp_flags & NEWS_M_MAILMODERATE)) {
          ga[g]->grp_flags &= ~NEWS_M_MAILMODERATE;
          ga[g]->grp_flags |= mod;
          }
        if (ga[g]->grp_life) ga[g]->grp_life = 0;
        update_newsgrp(g);
        }
      if (fpm || (fpm = fopen(mail_scratch,"w","rfm=var","rat=cr"))) {
        if (switches & EXECUTE_CONTROL) {
	  fprintf(fpm,"Control: newgroup - modified %s/NOLOCAL%s/HOLD\n",ga[g]->grp_name,(mod ? "/MOD" : ""));			/* modified */
        } else {
          fprintf(fpm,"$! Generated Command file from newgroup message:\n");
          fprintf(fpm,"$! use EXTRACT/NOHEAD then execute this file\n$!\n");
          fprintf(fpm,"$ DEFINE/USER NEWSRC NL:\n");
          fprintf(fpm,"$ NEWS/NOSCREEN\n");
          if (ga[g]->grp_flags & NEWS_M_LOCAL)
	    fprintf(fpm,"Set NEWSGROUP \"%s\"/NOLOCAL\n",ga[g]->grp_name);
          if (mod)
	    fprintf(fpm,"Set NEWSGROUP \"%s\"/MOD\n\n",ga[g]->grp_name);
          else if (!mod && (ga[g]->grp_flags & NEWS_M_MAILMODERATE))
            fprintf(fpm,"Set NEWSGROUP \"%s\"/NOMOD\n",ga[g]->grp_name);
          if (ga[g]->grp_life)
            fprintf(fpm,"Set NEWSGROUP \"%s\"/HOLD\n",ga[g]->grp_name);
	  if (*descrip)
	    fprintf(fpm,"Set NEWSGROUP \"%s\"/TITLE=\"%s\"\n",
			ga[g]->grp_name,quotes(descrip));
          fprintf(fpm,"EXIT\n");
          }
        }
      }
    }
  else if (fpm || (fpm = fopen(mail_scratch,"w","rfm=var","rat=cr"))) {
    fprintf(fpm,"$! Generated Command file from newgroup message:\n");
    fprintf(fpm,"$! use EXTRACT/NOHEAD then execute this file\n$!\n");
    fprintf(fpm,"$ DEFINE/USER NEWSRC NL:\n");
    fprintf(fpm,"$ NEWS/NOSCREEN\n");
    fprintf(fpm,"CREATE NEWSGROUP/NOCONFIRM \"%s\"",locname);
    if (mod) fprintf(fpm,"/MOD");
    fprintf(fpm,"\n");
    if (mod) fprintf(fpm,"\n");
    if (*descrip)
      fprintf(fpm,"Set NEWSGROUP \"%s\"/TITLE=\"%s\"\n",
        locname,quotes(descrip));
    fprintf(fpm,"EXIT\n");
    }
  else return(NOSCRATCHFILE);

  if (fpm) {
    char subjline[256];

    append_control(fn,fpm);
    fclose(fpm);
    sprintf(err_oline,"\tControl: newgroup %s%s - Mail to Usenet\n",locname,mod ? " moderated" : "");
    sprintf(subjline,"NEWS: newgroup %s%s",locname,mod ? " moderated" :"");
    sprintf(err_oline,"\tControl: newgroup %s - Mail to Usenet\n",locname);
    sprintf(subjline,"NEWS: newgroup %s",locname);
    call_mail(mail_scratch,subjline,USENET,err_oline);
    delete(mail_scratch);
    return(CTLOK);
    }
  if (g) return(CTLOK);
  return(NEWGROUPERROR);
}

/*
 *      control_rmgroup
 *
 *      Delete a newsgroup control message - if auto execution enabled then
 *      newsgroup is deleted, otherwise DCL is posted to USENET.
 */

static int control_rmgroup(argv,fn,switches)
  char *argv[], *fn;
  int switches;
{
  char *c, locname[132];
  int g;
  FILE *fpm = 0;

  if (!itm_approved) return(NONAPPROVEDCONTROL);

  c = argv[CONTROL] + 7;
  while (isspace(*c)) c++;
  if (!*c) return(NONEWSGROUPFIELD);

  strcpy(locname,c);
  while (isgraph(*c)) c++;
  *c = '\0';
  util_cvrt(locname,locname);
  if (!*mail_scratch) sprintf(mail_scratch,"SYS$SCRATCH:NEWS_MAIL_%X.TMP",getpid());

  if (!(g = ga_exact_name(locname))) return(NORMNEWSGROUP);

  if (ga[g]->grp_flags & (NEWS_M_LOCAL | NEWS_M_MAILGROUP)) return(LOCALNEWSGROUP);

  fpm = fopen(mail_scratch,"w","rfm=var","rat=cr");
  if (switches & EXECUTE_DELETE) {
    delgrp_file(g);
    if (fpm) fprintf(fpm,"Control: rmgroup %s - Deleted\n",locname);
    }
  else if (switches & EXECUTE_LOCAL) {
    ga[g]->grp_flags |= NEWS_M_LOCAL;
    update_newsgrp(g);
    if (fpm) fprintf(fpm,"Control: rmgroup %s - Set /LOCAL\n",locname);
    }
  else if (switches & EXECUTE_HOLD) {
    ga[g]->grp_life = 14;
    update_newsgrp(g);
    if (fpm) fprintf(fpm,"Control: rmgroup %s - Set /HOLD=14\n",locname);
    }
  else if (fpm) {
    fprintf(fpm,"$ ! Generated Command file from rmgroup message\n");
    fprintf(fpm,"$ ! use EXTRACT/NOHEAD then execute this file\n$ !\n");
    fprintf(fpm,"$ DEFINE/USER NEWSRC NL:\n");
    fprintf(fpm,"$ NEWS/NOSCREEN\n");
    fprintf(fpm,"! Remove NEWSGROUP %s (or set to /LOCAL)\n",ga[g]->grp_name);
    fprintf(fpm,"!SET NEWSGROUP \"%s\"/LOCAL\n",ga[g]->grp_name);
    fprintf(fpm,"! Or:\n");
    fprintf(fpm,"DELETE NEWSGROUP \"%s\"/NoConfirm\n",ga[g]->grp_name);
    fprintf(fpm,"EXIT\n");
    }
  else return(NOSCRATCHFILE);

  if (fpm) {
    char subjline[256];

    append_control(fn,fpm);
    fclose(fpm);
    sprintf(err_oline,"\tControl: rmgroup %s - Mail to Usenet\n",locname);
    sprintf(subjline,"NEWS: rmgroup %s",locname);
    call_mail(mail_scratch,subjline,USENET,err_oline);
    delete(mail_scratch);
    }
  return(CTLOK);
}

/*
 *  control_checkgroups
 *
 *  Check the local newsgroup set against the network distribution list,
 *  removing any detected inconsistencies
 */

/*
 * Domain support code - checkgroup messages generally refer to a "family" of newsgroups
 * which have a common root (or roots) in the newsgroup name hierarchy. This code supports
 * these common domains
 */

static struct domain_list {
  char *dom_name;
  struct domain_list *dom_next;
  } *dom_head = 0;

static void clear_domain()
{
  struct domain_list *tmp = dom_head;

  while (dom_head) {
    news_free(dom_head->dom_name);
    tmp = dom_head->dom_next;
    news_free(dom_head);
    dom_head = tmp;
    }
}

static struct domain_list *match_domain_list(n)
  char *n;
{
  struct domain_list *tmp = dom_head;

  while (tmp) {
    if (wild_match(tmp->dom_name,n)) break;
    tmp = tmp->dom_next;
    }
  return(tmp);
}

static void add_domain_list(s)
  char *s;
{
  struct domain_list *tmp;

  tmp = (struct domain_list *) news_malloc(sizeof *tmp);
  tmp->dom_next = dom_head;
  dom_head = tmp;
  tmp->dom_name = (char *) news_malloc(strlen(s) + 2);
  strcpy(tmp->dom_name,s);
  strcat(tmp->dom_name,".");
}

static int domain_match(lname)	char *lname;
{
  char *cmpname, *cp;
  struct domain_list *d_ent;

  strcpy((cmpname = (char *) news_malloc(strlen(lname) + 3)),lname);
  strcat(cmpname,".*");
  do {
    if ((d_ent = match_domain_list(cmpname)) != 0) {
      if ((cp = strrchr(cmpname,'*')) != 0) *cp = '\0';
      strcpy(d_ent->dom_name,cmpname);
      break;
      }
    if ((cp = strrchr(cmpname,'.')) != 0) {
      if (*(cp + 1) == '*') {
        *cp = '\0';
        if (!(cp = strrchr(cmpname,'.'))) break;
        }
      strcpy(cp,".*");
      }
    } while (cp);
  return((int) d_ent);
}

static void add_domain(lname)
  char *lname;
{
  if (!domain_match(lname)) add_domain_list(lname);
}

static int control_checkgroups(argv,fn,switches)
  char *argv[], *fn;
  int switches;
{
  char locname[132], xbuf[512];
  FILE *fpr, *fpm = 0;
  GRP_PTR *ca;
  int i, g, mail_msg = 0, indx, tpc;
  struct glist {
        struct glist *gnext;
        char *gtitle;
        int glmod;
        char glname[SUBJLEN];
        } *add_head = 0, *del_head = 0, *at = 0, *dt = 0, *tmp;

  if (!itm_approved) return(NONAPPROVEDCONTROL);
  if (!(fpr = fopen(fn,"r"))) return(NOACCESSINFILE);

  if (!*mail_scratch) sprintf(mail_scratch,"SYS$SCRATCH:NEWS_MAIL_%X.TMP",getpid());
  if (!(fpm = fopen(mail_scratch,"w","rfm=var","rat=cr"))) {
    fclose(fpr);
    return(NOSCRATCHFILE);
    }

  if (switches & EXECUTE_CONTROL)
    fprintf(fpm,"Generated Log file from checkgroup message\n");
  else {
    fprintf(fpm,"$! Generated Command file from checkgroup message\n");
    fprintf(fpm,"$! use EXTRACT/NOHEAD then execute this item\n$!\n");
    fprintf(fpm,"$ DEFINE/USER NEWSRC NL:\n");
    fprintf(fpm,"$ NEWS/NOSCREEN\n");
    }

  ca = (GRP_PTR *) news_malloc((ga_size + 1) * (sizeof *ca));
  for (i = 1 ; i <= ga_size ; ++i) ca[i] = ga[i];

  while ((fgets(xbuf,512,fpr)) && (*xbuf != '\n'));      /* skip news headers */
  while (fgets(xbuf,510,fpr)) {                            /* newsgroup entry */
    if (*xbuf == '\n') continue;
    /*
     * checkgroup messages sometimes have imbedded comments that start with '!'
     */
    if ('!' == *xbuf) continue;
    chop_str(xbuf,'\n');
    if ((indx = strcspn(xbuf," \t")) != 0) xbuf[indx] = '\0';
    util_cvrt(locname,xbuf);                             /* extract newsgroup */
    add_domain(locname);

    if ((tpc = indx) != 0) {				/* extract title */
      tpc++;
      while (isspace(xbuf[tpc])) tpc++;
      }

    if ((g = ga_exact_name(locname)) != 0
     && !(ga[g]->grp_flags & NEWS_M_MAILGROUP)) {
      ca[g] = 0;

      if (tpc && xbuf[tpc]) {                          /* update topic string */
        if (strncmp(ga[g]->grp_topic,&xbuf[tpc],124)) {
          ++mail_msg;
          if (switches & EXECUTE_CONTROL) {
            strncpy(ga[g]->grp_topic,&xbuf[tpc],124);
            ga[g]->grp_topic[124] = '\0';
            fprintf(fpm,"Set Newsgroup %s/Title=\"%s\"\n",locname,&xbuf[tpc]);
            }
          else fprintf(fpm,"SET NEWSGROUP \"%s\"/TITLE=\"%s\"\n",locname,quotes(&xbuf[tpc]));
          }
        }

      if (tpc && substrcmp(&xbuf[tpc],"(Moderated)")) {          /* moderated */
        if (!(ga[g]->grp_flags & NEWS_M_MAILMODERATE)) {    /* set mod status */
          ++mail_msg;
          if (switches & EXECUTE_CONTROL) {
            ga[g]->grp_flags |= NEWS_M_MAILMODERATE;
            fprintf(fpm,"Set Newsgroup %s/Moderated\n",locname);
            }
          else fprintf(fpm,"SET NEWSGROUP \"%s\"/MOD\n\n",locname);
          }
        }

      else if (ga[g]->grp_flags & NEWS_M_MAILMODERATE) {       /* unmoderated */
        ++mail_msg;                                       /* reset mod status */
        if (switches & EXECUTE_CONTROL) {
          ga[g]->grp_flags &= ~NEWS_M_MAILMODERATE;
          fprintf(fpm,"Set Newsgroup %s/Nomoderated\n",locname);
          }
        else fprintf(fpm,"SET NEWSGROUP \"%s\"/NOMOD\n",locname);
        }

      time((time_t *) &(ga[g]->grp_entdate));     /* update time stamp */
      update_newsgrp(g);
      }

    else if (!g) {                                        /* no local newsgroup match */
      ++mail_msg;
      if (switches & EXECUTE_CONTROL) {
        tmp = (struct glist *) news_malloc(sizeof *tmp);
        if (at) at->gnext = tmp; else add_head = tmp;
        at = tmp;
        at->gtitle = 0;
        at->gnext = 0;
        strcpy(at->glname,locname);
        at->glmod = 0;
        }
      else fprintf(fpm,"CREATE NEWSGROUP/NOCONFIRM \"%s\"\n",locname);

      if (tpc && xbuf[tpc]) {                          /* update topic string */
        if (switches & EXECUTE_CONTROL) {
          at->gtitle = (char *) news_malloc(strlen(&xbuf[tpc]) + 1);
          strcpy(at->gtitle,&xbuf[tpc]);
          }
        else fprintf(fpm,"SET NEWSGROUP \"%s\"/TITLE=\"%s\"\n",locname,quotes(&xbuf[tpc]));
        }

      if (tpc && substrcmp(&xbuf[tpc],"(Moderated)")) {          /* moderated */
        if (switches & EXECUTE_CONTROL) at->glmod = 1;
        else fprintf(fpm,"SET NEWSGROUP \"%s\"/MOD\n\n",locname);
        }
      }
    } /*----- END of pass through input file */
  fclose(fpr);

  g = 1;
  while (g <= ga_size) {
    if (ca[g] && domain_match(ga[g]->grp_name) && !(ga[g]->grp_flags & (NEWS_M_LOCAL | NEWS_M_MAILGROUP))) {
      ++mail_msg;
      if (switches & EXECUTE_CONTROL) {
        tmp = (struct glist *) news_malloc(sizeof *tmp);
        if (dt) dt->gnext = tmp; else del_head = tmp;
        dt = tmp;
        dt->gnext = 0;
        strcpy(dt->glname,ga[g]->grp_name);
        }
      else {
        fprintf(fpm,"! Remove NEWSGROUP %s\n",ga[g]->grp_name);
        fprintf(fpm,"SET NEWSGROUP \"%s\"/HOLD=14\n",ga[g]->grp_name);
        fprintf(fpm,"! Or:\n");
        fprintf(fpm,"!SET NEWSGROUP \"%s\"/LOCAL\n",ga[g]->grp_name);
        fprintf(fpm,"!DELETE NEWSGROUP \"%s\"/NOCONFIRM\n",ga[g]->grp_name);
        }
      }
    g++;
    }
  news_free(ca);

  at = add_head;
  while (add_head) {
    int sc = auto_cre_grp;

    auto_cre_grp = 1;
    tmp = add_head ;
    do_newg(tmp->glname,0,1);
    auto_cre_grp = sc;
    if ((g = ga_exact_name(tmp->glname)) != 0) {
      ++mail_msg;
      fprintf(fpm,"Created new Network Newsgroup: %s",tmp->glname);
      if (tmp->glmod) {
        fprintf(fpm,"/Moderated");
        ga[g]->grp_flags |= NEWS_M_MAILMODERATE;
        }
      if (tmp->gtitle) {
        strncpy(ga[g]->grp_topic,tmp->gtitle,124);
        ga[g]->grp_topic[124] = '\0';
        fprintf(fpm,"/Title=\"%s\"",tmp->gtitle);
        }
      update_newsgrp(g);
      tmp->glmod = 0;
      fprintf(fpm,"\n");
      }
    else tmp->glmod = 1;
    add_head = add_head->gnext;
    }

  while (del_head) {
    tmp = del_head;
    ++mail_msg;
    if ((g = ga_exact_name(tmp->glname)) != 0) {
      if (switches & EXECUTE_DELETE) {
        fprintf(fpm,"Delete Newsgroup %s\n",tmp->glname);
        delgrp_file(g);
        }
      else if (switches & EXECUTE_LOCAL) {
        fprintf(fpm,"Set Newsgroup %s/LOCAL\n",tmp->glname);
        ga[g]->grp_flags |= NEWS_M_LOCAL;
        update_newsgrp(g);
        }
      else {
        fprintf(fpm,"Set Newsgroup %s/HOLD=14\n",tmp->glname);
        ga[g]->grp_life = 14;
        update_newsgrp(g);
        }
      }
    else fprintf(fpm,"Cannot locate Newsgroup %s (Delete required)\n",tmp->glname);
    del_head = del_head->gnext;
    news_free(tmp);
    }

  add_head = at;
  while (add_head) {
    tmp = add_head ;
    if (tmp->glmod) {
      ++mail_msg;
      fprintf(fpm,"\n!** NO Create of Newsgroup: %s - SYS filter reject\n",tmp->glname);
      }
    add_head = add_head->gnext;
    if (tmp->gtitle) news_free(tmp->gtitle);
    news_free(tmp);
    }

  if (!(switches & EXECUTE_CONTROL)) fprintf(fpm,"EXIT\n");
  fclose(fpm);
  if (mail_msg) {
    sprintf(err_oline,"\tControl: checkgroups - Mail to Usenet\n");
    call_mail(mail_scratch,"Control: checkgroups",USENET,err_oline);
    }
  delete_file_versions(mail_scratch);
  clear_domain();
  if (mail_msg) return(CTLOK);
  return(CTLOKNOCHG);
}

/*
 *  control_cancel
 *
 *  Delete a newsitem - message id added to history file.
 */

static int control_cancel(argv,fn,switches)
  char *argv[], *fn;
  int switches;
{
  char *id, *ide, control[132], net_sender[132], *mail_sender;

  mail_sender = argv[SENDER];
  if (!mail_sender || !*mail_sender) mail_sender = argv[FROM];
  if (!mail_sender || !*mail_sender) return(NOSENDERADDRESS);

  if (!mail_add_expiry) mail_add_expiry = 2;
  if (mail_add_expiry > 7) mail_add_expiry = 7;
  strcpy(control,argv[CONTROL]);
  if ((id = strchr(control,'<')) && (ide = strchr(control,'>'))) {
    *(ide + 1) = '\0';
    if ((ide = strchr(mail_sender,'<')) != 0) {
      strcpy(net_sender,++ide);
      chop_str(net_sender,'>');
      chop_str(net_sender,'\n');
      }
    else {
      char *cp = net_sender,
           *cp1 = mail_sender;

      while (isspace(*cp1)) cp1++;
      while ((*cp1) && (!isspace(*cp1))) *cp++ = *cp1++;
      *cp = '\0';
      chop_str(net_sender,'(');
      }
    if (del_id(id,net_sender)) {
#if CANCELMAIL
      sprintf(err_oline,"Control: cancel - item located and deleted");
      call_mail(fn,"Control: cancel (performed)",USENET,err_oline);
#endif
      }
    else {
      hist_add(id);
#if CANCELMAIL
      sprintf(err_oline,"Control: cancel - item NOT located - added to History file");
      call_mail(fn,"Control: cancel (NOT successful)",USENET,err_oline);
#endif

      }
    return(CTLOK);
    }
  return(BADFORMAT);
}

/*
 *  c_ihave_id
 *
 *  If a message identifier is NOT located in the local id file, and the
 *  id is NOT in the history file, then write the id to the request file.
 */

static int c_ihave_id(id,f)
  char *id;
  FILE *f;
{
  int status;
  char l_id[IDLEN + 4];

  memset(&l_id[IDLEN],0,4);
  util_idcpy(l_id,id);

  itmrab.rab$l_kbf = l_id;
  itmrab.rab$b_ksz = IDLEN + 4;
  itmrab.rab$b_krf = 1;
  itmrab.rab$l_rop = RAB$M_KGE | RAB$M_RRL | RAB$M_NLK;
  itmrab.rab$b_rac = RAB$C_KEY;

  if (   (!sys_get_nornf(&itmrab) || strcmp(l_id,newsitm.itm_id))
      && !hist_check(id))
    return(fprintf(f,"%s\n",id),1);
  return(0);
}

/*
 *  control_ihave
 *
 *  Check the local database of message-ids against the incoming ihave
 *  set. Any ids not held locally are added to a request file for a
 *  sendme response.
 */

static int control_ihave(argv,fn,switches)
  char *argv[], *fn;
  int switches;
{
  char send_scr_file[256], s_str[132], newsgroups[132], subject[132], xbuf[512],
       *il, *cp;
  FILE *fpw = 0, *fpr;
  int line_count = 0, cre_grp[2];

  il = argv[CONTROL] + 5;;
  sprintf(s_str,"to.%s",news_pathname);
  strcpy(newsgroups,argv[NEWSGROUPS]);
  lower_case(newsgroups);
  strcpy(xbuf,newsgroups);
  if ((cp = strrchr(xbuf, '.')) && !strcmp(".ctl",cp)) *cp = '\0';
  if (strcmp(xbuf,s_str)) return(NOTTOME);

  if ((cp = strrchr(il,'>')) != 0) il = ++cp;
  while (isspace(*il)) il++;
  if (!*il) return(NOREMOTESYS);

  strcpy(s_str,"to.");
  strcat(s_str,il);
  cp = s_str;
  while (isgraph(*cp)) cp++;
  *cp++ = '\0';

  if (!s_str[3]) return(NOREMOTESYS);

  lower_case(s_str);
  sprintf(send_scr_file,"SYS$SCRATCH:SENDME_MAIL_%X.TMP",getpid());
  fpw = fopen(send_scr_file,"w","mbc=32");

  il = argv[CONTROL] + 5;
  while ((cp = strchr(il,'<')) != 0) {
    il = cp;
    if ((cp = strchr(il,'>')) != 0) {
      *++cp = '\0';
      line_count += c_ihave_id(il,fpw);
      il = cp++;
      }
    else ++il;
    }

  if ((fpr = fopen(fn,"r","mbc=32","mbf=2")) != 0) {
    while ((fgets(xbuf,510,fpr)) && (*xbuf != '\n'));
    while (fgets(xbuf,510,fpr)) {
      if (*xbuf == '\n') continue;
      if ((cp = strchr(xbuf,'<')) != 0 && (il = strchr(cp,'>')) != 0) {
        *++il = '\0';
        line_count += c_ihave_id(cp,fpw);
        }
      }
    }
  fclose(fpw);

  if (line_count) {
    char *p,
         post_dist = '\0',
         post_path[132],
         loc_id[IDLEN];
     int cur_time;
     struct tm *stm;

     fpr = fopen(send_scr_file,"r","mbc=32","mbf=2");
     fpw = fopen(send_scr_file,"w","mbc=32");
     sprintf(post_path," %s!%s",news_pathname,usr_username);
     fprintf(fpw,"Path: %s\n",post_path);
     fprintf(fpw,"From: %s@%s\n",usr_username,Node_address);
     fprintf(fpw,"Newsgroups: %s\n",s_str);
     sprintf(subject,"sendme %s",news_pathname);
     fprintf(fpw,"Control: %s\n",subject);
     fprintf(fpw,"Subject: %s\n",subject);
     time((time_t *) &cur_time);
     p = ctime((time_t *) &cur_time);
     p += 4;
     stm = localtime((time_t *) &cur_time);
     strcpy(loc_id,gen_id());
     fprintf(fpw,"Message-ID: %s\n",loc_id);
     fprintf(fpw,"Date: %d %.3s %d %02d:%02d:%02d %s\n",
            stm->tm_mday,p,stm->tm_year,stm->tm_hour,stm->tm_min,stm->tm_sec,news_timezone);
     fprintf(fpw,"Organization: %s\n",Organisation_name);
     fprintf(fpw,"Lines: %d\n",line_count);
     fprintf(fpw,"\n");
     while (fgets(xbuf,510,fpr)) fputs(xbuf,fpw);
     fclose(fpr);
     fclose(fpw);
     sys_remote_send(post_path,s_str,&post_dist,send_scr_file,loc_id,0);
     if (sys_local_accept(s_str,&post_dist)) {
       do_new_group(s_str,0,(unsigned int *) cre_grp);
       if (cre_grp[0])
         do_new_item((unsigned int *)cre_grp, loc_id, subject, argv[FROM],
                     send_scr_file, 1, 0, line_count, 0);
       }
     }
  delete_file_versions(send_scr_file);
  return(CTLOK);
}

/*
 *  control_sendme
 *
 *  Check the requested message-id list against the local database
 *  and forward all requested messages that are held locally.
 */

static int control_sendme(argv,fn,switches)
  char *argv[], *fn;
  int switches;
{
  int status;
  char s_str[132], *il, *cp, l_id[IDLEN + 4], post_dist = '\0', post_path[132],
       fname[256], xbuf[512];
  int gn;
  FILE *fpr = NULL, *fpi = NULL;

  il = argv[CONTROL] + 6;
  sprintf(post_path," %s!%s",news_pathname,usr_username);
  itmrab.rab$l_kbf = l_id;
  itmrab.rab$b_ksz = IDLEN + 4;
  itmrab.rab$b_krf = 1;
  itmrab.rab$l_rop = RAB$M_KGE | RAB$M_RRL | RAB$M_NLK;
  itmrab.rab$b_rac = RAB$C_KEY;

  sprintf(s_str,"to.%s",news_pathname);
  strcpy(xbuf,argv[NEWSGROUPS]);
  lower_case(xbuf);
  if ((cp = strrchr(xbuf,'.')) && !strcmp(".ctl",cp)) *cp = '\0';
  if (strcmp(xbuf,s_str)) return(NOTTOME);

  if ((cp = strrchr(il,'>')) != 0) il = ++cp;
  while (isspace(*il)) il++;
  if (!*il) return(NOREMOTESYS);

  strcpy(s_str,"to.");
  strcat(s_str,il);
  cp = s_str;
  while (isgraph(*cp)) cp++;
  *cp++ = '\0';
  if (!s_str[3]) return(NOREMOTESYS);

  lower_case(s_str);
  il = argv[CONTROL] + 6;
  while ((cp = strchr(il,'<')) != 0) {
    il = cp;
    if ((cp = strchr(il,'>')) != 0) {
      *++cp = '\0';
      memset(&l_id[IDLEN],0,4);
      util_idcpy(l_id,il);
      if (   sys_get_nornf(&itmrab)
	  && !strcmp(l_id,newsitm.itm_id)
	  && (gn = ga_locate(newsitm.itm_grp))
	  && (fpr = do_oitem(&newsitm,"r",ga[gn]->grp_name,0,
	                     (ga[gn]->grp_flags & NEWS_M_RESTRICT_SET),0))) {
        fgetname(fpr,fname);
        fclose(fpr);
        sys_remote_send(post_path,s_str,&post_dist,fname,l_id,0);
        }
      il = ++cp;
      }
    else ++il;
    }
  if ((fpi = fopen(fn,"r")) != 0) {
    while ((fgets(xbuf,510,fpi)) && (*xbuf != '\n'));
    while (fgets(xbuf,510,fpi)) {
      if (*xbuf == '\n') continue;
      if ((cp = strchr(xbuf,'<')) != 0 && (il = strchr(cp,'>')) != 0) {
        *++il = '\0';
        memset(&l_id[IDLEN],0,4);
        util_idcpy(l_id,cp);
        if (   sys_get_nornf(&itmrab)
	    && !strcmp(l_id,newsitm.itm_id)
	    && (gn = ga_locate(newsitm.itm_grp))
	    && (fpr = do_oitem(&newsitm,"r",ga[gn]->grp_name,0,
	                       (ga[gn]->grp_flags & NEWS_M_RESTRICT_SET),0))) {
          fgetname(fpr,fname);
          fclose(fpr);
          sys_remote_send(post_path,s_str,&post_dist,itm_fname,l_id,0);
          }
        }
      }
    if (fpi) fclose(fpi);
    }
  return(CTLOK);
}

/*
 *  parse_control_item
 *
 *  Parse a Control: header item and call appropriate execution routine.
 */

int parse_control_item(argv,fn,switches,newsgroup)
  char *argv[], *fn;
  int switches;
  char *newsgroup;
{
  char loc_cmd[132];
  char *ng,*hier;

  if (strlen(argv[CONTROL]) > (sizeof(loc_cmd) - 1))
    argv[CONTROL][sizeof(loc_cmd)-1] = '\0';
  strcpy(loc_cmd,argv[CONTROL]);
  lower_case(loc_cmd);
/*
 *  make up a group name for the article, ie control.type.hier
 */
  ng = news_malloc(8+strlen(loc_cmd)+strlen(argv[NEWSGROUPS])+1);
  strcpy(ng,"control.");
  strcat(ng,loc_cmd);
  chop_str(ng,' ');
  strcat(ng,".");
  strcpy(hier=news_malloc(strlen(argv[NEWSGROUPS])+1),argv[NEWSGROUPS]);
  chop_str(hier,','); /* only use first newgroup on the line */
  chop_str(hier,'.'); /* only use the first component of the newsgroup */
  strcat(ng,hier);
  if ( ga_exact_name(ng) )
    strcpy(newsgroup,ng);
  else {
    strcpy(ng,"control.");
    strcat(ng,loc_cmd);
    chop_str(ng,' ');
    if ( ga_exact_name(ng) )
      strcpy(newsgroup,ng);
  }
  news_free(ng);
  news_free(hier);
/*
 * now newsgroup has changed to control.type.hier if that exists, 
 * or control.type if that exists, or no change if none exist.
 */
  if (!strncmp(loc_cmd,"version",7)) return(control_version(argv,fn,switches));
  if (!strncmp(loc_cmd,"senduuname",10)) return(control_senduuname(argv,fn,switches));
  if (!strncmp(loc_cmd,"sendsys",7)) return(control_sendsys(argv,fn,switches));
  if (!strncmp(loc_cmd,"newgroup",8)) return(control_newgroup(argv,fn,switches));
  if (!strncmp(loc_cmd,"rmgroup",7)) return(control_rmgroup(argv,fn,switches));
  if (!strncmp(loc_cmd,"checkgroups",11)) return(control_checkgroups(argv,fn,switches));
  if (!strncmp(loc_cmd,"cancel",6)) return(control_cancel(argv,fn,switches));
  if (!strncmp(loc_cmd,"ihave",5)) return(control_ihave(argv,fn,switches));
  if (!strncmp(loc_cmd,"sendme",6)) return(control_sendme(argv,fn,switches));
  sprintf(err_oline,"Control: No handler defined for %s",loc_cmd);
#if UNKCONTROLMAIL
  call_mail(fn,"Unrecognised NEWS Control: message",USENET,err_oline);
#endif
  return(UNKCTL);
}
#endif
