/*
**++
**  FACILITY:
**	NEWSPROFILE
**
**  ABSTRACT:
**      Manage the per-user newsrc context file for read/unread items,
**      kill filters and marks.
**
**  AUTHOR:
**      Geoff Huston
**
**  COPYRIGHT:
**      Copyright  1989, 1990,1991,1992
**
**  MODIFICATION HISTORY:
**	V6.1	gh	31/1/92
**	- mod show profile to correctly display fastload setting
**	V6.1	 9-Feb-1992	rankin@eql.caltech.edu
**	- Lint cleanup
**	V6.1b5	17-Feb-1993	bailey@genetics.upenn.edu
**	- replaced 'unseenitems' with 'unreaditems' throughout
**	V6.1b6	22-Mar-1993	bailey@genetics.upenn.edu
**	- corrected Call/Spawn display of of editor
**	V6.1b8	 9-Mar-1994	mark.martinec@ijs.si
**	- don't call set_[no]broad and set_[no]line from dsetprofile(),
**	  since label "BOOL" is not defined in the SET PROFILE syntax
**	V6.1b9	15-Dec-1994	bailey@genetics.upenn.edu
**	- don't ask about registering for new groups during News/[All]Scan
**
**--
**/

#ifdef vaxc
#module NEWSPROFILE "V6.1"
#endif

#define _NEWSPROFILE_C
#define module_name "NEWSPROFILE"

#include "newsinclude.h"
#include "newsextern.h"

extern
time_t sys_reg_time;

static
char profile_signature[256] = {"@"},
     profile_locsignature[256] = {"@"};

char profile_organization[256] = {""};

int profile_flags =
#if CONFIRM_NEW_GROUPS
PROFILE_NEWREGISTER;
#else
0;
#endif

int set_profile_display_lines = 0,
    set_profile_display_postmark = 0,
    profile_search_val = 0,
    profile_minfromlen = 19,
    profile_auto = 0,
    profile_rc_scansize = -1,
    profile_display_stk = 0,
    profile_showdirs_val = DISPLAY_BY_NEWSGROUPS,
    pw_allitems = 0;

char profile_rc_signature[256],
     profile_rc_locsignature[256],
     profile_rc_organization[256] = {""},
     profile_autostr[132],
     profile_classstr[132],
     profile_editstr[132],
     profile_viewstr[132],
     profile_persnamestr[132],
     profile_printstr[132],
     profile_newregstr[256] = {"*/confirm"},
     profile_keydefs[256],
     profile_reply_to_str[256];

int reggroup(grpname)
  char *grpname;
{
  static int profile_setting = -1, g, notall, conf = 0, do_it = 2, regit;
  static char *slash, *cp1, *cp2, prompt[128];

  chop_str(grpname,'\n');
  for (g = 1; g <= ga_size ; ++g) {
    if (!strcmp(ga[g]->grp_name,grpname)) {
      if (ga[g]->grp_reg) return 0;
      break;
      }
    }
  if (profile_setting == -1) {
    profile_setting = ((profile_flags & PROFILE_NEWREGISTER)
		    && strcmp(profile_newregstr,"none"));
    if (profile_setting) {
      time(&sys_reg_time);
      if ((slash = strchr(profile_newregstr,'/')) != 0) {
        conf = !strncmp(slash,"/confirm",strlen(slash));
        *slash = '\0';
        }
      confirm_specified = conf;
      notall = strcmp(profile_newregstr,"*");
      }
    else {
      if (NEWGROUP_REG) time(&sys_reg_time);
      }
    }

  if (profile_setting) {
    if (slash) *slash = '/';
    if (!do_it) {
      sprintf(err_oline,"New newsgroup: %s, not registered",grpname);
      err_line(err_oline);
      return 0;
      }
    if (notall) {
      cp1 = profile_newregstr;
      while (cp1) {
	cp2 = chop_str(cp1,',');
        if (wild_match(grpname,cp1)) {
          sprintf(err_oline,"New newsgroup: %s, Registered",grpname);
          regit = 1;
          if (conf) {
	    sprintf(prompt,"New newsgroup: %s, Register? (Yes, No, Quit, All) [Y]",grpname);
	    if ((do_it = ask_confirm(prompt,'Y')) != 1) {
              regit = 0;
              }
            *err_oline = '\0';
	    }
          if (cp2) *cp2 = ',';
          break;
          }
        if (cp2) *cp2++ = ',';
        cp1 = cp2;
        }
      }
    else {
      sprintf(err_oline,"New newsgroup: %s, Registered",grpname);
      regit = 1;
      if (conf) {
        sprintf(prompt,"New newsgroup: %s, Register? (Yes, No, Quit, All) [Y]",grpname);
	if ((do_it = ask_confirm(prompt,'Y')) != 1) {
          regit = 0;
          }
        *err_oline = '\0';
	}
      }
    if (slash) *slash = '/';
    }
  else {
    regit = NEWGROUP_REG;
    if (NEWGROUP_REG) sprintf(err_oline,"New newsgroup: %s, Registered",grpname);
    else sprintf(err_oline,"New newsgroup: %s, not registered",grpname);
    }
  err_line(err_oline);
  if (!regit) return 0;

  for (g = 1; g <= ga_size ; ++g) {
    if (!strcmp(ga[g]->grp_name,grpname)) return(ga[g]->grp_reg = regit);
    }
#if FAST_LOAD
  read_group(grpname);
  for (g = 1; g <= ga_size ; ++g) {
    if (!strcmp(ga[g]->grp_name,grpname)) return(ga[g]->grp_reg = regit);
    }
#endif
  return 0;
}

void profile_read(f)
  FILE *f;
{
  int status;
  char *in_line;
  int g;
  struct dir_class *ch = c_head;

#if FAST_LOAD
  if (fast_loading) profile_flags |= PROFILE_FASTLOAD;
#endif

  while ((in_line = fgetl(f)) != 0) {
    chop_str(in_line,'\n');
    if (in_line[0] == '#') continue;
    if (!strncmp(in_line,"AUTO=",5)) {
      profile_flags |= PROFILE_AUTO;
      strcpy(profile_autostr,&in_line[5]);
      }
    else if (!strncmp(in_line,"LINE",4)) {
      profile_flags |= PROFILE_LINEEDIT;
      line_editing = 1;
      }
    else if (!strncmp(in_line,"UNSEENSTACK",11)) {
      profile_flags |= PROFILE_UNSEENSTACK;
      display_unseen_stack = 1;
      if ((in_line[11] == '=') && (sscanf(&in_line[12],"%d",&profile_display_stk) == 1))
        display_stk = profile_display_stk;
      else profile_display_stk = display_stk;
      }
    else if (!strncmp(in_line,"UNREADITEMS",11)) {
      profile_flags |= PROFILE_UNREADITEMS;
      display_unseen_items = 1;
      }
    else if (!strncmp(in_line,"ALLITEMS",8)) {
      profile_flags &= ~PROFILE_UNREADITEMS;
      profile_flags &= ~PROFILE_UNSEENSTACK;
      display_unseen_stack = 0;
      display_unseen_items = 0;
      pw_allitems = 1;
      }
    else if (!strncmp(in_line,"NOBROADCAST",11)) {
      profile_flags |= PROFILE_BROADCAST;
      set_nobroad();
      }
    else if (!strncmp(in_line,"DISPLAY=",8)) {
      profile_flags |= PROFILE_DISPLAY;
      in_line += 8;
      while (*in_line) {
        if (toupper(*in_line) == 'L') {
          set_profile_display_lines = profile_display_lines = (*(++in_line) == '1');
          }
        else if (toupper(*in_line) == 'P') {
          set_profile_display_postmark = profile_display_postmark = (*(++in_line) == '1');
          }
	++in_line;
       }
      }
    else if (!strncmp(in_line,"CLASS=",6)) {
      profile_flags |= PROFILE_CLASS;
      strcpy(profile_classstr,&in_line[6]);
      }
    else if (!strncmp(in_line,"SEARCH-LITERAL",14)) {
      profile_flags |= PROFILE_SEARCH;
      profile_search_val = 0;
      }
    else if (!strncmp(in_line,"SEARCH-WILD",11)) {
      profile_flags |= PROFILE_SEARCH;
      profile_search_val = 1;
      }
    else if (!strncmp(in_line,"SEARCH-PATTERNS",15)) {
      profile_flags |= PROFILE_SEARCH;
      profile_search_val = 2;
      }
    else if (!strncmp(in_line,"SHOWDIRS",8)) {
      profile_flags |= PROFILE_SHOWDIRS;
      showdirs_val = profile_showdirs_val = 1;
      }
    else if (!strncmp(in_line,"NOSHOWDIRS",10)) {
      profile_flags |= PROFILE_SHOWDIRS;
      showdirs_val = profile_showdirs_val = 0;
      }
    else if (!strncmp(in_line,"RC-ORDER",8)) {
      reorder_groups = 1;
      if (!profile_filter) include_all_groups = 1;
      }
    else if (!strncmp(in_line,"RC-FILTER",9)) {
      profile_filter = 1;
      include_all_groups = 0;
      }
    else if (!strncmp(in_line,"RC-REG",6)) profile_flags |= PROFILE_RCREG;
    else if (!strncmp(in_line,"EDIT=",5)) {
      profile_flags |= PROFILE_EDIT;
      strcpy(profile_editstr,&in_line[5]);
      }
    else if (!strncmp(in_line,"VIEW=",5)) {
      profile_flags |= PROFILE_VIEW;
      strcpy(profile_viewstr,&in_line[5]);
      }
    else if (!strncmp(in_line,"PERSNAME=",9)) {
      profile_flags |= PROFILE_PERSNAME;
      strcpy(profile_persnamestr,&in_line[9]);
      }
    else if (!strncmp(in_line,"PRINT=",6)) {
      profile_flags |= PROFILE_PRINT;
      strcpy(profile_printstr,&in_line[6]);
      }
    else if (!strncmp(in_line,"DIRECTORY=",10)) {
      profile_flags |= PROFILE_DIRECTORY;
      strcpy(profile_dirstr,&in_line[10]);
      }
    else if (!strncmp(in_line,"NEWREGISTER=",12)) {
      profile_flags |= PROFILE_NEWREGISTER;
      strcpy(profile_newregstr,&in_line[12]);
      lower_case(profile_newregstr);
      }
    else if (!strncmp(in_line,"SCANSIZE=",9)) {
      profile_flags |= PROFILE_SCANSIZE;
      if (sscanf(&in_line[9],"%d",&profile_scansize) != 1)
        profile_scansize = -1;
      profile_rc_scansize = profile_scansize;
      }
    else if (!strncmp(in_line,"FROMSIZE=",9)) {
      if (sscanf(&in_line[9],"%d",&profile_minfromlen) != 1)
        profile_minfromlen = minfromlen;
      else minfromlen = profile_minfromlen;
      }
    else if (!strncmp(in_line,"SIGNATURE=",10)) {
      profile_flags |= PROFILE_SIGNATURE;
      strcpy(profile_rc_signature,&in_line[10]);
      strcpy(profile_signature,profile_rc_signature);
      }
    else if (!strncmp(in_line,"LOCSIGNATURE=",13)) {
      profile_flags |= PROFILE_LOCSIGNATURE;
      strcpy(profile_rc_locsignature,&in_line[13]);
      strcpy(profile_locsignature,profile_rc_locsignature);
      }
    else if (!strncmp(in_line,"ORGANIZATION=",13)) {
      profile_flags |= PROFILE_ORGANIZATION;
      strcpy(profile_rc_organization,&in_line[13]);
      strcpy(profile_organization,profile_rc_organization);
      }
    else if (!strncmp(in_line,"KEYDEFS=",8)) {
      profile_flags |= PROFILE_KEYDEFS;
      strcpy(profile_keydefs,&in_line[8]);
      }
    else if (!strncmp(in_line,"REPLYTO=",8)) {
      profile_flags |= PROFILE_REPLYTO;
      strcpy(profile_reply_to_str,&in_line[8]);
      strcpy(profile_reply_to,profile_reply_to_str);
      }
    }
  if (profile_flags & PROFILE_KEYDEFS) {
    FILE *fpr;

    if ((fpr = fopen(profile_keydefs,"r")) != 0) {
    /* use the access() call instead */
      fclose(fpr);
      if (keytab) smg$load_key_defs(&keytab,c$dsc(profile_keydefs),0,c$ac(0));
      }
    }
  if (profile_flags & PROFILE_AUTO) {
    if (!strcmp(profile_autostr,"UNSEEN")) profile_auto = PROFILE_UNSEEN;
    else if (!strcmp(profile_autostr,"DIRECTORY")) profile_auto = PROFILE_DIR;
    else profile_flags &= ~PROFILE_AUTO;
    }
  if (profile_flags & PROFILE_CLASS) {
    if (!strcmp(profile_classstr,"MAIN")) *n_class_name = '\0';
    else if (*profile_classstr) {
      lower_case(profile_classstr);
      while (ch) {
        if (!strcmp(profile_classstr,ch->c_name)) break;
        ch = ch->c_next;
        }
      if (ch) strcpy(n_class_name,profile_classstr);
      }
    else profile_flags &= ~PROFILE_CLASS;
    }
  if (profile_flags & PROFILE_EDIT) {
    lower_case(profile_editstr);
    if (!strcmp(profile_editstr,"callable_tpu")) *Editor = *Viewer = '\0';
    else if (!strcmp(profile_editstr,"eve")) *Editor = *Viewer = '\0';
    else if (!strcmp(profile_editstr,"tpu")) *Editor = *Viewer = '\0';
    else if (!strcmp(profile_editstr,"callable_edt")) {
      strcpy(Editor,"edt");
      strcpy(Viewer,"edt");
      }
    else {
      if (captive_account()) strcpy(profile_editstr,"edt");
      strcpy(Editor,profile_editstr);
      strcpy(Viewer,profile_editstr);
      }
    }
  if (profile_flags & PROFILE_VIEW) {
    lower_case(profile_viewstr);
    if (!strcmp(profile_viewstr,"callable_tpu")) *Viewer = '\0';
    else if (!strcmp(profile_viewstr,"eve")) *Viewer = '\0';
    else if (!strcmp(profile_viewstr,"tpu")) *Viewer = '\0';
    else if (!strcmp(profile_viewstr,"callable_edt")) strcpy(Viewer,"edt");
    else if (captive_account()) strcpy(profile_viewstr,"edt");
    }
  if (profile_flags & PROFILE_PERSNAME) {
    strcpy(usr_persname,profile_persnamestr);
    }
  if (profile_flags & PROFILE_PRINT) {
    char try_command[256];

    sprintf(try_command,"PRINT%s 1",profile_printstr);
    if (!(cli$dcl_parse(c$dsc(try_command),CLICMDTBL,0,0,0) & 1))
      profile_flags &= ~PROFILE_PRINT;
    else strcpy(print_constant,profile_printstr);
    }
  if (profile_flags & PROFILE_DIRECTORY) {
    char try_command[256];

    strcpy(try_command,"DIRECTORY/");
    strcat(try_command,profile_dirstr);
    if (!(cli$dcl_parse(c$dsc(try_command),CLICMDTBL,0,0,0) & 1)) {
      profile_flags &= ~PROFILE_DIRECTORY;
      *profile_dirstr = '\0';
      }
    }

  if (init_scanning) return;
  

#if FAST_LOAD
  if (all_loaded) return;
  if (fast_loading) {
    time_t nowtime, savetime;

    time(&nowtime);
			/* ten hour min time for group refresh */
    if (nowtime < (sys_reg_time + (10 * 60 * 60))) return;
    if (nntp_client) {
      struct tm *stm;

      stm = localtime(&sys_reg_time);
      sprintf(err_oline,"NEWGROUPS %02d%02d%02d %02d%02d%02d",stm->tm_year,stm->tm_mon + 1,stm->tm_mday,stm->tm_hour,stm->tm_min,stm->tm_sec);
      nntp_get_info(nntp_node,nntp_proto,err_oline,231,0,reggroup);
      }
    else {
      savetime = sys_reg_time;
      grprab.rab$l_rop = RAB$M_RRL | RAB$M_NLK;
      grprab.rab$b_krf = 0;
      grprab.rab$b_rac = RAB$C_SEQ;
      sys_rewind(&grprab);
      while (sys_get_noeof(&grprab)) {
        if (!newsgrp.grp_num) continue;
        if (newsgrp.grp_credate >= sys_reg_time) {
	  reggroup(newsgrp.grp_name);
          sys_reg_time = savetime;    /* inserted - mark.martinec@ijs.si */
	  if (grprab.rab$b_rac != RAB$C_SEQ) {
	    grprab.rab$l_rop = RAB$M_RRL | RAB$M_NLK;
	    grprab.rab$b_krf = 0;
	    grprab.rab$b_rac = RAB$C_SEQ;
	 /* sys_reg_time = savetime;  (* removed  - mark.martinec@ijs.si */
	    }
	  }
        }
      }
    sys_reg_time = nowtime;
    return;
    }
#endif
  if (init_scanning || first_time_user || profile_filter) return;
  for (g = 1; g <= ga_size ; ++g) {
    if (!(ga[g]->grp_flags & NEWS_M_OLDGROUP)) {
      ga[g]->grp_reg = 0;
      reggroup(ga[g]->grp_name);
      }
    }
}

void profile_write(f)
  FILE *f;
{
  if (!profile_flags && !reorder_groups && !profile_filter) return;
  putl("PROFILE\n",f);
  if ((profile_flags & PROFILE_AUTO) && *profile_autostr) {
    putl("AUTO=",f);
    putl(profile_autostr,f);
    putl("\n",f);
    }
  if (profile_flags & PROFILE_LINEEDIT)
    putl("LINE\n",f);
  if (profile_flags & PROFILE_SHOWDIRS) {
    if (!profile_showdirs_val) putl("NO",f);
    putl("SHOWDIRS\n",f);
    }
  if (profile_flags & PROFILE_UNSEENSTACK) {
    if (profile_display_stk)
      sprintf(err_oline,"UNSEENSTACK=%d\n",profile_display_stk);
    else sprintf(err_oline,"UNSEENSTACK\n");
    putl(err_oline,f);
    }
  if (profile_flags & PROFILE_UNREADITEMS)
    putl("UNREADITEMS\n",f);
  if (pw_allitems && !(profile_flags & PROFILE_UNSEENSTACK) && !(profile_flags & PROFILE_UNREADITEMS))
    putl("ALLITEMS\n",f);
  if (profile_flags & PROFILE_BROADCAST)
    putl("NOBROADCAST\n",f);
  if (profile_flags & PROFILE_DISPLAY) {
    sprintf(err_oline,"DISPLAY=(L%c,P%c)\n",
      (set_profile_display_lines ? '1' : '0'),
      (set_profile_display_postmark ? '1' : '0'));
    putl(err_oline,f);
    }
  if (profile_flags & PROFILE_SEARCH) {
    if (profile_search_val == 1) putl("SEARCH-WILD\n",f);
    else if (profile_search_val == 2) putl("SEARCH-PATTERNS\n",f);
    else putl("SEARCH-LITERAL\n",f);
    }
  if (reorder_groups) putl("RC-ORDER\n",f);
  if (profile_filter) putl("RC-FILTER\n",f);
  if (profile_flags & PROFILE_RCREG) putl("RC-REG\n",f);
  if ((profile_flags & PROFILE_CLASS) && *profile_classstr) {
    putl("CLASS=",f);
    putl(profile_classstr,f);
    putl("\n",f);
    }
  if ((profile_flags & PROFILE_EDIT) && *profile_editstr) {
    putl("EDIT=",f);
    putl(profile_editstr,f);
    putl("\n",f);
    }
  if ((profile_flags & PROFILE_VIEW) && *profile_viewstr) {
    putl("VIEW=",f);
    putl(profile_viewstr,f);
    putl("\n",f);
    }
  if ((profile_flags & PROFILE_PERSNAME) && *profile_persnamestr) {
    putl("PERSNAME=",f);
    putl(profile_persnamestr,f);
    putl("\n",f);
    }
  if ((profile_flags & PROFILE_PRINT) && *profile_printstr) {
    putl("PRINT=",f);
    putl(profile_printstr,f);
    putl("\n",f);
    }
  if ((profile_flags & PROFILE_DIRECTORY) && *profile_dirstr) {
    putl("DIRECTORY=",f);
    putl(profile_dirstr,f);
    putl("\n",f);
    }
  if ((profile_flags & PROFILE_NEWREGISTER) && *profile_newregstr) {
    putl("NEWREGISTER=",f);
    putl(profile_newregstr,f);
    putl("\n",f);
    }
  if (profile_flags & PROFILE_SCANSIZE) {
    sprintf(err_oline,"SCANSIZE=%d\n",profile_rc_scansize);
    putl(err_oline,f);
    }
  if (profile_minfromlen != 19) {
    sprintf(err_oline,"FROMSIZE=%d\n",profile_minfromlen);
    putl(err_oline,f);
    }
  if ((profile_flags & PROFILE_SIGNATURE) && *profile_rc_signature) {
    sprintf(err_oline,"SIGNATURE=%s\n",profile_rc_signature);
    putl(err_oline,f);
    }
  if ((profile_flags & PROFILE_LOCSIGNATURE) && *profile_rc_locsignature) {
    sprintf(err_oline,"LOCSIGNATURE=%s\n",profile_rc_locsignature);
    putl(err_oline,f);
    }
  if ((profile_flags & PROFILE_ORGANIZATION) && *profile_rc_organization) {
    sprintf(err_oline,"ORGANIZATION=%s\n",profile_rc_organization);
    putl(err_oline,f);
    }
  if ((profile_flags & PROFILE_KEYDEFS) && *profile_keydefs) {
    sprintf(err_oline,"KEYDEFS=%s\n",profile_keydefs);
    putl(err_oline,f);
    }
  if ((profile_flags & PROFILE_REPLYTO) && *profile_reply_to_str) {
    sprintf(err_oline,"REPLYTO=%s\n",profile_reply_to_str);
    putl(err_oline,f);
    }
}

static int dsprofile()
{
  char *tpucall;

  start_header(T_DISPLAY_LOOP,SHOW_PROFILE);
  sprintf(err_oline,"NEWS PROFILE for user: %s",usr_username);
  put_line(err_oline,0,T_DISPLAY_LOOP);
  put_line("",0,T_DISPLAY_LOOP);
  end_header(T_DISPLAY_LOOP);
  if (profile_flags & PROFILE_AUTO) {
    sprintf(err_oline,"OPEN action display: %s",profile_autostr);
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  if (profile_flags & PROFILE_CLASS) {
    sprintf(err_oline,"Default CLASS on startup: %s",profile_classstr);
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }

  if (!(tpucall = news_getenv("TPU$SECTION",0))) tpucall = news_getenv("TPUSECINI",0);
  sprintf(err_oline,"%s Editor: ",((profile_flags & PROFILE_EDIT) ? "Profile" : "Default"));
  if (!*Editor) {
    strcat(err_oline,"Call EVE");
    if (tpucall) {
      strcat(err_oline,"/SECTION=");
      strcat(err_oline,tpucall);
      }
    }
  else if (!strcmp(Editor,"edt")) strcat(err_oline,"Call EDT");
  else {
    if (!strncmp(Editor,"callable_",9)) {
      strcat(err_oline,"Call ");
      strcat(err_oline,Editor+9);
      }
    else {
      strcat(err_oline,"Spawn ");
      strcat(err_oline,Editor);
      }
    }
  put_line(err_oline,0,T_DISPLAY_LOOP);

  sprintf(err_oline,"%s Viewer: ",((profile_flags & PROFILE_VIEW) ? "Profile" : "Default"));
  if (!*Viewer) {
    strcat(err_oline,"Call EVE");
    if (tpucall) {
      strcat(err_oline,"/SECTION=");
      strcat(err_oline,tpucall);
      }
    }
  else if (!strcmp(Viewer,"edt")) strcat(err_oline,"Call EDT");
  else {
    if (!strncmp(Editor,"callable_",9)) {
      strcat(err_oline,"Call ");
      strcat(err_oline,Editor+9);
      }
    else {
      strcat(err_oline,"Spawn ");
      strcat(err_oline,Editor);
      }
    }
  put_line(err_oline,0,T_DISPLAY_LOOP);

  if (*usr_persname) {
    sprintf(err_oline,"%s Personal Name: %s",
            ((profile_flags & PROFILE_PERSNAME) ? "Profile" : "Default"),
            usr_persname);
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  if (profile_flags & PROFILE_PRINT) {
    sprintf(err_oline,"PRINT default qualifiers: %s",profile_printstr);
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  if (*profile_dirstr) {
    sprintf(err_oline,"%s Start Directory: DIR/%s",
            ((profile_flags & PROFILE_DIRECTORY) ? "Profile" : "Default"),
            profile_dirstr);
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  if ((profile_flags & PROFILE_NEWREGISTER)  && strcmp(profile_newregstr,"none")) {
    sprintf(err_oline,"Automatic REGISTER of new newsgroups: %s",profile_newregstr);
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  if (profile_flags & PROFILE_SHOWDIRS) {
    sprintf(err_oline,"Read/NEW action: %sisplay newsitem directories",((profile_showdirs_val) ? "D" : "Don't d"));
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  if (profile_flags & PROFILE_SCANSIZE) {
    if (profile_scansize == -1) sprintf(err_oline,"Scan lines: Use defaults");
    else sprintf(err_oline,"Scan lines: %d",profile_rc_scansize);
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  if (profile_minfromlen != 19) {
    if (!profile_minfromlen) 
      put_line("From addresses NOT displayed",0,T_DISPLAY_LOOP);
    else {
      sprintf(err_oline,"Fromsize: %d",profile_minfromlen);
      put_line(err_oline,0,T_DISPLAY_LOOP);
      }
    }
  if (*profile_reply_to) {
    sprintf(err_oline,"Reply-To: \"%s\"",profile_reply_to);
    if ((!(profile_flags & PROFILE_REPLYTO)) || strcmp(profile_reply_to,profile_reply_to_str)) 
      strcat(err_oline," <this session only>");
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  if (profile_flags & PROFILE_REPLYTO) {
    if (strcmp(profile_reply_to,profile_reply_to_str)) {
      sprintf(err_oline,"Reply-To: \"%s\"",profile_reply_to_str);
      strcat(err_oline," <Profile value>");
      put_line(err_oline,0,T_DISPLAY_LOOP);
      }
    }

  if (profile_flags & PROFILE_SIGNATURE) {
    sprintf(err_oline,"Signature file: %s",profile_rc_signature);
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  if (profile_flags & PROFILE_LOCSIGNATURE) {
    sprintf(err_oline,"Local postings signature file: %s",profile_rc_locsignature);
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  if (*profile_rc_organization) {
    sprintf(err_oline,"Postings: Organization - %s",profile_rc_organization);
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  if (strcmp(profile_rc_organization,profile_organization)) {
    sprintf(err_oline,"Postings: Organization/TEMP - %s",profile_organization);
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  if (profile_flags & PROFILE_DISPLAY) {
    sprintf(err_oline,"DISPLAY action (NNTP client): DISPLAY=(%sLINES,%sPOSTMARK)",
      (set_profile_display_lines ? "" : "No"),
      (set_profile_display_postmark ? "" : "No"));
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  if (profile_flags & PROFILE_UNREADITEMS)
    put_line("UNREADITEMS - display unseen items only",0,T_DISPLAY_LOOP);
  else if (profile_flags & PROFILE_UNSEENSTACK) {
    if (profile_display_stk) {
      sprintf(err_oline,"UNSEENSTACK - display unseen stack (+%d)",profile_display_stk);
      put_line(err_oline,0,T_DISPLAY_LOOP);
      }
    else put_line("UNSEENSTACK - display unseen items only",0,T_DISPLAY_LOOP);
    }
  if (pw_allitems && !(profile_flags & PROFILE_UNSEENSTACK) && !(profile_flags & PROFILE_UNREADITEMS))
    put_line("ALLITEMS - display all items",0,T_DISPLAY_LOOP);
#if FAST_LOAD
  if (profile_flags & PROFILE_FASTLOAD)
    put_line("FASTLOAD - save only registered newsgroups in newsrc",0,T_DISPLAY_LOOP);
#endif
  if (profile_flags & PROFILE_LINEEDIT)
    put_line("LINE mode - cursor keys perform line edit functions",0,T_DISPLAY_LOOP);
  else
    put_line("NOLINE mode - cursor keys perform cursor movement functions",0,T_DISPLAY_LOOP);
  if (profile_flags & PROFILE_BROADCAST)
    put_line("NOBROADCAST_TRAPPING mode",0,T_DISPLAY_LOOP);
  else put_line("BROADCAST_TRAPPING mode",0,T_DISPLAY_LOOP);
  if (profile_flags & PROFILE_KEYDEFS) {
    sprintf(err_oline,"KeyDefs File: %s",profile_keydefs);
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  if (reorder_groups)
    put_line("RCOrder: Newsgroup Display ordering taken from newsrc file",0,T_DISPLAY_LOOP);
  if (profile_filter)
    put_line("RCFilter: Newsgroups Filter based on  newsrc file",0,T_DISPLAY_LOOP);
  if (profile_flags & PROFILE_RCREG)
    put_line("RCReg: Record seen/unseen status only for registered Newsgroups",0,T_DISPLAY_LOOP);
  if (profile_flags & PROFILE_DOTNEWSRC)
    put_line("DotNewsRC mode - maintain rn-style .newsrc file",0,T_DISPLAY_LOOP);
  if (profile_flags & PROFILE_SEARCH) {
    strcpy(err_oline,"Search Routine: ");
    if (profile_search_val == 1) strcat(err_oline,"Wildcard match");
    else if (profile_search_val == 2) strcat(err_oline,"Pattern match");
    else strcat(err_oline,"Literal match");
    put_line(err_oline,0,T_DISPLAY_LOOP);
    }
  put_line("",1,T_DISPLAY_LOOP);
  return(0);
}

int do_show_profile()
{
  return(unwind_display(I_DISPLAY_LOOP,dsprofile));
}

static int dsetprofile()
{
  int status;
  char val[256], *cp;
  int temp, errs = 0, do_update_call = 0;
  unsigned short val_len;
  $DESCRIPTOR(val_dsc,val);

  strcpy(err_oline,"Set Profile");
  temp = (cli$present(c$dsc("TEMPORARY")) & 1);
  if (temp) strcat(err_oline,"/TEMPORARY: ");
  else strcat(err_oline,": ");

  if (cli$present(c$dsc("SEARCH")) & 1) {
    profile_search_val = 0;
    profile_flags |= PROFILE_SEARCH;
    if (   (cli$get_value(c$dsc("SEARCH"),&val_dsc,&val_len) & 1)
        && val_len) {
      val[val_len] = '\0';
      s_to_upper(val);
      if (!strncmp(val,"LITERAL",val_len)) profile_search_val = 0;
      else if (!strncmp(val,"WILDCARD",val_len)) profile_search_val = 1;
      else if (!strncmp(val,"PATTERN",val_len)) profile_search_val = 2;
      }
    }

  while (cli$get_value(c$dsc("DISPLAY"),&val_dsc,&val_len) & 1)  {
    if (val_len) {
      int ass_val = 1;
      char *valp = val;

      val[val_len] = '\0';
      s_to_upper(val);
      if (!strncmp(val,"NO",2)) {
        val_len -= 2;
        ass_val = 0;
	valp += 2;
        }
      if ((cp = chop_str_plus(valp,':')) != 0) val_len = strlen(valp);

      if (!strncmp(valp,"ALLITEMS",val_len)) {
        if (ass_val) {
          if (display_unseen_stack || display_unseen_items) do_update_call = 1;
          display_unseen_stack = display_unseen_items = 0;
          if (!temp) {
            profile_flags &= ~PROFILE_UNSEENSTACK;
            profile_flags &= ~PROFILE_UNREADITEMS;
            pw_allitems = 1;
            }
          }
        else {
          if (!display_unseen_items || display_unseen_stack) do_update_call = 1;
          display_unseen_items = 1;
          display_unseen_stack = 0;
          if (!temp) {
            profile_flags &= PROFILE_UNREADITEMS;
	    profile_flags |= ~PROFILE_UNSEENSTACK;
            pw_allitems = 0;
            }
          }
        }
      else if (!strncmp(valp,"UNREADITEMS",val_len)) {
        if (ass_val) {
          if (display_unseen_stack || !display_unseen_items) do_update_call = 1;
          display_unseen_stack = 0;
          display_unseen_items = 1;
          if (!temp) {
            profile_flags &= ~PROFILE_UNSEENSTACK;
            profile_flags |= PROFILE_UNREADITEMS;
            }
          }
        else {
          if (display_unseen_stack || display_unseen_items) do_update_call = 1;
          display_unseen_stack = display_unseen_items = 0;
          if (!temp) {
            profile_flags &= ~PROFILE_UNSEENSTACK;
            profile_flags &= ~PROFILE_UNREADITEMS;
            }
          }
        }
      else if (!strncmp(valp,"UNSEENSTACK",val_len)) {
        if (ass_val) {
          int tmp;

          if (!display_unseen_stack || display_unseen_items) do_update_call = 1;
          display_unseen_stack = 1;
          display_unseen_items = 0;
          if (cp && (sscanf(cp,"%d",&tmp) == 1)) display_stk = tmp;
          if (!temp) {
            profile_flags |= PROFILE_UNSEENSTACK;
            profile_flags &= ~PROFILE_UNREADITEMS;
            profile_display_stk = display_stk;
            }
          }
        else {
          if (display_unseen_stack || display_unseen_items) do_update_call = 1;
          display_unseen_stack = display_unseen_items = 0;
          if (!temp) {
            profile_flags &= ~PROFILE_UNSEENSTACK;
            profile_flags &= ~PROFILE_UNREADITEMS;
            }
          }
        }
      else if (!strncmp(valp,"LINES",val_len)) {
        profile_display_lines = ass_val;
        if (!temp) {
	  set_profile_display_lines = ass_val;
          profile_flags |= PROFILE_DISPLAY;
          }
        }
      else if (!strncmp(valp,"POSTMARKS",val_len)) {
        profile_display_postmark = ass_val;
        if (!temp) {
	  set_profile_display_postmark = ass_val;
          profile_flags |= PROFILE_DISPLAY;
          }
        }
      else {
	strcat(err_oline,"Invalid DISPLAY value");
      	++errs;
	}
      }
    }

  if ((status = cli$present(c$dsc("SHOWDIRS"))) == CLI$_NEGATED) {
    showdirs_val = 0;
    if (!temp) {
      profile_flags |= PROFILE_SHOWDIRS;
      profile_showdirs_val = 0;
      }
    }
  else if (status & 1) {
    showdirs_val = 1;
    if (!temp) {
      profile_flags |= PROFILE_SHOWDIRS;
      profile_showdirs_val = 1;
      }
    }
  if ((status = cli$present(c$dsc("LINE_EDITING"))) == CLI$_NEGATED) {
    line_editing = 0;  /* set_noline(); */
    if (!temp) profile_flags &= ~PROFILE_LINEEDIT;
    }
  else if (status & 1) {
    set_line();
    if (!temp) {
      if (line_editing) profile_flags |= PROFILE_LINEEDIT;
      else profile_flags &= ~PROFILE_LINEEDIT;
      }
    }
  if ((status = cli$present(c$dsc("DOTNEWSRC"))) == CLI$_NEGATED) {
    set_nodotnewsrc();
    if (!temp) profile_flags &= ~PROFILE_DOTNEWSRC;
    }
  else if (status & 1) {
    set_dotnewsrc();
    if (!temp) {
      if (usedotnewsrc) profile_flags |= PROFILE_DOTNEWSRC;
      else profile_flags &= ~PROFILE_DOTNEWSRC;
      }
    }
  if ((status = cli$present(c$dsc("BROADCAST_TRAPPING"))) == CLI$_NEGATED) {
    no_broad_trap();  /* set_nobroad(); */
    if (!temp) profile_flags |= PROFILE_BROADCAST;
    }
  else if (status & 1) {
    set_broad();
    if (!temp) {
      if (broadcast_trapping_requested) profile_flags &= ~PROFILE_BROADCAST;
      else profile_flags |= PROFILE_BROADCAST;
      }
    }
  if ((status = cli$present(c$dsc("AUTOMATIC"))) == CLI$_NEGATED) {
    profile_auto = 0;
    if (!temp) profile_flags &= ~PROFILE_AUTO;
    }
  else if (   (status & 1)
           && (cli$get_value(c$dsc("AUTOMATIC"),&val_dsc,&val_len) & 1)
           && val_len) {
    val[val_len] = '\0';
    s_to_upper(val);
    if (!strncmp(val,"UNSEEN",val_len)) {
      profile_auto = PROFILE_UNSEEN;
      strcpy(profile_autostr,"UNSEEN");
      profile_flags |= PROFILE_AUTO;
      }
    else if (!strncmp(val,"DIRECTORY",val_len)) {
      profile_auto = PROFILE_DIRECTORY;
      strcpy(profile_autostr,"DIRECTORY");
      profile_flags |= PROFILE_AUTO;
      }
    else {
      profile_flags &= ~PROFILE_AUTO;
      strcat(err_oline,"Invalid AUTO value ");
      ++errs;
      }
    }

  if ((status = cli$present(c$dsc("RCORDER"))) == CLI$_NEGATED)
    reorder_groups = 0;
  else
    if (status & 1) reorder_groups = 1;

  if ((status = cli$present(c$dsc("RCFILTER"))) == CLI$_NEGATED)
    profile_filter = 0;
  else
    if (status & 1) profile_filter = 1;

  if ((status = cli$present(c$dsc("RCREG"))) == CLI$_NEGATED)
    profile_flags &= ~PROFILE_RCREG;
  else
    if (status & 1) profile_flags |= PROFILE_RCREG;

  if ((status = cli$present(c$dsc("CLASS"))) == CLI$_NEGATED) {
    if (!temp) profile_flags &= ~PROFILE_CLASS;
    }
  else if (!temp && (status & 1)) {
    profile_flags |= PROFILE_CLASS;
    strcpy(profile_classstr,"MAIN");
    if ((cli$get_value(c$dsc("CLASS"),&val_dsc,&val_len) & 1) && val_len) {
      val[val_len] = '\0';
      strcpy(profile_classstr,val);
      }
    }

  if ((status = cli$present(c$dsc("EDITOR"))) == CLI$_NEGATED) {
    if (!temp) profile_flags &= ~PROFILE_EDIT;
    }
  else if (status & 1) {
    *Editor = *Viewer = '\0';
    if (!temp) {
      strcpy(profile_editstr,"eve");
      profile_flags |= PROFILE_EDIT;
      }
    if ((cli$get_value(c$dsc("EDITOR"),&val_dsc,&val_len) & 1) && val_len) {
      val[val_len] = '\0';
      lower_case(val);
      if (!strcmp(val,"callable_tpu")) {
        *Editor = *Viewer = '\0';
        strcpy(val,"eve");
        }
      if (!strcmp(val,"tpu")) {
        *Editor = *Viewer = '\0';
        strcpy(val,"eve");
        }
      else if (!strcmp(val,"eve")) *Editor = *Viewer = '\0';
      else if (!strcmp(val,"callable_edt")) {
        strcpy(Editor,"edt");
        strcpy(Viewer,"edt");
        strcpy(val,"edt");
        }
      else {
        if (captive_account()) strcpy(val, "edt"); /* disable CAPTIVE accounts from SPAWNing */
        strcpy(Editor,val);
        strcpy(Viewer,val);
        }
      if (!temp) strcpy(profile_editstr,val);
      }
    }

  if ((status = cli$present(c$dsc("VIEWER"))) == CLI$_NEGATED) {
    if (!temp) profile_flags &= ~PROFILE_VIEW;
    }
  else if (status & 1) {
    *Viewer = '\0';
    if (!temp) {
      strcpy(profile_viewstr,"eve");
      profile_flags |= PROFILE_VIEW;
      }
    if ((cli$get_value(c$dsc("VIEWER"),&val_dsc,&val_len) & 1) && val_len) {
      val[val_len] = '\0';
      lower_case(val);
      if (!strcmp(val,"callable_tpu")) {
        *Viewer = '\0';
        strcpy(val,"eve");
        }
      if (!strcmp(val,"tpu")) {
        *Viewer = '\0';
        strcpy(val,"eve");
        }
      else if (!strcmp(val,"eve")) *Viewer = '\0';
      else if (!strcmp(val,"callable_edt")) {
        strcpy(Viewer,"edt");
        strcpy(val,"edt");
        }
      else {
        if (captive_account()) strcpy(val, "edt"); /* disable CAPTIVE accounts from SPAWNing */
        strcpy(Viewer,val);
        }
      if (!temp) strcpy(profile_viewstr,val);
      }
    }

  if ((status = cli$present(c$dsc("PERSONAL_NAME"))) == CLI$_NEGATED) {
    *usr_persname = '\0';
    if (!temp) profile_flags &= ~PROFILE_PERSNAME;
    }
  else if (   (status & 1)
           && (cli$get_value(c$dsc("PERSONAL_NAME"),&val_dsc,&val_len) & 1)
           && val_len) {
    val[val_len] = '\0';
    strcpy(usr_persname,val);
    if (!temp) {
      strcpy(profile_persnamestr,val);
      profile_flags |= PROFILE_PERSNAME;
      }
    }

  if ((status = cli$present(c$dsc("REPLYTO"))) == CLI$_NEGATED) {
    *profile_reply_to = '\0';
    if (!temp) profile_flags &= ~PROFILE_REPLYTO;
    }
  else if (   (status & 1)
           && (cli$get_value(c$dsc("REPLYTO"),&val_dsc,&val_len) & 1)
           && val_len) {
    val[val_len] = '\0';
    strcpy(profile_reply_to,val);
    if (!temp) {
      strcpy(profile_reply_to_str,val);
      profile_flags |= PROFILE_REPLYTO;
      }
    }

  if (   ((status = cli$present(c$dsc("NEWREGISTER"))) == CLI$_NEGATED)
      && !temp) {
    profile_flags |= PROFILE_NEWREGISTER;
    strcpy(profile_newregstr,"none");
    }
  else if ((status & 1) && !temp) {
    profile_flags |= PROFILE_NEWREGISTER;
    strcpy(profile_newregstr,"*");
    if ((cli$get_value(c$dsc("NEWREGISTER"),&val_dsc,&val_len) & 1)
        && val_len) {
      val[val_len] = '\0';
      lower_case(val);
      strcpy(profile_newregstr,val);
      }
    }
  if ((status = cli$present(c$dsc("SCANSIZE"))) == CLI$_NEGATED) {
    if (!temp) profile_flags &= ~PROFILE_SCANSIZE;
    else profile_scansize = -1;
    }
  else if (   (status & 1)
           && (cli$get_value(c$dsc("SCANSIZE"),&val_dsc,&val_len) & 1)
           && val_len) {
    val[val_len] = '\0';
    if (sscanf(val,"%d",&profile_scansize) != 1) profile_scansize = 0;
    else if (!temp) profile_flags |= PROFILE_SCANSIZE;
    if (!temp) profile_rc_scansize = profile_scansize;
    }
  if ((status = cli$present(c$dsc("FROMSIZE"))) == CLI$_NEGATED) {
    do_update_call = 1;
    if (!temp) profile_minfromlen = 0;
    minfromlen = 0;
    }
  else if (status & 1) {
    do_update_call = 1;
    if (!temp) profile_minfromlen = 19;
    minfromlen = 19;
    if (  (cli$get_value(c$dsc("FROMSIZE"),&val_dsc,&val_len) & 1) && val_len) {
      val[val_len] = '\0';
      if (sscanf(val,"%d",&minfromlen) != 1) minfromlen = 19;
      else if (!temp) profile_minfromlen = minfromlen;
      }
    }
  if ((status = cli$present(c$dsc("SIGNATURE"))) == CLI$_NEGATED) {
    if (!temp) profile_flags &= ~PROFILE_SIGNATURE;
    else *profile_signature = '\0';
    }
  else if (   (status & 1)
           && (cli$get_value(c$dsc("SIGNATURE"),&val_dsc,&val_len) & 1)
           && val_len) {
    val[val_len] = '\0';
    if (!temp) {
      profile_flags |= PROFILE_SIGNATURE;
      strcpy(profile_rc_signature,val);
      }
    strcpy(profile_signature,val);
    }
  if ((status = cli$present(c$dsc("LOCSIGNATURE"))) == CLI$_NEGATED) {
    if (!temp) profile_flags &= ~PROFILE_LOCSIGNATURE;
    else *profile_locsignature = '\0';
    }
  else if (   (status & 1)
           && (cli$get_value(c$dsc("LOCSIGNATURE"),&val_dsc,&val_len) & 1)
           && val_len) {
    val[val_len] = '\0';
    if (!temp) {
      profile_flags |= PROFILE_LOCSIGNATURE;
      strcpy(profile_rc_locsignature,val);
      }
    strcpy(profile_locsignature,val);
    }
  if ((status = cli$present(c$dsc("ORGANIZATION"))) == CLI$_NEGATED) {
    if (!temp) {
      profile_flags &= ~PROFILE_ORGANIZATION;
      *profile_rc_organization = '\0';
      }
    *profile_organization = '\0';
    }
  else if (   (status & 1)
           && (cli$get_value(c$dsc("ORGANIZATION"),&val_dsc,&val_len) & 1)
           && val_len) {
    val[val_len] = '\0';
    if (!temp) {
      profile_flags |= PROFILE_ORGANIZATION;
      strcpy(profile_rc_organization,val);
      }
    strcpy(profile_organization,val);
    }
  if ((status = cli$present(c$dsc("KEYDEFS"))) == CLI$_NEGATED) {
    if (!temp) {
      profile_flags &= ~PROFILE_KEYDEFS;
      *profile_keydefs = '\0';
      }
    }
  else if (   (status & 1)
           && (cli$get_value(c$dsc("KEYDEFS"),&val_dsc,&val_len) & 1)
           && val_len) {
    FILE *fpr;

    val[val_len] = '\0';
    if (!temp) {
      profile_flags |= PROFILE_KEYDEFS;
      strcpy(profile_keydefs,val);
      }
    if ((fpr = fopen(val,"r")) != 0) {  /* use the access() call instead */
      fclose(fpr);
      smg$load_key_defs(&keytab,c$dsc(val),0,c$ac(0));
      }
    }
  if ((status = cli$present(c$dsc("FASTLOAD"))) == CLI$_NEGATED) {
    if (!temp) {
      profile_flags &= ~PROFILE_FASTLOAD;
#if FAST_LOAD
      if (fast_loading) {
        fast_loading = 0;
        do_update_call = 1;
        }
#endif
      }
    }
  else if (status & 1) {
    profile_flags |= PROFILE_FASTLOAD;
#if FAST_LOAD
    if (!fast_loading) {
      fast_loading = 1;
      all_loaded = 1;
      }
#endif
    }

  if ((status = cli$present(c$dsc("PRINT"))) == CLI$_NEGATED) {
    *print_constant = '\0';
    if (!temp) profile_flags &= ~PROFILE_PRINT;
    }
  else if (   (status & 1)
           && (cli$get_value(c$dsc("PRINT"),&val_dsc,&val_len) & 1)
           && val_len) {
    char try_command[256];

    val[val_len] = '\0';
    sprintf(try_command,"PRINT%s 1",val);
    if (cli$dcl_parse(c$dsc(try_command),CLICMDTBL,0,0,0) & 1) {
      strcpy(print_constant,val);
      if (!temp) {
        profile_flags |= PROFILE_PRINT;
        strcpy(profile_printstr,val);
        }
      }
    else {
      strcat(err_oline,"Invalid PRINT value");
      ++errs;
      }
    }

  if ((status = cli$present(c$dsc("DIRECTORY"))) == CLI$_NEGATED) {
    *profile_dirstr = '\0';
    if (!temp) profile_flags &= ~PROFILE_DIRECTORY;
    }
  else if (   (status & 1)
           && (cli$get_value(c$dsc("DIRECTORY"),&val_dsc,&val_len) & 1)
           && val_len) {
    char try_command[256];

    val[val_len] = '\0';
    strcpy(try_command,"DIRECTORY/");
    strcat(try_command,val);
    if (cli$dcl_parse(c$dsc(try_command),CLICMDTBL,0,0,0) & 1) {
      if (!temp) {
        profile_flags |= PROFILE_DIRECTORY;
        strcpy(profile_dirstr,val);
        }
      }
    else {
      strcat(err_oline,"Invalid DIRECTORY value");
      ++errs;
      }
    }

  if (!errs) strcat(err_oline,"Profile modified");
  err_line(err_oline);
  dsprofile();
  if (do_update_call) unwind_display(OUTER_LOOP,do_update_1);
  return(0);
}

int do_set_profile()
{
  return(unwind_display(I_DISPLAY_LOOP,dsetprofile));
}

void selgrp_action()
{
  if ((profile_flags & PROFILE_AUTO) && (profile_auto == PROFILE_UNSEEN)) {
    while (display_new(1,0,0,0) == 2);
    }
}

FILE *profile_sigfile(groups)
  char *groups;
{
  FILE *fpr = 0;

  if (groups && all_local(groups)) {
    if ( !(profile_flags & PROFILE_SIGNATURE)
      || !*profile_locsignature
      || (*profile_locsignature == '@')) {
      if (   (fpr = fopen("LOCSIGNATURE","r"))
          || (fpr = fopen("MAIL_LOCSIGNATURE","r"))
          || (fpr = fopen("SYS$LOGIN:MAIL.LOCSIGNATURE","r")))
        return(fpr);
      }
    else return(fpr = fopen(profile_locsignature,"r"));
    }

  if ( !(profile_flags & PROFILE_SIGNATURE)
    || !*profile_signature
    || (*profile_signature == '@')) {
    if (   (fpr = fopen("SIGNATURE","r"))
        || (fpr = fopen("MAIL_SIGNATURE","r"))
        || (fpr = fopen("SYS$LOGIN:MAIL.SIGNATURE","r")))
      return(fpr);
      else return(0);
    }
  return(fpr = fopen(profile_signature,"r"));
}

/*
 * This function clears the profile_flags variable.  It is used by arbitron
 * in between reading NEWSRC files of different users.  It must be included
 * in newsrc.c because profile_flags is static to that module.
 */

void clear_profile_flags()
{
  profile_flags = 0;
}

/*### **********************
static struct pro_record {
   char pro_signature[256];
   char pro_locsignature[256];
   char pro_organisation[256];
   } pro_actual, pro_df, pro_min, pro_max;

static int pro_form;

char *frm$news_malloc(size) unsigned size; { return(news_malloc(size)); }

frm$news_free(p) void *p; { news_free(p); }

void edit_profile()
{
   frm$set(&pro_form,&pro_actual,&pro_df,&pro_min,&pro_max);
   frm$txt(1,1,pro_form,"NEWS Profile Settings");
   frm$ustr(3,1,0,pro_form,"Signature File: ","Name of file containing Signature text",60);

}

*************************/
