/*
**++
**  FACILITY:
**      NEWSREMCLIENT
**
**  ABSTRACT:
**      Interface between local NEWS reader and a remote NNTP implementation.
**      This module contains all necessary calls to access the remote site
**      using NNTP.
**
**  AUTHOR:
**      Geoff Huston
**
**  COPYRIGHT:
**      Copyright Geoff Huston  1988,1989,1990,1991
**
**  MODIFICATION HISTORY:
**      V5.5     7-Oct-1988     GIH
**        - Included modifications for WIN TCP/IP V3.0 support by Jim Patterson
**          (jimp@cognos.uucp)
**          NOTE: I have NOT tested the WIN support modifications.
**        - Changed DECnet object name to "NNTP".
**        - Allow task definition by the logical name NEWS_node_TASK
**          i.e. DEFINE NEWS_CSC_TASK "NEWS="
**      V5.6    11-Nov-1988     GIH
**        - Add get_server_header to get the remote item header only
**      V5.7     1-Dec-1988     PK
**        - Changes to support WIN TCP 3.2 - Pekka Kytolaakso
**        - Add XHDR call to speed up title search if supported at remote end
**        - Bug fix in open_net() - Pekka Kytolaasko
**      V5.8    19-Jan-1989     GIH
**        - Bug fix in open_net() - Pekka Kytolaasko
**        - Added conditional compilation for socket stream emulation in the
**          DECnet QIO interface - "DECNETSTREAM" controls inclusion of stream
**          QIO calls.
**        - If a control-c is typed while NEWS is making a connection to
**          a remote server, the program terminates and all items in the
**          database at that time become inaccessible, even for subsequent
**          NEWS invocations (connection attempts to the remote server are
**          lost.) net_write() - Stuart McGraw
**        - Add support for MULTINET Multinet - Mats Sundvall
**	V6.0	12-Jun-1989	GIH
**	  - Add support for UCX
**	V6.0-1	 1-Nov-1990	SD
**	  - Add support for NNTP-II posting authorization - Sean Donelan
**	V6.0-1	 5-Nov-1990	glass@mgi.com
**	  - Alter response in post-to-server to display server's response
**	V6.0-3	 4-Feb-1991	glass@mgi.com
**	  - Alter nntp_one_call to detect no answer response
**	V6.0-3	11-Mar-1991	GIH
**	  - Add support for the LIST NEWSGROUPS call instead of XGTITLE
**	V6.0-4	20-Mar-1991      Volker C. Huck huck@linac.kernphysik.th-darmstadt.de
**	  - UCX bug fixes
**	V6.0-5	25-May-1991	volz@process.com
**	  - Added TCPware support
**	V6.1	 9-Jun-1991	Andrew Greer (Andrew.Greer@vuw.ac.nz)
**	  - Fix ARTICLE retrieve not reestablishing GROUP context
**	V6.1	 8-Feb-1992	rankin@eql.caltech.edu
**	  - Lint cleanup
**	V6.1b7	15-May-1993	Charles Bailey  bailey@genetics.upenn.edu
**	  - optimize item number lookup and message-ID handling
**	V6.1b9	17-Aug-1994     Mark Martinec   mark.martinec@ijs.si
**	  - code cleanup to make it compile under gcc 2.6.0 with full
**	    warnings reporting turned on - with no or very few harmless warnings
**	V6.1b9	17-Sep-1994     Mark Martinec   mark.martinec@ijs.si
**	  - code cleanup to preserve the read-only nature of string literals
**	    (strategically placed 'const' attribute to string parameters)
**	V6.1b9	23-Sep-1994     Mark Martinec   mark.martinec@ijs.si
**	  - replace testing macros MULTINET, UCX, TWG, ... for value
**	    with a test for defined(), to prevent evaluating an undefined macro
**	V6.1b9	20-Oct-1994     Mark Martinec   mark.martinec@ijs.si
**	  - nntp_read() and nntp_read_line():  use memcpy for efficiency,
**	    be more careful and avoid writing beyond buffer size
**	  - nntp_write(): bump obuf buffer size to accomodate
**	    inpline buffer in a call to nntp_write() from post_to_server()
**	  - remove definition of MAX_NNTP_SIZE as it is not used anymore
**
**  CONTRIBUTIONS:
**	  - Add support for Excelan - Pekka Kyt|laakso
**--
**/

#ifdef vaxc
#module NEWSREMCLIENT "V6.1"
#endif

#define _NEWSREMCLIENT_C
#define module_name "NEWSREMCLIENT"

#include "newsinclude.h"
#include "newsextern.h"
#include "nntpinclude.h"  /* must follow #include "newsextern.h" */


#define wmg(s)	if (smg_active) {			\
		  smg$end_pasteboard_update(&pid);	\
		  smg$begin_pasteboard_update(&pid);	\
		  }					\
		err_line(s);				\
		if (smg_active) {			\
		  smg$end_pasteboard_update(&pid);	\
		  smg$begin_pasteboard_update(&pid);	\
		  }

#define WEEK		(60*60*24*7)

#define NNTP_PORT       (119)

#define CMUTCP     	1
#define WINTCP     	2
#define MULTINETTCP     3
#define UCXTCP     	4
#define EXOSTCP		5
#define TCPWARETCP	6

#define AUTH 1

#if defined(TWG) || defined(MULTINET) || defined(TCPWARE)
struct hostent *dest_host;
struct sockaddr_in data_socket = {0};
#endif

#ifdef UCX

#ifdef UCXQIOS
/*   VCH001 changed sockaddr to socketaddr for name cludge with UCX V1.3  */
struct socketaddr {
  short inet_family;
  short inet_port;
  int inet_adrs;
  char bklb[8];
  };

struct itlist { int lgth; struct socketaddr *hst; };

static short sck_parm[2];
static struct socketaddr local_host, remote_host;
struct itlist lhst_adrs, rhst_adrs;
#endif
#endif

#ifdef EXOS
#define EXOS 1
#include        "exos_etc:ttcp.h"
struct ex_iosb iosb;	      	/* I/O status block */
struct SOioctl sock;    	/* Socket control structure     */
struct SOioctl remsock;         /* remote socket control structure */
$DESCRIPTOR(exos, "EX");    	/* EXOS device name descriptor */
long  addr;                   	/* Internet address */

static struct sockaddr_in send_socket = { AF_INET };
static struct sockaddr_in recv_socket = { AF_INET };
#endif

#define CLIENT_TIMER 500
#define RESP_TIMER 300
#define TMP_FILE_NAME "SYS$SCRATCH:NEWS_%X_NNTP.TMP"
#define TMP_HEAD_NAME "SYS$SCRATCH:NEWS_%X_NNTPH.TMP"
#define X_BUF_SIZE 1024

char client_msg[132];
char get_server_title[132];
char get_server_from[132];
char get_server_id[132];
int  get_server_size;

static
char net_open_chan[128] = {""},
     ibuf[X_BUF_SIZE + 1],
     sav_id[256] = {""},
     sav_idh[256] = {""},
     scratch_file[256] = {""},
     scratch_head[256] = {""};

static
unsigned short net_chan;

extern char nntp_currgroup[];
#ifdef UCXSOCKETS
static int socket_1;					/* socket */
static struct sockaddr_in socket_2_name;         	/* Address struct for remote server */
static struct in_addr inetadd;
static struct hostent hostentstruct, *hostentptr;
#endif

static
int net_read_status = 0,
    cmd_code,
    net_proto = 0,
    sav_itm = 0,
    sav_grp = 0;

#ifdef __DECC
#pragma member_alignment save
#pragma nomember_alignment   /* no member alignment - this is a VMS structure */
#endif
static
struct iosb {
    unsigned short iostatus;
    unsigned short iosize;
    int netinfo;
    } read_iosb;
#ifdef __DECC
#pragma member_alignment restore
#endif

static
struct nol {
    char *nodename;
    char *taskname;
    struct nol *next;
    } *nh = 0;


#ifdef UCXQIOS
static unsigned int get_local_adrs(void)
{
  static unsigned int localhost = 0;

  if (!localhost) {
    const char *gotenv;
    short nam_chan, retlen;
    int comm = INETACP$C_TRANS * 256 + INETACP_FUNC$C_GETHOSTBYNAME;
    struct dsc$descriptor command = {4,DSC$K_CLASS_S, DSC$K_DTYPE_T,(char *)&comm};
    struct dsc$descriptor host_ad = {4,DSC$K_CLASS_S, DSC$K_DTYPE_T,(char *)&localhost};
    struct iosb nam_iosb;

    if (!(gotenv = news_getenv("UCX$INET_HOST",1))) gotenv = "localhost";
    if (sys$assign(c$dsc("BG0:"),&nam_chan,0,0) & 1) {
      if (!(   (sys$qiow(0,nam_chan,IO$_ACPCONTROL,&nam_iosb,0,0,
                       &command,c$dsc(gotenv),&retlen,&host_ad,0,0) & 1)
            && (nam_iosb.iostatus & 1))) localhost = (1 << 24) + 127;
      sys$dassgn(nam_chan);
      }
    }
  return(localhost);
}

static unsigned int get_host_adrs(host_name)
  char *host_name;
{
  unsigned int hostadrs = 0;
  short nam_chan, retlen;
  int comm = INETACP$C_TRANS * 256 + INETACP_FUNC$C_GETHOSTBYNAME;
  struct dsc$descriptor command = {4,DSC$K_CLASS_S, DSC$K_DTYPE_T,(char *)&comm};
  struct dsc$descriptor host_ad = {4,DSC$K_CLASS_S, DSC$K_DTYPE_T,(char *)&hostadrs};
  struct iosb nam_iosb;

  if (sys$assign(c$dsc("BG0:"),&nam_chan,0,0) & 1) {
    if (!(   (sys$qiow(0,nam_chan,IO$_ACPCONTROL,&nam_iosb,0,0,
                       &command,c$dsc(host_name),&retlen,&host_ad,0,0) & 1)
          && (nam_iosb.iostatus & 1))) hostadrs = 0;
    sys$dassgn(nam_chan);
    }
  return(hostadrs);
}
#endif	/*UCXQIOS*/

/*
 *  nntp_write
 *
 *  Synchronous write of a string to the net channel
 */

static int nntp_write(b)
  const char *b;
{
  static char obuf[1026];
  struct iosb write_iosb;
  int sts;

  strcpy(obuf,b);
  strcat(obuf,"\r\n");

#ifdef UCXSOCKETS
  if (send(socket_1,obuf,strlen(obuf),0) < 0) {
    sprintf(client_msg,"NNTP: Lost connection to %s (%s)",net_open_chan,strerror(errno));
    close_net();
    net_read_status = 0;
    return(0);
    }
#else
#if defined(DECNETSTREAM)
  sts = sys$qiow(0,net_chan,IO$_WRITEVBLK | IO$M_MULTIPLE,&write_iosb,0,0,obuf,strlen(obuf),0,(net_proto == CMUTCP),0,0);
#elif defined(EXOS)
  sts = sys$qiow(0,net_chan,EX__WRITE,&write_iosb,0,0,obuf,strlen(obuf),0,0,0,0);
#else
  sts = sys$qiow(0,net_chan,IO$_WRITEVBLK,&write_iosb,0,0,obuf,strlen(obuf),0,(net_proto == CMUTCP),0,0);
#endif
  if (!(sts & 1) || (!(write_iosb.iostatus & 1))) {
    sprintf(client_msg,"NNTP: Lost connection to Server: %s",net_open_chan);
    close_net();
    net_read_status = 0;
    return(0);
    }
#endif
  return(1);
}

/*
 *  nntp_read
 *
 *  Timed read of the net channel for a full line
 */

static jmp_buf nntp_env;

static void cancel_nntp_read(sig)
  int sig;	/* not used */
{
  sprintf(client_msg,"NNTP: Lost connection to Server: %s",net_open_chan);
  close_net();
  longjmp(nntp_env,1);
}

static int nntp_read(b,timer)
  char *b;
  int timer;
{
  static char ibuf_local[X_BUF_SIZE + 1];
  struct iosb r_iosb;
  int ibuf_local_l;
  int sts;

  *b = '\0';

#ifdef UCXSOCKETS
  if ((r_iosb.iosize = recv(socket_1,ibuf_local,X_BUF_SIZE,0)) <= 0) {
    sprintf(client_msg,"NNTP: Lost link to %s (%s)",net_open_chan,strerror(errno));
    close_net();
    return(-1);
    }
  else {
    alarm(0);
    sts = 1;
    r_iosb.iostatus = 1;
    }
#else
  if (setjmp(nntp_env)) return(-1);
  signal(SIGALRM,cancel_nntp_read);
  alarm(timer);
#if defined(DECNETSTREAM)
  sts = sys$qiow(0,net_chan,IO$_READVBLK | IO$M_MULTIPLE,&r_iosb,0,0,ibuf_local,X_BUF_SIZE,0,0,0,0);
#elif defined(EXOS)
  sts = sys$qiow(0,net_chan,EX__READ,&r_iosb,0,0,ibuf_local,X_BUF_SIZE,0,0,0,0);
#else
  sts = sys$qiow(0,net_chan,IO$_READVBLK,&r_iosb,0,0,ibuf_local,X_BUF_SIZE,0,0,0,0);
#endif
  alarm(0);
  if (!(sts & 1) || (!(r_iosb.iostatus & 1))) {
    sprintf(client_msg,"NNTP: Lost connection to Server: %s",net_open_chan);
    close_net();
    return(-1);
    }
  if (!r_iosb.iosize) {
    sprintf(client_msg,"NNTP: Read error - Lost connection to Server: %s",net_open_chan);
    close_net();
    return(-1);
    }
#endif
  ibuf_local[r_iosb.iosize] = '\0';
  ibuf_local_l = r_iosb.iosize;

  { char *rp, *cp;
    int rp_l, ll;
    int b_l = 0;

 /* copy ibuf_local to b, removing all <CR> characters */
    for (rp=ibuf_local, rp_l=ibuf_local_l;  rp_l>0; ) {
      if ((cp = memchr(rp,'\r',rp_l)) == NULL)
        { memcpy(&b[b_l],rp,ll=rp_l); rp_l = 0; rp = NULL; }
      else
        { memcpy(&b[b_l],rp,ll=cp-rp); rp_l -= ll+1; rp = cp+1; }
      b_l += ll;
      }
    b[b_l] = '\0';
    return(b_l);
  }
}

static char lbuf[X_BUF_SIZE + 1] = {""};
static int  lbuf_l = 0;     /* actual length of string in lbuf */
static int  lbuf_ind = 0;   /* index of the next character in lbuf to be read */

static int nntp_read_line(b,timer)
  char *b;
  int timer;
{
  char *cp;
  int b_l,ll;

  *b = '\0';
  for (b_l=0; b_l<X_BUF_SIZE; ) {
    if (lbuf_l-lbuf_ind <= 0) {
      lbuf_ind = 0; lbuf_l = nntp_read(lbuf,timer);
      if (lbuf_l < 0) { lbuf_l = 0; return(0); }
      }
    cp = memchr(&lbuf[lbuf_ind], '\n', ll=min(lbuf_l-lbuf_ind,X_BUF_SIZE-b_l));
    if (cp != NULL) ll=cp-&lbuf[lbuf_ind]+1;   /* take string only up to '\n' */
    memcpy(&b[b_l],&lbuf[lbuf_ind],ll); b_l += ll; b[b_l] = '\0';
    lbuf_ind += ll;
    if (cp != NULL) return(1);           /* return if we have a complete line */
    }
  return(1);/* return full buffer even if it does not contain a complete line */
}

/*
 *  wait_net_response
 *
 *  Wait for a response from remote system - cmd indicates that all text
 *  should be skipped until NNTP command response is obtained
 */

int wait_net_response(secs,cmd)
  int secs,
      cmd;
{
  if (nntp_read_line(ibuf,secs)) {
    if (!cmd) return(cmd_code = 1);
    if (sscanf(ibuf,"%d",&cmd_code) == 1) {
      if ((cmd_code == 400) || (cmd_code == 205)) {
        close_net();
        return(0);
        }
      else return(cmd_code);
      }
    else return(cmd_code = 1);
    }
  return(cmd_code = 0);
}

/*
 *  open_net
 *
 *  Open network link
 */

int open_net(node,proto)
  char *node;
  int proto;
{
  char netobj[64];

  strcpy(netobj,node);
  switch (proto) {

  case CMUTCP:
#ifdef CMU
    if (sys$assign(c$dsc("INET$DEVICE"),&net_chan,0,0) & 1) {
		/* was     if (sys$assign(c$dsc("IP:"),&net_chan,0,0) & 1) { */
      if (   (sys$qiow(0,net_chan,IO$_CREATE,&read_iosb,0,0,netobj,NNTP_PORT,0,1,0,300) & 1)
          && (read_iosb.iostatus & 1))
        return(1);
      sys$dassgn(net_chan);
      }
#else
    sprintf(err_oline,"\tERROR - CMU TCP/IP support not available\n");
    wmg(err_oline);
#endif
    return(0);

  case WINTCP:
#ifdef TWG
    if (sys$assign(c$dsc("INET:"),&net_chan,0,0) & 1) {
      if (   (sys$qiow(0,net_chan,IO$_SOCKET,&read_iosb,0,0,AF_INET,SOCK_STREAM,0,0,0,0) & 1)
          && (read_iosb.iostatus & 1)) {
        if (!(dest_host = gethostbyname(netobj))) {
	  sys$dassgn(net_chan);
	  return(0);
	  }
        data_socket.sin_addr.s_addr = *((u_long *) dest_host->h_addr);
        data_socket.sin_family = AF_INET;
        data_socket.sin_port = htons(NNTP_PORT);
        if (   (sys$qiow(0,net_chan,IO$_CONNECT,&read_iosb,0,0,&data_socket,sizeof(data_socket),0,0,0,0) & 1)
            && (read_iosb.iostatus & 1))
	  return(1);
        }
      }
    sys$dassgn(net_chan);
#else
    sprintf(err_oline,"\tERROR - WIN TCP/IP support not available\n");
    wmg(err_oline);
#endif
    return(0);

  case MULTINETTCP:
  case TCPWARETCP:
#if defined(MULTINET) || defined(TCPWARE)
    if (sys$assign(c$dsc("INET0:"),&net_chan,0,0) & 1) {
      if (   (sys$qiow(0,net_chan,IO$_SOCKET,&read_iosb,0,0,AF_INET,SOCK_STREAM,0,0,0,0) & 1)
          && (read_iosb.iostatus & 1)) {
        if (!(dest_host = gethostbyname(netobj))) {
	  sys$dassgn(net_chan);
	  return(0);
	  }
        data_socket.sin_addr.s_addr = *((u_long *)dest_host->h_addr);
        data_socket.sin_family = AF_INET;
        data_socket.sin_port = htons(NNTP_PORT);
        if (   (sys$qiow(0,net_chan,IO$_CONNECT,&read_iosb,0,0,&data_socket,sizeof(data_socket),0,0,0,0) & 1)
            && (read_iosb.iostatus & 1))
	  return(1);
        }
      }
    sys$dassgn(net_chan);
#else
    sprintf(err_oline,"\tERROR - %s TCP/IP support not available\n",
		proto == MULTINETTCP ? "MultiNet" : "TCPware");
    wmg(err_oline);
#endif
    return(0);

  case UCXTCP:
#ifdef UCX
#ifdef UCXQIOS
    if (sys$assign(c$dsc("BG:"),&net_chan,0,0) & 1) {
      sck_parm[0] = INET$C_TCP;
      sck_parm[1] = INET_PROTYP$C_STREAM;
      local_host.inet_family = INET$C_AF_INET;
      local_host.inet_port = 0;  /* dynamic port assignment */
      local_host.inet_adrs = get_local_adrs();
      lhst_adrs.lgth = sizeof local_host;
      lhst_adrs.hst = &local_host;
      if (   (sys$qiow(0,net_chan,IO$_SETMODE,&read_iosb,0,0,&sck_parm,0x01000000|SOCKOPT$M_KEEPALIVE,&lhst_adrs,0,0,0) & 1)
          && (read_iosb.iostatus & 1)) {
        remote_host.inet_family = INET$C_AF_INET;
        remote_host.inet_port = htons(NNTP_PORT);
        if (!(remote_host.inet_adrs = get_host_adrs(netobj)))  {
          char *c, *cd, resnum[4];
          unsigned int indx = 0, *i;

	  c = netobj;
          while (c && (indx <= 3)) {
	    if ((cd = strchr(c,'.')) != 0) *cd = '\0';
            resnum[indx++] = atoi(c);
            if (cd) {
              *cd++ = '.';
              c = cd;
              }
            else c = 0;
            }
          if (c || indx != 4) {
            sys$qiow(0,net_chan, IO$_DEACCESS | IO$M_SHUTDOWN,read_iosb,0,0,0,0,0,UCX$C_DSC_ALL,0,0);
	    sys$dassgn(net_chan);
	    return(0);
            }
          i = (unsigned int *) &resnum[0];
          remote_host.inet_adrs = *i;
          }
        rhst_adrs.lgth = sizeof remote_host;
        rhst_adrs.hst = &remote_host;
        if (   (sys$qiow(0,net_chan,IO$_ACCESS,&read_iosb,0,0,0,0,&rhst_adrs,0,0,0) & 1)
            && (read_iosb.iostatus & 1))
          return(1);
        sys$qiow(0,net_chan, IO$_DEACCESS | IO$M_SHUTDOWN,read_iosb,0,0,0,0,0,UCX$C_DSC_ALL,0,0);
        }
      sys$dassgn(net_chan);
      }
#else	/* !UCXQIOS */
    if ((socket_1 = socket(AF_INET,SOCK_STREAM,0)) == -1) {
      perror("socket");
      return(0);
      }
    socket_2_name.sin_port = htons(NNTP_PORT);
    if ((inetadd.S_un.S_addr = inet_addr(netobj)) != -1) {
      socket_2_name.sin_family = AF_INET;
      socket_2_name.sin_addr = inetadd;
      }
    else {
      if ((hostentptr = gethostbyname(netobj)) == NULL) {
        perror("gethostbyname");
        close(socket_1);
        return(0);
        }
      hostentstruct = *hostentptr;
      socket_2_name.sin_family = hostentstruct.h_addrtype;
      socket_2_name.sin_addr = *((struct in_addr *) hostentstruct.h_addr);
      }
    if (!connect(socket_1,&socket_2_name,sizeof(socket_2_name)))
      return(1);	/* success */
    else {
      perror("connect");
      close(socket_1);
      }
#endif	/* ?UCXQIOS */
#else
    sprintf(err_oline,"\tERROR - UCX TCP/IP support not available\n");
    wmg(err_oline);
#endif
    return(0);   

  case EXOSTCP:
#ifdef EXOS
    char *host=netobj;

    if ((addr = rhost(&host)) == -1) return(0);

    /* set up data structure here   */
    send_socket.sin_port = 0;       /* use any port address for local */
    recv_socket.sin_port = htons(NNTP_PORT);
    recv_socket.sin_addr.s_addr = addr;
    sock.hassa = 1;
    sock.sa.sin = send_socket;
    sock.hassp = 0;
    sock.type = SOCK_STREAM;
    sock.options = 0;
    remsock.hassa = 1;
    remsock.sa.sin = recv_socket;
    remsock.hassp = 0;
    remsock.type = 0;
    remsock.options = 0;

    if (   (sys$assign(&exos,&net_chan,0,0) & 1)
        && (sys$qiow(0,net_chan,EX__SOCKET,&iosb,0,0,0,0,&sock,0,0,0) & 1)
        && (iosb.ex_stat & 1)
	&& (sys$qiow(0,net_chan,EX__CONNECT,&iosb,0,0,0,0,&remsock,0,0,0) & 1)
        && (iosb.ex_stat & 1))
      return(1);
#else
    sprintf(err_oline,"\tERROR - EXOS TCP/IP support not available\n");
    wmg(err_oline);
#endif
    return(0);

  default:	/* DECnet */
  {
    struct nol *tmp;

    lower_case(node);
    tmp = nh;
    while (tmp) {
      if (!strcmp(node,tmp->nodename)) break;
      tmp = tmp->next;
      }
    if (!tmp) {
      char logname[64],
           *gotenv;

      sprintf(logname,"NEWS_%s_TASK",node);
      s_to_upper(logname);
      if ( (gotenv = news_getenv(logname,0)) ) strcpy(logname,gotenv);
      else sprintf(logname,"TASK=NNTP");
      s_to_upper(logname);

      tmp = (struct nol *) news_malloc(sizeof *tmp);
      strcpy((tmp->nodename = (char *) news_malloc(strlen(node) + 1)),node);
      strcpy((tmp->taskname = (char *) news_malloc(strlen(logname) + 1)),logname);
      tmp->next = nh;
      nh = tmp;
      }
    sprintf(netobj,"%s::\"%s\"",node,tmp->taskname);
    return(sys$assign(c$dsc(netobj),&net_chan,0,0) & 1);
    }
  } /*switch(proto)*/
  return(0);
}

/*
 *  close_net
 *
 *  Close network link
 */

void close_net()
{
#ifdef UCX
  if (net_proto == UCXTCP) {
#ifdef UCXSOCKETS
    shutdown(socket_1,2);
    close(socket_1);
#else
    sys$cancel(net_chan);
    sys$qiow(0,net_chan,IO$_DEACCESS,0,0,0,0,0,0,0,0,0);
    sys$dassgn(net_chan);
#endif
    }
  else 
#endif
#ifdef EXOS
  if (net_proto == EXOSTCP) {
    int portion=1;

    sys$qiow(0,net_chan,EX__IOCTL,0,0,0,0,0,&portion,SIOCDONE,0,0);
    sys$dassgn(net_chan);
    }
  else
#endif
#if defined(MULTINET) || defined(TWG) || defined(CMU)
  if ((net_proto == MULTINETTCP) || (net_proto == WINTCP) || (net_proto ==CMUTCP)) {
    sys$cancel(net_chan);
    sys$qiow(0,net_chan,IO$_DELETE,0,0,0,0,0,0,0,0,0);
    sys$dassgn(net_chan);
    }
  else
#endif
  sys$dassgn(net_chan);
  *net_open_chan = 0;
  *lbuf = '\0'; lbuf_l = 0;
}

/*
 *  toggle_link
 *
 *  Write a non-command to the net chan and wait for the command-not-recognised
 *  response from the server.
 */

int toggle_link()
{
  if (!nntp_write("STAT") || (!(wait_net_response(RESP_TIMER,1)))) {
    if (*net_open_chan) {
      sprintf(client_msg,"Lost connection to NNTP-Server %s",net_open_chan);
      close_net();
      }
    return(0);
    }
  return(1);
}

/*
 *  open_rem_chan
 *
 *  Open channel to remote NNTP server on host name node, using protocol
 *  proto. Values for proto are:
 *    0 - DECNET
 *    1 - CMUTCP
 *    2 - WINTCP
 *    3 - MULTINETTCP
 *    4 - UCXTCP
 *    5 - EXOSTCP
 *    6 - TCPWARETCP
 */

int open_rem_chan(node,proto)
  const char *node;
  int proto;
{
  char *p;

  if (!*scratch_file) sprintf(scratch_file,TMP_FILE_NAME,getpid());
  if (!*scratch_head) sprintf(scratch_head,TMP_HEAD_NAME,getpid());
  if (*net_open_chan) {
    if ((proto != net_proto) || (news_strncasecmp(node,net_open_chan,0))) {
      close_net();
      net_read_status = 0;
      }
/*
    else toggle_link();
*/
    }

  if (!*net_open_chan) {
    strcpy(net_open_chan,node);
    lower_case(net_open_chan);
    if (!open_net(net_open_chan,proto)) {
      sprintf(client_msg,"Cannot connect to NNTP-Server %s",net_open_chan);
      net_read_status = 0;
      *net_open_chan = 0;
      return(0);
      }
    else net_proto = proto;
    if (!(wait_net_response(CLIENT_TIMER,1))) {
      net_read_status = 0;
      sprintf(client_msg,"Cannot connect to NNTP-Server %s",net_open_chan);
      close_net();
      return(0);
      }
    else {
      nntp_no_posting_allowed = (cmd_code == 201);
      nntp_anu_news_server = (strncmp(&ibuf[4],"ANU NEWS/NNTP",13) == 0);
      if ( (p=strchr(&ibuf[4],' ')) )
	  if ( !strncmp(p+1,"InterNetNews server",19) )
	      if ( nntp_write("mode reader") && wait_net_response(CLIENT_TIMER,1))
		  nntp_no_posting_allowed = (cmd_code == 201);
      }
    }
  return(1);
}

/*
 *  get_group_topnum
 */

int get_group_topnum(node,proto,group)
  char *node, *group;
  int proto;
{
  int resp,count,first,topnum;
  char cmd[256];

  if (!open_rem_chan(node,proto)) {
    sprintf(err_oline,"\tERROR - Cannot connect to NNTP SERVER (%s)\n",node);
    wmg(err_oline);
    return(-4);
    }
  sprintf(cmd,"GROUP %s",group);
  if (   !nntp_write(cmd)
      || (   ((resp = wait_net_response(CLIENT_TIMER,1)) != 211)
          && (resp != 411))) {
    sprintf(err_oline,"\tERROR - Lost link to NEWS SERVER (%s)\n",node);
    wmg(err_oline);
    return(-4);
    }
  if (resp == 411) return(-2);
  if (sscanf(ibuf,"%*d %d %d %d %*s",&count,&first,&topnum) < 3) return(-3);
  if (!count) return(-1);
  return(topnum);
}

/*
 *  get_server_file
 *
 *  Get a newsitem text from remote system
 */

FILE *get_server_file(id,node,tmp_name,put_file,proto,g,i)
  const char *id, *node, *put_file;
  char *tmp_name;
  int proto, g, i;
{
  FILE *fpw;
  char cmd[132];
  int data = 0;
  struct rms_file *ofile, *ifile;

  if (!strcmp(id,sav_id) && (fpw = fopen(scratch_file,"r"))) {
    *tmp_name = 1;
    return(fpw);
    }
  *sav_id = '\0';
  sav_itm = sav_grp = 0;
  while (!delete(scratch_file));

  if (!tmp_name) tmp_name = cmd;
  *tmp_name = '\0';
  if (!open_rem_chan(node,proto)) {
    sprintf(err_oline,"\tERROR - Cannot connect to NEWS SERVER (%s)\n",node);
    wmg(err_oline);
    return(0);
    }
  sprintf(cmd,"ARTICLE %s",id);
  if (!nntp_write(cmd) || (wait_net_response(CLIENT_TIMER,1) != 220)) {
    char *nntp_response;
    unsigned int inum;

    sprintf(err_oline,"GROUP %s",ga[g]->grp_name);
    if (nntp_one_call(node,proto,err_oline,&nntp_response) != 211) {
      sprintf(err_oline,"\tNNTP error - newsgroup %s",ga[g]->grp_name);
      wmg(err_oline);
      return(0);
      }
    strcpy(nntp_currgroup,ga[g]->grp_name);
    if (!find_itm_by_id(g,id,&inum)) {
      sprintf(err_oline,"\tItem search failed for %s",id);
      wmg(err_oline);
      return(0);
      }
    sprintf(cmd,"ARTICLE %d",inum);
    if (!nntp_write(cmd) || (wait_net_response(CLIENT_TIMER,1) != 220)) {
      sprintf(err_oline,"\tERROR - Item not returned from SERVER (%s)\n",node);
      wmg(err_oline);
      return(0);
      }
    }
  fpw = fopen(scratch_file,"w");
  get_server_size = 0;
  *get_server_title = '\0';
  *get_server_from = '\0';
  *get_server_id = 0;
  while (wait_net_response(RESP_TIMER,0) == 1) {
    if (!strcmp(ibuf,".\n")) break;
    if (*ibuf == '.') memmove(ibuf,ibuf + 1,strlen(ibuf));
    ++data;
    fputs(ibuf,fpw);
    if (!get_server_size && !strncmp(ibuf,"Lines:",6)) {
      if (!sscanf(ibuf,"Lines: %d",&get_server_size)) get_server_size = 0;
      }
    else if ((!*get_server_title) && !strncmp(ibuf,"Subject:",8)) {
      char *cp;
      strcpy(get_server_title,ibuf + 9);
      if ((cp = strchr(get_server_title,'\n')) != 0) *cp = '\0';
      }
    else if ((!*get_server_from) && !strncmp(ibuf,"From:",5)) {
      char *cp;
      strcpy(get_server_from,ibuf + 6);
      if ((cp = strchr(get_server_from,'\n')) != 0) *cp = '\0';
      }
    }
  fclose(fpw);
  if ((!data) || (strcmp(ibuf,".\n"))) {
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    while (!delete(scratch_file));
    sprintf(err_oline,"\tERROR - Item not located SERVER (%s)\n",node);
    err_line(err_oline);
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    return(0);
    }
  if (put_file) {
    sysprv();
    if (   (!(ifile = rms_open(scratch_file,0,0)))
        || (!(ofile = rms_create(put_file,0xa000,0)))) {
      if (ifile) rms_close(ifile);
      nosysprv();
      if ((fpw = fopen(scratch_file,"r")) != 0) {
        strcpy(sav_id,id);
        *tmp_name = 1;
        }
      else while (!delete(scratch_file));
      clear_err_line();
      return(fpw);
      }
    while (rms_get(ibuf,511,ifile)) rms_put(ibuf,ofile);
    rms_close(ifile);
    rms_close(ofile);
    nosysprv();
    while (!delete(scratch_file));
    if ((fpw = fopen(put_file,"r")) != 0) *tmp_name = 1;
    clear_err_line();
    return(fpw);
    }
  if ((fpw = fopen(scratch_file,"r")) != 0) {
    strcpy(sav_id,id);
    *tmp_name = 1;
    }
  else while (!delete(scratch_file));
  clear_err_line();
  return(fpw);
}

FILE *getno_server(itm,grp,node,tmp_name,put_file,proto)
  int itm,grp;
  const char *node, *put_file;
  char *tmp_name;
  int proto;
{
  FILE *fpw;
  char cmd[132];
  int data = 0, netresp = 0;
  struct rms_file *ifile, *ofile;

  if ((itm == sav_itm) && (grp == sav_grp)
   && ((fpw = fopen(scratch_file,"r")) != 0)) {
    *tmp_name = 1;
    return(fpw);
    }
  *sav_id = '\0';
  sav_itm = sav_grp = 0;
  while (!delete(scratch_file));

  if (!tmp_name) tmp_name = cmd;
  *tmp_name = '\0';
  if (!open_rem_chan(node,proto)) {
    sprintf(err_oline,"\tERROR - Cannot connect to NEWS SERVER (%s)\n",node);
    wmg(err_oline);
    return(0);
    }
  sprintf(cmd,"ARTICLE %d",itm);
  if (nntp_write(cmd)) {
    netresp = wait_net_response(CLIENT_TIMER,1);
    if (netresp == 412) {	/* 412 is no group specified so set group then try to get article */
      sprintf(cmd,"GROUP %s",ga[curr_g]->grp_name);
      if (!nntp_write(cmd) || (wait_net_response(CLIENT_TIMER,1) != 211)) {
        sprintf(err_oline,"\tERROR - Item not returned from SERVER (%s)\n",node);
        wmg(err_oline);
        return(0);
        }
      sprintf(cmd,"ARTICLE %d",itm);
      if (nntp_write(cmd)) netresp = wait_net_response(CLIENT_TIMER,1);
      else {
        sprintf(err_oline,"\tERROR - Item not located on SERVER (%s)\n",node);
        wmg(err_oline);
        return(0);
        }
      }
    }
  if (netresp != 220) {
    sprintf(err_oline,"\tERROR - Item not located on SERVER (%s)\n",node);
    wmg(err_oline);
    return(0);
    }
  fpw = fopen(scratch_file,"w");
  get_server_size = 0;
  *get_server_title = '\0';
  *get_server_from = '\0';
  *get_server_id = '\0';
  while (wait_net_response(RESP_TIMER,0) == 1) {
    if (!strcmp(ibuf,".\n")) break;
    if (*ibuf == '.') memmove(ibuf,ibuf + 1,strlen(ibuf));
    ++data;
    fputs(ibuf,fpw);
    if (!get_server_size && !strncmp(ibuf,"Lines:",6)) {
      if (!sscanf(ibuf,"Lines: %d",&get_server_size)) get_server_size = 0;
      }
    else if ((!*get_server_title) && !strncmp(ibuf,"Subject:",8)) {
      char *cp;
      strcpy(get_server_title,ibuf + 9);
      if ((cp = strchr(get_server_title,'\n')) != 0) *cp = '\0';
      }
    else if ((!*get_server_from) && !strncmp(ibuf,"From:",5)) {
      char *cp;
      strcpy(get_server_from,ibuf + 6);
      if ((cp = strchr(get_server_from,'\n')) != 0) *cp = '\0';
      }
    else if ((!*get_server_id) && !strncmp(ibuf,"Message-Id:",11)) {
      char *cp;
      strcpy(get_server_id,ibuf + 12);
      if ((cp = strchr(get_server_id,'\n')) != 0) *cp = '\0';
      }
    }
  fclose(fpw);
  if ((!data) || (strcmp(ibuf,".\n"))) {
    while (!delete(scratch_file));
    sprintf(err_oline,"\tERROR - Item not located on SERVER (%s)\n",node);
    wmg(err_oline);
    return(0);
    }
  if (put_file) {
    sysprv();
    if (   (!(ifile = rms_open(scratch_file,(char *)0,0)))
        || (!(ofile = rms_create(put_file,0xa000,0)))) {
      if (ifile) rms_close(ifile);
      nosysprv();
      if ((fpw = fopen(scratch_file,"r")) != 0) {
        sav_itm = itm; sav_grp = grp;
        *tmp_name = 1;
        }
      else while (!delete(scratch_file));
      clear_err_line();
      return(fpw);
      }
    while (rms_get(ibuf,511,ifile)) rms_put(ibuf,ofile);
    rms_close(ifile);
    rms_close(ofile);
    nosysprv();
    while (!delete(scratch_file));
    if ((fpw = fopen(put_file,"r")) != 0) *tmp_name = 1;
    clear_err_line();
    return(fpw);
    }
  if ((fpw = fopen(scratch_file,"r")) != 0) {
    sav_itm = itm; sav_grp = grp;
    *tmp_name = 1;
    }
  else while (!delete(scratch_file));
  clear_err_line();
  return(fpw);
}

/*
 *  get_server_header
 *
 *  Get a newsitem header from remote system
 */

FILE *get_server_header(id,node,tmp_name,proto,grp_name,itm_num)
  const char *id, *node, *grp_name;
  char *tmp_name;
  int proto, itm_num;
{
  FILE *fpw;
  char cmd[132];
  int data = 0, no_retrieve = 1;

  if (*id && !strcmp(id,sav_id) && (fpw = fopen(scratch_file,"r")) != 0) {
    *tmp_name = 1;
    return(fpw);
    }

  if (*id && !strcmp(id,sav_idh) && (fpw = fopen(scratch_head,"r")) != 0) {
    *tmp_name = 1;
    return(fpw);
    }
  *sav_idh = '\0';
  while (!delete(scratch_head));

  sprintf(err_oline,"\tConnecting to NEWS SERVER (%s) ...\n",node);
  err_line(err_oline);
  if (!tmp_name) tmp_name = cmd;
  *tmp_name = '\0';
  if (!open_rem_chan(node,proto)) {
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    sprintf(err_oline,"\tERROR - Cannot connect to NEWS SERVER (%s)\n",node);
    err_line(err_oline);
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    return(0);
    }

  if (*id) {
    sprintf(cmd,"HEAD %s",id);
    no_retrieve = (!nntp_write(cmd) || (wait_net_response(CLIENT_TIMER,1) != 221));
    }
  if (no_retrieve && nntp_client && *grp_name) {
    *nntp_currgroup = '\0';
    sprintf(cmd,"GROUP %s",grp_name);
    if (!(no_retrieve = (!nntp_write(cmd) || (wait_net_response(CLIENT_TIMER,1) != 211)))) {
      strcpy(nntp_currgroup,grp_name);
      sprintf(cmd,"HEAD %d",itm_num);
      no_retrieve = (!nntp_write(cmd) || (wait_net_response(CLIENT_TIMER,1) != 221));
      }
    }
  if (no_retrieve && !nntp_client) {
    char *nntp_response;
    unsigned int inum;
    int gidx;

    if (!(gidx = ga_exact_name(grp_name))) {
      sprintf(err_oline,"Cannot locate group %s",grp_name);
      err_line(err_oline);
      return(0);
      }
    sprintf(err_oline,"GROUP %s",grp_name);
    if (nntp_one_call(node,proto,err_oline,&nntp_response) == 211) {
      strcpy(nntp_currgroup,grp_name);
      if (!find_itm_by_id(gidx,id,&inum)) {
        sprintf(err_oline,"Can't locate item %s",id);
        err_line(err_oline);
        return(0);
        }
      sprintf(cmd,"HEAD %d",inum);
      no_retrieve = (!nntp_write(cmd) || (wait_net_response(CLIENT_TIMER,1) != 221));
      }
    }
  if (no_retrieve) {
    sprintf(err_oline,"NNTP> HEAD Error: %s",ibuf);
    wmg(err_oline);
    return(0);
    }
  fpw = fopen(scratch_head,"w");
  get_server_size = 0;
  *get_server_title = '\0';
  *get_server_from = '\0';
  while (wait_net_response(RESP_TIMER,0) == 1) {
    if (!strcmp(ibuf,".\n")) break;
    if (*ibuf == '.') memmove(ibuf,ibuf + 1,strlen(ibuf));
    ++data;
    fputs(ibuf,fpw);
    if (!get_server_size && !strncmp(ibuf,"Lines:",6)) {
      if (!sscanf(ibuf,"Lines: %d",&get_server_size)) get_server_size = 0;
      }
    else if ((!*get_server_title) && !strncmp(ibuf,"Subject:",8)) {
      char *cp;

      strcpy(get_server_title,ibuf + 9);
      if ((cp = strchr(get_server_title,'\n')) != 0) *cp = '\0';
      }
    else if ((!*get_server_from) && !strncmp(ibuf,"From:",5)) {
      char *cp;
      strcpy(get_server_from,ibuf + 6);
      if ((cp = strchr(get_server_from,'\n')) != 0) *cp = '\0';
      }
    }
  fclose(fpw);
  if ((!data) || (strcmp(ibuf,".\n"))) {
    while (!delete(scratch_head));
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    sprintf(err_oline,"\tERROR - Lost link to NEWS SERVER (%s)\n",node);
    err_line(err_oline);
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    return(0);
    }
  if ((fpw = fopen(scratch_head,"r")) != 0) {
    strcpy(sav_idh,id);
    *tmp_name = 1;
    }
  else while (!delete(scratch_head));
  clear_err_line();
  return(fpw);
}

/*
 *  server_check_ids
 *
 *  Obtain a directory from the remote system for a newsgroup
 */

void server_check_ids(check_id_head)
  struct grps *check_id_head;
{
  time_t daytime;
  int resp_val;
  struct grps *reply = check_id_head;
  struct itms *tmp;
  char cmd[132];
  struct tm *stm;

  while (reply) {
    if (open_rem_chan(reply->grpnode,reply->nproto)) {
      time(&daytime);
      if (   (reply->grptime > daytime)
          || (reply->grptime < 0))
	 reply->grptime = daytime - WEEK;
      stm = localtime((time_t *) &(reply->grptime));
      sprintf(cmd,"XUPDGROUP %s %02d%02d%02d %02d%02d%02d",reply->grpname,
	stm->tm_year,stm->tm_mon + 1,stm->tm_mday,
	stm->tm_hour,stm->tm_min,stm->tm_sec);
      if (!(NNTP_USE_XUPDGROUP) || nntp_write(cmd)) {
        if (!(NNTP_USE_XUPDGROUP)) resp_val = 500;
        else if ((resp_val = wait_net_response(CLIENT_TIMER,1)) == 480) {
          printf("\t\tNewsgroup not defined on server\n");
          reply = reply->grpsnext;
          continue;
          }
        if (resp_val == 380) {
          tmp = reply->ids;
          while (tmp) {
            sprintf(cmd,"%s",tmp->itmid);
            if (!nntp_write(cmd)) {
              printf("\t\tLost link to NEWS Server on %s\n",reply->grpnode);
              reply = reply->grpsnext;
              continue;
              }
            tmp = tmp->itmsnext;
            }
          if (!nntp_write(".")) {
            printf("\t\tLost link to NEWS Server on %s\n",reply->grpnode);
            reply = reply->grpsnext;
            continue;
            }
          if (!(wait_net_response(CLIENT_TIMER,1) == 280)) {
            printf("\t\tLost link to NEWS Server on %s\n",reply->grpnode);
            reply = reply->grpsnext;
            continue;
            }
          while (wait_net_response(RESP_TIMER,0) == 1) {
            if (!strcmp(ibuf,".\n")) break;
            tmp = (struct itms *) news_malloc(sizeof *tmp);
            if (*ibuf == 'D') {
              ibuf[strlen(ibuf) - 1] = '\0';
              util_idcpy(tmp->itmid,&ibuf[2]);
              tmp->itmdate = 0;
              tmp->itmsnext = reply->idr;
              reply->idr = tmp;
              }
            else if (sscanf(ibuf,"%s %d %d %[^\n]",tmp->itmid,&(tmp->itmdate),&(tmp->itmsize),tmp->itmtitle) == 4) {
              tmp->itmsnext = reply->idr;
              reply->idr = tmp;
              }
            else news_free(tmp);
            }
          }
        else {
          sprintf(cmd,"NEWNEWS %s %02d%02d%02d %02d%02d%02d",reply->grpname,stm->tm_year,stm->tm_mon + 1,stm->tm_mday,stm->tm_hour,stm->tm_min,stm->tm_sec);
          if (nntp_write(cmd) && (wait_net_response(CLIENT_TIMER,1) == 230)) {
            while (wait_net_response(RESP_TIMER,0) == 1) {
              if (!strcmp(ibuf,".\n")) break;
              tmp = (struct itms *) news_malloc(sizeof *tmp);
              tmp->itmsnext = reply->idr;
              reply->idr = tmp;
              sscanf(ibuf,"%s",tmp->itmid);
              time((time_t *) &tmp->itmdate);
              tmp->itmsize = 0;
              }
            }
          else printf("\t\tLost link to NEWS Server on %s\n",reply->grpnode);
          }
        }
      else printf("\t\tLost link to NEWS Server on %s\n",reply->grpnode);
      }
    else printf("\t\tNo link to NEWS Server on %s\n",reply->grpnode);
    reply = reply->grpsnext;
    }
}

void server_get_titles(reply,irecheck)
  struct grps *reply;
  struct itms *irecheck;
{
  char cmd[132], *cp, *nntp_response;
  int count, first, last,
      try_xhdr = NNTP_USE_XHDR, try_head, resp_code;

  if (!irecheck) return;
  printf("\tTitle search on %s (Server: %s)\n",reply->grpname,reply->grpnode);
  sprintf(err_oline,"GROUP %s",reply->grpname);
  if ((nntp_one_call(reply->grpnode,reply->nproto,err_oline,&nntp_response) != 211)  ||
      (sscanf(nntp_response,"211 %d %d %d",&count,&first,&last) != 3)) {
    sprintf(err_oline,"NNTP: Group %s -> %s",reply->grpnode,nntp_response);
    err_line(err_oline);     
    return;
    }
  strcpy(nntp_currgroup,reply->grpname);
  while (irecheck) {
    *(irecheck->itmtitle) = '\0';
    *(irecheck->itmfrom) = '\0';
    irecheck->itmvalid = 0;
    try_head = 1;
    if (try_xhdr) {
      resp_code = 0;
      sprintf(cmd,"XHDR Subject %s",irecheck->itmid);
      if ((nntp_write(cmd)) && ((resp_code = wait_net_response(CLIENT_TIMER,1)) == 221)) {
        irecheck->itmvalid = 1;
        while (wait_net_response(RESP_TIMER,0) == 1) {
          if (!strcmp(ibuf,".\n")) break;
	  if ((cp = strchr(ibuf,'\n')) != 0) *cp = '\0';
	  if ((cp = strchr(ibuf,'>')) != 0) {
            cp += 2;
            util_subjcpy(irecheck->itmtitle,cp);
            try_head = 0;
            }
          }
        sprintf(cmd,"XHDR From %s",irecheck->itmid);
        if ((nntp_write(cmd)) && ((resp_code = wait_net_response(CLIENT_TIMER,1)) == 221)) {
          while (wait_net_response(RESP_TIMER,0) == 1) {
            if (!strcmp(ibuf,".\n")) break;
	    if ((cp = strchr(ibuf,'\n')) != 0) *cp = '\0';
	    if ((cp = strchr(ibuf,'>')) != 0) {
              cp += 2;
              util_fromcpy(irecheck->itmfrom,cp);
              }
            }
          }
        sprintf(cmd,"XHDR Date %s",irecheck->itmid);
        if ((nntp_write(cmd)) && ((resp_code = wait_net_response(CLIENT_TIMER,1)) == 221)) {
          while (wait_net_response(RESP_TIMER,0) == 1) {
            if (!strcmp(ibuf,".\n")) break;
	    if ((cp = strchr(ibuf,'\n')) != 0) *cp = '\0';
	    if ((cp = strchr(ibuf,'>')) != 0) {
              cp += 2;
              irecheck->itmdate = parse_usenet_date(cp);
              }
            }
          }
        if (!(irecheck->itmsize)) {
          sprintf(cmd,"XHDR lines %s",irecheck->itmid);
          if ((nntp_write(cmd)) && (wait_net_response(CLIENT_TIMER,1) == 221)) {
            while (wait_net_response(RESP_TIMER,0) == 1) {
              if (!strcmp(ibuf,".\n")) break;
	      if ((cp = strchr(ibuf,'\n')) != 0) *cp = '\0';
	      if ((cp = strchr(ibuf,'>')) != 0) {
                cp += 2;
                if (!sscanf(cp,"%d",&(irecheck->itmsize))) irecheck->itmsize = 0;
                }
              }
            }
          }
        }
      else if (resp_code == 500) try_xhdr = 0;
      }
    if (try_head) {
      unsigned int inum;
      int gidx;

      sprintf(cmd,"HEAD %s",irecheck->itmid);
      if (!nntp_write(cmd)) {
        sprintf(err_oline,"NNTP: Channel closed - write fail");
        err_line(err_oline);     
        return;
        }
      if ((resp_code = wait_net_response(CLIENT_TIMER,1)) != 221) {
      if (!(gidx = ga_exact_name(reply->grpname))) {
        sprintf(err_oline,"Cannot locate group %s",reply->grpname);
        err_line(err_oline);
        return;
        }
      if (!find_itm_by_id(gidx,irecheck->itmid,&inum)) {
        sprintf(err_oline,"Can't locate item %s",irecheck->itmid);
        err_line(err_oline);
        return;
        }
      sprintf(cmd,"HEAD %d",inum);
      if (!nntp_write(cmd)) {
        sprintf(err_oline,"NNTP: Channel closed - write fail");
        err_line(err_oline);     
        return;
        }
      resp_code = wait_net_response(CLIENT_TIMER,1);
      }
      if (resp_code == 221) {
        irecheck->itmvalid = 1;
        while (wait_net_response(RESP_TIMER,0) == 1) {
          if (!strcmp(ibuf,".\n")) break;
	  if (*ibuf == '.') memmove(ibuf,ibuf + 1,strlen(ibuf));
	  if ((cp = strchr(ibuf,'\n')) != 0) *cp = '\0';
          if (!strncmp(ibuf,"Subject:",8)) {
	    util_subjcpy(irecheck->itmtitle,ibuf + 9);
	    }
          else if (!strncmp(ibuf,"From:",5)) {
	    util_fromcpy(irecheck->itmfrom,ibuf + 6);
	    }
          else if (!strncmp(ibuf,"Date:",5)) {
	    irecheck->itmdate = parse_usenet_date(ibuf + 6);
	    }
          else if (!strncmp(ibuf,"Lines:",6)) {
            if (!sscanf(ibuf,"Lines: %d",&(irecheck->itmsize))) irecheck->itmsize = 0;
            }
          }
        }
      }
    if (!*(irecheck->itmtitle)) irecheck->itmvalid = 0;
    irecheck = irecheck->itmsnext;
    }
}

char *server_get_newsgroups(node,proto,sincetime)
  char *node;
  int proto;
  time_t sincetime;
{
  char cmd[132];
  int malloc_size = 0,
      resp_size = 0;
  char *return_buf = 0;
  struct tm *stm;

  sprintf(err_oline,"\tConnecting to NEWS SERVER (%s) ...\n",node);
  err_line(err_oline);
  if (!open_rem_chan(node,proto)) {
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    sprintf(err_oline,"\tERROR - Cannot connect to NEWS SERVER (%s)\n",node);
    err_line(err_oline);
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    return(0);
    }
  if (sincetime) {
    stm = localtime(&sincetime);
    sprintf(cmd,"NEWGROUPS %02d%02d%02d %02d%02d%02d",stm->tm_year,stm->tm_mon + 1,stm->tm_mday,stm->tm_hour,stm->tm_min,stm->tm_sec);
    }
  else sprintf(cmd,"LIST");
  if (!nntp_write(cmd) || (wait_net_response(CLIENT_TIMER,1) != 215)) {
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    sprintf(err_oline,"\tERROR - Lost link to NEWS SERVER (%s)\n",node);
    err_line(err_oline);
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    return(0);
    }
  return_buf = (char *) news_malloc(malloc_size = 512);
  *return_buf = '\0';
  while (wait_net_response(RESP_TIMER,0) == 1) {
    if (!strcmp(ibuf,".\n")) break;
    resp_size += strlen(ibuf);
    if (malloc_size < resp_size + 10)
      return_buf = (char *) news_realloc(return_buf,
        (malloc_size += (resp_size + 512)));
    strcat(return_buf,ibuf);
    }
  if (strcmp(ibuf,".\n")) {
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    sprintf(err_oline,"\tERROR - Lost link to NEWS SERVER (%s)\n",node);
    err_line(err_oline);
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    }
  return(return_buf);
}

/*
 *  s_to_upper
 *
 *  convert a string to lower case
 */

char *s_to_upper(s)
    char *s;
{
  char *save = s;

  while (*s) {
    *s = toupper(*s);
    s++;
    }
  return(save);
}

void close_nntp_file()
{
  if (*scratch_file) while (!delete(scratch_file));
  if (*scratch_head) while (!delete(scratch_head));
}

static char *nc_resp = 0;
static int nc_len = 0, nc_malloc = 0;


int nntp_call(node,proto,xcmd,retbuf)
  char *node; int proto; char *xcmd; char **retbuf;
{
  int respval;

  if (nc_resp) news_free(nc_resp);
  nc_len = nc_malloc = 0,  nc_resp = 0;
  if (!open_rem_chan(node,proto)) {
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    sprintf(err_oline,"\tERROR - Cannot connect to NEWS SERVER (%s)\n",node);
    err_line(err_oline);
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    return(0);
    }
  if (!nntp_write(xcmd) || !(respval = wait_net_response(CLIENT_TIMER,1))) {
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    sprintf(err_oline,"\tERROR - Lost link to NEWS SERVER (%s)\n",node);
    err_line(err_oline);
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    *retbuf = 0;
    return 0;
    }
  nc_resp = news_malloc(nc_malloc = ((nc_len = strlen(ibuf)) + 512));
  strcpy(nc_resp,ibuf);
  if ((respval / 100) != 2) {	/* galss@mgi.com  V6.0-3  4/2/91 */
    *retbuf = nc_resp;
    return(respval);
    }
  while (wait_net_response(RESP_TIMER,0) == 1) {
    if (!strcmp(ibuf,".\n")) break;
    nc_len += strlen(ibuf);
    if (nc_malloc < nc_len + 10)
      nc_resp = news_realloc(nc_resp,
                          (nc_malloc += (nc_len + 512)));
    strcat(nc_resp,ibuf);
    }
  if (strcmp(ibuf,".\n")) {
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    sprintf(err_oline,"\tERROR - Lost link to NEWS SERVER (%s)\n",node);
    err_line(err_oline);
    if (smg_active) {
      smg$end_pasteboard_update(&pid);
      smg$begin_pasteboard_update(&pid);
      }
    }
  *retbuf = nc_resp;
  return(respval);
}

int nntp_one_call(node,proto,xcmd,retbuf)
  const char *node; int proto; const char *xcmd; char **retbuf;
{
  int respval;

  if (nc_resp) news_free(nc_resp);
  nc_len = nc_malloc = 0,  nc_resp = 0;
  if (!open_rem_chan(node,proto)) return(0);
  if (!nntp_write(xcmd) || !(respval = wait_net_response(CLIENT_TIMER,1))) {
    *retbuf = 0;
    return 0;
  }
  nc_resp = news_malloc(strlen(ibuf) + 1);
  strcpy(nc_resp,ibuf);
  *retbuf = nc_resp;
  return(respval);
}


int nntp_get_info(node,proto,xcmd,code,fcall,infocall)
  const char *node;
  int proto;
  const char *xcmd;
  int code;
  int (*fcall)();
  int (*infocall)();
{
  int respval;

  if (!open_rem_chan(node,proto)) return(0);
  if (!nntp_write(xcmd) || !(respval = wait_net_response(CLIENT_TIMER,1)))
    return(0);
  if (respval != code) return(0);
  if (fcall) (*fcall)(ibuf);
  while (wait_net_response(RESP_TIMER,0) == 1) {
    if (!strcmp(ibuf,".\n")) break;
    (*infocall)(ibuf);
    }
  return(1);
}

/*
 * post_to_server
 *
 * Post the item in "filename" to the server using the POST command
 */

int  post_to_server(node,proto,filename)	char *node; int proto; char *filename;
{
  int status;
  int respval;
  FILE *fpr;
  char *cp, inpline[1024];

  if (!(fpr = fopen(filename,"r"))) {
    sprintf(err_oline,"NNTP-POST: Error: Cannot open file %s",filename);
    err_line(err_oline);
    return(0);
    }
  if (!open_rem_chan(node,proto)) {
    sprintf(err_oline,"NNTP-POST: Error: Cannot connect to %s",node);
    err_line(err_oline);
    fclose(fpr);
    return(0);
    }
again:
  if (!nntp_write("POST") || !(respval = wait_net_response(CLIENT_TIMER,1))) {
    sprintf(err_oline,"NNTP-POST: Error: I/O error to %s",node);
    err_line(err_oline);
    fclose(fpr);
    return(0);
    }
#ifdef AUTH
  if (respval == 381 || respval == 480) {
    if ((cp = strchr(ibuf,'\n')) != 0) *cp = '\0';
    sprintf(err_oline,"NNTP-POST: %s",ibuf);
    err_line(err_oline);
    status = get_input(&usr_inp_dsc,c$dsc("NNTP-Username: "),&usr_inp_l);
    if ((status == RMS$_EOF) || (status == SMG$_EOF)) {
      fclose(fpr);
      return(0);
      }
    if ((status & 1) && (usr_inp_l)) {
      usr_inp[usr_inp_l] = '\0';
      sprintf(inpline,"AUTHINFO USER %s\n",usr_inp);
      }
    if (!nntp_write(inpline) || !(respval = wait_net_response(CLIENT_TIMER,1))) {
      sprintf(err_oline,"NNTP-POST: Error: I/O error to %s",node);
      err_line(err_oline);
      fclose(fpr);
      return(0);
      }
    if (respval == 381) {
      if ((cp = strchr(ibuf,'\n')) != 0) *cp = '\0';
      sprintf(err_oline,"NNTP-POST: %s",ibuf);
      err_line(err_oline);
      status = get_input(&usr_inp_dsc,c$dsc("NNTP-Password: "),&usr_inp_l);
      if ((status == RMS$_EOF) || (status == SMG$_EOF)) {
        fclose(fpr);
        return(0);
        }
      if ((status & 1) && (usr_inp_l)) {
        usr_inp[usr_inp_l] = '\0';
        sprintf(inpline,"AUTHINFO PASS %s\n",usr_inp);
        }
      if (!nntp_write(inpline) || !(respval = wait_net_response(CLIENT_TIMER,1))) {
        sprintf(err_oline,"NNTP-POST: Error: I/O error to %s",node);
        err_line(err_oline);
        fclose(fpr);
        return(0);
        }
    if (respval == 281)
      goto again;
    }
    sprintf(err_oline,"NNTP-POST: Error: %s",ibuf);
    err_line(err_oline);
    fclose(fpr);
    return(0);
    }
#endif

  if (respval != 340) {
    if ((cp = strchr(ibuf,'\n')) != 0) *cp = '\0';
    sprintf(err_oline,"NNTP-POST Refused: %s",ibuf);
    err_line(err_oline);
    fclose(fpr);
    return(0);
    }
  while (fgets(inpline,1024,fpr)) {
    if ((cp = strchr(inpline,'\n')) != 0) *cp = '\0';
    if (*inpline == '.') memmove(inpline + 1,inpline,strlen(inpline) + 1);
    if (!nntp_write(inpline)) {
      sprintf(err_oline,"NNTP-POST: Error: I/O error to %s",node);
      err_line(err_oline);
      fclose(fpr);
      return(0);
      }
    }
  if (!nntp_write(".")) {
    sprintf(err_oline,"NNTP-POST: Error: I/O error to %s",node);
    err_line(err_oline);
    fclose(fpr);
    return(0);
    }
  if (!(respval = wait_net_response(CLIENT_TIMER,1))) {
    sprintf(err_oline,"NNTP-POST: Error: I/O error to %s",node);
    err_line(err_oline);
    fclose(fpr);
    return(0);
    }
  if ((cp = strchr(ibuf,'\n')) != 0) *cp = '\0';
  if (respval != 240) {
    sprintf(err_oline,"NNTP-POST Refused: %s",ibuf);
    err_line(err_oline);
    fclose(fpr);
    return(0);
    }
  sprintf(err_oline,"NNTP-POST Accepted: %s",ibuf);
  err_line(err_oline);
  fclose(fpr);
  return(1);
}
